/* $Id: threads.h,v 1.10 2010/06/05 19:36:35 fredette Exp $ */

/* tme/threads.h - header file for threads: */

/*
 * Copyright (c) 2003 Matt Fredette
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Matt Fredette.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef _TME_THREADS_H
#define _TME_THREADS_H

#include <tme/common.h>

/* includes: */
#include <errno.h>

/* our errno convention: */
#define TME_EDEADLK		EDEADLK
#define TME_EBUSY		EBUSY
#define TME_THREADS_ERRNO(rc)	(rc)

typedef tme_uint64_t tme_time_t;
#define TME_NSEC_PER_TICK (1000000000 / TME_FRAC_PER_SEC)
#define TME_FRAC_PER_MSEC (TME_FRAC_PER_SEC / 1000)
#define TME_FRAC_PER_USEC (TME_FRAC_PER_SEC / 1000000)

#define TME_TIME_GET_SEC(a) ((a) / TME_FRAC_PER_SEC)
#define TME_TIME_GET_MSEC(a) ((a) / TME_FRAC_PER_MSEC)
#define TME_TIME_GET_USEC(a) ((a) / TME_FRAC_PER_USEC)
#define TME_TIME_GET_NSEC(a) ((a) * TME_NSEC_PER_TICK)
#define TME_TIME_SET_SEC(a) ((tme_time_t)(a) * TME_FRAC_PER_SEC)
#define TME_TIME_SET_MSEC(a) ((tme_time_t)(a) * TME_FRAC_PER_MSEC)
#define TME_TIME_SET_USEC(a) ((tme_time_t)(a) * TME_FRAC_PER_USEC)
#define TME_TIME_SET_NSEC(a) ((tme_time_t)(a) / TME_NSEC_PER_TICK)

/* setjmp/longjmp threading: */

#define glib 0
#define sdl 1

#if TME_THREADS_TYPE == glib && defined(_TME_HAVE_GLIB)
#include "threads-glib.h"
#define TME_THREADS_NAME "glib"
#endif

#if TME_THREADS_TYPE == sdl && defined(_TME_HAVE_SDL3)
#include "threads-sdl.h"
#define TME_THREADS_NAME "sdl"
#endif

#ifndef TME_THREADS_NAME
#include "threads-posix.h"
#define TME_THREADS_NAME "posix"
#define TME_THREADS_POSIX
#endif

#undef glib
#undef sdl

#include "threads-fiber.h"

extern int thread_mode;

typedef struct tme_rwlock {
  union {
    tme_thread_rwlock_t thread;
    tme_fiber_rwlock_t fiber;
  } lock;
  _tme_threadid_t writer;
} tme_rwlock_t;

extern tme_rwlock_t tme_rwlock_suspere;

#define tme_thread_op(func,arg) ((thread_mode) ? (tme_thread_##func(&(arg)->thread)) : (tme_fiber_##func(&(arg)->fiber)))
#define tme_thread_opt(func,arg) ((thread_mode) ? (tme_thread_##func((arg).thread)) : (tme_fiber_##func((arg).fiber)))
#define tme_thread_op2(func,arg,arg2) ((thread_mode) ? (tme_thread_##func(&(arg)->thread,&(arg2)->thread)) : (tme_fiber_##func(&(arg)->fiber,&(arg2)->fiber)))
#define tme_thread_opt3(func,arg,arg2,arg3) ((thread_mode) ? (tme_thread_##func(&(arg)->thread,&(arg2)->thread,(arg3).thread)) : (tme_fiber_##func(&(arg)->fiber,&(arg2)->fiber,(arg3).fiber)))

static _tme_inline void tme_rwlock_init _TME_P((tme_rwlock_t *l)) {
  (l)->writer = 0;
  tme_thread_op(rwlock_init,&(l)->lock);
}

#define tme_rwlock_destroy(l) tme_thread_op(rwlock_destroy,&(l)->lock)

#define _tme_rwlock_rdlock(l)           tme_thread_op(rwlock_rdlock,&(l)->lock);
#define _tme_rwlock_wrlock(l)           tme_thread_op(rwlock_wrlock,&(l)->lock);
#define _tme_rwlock_wrunlock(l)         tme_thread_op(rwlock_wrunlock,&(l)->lock)

#define tme_rwlock_rdunlock(l) tme_thread_op(rwlock_rdunlock,&(l)->lock)
#define tme_rwlock_tryrdlock(l) tme_thread_op(rwlock_tryrdlock,&(l)->lock)
int tme_rwlock_rdlock _TME_P((tme_rwlock_t *l));
int tme_rwlock_wrlock _TME_P((tme_rwlock_t *l));
int tme_rwlock_wrunlock _TME_P((tme_rwlock_t *l));
int tme_rwlock_trywrlock _TME_P((tme_rwlock_t *l));

/* thread suspension: */
#ifndef tme_thread_cooperative
#define tme_thread_cooperative() (thread_mode == FALSE)
#endif

#define _tme_thread_suspended()	        if(!tme_thread_cooperative()) tme_rwlock_rdunlock(&tme_rwlock_suspere)
#define _tme_thread_resumed()	        if(!tme_thread_cooperative()) _tme_rwlock_rdlock(&tme_rwlock_suspere)
#define tme_thread_suspend_others()	_tme_thread_suspended();if(!tme_thread_cooperative()) _tme_rwlock_wrlock(&tme_rwlock_suspere)
#define tme_thread_resume_others()	if(!tme_thread_cooperative()) _tme_rwlock_wrunlock(&tme_rwlock_suspere);_tme_thread_resumed()

#ifdef tme_thread_rwlock_timedrdlock
int tme_rwlock_timedlock _TME_P((tme_rwlock_t *l, tme_time_t abstime, int write));
#define tme_rwlock_timedrdlock(l,sec) tme_rwlock_timedlock(l,sec,0)
#define tme_rwlock_timedwrlock(l,sec) tme_rwlock_timedlock(l,sec,1)
#else
// TODO: why do thread_mode trylocks not work here?
#define tme_rwlock_timedrdlock(l,sec)     ((thread_mode) ? (tme_rwlock_rdlock(l)) : (tme_rwlock_tryrdlock(l)))
#define tme_rwlock_timedwrlock(l,sec)     ((thread_mode) ? (tme_rwlock_wrlock(l)) : (tme_rwlock_trywrlock(l)))
#endif

/* mutexes: */
typedef union {
  tme_thread_mutex_t thread;
  tme_fiber_mutex_t fiber;
} tme_mutex_t;
  
#define tme_mutex_init(m) tme_thread_op(mutex_init,m)
#define tme_mutex_destroy(m) tme_thread_op(mutex_clear,m)
#define tme_mutex_trylock(m) tme_thread_op(mutex_trylock,m)
#define tme_mutex_unlock(m) tme_thread_op(mutex_unlock,m)

#ifdef tme_thread_mutex_timedlock
int tme_mutex_timedlock _TME_P((tme_mutex_t *m, tme_time_t abstime));
#else
#define tme_mutex_timedlock(m,t) tme_mutex_trylock(m)
#endif

/* conditions: */
typedef union {
  tme_thread_cond_t thread;
  tme_fiber_cond_t fiber;
} tme_cond_t;

#define tme_cond_init(c) tme_thread_op(cond_init, c)
#define tme_cond_destroy(c) tme_thread_op(cond_clear, c)
#define tme_cond_wait(c) tme_thread_op(cond_wait, c)
#define tme_cond_notify(cond,bc) tme_thread_op(cond_notify##bc, cond)
int tme_cond_wait_yield _TME_P((tme_cond_t *cond, tme_mutex_t *mutex));
int tme_cond_sleep_yield _TME_P((tme_cond_t *cond, tme_mutex_t *mutex, tme_time_t sleep));

typedef union {
  tme_thread_time_t thread;
  tme_time_t fiber;
} tme_timeout_t;

#define tme_get_timeout(s,t,a) ((thread_mode) ? (tme_thread_get_timeout(s,&((t).thread),a)) : ((t).fiber=(s)))

void tme_thread_yield _TME_P((void));

#ifdef _TME_HAVE_ZLIB
#include "zlib.h"
struct tme_zlib_handle {
  gzFile handle;
  int fd;
};
struct tme_zlib_handle  *tme_zlib_open _TME_P((const char *path, int flags));
#endif

typedef void (*tme_threads_fn) _TME_P((void));
typedef void (*tme_threads_fn1) _TME_P((void *));

static _tme_inline void tme_mutex_lock _TME_P((tme_mutex_t *mutex)) { 
  _tme_thread_suspended();
  
  tme_thread_op(mutex_lock, mutex);

  _tme_thread_resumed();
}

static _tme_inline void tme_thread_enter _TME_P((tme_mutex_t *mutex)) {
#ifndef WIN32
  static int init=TRUE;
  if(!init) {
    return;
  }
  init=thread_mode;
#endif
  _tme_thread_resumed();
  if(mutex)
    tme_thread_op(mutex_lock, mutex);
}

int tme_thread_sleep_yield _TME_P((tme_time_t time, tme_mutex_t *mutex));

void tme_threads_init(int mode);

/* time: */
#define tme_thread_get_time() ((thread_mode) ? (tme_thread_time()) : (tme_fiber_get_time()))

#if defined(_TME_HAVE_GMTIME_R) || defined(_TME_HAVE_GMTIME_S) || defined(_TME_HAVE_GMTIME)
typedef struct tm tme_date_t;
static _tme_inline tme_date_t *tme_time_get_date _TME_P((tme_time_t time, tme_date_t *date)) {
  time_t sec = TME_TIME_GET_SEC(time);
#ifdef _TME_HAVE_GMTIME_S
  gmtime_s(date, &sec);
  return date;
#elif defined(_TME_HAVE_GMTIME_R)
  return gmtime_r(&sec, date);
#else
  return date = gmtime(&sec);
#endif
}
#define TME_DATE_SEC(date) ((date)->tm_sec)
#define TME_DATE_MIN(date) ((date)->tm_min)
#define TME_DATE_HOUR(date) ((date)->tm_hour)
#define TME_DATE_WDAY(date) ((date)->tm_wday)
#define TME_DATE_MDAY(date) ((date)->tm_mday)
#define TME_DATE_MONTH(date) ((date)->tm_mon + 1)
#define TME_DATE_YEAR(date) ((date)->tm_year)
#endif

/* I/O: */
#ifdef WIN32

/* file flags: */
#define TME_FILE_RO		GENERIC_READ
#define TME_FILE_WO		GENERIC_WRITE
#define TME_FILE_RW		GENERIC_READ | GENERIC_WRITE
#define TME_FILE_NB		0

#ifdef TME_HAVE_INT64_T
typedef tme_int64_t tme_off_t;
#else
#error "No support for 32-bit file offsets on Windows"
#endif

typedef struct tme_win32_handle *tme_thread_handle_t;

extern tme_thread_handle_t win32_stdin;
extern tme_thread_handle_t win32_stdout;
extern tme_thread_handle_t win32_stderr;

tme_thread_handle_t tme_thread_open _TME_P((const char *path, int flags));
void tme_thread_close _TME_P((tme_thread_handle_t));

int tme_read _TME_P((tme_thread_handle_t hand, void *data, int len));
int tme_write _TME_P((tme_thread_handle_t hand, void *data, int len));

#define tme_thread_fd(hand, flags) _open_osfhandle((intptr_t)TME_THREAD_HANDLE(hand), flags);
#define TME_SEEK_SET FILE_BEGIN
#define TME_SEEK_CUR FILE_CURRENT
#define TME_SEEK_END FILE_END

typedef tme_thread_handle_t tme_event_t;

#define TME_STD_HANDLE(hand) win32_##hand

#define TME_THREAD_HANDLE(hand) (*(HANDLE *)(hand))
#define TME_EVENT_HANDLE(hand) hand
#define TME_INVALID_HANDLE NULL
tme_off_t tme_thread_seek _TME_P((tme_thread_handle_t hand, tme_off_t off, int where));
#elif defined(USE_ZLIB) && defined(_TME_HAVE_ZLIB)
/* file flags: */
typedef struct tme_zlib_handle  *tme_thread_handle_t;
#define TME_FILE_RO		O_RDONLY
#define TME_FILE_WO		O_WRONLY
#define TME_FILE_RW		O_RDWR
#define TME_FILE_NB		0
#define TME_THREAD_HANDLE(hand) hand->handle
#define TME_EVENT_HANDLE(hand) hand->fd
#define TME_INVALID_HANDLE NULL
#define TME_STD_HANDLE(hand) fileno(hand)
typedef uintptr_t tme_event_t;
#define tme_thread_fd(hand, flags) TME_EVENT_HANDLE(hand)
#define tme_read(hand,buf,len) gzread(TME_THREAD_HANDLE(hand),buf,len)
#define tme_write(hand,buf,len) gzwrite(TME_THREAD_HANDLE(hand),buf,len)
#define TME_SEEK_SET SEEK_SET
#define TME_SEEK_CUR SEEK_CUR
#define TME_SEEK_END SEEK_END
typedef z_off_t tme_off_t;
#define tme_thread_seek(hand,off,where) gzseek(TME_THREAD_HANDLE(hand),off,where)
#define tme_thread_open tme_zlib_open
#define tme_thread_close(hand) gzclose(TME_THREAD_HANDLE(hand))

#else // HAVE_ZLIB
/* file flags: */
#define TME_FILE_RO		O_RDONLY
#define TME_FILE_WO		O_WRONLY
#define TME_FILE_RW		O_RDWR
#define TME_FILE_NB		O_NONBLOCK
#define TME_THREAD_HANDLE(hand) hand
#define TME_EVENT_HANDLE(hand) hand
#define TME_INVALID_HANDLE -1
#define TME_STD_HANDLE(hand) fileno(hand)
typedef uintptr_t tme_event_t;
typedef tme_event_t tme_thread_handle_t;
#define tme_thread_fd(hand, flags) hand
#define tme_read read
#define tme_write write
#define TME_SEEK_SET SEEK_SET
#define TME_SEEK_CUR SEEK_CUR
#define TME_SEEK_END SEEK_END
typedef off_t tme_off_t;
#define tme_thread_seek lseek
#define tme_thread_open open
#define tme_thread_close close
#endif // !WIN32

int tme_event_yield _TME_P((tme_event_t hand, void *data, size_t len, bool read, tme_mutex_t *mutex));

static _tme_inline
int tme_thread_read(tme_thread_handle_t hand, void *data, size_t len, tme_mutex_t *mutex) {
  tme_event_yield(TME_EVENT_HANDLE(hand), data, len, true, mutex);
  return tme_read(hand, data, len);
}

static _tme_inline
int tme_thread_write(tme_thread_handle_t hand, void *data, size_t len, tme_mutex_t *mutex) {
  tme_event_yield(TME_EVENT_HANDLE(hand), data, len, false, mutex);
  return tme_write(hand, data, len);
}

typedef union {
  tme_thread_threadid_t thread;
  tme_fiber_thread_t *fiber;
} tme_threadid_t;

static _tme_inline
void tme_thread_create_named _TME_P((tme_threadid_t *thr, const char *name, tme_thread_t func, void *arg)) {
  if(thread_mode) 
    (thr)->thread = tme_thread_new(name,func,arg);
  else
    (thr)->fiber = tme_fiber_new(name,func,arg);
}

#define tme_thread_create(t,f,a) tme_thread_create_named(t,element->tme_element_args[0],f,a)
#define tme_thread_exit(m) return _tme_thread_exit(m)

static _tme_inline _tme_thret _tme_thread_exit _TME_P((tme_mutex_t *mutex)) {
  if(thread_mode) {
    _tme_thread_suspended();  
    if(mutex)
      tme_thread_mutex_unlock(&mutex->thread);
  } else tme_fiber_exit(&mutex->fiber);
  return NULL;
}

/* A default main iterator for use in the main thread loop */
static _tme_inline void tme_threads_main_iter _TME_P((void *usec)) {
  if(!thread_mode) tme_fiber_main_iter(usec);
  //  g_usleep((usec) ? (uintptr_t)usec : 1000000);
}

#define _tme_threads_main_iter(fn) if(fn) fn()
#endif /* !_TME_THREADS_H */
