










with Ada.Containers;                  use Ada.Containers;
with Ada.Containers.Hashed_Maps;
with Ada.Containers.Vectors;
with Ada.Directories;
with Ada.Exceptions;
with Ada.Finalization;
with Ada.Strings.Unbounded;           use Ada.Strings.Unbounded;
with Ada.Strings.Wide_Wide_Unbounded; use Ada.Strings.Wide_Wide_Unbounded;

pragma Warnings (Off, "internal");
with Ada.Strings.Wide_Wide_Unbounded.Aux;
use Ada.Strings.Wide_Wide_Unbounded.Aux;
pragma Warnings (On, "internal");

with Ada.Text_IO;                     use Ada.Text_IO;
with Ada.Unchecked_Conversion;
with Ada.Unchecked_Deallocation;
with System;
with System.Memory;

with GNAT.Task_Lock;
with GNAT.Traceback.Symbolic;

with GNATCOLL.Traces;

with Liblktlang_Support.Adalog.Debug;
with Liblktlang_Support.Generic_API.Analysis;
with Liblktlang_Support.Generic_API.Introspection;
with Liblktlang_Support.Hashes; use Liblktlang_Support.Hashes;
with Liblktlang_Support.Images; use Liblktlang_Support.Images;
with Liblktlang_Support.Names;  use Liblktlang_Support.Names;
with Liblktlang_Support.Relative_Get;

with Liblktlang.Private_Converters;
use Liblktlang.Private_Converters;

pragma Warnings (Off, "referenced");

          with Liblktlang.Default_Provider;
          with Liblktlang.Implementation.Extensions;
            use Liblktlang.Implementation.Extensions;

pragma Warnings (On, "referenced");



package body Liblktlang.Implementation is

   use Precomputed_Symbols;

   pragma Warnings (Off, "has no effect");
   use Solver;
   pragma Warnings (On, "has no effect");

   package Context_Vectors is new Ada.Containers.Vectors
     (Index_Type   => Positive,
      Element_Type => Internal_Context);

   type Contexts_Destructor is limited
      new Ada.Finalization.Limited_Controlled with null record;
   overriding procedure Finalize (CD : in out Contexts_Destructor);
   --  Helper to destroy all contexts when terminating the process

   package Context_Pool is

      procedure Acquire (Context : out Internal_Context)
         with Post => Context /= null;
      --  If a context is free for reuse, increment its serial number and
      --  return it. Otherwise, allocate a new one. In any case, this does not
      --  initialize it, except for the Serial_Number field.

      procedure Release (Context : in out Internal_Context)
         with Pre  => Context /= null,
              Post => Context = null;
      --  Tag Context as free for reuse and set it to null

      procedure Free;
      --  Free all contexts in this pool. Intended to be called only when the
      --  process is terminating, to avoid reported memory leaks.

   private

      Available : Context_Vectors.Vector;
      --  List of allocated contexts that can be re-used right now

      CD : Contexts_Destructor with Unreferenced;
      --  Singleton whose only purpose is to free all contexts in Available
      --  when finalized.

   end Context_Pool;

   procedure Register_Destroyable_Helper
     (Unit    : Internal_Unit;
      Object  : System.Address;
      Destroy : Destroy_Procedure);
   --  Common underlying implementation for Register_Destroyable_Gen

   pragma Warnings (Off, "referenced");
   function Construct_Entity_Array
     (V : AST_Envs.Entity_Vectors.Vector) return Internal_Entity_Array_Access;
   pragma Warnings (On, "referenced");

   procedure Reset_Envs_Caches (Unit : Internal_Unit);
   --  Reset the env caches of all lexical environments created for ``Unit``

   procedure Destroy (Env : in out Lexical_Env_Access);

   function Snaps_At_Start (Self : Bare_Lkt_Node) return Boolean;
   function Snaps_At_End (Self : Bare_Lkt_Node) return Boolean;

   --  Those maps are used to give unique ids to lexical envs while pretty
   --  printing them.

   package Address_To_Id_Maps is new Ada.Containers.Hashed_Maps
     (Lexical_Env, Integer, Hash, "=");

   type Dump_Lexical_Env_State is record
      Env_Ids : Address_To_Id_Maps.Map;
      --  Mapping: Lexical_Env -> Integer, used to remember which unique Ids we
      --  assigned to the lexical environments we found.

      Next_Id : Positive := 1;
      --  Id to assign to the next unknown lexical environment

      Root_Env : Lexical_Env;
      --  Lexical environment we consider a root (this is the Root_Scope from
      --  the current analysis context), or null if unknown.
   end record;
   --  Holder for the state of lexical environment dumpers

   function Get_Env_Id
     (E : Lexical_Env; State : in out Dump_Lexical_Env_State) return String;
   --  If E is known, return its unique Id from State. Otherwise, assign it a
   --  new unique Id and return it.

   procedure Print
     (Node        : Liblktlang_Support.Generic_API.Analysis.Lk_Node;
      Show_Slocs  : Boolean;
      Line_Prefix : String := "");
   --  Helper for the public overload, but working on the generic API node type

   ------------------------
   -- Precomputed_Symbol --
   ------------------------

   pragma Warnings (Off, "referenced");
   function Precomputed_Symbol
     (Index : Precomputed_Symbol_Index) return Text_Type is
   pragma Warnings (On, "referenced");
   begin
         declare
            Raw_Text : constant Text_Type := (case Index is
            
               when Precomputed_Sym_Abstract => "abstract",
               when Precomputed_Sym_All => "all",
               when Precomputed_Sym_Analysisunit => "AnalysisUnit",
               when Precomputed_Sym_Any => "any",
               when Precomputed_Sym_Array => "Array",
               when Precomputed_Sym_As => "as",
               when Precomputed_Sym_As_Bool => "as_bool",
               when Precomputed_Sym_Astlist => "ASTList",
               when Precomputed_Sym_Basictrait => "BasicTrait",
               when Precomputed_Sym_Bigint => "BigInt",
               when Precomputed_Sym_Bool => "Bool",
               when Precomputed_Sym_Builder => "builder",
               when Precomputed_Sym_Call => "__call__",
               when Precomputed_Sym_Char => "Char",
               when Precomputed_Sym_Dedent => "dedent",
               when Precomputed_Sym_Domain => "domain",
               when Precomputed_Sym_Dont_Skip => "dont_skip",
               when Precomputed_Sym_Entity => "Entity",
               when Precomputed_Sym_Env_Spec => "env_spec",
               when Precomputed_Sym_Envaction => "EnvAction",
               when Precomputed_Sym_Equation => "Equation",
               when Precomputed_Sym_Errornode => "ErrorNode",
               when Precomputed_Sym_False => "false",
               when Precomputed_Sym_Family => "family",
               when Precomputed_Sym_Ignore_Constructor_Arg => "ignore_constructor_arg",
               when Precomputed_Sym_Indent => "indent",
               when Precomputed_Sym_Indexable => "Indexable",
               when Precomputed_Sym_Int => "Int",
               when Precomputed_Sym_Internal => "__internal",
               when Precomputed_Sym_Invalid => "invalid",
               when Precomputed_Sym_Iterator => "Iterator",
               when Precomputed_Sym_Keep => "keep",
               when Precomputed_Sym_Lazy => "lazy",
               when Precomputed_Sym_List => "list",
               when Precomputed_Sym_List_Elements => "list_elements",
               when Precomputed_Sym_Logicvar => "LogicVar",
               when Precomputed_Sym_Metadata => "Metadata",
               when Precomputed_Sym_Metadata_46 => "metadata",
               when Precomputed_Sym_Newline => "newline",
               when Precomputed_Sym_No_Case => "no_case",
               when Precomputed_Sym_Node => "Node",
               when Precomputed_Sym_Node_49 => "node",
               when Precomputed_Sym_Nodebuilder => "NodeBuilder",
               when Precomputed_Sym_Null_Field => "null_field",
               when Precomputed_Sym_Nullable => "nullable",
               when Precomputed_Sym_Open => "open",
               when Precomputed_Sym_Parse_Field => "parse_field",
               when Precomputed_Sym_Pick => "pick",
               when Precomputed_Sym_Previous_Token => "previous_token",
               when Precomputed_Sym_Property => "property",
               when Precomputed_Sym_Propertyerror => "PropertyError",
               when Precomputed_Sym_Qualifier => "qualifier",
               when Precomputed_Sym_Regexp => "Regexp",
               when Precomputed_Sym_Root_Node => "root_node",
               when Precomputed_Sym_Rootnode => "RootNode__",
               when Precomputed_Sym_Self => "self",
               when Precomputed_Sym_Send => "send",
               when Precomputed_Sym_Skip => "skip",
               when Precomputed_Sym_Stop_Cut => "stop_cut",
               when Precomputed_Sym_String => "String",
               when Precomputed_Sym_Super => "super",
               when Precomputed_Sym_Symbol => "Symbol",
               when Precomputed_Sym_Synthetic => "synthetic",
               when Precomputed_Sym_T => "T",
               when Precomputed_Sym_Tokennode => "TokenNode",
               when Precomputed_Sym_True => "true",
               when Precomputed_Sym_Update => "update",
               when Precomputed_Sym_Var => "var",
               when Precomputed_Sym_With_Dynvars => "with_dynvars"
            );

            Symbol : constant Symbolization_Result :=
                  Create_Symbol (Raw_Text)
            ;
         begin
            if Symbol.Success then
               return Symbol.Symbol;
            else
               raise Program_Error with
                 "Cannot canonicalize symbol literal: " & Image (Raw_Text);
            end if;
         end;
   end Precomputed_Symbol;

   ----------------------------
   -- Construct_Entity_Array --
   ----------------------------

   function Construct_Entity_Array
     (V : AST_Envs.Entity_Vectors.Vector) return Internal_Entity_Array_Access
   is
      Ret : Internal_Entity_Array_Access :=
        Create_Internal_Entity_Array (V.Length);
   begin
      for J in V.First_Index .. V.Last_Index loop
         Ret.Items (J) := V.Get (J);
      end loop;

      declare
         Tmp : AST_Envs.Entity_Vectors.Vector := V;
      begin
         Tmp.Destroy;
      end;

      return Ret;
   end Construct_Entity_Array;

   -----------
   -- Image --
   -----------

   function Image (Self : Symbol_Type) return String_Type is
   begin
      return Create_String (Image (Self));
   end Image;

   ------------------
   -- Context_Pool --
   ------------------

   package body Context_Pool is

      -------------
      -- Acquire --
      -------------

      procedure Acquire (Context : out Internal_Context) is
      begin
         GNAT.Task_Lock.Lock;

         if Available.Is_Empty then
            Context := new Analysis_Context_Type;
            Context.Serial_Number := 1;
         else
            Context := Available.Last_Element;
            Available.Delete_Last;
         end if;

         GNAT.Task_Lock.Unlock;

         Context.Initialized := False;
         Context.Ref_Count := 1;

      exception
         when others =>
            GNAT.Task_Lock.Unlock;
            raise;
      end Acquire;

      -------------
      -- Release --
      -------------

      procedure Release (Context : in out Internal_Context) is
      begin
         GNAT.Task_Lock.Lock;

         Available.Append (Context);
         Context.Serial_Number := Context.Serial_Number + 1;
         Context := null;

         GNAT.Task_Lock.Unlock;

      exception
         when others =>
            GNAT.Task_Lock.Unlock;
            raise;
      end Release;

      ----------
      -- Free --
      ----------

      procedure Free is
      begin
         GNAT.Task_Lock.Lock;

         for C of Available loop
            Free (C);
         end loop;

         GNAT.Task_Lock.Unlock;

      exception
         when others =>
            GNAT.Task_Lock.Unlock;
            raise;
      end Free;

   end Context_Pool;

   --------------
   -- Finalize --
   --------------

   overriding procedure Finalize (CD : in out Contexts_Destructor) is
      pragma Unreferenced (CD);
   begin
      Context_Pool.Free;
   end Finalize;

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (File_Reader : in out Internal_File_Reader_Access) is
      procedure Destroy is new Ada.Unchecked_Deallocation
        (Internal_File_Reader'Class, Internal_File_Reader_Access);
   begin
      if File_Reader /= null and then File_Reader.all.Dec_Ref then
         Destroy (File_Reader);
      end if;
   end Dec_Ref;

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (Self : in out Internal_Event_Handler_Access) is
      procedure Destroy is new Ada.Unchecked_Deallocation
        (Internal_Event_Handler'Class, Internal_Event_Handler_Access);
   begin
      if Self /= null and then Self.all.Dec_Ref then
         Destroy (Self);
      end if;
   end Dec_Ref;

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (Provider : in out Internal_Unit_Provider_Access) is
      procedure Destroy is new Ada.Unchecked_Deallocation
        (Internal_Unit_Provider'Class, Internal_Unit_Provider_Access);
   begin
      if Provider /= null and then Provider.all.Dec_Ref then
         Destroy (Provider);
      end if;
   end Dec_Ref;

   ----------------
   -- Get_Env_Id --
   ----------------

   function Get_Env_Id
     (E : Lexical_Env; State : in out Dump_Lexical_Env_State) return String
   is
      C        : Address_To_Id_Maps.Cursor;
      Inserted : Boolean;
   begin
      if E = Null_Lexical_Env then
         return "$null";

      elsif E = State.Root_Env then
         --  Insert root env with a special Id so that we only print it once
         State.Env_Ids.Insert (E, -1, C, Inserted);
         return "$root";
      end if;

      State.Env_Ids.Insert (E, State.Next_Id, C, Inserted);
      if Inserted then
         State.Next_Id := State.Next_Id + 1;
      end if;

      return '@' & Stripped_Image (Address_To_Id_Maps.Element (C));
   end Get_Env_Id;

   pragma Warnings (Off, "referenced");
   function To_Lookup_Kind_Type (K : Lookup_Kind) return Lookup_Kind_Type
   is
     (Lookup_Kind_Type'Val (Lookup_Kind'Pos (K)));
   pragma Warnings (On, "referenced");

   ----------------------
   -- Allocate_Context --
   ----------------------

   function Allocate_Context return Internal_Context is
   begin
      return Context : Internal_Context do
         Context_Pool.Acquire (Context);
      end return;
   end Allocate_Context;

   ------------------------
   -- Initialize_Context --
   ------------------------

   procedure Initialize_Context
     (Context        : Internal_Context;
      Charset        : String;
      File_Reader    : Internal_File_Reader_Access;
      Unit_Provider  : Internal_Unit_Provider_Access;
      Event_Handler  : Internal_Event_Handler_Access;
      With_Trivia    : Boolean;
      Tab_Stop       : Positive)
   is
      Actual_Charset : constant String :=
        (if Charset = "" then Default_Charset else Charset);
      Symbols        : constant Precomputed_Symbol_Table
        := Create_Symbol_Table;
   begin
      Context.Rewriting_Handle := System.Null_Address;
      Context.Rewriting_Version := 1;
      Context.Initialized := True;
      Context.Symbols := Symbol_Table (Symbols);
      Context.Charset := To_Unbounded_String (Actual_Charset);
      Context.Tab_Stop := Tab_Stop;
      Context.With_Trivia := With_Trivia;
      Context.Root_Scope := Create_Static_Lexical_Env
        (Parent    => Null_Lexical_Env,
         Node      => null,
         Sym_Table => Context.Symbols);

      --  Create a new ownership share for Event_Handler so that it lives at
      --  least as long as this analysis context.
      Context.Event_Handler := Event_Handler;
      if Context.Event_Handler /= null then
         Context.Event_Handler.Inc_Ref;
      end if;

      --  Create a new ownership share for File_Reader so that it lives at
      --  least as long as this analysis context.
      Context.File_Reader := File_Reader;
      if Context.File_Reader /= null then
         Context.File_Reader.Inc_Ref;
      end if;

      --  Create a new ownership share for Unit_Provider so that it lives at
      --  least as long as this analysis context.
      Context.Unit_Provider := Unit_Provider;
      if Context.Unit_Provider /= null then
         Context.Unit_Provider.Inc_Ref;
      end if;

         if Context.Unit_Provider = null then
            Context.Unit_Provider := Liblktlang.Default_Provider.Create;
         end if;

      Initialize (Context.Parser);

      Context.Discard_Errors_In_Populate_Lexical_Env := True;
      Context.Logic_Resolution_Timeout :=
        Liblktlang_Support.Adalog.Default_Timeout_Ticks_Number;
      Context.In_Populate_Lexical_Env := False;
      Context.Cache_Version := 0;
      Context.Reparse_Cache_Version := 0;

      Context.Templates_Unit := No_Analysis_Unit;

      Context.Available_Rebindings := Env_Rebindings_Vectors.Empty_Vector;

      

   end Initialize_Context;

   -----------------
   -- Create_Unit --
   -----------------

   function Create_Unit
     (Context             : Internal_Context;
      Normalized_Filename : Virtual_File;
      Charset             : String;
      Rule                : Grammar_Rule) return Internal_Unit
   is
      use Units_Maps;

      Unit : Internal_Unit;
   begin
      Unit := Create_Special_Unit
        (Context, Normalized_Filename, Charset, Rule);
      Context.Units.Insert (Normalized_Filename, Unit);
      return Unit;
   end Create_Unit;

   --------------
   -- Get_Unit --
   --------------

   function Get_Unit
     (Context           : Internal_Context;
      Filename, Charset : String;
      Reparse           : Boolean;
      Input             : Liblktlang_Support.Internal.Analysis.Lexer_Input;
      Rule              : Grammar_Rule;
      Is_Internal       : Boolean := False) return Internal_Unit
   is
      use Units_Maps;

      Normalized_Filename : constant GNATCOLL.VFS.Virtual_File :=
         Normalized_Unit_Filename (Context, Filename);

      Cur     : constant Cursor :=
         Context.Units.Find (Normalized_Filename);
      Created : constant Boolean := Cur = No_Element;
      Unit    : Internal_Unit;

      Actual_Charset : Unbounded_String;
      Refined_Input  : Liblktlang_Support.Internal.Analysis.Lexer_Input := Input;

      Parsing_Happened : Boolean := False;

   begin
      --  Determine which encoding to use. Use the Charset parameter (if
      --  provided), otherwise use the context-wide default.

      Actual_Charset := (if Charset'Length /= 0
                         then To_Unbounded_String (Charset)
                         else Context.Charset);

      if Refined_Input.Kind = File then
         Refined_Input.Filename := Normalized_Filename;
      end if;

      if Refined_Input.Kind in File | Bytes_Buffer then
         Refined_Input.Charset := Actual_Charset;

         --  Unless the caller requested a specific charset for this unit,
         --  allow the lexer to automatically discover the source file encoding
         --  before defaulting to the context-specific one. We do this trying
         --  to match a byte order mark.

         Refined_Input.Read_BOM := Charset'Length = 0;
      end if;

      --  Create the Internal_Unit if needed

      Unit :=
        (if Created
         then Create_Unit (Context, Normalized_Filename,
                           To_String (Actual_Charset), Rule)
         else Element (Cur));

      --  If an internal unit is requested, set the corresponding flag.
      --  Otherwise, make sure that the unit we return isn't internal.

      if Is_Internal then
         Unit.Is_Internal := True;
      end if;

      --  (Re)parse it if needed

      if Created or else Reparse then

         --  It is illegal to reparse an internal unit for public API users.
         --  Since public APIs do not allow to pass True to Is_Internal, we can
         --  check here that only the implementation can ask to reparse an
         --  internal unit.

         if Unit.Is_Internal and then not Is_Internal then
            raise Precondition_Failure with "cannot reparse an internal unit";
         end if;

         declare
            Reparsed : Reparsed_Unit;
         begin
            Do_Parsing (Unit, Refined_Input, Reparsed);
            Parsing_Happened := Reparsed.Present;
            Update_After_Reparse (Unit, Reparsed);
         end;

         --  Now that we have removed reparsed the unit, update its current
         --  charset.

         Unit.Charset := Actual_Charset;
      end if;

      if Context.Event_Handler /= null then
         Context.Event_Handler.Unit_Parsed_Callback
           (Context  => Context,
            Unit     => Unit,
            Reparsed => Parsing_Happened and then not Created);
      end if;

      return Unit;
   end Get_Unit;

   --------------
   -- Has_Unit --
   --------------

   function Has_Unit
     (Context : Internal_Context; Unit_Filename : String) return Boolean is
   begin
      return Context.Units.Contains
        (Normalized_Unit_Filename (Context, Unit_Filename));
   end Has_Unit;

   -------------------
   -- Get_From_File --
   -------------------

   function Get_From_File
     (Context  : Internal_Context;
      Filename : String;
      Charset  : String;
      Reparse  : Boolean;
      Rule     : Grammar_Rule) return Internal_Unit
   is
      Input : constant Liblktlang_Support.Internal.Analysis.Lexer_Input :=
        (Kind     => File,
         Charset  => <>,
         Read_BOM => False,
         Filename => <>);
   begin
      if Reparse and then Has_Rewriting_Handle (Context) then
         raise Precondition_Failure with
            "cannot reparse during tree rewriting";
      end if;

      return Get_Unit (Context, Filename, Charset, Reparse, Input, Rule);
   end Get_From_File;

   ---------------------
   -- Get_From_Buffer --
   ---------------------

   function Get_From_Buffer
     (Context  : Internal_Context;
      Filename : String;
      Charset  : String;
      Buffer   : String;
      Rule     : Grammar_Rule) return Internal_Unit
   is
      Input : constant Liblktlang_Support.Internal.Analysis.Lexer_Input :=
        (Kind        => Bytes_Buffer,
         Charset     => <>,
         Read_BOM    => False,
         Bytes       => Buffer'Address,
         Bytes_Count => Buffer'Length);
   begin
      if Has_Rewriting_Handle (Context) then
         raise Precondition_Failure with
            "cannot parse from buffer during tree rewriting";

      elsif Context.File_Reader /= null then
         raise Precondition_Failure with
            "cannot parse from buffer with a file reader";
      end if;

      return Get_Unit (Context, Filename, Charset, True, Input, Rule);
   end Get_From_Buffer;

   --------------------
   -- Get_With_Error --
   --------------------

   function Get_With_Error
     (Context  : Internal_Context;
      Filename : String;
      Error    : Text_Type;
      Charset  : String;
      Rule     : Grammar_Rule) return Internal_Unit
   is
      use Units_Maps;

      Normalized_Filename : constant Virtual_File :=
         Normalized_Unit_Filename (Context, Filename);
      Cur                 : constant Cursor :=
         Context.Units.Find (Normalized_Filename);
   begin
      if Cur = No_Element then
         declare
            Unit : constant Internal_Unit := Create_Unit
              (Context, Normalized_Filename, Charset, Rule);
         begin
            Append (Unit.Diagnostics, No_Source_Location_Range, Error);
            return Unit;
         end;
      else
         return Element (Cur);
      end if;
   end Get_With_Error;


   -----------------------
   -- Get_From_Provider --
   -----------------------

   function Get_From_Provider
     (Context : Internal_Context;
      Name    : Text_Type;
      Kind    : Analysis_Unit_Kind;
      Charset : String;
      Reparse : Boolean) return Internal_Unit is
   begin
      if Reparse and then Has_Rewriting_Handle (Context) then
         raise Precondition_Failure with
            "cannot reparse during tree rewriting";
      end if;

      declare
         Result      : Internal_Unit;
         Dummy_Index : Positive;
      begin
         Context.Unit_Provider.Get_Unit_And_PLE_Root
           (Context, Name, Kind, Charset, Reparse, Result, Dummy_Index);
         return Result;
      exception
         when Property_Error =>
            raise Invalid_Unit_Name_Error with
               "Invalid unit name: " & Image (Name, With_Quotes => True)
               & " (" & Analysis_Unit_Kind'Image (Kind) & ")";
      end;
   end Get_From_Provider;


   -------------------
   -- Unit_Provider --
   -------------------

   function Unit_Provider
     (Context : Internal_Context) return Internal_Unit_Provider_Access
   is (Context.Unit_Provider);

   ------------------
   -- Resolve_Unit --
   ------------------

   procedure Resolve_Unit
     (Context : Internal_Context;
      Name    : Text_Type;
      Kind    : Analysis_Unit_Kind;
      Unit    : out Resolved_Unit)
   is
      --  Look for the cache entry corresponding to Unit; create one if needed

      Dummy    : Resolved_Unit_Array;
      Key      : constant Symbol_Type := Find (Context.Symbols, Name);
      Pos      : Unit_Provider_Cache_Maps.Cursor;
      Inserted : Boolean;
   begin
      Context.Unit_Provider_Cache.Insert (Key, Dummy, Pos, Inserted);
      declare
         Units : Resolved_Unit_Array renames
           Context.Unit_Provider_Cache.Reference (Pos);
         U     : Resolved_Unit renames Units (Kind);
      begin
         --  If the cache entry is not populated for the requested kind, run
         --  the query and save the result for later requests.

         if U.Filename = null then
            declare
               Provider : Internal_Unit_Provider'Class renames
                 Context.Unit_Provider.all;
               Filename : Unbounded_String;
            begin
               Provider.Get_Unit_Location
                 (Name           => Name,
                  Kind           => Kind,
                  Filename       => Filename,
                  PLE_Root_Index => U.PLE_Root_Index);
               Provider.Get_Unit_And_PLE_Root
                 (Context        => Context,
                  Name           => Name,
                  Kind           => Kind,
                  Unit           => U.Unit,
                  PLE_Root_Index => U.PLE_Root_Index);
               U.Filename := new String'(To_String (Filename));
            end;
         end if;

         Unit := U;
      end;
   end Resolve_Unit;

   -----------------------
   -- Get_Unit_Location --
   -----------------------

   procedure Get_Unit_Location
     (Context        : Internal_Context;
      Name           : Text_Type;
      Kind           : Analysis_Unit_Kind;
      Filename       : out String_Access;
      PLE_Root_Index : out Positive)
   is
      U : Resolved_Unit;
   begin
      Resolve_Unit (Context, Name, Kind, U);
      Filename := U.Filename;
      PLE_Root_Index := U.PLE_Root_Index;
   end Get_Unit_Location;

   ---------------------------
   -- Get_Unit_And_PLE_Root --
   ---------------------------

   procedure Get_Unit_And_PLE_Root
     (Context        : Internal_Context;
      Name           : Text_Type;
      Kind           : Analysis_Unit_Kind;
      Unit           : out Internal_Unit;
      PLE_Root_Index : out Positive)
   is
      U : Resolved_Unit;
   begin
      Resolve_Unit (Context, Name, Kind, U);
      Unit := U.Unit;
      PLE_Root_Index := U.PLE_Root_Index;
   end Get_Unit_And_PLE_Root;

   ----------
   -- Hash --
   ----------

   function Hash (Context : Internal_Context) return Hash_Type is
      function H is new Hash_Access (Analysis_Context_Type, Internal_Context);
   begin
      return H (Context);
   end Hash;

   ---------------------
   -- Has_With_Trivia --
   ---------------------

   function Has_With_Trivia (Context : Internal_Context) return Boolean is
   begin
      return Context.With_Trivia;
   end Has_With_Trivia;

   --------------------------------------------
   -- Discard_Errors_In_Populate_Lexical_Env --
   --------------------------------------------

   procedure Discard_Errors_In_Populate_Lexical_Env
     (Context : Internal_Context; Discard : Boolean) is
   begin
      Context.Discard_Errors_In_Populate_Lexical_Env := Discard;
   end Discard_Errors_In_Populate_Lexical_Env;

   ----------------------------------
   -- Set_Logic_Resolution_Timeout --
   ----------------------------------

   procedure Set_Logic_Resolution_Timeout
     (Context : Internal_Context; Timeout : Natural) is
   begin
      Context.Logic_Resolution_Timeout := Timeout;
   end Set_Logic_Resolution_Timeout;

   --------------------------
   -- Has_Rewriting_Handle --
   --------------------------

   function Has_Rewriting_Handle (Context : Internal_Context) return Boolean is
   begin
      return Context.Rewriting_Handle /= System.Null_Address;
   end Has_Rewriting_Handle;

   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (Context : Internal_Context) is
   begin
      if Context /= null then
         Context.Ref_Count := Context.Ref_Count + 1;
      end if;
   end Inc_Ref;

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (Context : in out Internal_Context) is
   begin
      if Context /= null then
         Context.Ref_Count := Context.Ref_Count - 1;
         if Context.Ref_Count = 0 then

            --  If this context was not completely initialized, just release
            --  the allocated object. Do the full destruction otherwise.
            if Context.Initialized then
               Destroy (Context);
            end if;
            Context_Pool.Release (Context);
         end if;
      end if;
   end Dec_Ref;

   -------------
   -- Destroy --
   -------------

   procedure Destroy (Context : Internal_Context) is
   begin
      --  Destroy all named environment data structures
      for Desc of Context.Named_Envs loop
         for V of Desc.Foreign_Nodes loop
            V.Destroy;
         end loop;
         Destroy (Desc);
      end loop;
      Context.Named_Envs.Clear;

      --  If we are asked to free this context, it means that no one else have
      --  references to its analysis units, so it's safe to destroy these.
      for Unit of Context.Units loop
         Destroy (Unit);
      end loop;
      Context.Units := Units_Maps.Empty_Map;
      Context.Filenames :=
        Liblktlang_Support.Internal.Analysis.Empty_Virtual_File_Cache;

      declare
         procedure Destroy is new Ada.Unchecked_Deallocation
           (Env_Rebindings_Type, Env_Rebindings);

         AR : Env_Rebindings_Vectors.Vector renames
            Context.Available_Rebindings;
         R  : Env_Rebindings;
      begin
         for I in AR.First_Index .. AR.Last_Index loop
            R := AR.Get (I);
            Destroy (R);
         end loop;
         AR.Destroy;
      end;

      for Pos in Context.Unit_Provider_Cache.Iterate loop
         declare
            Units : Resolved_Unit_Array renames
              Context.Unit_Provider_Cache.Reference (Pos);
         begin
            for U of Units loop
               Free (U.Filename);
            end loop;
         end;
      end loop;
      Context.Unit_Provider_Cache.Clear;

      Destroy (Context.Templates_Unit);
      AST_Envs.Destroy (Context.Root_Scope);
      Destroy (Context.Symbols);
      Destroy (Context.Parser);
      Dec_Ref (Context.File_Reader);
      Dec_Ref (Context.Unit_Provider);
      Dec_Ref (Context.Event_Handler);
   end Destroy;

   -------------
   -- Context --
   -------------

   function Context (Unit : Internal_Unit) return Internal_Context is
   begin
      return Unit.Context;
   end Context;

   ----------
   -- Hash --
   ----------

   function Hash (Unit : Internal_Unit) return Hash_Type is
      function H is new Hash_Access (Analysis_Unit_Type, Internal_Unit);
   begin
      return H (Unit);
   end Hash;

   -------------
   -- Reparse --
   -------------

   procedure Reparse (Unit : Internal_Unit; Charset : String) is
      Dummy : constant Internal_Unit := Get_From_File
        (Unit.Context, +Unit.Filename.Full_Name, Charset,
         Reparse => True,
         Rule    => Unit.Rule);
   begin
      null;
   end Reparse;

   -------------
   -- Reparse --
   -------------

   procedure Reparse (Unit : Internal_Unit; Charset : String; Buffer : String)
   is
      Dummy : constant Internal_Unit := Get_From_Buffer
        (Unit.Context, +Unit.Filename.Full_Name, Charset, Buffer, Unit.Rule);
   begin
      null;
   end Reparse;

   -----------------------
   -- Reset_Envs_Caches --
   -----------------------

   procedure Reset_Envs_Caches (Unit : Internal_Unit) is
      procedure Internal (Node : Bare_Lkt_Node);
      --  Reset env caches in ``Node`` and then in its children recursively

      Generic_Unit : constant Generic_Unit_Ptr := Convert_Unit (Unit);

      --------------
      -- Internal --
      --------------

      procedure Internal (Node : Bare_Lkt_Node) is
      begin
         if Node = null then
            return;
         end if;
         --  Make sure to only reset caches of envs belonging to this unit
         if Node.Self_Env.Owner = Generic_Unit then
            Reset_Caches (Node.Self_Env);
         end if;
         for I in 1 .. Children_Count (Node) loop
            Internal (Child (Node, I));
         end loop;
      end Internal;
   begin
      Internal (Unit.Ast_Root);
   end Reset_Envs_Caches;

   --------------------------
   -- Populate_Lexical_Env --
   --------------------------

   procedure Populate_Lexical_Env
     (Unit           : Internal_Unit;
      PLE_Root_Index : Positive
         := 1
   ) is
      Context : constant Internal_Context := Unit.Context;

      Saved_In_Populate_Lexical_Env : constant Boolean :=
        Context.In_Populate_Lexical_Env;

      Has_Errors : Boolean := False;
      --  Whether at least one Property_Error occurred during this PLE pass

   begin
      --  TODO??? Handle env invalidation when reparsing a unit and when a
      --  previous call raised a Property_Error.

      --  If we have already run PLE on this root, there is nothing to do.
      --  Otherwise, keep track of the fact that PLE was requested for it,
      --  possibly extending the vector if needed.

      if Unit.Env_Populated_Roots.Last_Index >= PLE_Root_Index
         and then Unit.Env_Populated_Roots.Get (PLE_Root_Index)
      then
         return;
      end if;
      for Dummy in Unit.Env_Populated_Roots.Last_Index + 1 .. PLE_Root_Index
      loop
         Unit.Env_Populated_Roots.Append (False);
      end loop;
      Unit.Env_Populated_Roots.Set (PLE_Root_Index, True);

      --  Create context for the PLE run: all exit points must call the Cleanup
      --  procedure above first to clean this context.

      Context.In_Populate_Lexical_Env := True;
      if Main_Trace.Active then
         Main_Trace.Trace
           ("Populating lexical envs for"
            & " unit: " & Basename (Unit));
         Main_Trace.Increase_Indent;
      end if;

      --  Fetch the node on which to run PLE: it's the unit root node, or one
      --  of its children if PLE roots are enabled and the unit has a list of
      --  PLE roots. Then run PLE itself.

      declare
         PLE_Root : Bare_Lkt_Node := Unit.Ast_Root;
      begin

         if PLE_Root /= null then
            Has_Errors := Populate_Lexical_Env (PLE_Root);
         end if;
      end;

      --  Restore the context for PLE run (undo what was done above)

      Context.In_Populate_Lexical_Env := Saved_In_Populate_Lexical_Env;
      if Main_Trace.Active then
         Main_Trace.Decrease_Indent;
         Main_Trace.Trace
           ("Finished populating lexical envs for"
            & " unit: " & Basename (Unit));
      end if;

      Reset_Envs_Caches (Unit);

      if Has_Errors and then not Context.Discard_Errors_In_Populate_Lexical_Env
      then
         raise Property_Error with
            "errors occurred in Populate_Lexical_Env";
      end if;
   end Populate_Lexical_Env;

   -----------------------------------
   -- Populate_Lexical_Env_For_Unit --
   -----------------------------------

   procedure Populate_Lexical_Env_For_Unit (Node : Bare_Lkt_Node) is
      Root  : Bare_Lkt_Node;
      Index : Natural;
   begin
      Lookup_PLE_Root (Node, Root, Index);
      if Index = 0 then
         Index := 1;
      end if;
      Populate_Lexical_Env (Node.Unit, Index);
   end Populate_Lexical_Env_For_Unit;

   ------------------
   -- Get_Filename --
   ------------------

   function Get_Filename (Unit : Internal_Unit) return String is
     (+Unit.Filename.Full_Name);

   -----------------
   -- Get_Charset --
   -----------------

   function Get_Charset (Unit : Internal_Unit) return String is
   begin
      return To_String (Unit.Charset);
   end Get_Charset;

   ---------------------
   -- Has_Diagnostics --
   ---------------------

   function Has_Diagnostics (Unit : Internal_Unit) return Boolean is
   begin
      return not Unit.Diagnostics.Is_Empty;
   end Has_Diagnostics;

   -----------------
   -- Diagnostics --
   -----------------

   function Diagnostics (Unit : Internal_Unit) return Diagnostics_Array is
      Result : Diagnostics_Array (1 .. Natural (Unit.Diagnostics.Length));
      I      : Natural := 1;
   begin
      for D of Unit.Diagnostics loop
         Result (I) := D;
         I := I + 1;
      end loop;
      return Result;
   end Diagnostics;

   ---------------------------
   -- Format_GNU_Diagnostic --
   ---------------------------

   function Format_GNU_Diagnostic
     (Unit : Internal_Unit; D : Diagnostic) return String
   is
      Filename : constant String := Basename (Unit);
      Sloc     : constant Source_Location := Start_Sloc (D.Sloc_Range);
      Msg      : constant String :=
         Image
           (Ada.Strings.Wide_Wide_Unbounded.To_Wide_Wide_String (D.Message));
   begin
      return (Filename
              & (if Sloc = No_Source_Location then "" else ":" & Image (Sloc))
              & ": " & Msg);
   end Format_GNU_Diagnostic;

   ----------
   -- Root --
   ----------

   function Root (Unit : Internal_Unit) return Bare_Lkt_Node is
     (Unit.Ast_Root);

   -----------------
   -- First_Token --
   -----------------

   function First_Token (Unit : Internal_Unit) return Token_Reference is
     (Wrap_Token_Reference (Unit.Context,
                            Unit.TDH'Access,
                            First_Token_Or_Trivia (Unit.TDH)));

   ----------------
   -- Last_Token --
   ----------------

   function Last_Token (Unit : Internal_Unit) return Token_Reference is
     (Wrap_Token_Reference (Unit.Context,
                            Unit.TDH'Access,
                            Last_Token_Or_Trivia (Unit.TDH)));

   -----------------
   -- Token_Count --
   -----------------

   function Token_Count (Unit : Internal_Unit) return Natural is
     (Unit.TDH.Tokens.Length);

   ------------------
   -- Trivia_Count --
   ------------------

   function Trivia_Count (Unit : Internal_Unit) return Natural is
     (Unit.TDH.Trivias.Length);

   ----------
   -- Text --
   ----------

   function Text (Unit : Internal_Unit) return Text_Type is
   begin
      return Text (First_Token (Unit), Last_Token (Unit));
   end Text;

   ------------------
   -- Lookup_Token --
   ------------------

   function Lookup_Token
     (Unit : Internal_Unit; Sloc : Source_Location) return Token_Reference
   is
      Result : constant Token_Or_Trivia_Index := Lookup_Token (Unit.TDH, Sloc);
   begin
      return Wrap_Token_Reference (Unit.Context, Unit.TDH'Access, Result);
   end Lookup_Token;

   ---------------------
   -- Lookup_PLE_Root --
   ---------------------

   procedure Lookup_PLE_Root
     (Node  : Bare_Lkt_Node;
      Root  : out Bare_Lkt_Node;
      Index : out Natural)
   is
      Unit : constant Internal_Unit := Node.Unit;
   begin
      --  If this unit does not contain a list of PLE roots, just return the
      --  unit root node.

      if Unit.PLE_Roots_Starting_Token.Is_Empty then
         Root := Unit.Ast_Root;
         Index := 0;
         return;
      end if;

      --  Otherwise, look for the last PLE root whose first token (in
      --  Unit.PLE_Roots_Starting_Token) appears before Node's (T). This vector
      --  is sorted by construction, so we can perform a binary search.

      declare
         T      : constant Token_Index := Node.Token_Start_Index;
         Tokens : Token_Index_Vectors.Vector renames
           Unit.PLE_Roots_Starting_Token;

         First : Positive := Tokens.First_Index;
         Last  : Positive := Tokens.Last_Index;
         I     : Positive;
      begin
         while First < Last loop

            --  Because we look for the "floor" (last element that is <= T), we
            --  need to look at the value in Last when there are only two
            --  elements left to look at. If we did not do that, then we would
            --  go into an infinite loop when Tokens[First] < T.

            I := (if First + 1 = Last
                  then Last
                  else (First + Last) / 2);
            declare
               I_T : constant Token_Index := Tokens.Get (I);
            begin
               if I_T <= T then
                  First := I;
               else
                  Last := I - 1;
               end if;
            end;
         end loop;

         Root := Child (Unit.Ast_Root, First);
         Index := First;
      end;
   end Lookup_PLE_Root;

   --------------
   -- Ple_Root --
   --------------

   function Ple_Root
     (Node : Bare_Lkt_Node) return Bare_Lkt_Node
   is
      Root        : Bare_Lkt_Node;
      Dummy_Index : Natural;
   begin
      if Node = null then
         raise Property_Error with "null node dereference";
      end if;
      Lookup_PLE_Root (Node, Root, Dummy_Index);
      return Root;
   end Ple_Root;

   ----------------------
   -- Dump_Lexical_Env --
   ----------------------

   procedure Dump_Lexical_Env (Unit : Internal_Unit) is
      Node     : constant Bare_Lkt_Node := Unit.Ast_Root;
      Root_Env : constant Lexical_Env := Unit.Context.Root_Scope;
      State    : Dump_Lexical_Env_State := (Root_Env => Root_Env, others => <>);

      function Get_Parent (Env : Lexical_Env) return Lexical_Env
      is (Unwrap (Env).Parent);

      --------------------------
      -- Explore_Parent_Chain --
      --------------------------

      procedure Explore_Parent_Chain (Env : Lexical_Env) is
         P : Lexical_Env;
      begin
         if Env /= Null_Lexical_Env then
            P := Get_Parent (Env);
            Dump_One_Lexical_Env
              (Env, Get_Env_Id (Env, State), Get_Env_Id (P, State));
            Explore_Parent_Chain (P);
         end if;
      end Explore_Parent_Chain;

      --------------
      -- Internal --
      --------------

      procedure Internal (Current : Bare_Lkt_Node) is
         Explore_Parent : Boolean := False;
         Env, Parent    : Lexical_Env;
      begin
         if Current = null then
            return;
         end if;

         --  We only dump environments that we haven't dumped before. This way
         --  we'll only dump environments at the site of their creation, and
         --  not in any subsequent link. We use the Env_Ids map to check which
         --  envs we have already seen or not.
         if not State.Env_Ids.Contains (Current.Self_Env) then
            Env := Current.Self_Env;
            Parent := Get_Parent (Env);
            Explore_Parent := not State.Env_Ids.Contains (Parent);

            Dump_One_Lexical_Env
              (Env, Get_Env_Id (Env, State), Get_Env_Id (Parent, State));

            if Explore_Parent then
               Explore_Parent_Chain (Parent);
            end if;
         end if;

         for Child of Internal_Bare_Lkt_Node_Array'(Children (Current))
         loop
            Internal (Child);
         end loop;
      end Internal;
      --  This procedure implements the main recursive logic of dumping the
      --  environments.
   begin
      Internal (Bare_Lkt_Node (Node));
   end Dump_Lexical_Env;

   --------------
   -- Get_Line --
   --------------

   function Get_Line
     (Unit : Internal_Unit; Line_Number : Positive) return Text_Type
   is
   begin
      return Get_Line (Unit.TDH, Line_Number);
   end Get_Line;

   -----------
   -- Print --
   -----------

   procedure Print (Unit : Internal_Unit; Show_Slocs : Boolean) is
   begin
      if Unit.Ast_Root = null then
         Put_Line ("<empty analysis unit>");
      else
         Print (Unit.Ast_Root, Show_Slocs);
      end if;
   end Print;

   ---------------
   -- PP_Trivia --
   ---------------

   procedure PP_Trivia (Unit : Internal_Unit) is

      procedure Process (Trivia : Token_Index) is
         Data : constant Stored_Token_Data :=
            Unit.TDH.Trivias.Get (Natural (Trivia)).T;
      begin
         Put_Line (Image (Text (Unit.TDH, Data)));
      end Process;

      Last_Token : constant Token_Index :=
         Token_Index (Token_Vectors.Last_Index (Unit.TDH.Tokens) - 1);
      --  Index for the last token in Unit excluding the Termination token
      --  (hence the -1).
   begin
      for Tok of Get_Leading_Trivias (Unit.TDH) loop
         Process (Tok);
      end loop;

      PP_Trivia (Unit.Ast_Root);

      for Tok of Get_Trivias (Unit.TDH, Last_Token) loop
         Process (Tok);
      end loop;
   end PP_Trivia;

   -------------
   -- Destroy --
   -------------

   procedure Destroy (Unit : in out Internal_Unit) is
   begin
      if Unit = No_Analysis_Unit then
         return;
      end if;

      --  Clear the env caches while the unit is still fully alive to make sure
      --  that what is accessed in ``Lexical_Env_Cache_Updated`` is still
      --  valid, as it will be called back by lexical envs that are being
      --  destroyed.
      Reset_Envs_Caches (Unit);

      Unit.PLE_Roots_Starting_Token.Destroy;
      Unit.Env_Populated_Roots.Destroy;

      Unit.Exiled_Entries.Destroy;
      Unit.Foreign_Nodes.Destroy;
      Unit.Exiled_Entries_In_NED.Destroy;
      Unit.Exiled_Envs.Destroy;
      Unit.Named_Envs.Destroy;

         Destroy (Unit.Memoization_Map);

      Destroy_Rebindings (Unit.Rebindings'Access);
      Unit.Rebindings.Destroy;

      if Unit.Ast_Root /= null then
         Destroy (Unit.Ast_Root);
      end if;

      Free (Unit.TDH);
      Free (Unit.Ast_Mem_Pool);
      Destroy_Unit_Destroyables (Unit);
      Destroyable_Vectors.Destroy (Unit.Destroyables);
      

      Free (Unit);
   end Destroy;

   -------------------
   -- Is_Token_Node --
   -------------------

   function Is_Token_Node (Node : Bare_Lkt_Node) return Boolean is
   begin
      return Is_Token_Node (Node.Kind);
   end Is_Token_Node;

   ------------------
   -- Is_Synthetic --
   ------------------

   function Is_Synthetic (Node : Bare_Lkt_Node) return Boolean is
   begin
      return Node.Kind in Synthetic_Nodes;
   end Is_Synthetic;

   ------------------------------
   -- Raise_Property_Exception --
   ------------------------------

   procedure Raise_Property_Exception
     (Node    : Bare_Lkt_Node;
      Exc     : Ada.Exceptions.Exception_Id;
      Message : String)
   is
      Sloc_Prefix : constant String :=
        (if Node = null
         then ""
         else Ada.Directories.Simple_Name (Get_Filename (Unit (Node)))
              & ":" & Image (Sloc_Range (Node)) & ": ");
   begin
      Ada.Exceptions.Raise_Exception (Exc, Sloc_Prefix & Message);
   end Raise_Property_Exception;

   ------------------------------
   -- Register_Destroyable_Gen --
   ------------------------------

   procedure Register_Destroyable_Gen
     (Unit : Internal_Unit; Object : T_Access)
   is
      function Convert is new Ada.Unchecked_Conversion
        (System.Address, Destroy_Procedure);
      procedure Destroy_Procedure (Object : in out T_Access) renames Destroy;
   begin
      Register_Destroyable_Helper
        (Unit,
         Object.all'Address,
         Convert (Destroy_Procedure'Address));
   end Register_Destroyable_Gen;

      

   


   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Bare_Lkt_Node_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Bare_Lkt_Node
   is
      function Absolute_Get
        (T : Bare_Lkt_Node_Array_Access; Index : Integer)
         return Bare_Lkt_Node
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Bare_Lkt_Node,
         Sequence_Type => Bare_Lkt_Node_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Bare_Lkt_Node;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Bare_Lkt_Node;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Bare_Lkt_Node_Array_Access) return Bare_Lkt_Node_Array_Access is
      Ret : Bare_Lkt_Node_Array_Access := Create_Bare_Lkt_Node_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Bare_Lkt_Node_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Bare_Lkt_Node_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Bare_Lkt_Node_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Bare_Lkt_Node_Array (Items_Count : Natural) return Bare_Lkt_Node_Array_Access
   is (if Items_Count = 0
       then No_Bare_Lkt_Node_Array_Type
       else new Bare_Lkt_Node_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Bare_Lkt_Node_Array
     (Items : Internal_Bare_Lkt_Node_Array) return Bare_Lkt_Node_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Bare_Lkt_Node_Array_Type;
      end if;

      return new Bare_Lkt_Node_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Bare_Lkt_Node_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Bare_Lkt_Node_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;




      

   


   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Bare_Type_Ref_Node_Builder_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Bare_Type_Ref_Node_Builder
   is
      function Absolute_Get
        (T : Bare_Type_Ref_Node_Builder_Array_Access; Index : Integer)
         return Bare_Type_Ref_Node_Builder
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Bare_Type_Ref_Node_Builder,
         Sequence_Type => Bare_Type_Ref_Node_Builder_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Bare_Type_Ref_Node_Builder;
   begin
      if Relative_Get (T, Index, Result) then
            Inc_Ref (Result);
         return Result;
      elsif Or_Null then
         return null;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Bare_Type_Ref_Node_Builder_Array_Access) return Bare_Type_Ref_Node_Builder_Array_Access is
      Ret : Bare_Type_Ref_Node_Builder_Array_Access := Create_Bare_Type_Ref_Node_Builder_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
         for Item of Ret.Items loop
            Inc_Ref (Item);
         end loop;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Bare_Type_Ref_Node_Builder_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Bare_Type_Ref_Node_Builder_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Bare_Type_Ref_Node_Builder_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
            for Item of T.Items loop
               Dec_Ref (Item);
            end loop;
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Bare_Type_Ref_Node_Builder_Array (Items_Count : Natural) return Bare_Type_Ref_Node_Builder_Array_Access
   is (if Items_Count = 0
       then No_Bare_Type_Ref_Node_Builder_Array_Type
       else new Bare_Type_Ref_Node_Builder_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Bare_Type_Ref_Node_Builder_Array
     (Items : Internal_Bare_Type_Ref_Node_Builder_Array) return Bare_Type_Ref_Node_Builder_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Bare_Type_Ref_Node_Builder_Array_Type;
      end if;

         for El of Items loop
            Inc_Ref (El);
         end loop;
      return new Bare_Type_Ref_Node_Builder_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Bare_Type_Ref_Node_Builder_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Bare_Type_Ref_Node_Builder_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;




      

   

      package Integer_Vectors is new Liblktlang_Support.Vectors (Integer);

   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Integer_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Integer
   is
      function Absolute_Get
        (T : Integer_Array_Access; Index : Integer)
         return Integer
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Integer,
         Sequence_Type => Integer_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Integer;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return 0;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Integer_Array_Access) return Integer_Array_Access is
      Ret : Integer_Array_Access := Create_Integer_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Integer_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Integer_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Integer_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Integer_Array (Items_Count : Natural) return Integer_Array_Access
   is (if Items_Count = 0
       then No_Integer_Array_Type
       else new Integer_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Integer_Array
     (Items : Internal_Integer_Array) return Integer_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Integer_Array_Type;
      end if;

      return new Integer_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Integer_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Integer_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;




      

   

      package Internal_Complete_Item_Vectors is new Liblktlang_Support.Vectors (Internal_Complete_Item);

   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Internal_Complete_Item_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Complete_Item
   is
      function Absolute_Get
        (T : Internal_Complete_Item_Array_Access; Index : Integer)
         return Internal_Complete_Item
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Internal_Complete_Item,
         Sequence_Type => Internal_Complete_Item_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Internal_Complete_Item;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Complete_Item;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Internal_Complete_Item_Array_Access) return Internal_Complete_Item_Array_Access is
      Ret : Internal_Complete_Item_Array_Access := Create_Internal_Complete_Item_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Internal_Complete_Item_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Internal_Complete_Item_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Internal_Complete_Item_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Internal_Complete_Item_Array (Items_Count : Natural) return Internal_Complete_Item_Array_Access
   is (if Items_Count = 0
       then No_Internal_Complete_Item_Array_Type
       else new Internal_Complete_Item_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Internal_Complete_Item_Array
     (Items : Internal_Internal_Complete_Item_Array) return Internal_Complete_Item_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Complete_Item_Array_Type;
      end if;

      return new Internal_Complete_Item_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Internal_Complete_Item_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Internal_Complete_Item_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;




      

   

      package Internal_Entity_Argument_Vectors is new Liblktlang_Support.Vectors (Internal_Entity_Argument);

   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Internal_Entity_Argument_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Argument
   is
      function Absolute_Get
        (T : Internal_Entity_Argument_Array_Access; Index : Integer)
         return Internal_Entity_Argument
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Internal_Entity_Argument,
         Sequence_Type => Internal_Entity_Argument_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Internal_Entity_Argument;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Argument;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Internal_Entity_Argument_Array_Access) return Internal_Entity_Argument_Array_Access is
      Ret : Internal_Entity_Argument_Array_Access := Create_Internal_Entity_Argument_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Internal_Entity_Argument_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Internal_Entity_Argument_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Internal_Entity_Argument_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Internal_Entity_Argument_Array (Items_Count : Natural) return Internal_Entity_Argument_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Argument_Array_Type
       else new Internal_Entity_Argument_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Internal_Entity_Argument_Array
     (Items : Internal_Internal_Entity_Argument_Array) return Internal_Entity_Argument_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Argument_Array_Type;
      end if;

      return new Internal_Entity_Argument_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Internal_Entity_Argument_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Internal_Entity_Argument_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;




      

   

      package Internal_Entity_Vectors is new Liblktlang_Support.Vectors (Internal_Entity);

   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Internal_Entity_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity
   is
      function Absolute_Get
        (T : Internal_Entity_Array_Access; Index : Integer)
         return Internal_Entity
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Internal_Entity,
         Sequence_Type => Internal_Entity_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Internal_Entity;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Internal_Entity_Array_Access) return Internal_Entity_Array_Access is
      Ret : Internal_Entity_Array_Access := Create_Internal_Entity_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Internal_Entity_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Internal_Entity_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Internal_Entity_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Internal_Entity_Array (Items_Count : Natural) return Internal_Entity_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Array_Type
       else new Internal_Entity_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));

   function Create_Internal_Entity_Array
     (Items : AST_Envs.Entity_Array) return Internal_Entity_Array_Access
   is (if Items'Length = 0
       then No_Internal_Entity_Array_Type
       else new Internal_Entity_Array_Record'
         (N         => Items'Length,
          Items     => Implementation.Internal_Internal_Entity_Array (Items),
          Ref_Count => 1));

   function Create_Internal_Entity_Array
     (Items : Internal_Internal_Entity_Array) return Internal_Entity_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Array_Type;
      end if;

      return new Internal_Entity_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Internal_Entity_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Internal_Entity_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;




      

   

      package Internal_Entity_Def_Id_Vectors is new Liblktlang_Support.Vectors (Internal_Entity_Def_Id);

   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Internal_Entity_Def_Id_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Def_Id
   is
      function Absolute_Get
        (T : Internal_Entity_Def_Id_Array_Access; Index : Integer)
         return Internal_Entity_Def_Id
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Internal_Entity_Def_Id,
         Sequence_Type => Internal_Entity_Def_Id_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Internal_Entity_Def_Id;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Def_Id;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Internal_Entity_Def_Id_Array_Access) return Internal_Entity_Def_Id_Array_Access is
      Ret : Internal_Entity_Def_Id_Array_Access := Create_Internal_Entity_Def_Id_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Internal_Entity_Def_Id_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Internal_Entity_Def_Id_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Internal_Entity_Def_Id_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Internal_Entity_Def_Id_Array (Items_Count : Natural) return Internal_Entity_Def_Id_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Def_Id_Array_Type
       else new Internal_Entity_Def_Id_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Internal_Entity_Def_Id_Array
     (Items : Internal_Internal_Entity_Def_Id_Array) return Internal_Entity_Def_Id_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Def_Id_Array_Type;
      end if;

      return new Internal_Entity_Def_Id_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Internal_Entity_Def_Id_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Internal_Entity_Def_Id_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;




      

   

      package Internal_Entity_Enum_Class_Alt_Decl_Vectors is new Liblktlang_Support.Vectors (Internal_Entity_Enum_Class_Alt_Decl);

   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Internal_Entity_Enum_Class_Alt_Decl_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Enum_Class_Alt_Decl
   is
      function Absolute_Get
        (T : Internal_Entity_Enum_Class_Alt_Decl_Array_Access; Index : Integer)
         return Internal_Entity_Enum_Class_Alt_Decl
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Internal_Entity_Enum_Class_Alt_Decl,
         Sequence_Type => Internal_Entity_Enum_Class_Alt_Decl_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Internal_Entity_Enum_Class_Alt_Decl;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Enum_Class_Alt_Decl;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Internal_Entity_Enum_Class_Alt_Decl_Array_Access) return Internal_Entity_Enum_Class_Alt_Decl_Array_Access is
      Ret : Internal_Entity_Enum_Class_Alt_Decl_Array_Access := Create_Internal_Entity_Enum_Class_Alt_Decl_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Internal_Entity_Enum_Class_Alt_Decl_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Internal_Entity_Enum_Class_Alt_Decl_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Internal_Entity_Enum_Class_Alt_Decl_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Internal_Entity_Enum_Class_Alt_Decl_Array (Items_Count : Natural) return Internal_Entity_Enum_Class_Alt_Decl_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Enum_Class_Alt_Decl_Array_Type
       else new Internal_Entity_Enum_Class_Alt_Decl_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Internal_Entity_Enum_Class_Alt_Decl_Array
     (Items : Internal_Internal_Entity_Enum_Class_Alt_Decl_Array) return Internal_Entity_Enum_Class_Alt_Decl_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Enum_Class_Alt_Decl_Array_Type;
      end if;

      return new Internal_Entity_Enum_Class_Alt_Decl_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Internal_Entity_Enum_Class_Alt_Decl_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Internal_Entity_Enum_Class_Alt_Decl_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;




      

   

      package Internal_Entity_Expr_Vectors is new Liblktlang_Support.Vectors (Internal_Entity_Expr);

   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Internal_Entity_Expr_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Expr
   is
      function Absolute_Get
        (T : Internal_Entity_Expr_Array_Access; Index : Integer)
         return Internal_Entity_Expr
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Internal_Entity_Expr,
         Sequence_Type => Internal_Entity_Expr_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Internal_Entity_Expr;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Expr;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Internal_Entity_Expr_Array_Access) return Internal_Entity_Expr_Array_Access is
      Ret : Internal_Entity_Expr_Array_Access := Create_Internal_Entity_Expr_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Internal_Entity_Expr_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Internal_Entity_Expr_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Internal_Entity_Expr_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Internal_Entity_Expr_Array (Items_Count : Natural) return Internal_Entity_Expr_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Expr_Array_Type
       else new Internal_Entity_Expr_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Internal_Entity_Expr_Array
     (Items : Internal_Internal_Entity_Expr_Array) return Internal_Entity_Expr_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Expr_Array_Type;
      end if;

      return new Internal_Entity_Expr_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Internal_Entity_Expr_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Internal_Entity_Expr_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;




      

   

      package Internal_Entity_Field_Decl_Vectors is new Liblktlang_Support.Vectors (Internal_Entity_Field_Decl);

   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Internal_Entity_Field_Decl_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Field_Decl
   is
      function Absolute_Get
        (T : Internal_Entity_Field_Decl_Array_Access; Index : Integer)
         return Internal_Entity_Field_Decl
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Internal_Entity_Field_Decl,
         Sequence_Type => Internal_Entity_Field_Decl_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Internal_Entity_Field_Decl;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Field_Decl;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Internal_Entity_Field_Decl_Array_Access) return Internal_Entity_Field_Decl_Array_Access is
      Ret : Internal_Entity_Field_Decl_Array_Access := Create_Internal_Entity_Field_Decl_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Internal_Entity_Field_Decl_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Internal_Entity_Field_Decl_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Internal_Entity_Field_Decl_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Internal_Entity_Field_Decl_Array (Items_Count : Natural) return Internal_Entity_Field_Decl_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Field_Decl_Array_Type
       else new Internal_Entity_Field_Decl_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Internal_Entity_Field_Decl_Array
     (Items : Internal_Internal_Entity_Field_Decl_Array) return Internal_Entity_Field_Decl_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Field_Decl_Array_Type;
      end if;

      return new Internal_Entity_Field_Decl_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Internal_Entity_Field_Decl_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Internal_Entity_Field_Decl_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;




      

   

      package Internal_Entity_Full_Decl_Vectors is new Liblktlang_Support.Vectors (Internal_Entity_Full_Decl);

   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Internal_Entity_Full_Decl_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Full_Decl
   is
      function Absolute_Get
        (T : Internal_Entity_Full_Decl_Array_Access; Index : Integer)
         return Internal_Entity_Full_Decl
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Internal_Entity_Full_Decl,
         Sequence_Type => Internal_Entity_Full_Decl_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Internal_Entity_Full_Decl;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Full_Decl;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Internal_Entity_Full_Decl_Array_Access) return Internal_Entity_Full_Decl_Array_Access is
      Ret : Internal_Entity_Full_Decl_Array_Access := Create_Internal_Entity_Full_Decl_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Internal_Entity_Full_Decl_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Internal_Entity_Full_Decl_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Internal_Entity_Full_Decl_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Internal_Entity_Full_Decl_Array (Items_Count : Natural) return Internal_Entity_Full_Decl_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Full_Decl_Array_Type
       else new Internal_Entity_Full_Decl_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Internal_Entity_Full_Decl_Array
     (Items : Internal_Internal_Entity_Full_Decl_Array) return Internal_Entity_Full_Decl_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Full_Decl_Array_Type;
      end if;

      return new Internal_Entity_Full_Decl_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Internal_Entity_Full_Decl_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Internal_Entity_Full_Decl_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;




      

   

      package Internal_Entity_Fun_Decl_Vectors is new Liblktlang_Support.Vectors (Internal_Entity_Fun_Decl);

   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Internal_Entity_Fun_Decl_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Fun_Decl
   is
      function Absolute_Get
        (T : Internal_Entity_Fun_Decl_Array_Access; Index : Integer)
         return Internal_Entity_Fun_Decl
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Internal_Entity_Fun_Decl,
         Sequence_Type => Internal_Entity_Fun_Decl_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Internal_Entity_Fun_Decl;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Fun_Decl;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Internal_Entity_Fun_Decl_Array_Access) return Internal_Entity_Fun_Decl_Array_Access is
      Ret : Internal_Entity_Fun_Decl_Array_Access := Create_Internal_Entity_Fun_Decl_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Internal_Entity_Fun_Decl_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Internal_Entity_Fun_Decl_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Internal_Entity_Fun_Decl_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Internal_Entity_Fun_Decl_Array (Items_Count : Natural) return Internal_Entity_Fun_Decl_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Fun_Decl_Array_Type
       else new Internal_Entity_Fun_Decl_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Internal_Entity_Fun_Decl_Array
     (Items : Internal_Internal_Entity_Fun_Decl_Array) return Internal_Entity_Fun_Decl_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Fun_Decl_Array_Type;
      end if;

      return new Internal_Entity_Fun_Decl_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Internal_Entity_Fun_Decl_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Internal_Entity_Fun_Decl_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;




      

   

      package Internal_Entity_Generic_Param_Type_Decl_Vectors is new Liblktlang_Support.Vectors (Internal_Entity_Generic_Param_Type_Decl);

   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Internal_Entity_Generic_Param_Type_Decl_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Generic_Param_Type_Decl
   is
      function Absolute_Get
        (T : Internal_Entity_Generic_Param_Type_Decl_Array_Access; Index : Integer)
         return Internal_Entity_Generic_Param_Type_Decl
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Internal_Entity_Generic_Param_Type_Decl,
         Sequence_Type => Internal_Entity_Generic_Param_Type_Decl_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Internal_Entity_Generic_Param_Type_Decl;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Generic_Param_Type_Decl;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Internal_Entity_Generic_Param_Type_Decl_Array_Access) return Internal_Entity_Generic_Param_Type_Decl_Array_Access is
      Ret : Internal_Entity_Generic_Param_Type_Decl_Array_Access := Create_Internal_Entity_Generic_Param_Type_Decl_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Internal_Entity_Generic_Param_Type_Decl_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Internal_Entity_Generic_Param_Type_Decl_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Internal_Entity_Generic_Param_Type_Decl_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Internal_Entity_Generic_Param_Type_Decl_Array (Items_Count : Natural) return Internal_Entity_Generic_Param_Type_Decl_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Generic_Param_Type_Decl_Array_Type
       else new Internal_Entity_Generic_Param_Type_Decl_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Internal_Entity_Generic_Param_Type_Decl_Array
     (Items : Internal_Internal_Entity_Generic_Param_Type_Decl_Array) return Internal_Entity_Generic_Param_Type_Decl_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Generic_Param_Type_Decl_Array_Type;
      end if;

      return new Internal_Entity_Generic_Param_Type_Decl_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Internal_Entity_Generic_Param_Type_Decl_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Internal_Entity_Generic_Param_Type_Decl_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;




      

   

      package Internal_Entity_Ref_Id_Vectors is new Liblktlang_Support.Vectors (Internal_Entity_Ref_Id);

   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Internal_Entity_Ref_Id_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Ref_Id
   is
      function Absolute_Get
        (T : Internal_Entity_Ref_Id_Array_Access; Index : Integer)
         return Internal_Entity_Ref_Id
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Internal_Entity_Ref_Id,
         Sequence_Type => Internal_Entity_Ref_Id_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Internal_Entity_Ref_Id;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Ref_Id;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Internal_Entity_Ref_Id_Array_Access) return Internal_Entity_Ref_Id_Array_Access is
      Ret : Internal_Entity_Ref_Id_Array_Access := Create_Internal_Entity_Ref_Id_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Internal_Entity_Ref_Id_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Internal_Entity_Ref_Id_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Internal_Entity_Ref_Id_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Internal_Entity_Ref_Id_Array (Items_Count : Natural) return Internal_Entity_Ref_Id_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Ref_Id_Array_Type
       else new Internal_Entity_Ref_Id_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Internal_Entity_Ref_Id_Array
     (Items : Internal_Internal_Entity_Ref_Id_Array) return Internal_Entity_Ref_Id_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Ref_Id_Array_Type;
      end if;

      return new Internal_Entity_Ref_Id_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Internal_Entity_Ref_Id_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Internal_Entity_Ref_Id_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;




      

   

      package Internal_Entity_Type_Decl_Vectors is new Liblktlang_Support.Vectors (Internal_Entity_Type_Decl);

   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Internal_Entity_Type_Decl_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Type_Decl
   is
      function Absolute_Get
        (T : Internal_Entity_Type_Decl_Array_Access; Index : Integer)
         return Internal_Entity_Type_Decl
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Internal_Entity_Type_Decl,
         Sequence_Type => Internal_Entity_Type_Decl_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Internal_Entity_Type_Decl;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Type_Decl;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Internal_Entity_Type_Decl_Array_Access) return Internal_Entity_Type_Decl_Array_Access is
      Ret : Internal_Entity_Type_Decl_Array_Access := Create_Internal_Entity_Type_Decl_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Internal_Entity_Type_Decl_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Internal_Entity_Type_Decl_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Internal_Entity_Type_Decl_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Internal_Entity_Type_Decl_Array (Items_Count : Natural) return Internal_Entity_Type_Decl_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Type_Decl_Array_Type
       else new Internal_Entity_Type_Decl_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Internal_Entity_Type_Decl_Array
     (Items : Internal_Internal_Entity_Type_Decl_Array) return Internal_Entity_Type_Decl_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Type_Decl_Array_Type;
      end if;

      return new Internal_Entity_Type_Decl_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Internal_Entity_Type_Decl_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Internal_Entity_Type_Decl_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;


      ----------
      -- Hash --
      ----------

      function Hash (R : Internal_Entity_Type_Decl_Array_Access) return Hash_Type is
         Result : Hash_Type := Initial_Hash;
      begin
         for I in R.Items'Range loop
            Result := Combine (Result, Hash (R.Items (I)));
         end loop;
         return Result;
      end Hash;


      

   

      package Internal_Env_Assoc_Vectors is new Liblktlang_Support.Vectors (Internal_Env_Assoc);

   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Internal_Env_Assoc_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Env_Assoc
   is
      function Absolute_Get
        (T : Internal_Env_Assoc_Array_Access; Index : Integer)
         return Internal_Env_Assoc
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Internal_Env_Assoc,
         Sequence_Type => Internal_Env_Assoc_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Internal_Env_Assoc;
   begin
      if Relative_Get (T, Index, Result) then
            Inc_Ref (Result);
         return Result;
      elsif Or_Null then
         return No_Env_Assoc;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Internal_Env_Assoc_Array_Access) return Internal_Env_Assoc_Array_Access is
      Ret : Internal_Env_Assoc_Array_Access := Create_Internal_Env_Assoc_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
         for Item of Ret.Items loop
            Inc_Ref (Item);
         end loop;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Internal_Env_Assoc_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Internal_Env_Assoc_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Internal_Env_Assoc_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
            for Item of T.Items loop
               Dec_Ref (Item);
            end loop;
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Internal_Env_Assoc_Array (Items_Count : Natural) return Internal_Env_Assoc_Array_Access
   is (if Items_Count = 0
       then No_Internal_Env_Assoc_Array_Type
       else new Internal_Env_Assoc_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Internal_Env_Assoc_Array
     (Items : Internal_Internal_Env_Assoc_Array) return Internal_Env_Assoc_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Env_Assoc_Array_Type;
      end if;

         for El of Items loop
            Inc_Ref (El);
         end loop;
      return new Internal_Env_Assoc_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Internal_Env_Assoc_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               not Equivalent (L.Items (I), R.Items (I))
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Internal_Env_Assoc_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;




      

   

      package Internal_Inner_Env_Assoc_Vectors is new Liblktlang_Support.Vectors (Internal_Inner_Env_Assoc);

   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Internal_Inner_Env_Assoc_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Inner_Env_Assoc
   is
      function Absolute_Get
        (T : Internal_Inner_Env_Assoc_Array_Access; Index : Integer)
         return Internal_Inner_Env_Assoc
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Internal_Inner_Env_Assoc,
         Sequence_Type => Internal_Inner_Env_Assoc_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Internal_Inner_Env_Assoc;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Inner_Env_Assoc;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Internal_Inner_Env_Assoc_Array_Access) return Internal_Inner_Env_Assoc_Array_Access is
      Ret : Internal_Inner_Env_Assoc_Array_Access := Create_Internal_Inner_Env_Assoc_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Internal_Inner_Env_Assoc_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Internal_Inner_Env_Assoc_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Internal_Inner_Env_Assoc_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Internal_Inner_Env_Assoc_Array (Items_Count : Natural) return Internal_Inner_Env_Assoc_Array_Access
   is (if Items_Count = 0
       then No_Internal_Inner_Env_Assoc_Array_Type
       else new Internal_Inner_Env_Assoc_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Internal_Inner_Env_Assoc_Array
     (Items : Internal_Internal_Inner_Env_Assoc_Array) return Internal_Inner_Env_Assoc_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Inner_Env_Assoc_Array_Type;
      end if;

      return new Internal_Inner_Env_Assoc_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Internal_Inner_Env_Assoc_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Internal_Inner_Env_Assoc_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;




      

   


   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Internal_Logic_Context_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Logic_Context
   is
      function Absolute_Get
        (T : Internal_Logic_Context_Array_Access; Index : Integer)
         return Internal_Logic_Context
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Internal_Logic_Context,
         Sequence_Type => Internal_Logic_Context_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Internal_Logic_Context;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Logic_Context;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Internal_Logic_Context_Array_Access) return Internal_Logic_Context_Array_Access is
      Ret : Internal_Logic_Context_Array_Access := Create_Internal_Logic_Context_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Internal_Logic_Context_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Internal_Logic_Context_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Internal_Logic_Context_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Internal_Logic_Context_Array (Items_Count : Natural) return Internal_Logic_Context_Array_Access
   is (if Items_Count = 0
       then No_Internal_Logic_Context_Array_Type
       else new Internal_Logic_Context_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Internal_Logic_Context_Array
     (Items : Internal_Internal_Logic_Context_Array) return Internal_Logic_Context_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Logic_Context_Array_Type;
      end if;

      return new Internal_Logic_Context_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Internal_Logic_Context_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Internal_Logic_Context_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;




      

   

      package Internal_Param_Match_Vectors is new Liblktlang_Support.Vectors (Internal_Param_Match);

   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Internal_Param_Match_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Param_Match
   is
      function Absolute_Get
        (T : Internal_Param_Match_Array_Access; Index : Integer)
         return Internal_Param_Match
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Internal_Param_Match,
         Sequence_Type => Internal_Param_Match_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Internal_Param_Match;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Param_Match;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Internal_Param_Match_Array_Access) return Internal_Param_Match_Array_Access is
      Ret : Internal_Param_Match_Array_Access := Create_Internal_Param_Match_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Internal_Param_Match_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Internal_Param_Match_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Internal_Param_Match_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Internal_Param_Match_Array (Items_Count : Natural) return Internal_Param_Match_Array_Access
   is (if Items_Count = 0
       then No_Internal_Param_Match_Array_Type
       else new Internal_Param_Match_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Internal_Param_Match_Array
     (Items : Internal_Internal_Param_Match_Array) return Internal_Param_Match_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Param_Match_Array_Type;
      end if;

      return new Internal_Param_Match_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Internal_Param_Match_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Internal_Param_Match_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;




      

   

      package Internal_Ref_Result_Vectors is new Liblktlang_Support.Vectors (Internal_Ref_Result);

   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Internal_Ref_Result_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Ref_Result
   is
      function Absolute_Get
        (T : Internal_Ref_Result_Array_Access; Index : Integer)
         return Internal_Ref_Result
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Internal_Ref_Result,
         Sequence_Type => Internal_Ref_Result_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Internal_Ref_Result;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Ref_Result;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Internal_Ref_Result_Array_Access) return Internal_Ref_Result_Array_Access is
      Ret : Internal_Ref_Result_Array_Access := Create_Internal_Ref_Result_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Internal_Ref_Result_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Internal_Ref_Result_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Internal_Ref_Result_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Internal_Ref_Result_Array (Items_Count : Natural) return Internal_Ref_Result_Array_Access
   is (if Items_Count = 0
       then No_Internal_Ref_Result_Array_Type
       else new Internal_Ref_Result_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Internal_Ref_Result_Array
     (Items : Internal_Internal_Ref_Result_Array) return Internal_Ref_Result_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Ref_Result_Array_Type;
      end if;

      return new Internal_Ref_Result_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Internal_Ref_Result_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Internal_Ref_Result_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;




      

   

      package Internal_Resolved_Param_Vectors is new Liblktlang_Support.Vectors (Internal_Resolved_Param);

   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Internal_Resolved_Param_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Resolved_Param
   is
      function Absolute_Get
        (T : Internal_Resolved_Param_Array_Access; Index : Integer)
         return Internal_Resolved_Param
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Internal_Resolved_Param,
         Sequence_Type => Internal_Resolved_Param_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Internal_Resolved_Param;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Resolved_Param;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Internal_Resolved_Param_Array_Access) return Internal_Resolved_Param_Array_Access is
      Ret : Internal_Resolved_Param_Array_Access := Create_Internal_Resolved_Param_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Internal_Resolved_Param_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Internal_Resolved_Param_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Internal_Resolved_Param_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Internal_Resolved_Param_Array (Items_Count : Natural) return Internal_Resolved_Param_Array_Access
   is (if Items_Count = 0
       then No_Internal_Resolved_Param_Array_Type
       else new Internal_Resolved_Param_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Internal_Resolved_Param_Array
     (Items : Internal_Internal_Resolved_Param_Array) return Internal_Resolved_Param_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Resolved_Param_Array_Type;
      end if;

      return new Internal_Resolved_Param_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Internal_Resolved_Param_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Internal_Resolved_Param_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;


      ----------
      -- Hash --
      ----------

      function Hash (R : Internal_Resolved_Param_Array_Access) return Hash_Type is
         Result : Hash_Type := Initial_Hash;
      begin
         for I in R.Items'Range loop
            Result := Combine (Result, Hash (R.Items (I)));
         end loop;
         return Result;
      end Hash;


      

   


   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Internal_Solver_Diagnostic_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Solver_Diagnostic
   is
      function Absolute_Get
        (T : Internal_Solver_Diagnostic_Array_Access; Index : Integer)
         return Internal_Solver_Diagnostic
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Internal_Solver_Diagnostic,
         Sequence_Type => Internal_Solver_Diagnostic_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Internal_Solver_Diagnostic;
   begin
      if Relative_Get (T, Index, Result) then
            Inc_Ref (Result);
         return Result;
      elsif Or_Null then
         return No_Solver_Diagnostic;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Internal_Solver_Diagnostic_Array_Access) return Internal_Solver_Diagnostic_Array_Access is
      Ret : Internal_Solver_Diagnostic_Array_Access := Create_Internal_Solver_Diagnostic_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
         for Item of Ret.Items loop
            Inc_Ref (Item);
         end loop;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Internal_Solver_Diagnostic_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Internal_Solver_Diagnostic_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Internal_Solver_Diagnostic_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
            for Item of T.Items loop
               Dec_Ref (Item);
            end loop;
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Internal_Solver_Diagnostic_Array (Items_Count : Natural) return Internal_Solver_Diagnostic_Array_Access
   is (if Items_Count = 0
       then No_Internal_Solver_Diagnostic_Array_Type
       else new Internal_Solver_Diagnostic_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Internal_Solver_Diagnostic_Array
     (Items : Internal_Internal_Solver_Diagnostic_Array) return Internal_Solver_Diagnostic_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Solver_Diagnostic_Array_Type;
      end if;

         for El of Items loop
            Inc_Ref (El);
         end loop;
      return new Internal_Solver_Diagnostic_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Internal_Solver_Diagnostic_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               not Equivalent (L.Items (I), R.Items (I))
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Internal_Solver_Diagnostic_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;




      

   


   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Internal_Unit_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Unit
   is
      function Absolute_Get
        (T : Internal_Unit_Array_Access; Index : Integer)
         return Internal_Unit
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Internal_Unit,
         Sequence_Type => Internal_Unit_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Internal_Unit;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return null;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Internal_Unit_Array_Access) return Internal_Unit_Array_Access is
      Ret : Internal_Unit_Array_Access := Create_Internal_Unit_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Internal_Unit_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Internal_Unit_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Internal_Unit_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Internal_Unit_Array (Items_Count : Natural) return Internal_Unit_Array_Access
   is (if Items_Count = 0
       then No_Internal_Unit_Array_Type
       else new Internal_Unit_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Internal_Unit_Array
     (Items : Internal_Internal_Unit_Array) return Internal_Unit_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Unit_Array_Type;
      end if;

      return new Internal_Unit_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Internal_Unit_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Internal_Unit_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;




      

   

      package Lexical_Env_Vectors is new Liblktlang_Support.Vectors (Lexical_Env);

   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Lexical_Env_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Lexical_Env
   is
      function Absolute_Get
        (T : Lexical_Env_Array_Access; Index : Integer)
         return Lexical_Env
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Lexical_Env,
         Sequence_Type => Lexical_Env_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Lexical_Env;
   begin
      if Relative_Get (T, Index, Result) then
            Inc_Ref (Result);
         return Result;
      elsif Or_Null then
         return Empty_Env;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Lexical_Env_Array_Access) return Lexical_Env_Array_Access is
      Ret : Lexical_Env_Array_Access := Create_Lexical_Env_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
         for Item of Ret.Items loop
            Inc_Ref (Item);
         end loop;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Lexical_Env_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Lexical_Env_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Lexical_Env_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
            for Item of T.Items loop
               Dec_Ref (Item);
            end loop;
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Lexical_Env_Array (Items_Count : Natural) return Lexical_Env_Array_Access
   is (if Items_Count = 0
       then No_Lexical_Env_Array_Type
       else new Lexical_Env_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Lexical_Env_Array
     (Items : Internal_Lexical_Env_Array) return Lexical_Env_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Lexical_Env_Array_Type;
      end if;

         for El of Items loop
            Inc_Ref (El);
         end loop;
      return new Lexical_Env_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Lexical_Env_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               not Equivalent (L.Items (I), R.Items (I))
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Lexical_Env_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;




      

   

      package Logic_Equation_Vectors is new Liblktlang_Support.Vectors (Logic_Equation);

   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Logic_Equation_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Logic_Equation
   is
      function Absolute_Get
        (T : Logic_Equation_Array_Access; Index : Integer)
         return Logic_Equation
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Logic_Equation,
         Sequence_Type => Logic_Equation_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Logic_Equation;
   begin
      if Relative_Get (T, Index, Result) then
            Inc_Ref (Result);
         return Result;
      elsif Or_Null then
         return Null_Logic_Equation;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Logic_Equation_Array_Access) return Logic_Equation_Array_Access is
      Ret : Logic_Equation_Array_Access := Create_Logic_Equation_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
         for Item of Ret.Items loop
            Inc_Ref (Item);
         end loop;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Logic_Equation_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Logic_Equation_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Logic_Equation_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
            for Item of T.Items loop
               Dec_Ref (Item);
            end loop;
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Logic_Equation_Array (Items_Count : Natural) return Logic_Equation_Array_Access
   is (if Items_Count = 0
       then No_Logic_Equation_Array_Type
       else new Logic_Equation_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Logic_Equation_Array
     (Items : Internal_Logic_Equation_Array) return Logic_Equation_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Logic_Equation_Array_Type;
      end if;

         for El of Items loop
            Inc_Ref (El);
         end loop;
      return new Logic_Equation_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Logic_Equation_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Logic_Equation_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;




      

   

      package Logic_Var_Vectors is new Liblktlang_Support.Vectors (Logic_Var);

   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Logic_Var_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Logic_Var
   is
      function Absolute_Get
        (T : Logic_Var_Array_Access; Index : Integer)
         return Logic_Var
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Logic_Var,
         Sequence_Type => Logic_Var_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Logic_Var;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return null;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Logic_Var_Array_Access) return Logic_Var_Array_Access is
      Ret : Logic_Var_Array_Access := Create_Logic_Var_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Logic_Var_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Logic_Var_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Logic_Var_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Logic_Var_Array (Items_Count : Natural) return Logic_Var_Array_Access
   is (if Items_Count = 0
       then No_Logic_Var_Array_Type
       else new Logic_Var_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Logic_Var_Array
     (Items : Internal_Logic_Var_Array) return Logic_Var_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Logic_Var_Array_Type;
      end if;

      return new Logic_Var_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Logic_Var_Array_Access) return Boolean is
      use type Logic_Var;
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Logic_Var_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;




      

   

      package String_Type_Vectors is new Liblktlang_Support.Vectors (String_Type);

   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : String_Type_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return String_Type
   is
      function Absolute_Get
        (T : String_Type_Array_Access; Index : Integer)
         return String_Type
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => String_Type,
         Sequence_Type => String_Type_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : String_Type;
   begin
      if Relative_Get (T, Index, Result) then
            Inc_Ref (Result);
         return Result;
      elsif Or_Null then
         return Empty_String;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : String_Type_Array_Access) return String_Type_Array_Access is
      Ret : String_Type_Array_Access := Create_String_Type_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
         for Item of Ret.Items loop
            Inc_Ref (Item);
         end loop;
      return Ret;
   end Concat;

      ------------------
      -- Join_Strings --
      ------------------

      function Join_Strings
        (Separator : String_Type;
         Strings   : String_Type_Array_Access) return String_Type
      is
         Separator_Length : constant Natural := Separator.Length;
         Length           : Natural := 0;
         First            : Boolean;
      begin
         --  First, compute the length of the result: the sum of all string
         --  lengths in Strings.
         First := True;
         for S of Strings.Items loop
            if First then
               First := False;
            else
               Length := Length + Separator_Length;
            end if;
            Length := Length + S.Length;
         end loop;

         --  Create the result string with the correct length. Do not use our
         --  constructor and initialize the content in-place, to avoid extra
         --  copies.
         return Result : constant String_Type :=
            new String_Record (Length)
         do
            Result.Ref_Count := 1;

            --  Now copy the content of all strings into the result
            declare
               Last : Natural := 0;
            begin
               First := True;
               for S of Strings.Items loop
                  if First then
                     First := False;
                  else
                     Result.Content (Last + 1 .. Last + Separator_Length) :=
                        Separator.Content;
                     Last := Last + Separator_Length;
                  end if;
                  Result.Content (Last + 1 .. Last + S.Length) := S.Content;
                  Last := Last + S.Length;
               end loop;
            end;
         end return;
      end Join_Strings;

   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : String_Type_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : String_Type_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out String_Type_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
            for Item of T.Items loop
               Dec_Ref (Item);
            end loop;
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_String_Type_Array (Items_Count : Natural) return String_Type_Array_Access
   is (if Items_Count = 0
       then No_String_Type_Array_Type
       else new String_Type_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_String_Type_Array
     (Items : Internal_String_Type_Array) return String_Type_Array_Access is
   begin
      if Items'Length = 0 then
         return No_String_Type_Array_Type;
      end if;

         for El of Items loop
            Inc_Ref (El);
         end loop;
      return new String_Type_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : String_Type_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               not Equivalent (L.Items (I), R.Items (I))
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : String_Type_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;




      

   

      package Symbol_Type_Vectors is new Liblktlang_Support.Vectors (Symbol_Type);

   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Symbol_Type_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Symbol_Type
   is
      function Absolute_Get
        (T : Symbol_Type_Array_Access; Index : Integer)
         return Symbol_Type
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Symbol_Type,
         Sequence_Type => Symbol_Type_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Symbol_Type;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Symbol;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Symbol_Type_Array_Access) return Symbol_Type_Array_Access is
      Ret : Symbol_Type_Array_Access := Create_Symbol_Type_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Symbol_Type_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Symbol_Type_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Symbol_Type_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Symbol_Type_Array (Items_Count : Natural) return Symbol_Type_Array_Access
   is (if Items_Count = 0
       then No_Symbol_Type_Array_Type
       else new Symbol_Type_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Symbol_Type_Array
     (Items : Internal_Symbol_Type_Array) return Symbol_Type_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Symbol_Type_Array_Type;
      end if;

      return new Symbol_Type_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Symbol_Type_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Symbol_Type_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;





         

   

   ----------
   -- Next --
   ----------

   function Next
     (T       : Bare_Lkt_Node_Iterator_Access;
      Element : out Bare_Lkt_Node) return Boolean is
   begin
      if T = null then
         raise Property_Error with "null access dereference";
      end if;
      Check_Safety_Net (T.Safety_Net);

      if T.Index > T.Elements.Items'Last then
         return False;
      else
         Element := T.Elements.Items (T.Index);
         T.Index := T.Index + 1;
         return True;
      end if;
   end Next;

   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Bare_Lkt_Node_Iterator_Access) is
   begin
      if T /= null and then T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Bare_Lkt_Node_Iterator_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
         Dec_Ref (T.Elements);
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Bare_Lkt_Node_Iterator_Access) return String is
      begin
         return "<Iterator of LktNode, index="
                & A.Index'Image & ">";
      end Trace_Image;


         

   

   ----------
   -- Next --
   ----------

   function Next
     (T       : Internal_Entity_Iterator_Access;
      Element : out Internal_Entity) return Boolean is
   begin
      if T = null then
         raise Property_Error with "null access dereference";
      end if;
      Check_Safety_Net (T.Safety_Net);

      if T.Index > T.Elements.Items'Last then
         return False;
      else
         Element := T.Elements.Items (T.Index);
         T.Index := T.Index + 1;
         return True;
      end if;
   end Next;

   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Internal_Entity_Iterator_Access) is
   begin
      if T /= null and then T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Internal_Entity_Iterator_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
         Dec_Ref (T.Elements);
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Internal_Entity_Iterator_Access) return String is
      begin
         return "<Iterator of Entity[LktNode], index="
                & A.Index'Image & ">";
      end Trace_Image;


         

   

   ----------
   -- Next --
   ----------

   function Next
     (T       : Internal_Inner_Env_Assoc_Iterator_Access;
      Element : out Internal_Inner_Env_Assoc) return Boolean is
   begin
      if T = null then
         raise Property_Error with "null access dereference";
      end if;
      Check_Safety_Net (T.Safety_Net);

      if T.Index > T.Elements.Items'Last then
         return False;
      else
         Element := T.Elements.Items (T.Index);
         T.Index := T.Index + 1;
         return True;
      end if;
   end Next;

   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Internal_Inner_Env_Assoc_Iterator_Access) is
   begin
      if T /= null and then T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Internal_Inner_Env_Assoc_Iterator_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
         Dec_Ref (T.Elements);
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Internal_Inner_Env_Assoc_Iterator_Access) return String is
      begin
         return "<Iterator of InnerEnvAssoc, index="
                & A.Index'Image & ">";
      end Trace_Image;




   ---------
   -- "<" --
   ---------

   function "<" (Left, Right : Internal_Unit) return Boolean is
   begin
      return Left.Filename < Right.Filename;
   end "<";

      



function Hash (Key : Mmz_Key_Item) return Hash_Type;
function Equivalent (L, R : Mmz_Key_Item) return Boolean;
procedure Destroy (Key : in out Mmz_Key_Array_Access);

----------------
-- Equivalent --
----------------

function Equivalent (L, R : Mmz_Key_Item) return Boolean is
begin
   if L.Kind /= R.Kind then
      return False;
   end if;

   case L.Kind is
         when Mmz_Bare_Argument_List =>
            
               return L.As_Bare_Argument_List = R.As_Bare_Argument_List;
         when Mmz_Bare_Basic_Class_Decl =>
            
               return L.As_Bare_Basic_Class_Decl = R.As_Bare_Basic_Class_Decl;
         when Mmz_Bare_Decl =>
            
               return L.As_Bare_Decl = R.As_Bare_Decl;
         when Mmz_Bare_Dot_Expr =>
            
               return L.As_Bare_Dot_Expr = R.As_Bare_Dot_Expr;
         when Mmz_Bare_Expr =>
            
               return L.As_Bare_Expr = R.As_Bare_Expr;
         when Mmz_Bare_Lexer_Decl =>
            
               return L.As_Bare_Lexer_Decl = R.As_Bare_Lexer_Decl;
         when Mmz_Bare_Lkt_Node =>
            
               return L.As_Bare_Lkt_Node = R.As_Bare_Lkt_Node;
         when Mmz_Bare_Ref_Id =>
            
               return L.As_Bare_Ref_Id = R.As_Bare_Ref_Id;
         when Mmz_Bare_Struct_Decl =>
            
               return L.As_Bare_Struct_Decl = R.As_Bare_Struct_Decl;
         when Mmz_Bare_Type_Decl =>
            
               return L.As_Bare_Type_Decl = R.As_Bare_Type_Decl;
         when Mmz_Bare_Type_Ref =>
            
               return L.As_Bare_Type_Ref = R.As_Bare_Type_Ref;
         when Mmz_Boolean =>
            
               return L.As_Boolean = R.As_Boolean;
         when Mmz_Env_Rebindings =>
            
               return L.As_Env_Rebindings = R.As_Env_Rebindings;
         when Mmz_Internal_Entity =>
            
               return L.As_Internal_Entity = R.As_Internal_Entity;
         when Mmz_Internal_Entity_Argument_List =>
            
               return L.As_Internal_Entity_Argument_List = R.As_Internal_Entity_Argument_List;
         when Mmz_Internal_Entity_Decl =>
            
               return L.As_Internal_Entity_Decl = R.As_Internal_Entity_Decl;
         when Mmz_Internal_Entity_Info =>
            
               return L.As_Internal_Entity_Info = R.As_Internal_Entity_Info;
         when Mmz_Internal_Entity_Type_Decl =>
            
               return L.As_Internal_Entity_Type_Decl = R.As_Internal_Entity_Type_Decl;
         when Mmz_Internal_Entity_Type_Decl_Array_Access =>
            
               return Equivalent (L.As_Internal_Entity_Type_Decl_Array_Access, R.As_Internal_Entity_Type_Decl_Array_Access);
         when Mmz_Internal_Metadata =>
            
               return L.As_Internal_Metadata = R.As_Internal_Metadata;
         when Mmz_Internal_Resolved_Param_Array_Access =>
            
               return Equivalent (L.As_Internal_Resolved_Param_Array_Access, R.As_Internal_Resolved_Param_Array_Access);
         when Mmz_Symbol_Type =>
            
               return L.As_Symbol_Type = R.As_Symbol_Type;
   end case;
end Equivalent;

----------
-- Hash --
----------

function Hash (Key : Mmz_Key_Item) return Hash_Type is
begin
   case Key.Kind is
         when Mmz_Bare_Argument_List =>
            return Hash (Key.As_Bare_Argument_List);
         when Mmz_Bare_Basic_Class_Decl =>
            return Hash (Key.As_Bare_Basic_Class_Decl);
         when Mmz_Bare_Decl =>
            return Hash (Key.As_Bare_Decl);
         when Mmz_Bare_Dot_Expr =>
            return Hash (Key.As_Bare_Dot_Expr);
         when Mmz_Bare_Expr =>
            return Hash (Key.As_Bare_Expr);
         when Mmz_Bare_Lexer_Decl =>
            return Hash (Key.As_Bare_Lexer_Decl);
         when Mmz_Bare_Lkt_Node =>
            return Hash (Key.As_Bare_Lkt_Node);
         when Mmz_Bare_Ref_Id =>
            return Hash (Key.As_Bare_Ref_Id);
         when Mmz_Bare_Struct_Decl =>
            return Hash (Key.As_Bare_Struct_Decl);
         when Mmz_Bare_Type_Decl =>
            return Hash (Key.As_Bare_Type_Decl);
         when Mmz_Bare_Type_Ref =>
            return Hash (Key.As_Bare_Type_Ref);
         when Mmz_Boolean =>
            return Hash (Key.As_Boolean);
         when Mmz_Env_Rebindings =>
            return Hash (Key.As_Env_Rebindings);
         when Mmz_Internal_Entity =>
            return Hash (Key.As_Internal_Entity);
         when Mmz_Internal_Entity_Argument_List =>
            return Hash (Key.As_Internal_Entity_Argument_List);
         when Mmz_Internal_Entity_Decl =>
            return Hash (Key.As_Internal_Entity_Decl);
         when Mmz_Internal_Entity_Info =>
            return Hash (Key.As_Internal_Entity_Info);
         when Mmz_Internal_Entity_Type_Decl =>
            return Hash (Key.As_Internal_Entity_Type_Decl);
         when Mmz_Internal_Entity_Type_Decl_Array_Access =>
            return Hash (Key.As_Internal_Entity_Type_Decl_Array_Access);
         when Mmz_Internal_Metadata =>
            return Hash (Key.As_Internal_Metadata);
         when Mmz_Internal_Resolved_Param_Array_Access =>
            return Hash (Key.As_Internal_Resolved_Param_Array_Access);
         when Mmz_Symbol_Type =>
            return Hash (Key.As_Symbol_Type);
   end case;
end Hash;

----------
-- Hash --
----------

function Hash (Key : Mmz_Key) return Hash_Type is
   Result : Hash_Type := Mmz_Property'Pos (Key.Property);
begin
   for K of Key.Items.all loop
      Result := Combine (Result, Hash (K));
   end loop;
   return Result;
end Hash;

----------------
-- Equivalent --
----------------

function Equivalent (L, R : Mmz_Key) return Boolean is
   L_Items : Mmz_Key_Array renames L.Items.all;
   R_Items : Mmz_Key_Array renames R.Items.all;
begin
   if L.Property /= R.Property or else L_Items'Length /= R_Items'Length then
      return False;
   end if;

   for I in L_Items'Range loop
      if not Equivalent (L_Items (I), R_Items (I)) then
         return False;
      end if;
   end loop;

   return True;
end Equivalent;

-------------
-- Destroy --
-------------

procedure Destroy (Map : in out Memoization_Maps.Map) is
   use Memoization_Maps;

   --  We need keys and values to be valid when clearing the memoization map,
   --  but on the other hand we need to free keys and values as well. To
   --  achieve both goals, we first copy key and values into arrays, then we
   --  clear the map, and then we free keys/values in the arrays. Allocate both
   --  arrays on the heap to avoid stack overflow, as they can be quite big.

   Length : constant Natural := Natural (Map.Length);

   type Key_Array is array (1 .. Length) of Mmz_Key_Array_Access;
   type Key_Array_Access is access Key_Array;
   procedure Free is new Ada.Unchecked_Deallocation
     (Key_Array, Key_Array_Access);

   type Value_Array is array (1 .. Length) of Mmz_Value;
   type Value_Array_Access is access Value_Array;
   procedure Free is new Ada.Unchecked_Deallocation
     (Value_Array, Value_Array_Access);

   Keys   : Key_Array_Access := new Key_Array;
   Values : Value_Array_Access := new Value_Array;
   I      : Positive := 1;
begin
   for Cur in Map.Iterate loop
      Keys (I) := Key (Cur).Items;
      Values (I) := Element (Cur);
      I := I + 1;
   end loop;

   Map.Clear;

   for K_Array of Keys.all loop
      Destroy (K_Array);
   end loop;

   
   for V of Values.all loop
      case V.Kind is
         when Mmz_Error =>
            Free_Memoized_Error (V.Exc_Id, V.Exc_Msg);

            when Mmz_Internal_Env_Assoc_Array_Access =>
               Dec_Ref (V.As_Internal_Env_Assoc_Array_Access);
            when Mmz_Internal_Param_Match_Array_Access =>
               Dec_Ref (V.As_Internal_Param_Match_Array_Access);
            when Mmz_Internal_Resolved_Param_Array_Access =>
               Dec_Ref (V.As_Internal_Resolved_Param_Array_Access);
            when Mmz_Internal_Solver_Result =>
               Dec_Ref (V.As_Internal_Solver_Result);
            when Mmz_Lexical_Env =>
               Dec_Ref (V.As_Lexical_Env);

         when others => null;
      end case;
   end loop;

   Free (Keys);
   Free (Values);
end Destroy;

-------------
-- Destroy --
-------------

procedure Destroy (Key : in out Mmz_Key_Array_Access) is
   procedure Free is new Ada.Unchecked_Deallocation
     (Mmz_Key_Array, Mmz_Key_Array_Access);
begin
   

      for K of Key.all loop
         case K.Kind is
               when Mmz_Internal_Entity_Type_Decl_Array_Access =>
                  Dec_Ref (K.As_Internal_Entity_Type_Decl_Array_Access);
               when Mmz_Internal_Resolved_Param_Array_Access =>
                  Dec_Ref (K.As_Internal_Resolved_Param_Array_Access);

            when others => null;
         end case;
      end loop;
   Free (Key);
end Destroy;

-------------------------
-- Find_Memoized_Value --
-------------------------

function Find_Memoized_Value
  (Unit       : Internal_Unit;
   Handle     : out Memoization_Handle;
   Value      : out Mmz_Value;
   Create_Key : access function return Mmz_Key) return Boolean
is
   Inserted : Boolean;
begin
   --  Make sure that we don't lookup stale caches
   Reset_Caches (Unit);

   --  Initialize handle: create the key and create a cursor pointing to an
   --  existing entry.
   Handle.Key := Create_Key.all;
   Handle.Cache_Version := Unit.Cache_Version;
   Value := (Kind => Mmz_Evaluating);
   Unit.Memoization_Map.Insert (Handle.Key, Value, Handle.Cur, Inserted);

   --  No existing entry yet? The above just created one. Otherwise, destroy
   --  our key and reuse the existing entry's.
   if not Inserted then
      Destroy (Handle.Key.Items);
      Handle.Key := Memoization_Maps.Key (Handle.Cur);
      Value := Memoization_Maps.Element (Handle.Cur);
   end if;

   return not Inserted;
end Find_Memoized_Value;

------------------------
-- Add_Memoized_Value --
------------------------

procedure Add_Memoized_Value
  (Unit   : Internal_Unit;
   Handle : in out Memoization_Handle;
   Value  : Mmz_Value;
   Stored : out Boolean) is
begin
   --  If Handle was created using a memoization map that has been since then
   --  reset, do nothing: the result can be partly stale due to the event that
   --  triggered the memoization tables reset.

   Stored := Unit.Cache_Version <= Handle.Cache_Version;
   if Stored then
      Unit.Memoization_Map.Replace_Element (Handle.Cur, Value);
   end if;
end Add_Memoized_Value;

------------------------
-- Add_Memoized_Error --
------------------------

procedure Add_Memoized_Error
  (Unit   : Internal_Unit;
   Handle : in out Memoization_Handle;
   Exc    : Ada.Exceptions.Exception_Occurrence;
   Stored : out Boolean)
is
   Value : Mmz_Value (Kind => Mmz_Error);
begin
   Store_Memoized_Error (Exc, Value.Exc_Id, Value.Exc_Msg);
   Add_Memoized_Value (Unit, Handle, Value, Stored);
   if not Stored then
      Free_Memoized_Error (Value.Exc_Id, Value.Exc_Msg);
   end if;
end Add_Memoized_Error;

----------------------------
-- Reraise_Memoized_Error --
----------------------------

procedure Reraise_Memoized_Error (Value : Mmz_Value) is
begin
   Reraise_Memoized_Error (Value.Exc_Id, Value.Exc_Msg);
end Reraise_Memoized_Error;

--------------------------
-- Store_Memoized_Error --
--------------------------

procedure Store_Memoized_Error
  (Exc     : Ada.Exceptions.Exception_Occurrence;
   Exc_Id  : out Ada.Exceptions.Exception_Id;
   Exc_Msg : out String_Access) is
begin
   Exc_Id := Ada.Exceptions.Exception_Identity (Exc);
   Exc_Msg := new String'(Ada.Exceptions.Exception_Message (Exc));
end Store_Memoized_Error;

-------------------------
-- Free_Memoized_Error --
-------------------------

procedure Free_Memoized_Error
  (Exc_Id  : in out Ada.Exceptions.Exception_Id;
   Exc_Msg : in out String_Access)
is
   pragma Unreferenced (Exc_Id);
begin
   Free (Exc_Msg);
end Free_Memoized_Error;

----------------------------
-- Reraise_Memoized_Error --
----------------------------

procedure Reraise_Memoized_Error
  (Exc_Id  : Ada.Exceptions.Exception_Id;
   Exc_Msg : String_Access) is
begin
   Ada.Exceptions.Raise_Exception (Exc_Id, Exc_Msg.all & " (memoized)");
end Reraise_Memoized_Error;



   package Solver_Diagnostic_Vectors is new Liblktlang_Support.Vectors
     (Internal_Solver_Diagnostic);

   ----------------------------
   -- Allocate_Logic_Context --
   ----------------------------

   function Allocate_Logic_Context
     (Ctx : Internal_Logic_Context) return Internal_Logic_Context_Access
   is ((if Ctx.Ref_Node = No_Entity
           and then Ctx.Decl_Node = No_Entity
        then null
        else new Internal_Logic_Context'(Ctx)));

   -------------------------
   -- Trace_Logic_Context --
   -------------------------

   function Trace_Logic_Context
     (Ctx : Internal_Logic_Context_Access) return String
   is (Trace_Image (Ctx.all));

   -----------------
   -- Deep_Equals --
   -----------------

   function Deep_Equals
     (X, Y : Internal_Logic_Context_Access) return Boolean
   is (X.all = Y.all);

   ------------------------
   -- Free_Logic_Context --
   ------------------------

   procedure Free_Logic_Context
     (Ctx : in out Internal_Logic_Context_Access)
   is
      procedure Free is new Ada.Unchecked_Deallocation
        (Internal_Logic_Context, Internal_Logic_Context_Access);
   begin
      Free (Ctx);
   end Free_Logic_Context;

   -------------------
   -- Solve_Wrapper --
   -------------------

   function Solve_Wrapper
     (R            : Solver.Relation;
      Context_Node : Bare_Lkt_Node) return Boolean is
   begin
      if Context_Node /= null and then Liblktlang_Support.Adalog.Debug.Debug then
         Assign_Names_To_Logic_Vars (Context_Node);
      end if;

      begin
         return Solver.Solve_First
           (R, Timeout => Context_Node.Unit.Context.Logic_Resolution_Timeout);
      exception
         when Liblktlang_Support.Adalog.Early_Binding_Error =>
            Raise_Property_Exception
              (Context_Node,
               Property_Error'Identity,
               "invalid equation for logic resolution");
         when Liblktlang_Support.Adalog.Timeout_Error =>
            Raise_Property_Exception
              (Context_Node,
               Property_Error'Identity,
               "logic resolution timed out");
      end;
   end Solve_Wrapper;

   ----------------------------
   -- Solve_With_Diagnostics --
   ----------------------------

   function Solve_With_Diagnostics
     (R            : Solver.Relation;
      Context_Node : Bare_Lkt_Node) return Internal_Solver_Result
   is
      Ret : Internal_Solver_Result :=
        (True, No_Internal_Solver_Diagnostic_Array_Type);

      Acc : Solver_Diagnostic_Vectors.Vector;
      --  Vectors that will accumulate diagnostic emitted during resolution

      procedure Emit_Diagnostic (Diag : Internal_Solver_Diagnostic) is
      begin
         Acc.Append (Diag);
      end Emit_Diagnostic;
   begin
      Ret.Success := Solve_Wrapper (R, Context_Node);

      if not Ret.Success then
         Ret.Success := Solver.Solve_First
           (R,
            Solve_Options => (Report_Errors => True),
            Diag_Emitter  => Emit_Diagnostic'Unrestricted_Access,
            Timeout       =>
              Context_Node.Unit.Context.Logic_Resolution_Timeout);
         Ret.Diagnostics := Create_Internal_Solver_Diagnostic_Array
           (Acc.Length);
         for I in 1 .. Acc.Length loop
            Ret.Diagnostics.Items (I) := Acc.Get (I);
         end loop;
         Acc.Destroy;
      end if;
      return Ret;
   end Solve_With_Diagnostics;

   -------------
   -- Destroy --
   -------------

   procedure Destroy (Env : in out Lexical_Env_Access) is
      Mutable_Env : Lexical_Env :=
        (Wrap (Env), 0, Env.Kind, No_Generic_Unit, 0);
   begin
      Destroy (Mutable_Env);
      Env := null;
   end Destroy;

   ----------------
   -- Initialize --
   ----------------

   procedure Initialize
     (Self              : Bare_Lkt_Node;
      Kind              : Lkt_Node_Kind_Type;
      Unit              : Internal_Unit;
      Token_Start_Index : Token_Index;
      Token_End_Index   : Token_Index;
      Parent            : Bare_Lkt_Node := null;
      Self_Env          : Lexical_Env := AST_Envs.Empty_Env) is
   begin
      pragma Unreferenced (Kind);
      Self.Parent := Parent;
      Self.Unit := Unit;

      Self.Token_Start_Index := Token_Start_Index;
      Self.Token_End_Index := Token_End_Index;

      Self.Self_Env := Self_Env;
      Self.Last_Attempted_Child := -1;

      

   end Initialize;

   --------------------------------------
   -- Allocate_Synthetic_List_Children --
   --------------------------------------

   function Allocate_Synthetic_List_Children
     (Self  : Bare_Lkt_Node_Base_List;
      Count : Natural) return Alloc_AST_List_Array.Element_Array_Access
   is
      use Alloc_AST_List_Array;
      use System.Memory;

      Size : constant size_t :=
        Bare_Lkt_Node'Max_Size_In_Storage_Elements * size_t (Count);
   begin
      return Result : constant Element_Array_Access :=
        (if Count = 0
         then Empty_Array_Access
         else To_Pointer (System.Memory.Alloc (Size)))
      do
         Self.Count := Count;
         Self.Nodes := Result;
      end return;
   end Allocate_Synthetic_List_Children;

   ----------------------------------
   -- Free_Synthetic_List_Children --
   ----------------------------------

   procedure Free_Synthetic_List_Children
     (Self : Bare_Lkt_Node_Base_List)
   is
      use Alloc_AST_List_Array;
      use System.Memory;
   begin
      if Self.Nodes /= Empty_Array_Access then
         Free (To_Address (Self.Nodes));
      end if;
   end Free_Synthetic_List_Children;

   --------------------
   -- Use_Direct_Env --
   --------------------

   procedure Use_Direct_Env (State : in out PLE_Node_State; Env : Lexical_Env)
   is
   begin
      State.Current_Env := Env;
      State.Current_NED := null;
   end Use_Direct_Env;

   -------------------
   -- Use_Named_Env --
   -------------------

   procedure Use_Named_Env
     (State   : in out PLE_Node_State;
      Context : Internal_Context;
      Name    : Symbol_Type) is
   begin
      State.Current_NED := Get_Named_Env_Descriptor (Context, Name);
      State.Current_Env := State.Current_NED.Env_With_Precedence;
   end Use_Named_Env;

   ---------------------
   -- Set_Initial_Env --
   ---------------------

   procedure Set_Initial_Env
     (Self         : Bare_Lkt_Node;
      State        : in out PLE_Node_State;
      Env          : Internal_Designated_Env;
      DSL_Location : String) is
   begin
      case Env.Kind is
         when None =>
            Use_Direct_Env (State, Empty_Env);

         when Current_Env =>
            null;

         when Named_Env =>
            Use_Named_Env (State, Self.Unit.Context, Env.Env_Name);

         when Direct_Env =>

            --  Sanitize this environment value: make sure it's a non-foreign
            --  and primary environment.

            if Env.Direct_Env.Kind /= Static_Primary then
               Raise_Property_Exception
                 (Self,
                  Property_Error'Identity,
                  "Cannot set an env that is not static-primary as the"
                  & " initial env");

            elsif Is_Foreign_Strict (Env.Direct_Env, Self) then
               Raise_Property_Exception
                 (Self,
                  Property_Error'Identity,
                  "unsound foreign environment in SetInitialEnv ("
                  & DSL_Location & ")");
            end if;
            Use_Direct_Env (State, Env.Direct_Env);
      end case;
   end Set_Initial_Env;

   ----------------
   -- Add_To_Env --
   ----------------

   procedure Add_To_Env
     (Self         : Bare_Lkt_Node;
      State        : PLE_Node_State;
      Key          : Symbol_Type;
      Value        : Bare_Lkt_Node;
      Md           : Internal_Metadata;
      Resolver     : Entity_Resolver;
      Dest_Env     : Internal_Designated_Env;
      DSL_Location : String)
   is
      Context    : constant Internal_Context := Self.Unit.Context;
      Root_Scope : Lexical_Env renames Context.Root_Scope;
      --  Shortcuts

      Actual_Dest_Env : Lexical_Env;
      Dest_NED        : Named_Env_Descriptor_Access;
      --  Description for the destination environment
   begin
      --  Skip the env addition if explicitly requested

      if Key = No_Symbol
         or else Value = null
         or else (case Dest_Env.Kind is
                  when None        => True,
                  when Current_Env => False,
                  when Named_Env   => Dest_Env.Env_Name = No_Symbol,
                  when Direct_Env  => Dest_Env.Direct_Env = Empty_Env)
      then
         return;
      end if;

      if Value.Unit /= Self.Unit then
         Raise_Property_Exception
           (Self,
            Property_Error'Identity,
            "Cannot add_to_env an AST node that comes from another analysis"
            & " unit");
      end if;

      

      --  Then determine the destination environment

      case Dest_Env.Kind is
         when None =>
            raise Program_Error with "unreachable code";

         when Current_Env =>
            --  Just use the current environment
            Dest_NED := State.Current_NED;
            Actual_Dest_Env := State.Current_Env;

         when Named_Env =>
            --  There is an environment name: just lookup the corresponding
            --  NED/env.
            Dest_NED := Get_Named_Env_Descriptor (Context, Dest_Env.Env_Name);
            Actual_Dest_Env := Dest_NED.Env_With_Precedence;

         when Direct_Env =>
            --  There is an explicit destination environment
            Dest_NED := null;
            Actual_Dest_Env := Dest_Env.Direct_Env;
      end case;

      --  Sanitize it

      if Actual_Dest_Env.Kind /= Static_Primary then
         Raise_Property_Exception
           (Self,
            Property_Error'Identity,
            "Cannot add elements to a lexical env that is not static-primary");

      elsif
         --  Since lexical envs need to sort the foreign nodes they contain,
         --  and that the total order on nodes is not defined for synthetic
         --  nodes, it is not possible to add a synthetic node to a foreign
         --  lexical environment.
         --
         --  This reasoning applies to environments that belong to foreign
         --  units, but also to the root environment.
         Is_Foreign (Actual_Dest_Env, Self) and then Is_Synthetic (Value)
      then
         Raise_Property_Exception
           (Self,
            Property_Error'Identity,
            "Cannot add a synthetic node to a lexical env from another"
            & " analysis unit");

      elsif
         --  Reject direct references to foreign destination environments.
         --
         --  This is an attempt at identifying uses of the unsound relocation
         --  mechanism (as opposed to named environments), so this applies to
         --  all foreign environments (root scope included).
         DSL_Location'Length > 0
         and then Dest_Env.Kind = Direct_Env
         and then Is_Foreign_Strict (Actual_Dest_Env, Self)
      then
         Raise_Property_Exception
           (Self,
            Property_Error'Identity,
            "unsound foreign environment in AddToEnv (" & DSL_Location & ")");
      end if;

      --  Now that everything is sanitized, we can proceed with the actual
      --  key/value pair addition. Note that this does nothing if
      --  Actual_Dest_Env ended up empty.
      Add (Actual_Dest_Env, Thin (Key), Value, Md, Resolver);

      --  If we're adding the element to an environment by env name, we must
      --  register this association in two places: in the target named env
      --  entry, and in Value's unit.
      if Dest_NED /= null then
         declare
            use NED_Assoc_Maps;

            FN    : Map renames Dest_NED.Foreign_Nodes;
            Dummy : Boolean;
            Cur   : Cursor;
         begin
            FN.Insert (Key      => Key,
                       New_Item => Internal_Map_Node_Vectors.Empty_Vector,
                       Position => Cur,
                       Inserted => Dummy);
            declare
               V : Internal_Map_Node_Vectors.Vector renames
                  FN.Reference (Cur);
            begin
               V.Append ((Value, null, Md, Resolver));
               --  Note that the rebindings field is unused by the relocation
               --  mechanism (since we don't even allow adding env entries
               --  with custom rebindings), hence we simply leave it to null.
            end;
         end;
         Value.Unit.Exiled_Entries_In_NED.Append ((Dest_NED, Key, Value));

      --  Otherwise, if we're adding the element to an environment that belongs
      --  to a different unit, or to the root scope, then...
      elsif Is_Foreign_Not_Empty (Actual_Dest_Env, Self) then
         --  Add the Key/Value association to the list of entries contained in
         --  other units, so we can remove them when reparsing Value's unit.
         Value.Unit.Exiled_Entries.Append ((Actual_Dest_Env, Key, Value));

         if Actual_Dest_Env /= Root_Scope then
            --  Add Val to the list of foreign nodes that Actual_Dest_Env's
            --  unit contains, so that when that unit is reparsed, we can call
            --  Add_To_Env again on those nodes.
            Convert_Unit (Actual_Dest_Env.Owner).Foreign_Nodes.Append
              ((Value, Self.Unit));
         end if;
      end if;
   end Add_To_Env;

   -------------
   -- Ref_Env --
   -------------

   procedure Ref_Env
     (Self                : Bare_Lkt_Node;
      Dest_Env            : Lexical_Env;
      Ref_Env_Nodes       : in out Bare_Lkt_Node_Array_Access;
      Resolver            : Lexical_Env_Resolver;
      Kind                : Ref_Kind;
      Cats                : Ref_Categories;
      Shed_Rebindings     : Boolean) is
   begin
      for N of Ref_Env_Nodes.Items loop
         if N /= null then
            if N.Unit /= Self.Unit then
               Raise_Property_Exception
                 (Self,
                  Property_Error'Identity,
                  "attempt to add a referenced environment to a foreign unit");
            end if;
            Reference (Dest_Env, N, Resolver, Kind, Cats, Shed_Rebindings);
         end if;
      end loop;
      Dec_Ref (Ref_Env_Nodes);
   end Ref_Env;

   -------------
   -- Add_Env --
   -------------

   procedure Add_Env
     (Self              : Bare_Lkt_Node;
      State             : in out PLE_Node_State;
      No_Parent         : Boolean;
      Transitive_Parent : Boolean;
      Names             : in out Symbol_Type_Array_Access)
   is
      Parent_From_Name : constant Boolean := State.Current_NED /= null;
      --  Does the parent environment comes from a named environment lookup?

      --  Determine the parent of this new environment:
      --
      --  (1) no parent if requested;
      --  (2) the current environment as the static parent if it comes from a
      --      named env lookup or if it is not foreign (or is the empty/root
      --      environment).
      Parent : constant Lexical_Env :=
        (if No_Parent
         then Null_Lexical_Env
         else State.Current_Env);
   begin
      --  Create the environment itself
      Self.Self_Env := Create_Static_Lexical_Env
        (Parent            => Parent,
         Node              => Self,
         Transitive_Parent => Transitive_Parent,
         Sym_Table         => Self.Unit.Context.Symbols);

      --  If the parent of this new environment comes from a named environment
      --  lookup, register this new environment so that its parent is updated
      --  when the precence for this named environment changes.
      if Parent_From_Name then
         declare
            NED : constant Named_Env_Descriptor_Access := State.Current_NED;
         begin
            Self.Unit.Exiled_Envs.Append ((NED, Self.Self_Env));
            NED.Foreign_Envs.Insert (Self, Self.Self_Env);
         end;
      end if;

      --  From now on, the current environment is Self.Self_Env, with a direct
      --  access to it. It does not go through the env naming scheme, since
      --  only this node and its children (i.e. non-foreign nodes) will access
      --  it as a "current" environment during PLE.
      Use_Direct_Env (State, Self.Self_Env);

      --  Register the environment we just created on all the requested names
      if Names /= null then
         declare
            Context   : constant Internal_Context := Self.Unit.Context;
            Env       : constant Lexical_Env := Self.Self_Env;
            NENU      : NED_Maps.Map renames
               State.Unit_State.Named_Envs_Needing_Update;
         begin
            for N of Names.Items loop
               Register_Named_Env (Context, N, Env, NENU);
            end loop;
            Dec_Ref (Names);
         end;
      end if;
   end Add_Env;

   ---------------------
   -- Pre_Env_Actions --
   ---------------------

   procedure Pre_Env_Actions
     (Self            : Bare_Lkt_Node;
      State           : in out PLE_Node_State;
      Add_To_Env_Only : Boolean := False) is
   begin

      
   

   case Self.Kind is
            when Lkt_Decl_Block =>
            
            Decl_Block_Pre_Env_Actions (Self, State, Add_To_Env_Only);
      
            when Lkt_Langkit_Root =>
            
            Langkit_Root_Pre_Env_Actions (Self, State, Add_To_Env_Only);
      
            when Lkt_Import =>
            
            Import_Pre_Env_Actions (Self, State, Add_To_Env_Only);
      
            when Lkt_Lambda_Expr =>
            
            Lambda_Expr_Pre_Env_Actions (Self, State, Add_To_Env_Only);
      
            when Lkt_Block_Expr =>
            
            Block_Expr_Pre_Env_Actions (Self, State, Add_To_Env_Only);
      
            when Lkt_Enum_Type_Decl =>
            
            Enum_Type_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
      
            when Lkt_Enum_Class_Decl =>
            
            Enum_Class_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
      
            when Lkt_Enum_Class_Alt_Decl =>
            
            null;
      
            when Lkt_Lexer_Decl =>
            
            Lexer_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
      
            when Lkt_Grammar_Decl =>
            
            Grammar_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
      
            when Lkt_Generic_Decl =>
            
            Generic_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
      
            when Lkt_Env_Spec_Decl =>
            
            Env_Spec_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
      
            when Lkt_Fun_Decl =>
            
            Fun_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
      
            when Lkt_Field_Decl =>
            
            Field_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
      
            when Lkt_Enum_Lit_Decl =>
            
            null;
      
            when Lkt_Grammar_Rule_Decl .. Lkt_Binding_Val_Decl | Lkt_Fun_Param_Decl .. Lkt_Val_Decl | Lkt_Error_Decl | Lkt_Lexer_Family_Decl .. Lkt_Any_Type_Decl | Lkt_Function_Type .. Lkt_Class_Decl | Lkt_Struct_Decl .. Lkt_Trait_Decl =>
            
            Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
      
            when Lkt_Match_Branch .. Lkt_Pattern_Match_Branch =>
            
            Base_Match_Branch_Pre_Env_Actions (Self, State, Add_To_Env_Only);
      
      when others =>  null; 
   end case;


   end Pre_Env_Actions;

   ----------------------
   -- Post_Env_Actions --
   ----------------------

   procedure Post_Env_Actions
     (Self : Bare_Lkt_Node; State : in out PLE_Node_State) is
   begin
      
   

   case Self.Kind is
            when Lkt_Decl_Block =>
            
            null;
      
            when Lkt_Langkit_Root =>
            
            Langkit_Root_Post_Env_Actions (Self, State);
      
            when Lkt_Import =>
            
            null;
      
            when Lkt_Lambda_Expr =>
            
            null;
      
            when Lkt_Block_Expr =>
            
            null;
      
            when Lkt_Enum_Type_Decl =>
            
            Enum_Type_Decl_Post_Env_Actions (Self, State);
      
            when Lkt_Enum_Class_Decl =>
            
            Enum_Class_Decl_Post_Env_Actions (Self, State);
      
            when Lkt_Enum_Class_Alt_Decl =>
            
            null;
      
            when Lkt_Lexer_Decl =>
            
            null;
      
            when Lkt_Grammar_Decl =>
            
            null;
      
            when Lkt_Generic_Decl =>
            
            null;
      
            when Lkt_Env_Spec_Decl =>
            
            null;
      
            when Lkt_Fun_Decl =>
            
            null;
      
            when Lkt_Field_Decl =>
            
            null;
      
            when Lkt_Enum_Lit_Decl =>
            
            null;
      
            when Lkt_Grammar_Rule_Decl .. Lkt_Binding_Val_Decl | Lkt_Fun_Param_Decl .. Lkt_Val_Decl | Lkt_Error_Decl | Lkt_Lexer_Family_Decl .. Lkt_Any_Type_Decl | Lkt_Function_Type .. Lkt_Class_Decl | Lkt_Struct_Decl .. Lkt_Trait_Decl =>
            
            null;
      
            when Lkt_Match_Branch .. Lkt_Pattern_Match_Branch =>
            
            null;
      
      when others =>  null; 
   end case;


   end Post_Env_Actions;

   ----------------
   -- Get_Symbol --
   ----------------

   function Get_Symbol
     (Node : Bare_Lkt_Node) return Symbol_Type is
   begin
      if Node = null then
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "cannot get the symbol of a null node");
      end if;
      return Get_Symbol (Token (Node, Node.Token_Start_Index));
   end Get_Symbol;

   ----------
   -- Text --
   ----------

   function Text
     (Node : Bare_Lkt_Node) return Text_Type
   is
   begin
      if Node = null then
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "cannot get the text of a null node");
      end if;

      declare
         Start_T : constant Token_Reference :=
            Token (Node, Node.Token_Start_Index);
         End_T   : constant Token_Reference :=
            Token (Node, Node.Token_End_Index);
      begin
         --  No text is associated to synthetic and ghost nodes

         if Is_Synthetic (Node) then
            return "";
         end if;

         if Is_Ghost (Node) then
            return "";
         end if;

         return Text (Start_T, End_T);
      end;
   end Text;

   ----------
   -- Unit --
   ----------

   function Unit (Node : Bare_Lkt_Node) return Internal_Unit is
   begin
      return Node.Unit;
   end Unit;

   function Lookup_Internal
     (Node : Bare_Lkt_Node;
      Sloc : Source_Location) return Bare_Lkt_Node;
   procedure Lookup_Relative
     (Node       : Bare_Lkt_Node;
      Sloc       : Source_Location;
      Position   : out Relative_Position;
      Node_Found : out Bare_Lkt_Node);
   --  Implementation helpers for the looking up process

   -----------------
   -- Set_Parents --
   -----------------

   procedure Set_Parents
     (Node, Parent : Bare_Lkt_Node)
   is
   begin
      if Node = null then
         return;
      end if;

      Node.Parent := Bare_Lkt_Node (Parent);

      for I in 1 .. Children_Count (Node) loop
         Set_Parents (Child (Node, I), Node);
      end loop;
   end Set_Parents;

   -------------
   -- Destroy --
   -------------

   procedure Destroy (Node : Bare_Lkt_Node) is
   begin
      if Node = null then
         return;
      end if;

      Free_User_Fields (Node);
      for I in 1 .. Children_Count (Node) loop
         Destroy (Child (Node, I));
      end loop;
   end Destroy;

   -----------
   -- Child --
   -----------

   function Child (Node  : Bare_Lkt_Node;
                   Index : Positive) return Bare_Lkt_Node
   is
      Result          : Bare_Lkt_Node;
      Index_In_Bounds : Boolean;
   begin
      Get_Child (Node, Index, Index_In_Bounds, Result);
      return Result;
   end Child;

   --------------
   -- Traverse --
   --------------

   function Traverse
     (Node  : Bare_Lkt_Node;
      Visit : access function (Node : Bare_Lkt_Node)
              return Visit_Status)
     return Visit_Status
   is
      Status : Visit_Status := Into;

   begin
      if Node /= null then
         Status := Visit (Node);

         --  Skip processing the child nodes if the returned status is Over
         --  or Stop. In the former case the previous call to Visit has taken
         --  care of processing the needed childs, and in the latter case we
         --  must immediately stop processing the tree.

         if Status = Into then
            for I in 1 .. Children_Count (Node) loop
               declare
                  Cur_Child : constant Bare_Lkt_Node :=
                     Child (Node, I);

               begin
                  if Cur_Child /= null then
                     Status := Traverse (Cur_Child, Visit);
                     exit when Status /= Into;
                  end if;
               end;
            end loop;
         end if;
      end if;

      if Status = Stop then
         return Stop;

      --  At this stage the Over status has no sense and we just continue
      --  processing the tree.

      else
         return Into;
      end if;
   end Traverse;

   --------------
   -- Traverse --
   --------------

   procedure Traverse
     (Node  : Bare_Lkt_Node;
      Visit : access function (Node : Bare_Lkt_Node)
                               return Visit_Status)
   is
      Result_Status : Visit_Status;
      pragma Unreferenced (Result_Status);
   begin
      Result_Status := Traverse (Node, Visit);
   end Traverse;

   ------------------------
   -- Traverse_With_Data --
   ------------------------

   function Traverse_With_Data
     (Node  : Bare_Lkt_Node;
      Visit : access function (Node : Bare_Lkt_Node;
                               Data : in out Data_Type)
                               return Visit_Status;
      Data  : in out Data_Type)
      return Visit_Status
   is
      function Helper (Node : Bare_Lkt_Node) return Visit_Status;

      ------------
      -- Helper --
      ------------

      function Helper (Node : Bare_Lkt_Node) return Visit_Status is
      begin
         return Visit (Node, Data);
      end Helper;

      Saved_Data : Data_Type;
      Result     : Visit_Status;

   begin
      if Reset_After_Traversal then
         Saved_Data := Data;
      end if;
      Result := Traverse (Node, Helper'Access);
      if Reset_After_Traversal then
         Data := Saved_Data;
      end if;
      return Result;
   end Traverse_With_Data;

   ----------------
   -- Sloc_Range --
   ----------------

   function Sloc_Range
     (Node : Bare_Lkt_Node) return Source_Location_Range
   is
      type Token_Anchor is (T_Start, T_End);
      type Token_Pos is record
         Pos    : Token_Index;
         Anchor : Token_Anchor;
      end record;

      TDH                    : Token_Data_Handler renames Node.Unit.TDH;
      Token_Start, Token_End : Token_Pos;

      function Get (Index : Token_Index) return Stored_Token_Data is
        (Get_Token (TDH, Index));

      function Sloc (T : Token_Pos) return Source_Location is
        (if T.Anchor = T_Start
         then Sloc_Start (TDH, Get (T.Pos))
         else Sloc_End (TDH, Get (T.Pos)));

   begin
      if Is_Synthetic (Node) then
         return (if Node.Parent = null
                 then No_Source_Location_Range
                 else Sloc_Range (Node.Parent));
      end if;

      if Is_Ghost (Node) then
         Token_Start := (if Node.Token_Start_Index = 1
                         then (1, T_Start)
                         else (Node.Token_Start_Index - 1, T_End));
         Token_End := Token_Start;
      else
         Token_Start := (Node.Token_Start_Index, T_Start);
         Token_End := (Node.Token_End_Index, T_End);
      end if;

      if Snaps_At_Start (Node)
         and then not Is_Ghost (Node)
         and then Token_Start.Pos /= 1
      then
         Token_Start := (Token_Start.Pos - 1, T_End);
      end if;

      if Snaps_At_End (Node) and then Token_End.Pos /= Last_Token (TDH) then
         Token_End := (Token_End.Pos + 1, T_Start);
      end if;

      return Make_Range (Sloc (Token_Start), Sloc (Token_End));
   end Sloc_Range;

   ------------
   -- Lookup --
   ------------

   function Lookup
     (Node : Bare_Lkt_Node;
      Sloc : Source_Location) return Bare_Lkt_Node
   is
      Position : Relative_Position;
      Result   : Bare_Lkt_Node;
   begin
      if Sloc = No_Source_Location then
         return null;
      end if;

      Lookup_Relative
        (Bare_Lkt_Node (Node), Sloc, Position, Result);
      return Result;
   end Lookup;

   ---------------------
   -- Lookup_Internal --
   ---------------------

   function Lookup_Internal
     (Node : Bare_Lkt_Node;
      Sloc : Source_Location) return Bare_Lkt_Node
   is
      --  For this implementation helper (i.e. internal primitive), we can
      --  assume that all lookups fall into this node's sloc range.
      pragma Assert (Compare (Sloc_Range (Node), Sloc) = Inside);

      Children : constant Internal_Bare_Lkt_Node_Array :=
         Implementation.Children (Node);
      Pos      : Relative_Position;
      Result   : Bare_Lkt_Node;
   begin
      --  Look for a child node that contains Sloc (i.e. return the most
      --  precise result).

      for Child of Children loop
         --  Note that we assume here that child nodes are ordered so that the
         --  first one has a sloc range that is before the sloc range of the
         --  second child node, etc.

         if Child /= null then
            Lookup_Relative (Child, Sloc, Pos, Result);
            case Pos is
               when Before =>
                   --  If this is the first node, Sloc is before it, so we can
                   --  stop here.  Otherwise, Sloc is between the previous
                   --  child node and the next one...  so we can stop here,
                   --  too.
                   return Node;

               when Inside =>
                   return Result;

               when After =>
                   --  Sloc is after the current child node, so see with the
                   --  next one.
                   null;
            end case;
         end if;
      end loop;

      --  If we reach this point, we found no children that covers Sloc, but
      --  Node still covers it (see the assertion).
      return Node;
   end Lookup_Internal;

   -------------
   -- Compare --
   -------------

   function Compare
     (Node : Bare_Lkt_Node;
      Sloc : Source_Location) return Relative_Position is
   begin
      return Compare (Sloc_Range (Node), Sloc);
   end Compare;

   ---------------------
   -- Lookup_Relative --
   ---------------------

   procedure Lookup_Relative
     (Node       : Bare_Lkt_Node;
      Sloc       : Source_Location;
      Position   : out Relative_Position;
      Node_Found : out Bare_Lkt_Node)
   is
      Result : constant Relative_Position :=
        Compare (Node, Sloc);
   begin
      Position := Result;
      Node_Found := (if Result = Inside
                     then Lookup_Internal (Node, Sloc)
                     else null);
   end Lookup_Relative;

   -------------
   -- Compare --
   -------------

   function Compare
     (Self, Left, Right : Bare_Lkt_Node;
      Relation          : Comparison_Relation) return Boolean
   is
      LS, RS : Source_Location;
   begin
      if Left = null or else Right = null or else Left.Unit /= Right.Unit then
         Raise_Property_Exception
           (Self,
            Property_Error'Identity,
            "invalid node comparison");
      end if;

      LS := Start_Sloc (Sloc_Range (Left));
      RS := Start_Sloc (Sloc_Range (Right));
      return (case Relation is
              when Liblktlang_Support.Types.Less_Than        => LS < RS,
              when Liblktlang_Support.Types.Less_Or_Equal    => LS <= RS,
              when Liblktlang_Support.Types.Greater_Than     => LS > RS,
              when Liblktlang_Support.Types.Greater_Or_Equal => LS >= RS);
   end Compare;

   --------------
   -- Children --
   --------------

   function Children
     (Node : Bare_Lkt_Node) return Internal_Bare_Lkt_Node_Array
   is
      First : constant Integer := Bare_Lkt_Node_Vectors.Index_Type'First;
      Last  : constant Integer := First + Children_Count (Node) - 1;
   begin
      return A : Internal_Bare_Lkt_Node_Array (First .. Last)
      do
         for I in First .. Last loop
            A (I) := Child (Node, I);
         end loop;
      end return;
   end Children;

   function Children
     (Node : Bare_Lkt_Node) return Bare_Lkt_Node_Array_Access
   is
      C : Internal_Bare_Lkt_Node_Array := Children (Node);
   begin
      return Ret : Bare_Lkt_Node_Array_Access :=
         Create_Bare_Lkt_Node_Array (C'Length)
      do
         Ret.Items := C;
      end return;
   end Children;

   ---------
   -- Get --
   ---------

   function Get
     (Self    : Bare_Lkt_Node;
      Node    : Bare_Lkt_Node_Base_List;
      Index   : Integer;
      Or_Null : Boolean := False) return Bare_Lkt_Node
   is
      function Length (Node : Bare_Lkt_Node_Base_List) return Natural
      is (Node.Count);
      --  Wrapper around the Length primitive to get the compiler happy for the
      --  the package instantiation below.

      function Absolute_Get
        (L     : Bare_Lkt_Node_Base_List;
         Index : Integer) return Bare_Lkt_Node
      is (L.Nodes.all (Index + 1));
      --  L.Nodes is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Bare_Lkt_Node,
         Sequence_Type => Bare_Lkt_Node_Base_List,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Bare_Lkt_Node;
   begin
      if Node = null and then Or_Null then
         return null;
      elsif Relative_Get (Node, Index, Result) then
         return Result;
      elsif Or_Null then
         return null;
      else
         Raise_Property_Exception
           (Self, Property_Error'Identity, "out-of-bounds AST list access");
      end if;
   end Get;

   ---------------
   -- PP_Trivia --
   ---------------

   procedure PP_Trivia
     (Node        : Bare_Lkt_Node;
      Line_Prefix : String := "")
   is
      Children_Prefix : constant String := Line_Prefix & "|  ";
   begin
      if Node = null then
         Put_Line (Line_Prefix & "None");
         return;
      end if;
      Put_Line (Line_Prefix & Kind_Name (Node));
      for C of Children_And_Trivia (Node) loop
         case C.Kind is
            when Trivia =>
               Put_Line (Children_Prefix & Image (Text (C.Trivia)));
            when Child =>
               PP_Trivia (C.Node, Children_Prefix);
         end case;
      end loop;
   end PP_Trivia;

   --------------------------
   -- Populate_Lexical_Env --
   --------------------------

   function Populate_Lexical_Env (Node : Bare_Lkt_Node) return Boolean is

      Context    : constant Internal_Context := Node.Unit.Context;
      Unit_State : aliased PLE_Unit_State := (Named_Envs_Needing_Update => <>);
      Root_State : constant PLE_Node_State :=
        (Unit_State  => Unit_State'Unchecked_Access,
         Current_Env => Context.Root_Scope,
         Current_NED => null);

      function Populate_Internal
        (Node         : Bare_Lkt_Node;
         Parent_State : PLE_Node_State) return Boolean;
      --  Do the lexical env population on Node and recurse on its children

      procedure Register_Foreign_Env
        (Node : Bare_Lkt_Node; State : PLE_Node_State);
      --  Given a node and its PLE state, register Node.Self_Env as being
      --  initialized through the named environment mechanism, if that's indeed
      --  the case. Do nothing otherwise.

      -----------------------
      -- Populate_Internal --
      -----------------------

      function Populate_Internal
        (Node         : Bare_Lkt_Node;
         Parent_State : PLE_Node_State) return Boolean
      is
         Result : Boolean := False;
         State  : PLE_Node_State := Parent_State;
      begin
         if Node = null then
            return Result;
         end if;

         --  By default (i.e. unless env actions add a new env), the
         --  environment we store in Node is the current one.
         Node.Self_Env := State.Current_Env;

         --  Run pre/post actions, and run PLE on children in between. Make
         --  sure we register the potential foreign Node.Self_Env environment
         --  at the end, even when an exception interrupts PLE to keep the
         --  state consistent.
         begin
            Pre_Env_Actions (Node, State);
            if State.Current_Env /= Null_Lexical_Env then
               Node.Self_Env := State.Current_Env;
               Register_Foreign_Env (Node, State);
            end if;

            --  Call recursively on children
            for I in First_Child_Index (Node) .. Last_Child_Index (Node) loop
               Result := Populate_Internal
                 (Child (Node, I), State) or else Result;
            end loop;

            Post_Env_Actions (Node, State);
         exception
            when Exc : Property_Error =>
               if PLE_Errors_Trace.Is_Active then
                   GNATCOLL.Traces.Trace
                     (PLE_Errors_Trace,
                      "Exception raised during PLE "
                      & Ada.Exceptions.Exception_Name (Exc) & " : "
                      & Ada.Exceptions.Exception_Message (Exc));
                   GNATCOLL.Traces.Trace
                     (PLE_Errors_Trace,
                      GNAT.Traceback.Symbolic.Symbolic_Traceback (Exc));
               end if;
               Register_Foreign_Env (Node, State);
               return True;
         end;

         return Result;
      end Populate_Internal;

      --------------------------
      -- Register_Foreign_Env --
      --------------------------

      procedure Register_Foreign_Env
        (Node : Bare_Lkt_Node; State : PLE_Node_State) is
      begin
         if State.Current_NED /= null then
            State.Current_NED.Nodes_With_Foreign_Env.Insert (Node);
            Node.Unit.Nodes_With_Foreign_Env.Insert (Node, State.Current_NED);
         end if;
      end Register_Foreign_Env;

   begin
      --  This is intended to be called on the root node only (when there is no
      --  PLE root) or on a PLE root (child of the root node with a specific
      --  kind).
      if
         Node.Parent /= null
      then
         raise Program_Error;
      end if;

      return Result : constant Boolean :=
         Populate_Internal (Node, Root_State)
      do
         Update_Named_Envs (Context, Unit_State.Named_Envs_Needing_Update);
      end return;
   end Populate_Lexical_Env;

   ------------------------------
   -- AST_Envs_Node_Text_Image --
   ------------------------------

   function AST_Envs_Node_Text_Image
     (Node  : Bare_Lkt_Node;
      Short : Boolean := True) return Text_Type is
   begin
      if Short then
         if Node = null then
            return "null";
         end if;
         return To_Text (Basename (Node.Unit))
           & ":" & To_Text (Image (Start_Sloc (Sloc_Range (Node))));
      else
         return Short_Text_Image (Node);
      end if;
   end AST_Envs_Node_Text_Image;

   -------------------
   -- Is_Rebindable --
   -------------------

   function Is_Rebindable (Node : Bare_Lkt_Node) return Boolean is
   begin
      
         return Node.Kind in Lkt_Generic_Decl;
   end Is_Rebindable;

   -----------------------
   -- Acquire_Rebinding --
   -----------------------

   function Acquire_Rebinding
     (Node             : Bare_Lkt_Node;
      Parent           : Env_Rebindings;
      Old_Env, New_Env : Lexical_Env) return Env_Rebindings
   is
      Result    : Env_Rebindings;
      Available : Env_Rebindings_Vectors.Vector renames
         Node.Unit.Context.Available_Rebindings;
   begin
      --  Use an existing and available Env_Rebindings_Type record for Node's
      --  Context, otherwise allocate a new rebinding.
      Result := (if Available.Is_Empty
                 then new Env_Rebindings_Type'(Version => 0, others => <>)
                 else Available.Pop);

      Result.Parent := Parent;
      Result.Old_Env := Old_Env;
      Result.New_Env := New_Env;
      Result.Children := Env_Rebindings_Vectors.Empty_Vector;
      return Result;
   end Acquire_Rebinding;

   -----------------------
   -- Release_Rebinding --
   -----------------------

   procedure Release_Rebinding (Self : in out Env_Rebindings) is
      Available : Env_Rebindings_Vectors.Vector renames
         Unwrap (Self.Old_Env).Node.Unit.Context.Available_Rebindings;
   begin
      --  Bumping the version number, to invalidate existing references to
      --  Self.
      Self.Version := Self.Version + 1;

      Self.Children.Destroy;
      Available.Append (Self);
      Self := null;
   end Release_Rebinding;

   ------------------------
   -- Register_Rebinding --
   ------------------------

   procedure Register_Rebinding
     (Node : Bare_Lkt_Node; Rebinding : Env_Rebindings) is
   begin
      Node.Unit.Rebindings.Append (Rebinding);
   end Register_Rebinding;


   --------------------
   -- Element_Parent --
   --------------------

   function Element_Parent
     (Node : Bare_Lkt_Node) return Bare_Lkt_Node
   is (Node.Parent);

   ---------------
   -- Node_Unit --
   ---------------

   function Node_Unit (Node : Bare_Lkt_Node) return Generic_Unit_Ptr is
   begin
      return Convert_Unit (Node.Unit);
   end Node_Unit;

   ----------
   -- Hash --
   ----------

   function Hash (Node : Bare_Lkt_Node) return Hash_Type
   is
      function H is new Hash_Access
        (Root_Node_Record, Bare_Lkt_Node);
   begin
      return H (Node);
   end Hash;

      function Hash (B : Boolean) return Hash_Type is (Boolean'Pos (B));





   ------------------------
   -- Named environments --
   ------------------------

   ---------
   -- Add --
   ---------

   procedure Add
     (Self : in out NED_Assoc_Maps.Map;
      Key  : Symbol_Type;
      Node : AST_Envs.Internal_Map_Node)
   is
      use NED_Assoc_Maps;

      Pos   : Cursor;
      Dummy : Boolean;
   begin
      --  Make sure there is a vector entry for Key
      Self.Insert (Key, Internal_Map_Node_Vectors.Empty_Vector, Pos, Dummy);

      --  Append Node to that vector
      declare
         V : Internal_Map_Node_Vectors.Vector renames Self.Reference (Pos);
      begin
         V.Append (Node);
      end;
   end Add;

   ------------
   -- Remove --
   ------------

   procedure Remove
     (Self : in out NED_Assoc_Maps.Map;
      Key  : Symbol_Type;
      Node : Bare_Lkt_Node)
   is
      use NED_Assoc_Maps;

      V : Internal_Map_Node_Vectors.Vector renames Self.Reference (Key);
   begin
      --  Remove the (assumed unique) entry in V whose node is Node. The order
      --  of items in V is not significant, so we can use Pop for efficient
      --  removal. Do the traversal in reverse order for correctness.
      for I in reverse 1 .. V.Length loop
         if V.Get_Access (I).Node = Node then
            V.Pop (I);
            exit;
         end if;
      end loop;
   end Remove;

   ------------------------------
   -- Get_Named_Env_Descriptor --
   ------------------------------

   function Get_Named_Env_Descriptor
     (Context : Internal_Context;
      Name    : Symbol_Type) return Named_Env_Descriptor_Access
   is
      use NED_Maps;

      --  Look for an existing entry for Name
      Pos : constant Cursor := Context.Named_Envs.Find (Name);
   begin
      if Has_Element (Pos) then
         return Element (Pos);
      end if;

      --  There is no such entry: create one
      return Result : constant Named_Env_Descriptor_Access :=
         new Named_Env_Descriptor'
           (Name                   => Name,
            Envs                   => <>,
            Env_With_Precedence    => Empty_Env,
            Foreign_Nodes          => <>,
            Foreign_Envs           => <>,
            Nodes_With_Foreign_Env => <>)
      do
         Context.Named_Envs.Insert (Name, Result);
      end return;
   end Get_Named_Env_Descriptor;

   ------------------------
   -- Register_Named_Env --
   ------------------------

   procedure Register_Named_Env
     (Context                   : Internal_Context;
      Name                      : Symbol_Type;
      Env                       : Lexical_Env;
      Named_Envs_Needing_Update : in out NED_Maps.Map)
   is
      NED_Access : constant Named_Env_Descriptor_Access :=
         Get_Named_Env_Descriptor (Context, Name);
      NED        : Named_Env_Descriptor renames NED_Access.all;
      Node       : constant Bare_Lkt_Node := Env_Node (Env);
   begin
      NED.Envs.Insert (Node, Env);
      Node.Unit.Named_Envs.Append ((Name, Env));

      --  If that insertion must change the env that has precedence, signal
      --  that NED requires an update.

      if NED.Envs.First_Element /= NED.Env_With_Precedence then
         Named_Envs_Needing_Update.Include (Name, NED_Access);
      end if;
   end Register_Named_Env;

   ----------------------
   -- Update_Named_Env --
   ----------------------

   procedure Update_Named_Envs
     (Context : Internal_Context; Named_Envs : NED_Maps.Map)
   is
      Require_Cache_Reset : Boolean := False;
   begin
      for Cur in Named_Envs.Iterate loop
         declare
            NE      : Named_Env_Descriptor renames NED_Maps.Element (Cur).all;
            New_Env : constant Lexical_Env :=
              (if NE.Envs.Is_Empty
               then Empty_Env
               else NE.Envs.First_Element);
         begin
            --  If there was an environment with precedence, remove its foreign
            --  nodes.
            if NE.Env_With_Precedence /= Empty_Env then
               for Cur in NE.Foreign_Nodes.Iterate loop
                  declare
                     Key   : constant Symbol_Type :=
                        NED_Assoc_Maps.Key (Cur);
                     Nodes : Internal_Map_Node_Vectors.Vector renames
                        NE.Foreign_Nodes.Reference (Cur);
                  begin
                     for N of Nodes loop
                        Remove (NE.Env_With_Precedence, Thin (Key), N.Node);
                     end loop;
                  end;
               end loop;
            end if;

            --  Now, set the new environment that has precedence
            NE.Env_With_Precedence := New_Env;

            --  Add the foreign nodes to the new environment with precedence,
            --  if any.
            for Cur in NE.Foreign_Nodes.Iterate loop
               declare
                  Key   : constant Symbol_Type :=
                     NED_Assoc_Maps.Key (Cur);
                  Nodes : Internal_Map_Node_Vectors.Vector renames
                     NE.Foreign_Nodes.Reference (Cur);
               begin
                  for N of Nodes loop
                     Add (New_Env, Thin (Key), N.Node, N.Md, N.Resolver);
                  end loop;
               end;
            end loop;

            --  Set the parent environment of all foreign environments
            for Cur in NE.Foreign_Envs.Iterate loop
               declare
                  Env : Lexical_Env_Record renames
                     Unwrap (Sorted_Env_Maps.Element (Cur)).all;
               begin
                  Env.Parent := New_Env;

                  --  We have updated the lexical env hierarchy (e.g. an env
                  --  which had no parent may have one now), so the cached
                  --  entries for queries that traveresed the old env hierarchy
                  --  need to be invalidated.
                  Require_Cache_Reset := True;
               end;
            end loop;

            --  Update nodes whose environment was the old env with precedence
            for N of NE.Nodes_With_Foreign_Env loop
               N.Self_Env := New_Env;
            end loop;
         end;
      end loop;
      if Require_Cache_Reset then
         Invalidate_Caches (Context, Invalidate_Envs => True);
      end if;
   end Update_Named_Envs;

   --------------------------
   -- Big integers wrapper --
   --------------------------

   ------------------------
   -- Create_Big_Integer --
   ------------------------

   function Create_Big_Integer
     (Image : String; Base : Integer := 10) return Big_Integer_Type
   is
      use GNATCOLL.GMP;
      use GNATCOLL.GMP.Integers;
   begin
      return new Big_Integer_Record'(Value     => Make (Image, Int (Base)),
                                     Ref_Count => 1);
   end Create_Big_Integer;

   ------------------------
   -- Create_Big_Integer --
   ------------------------

   function Create_Big_Integer
     (Big_Int : GNATCOLL.GMP.Integers.Big_Integer) return Big_Integer_Type
   is
      Result : constant Big_Integer_Type :=
         new Big_Integer_Record'(Value     => <>,
                                 Ref_Count => 1);
   begin
      Result.Value.Set (Big_Int);
      return Result;
   end Create_Big_Integer;

   ------------------------
   -- Create_Big_Integer --
   ------------------------

   function Create_Big_Integer (Int : Integer) return Big_Integer_Type is
      Result : constant Big_Integer_Type :=
         new Big_Integer_Record'(Value     => <>,
                                 Ref_Count => 1);
   begin
      Result.Value.Set (GNATCOLL.GMP.Long (Int));
      return Result;
   end Create_Big_Integer;

   -------------------------------
   -- Create_Public_Big_Integer --
   -------------------------------

   function Create_Public_Big_Integer
     (Big_Int : Big_Integer_Type) return GNATCOLL.GMP.Integers.Big_Integer is
   begin
      return Result : GNATCOLL.GMP.Integers.Big_Integer do
         Result.Set (Big_Int.Value);
      end return;
   end Create_Public_Big_Integer;

   -----------------
   -- Trace_Image --
   -----------------

   function Trace_Image (I : Big_Integer_Type) return String is
   begin
      return GNATCOLL.GMP.Integers.Image (I.Value);
   end Trace_Image;

   ----------------
   -- To_Integer --
   ----------------

   function To_Integer
     (Self    : Bare_Lkt_Node;
      Big_Int : Big_Integer_Type) return Integer
   is
      Image : constant String := Big_Int.Value.Image;
   begin
      return Integer'Value (Image);
   exception
      when Constraint_Error =>
         Raise_Property_Exception
           (Self, Property_Error'Identity, "out of range big integer");
   end To_Integer;

   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (Big_Int : Big_Integer_Type) is
   begin
      if Big_Int.Ref_Count /= -1 then
         Big_Int.Ref_Count := Big_Int.Ref_Count + 1;
      end if;
   end Inc_Ref;

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (Big_Int : in out Big_Integer_Type) is
      procedure Destroy is new Ada.Unchecked_Deallocation
        (Big_Integer_Record, Big_Integer_Type);
   begin
      if Big_Int = null or else Big_Int.Ref_Count = -1 then
         return;
      end if;

      Big_Int.Ref_Count := Big_Int.Ref_Count - 1;
      if Big_Int.Ref_Count = 0 then
         Destroy (Big_Int);
      else
         Big_Int := null;
      end if;
   end Dec_Ref;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (Left, Right : Big_Integer_Type) return Boolean is
      use type GNATCOLL.GMP.Integers.Big_Integer;
   begin
      return Left.Value = Right.Value;
   end Equivalent;

   ---------
   -- "<" --
   ---------

   function "<" (Left, Right : Big_Integer_Type) return Boolean is
      use type GNATCOLL.GMP.Integers.Big_Integer;
   begin
      return Left.Value < Right.Value;
   end "<";

   ----------
   -- "<=" --
   ----------

   function "<=" (Left, Right : Big_Integer_Type) return Boolean is
      use type GNATCOLL.GMP.Integers.Big_Integer;
   begin
      return Left.Value <= Right.Value;
   end "<=";

   ---------
   -- ">" --
   ---------

   function ">" (Left, Right : Big_Integer_Type) return Boolean is
      use type GNATCOLL.GMP.Integers.Big_Integer;
   begin
      return Left.Value > Right.Value;
   end ">";

   ----------
   -- ">=" --
   ----------

   function ">=" (Left, Right : Big_Integer_Type) return Boolean is
      use type GNATCOLL.GMP.Integers.Big_Integer;
   begin
      return Left.Value >= Right.Value;
   end ">=";

   ---------
   -- "+" --
   ---------

   function "+" (Left, Right : Big_Integer_Type) return Big_Integer_Type is
      use type GNATCOLL.GMP.Integers.Big_Integer;
   begin
      return Create_Big_Integer (Left.Value + Right.Value);
   end "+";

   ---------
   -- "-" --
   ---------

   function "-" (Left, Right : Big_Integer_Type) return Big_Integer_Type is
      use type GNATCOLL.GMP.Integers.Big_Integer;
   begin
      return Create_Big_Integer (Left.Value - Right.Value);
   end "-";

   ---------
   -- "-" --
   ---------

   function "-" (Value : Big_Integer_Type) return Big_Integer_Type is
      use type GNATCOLL.GMP.Integers.Big_Integer;
   begin
      return Create_Big_Integer (-Value.Value);
   end "-";

   ------------------
   -- Unit_Version --
   ------------------

   function Unit_Version (Unit : Generic_Unit_Ptr) return Version_Number is
   begin
      return Convert_Unit (Unit).Unit_Version;
   end Unit_Version;

   -------------------------
   -- Get_Context_Version --
   -------------------------

   function Get_Context_Version
     (Node : Bare_Lkt_Node) return Version_Number is
   begin
      return Node.Unit.Context.Cache_Version;
   end Get_Context_Version;

   ---------------
   --  Self_Env --
   ---------------

   function Self_Env (Node : Bare_Lkt_Node) return Lexical_Env is
   begin
      return Node.Self_Env;
   end Self_Env;

   --------------------------
   -- Properties_May_Raise --
   --------------------------

   function Properties_May_Raise
     (Exc : Ada.Exceptions.Exception_Occurrence) return Boolean is
   begin
      return Ada.Exceptions.Exception_Identity (Exc) in
            Property_Error'Identity
      ;
   end Properties_May_Raise;

   ----------------------
   -- Short_Text_Image --
   ----------------------

   function Short_Text_Image (Self : Bare_Lkt_Node) return Text_Type
   is
   begin
      if Self = null then
         return "None";
      end if;

      
   

   case Self.Kind is
            when Lkt_Id .. Lkt_Ref_Id =>
            
         return Id_Short_Image (Self);
      
            when Lkt_Grammar_Rule_Decl .. Lkt_Trait_Decl =>
            
         return Decl_Short_Image (Self);
      
      when others => 
         return "<" & To_Text (Kind_Name (Self))
                & " " & Node_Sloc_Image (Self) & ">";
      
   end case;

   end Short_Text_Image;

   ----------------------
   --- Node_Sloc_Image --
   ----------------------

   function Node_Sloc_Image (Self : Bare_Lkt_Node) return Text_Type
   is
      
   begin
         return To_Text
                  (Ada.Directories.Simple_Name (Get_Filename (Unit (Self))))
                & ":" & To_Text (Image (Sloc_Range (Self)));
   end Node_Sloc_Image;

   --------------------
   -- Snaps_At_Start --
   --------------------

   function Snaps_At_Start (Self : Bare_Lkt_Node) return Boolean is
   begin
      
   

   case Self.Kind is
      when others => 
         return False;
      
   end case;

   end Snaps_At_Start;

   ------------------
   -- Snaps_At_End --
   ------------------

   function Snaps_At_End (Self : Bare_Lkt_Node) return Boolean is
   begin
      
   

   case Self.Kind is
      when others => 
         return Is_Incomplete (Self);
      
   end case;

   end Snaps_At_End;

   -------------
   -- Parents --
   -------------

   function Parents
     (Node      : Bare_Lkt_Node;
      With_Self : Boolean := True)
      return Bare_Lkt_Node_Array_Access
   is
      Count : Natural := 0;
      Start : Bare_Lkt_Node :=
        (if With_Self then Node else Node.Parent);
      Cur   : Bare_Lkt_Node := Start;
   begin
      while Cur /= null loop
         Count := Count + 1;
         Cur := Cur.Parent;
      end loop;

      declare
         Result : constant Bare_Lkt_Node_Array_Access :=
            Create_Bare_Lkt_Node_Array (Count);
      begin
         Cur := Start;
         for I in Result.Items'Range loop
            Result.Items (I) := Cur;
            Cur := Cur.Parent;
         end loop;
         return Result;
      end;
   end Parents;

   -----------------------
   -- First_Child_Index --
   -----------------------

   function First_Child_Index (Node : Bare_Lkt_Node) return Natural
   is (1);

   ----------------------
   -- Last_Child_Index --
   ----------------------

   function Last_Child_Index (Node : Bare_Lkt_Node) return Natural
   is (Children_Count (Node));

   ---------------
   -- Get_Child --
   ---------------

   procedure Get_Child
     (Node            : Bare_Lkt_Node;
      Index           : Positive;
      Index_In_Bounds : out Boolean;
      Result          : out Bare_Lkt_Node)
   is
      K : constant Lkt_Node_Kind_Type := Node.Kind;
   begin
      

      Index_In_Bounds := True;
      Result := null;
      case Lkt_Lkt_Node (K) is
when Lkt_Argument_Range =>
declare
N_Bare_Argument : constant Bare_Argument := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Argument.Argument_F_Name;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Argument.Argument_F_Value;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Lexer_Case_Rule_Cond_Alt_Range =>
declare
N_Bare_Lexer_Case_Rule_Cond_Alt : constant Bare_Lexer_Case_Rule_Cond_Alt := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Lexer_Case_Rule_Cond_Alt.Lexer_Case_Rule_Cond_Alt_F_Cond_Exprs;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Lexer_Case_Rule_Cond_Alt.Lexer_Case_Rule_Cond_Alt_F_Send;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Lexer_Case_Rule_Default_Alt_Range =>
declare
N_Bare_Lexer_Case_Rule_Default_Alt : constant Bare_Lexer_Case_Rule_Default_Alt := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Lexer_Case_Rule_Default_Alt.Lexer_Case_Rule_Default_Alt_F_Send;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Match_Branch_Range =>
declare
N_Bare_Match_Branch : constant Bare_Match_Branch := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Match_Branch.Match_Branch_F_Decl;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Match_Branch.Match_Branch_F_Expr;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Pattern_Match_Branch_Range =>
declare
N_Bare_Pattern_Match_Branch : constant Bare_Pattern_Match_Branch := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Pattern_Match_Branch.Pattern_Match_Branch_F_Pattern;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Pattern_Match_Branch.Pattern_Match_Branch_F_Expr;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Block_Expr_Clause_Range =>
declare
N_Bare_Block_Expr_Clause : constant Bare_Block_Expr_Clause := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Block_Expr_Clause.Block_Expr_Clause_F_Clause;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Grammar_Rule_Decl_Range =>
declare
N_Bare_Grammar_Rule_Decl : constant Bare_Grammar_Rule_Decl := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Grammar_Rule_Decl.Grammar_Rule_Decl_F_Syn_Name;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Grammar_Rule_Decl.Grammar_Rule_Decl_F_Expr;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Binding_Val_Decl_Range =>
declare
N_Bare_Binding_Val_Decl : constant Bare_Binding_Val_Decl := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Binding_Val_Decl.Binding_Val_Decl_F_Syn_Name;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Enum_Lit_Decl_Range =>
declare
N_Bare_Enum_Lit_Decl : constant Bare_Enum_Lit_Decl := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Enum_Lit_Decl.Enum_Lit_Decl_F_Syn_Name;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Field_Decl_Range =>
declare
N_Bare_Field_Decl : constant Bare_Field_Decl := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Field_Decl.Field_Decl_F_Syn_Name;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Field_Decl.Field_Decl_F_Decl_Type;
                            return;
                    

                        when 3 =>
                            Result := N_Bare_Field_Decl.Field_Decl_F_Trait_Ref;
                            return;
                    

                        when 4 =>
                            Result := N_Bare_Field_Decl.Field_Decl_F_Default_Val;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Fun_Param_Decl_Range =>
declare
N_Bare_Fun_Param_Decl : constant Bare_Fun_Param_Decl := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Fun_Param_Decl.Fun_Param_Decl_F_Decl_Annotations;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Fun_Param_Decl.Fun_Param_Decl_F_Syn_Name;
                            return;
                    

                        when 3 =>
                            Result := N_Bare_Fun_Param_Decl.Fun_Param_Decl_F_Decl_Type;
                            return;
                    

                        when 4 =>
                            Result := N_Bare_Fun_Param_Decl.Fun_Param_Decl_F_Default_Val;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Lambda_Param_Decl_Range =>
declare
N_Bare_Lambda_Param_Decl : constant Bare_Lambda_Param_Decl := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Lambda_Param_Decl.Lambda_Param_Decl_F_Syn_Name;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Lambda_Param_Decl.Lambda_Param_Decl_F_Decl_Type;
                            return;
                    

                        when 3 =>
                            Result := N_Bare_Lambda_Param_Decl.Lambda_Param_Decl_F_Default_Val;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Dyn_Var_Decl_Range =>
declare
N_Bare_Dyn_Var_Decl : constant Bare_Dyn_Var_Decl := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Dyn_Var_Decl.Dyn_Var_Decl_F_Syn_Name;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Dyn_Var_Decl.Dyn_Var_Decl_F_Decl_Type;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Match_Val_Decl_Range =>
declare
N_Bare_Match_Val_Decl : constant Bare_Match_Val_Decl := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Match_Val_Decl.Match_Val_Decl_F_Syn_Name;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Match_Val_Decl.Match_Val_Decl_F_Decl_Type;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Val_Decl_Range =>
declare
N_Bare_Val_Decl : constant Bare_Val_Decl := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Val_Decl.Val_Decl_F_Syn_Name;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Val_Decl.Val_Decl_F_Decl_Type;
                            return;
                    

                        when 3 =>
                            Result := N_Bare_Val_Decl.Val_Decl_F_Expr;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Fun_Decl_Range =>
declare
N_Bare_Fun_Decl : constant Bare_Fun_Decl := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Fun_Decl.Fun_Decl_F_Syn_Name;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Fun_Decl.Fun_Decl_F_Params;
                            return;
                    

                        when 3 =>
                            Result := N_Bare_Fun_Decl.Fun_Decl_F_Return_Type;
                            return;
                    

                        when 4 =>
                            Result := N_Bare_Fun_Decl.Fun_Decl_F_Trait_Ref;
                            return;
                    

                        when 5 =>
                            Result := N_Bare_Fun_Decl.Fun_Decl_F_Body;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Env_Spec_Decl_Range =>
declare
N_Bare_Env_Spec_Decl : constant Bare_Env_Spec_Decl := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Env_Spec_Decl.Env_Spec_Decl_F_Syn_Name;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Env_Spec_Decl.Env_Spec_Decl_F_Actions;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Generic_Decl_Range =>
declare
N_Bare_Generic_Decl : constant Bare_Generic_Decl := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Generic_Decl.Generic_Decl_F_Generic_Param_Decls;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Generic_Decl.Generic_Decl_F_Decl;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Grammar_Decl_Range =>
declare
N_Bare_Grammar_Decl : constant Bare_Grammar_Decl := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Grammar_Decl.Grammar_Decl_F_Syn_Name;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Grammar_Decl.Grammar_Decl_F_Rules;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Lexer_Decl_Range =>
declare
N_Bare_Lexer_Decl : constant Bare_Lexer_Decl := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Lexer_Decl.Lexer_Decl_F_Syn_Name;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Lexer_Decl.Lexer_Decl_F_Rules;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Lexer_Family_Decl_Range =>
declare
N_Bare_Lexer_Family_Decl : constant Bare_Lexer_Family_Decl := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Lexer_Family_Decl.Lexer_Family_Decl_F_Syn_Name;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Lexer_Family_Decl.Lexer_Family_Decl_F_Rules;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Any_Type_Decl_Range =>
declare
N_Bare_Any_Type_Decl : constant Bare_Any_Type_Decl := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Any_Type_Decl.Any_Type_Decl_F_Traits;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Enum_Class_Alt_Decl_Range =>
declare
N_Bare_Enum_Class_Alt_Decl : constant Bare_Enum_Class_Alt_Decl := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Enum_Class_Alt_Decl.Enum_Class_Alt_Decl_F_Syn_Name;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Enum_Class_Alt_Decl.Enum_Class_Alt_Decl_F_Traits;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Function_Type_Range =>
declare
N_Bare_Function_Type : constant Bare_Function_Type := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Function_Type.Function_Type_F_Traits;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Generic_Param_Type_Decl_Range =>
declare
N_Bare_Generic_Param_Type_Decl : constant Bare_Generic_Param_Type_Decl := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Generic_Param_Type_Decl.Generic_Param_Type_Decl_F_Has_Class;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Generic_Param_Type_Decl.Generic_Param_Type_Decl_F_Syn_Name;
                            return;
                    

                        when 3 =>
                            Result := N_Bare_Generic_Param_Type_Decl.Generic_Param_Type_Decl_F_Traits;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Basic_Class_Decl =>
declare
N_Bare_Basic_Class_Decl : constant Bare_Basic_Class_Decl := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Basic_Class_Decl.Basic_Class_Decl_F_Syn_Name;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Basic_Class_Decl.Basic_Class_Decl_F_Syn_Base_Type;
                            return;
                    

                        when 3 =>
                            Result := N_Bare_Basic_Class_Decl.Basic_Class_Decl_F_Traits;
                            return;
                    

                        when others => null;
                    end case;
                
case Lkt_Basic_Class_Decl (K) is
when Lkt_Class_Decl_Range =>
declare
N_Bare_Class_Decl : constant Bare_Class_Decl := N_Bare_Basic_Class_Decl;
begin
case Index is

                        when 4 =>
                            Result := N_Bare_Class_Decl.Class_Decl_F_Decls;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Enum_Class_Decl_Range =>
declare
N_Bare_Enum_Class_Decl : constant Bare_Enum_Class_Decl := N_Bare_Basic_Class_Decl;
begin
case Index is

                        when 4 =>
                            Result := N_Bare_Enum_Class_Decl.Enum_Class_Decl_F_Branches;
                            return;
                    

                        when 5 =>
                            Result := N_Bare_Enum_Class_Decl.Enum_Class_Decl_F_Decls;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when others => null;
end case;
end;
when Lkt_Enum_Type_Decl_Range =>
declare
N_Bare_Enum_Type_Decl : constant Bare_Enum_Type_Decl := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Enum_Type_Decl.Enum_Type_Decl_F_Syn_Name;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Enum_Type_Decl.Enum_Type_Decl_F_Traits;
                            return;
                    

                        when 3 =>
                            Result := N_Bare_Enum_Type_Decl.Enum_Type_Decl_F_Literals;
                            return;
                    

                        when 4 =>
                            Result := N_Bare_Enum_Type_Decl.Enum_Type_Decl_F_Decls;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Struct_Decl_Range =>
declare
N_Bare_Struct_Decl : constant Bare_Struct_Decl := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Struct_Decl.Struct_Decl_F_Syn_Name;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Struct_Decl.Struct_Decl_F_Traits;
                            return;
                    

                        when 3 =>
                            Result := N_Bare_Struct_Decl.Struct_Decl_F_Decls;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Trait_Decl_Range =>
declare
N_Bare_Trait_Decl : constant Bare_Trait_Decl := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Trait_Decl.Trait_Decl_F_Syn_Name;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Trait_Decl.Trait_Decl_F_Traits;
                            return;
                    

                        when 3 =>
                            Result := N_Bare_Trait_Decl.Trait_Decl_F_Decls;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Decl_Annotation_Range =>
declare
N_Bare_Decl_Annotation : constant Bare_Decl_Annotation := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Decl_Annotation.Decl_Annotation_F_Name;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Decl_Annotation.Decl_Annotation_F_Args;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Decl_Annotation_Args_Range =>
declare
N_Bare_Decl_Annotation_Args : constant Bare_Decl_Annotation_Args := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Decl_Annotation_Args.Decl_Annotation_Args_F_Args;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Elsif_Branch_Range =>
declare
N_Bare_Elsif_Branch : constant Bare_Elsif_Branch := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Elsif_Branch.Elsif_Branch_F_Cond_Expr;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Elsif_Branch.Elsif_Branch_F_Then_Expr;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Enum_Class_Case_Range =>
declare
N_Bare_Enum_Class_Case : constant Bare_Enum_Class_Case := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Enum_Class_Case.Enum_Class_Case_F_Decls;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Any_Of_Range =>
declare
N_Bare_Any_Of : constant Bare_Any_Of := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Any_Of.Any_Of_F_Expr;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Any_Of.Any_Of_F_Values;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Array_Literal_Range =>
declare
N_Bare_Array_Literal : constant Bare_Array_Literal := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Array_Literal.Array_Literal_F_Exprs;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Array_Literal.Array_Literal_F_Element_Type;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Base_Call_Expr =>
declare
N_Bare_Base_Call_Expr : constant Bare_Base_Call_Expr := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Base_Call_Expr.Base_Call_Expr_F_Name;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Base_Call_Expr.Base_Call_Expr_F_Args;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Bin_Op_Range =>
declare
N_Bare_Bin_Op : constant Bare_Bin_Op := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Bin_Op.Bin_Op_F_Left;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Bin_Op.Bin_Op_F_Op;
                            return;
                    

                        when 3 =>
                            Result := N_Bare_Bin_Op.Bin_Op_F_Right;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Block_Expr_Range =>
declare
N_Bare_Block_Expr : constant Bare_Block_Expr := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Block_Expr.Block_Expr_F_Clauses;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Cast_Expr_Range =>
declare
N_Bare_Cast_Expr : constant Bare_Cast_Expr := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Cast_Expr.Cast_Expr_F_Expr;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Cast_Expr.Cast_Expr_F_Null_Cond;
                            return;
                    

                        when 3 =>
                            Result := N_Bare_Cast_Expr.Cast_Expr_F_Excludes_Null;
                            return;
                    

                        when 4 =>
                            Result := N_Bare_Cast_Expr.Cast_Expr_F_Dest_Type;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Dot_Expr_Range =>
declare
N_Bare_Dot_Expr : constant Bare_Dot_Expr := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Dot_Expr.Dot_Expr_F_Prefix;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Dot_Expr.Dot_Expr_F_Null_Cond;
                            return;
                    

                        when 3 =>
                            Result := N_Bare_Dot_Expr.Dot_Expr_F_Suffix;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Error_On_Null_Range =>
declare
N_Bare_Error_On_Null : constant Bare_Error_On_Null := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Error_On_Null.Error_On_Null_F_Expr;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Generic_Instantiation_Range =>
declare
N_Bare_Generic_Instantiation : constant Bare_Generic_Instantiation := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Generic_Instantiation.Generic_Instantiation_F_Name;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Generic_Instantiation.Generic_Instantiation_F_Args;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Grammar_Discard_Range =>
declare
N_Bare_Grammar_Discard : constant Bare_Grammar_Discard := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Grammar_Discard.Grammar_Discard_F_Expr;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Grammar_Dont_Skip_Range =>
declare
N_Bare_Grammar_Dont_Skip : constant Bare_Grammar_Dont_Skip := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Grammar_Dont_Skip.Grammar_Dont_Skip_F_Expr;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Grammar_Dont_Skip.Grammar_Dont_Skip_F_Dont_Skip;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Grammar_List_Range =>
declare
N_Bare_Grammar_List : constant Bare_Grammar_List := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Grammar_List.Grammar_List_F_List_Type;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Grammar_List.Grammar_List_F_Kind;
                            return;
                    

                        when 3 =>
                            Result := N_Bare_Grammar_List.Grammar_List_F_Expr;
                            return;
                    

                        when 4 =>
                            Result := N_Bare_Grammar_List.Grammar_List_F_Sep;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Grammar_Null_Range =>
declare
N_Bare_Grammar_Null : constant Bare_Grammar_Null := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Grammar_Null.Grammar_Null_F_Name;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Grammar_Opt_Range =>
declare
N_Bare_Grammar_Opt : constant Bare_Grammar_Opt := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Grammar_Opt.Grammar_Opt_F_Expr;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Grammar_Opt_Error_Range =>
declare
N_Bare_Grammar_Opt_Error : constant Bare_Grammar_Opt_Error := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Grammar_Opt_Error.Grammar_Opt_Error_F_Expr;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Grammar_Opt_Error_Group_Range =>
declare
N_Bare_Grammar_Opt_Error_Group : constant Bare_Grammar_Opt_Error_Group := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Grammar_Opt_Error_Group.Grammar_Opt_Error_Group_F_Expr;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Grammar_Opt_Group_Range =>
declare
N_Bare_Grammar_Opt_Group : constant Bare_Grammar_Opt_Group := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Grammar_Opt_Group.Grammar_Opt_Group_F_Expr;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Grammar_Or_Expr_Range =>
declare
N_Bare_Grammar_Or_Expr : constant Bare_Grammar_Or_Expr := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Grammar_Or_Expr.Grammar_Or_Expr_F_Sub_Exprs;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Grammar_Pick_Range =>
declare
N_Bare_Grammar_Pick : constant Bare_Grammar_Pick := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Grammar_Pick.Grammar_Pick_F_Exprs;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Grammar_Predicate_Range =>
declare
N_Bare_Grammar_Predicate : constant Bare_Grammar_Predicate := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Grammar_Predicate.Grammar_Predicate_F_Expr;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Grammar_Predicate.Grammar_Predicate_F_Prop_Ref;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Grammar_Rule_Ref_Range =>
declare
N_Bare_Grammar_Rule_Ref : constant Bare_Grammar_Rule_Ref := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Grammar_Rule_Ref.Grammar_Rule_Ref_F_Node_Name;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Grammar_Skip_Range =>
declare
N_Bare_Grammar_Skip : constant Bare_Grammar_Skip := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Grammar_Skip.Grammar_Skip_F_Name;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Grammar_Stop_Cut_Range =>
declare
N_Bare_Grammar_Stop_Cut : constant Bare_Grammar_Stop_Cut := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Grammar_Stop_Cut.Grammar_Stop_Cut_F_Expr;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Parse_Node_Expr_Range =>
declare
N_Bare_Parse_Node_Expr : constant Bare_Parse_Node_Expr := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Parse_Node_Expr.Parse_Node_Expr_F_Node_Name;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Parse_Node_Expr.Parse_Node_Expr_F_Sub_Exprs;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Token_No_Case_Lit_Range =>
declare
N_Bare_Token_No_Case_Lit : constant Bare_Token_No_Case_Lit := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Token_No_Case_Lit.Token_No_Case_Lit_F_Lit;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Token_Pattern_Concat_Range =>
declare
N_Bare_Token_Pattern_Concat : constant Bare_Token_Pattern_Concat := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Token_Pattern_Concat.Token_Pattern_Concat_F_Left;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Token_Pattern_Concat.Token_Pattern_Concat_F_Right;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Token_Ref_Range =>
declare
N_Bare_Token_Ref : constant Bare_Token_Ref := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Token_Ref.Token_Ref_F_Token_Name;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Token_Ref.Token_Ref_F_Expr;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_If_Expr_Range =>
declare
N_Bare_If_Expr : constant Bare_If_Expr := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_If_Expr.If_Expr_F_Cond_Expr;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_If_Expr.If_Expr_F_Then_Expr;
                            return;
                    

                        when 3 =>
                            Result := N_Bare_If_Expr.If_Expr_F_Alternatives;
                            return;
                    

                        when 4 =>
                            Result := N_Bare_If_Expr.If_Expr_F_Else_Expr;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Isa_Range =>
declare
N_Bare_Isa : constant Bare_Isa := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Isa.Isa_F_Expr;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Isa.Isa_F_Pattern;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Keep_Expr_Range =>
declare
N_Bare_Keep_Expr : constant Bare_Keep_Expr := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Keep_Expr.Keep_Expr_F_Expr;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Keep_Expr.Keep_Expr_F_Null_Cond;
                            return;
                    

                        when 3 =>
                            Result := N_Bare_Keep_Expr.Keep_Expr_F_Keep_Type;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Lambda_Expr_Range =>
declare
N_Bare_Lambda_Expr : constant Bare_Lambda_Expr := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Lambda_Expr.Lambda_Expr_F_Params;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Lambda_Expr.Lambda_Expr_F_Return_Type;
                            return;
                    

                        when 3 =>
                            Result := N_Bare_Lambda_Expr.Lambda_Expr_F_Body;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Null_Lit_Range =>
declare
N_Bare_Null_Lit : constant Bare_Null_Lit := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Null_Lit.Null_Lit_F_Dest_Type;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Block_String_Lit_Range =>
declare
N_Bare_Block_String_Lit : constant Bare_Block_String_Lit := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Block_String_Lit.Block_String_Lit_F_Lines;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Logic_Assign_Range =>
declare
N_Bare_Logic_Assign : constant Bare_Logic_Assign := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Logic_Assign.Logic_Assign_F_Dest_Var;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Logic_Assign.Logic_Assign_F_Value;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Logic_Expr_Range =>
declare
N_Bare_Logic_Expr : constant Bare_Logic_Expr := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Logic_Expr.Logic_Expr_F_Expr;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Logic_Propagate_Range =>
declare
N_Bare_Logic_Propagate : constant Bare_Logic_Propagate := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Logic_Propagate.Logic_Propagate_F_Dest_Var;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Logic_Propagate.Logic_Propagate_F_Call;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Logic_Unify_Range =>
declare
N_Bare_Logic_Unify : constant Bare_Logic_Unify := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Logic_Unify.Logic_Unify_F_Lhs;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Logic_Unify.Logic_Unify_F_Rhs;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Match_Expr_Range =>
declare
N_Bare_Match_Expr : constant Bare_Match_Expr := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Match_Expr.Match_Expr_F_Match_Expr;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Match_Expr.Match_Expr_F_Branches;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Not_Expr_Range =>
declare
N_Bare_Not_Expr : constant Bare_Not_Expr := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Not_Expr.Not_Expr_F_Expr;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Paren_Expr_Range =>
declare
N_Bare_Paren_Expr : constant Bare_Paren_Expr := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Paren_Expr.Paren_Expr_F_Expr;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Raise_Expr_Range =>
declare
N_Bare_Raise_Expr : constant Bare_Raise_Expr := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Raise_Expr.Raise_Expr_F_Dest_Type;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Raise_Expr.Raise_Expr_F_Except_Expr;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Subscript_Expr_Range =>
declare
N_Bare_Subscript_Expr : constant Bare_Subscript_Expr := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Subscript_Expr.Subscript_Expr_F_Prefix;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Subscript_Expr.Subscript_Expr_F_Null_Cond;
                            return;
                    

                        when 3 =>
                            Result := N_Bare_Subscript_Expr.Subscript_Expr_F_Index;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Try_Expr_Range =>
declare
N_Bare_Try_Expr : constant Bare_Try_Expr := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Try_Expr.Try_Expr_F_Try_Expr;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Try_Expr.Try_Expr_F_Or_Expr;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Un_Op_Range =>
declare
N_Bare_Un_Op : constant Bare_Un_Op := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Un_Op.Un_Op_F_Op;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Un_Op.Un_Op_F_Expr;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Full_Decl_Range =>
declare
N_Bare_Full_Decl : constant Bare_Full_Decl := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Full_Decl.Full_Decl_F_Doc;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Full_Decl.Full_Decl_F_Decl_Annotations;
                            return;
                    

                        when 3 =>
                            Result := N_Bare_Full_Decl.Full_Decl_F_Decl;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Grammar_List_Sep_Range =>
declare
N_Bare_Grammar_List_Sep : constant Bare_Grammar_List_Sep := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Grammar_List_Sep.Grammar_List_Sep_F_Token;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Grammar_List_Sep.Grammar_List_Sep_F_Extra;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Import_Range =>
declare
N_Bare_Import : constant Bare_Import := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Import.Import_F_Name;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Langkit_Root_Range =>
declare
N_Bare_Langkit_Root : constant Bare_Langkit_Root := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Langkit_Root.Langkit_Root_F_Imports;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Langkit_Root.Langkit_Root_F_Decls;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Lexer_Case_Rule_Range =>
declare
N_Bare_Lexer_Case_Rule : constant Bare_Lexer_Case_Rule := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Lexer_Case_Rule.Lexer_Case_Rule_F_Expr;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Lexer_Case_Rule.Lexer_Case_Rule_F_Alts;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Lexer_Case_Rule_Send_Range =>
declare
N_Bare_Lexer_Case_Rule_Send : constant Bare_Lexer_Case_Rule_Send := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Lexer_Case_Rule_Send.Lexer_Case_Rule_Send_F_Sent;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Lexer_Case_Rule_Send.Lexer_Case_Rule_Send_F_Match_Size;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Lkt_Node_Base_List =>
declare
N_Bare_Lkt_Node_Base_List : constant Bare_Lkt_Node_Base_List := Node;
begin

                    if Index > N_Bare_Lkt_Node_Base_List.Count then
                        Index_In_Bounds := False;
                    else
                        Result := N_Bare_Lkt_Node_Base_List.Nodes (Index);
                    end if;
                    return;
                
end;
when Lkt_Binding_Pattern_Range =>
declare
N_Bare_Binding_Pattern : constant Bare_Binding_Pattern := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Binding_Pattern.Binding_Pattern_F_Decl;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Binding_Pattern.Binding_Pattern_F_Sub_Pattern;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Ellipsis_Pattern_Range =>
declare
N_Bare_Ellipsis_Pattern : constant Bare_Ellipsis_Pattern := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Ellipsis_Pattern.Ellipsis_Pattern_F_Binding;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Extended_Pattern_Range =>
declare
N_Bare_Extended_Pattern : constant Bare_Extended_Pattern := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Extended_Pattern.Extended_Pattern_F_Sub_Pattern;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Extended_Pattern.Extended_Pattern_F_Details;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Filtered_Pattern_Range =>
declare
N_Bare_Filtered_Pattern : constant Bare_Filtered_Pattern := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Filtered_Pattern.Filtered_Pattern_F_Sub_Pattern;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Filtered_Pattern.Filtered_Pattern_F_Predicate;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_List_Pattern_Range =>
declare
N_Bare_List_Pattern : constant Bare_List_Pattern := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_List_Pattern.List_Pattern_F_Sub_Patterns;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Not_Pattern_Range =>
declare
N_Bare_Not_Pattern : constant Bare_Not_Pattern := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Not_Pattern.Not_Pattern_F_Sub_Pattern;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Or_Pattern_Range =>
declare
N_Bare_Or_Pattern : constant Bare_Or_Pattern := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Or_Pattern.Or_Pattern_F_Left_Sub_Pattern;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Or_Pattern.Or_Pattern_F_Right_Sub_Pattern;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Paren_Pattern_Range =>
declare
N_Bare_Paren_Pattern : constant Bare_Paren_Pattern := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Paren_Pattern.Paren_Pattern_F_Sub_Pattern;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Tuple_Pattern_Range =>
declare
N_Bare_Tuple_Pattern : constant Bare_Tuple_Pattern := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Tuple_Pattern.Tuple_Pattern_F_Sub_Patterns;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Type_Pattern_Range =>
declare
N_Bare_Type_Pattern : constant Bare_Type_Pattern := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Type_Pattern.Type_Pattern_F_Type_Name;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Field_Pattern_Detail_Range =>
declare
N_Bare_Field_Pattern_Detail : constant Bare_Field_Pattern_Detail := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Field_Pattern_Detail.Field_Pattern_Detail_F_Id;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Field_Pattern_Detail.Field_Pattern_Detail_F_Expected_Value;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Property_Pattern_Detail_Range =>
declare
N_Bare_Property_Pattern_Detail : constant Bare_Property_Pattern_Detail := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Property_Pattern_Detail.Property_Pattern_Detail_F_Call;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Property_Pattern_Detail.Property_Pattern_Detail_F_Expected_Value;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Selector_Pattern_Detail_Range =>
declare
N_Bare_Selector_Pattern_Detail : constant Bare_Selector_Pattern_Detail := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Selector_Pattern_Detail.Selector_Pattern_Detail_F_Call;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Selector_Pattern_Detail.Selector_Pattern_Detail_F_Sub_Pattern;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Selector_Call_Range =>
declare
N_Bare_Selector_Call : constant Bare_Selector_Call := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Selector_Call.Selector_Call_F_Quantifier;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Selector_Call.Selector_Call_F_Binding;
                            return;
                    

                        when 3 =>
                            Result := N_Bare_Selector_Call.Selector_Call_F_Selector_Call;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Function_Type_Ref_Range =>
declare
N_Bare_Function_Type_Ref : constant Bare_Function_Type_Ref := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Function_Type_Ref.Function_Type_Ref_F_Param_Types;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Function_Type_Ref.Function_Type_Ref_F_Return_Type;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Generic_Type_Ref_Range =>
declare
N_Bare_Generic_Type_Ref : constant Bare_Generic_Type_Ref := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Generic_Type_Ref.Generic_Type_Ref_F_Type_Name;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Generic_Type_Ref.Generic_Type_Ref_F_Args;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Simple_Type_Ref_Range =>
declare
N_Bare_Simple_Type_Ref : constant Bare_Simple_Type_Ref := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Simple_Type_Ref.Simple_Type_Ref_F_Type_Name;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Var_Bind_Range =>
declare
N_Bare_Var_Bind : constant Bare_Var_Bind := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Var_Bind.Var_Bind_F_Name;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Var_Bind.Var_Bind_F_Expr;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when others => null;
end case;

      --  Execution should reach this point iff nothing matched this index, so
      --  we must be out of bounds.
      Index_In_Bounds := False;
   end Get_Child;

   -----------
   -- Print --
   -----------

   procedure Print
     (Node        : Liblktlang_Support.Generic_API.Analysis.Lk_Node;
      Show_Slocs  : Boolean;
      Line_Prefix : String := "")
   is
      use Liblktlang_Support.Generic_API.Analysis;
      use Liblktlang_Support.Generic_API.Introspection;

      T : Type_Ref;
   begin
      if Node.Is_Null then
         Put_Line ("None");
         return;
      end if;

      T := Type_Of (Node);
      Put (Line_Prefix & Image (Node_Type_Repr_Name (T)));
      if Show_Slocs then
         Put ("[" & Image (Node.Sloc_Range) & "]");
      end if;

      if Node.Is_Incomplete then
         Put (" <<INCOMPLETE>>");
      end if;

      if Node.Is_Token_Node then
         Put_Line (": " & Image (Node.Text));

      elsif Is_List_Node (Node) then

         --  List nodes are displayed in a special way (they have no field)

         declare
            Count : constant Natural := Node.Children_Count;
            Child : Lk_Node;
         begin
            if Count = 0 then
               Put_Line (": <empty list>");
               return;
            end if;
            New_Line;

            for I in 1 .. Count loop
               Child := Node.Child (I);
               if not Child.Is_Null then
                  Print (Child, Show_Slocs, Line_Prefix & "|  ");
               end if;
            end loop;
         end;

      else
         --  This is for regular nodes: display each syntax field (i.e.
         --  non-property member).

         declare
            Attr_Prefix     : constant String := Line_Prefix & "|";
            Children_Prefix : constant String := Line_Prefix & "|  ";
            M_List          : constant Struct_Member_Ref_Array := Members (T);
            Child           : Lk_Node;
         begin
            New_Line;
            for M of M_List loop
               if not Is_Property (M) and then not Is_Null_For (M, T) then
                  Child := As_Node (Eval_Node_Member (Node, M));
                  Put (Attr_Prefix
                       & Image (Format_Name (Member_Name (M), Lower)) & ":");
                  if Child.Is_Null then
                     Put_Line (" <null>");
                  else
                     New_Line;
                     Print (Child, Show_Slocs, Children_Prefix);
                  end if;
               end if;
            end loop;
         end;
      end if;
   end Print;

   -----------
   -- Print --
   -----------

   procedure Print
     (Node        : Bare_Lkt_Node;
      Show_Slocs  : Boolean;
      Line_Prefix : String := "")
   is
      Entity : constant Internal_Entity := (Node, No_Entity_Info);
   begin
      Print (To_Generic_Node (Entity), Show_Slocs, Line_Prefix);
   end Print;

   ------------
   -- Parent --
   ------------

   function Parent (Node : Bare_Lkt_Node) return Bare_Lkt_Node is
   begin
      return Node.Parent;
   end Parent;

   ------------------
   -- Stored_Token --
   ------------------

   function Stored_Token
     (Node  : Bare_Lkt_Node;
      Token : Token_Reference) return Token_Index
   is
      Index : constant Token_Or_Trivia_Index := Get_Token_Index (Token);
   begin
      if Node.Unit.TDH'Access /= Get_Token_TDH (Token) then
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "Cannot associate a token and a node from different analysis"
            & " units");
      elsif Index.Trivia /= No_Token_Index then
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "A node cannot hold trivia");
      end if;

      return Index.Token;
   end Stored_Token;

   -------------------------
   -- Children_And_Trivia --
   -------------------------

   function Children_And_Trivia
     (Node : Bare_Lkt_Node) return Bare_Children_Vector
   is
      Ret_Vec : Bare_Children_Vector;
      Ctx     : Internal_Context renames Node.Unit.Context;
      TDH     : Token_Data_Handler renames Node.Unit.TDH;

      procedure Append_Trivias (First, Last : Token_Index);
      --  Append all the trivias of tokens between indices First and Last to
      --  the returned vector.

      function Filter_Children
        (Parent : Bare_Lkt_Node)
         return Internal_Bare_Lkt_Node_Array;
      --  Return an array for all children in Parent that are not null

      --------------------
      -- Append_Trivias --
      --------------------

      procedure Append_Trivias (First, Last : Token_Index) is
      begin
         for I in First .. Last loop
            for D of Get_Trivias (TDH, I) loop
               Ret_Vec.Append
                 (Bare_Child_Record'
                    (Kind   => Trivia,
                     Trivia => Wrap_Token_Reference
                                 (Ctx, TDH'Access, (I, D))));
            end loop;
         end loop;
      end Append_Trivias;

      ---------------------
      -- Filter_Children --
      ---------------------

      function Filter_Children
        (Parent : Bare_Lkt_Node)
         return Internal_Bare_Lkt_Node_Array
      is
         Children : constant Internal_Bare_Lkt_Node_Array :=
            Implementation.Children (Parent);
         Result   : Internal_Bare_Lkt_Node_Array (Children'Range);
         Next     : Integer := Result'First;
      begin
         for I in Children'Range loop
            if Children (I) /= null then
               Result (Next) := Children (I);
               Next := Next + 1;
            end if;
         end loop;
         return Result (Result'First .. Next - 1);
      end Filter_Children;

      First_Child : constant Positive := 1;
      N_Children  : constant Internal_Bare_Lkt_Node_Array :=
         Filter_Children (Node);
   begin
      if N_Children'Length > 0
        and then (Node.Token_Start_Index
                    /= N_Children (First_Child).Token_Start_Index)
      then
         Append_Trivias (Node.Token_Start_Index,
                         N_Children (First_Child).Token_Start_Index - 1);
      end if;

      --  Append each node to Ret_Vec, and append trivia that follow after each
      --  non-ghost nodes.
      for I in N_Children'Range loop
         Ret_Vec.Append (Bare_Child_Record'(Child, N_Children (I)));
         if not Is_Ghost (N_Children (I)) then
            Append_Trivias (N_Children (I).Token_End_Index,
                            (if I = N_Children'Last
                             then Node.Token_End_Index - 1
                             else N_Children (I + 1).Token_Start_Index - 1));
         end if;
      end loop;

      return Ret_Vec;
   end Children_And_Trivia;

   --------------
   -- Is_Ghost --
   --------------

   function Is_Ghost (Node : Bare_Lkt_Node) return Boolean
   is (Node.Token_End_Index = No_Token_Index);

   -------------------
   -- Is_Incomplete --
   -------------------

   function Is_Incomplete (Node : Bare_Lkt_Node) return Boolean
   is
      LGC : Bare_Lkt_Node;
   begin
     if Is_List_Node (Node.Kind) then
        LGC := (if Last_Child_Index (Node) /= 0
                then Child (Node, Last_Child_Index (Node))
                else null);
        return LGC /= null and then Is_Incomplete (LGC);
      else
         return Node.Last_Attempted_Child > -1;
      end if;
   end;

   -----------------
   -- Token_Start --
   -----------------

   function Token_Start (Node : Bare_Lkt_Node) return Token_Reference
   is (Token (Node, Node.Token_Start_Index));

   ---------------
   -- Token_End --
   ---------------

   function Token_End (Node : Bare_Lkt_Node) return Token_Reference
   is
     (if Node.Token_End_Index = No_Token_Index
      then Token_Start (Node)
      else Token (Node, Node.Token_End_Index));

   -----------
   -- Token --
   -----------

   function Token
     (Node  : Bare_Lkt_Node;
      Index : Token_Index) return Token_Reference
   is
      Unit    : constant Internal_Unit := Node.Unit;
      Context : constant Internal_Context := Unit.Context;
   begin
      return Wrap_Token_Reference
        (Context, Token_Data (Unit), (Index, No_Token_Index));
   end Token;

   ---------
   -- "<" --
   ---------

   function "<" (Left, Right : Bare_Lkt_Node) return Boolean is
   begin
      --  Reject invalid inputs
      if Left /= null and Is_Synthetic (Left) then
         raise Property_Error with "left node is synthetic";
      elsif Right /= null and Is_Synthetic (Right) then
         raise Property_Error with "right node is synthetic";
      end if;

      --  Null nodes come first
      if Left = null then
         return Right /= null;
      elsif Right = null then
         return False;
      end if;

      --  So we have two non-null nodes. Sort by unit filename
      if Left.Unit < Right.Unit then
         return True;
      elsif Left.Unit /= Right.Unit then
         return False;
      end if;

      --  Both nodes come from the same unit: compare their token indexes
      if Left.Token_Start_Index < Right.Token_Start_Index then
         return True;
      elsif Left.Token_Start_Index > Right.Token_Start_Index then
         return False;
      else
         return Left.Token_End_Index < Right.Token_End_Index;
      end if;
   end "<";

   -------------
   -- Is_Null --
   -------------

   function Is_Null (Node : Bare_Lkt_Node) return Boolean
   is (Node = null);

   ----------
   -- Kind --
   ----------

   function Kind (Node : Bare_Lkt_Node) return Lkt_Node_Kind_Type
   is (Node.Kind);

   -----------------
   -- Child_Index --
   -----------------

   function Child_Index (Node : Bare_Lkt_Node) return Integer
   is
      N : Bare_Lkt_Node := null;
   begin
      if Node.Parent = null then
         raise Property_Error with
            "Trying to get the child index of a root node";
      end if;

      for I in First_Child_Index (Node.Parent)
            .. Last_Child_Index (Node.Parent)
      loop
         N := Child (Node.Parent, I);
         if N = Node then
            return I - 1;
         end if;
      end loop;

      --  If we reach this point, then Node isn't a Child of Node.Parent. This
      --  is not supposed to happen.
      raise Program_Error;
   end Child_Index;

   -------------------
   -- Fetch_Sibling --
   -------------------

   function Fetch_Sibling
     (Node   : Bare_Lkt_Node;
      Offset : Integer) return Bare_Lkt_Node is
   begin
      --  Root nodes have no sibling: handle them now to avoid invalid requests
      --  in the code below.
      if Node.Parent = null then
         return null;
      end if;

      declare
         Node_Index : constant Positive := Child_Index (Node) + 1;
         --  Child_Index is 0-based, but the Child primitive expects a 1-based
         --  index.

         Sibling_Index : constant Integer := Node_Index + Offset;
      begin
         --  Child returns null for out-of-bound indexes

         return (if Sibling_Index >= 1
                 then Child (Node.Parent, Sibling_Index)
                 else null);
      end;
   end Fetch_Sibling;

   -------------------
   -- Fetch_Sibling --
   -------------------

   function Fetch_Sibling
     (Node   : Bare_Lkt_Node;
      E_Info : Internal_Entity_Info;
      Offset : Integer) return Internal_Entity
   is
      Sibling : constant Bare_Lkt_Node := Fetch_Sibling (Node, Offset);
   begin
      --  Don't forget to clear entity info if the result is nul

      return (if Sibling = null
              then No_Entity
              else (Sibling, E_Info));
   end Fetch_Sibling;

   ----------------------
   -- Previous_Sibling --
   ----------------------

   function Previous_Sibling
     (Node   : Bare_Lkt_Node;
      E_Info : Internal_Entity_Info := No_Entity_Info)
      return Internal_Entity is
   begin
      return Fetch_Sibling (Node, E_Info, -1);
   end Previous_Sibling;

   ------------------
   -- Next_Sibling --
   ------------------

   function Next_Sibling
     (Node   : Bare_Lkt_Node;
      E_Info : Internal_Entity_Info := No_Entity_Info)
      return Internal_Entity is
   begin
      return Fetch_Sibling (Node, E_Info, 1);
   end Next_Sibling;

   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (Self : Node_Builder_Type) is
   begin
      if Self.Ref_Count > 0 then
         Self.Ref_Count := Self.Ref_Count + 1;
      end if;
   end Inc_Ref;

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (Self : in out Node_Builder_Type) is
   begin
      if Self = null or else Self.Ref_Count < 0 then
         return;
      elsif Self.Ref_Count = 1 then
         Self.Release;
         Free (Self);
      else
         Self.Ref_Count := Self.Ref_Count - 1;
      end if;
   end Dec_Ref;

   ------------------------------
   -- Create_Copy_Node_Builder --
   ------------------------------

   function Create_Copy_Node_Builder
     (Value : Bare_Lkt_Node) return Node_Builder_Type is
   begin
      --  No need to allocate a new builder if in practice it cannot be
      --  distinguished from the "null" builder.

      if Value = null then
         return Null_Node_Builder;
      else
         return new Copy_Node_Builder_Record'(Ref_Count => 1, Value => Value);
      end if;
   end Create_Copy_Node_Builder;

         

         type Bare_Synthetic_Type_Ref_List_Node_Builder_Record is new Node_Builder_Record with
            record
               List_Elements : Bare_Type_Ref_Node_Builder_Array_Access;
            end record;
         type Bare_Synthetic_Type_Ref_List_Node_Builder_Access is access all Bare_Synthetic_Type_Ref_List_Node_Builder_Record;

         overriding function Build
           (Self              : Bare_Synthetic_Type_Ref_List_Node_Builder_Record;
            Parent, Self_Node : Bare_Lkt_Node)
            return Bare_Lkt_Node;

         overriding function Trace_Image
           (Self : Bare_Synthetic_Type_Ref_List_Node_Builder_Record) return String
         is ("<NodeBuilder to synthetize SyntheticTypeRefList>");

            overriding procedure Release (Self : in out Bare_Synthetic_Type_Ref_List_Node_Builder_Record);

         -----------
         -- Build --
         -----------

         overriding function Build
           (Self              : Bare_Synthetic_Type_Ref_List_Node_Builder_Record;
            Parent, Self_Node : Bare_Lkt_Node)
            return Bare_Lkt_Node
         is
            Result : Bare_Lkt_Node;
            Unit   : constant Internal_Unit := Self_Node.Unit;
            Env    : constant Lexical_Env :=
              (if Parent = null
               then Empty_Env
               else Parent.Self_Env);
         begin
            if Parent /= null and then Parent.Unit /= Unit then
               Raise_Property_Exception
                 (Self_Node,
                  Property_Error'Identity,
                  "synthetic node parents must belong to the same unit as the"
                  & " nodes that trigger node synthetization");
            end if;

            if Is_Foreign_Strict (Env, Parent) then
               Raise_Property_Exception
                 (Self_Node,
                  Property_Error'Identity,
                  "synthetic nodes cannot have foreign lexical envs");
            end if;

            Result := new Root_Node_Record
              (Lkt_Synthetic_Type_Ref_List);
            Initialize
              (Self => Result,
               Kind => Lkt_Synthetic_Type_Ref_List,
               Unit => Unit,

               Token_Start_Index => No_Token_Index,
               Token_End_Index   => No_Token_Index,

               Parent   => Parent,
               Self_Env => Env);
            Register_Destroyable (Unit, Result);

               declare
                  Elements_Builders : Internal_Bare_Type_Ref_Node_Builder_Array renames Self.List_Elements.Items;
                  Children          : constant
                    Alloc_AST_List_Array.Element_Array_Access :=
                      Allocate_Synthetic_List_Children
                        (Result, Elements_Builders'Length);
               begin
                  for I in Elements_Builders'Range loop
                     Children (I) :=
                       Elements_Builders (I).Build (Result, Self_Node);
                  end loop;
               end;


            return Result;
         end Build;


            -------------
            -- Release --
            -------------

            overriding procedure Release (Self : in out Bare_Synthetic_Type_Ref_List_Node_Builder_Record) is
            begin
                     Dec_Ref (Self.List_Elements);
            end Release;

         function Create_Bare_Synthetic_Type_Ref_List_Node_Builder
             (List_Elements : Bare_Type_Ref_Node_Builder_Array_Access)
           return Bare_Synthetic_Type_Ref_List_Node_Builder
         is
            Builder : constant Bare_Synthetic_Type_Ref_List_Node_Builder_Access := new Bare_Synthetic_Type_Ref_List_Node_Builder_Record;
         begin
            Builder.Ref_Count := 1;
               Builder.List_Elements := List_Elements;
                  Inc_Ref (Builder.List_Elements);
            return Node_Builder_Type (Builder);
         end Create_Bare_Synthetic_Type_Ref_List_Node_Builder;


   ----------------------
   -- Compare_Metadata --
   ----------------------

   --  Deactivate "not referenced" warnings because if the metadata struct has
   --  no fields, formals and temporaries won't be referenced in the two
   --  following functions.
   pragma Warnings (Off, "referenced");
   function Compare_Metadata (L, R : Internal_Metadata) return Boolean is
   begin
      return True;
   end Compare_Metadata;

   ----------
   -- Hash --
   ----------

   function Hash (Self : Internal_Metadata) return Hash_Type is
      Ret : Hash_Type := Liblktlang_Support.Hashes.Initial_Hash;
   begin
      return Ret;
   end Hash;
   pragma Warnings (On, "referenced");

   -------------
   -- Combine --
   -------------

   function Combine
     (L, R : Internal_Metadata) return Internal_Metadata
   is
      pragma Unreferenced (L, R);
      Ret : Internal_Metadata := No_Metadata;
   begin
      return Ret;
   end Combine;

   -------------------------------
   -- Create_Static_Lexical_Env --
   -------------------------------

   function Create_Static_Lexical_Env
     (Parent            : Lexical_Env;
      Node              : Bare_Lkt_Node;
      Sym_Table         : Symbol_Table;
      Transitive_Parent : Boolean := False) return Lexical_Env
   is
      Unit : constant Internal_Unit :=
        (if Node = null then null else Node.Unit);
   begin
      return Result : Lexical_Env := Create_Lexical_Env
        (Parent, Node, Transitive_Parent, Sym_Table, Convert_Unit (Unit))
      do
         if Unit /= null then
            Register_Destroyable (Unit, Unwrap (Result.Env));
         end if;
      end return;
   end Create_Static_Lexical_Env;

   ---------
   -- Get --
   ---------

   function Get
     (Self  : Bare_Lkt_Node;
      A     : AST_Envs.Entity_Array;
      Index : Integer) return Internal_Entity
   is
      function Length (A : AST_Envs.Entity_Array) return Natural
      is (A'Length);

      function Get
        (A     : AST_Envs.Entity_Array;
         Index : Integer) return Internal_Entity
      is (A (Index + 1)); --  A is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Entity,
         Sequence_Type => AST_Envs.Entity_Array,
         Length        => Length,
         Get           => Get);
      Result : Internal_Entity;
   begin
      if Relative_Get (A, Index, Result) then
         return Result;
      else
         Raise_Property_Exception
           (Self, Property_Error'Identity, "out-of-bounds array access");
      end if;
   end Get;

   -----------
   -- Group --
   -----------

   function Group
     (Envs   : Lexical_Env_Array_Access;
      Env_Md : Internal_Metadata := No_Metadata) return Lexical_Env
   is (Group (Lexical_Env_Array (Envs.Items), Env_Md));

       

   

      --
      --  Untyped wrappers for Bare_Lkt_Node
      --

         
function Lkt_Node_P_Shed_Rebindings
  (E : Entity
  ) return Internal_Entity;



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   

      --
      --  Untyped wrappers for Bare_Dyn_Env_Wrapper
      --

         
function Dyn_Env_Wrapper_P_Instantiation_Bindings
  (E : Entity
  ) return Internal_Inner_Env_Assoc_Array_Access;



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   

      --
      --  Untyped wrappers for Bare_Langkit_Root
      --

         
function Langkit_Root_P_Internal_Env
  (E : Entity
  ) return Lexical_Env;



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   




   ------------------
   -- Children_Env --
   ------------------

   function Children_Env
     (Node   : Bare_Lkt_Node;
      E_Info : Internal_Entity_Info := No_Entity_Info)
      return Lexical_Env
   is (Rebind_Env (Node.Self_Env, E_Info));

   --------------
   -- Node_Env --
   --------------

   function Node_Env
     (Node   : Bare_Lkt_Node;
      E_Info : Internal_Entity_Info := No_Entity_Info)
      return Lexical_Env
   is
      function Get_Base_Env return Lexical_Env;
      --  Return the environment that we need to rebind before returning

      ------------------
      -- Get_Base_Env --
      ------------------

      function Get_Base_Env return Lexical_Env is
         pragma Warnings (Off, "referenced");
         function Get_Parent_Env return Lexical_Env;
         pragma Warnings (On, "referenced");

         --------------------
         -- Get_Parent_Env --
         --------------------

         function Get_Parent_Env return Lexical_Env is
            Parent : constant Lexical_Env := AST_Envs.Parent (Node.Self_Env);
         begin
            --  If Node is the root scope or the empty environment, Parent can
            --  be a wrapper around the null node. Turn this into the
            --  Empty_Env, as null envs are erroneous values in properties.
            return (if Unwrap (Parent) = null
                    then Empty_Env
                    else Parent);
         end Get_Parent_Env;

      begin
         
         return
           (if Node.Kind in Lkt_Match_Branch | Lkt_Pattern_Match_Branch | Lkt_Field_Decl | Lkt_Fun_Decl | Lkt_Env_Spec_Decl | Lkt_Generic_Decl | Lkt_Grammar_Decl | Lkt_Lexer_Decl | Lkt_Block_Expr | Lkt_Lambda_Expr | Lkt_Decl_Block
            then Get_Parent_Env
            else Node.Self_Env);
      end Get_Base_Env;

      Base_Env : Lexical_Env := Get_Base_Env;
      Result   : constant Lexical_Env := Rebind_Env (Base_Env, E_Info);
   begin
      Dec_Ref (Base_Env);
      return Result;
   end Node_Env;

   ------------
   -- Parent --
   ------------

   function Parent
     (Node   : Bare_Lkt_Node;
      E_Info : Internal_Entity_Info := No_Entity_Info)
      return Internal_Entity is
   begin
      --  TODO: shed entity information as appropriate
      return (Node.Parent, E_Info);
   end Parent;

   -------------
   -- Parents --
   -------------

   function Parents
     (Node      : Bare_Lkt_Node;
      With_Self : Boolean := True;
      E_Info    : Internal_Entity_Info := No_Entity_Info)
      return Internal_Entity_Array_Access
   is
      Bare_Parents : Bare_Lkt_Node_Array_Access := Parents (Node, With_Self);
      Result       : Internal_Entity_Array_Access :=
         Create_Internal_Entity_Array (Bare_Parents.N);
   begin
      --  TODO: shed entity information as appropriate
      for I in Bare_Parents.Items'Range loop
         Result.Items (I) := (Bare_Parents.Items (I), E_Info);
      end loop;
      Dec_Ref (Bare_Parents);
      return Result;
   end Parents;

   --------------
   -- Children --
   --------------

   function Children
     (Node   : Bare_Lkt_Node;
      E_Info : Internal_Entity_Info := No_Entity_Info)
      return Internal_Entity_Array_Access
   is
      Bare_Children : Bare_Lkt_Node_Array_Access := Children (Node);
      Result        : Internal_Entity_Array_Access :=
         Create_Internal_Entity_Array (Bare_Children.N);
   begin
      --  TODO: shed entity information as appropriate
      for I in Bare_Children.Items'Range loop
         Result.Items (I) := (Bare_Children.Items (I), E_Info);
      end loop;
      Dec_Ref (Bare_Children);
      return Result;
   end Children;

   ---------------------
   -- New_Unit_String --
   ---------------------

   function New_Unit_String
     (Unit : Internal_Unit; Str : String) return String_Access
   is
      procedure Register_Destroyable_String is new Register_Destroyable_Gen
        (String, String_Access, Free);
   begin
      return Ret : String_Access := new String'(Str) do
         Register_Destroyable_String (Unit, Ret);
      end return;
   end New_Unit_String;

   --------------------------------
   -- Assign_Names_To_Logic_Vars --
   --------------------------------

   procedure Assign_Names_To_Logic_Vars (Node : Bare_Lkt_Node) is

      pragma Warnings (Off, "referenced");

      procedure Assign
        (Node  : Bare_Lkt_Node;
         LV    : in out Logic_Var_Record;
         Field : String);
      --  Assign a name to the LV logic variable. Node must be the node that
      --  owns LV, and Field must be the name of the field in Node that holds
      --  LV.

      ------------
      -- Assign --
      ------------

      procedure Assign
        (Node  : Bare_Lkt_Node;
         LV    : in out Logic_Var_Record;
         Field : String) is
      begin
         LV.Dbg_Name :=
           new String'(Image (Short_Text_Image (Node)) & "." & Field);
      end Assign;

      K : constant Lkt_Node_Kind_Type := Node.Kind;

      pragma Warnings (On, "referenced");

   begin
      
      case Lkt_Lkt_Node (K) is
when Lkt_Lambda_Param_Decl_Range =>
declare
N_Bare_Lambda_Param_Decl : constant Bare_Lambda_Param_Decl := Node;
begin
Assign (N_Bare_Lambda_Param_Decl,        N_Bare_Lambda_Param_Decl.Lambda_Param_Decl_F_Type_Var,        "type_var");
end;
when Lkt_Expr =>
declare
N_Bare_Expr : constant Bare_Expr := Node;
begin
Assign (N_Bare_Expr,        N_Bare_Expr.Expr_F_Expected_Type_Var,        "expected_type_var");
Assign (N_Bare_Expr,        N_Bare_Expr.Expr_F_Actual_Type_Var,        "actual_type_var");
Assign (N_Bare_Expr,        N_Bare_Expr.Expr_F_Generic_Func_Type_Var,        "generic_func_type_var");
case Lkt_Expr (K) is
when Lkt_Array_Literal_Range =>
declare
N_Bare_Array_Literal : constant Bare_Array_Literal := N_Bare_Expr;
begin
Assign (N_Bare_Array_Literal,        N_Bare_Array_Literal.Array_Literal_F_Expected_Exprs_Type_Var,        "expected_exprs_type_var");
Assign (N_Bare_Array_Literal,        N_Bare_Array_Literal.Array_Literal_F_Actual_Element_Type,        "actual_element_type");
end;
when Lkt_Generic_Instantiation_Range =>
declare
N_Bare_Generic_Instantiation : constant Bare_Generic_Instantiation := N_Bare_Expr;
begin
Assign (N_Bare_Generic_Instantiation,        N_Bare_Generic_Instantiation.Generic_Instantiation_F_Rebinded_Var,        "rebinded_var");
end;
when Lkt_Ref_Id_Range =>
declare
N_Bare_Ref_Id : constant Bare_Ref_Id := N_Bare_Expr;
begin
Assign (N_Bare_Ref_Id,        N_Bare_Ref_Id.Ref_Id_F_Ref_Var,        "ref_var");
end;
when Lkt_If_Expr_Range =>
declare
N_Bare_If_Expr : constant Bare_If_Expr := N_Bare_Expr;
begin
Assign (N_Bare_If_Expr,        N_Bare_If_Expr.If_Expr_F_Expected_Branch_Type_Var,        "expected_branch_type_var");
end;
when Lkt_Keep_Expr_Range =>
declare
N_Bare_Keep_Expr : constant Bare_Keep_Expr := N_Bare_Expr;
begin
Assign (N_Bare_Keep_Expr,        N_Bare_Keep_Expr.Keep_Expr_F_Array_Element_Type,        "array_element_type");
end;
when Lkt_Match_Expr_Range =>
declare
N_Bare_Match_Expr : constant Bare_Match_Expr := N_Bare_Expr;
begin
Assign (N_Bare_Match_Expr,        N_Bare_Match_Expr.Match_Expr_F_Expected_Branch_Type_Var,        "expected_branch_type_var");
end;
when Lkt_Try_Expr_Range =>
declare
N_Bare_Try_Expr : constant Bare_Try_Expr := N_Bare_Expr;
begin
Assign (N_Bare_Try_Expr,        N_Bare_Try_Expr.Try_Expr_F_Expected_Expr_Type_Var,        "expected_expr_type_var");
end;
when others => null;
end case;
end;
when Lkt_Type_Ref =>
declare
N_Bare_Type_Ref : constant Bare_Type_Ref := Node;
begin
Assign (N_Bare_Type_Ref,        N_Bare_Type_Ref.Type_Ref_F_Type_Var,        "type_var");
end;
when others => null;
end case;
      for Child of Internal_Bare_Lkt_Node_Array'(Children (Node)) loop
         if Child /= null then
            Assign_Names_To_Logic_Vars (Child);
         end if;
      end loop;
   end Assign_Names_To_Logic_Vars;

   --------------------------
   -- Initialization_Error --
   --------------------------

   function Initialization_Error
     (Exc : Ada.Exceptions.Exception_Occurrence)
      return Error_Initialization_State
   is
      use Ada.Exceptions;
   begin
      if
         
            Exception_Identity (Exc) = Property_Error'Identity
         then
            return Raised_Property_Error;
      else
         raise Program_Error;
      end if;
   end Initialization_Error;

   ----------------------------------
   -- Reraise_Initialization_Error --
   ----------------------------------

   procedure Reraise_Initialization_Error
     (Node    : Bare_Lkt_Node;
      State   : Error_Initialization_State;
      Message : String)
   is
      Exc : Ada.Exceptions.Exception_Id;
   begin
      case State is
            when Raised_Property_Error =>
               Exc := Property_Error'Identity;
      end case;
      Raise_Property_Exception (Node, Exc, Message);
   end Reraise_Initialization_Error;

   ----------------
   -- Text_Image --
   ----------------

   function Text_Image (Ent : Internal_Entity) return Text_Type is
   begin
      if Ent.Node /= null then
         declare
            Node_Image : constant Text_Type := Short_Text_Image (Ent.Node);
         begin
            return
            (if Ent.Info.Rebindings /= null
             then "<| "
             & Node_Image (Node_Image'First + 1 .. Node_Image'Last - 1) & " "
             & AST_Envs.Text_Image (Ent.Info.Rebindings) & " |>"
             else Node_Image);
         end;
      else
         return "None";
      end if;
   end Text_Image;

   ---------------------
   -- Full_Sloc_Image --
   ---------------------

   function Full_Sloc_Image (Node : Bare_Lkt_Node) return String_Type
   is
      Res      : constant Text_Type :=
        To_Text
          (Ada.Directories.Simple_Name
             (Get_Filename (Unit (Node))))
           & ":" & To_Text (Image (Start_Sloc (Sloc_Range (Node)))) & ": ";
   begin
      return Create_String (Res);
   end Full_Sloc_Image;

   ---------------------------------
   -- Completion_Item_Kind_To_Int --
   ---------------------------------

   function Completion_Item_Kind_To_Int
     (Node : Bare_Lkt_Node;
      Kind : Completion_Item_Kind)
      return Integer
   is
      pragma Unreferenced (Node);
   begin
      return Completion_Item_Kind'Enum_Rep (Kind) + 1;
   end Completion_Item_Kind_To_Int;

   -----------
   -- Image --
   -----------

   function Image (Ent : Internal_Entity) return String is
      Result : constant Text_Type := Text_Image (Ent);
   begin
      return Image (Result);
   end Image;

   ---------------
   -- Can_Reach --
   ---------------

   function Can_Reach (El, From : Bare_Lkt_Node) return Boolean is
   begin
      return Lkt_Node_P_Can_Reach (El, From);
   end Can_Reach;

   -----------------
   -- Hash_Entity --
   -----------------

   function Hash_Entity (Self : Internal_Entity) return Hash_Type is
   begin
      return Combine
        ((Hash (Self.Node), Hash (Self.Info.Rebindings), Hash (Self.Info.Md)));
   end Hash_Entity;

   --------------------
   -- Compare_Entity --
   --------------------

   function Compare_Entity (Left, Right : Internal_Entity) return Boolean
   is
   begin
      return Left.Node = Right.Node
             and then Left.Info.Rebindings = Right.Info.Rebindings
             and then Compare_Metadata (Left.Info.Md, Right.Info.Md);
   end Compare_Entity;

   --------------------------------
   -- Create_Dynamic_Lexical_Env --
   --------------------------------

   function Create_Dynamic_Lexical_Env
     (Self              : Bare_Lkt_Node;
      Assocs_Getter     : Inner_Env_Assocs_Resolver;
      Assoc_Resolver    : Entity_Resolver;
      Transitive_Parent : Boolean;
      Sym_Table         : Symbol_Table) return Lexical_Env
   is
      Unit : constant Internal_Unit := Self.Unit;
   begin
      --  This restriction is necessary to avoid relocation issues when
      --  Self.Self_Env is terminated.
      if Is_Foreign_Strict (Self.Self_Env, Self) then
         Raise_Property_Exception
           (Self,
            Property_Error'Identity,
            "cannot create a dynamic lexical env when Self.Self_Env is"
            & " foreign");
      end if;

      return Result : constant Lexical_Env := Create_Dynamic_Lexical_Env
        (Parent            => Null_Lexical_Env,
         Node              => Self,
         Transitive_Parent => Transitive_Parent,
         Owner             => Convert_Unit (Unit),
         Assocs_Getter     => Assocs_Getter,
         Assoc_Resolver    => Assoc_Resolver,
         Sym_Table         => Sym_Table)
      do
         --  Since dynamic lexical environments can only be created in lazy
         --  field initializers, it is fine to tie Result's lifetime to the
         --  its owning unit's lifetime.
         Register_Destroyable (Unit, Unwrap (Result));
      end return;
   end Create_Dynamic_Lexical_Env;

   procedure Destroy_Synthetic_Node (Node : in out Bare_Lkt_Node);
   --  Helper for the Register_Destroyable above

   ------------
   -- Length --
   ------------

   function Length (Node : Bare_Lkt_Node_Base_List) return Natural
   is (if Node = null then 0 else Children_Count (Node));


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (B : Boolean) return String is
      begin
         return (if B then "True" else "False");
      end Trace_Image;

      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (I : Integer) return String is
      begin
         return Integer'Image (I);
      end Trace_Image;

      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (S : Symbol_Type) return String is
      begin
         return (if S = No_Symbol
                 then "None"
                 else Image (S, With_Quotes => True));
      end Trace_Image;

      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (C : Character_Type) return String is
         C_Str : constant Text_Type := (1 => C);
      begin
         return "'" & Image (C_Str) & "'";
      end Trace_Image;

      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (S : String_Type) return String is
      begin
         return Image (S.Content, With_Quotes => True);
      end Trace_Image;

      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (Env : Lexical_Env) return String is
      begin
         case Env.Kind is
         when Static_Primary =>
            return "<LexicalEnv static-primary for "
                   & Trace_Image (Env_Node (Env)) & ">";
         when others =>
            return "<LexicalEnv synthetic>";
         end case;
      end Trace_Image;

      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (R : Env_Rebindings) return String is
      begin
         return Image (Text_Image (R));
      end Trace_Image;

      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (Unit : Internal_Unit) return String is
      begin
         return "Internal_Unit (""" & Basename (Unit) & """)";
      end Trace_Image;

      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (Eq : Logic_Equation) return String is
         pragma Unreferenced (Eq);
      begin
         return "<LogicEquation>";
      end Trace_Image;

      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (Var : Logic_Var) return String is
         pragma Unreferenced (Var);
      begin
         return "<LogicVariable>";
      end Trace_Image;

      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (K : Analysis_Unit_Kind) return String is
      begin
         return Analysis_Unit_Kind'Image (K);
      end Trace_Image;

      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (Self : Ref_Categories) return String is
         Result : Unbounded_String;
         First  : Boolean := True;
      begin
         Append (Result, "RefCategories(");
         for C in Ref_Category loop
            if Self (C) then
               if First then
                  First := False;
               else
                  Append (Result, ", ");
               end if;
               Append (Result, C'Image);
            end if;
         end loop;
         Append (Result, ")");
         return To_String (Result);
      end Trace_Image;


   

   




   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Metadata) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                  & "null record"
               & ")");
      end Trace_Image;


   

   




   

      ----------
      -- Hash --
      ----------

      pragma Warnings (Off, "referenced");
      function Hash (R : Internal_Entity_Info) return Hash_Type is
         pragma Warnings (On, "referenced");
      begin
         
            return Combine ((Hash (R.Md), Hash (R.Rebindings), Hash (R.From_Rebound)));
      end Hash;


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Info) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Md => "
                     & Trace_Image (R.Md)
                        & ", "
                     & "Rebindings => "
                     & Trace_Image (R.Rebindings)
                        & ", "
                     & "From_Rebound => "
                     & Trace_Image (R.From_Rebound)
               & ")");
      end Trace_Image;


   

   



      function Create_Internal_Entity
        (Node : Bare_Lkt_Node; Info : Internal_Entity_Info)
         return Internal_Entity is
      begin
         if Node = null then
            return No_Entity;
         end if;
         return (Node => Node, Info => Info);
      end;



   

      ----------
      -- Hash --
      ----------

      pragma Warnings (Off, "referenced");
      function Hash (R : Internal_Entity) return Hash_Type is
         pragma Warnings (On, "referenced");
      begin
         
            return Combine
              (Hash (R.Node), Hash (R.Info));
      end Hash;


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Decl
        (Node : Bare_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Decl is
      begin
         if Node = null then
            return No_Entity_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   

      ----------
      -- Hash --
      ----------

      pragma Warnings (Off, "referenced");
      function Hash (R : Internal_Entity_Decl) return Hash_Type is
         pragma Warnings (On, "referenced");
      begin
         
            return Combine
              (Hash (R.Node), Hash (R.Info));
      end Hash;


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   




   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Complete_Item) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Declaration => "
                     & Trace_Image (R.Declaration)
               & ")");
      end Trace_Image;


   

   


      -------------
      -- Inc_Ref --
      -------------

      procedure Inc_Ref (R : Internal_Decoded_Char_Value) is
      begin
               Inc_Ref (R.Error_Message);
      end Inc_Ref;

      -------------
      -- Dec_Ref --
      -------------

      procedure Dec_Ref (R : in out Internal_Decoded_Char_Value) is
      begin
               Dec_Ref (R.Error_Message);
      end Dec_Ref;




      ----------------
      -- Equivalent --
      ----------------

      function Equivalent (L, R : Internal_Decoded_Char_Value) return Boolean is
      begin
         return L.Value = R.Value and then L.Has_Error = R.Has_Error and then L.Error_Sloc = R.Error_Sloc and then Equivalent (L.Error_Message, R.Error_Message);
      end Equivalent;


   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Decoded_Char_Value) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Value => "
                     & Trace_Image (R.Value)
                        & ", "
                     & "Has_Error => "
                     & Trace_Image (R.Has_Error)
                        & ", "
                     & "Error_Sloc => "
                     & Trace_Image (R.Error_Sloc)
                        & ", "
                     & "Error_Message => "
                     & Trace_Image (R.Error_Message)
               & ")");
      end Trace_Image;


   

   


      -------------
      -- Inc_Ref --
      -------------

      procedure Inc_Ref (R : Internal_Decoded_String_Value) is
      begin
               Inc_Ref (R.Value);
               Inc_Ref (R.Error_Message);
      end Inc_Ref;

      -------------
      -- Dec_Ref --
      -------------

      procedure Dec_Ref (R : in out Internal_Decoded_String_Value) is
      begin
               Dec_Ref (R.Value);
               Dec_Ref (R.Error_Message);
      end Dec_Ref;




      ----------------
      -- Equivalent --
      ----------------

      function Equivalent (L, R : Internal_Decoded_String_Value) return Boolean is
      begin
         return Equivalent (L.Value, R.Value) and then L.Has_Error = R.Has_Error and then L.Error_Sloc = R.Error_Sloc and then Equivalent (L.Error_Message, R.Error_Message);
      end Equivalent;


   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Decoded_String_Value) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Value => "
                     & Trace_Image (R.Value)
                        & ", "
                     & "Has_Error => "
                     & Trace_Image (R.Has_Error)
                        & ", "
                     & "Error_Sloc => "
                     & Trace_Image (R.Error_Sloc)
                        & ", "
                     & "Error_Message => "
                     & Trace_Image (R.Error_Message)
               & ")");
      end Trace_Image;


   

   


      -------------
      -- Inc_Ref --
      -------------

      procedure Inc_Ref (R : Internal_Designated_Env) is
      begin
               Inc_Ref (R.Direct_Env);
      end Inc_Ref;

      -------------
      -- Dec_Ref --
      -------------

      procedure Dec_Ref (R : in out Internal_Designated_Env) is
      begin
               Dec_Ref (R.Direct_Env);
      end Dec_Ref;




      ----------------
      -- Equivalent --
      ----------------

      function Equivalent (L, R : Internal_Designated_Env) return Boolean is
      begin
         return L.Kind = R.Kind and then L.Env_Name = R.Env_Name and then Equivalent (L.Direct_Env, R.Direct_Env);
      end Equivalent;


   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Designated_Env) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Kind => "
                     & Trace_Image (R.Kind)
                        & ", "
                     & "Env_Name => "
                     & Trace_Image (R.Env_Name)
                        & ", "
                     & "Direct_Env => "
                     & Trace_Image (R.Direct_Env)
               & ")");
      end Trace_Image;


   

   



      function Create_Internal_Entity_Expr
        (Node : Bare_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Expr is
      begin
         if Node = null then
            return No_Entity_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Any_Of
        (Node : Bare_Any_Of; Info : Internal_Entity_Info)
         return Internal_Entity_Any_Of is
      begin
         if Node = null then
            return No_Entity_Any_Of;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Any_Of) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Lkt_Node_Base_List
        (Node : Bare_Lkt_Node_Base_List; Info : Internal_Entity_Info)
         return Internal_Entity_Lkt_Node_Base_List is
      begin
         if Node = null then
            return No_Entity_Lkt_Node_Base_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Lkt_Node_Base_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Expr_List
        (Node : Bare_Expr_List; Info : Internal_Entity_Info)
         return Internal_Entity_Expr_List is
      begin
         if Node = null then
            return No_Entity_Expr_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Expr_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Any_Of_List
        (Node : Bare_Any_Of_List; Info : Internal_Entity_Info)
         return Internal_Entity_Any_Of_List is
      begin
         if Node = null then
            return No_Entity_Any_Of_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Any_Of_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Type_Decl
        (Node : Bare_Type_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Type_Decl is
      begin
         if Node = null then
            return No_Entity_Type_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   

      ----------
      -- Hash --
      ----------

      pragma Warnings (Off, "referenced");
      function Hash (R : Internal_Entity_Type_Decl) return Hash_Type is
         pragma Warnings (On, "referenced");
      begin
         
            return Combine
              (Hash (R.Node), Hash (R.Info));
      end Hash;


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Type_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Any_Type_Decl
        (Node : Bare_Any_Type_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Any_Type_Decl is
      begin
         if Node = null then
            return No_Entity_Any_Type_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Any_Type_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Pattern
        (Node : Bare_Pattern; Info : Internal_Entity_Info)
         return Internal_Entity_Pattern is
      begin
         if Node = null then
            return No_Entity_Pattern;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Pattern) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Any_Type_Pattern
        (Node : Bare_Any_Type_Pattern; Info : Internal_Entity_Info)
         return Internal_Entity_Any_Type_Pattern is
      begin
         if Node = null then
            return No_Entity_Any_Type_Pattern;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Any_Type_Pattern) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Argument
        (Node : Bare_Argument; Info : Internal_Entity_Info)
         return Internal_Entity_Argument is
      begin
         if Node = null then
            return No_Entity_Argument;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Argument) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Argument_List
        (Node : Bare_Argument_List; Info : Internal_Entity_Info)
         return Internal_Entity_Argument_List is
      begin
         if Node = null then
            return No_Entity_Argument_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   

      ----------
      -- Hash --
      ----------

      pragma Warnings (Off, "referenced");
      function Hash (R : Internal_Entity_Argument_List) return Hash_Type is
         pragma Warnings (On, "referenced");
      begin
         
            return Combine
              (Hash (R.Node), Hash (R.Info));
      end Hash;


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Argument_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Array_Literal
        (Node : Bare_Array_Literal; Info : Internal_Entity_Info)
         return Internal_Entity_Array_Literal is
      begin
         if Node = null then
            return No_Entity_Array_Literal;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Array_Literal) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Base_Call_Expr
        (Node : Bare_Base_Call_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Base_Call_Expr is
      begin
         if Node = null then
            return No_Entity_Base_Call_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Base_Call_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Base_Grammar_Rule_Decl
        (Node : Bare_Base_Grammar_Rule_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Base_Grammar_Rule_Decl is
      begin
         if Node = null then
            return No_Entity_Base_Grammar_Rule_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Base_Grammar_Rule_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Base_Lexer_Case_Rule_Alt
        (Node : Bare_Base_Lexer_Case_Rule_Alt; Info : Internal_Entity_Info)
         return Internal_Entity_Base_Lexer_Case_Rule_Alt is
      begin
         if Node = null then
            return No_Entity_Base_Lexer_Case_Rule_Alt;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Base_Lexer_Case_Rule_Alt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Base_Lexer_Case_Rule_Alt_List
        (Node : Bare_Base_Lexer_Case_Rule_Alt_List; Info : Internal_Entity_Info)
         return Internal_Entity_Base_Lexer_Case_Rule_Alt_List is
      begin
         if Node = null then
            return No_Entity_Base_Lexer_Case_Rule_Alt_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Base_Lexer_Case_Rule_Alt_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Base_Match_Branch
        (Node : Bare_Base_Match_Branch; Info : Internal_Entity_Info)
         return Internal_Entity_Base_Match_Branch is
      begin
         if Node = null then
            return No_Entity_Base_Match_Branch;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Base_Match_Branch) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Base_Match_Branch_List
        (Node : Bare_Base_Match_Branch_List; Info : Internal_Entity_Info)
         return Internal_Entity_Base_Match_Branch_List is
      begin
         if Node = null then
            return No_Entity_Base_Match_Branch_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Base_Match_Branch_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Base_Val_Decl
        (Node : Bare_Base_Val_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Base_Val_Decl is
      begin
         if Node = null then
            return No_Entity_Base_Val_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Base_Val_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Named_Type_Decl
        (Node : Bare_Named_Type_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Named_Type_Decl is
      begin
         if Node = null then
            return No_Entity_Named_Type_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Named_Type_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Basic_Class_Decl
        (Node : Bare_Basic_Class_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Basic_Class_Decl is
      begin
         if Node = null then
            return No_Entity_Basic_Class_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Basic_Class_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Lit
        (Node : Bare_Lit; Info : Internal_Entity_Info)
         return Internal_Entity_Lit is
      begin
         if Node = null then
            return No_Entity_Lit;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Lit) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Big_Num_Lit
        (Node : Bare_Big_Num_Lit; Info : Internal_Entity_Info)
         return Internal_Entity_Big_Num_Lit is
      begin
         if Node = null then
            return No_Entity_Big_Num_Lit;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Big_Num_Lit) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Bin_Op
        (Node : Bare_Bin_Op; Info : Internal_Entity_Info)
         return Internal_Entity_Bin_Op is
      begin
         if Node = null then
            return No_Entity_Bin_Op;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Bin_Op) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Binding_Pattern
        (Node : Bare_Binding_Pattern; Info : Internal_Entity_Info)
         return Internal_Entity_Binding_Pattern is
      begin
         if Node = null then
            return No_Entity_Binding_Pattern;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Binding_Pattern) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_User_Val_Decl
        (Node : Bare_User_Val_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_User_Val_Decl is
      begin
         if Node = null then
            return No_Entity_User_Val_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_User_Val_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Binding_Val_Decl
        (Node : Bare_Binding_Val_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Binding_Val_Decl is
      begin
         if Node = null then
            return No_Entity_Binding_Val_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Binding_Val_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Block_Expr
        (Node : Bare_Block_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Block_Expr is
      begin
         if Node = null then
            return No_Entity_Block_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Block_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Block_Expr_Clause
        (Node : Bare_Block_Expr_Clause; Info : Internal_Entity_Info)
         return Internal_Entity_Block_Expr_Clause is
      begin
         if Node = null then
            return No_Entity_Block_Expr_Clause;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Block_Expr_Clause) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Block_String_Line
        (Node : Bare_Block_String_Line; Info : Internal_Entity_Info)
         return Internal_Entity_Block_String_Line is
      begin
         if Node = null then
            return No_Entity_Block_String_Line;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Block_String_Line) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Block_String_Line_List
        (Node : Bare_Block_String_Line_List; Info : Internal_Entity_Info)
         return Internal_Entity_Block_String_Line_List is
      begin
         if Node = null then
            return No_Entity_Block_String_Line_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Block_String_Line_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_String_Lit
        (Node : Bare_String_Lit; Info : Internal_Entity_Info)
         return Internal_Entity_String_Lit is
      begin
         if Node = null then
            return No_Entity_String_Lit;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_String_Lit) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Block_String_Lit
        (Node : Bare_Block_String_Lit; Info : Internal_Entity_Info)
         return Internal_Entity_Block_String_Lit is
      begin
         if Node = null then
            return No_Entity_Block_String_Lit;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Block_String_Lit) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Bool_Pattern
        (Node : Bare_Bool_Pattern; Info : Internal_Entity_Info)
         return Internal_Entity_Bool_Pattern is
      begin
         if Node = null then
            return No_Entity_Bool_Pattern;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Bool_Pattern) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Bool_Pattern_False
        (Node : Bare_Bool_Pattern_False; Info : Internal_Entity_Info)
         return Internal_Entity_Bool_Pattern_False is
      begin
         if Node = null then
            return No_Entity_Bool_Pattern_False;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Bool_Pattern_False) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Bool_Pattern_True
        (Node : Bare_Bool_Pattern_True; Info : Internal_Entity_Info)
         return Internal_Entity_Bool_Pattern_True is
      begin
         if Node = null then
            return No_Entity_Bool_Pattern_True;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Bool_Pattern_True) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Call_Expr
        (Node : Bare_Call_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Call_Expr is
      begin
         if Node = null then
            return No_Entity_Call_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Call_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Call_Expr_List
        (Node : Bare_Call_Expr_List; Info : Internal_Entity_Info)
         return Internal_Entity_Call_Expr_List is
      begin
         if Node = null then
            return No_Entity_Call_Expr_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Call_Expr_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Cast_Expr
        (Node : Bare_Cast_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Cast_Expr is
      begin
         if Node = null then
            return No_Entity_Cast_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Cast_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Char_Lit
        (Node : Bare_Char_Lit; Info : Internal_Entity_Info)
         return Internal_Entity_Char_Lit is
      begin
         if Node = null then
            return No_Entity_Char_Lit;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Char_Lit) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Class_Decl
        (Node : Bare_Class_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Class_Decl is
      begin
         if Node = null then
            return No_Entity_Class_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Class_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Class_Qualifier
        (Node : Bare_Class_Qualifier; Info : Internal_Entity_Info)
         return Internal_Entity_Class_Qualifier is
      begin
         if Node = null then
            return No_Entity_Class_Qualifier;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Class_Qualifier) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Class_Qualifier_Absent
        (Node : Bare_Class_Qualifier_Absent; Info : Internal_Entity_Info)
         return Internal_Entity_Class_Qualifier_Absent is
      begin
         if Node = null then
            return No_Entity_Class_Qualifier_Absent;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Class_Qualifier_Absent) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Class_Qualifier_Present
        (Node : Bare_Class_Qualifier_Present; Info : Internal_Entity_Info)
         return Internal_Entity_Class_Qualifier_Present is
      begin
         if Node = null then
            return No_Entity_Class_Qualifier_Present;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Class_Qualifier_Present) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Explicitly_Typed_Decl
        (Node : Bare_Explicitly_Typed_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Explicitly_Typed_Decl is
      begin
         if Node = null then
            return No_Entity_Explicitly_Typed_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Explicitly_Typed_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Component_Decl
        (Node : Bare_Component_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Component_Decl is
      begin
         if Node = null then
            return No_Entity_Component_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Component_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Decl_Annotation
        (Node : Bare_Decl_Annotation; Info : Internal_Entity_Info)
         return Internal_Entity_Decl_Annotation is
      begin
         if Node = null then
            return No_Entity_Decl_Annotation;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Decl_Annotation) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Decl_Annotation_Args
        (Node : Bare_Decl_Annotation_Args; Info : Internal_Entity_Info)
         return Internal_Entity_Decl_Annotation_Args is
      begin
         if Node = null then
            return No_Entity_Decl_Annotation_Args;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Decl_Annotation_Args) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Decl_Annotation_List
        (Node : Bare_Decl_Annotation_List; Info : Internal_Entity_Info)
         return Internal_Entity_Decl_Annotation_List is
      begin
         if Node = null then
            return No_Entity_Decl_Annotation_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Decl_Annotation_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Full_Decl_List
        (Node : Bare_Full_Decl_List; Info : Internal_Entity_Info)
         return Internal_Entity_Full_Decl_List is
      begin
         if Node = null then
            return No_Entity_Full_Decl_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Full_Decl_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Decl_Block
        (Node : Bare_Decl_Block; Info : Internal_Entity_Info)
         return Internal_Entity_Decl_Block is
      begin
         if Node = null then
            return No_Entity_Decl_Block;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Decl_Block) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Id
        (Node : Bare_Id; Info : Internal_Entity_Info)
         return Internal_Entity_Id is
      begin
         if Node = null then
            return No_Entity_Id;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Id) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Def_Id
        (Node : Bare_Def_Id; Info : Internal_Entity_Info)
         return Internal_Entity_Def_Id is
      begin
         if Node = null then
            return No_Entity_Def_Id;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Def_Id) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Type_Ref
        (Node : Bare_Type_Ref; Info : Internal_Entity_Info)
         return Internal_Entity_Type_Ref is
      begin
         if Node = null then
            return No_Entity_Type_Ref;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Type_Ref) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Default_List_Type_Ref
        (Node : Bare_Default_List_Type_Ref; Info : Internal_Entity_Info)
         return Internal_Entity_Default_List_Type_Ref is
      begin
         if Node = null then
            return No_Entity_Default_List_Type_Ref;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Default_List_Type_Ref) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Dot_Expr
        (Node : Bare_Dot_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Dot_Expr is
      begin
         if Node = null then
            return No_Entity_Dot_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Dot_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Dyn_Env_Wrapper
        (Node : Bare_Dyn_Env_Wrapper; Info : Internal_Entity_Info)
         return Internal_Entity_Dyn_Env_Wrapper is
      begin
         if Node = null then
            return No_Entity_Dyn_Env_Wrapper;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Dyn_Env_Wrapper) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Dyn_Var_Decl
        (Node : Bare_Dyn_Var_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Dyn_Var_Decl is
      begin
         if Node = null then
            return No_Entity_Dyn_Var_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Dyn_Var_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Ellipsis_Pattern
        (Node : Bare_Ellipsis_Pattern; Info : Internal_Entity_Info)
         return Internal_Entity_Ellipsis_Pattern is
      begin
         if Node = null then
            return No_Entity_Ellipsis_Pattern;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Ellipsis_Pattern) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Elsif_Branch
        (Node : Bare_Elsif_Branch; Info : Internal_Entity_Info)
         return Internal_Entity_Elsif_Branch is
      begin
         if Node = null then
            return No_Entity_Elsif_Branch;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Elsif_Branch) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Elsif_Branch_List
        (Node : Bare_Elsif_Branch_List; Info : Internal_Entity_Info)
         return Internal_Entity_Elsif_Branch_List is
      begin
         if Node = null then
            return No_Entity_Elsif_Branch_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Elsif_Branch_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Enum_Class_Alt_Decl
        (Node : Bare_Enum_Class_Alt_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Enum_Class_Alt_Decl is
      begin
         if Node = null then
            return No_Entity_Enum_Class_Alt_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Enum_Class_Alt_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Enum_Class_Alt_Decl_List
        (Node : Bare_Enum_Class_Alt_Decl_List; Info : Internal_Entity_Info)
         return Internal_Entity_Enum_Class_Alt_Decl_List is
      begin
         if Node = null then
            return No_Entity_Enum_Class_Alt_Decl_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Enum_Class_Alt_Decl_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Enum_Class_Case
        (Node : Bare_Enum_Class_Case; Info : Internal_Entity_Info)
         return Internal_Entity_Enum_Class_Case is
      begin
         if Node = null then
            return No_Entity_Enum_Class_Case;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Enum_Class_Case) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Enum_Class_Case_List
        (Node : Bare_Enum_Class_Case_List; Info : Internal_Entity_Info)
         return Internal_Entity_Enum_Class_Case_List is
      begin
         if Node = null then
            return No_Entity_Enum_Class_Case_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Enum_Class_Case_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Enum_Class_Decl
        (Node : Bare_Enum_Class_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Enum_Class_Decl is
      begin
         if Node = null then
            return No_Entity_Enum_Class_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Enum_Class_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Enum_Lit_Decl
        (Node : Bare_Enum_Lit_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Enum_Lit_Decl is
      begin
         if Node = null then
            return No_Entity_Enum_Lit_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Enum_Lit_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Enum_Lit_Decl_List
        (Node : Bare_Enum_Lit_Decl_List; Info : Internal_Entity_Info)
         return Internal_Entity_Enum_Lit_Decl_List is
      begin
         if Node = null then
            return No_Entity_Enum_Lit_Decl_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Enum_Lit_Decl_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Enum_Type_Decl
        (Node : Bare_Enum_Type_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Enum_Type_Decl is
      begin
         if Node = null then
            return No_Entity_Enum_Type_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Enum_Type_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Env_Spec_Decl
        (Node : Bare_Env_Spec_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Env_Spec_Decl is
      begin
         if Node = null then
            return No_Entity_Env_Spec_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Env_Spec_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Error_Decl
        (Node : Bare_Error_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Error_Decl is
      begin
         if Node = null then
            return No_Entity_Error_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Error_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Error_On_Null
        (Node : Bare_Error_On_Null; Info : Internal_Entity_Info)
         return Internal_Entity_Error_On_Null is
      begin
         if Node = null then
            return No_Entity_Error_On_Null;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Error_On_Null) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Excludes_Null
        (Node : Bare_Excludes_Null; Info : Internal_Entity_Info)
         return Internal_Entity_Excludes_Null is
      begin
         if Node = null then
            return No_Entity_Excludes_Null;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Excludes_Null) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Excludes_Null_Absent
        (Node : Bare_Excludes_Null_Absent; Info : Internal_Entity_Info)
         return Internal_Entity_Excludes_Null_Absent is
      begin
         if Node = null then
            return No_Entity_Excludes_Null_Absent;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Excludes_Null_Absent) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Excludes_Null_Present
        (Node : Bare_Excludes_Null_Present; Info : Internal_Entity_Info)
         return Internal_Entity_Excludes_Null_Present is
      begin
         if Node = null then
            return No_Entity_Excludes_Null_Present;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Excludes_Null_Present) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Extended_Pattern
        (Node : Bare_Extended_Pattern; Info : Internal_Entity_Info)
         return Internal_Entity_Extended_Pattern is
      begin
         if Node = null then
            return No_Entity_Extended_Pattern;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Extended_Pattern) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Field_Decl
        (Node : Bare_Field_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Field_Decl is
      begin
         if Node = null then
            return No_Entity_Field_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Field_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Pattern_Detail
        (Node : Bare_Pattern_Detail; Info : Internal_Entity_Info)
         return Internal_Entity_Pattern_Detail is
      begin
         if Node = null then
            return No_Entity_Pattern_Detail;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Pattern_Detail) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Field_Pattern_Detail
        (Node : Bare_Field_Pattern_Detail; Info : Internal_Entity_Info)
         return Internal_Entity_Field_Pattern_Detail is
      begin
         if Node = null then
            return No_Entity_Field_Pattern_Detail;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Field_Pattern_Detail) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Filtered_Pattern
        (Node : Bare_Filtered_Pattern; Info : Internal_Entity_Info)
         return Internal_Entity_Filtered_Pattern is
      begin
         if Node = null then
            return No_Entity_Filtered_Pattern;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Filtered_Pattern) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Full_Decl
        (Node : Bare_Full_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Full_Decl is
      begin
         if Node = null then
            return No_Entity_Full_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Full_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Fun_Decl
        (Node : Bare_Fun_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Fun_Decl is
      begin
         if Node = null then
            return No_Entity_Fun_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Fun_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Fun_Param_Decl
        (Node : Bare_Fun_Param_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Fun_Param_Decl is
      begin
         if Node = null then
            return No_Entity_Fun_Param_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Fun_Param_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Fun_Param_Decl_List
        (Node : Bare_Fun_Param_Decl_List; Info : Internal_Entity_Info)
         return Internal_Entity_Fun_Param_Decl_List is
      begin
         if Node = null then
            return No_Entity_Fun_Param_Decl_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Fun_Param_Decl_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Function_Type
        (Node : Bare_Function_Type; Info : Internal_Entity_Info)
         return Internal_Entity_Function_Type is
      begin
         if Node = null then
            return No_Entity_Function_Type;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Function_Type) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Function_Type_Ref
        (Node : Bare_Function_Type_Ref; Info : Internal_Entity_Info)
         return Internal_Entity_Function_Type_Ref is
      begin
         if Node = null then
            return No_Entity_Function_Type_Ref;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Function_Type_Ref) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Generic_Decl
        (Node : Bare_Generic_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Generic_Decl is
      begin
         if Node = null then
            return No_Entity_Generic_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Generic_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Generic_Instantiation
        (Node : Bare_Generic_Instantiation; Info : Internal_Entity_Info)
         return Internal_Entity_Generic_Instantiation is
      begin
         if Node = null then
            return No_Entity_Generic_Instantiation;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Generic_Instantiation) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Generic_Param_Decl_List
        (Node : Bare_Generic_Param_Decl_List; Info : Internal_Entity_Info)
         return Internal_Entity_Generic_Param_Decl_List is
      begin
         if Node = null then
            return No_Entity_Generic_Param_Decl_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Generic_Param_Decl_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Generic_Param_Type_Decl
        (Node : Bare_Generic_Param_Type_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Generic_Param_Type_Decl is
      begin
         if Node = null then
            return No_Entity_Generic_Param_Type_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Generic_Param_Type_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Generic_Type_Ref
        (Node : Bare_Generic_Type_Ref; Info : Internal_Entity_Info)
         return Internal_Entity_Generic_Type_Ref is
      begin
         if Node = null then
            return No_Entity_Generic_Type_Ref;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Generic_Type_Ref) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Grammar_Expr
        (Node : Bare_Grammar_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Grammar_Expr is
      begin
         if Node = null then
            return No_Entity_Grammar_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Grammar_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Grammar_Cut
        (Node : Bare_Grammar_Cut; Info : Internal_Entity_Info)
         return Internal_Entity_Grammar_Cut is
      begin
         if Node = null then
            return No_Entity_Grammar_Cut;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Grammar_Cut) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Grammar_Decl
        (Node : Bare_Grammar_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Grammar_Decl is
      begin
         if Node = null then
            return No_Entity_Grammar_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Grammar_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Grammar_Discard
        (Node : Bare_Grammar_Discard; Info : Internal_Entity_Info)
         return Internal_Entity_Grammar_Discard is
      begin
         if Node = null then
            return No_Entity_Grammar_Discard;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Grammar_Discard) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Grammar_Dont_Skip
        (Node : Bare_Grammar_Dont_Skip; Info : Internal_Entity_Info)
         return Internal_Entity_Grammar_Dont_Skip is
      begin
         if Node = null then
            return No_Entity_Grammar_Dont_Skip;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Grammar_Dont_Skip) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Grammar_Expr_List
        (Node : Bare_Grammar_Expr_List; Info : Internal_Entity_Info)
         return Internal_Entity_Grammar_Expr_List is
      begin
         if Node = null then
            return No_Entity_Grammar_Expr_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Grammar_Expr_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Grammar_Expr_List_List
        (Node : Bare_Grammar_Expr_List_List; Info : Internal_Entity_Info)
         return Internal_Entity_Grammar_Expr_List_List is
      begin
         if Node = null then
            return No_Entity_Grammar_Expr_List_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Grammar_Expr_List_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Grammar_Pick
        (Node : Bare_Grammar_Pick; Info : Internal_Entity_Info)
         return Internal_Entity_Grammar_Pick is
      begin
         if Node = null then
            return No_Entity_Grammar_Pick;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Grammar_Pick) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Grammar_Implicit_Pick
        (Node : Bare_Grammar_Implicit_Pick; Info : Internal_Entity_Info)
         return Internal_Entity_Grammar_Implicit_Pick is
      begin
         if Node = null then
            return No_Entity_Grammar_Implicit_Pick;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Grammar_Implicit_Pick) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Grammar_List
        (Node : Bare_Grammar_List; Info : Internal_Entity_Info)
         return Internal_Entity_Grammar_List is
      begin
         if Node = null then
            return No_Entity_Grammar_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Grammar_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Grammar_List_Sep
        (Node : Bare_Grammar_List_Sep; Info : Internal_Entity_Info)
         return Internal_Entity_Grammar_List_Sep is
      begin
         if Node = null then
            return No_Entity_Grammar_List_Sep;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Grammar_List_Sep) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Grammar_Null
        (Node : Bare_Grammar_Null; Info : Internal_Entity_Info)
         return Internal_Entity_Grammar_Null is
      begin
         if Node = null then
            return No_Entity_Grammar_Null;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Grammar_Null) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Grammar_Opt
        (Node : Bare_Grammar_Opt; Info : Internal_Entity_Info)
         return Internal_Entity_Grammar_Opt is
      begin
         if Node = null then
            return No_Entity_Grammar_Opt;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Grammar_Opt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Grammar_Opt_Error
        (Node : Bare_Grammar_Opt_Error; Info : Internal_Entity_Info)
         return Internal_Entity_Grammar_Opt_Error is
      begin
         if Node = null then
            return No_Entity_Grammar_Opt_Error;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Grammar_Opt_Error) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Grammar_Opt_Error_Group
        (Node : Bare_Grammar_Opt_Error_Group; Info : Internal_Entity_Info)
         return Internal_Entity_Grammar_Opt_Error_Group is
      begin
         if Node = null then
            return No_Entity_Grammar_Opt_Error_Group;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Grammar_Opt_Error_Group) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Grammar_Opt_Group
        (Node : Bare_Grammar_Opt_Group; Info : Internal_Entity_Info)
         return Internal_Entity_Grammar_Opt_Group is
      begin
         if Node = null then
            return No_Entity_Grammar_Opt_Group;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Grammar_Opt_Group) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Grammar_Or_Expr
        (Node : Bare_Grammar_Or_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Grammar_Or_Expr is
      begin
         if Node = null then
            return No_Entity_Grammar_Or_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Grammar_Or_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Grammar_Predicate
        (Node : Bare_Grammar_Predicate; Info : Internal_Entity_Info)
         return Internal_Entity_Grammar_Predicate is
      begin
         if Node = null then
            return No_Entity_Grammar_Predicate;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Grammar_Predicate) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Grammar_Rule_Decl
        (Node : Bare_Grammar_Rule_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Grammar_Rule_Decl is
      begin
         if Node = null then
            return No_Entity_Grammar_Rule_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Grammar_Rule_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Grammar_Rule_Ref
        (Node : Bare_Grammar_Rule_Ref; Info : Internal_Entity_Info)
         return Internal_Entity_Grammar_Rule_Ref is
      begin
         if Node = null then
            return No_Entity_Grammar_Rule_Ref;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Grammar_Rule_Ref) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Grammar_Skip
        (Node : Bare_Grammar_Skip; Info : Internal_Entity_Info)
         return Internal_Entity_Grammar_Skip is
      begin
         if Node = null then
            return No_Entity_Grammar_Skip;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Grammar_Skip) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Grammar_Stop_Cut
        (Node : Bare_Grammar_Stop_Cut; Info : Internal_Entity_Info)
         return Internal_Entity_Grammar_Stop_Cut is
      begin
         if Node = null then
            return No_Entity_Grammar_Stop_Cut;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Grammar_Stop_Cut) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_If_Expr
        (Node : Bare_If_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_If_Expr is
      begin
         if Node = null then
            return No_Entity_If_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_If_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Import
        (Node : Bare_Import; Info : Internal_Entity_Info)
         return Internal_Entity_Import is
      begin
         if Node = null then
            return No_Entity_Import;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Import) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Import_List
        (Node : Bare_Import_List; Info : Internal_Entity_Info)
         return Internal_Entity_Import_List is
      begin
         if Node = null then
            return No_Entity_Import_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Import_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Integer_Pattern
        (Node : Bare_Integer_Pattern; Info : Internal_Entity_Info)
         return Internal_Entity_Integer_Pattern is
      begin
         if Node = null then
            return No_Entity_Integer_Pattern;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Integer_Pattern) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Isa
        (Node : Bare_Isa; Info : Internal_Entity_Info)
         return Internal_Entity_Isa is
      begin
         if Node = null then
            return No_Entity_Isa;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Isa) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Keep_Expr
        (Node : Bare_Keep_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Keep_Expr is
      begin
         if Node = null then
            return No_Entity_Keep_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Keep_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Lambda_Expr
        (Node : Bare_Lambda_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Lambda_Expr is
      begin
         if Node = null then
            return No_Entity_Lambda_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Lambda_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Lambda_Param_Decl
        (Node : Bare_Lambda_Param_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Lambda_Param_Decl is
      begin
         if Node = null then
            return No_Entity_Lambda_Param_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Lambda_Param_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Lambda_Param_Decl_List
        (Node : Bare_Lambda_Param_Decl_List; Info : Internal_Entity_Info)
         return Internal_Entity_Lambda_Param_Decl_List is
      begin
         if Node = null then
            return No_Entity_Lambda_Param_Decl_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Lambda_Param_Decl_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Langkit_Root
        (Node : Bare_Langkit_Root; Info : Internal_Entity_Info)
         return Internal_Entity_Langkit_Root is
      begin
         if Node = null then
            return No_Entity_Langkit_Root;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Langkit_Root) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Lexer_Case_Rule
        (Node : Bare_Lexer_Case_Rule; Info : Internal_Entity_Info)
         return Internal_Entity_Lexer_Case_Rule is
      begin
         if Node = null then
            return No_Entity_Lexer_Case_Rule;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Lexer_Case_Rule) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Lexer_Case_Rule_Cond_Alt
        (Node : Bare_Lexer_Case_Rule_Cond_Alt; Info : Internal_Entity_Info)
         return Internal_Entity_Lexer_Case_Rule_Cond_Alt is
      begin
         if Node = null then
            return No_Entity_Lexer_Case_Rule_Cond_Alt;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Lexer_Case_Rule_Cond_Alt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Lexer_Case_Rule_Default_Alt
        (Node : Bare_Lexer_Case_Rule_Default_Alt; Info : Internal_Entity_Info)
         return Internal_Entity_Lexer_Case_Rule_Default_Alt is
      begin
         if Node = null then
            return No_Entity_Lexer_Case_Rule_Default_Alt;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Lexer_Case_Rule_Default_Alt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Lexer_Case_Rule_Send
        (Node : Bare_Lexer_Case_Rule_Send; Info : Internal_Entity_Info)
         return Internal_Entity_Lexer_Case_Rule_Send is
      begin
         if Node = null then
            return No_Entity_Lexer_Case_Rule_Send;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Lexer_Case_Rule_Send) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Lexer_Decl
        (Node : Bare_Lexer_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Lexer_Decl is
      begin
         if Node = null then
            return No_Entity_Lexer_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Lexer_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Lexer_Family_Decl
        (Node : Bare_Lexer_Family_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Lexer_Family_Decl is
      begin
         if Node = null then
            return No_Entity_Lexer_Family_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Lexer_Family_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_List_Kind
        (Node : Bare_List_Kind; Info : Internal_Entity_Info)
         return Internal_Entity_List_Kind is
      begin
         if Node = null then
            return No_Entity_List_Kind;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_List_Kind) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_List_Kind_One
        (Node : Bare_List_Kind_One; Info : Internal_Entity_Info)
         return Internal_Entity_List_Kind_One is
      begin
         if Node = null then
            return No_Entity_List_Kind_One;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_List_Kind_One) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_List_Kind_Zero
        (Node : Bare_List_Kind_Zero; Info : Internal_Entity_Info)
         return Internal_Entity_List_Kind_Zero is
      begin
         if Node = null then
            return No_Entity_List_Kind_Zero;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_List_Kind_Zero) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_List_Pattern
        (Node : Bare_List_Pattern; Info : Internal_Entity_Info)
         return Internal_Entity_List_Pattern is
      begin
         if Node = null then
            return No_Entity_List_Pattern;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_List_Pattern) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Lkt_Node_List
        (Node : Bare_Lkt_Node_List; Info : Internal_Entity_Info)
         return Internal_Entity_Lkt_Node_List is
      begin
         if Node = null then
            return No_Entity_Lkt_Node_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Lkt_Node_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Logic_Assign
        (Node : Bare_Logic_Assign; Info : Internal_Entity_Info)
         return Internal_Entity_Logic_Assign is
      begin
         if Node = null then
            return No_Entity_Logic_Assign;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Logic_Assign) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Logic_Call_Expr
        (Node : Bare_Logic_Call_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Logic_Call_Expr is
      begin
         if Node = null then
            return No_Entity_Logic_Call_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Logic_Call_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Logic_Expr
        (Node : Bare_Logic_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Logic_Expr is
      begin
         if Node = null then
            return No_Entity_Logic_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Logic_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Logic_Predicate
        (Node : Bare_Logic_Predicate; Info : Internal_Entity_Info)
         return Internal_Entity_Logic_Predicate is
      begin
         if Node = null then
            return No_Entity_Logic_Predicate;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Logic_Predicate) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Logic_Propagate
        (Node : Bare_Logic_Propagate; Info : Internal_Entity_Info)
         return Internal_Entity_Logic_Propagate is
      begin
         if Node = null then
            return No_Entity_Logic_Propagate;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Logic_Propagate) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Logic_Propagate_Call
        (Node : Bare_Logic_Propagate_Call; Info : Internal_Entity_Info)
         return Internal_Entity_Logic_Propagate_Call is
      begin
         if Node = null then
            return No_Entity_Logic_Propagate_Call;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Logic_Propagate_Call) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Logic_Unify
        (Node : Bare_Logic_Unify; Info : Internal_Entity_Info)
         return Internal_Entity_Logic_Unify is
      begin
         if Node = null then
            return No_Entity_Logic_Unify;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Logic_Unify) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Match_Branch
        (Node : Bare_Match_Branch; Info : Internal_Entity_Info)
         return Internal_Entity_Match_Branch is
      begin
         if Node = null then
            return No_Entity_Match_Branch;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Match_Branch) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Match_Expr
        (Node : Bare_Match_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Match_Expr is
      begin
         if Node = null then
            return No_Entity_Match_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Match_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Match_Val_Decl
        (Node : Bare_Match_Val_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Match_Val_Decl is
      begin
         if Node = null then
            return No_Entity_Match_Val_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Match_Val_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Module_Ref_Id
        (Node : Bare_Module_Ref_Id; Info : Internal_Entity_Info)
         return Internal_Entity_Module_Ref_Id is
      begin
         if Node = null then
            return No_Entity_Module_Ref_Id;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Module_Ref_Id) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Node_Decl
        (Node : Bare_Node_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Node_Decl is
      begin
         if Node = null then
            return No_Entity_Node_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Node_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Not_Expr
        (Node : Bare_Not_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Not_Expr is
      begin
         if Node = null then
            return No_Entity_Not_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Not_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Not_Pattern
        (Node : Bare_Not_Pattern; Info : Internal_Entity_Info)
         return Internal_Entity_Not_Pattern is
      begin
         if Node = null then
            return No_Entity_Not_Pattern;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Not_Pattern) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Null_Cond_Qualifier
        (Node : Bare_Null_Cond_Qualifier; Info : Internal_Entity_Info)
         return Internal_Entity_Null_Cond_Qualifier is
      begin
         if Node = null then
            return No_Entity_Null_Cond_Qualifier;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Null_Cond_Qualifier) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Null_Cond_Qualifier_Absent
        (Node : Bare_Null_Cond_Qualifier_Absent; Info : Internal_Entity_Info)
         return Internal_Entity_Null_Cond_Qualifier_Absent is
      begin
         if Node = null then
            return No_Entity_Null_Cond_Qualifier_Absent;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Null_Cond_Qualifier_Absent) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Null_Cond_Qualifier_Present
        (Node : Bare_Null_Cond_Qualifier_Present; Info : Internal_Entity_Info)
         return Internal_Entity_Null_Cond_Qualifier_Present is
      begin
         if Node = null then
            return No_Entity_Null_Cond_Qualifier_Present;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Null_Cond_Qualifier_Present) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Null_Lit
        (Node : Bare_Null_Lit; Info : Internal_Entity_Info)
         return Internal_Entity_Null_Lit is
      begin
         if Node = null then
            return No_Entity_Null_Lit;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Null_Lit) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Null_Pattern
        (Node : Bare_Null_Pattern; Info : Internal_Entity_Info)
         return Internal_Entity_Null_Pattern is
      begin
         if Node = null then
            return No_Entity_Null_Pattern;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Null_Pattern) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Num_Lit
        (Node : Bare_Num_Lit; Info : Internal_Entity_Info)
         return Internal_Entity_Num_Lit is
      begin
         if Node = null then
            return No_Entity_Num_Lit;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Num_Lit) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Op
        (Node : Bare_Op; Info : Internal_Entity_Info)
         return Internal_Entity_Op is
      begin
         if Node = null then
            return No_Entity_Op;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Op_Amp
        (Node : Bare_Op_Amp; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Amp is
      begin
         if Node = null then
            return No_Entity_Op_Amp;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Amp) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Op_And
        (Node : Bare_Op_And; Info : Internal_Entity_Info)
         return Internal_Entity_Op_And is
      begin
         if Node = null then
            return No_Entity_Op_And;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_And) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Op_Div
        (Node : Bare_Op_Div; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Div is
      begin
         if Node = null then
            return No_Entity_Op_Div;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Div) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Op_Eq
        (Node : Bare_Op_Eq; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Eq is
      begin
         if Node = null then
            return No_Entity_Op_Eq;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Eq) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Op_Gt
        (Node : Bare_Op_Gt; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Gt is
      begin
         if Node = null then
            return No_Entity_Op_Gt;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Gt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Op_Gte
        (Node : Bare_Op_Gte; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Gte is
      begin
         if Node = null then
            return No_Entity_Op_Gte;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Gte) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Op_Logic_And
        (Node : Bare_Op_Logic_And; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Logic_And is
      begin
         if Node = null then
            return No_Entity_Op_Logic_And;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Logic_And) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Op_Logic_Or
        (Node : Bare_Op_Logic_Or; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Logic_Or is
      begin
         if Node = null then
            return No_Entity_Op_Logic_Or;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Logic_Or) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Op_Lt
        (Node : Bare_Op_Lt; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Lt is
      begin
         if Node = null then
            return No_Entity_Op_Lt;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Lt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Op_Lte
        (Node : Bare_Op_Lte; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Lte is
      begin
         if Node = null then
            return No_Entity_Op_Lte;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Lte) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Op_Minus
        (Node : Bare_Op_Minus; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Minus is
      begin
         if Node = null then
            return No_Entity_Op_Minus;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Minus) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Op_Mult
        (Node : Bare_Op_Mult; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Mult is
      begin
         if Node = null then
            return No_Entity_Op_Mult;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Mult) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Op_Ne
        (Node : Bare_Op_Ne; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Ne is
      begin
         if Node = null then
            return No_Entity_Op_Ne;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Ne) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Op_Or
        (Node : Bare_Op_Or; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Or is
      begin
         if Node = null then
            return No_Entity_Op_Or;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Or) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Op_Or_Int
        (Node : Bare_Op_Or_Int; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Or_Int is
      begin
         if Node = null then
            return No_Entity_Op_Or_Int;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Or_Int) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Op_Plus
        (Node : Bare_Op_Plus; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Plus is
      begin
         if Node = null then
            return No_Entity_Op_Plus;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Plus) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Or_Pattern
        (Node : Bare_Or_Pattern; Info : Internal_Entity_Info)
         return Internal_Entity_Or_Pattern is
      begin
         if Node = null then
            return No_Entity_Or_Pattern;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Or_Pattern) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Paren_Expr
        (Node : Bare_Paren_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Paren_Expr is
      begin
         if Node = null then
            return No_Entity_Paren_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Paren_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Paren_Pattern
        (Node : Bare_Paren_Pattern; Info : Internal_Entity_Info)
         return Internal_Entity_Paren_Pattern is
      begin
         if Node = null then
            return No_Entity_Paren_Pattern;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Paren_Pattern) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Parse_Node_Expr
        (Node : Bare_Parse_Node_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Parse_Node_Expr is
      begin
         if Node = null then
            return No_Entity_Parse_Node_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Parse_Node_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Pattern_Detail_List
        (Node : Bare_Pattern_Detail_List; Info : Internal_Entity_Info)
         return Internal_Entity_Pattern_Detail_List is
      begin
         if Node = null then
            return No_Entity_Pattern_Detail_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Pattern_Detail_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Pattern_List
        (Node : Bare_Pattern_List; Info : Internal_Entity_Info)
         return Internal_Entity_Pattern_List is
      begin
         if Node = null then
            return No_Entity_Pattern_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Pattern_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Pattern_Match_Branch
        (Node : Bare_Pattern_Match_Branch; Info : Internal_Entity_Info)
         return Internal_Entity_Pattern_Match_Branch is
      begin
         if Node = null then
            return No_Entity_Pattern_Match_Branch;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Pattern_Match_Branch) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Single_Line_String_Lit
        (Node : Bare_Single_Line_String_Lit; Info : Internal_Entity_Info)
         return Internal_Entity_Single_Line_String_Lit is
      begin
         if Node = null then
            return No_Entity_Single_Line_String_Lit;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Single_Line_String_Lit) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Pattern_Single_Line_String_Lit
        (Node : Bare_Pattern_Single_Line_String_Lit; Info : Internal_Entity_Info)
         return Internal_Entity_Pattern_Single_Line_String_Lit is
      begin
         if Node = null then
            return No_Entity_Pattern_Single_Line_String_Lit;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Pattern_Single_Line_String_Lit) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Property_Pattern_Detail
        (Node : Bare_Property_Pattern_Detail; Info : Internal_Entity_Info)
         return Internal_Entity_Property_Pattern_Detail is
      begin
         if Node = null then
            return No_Entity_Property_Pattern_Detail;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Property_Pattern_Detail) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Raise_Expr
        (Node : Bare_Raise_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Raise_Expr is
      begin
         if Node = null then
            return No_Entity_Raise_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Raise_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Ref_Id
        (Node : Bare_Ref_Id; Info : Internal_Entity_Info)
         return Internal_Entity_Ref_Id is
      begin
         if Node = null then
            return No_Entity_Ref_Id;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Ref_Id) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Ref_Id_List
        (Node : Bare_Ref_Id_List; Info : Internal_Entity_Info)
         return Internal_Entity_Ref_Id_List is
      begin
         if Node = null then
            return No_Entity_Ref_Id_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Ref_Id_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Regex_Pattern
        (Node : Bare_Regex_Pattern; Info : Internal_Entity_Info)
         return Internal_Entity_Regex_Pattern is
      begin
         if Node = null then
            return No_Entity_Regex_Pattern;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Regex_Pattern) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Selector_Call
        (Node : Bare_Selector_Call; Info : Internal_Entity_Info)
         return Internal_Entity_Selector_Call is
      begin
         if Node = null then
            return No_Entity_Selector_Call;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Selector_Call) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Selector_Pattern_Detail
        (Node : Bare_Selector_Pattern_Detail; Info : Internal_Entity_Info)
         return Internal_Entity_Selector_Pattern_Detail is
      begin
         if Node = null then
            return No_Entity_Selector_Pattern_Detail;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Selector_Pattern_Detail) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Self_Decl
        (Node : Bare_Self_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Self_Decl is
      begin
         if Node = null then
            return No_Entity_Self_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Self_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Simple_Type_Ref
        (Node : Bare_Simple_Type_Ref; Info : Internal_Entity_Info)
         return Internal_Entity_Simple_Type_Ref is
      begin
         if Node = null then
            return No_Entity_Simple_Type_Ref;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Simple_Type_Ref) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Struct_Decl
        (Node : Bare_Struct_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Struct_Decl is
      begin
         if Node = null then
            return No_Entity_Struct_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Struct_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Subscript_Expr
        (Node : Bare_Subscript_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Subscript_Expr is
      begin
         if Node = null then
            return No_Entity_Subscript_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Subscript_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Synth_Fun_Decl
        (Node : Bare_Synth_Fun_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Synth_Fun_Decl is
      begin
         if Node = null then
            return No_Entity_Synth_Fun_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Synth_Fun_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Synth_Param_Decl
        (Node : Bare_Synth_Param_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Synth_Param_Decl is
      begin
         if Node = null then
            return No_Entity_Synth_Param_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Synth_Param_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Synthetic_Lexer_Decl
        (Node : Bare_Synthetic_Lexer_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Synthetic_Lexer_Decl is
      begin
         if Node = null then
            return No_Entity_Synthetic_Lexer_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Synthetic_Lexer_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Type_Ref_List
        (Node : Bare_Type_Ref_List; Info : Internal_Entity_Info)
         return Internal_Entity_Type_Ref_List is
      begin
         if Node = null then
            return No_Entity_Type_Ref_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Type_Ref_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Synthetic_Type_Ref_List
        (Node : Bare_Synthetic_Type_Ref_List; Info : Internal_Entity_Info)
         return Internal_Entity_Synthetic_Type_Ref_List is
      begin
         if Node = null then
            return No_Entity_Synthetic_Type_Ref_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Synthetic_Type_Ref_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Token_Lit
        (Node : Bare_Token_Lit; Info : Internal_Entity_Info)
         return Internal_Entity_Token_Lit is
      begin
         if Node = null then
            return No_Entity_Token_Lit;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Token_Lit) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Token_No_Case_Lit
        (Node : Bare_Token_No_Case_Lit; Info : Internal_Entity_Info)
         return Internal_Entity_Token_No_Case_Lit is
      begin
         if Node = null then
            return No_Entity_Token_No_Case_Lit;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Token_No_Case_Lit) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Token_Pattern_Concat
        (Node : Bare_Token_Pattern_Concat; Info : Internal_Entity_Info)
         return Internal_Entity_Token_Pattern_Concat is
      begin
         if Node = null then
            return No_Entity_Token_Pattern_Concat;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Token_Pattern_Concat) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Token_Pattern_Lit
        (Node : Bare_Token_Pattern_Lit; Info : Internal_Entity_Info)
         return Internal_Entity_Token_Pattern_Lit is
      begin
         if Node = null then
            return No_Entity_Token_Pattern_Lit;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Token_Pattern_Lit) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Token_Ref
        (Node : Bare_Token_Ref; Info : Internal_Entity_Info)
         return Internal_Entity_Token_Ref is
      begin
         if Node = null then
            return No_Entity_Token_Ref;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Token_Ref) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Trait_Decl
        (Node : Bare_Trait_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Trait_Decl is
      begin
         if Node = null then
            return No_Entity_Trait_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Trait_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Try_Expr
        (Node : Bare_Try_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Try_Expr is
      begin
         if Node = null then
            return No_Entity_Try_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Try_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Tuple_Pattern
        (Node : Bare_Tuple_Pattern; Info : Internal_Entity_Info)
         return Internal_Entity_Tuple_Pattern is
      begin
         if Node = null then
            return No_Entity_Tuple_Pattern;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Tuple_Pattern) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Type_Pattern
        (Node : Bare_Type_Pattern; Info : Internal_Entity_Info)
         return Internal_Entity_Type_Pattern is
      begin
         if Node = null then
            return No_Entity_Type_Pattern;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Type_Pattern) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Un_Op
        (Node : Bare_Un_Op; Info : Internal_Entity_Info)
         return Internal_Entity_Un_Op is
      begin
         if Node = null then
            return No_Entity_Un_Op;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Un_Op) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Val_Decl
        (Node : Bare_Val_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Val_Decl is
      begin
         if Node = null then
            return No_Entity_Val_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Val_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Var_Bind
        (Node : Bare_Var_Bind; Info : Internal_Entity_Info)
         return Internal_Entity_Var_Bind is
      begin
         if Node = null then
            return No_Entity_Var_Bind;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Var_Bind) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   


      -------------
      -- Inc_Ref --
      -------------

      procedure Inc_Ref (R : Internal_Env_Assoc) is
      begin
               Inc_Ref (R.Dest_Env);
      end Inc_Ref;

      -------------
      -- Dec_Ref --
      -------------

      procedure Dec_Ref (R : in out Internal_Env_Assoc) is
      begin
               Dec_Ref (R.Dest_Env);
      end Dec_Ref;




      ----------------
      -- Equivalent --
      ----------------

      function Equivalent (L, R : Internal_Env_Assoc) return Boolean is
      begin
         return L.Key = R.Key and then L.Value = R.Value and then Equivalent (L.Dest_Env, R.Dest_Env) and then L.Metadata = R.Metadata;
      end Equivalent;


   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Env_Assoc) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Key => "
                     & Trace_Image (R.Key)
                        & ", "
                     & "Value => "
                     & Trace_Image (R.Value)
                        & ", "
                     & "Dest_Env => "
                     & Trace_Image (R.Dest_Env)
                        & ", "
                     & "Metadata => "
                     & Trace_Image (R.Metadata)
               & ")");
      end Trace_Image;


   

   




   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Inner_Env_Assoc) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Key => "
                     & Trace_Image (R.Key)
                        & ", "
                     & "Value => "
                     & Trace_Image (R.Value)
                        & ", "
                     & "Rebindings => "
                     & Trace_Image (R.Rebindings)
                        & ", "
                     & "Metadata => "
                     & Trace_Image (R.Metadata)
               & ")");
      end Trace_Image;


   

   




   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Logic_Context) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Ref_Node => "
                     & Trace_Image (R.Ref_Node)
                        & ", "
                     & "Decl_Node => "
                     & Trace_Image (R.Decl_Node)
               & ")");
      end Trace_Image;


   

   




   

      ----------
      -- Hash --
      ----------

      pragma Warnings (Off, "referenced");
      function Hash (R : Internal_Resolved_Param) return Hash_Type is
         pragma Warnings (On, "referenced");
      begin
         
            return Combine ((Hash (R.Name), Hash (R.Param_Type), Hash (R.Has_Default_Value), Hash (R.Accept_Logical_Var), Hash (R.Decl)));
      end Hash;


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Resolved_Param) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Name => "
                     & Trace_Image (R.Name)
                        & ", "
                     & "Param_Type => "
                     & Trace_Image (R.Param_Type)
                        & ", "
                     & "Has_Default_Value => "
                     & Trace_Image (R.Has_Default_Value)
                        & ", "
                     & "Accept_Logical_Var => "
                     & Trace_Image (R.Accept_Logical_Var)
                        & ", "
                     & "Decl => "
                     & Trace_Image (R.Decl)
               & ")");
      end Trace_Image;


   

   




   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Param_Match) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Has_Matched => "
                     & Trace_Image (R.Has_Matched)
                        & ", "
                     & "Arg => "
                     & Trace_Image (R.Arg)
                        & ", "
                     & "Param => "
                     & Trace_Image (R.Param)
               & ")");
      end Trace_Image;


   

   




   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Ref_Result) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Ref => "
                     & Trace_Image (R.Ref)
               & ")");
      end Trace_Image;


   

   


      -------------
      -- Inc_Ref --
      -------------

      procedure Inc_Ref (R : Internal_Solver_Diagnostic) is
      begin
               Inc_Ref (R.Message_Template);
               Inc_Ref (R.Args);
               Inc_Ref (R.Contexts);
      end Inc_Ref;

      -------------
      -- Dec_Ref --
      -------------

      procedure Dec_Ref (R : in out Internal_Solver_Diagnostic) is
      begin
               Dec_Ref (R.Message_Template);
               Dec_Ref (R.Args);
               Dec_Ref (R.Contexts);
      end Dec_Ref;




      ----------------
      -- Equivalent --
      ----------------

      function Equivalent (L, R : Internal_Solver_Diagnostic) return Boolean is
      begin
         return Equivalent (L.Message_Template, R.Message_Template) and then Equivalent (L.Args, R.Args) and then L.Location = R.Location and then Equivalent (L.Contexts, R.Contexts) and then L.Round = R.Round;
      end Equivalent;


   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Solver_Diagnostic) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Message_Template => "
                     & Trace_Image (R.Message_Template)
                        & ", "
                     & "Args => "
                     & Trace_Image (R.Args)
                        & ", "
                     & "Location => "
                     & Trace_Image (R.Location)
                        & ", "
                     & "Contexts => "
                     & Trace_Image (R.Contexts)
                        & ", "
                     & "Round => "
                     & Trace_Image (R.Round)
               & ")");
      end Trace_Image;


   

   


      -------------
      -- Inc_Ref --
      -------------

      procedure Inc_Ref (R : Internal_Solver_Result) is
      begin
               Inc_Ref (R.Diagnostics);
      end Inc_Ref;

      -------------
      -- Dec_Ref --
      -------------

      procedure Dec_Ref (R : in out Internal_Solver_Result) is
      begin
               Dec_Ref (R.Diagnostics);
      end Dec_Ref;




      ----------------
      -- Equivalent --
      ----------------

      function Equivalent (L, R : Internal_Solver_Result) return Boolean is
      begin
         return L.Success = R.Success and then Equivalent (L.Diagnostics, R.Diagnostics);
      end Equivalent;


   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Solver_Result) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Success => "
                     & Trace_Image (R.Success)
                        & ", "
                     & "Diagnostics => "
                     & Trace_Image (R.Diagnostics)
               & ")");
      end Trace_Image;



   
         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

   

   

   

   type Lkt_Node_P_Unmatched_Argument_0_Predicate is
   new Solver_Ifc.N_Predicate_Type
   with record
         Error_Location : Bare_Lkt_Node;
   end record;

   
      overriding function Call
        (Self : Lkt_Node_P_Unmatched_Argument_0_Predicate;
            Entities : Entity_Vars.Value_Array
        ) return Boolean
   ;

   
      overriding procedure Failed
        (Self : Lkt_Node_P_Unmatched_Argument_0_Predicate;
            Entities : Entity_Vars.Value_Array;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   ;

   overriding function Image (Self : Lkt_Node_P_Unmatched_Argument_0_Predicate) return String;

   

   function Create_Lkt_Node_P_Unmatched_Argument_0_Predicate
   (
         N : Positive;
         Error_Location : Bare_Lkt_Node
   )
      return Lkt_Node_P_Unmatched_Argument_0_Predicate is
   begin
      
      return Lkt_Node_P_Unmatched_Argument_0_Predicate'(N => N, Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Error_Location => Error_Location);
   end;

   ----------
   -- Call --
   ----------

   
      overriding function Call
        (Self : Lkt_Node_P_Unmatched_Argument_0_Predicate;
            Entities : Entity_Vars.Value_Array
        ) return Boolean
   
   is
         pragma Unreferenced (Self);

         Entity : Internal_Entity := Entities (1);


      
      Node : Bare_Lkt_Node;

      Ret : Boolean;
   begin

      
            if Entities (2).Node /= null
               and then Entities (2).Node.Kind
                        not in Lkt_Type_Decl
            then
               Raise_Property_Exception
                 (Node, Property_Error'Identity, "mismatching node type");
            end if;

      Node := Entity.Node;


      

      Ret := Lkt_Node_P_Unmatched_Argument (Node, (Node => Entities (2).Node, Info => Entities (2).Info), E_Info => Entity.Info);


      return Ret;
   end Call;

   ------------
   -- Failed --
   ------------

   
      overriding procedure Failed
        (Self : Lkt_Node_P_Unmatched_Argument_0_Predicate;
            Entities : Entity_Vars.Value_Array;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   
   is
      
      Args : Internal_Entity_Array_Access :=
         Create_Internal_Entity_Array (0);

      Contexts : Internal_Logic_Context_Array_Access :=
         Create_Internal_Logic_Context_Array (Ctxs'Length);

      Diag : constant Internal_Solver_Diagnostic :=
        (Message_Template => Create_String ("unmatched arguments"),
         Args             => Args,
         Contexts         => Contexts,
         Location         => Self.Error_Location,
         Round            => Round);
   begin
      for I in Ctxs'Range loop
         Contexts.Items (I) := Ctxs (I).all;
      end loop;
      Emitter (Diag);
   end Failed;

   -----------
   -- Image --
   -----------

   overriding function Image (Self : Lkt_Node_P_Unmatched_Argument_0_Predicate) return String is
   begin
      return "LktNode.unmatched_argument";
   end Image;



         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

   

   

   

   type Decl_P_Is_Type_Decl_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         Error_Location : Bare_Lkt_Node;
   end record;

   
      overriding function Call
        (Self : Decl_P_Is_Type_Decl_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;

   
      overriding procedure Failed
        (Self : Decl_P_Is_Type_Decl_0_Predicate;
            Entity : Internal_Entity;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   ;

   overriding function Image (Self : Decl_P_Is_Type_Decl_0_Predicate) return String;

   

   function Create_Decl_P_Is_Type_Decl_0_Predicate
   (
         Error_Location : Bare_Lkt_Node
   )
      return Decl_P_Is_Type_Decl_0_Predicate is
   begin
      
      return Decl_P_Is_Type_Decl_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Error_Location => Error_Location);
   end;

   ----------
   -- Call --
   ----------

   
      overriding function Call
        (Self : Decl_P_Is_Type_Decl_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   
   is
         pragma Unreferenced (Self);



      
      Node : Bare_Decl;

      Ret : Boolean;
   begin

      
            if Entity.Node /= null
               and then Entity.Node.Kind
                        not in Lkt_Decl
            then
               Raise_Property_Exception
                 (Node, Property_Error'Identity, "mismatching node type");
            end if;

      Node := Entity.Node;


      

      Ret := Decl_P_Is_Type_Decl (Node);


      return Ret;
   end Call;

   ------------
   -- Failed --
   ------------

   
      overriding procedure Failed
        (Self : Decl_P_Is_Type_Decl_0_Predicate;
            Entity : Internal_Entity;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   
   is
      
      Args : Internal_Entity_Array_Access :=
         Create_Internal_Entity_Array (1);

      Contexts : Internal_Logic_Context_Array_Access :=
         Create_Internal_Logic_Context_Array (Ctxs'Length);

      Diag : constant Internal_Solver_Diagnostic :=
        (Message_Template => Create_String ("{} is not a type"),
         Args             => Args,
         Contexts         => Contexts,
         Location         => Self.Error_Location,
         Round            => Round);
   begin
      Args.Items (1) := Entity;
      for I in Ctxs'Range loop
         Contexts.Items (I) := Ctxs (I).all;
      end loop;
      Emitter (Diag);
   end Failed;

   -----------
   -- Image --
   -----------

   overriding function Image (Self : Decl_P_Is_Type_Decl_0_Predicate) return String is
   begin
      return "Decl.is_type_decl";
   end Image;



         

         

   

   

   

   type Decl_P_Is_Defined_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         Error_Location : Bare_Lkt_Node;
   end record;

   
      overriding function Call
        (Self : Decl_P_Is_Defined_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;

   
      overriding procedure Failed
        (Self : Decl_P_Is_Defined_0_Predicate;
            Entity : Internal_Entity;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   ;

   overriding function Image (Self : Decl_P_Is_Defined_0_Predicate) return String;

   

   function Create_Decl_P_Is_Defined_0_Predicate
   (
         Error_Location : Bare_Lkt_Node
   )
      return Decl_P_Is_Defined_0_Predicate is
   begin
      
      return Decl_P_Is_Defined_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Error_Location => Error_Location);
   end;

   ----------
   -- Call --
   ----------

   
      overriding function Call
        (Self : Decl_P_Is_Defined_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   
   is
         pragma Unreferenced (Self);



      
      Node : Bare_Decl;

      Ret : Boolean;
   begin

      
            if Entity.Node /= null
               and then Entity.Node.Kind
                        not in Lkt_Decl
            then
               Raise_Property_Exception
                 (Node, Property_Error'Identity, "mismatching node type");
            end if;

      Node := Entity.Node;


      

      Ret := Decl_P_Is_Defined (Node);


      return Ret;
   end Call;

   ------------
   -- Failed --
   ------------

   
      overriding procedure Failed
        (Self : Decl_P_Is_Defined_0_Predicate;
            Entity : Internal_Entity;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   
   is
      
      Args : Internal_Entity_Array_Access :=
         Create_Internal_Entity_Array (0);

      Contexts : Internal_Logic_Context_Array_Access :=
         Create_Internal_Logic_Context_Array (Ctxs'Length);

      Diag : constant Internal_Solver_Diagnostic :=
        (Message_Template => Create_String ("undefined entity"),
         Args             => Args,
         Contexts         => Contexts,
         Location         => Self.Error_Location,
         Round            => Round);
   begin
      for I in Ctxs'Range loop
         Contexts.Items (I) := Ctxs (I).all;
      end loop;
      Emitter (Diag);
   end Failed;

   -----------
   -- Image --
   -----------

   overriding function Image (Self : Decl_P_Is_Defined_0_Predicate) return String is
   begin
      return "Decl.is_defined";
   end Image;



         

         

         

         

         

   

   type Decl_P_Function_Type_0_Functor is new Solver_Ifc.Converter_Type with
      null record;

   overriding function Convert (Self : Decl_P_Function_Type_0_Functor; From : Internal_Entity) return Internal_Entity with Inline;
   overriding function Image (Self : Decl_P_Function_Type_0_Functor) return String;


   

   function Create_Decl_P_Function_Type_0_Functor
      return Decl_P_Function_Type_0_Functor is
   begin
      
      return Decl_P_Function_Type_0_Functor'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1);
   end;

   overriding function Convert (Self : Decl_P_Function_Type_0_Functor; From : Internal_Entity) return Internal_Entity is
         pragma Unreferenced (Self);



      

      Ret : Internal_Entity_Function_Type;
   begin
            if From.Node /= null
               and then From.Node.Kind not in
                 Lkt_Decl
            then
               Raise_Property_Exception
                 (From.Node,
                  Property_Error'Identity,
                  "mismatching node type for conv_prop");
            end if;


      
      Ret := Decl_P_Function_Type (Node => From.Node, E_Info => From.Info);


      return (Node => Ret.Node, Info => Ret.Info);
   end Convert;

   -----------
   -- Image --
   -----------

   overriding function Image (Self : Decl_P_Function_Type_0_Functor) return String is
   begin
      return ("Decl.function_type");
   end Image;



         

         

         

         

         

         

         

         

   

   type Decl_P_Get_Keep_Type_0_Functor is new Solver_Ifc.Combiner_Type with
      null record;

   overriding function Combine (Self : Decl_P_Get_Keep_Type_0_Functor; Vals : Entity_Vars.Value_Array) return Internal_Entity with Inline;
   overriding function Image (Self : Decl_P_Get_Keep_Type_0_Functor) return String;


   

   function Create_Decl_P_Get_Keep_Type_0_Functor
   (
         N : Positive
   )
      return Decl_P_Get_Keep_Type_0_Functor is
   begin
      
      return Decl_P_Get_Keep_Type_0_Functor'(N => N, Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1);
   end;

   overriding function Combine (Self : Decl_P_Get_Keep_Type_0_Functor; Vals : Entity_Vars.Value_Array) return Internal_Entity is
         pragma Unreferenced (Self);

         From : constant Internal_Entity := Vals (1);


      

      Ret : Internal_Entity_Type_Decl;
   begin
            if From.Node /= null
               and then From.Node.Kind not in
                 Lkt_Decl
            then
               Raise_Property_Exception
                 (From.Node,
                  Property_Error'Identity,
                  "mismatching node type for comb_prop");
            end if;
            if Vals (2).Node /= null
               and then Vals (2).Node.Kind not in
                 Lkt_Type_Decl
            then
               Raise_Property_Exception
                 (From.Node,
                  Property_Error'Identity,
                  "mismatching node type for comb_prop");
            end if;


      
      Ret := Decl_P_Get_Keep_Type (Node => From.Node, Keep_Type => (Node => Vals (2).Node,  Info => Vals (2).Info), E_Info => From.Info);


      return (Node => Ret.Node, Info => Ret.Info);
   end Combine;

   -----------
   -- Image --
   -----------

   overriding function Image (Self : Decl_P_Get_Keep_Type_0_Functor) return String is
   begin
      return ("Decl.get_keep_type");
   end Image;



         

         

         

         

         

         

         

         

         

         

         

   

   

   

   type Decl_P_Is_Generic_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         Error_Location : Bare_Lkt_Node;
   end record;

   
      overriding function Call
        (Self : Decl_P_Is_Generic_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;

   
      overriding procedure Failed
        (Self : Decl_P_Is_Generic_0_Predicate;
            Entity : Internal_Entity;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   ;

   overriding function Image (Self : Decl_P_Is_Generic_0_Predicate) return String;

   

   function Create_Decl_P_Is_Generic_0_Predicate
   (
         Error_Location : Bare_Lkt_Node
   )
      return Decl_P_Is_Generic_0_Predicate is
   begin
      
      return Decl_P_Is_Generic_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Error_Location => Error_Location);
   end;

   ----------
   -- Call --
   ----------

   
      overriding function Call
        (Self : Decl_P_Is_Generic_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   
   is
         pragma Unreferenced (Self);



      
      Node : Bare_Decl;

      Ret : Boolean;
   begin

      
            if Entity.Node /= null
               and then Entity.Node.Kind
                        not in Lkt_Decl
            then
               Raise_Property_Exception
                 (Node, Property_Error'Identity, "mismatching node type");
            end if;

      Node := Entity.Node;


      

      Ret := Decl_P_Is_Generic (Node, E_Info => Entity.Info);


      return Ret;
   end Call;

   ------------
   -- Failed --
   ------------

   
      overriding procedure Failed
        (Self : Decl_P_Is_Generic_0_Predicate;
            Entity : Internal_Entity;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   
   is
      
      Args : Internal_Entity_Array_Access :=
         Create_Internal_Entity_Array (1);

      Contexts : Internal_Logic_Context_Array_Access :=
         Create_Internal_Logic_Context_Array (Ctxs'Length);

      Diag : constant Internal_Solver_Diagnostic :=
        (Message_Template => Create_String ("{} is not a generic declaration"),
         Args             => Args,
         Contexts         => Contexts,
         Location         => Self.Error_Location,
         Round            => Round);
   begin
      Args.Items (1) := Entity;
      for I in Ctxs'Range loop
         Contexts.Items (I) := Ctxs (I).all;
      end loop;
      Emitter (Diag);
   end Failed;

   -----------
   -- Image --
   -----------

   overriding function Image (Self : Decl_P_Is_Generic_0_Predicate) return String is
   begin
      return "Decl.is_generic";
   end Image;



         

         

   

   

   

   type Decl_P_Return_Type_Is_Instantiated_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         null;
   end record;

   
      overriding function Call
        (Self : Decl_P_Return_Type_Is_Instantiated_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;


   overriding function Image (Self : Decl_P_Return_Type_Is_Instantiated_0_Predicate) return String;

   

   function Create_Decl_P_Return_Type_Is_Instantiated_0_Predicate
      return Decl_P_Return_Type_Is_Instantiated_0_Predicate is
   begin
      
      return Decl_P_Return_Type_Is_Instantiated_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1);
   end;

   ----------
   -- Call --
   ----------

   
      overriding function Call
        (Self : Decl_P_Return_Type_Is_Instantiated_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   
   is
         pragma Unreferenced (Self);



      
      Node : Bare_Decl;

      Ret : Boolean;
   begin

      
            if Entity.Node /= null
               and then Entity.Node.Kind
                        not in Lkt_Decl
            then
               Raise_Property_Exception
                 (Node, Property_Error'Identity, "mismatching node type");
            end if;

      Node := Entity.Node;


      

      Ret := Decl_P_Return_Type_Is_Instantiated (Node, E_Info => Entity.Info);


      return Ret;
   end Call;


   -----------
   -- Image --
   -----------

   overriding function Image (Self : Decl_P_Return_Type_Is_Instantiated_0_Predicate) return String is
   begin
      return "Decl.return_type_is_instantiated";
   end Image;



         

         

         

         

   

   

   

   type Decl_P_Has_Correct_Type_Arg_Number_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         Field_0 : Integer;
         Error_Location : Bare_Lkt_Node;
   end record;

   
      overriding function Call
        (Self : Decl_P_Has_Correct_Type_Arg_Number_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;

   
      overriding procedure Failed
        (Self : Decl_P_Has_Correct_Type_Arg_Number_0_Predicate;
            Entity : Internal_Entity;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   ;

   overriding function Image (Self : Decl_P_Has_Correct_Type_Arg_Number_0_Predicate) return String;

   

   function Create_Decl_P_Has_Correct_Type_Arg_Number_0_Predicate
   (
         Nb_Types : Integer;
         Error_Location : Bare_Lkt_Node
   )
      return Decl_P_Has_Correct_Type_Arg_Number_0_Predicate is
   begin
      
         
      return Decl_P_Has_Correct_Type_Arg_Number_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Error_Location => Error_Location, Field_0 => Nb_Types);
   end;

   ----------
   -- Call --
   ----------

   
      overriding function Call
        (Self : Decl_P_Has_Correct_Type_Arg_Number_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   
   is



      
      Node : Bare_Decl;

      Ret : Boolean;
   begin

      
            if Entity.Node /= null
               and then Entity.Node.Kind
                        not in Lkt_Decl
            then
               Raise_Property_Exception
                 (Node, Property_Error'Identity, "mismatching node type");
            end if;

      Node := Entity.Node;


      

      Ret := Decl_P_Has_Correct_Type_Arg_Number (Node, Nb_Types => Self.Field_0, E_Info => Entity.Info);


      return Ret;
   end Call;

   ------------
   -- Failed --
   ------------

   
      overriding procedure Failed
        (Self : Decl_P_Has_Correct_Type_Arg_Number_0_Predicate;
            Entity : Internal_Entity;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   
   is
      
      Args : Internal_Entity_Array_Access :=
         Create_Internal_Entity_Array (0);

      Contexts : Internal_Logic_Context_Array_Access :=
         Create_Internal_Logic_Context_Array (Ctxs'Length);

      Diag : constant Internal_Solver_Diagnostic :=
        (Message_Template => Create_String ("Incorrect number of parameter types"),
         Args             => Args,
         Contexts         => Contexts,
         Location         => Self.Error_Location,
         Round            => Round);
   begin
      for I in Ctxs'Range loop
         Contexts.Items (I) := Ctxs (I).all;
      end loop;
      Emitter (Diag);
   end Failed;

   -----------
   -- Image --
   -----------

   overriding function Image (Self : Decl_P_Has_Correct_Type_Arg_Number_0_Predicate) return String is
   begin
      return "Decl.has_correct_type_arg_number";
   end Image;



         

         

   

   

   

   type Decl_P_Could_Infer_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         Field_0 : Internal_Entity_Function_Type;
         Error_Location : Bare_Lkt_Node;
   end record;

   
      overriding function Call
        (Self : Decl_P_Could_Infer_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;

   
      overriding procedure Failed
        (Self : Decl_P_Could_Infer_0_Predicate;
            Entity : Internal_Entity;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   ;

   overriding function Image (Self : Decl_P_Could_Infer_0_Predicate) return String;

   

   function Create_Decl_P_Could_Infer_0_Predicate
   (
         Generic_Type : Internal_Entity_Function_Type;
         Error_Location : Bare_Lkt_Node
   )
      return Decl_P_Could_Infer_0_Predicate is
   begin
      
         
      return Decl_P_Could_Infer_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Error_Location => Error_Location, Field_0 => Generic_Type);
   end;

   ----------
   -- Call --
   ----------

   
      overriding function Call
        (Self : Decl_P_Could_Infer_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   
   is



      
      Node : Bare_Decl;

      Ret : Boolean;
   begin

      
            if Entity.Node /= null
               and then Entity.Node.Kind
                        not in Lkt_Decl
            then
               Raise_Property_Exception
                 (Node, Property_Error'Identity, "mismatching node type");
            end if;

      Node := Entity.Node;


      

      Ret := Decl_P_Could_Infer (Node, Generic_Type => Self.Field_0, E_Info => Entity.Info);


      return Ret;
   end Call;

   ------------
   -- Failed --
   ------------

   
      overriding procedure Failed
        (Self : Decl_P_Could_Infer_0_Predicate;
            Entity : Internal_Entity;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   
   is
      
      Args : Internal_Entity_Array_Access :=
         Create_Internal_Entity_Array (0);

      Contexts : Internal_Logic_Context_Array_Access :=
         Create_Internal_Logic_Context_Array (Ctxs'Length);

      Diag : constant Internal_Solver_Diagnostic :=
        (Message_Template => Create_String ("could not determine callee type"),
         Args             => Args,
         Contexts         => Contexts,
         Location         => Self.Error_Location,
         Round            => Round);
   begin
      for I in Ctxs'Range loop
         Contexts.Items (I) := Ctxs (I).all;
      end loop;
      Emitter (Diag);
   end Failed;

   -----------
   -- Image --
   -----------

   overriding function Image (Self : Decl_P_Could_Infer_0_Predicate) return String is
   begin
      return "Decl.could_infer";
   end Image;



         

         

         

   

   type Decl_P_Instantiate_Generic_Decl_0_Functor is new Solver_Ifc.Combiner_Type with
      null record;

   overriding function Combine (Self : Decl_P_Instantiate_Generic_Decl_0_Functor; Vals : Entity_Vars.Value_Array) return Internal_Entity with Inline;
   overriding function Image (Self : Decl_P_Instantiate_Generic_Decl_0_Functor) return String;


   

   function Create_Decl_P_Instantiate_Generic_Decl_0_Functor
   (
         N : Positive
   )
      return Decl_P_Instantiate_Generic_Decl_0_Functor is
   begin
      
      return Decl_P_Instantiate_Generic_Decl_0_Functor'(N => N, Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1);
   end;

   overriding function Combine (Self : Decl_P_Instantiate_Generic_Decl_0_Functor; Vals : Entity_Vars.Value_Array) return Internal_Entity is
         pragma Unreferenced (Self);

         From : constant Internal_Entity := Vals (1);

         
         Args : Internal_Entity_Type_Decl_Array_Access :=
           Create_Internal_Entity_Type_Decl_Array (Vals'Length - 1);

      

      Ret : Internal_Entity_Decl;
   begin
            if From.Node /= null
               and then From.Node.Kind not in
                 Lkt_Decl
            then
               Raise_Property_Exception
                 (From.Node,
                  Property_Error'Identity,
                  "mismatching node type for comb_prop");
            end if;

      
      for I in 2 .. Vals'Last loop
            if Vals (I).Node /= null
               and then Vals (I).Node.Kind not in
                  Lkt_Type_Decl
            then
               Raise_Property_Exception
                 (Vals (I).Node,
                  Property_Error'Identity,
                  "mismatching node type for comb_prop");
            end if;
         Args.Items (I - 1) := (Vals (I).Node, Vals (I).Info);
      end loop;

      
      Ret := Decl_P_Instantiate_Generic_Decl (Node => From.Node, Param_Types => Args, E_Info => From.Info);

      Dec_Ref (Args);

      return (Node => Ret.Node, Info => Ret.Info);
   exception
      when Exc : Property_Error =>
         pragma Unreferenced (Exc);
         Dec_Ref (Args);
         raise;
   end Combine;

   -----------
   -- Image --
   -----------

   overriding function Image (Self : Decl_P_Instantiate_Generic_Decl_0_Functor) return String is
   begin
      return ("Decl.instantiate_generic_decl");
   end Image;



         

         

         

   

   

   

   type Decl_P_Is_Dynvar_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         Error_Location : Bare_Lkt_Node;
   end record;

   
      overriding function Call
        (Self : Decl_P_Is_Dynvar_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;

   
      overriding procedure Failed
        (Self : Decl_P_Is_Dynvar_0_Predicate;
            Entity : Internal_Entity;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   ;

   overriding function Image (Self : Decl_P_Is_Dynvar_0_Predicate) return String;

   

   function Create_Decl_P_Is_Dynvar_0_Predicate
   (
         Error_Location : Bare_Lkt_Node
   )
      return Decl_P_Is_Dynvar_0_Predicate is
   begin
      
      return Decl_P_Is_Dynvar_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Error_Location => Error_Location);
   end;

   ----------
   -- Call --
   ----------

   
      overriding function Call
        (Self : Decl_P_Is_Dynvar_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   
   is
         pragma Unreferenced (Self);



      
      Node : Bare_Decl;

      Ret : Boolean;
   begin

      
            if Entity.Node /= null
               and then Entity.Node.Kind
                        not in Lkt_Decl
            then
               Raise_Property_Exception
                 (Node, Property_Error'Identity, "mismatching node type");
            end if;

      Node := Entity.Node;


      

      Ret := Decl_P_Is_Dynvar (Node, E_Info => Entity.Info);


      return Ret;
   end Call;

   ------------
   -- Failed --
   ------------

   
      overriding procedure Failed
        (Self : Decl_P_Is_Dynvar_0_Predicate;
            Entity : Internal_Entity;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   
   is
      
      Args : Internal_Entity_Array_Access :=
         Create_Internal_Entity_Array (1);

      Contexts : Internal_Logic_Context_Array_Access :=
         Create_Internal_Logic_Context_Array (Ctxs'Length);

      Diag : constant Internal_Solver_Diagnostic :=
        (Message_Template => Create_String ("{} is not a dynamic variable"),
         Args             => Args,
         Contexts         => Contexts,
         Location         => Self.Error_Location,
         Round            => Round);
   begin
      Args.Items (1) := Entity;
      for I in Ctxs'Range loop
         Contexts.Items (I) := Ctxs (I).all;
      end loop;
      Emitter (Diag);
   end Failed;

   -----------
   -- Image --
   -----------

   overriding function Image (Self : Decl_P_Is_Dynvar_0_Predicate) return String is
   begin
      return "Decl.is_dynvar";
   end Image;



         

         

         

         

         

         

         

         

         

         

         

         

   

   

   

   type Decl_P_Is_Directly_Referenceable_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         Error_Location : Bare_Lkt_Node;
   end record;

   
      overriding function Call
        (Self : Decl_P_Is_Directly_Referenceable_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;

   
      overriding procedure Failed
        (Self : Decl_P_Is_Directly_Referenceable_0_Predicate;
            Entity : Internal_Entity;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   ;

   overriding function Image (Self : Decl_P_Is_Directly_Referenceable_0_Predicate) return String;

   

   function Create_Decl_P_Is_Directly_Referenceable_0_Predicate
   (
         Error_Location : Bare_Lkt_Node
   )
      return Decl_P_Is_Directly_Referenceable_0_Predicate is
   begin
      
      return Decl_P_Is_Directly_Referenceable_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Error_Location => Error_Location);
   end;

   ----------
   -- Call --
   ----------

   
      overriding function Call
        (Self : Decl_P_Is_Directly_Referenceable_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   
   is
         pragma Unreferenced (Self);



      
      Node : Bare_Decl;

      Ret : Boolean;
   begin

      
            if Entity.Node /= null
               and then Entity.Node.Kind
                        not in Lkt_Decl
            then
               Raise_Property_Exception
                 (Node, Property_Error'Identity, "mismatching node type");
            end if;

      Node := Entity.Node;


      

      Ret := Decl_P_Is_Directly_Referenceable (Node, E_Info => Entity.Info);


      return Ret;
   end Call;

   ------------
   -- Failed --
   ------------

   
      overriding procedure Failed
        (Self : Decl_P_Is_Directly_Referenceable_0_Predicate;
            Entity : Internal_Entity;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   
   is
      
      Args : Internal_Entity_Array_Access :=
         Create_Internal_Entity_Array (1);

      Contexts : Internal_Logic_Context_Array_Access :=
         Create_Internal_Logic_Context_Array (Ctxs'Length);

      Diag : constant Internal_Solver_Diagnostic :=
        (Message_Template => Create_String ("{} cannot be directly referenced"),
         Args             => Args,
         Contexts         => Contexts,
         Location         => Self.Error_Location,
         Round            => Round);
   begin
      Args.Items (1) := Entity;
      for I in Ctxs'Range loop
         Contexts.Items (I) := Ctxs (I).all;
      end loop;
      Emitter (Diag);
   end Failed;

   -----------
   -- Image --
   -----------

   overriding function Image (Self : Decl_P_Is_Directly_Referenceable_0_Predicate) return String is
   begin
      return "Decl.is_directly_referenceable";
   end Image;



         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

   

   

   

   type Type_Decl_P_Is_Equation_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         Error_Location : Bare_Lkt_Node;
   end record;

   
      overriding function Call
        (Self : Type_Decl_P_Is_Equation_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;

   
      overriding procedure Failed
        (Self : Type_Decl_P_Is_Equation_0_Predicate;
            Entity : Internal_Entity;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   ;

   overriding function Image (Self : Type_Decl_P_Is_Equation_0_Predicate) return String;

   

   function Create_Type_Decl_P_Is_Equation_0_Predicate
   (
         Error_Location : Bare_Lkt_Node
   )
      return Type_Decl_P_Is_Equation_0_Predicate is
   begin
      
      return Type_Decl_P_Is_Equation_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Error_Location => Error_Location);
   end;

   ----------
   -- Call --
   ----------

   
      overriding function Call
        (Self : Type_Decl_P_Is_Equation_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   
   is
         pragma Unreferenced (Self);



      
      Node : Bare_Type_Decl;

      Ret : Boolean;
   begin

      
            if Entity.Node /= null
               and then Entity.Node.Kind
                        not in Lkt_Type_Decl
            then
               Raise_Property_Exception
                 (Node, Property_Error'Identity, "mismatching node type");
            end if;

      Node := Entity.Node;


      

      Ret := Type_Decl_P_Is_Equation (Node, E_Info => Entity.Info);


      return Ret;
   end Call;

   ------------
   -- Failed --
   ------------

   
      overriding procedure Failed
        (Self : Type_Decl_P_Is_Equation_0_Predicate;
            Entity : Internal_Entity;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   
   is
      
      Args : Internal_Entity_Array_Access :=
         Create_Internal_Entity_Array (1);

      Contexts : Internal_Logic_Context_Array_Access :=
         Create_Internal_Logic_Context_Array (Ctxs'Length);

      Diag : constant Internal_Solver_Diagnostic :=
        (Message_Template => Create_String ("operands of %and and %or must be equations, got {}"),
         Args             => Args,
         Contexts         => Contexts,
         Location         => Self.Error_Location,
         Round            => Round);
   begin
      Args.Items (1) := Entity;
      for I in Ctxs'Range loop
         Contexts.Items (I) := Ctxs (I).all;
      end loop;
      Emitter (Diag);
   end Failed;

   -----------
   -- Image --
   -----------

   overriding function Image (Self : Type_Decl_P_Is_Equation_0_Predicate) return String is
   begin
      return "TypeDecl.is_equation";
   end Image;



         

         

   

   

   

   type Type_Decl_P_Is_Bool_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         Error_Location : Bare_Lkt_Node;
   end record;

   
      overriding function Call
        (Self : Type_Decl_P_Is_Bool_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;

   
      overriding procedure Failed
        (Self : Type_Decl_P_Is_Bool_0_Predicate;
            Entity : Internal_Entity;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   ;

   overriding function Image (Self : Type_Decl_P_Is_Bool_0_Predicate) return String;

   

   function Create_Type_Decl_P_Is_Bool_0_Predicate
   (
         Error_Location : Bare_Lkt_Node
   )
      return Type_Decl_P_Is_Bool_0_Predicate is
   begin
      
      return Type_Decl_P_Is_Bool_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Error_Location => Error_Location);
   end;

   ----------
   -- Call --
   ----------

   
      overriding function Call
        (Self : Type_Decl_P_Is_Bool_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   
   is
         pragma Unreferenced (Self);



      
      Node : Bare_Type_Decl;

      Ret : Boolean;
   begin

      
            if Entity.Node /= null
               and then Entity.Node.Kind
                        not in Lkt_Type_Decl
            then
               Raise_Property_Exception
                 (Node, Property_Error'Identity, "mismatching node type");
            end if;

      Node := Entity.Node;


      

      Ret := Type_Decl_P_Is_Bool (Node, E_Info => Entity.Info);


      return Ret;
   end Call;

   ------------
   -- Failed --
   ------------

   
      overriding procedure Failed
        (Self : Type_Decl_P_Is_Bool_0_Predicate;
            Entity : Internal_Entity;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   
   is
      
      Args : Internal_Entity_Array_Access :=
         Create_Internal_Entity_Array (1);

      Contexts : Internal_Logic_Context_Array_Access :=
         Create_Internal_Logic_Context_Array (Ctxs'Length);

      Diag : constant Internal_Solver_Diagnostic :=
        (Message_Template => Create_String ("operands of binary logic operators must be of boolean type, got {}"),
         Args             => Args,
         Contexts         => Contexts,
         Location         => Self.Error_Location,
         Round            => Round);
   begin
      Args.Items (1) := Entity;
      for I in Ctxs'Range loop
         Contexts.Items (I) := Ctxs (I).all;
      end loop;
      Emitter (Diag);
   end Failed;

   -----------
   -- Image --
   -----------

   overriding function Image (Self : Type_Decl_P_Is_Bool_0_Predicate) return String is
   begin
      return "TypeDecl.is_bool";
   end Image;



         

         

   

   

   

   type Type_Decl_P_Is_String_Or_Array_Type_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         Error_Location : Bare_Lkt_Node;
   end record;

   
      overriding function Call
        (Self : Type_Decl_P_Is_String_Or_Array_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;

   
      overriding procedure Failed
        (Self : Type_Decl_P_Is_String_Or_Array_Type_0_Predicate;
            Entity : Internal_Entity;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   ;

   overriding function Image (Self : Type_Decl_P_Is_String_Or_Array_Type_0_Predicate) return String;

   

   function Create_Type_Decl_P_Is_String_Or_Array_Type_0_Predicate
   (
         Error_Location : Bare_Lkt_Node
   )
      return Type_Decl_P_Is_String_Or_Array_Type_0_Predicate is
   begin
      
      return Type_Decl_P_Is_String_Or_Array_Type_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Error_Location => Error_Location);
   end;

   ----------
   -- Call --
   ----------

   
      overriding function Call
        (Self : Type_Decl_P_Is_String_Or_Array_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   
   is
         pragma Unreferenced (Self);



      
      Node : Bare_Type_Decl;

      Ret : Boolean;
   begin

      
            if Entity.Node /= null
               and then Entity.Node.Kind
                        not in Lkt_Type_Decl
            then
               Raise_Property_Exception
                 (Node, Property_Error'Identity, "mismatching node type");
            end if;

      Node := Entity.Node;


      

      Ret := Type_Decl_P_Is_String_Or_Array_Type (Node, E_Info => Entity.Info);


      return Ret;
   end Call;

   ------------
   -- Failed --
   ------------

   
      overriding procedure Failed
        (Self : Type_Decl_P_Is_String_Or_Array_Type_0_Predicate;
            Entity : Internal_Entity;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   
   is
      
      Args : Internal_Entity_Array_Access :=
         Create_Internal_Entity_Array (1);

      Contexts : Internal_Logic_Context_Array_Access :=
         Create_Internal_Logic_Context_Array (Ctxs'Length);

      Diag : constant Internal_Solver_Diagnostic :=
        (Message_Template => Create_String ("expression must be a string or an array, got {}"),
         Args             => Args,
         Contexts         => Contexts,
         Location         => Self.Error_Location,
         Round            => Round);
   begin
      Args.Items (1) := Entity;
      for I in Ctxs'Range loop
         Contexts.Items (I) := Ctxs (I).all;
      end loop;
      Emitter (Diag);
   end Failed;

   -----------
   -- Image --
   -----------

   overriding function Image (Self : Type_Decl_P_Is_String_Or_Array_Type_0_Predicate) return String is
   begin
      return "TypeDecl.is_string_or_array_type";
   end Image;



         

         

   

   

   

   type Type_Decl_P_Is_Int_Type_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         Error_Location : Bare_Lkt_Node;
   end record;

   
      overriding function Call
        (Self : Type_Decl_P_Is_Int_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;

   
      overriding procedure Failed
        (Self : Type_Decl_P_Is_Int_Type_0_Predicate;
            Entity : Internal_Entity;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   ;

   overriding function Image (Self : Type_Decl_P_Is_Int_Type_0_Predicate) return String;

   

   function Create_Type_Decl_P_Is_Int_Type_0_Predicate
   (
         Error_Location : Bare_Lkt_Node
   )
      return Type_Decl_P_Is_Int_Type_0_Predicate is
   begin
      
      return Type_Decl_P_Is_Int_Type_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Error_Location => Error_Location);
   end;

   ----------
   -- Call --
   ----------

   
      overriding function Call
        (Self : Type_Decl_P_Is_Int_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   
   is
         pragma Unreferenced (Self);



      
      Node : Bare_Type_Decl;

      Ret : Boolean;
   begin

      
            if Entity.Node /= null
               and then Entity.Node.Kind
                        not in Lkt_Type_Decl
            then
               Raise_Property_Exception
                 (Node, Property_Error'Identity, "mismatching node type");
            end if;

      Node := Entity.Node;


      

      Ret := Type_Decl_P_Is_Int_Type (Node, E_Info => Entity.Info);


      return Ret;
   end Call;

   ------------
   -- Failed --
   ------------

   
      overriding procedure Failed
        (Self : Type_Decl_P_Is_Int_Type_0_Predicate;
            Entity : Internal_Entity;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   
   is
      
      Args : Internal_Entity_Array_Access :=
         Create_Internal_Entity_Array (1);

      Contexts : Internal_Logic_Context_Array_Access :=
         Create_Internal_Logic_Context_Array (Ctxs'Length);

      Diag : constant Internal_Solver_Diagnostic :=
        (Message_Template => Create_String ("expression must be of type Int or BigInt, got {}"),
         Args             => Args,
         Contexts         => Contexts,
         Location         => Self.Error_Location,
         Round            => Round);
   begin
      Args.Items (1) := Entity;
      for I in Ctxs'Range loop
         Contexts.Items (I) := Ctxs (I).all;
      end loop;
      Emitter (Diag);
   end Failed;

   -----------
   -- Image --
   -----------

   overriding function Image (Self : Type_Decl_P_Is_Int_Type_0_Predicate) return String is
   begin
      return "TypeDecl.is_int_type";
   end Image;



         

         

   

   

   

   type Type_Decl_P_Is_Int_Or_Node_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         Error_Location : Bare_Lkt_Node;
   end record;

   
      overriding function Call
        (Self : Type_Decl_P_Is_Int_Or_Node_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;

   
      overriding procedure Failed
        (Self : Type_Decl_P_Is_Int_Or_Node_0_Predicate;
            Entity : Internal_Entity;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   ;

   overriding function Image (Self : Type_Decl_P_Is_Int_Or_Node_0_Predicate) return String;

   

   function Create_Type_Decl_P_Is_Int_Or_Node_0_Predicate
   (
         Error_Location : Bare_Lkt_Node
   )
      return Type_Decl_P_Is_Int_Or_Node_0_Predicate is
   begin
      
      return Type_Decl_P_Is_Int_Or_Node_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Error_Location => Error_Location);
   end;

   ----------
   -- Call --
   ----------

   
      overriding function Call
        (Self : Type_Decl_P_Is_Int_Or_Node_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   
   is
         pragma Unreferenced (Self);



      
      Node : Bare_Type_Decl;

      Ret : Boolean;
   begin

      
            if Entity.Node /= null
               and then Entity.Node.Kind
                        not in Lkt_Type_Decl
            then
               Raise_Property_Exception
                 (Node, Property_Error'Identity, "mismatching node type");
            end if;

      Node := Entity.Node;


      

      Ret := Type_Decl_P_Is_Int_Or_Node (Node, E_Info => Entity.Info);


      return Ret;
   end Call;

   ------------
   -- Failed --
   ------------

   
      overriding procedure Failed
        (Self : Type_Decl_P_Is_Int_Or_Node_0_Predicate;
            Entity : Internal_Entity;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   
   is
      
      Args : Internal_Entity_Array_Access :=
         Create_Internal_Entity_Array (1);

      Contexts : Internal_Logic_Context_Array_Access :=
         Create_Internal_Logic_Context_Array (Ctxs'Length);

      Diag : constant Internal_Solver_Diagnostic :=
        (Message_Template => Create_String ("expression must be of type Int, BigInt or Node, got {}"),
         Args             => Args,
         Contexts         => Contexts,
         Location         => Self.Error_Location,
         Round            => Round);
   begin
      Args.Items (1) := Entity;
      for I in Ctxs'Range loop
         Contexts.Items (I) := Ctxs (I).all;
      end loop;
      Emitter (Diag);
   end Failed;

   -----------
   -- Image --
   -----------

   overriding function Image (Self : Type_Decl_P_Is_Int_Or_Node_0_Predicate) return String is
   begin
      return "TypeDecl.is_int_or_node";
   end Image;



         

         

         

         

   

   

   

   type Type_Decl_P_Is_Subtype_Or_Eq_0_Predicate is
   new Solver_Ifc.N_Predicate_Type
   with record
         Field_0 : Boolean;
         Error_Location : Bare_Lkt_Node;
   end record;

   
      overriding function Call
        (Self : Type_Decl_P_Is_Subtype_Or_Eq_0_Predicate;
            Entities : Entity_Vars.Value_Array
        ) return Boolean
   ;

   
      overriding procedure Failed
        (Self : Type_Decl_P_Is_Subtype_Or_Eq_0_Predicate;
            Entities : Entity_Vars.Value_Array;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   ;

   overriding function Image (Self : Type_Decl_P_Is_Subtype_Or_Eq_0_Predicate) return String;

   

   function Create_Type_Decl_P_Is_Subtype_Or_Eq_0_Predicate
   (
         N : Positive;
         Allow_Entity : Boolean;
         Error_Location : Bare_Lkt_Node
   )
      return Type_Decl_P_Is_Subtype_Or_Eq_0_Predicate is
   begin
      
         
      return Type_Decl_P_Is_Subtype_Or_Eq_0_Predicate'(N => N, Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Error_Location => Error_Location, Field_0 => Allow_Entity);
   end;

   ----------
   -- Call --
   ----------

   
      overriding function Call
        (Self : Type_Decl_P_Is_Subtype_Or_Eq_0_Predicate;
            Entities : Entity_Vars.Value_Array
        ) return Boolean
   
   is

         Entity : Internal_Entity := Entities (1);


      
      Node : Bare_Type_Decl;

      Ret : Boolean;
   begin

      
            if Entity.Node /= null
               and then Entity.Node.Kind
                        not in Lkt_Type_Decl
            then
               Raise_Property_Exception
                 (Node, Property_Error'Identity, "mismatching node type");
            end if;
            if Entities (2).Node /= null
               and then Entities (2).Node.Kind
                        not in Lkt_Type_Decl
            then
               Raise_Property_Exception
                 (Node, Property_Error'Identity, "mismatching node type");
            end if;

      Node := Entity.Node;


      

      Ret := Type_Decl_P_Is_Subtype_Or_Eq (Node, (Node => Entities (2).Node, Info => Entities (2).Info), Allow_Entity => Self.Field_0, E_Info => Entity.Info);


      return Ret;
   end Call;

   ------------
   -- Failed --
   ------------

   
      overriding procedure Failed
        (Self : Type_Decl_P_Is_Subtype_Or_Eq_0_Predicate;
            Entities : Entity_Vars.Value_Array;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   
   is
      
      Args : Internal_Entity_Array_Access :=
         Create_Internal_Entity_Array (2);

      Contexts : Internal_Logic_Context_Array_Access :=
         Create_Internal_Logic_Context_Array (Ctxs'Length);

      Diag : constant Internal_Solver_Diagnostic :=
        (Message_Template => Create_String ("{} is not a subtype of {}"),
         Args             => Args,
         Contexts         => Contexts,
         Location         => Self.Error_Location,
         Round            => Round);
   begin
      Args.Items (1) := Entities (2);
      Args.Items (2) := Entities (1);
      for I in Ctxs'Range loop
         Contexts.Items (I) := Ctxs (I).all;
      end loop;
      Emitter (Diag);
   end Failed;

   -----------
   -- Image --
   -----------

   overriding function Image (Self : Type_Decl_P_Is_Subtype_Or_Eq_0_Predicate) return String is
   begin
      return "TypeDecl.is_subtype_or_eq";
   end Image;



         

         

         

         

         

   

   type Type_Decl_P_Imprecise_Common_Ancestor_List_0_Functor is new Solver_Ifc.Combiner_Type with
      null record;

   overriding function Combine (Self : Type_Decl_P_Imprecise_Common_Ancestor_List_0_Functor; Vals : Entity_Vars.Value_Array) return Internal_Entity with Inline;
   overriding function Image (Self : Type_Decl_P_Imprecise_Common_Ancestor_List_0_Functor) return String;


   

   function Create_Type_Decl_P_Imprecise_Common_Ancestor_List_0_Functor
   (
         N : Positive
   )
      return Type_Decl_P_Imprecise_Common_Ancestor_List_0_Functor is
   begin
      
      return Type_Decl_P_Imprecise_Common_Ancestor_List_0_Functor'(N => N, Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1);
   end;

   overriding function Combine (Self : Type_Decl_P_Imprecise_Common_Ancestor_List_0_Functor; Vals : Entity_Vars.Value_Array) return Internal_Entity is
         pragma Unreferenced (Self);

         From : constant Internal_Entity := Vals (1);

         
         Args : Internal_Entity_Type_Decl_Array_Access :=
           Create_Internal_Entity_Type_Decl_Array (Vals'Length - 1);

      

      Ret : Internal_Entity_Type_Decl;
   begin
            if From.Node /= null
               and then From.Node.Kind not in
                 Lkt_Type_Decl
            then
               Raise_Property_Exception
                 (From.Node,
                  Property_Error'Identity,
                  "mismatching node type for comb_prop");
            end if;

      
      for I in 2 .. Vals'Last loop
            if Vals (I).Node /= null
               and then Vals (I).Node.Kind not in
                  Lkt_Type_Decl
            then
               Raise_Property_Exception
                 (Vals (I).Node,
                  Property_Error'Identity,
                  "mismatching node type for comb_prop");
            end if;
         Args.Items (I - 1) := (Vals (I).Node, Vals (I).Info);
      end loop;

      
      Ret := Type_Decl_P_Imprecise_Common_Ancestor_List (Node => From.Node, Other_Types => Args, E_Info => From.Info);

      Dec_Ref (Args);

      return (Node => Ret.Node, Info => Ret.Info);
   exception
      when Exc : Property_Error =>
         pragma Unreferenced (Exc);
         Dec_Ref (Args);
         raise;
   end Combine;

   -----------
   -- Image --
   -----------

   overriding function Image (Self : Type_Decl_P_Imprecise_Common_Ancestor_List_0_Functor) return String is
   begin
      return ("TypeDecl.imprecise_common_ancestor_list");
   end Image;



         

   

   

   

   type Type_Decl_P_Commutative_Matching_Type_0_Predicate is
   new Solver_Ifc.N_Predicate_Type
   with record
         Error_Location : Bare_Lkt_Node;
   end record;

   
      overriding function Call
        (Self : Type_Decl_P_Commutative_Matching_Type_0_Predicate;
            Entities : Entity_Vars.Value_Array
        ) return Boolean
   ;

   
      overriding procedure Failed
        (Self : Type_Decl_P_Commutative_Matching_Type_0_Predicate;
            Entities : Entity_Vars.Value_Array;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   ;

   overriding function Image (Self : Type_Decl_P_Commutative_Matching_Type_0_Predicate) return String;

   

   function Create_Type_Decl_P_Commutative_Matching_Type_0_Predicate
   (
         N : Positive;
         Error_Location : Bare_Lkt_Node
   )
      return Type_Decl_P_Commutative_Matching_Type_0_Predicate is
   begin
      
      return Type_Decl_P_Commutative_Matching_Type_0_Predicate'(N => N, Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Error_Location => Error_Location);
   end;

   ----------
   -- Call --
   ----------

   
      overriding function Call
        (Self : Type_Decl_P_Commutative_Matching_Type_0_Predicate;
            Entities : Entity_Vars.Value_Array
        ) return Boolean
   
   is
         pragma Unreferenced (Self);

         Entity : Internal_Entity := Entities (1);


      
      Node : Bare_Type_Decl;

      Ret : Boolean;
   begin

      
            if Entity.Node /= null
               and then Entity.Node.Kind
                        not in Lkt_Type_Decl
            then
               Raise_Property_Exception
                 (Node, Property_Error'Identity, "mismatching node type");
            end if;
            if Entities (2).Node /= null
               and then Entities (2).Node.Kind
                        not in Lkt_Type_Decl
            then
               Raise_Property_Exception
                 (Node, Property_Error'Identity, "mismatching node type");
            end if;

      Node := Entity.Node;


      

      Ret := Type_Decl_P_Commutative_Matching_Type (Node, (Node => Entities (2).Node, Info => Entities (2).Info), E_Info => Entity.Info);


      return Ret;
   end Call;

   ------------
   -- Failed --
   ------------

   
      overriding procedure Failed
        (Self : Type_Decl_P_Commutative_Matching_Type_0_Predicate;
            Entities : Entity_Vars.Value_Array;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   
   is
      
      Args : Internal_Entity_Array_Access :=
         Create_Internal_Entity_Array (2);

      Contexts : Internal_Logic_Context_Array_Access :=
         Create_Internal_Logic_Context_Array (Ctxs'Length);

      Diag : constant Internal_Solver_Diagnostic :=
        (Message_Template => Create_String ("expected {}, got {}"),
         Args             => Args,
         Contexts         => Contexts,
         Location         => Self.Error_Location,
         Round            => Round);
   begin
      Args.Items (1) := Entities (1);
      Args.Items (2) := Entities (2);
      for I in Ctxs'Range loop
         Contexts.Items (I) := Ctxs (I).all;
      end loop;
      Emitter (Diag);
   end Failed;

   -----------
   -- Image --
   -----------

   overriding function Image (Self : Type_Decl_P_Commutative_Matching_Type_0_Predicate) return String is
   begin
      return "TypeDecl.commutative_matching_type";
   end Image;



   

   

   

   type Type_Decl_P_Commutative_Matching_Type_1_Predicate is
   new Solver_Ifc.N_Predicate_Type
   with record
         Field_0 : Boolean;
         Error_Location : Bare_Lkt_Node;
   end record;

   
      overriding function Call
        (Self : Type_Decl_P_Commutative_Matching_Type_1_Predicate;
            Entities : Entity_Vars.Value_Array
        ) return Boolean
   ;

   
      overriding procedure Failed
        (Self : Type_Decl_P_Commutative_Matching_Type_1_Predicate;
            Entities : Entity_Vars.Value_Array;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   ;

   overriding function Image (Self : Type_Decl_P_Commutative_Matching_Type_1_Predicate) return String;

   

   function Create_Type_Decl_P_Commutative_Matching_Type_1_Predicate
   (
         N : Positive;
         Allow_Common_Ancestor : Boolean;
         Error_Location : Bare_Lkt_Node
   )
      return Type_Decl_P_Commutative_Matching_Type_1_Predicate is
   begin
      
         
      return Type_Decl_P_Commutative_Matching_Type_1_Predicate'(N => N, Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Error_Location => Error_Location, Field_0 => Allow_Common_Ancestor);
   end;

   ----------
   -- Call --
   ----------

   
      overriding function Call
        (Self : Type_Decl_P_Commutative_Matching_Type_1_Predicate;
            Entities : Entity_Vars.Value_Array
        ) return Boolean
   
   is

         Entity : Internal_Entity := Entities (1);


      
      Node : Bare_Type_Decl;

      Ret : Boolean;
   begin

      
            if Entity.Node /= null
               and then Entity.Node.Kind
                        not in Lkt_Type_Decl
            then
               Raise_Property_Exception
                 (Node, Property_Error'Identity, "mismatching node type");
            end if;
            if Entities (2).Node /= null
               and then Entities (2).Node.Kind
                        not in Lkt_Type_Decl
            then
               Raise_Property_Exception
                 (Node, Property_Error'Identity, "mismatching node type");
            end if;

      Node := Entity.Node;


      

      Ret := Type_Decl_P_Commutative_Matching_Type (Node, (Node => Entities (2).Node, Info => Entities (2).Info), Allow_Common_Ancestor => Self.Field_0, E_Info => Entity.Info);


      return Ret;
   end Call;

   ------------
   -- Failed --
   ------------

   
      overriding procedure Failed
        (Self : Type_Decl_P_Commutative_Matching_Type_1_Predicate;
            Entities : Entity_Vars.Value_Array;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   
   is
      
      Args : Internal_Entity_Array_Access :=
         Create_Internal_Entity_Array (2);

      Contexts : Internal_Logic_Context_Array_Access :=
         Create_Internal_Logic_Context_Array (Ctxs'Length);

      Diag : constant Internal_Solver_Diagnostic :=
        (Message_Template => Create_String ("expected {}, got {}"),
         Args             => Args,
         Contexts         => Contexts,
         Location         => Self.Error_Location,
         Round            => Round);
   begin
      Args.Items (1) := Entities (1);
      Args.Items (2) := Entities (2);
      for I in Ctxs'Range loop
         Contexts.Items (I) := Ctxs (I).all;
      end loop;
      Emitter (Diag);
   end Failed;

   -----------
   -- Image --
   -----------

   overriding function Image (Self : Type_Decl_P_Commutative_Matching_Type_1_Predicate) return String is
   begin
      return "TypeDecl.commutative_matching_type";
   end Image;



         

         

   

   

   

   type Type_Decl_P_Could_Determine_Type_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         Error_Location : Bare_Lkt_Node;
   end record;

   
      overriding function Call
        (Self : Type_Decl_P_Could_Determine_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;

   
      overriding procedure Failed
        (Self : Type_Decl_P_Could_Determine_Type_0_Predicate;
            Entity : Internal_Entity;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   ;

   overriding function Image (Self : Type_Decl_P_Could_Determine_Type_0_Predicate) return String;

   

   function Create_Type_Decl_P_Could_Determine_Type_0_Predicate
   (
         Error_Location : Bare_Lkt_Node
   )
      return Type_Decl_P_Could_Determine_Type_0_Predicate is
   begin
      
      return Type_Decl_P_Could_Determine_Type_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Error_Location => Error_Location);
   end;

   ----------
   -- Call --
   ----------

   
      overriding function Call
        (Self : Type_Decl_P_Could_Determine_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   
   is
         pragma Unreferenced (Self);



      
      Node : Bare_Type_Decl;

      Ret : Boolean;
   begin

      
            if Entity.Node /= null
               and then Entity.Node.Kind
                        not in Lkt_Type_Decl
            then
               Raise_Property_Exception
                 (Node, Property_Error'Identity, "mismatching node type");
            end if;

      Node := Entity.Node;


      

      Ret := Type_Decl_P_Could_Determine_Type (Node);


      return Ret;
   end Call;

   ------------
   -- Failed --
   ------------

   
      overriding procedure Failed
        (Self : Type_Decl_P_Could_Determine_Type_0_Predicate;
            Entity : Internal_Entity;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   
   is
      
      Args : Internal_Entity_Array_Access :=
         Create_Internal_Entity_Array (0);

      Contexts : Internal_Logic_Context_Array_Access :=
         Create_Internal_Logic_Context_Array (Ctxs'Length);

      Diag : constant Internal_Solver_Diagnostic :=
        (Message_Template => Create_String ("could not determine type"),
         Args             => Args,
         Contexts         => Contexts,
         Location         => Self.Error_Location,
         Round            => Round);
   begin
      for I in Ctxs'Range loop
         Contexts.Items (I) := Ctxs (I).all;
      end loop;
      Emitter (Diag);
   end Failed;

   -----------
   -- Image --
   -----------

   overriding function Image (Self : Type_Decl_P_Could_Determine_Type_0_Predicate) return String is
   begin
      return "TypeDecl.could_determine_type";
   end Image;



         

         

         

         

   

   

   

   type Type_Decl_P_Matching_Type_0_Predicate is
   new Solver_Ifc.N_Predicate_Type
   with record
         Error_Location : Bare_Lkt_Node;
   end record;

   
      overriding function Call
        (Self : Type_Decl_P_Matching_Type_0_Predicate;
            Entities : Entity_Vars.Value_Array
        ) return Boolean
   ;

   
      overriding procedure Failed
        (Self : Type_Decl_P_Matching_Type_0_Predicate;
            Entities : Entity_Vars.Value_Array;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   ;

   overriding function Image (Self : Type_Decl_P_Matching_Type_0_Predicate) return String;

   

   function Create_Type_Decl_P_Matching_Type_0_Predicate
   (
         N : Positive;
         Error_Location : Bare_Lkt_Node
   )
      return Type_Decl_P_Matching_Type_0_Predicate is
   begin
      
      return Type_Decl_P_Matching_Type_0_Predicate'(N => N, Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Error_Location => Error_Location);
   end;

   ----------
   -- Call --
   ----------

   
      overriding function Call
        (Self : Type_Decl_P_Matching_Type_0_Predicate;
            Entities : Entity_Vars.Value_Array
        ) return Boolean
   
   is
         pragma Unreferenced (Self);

         Entity : Internal_Entity := Entities (1);


      
      Node : Bare_Type_Decl;

      Ret : Boolean;
   begin

      
            if Entity.Node /= null
               and then Entity.Node.Kind
                        not in Lkt_Type_Decl
            then
               Raise_Property_Exception
                 (Node, Property_Error'Identity, "mismatching node type");
            end if;
            if Entities (2).Node /= null
               and then Entities (2).Node.Kind
                        not in Lkt_Type_Decl
            then
               Raise_Property_Exception
                 (Node, Property_Error'Identity, "mismatching node type");
            end if;

      Node := Entity.Node;


      

      Ret := Type_Decl_P_Matching_Type (Node, (Node => Entities (2).Node, Info => Entities (2).Info), E_Info => Entity.Info);


      return Ret;
   end Call;

   ------------
   -- Failed --
   ------------

   
      overriding procedure Failed
        (Self : Type_Decl_P_Matching_Type_0_Predicate;
            Entities : Entity_Vars.Value_Array;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   
   is
      
      Args : Internal_Entity_Array_Access :=
         Create_Internal_Entity_Array (2);

      Contexts : Internal_Logic_Context_Array_Access :=
         Create_Internal_Logic_Context_Array (Ctxs'Length);

      Diag : constant Internal_Solver_Diagnostic :=
        (Message_Template => Create_String ("expected {}, got {}"),
         Args             => Args,
         Contexts         => Contexts,
         Location         => Self.Error_Location,
         Round            => Round);
   begin
      Args.Items (1) := Entities (1);
      Args.Items (2) := Entities (2);
      for I in Ctxs'Range loop
         Contexts.Items (I) := Ctxs (I).all;
      end loop;
      Emitter (Diag);
   end Failed;

   -----------
   -- Image --
   -----------

   overriding function Image (Self : Type_Decl_P_Matching_Type_0_Predicate) return String is
   begin
      return "TypeDecl.matching_type";
   end Image;



         

         

   

   

   

   type Type_Decl_P_Matching_Logic_Type_0_Predicate is
   new Solver_Ifc.N_Predicate_Type
   with record
         Error_Location : Bare_Lkt_Node;
   end record;

   
      overriding function Call
        (Self : Type_Decl_P_Matching_Logic_Type_0_Predicate;
            Entities : Entity_Vars.Value_Array
        ) return Boolean
   ;

   
      overriding procedure Failed
        (Self : Type_Decl_P_Matching_Logic_Type_0_Predicate;
            Entities : Entity_Vars.Value_Array;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   ;

   overriding function Image (Self : Type_Decl_P_Matching_Logic_Type_0_Predicate) return String;

   

   function Create_Type_Decl_P_Matching_Logic_Type_0_Predicate
   (
         N : Positive;
         Error_Location : Bare_Lkt_Node
   )
      return Type_Decl_P_Matching_Logic_Type_0_Predicate is
   begin
      
      return Type_Decl_P_Matching_Logic_Type_0_Predicate'(N => N, Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Error_Location => Error_Location);
   end;

   ----------
   -- Call --
   ----------

   
      overriding function Call
        (Self : Type_Decl_P_Matching_Logic_Type_0_Predicate;
            Entities : Entity_Vars.Value_Array
        ) return Boolean
   
   is
         pragma Unreferenced (Self);

         Entity : Internal_Entity := Entities (1);


      
      Node : Bare_Type_Decl;

      Ret : Boolean;
   begin

      
            if Entity.Node /= null
               and then Entity.Node.Kind
                        not in Lkt_Type_Decl
            then
               Raise_Property_Exception
                 (Node, Property_Error'Identity, "mismatching node type");
            end if;
            if Entities (2).Node /= null
               and then Entities (2).Node.Kind
                        not in Lkt_Type_Decl
            then
               Raise_Property_Exception
                 (Node, Property_Error'Identity, "mismatching node type");
            end if;

      Node := Entity.Node;


      

      Ret := Type_Decl_P_Matching_Logic_Type (Node, (Node => Entities (2).Node, Info => Entities (2).Info), E_Info => Entity.Info);


      return Ret;
   end Call;

   ------------
   -- Failed --
   ------------

   
      overriding procedure Failed
        (Self : Type_Decl_P_Matching_Logic_Type_0_Predicate;
            Entities : Entity_Vars.Value_Array;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   
   is
      
      Args : Internal_Entity_Array_Access :=
         Create_Internal_Entity_Array (2);

      Contexts : Internal_Logic_Context_Array_Access :=
         Create_Internal_Logic_Context_Array (Ctxs'Length);

      Diag : constant Internal_Solver_Diagnostic :=
        (Message_Template => Create_String ("expected {}, got {}"),
         Args             => Args,
         Contexts         => Contexts,
         Location         => Self.Error_Location,
         Round            => Round);
   begin
      Args.Items (1) := Entities (1);
      Args.Items (2) := Entities (2);
      for I in Ctxs'Range loop
         Contexts.Items (I) := Ctxs (I).all;
      end loop;
      Emitter (Diag);
   end Failed;

   -----------
   -- Image --
   -----------

   overriding function Image (Self : Type_Decl_P_Matching_Logic_Type_0_Predicate) return String is
   begin
      return "TypeDecl.matching_logic_type";
   end Image;



         

         

   

   

   

   type Type_Decl_P_Is_Of_Array_Type_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         Error_Location : Bare_Lkt_Node;
   end record;

   
      overriding function Call
        (Self : Type_Decl_P_Is_Of_Array_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;

   
      overriding procedure Failed
        (Self : Type_Decl_P_Is_Of_Array_Type_0_Predicate;
            Entity : Internal_Entity;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   ;

   overriding function Image (Self : Type_Decl_P_Is_Of_Array_Type_0_Predicate) return String;

   

   function Create_Type_Decl_P_Is_Of_Array_Type_0_Predicate
   (
         Error_Location : Bare_Lkt_Node
   )
      return Type_Decl_P_Is_Of_Array_Type_0_Predicate is
   begin
      
      return Type_Decl_P_Is_Of_Array_Type_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Error_Location => Error_Location);
   end;

   ----------
   -- Call --
   ----------

   
      overriding function Call
        (Self : Type_Decl_P_Is_Of_Array_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   
   is
         pragma Unreferenced (Self);



      
      Node : Bare_Type_Decl;

      Ret : Boolean;
   begin

      
            if Entity.Node /= null
               and then Entity.Node.Kind
                        not in Lkt_Type_Decl
            then
               Raise_Property_Exception
                 (Node, Property_Error'Identity, "mismatching node type");
            end if;

      Node := Entity.Node;


      

      Ret := Type_Decl_P_Is_Of_Array_Type (Node, E_Info => Entity.Info);


      return Ret;
   end Call;

   ------------
   -- Failed --
   ------------

   
      overriding procedure Failed
        (Self : Type_Decl_P_Is_Of_Array_Type_0_Predicate;
            Entity : Internal_Entity;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   
   is
      
      Args : Internal_Entity_Array_Access :=
         Create_Internal_Entity_Array (1);

      Contexts : Internal_Logic_Context_Array_Access :=
         Create_Internal_Logic_Context_Array (Ctxs'Length);

      Diag : constant Internal_Solver_Diagnostic :=
        (Message_Template => Create_String ("{} is not an array"),
         Args             => Args,
         Contexts         => Contexts,
         Location         => Self.Error_Location,
         Round            => Round);
   begin
      Args.Items (1) := Entity;
      for I in Ctxs'Range loop
         Contexts.Items (I) := Ctxs (I).all;
      end loop;
      Emitter (Diag);
   end Failed;

   -----------
   -- Image --
   -----------

   overriding function Image (Self : Type_Decl_P_Is_Of_Array_Type_0_Predicate) return String is
   begin
      return "TypeDecl.is_of_array_type";
   end Image;



         

         

   

   

   

   type Type_Decl_P_Is_Callable_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         Error_Location : Bare_Lkt_Node;
   end record;

   
      overriding function Call
        (Self : Type_Decl_P_Is_Callable_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;

   
      overriding procedure Failed
        (Self : Type_Decl_P_Is_Callable_0_Predicate;
            Entity : Internal_Entity;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   ;

   overriding function Image (Self : Type_Decl_P_Is_Callable_0_Predicate) return String;

   

   function Create_Type_Decl_P_Is_Callable_0_Predicate
   (
         Error_Location : Bare_Lkt_Node
   )
      return Type_Decl_P_Is_Callable_0_Predicate is
   begin
      
      return Type_Decl_P_Is_Callable_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Error_Location => Error_Location);
   end;

   ----------
   -- Call --
   ----------

   
      overriding function Call
        (Self : Type_Decl_P_Is_Callable_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   
   is
         pragma Unreferenced (Self);



      
      Node : Bare_Type_Decl;

      Ret : Boolean;
   begin

      
            if Entity.Node /= null
               and then Entity.Node.Kind
                        not in Lkt_Type_Decl
            then
               Raise_Property_Exception
                 (Node, Property_Error'Identity, "mismatching node type");
            end if;

      Node := Entity.Node;


      

      Ret := Type_Decl_P_Is_Callable (Node, E_Info => Entity.Info);


      return Ret;
   end Call;

   ------------
   -- Failed --
   ------------

   
      overriding procedure Failed
        (Self : Type_Decl_P_Is_Callable_0_Predicate;
            Entity : Internal_Entity;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   
   is
      
      Args : Internal_Entity_Array_Access :=
         Create_Internal_Entity_Array (1);

      Contexts : Internal_Logic_Context_Array_Access :=
         Create_Internal_Logic_Context_Array (Ctxs'Length);

      Diag : constant Internal_Solver_Diagnostic :=
        (Message_Template => Create_String ("{} is not callable"),
         Args             => Args,
         Contexts         => Contexts,
         Location         => Self.Error_Location,
         Round            => Round);
   begin
      Args.Items (1) := Entity;
      for I in Ctxs'Range loop
         Contexts.Items (I) := Ctxs (I).all;
      end loop;
      Emitter (Diag);
   end Failed;

   -----------
   -- Image --
   -----------

   overriding function Image (Self : Type_Decl_P_Is_Callable_0_Predicate) return String is
   begin
      return "TypeDecl.is_callable";
   end Image;



         

         

         

   

   type Type_Decl_P_Match_Param_Get_Type_0_Functor is new Solver_Ifc.Converter_Type with
      record
            Current_Name : Internal_Entity_Argument;
            In_Logic_Call : Boolean;
      end record;

   overriding function Convert (Self : Type_Decl_P_Match_Param_Get_Type_0_Functor; From : Internal_Entity) return Internal_Entity with Inline;
   overriding function Image (Self : Type_Decl_P_Match_Param_Get_Type_0_Functor) return String;


   

   function Create_Type_Decl_P_Match_Param_Get_Type_0_Functor
   (
         Current_Name : Internal_Entity_Argument;
         In_Logic_Call : Boolean
   )
      return Type_Decl_P_Match_Param_Get_Type_0_Functor is
   begin
      
         
         
      return Type_Decl_P_Match_Param_Get_Type_0_Functor'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Current_Name => Current_Name, In_Logic_Call => In_Logic_Call);
   end;

   overriding function Convert (Self : Type_Decl_P_Match_Param_Get_Type_0_Functor; From : Internal_Entity) return Internal_Entity is



      

      Ret : Internal_Entity_Decl;
   begin
            if From.Node /= null
               and then From.Node.Kind not in
                 Lkt_Type_Decl
            then
               Raise_Property_Exception
                 (From.Node,
                  Property_Error'Identity,
                  "mismatching node type for conv_prop");
            end if;


      
      Ret := Type_Decl_P_Match_Param_Get_Type (Node => From.Node, Current_Name => Self.Current_Name, In_Logic_Call => Self.In_Logic_Call, E_Info => From.Info);


      return (Node => Ret.Node, Info => Ret.Info);
   end Convert;

   -----------
   -- Image --
   -----------

   overriding function Image (Self : Type_Decl_P_Match_Param_Get_Type_0_Functor) return String is
   begin
      return ("TypeDecl.match_param_get_type");
   end Image;



         

         

   

   type Type_Decl_P_Match_Param_Get_Decl_0_Functor is new Solver_Ifc.Converter_Type with
      record
            Current_Name : Internal_Entity_Argument;
            In_Logic_Call : Boolean;
      end record;

   overriding function Convert (Self : Type_Decl_P_Match_Param_Get_Decl_0_Functor; From : Internal_Entity) return Internal_Entity with Inline;
   overriding function Image (Self : Type_Decl_P_Match_Param_Get_Decl_0_Functor) return String;


   

   function Create_Type_Decl_P_Match_Param_Get_Decl_0_Functor
   (
         Current_Name : Internal_Entity_Argument;
         In_Logic_Call : Boolean
   )
      return Type_Decl_P_Match_Param_Get_Decl_0_Functor is
   begin
      
         
         
      return Type_Decl_P_Match_Param_Get_Decl_0_Functor'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Current_Name => Current_Name, In_Logic_Call => In_Logic_Call);
   end;

   overriding function Convert (Self : Type_Decl_P_Match_Param_Get_Decl_0_Functor; From : Internal_Entity) return Internal_Entity is



      

      Ret : Internal_Entity_Decl;
   begin
            if From.Node /= null
               and then From.Node.Kind not in
                 Lkt_Type_Decl
            then
               Raise_Property_Exception
                 (From.Node,
                  Property_Error'Identity,
                  "mismatching node type for conv_prop");
            end if;


      
      Ret := Type_Decl_P_Match_Param_Get_Decl (Node => From.Node, Current_Name => Self.Current_Name, In_Logic_Call => Self.In_Logic_Call, E_Info => From.Info);


      return (Node => Ret.Node, Info => Ret.Info);
   end Convert;

   -----------
   -- Image --
   -----------

   overriding function Image (Self : Type_Decl_P_Match_Param_Get_Decl_0_Functor) return String is
   begin
      return ("TypeDecl.match_param_get_decl");
   end Image;



         

         

         

   

   

   

   type Type_Decl_P_Is_Valid_Call_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         Field_0 : Internal_Entity_Argument_List;
         Field_1 : Boolean;
         Error_Location : Bare_Lkt_Node;
   end record;

   
      overriding function Call
        (Self : Type_Decl_P_Is_Valid_Call_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;

   
      overriding procedure Failed
        (Self : Type_Decl_P_Is_Valid_Call_0_Predicate;
            Entity : Internal_Entity;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   ;

   overriding function Image (Self : Type_Decl_P_Is_Valid_Call_0_Predicate) return String;

   

   function Create_Type_Decl_P_Is_Valid_Call_0_Predicate
   (
         Args : Internal_Entity_Argument_List;
         In_Logic_Call : Boolean;
         Error_Location : Bare_Lkt_Node
   )
      return Type_Decl_P_Is_Valid_Call_0_Predicate is
   begin
      
         
         
      return Type_Decl_P_Is_Valid_Call_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Error_Location => Error_Location, Field_0 => Args, Field_1 => In_Logic_Call);
   end;

   ----------
   -- Call --
   ----------

   
      overriding function Call
        (Self : Type_Decl_P_Is_Valid_Call_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   
   is



      
      Node : Bare_Type_Decl;

      Ret : Boolean;
   begin

      
            if Entity.Node /= null
               and then Entity.Node.Kind
                        not in Lkt_Type_Decl
            then
               Raise_Property_Exception
                 (Node, Property_Error'Identity, "mismatching node type");
            end if;

      Node := Entity.Node;


      

      Ret := Type_Decl_P_Is_Valid_Call (Node, Args => Self.Field_0, In_Logic_Call => Self.Field_1, E_Info => Entity.Info);


      return Ret;
   end Call;

   ------------
   -- Failed --
   ------------

   
      overriding procedure Failed
        (Self : Type_Decl_P_Is_Valid_Call_0_Predicate;
            Entity : Internal_Entity;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   
   is
      
      Args : Internal_Entity_Array_Access :=
         Create_Internal_Entity_Array (1);

      Contexts : Internal_Logic_Context_Array_Access :=
         Create_Internal_Logic_Context_Array (Ctxs'Length);

      Diag : constant Internal_Solver_Diagnostic :=
        (Message_Template => Create_String ("invalid call, expected {}"),
         Args             => Args,
         Contexts         => Contexts,
         Location         => Self.Error_Location,
         Round            => Round);
   begin
      Args.Items (1) := Entity;
      for I in Ctxs'Range loop
         Contexts.Items (I) := Ctxs (I).all;
      end loop;
      Emitter (Diag);
   end Failed;

   -----------
   -- Image --
   -----------

   overriding function Image (Self : Type_Decl_P_Is_Valid_Call_0_Predicate) return String is
   begin
      return "TypeDecl.is_valid_call";
   end Image;



         

         

         

   

   type Type_Decl_P_Get_Return_Type_0_Functor is new Solver_Ifc.Converter_Type with
      null record;

   overriding function Convert (Self : Type_Decl_P_Get_Return_Type_0_Functor; From : Internal_Entity) return Internal_Entity with Inline;
   overriding function Image (Self : Type_Decl_P_Get_Return_Type_0_Functor) return String;


   

   function Create_Type_Decl_P_Get_Return_Type_0_Functor
      return Type_Decl_P_Get_Return_Type_0_Functor is
   begin
      
      return Type_Decl_P_Get_Return_Type_0_Functor'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1);
   end;

   overriding function Convert (Self : Type_Decl_P_Get_Return_Type_0_Functor; From : Internal_Entity) return Internal_Entity is
         pragma Unreferenced (Self);



      

      Ret : Internal_Entity_Type_Decl;
   begin
            if From.Node /= null
               and then From.Node.Kind not in
                 Lkt_Type_Decl
            then
               Raise_Property_Exception
                 (From.Node,
                  Property_Error'Identity,
                  "mismatching node type for conv_prop");
            end if;


      
      Ret := Type_Decl_P_Get_Return_Type (Node => From.Node, E_Info => From.Info);


      return (Node => Ret.Node, Info => Ret.Info);
   end Convert;

   -----------
   -- Image --
   -----------

   overriding function Image (Self : Type_Decl_P_Get_Return_Type_0_Functor) return String is
   begin
      return ("TypeDecl.get_return_type");
   end Image;



         

         

   

   type Type_Decl_P_Create_Function_Type_0_Functor is new Solver_Ifc.Combiner_Type with
      null record;

   overriding function Combine (Self : Type_Decl_P_Create_Function_Type_0_Functor; Vals : Entity_Vars.Value_Array) return Internal_Entity with Inline;
   overriding function Image (Self : Type_Decl_P_Create_Function_Type_0_Functor) return String;


   

   function Create_Type_Decl_P_Create_Function_Type_0_Functor
   (
         N : Positive
   )
      return Type_Decl_P_Create_Function_Type_0_Functor is
   begin
      
      return Type_Decl_P_Create_Function_Type_0_Functor'(N => N, Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1);
   end;

   overriding function Combine (Self : Type_Decl_P_Create_Function_Type_0_Functor; Vals : Entity_Vars.Value_Array) return Internal_Entity is
         pragma Unreferenced (Self);

         From : constant Internal_Entity := Vals (1);

         
         Args : Internal_Entity_Type_Decl_Array_Access :=
           Create_Internal_Entity_Type_Decl_Array (Vals'Length - 1);

      

      Ret : Internal_Entity_Type_Decl;
   begin
            if From.Node /= null
               and then From.Node.Kind not in
                 Lkt_Type_Decl
            then
               Raise_Property_Exception
                 (From.Node,
                  Property_Error'Identity,
                  "mismatching node type for comb_prop");
            end if;

      
      for I in 2 .. Vals'Last loop
            if Vals (I).Node /= null
               and then Vals (I).Node.Kind not in
                  Lkt_Type_Decl
            then
               Raise_Property_Exception
                 (Vals (I).Node,
                  Property_Error'Identity,
                  "mismatching node type for comb_prop");
            end if;
         Args.Items (I - 1) := (Vals (I).Node, Vals (I).Info);
      end loop;

      
      Ret := Type_Decl_P_Create_Function_Type (Node => From.Node, Params => Args, E_Info => From.Info);

      Dec_Ref (Args);

      return (Node => Ret.Node, Info => Ret.Info);
   exception
      when Exc : Property_Error =>
         pragma Unreferenced (Exc);
         Dec_Ref (Args);
         raise;
   end Combine;

   -----------
   -- Image --
   -----------

   overriding function Image (Self : Type_Decl_P_Create_Function_Type_0_Functor) return String is
   begin
      return ("TypeDecl.create_function_type");
   end Image;



         

         

   

   type Type_Decl_P_Make_Array_Type_0_Functor is new Solver_Ifc.Converter_Type with
      null record;

   overriding function Convert (Self : Type_Decl_P_Make_Array_Type_0_Functor; From : Internal_Entity) return Internal_Entity with Inline;
   overriding function Image (Self : Type_Decl_P_Make_Array_Type_0_Functor) return String;


   

   function Create_Type_Decl_P_Make_Array_Type_0_Functor
      return Type_Decl_P_Make_Array_Type_0_Functor is
   begin
      
      return Type_Decl_P_Make_Array_Type_0_Functor'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1);
   end;

   overriding function Convert (Self : Type_Decl_P_Make_Array_Type_0_Functor; From : Internal_Entity) return Internal_Entity is
         pragma Unreferenced (Self);



      

      Ret : Internal_Entity_Type_Decl;
   begin
            if From.Node /= null
               and then From.Node.Kind not in
                 Lkt_Type_Decl
            then
               Raise_Property_Exception
                 (From.Node,
                  Property_Error'Identity,
                  "mismatching node type for conv_prop");
            end if;


      
      Ret := Type_Decl_P_Make_Array_Type (Node => From.Node, E_Info => From.Info);


      return (Node => Ret.Node, Info => Ret.Info);
   end Convert;

   -----------
   -- Image --
   -----------

   overriding function Image (Self : Type_Decl_P_Make_Array_Type_0_Functor) return String is
   begin
      return ("TypeDecl.make_array_type");
   end Image;



         

         

   

   type Type_Decl_P_Get_Array_Content_Type_0_Functor is new Solver_Ifc.Converter_Type with
      null record;

   overriding function Convert (Self : Type_Decl_P_Get_Array_Content_Type_0_Functor; From : Internal_Entity) return Internal_Entity with Inline;
   overriding function Image (Self : Type_Decl_P_Get_Array_Content_Type_0_Functor) return String;


   

   function Create_Type_Decl_P_Get_Array_Content_Type_0_Functor
      return Type_Decl_P_Get_Array_Content_Type_0_Functor is
   begin
      
      return Type_Decl_P_Get_Array_Content_Type_0_Functor'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1);
   end;

   overriding function Convert (Self : Type_Decl_P_Get_Array_Content_Type_0_Functor; From : Internal_Entity) return Internal_Entity is
         pragma Unreferenced (Self);



      

      Ret : Internal_Entity_Type_Decl;
   begin
            if From.Node /= null
               and then From.Node.Kind not in
                 Lkt_Type_Decl
            then
               Raise_Property_Exception
                 (From.Node,
                  Property_Error'Identity,
                  "mismatching node type for conv_prop");
            end if;


      
      Ret := Type_Decl_P_Get_Array_Content_Type (Node => From.Node, E_Info => From.Info);


      return (Node => Ret.Node, Info => Ret.Info);
   end Convert;

   -----------
   -- Image --
   -----------

   overriding function Image (Self : Type_Decl_P_Get_Array_Content_Type_0_Functor) return String is
   begin
      return ("TypeDecl.get_array_content_type");
   end Image;



         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

   

   

   

   type Function_Type_P_Returns_Entity_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         Error_Location : Bare_Lkt_Node;
   end record;

   
      overriding function Call
        (Self : Function_Type_P_Returns_Entity_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;

   
      overriding procedure Failed
        (Self : Function_Type_P_Returns_Entity_0_Predicate;
            Entity : Internal_Entity;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   ;

   overriding function Image (Self : Function_Type_P_Returns_Entity_0_Predicate) return String;

   

   function Create_Function_Type_P_Returns_Entity_0_Predicate
   (
         Error_Location : Bare_Lkt_Node
   )
      return Function_Type_P_Returns_Entity_0_Predicate is
   begin
      
      return Function_Type_P_Returns_Entity_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Error_Location => Error_Location);
   end;

   ----------
   -- Call --
   ----------

   
      overriding function Call
        (Self : Function_Type_P_Returns_Entity_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   
   is
         pragma Unreferenced (Self);



      
      Node : Bare_Function_Type;

      Ret : Boolean;
   begin

      
            if Entity.Node /= null
               and then Entity.Node.Kind
                        not in Lkt_Function_Type_Range
            then
               Raise_Property_Exception
                 (Node, Property_Error'Identity, "mismatching node type");
            end if;

      Node := Entity.Node;


      

      Ret := Function_Type_P_Returns_Entity (Node, E_Info => Entity.Info);


      return Ret;
   end Call;

   ------------
   -- Failed --
   ------------

   
      overriding procedure Failed
        (Self : Function_Type_P_Returns_Entity_0_Predicate;
            Entity : Internal_Entity;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   
   is
      
      Args : Internal_Entity_Array_Access :=
         Create_Internal_Entity_Array (0);

      Contexts : Internal_Logic_Context_Array_Access :=
         Create_Internal_Logic_Context_Array (Ctxs'Length);

      Diag : constant Internal_Solver_Diagnostic :=
        (Message_Template => Create_String ("Propagate functions must return entities"),
         Args             => Args,
         Contexts         => Contexts,
         Location         => Self.Error_Location,
         Round            => Round);
   begin
      for I in Ctxs'Range loop
         Contexts.Items (I) := Ctxs (I).all;
      end loop;
      Emitter (Diag);
   end Failed;

   -----------
   -- Image --
   -----------

   overriding function Image (Self : Function_Type_P_Returns_Entity_0_Predicate) return String is
   begin
      return "FunctionType.returns_entity";
   end Image;



         

         

   

   

   

   type Function_Type_P_Returns_Bool_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         Error_Location : Bare_Lkt_Node;
   end record;

   
      overriding function Call
        (Self : Function_Type_P_Returns_Bool_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;

   
      overriding procedure Failed
        (Self : Function_Type_P_Returns_Bool_0_Predicate;
            Entity : Internal_Entity;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   ;

   overriding function Image (Self : Function_Type_P_Returns_Bool_0_Predicate) return String;

   

   function Create_Function_Type_P_Returns_Bool_0_Predicate
   (
         Error_Location : Bare_Lkt_Node
   )
      return Function_Type_P_Returns_Bool_0_Predicate is
   begin
      
      return Function_Type_P_Returns_Bool_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Error_Location => Error_Location);
   end;

   ----------
   -- Call --
   ----------

   
      overriding function Call
        (Self : Function_Type_P_Returns_Bool_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   
   is
         pragma Unreferenced (Self);



      
      Node : Bare_Function_Type;

      Ret : Boolean;
   begin

      
            if Entity.Node /= null
               and then Entity.Node.Kind
                        not in Lkt_Function_Type_Range
            then
               Raise_Property_Exception
                 (Node, Property_Error'Identity, "mismatching node type");
            end if;

      Node := Entity.Node;


      

      Ret := Function_Type_P_Returns_Bool (Node, E_Info => Entity.Info);


      return Ret;
   end Call;

   ------------
   -- Failed --
   ------------

   
      overriding procedure Failed
        (Self : Function_Type_P_Returns_Bool_0_Predicate;
            Entity : Internal_Entity;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   
   is
      
      Args : Internal_Entity_Array_Access :=
         Create_Internal_Entity_Array (0);

      Contexts : Internal_Logic_Context_Array_Access :=
         Create_Internal_Logic_Context_Array (Ctxs'Length);

      Diag : constant Internal_Solver_Diagnostic :=
        (Message_Template => Create_String ("Predicate functions must return booleans"),
         Args             => Args,
         Contexts         => Contexts,
         Location         => Self.Error_Location,
         Round            => Round);
   begin
      for I in Ctxs'Range loop
         Contexts.Items (I) := Ctxs (I).all;
      end loop;
      Emitter (Diag);
   end Failed;

   -----------
   -- Image --
   -----------

   overriding function Image (Self : Function_Type_P_Returns_Bool_0_Predicate) return String is
   begin
      return "FunctionType.returns_bool";
   end Image;



         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         



      

   --
   --  Primitives for Bare_Lkt_Node
   --

   





   







--# property-start LktNode.root_get nodes.lkt:64
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Root_Get
  
  (Node : Bare_Lkt_Node
      ; Entity_Name : Symbol_Type
  )

   return Internal_Entity_Decl
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
      --# bind node Self

   

   --# bind entity_name Entity_Name

   Property_Result : Internal_Entity_Decl;

      
            procedure Finalizer_Scope_21 with Inline_Always;

      Fld : Internal_Unit;
Fld_1 : Bare_Lkt_Node;
Fld_2 : Lexical_Env := Empty_Env;
Env_Get_Result : Internal_Entity;
Cast_Result : Internal_Entity_Decl;

            procedure Finalizer_Scope_21 is
            begin
                     Dec_Ref (Fld_2);
            end Finalizer_Scope_21;


begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      begin
         
   --# scope-start

         --# expr-start 5 Cast Cast_Result nodes.lkt:65





--# expr-start 4 .get_first Env_Get_Result nodes.lkt:65
--# expr-start 3 LktNode.node_env Fld_2 nodes.lkt:65



--# expr-start 2 AnalysisUnit.root Fld_1 nodes.lkt:65



--# expr-start 1 LktNode.unit Fld nodes.lkt:65







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld := Liblktlang.Implementation.Unit (Node => Self);
--# expr-done 1



   if Fld = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_1 := Fld.Ast_Root;
--# expr-done 2



   if Fld_1 = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_2 := Liblktlang.Implementation.Node_Env (Node => Fld_1);
--# expr-done 3



Env_Get_Result := AST_Envs.Get_First (Self => Fld_2, Key => Thin (Entity_Name), Lookup_Kind => To_Lookup_Kind_Type (Recursive), Categories => (Nocat => True)); 
--# expr-done 4



   if Env_Get_Result.Node = null
      or else Env_Get_Result.Node.Kind in Lkt_Decl
   then
      
      Cast_Result := Create_Internal_Entity_Decl
        (Node => Env_Get_Result.Node,
         Info => Env_Get_Result.Info);

   else
         Raise_Property_Exception
           (Self, Property_Error'Identity, "invalid object cast");
   end if;


--# expr-done 5

         Property_Result := Cast_Result;
         
   --# end
      Finalizer_Scope_21;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_21;




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Root_Get;
--# end

   







--# property-start LktNode.get_builtin_type nodes.lkt:68
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Get_Builtin_Type
  
  (Node : Bare_Lkt_Node
      ; Entity_Name : Symbol_Type
  )

   return Internal_Entity_Named_Type_Decl
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
      --# bind node Self

   

   --# bind entity_name Entity_Name

   Property_Result : Internal_Entity_Named_Type_Decl;

      

      Fld : Internal_Entity_Decl;
Cast_Result : Internal_Entity_Named_Type_Decl;


      
      Mmz_Handle : Memoization_Handle;
      Mmz_Val    : Mmz_Value;

      Mmz_Stored : Boolean;
      --  Whether the memoization couple was actually stored. Used to determine
      --  whether to inc-ref the memoized value.

      function Create_Mmz_Key return Mmz_Key;
      --  Create a memoization key for this property call and return it

      --------------------
      -- Create_Mmz_Key --
      --------------------

      function Create_Mmz_Key return Mmz_Key is
      begin
         return Mmz_K : Mmz_Key :=
           (Property => Mmz_Bare_Lkt_Node_Lkt_Node_P_Get_Builtin_Type,
            Items    => new Mmz_Key_Array (1 ..  2))
         do
            Mmz_K.Items (1) := (Kind => Mmz_Bare_Lkt_Node,
                                As_Bare_Lkt_Node => Self);
               Mmz_K.Items (2) := (Kind => Mmz_Symbol_Type,
                                      As_Symbol_Type => Entity_Name);
         end return;
      end Create_Mmz_Key;

begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;

      if Self /= null then

         if not Self.Unit.Context.In_Populate_Lexical_Env
            and then Find_Memoized_Value
                       (Self.Unit, Mmz_Handle, Mmz_Val, Create_Mmz_Key'Access)
         then
            --# memoization-lookup

            if Mmz_Val.Kind = Mmz_Evaluating then
               --# memoization-return
               Raise_Property_Exception
                 (Self,
                  Property_Error'Identity,
                  "Infinite recursion detected");

            elsif Mmz_Val.Kind = Mmz_Error then
               --# memoization-return
               Reraise_Memoized_Error (Mmz_Val);

            else
               Property_Result := Mmz_Val.As_Internal_Entity_Named_Type_Decl;

               --# memoization-return
               return Property_Result;
            end if;
            --# end
         end if;
      end if;

      begin
         
   --# scope-start

         --# expr-start 7 Cast Cast_Result nodes.lkt:69





--# expr-start 6 LktNode.root_get Fld nodes.lkt:69







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;



--# property-call-start LktNode.root_get
Fld := Liblktlang.Implementation.Lkt_Node_P_Root_Get (Node => Self, Entity_Name => Entity_Name);
--# end
--# expr-done 6



   if Fld.Node = null
      or else Fld.Node.Kind in Lkt_Named_Type_Decl
   then
      
      Cast_Result := Create_Internal_Entity_Named_Type_Decl
        (Node => Fld.Node,
         Info => Fld.Info);

   else
         Raise_Property_Exception
           (Self, Property_Error'Identity, "invalid object cast");
   end if;


--# expr-done 7

         Property_Result := Cast_Result;
         
   --# end


      exception
         when Exc : Property_Error =>


               if Self /= null
                  and then not Self.Unit.Context.In_Populate_Lexical_Env
               then
                     Add_Memoized_Error
                       (Self.Unit, Mmz_Handle, Exc, Mmz_Stored);
               end if;


            raise;
      end;

      if Self /= null
         and then not Self.Unit.Context.In_Populate_Lexical_Env
      then
         Mmz_Val := (Kind => Mmz_Internal_Entity_Named_Type_Decl,
                     As_Internal_Entity_Named_Type_Decl => Property_Result);
         Add_Memoized_Value (Self.Unit, Mmz_Handle, Mmz_Val, Mmz_Stored);
      end if;



   return Property_Result;
end Lkt_Node_P_Get_Builtin_Type;
--# end

   







--# property-start LktNode.get_builtin_gen_decl nodes.lkt:71
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Get_Builtin_Gen_Decl
  
  (Node : Bare_Lkt_Node
      ; Entity_Name : Symbol_Type
  )

   return Internal_Entity_Generic_Decl
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
      --# bind node Self

   

   --# bind entity_name Entity_Name

   Property_Result : Internal_Entity_Generic_Decl;

      

      Fld : Internal_Entity_Decl;
Cast_Result : Internal_Entity_Generic_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      begin
         
   --# scope-start

         --# expr-start 9 Cast Cast_Result nodes.lkt:72





--# expr-start 8 LktNode.root_get Fld nodes.lkt:72







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;



--# property-call-start LktNode.root_get
Fld := Liblktlang.Implementation.Lkt_Node_P_Root_Get (Node => Self, Entity_Name => Entity_Name);
--# end
--# expr-done 8



   if Fld.Node = null
      or else Fld.Node.Kind in Lkt_Generic_Decl_Range
   then
      
      Cast_Result := Create_Internal_Entity_Generic_Decl
        (Node => Fld.Node,
         Info => Fld.Info);

   else
         Raise_Property_Exception
           (Self, Property_Error'Identity, "invalid object cast");
   end if;


--# expr-done 9

         Property_Result := Cast_Result;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Get_Builtin_Gen_Decl;
--# end

   







--# property-start LktNode.basic_trait_gen nodes.lkt:76
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Basic_Trait_Gen
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Generic_Decl
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
      --# bind node Self

   


   Property_Result : Internal_Entity_Generic_Decl;

      

      Sym : Symbol_Type;
Fld : Internal_Entity_Generic_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 11 LktNode.get_builtin_gen_decl Fld nodes.lkt:77







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# expr-start 10 SymbolLiteral Sym nodes.lkt:77
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Basictrait); 
--# expr-done 10
--# property-call-start LktNode.get_builtin_gen_decl
Fld := Liblktlang.Implementation.Lkt_Node_P_Get_Builtin_Gen_Decl (Node => Self, Entity_Name => Sym);
--# end
--# expr-done 11

         Property_Result := Fld;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Basic_Trait_Gen;
--# end

   







--# property-start LktNode.basic_trait nodes.lkt:81
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Basic_Trait
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Trait_Decl
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
      --# bind node Self

   


   Property_Result : Internal_Entity_Trait_Decl;

      

      Fld : Internal_Entity_Generic_Decl;
Fld_1 : Internal_Entity_Decl;
Cast_Result : Internal_Entity_Trait_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 14 Cast Cast_Result nodes.lkt:82





--# expr-start 13 GenericDecl.decl Fld_1 nodes.lkt:82



--# expr-start 12 LktNode.basic_trait_gen Fld nodes.lkt:82







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start LktNode.basic_trait_gen
Fld := Liblktlang.Implementation.Lkt_Node_P_Basic_Trait_Gen (Node => Self);
--# end
--# expr-done 12



   if Fld.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_1 := Create_Internal_Entity_Decl (Node => Fld.Node.Generic_Decl_F_Decl, Info => Fld.Info);
--# expr-done 13



   if Fld_1.Node = null
      or else Fld_1.Node.Kind in Lkt_Trait_Decl_Range
   then
      
      Cast_Result := Create_Internal_Entity_Trait_Decl
        (Node => Fld_1.Node,
         Info => Fld_1.Info);

   else
         Cast_Result := No_Entity_Trait_Decl;
   end if;


--# expr-done 14

         Property_Result := Cast_Result;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Basic_Trait;
--# end

   







--# property-start LktNode.node_gen_trait nodes.lkt:86
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Node_Gen_Trait
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Generic_Decl
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
      --# bind node Self

   


   Property_Result : Internal_Entity_Generic_Decl;

      

      Sym : Symbol_Type;
Fld : Internal_Entity_Generic_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 16 LktNode.get_builtin_gen_decl Fld nodes.lkt:87







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# expr-start 15 SymbolLiteral Sym nodes.lkt:87
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Node); 
--# expr-done 15
--# property-call-start LktNode.get_builtin_gen_decl
Fld := Liblktlang.Implementation.Lkt_Node_P_Get_Builtin_Gen_Decl (Node => Self, Entity_Name => Sym);
--# end
--# expr-done 16

         Property_Result := Fld;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Node_Gen_Trait;
--# end

   







--# property-start LktNode.node_trait nodes.lkt:91
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Node_Trait
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Trait_Decl
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
      --# bind node Self

   


   Property_Result : Internal_Entity_Trait_Decl;

      

      Fld : Internal_Entity_Generic_Decl;
Fld_1 : Internal_Entity_Decl;
Cast_Result : Internal_Entity_Trait_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 19 Cast Cast_Result nodes.lkt:92





--# expr-start 18 GenericDecl.decl Fld_1 nodes.lkt:92



--# expr-start 17 LktNode.node_gen_trait Fld nodes.lkt:92







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start LktNode.node_gen_trait
Fld := Liblktlang.Implementation.Lkt_Node_P_Node_Gen_Trait (Node => Self);
--# end
--# expr-done 17



   if Fld.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_1 := Create_Internal_Entity_Decl (Node => Fld.Node.Generic_Decl_F_Decl, Info => Fld.Info);
--# expr-done 18



   if Fld_1.Node = null
      or else Fld_1.Node.Kind in Lkt_Trait_Decl_Range
   then
      
      Cast_Result := Create_Internal_Entity_Trait_Decl
        (Node => Fld_1.Node,
         Info => Fld_1.Info);

   else
         Cast_Result := No_Entity_Trait_Decl;
   end if;


--# expr-done 19

         Property_Result := Cast_Result;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Node_Trait;
--# end

   







--# property-start LktNode.indexable_gen_trait nodes.lkt:96
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Indexable_Gen_Trait
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Generic_Decl
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
      --# bind node Self

   


   Property_Result : Internal_Entity_Generic_Decl;

      

      Sym : Symbol_Type;
Fld : Internal_Entity_Generic_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 21 LktNode.get_builtin_gen_decl Fld nodes.lkt:97







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# expr-start 20 SymbolLiteral Sym nodes.lkt:97
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Indexable); 
--# expr-done 20
--# property-call-start LktNode.get_builtin_gen_decl
Fld := Liblktlang.Implementation.Lkt_Node_P_Get_Builtin_Gen_Decl (Node => Self, Entity_Name => Sym);
--# end
--# expr-done 21

         Property_Result := Fld;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Indexable_Gen_Trait;
--# end

   







--# property-start LktNode.indexable_trait nodes.lkt:101
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Indexable_Trait
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Trait_Decl
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
      --# bind node Self

   


   Property_Result : Internal_Entity_Trait_Decl;

      

      Fld : Internal_Entity_Generic_Decl;
Fld_1 : Internal_Entity_Decl;
Cast_Result : Internal_Entity_Trait_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 24 Cast Cast_Result nodes.lkt:102





--# expr-start 23 GenericDecl.decl Fld_1 nodes.lkt:102



--# expr-start 22 LktNode.indexable_gen_trait Fld nodes.lkt:102







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start LktNode.indexable_gen_trait
Fld := Liblktlang.Implementation.Lkt_Node_P_Indexable_Gen_Trait (Node => Self);
--# end
--# expr-done 22



   if Fld.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_1 := Create_Internal_Entity_Decl (Node => Fld.Node.Generic_Decl_F_Decl, Info => Fld.Info);
--# expr-done 23



   if Fld_1.Node = null
      or else Fld_1.Node.Kind in Lkt_Trait_Decl_Range
   then
      
      Cast_Result := Create_Internal_Entity_Trait_Decl
        (Node => Fld_1.Node,
         Info => Fld_1.Info);

   else
         Cast_Result := No_Entity_Trait_Decl;
   end if;


--# expr-done 24

         Property_Result := Cast_Result;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Indexable_Trait;
--# end

   







--# property-start LktNode.token_node_trait nodes.lkt:106
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Token_Node_Trait
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Named_Type_Decl
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
      --# bind node Self

   


   Property_Result : Internal_Entity_Named_Type_Decl;

      

      Sym : Symbol_Type;
Fld : Internal_Entity_Named_Type_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 26 LktNode.get_builtin_type Fld nodes.lkt:107







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# expr-start 25 SymbolLiteral Sym nodes.lkt:107
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Tokennode); 
--# expr-done 25
--# property-call-start LktNode.get_builtin_type
Fld := Liblktlang.Implementation.Lkt_Node_P_Get_Builtin_Type (Node => Self, Entity_Name => Sym);
--# end
--# expr-done 26

         Property_Result := Fld;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Token_Node_Trait;
--# end

   







--# property-start LktNode.error_node_trait nodes.lkt:111
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Error_Node_Trait
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Named_Type_Decl
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
      --# bind node Self

   


   Property_Result : Internal_Entity_Named_Type_Decl;

      

      Sym : Symbol_Type;
Fld : Internal_Entity_Named_Type_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 28 LktNode.get_builtin_type Fld nodes.lkt:112







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# expr-start 27 SymbolLiteral Sym nodes.lkt:112
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Errornode); 
--# expr-done 27
--# property-call-start LktNode.get_builtin_type
Fld := Liblktlang.Implementation.Lkt_Node_P_Get_Builtin_Type (Node => Self, Entity_Name => Sym);
--# end
--# expr-done 28

         Property_Result := Fld;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Error_Node_Trait;
--# end

   







--# property-start LktNode.char_type nodes.lkt:116
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Char_Type
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Named_Type_Decl
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
      --# bind node Self

   


   Property_Result : Internal_Entity_Named_Type_Decl;

      

      Sym : Symbol_Type;
Fld : Internal_Entity_Named_Type_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 30 LktNode.get_builtin_type Fld nodes.lkt:116







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# expr-start 29 SymbolLiteral Sym nodes.lkt:116
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Char); 
--# expr-done 29
--# property-call-start LktNode.get_builtin_type
Fld := Liblktlang.Implementation.Lkt_Node_P_Get_Builtin_Type (Node => Self, Entity_Name => Sym);
--# end
--# expr-done 30

         Property_Result := Fld;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Char_Type;
--# end

   







--# property-start LktNode.int_type nodes.lkt:120
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Int_Type
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Named_Type_Decl
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
      --# bind node Self

   


   Property_Result : Internal_Entity_Named_Type_Decl;

      

      Sym : Symbol_Type;
Fld : Internal_Entity_Named_Type_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 32 LktNode.get_builtin_type Fld nodes.lkt:120







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# expr-start 31 SymbolLiteral Sym nodes.lkt:120
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Int); 
--# expr-done 31
--# property-call-start LktNode.get_builtin_type
Fld := Liblktlang.Implementation.Lkt_Node_P_Get_Builtin_Type (Node => Self, Entity_Name => Sym);
--# end
--# expr-done 32

         Property_Result := Fld;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Int_Type;
--# end

   







--# property-start LktNode.bool_type nodes.lkt:124
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Bool_Type
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Named_Type_Decl
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
      --# bind node Self

   


   Property_Result : Internal_Entity_Named_Type_Decl;

      

      Sym : Symbol_Type;
Fld : Internal_Entity_Named_Type_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 34 LktNode.get_builtin_type Fld nodes.lkt:124







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# expr-start 33 SymbolLiteral Sym nodes.lkt:124
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Bool); 
--# expr-done 33
--# property-call-start LktNode.get_builtin_type
Fld := Liblktlang.Implementation.Lkt_Node_P_Get_Builtin_Type (Node => Self, Entity_Name => Sym);
--# end
--# expr-done 34

         Property_Result := Fld;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Bool_Type;
--# end

   







--# property-start LktNode.bigint_type nodes.lkt:128
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Bigint_Type
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Named_Type_Decl
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
      --# bind node Self

   


   Property_Result : Internal_Entity_Named_Type_Decl;

      

      Sym : Symbol_Type;
Fld : Internal_Entity_Named_Type_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 36 LktNode.get_builtin_type Fld nodes.lkt:128







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# expr-start 35 SymbolLiteral Sym nodes.lkt:128
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Bigint); 
--# expr-done 35
--# property-call-start LktNode.get_builtin_type
Fld := Liblktlang.Implementation.Lkt_Node_P_Get_Builtin_Type (Node => Self, Entity_Name => Sym);
--# end
--# expr-done 36

         Property_Result := Fld;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Bigint_Type;
--# end

   







--# property-start LktNode.string_type nodes.lkt:132
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_String_Type
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Named_Type_Decl
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
      --# bind node Self

   


   Property_Result : Internal_Entity_Named_Type_Decl;

      

      Sym : Symbol_Type;
Fld : Internal_Entity_Named_Type_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 38 LktNode.get_builtin_type Fld nodes.lkt:132







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# expr-start 37 SymbolLiteral Sym nodes.lkt:132
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_String); 
--# expr-done 37
--# property-call-start LktNode.get_builtin_type
Fld := Liblktlang.Implementation.Lkt_Node_P_Get_Builtin_Type (Node => Self, Entity_Name => Sym);
--# end
--# expr-done 38

         Property_Result := Fld;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_String_Type;
--# end

   







--# property-start LktNode.symbol_type nodes.lkt:136
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Symbol_Type
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Named_Type_Decl
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
      --# bind node Self

   


   Property_Result : Internal_Entity_Named_Type_Decl;

      

      Sym : Symbol_Type;
Fld : Internal_Entity_Named_Type_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 40 LktNode.get_builtin_type Fld nodes.lkt:136







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# expr-start 39 SymbolLiteral Sym nodes.lkt:136
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Symbol); 
--# expr-done 39
--# property-call-start LktNode.get_builtin_type
Fld := Liblktlang.Implementation.Lkt_Node_P_Get_Builtin_Type (Node => Self, Entity_Name => Sym);
--# end
--# expr-done 40

         Property_Result := Fld;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Symbol_Type;
--# end

   







--# property-start LktNode.property_error_type nodes.lkt:140
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Property_Error_Type
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Named_Type_Decl
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
      --# bind node Self

   


   Property_Result : Internal_Entity_Named_Type_Decl;

      

      Sym : Symbol_Type;
Fld : Internal_Entity_Named_Type_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 42 LktNode.get_builtin_type Fld nodes.lkt:141







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# expr-start 41 SymbolLiteral Sym nodes.lkt:141
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Propertyerror); 
--# expr-done 41
--# property-call-start LktNode.get_builtin_type
Fld := Liblktlang.Implementation.Lkt_Node_P_Get_Builtin_Type (Node => Self, Entity_Name => Sym);
--# end
--# expr-done 42

         Property_Result := Fld;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Property_Error_Type;
--# end

   







--# property-start LktNode.regexp_type nodes.lkt:145
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Regexp_Type
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Named_Type_Decl
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
      --# bind node Self

   


   Property_Result : Internal_Entity_Named_Type_Decl;

      

      Sym : Symbol_Type;
Fld : Internal_Entity_Named_Type_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 44 LktNode.get_builtin_type Fld nodes.lkt:145







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# expr-start 43 SymbolLiteral Sym nodes.lkt:145
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Regexp); 
--# expr-done 43
--# property-call-start LktNode.get_builtin_type
Fld := Liblktlang.Implementation.Lkt_Node_P_Get_Builtin_Type (Node => Self, Entity_Name => Sym);
--# end
--# expr-done 44

         Property_Result := Fld;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Regexp_Type;
--# end

   







--# property-start LktNode.entity_gen_type nodes.lkt:149
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Entity_Gen_Type
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Generic_Decl
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
      --# bind node Self

   


   Property_Result : Internal_Entity_Generic_Decl;

      

      Sym : Symbol_Type;
Fld : Internal_Entity_Generic_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 46 LktNode.get_builtin_gen_decl Fld nodes.lkt:150







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# expr-start 45 SymbolLiteral Sym nodes.lkt:150
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Entity); 
--# expr-done 45
--# property-call-start LktNode.get_builtin_gen_decl
Fld := Liblktlang.Implementation.Lkt_Node_P_Get_Builtin_Gen_Decl (Node => Self, Entity_Name => Sym);
--# end
--# expr-done 46

         Property_Result := Fld;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Entity_Gen_Type;
--# end

   







--# property-start LktNode.entity_type nodes.lkt:154
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Entity_Type
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Named_Type_Decl
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
      --# bind node Self

   


   Property_Result : Internal_Entity_Named_Type_Decl;

      

      Fld : Internal_Entity_Generic_Decl;
Fld_1 : Internal_Entity_Decl;
Cast_Result : Internal_Entity_Named_Type_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 49 Cast Cast_Result nodes.lkt:155





--# expr-start 48 GenericDecl.decl Fld_1 nodes.lkt:155



--# expr-start 47 LktNode.entity_gen_type Fld nodes.lkt:155







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start LktNode.entity_gen_type
Fld := Liblktlang.Implementation.Lkt_Node_P_Entity_Gen_Type (Node => Self);
--# end
--# expr-done 47



   if Fld.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_1 := Create_Internal_Entity_Decl (Node => Fld.Node.Generic_Decl_F_Decl, Info => Fld.Info);
--# expr-done 48



   if Fld_1.Node = null
      or else Fld_1.Node.Kind in Lkt_Named_Type_Decl
   then
      
      Cast_Result := Create_Internal_Entity_Named_Type_Decl
        (Node => Fld_1.Node,
         Info => Fld_1.Info);

   else
         Cast_Result := No_Entity_Named_Type_Decl;
   end if;


--# expr-done 49

         Property_Result := Cast_Result;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Entity_Type;
--# end

   







--# property-start LktNode.logicvar_type nodes.lkt:159
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Logicvar_Type
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Named_Type_Decl
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
      --# bind node Self

   


   Property_Result : Internal_Entity_Named_Type_Decl;

      

      Sym : Symbol_Type;
Fld : Internal_Entity_Named_Type_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 51 LktNode.get_builtin_type Fld nodes.lkt:160







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# expr-start 50 SymbolLiteral Sym nodes.lkt:160
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Logicvar); 
--# expr-done 50
--# property-call-start LktNode.get_builtin_type
Fld := Liblktlang.Implementation.Lkt_Node_P_Get_Builtin_Type (Node => Self, Entity_Name => Sym);
--# end
--# expr-done 51

         Property_Result := Fld;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Logicvar_Type;
--# end

   







--# property-start LktNode.equation_type nodes.lkt:164
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Equation_Type
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Named_Type_Decl
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
      --# bind node Self

   


   Property_Result : Internal_Entity_Named_Type_Decl;

      

      Sym : Symbol_Type;
Fld : Internal_Entity_Named_Type_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 53 LktNode.get_builtin_type Fld nodes.lkt:165







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# expr-start 52 SymbolLiteral Sym nodes.lkt:165
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Equation); 
--# expr-done 52
--# property-call-start LktNode.get_builtin_type
Fld := Liblktlang.Implementation.Lkt_Node_P_Get_Builtin_Type (Node => Self, Entity_Name => Sym);
--# end
--# expr-done 53

         Property_Result := Fld;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Equation_Type;
--# end

   







--# property-start LktNode.array_gen_type nodes.lkt:169
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Array_Gen_Type
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Generic_Decl
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
      --# bind node Self

   


   Property_Result : Internal_Entity_Generic_Decl;

      

      Sym : Symbol_Type;
Fld : Internal_Entity_Generic_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 55 LktNode.get_builtin_gen_decl Fld nodes.lkt:170







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# expr-start 54 SymbolLiteral Sym nodes.lkt:170
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Array); 
--# expr-done 54
--# property-call-start LktNode.get_builtin_gen_decl
Fld := Liblktlang.Implementation.Lkt_Node_P_Get_Builtin_Gen_Decl (Node => Self, Entity_Name => Sym);
--# end
--# expr-done 55

         Property_Result := Fld;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Array_Gen_Type;
--# end

   







--# property-start LktNode.array_type nodes.lkt:174
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Array_Type
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Named_Type_Decl
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
      --# bind node Self

   


   Property_Result : Internal_Entity_Named_Type_Decl;

      

      Fld : Internal_Entity_Generic_Decl;
Fld_1 : Internal_Entity_Decl;
Cast_Result : Internal_Entity_Named_Type_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 58 Cast Cast_Result nodes.lkt:175





--# expr-start 57 GenericDecl.decl Fld_1 nodes.lkt:175



--# expr-start 56 LktNode.array_gen_type Fld nodes.lkt:175







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start LktNode.array_gen_type
Fld := Liblktlang.Implementation.Lkt_Node_P_Array_Gen_Type (Node => Self);
--# end
--# expr-done 56



   if Fld.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_1 := Create_Internal_Entity_Decl (Node => Fld.Node.Generic_Decl_F_Decl, Info => Fld.Info);
--# expr-done 57



   if Fld_1.Node = null
      or else Fld_1.Node.Kind in Lkt_Named_Type_Decl
   then
      
      Cast_Result := Create_Internal_Entity_Named_Type_Decl
        (Node => Fld_1.Node,
         Info => Fld_1.Info);

   else
         Cast_Result := No_Entity_Named_Type_Decl;
   end if;


--# expr-done 58

         Property_Result := Cast_Result;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Array_Type;
--# end

   







--# property-start LktNode.astlist_gen_type nodes.lkt:179
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Astlist_Gen_Type
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Generic_Decl
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
      --# bind node Self

   


   Property_Result : Internal_Entity_Generic_Decl;

      

      Sym : Symbol_Type;
Fld : Internal_Entity_Generic_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 60 LktNode.get_builtin_gen_decl Fld nodes.lkt:180







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# expr-start 59 SymbolLiteral Sym nodes.lkt:180
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Astlist); 
--# expr-done 59
--# property-call-start LktNode.get_builtin_gen_decl
Fld := Liblktlang.Implementation.Lkt_Node_P_Get_Builtin_Gen_Decl (Node => Self, Entity_Name => Sym);
--# end
--# expr-done 60

         Property_Result := Fld;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Astlist_Gen_Type;
--# end

   







--# property-start LktNode.astlist_type nodes.lkt:184
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Astlist_Type
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Named_Type_Decl
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
      --# bind node Self

   


   Property_Result : Internal_Entity_Named_Type_Decl;

      

      Fld : Internal_Entity_Generic_Decl;
Fld_1 : Internal_Entity_Decl;
Cast_Result : Internal_Entity_Named_Type_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 63 Cast Cast_Result nodes.lkt:185





--# expr-start 62 GenericDecl.decl Fld_1 nodes.lkt:185



--# expr-start 61 LktNode.astlist_gen_type Fld nodes.lkt:185







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start LktNode.astlist_gen_type
Fld := Liblktlang.Implementation.Lkt_Node_P_Astlist_Gen_Type (Node => Self);
--# end
--# expr-done 61



   if Fld.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_1 := Create_Internal_Entity_Decl (Node => Fld.Node.Generic_Decl_F_Decl, Info => Fld.Info);
--# expr-done 62



   if Fld_1.Node = null
      or else Fld_1.Node.Kind in Lkt_Named_Type_Decl
   then
      
      Cast_Result := Create_Internal_Entity_Named_Type_Decl
        (Node => Fld_1.Node,
         Info => Fld_1.Info);

   else
         Cast_Result := No_Entity_Named_Type_Decl;
   end if;


--# expr-done 63

         Property_Result := Cast_Result;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Astlist_Type;
--# end

   







--# property-start LktNode.node_builder_gen_type nodes.lkt:189
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Node_Builder_Gen_Type
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Generic_Decl
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
      --# bind node Self

   


   Property_Result : Internal_Entity_Generic_Decl;

      

      Sym : Symbol_Type;
Fld : Internal_Entity_Generic_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 65 LktNode.get_builtin_gen_decl Fld nodes.lkt:190







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# expr-start 64 SymbolLiteral Sym nodes.lkt:190
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Nodebuilder); 
--# expr-done 64
--# property-call-start LktNode.get_builtin_gen_decl
Fld := Liblktlang.Implementation.Lkt_Node_P_Get_Builtin_Gen_Decl (Node => Self, Entity_Name => Sym);
--# end
--# expr-done 65

         Property_Result := Fld;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Node_Builder_Gen_Type;
--# end

   







--# property-start LktNode.node_builder_type nodes.lkt:194
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Node_Builder_Type
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Named_Type_Decl
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
      --# bind node Self

   


   Property_Result : Internal_Entity_Named_Type_Decl;

      

      Fld : Internal_Entity_Generic_Decl;
Fld_1 : Internal_Entity_Decl;
Cast_Result : Internal_Entity_Named_Type_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 68 Cast Cast_Result nodes.lkt:195





--# expr-start 67 GenericDecl.decl Fld_1 nodes.lkt:195



--# expr-start 66 LktNode.node_builder_gen_type Fld nodes.lkt:195







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start LktNode.node_builder_gen_type
Fld := Liblktlang.Implementation.Lkt_Node_P_Node_Builder_Gen_Type (Node => Self);
--# end
--# expr-done 66



   if Fld.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_1 := Create_Internal_Entity_Decl (Node => Fld.Node.Generic_Decl_F_Decl, Info => Fld.Info);
--# expr-done 67



   if Fld_1.Node = null
      or else Fld_1.Node.Kind in Lkt_Named_Type_Decl
   then
      
      Cast_Result := Create_Internal_Entity_Named_Type_Decl
        (Node => Fld_1.Node,
         Info => Fld_1.Info);

   else
         Cast_Result := No_Entity_Named_Type_Decl;
   end if;


--# expr-done 68

         Property_Result := Cast_Result;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Node_Builder_Type;
--# end

   







--# property-start LktNode.iterator_gen_trait nodes.lkt:199
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Iterator_Gen_Trait
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Generic_Decl
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
      --# bind node Self

   


   Property_Result : Internal_Entity_Generic_Decl;

      

      Sym : Symbol_Type;
Fld : Internal_Entity_Generic_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 70 LktNode.get_builtin_gen_decl Fld nodes.lkt:200







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# expr-start 69 SymbolLiteral Sym nodes.lkt:200
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Iterator); 
--# expr-done 69
--# property-call-start LktNode.get_builtin_gen_decl
Fld := Liblktlang.Implementation.Lkt_Node_P_Get_Builtin_Gen_Decl (Node => Self, Entity_Name => Sym);
--# end
--# expr-done 70

         Property_Result := Fld;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Iterator_Gen_Trait;
--# end

   







--# property-start LktNode.iterator_trait nodes.lkt:204
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Iterator_Trait
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Trait_Decl
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
      --# bind node Self

   


   Property_Result : Internal_Entity_Trait_Decl;

      

      Fld : Internal_Entity_Generic_Decl;
Fld_1 : Internal_Entity_Decl;
Cast_Result : Internal_Entity_Trait_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 73 Cast Cast_Result nodes.lkt:205





--# expr-start 72 GenericDecl.decl Fld_1 nodes.lkt:205



--# expr-start 71 LktNode.iterator_gen_trait Fld nodes.lkt:205







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start LktNode.iterator_gen_trait
Fld := Liblktlang.Implementation.Lkt_Node_P_Iterator_Gen_Trait (Node => Self);
--# end
--# expr-done 71



   if Fld.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_1 := Create_Internal_Entity_Decl (Node => Fld.Node.Generic_Decl_F_Decl, Info => Fld.Info);
--# expr-done 72



   if Fld_1.Node = null
      or else Fld_1.Node.Kind in Lkt_Trait_Decl_Range
   then
      
      Cast_Result := Create_Internal_Entity_Trait_Decl
        (Node => Fld_1.Node,
         Info => Fld_1.Info);

   else
         Cast_Result := No_Entity_Trait_Decl;
   end if;


--# expr-done 73

         Property_Result := Cast_Result;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Iterator_Trait;
--# end

   







--# property-start LktNode.analysis_unit_gen_trait nodes.lkt:209
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Analysis_Unit_Gen_Trait
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Generic_Decl
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
      --# bind node Self

   


   Property_Result : Internal_Entity_Generic_Decl;

      

      Sym : Symbol_Type;
Fld : Internal_Entity_Generic_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 75 LktNode.get_builtin_gen_decl Fld nodes.lkt:210







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# expr-start 74 SymbolLiteral Sym nodes.lkt:210
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Analysisunit); 
--# expr-done 74
--# property-call-start LktNode.get_builtin_gen_decl
Fld := Liblktlang.Implementation.Lkt_Node_P_Get_Builtin_Gen_Decl (Node => Self, Entity_Name => Sym);
--# end
--# expr-done 75

         Property_Result := Fld;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Analysis_Unit_Gen_Trait;
--# end

   







--# property-start LktNode.analysis_unit_trait nodes.lkt:214
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Analysis_Unit_Trait
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Trait_Decl
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
      --# bind node Self

   


   Property_Result : Internal_Entity_Trait_Decl;

      

      Fld : Internal_Entity_Generic_Decl;
Fld_1 : Internal_Entity_Decl;
Cast_Result : Internal_Entity_Trait_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 78 Cast Cast_Result nodes.lkt:215





--# expr-start 77 GenericDecl.decl Fld_1 nodes.lkt:215



--# expr-start 76 LktNode.analysis_unit_gen_trait Fld nodes.lkt:215







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start LktNode.analysis_unit_gen_trait
Fld := Liblktlang.Implementation.Lkt_Node_P_Analysis_Unit_Gen_Trait (Node => Self);
--# end
--# expr-done 76



   if Fld.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_1 := Create_Internal_Entity_Decl (Node => Fld.Node.Generic_Decl_F_Decl, Info => Fld.Info);
--# expr-done 77



   if Fld_1.Node = null
      or else Fld_1.Node.Kind in Lkt_Trait_Decl_Range
   then
      
      Cast_Result := Create_Internal_Entity_Trait_Decl
        (Node => Fld_1.Node,
         Info => Fld_1.Info);

   else
         Cast_Result := No_Entity_Trait_Decl;
   end if;


--# expr-done 78

         Property_Result := Cast_Result;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Analysis_Unit_Trait;
--# end

   







--# property-start LktNode.get_empty_type_ref_list nodes.lkt:219
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Get_Empty_Type_Ref_List
  
  (Node : Bare_Lkt_Node
  )

   return Bare_Synthetic_Type_Ref_List
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
      --# bind node Self

   


   Property_Result : Bare_Synthetic_Type_Ref_List;

      

      Fld : Internal_Unit;
Fld_1 : Bare_Lkt_Node;
Cast_Result : Bare_Langkit_Root;
Fld_2 : Bare_Synthetic_Type_Ref_List;



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 82 LangkitRoot.empty_type_ref_list Fld_2 nodes.lkt:220



--# expr-start 81 Cast Cast_Result nodes.lkt:220





--# expr-start 80 AnalysisUnit.root Fld_1 nodes.lkt:220



--# expr-start 79 LktNode.unit Fld nodes.lkt:220







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld := Liblktlang.Implementation.Unit (Node => Self);
--# expr-done 79



   if Fld = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_1 := Fld.Ast_Root;
--# expr-done 80



   if Fld_1 = null
      or else Fld_1.Kind in Lkt_Langkit_Root_Range
   then
      
      Cast_Result := Fld_1;

   else
         Cast_Result := No_Bare_Lkt_Node;
   end if;


--# expr-done 81



   if Cast_Result = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start LangkitRoot.empty_type_ref_list
Fld_2 := Liblktlang.Implementation.Langkit_Root_F_Empty_Type_Ref_List (Node => Cast_Result);
--# end
--# expr-done 82

         Property_Result := Fld_2;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Get_Empty_Type_Ref_List;
--# end

   







--# property-start LktNode.any_type nodes.lkt:224
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Any_Type
  
  (Node : Bare_Lkt_Node
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity_Type_Decl
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
     Ent : Internal_Entity :=
       Internal_Entity'(Node => Self, Info => E_Info);
      --# bind self Ent

   


   Property_Result : Internal_Entity_Type_Decl;

      

      Fld : Bare_Synthetic_Type_Ref_List;
New_Node : Bare_Any_Type_Decl;
As_Entity : Internal_Entity_Any_Type_Decl;
Is_Null : Boolean;
If_Result : Internal_Entity_Any_Type_Decl;
Cast_Result : Internal_Entity_Type_Decl;


      
      Mmz_Handle : Memoization_Handle;
      Mmz_Val    : Mmz_Value;

      Mmz_Stored : Boolean;
      --  Whether the memoization couple was actually stored. Used to determine
      --  whether to inc-ref the memoized value.

      function Create_Mmz_Key return Mmz_Key;
      --  Create a memoization key for this property call and return it

      --------------------
      -- Create_Mmz_Key --
      --------------------

      function Create_Mmz_Key return Mmz_Key is
      begin
         return Mmz_K : Mmz_Key :=
           (Property => Mmz_Bare_Lkt_Node_Lkt_Node_P_Any_Type,
            Items    => new Mmz_Key_Array (1 ..  2))
         do
            Mmz_K.Items (1) := (Kind => Mmz_Bare_Lkt_Node,
                                As_Bare_Lkt_Node => Self);
               Mmz_K.Items (2) :=
                 (Kind => Mmz_Internal_Entity_Info,
                  As_Internal_Entity_Info => E_Info);
         end return;
      end Create_Mmz_Key;

begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;

      if Self /= null then

         if not Self.Unit.Context.In_Populate_Lexical_Env
            and then Find_Memoized_Value
                       (Self.Unit, Mmz_Handle, Mmz_Val, Create_Mmz_Key'Access)
         then
            --# memoization-lookup

            if Mmz_Val.Kind = Mmz_Evaluating then
               --# memoization-return
               Raise_Property_Exception
                 (Self,
                  Property_Error'Identity,
                  "Infinite recursion detected");

            elsif Mmz_Val.Kind = Mmz_Error then
               --# memoization-return
               Reraise_Memoized_Error (Mmz_Val);

            else
               Property_Result := Mmz_Val.As_Internal_Entity_Type_Decl;

               --# memoization-return
               return Property_Result;
            end if;
            --# end
         end if;
      end if;

      begin
         
   --# scope-start

         




--# expr-start 85 .as_entity If_Result nodes.lkt:225



--# expr-start 84 'New[AnyTypeDecl]' New_Node nodes.lkt:225
--# expr-start 83 LktNode.get_empty_type_ref_list Fld nodes.lkt:225







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start LktNode.get_empty_type_ref_list
Fld := Liblktlang.Implementation.Lkt_Node_P_Get_Empty_Type_Ref_List (Node => Self);
--# end
--# expr-done 83


if Is_Foreign_Strict (Self.Self_Env, Self) then
   Raise_Property_Exception
     (Self,
      Property_Error'Identity,
      "synthetic nodes cannot have foreign lexical envs");
end if;


New_Node := new Root_Node_Record
  (Lkt_Any_Type_Decl);
Initialize
  (Self => New_Node,
   Kind => Lkt_Any_Type_Decl,
   Unit => Self.Unit,

   Token_Start_Index => No_Token_Index,
   Token_End_Index   => No_Token_Index,

   Parent => Self,

   Self_Env => Self.Self_Env);
Register_Destroyable (Self.Unit, New_Node);

   Initialize_Fields_For_Any_Type_Decl
     (Self => New_Node,
      Any_Type_Decl_F_Traits => Fld);


--# expr-done 84
Is_Null := New_Node = null; 
if Is_Null then
   
   If_Result := No_Entity_Any_Type_Decl;
else
   

As_Entity := (Info => E_Info, Node => New_Node); 
   If_Result := As_Entity;
end if;



--# expr-done 85



   
      Cast_Result := Create_Internal_Entity_Type_Decl
        (Node => If_Result.Node,
         Info => If_Result.Info);



         Property_Result := Cast_Result;
         
   --# end


      exception
         when Exc : Property_Error =>


               if Self /= null
                  and then not Self.Unit.Context.In_Populate_Lexical_Env
               then
                     Add_Memoized_Error
                       (Self.Unit, Mmz_Handle, Exc, Mmz_Stored);
               end if;


            raise;
      end;

      if Self /= null
         and then not Self.Unit.Context.In_Populate_Lexical_Env
      then
         Mmz_Val := (Kind => Mmz_Internal_Entity_Type_Decl,
                     As_Internal_Entity_Type_Decl => Property_Result);
         Add_Memoized_Value (Self.Unit, Mmz_Handle, Mmz_Val, Mmz_Stored);
      end if;



   return Property_Result;
end Lkt_Node_P_Any_Type;
--# end

   







--# property-start LktNode.topmost_invalid_decl nodes.lkt:234
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Topmost_Invalid_Decl
  
  (Node : Bare_Lkt_Node
  )

   return Bare_Lkt_Node
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
      --# bind node Self

   


   Property_Result : Bare_Lkt_Node;

      
            procedure Finalizer_Scope_57 with Inline_Always;

      Node_Parents : Bare_Lkt_Node_Array_Access;
Cast_Result : Bare_Full_Decl;
Var_Expr : Bare_Full_Decl;
Sym : Symbol_Type;
Fld : Boolean;
Scope_Result : Boolean;
Result_Var : Boolean;
Map_Result : Bare_Lkt_Node_Array_Access;
Neg : Integer;
Get_Result : Bare_Lkt_Node;

            procedure Finalizer_Scope_57 is
            begin
                     Dec_Ref (Node_Parents);
                     Dec_Ref (Map_Result);
            end Finalizer_Scope_57;


begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 93 .at Get_Result nodes.lkt:235

--# expr-start 91 .filter Map_Result nodes.lkt:235








   

   --# expr-start 86 LktNode.parents Node_Parents nodes.lkt:235







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;



Node_Parents := Parents (Self, True); 
--# expr-done 86

   
      declare
         Map_Result_Vec : Bare_Lkt_Node_Vectors.Vector;
      begin
   

   

      

      declare
         
         Collection : constant Bare_Lkt_Node_Array_Access := Node_Parents;
      begin
         for Item of
               Collection.Items
         loop

            
   --# scope-start


               --# bind p Item

            
         --# expr-start 90 .do Result_Var nodes.lkt:236



--# expr-start 87 Cast Cast_Result nodes.lkt:236









   if Item = null
      or else Item.Kind in Lkt_Full_Decl_Range
   then
      
      Cast_Result := Item;

   else
         Cast_Result := No_Bare_Lkt_Node;
   end if;


--# expr-done 87
Var_Expr := Cast_Result;





if Var_Expr /= No_Bare_Lkt_Node then
   



   --# scope-start



--# expr-start 89 FullDecl.has_annotation Fld nodes.lkt:236







   if Var_Expr = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# expr-start 88 SymbolLiteral Sym nodes.lkt:236
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Invalid); 
--# expr-done 88
--# property-call-start FullDecl.has_annotation
Fld := Liblktlang.Implementation.Full_Decl_P_Has_Annotation (Node => Var_Expr, Name => Sym);
--# end
--# expr-done 89
Scope_Result := Fld;


   --# end


   Result_Var := Scope_Result;
else
   
   Result_Var := False;
end if;



--# expr-done 90
         if Result_Var then
            
         
      
      

         declare
            Item_To_Append : constant Bare_Lkt_Node := Item;
         begin
            Bare_Lkt_Node_Vectors.Append (Map_Result_Vec, Item_To_Append);
         end;
   
   
         end if;
   

            
   --# end



         end loop;
      end;
   

   

         Map_Result := Create_Bare_Lkt_Node_Array
           (Items_Count => Natural (Bare_Lkt_Node_Vectors.Length (Map_Result_Vec)));
         for I in Map_Result.Items'Range loop
            Map_Result.Items (I) := Bare_Lkt_Node_Vectors.Get
              (Map_Result_Vec,
               I + Bare_Lkt_Node_Vectors.Index_Type'First - Map_Result.Items'First);
         end loop;
         Bare_Lkt_Node_Vectors.Destroy (Map_Result_Vec);

      end;
   



--# expr-done 91
--# expr-start 92 UnaryNeg Neg nodes.lkt:237

Neg := -1; 
--# expr-done 92

Get_Result := Get (Self, Map_Result, Neg, True); 
--# expr-done 93

         Property_Result := Get_Result;
         
   --# end
      Finalizer_Scope_57;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_57;




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Topmost_Invalid_Decl;
--# end

   







--# property-start LktNode.nameres_diagnostics nodes.lkt:242
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Nameres_Diagnostics
  
  (Node : Bare_Lkt_Node
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Solver_Diagnostic_Array_Access
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
     Ent : Internal_Entity :=
       Internal_Entity'(Node => Self, Info => E_Info);
      --# bind self Ent

   


   Property_Result : Internal_Solver_Diagnostic_Array_Access;

      
            procedure Finalizer_Scope_58 with Inline_Always;

      Fld : Internal_Solver_Result := No_Solver_Result;
Fld_1 : Internal_Solver_Diagnostic_Array_Access;

            procedure Finalizer_Scope_58 is
            begin
                     Dec_Ref (Fld);
                     Dec_Ref (Fld_1);
            end Finalizer_Scope_58;


begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 95 SolverResult.diagnostics Fld_1 nodes.lkt:244



--# expr-start 94 LktNode.solve_equation Fld nodes.lkt:244







   if Ent.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start LktNode.solve_equation
Fld := Liblktlang.Implementation.Lkt_Node_P_Solve_Equation (Node => Ent.Node, E_Info => Ent.Info);
--# end
--# expr-done 94





Fld_1 := Fld.Diagnostics;
Inc_Ref (Fld_1);
--# expr-done 95

         Property_Result := Fld_1;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_58;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_58;




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Nameres_Diagnostics;
--# end

   







--# property-start LktNode.solve_equation nodes.lkt:250
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Solve_Equation
  
  (Node : Bare_Lkt_Node
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Solver_Result
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
     Ent : Internal_Entity :=
       Internal_Entity'(Node => Self, Info => E_Info);
      --# bind self Ent

   


   Property_Result : Internal_Solver_Result;

      
            procedure Finalizer_Scope_59 with Inline_Always;

      Fld : Logic_Equation;
Solve_Success : Internal_Solver_Result := No_Solver_Result;

            procedure Finalizer_Scope_59 is
            begin
                     Dec_Ref (Fld);
                     Dec_Ref (Solve_Success);
            end Finalizer_Scope_59;

      
      Mmz_Handle : Memoization_Handle;
      Mmz_Val    : Mmz_Value;

      Mmz_Stored : Boolean;
      --  Whether the memoization couple was actually stored. Used to determine
      --  whether to inc-ref the memoized value.

      function Create_Mmz_Key return Mmz_Key;
      --  Create a memoization key for this property call and return it

      --------------------
      -- Create_Mmz_Key --
      --------------------

      function Create_Mmz_Key return Mmz_Key is
      begin
         return Mmz_K : Mmz_Key :=
           (Property => Mmz_Bare_Lkt_Node_Lkt_Node_P_Solve_Equation,
            Items    => new Mmz_Key_Array (1 ..  2))
         do
            Mmz_K.Items (1) := (Kind => Mmz_Bare_Lkt_Node,
                                As_Bare_Lkt_Node => Self);
               Mmz_K.Items (2) :=
                 (Kind => Mmz_Internal_Entity_Info,
                  As_Internal_Entity_Info => E_Info);
         end return;
      end Create_Mmz_Key;

begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;

      if Self /= null then

         if not Self.Unit.Context.In_Populate_Lexical_Env
            and then Find_Memoized_Value
                       (Self.Unit, Mmz_Handle, Mmz_Val, Create_Mmz_Key'Access)
         then
            --# memoization-lookup

            if Mmz_Val.Kind = Mmz_Evaluating then
               --# memoization-return
               Raise_Property_Exception
                 (Self,
                  Property_Error'Identity,
                  "Infinite recursion detected");

            elsif Mmz_Val.Kind = Mmz_Error then
               --# memoization-return
               Reraise_Memoized_Error (Mmz_Val);

            else
               Property_Result := Mmz_Val.As_Internal_Solver_Result;
                  Inc_Ref (Property_Result);

               --# memoization-return
               return Property_Result;
            end if;
            --# end
         end if;
      end if;

      begin
         
   --# scope-start

         --# expr-start 97 .solve_with_diagnostics Solve_Success nodes.lkt:251
--# expr-start 96 LktNode.xref_equation Fld nodes.lkt:251







   if Ent.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start '[dispatcher]LktNode.xref_equation'
Fld := Liblktlang.Implementation.Dispatcher_Lkt_Node_P_Xref_Equation (Node => Ent.Node, E_Info => Ent.Info);
--# end
--# expr-done 96

Solve_Success := Solve_With_Diagnostics (Fld, Self); 
--# expr-done 97

         Property_Result := Solve_Success;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_59;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_59;


               if Self /= null
                  and then not Self.Unit.Context.In_Populate_Lexical_Env
               then
                     Add_Memoized_Error
                       (Self.Unit, Mmz_Handle, Exc, Mmz_Stored);
               end if;


            raise;
      end;

      if Self /= null
         and then not Self.Unit.Context.In_Populate_Lexical_Env
      then
         Mmz_Val := (Kind => Mmz_Internal_Solver_Result,
                     As_Internal_Solver_Result => Property_Result);
         Add_Memoized_Value (Self.Unit, Mmz_Handle, Mmz_Val, Mmz_Stored);
            if Mmz_Stored then
               Inc_Ref (Property_Result);
            end if;
      end if;



   return Property_Result;
end Lkt_Node_P_Solve_Equation;
--# end

   







--# property-start LktNode.solve_enclosing_context nodes.lkt:258
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Solve_Enclosing_Context
  
  (Node : Bare_Lkt_Node
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Solver_Result
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
     Ent : Internal_Entity :=
       Internal_Entity'(Node => Self, Info => E_Info);
      --# bind self Ent

   


   Property_Result : Internal_Solver_Result;

      
            procedure Finalizer_Scope_60 with Inline_Always;
            procedure Finalizer_Scope_455 with Inline_Always;

      Fld : Internal_Entity_Array_Access;
Fld_1 : Boolean;
Find_Result : Internal_Entity;
Local_Enclosing_Context : Internal_Entity;
Var_Expr : Internal_Entity;
Fld_2 : Internal_Solver_Result := No_Solver_Result;
Result_Var : Internal_Solver_Result := No_Solver_Result;
Let_Result : Internal_Solver_Result := No_Solver_Result;
Scope_Result : Internal_Solver_Result := No_Solver_Result;

            procedure Finalizer_Scope_60 is
            begin
                     Dec_Ref (Scope_Result);
            end Finalizer_Scope_60;
            procedure Finalizer_Scope_455 is
            begin
                     Dec_Ref (Fld);
                     Dec_Ref (Fld_2);
                     Dec_Ref (Result_Var);
                     Dec_Ref (Let_Result);
            end Finalizer_Scope_455;

      
      Mmz_Handle : Memoization_Handle;
      Mmz_Val    : Mmz_Value;

      Mmz_Stored : Boolean;
      --  Whether the memoization couple was actually stored. Used to determine
      --  whether to inc-ref the memoized value.

      function Create_Mmz_Key return Mmz_Key;
      --  Create a memoization key for this property call and return it

      --------------------
      -- Create_Mmz_Key --
      --------------------

      function Create_Mmz_Key return Mmz_Key is
      begin
         return Mmz_K : Mmz_Key :=
           (Property => Mmz_Bare_Lkt_Node_Lkt_Node_P_Solve_Enclosing_Context,
            Items    => new Mmz_Key_Array (1 ..  2))
         do
            Mmz_K.Items (1) := (Kind => Mmz_Bare_Lkt_Node,
                                As_Bare_Lkt_Node => Self);
               Mmz_K.Items (2) :=
                 (Kind => Mmz_Internal_Entity_Info,
                  As_Internal_Entity_Info => E_Info);
         end return;
      end Create_Mmz_Key;

begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;

      if Self /= null then

         if not Self.Unit.Context.In_Populate_Lexical_Env
            and then Find_Memoized_Value
                       (Self.Unit, Mmz_Handle, Mmz_Val, Create_Mmz_Key'Access)
         then
            --# memoization-lookup

            if Mmz_Val.Kind = Mmz_Evaluating then
               --# memoization-return
               Raise_Property_Exception
                 (Self,
                  Property_Error'Identity,
                  "Infinite recursion detected");

            elsif Mmz_Val.Kind = Mmz_Error then
               --# memoization-return
               Reraise_Memoized_Error (Mmz_Val);

            else
               Property_Result := Mmz_Val.As_Internal_Solver_Result;
                  Inc_Ref (Property_Result);

               --# memoization-return
               return Property_Result;
            end if;
            --# end
         end if;
      end if;

      begin
         
   --# scope-start

         



   --# scope-start



--# expr-start 102 ValDecl Let_Result nodes.lkt:259
--# scope-start
--# expr-start 100 .find Find_Result nodes.lkt:260








   

   --# expr-start 98 LktNode.parents Fld nodes.lkt:260







   if Ent.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;



Fld := Liblktlang.Implementation.Parents (Node => Ent.Node, With_Self => True, E_Info => Ent.Info);
--# expr-done 98

   
      Find_Result := No_Entity;
   

   

      

      declare
         
         Collection : constant Internal_Entity_Array_Access := Fld;
      begin
         for Item of
               Collection.Items
         loop

            
   --# scope-start


               --# bind p Item

            
      --# expr-start 99 LktNode.xref_entry_point Fld_1 nodes.lkt:260







   if Item.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start '[dispatcher]LktNode.xref_entry_point'
Fld_1 := Liblktlang.Implementation.Dispatcher_Lkt_Node_P_Xref_Entry_Point (Node => Item.Node, E_Info => Item.Info);
--# end
--# expr-done 99
      if Fld_1 then
         Find_Result := Item; 
      end if;
   

            
   --# end


               exit when Fld_1;

         end loop;
      end;
   

   
   



--# expr-done 100
Local_Enclosing_Context := Find_Result; 
--# bind enclosing_context Local_Enclosing_Context




Var_Expr := Local_Enclosing_Context;





if Var_Expr /= No_Entity then
   --# expr-start 101 LktNode.solve_equation Fld_2 nodes.lkt:262







   if Var_Expr.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start LktNode.solve_equation
Fld_2 := Liblktlang.Implementation.Lkt_Node_P_Solve_Equation (Node => Var_Expr.Node, E_Info => Var_Expr.Info);
--# end
--# expr-done 101
   Result_Var := Fld_2;
else
   
   Result_Var := No_Solver_Result;
end if;

      Inc_Ref (Result_Var);


Let_Result := Result_Var; Inc_Ref (Let_Result);
--# end
--# expr-done 102
Scope_Result := Let_Result;
   Inc_Ref (Scope_Result);


   --# end
      Finalizer_Scope_455;



         Property_Result := Scope_Result;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_60;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_60;
                     Finalizer_Scope_455;


               if Self /= null
                  and then not Self.Unit.Context.In_Populate_Lexical_Env
               then
                     Add_Memoized_Error
                       (Self.Unit, Mmz_Handle, Exc, Mmz_Stored);
               end if;


            raise;
      end;

      if Self /= null
         and then not Self.Unit.Context.In_Populate_Lexical_Env
      then
         Mmz_Val := (Kind => Mmz_Internal_Solver_Result,
                     As_Internal_Solver_Result => Property_Result);
         Add_Memoized_Value (Self.Unit, Mmz_Handle, Mmz_Val, Mmz_Stored);
            if Mmz_Stored then
               Inc_Ref (Property_Result);
            end if;
      end if;



   return Property_Result;
end Lkt_Node_P_Solve_Enclosing_Context;
--# end

   







--# property-start LktNode.expected_type_entry_point nodes.lkt:265
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Expected_Type_Entry_Point
  
  (Node : Bare_Lkt_Node
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Boolean
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
     Ent : Internal_Entity :=
       Internal_Entity'(Node => Self, Info => E_Info);
      --# bind self Ent

   


   Property_Result : Boolean;

      

      Ignored : Internal_Entity_Cast_Expr;
Ignored_1 : Internal_Entity_Isa;
Ignored_2 : Internal_Entity_Match_Expr;
Ignored_3 : Internal_Entity_Dot_Expr;
Ignored_4 : Internal_Entity_Call_Expr;
Ignored_5 : Internal_Entity_Logic_Predicate;
Ignored_6 : Internal_Entity_Logic_Propagate;
Local_P : Internal_Entity_Paren_Expr;
Fld : Internal_Entity;
Fld_1 : Boolean;
Local_P_1 : Internal_Entity_Lambda_Expr;
Fld_2 : Boolean;
Ignored_7 : Internal_Entity;
Match_Prefix : Internal_Entity;
Cast_Result : Internal_Entity_Cast_Expr;
Let_Result : Boolean;
Scope_Result : Boolean;
Cast_Result_1 : Internal_Entity_Isa;
Let_Result_1 : Boolean;
Scope_Result_1 : Boolean;
Cast_Result_2 : Internal_Entity_Match_Expr;
Let_Result_2 : Boolean;
Scope_Result_2 : Boolean;
Cast_Result_3 : Internal_Entity_Dot_Expr;
Let_Result_3 : Boolean;
Scope_Result_3 : Boolean;
Cast_Result_4 : Internal_Entity_Call_Expr;
Let_Result_4 : Boolean;
Scope_Result_4 : Boolean;
Cast_Result_5 : Internal_Entity_Logic_Predicate;
Let_Result_5 : Boolean;
Scope_Result_5 : Boolean;
Cast_Result_6 : Internal_Entity_Logic_Propagate;
Let_Result_6 : Boolean;
Scope_Result_6 : Boolean;
Cast_Result_7 : Internal_Entity_Paren_Expr;
Let_Result_7 : Boolean;
Scope_Result_7 : Boolean;
Cast_Result_8 : Internal_Entity_Lambda_Expr;
Let_Result_8 : Boolean;
Scope_Result_8 : Boolean;
Cast_Result_9 : Internal_Entity;
Let_Result_9 : Boolean;
Scope_Result_9 : Boolean;
Match_Result : Boolean;



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 106 Match Match_Result nodes.lkt:265










   if Ent.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Match_Prefix := Ent; 

case Lkt_Lkt_Node (Match_Prefix.Node.Kind) is
         when Lkt_Cast_Expr =>
            



   --# scope-start



--# scope-start









   
      Cast_Result := Create_Internal_Entity_Cast_Expr
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Ignored := Cast_Result; 
--# bind _ Ignored

Let_Result := True; 
--# end
Scope_Result := Let_Result;


   --# end


            Match_Result := Scope_Result; 
         when Lkt_Isa =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_1 := Create_Internal_Entity_Isa
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Ignored_1 := Cast_Result_1; 
--# bind _ Ignored_1

Let_Result_1 := True; 
--# end
Scope_Result_1 := Let_Result_1;


   --# end


            Match_Result := Scope_Result_1; 
         when Lkt_Match_Expr =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_2 := Create_Internal_Entity_Match_Expr
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Ignored_2 := Cast_Result_2; 
--# bind _ Ignored_2

Let_Result_2 := True; 
--# end
Scope_Result_2 := Let_Result_2;


   --# end


            Match_Result := Scope_Result_2; 
         when Lkt_Dot_Expr =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_3 := Create_Internal_Entity_Dot_Expr
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Ignored_3 := Cast_Result_3; 
--# bind _ Ignored_3

Let_Result_3 := True; 
--# end
Scope_Result_3 := Let_Result_3;


   --# end


            Match_Result := Scope_Result_3; 
         when Lkt_Call_Expr =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_4 := Create_Internal_Entity_Call_Expr
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Ignored_4 := Cast_Result_4; 
--# bind _ Ignored_4

Let_Result_4 := True; 
--# end
Scope_Result_4 := Let_Result_4;


   --# end


            Match_Result := Scope_Result_4; 
         when Lkt_Logic_Predicate =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_5 := Create_Internal_Entity_Logic_Predicate
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Ignored_5 := Cast_Result_5; 
--# bind _ Ignored_5

Let_Result_5 := True; 
--# end
Scope_Result_5 := Let_Result_5;


   --# end


            Match_Result := Scope_Result_5; 
         when Lkt_Logic_Propagate =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_6 := Create_Internal_Entity_Logic_Propagate
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Ignored_6 := Cast_Result_6; 
--# bind _ Ignored_6

Let_Result_6 := True; 
--# end
Scope_Result_6 := Let_Result_6;


   --# end


            Match_Result := Scope_Result_6; 
         when Lkt_Paren_Expr =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_7 := Create_Internal_Entity_Paren_Expr
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Local_P := Cast_Result_7; 
--# bind p Local_P
--# expr-start 104 LktNode.expected_type_entry_point Fld_1 nodes.lkt:273



--# expr-start 103 .parent Fld nodes.lkt:273







   if Local_P.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld := Liblktlang.Implementation.Parent (Node => Local_P.Node, E_Info => Local_P.Info);
--# expr-done 103



   if Fld.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start LktNode.expected_type_entry_point
Fld_1 := Liblktlang.Implementation.Lkt_Node_P_Expected_Type_Entry_Point (Node => Fld.Node, E_Info => Fld.Info);
--# end
--# expr-done 104
Let_Result_7 := Fld_1; 
--# end
Scope_Result_7 := Let_Result_7;


   --# end


            Match_Result := Scope_Result_7; 
         when Lkt_Lambda_Expr =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_8 := Create_Internal_Entity_Lambda_Expr
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Local_P_1 := Cast_Result_8; 
--# bind p Local_P_1
--# expr-start 105 Expr.xref_entry_point Fld_2 nodes.lkt:274







   if Local_P_1.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start '[dispatcher]LktNode.xref_entry_point'
Fld_2 := Liblktlang.Implementation.Dispatcher_Lkt_Node_P_Xref_Entry_Point (Node => Local_P_1.Node, E_Info => Local_P_1.Info);
--# end
--# expr-done 105
Let_Result_8 := Fld_2; 
--# end
Scope_Result_8 := Let_Result_8;


   --# end


            Match_Result := Scope_Result_8; 
         when Lkt_Argument .. Lkt_Array_Literal | Lkt_Logic_Propagate_Call .. Lkt_Block_Expr | Lkt_Error_On_Null .. Lkt_If_Expr | Lkt_Keep_Expr | Lkt_Big_Num_Lit .. Lkt_Logic_Expr | Lkt_Logic_Unify | Lkt_Not_Expr | Lkt_Raise_Expr .. Lkt_Var_Bind =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_9 := Create_Internal_Entity
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Ignored_7 := Cast_Result_9; 
--# bind _ Ignored_7

Let_Result_9 := False; 
--# end
Scope_Result_9 := Let_Result_9;


   --# end


            Match_Result := Scope_Result_9; 
end case;

--# expr-done 106

         Property_Result := Match_Result;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Expected_Type_Entry_Point;
--# end

   







--# property-start LktNode.solve_expected_types nodes.lkt:280
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Solve_Expected_Types
  
  (Node : Bare_Lkt_Node
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Boolean
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
     Ent : Internal_Entity :=
       Internal_Entity'(Node => Self, Info => E_Info);
      --# bind self Ent

   


   Property_Result : Boolean;

      
            procedure Finalizer_Scope_62 with Inline_Always;

      Fld : Logic_Equation;
Solve_Success : Boolean;

            procedure Finalizer_Scope_62 is
            begin
                     Dec_Ref (Fld);
            end Finalizer_Scope_62;

      
      Mmz_Handle : Memoization_Handle;
      Mmz_Val    : Mmz_Value;

      Mmz_Stored : Boolean;
      --  Whether the memoization couple was actually stored. Used to determine
      --  whether to inc-ref the memoized value.

      function Create_Mmz_Key return Mmz_Key;
      --  Create a memoization key for this property call and return it

      --------------------
      -- Create_Mmz_Key --
      --------------------

      function Create_Mmz_Key return Mmz_Key is
      begin
         return Mmz_K : Mmz_Key :=
           (Property => Mmz_Bare_Lkt_Node_Lkt_Node_P_Solve_Expected_Types,
            Items    => new Mmz_Key_Array (1 ..  2))
         do
            Mmz_K.Items (1) := (Kind => Mmz_Bare_Lkt_Node,
                                As_Bare_Lkt_Node => Self);
               Mmz_K.Items (2) :=
                 (Kind => Mmz_Internal_Entity_Info,
                  As_Internal_Entity_Info => E_Info);
         end return;
      end Create_Mmz_Key;

begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;

      if Self /= null then

         if not Self.Unit.Context.In_Populate_Lexical_Env
            and then Find_Memoized_Value
                       (Self.Unit, Mmz_Handle, Mmz_Val, Create_Mmz_Key'Access)
         then
            --# memoization-lookup

            if Mmz_Val.Kind = Mmz_Evaluating then
               --# memoization-return
               Raise_Property_Exception
                 (Self,
                  Property_Error'Identity,
                  "Infinite recursion detected");

            elsif Mmz_Val.Kind = Mmz_Error then
               --# memoization-return
               Reraise_Memoized_Error (Mmz_Val);

            else
               Property_Result := Mmz_Val.As_Boolean;

               --# memoization-return
               return Property_Result;
            end if;
            --# end
         end if;
      end if;

      begin
         
   --# scope-start

         --# expr-start 108 .solve Solve_Success nodes.lkt:280
--# expr-start 107 LktNode.expected_type_equation Fld nodes.lkt:280







   if Ent.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start '[dispatcher]LktNode.expected_type_equation'
Fld := Liblktlang.Implementation.Dispatcher_Lkt_Node_P_Expected_Type_Equation (Node => Ent.Node, E_Info => Ent.Info);
--# end
--# expr-done 107

Solve_Success := Solve_Wrapper (Fld, Self); 
--# expr-done 108

         Property_Result := Solve_Success;
         
   --# end
      Finalizer_Scope_62;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_62;


               if Self /= null
                  and then not Self.Unit.Context.In_Populate_Lexical_Env
               then
                     Add_Memoized_Error
                       (Self.Unit, Mmz_Handle, Exc, Mmz_Stored);
               end if;


            raise;
      end;

      if Self /= null
         and then not Self.Unit.Context.In_Populate_Lexical_Env
      then
         Mmz_Val := (Kind => Mmz_Boolean,
                     As_Boolean => Property_Result);
         Add_Memoized_Value (Self.Unit, Mmz_Handle, Mmz_Val, Mmz_Stored);
      end if;



   return Property_Result;
end Lkt_Node_P_Solve_Expected_Types;
--# end

   







--# property-start LktNode.generic_type_entry_point nodes.lkt:282
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Generic_Type_Entry_Point
  
  (Node : Bare_Lkt_Node
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Boolean
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
     Ent : Internal_Entity :=
       Internal_Entity'(Node => Self, Info => E_Info);
      --# bind self Ent

   


   Property_Result : Boolean;

      

      Is_A : Boolean;



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 109 IsA Is_A nodes.lkt:283

Is_A := Ent.Node /= null 
and then Ent.Node.Kind in Lkt_Call_Expr_Range | Lkt_Logic_Propagate_Range | Lkt_Logic_Predicate_Range; 
--# expr-done 109

         Property_Result := Is_A;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Generic_Type_Entry_Point;
--# end

   







--# property-start LktNode.solve_generic_types nodes.lkt:287
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Solve_Generic_Types
  
  (Node : Bare_Lkt_Node
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Boolean
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
     Ent : Internal_Entity :=
       Internal_Entity'(Node => Self, Info => E_Info);
      --# bind self Ent

   


   Property_Result : Boolean;

      
            procedure Finalizer_Scope_64 with Inline_Always;

      Fld : Logic_Equation;
Solve_Success : Boolean;

            procedure Finalizer_Scope_64 is
            begin
                     Dec_Ref (Fld);
            end Finalizer_Scope_64;

      
      Mmz_Handle : Memoization_Handle;
      Mmz_Val    : Mmz_Value;

      Mmz_Stored : Boolean;
      --  Whether the memoization couple was actually stored. Used to determine
      --  whether to inc-ref the memoized value.

      function Create_Mmz_Key return Mmz_Key;
      --  Create a memoization key for this property call and return it

      --------------------
      -- Create_Mmz_Key --
      --------------------

      function Create_Mmz_Key return Mmz_Key is
      begin
         return Mmz_K : Mmz_Key :=
           (Property => Mmz_Bare_Lkt_Node_Lkt_Node_P_Solve_Generic_Types,
            Items    => new Mmz_Key_Array (1 ..  2))
         do
            Mmz_K.Items (1) := (Kind => Mmz_Bare_Lkt_Node,
                                As_Bare_Lkt_Node => Self);
               Mmz_K.Items (2) :=
                 (Kind => Mmz_Internal_Entity_Info,
                  As_Internal_Entity_Info => E_Info);
         end return;
      end Create_Mmz_Key;

begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;

      if Self /= null then

         if not Self.Unit.Context.In_Populate_Lexical_Env
            and then Find_Memoized_Value
                       (Self.Unit, Mmz_Handle, Mmz_Val, Create_Mmz_Key'Access)
         then
            --# memoization-lookup

            if Mmz_Val.Kind = Mmz_Evaluating then
               --# memoization-return
               Raise_Property_Exception
                 (Self,
                  Property_Error'Identity,
                  "Infinite recursion detected");

            elsif Mmz_Val.Kind = Mmz_Error then
               --# memoization-return
               Reraise_Memoized_Error (Mmz_Val);

            else
               Property_Result := Mmz_Val.As_Boolean;

               --# memoization-return
               return Property_Result;
            end if;
            --# end
         end if;
      end if;

      begin
         
   --# scope-start

         --# expr-start 111 .solve Solve_Success nodes.lkt:287
--# expr-start 110 LktNode.generic_type_equation Fld nodes.lkt:287







   if Ent.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start '[dispatcher]LktNode.generic_type_equation'
Fld := Liblktlang.Implementation.Dispatcher_Lkt_Node_P_Generic_Type_Equation (Node => Ent.Node, E_Info => Ent.Info);
--# end
--# expr-done 110

Solve_Success := Solve_Wrapper (Fld, Self); 
--# expr-done 111

         Property_Result := Solve_Success;
         
   --# end
      Finalizer_Scope_64;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_64;


               if Self /= null
                  and then not Self.Unit.Context.In_Populate_Lexical_Env
               then
                     Add_Memoized_Error
                       (Self.Unit, Mmz_Handle, Exc, Mmz_Stored);
               end if;


            raise;
      end;

      if Self /= null
         and then not Self.Unit.Context.In_Populate_Lexical_Env
      then
         Mmz_Val := (Kind => Mmz_Boolean,
                     As_Boolean => Property_Result);
         Add_Memoized_Value (Self.Unit, Mmz_Handle, Mmz_Val, Mmz_Stored);
      end if;



   return Property_Result;
end Lkt_Node_P_Solve_Generic_Types;
--# end

   







--# property-start LktNode.unmatched_argument nodes.lkt:292
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Unmatched_Argument
  
  (Node : Bare_Lkt_Node
      ; Callee_Type : Internal_Entity_Type_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Boolean
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
     Ent : Internal_Entity :=
       Internal_Entity'(Node => Self, Info => E_Info);
      --# bind self Ent

   

   --# bind callee_type Callee_Type

   Property_Result : Boolean;

      

      Is_Null : Boolean;
Not_Val : Boolean;
Cast_Result : Internal_Entity_Function_Type;
Var_Expr : Internal_Entity_Function_Type;
Fld : Boolean;
Result_Var : Boolean;
If_Result : Boolean;



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 116 BooleanOr If_Result nodes.lkt:293



--# expr-start 113 Not Not_Val nodes.lkt:293
--# expr-start 112 .is_null Is_Null nodes.lkt:293

Is_Null := Ent.Node = null; 
--# expr-done 112
Not_Val := not (Is_Null); 
--# expr-done 113
if Not_Val then
   
   If_Result := True;
else
   


--# expr-start 114 Cast Cast_Result nodes.lkt:294









   if Callee_Type.Node = null
      or else Callee_Type.Node.Kind in Lkt_Function_Type_Range
   then
      
      Cast_Result := Create_Internal_Entity_Function_Type
        (Node => Callee_Type.Node,
         Info => Callee_Type.Info);

   else
         Cast_Result := No_Entity_Function_Type;
   end if;


--# expr-done 114
Var_Expr := Cast_Result;





if Var_Expr /= No_Entity_Function_Type then
   --# expr-start 115 FunctionType.should_ignore_constructor_arg Fld nodes.lkt:294







   if Var_Expr.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start FunctionType.should_ignore_constructor_arg
Fld := Liblktlang.Implementation.Function_Type_P_Should_Ignore_Constructor_Arg (Node => Var_Expr.Node, E_Info => Var_Expr.Info);
--# end
--# expr-done 115
   Result_Var := Fld;
else
   
   Result_Var := False;
end if;



   If_Result := Result_Var;
end if;



--# expr-done 116

         Property_Result := If_Result;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Unmatched_Argument;
--# end

   







--# property-start LktNode.function_type_helper nodes.lkt:298
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Function_Type_Helper
  
  (Node : Bare_Lkt_Node
      ; Param_Types : Internal_Entity_Type_Decl_Array_Access
      ; Return_Type : Internal_Entity_Type_Decl
      ; Origin : Internal_Entity_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity_Function_Type
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
     Ent : Internal_Entity :=
       Internal_Entity'(Node => Self, Info => E_Info);
      --# bind self Ent

   

   --# bind param_types Param_Types
   --# bind return_type Return_Type
   --# bind origin Origin

   Property_Result : Internal_Entity_Function_Type;

      

      Fld : Bare_Synthetic_Type_Ref_List;
New_Node : Bare_Function_Type;
As_Entity : Internal_Entity_Function_Type;
Is_Null : Boolean;
If_Result : Internal_Entity_Function_Type;
Cast_Result : Internal_Entity;
Cast_Expr : Internal_Entity_Type_Decl;
Cast_Result_1 : Internal_Entity;
Is_Equiv : Boolean;
Quantifier_Result : Boolean;
Is_Null_1 : Boolean;
If_Result_1 : Boolean;
If_Result_2 : Internal_Entity_Function_Type;


      
      Mmz_Handle : Memoization_Handle;
      Mmz_Val    : Mmz_Value;

      Mmz_Stored : Boolean;
      --  Whether the memoization couple was actually stored. Used to determine
      --  whether to inc-ref the memoized value.

      function Create_Mmz_Key return Mmz_Key;
      --  Create a memoization key for this property call and return it

      --------------------
      -- Create_Mmz_Key --
      --------------------

      function Create_Mmz_Key return Mmz_Key is
      begin
         return Mmz_K : Mmz_Key :=
           (Property => Mmz_Bare_Lkt_Node_Lkt_Node_P_Function_Type_Helper,
            Items    => new Mmz_Key_Array (1 ..  5))
         do
            Mmz_K.Items (1) := (Kind => Mmz_Bare_Lkt_Node,
                                As_Bare_Lkt_Node => Self);
               Mmz_K.Items (2) := (Kind => Mmz_Internal_Entity_Type_Decl_Array_Access,
                                      As_Internal_Entity_Type_Decl_Array_Access => Param_Types);
                  Inc_Ref (Mmz_K.Items (2).As_Internal_Entity_Type_Decl_Array_Access);
               Mmz_K.Items (3) := (Kind => Mmz_Internal_Entity_Type_Decl,
                                      As_Internal_Entity_Type_Decl => Return_Type);
               Mmz_K.Items (4) := (Kind => Mmz_Internal_Entity_Decl,
                                      As_Internal_Entity_Decl => Origin);
               Mmz_K.Items (5) :=
                 (Kind => Mmz_Internal_Entity_Info,
                  As_Internal_Entity_Info => E_Info);
         end return;
      end Create_Mmz_Key;

begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;

      if Self /= null then

         if not Self.Unit.Context.In_Populate_Lexical_Env
            and then Find_Memoized_Value
                       (Self.Unit, Mmz_Handle, Mmz_Val, Create_Mmz_Key'Access)
         then
            --# memoization-lookup

            if Mmz_Val.Kind = Mmz_Evaluating then
               --# memoization-return
               Raise_Property_Exception
                 (Self,
                  Property_Error'Identity,
                  "Infinite recursion detected");

            elsif Mmz_Val.Kind = Mmz_Error then
               --# memoization-return
               Reraise_Memoized_Error (Mmz_Val);

            else
               Property_Result := Mmz_Val.As_Internal_Entity_Function_Type;

               --# memoization-return
               return Property_Result;
            end if;
            --# end
         end if;
      end if;

      begin
         
   --# scope-start

         --# expr-start 123 If If_Result_2 nodes.lkt:303



--# expr-start 119 BooleanOr If_Result_1 nodes.lkt:303



--# expr-start 117 .contains Quantifier_Result nodes.lkt:303








   

   

   
      Quantifier_Result := False;
   

   

      

      declare
         
         Collection : constant Internal_Entity_Type_Decl_Array_Access := Param_Types;
      begin
         for Item of
               Collection.Items
         loop

            
   --# scope-start



            
      








   
      Cast_Result := Create_Internal_Entity
        (Node => Item.Node,
         Info => Item.Info);








Cast_Expr := No_Entity_Type_Decl; 



   
      Cast_Result_1 := Create_Internal_Entity
        (Node => Cast_Expr.Node,
         Info => Cast_Expr.Info);


Is_Equiv := Equivalent (Cast_Result, Cast_Result_1); 
      Quantifier_Result := Is_Equiv;
   

            
   --# end


               exit when Quantifier_Result;

         end loop;
      end;
   

   
   



--# expr-done 117
if Quantifier_Result then
   
   If_Result_1 := True;
else
   --# expr-start 118 .is_null Is_Null_1 nodes.lkt:303

Is_Null_1 := Return_Type.Node = null; 
--# expr-done 118
   If_Result_1 := Is_Null_1;
end if;



--# expr-done 119
if If_Result_1 then
   
   If_Result_2 := No_Entity_Function_Type;
else
   --# expr-start 122 .as_entity If_Result nodes.lkt:306



--# expr-start 121 'New[FunctionType]' New_Node nodes.lkt:306



--# expr-start 120 LktNode.get_empty_type_ref_list Fld nodes.lkt:310







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start LktNode.get_empty_type_ref_list
Fld := Liblktlang.Implementation.Lkt_Node_P_Get_Empty_Type_Ref_List (Node => Self);
--# end
--# expr-done 120
Inc_Ref (Param_Types);


if Is_Foreign_Strict (Self.Self_Env, Self) then
   Raise_Property_Exception
     (Self,
      Property_Error'Identity,
      "synthetic nodes cannot have foreign lexical envs");
end if;


New_Node := new Root_Node_Record
  (Lkt_Function_Type);
Initialize
  (Self => New_Node,
   Kind => Lkt_Function_Type,
   Unit => Self.Unit,

   Token_Start_Index => No_Token_Index,
   Token_End_Index   => No_Token_Index,

   Parent => Self,

   Self_Env => Self.Self_Env);
Register_Destroyable (Self.Unit, New_Node);

   Initialize_Fields_For_Function_Type
     (Self => New_Node,
      Function_Type_F_Traits => Fld);

      New_Node.Function_Type_F_Origin :=
         Origin;
      New_Node.Function_Type_F_Params :=
         Param_Types;
      New_Node.Function_Type_F_Return_Type :=
         Return_Type;

--# expr-done 121
Is_Null := New_Node = null; 
if Is_Null then
   
   If_Result := No_Entity_Function_Type;
else
   

As_Entity := (Info => E_Info, Node => New_Node); 
   If_Result := As_Entity;
end if;



--# expr-done 122
   If_Result_2 := If_Result;
end if;



--# expr-done 123

         Property_Result := If_Result_2;
         
   --# end


      exception
         when Exc : Property_Error =>


               if Self /= null
                  and then not Self.Unit.Context.In_Populate_Lexical_Env
               then
                     Add_Memoized_Error
                       (Self.Unit, Mmz_Handle, Exc, Mmz_Stored);
               end if;


            raise;
      end;

      if Self /= null
         and then not Self.Unit.Context.In_Populate_Lexical_Env
      then
         Mmz_Val := (Kind => Mmz_Internal_Entity_Function_Type,
                     As_Internal_Entity_Function_Type => Property_Result);
         Add_Memoized_Value (Self.Unit, Mmz_Handle, Mmz_Val, Mmz_Stored);
      end if;



   return Property_Result;
end Lkt_Node_P_Function_Type_Helper;
--# end

   







--# property-start LktNode.shed_rebindings nodes.lkt:316
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Shed_Rebindings
  
  (Node : Bare_Lkt_Node
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
     Ent : Internal_Entity :=
       Internal_Entity'(Node => Self, Info => E_Info);
      --# bind self Ent

   


   Property_Result : Internal_Entity;

      
            procedure Finalizer_Scope_68 with Inline_Always;

      Fld : Lexical_Env := Empty_Env;
Fld_1 : Internal_Entity_Info;
Shed_Entity_Info : Internal_Entity_Info;
New_Struct : Internal_Entity;

            procedure Finalizer_Scope_68 is
            begin
                     Dec_Ref (Fld);
            end Finalizer_Scope_68;


begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      begin
         
   --# scope-start

         --# expr-start 127 'New[Entity[LktNode]]' New_Struct nodes.lkt:317
--# expr-start 126 .shed_rebindings Shed_Entity_Info nodes.lkt:319
--# expr-start 124 'Entity[LktNode].info' Fld_1 nodes.lkt:319









Fld_1 := Ent.Info;
--# expr-done 124
--# expr-start 125 LktNode.children_env Fld nodes.lkt:319







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld := Liblktlang.Implementation.Children_Env (Node => Self);
--# expr-done 125
Shed_Entity_Info := AST_Envs.Shed_Rebindings (Fld_1, Fld); 
--# expr-done 126

New_Struct := (Info => Shed_Entity_Info, Node => Self); 
--# expr-done 127

         Property_Result := New_Struct;
         
   --# end
      Finalizer_Scope_68;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_68;




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Shed_Rebindings;
--# end

   







--# property-start '[dispatcher]LktNode.xref_entry_point' dispatcher
pragma Warnings (Off, "is not referenced");

function Dispatcher_Lkt_Node_P_Xref_Entry_Point
  
  (Node : Bare_Lkt_Node
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Boolean
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
     Ent : Internal_Entity :=
       Internal_Entity'(Node => Self, Info => E_Info);
      --# bind self Ent

   


   Property_Result : Boolean;



begin
   --# property-body-start

   pragma Assert (Self = Node);





      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dispatching on null node");
      end if;

      case Lkt_Lkt_Node (Self.Kind) is
               when Lkt_Argument .. Lkt_Self_Decl | Lkt_Error_Decl .. Lkt_Trait_Decl | Lkt_Decl_Annotation_Args .. Lkt_Excludes_Null_Present | Lkt_Full_Decl .. Lkt_Selector_Call =>
                  --# property-call-start 'LktNode.[root-static]xref_entry_point'
                  Property_Result := Lkt_Node_P_Xref_Entry_Point
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Field_Decl .. Lkt_Lambda_Param_Decl | Lkt_Val_Decl .. Lkt_Fun_Decl =>
                  --# property-call-start UserValDecl.xref_entry_point
                  Property_Result := User_Val_Decl_P_Xref_Entry_Point
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Binding_Val_Decl =>
                  --# property-call-start BindingValDecl.xref_entry_point
                  Property_Result := Binding_Val_Decl_P_Xref_Entry_Point
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Enum_Lit_Decl =>
                  --# property-call-start EnumLitDecl.xref_entry_point
                  Property_Result := Enum_Lit_Decl_P_Xref_Entry_Point
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Dyn_Var_Decl =>
                  --# property-call-start DynVarDecl.xref_entry_point
                  Property_Result := Dyn_Var_Decl_P_Xref_Entry_Point
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Match_Val_Decl =>
                  --# property-call-start MatchValDecl.xref_entry_point
                  Property_Result := Match_Val_Decl_P_Xref_Entry_Point
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Env_Spec_Decl =>
                  --# property-call-start EnvSpecDecl.xref_entry_point
                  Property_Result := Env_Spec_Decl_P_Xref_Entry_Point
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Decl_Annotation =>
                  --# property-call-start DeclAnnotation.xref_entry_point
                  Property_Result := Decl_Annotation_P_Xref_Entry_Point
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Any_Of .. Lkt_Un_Op =>
                  --# property-call-start Expr.xref_entry_point
                  Property_Result := Expr_P_Xref_Entry_Point
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Default_List_Type_Ref .. Lkt_Simple_Type_Ref =>
                  --# property-call-start TypeRef.xref_entry_point
                  Property_Result := Type_Ref_P_Xref_Entry_Point
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Var_Bind =>
                  --# property-call-start VarBind.xref_entry_point
                  Property_Result := Var_Bind_P_Xref_Entry_Point
                    (Self
                        , E_Info
                    );
                  --# end
      end case;




   return Property_Result;
end Dispatcher_Lkt_Node_P_Xref_Entry_Point;
--# end

   







--# property-start '[dispatcher]LktNode.xref_equation' dispatcher
pragma Warnings (Off, "is not referenced");

function Dispatcher_Lkt_Node_P_Xref_Equation
  
  (Node : Bare_Lkt_Node
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Logic_Equation
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
     Ent : Internal_Entity :=
       Internal_Entity'(Node => Self, Info => E_Info);
      --# bind self Ent

   


   Property_Result : Logic_Equation;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dispatching on null node");
      end if;

      case Lkt_Lkt_Node (Self.Kind) is
               when Lkt_Lexer_Case_Rule_Cond_Alt .. Lkt_Enum_Lit_Decl | Lkt_Dyn_Var_Decl .. Lkt_Match_Val_Decl | Lkt_Error_Decl .. Lkt_Trait_Decl | Lkt_Decl_Annotation_Args .. Lkt_Excludes_Null_Present | Lkt_Grammar_Cut .. Lkt_Module_Ref_Id | Lkt_Full_Decl .. Lkt_Default_List_Type_Ref =>
                  --# property-call-start 'LktNode.[root-static]xref_equation'
                  Property_Result := Lkt_Node_P_Xref_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Argument =>
                  --# property-call-start Argument.xref_equation
                  Property_Result := Argument_P_Xref_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Field_Decl .. Lkt_Lambda_Param_Decl =>
                  --# property-call-start ComponentDecl.xref_equation
                  Property_Result := Component_Decl_P_Xref_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Val_Decl =>
                  --# property-call-start ValDecl.xref_equation
                  Property_Result := Val_Decl_P_Xref_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Fun_Decl =>
                  --# property-call-start FunDecl.xref_equation
                  Property_Result := Fun_Decl_P_Xref_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Env_Spec_Decl =>
                  --# property-call-start EnvSpecDecl.xref_equation
                  Property_Result := Env_Spec_Decl_P_Xref_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Decl_Annotation =>
                  --# property-call-start DeclAnnotation.xref_equation
                  Property_Result := Decl_Annotation_P_Xref_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Any_Of =>
                  --# property-call-start AnyOf.xref_equation
                  Property_Result := Any_Of_P_Xref_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Array_Literal =>
                  --# property-call-start ArrayLiteral.xref_equation
                  Property_Result := Array_Literal_P_Xref_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Call_Expr | Lkt_Logic_Propagate_Call =>
                  --# property-call-start BaseCallExpr.xref_equation
                  Property_Result := Base_Call_Expr_P_Xref_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Logic_Predicate =>
                  --# property-call-start LogicPredicate.xref_equation
                  Property_Result := Logic_Predicate_P_Xref_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Bin_Op =>
                  --# property-call-start BinOp.xref_equation
                  Property_Result := Bin_Op_P_Xref_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Block_Expr =>
                  --# property-call-start BlockExpr.xref_equation
                  Property_Result := Block_Expr_P_Xref_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Cast_Expr =>
                  --# property-call-start CastExpr.xref_equation
                  Property_Result := Cast_Expr_P_Xref_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Dot_Expr =>
                  --# property-call-start DotExpr.xref_equation
                  Property_Result := Dot_Expr_P_Xref_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Error_On_Null =>
                  --# property-call-start ErrorOnNull.xref_equation
                  Property_Result := Error_On_Null_P_Xref_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Generic_Instantiation =>
                  --# property-call-start GenericInstantiation.xref_equation
                  Property_Result := Generic_Instantiation_P_Xref_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Ref_Id =>
                  --# property-call-start RefId.xref_equation
                  Property_Result := Ref_Id_P_Xref_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_If_Expr =>
                  --# property-call-start IfExpr.xref_equation
                  Property_Result := If_Expr_P_Xref_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Isa =>
                  --# property-call-start Isa.xref_equation
                  Property_Result := Isa_P_Xref_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Keep_Expr =>
                  --# property-call-start KeepExpr.xref_equation
                  Property_Result := Keep_Expr_P_Xref_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Lambda_Expr =>
                  --# property-call-start LambdaExpr.xref_equation
                  Property_Result := Lambda_Expr_P_Xref_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Big_Num_Lit =>
                  --# property-call-start BigNumLit.xref_equation
                  Property_Result := Big_Num_Lit_P_Xref_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Char_Lit =>
                  --# property-call-start CharLit.xref_equation
                  Property_Result := Char_Lit_P_Xref_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Null_Lit =>
                  --# property-call-start NullLit.xref_equation
                  Property_Result := Null_Lit_P_Xref_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Num_Lit =>
                  --# property-call-start NumLit.xref_equation
                  Property_Result := Num_Lit_P_Xref_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Block_String_Lit .. Lkt_Pattern_Single_Line_String_Lit =>
                  --# property-call-start StringLit.xref_equation
                  Property_Result := String_Lit_P_Xref_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Logic_Assign =>
                  --# property-call-start LogicAssign.xref_equation
                  Property_Result := Logic_Assign_P_Xref_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Logic_Expr =>
                  --# property-call-start LogicExpr.xref_equation
                  Property_Result := Logic_Expr_P_Xref_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Logic_Propagate =>
                  --# property-call-start LogicPropagate.xref_equation
                  Property_Result := Logic_Propagate_P_Xref_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Logic_Unify =>
                  --# property-call-start LogicUnify.xref_equation
                  Property_Result := Logic_Unify_P_Xref_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Match_Expr =>
                  --# property-call-start MatchExpr.xref_equation
                  Property_Result := Match_Expr_P_Xref_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Not_Expr =>
                  --# property-call-start NotExpr.xref_equation
                  Property_Result := Not_Expr_P_Xref_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Paren_Expr =>
                  --# property-call-start ParenExpr.xref_equation
                  Property_Result := Paren_Expr_P_Xref_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Raise_Expr =>
                  --# property-call-start RaiseExpr.xref_equation
                  Property_Result := Raise_Expr_P_Xref_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Subscript_Expr =>
                  --# property-call-start SubscriptExpr.xref_equation
                  Property_Result := Subscript_Expr_P_Xref_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Try_Expr =>
                  --# property-call-start TryExpr.xref_equation
                  Property_Result := Try_Expr_P_Xref_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Un_Op =>
                  --# property-call-start UnOp.xref_equation
                  Property_Result := Un_Op_P_Xref_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Function_Type_Ref =>
                  --# property-call-start FunctionTypeRef.xref_equation
                  Property_Result := Function_Type_Ref_P_Xref_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Generic_Type_Ref =>
                  --# property-call-start GenericTypeRef.xref_equation
                  Property_Result := Generic_Type_Ref_P_Xref_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Simple_Type_Ref =>
                  --# property-call-start SimpleTypeRef.xref_equation
                  Property_Result := Simple_Type_Ref_P_Xref_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Var_Bind =>
                  --# property-call-start VarBind.xref_equation
                  Property_Result := Var_Bind_P_Xref_Equation
                    (Self
                        , E_Info
                    );
                  --# end
      end case;




   return Property_Result;
end Dispatcher_Lkt_Node_P_Xref_Equation;
--# end

   







--# property-start '[dispatcher]LktNode.expected_type_equation' dispatcher
pragma Warnings (Off, "is not referenced");

function Dispatcher_Lkt_Node_P_Expected_Type_Equation
  
  (Node : Bare_Lkt_Node
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Logic_Equation
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
     Ent : Internal_Entity :=
       Internal_Entity'(Node => Self, Info => E_Info);
      --# bind self Ent

   


   Property_Result : Logic_Equation;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dispatching on null node");
      end if;

      case Lkt_Lkt_Node (Self.Kind) is
               when Lkt_Lexer_Case_Rule_Cond_Alt .. Lkt_Excludes_Null_Present | Lkt_Full_Decl .. Lkt_Var_Bind =>
                  --# property-call-start 'LktNode.[root-static]expected_type_equation'
                  Property_Result := Lkt_Node_P_Expected_Type_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Argument =>
                  --# property-call-start Argument.expected_type_equation
                  Property_Result := Argument_P_Expected_Type_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Any_Of .. Lkt_Array_Literal | Lkt_Bin_Op .. Lkt_Block_Expr | Lkt_Error_On_Null .. Lkt_If_Expr | Lkt_Keep_Expr | Lkt_Big_Num_Lit .. Lkt_Logic_Expr | Lkt_Logic_Unify .. Lkt_Not_Expr | Lkt_Raise_Expr .. Lkt_Un_Op =>
                  --# property-call-start Expr.expected_type_equation
                  Property_Result := Expr_P_Expected_Type_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Call_Expr | Lkt_Logic_Propagate_Call =>
                  --# property-call-start BaseCallExpr.expected_type_equation
                  Property_Result := Base_Call_Expr_P_Expected_Type_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Logic_Predicate =>
                  --# property-call-start LogicPredicate.expected_type_equation
                  Property_Result := Logic_Predicate_P_Expected_Type_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Cast_Expr =>
                  --# property-call-start CastExpr.expected_type_equation
                  Property_Result := Cast_Expr_P_Expected_Type_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Dot_Expr =>
                  --# property-call-start DotExpr.expected_type_equation
                  Property_Result := Dot_Expr_P_Expected_Type_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Isa =>
                  --# property-call-start Isa.expected_type_equation
                  Property_Result := Isa_P_Expected_Type_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Lambda_Expr =>
                  --# property-call-start LambdaExpr.expected_type_equation
                  Property_Result := Lambda_Expr_P_Expected_Type_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Logic_Propagate =>
                  --# property-call-start LogicPropagate.expected_type_equation
                  Property_Result := Logic_Propagate_P_Expected_Type_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Paren_Expr =>
                  --# property-call-start ParenExpr.expected_type_equation
                  Property_Result := Paren_Expr_P_Expected_Type_Equation
                    (Self
                        , E_Info
                    );
                  --# end
      end case;




   return Property_Result;
end Dispatcher_Lkt_Node_P_Expected_Type_Equation;
--# end

   







--# property-start '[dispatcher]LktNode.generic_type_equation' dispatcher
pragma Warnings (Off, "is not referenced");

function Dispatcher_Lkt_Node_P_Generic_Type_Equation
  
  (Node : Bare_Lkt_Node
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Logic_Equation
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
     Ent : Internal_Entity :=
       Internal_Entity'(Node => Self, Info => E_Info);
      --# bind self Ent

   


   Property_Result : Logic_Equation;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dispatching on null node");
      end if;

      case Lkt_Lkt_Node (Self.Kind) is
               when Lkt_Argument .. Lkt_Array_Literal | Lkt_Bin_Op .. Lkt_Cast_Expr | Lkt_Error_On_Null .. Lkt_Module_Ref_Id | Lkt_If_Expr .. Lkt_Keep_Expr | Lkt_Big_Num_Lit .. Lkt_Logic_Expr | Lkt_Logic_Unify .. Lkt_Var_Bind =>
                  --# property-call-start 'LktNode.[root-static]generic_type_equation'
                  Property_Result := Lkt_Node_P_Generic_Type_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Call_Expr | Lkt_Logic_Propagate_Call =>
                  --# property-call-start BaseCallExpr.generic_type_equation
                  Property_Result := Base_Call_Expr_P_Generic_Type_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Logic_Predicate =>
                  --# property-call-start LogicPredicate.generic_type_equation
                  Property_Result := Logic_Predicate_P_Generic_Type_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Dot_Expr =>
                  --# property-call-start DotExpr.generic_type_equation
                  Property_Result := Dot_Expr_P_Generic_Type_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Ref_Id =>
                  --# property-call-start RefId.generic_type_equation
                  Property_Result := Ref_Id_P_Generic_Type_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Lambda_Expr =>
                  --# property-call-start LambdaExpr.generic_type_equation
                  Property_Result := Lambda_Expr_P_Generic_Type_Equation
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Logic_Propagate =>
                  --# property-call-start LogicPropagate.generic_type_equation
                  Property_Result := Logic_Propagate_P_Generic_Type_Equation
                    (Self
                        , E_Info
                    );
                  --# end
      end case;




   return Property_Result;
end Dispatcher_Lkt_Node_P_Generic_Type_Equation;
--# end

   







--# property-start '[dispatcher]LktNode.complete' dispatcher
pragma Warnings (Off, "is not referenced");

function Dispatcher_Lkt_Node_P_Complete
  
  (Node : Bare_Lkt_Node
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Complete_Item_Array_Access
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
     Ent : Internal_Entity :=
       Internal_Entity'(Node => Self, Info => E_Info);
      --# bind self Ent

   


   Property_Result : Internal_Complete_Item_Array_Access;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dispatching on null node");
      end if;

      case Lkt_Lkt_Node (Self.Kind) is
               when Lkt_Argument .. Lkt_Cast_Expr | Lkt_Error_On_Null .. Lkt_Module_Ref_Id | Lkt_If_Expr .. Lkt_Var_Bind =>
                  --# property-call-start 'LktNode.[root-static]complete'
                  Property_Result := Lkt_Node_P_Complete
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Dot_Expr =>
                  --# property-call-start DotExpr.complete
                  Property_Result := Dot_Expr_P_Complete
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Ref_Id =>
                  --# property-call-start RefId.complete
                  Property_Result := Ref_Id_P_Complete
                    (Self
                        , E_Info
                    );
                  --# end
      end case;




   return Property_Result;
end Dispatcher_Lkt_Node_P_Complete;
--# end

   







--# property-start LktNode.can_reach '<builtin>:0'
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Can_Reach
  
  (Node : Bare_Lkt_Node
      ; From_Node : Bare_Lkt_Node
  )

   return Boolean
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
      --# bind node Self

   

   --# bind from_node From_Node

   Property_Result : Boolean;

      

      Is_Equal : Boolean;
Fld : Internal_Unit;
Fld_1 : Internal_Unit;
Is_Equal_1 : Boolean;
Not_Val : Boolean;
Node_Comp : Boolean;
If_Result : Boolean;
If_Result_1 : Boolean;



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         




Is_Equal := From_Node = No_Bare_Lkt_Node; 
if Is_Equal then
   
   If_Result_1 := True;
else
   









   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld := Liblktlang.Implementation.Unit (Node => Self);







   if From_Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_1 := Liblktlang.Implementation.Unit (Node => From_Node);
Is_Equal_1 := Fld = Fld_1; 
Not_Val := not (Is_Equal_1); 
if Not_Val then
   
   If_Result := True;
else
   


Node_Comp := Compare (Self, Self, From_Node, Less_Than); 
   If_Result := Node_Comp;
end if;



   If_Result_1 := If_Result;
end if;




         Property_Result := If_Result_1;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Can_Reach;
--# end

   







--# property-start 'LktNode.[root-static]xref_entry_point' nodes.lkt:326
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Xref_Entry_Point
  
  (Node : Bare_Lkt_Node
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Boolean
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
     Ent : Internal_Entity :=
       Internal_Entity'(Node => Self, Info => E_Info);
      --# bind self Ent

   


   Property_Result : Boolean;

      

      



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         

         Property_Result := False;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Xref_Entry_Point;
--# end

   







--# property-start 'LktNode.[root-static]xref_equation' nodes.lkt:331
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Xref_Equation
  
  (Node : Bare_Lkt_Node
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Logic_Equation
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
     Ent : Internal_Entity :=
       Internal_Entity'(Node => Self, Info => E_Info);
      --# bind self Ent

   


   Property_Result : Logic_Equation;

      

      



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         

         Property_Result := raise Property_Error with "Property LktNode.xref_equation not implemented";
            Inc_Ref (Property_Result);
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Xref_Equation;
--# end

   







--# property-start 'LktNode.[root-static]expected_type_equation' nodes.lkt:337
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Expected_Type_Equation
  
  (Node : Bare_Lkt_Node
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Logic_Equation
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
     Ent : Internal_Entity :=
       Internal_Entity'(Node => Self, Info => E_Info);
      --# bind self Ent

   


   Property_Result : Logic_Equation;

      

      



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         

         Property_Result := raise Property_Error with "Property LktNode.expected_type_equation not implemented";
            Inc_Ref (Property_Result);
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Expected_Type_Equation;
--# end

   







--# property-start 'LktNode.[root-static]generic_type_equation' nodes.lkt:343
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Generic_Type_Equation
  
  (Node : Bare_Lkt_Node
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Logic_Equation
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
     Ent : Internal_Entity :=
       Internal_Entity'(Node => Self, Info => E_Info);
      --# bind self Ent

   


   Property_Result : Logic_Equation;

      

      



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         

         Property_Result := raise Property_Error with "Property LktNode.generic_type_equation not implemented";
            Inc_Ref (Property_Result);
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Generic_Type_Equation;
--# end

   







--# property-start 'LktNode.[root-static]complete' nodes.lkt:350
pragma Warnings (Off, "is not referenced");

function Lkt_Node_P_Complete
  
  (Node : Bare_Lkt_Node
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Complete_Item_Array_Access
is
   Self : Bare_Lkt_Node :=
     Bare_Lkt_Node (Node);
     Ent : Internal_Entity :=
       Internal_Entity'(Node => Self, Info => E_Info);
      --# bind self Ent

   


   Property_Result : Internal_Complete_Item_Array_Access;

      
            procedure Finalizer_Scope_74 with Inline_Always;
            procedure Finalizer_Scope_468 with Inline_Always;

      Fld : Internal_Unit;
Fld_1 : Bare_Lkt_Node;
Fld_2 : Lexical_Env := Empty_Env;
Env_Get_Result : Internal_Entity_Array_Access;
Local_Items : Internal_Entity_Array_Access;
Cast_Result : Internal_Entity_Decl;
New_Struct : Internal_Complete_Item;
Map_Result : Internal_Complete_Item_Array_Access;
Fld_3 : Internal_Entity_Array_Access;
Is_A : Boolean;
Quantifier_Result : Boolean;
Cast_Result_1 : Internal_Entity_Decl;
New_Struct_1 : Internal_Complete_Item;
Is_A_1 : Boolean;
Map_Result_1 : Internal_Complete_Item_Array_Access;
If_Result : Internal_Complete_Item_Array_Access;
Let_Result : Internal_Complete_Item_Array_Access;
Scope_Result : Internal_Complete_Item_Array_Access;

            procedure Finalizer_Scope_74 is
            begin
                     Dec_Ref (Scope_Result);
            end Finalizer_Scope_74;
            procedure Finalizer_Scope_468 is
            begin
                     Dec_Ref (Fld_2);
                     Dec_Ref (Env_Get_Result);
                     Dec_Ref (Local_Items);
                     Dec_Ref (Map_Result);
                     Dec_Ref (Fld_3);
                     Dec_Ref (Map_Result_1);
                     Dec_Ref (If_Result);
                     Dec_Ref (Let_Result);
            end Finalizer_Scope_468;


begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      begin
         
   --# scope-start

         



   --# scope-start



--# expr-start 143 ValDecl Let_Result nodes.lkt:352
--# scope-start
--# expr-start 131 .get Env_Get_Result nodes.lkt:352
--# expr-start 130 LktNode.node_env Fld_2 nodes.lkt:352



--# expr-start 129 AnalysisUnit.root Fld_1 nodes.lkt:352



--# expr-start 128 LktNode.unit Fld nodes.lkt:352







   if Ent.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld := Liblktlang.Implementation.Unit (Node => Ent.Node);
--# expr-done 128



   if Fld = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_1 := Fld.Ast_Root;
--# expr-done 129



   if Fld_1 = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_2 := Liblktlang.Implementation.Node_Env (Node => Fld_1);
--# expr-done 130



Env_Get_Result := Construct_Entity_Array (AST_Envs.Get (Self => Fld_2, Key => Thin (No_Symbol), Lookup_Kind => To_Lookup_Kind_Type (Recursive), Categories => (Nocat => True))); 
--# expr-done 131
Local_Items := Env_Get_Result; Inc_Ref (Local_Items);
--# bind items Local_Items
--# expr-start 142 If If_Result nodes.lkt:354



--# expr-start 134 .any Quantifier_Result nodes.lkt:354








   

   --# expr-start 132 LktNode.parents Fld_3 nodes.lkt:354







   if Ent.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;



Fld_3 := Liblktlang.Implementation.Parents (Node => Ent.Node, With_Self => True, E_Info => Ent.Info);
--# expr-done 132

   
      Quantifier_Result := False;
   

   

      

      declare
         
         Collection : constant Internal_Entity_Array_Access := Fld_3;
      begin
         for Item_1 of
               Collection.Items
         loop

            
   --# scope-start


               --# bind p Item_1

            
      --# expr-start 133 IsA Is_A nodes.lkt:354

Is_A := Item_1.Node /= null 
and then Item_1.Node.Kind in Lkt_Type_Ref; 
--# expr-done 133
      Quantifier_Result := Is_A;
   

            
   --# end


               exit when Quantifier_Result;

         end loop;
      end;
   

   
   



--# expr-done 134
if Quantifier_Result then
   --# expr-start 138 .filtermap Map_Result_1 nodes.lkt:355








   

   

   
      declare
         Map_Result_1_Vec : Internal_Complete_Item_Vectors.Vector;
      begin
   

   

      

      declare
         
         Collection : constant Internal_Entity_Array_Access := Local_Items;
      begin
         for Item_2 of
               Collection.Items
         loop

            
   --# scope-start


               --# bind d Item_2

            
         --# expr-start 135 IsA Is_A_1 nodes.lkt:357

Is_A_1 := Item_2.Node /= null 
and then Item_2.Node.Kind in Lkt_Type_Decl; 
--# expr-done 135
         if Is_A_1 then
            
         
      --# expr-start 137 'New[CompleteItem]' New_Struct_1 nodes.lkt:356
--# expr-start 136 Cast Cast_Result_1 nodes.lkt:356









   if Item_2.Node = null
      or else Item_2.Node.Kind in Lkt_Decl
   then
      
      Cast_Result_1 := Create_Internal_Entity_Decl
        (Node => Item_2.Node,
         Info => Item_2.Info);

   else
         Cast_Result_1 := No_Entity_Decl;
   end if;


--# expr-done 136
New_Struct_1 := (Declaration => Cast_Result_1); 
--# expr-done 137
      

         declare
            Item_To_Append : constant Internal_Complete_Item := New_Struct_1;
         begin
            Internal_Complete_Item_Vectors.Append (Map_Result_1_Vec, Item_To_Append);
         end;
   
   
         end if;
   

            
   --# end



         end loop;
      end;
   

   

         Map_Result_1 := Create_Internal_Complete_Item_Array
           (Items_Count => Natural (Internal_Complete_Item_Vectors.Length (Map_Result_1_Vec)));
         for I in Map_Result_1.Items'Range loop
            Map_Result_1.Items (I) := Internal_Complete_Item_Vectors.Get
              (Map_Result_1_Vec,
               I + Internal_Complete_Item_Vectors.Index_Type'First - Map_Result_1.Items'First);
         end loop;
         Internal_Complete_Item_Vectors.Destroy (Map_Result_1_Vec);

      end;
   



--# expr-done 138
   If_Result := Map_Result_1;
else
   --# expr-start 141 .map Map_Result nodes.lkt:359








   

   

   
      declare
         Map_Result_Vec : Internal_Complete_Item_Vectors.Vector;
      begin
   

   

      

      declare
         
         Collection : constant Internal_Entity_Array_Access := Local_Items;
      begin
         for Item of
               Collection.Items
         loop

            
   --# scope-start


               --# bind d Item

            
         
         
      --# expr-start 140 'New[CompleteItem]' New_Struct nodes.lkt:359
--# expr-start 139 Cast Cast_Result nodes.lkt:359









   if Item.Node = null
      or else Item.Node.Kind in Lkt_Decl
   then
      
      Cast_Result := Create_Internal_Entity_Decl
        (Node => Item.Node,
         Info => Item.Info);

   else
         Cast_Result := No_Entity_Decl;
   end if;


--# expr-done 139
New_Struct := (Declaration => Cast_Result); 
--# expr-done 140
      

         declare
            Item_To_Append : constant Internal_Complete_Item := New_Struct;
         begin
            Internal_Complete_Item_Vectors.Append (Map_Result_Vec, Item_To_Append);
         end;
   
   
   

            
   --# end



         end loop;
      end;
   

   

         Map_Result := Create_Internal_Complete_Item_Array
           (Items_Count => Natural (Internal_Complete_Item_Vectors.Length (Map_Result_Vec)));
         for I in Map_Result.Items'Range loop
            Map_Result.Items (I) := Internal_Complete_Item_Vectors.Get
              (Map_Result_Vec,
               I + Internal_Complete_Item_Vectors.Index_Type'First - Map_Result.Items'First);
         end loop;
         Internal_Complete_Item_Vectors.Destroy (Map_Result_Vec);

      end;
   



--# expr-done 141
   If_Result := Map_Result;
end if;

      Inc_Ref (If_Result);


--# expr-done 142
Let_Result := If_Result; Inc_Ref (Let_Result);
--# end
--# expr-done 143
Scope_Result := Let_Result;
   Inc_Ref (Scope_Result);


   --# end
      Finalizer_Scope_468;



         Property_Result := Scope_Result;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_74;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_74;
                     Finalizer_Scope_468;




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Complete;
--# end


   
function Lkt_Node_P_Shed_Rebindings
  (E : Entity
  ) return Internal_Entity
is
   

      E_Info : Internal_Entity_Info :=
         Shed_Rebindings (E.Info, Children_Env (E.Node));
      
begin
   

   declare
      Result : constant Internal_Entity :=
         Lkt_Node_P_Shed_Rebindings (E.Node, E_Info);
   begin
         return (Node => Result.Node, Info => Result.Info);
   end;
end;


   


      

   --
   --  Primitives for Bare_Argument
   --

   



      
      procedure Initialize_Fields_For_Argument
        (Self : Bare_Argument
         ; Argument_F_Name : Bare_Ref_Id
         ; Argument_F_Value : Bare_Expr
        ) is
      begin

            Self.Argument_F_Name := Argument_F_Name;
            Self.Argument_F_Value := Argument_F_Value;
         

      end Initialize_Fields_For_Argument;

      
   function Argument_F_Name
     (Node : Bare_Argument) return Bare_Ref_Id
   is
      

   begin
         
         return Node.Argument_F_Name;
      
   end;

      
   function Argument_F_Value
     (Node : Bare_Argument) return Bare_Expr
   is
      

   begin
         
         return Node.Argument_F_Value;
      
   end;


   







--# property-start Argument.expected_type_equation nodes.lkt:5463
pragma Warnings (Off, "is not referenced");

function Argument_P_Expected_Type_Equation
  
  (Node : Bare_Argument
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Logic_Equation
is
   Self : Bare_Argument :=
     Bare_Argument (Node);
     Ent : Internal_Entity_Argument :=
       Internal_Entity_Argument'(Node => Self, Info => E_Info);
      --# bind self Ent

   


   Property_Result : Logic_Equation;

      
            procedure Finalizer_Scope_408 with Inline_Always;

      True_Rel : Logic_Equation;
Fld : Internal_Entity_Expr;
Fld_1 : Boolean;
Fld_2 : Internal_Entity_Expr;
Fld_3 : Logic_Var;
Cast_Expr : Internal_Entity_Type_Decl;
Cast_Result : Internal_Entity;
Bind_Result : Logic_Equation;
If_Result : Logic_Equation;

            procedure Finalizer_Scope_408 is
            begin
                     Dec_Ref (True_Rel);
                     Dec_Ref (Bind_Result);
                     Dec_Ref (If_Result);
            end Finalizer_Scope_408;


begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 150 If If_Result nodes.lkt:5467



--# expr-start 145 Expr.has_context_free_type Fld_1 nodes.lkt:5467



--# expr-start 144 Argument.value Fld nodes.lkt:5467







   if Ent.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld := Create_Internal_Entity_Expr (Node => Ent.Node.Argument_F_Value, Info => Ent.Info);
--# expr-done 144



   if Fld.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start '[dispatcher]Expr.has_context_free_type'
Fld_1 := Liblktlang.Implementation.Dispatcher_Expr_P_Has_Context_Free_Type (Node => Fld.Node, E_Info => Fld.Info);
--# end
--# expr-done 145
if Fld_1 then
   --# expr-start 148 LogicAssign Bind_Result nodes.lkt:5468
--# expr-start 147 Expr.expected_type_var Fld_3 nodes.lkt:5468



--# expr-start 146 Argument.value Fld_2 nodes.lkt:5468







   if Ent.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_2 := Create_Internal_Entity_Expr (Node => Ent.Node.Argument_F_Value, Info => Ent.Info);
--# expr-done 146



   if Fld_2.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_3 := Fld_2.Node.Expr_F_Expected_Type_Var'Unrestricted_Access;
--# expr-done 147
Fld_3.Value := No_Entity;
Entity_Vars.Reset (Fld_3);






Cast_Expr := No_Entity_Type_Decl; 



   
      Cast_Result := Create_Internal_Entity
        (Node => Cast_Expr.Node,
         Info => Cast_Expr.Info);


Bind_Result := Solver.Create_Assign (Fld_3, Cast_Result, Solver_Ifc.No_Converter, Debug_String => (if Liblktlang_Support.Adalog.Debug.Debug then New_Unit_String (Node.Unit, "nodes.lkt:5468:14") else null)); 
--# expr-done 148
   If_Result := Bind_Result;
else
   --# expr-start 149 LogicTrue True_Rel nodes.lkt:5469
True_Rel := Solver.Create_True ((if Liblktlang_Support.Adalog.Debug.Debug then New_Unit_String (Node.Unit, "nodes.lkt:5469:14") else null)); 
--# expr-done 149
   If_Result := True_Rel;
end if;

      Inc_Ref (If_Result);


--# expr-done 150

         Property_Result := If_Result;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_408;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_408;




            raise;
      end;



   return Property_Result;
end Argument_P_Expected_Type_Equation;
--# end

   







--# property-start Argument.xref_equation nodes.lkt:5471
pragma Warnings (Off, "is not referenced");

function Argument_P_Xref_Equation
  
  (Node : Bare_Argument
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Logic_Equation
is
   Self : Bare_Argument :=
     Bare_Argument (Node);
     Ent : Internal_Entity_Argument :=
       Internal_Entity_Argument'(Node => Self, Info => E_Info);
      --# bind self Ent

   


   Property_Result : Logic_Equation;

      
            procedure Finalizer_Scope_409 with Inline_Always;
            procedure Finalizer_Scope_865 with Inline_Always;

      Fld : Internal_Entity_Expr;
Fld_1 : Logic_Equation;
Fld_2 : Internal_Entity_Ref_Id;
Var_Expr : Internal_Entity_Ref_Id;
Fld_3 : Logic_Equation;
Scope_Result : Logic_Equation;
True_Rel : Logic_Equation;
Result_Var : Logic_Equation;
And_Pred : Logic_Equation;

            procedure Finalizer_Scope_409 is
            begin
                     Dec_Ref (Fld_1);
                     Dec_Ref (Scope_Result);
                     Dec_Ref (True_Rel);
                     Dec_Ref (Result_Var);
                     Dec_Ref (And_Pred);
            end Finalizer_Scope_409;
            procedure Finalizer_Scope_865 is
            begin
                     Dec_Ref (Fld_3);
            end Finalizer_Scope_865;


begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      begin
         
   --# scope-start

         --# expr-start 157 LogicAnd And_Pred nodes.lkt:5472
--# expr-start 152 LktNode.xref_equation Fld_1 nodes.lkt:5472



--# expr-start 151 Argument.value Fld nodes.lkt:5472







   if Ent.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld := Create_Internal_Entity_Expr (Node => Ent.Node.Argument_F_Value, Info => Ent.Info);
--# expr-done 151



   if Fld.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start '[dispatcher]LktNode.xref_equation'
Fld_1 := Liblktlang.Implementation.Dispatcher_Lkt_Node_P_Xref_Equation (Node => Fld.Node, E_Info => Fld.Info);
--# end
--# expr-done 152
--# expr-start 156 .do Result_Var nodes.lkt:5473



--# expr-start 153 Argument.name Fld_2 nodes.lkt:5473







   if Ent.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_2 := Create_Internal_Entity_Ref_Id (Node => Ent.Node.Argument_F_Name, Info => Ent.Info);
--# expr-done 153
Var_Expr := Fld_2;





if Var_Expr /= No_Entity_Ref_Id then
   



   --# scope-start



--# expr-start 154 RefId.xref_equation Fld_3 nodes.lkt:5473







   if Var_Expr.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start '[dispatcher]LktNode.xref_equation'
Fld_3 := Liblktlang.Implementation.Dispatcher_Lkt_Node_P_Xref_Equation (Node => Var_Expr.Node, E_Info => Var_Expr.Info);
--# end
--# expr-done 154
Scope_Result := Fld_3;
   Inc_Ref (Scope_Result);


   --# end
      Finalizer_Scope_865;


   Result_Var := Scope_Result;
else
   --# expr-start 155 LogicTrue True_Rel nodes.lkt:5473
True_Rel := Solver.Create_True ((if Liblktlang_Support.Adalog.Debug.Debug then New_Unit_String (Node.Unit, "nodes.lkt:5473:65") else null)); 
--# expr-done 155
   Result_Var := True_Rel;
end if;

      Inc_Ref (Result_Var);


--# expr-done 156
And_Pred := Create_And (Fld_1, Result_Var, (if Liblktlang_Support.Adalog.Debug.Debug then New_Unit_String (Node.Unit, "nodes.lkt:5472:9") else null)); 
--# expr-done 157

         Property_Result := And_Pred;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_409;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_409;
                     Finalizer_Scope_865;




            raise;
      end;



   return Property_Result;
end Argument_P_Xref_Equation;
--# end



   


      

   --
   --  Primitives for Bare_Base_Lexer_Case_Rule_Alt
   --

   




      
   function Base_Lexer_Case_Rule_Alt_F_Send
     (Node : Bare_Base_Lexer_Case_Rule_Alt) return Bare_Lexer_Case_Rule_Send
   is
      

         Kind : constant Lkt_Base_Lexer_Case_Rule_Alt := Node.Kind;
   begin
         case Kind is
               when Lkt_Lexer_Case_Rule_Cond_Alt =>
                     
         return Node.Lexer_Case_Rule_Cond_Alt_F_Send;
      
               when Lkt_Lexer_Case_Rule_Default_Alt =>
                     
         return Node.Lexer_Case_Rule_Default_Alt_F_Send;
      
         end case;
   end;




   


      

   --
   --  Primitives for Bare_Lexer_Case_Rule_Cond_Alt
   --

   



      
      procedure Initialize_Fields_For_Lexer_Case_Rule_Cond_Alt
        (Self : Bare_Lexer_Case_Rule_Cond_Alt
         ; Lexer_Case_Rule_Cond_Alt_F_Cond_Exprs : Bare_Ref_Id_List
         ; Lexer_Case_Rule_Cond_Alt_F_Send : Bare_Lexer_Case_Rule_Send
        ) is
      begin

            Self.Lexer_Case_Rule_Cond_Alt_F_Cond_Exprs := Lexer_Case_Rule_Cond_Alt_F_Cond_Exprs;
            Self.Lexer_Case_Rule_Cond_Alt_F_Send := Lexer_Case_Rule_Cond_Alt_F_Send;
         

      end Initialize_Fields_For_Lexer_Case_Rule_Cond_Alt;

      
   function Lexer_Case_Rule_Cond_Alt_F_Cond_Exprs
     (Node : Bare_Lexer_Case_Rule_Cond_Alt) return Bare_Ref_Id_List
   is
      

   begin
         
         return Node.Lexer_Case_Rule_Cond_Alt_F_Cond_Exprs;
      
   end;




   


      

   --
   --  Primitives for Bare_Lexer_Case_Rule_Default_Alt
   --

   



      
      procedure Initialize_Fields_For_Lexer_Case_Rule_Default_Alt
        (Self : Bare_Lexer_Case_Rule_Default_Alt
         ; Lexer_Case_Rule_Default_Alt_F_Send : Bare_Lexer_Case_Rule_Send
        ) is
      begin

            Self.Lexer_Case_Rule_Default_Alt_F_Send := Lexer_Case_Rule_Default_Alt_F_Send;
         

      end Initialize_Fields_For_Lexer_Case_Rule_Default_Alt;




   


      

   --
   --  Primitives for Bare_Base_Match_Branch
   --

   



   

   

   

   

   

   

   


      procedure Base_Match_Branch_Pre_Env_Actions
        (Self            : Bare_Base_Match_Branch;
         State           : in out PLE_Node_State;
         Add_To_Env_Only : Boolean := False) is
      begin
            


   
      if Add_To_Env_Only then
         return;
      end if;

      declare
         No_Parent         : constant Boolean :=
            False;
         Transitive_Parent : constant Boolean :=
            False;
         Names             : Symbol_Type_Array_Access :=
            null;
      begin
         Add_Env (Self, State, No_Parent, Transitive_Parent, Names);
      end;
   
   
      end;




      
   function Base_Match_Branch_F_Expr
     (Node : Bare_Base_Match_Branch) return Bare_Expr
   is
      

         Kind : constant Lkt_Base_Match_Branch := Node.Kind;
   begin
         case Kind is
               when Lkt_Match_Branch =>
                     
         return Node.Match_Branch_F_Expr;
      
               when Lkt_Pattern_Match_Branch =>
                     
         return Node.Pattern_Match_Branch_F_Expr;
      
         end case;
   end;


   







--# property-start '[dispatcher]BaseMatchBranch.match_part' dispatcher
pragma Warnings (Off, "is not referenced");

function Dispatcher_Base_Match_Branch_P_Match_Part
  
  (Node : Bare_Base_Match_Branch
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity
is
   Self : Bare_Base_Match_Branch :=
     Bare_Base_Match_Branch (Node);
     Ent : Internal_Entity_Base_Match_Branch :=
       Internal_Entity_Base_Match_Branch'(Node => Self, Info => E_Info);
      --# bind self Ent

   


   Property_Result : Internal_Entity;



begin
   --# property-body-start

   pragma Assert (Self = Node);





      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dispatching on null node");
      end if;

      case Lkt_Base_Match_Branch (Self.Kind) is
               when Lkt_Match_Branch =>
                  --# property-call-start MatchBranch.match_part
                  Property_Result := Match_Branch_P_Match_Part
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Pattern_Match_Branch =>
                  --# property-call-start PatternMatchBranch.match_part
                  Property_Result := Pattern_Match_Branch_P_Match_Part
                    (Self
                        , E_Info
                    );
                  --# end
      end case;




   return Property_Result;
end Dispatcher_Base_Match_Branch_P_Match_Part;
--# end



   


      

   --
   --  Primitives for Bare_Match_Branch
   --

   



      
      procedure Initialize_Fields_For_Match_Branch
        (Self : Bare_Match_Branch
         ; Match_Branch_F_Decl : Bare_Match_Val_Decl
         ; Match_Branch_F_Expr : Bare_Expr
        ) is
      begin

            Self.Match_Branch_F_Decl := Match_Branch_F_Decl;
            Self.Match_Branch_F_Expr := Match_Branch_F_Expr;
         

      end Initialize_Fields_For_Match_Branch;

      
   function Match_Branch_F_Decl
     (Node : Bare_Match_Branch) return Bare_Match_Val_Decl
   is
      

   begin
         
         return Node.Match_Branch_F_Decl;
      
   end;


   







--# property-start MatchBranch.match_part nodes.lkt:5411
pragma Warnings (Off, "is not referenced");

function Match_Branch_P_Match_Part
  
  (Node : Bare_Match_Branch
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity
is
   Self : Bare_Match_Branch :=
     Bare_Match_Branch (Node);
     Ent : Internal_Entity_Match_Branch :=
       Internal_Entity_Match_Branch'(Node => Self, Info => E_Info);
      --# bind self Ent

   


   Property_Result : Internal_Entity;

      

      Fld : Internal_Entity_Match_Val_Decl;
Cast_Result : Internal_Entity;



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         




--# expr-start 158 MatchBranch.decl Fld nodes.lkt:5411







   if Ent.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld := Create_Internal_Entity_Match_Val_Decl (Node => Ent.Node.Match_Branch_F_Decl, Info => Ent.Info);
--# expr-done 158



   
      Cast_Result := Create_Internal_Entity
        (Node => Fld.Node,
         Info => Fld.Info);



         Property_Result := Cast_Result;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Match_Branch_P_Match_Part;
--# end



   


      

   --
   --  Primitives for Bare_Pattern_Match_Branch
   --

   



      
      procedure Initialize_Fields_For_Pattern_Match_Branch
        (Self : Bare_Pattern_Match_Branch
         ; Pattern_Match_Branch_F_Pattern : Bare_Pattern
         ; Pattern_Match_Branch_F_Expr : Bare_Expr
        ) is
      begin

            Self.Pattern_Match_Branch_F_Pattern := Pattern_Match_Branch_F_Pattern;
            Self.Pattern_Match_Branch_F_Expr := Pattern_Match_Branch_F_Expr;
         

      end Initialize_Fields_For_Pattern_Match_Branch;

      
   function Pattern_Match_Branch_F_Pattern
     (Node : Bare_Pattern_Match_Branch) return Bare_Pattern
   is
      

   begin
         
         return Node.Pattern_Match_Branch_F_Pattern;
      
   end;


   







--# property-start PatternMatchBranch.match_part nodes.lkt:5423
pragma Warnings (Off, "is not referenced");

function Pattern_Match_Branch_P_Match_Part
  
  (Node : Bare_Pattern_Match_Branch
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity
is
   Self : Bare_Pattern_Match_Branch :=
     Bare_Pattern_Match_Branch (Node);
     Ent : Internal_Entity_Pattern_Match_Branch :=
       Internal_Entity_Pattern_Match_Branch'(Node => Self, Info => E_Info);
      --# bind self Ent

   


   Property_Result : Internal_Entity;

      

      Fld : Internal_Entity_Pattern;
Cast_Result : Internal_Entity;



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         




--# expr-start 159 PatternMatchBranch.pattern Fld nodes.lkt:5423







   if Ent.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld := Create_Internal_Entity_Pattern (Node => Ent.Node.Pattern_Match_Branch_F_Pattern, Info => Ent.Info);
--# expr-done 159



   
      Cast_Result := Create_Internal_Entity
        (Node => Fld.Node,
         Info => Fld.Info);



         Property_Result := Cast_Result;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Pattern_Match_Branch_P_Match_Part;
--# end



   


      

   --
   --  Primitives for Bare_Block_Expr_Clause
   --

   



      
      procedure Initialize_Fields_For_Block_Expr_Clause
        (Self : Bare_Block_Expr_Clause
         ; Block_Expr_Clause_F_Clause : Bare_Lkt_Node
        ) is
      begin

            Self.Block_Expr_Clause_F_Clause := Block_Expr_Clause_F_Clause;
         

      end Initialize_Fields_For_Block_Expr_Clause;

      
   function Block_Expr_Clause_F_Clause
     (Node : Bare_Block_Expr_Clause) return Bare_Lkt_Node
   is
      

   begin
         
         return Node.Block_Expr_Clause_F_Clause;
      
   end;




   


      

   --
   --  Primitives for Bare_Block_String_Line
   --

   







   


      

   --
   --  Primitives for Bare_Class_Qualifier
   --

   





   







--# property-start '[dispatcher]ClassQualifier.as_bool' dispatcher
pragma Warnings (Off, "is not referenced");

function Dispatcher_Class_Qualifier_P_As_Bool
  
  (Node : Bare_Class_Qualifier
  )

   return Boolean
is
   Self : Bare_Class_Qualifier :=
     Bare_Class_Qualifier (Node);
      --# bind node Self

   


   Property_Result : Boolean;



begin
   --# property-body-start

   pragma Assert (Self = Node);





      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dispatching on null node");
      end if;

      case Lkt_Class_Qualifier (Self.Kind) is
               when Lkt_Class_Qualifier_Absent =>
                  --# property-call-start ClassQualifier.Absent.as_bool
                  Property_Result := Class_Qualifier_Absent_P_As_Bool
                    (Self
                    );
                  --# end
               when Lkt_Class_Qualifier_Present =>
                  --# property-call-start ClassQualifier.Present.as_bool
                  Property_Result := Class_Qualifier_Present_P_As_Bool
                    (Self
                    );
                  --# end
      end case;




   return Property_Result;
end Dispatcher_Class_Qualifier_P_As_Bool;
--# end



   


      

   --
   --  Primitives for Bare_Class_Qualifier_Absent
   --

   





   







--# property-start ClassQualifier.Absent.as_bool nodes.lkt:394
pragma Warnings (Off, "is not referenced");

function Class_Qualifier_Absent_P_As_Bool
  
  (Node : Bare_Class_Qualifier_Absent
  )

   return Boolean
is
   Self : Bare_Class_Qualifier_Absent :=
     Bare_Class_Qualifier_Absent (Node);
      --# bind node Self

   


   Property_Result : Boolean;

      

      



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         

         Property_Result := False;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Class_Qualifier_Absent_P_As_Bool;
--# end



   


      

   --
   --  Primitives for Bare_Class_Qualifier_Present
   --

   





   







--# property-start ClassQualifier.Present.as_bool nodes.lkt:394
pragma Warnings (Off, "is not referenced");

function Class_Qualifier_Present_P_As_Bool
  
  (Node : Bare_Class_Qualifier_Present
  )

   return Boolean
is
   Self : Bare_Class_Qualifier_Present :=
     Bare_Class_Qualifier_Present (Node);
      --# bind node Self

   


   Property_Result : Boolean;

      

      



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         

         Property_Result := True;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Class_Qualifier_Present_P_As_Bool;
--# end



   


      

   --
   --  Primitives for Bare_Decl
   --

   



   

   

   

   

   

   

   


      procedure Decl_Pre_Env_Actions
        (Self            : Bare_Decl;
         State           : in out PLE_Node_State;
         Add_To_Env_Only : Boolean := False) is
      begin
            


   

      declare
         Resolver : constant Entity_Resolver :=
            null;

         

         Mappings : Internal_Env_Assoc_Array_Access :=
            Liblktlang.Implementation.Internal_Env_Mappings_1 (Node => 
Self);
      begin
         for Mapping of Mappings.Items loop

         Add_To_Env
           (Self,
            State,
            Mapping.Key,
            Mapping.Value,
            Mapping.Metadata,
            Resolver,
            Mapping.Dest_Env,
            DSL_Location => "nodes.lkt:1027:9");

         end loop;
         Dec_Ref (Mappings);
      end;
   
   
      end;




      
   function Decl_F_Syn_Name
     (Node : Bare_Decl) return Bare_Def_Id
   is
      

         Kind : constant Lkt_Decl := Node.Kind;
   begin
         case Kind is
               when Lkt_Error_Decl =>
                     return No_Bare_Lkt_Node;
               when Lkt_Grammar_Rule_Decl =>
                     
         return Node.Grammar_Rule_Decl_F_Syn_Name;
      
               when Lkt_Synthetic_Lexer_Decl =>
                     return No_Bare_Lkt_Node;
               when Lkt_Node_Decl =>
                     return No_Bare_Lkt_Node;
               when Lkt_Self_Decl =>
                     return No_Bare_Lkt_Node;
               when Lkt_Enum_Lit_Decl =>
                     
         return Node.Enum_Lit_Decl_F_Syn_Name;
      
               when Lkt_Field_Decl =>
                     
         return Node.Field_Decl_F_Syn_Name;
      
               when Lkt_Fun_Param_Decl =>
                     
         return Node.Fun_Param_Decl_F_Syn_Name;
      
               when Lkt_Lambda_Param_Decl =>
                     
         return Node.Lambda_Param_Decl_F_Syn_Name;
      
               when Lkt_Dyn_Var_Decl =>
                     
         return Node.Dyn_Var_Decl_F_Syn_Name;
      
               when Lkt_Binding_Val_Decl =>
                     
         return Node.Binding_Val_Decl_F_Syn_Name;
      
               when Lkt_Match_Val_Decl =>
                     
         return Node.Match_Val_Decl_F_Syn_Name;
      
               when Lkt_Val_Decl =>
                     
         return Node.Val_Decl_F_Syn_Name;
      
               when Lkt_Fun_Decl =>
                     
         return Node.Fun_Decl_F_Syn_Name;
      
               when Lkt_Env_Spec_Decl =>
                     
         return Node.Env_Spec_Decl_F_Syn_Name;
      
               when Lkt_Generic_Decl =>
                     return No_Bare_Lkt_Node;
               when Lkt_Grammar_Decl =>
                     
         return Node.Grammar_Decl_F_Syn_Name;
      
               when Lkt_Lexer_Decl =>
                     
         return Node.Lexer_Decl_F_Syn_Name;
      
               when Lkt_Lexer_Family_Decl =>
                     
         return Node.Lexer_Family_Decl_F_Syn_Name;
      
               when Lkt_Synth_Param_Decl =>
                     return No_Bare_Lkt_Node;
               when Lkt_Synth_Fun_Decl =>
                     return No_Bare_Lkt_Node;
               when Lkt_Any_Type_Decl =>
                     return No_Bare_Lkt_Node;
               when Lkt_Enum_Class_Alt_Decl =>
                     
         return Node.Enum_Class_Alt_Decl_F_Syn_Name;
      
               when Lkt_Function_Type =>
                     return No_Bare_Lkt_Node;
               when Lkt_Generic_Param_Type_Decl =>
                     
         return Node.Generic_Param_Type_Decl_F_Syn_Name;
      
               when Lkt_Class_Decl | Lkt_Enum_Class_Decl =>
                     
         return Node.Basic_Class_Decl_F_Syn_Name;
      
               when Lkt_Enum_Type_Decl =>
                     
         return Node.Enum_Type_Decl_F_Syn_Name;
      
               when Lkt_Struct_Decl =>
                     
         return Node.Struct_Decl_F_Syn_Name;
      
               when Lkt_Trait_Decl =>
                     
         return Node.Trait_Decl_F_Syn_Name;
      
         end case;
   end;


   







--# property-start '[dispatcher]Decl.decl_type_name' dispatcher
pragma Warnings (Off, "is not referenced");

function Dispatcher_Decl_P_Decl_Type_Name
  
  (Node : Bare_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return String_Type
is
   Self : Bare_Decl :=
     Bare_Decl (Node);
     Ent : Internal_Entity_Decl :=
       Internal_Entity_Decl'(Node => Self, Info => E_Info);
      --# bind self Ent

   


   Property_Result : String_Type;



begin
   --# property-body-start

   pragma Assert (Self = Node);





      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dispatching on null node");
      end if;

      case Lkt_Decl (Self.Kind) is
               when Lkt_Grammar_Rule_Decl =>
                  --# property-call-start GrammarRuleDecl.decl_type_name
                  Property_Result := Grammar_Rule_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Synthetic_Lexer_Decl =>
                  --# property-call-start SyntheticLexerDecl.decl_type_name
                  Property_Result := Synthetic_Lexer_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Node_Decl =>
                  --# property-call-start NodeDecl.decl_type_name
                  Property_Result := Node_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Self_Decl =>
                  --# property-call-start SelfDecl.decl_type_name
                  Property_Result := Self_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Binding_Val_Decl =>
                  --# property-call-start BindingValDecl.decl_type_name
                  Property_Result := Binding_Val_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Enum_Lit_Decl =>
                  --# property-call-start EnumLitDecl.decl_type_name
                  Property_Result := Enum_Lit_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Field_Decl =>
                  --# property-call-start FieldDecl.decl_type_name
                  Property_Result := Field_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Fun_Param_Decl =>
                  --# property-call-start FunParamDecl.decl_type_name
                  Property_Result := Fun_Param_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Lambda_Param_Decl =>
                  --# property-call-start LambdaParamDecl.decl_type_name
                  Property_Result := Lambda_Param_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Dyn_Var_Decl =>
                  --# property-call-start DynVarDecl.decl_type_name
                  Property_Result := Dyn_Var_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Match_Val_Decl =>
                  --# property-call-start MatchValDecl.decl_type_name
                  Property_Result := Match_Val_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Val_Decl =>
                  --# property-call-start ValDecl.decl_type_name
                  Property_Result := Val_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Fun_Decl =>
                  --# property-call-start FunDecl.decl_type_name
                  Property_Result := Fun_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Env_Spec_Decl =>
                  --# property-call-start EnvSpecDecl.decl_type_name
                  Property_Result := Env_Spec_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Error_Decl =>
                  --# property-call-start ErrorDecl.decl_type_name
                  Property_Result := Error_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Generic_Decl =>
                  --# property-call-start GenericDecl.decl_type_name
                  Property_Result := Generic_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Grammar_Decl =>
                  --# property-call-start GrammarDecl.decl_type_name
                  Property_Result := Grammar_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Lexer_Decl =>
                  --# property-call-start LexerDecl.decl_type_name
                  Property_Result := Lexer_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Lexer_Family_Decl =>
                  --# property-call-start LexerFamilyDecl.decl_type_name
                  Property_Result := Lexer_Family_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Synth_Fun_Decl =>
                  --# property-call-start SynthFunDecl.decl_type_name
                  Property_Result := Synth_Fun_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Synth_Param_Decl =>
                  --# property-call-start SynthParamDecl.decl_type_name
                  Property_Result := Synth_Param_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Any_Type_Decl =>
                  --# property-call-start AnyTypeDecl.decl_type_name
                  Property_Result := Any_Type_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Enum_Class_Alt_Decl =>
                  --# property-call-start EnumClassAltDecl.decl_type_name
                  Property_Result := Enum_Class_Alt_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Function_Type =>
                  --# property-call-start FunctionType.decl_type_name
                  Property_Result := Function_Type_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Generic_Param_Type_Decl =>
                  --# property-call-start GenericParamTypeDecl.decl_type_name
                  Property_Result := Generic_Param_Type_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Class_Decl =>
                  --# property-call-start ClassDecl.decl_type_name
                  Property_Result := Class_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Enum_Class_Decl =>
                  --# property-call-start EnumClassDecl.decl_type_name
                  Property_Result := Enum_Class_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Enum_Type_Decl =>
                  --# property-call-start EnumTypeDecl.decl_type_name
                  Property_Result := Enum_Type_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Struct_Decl =>
                  --# property-call-start StructDecl.decl_type_name
                  Property_Result := Struct_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Trait_Decl =>
                  --# property-call-start TraitDecl.decl_type_name
                  Property_Result := Trait_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
      end case;




   return Property_Result;
end Dispatcher_Decl_P_Decl_Type_Name;
--# end

   







--# property-start Decl.full_decl nodes.lkt:424
pragma Warnings (Off, "is not referenced");

function Decl_P_Full_Decl
  
  (Node : Bare_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity_Full_Decl
is
   Self : Bare_Decl :=
     Bare_Decl (Node);
     Ent : Internal_Entity_Decl :=
       Internal_Entity_Decl'(Node => Self, Info => E_Info);
      --# bind self Ent

   


   Property_Result : Internal_Entity_Full_Decl;

      

      Fld : Internal_Entity;
Local_Fd : Internal_Entity_Full_Decl;
Local_Gd : Internal_Entity_Generic_Decl;
Fld_1 : Internal_Entity_Full_Decl;
Ignored : Internal_Entity;
Match_Prefix : Internal_Entity;
Cast_Result : Internal_Entity_Full_Decl;
Let_Result : Internal_Entity_Full_Decl;
Scope_Result : Internal_Entity_Full_Decl;
Cast_Result_1 : Internal_Entity_Generic_Decl;
Let_Result_1 : Internal_Entity_Full_Decl;
Scope_Result_1 : Internal_Entity_Full_Decl;
Cast_Result_2 : Internal_Entity;
Let_Result_2 : Internal_Entity_Full_Decl;
Scope_Result_2 : Internal_Entity_Full_Decl;
Match_Result : Internal_Entity_Full_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 162 Match Match_Result nodes.lkt:424






--# expr-start 160 .parent Fld nodes.lkt:424







   if Ent.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld := Liblktlang.Implementation.Parent (Node => Ent.Node, E_Info => Ent.Info);
--# expr-done 160



   if Fld.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Match_Prefix := Fld; 

case Lkt_Lkt_Node (Match_Prefix.Node.Kind) is
         when Lkt_Full_Decl =>
            



   --# scope-start



--# scope-start









   
      Cast_Result := Create_Internal_Entity_Full_Decl
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Local_Fd := Cast_Result; 
--# bind fd Local_Fd

Let_Result := Local_Fd; 
--# end
Scope_Result := Let_Result;


   --# end


            Match_Result := Scope_Result; 
         when Lkt_Generic_Decl =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_1 := Create_Internal_Entity_Generic_Decl
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Local_Gd := Cast_Result_1; 
--# bind gd Local_Gd
--# expr-start 161 Decl.full_decl Fld_1 nodes.lkt:426







   if Local_Gd.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start Decl.full_decl
Fld_1 := Liblktlang.Implementation.Decl_P_Full_Decl (Node => Local_Gd.Node, E_Info => Local_Gd.Info);
--# end
--# expr-done 161
Let_Result_1 := Fld_1; 
--# end
Scope_Result_1 := Let_Result_1;


   --# end


            Match_Result := Scope_Result_1; 
         when Lkt_Argument .. Lkt_Error_Decl | Lkt_Grammar_Decl .. Lkt_Un_Op | Lkt_Grammar_List_Sep .. Lkt_Var_Bind =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_2 := Create_Internal_Entity
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Ignored := Cast_Result_2; 
--# bind _ Ignored

Let_Result_2 := No_Entity_Full_Decl; 
--# end
Scope_Result_2 := Let_Result_2;


   --# end


            Match_Result := Scope_Result_2; 
end case;

--# expr-done 162

         Property_Result := Match_Result;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Decl_P_Full_Decl;
--# end

   







--# property-start Decl.def_ids nodes.lkt:432
pragma Warnings (Off, "is not referenced");

function Decl_P_Def_Ids
  
  (Node : Bare_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity_Def_Id_Array_Access
is
   Self : Bare_Decl :=
     Bare_Decl (Node);
     Ent : Internal_Entity_Decl :=
       Internal_Entity_Decl'(Node => Self, Info => E_Info);
      --# bind self Ent

   


   Property_Result : Internal_Entity_Def_Id_Array_Access;

      
            procedure Finalizer_Scope_86 with Inline_Always;

      Fld : Internal_Entity_Def_Id;
Array_Lit : Internal_Entity_Def_Id_Array_Access;

            procedure Finalizer_Scope_86 is
            begin
                     Dec_Ref (Array_Lit);
            end Finalizer_Scope_86;


begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 164 ArrayLiteral Array_Lit nodes.lkt:434
--# expr-start 163 Decl.syn_name Fld nodes.lkt:434







   if Ent.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld := Create_Internal_Entity_Def_Id (Node => Implementation.Decl_F_Syn_Name (Ent.Node), Info => Ent.Info);
--# expr-done 163
Array_Lit := Create_Internal_Entity_Def_Id_Array (Internal_Internal_Entity_Def_Id_Array'(1 => Fld)); 
--# expr-done 164

         Property_Result := Array_Lit;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_86;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_86;




            raise;
      end;



   return Property_Result;
end Decl_P_Def_Ids;
--# end

   







--# property-start Decl.implements_node nodes.lkt:438
pragma Warnings (Off, "is not referenced");

function Decl_P_Implements_Node
  
  (Node : Bare_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Boolean
is
   Self : Bare_Decl :=
     Bare_Decl (Node);
     Ent : Internal_Entity_Decl :=
       Internal_Entity_Decl'(Node => Self, Info => E_Info);
      --# bind self Ent

   


   Property_Result : Boolean;

      
            procedure Finalizer_Scope_477 with Inline_Always;

      Cast_Result : Internal_Entity_Class_Decl;
Var_Expr : Internal_Entity_Class_Decl;
Fld : Internal_Entity_Type_Ref_List;
Item : Internal_Entity_Type_Ref;
Cast_Result_1 : Internal_Entity_Generic_Type_Ref;
Var_Expr_1 : Internal_Entity_Generic_Type_Ref;
Fld_1 : Lexical_Env := Empty_Env;
Fld_2 : Internal_Entity_Expr;
Cast_Result_2 : Internal_Entity_Ref_Id;
Var_Expr_2 : Internal_Entity_Ref_Id;
Fld_3 : Bare_Ref_Id;
Sym : Symbol_Type;
Result_Var : Symbol_Type;
Env_Get_Result : Internal_Entity;
Cast_Result_3 : Internal_Entity_Generic_Decl;
Fld_4 : Internal_Entity_Decl;
Scope_Result : Internal_Entity_Decl;
Result_Var_1 : Internal_Entity_Decl;
Fld_5 : Internal_Entity_Trait_Decl;
Cast_Result_4 : Internal_Entity;
Cast_Result_5 : Internal_Entity;
Is_Equiv : Boolean;
Fld_6 : Bare_Type_Ref_List;
Fld_7 : Internal_Entity_Info;
Bare_Item : Bare_Type_Ref;
As_Entity : Internal_Entity_Type_Ref;
Is_Null : Boolean;
If_Result : Internal_Entity_Type_Ref;
Find_Result : Internal_Entity_Type_Ref;
Is_Null_1 : Boolean;
Not_Val : Boolean;
Scope_Result_1 : Boolean;
Result_Var_2 : Boolean;

            procedure Finalizer_Scope_477 is
            begin
                     Dec_Ref (Fld_1);
            end Finalizer_Scope_477;


begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      begin
         
   --# scope-start

         --# expr-start 181 .do Result_Var_2 nodes.lkt:439



--# expr-start 165 Cast Cast_Result nodes.lkt:439









   if Ent.Node = null
      or else Ent.Node.Kind in Lkt_Class_Decl_Range
   then
      
      Cast_Result := Create_Internal_Entity_Class_Decl
        (Node => Ent.Node,
         Info => Ent.Info);

   else
         Cast_Result := No_Entity_Class_Decl;
   end if;


--# expr-done 165
Var_Expr := Cast_Result;





if Var_Expr /= No_Entity_Class_Decl then
   



   --# scope-start



--# expr-start 180 Not Not_Val nodes.lkt:441
--# expr-start 179 .is_null Is_Null_1 nodes.lkt:441
--# expr-start 178 .find Find_Result nodes.lkt:441








   

   --# expr-start 166 BasicClassDecl.traits Fld nodes.lkt:441







   if Var_Expr.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld := Create_Internal_Entity_Type_Ref_List (Node => Var_Expr.Node.Basic_Class_Decl_F_Traits, Info => Var_Expr.Info);
--# expr-done 166









Fld_6 := Fld.Node;

   
      Find_Result := No_Entity_Type_Ref;
   

   

      if Fld_6 = null then
         
      Find_Result := No_Entity_Type_Ref;
   
      else
         

      declare
         
         Collection : constant Bare_Type_Ref_List := Fld_6;
      begin
         for Untyped_Item of
               Collection.Nodes (1 .. Children_Count (Collection))
         loop
                  
                  Bare_Item := Untyped_Item; 
                  



Is_Null := Bare_Item = null; 
if Is_Null then
   
   If_Result := No_Entity_Type_Ref;
else
   








Fld_7 := Fld.Info;

As_Entity := (Info => Fld_7, Node => Bare_Item); 
   If_Result := As_Entity;
end if;



                  Item := If_Result; 

            
   --# scope-start


               --# bind t Item

            
      --# expr-start 177 Eq Is_Equiv nodes.lkt:443





--# expr-start 175 .do Result_Var_1 nodes.lkt:443



--# expr-start 167 Cast Cast_Result_1 nodes.lkt:443









   if Item.Node = null
      or else Item.Node.Kind in Lkt_Generic_Type_Ref_Range
   then
      
      Cast_Result_1 := Create_Internal_Entity_Generic_Type_Ref
        (Node => Item.Node,
         Info => Item.Info);

   else
         Cast_Result_1 := No_Entity_Generic_Type_Ref;
   end if;


--# expr-done 167
Var_Expr_1 := Cast_Result_1;





if Var_Expr_1 /= No_Entity_Generic_Type_Ref then
   



   --# scope-start



--# expr-start 174 GenericDecl.decl Fld_4 nodes.lkt:445



--# expr-start 173 Cast Cast_Result_3 nodes.lkt:445





--# expr-start 172 .get_first Env_Get_Result nodes.lkt:445
--# expr-start 168 LktNode.children_env Fld_1 nodes.lkt:445







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_1 := Liblktlang.Implementation.Children_Env (Node => Self);
--# expr-done 168



--# expr-start 170 Cast Cast_Result_2 nodes.lkt:446





--# expr-start 169 GenericTypeRef.type_name Fld_2 nodes.lkt:446







   if Var_Expr_1.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_2 := Create_Internal_Entity_Expr (Node => Var_Expr_1.Node.Generic_Type_Ref_F_Type_Name, Info => Var_Expr_1.Info);
--# expr-done 169



   if Fld_2.Node = null
      or else Fld_2.Node.Kind in Lkt_Ref_Id_Range
   then
      
      Cast_Result_2 := Create_Internal_Entity_Ref_Id
        (Node => Fld_2.Node,
         Info => Fld_2.Info);

   else
         Cast_Result_2 := No_Entity_Ref_Id;
   end if;


--# expr-done 170
Var_Expr_2 := Cast_Result_2;





if Var_Expr_2 /= No_Entity_Ref_Id then
   --# expr-start 171 .symbol Sym nodes.lkt:446









Fld_3 := Var_Expr_2.Node; 
Sym := Get_Symbol (Fld_3); 
--# expr-done 171
   Result_Var := Sym;
else
   
   Result_Var := No_Symbol;
end if;





Env_Get_Result := AST_Envs.Get_First (Self => Fld_1, Key => Thin (Result_Var), Lookup_Kind => To_Lookup_Kind_Type (Recursive), Categories => (Nocat => True)); 
--# expr-done 172



   if Env_Get_Result.Node = null
      or else Env_Get_Result.Node.Kind in Lkt_Generic_Decl_Range
   then
      
      Cast_Result_3 := Create_Internal_Entity_Generic_Decl
        (Node => Env_Get_Result.Node,
         Info => Env_Get_Result.Info);

   else
         Cast_Result_3 := No_Entity_Generic_Decl;
   end if;


--# expr-done 173



   if Cast_Result_3.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_4 := Create_Internal_Entity_Decl (Node => Cast_Result_3.Node.Generic_Decl_F_Decl, Info => Cast_Result_3.Info);
--# expr-done 174
Scope_Result := Fld_4;


   --# end
      Finalizer_Scope_477;


   Result_Var_1 := Scope_Result;
else
   
   Result_Var_1 := No_Entity_Decl;
end if;



--# expr-done 175



   
      Cast_Result_4 := Create_Internal_Entity
        (Node => Result_Var_1.Node,
         Info => Result_Var_1.Info);







--# expr-start 176 LktNode.node_trait Fld_5 nodes.lkt:451







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start LktNode.node_trait
Fld_5 := Liblktlang.Implementation.Lkt_Node_P_Node_Trait (Node => Self);
--# end
--# expr-done 176



   
      Cast_Result_5 := Create_Internal_Entity
        (Node => Fld_5.Node,
         Info => Fld_5.Info);


Is_Equiv := Equivalent (Cast_Result_4, Cast_Result_5); 
--# expr-done 177
      if Is_Equiv then
         Find_Result := Item; 
      end if;
   

            
   --# end


               exit when Is_Equiv;

         end loop;
      end;
   
      end if;

   
   



--# expr-done 178
Is_Null_1 := Find_Result.Node = null; 
--# expr-done 179
Not_Val := not (Is_Null_1); 
--# expr-done 180
Scope_Result_1 := Not_Val;


   --# end


   Result_Var_2 := Scope_Result_1;
else
   
   Result_Var_2 := False;
end if;



--# expr-done 181

         Property_Result := Result_Var_2;
         
   --# end


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_477;




            raise;
      end;



   return Property_Result;
end Decl_P_Implements_Node;
--# end

   







--# property-start Decl.as_bare_decl nodes.lkt:459
pragma Warnings (Off, "is not referenced");

function Decl_P_As_Bare_Decl
  
  (Node : Bare_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity_Decl
is
   Self : Bare_Decl :=
     Bare_Decl (Node);
     Ent : Internal_Entity_Decl :=
       Internal_Entity_Decl'(Node => Self, Info => E_Info);
      --# bind self Ent

   


   Property_Result : Internal_Entity_Decl;

      

      Fld : Bare_Decl;
As_Entity : Internal_Entity_Decl;
Is_Null : Boolean;
If_Result : Internal_Entity_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 183 .as_bare_entity If_Result nodes.lkt:464



--# expr-start 182 'Entity[Decl].node' Fld nodes.lkt:464









Fld := Ent.Node;
--# expr-done 182
Is_Null := Fld = null; 
if Is_Null then
   
   If_Result := No_Entity_Decl;
else
   

As_Entity := (Info => No_Entity_Info, Node => Fld); 
   If_Result := As_Entity;
end if;



--# expr-done 183

         Property_Result := If_Result;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Decl_P_As_Bare_Decl;
--# end

   







--# property-start Decl.is_type_decl nodes.lkt:467
pragma Warnings (Off, "is not referenced");

function Decl_P_Is_Type_Decl
  
  (Node : Bare_Decl
  )

   return Boolean
is
   Self : Bare_Decl :=
     Bare_Decl (Node);
      --# bind node Self

   


   Property_Result : Boolean;

      

      Is_A : Boolean;



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 184 IsA Is_A nodes.lkt:467

Is_A := Self /= null 
and then Self.Kind in Lkt_Type_Decl; 
--# expr-done 184

         Property_Result := Is_A;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Decl_P_Is_Type_Decl;
--# end

   







--# property-start Decl.is_defined nodes.lkt:470
pragma Warnings (Off, "is not referenced");

function Decl_P_Is_Defined
  
  (Node : Bare_Decl
  )

   return Boolean
is
   Self : Bare_Decl :=
     Bare_Decl (Node);
      --# bind node Self

   


   Property_Result : Boolean;

      

      Is_Null : Boolean;
Not_Val : Boolean;



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 186 Not Not_Val nodes.lkt:470
--# expr-start 185 .is_null Is_Null nodes.lkt:470

Is_Null := Self = null; 
--# expr-done 185
Not_Val := not (Is_Null); 
--# expr-done 186

         Property_Result := Not_Val;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Decl_P_Is_Defined;
--# end

   







--# property-start Decl.infer_function_type nodes.lkt:478
pragma Warnings (Off, "is not referenced");

function Decl_P_Infer_Function_Type
  
  (Node : Bare_Decl
      ; Expected_Call : Internal_Entity_Function_Type
      ; In_Logic_Call : Boolean
         := False
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity_Function_Type
is
   Self : Bare_Decl :=
     Bare_Decl (Node);
     Ent : Internal_Entity_Decl :=
       Internal_Entity_Decl'(Node => Self, Info => E_Info);
      --# bind self Ent

   

   --# bind expected_call Expected_Call
   --# bind in_logic_call In_Logic_Call

   Property_Result : Internal_Entity_Function_Type;

      
            procedure Finalizer_Scope_478 with Inline_Always;
            procedure Finalizer_Scope_479 with Inline_Always;
            procedure Finalizer_Scope_480 with Inline_Always;

      Cast_Result : Internal_Entity_Function_Type;
Var_Expr : Internal_Entity_Function_Type;
Fld : Internal_Entity_Decl;
Result_Var : Internal_Entity_Decl;
Left_Var : Internal_Entity_Decl;
Result_Var_1 : Internal_Entity_Decl;
Local_Origin : Internal_Entity_Decl;
Cast_Result_1 : Internal_Entity_Function_Type;
Var_Expr_1 : Internal_Entity_Decl;
Fld_1 : Internal_Entity_Function_Type;
Result_Var_2 : Internal_Entity_Function_Type;
Left_Var_1 : Internal_Entity_Function_Type;
Result_Var_3 : Internal_Entity_Function_Type;
Local_Generic_Func : Internal_Entity_Function_Type;
Fld_2 : Internal_Entity;
Cast_Result_2 : Internal_Entity_Generic_Decl;
Fld_3 : Internal_Entity_Generic_Param_Type_Decl_Array_Access;
Cast_Result_3 : Internal_Entity_Type_Decl;
Cast_Result_4 : Internal_Entity_Type_Decl;
Fld_4 : Internal_Entity_Type_Decl_Array_Access;
Var_Expr_2 : Internal_Entity_Type_Decl_Array_Access;
Get_Result : Internal_Entity_Type_Decl;
Var_Expr_3 : Internal_Entity_Type_Decl;
Fld_5 : Internal_Entity_Type_Decl;
Var_Expr_4 : Internal_Entity_Type_Decl;
Cast_Result_5 : Internal_Entity_Type_Decl;
Fld_6 : Boolean;
If_Result : Internal_Entity_Type_Decl;
Scope_Result : Internal_Entity_Type_Decl;
Result_Var_4 : Internal_Entity_Type_Decl;
Result_Var_5 : Internal_Entity_Type_Decl;
Scope_Result_1 : Internal_Entity_Type_Decl;
Result_Var_6 : Internal_Entity_Type_Decl;
Map_Result : Internal_Entity_Type_Decl_Array_Access;
Fld_7 : Internal_Entity_Decl;
Var_Expr_5 : Internal_Entity_Decl;
Fld_8 : Internal_Entity_Function_Type;
Fld_9 : Internal_Entity_Function_Type;
If_Result_1 : Internal_Entity_Function_Type;
Scope_Result_2 : Internal_Entity_Function_Type;
Result_Var_7 : Internal_Entity_Function_Type;
Fld_10 : Boolean;
Not_Val : Boolean;
Fld_11 : Internal_Entity_Function_Type;
Fld_12 : Internal_Entity_Function_Type;
If_Result_2 : Internal_Entity_Function_Type;
If_Result_3 : Internal_Entity_Function_Type;
Is_Null : Boolean;
Is_Null_1 : Boolean;
If_Result_4 : Boolean;
If_Result_5 : Internal_Entity_Function_Type;
Let_Result : Internal_Entity_Function_Type;
Let_Result_1 : Internal_Entity_Function_Type;
Scope_Result_3 : Internal_Entity_Function_Type;

            procedure Finalizer_Scope_478 is
            begin
                     Dec_Ref (Fld_3);
                     Dec_Ref (Map_Result);
            end Finalizer_Scope_478;
            procedure Finalizer_Scope_479 is
            begin
                     Dec_Ref (Fld_4);
            end Finalizer_Scope_479;
            procedure Finalizer_Scope_480 is
            begin
                     Dec_Ref (Var_Expr_2);
            end Finalizer_Scope_480;


begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      begin
         
   --# scope-start

         



   --# scope-start



--# expr-start 220 ValDecl Let_Result_1 nodes.lkt:481
--# scope-start
--# expr-start 189 'or?' Result_Var_1 nodes.lkt:481






--# expr-start 187 Cast Cast_Result nodes.lkt:481









   if Ent.Node = null
      or else Ent.Node.Kind in Lkt_Function_Type_Range
   then
      
      Cast_Result := Create_Internal_Entity_Function_Type
        (Node => Ent.Node,
         Info => Ent.Info);

   else
         Cast_Result := No_Entity_Function_Type;
   end if;


--# expr-done 187
Var_Expr := Cast_Result;





if Var_Expr /= No_Entity_Function_Type then
   --# expr-start 188 FunctionType.origin Fld nodes.lkt:481







   if Var_Expr.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld := Var_Expr.Node.Function_Type_F_Origin;
--# expr-done 188
   Result_Var := Fld;
else
   
   Result_Var := No_Entity_Decl;
end if;



Left_Var := Result_Var;





if Left_Var /= No_Entity_Decl then
   
   Result_Var_1 := Left_Var;
else
   
   Result_Var_1 := Ent;
end if;



--# expr-done 189
Local_Origin := Result_Var_1; 
--# bind origin Local_Origin
--# expr-start 219 ValDecl Let_Result nodes.lkt:482
--# scope-start
--# expr-start 192 'or?' Result_Var_3 nodes.lkt:482



--# expr-start 190 Cast Cast_Result_1 nodes.lkt:482









   if Ent.Node = null
      or else Ent.Node.Kind in Lkt_Function_Type_Range
   then
      
      Cast_Result_1 := Create_Internal_Entity_Function_Type
        (Node => Ent.Node,
         Info => Ent.Info);

   else
         Cast_Result_1 := No_Entity_Function_Type;
   end if;


--# expr-done 190
Left_Var_1 := Cast_Result_1;





if Left_Var_1 /= No_Entity_Function_Type then
   
   Result_Var_3 := Left_Var_1;
else
   



Var_Expr_1 := Ent;





if Var_Expr_1 /= No_Entity_Decl then
   --# expr-start 191 Decl.function_type Fld_1 nodes.lkt:482







   if Var_Expr_1.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start Decl.function_type
Fld_1 := Liblktlang.Implementation.Decl_P_Function_Type (Node => Var_Expr_1.Node, E_Info => Var_Expr_1.Info);
--# end
--# expr-done 191
   Result_Var_2 := Fld_1;
else
   
   Result_Var_2 := No_Entity_Function_Type;
end if;



   Result_Var_3 := Result_Var_2;
end if;



--# expr-done 192
Local_Generic_Func := Result_Var_3; 
--# bind generic_func Local_Generic_Func
--# expr-start 218 If If_Result_5 nodes.lkt:484



--# expr-start 195 BooleanOr If_Result_4 nodes.lkt:484



--# expr-start 193 .is_null Is_Null nodes.lkt:484

Is_Null := Expected_Call.Node = null; 
--# expr-done 193
if Is_Null then
   
   If_Result_4 := True;
else
   --# expr-start 194 .is_null Is_Null_1 nodes.lkt:484

Is_Null_1 := Local_Origin.Node = null; 
--# expr-done 194
   If_Result_4 := Is_Null_1;
end if;



--# expr-done 195
if If_Result_4 then
   
   If_Result_5 := No_Entity_Function_Type;
else
   --# expr-start 217 If If_Result_3 nodes.lkt:484



--# expr-start 197 Not Not_Val nodes.lkt:486
--# expr-start 196 Decl.is_generic Fld_10 nodes.lkt:486







   if Local_Origin.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start Decl.is_generic
Fld_10 := Liblktlang.Implementation.Decl_P_Is_Generic (Node => Local_Origin.Node, E_Info => Local_Origin.Info);
--# end
--# expr-done 196
Not_Val := not (Fld_10); 
--# expr-done 197
if Not_Val then
   --# expr-start 200 If If_Result_2 nodes.lkt:488




if In_Logic_Call then
   --# expr-start 198 Decl.logic_function_type Fld_12 nodes.lkt:488







   if Local_Origin.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start Decl.logic_function_type
Fld_12 := Liblktlang.Implementation.Decl_P_Logic_Function_Type (Node => Local_Origin.Node, E_Info => Local_Origin.Info);
--# end
--# expr-done 198
   If_Result_2 := Fld_12;
else
   --# expr-start 199 Decl.function_type Fld_11 nodes.lkt:489







   if Local_Origin.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start Decl.function_type
Fld_11 := Liblktlang.Implementation.Decl_P_Function_Type (Node => Local_Origin.Node, E_Info => Local_Origin.Info);
--# end
--# expr-done 199
   If_Result_2 := Fld_11;
end if;



--# expr-done 200
   If_Result_3 := If_Result_2;
else
   --# expr-start 216 .do Result_Var_7 nodes.lkt:491



--# expr-start 212 Decl.instantiate_generic_decl Fld_7 nodes.lkt:491







   if Local_Origin.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# expr-start 211 .map Map_Result nodes.lkt:492








   

   --# expr-start 203 GenericDecl.generic_params Fld_3 nodes.lkt:492



--# expr-start 202 Cast Cast_Result_2 nodes.lkt:492





--# expr-start 201 .parent Fld_2 nodes.lkt:492







   if Local_Origin.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_2 := Liblktlang.Implementation.Parent (Node => Local_Origin.Node, E_Info => Local_Origin.Info);
--# expr-done 201



   if Fld_2.Node = null
      or else Fld_2.Node.Kind in Lkt_Generic_Decl_Range
   then
      
      Cast_Result_2 := Create_Internal_Entity_Generic_Decl
        (Node => Fld_2.Node,
         Info => Fld_2.Info);

   else
         Raise_Property_Exception
           (Self, Property_Error'Identity, "invalid object cast");
   end if;


--# expr-done 202



   if Cast_Result_2.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start GenericDecl.generic_params
Fld_3 := Liblktlang.Implementation.Generic_Decl_P_Generic_Params (Node => Cast_Result_2.Node, E_Info => Cast_Result_2.Info);
--# end
--# expr-done 203

   
      declare
         Map_Result_Vec : Internal_Entity_Type_Decl_Vectors.Vector;
      begin
   

   

      

      declare
         
         Collection : constant Internal_Entity_Generic_Param_Type_Decl_Array_Access := Fld_3;
      begin
         for Item of
               Collection.Items
         loop

            
   --# scope-start


               --# bind gf Item

            
         
         
      --# expr-start 210 .do Result_Var_6 nodes.lkt:494



--# expr-start 204 TypeDecl.find_types_that_replace_ty Fld_4 nodes.lkt:494







   if Expected_Call.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;











   
      Cast_Result_3 := Create_Internal_Entity_Type_Decl
        (Node => Item.Node,
         Info => Item.Info);











   
      Cast_Result_4 := Create_Internal_Entity_Type_Decl
        (Node => Local_Generic_Func.Node,
         Info => Local_Generic_Func.Info);


--# property-call-start TypeDecl.find_types_that_replace_ty
Fld_4 := Liblktlang.Implementation.Type_Decl_P_Find_Types_That_Replace_Ty (Node => Expected_Call.Node, Ty => Cast_Result_3, Origin => Cast_Result_4, E_Info => Expected_Call.Info);
--# end
--# expr-done 204
Var_Expr_2 := Fld_4;

      Inc_Ref (Var_Expr_2);




if Var_Expr_2 /= No_Internal_Entity_Type_Decl_Array_Type then
   



   --# scope-start






--# expr-start 205 .at Get_Result nodes.lkt:497




Get_Result := Get (Self, Var_Expr_2, 0, True); 
--# expr-done 205
Var_Expr_3 := Get_Result;





if Var_Expr_3 /= No_Entity_Type_Decl then
   --# expr-start 209 .do Result_Var_4 nodes.lkt:497



--# expr-start 206 TypeDecl.imprecise_common_ancestor_list Fld_5 nodes.lkt:497







   if Var_Expr_3.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;



--# property-call-start TypeDecl.imprecise_common_ancestor_list
Fld_5 := Liblktlang.Implementation.Type_Decl_P_Imprecise_Common_Ancestor_List (Node => Var_Expr_3.Node, Other_Types => Var_Expr_2, E_Info => Var_Expr_3.Info);
--# end
--# expr-done 206
Var_Expr_4 := Fld_5;





if Var_Expr_4 /= No_Entity_Type_Decl then
   



   --# scope-start



--# expr-start 208 If If_Result nodes.lkt:499



--# expr-start 207 TypeDecl.matching_type Fld_6 nodes.lkt:499







   if Var_Expr_4.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;











   
      Cast_Result_5 := Create_Internal_Entity_Type_Decl
        (Node => Item.Node,
         Info => Item.Info);


--# property-call-start TypeDecl.matching_type
Fld_6 := Liblktlang.Implementation.Type_Decl_P_Matching_Type (Node => Var_Expr_4.Node, Other => Cast_Result_5, E_Info => Var_Expr_4.Info);
--# end
--# expr-done 207
if Fld_6 then
   
   If_Result := No_Entity_Type_Decl;
else
   
   If_Result := Var_Expr_4;
end if;



--# expr-done 208
Scope_Result := If_Result;


   --# end


   Result_Var_4 := Scope_Result;
else
   
   Result_Var_4 := No_Entity_Type_Decl;
end if;



--# expr-done 209
   Result_Var_5 := Result_Var_4;
else
   
   Result_Var_5 := No_Entity_Type_Decl;
end if;



Scope_Result_1 := Result_Var_5;


   --# end
      Finalizer_Scope_480;


   Result_Var_6 := Scope_Result_1;
else
   
   Result_Var_6 := No_Entity_Type_Decl;
end if;



--# expr-done 210
      

         declare
            Item_To_Append : constant Internal_Entity_Type_Decl := Result_Var_6;
         begin
            Internal_Entity_Type_Decl_Vectors.Append (Map_Result_Vec, Item_To_Append);
         end;
   
   
   

            
   --# end
      Finalizer_Scope_479;



         end loop;
      end;
   

   

         Map_Result := Create_Internal_Entity_Type_Decl_Array
           (Items_Count => Natural (Internal_Entity_Type_Decl_Vectors.Length (Map_Result_Vec)));
         for I in Map_Result.Items'Range loop
            Map_Result.Items (I) := Internal_Entity_Type_Decl_Vectors.Get
              (Map_Result_Vec,
               I + Internal_Entity_Type_Decl_Vectors.Index_Type'First - Map_Result.Items'First);
         end loop;
         Internal_Entity_Type_Decl_Vectors.Destroy (Map_Result_Vec);

      end;
   



--# expr-done 211
--# property-call-start Decl.instantiate_generic_decl
Fld_7 := Liblktlang.Implementation.Decl_P_Instantiate_Generic_Decl (Node => Local_Origin.Node, Param_Types => Map_Result, E_Info => Local_Origin.Info);
--# end
--# expr-done 212
Var_Expr_5 := Fld_7;





if Var_Expr_5 /= No_Entity_Decl then
   



   --# scope-start



--# expr-start 215 If If_Result_1 nodes.lkt:507




if In_Logic_Call then
   --# expr-start 213 Decl.logic_function_type Fld_9 nodes.lkt:507







   if Var_Expr_5.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start Decl.logic_function_type
Fld_9 := Liblktlang.Implementation.Decl_P_Logic_Function_Type (Node => Var_Expr_5.Node, E_Info => Var_Expr_5.Info);
--# end
--# expr-done 213
   If_Result_1 := Fld_9;
else
   --# expr-start 214 Decl.function_type Fld_8 nodes.lkt:508







   if Var_Expr_5.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start Decl.function_type
Fld_8 := Liblktlang.Implementation.Decl_P_Function_Type (Node => Var_Expr_5.Node, E_Info => Var_Expr_5.Info);
--# end
--# expr-done 214
   If_Result_1 := Fld_8;
end if;



--# expr-done 215
Scope_Result_2 := If_Result_1;


   --# end


   Result_Var_7 := Scope_Result_2;
else
   
   Result_Var_7 := No_Entity_Function_Type;
end if;



--# expr-done 216
   If_Result_3 := Result_Var_7;
end if;



--# expr-done 217
   If_Result_5 := If_Result_3;
end if;



--# expr-done 218
Let_Result := If_Result_5; 
--# end
--# expr-done 219
Let_Result_1 := Let_Result; 
--# end
--# expr-done 220
Scope_Result_3 := Let_Result_1;


   --# end
      Finalizer_Scope_478;



         Property_Result := Scope_Result_3;
         
   --# end


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_478;
                     Finalizer_Scope_479;
                     Finalizer_Scope_480;




            raise;
      end;



   return Property_Result;
end Decl_P_Infer_Function_Type;
--# end

   







--# property-start Decl.function_type nodes.lkt:513
pragma Warnings (Off, "is not referenced");

function Decl_P_Function_Type
  
  (Node : Bare_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity_Function_Type
is
   Self : Bare_Decl :=
     Bare_Decl (Node);
     Ent : Internal_Entity_Decl :=
       Internal_Entity_Decl'(Node => Self, Info => E_Info);
      --# bind self Ent

   


   Property_Result : Internal_Entity_Function_Type;

      

      Local_Ft : Internal_Entity_Function_Type;
Local_Td : Internal_Entity_Struct_Decl;
Fld : Internal_Entity_Function_Type;
Local_Td_1 : Internal_Entity_Class_Decl;
Fld_1 : Internal_Entity_Function_Type;
Local_Td_2 : Internal_Entity_Fun_Decl;
Fld_2 : Internal_Entity_Function_Type;
Local_Td_3 : Internal_Entity_Synth_Fun_Decl;
Fld_3 : Internal_Entity_Function_Type;
Local_Td_4 : Internal_Entity_Generic_Decl;
Fld_4 : Internal_Entity_Decl;
Fld_5 : Internal_Entity_Function_Type;
Ignored : Internal_Entity_Decl;
Match_Prefix : Internal_Entity_Decl;
Cast_Result : Internal_Entity_Function_Type;
Let_Result : Internal_Entity_Function_Type;
Scope_Result : Internal_Entity_Function_Type;
Cast_Result_1 : Internal_Entity_Struct_Decl;
Let_Result_1 : Internal_Entity_Function_Type;
Scope_Result_1 : Internal_Entity_Function_Type;
Cast_Result_2 : Internal_Entity_Class_Decl;
Let_Result_2 : Internal_Entity_Function_Type;
Scope_Result_2 : Internal_Entity_Function_Type;
Cast_Result_3 : Internal_Entity_Fun_Decl;
Let_Result_3 : Internal_Entity_Function_Type;
Scope_Result_3 : Internal_Entity_Function_Type;
Cast_Result_4 : Internal_Entity_Synth_Fun_Decl;
Let_Result_4 : Internal_Entity_Function_Type;
Scope_Result_4 : Internal_Entity_Function_Type;
Cast_Result_5 : Internal_Entity_Generic_Decl;
Let_Result_5 : Internal_Entity_Function_Type;
Scope_Result_5 : Internal_Entity_Function_Type;
Cast_Result_6 : Internal_Entity_Decl;
Let_Result_6 : Internal_Entity_Function_Type;
Scope_Result_6 : Internal_Entity_Function_Type;
Match_Result : Internal_Entity_Function_Type;
Is_Null : Boolean;
If_Result : Internal_Entity_Function_Type;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      begin
         
   --# scope-start

         --# expr-start 229 If If_Result nodes.lkt:514



--# expr-start 221 .is_null Is_Null nodes.lkt:514

Is_Null := Ent.Node = null; 
--# expr-done 221
if Is_Null then
   
   If_Result := No_Entity_Function_Type;
else
   --# expr-start 228 Match Match_Result nodes.lkt:516










   if Ent.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Match_Prefix := Ent; 

case Lkt_Decl (Match_Prefix.Node.Kind) is
         when Lkt_Function_Type =>
            



   --# scope-start



--# scope-start









   
      Cast_Result := Create_Internal_Entity_Function_Type
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Local_Ft := Cast_Result; 
--# bind ft Local_Ft

Let_Result := Local_Ft; 
--# end
Scope_Result := Let_Result;


   --# end


            Match_Result := Scope_Result; 
         when Lkt_Struct_Decl =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_1 := Create_Internal_Entity_Struct_Decl
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Local_Td := Cast_Result_1; 
--# bind td Local_Td
--# expr-start 222 StructDecl.function_type_aux Fld nodes.lkt:518







   if Local_Td.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start StructDecl.function_type_aux
Fld := Liblktlang.Implementation.Struct_Decl_P_Function_Type_Aux (Node => Local_Td.Node, E_Info => Local_Td.Info);
--# end
--# expr-done 222
Let_Result_1 := Fld; 
--# end
Scope_Result_1 := Let_Result_1;


   --# end


            Match_Result := Scope_Result_1; 
         when Lkt_Class_Decl =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_2 := Create_Internal_Entity_Class_Decl
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Local_Td_1 := Cast_Result_2; 
--# bind td Local_Td_1
--# expr-start 223 ClassDecl.function_type_aux Fld_1 nodes.lkt:519







   if Local_Td_1.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start ClassDecl.function_type_aux
Fld_1 := Liblktlang.Implementation.Class_Decl_P_Function_Type_Aux (Node => Local_Td_1.Node, E_Info => Local_Td_1.Info);
--# end
--# expr-done 223
Let_Result_2 := Fld_1; 
--# end
Scope_Result_2 := Let_Result_2;


   --# end


            Match_Result := Scope_Result_2; 
         when Lkt_Fun_Decl =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_3 := Create_Internal_Entity_Fun_Decl
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Local_Td_2 := Cast_Result_3; 
--# bind td Local_Td_2
--# expr-start 224 FunDecl.function_type_aux Fld_2 nodes.lkt:520







   if Local_Td_2.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start FunDecl.function_type_aux
Fld_2 := Liblktlang.Implementation.Fun_Decl_P_Function_Type_Aux (Node => Local_Td_2.Node, E_Info => Local_Td_2.Info);
--# end
--# expr-done 224
Let_Result_3 := Fld_2; 
--# end
Scope_Result_3 := Let_Result_3;


   --# end


            Match_Result := Scope_Result_3; 
         when Lkt_Synth_Fun_Decl =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_4 := Create_Internal_Entity_Synth_Fun_Decl
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Local_Td_3 := Cast_Result_4; 
--# bind td Local_Td_3
--# expr-start 225 SynthFunDecl.function_type_aux Fld_3 nodes.lkt:521







   if Local_Td_3.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start SynthFunDecl.function_type_aux
Fld_3 := Liblktlang.Implementation.Synth_Fun_Decl_P_Function_Type_Aux (Node => Local_Td_3.Node, E_Info => Local_Td_3.Info);
--# end
--# expr-done 225
Let_Result_4 := Fld_3; 
--# end
Scope_Result_4 := Let_Result_4;


   --# end


            Match_Result := Scope_Result_4; 
         when Lkt_Generic_Decl =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_5 := Create_Internal_Entity_Generic_Decl
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Local_Td_4 := Cast_Result_5; 
--# bind td Local_Td_4
--# expr-start 227 Decl.function_type Fld_5 nodes.lkt:522



--# expr-start 226 GenericDecl.decl Fld_4 nodes.lkt:522







   if Local_Td_4.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_4 := Create_Internal_Entity_Decl (Node => Local_Td_4.Node.Generic_Decl_F_Decl, Info => Local_Td_4.Info);
--# expr-done 226



   if Fld_4.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start Decl.function_type
Fld_5 := Liblktlang.Implementation.Decl_P_Function_Type (Node => Fld_4.Node, E_Info => Fld_4.Info);
--# end
--# expr-done 227
Let_Result_5 := Fld_5; 
--# end
Scope_Result_5 := Let_Result_5;


   --# end


            Match_Result := Scope_Result_5; 
         when Lkt_Grammar_Rule_Decl .. Lkt_Val_Decl | Lkt_Env_Spec_Decl .. Lkt_Error_Decl | Lkt_Grammar_Decl .. Lkt_Lexer_Family_Decl | Lkt_Synth_Param_Decl .. Lkt_Enum_Class_Alt_Decl | Lkt_Generic_Param_Type_Decl | Lkt_Enum_Class_Decl .. Lkt_Enum_Type_Decl | Lkt_Trait_Decl =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_6 := Create_Internal_Entity_Decl
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Ignored := Cast_Result_6; 
--# bind _ Ignored

Let_Result_6 := No_Entity_Function_Type; 
--# end
Scope_Result_6 := Let_Result_6;


   --# end


            Match_Result := Scope_Result_6; 
end case;

--# expr-done 228
   If_Result := Match_Result;
end if;



--# expr-done 229

         Property_Result := If_Result;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Decl_P_Function_Type;
--# end

   







--# property-start Decl.logic_function_type nodes.lkt:530
pragma Warnings (Off, "is not referenced");

function Decl_P_Logic_Function_Type
  
  (Node : Bare_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity_Function_Type
is
   Self : Bare_Decl :=
     Bare_Decl (Node);
     Ent : Internal_Entity_Decl :=
       Internal_Entity_Decl'(Node => Self, Info => E_Info);
      --# bind self Ent

   


   Property_Result : Internal_Entity_Function_Type;

      
            procedure Finalizer_Scope_491 with Inline_Always;

      Local_Td : Internal_Entity_Fun_Decl;
Fld : Internal_Entity_Function_Type;
Var_Expr : Internal_Entity_Function_Type;
Fld_1 : Internal_Entity_Named_Type_Decl;
Cast_Result : Internal_Entity_Type_Decl;
Array_Lit : Internal_Entity_Type_Decl_Array_Access;
Fld_2 : Internal_Entity_Type_Decl_Array_Access;
Concat_Result : Internal_Entity_Type_Decl_Array_Access;
Fld_3 : Internal_Entity_Type_Decl;
Fld_4 : Internal_Entity_Decl;
Fld_5 : Bare_Synthetic_Type_Ref_List;
New_Node : Bare_Function_Type;
As_Entity : Internal_Entity_Function_Type;
Is_Null : Boolean;
If_Result : Internal_Entity_Function_Type;
Fld_6 : Boolean;
Fld_7 : Internal_Entity_Named_Type_Decl;
Fld_8 : Internal_Entity_Type_Decl;
Array_Lit_1 : Internal_Entity_Type_Decl_Array_Access;
Fld_9 : Internal_Entity_Type_Decl;
Fld_10 : Internal_Entity_Decl;
Fld_11 : Bare_Synthetic_Type_Ref_List;
New_Node_1 : Bare_Function_Type;
As_Entity_1 : Internal_Entity_Function_Type;
Is_Null_1 : Boolean;
If_Result_1 : Internal_Entity_Function_Type;
If_Result_2 : Internal_Entity_Function_Type;
Scope_Result : Internal_Entity_Function_Type;
Result_Var : Internal_Entity_Function_Type;
Ignored : Internal_Entity_Decl;
Match_Prefix : Internal_Entity_Decl;
Cast_Result_1 : Internal_Entity_Fun_Decl;
Let_Result : Internal_Entity_Function_Type;
Scope_Result_1 : Internal_Entity_Function_Type;
Cast_Result_2 : Internal_Entity_Decl;
Let_Result_1 : Internal_Entity_Function_Type;
Scope_Result_2 : Internal_Entity_Function_Type;
Match_Result : Internal_Entity_Function_Type;

            procedure Finalizer_Scope_491 is
            begin
                     Dec_Ref (Array_Lit);
                     Dec_Ref (Fld_2);
                     Dec_Ref (Concat_Result);
                     Dec_Ref (Array_Lit_1);
            end Finalizer_Scope_491;

      
      Mmz_Handle : Memoization_Handle;
      Mmz_Val    : Mmz_Value;

      Mmz_Stored : Boolean;
      --  Whether the memoization couple was actually stored. Used to determine
      --  whether to inc-ref the memoized value.

      function Create_Mmz_Key return Mmz_Key;
      --  Create a memoization key for this property call and return it

      --------------------
      -- Create_Mmz_Key --
      --------------------

      function Create_Mmz_Key return Mmz_Key is
      begin
         return Mmz_K : Mmz_Key :=
           (Property => Mmz_Bare_Decl_Decl_P_Logic_Function_Type,
            Items    => new Mmz_Key_Array (1 ..  2))
         do
            Mmz_K.Items (1) := (Kind => Mmz_Bare_Decl,
                                As_Bare_Decl => Self);
               Mmz_K.Items (2) :=
                 (Kind => Mmz_Internal_Entity_Info,
                  As_Internal_Entity_Info => E_Info);
         end return;
      end Create_Mmz_Key;

begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;

      if Self /= null then

         if not Self.Unit.Context.In_Populate_Lexical_Env
            and then Find_Memoized_Value
                       (Self.Unit, Mmz_Handle, Mmz_Val, Create_Mmz_Key'Access)
         then
            --# memoization-lookup

            if Mmz_Val.Kind = Mmz_Evaluating then
               --# memoization-return
               Raise_Property_Exception
                 (Self,
                  Property_Error'Identity,
                  "Infinite recursion detected");

            elsif Mmz_Val.Kind = Mmz_Error then
               --# memoization-return
               Reraise_Memoized_Error (Mmz_Val);

            else
               Property_Result := Mmz_Val.As_Internal_Entity_Function_Type;

               --# memoization-return
               return Property_Result;
            end if;
            --# end
         end if;
      end if;

      begin
         
   --# scope-start

         --# expr-start 252 Match Match_Result nodes.lkt:530










   if Ent.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Match_Prefix := Ent; 

case Lkt_Decl (Match_Prefix.Node.Kind) is
         when Lkt_Fun_Decl =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_1 := Create_Internal_Entity_Fun_Decl
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Local_Td := Cast_Result_1; 
--# bind td Local_Td
--# expr-start 251 .do Result_Var nodes.lkt:532



--# expr-start 230 FunDecl.function_type_aux Fld nodes.lkt:532







   if Local_Td.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start FunDecl.function_type_aux
Fld := Liblktlang.Implementation.Fun_Decl_P_Function_Type_Aux (Node => Local_Td.Node, E_Info => Local_Td.Info);
--# end
--# expr-done 230
Var_Expr := Fld;





if Var_Expr /= No_Entity_Function_Type then
   



   --# scope-start



--# expr-start 250 If If_Result_2 nodes.lkt:534



--# expr-start 231 FunDecl.is_dynamic_combiner Fld_6 nodes.lkt:534







   if Local_Td.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start FunDecl.is_dynamic_combiner
Fld_6 := Liblktlang.Implementation.Fun_Decl_P_Is_Dynamic_Combiner (Node => Local_Td.Node, E_Info => Local_Td.Info);
--# end
--# expr-done 231
if Fld_6 then
   --# expr-start 239 .as_entity If_Result_1 nodes.lkt:536



--# expr-start 238 'New[FunctionType]' New_Node_1 nodes.lkt:536
--# expr-start 232 FunctionType.origin Fld_10 nodes.lkt:539







   if Var_Expr.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_10 := Var_Expr.Node.Function_Type_F_Origin;
--# expr-done 232
--# expr-start 235 ArrayLiteral Array_Lit_1 nodes.lkt:537
--# expr-start 234 TypeDecl.make_array_type Fld_8 nodes.lkt:537



--# expr-start 233 LktNode.logicvar_type Fld_7 nodes.lkt:537







   if Ent.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start LktNode.logicvar_type
Fld_7 := Liblktlang.Implementation.Lkt_Node_P_Logicvar_Type (Node => Ent.Node);
--# end
--# expr-done 233



   if Fld_7.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start TypeDecl.make_array_type
Fld_8 := Liblktlang.Implementation.Type_Decl_P_Make_Array_Type (Node => Fld_7.Node, E_Info => Fld_7.Info);
--# end
--# expr-done 234
Array_Lit_1 := Create_Internal_Entity_Type_Decl_Array (Internal_Internal_Entity_Type_Decl_Array'(1 => Fld_8)); 
--# expr-done 235
--# expr-start 236 FunctionType.return_type Fld_9 nodes.lkt:538







   if Var_Expr.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_9 := Var_Expr.Node.Function_Type_F_Return_Type;
--# expr-done 236
--# expr-start 237 LktNode.get_empty_type_ref_list Fld_11 nodes.lkt:540







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start LktNode.get_empty_type_ref_list
Fld_11 := Liblktlang.Implementation.Lkt_Node_P_Get_Empty_Type_Ref_List (Node => Self);
--# end
--# expr-done 237
Inc_Ref (Array_Lit_1);


if Is_Foreign_Strict (Self.Self_Env, Self) then
   Raise_Property_Exception
     (Self,
      Property_Error'Identity,
      "synthetic nodes cannot have foreign lexical envs");
end if;


New_Node_1 := new Root_Node_Record
  (Lkt_Function_Type);
Initialize
  (Self => New_Node_1,
   Kind => Lkt_Function_Type,
   Unit => Self.Unit,

   Token_Start_Index => No_Token_Index,
   Token_End_Index   => No_Token_Index,

   Parent => Self,

   Self_Env => Self.Self_Env);
Register_Destroyable (Self.Unit, New_Node_1);

   Initialize_Fields_For_Function_Type
     (Self => New_Node_1,
      Function_Type_F_Traits => Fld_11);

      New_Node_1.Function_Type_F_Origin :=
         Fld_10;
      New_Node_1.Function_Type_F_Params :=
         Array_Lit_1;
      New_Node_1.Function_Type_F_Return_Type :=
         Fld_9;

--# expr-done 238
Is_Null_1 := New_Node_1 = null; 
if Is_Null_1 then
   
   If_Result_1 := No_Entity_Function_Type;
else
   

As_Entity_1 := (Info => E_Info, Node => New_Node_1); 
   If_Result_1 := As_Entity_1;
end if;



--# expr-done 239
   If_Result_2 := If_Result_1;
else
   --# expr-start 249 .as_entity If_Result nodes.lkt:544



--# expr-start 248 'New[FunctionType]' New_Node nodes.lkt:544
--# expr-start 240 FunctionType.origin Fld_4 nodes.lkt:547







   if Var_Expr.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_4 := Var_Expr.Node.Function_Type_F_Origin;
--# expr-done 240
--# expr-start 245 ArrayConcat Concat_Result nodes.lkt:545
--# expr-start 243 ArrayLiteral Array_Lit nodes.lkt:545
--# expr-start 242 Cast Cast_Result nodes.lkt:545





--# expr-start 241 LktNode.logicvar_type Fld_1 nodes.lkt:545







   if Ent.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start LktNode.logicvar_type
Fld_1 := Liblktlang.Implementation.Lkt_Node_P_Logicvar_Type (Node => Ent.Node);
--# end
--# expr-done 241



   
      Cast_Result := Create_Internal_Entity_Type_Decl
        (Node => Fld_1.Node,
         Info => Fld_1.Info);


--# expr-done 242
Array_Lit := Create_Internal_Entity_Type_Decl_Array (Internal_Internal_Entity_Type_Decl_Array'(1 => Cast_Result)); 
--# expr-done 243
--# expr-start 244 FunctionType.params Fld_2 nodes.lkt:545







   if Var_Expr.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_2 := Var_Expr.Node.Function_Type_F_Params;
Inc_Ref (Fld_2);
--# expr-done 244
Concat_Result := Concat (Array_Lit, Fld_2); 
--# expr-done 245
--# expr-start 246 FunctionType.return_type Fld_3 nodes.lkt:546







   if Var_Expr.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_3 := Var_Expr.Node.Function_Type_F_Return_Type;
--# expr-done 246
--# expr-start 247 LktNode.get_empty_type_ref_list Fld_5 nodes.lkt:548







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start LktNode.get_empty_type_ref_list
Fld_5 := Liblktlang.Implementation.Lkt_Node_P_Get_Empty_Type_Ref_List (Node => Self);
--# end
--# expr-done 247
Inc_Ref (Concat_Result);


if Is_Foreign_Strict (Self.Self_Env, Self) then
   Raise_Property_Exception
     (Self,
      Property_Error'Identity,
      "synthetic nodes cannot have foreign lexical envs");
end if;


New_Node := new Root_Node_Record
  (Lkt_Function_Type);
Initialize
  (Self => New_Node,
   Kind => Lkt_Function_Type,
   Unit => Self.Unit,

   Token_Start_Index => No_Token_Index,
   Token_End_Index   => No_Token_Index,

   Parent => Self,

   Self_Env => Self.Self_Env);
Register_Destroyable (Self.Unit, New_Node);

   Initialize_Fields_For_Function_Type
     (Self => New_Node,
      Function_Type_F_Traits => Fld_5);

      New_Node.Function_Type_F_Origin :=
         Fld_4;
      New_Node.Function_Type_F_Params :=
         Concat_Result;
      New_Node.Function_Type_F_Return_Type :=
         Fld_3;

--# expr-done 248
Is_Null := New_Node = null; 
if Is_Null then
   
   If_Result := No_Entity_Function_Type;
else
   

As_Entity := (Info => E_Info, Node => New_Node); 
   If_Result := As_Entity;
end if;



--# expr-done 249
   If_Result_2 := If_Result;
end if;



--# expr-done 250
Scope_Result := If_Result_2;


   --# end
      Finalizer_Scope_491;


   Result_Var := Scope_Result;
else
   
   Result_Var := No_Entity_Function_Type;
end if;



--# expr-done 251
Let_Result := Result_Var; 
--# end
Scope_Result_1 := Let_Result;


   --# end


            Match_Result := Scope_Result_1; 
         when Lkt_Grammar_Rule_Decl .. Lkt_Val_Decl | Lkt_Env_Spec_Decl .. Lkt_Trait_Decl =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_2 := Create_Internal_Entity_Decl
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Ignored := Cast_Result_2; 
--# bind _ Ignored

Let_Result_1 := No_Entity_Function_Type; 
--# end
Scope_Result_2 := Let_Result_1;


   --# end


            Match_Result := Scope_Result_2; 
end case;

--# expr-done 252

         Property_Result := Match_Result;
         
   --# end


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_491;


               if Self /= null
                  and then not Self.Unit.Context.In_Populate_Lexical_Env
               then
                     Add_Memoized_Error
                       (Self.Unit, Mmz_Handle, Exc, Mmz_Stored);
               end if;


            raise;
      end;

      if Self /= null
         and then not Self.Unit.Context.In_Populate_Lexical_Env
      then
         Mmz_Val := (Kind => Mmz_Internal_Entity_Function_Type,
                     As_Internal_Entity_Function_Type => Property_Result);
         Add_Memoized_Value (Self.Unit, Mmz_Handle, Mmz_Val, Mmz_Stored);
      end if;



   return Property_Result;
end Decl_P_Logic_Function_Type;
--# end

   







--# property-start Decl.get_type nodes.lkt:559
pragma Warnings (Off, "is not referenced");

function Decl_P_Get_Type
  
  (Node : Bare_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity_Type_Decl
is
   Self : Bare_Decl :=
     Bare_Decl (Node);
     Ent : Internal_Entity_Decl :=
       Internal_Entity_Decl'(Node => Self, Info => E_Info);
      --# bind self Ent

   


   Property_Result : Internal_Entity_Type_Decl;

      
            procedure Finalizer_Scope_497 with Inline_Always;
            procedure Finalizer_Scope_505 with Inline_Always;

      Var_Expr : Internal_Entity_Decl;
Local_Td : Internal_Entity_Type_Decl;
Local_Fd : Internal_Entity_Fun_Decl;
Fld : Internal_Entity_Function_Type;
Fld_1 : Internal_Entity_Full_Decl;
Sym : Symbol_Type;
Fld_2 : Boolean;
Fld_3 : Internal_Entity_Type_Ref;
Fld_4 : Internal_Entity_Type_Decl;
Cast_Result : Internal_Entity_Type_Decl;
If_Result : Internal_Entity_Type_Decl;
Local_Eld : Internal_Entity_Enum_Lit_Decl;
Fld_5 : Internal_Entity_Type_Decl;
Local_Vd : Internal_Entity_Val_Decl;
Fld_6 : Internal_Solver_Result := No_Solver_Result;
Fld_7 : Boolean;
Fld_8 : Internal_Entity_Expr;
Fld_9 : Internal_Entity_Type_Decl;
If_Result_1 : Internal_Entity_Type_Decl;
Fld_10 : Internal_Entity_Type_Ref;
Is_Null : Boolean;
Not_Val : Boolean;
Fld_11 : Internal_Entity_Type_Ref;
Fld_12 : Internal_Entity_Type_Decl;
If_Result_2 : Internal_Entity_Type_Decl;
Local_Mvd : Internal_Entity_Match_Val_Decl;
Fld_13 : Internal_Entity_Expr;
Fld_14 : Internal_Entity_Type_Decl;
Var_Expr_1 : Internal_Entity_Type_Decl;
Fld_15 : Internal_Entity_Type_Ref;
Fld_16 : Internal_Entity_Type_Decl;
Fld_17 : Internal_Entity_Type_Decl;
Result_Var : Internal_Entity_Type_Decl;
Fld_18 : Internal_Entity_Type_Ref;
Is_Null_1 : Boolean;
Fld_19 : Internal_Entity_Expr;
Fld_20 : Internal_Entity_Type_Decl;
If_Result_3 : Internal_Entity_Type_Decl;
Local_Bvd : Internal_Entity_Binding_Val_Decl;
Fld_21 : Internal_Entity;
Fld_22 : Internal_Entity;
Cast_Result_1 : Internal_Entity_Pattern_Match_Branch;
Var_Expr_2 : Internal_Entity_Pattern_Match_Branch;
Fld_23 : Internal_Entity;
Fld_24 : Internal_Entity;
Cast_Result_2 : Internal_Entity_Match_Expr;
Fld_25 : Internal_Entity_Expr;
Fld_26 : Internal_Entity_Type_Decl;
Scope_Result : Internal_Entity_Type_Decl;
Result_Var_1 : Internal_Entity_Type_Decl;
Local_Lad : Internal_Entity_Lambda_Param_Decl;
Fld_27 : Internal_Entity_Type_Ref;
Fld_28 : Internal_Entity_Type_Decl;
Fld_29 : Internal_Entity_Type_Ref;
Is_Null_2 : Boolean;
Fld_30 : Internal_Entity;
Fld_31 : Internal_Entity;
Cast_Result_3 : Internal_Entity_Lambda_Expr;
Var_Expr_3 : Internal_Entity_Lambda_Expr;
Fld_32 : Boolean;
Fld_33 : Logic_Var;
Is_Logic_Var_Defined : Boolean;
Eq_Solution : Internal_Entity;
If_Result_4 : Internal_Entity;
Cast_Result_4 : Internal_Entity_Type_Decl;
If_Result_5 : Internal_Entity_Type_Decl;
Scope_Result_1 : Internal_Entity_Type_Decl;
Result_Var_2 : Internal_Entity_Type_Decl;
If_Result_6 : Internal_Entity_Type_Decl;
Local_Etd : Internal_Entity_Explicitly_Typed_Decl;
Fld_34 : Internal_Entity_Type_Ref;
Fld_35 : Internal_Entity_Type_Decl;
Local_Gd : Internal_Entity_Generic_Decl;
Fld_36 : Internal_Entity_Decl;
Fld_37 : Internal_Entity_Type_Decl;
Local_Sd : Internal_Entity_Self_Decl;
Fld_38 : Internal_Entity_Named_Type_Decl;
Fld_39 : Internal_Entity_Type_Decl;
Array_Lit : Internal_Entity_Type_Decl_Array_Access;
Fld_40 : Internal_Entity_Decl;
Cast_Result_5 : Internal_Entity_Type_Decl;
Local_Nd : Internal_Entity_Node_Decl;
Fld_41 : Internal_Entity_Type_Decl;
Ignored : Internal_Entity_Decl;
Match_Prefix : Internal_Entity_Decl;
Cast_Result_6 : Internal_Entity_Type_Decl;
Let_Result : Internal_Entity_Type_Decl;
Scope_Result_2 : Internal_Entity_Type_Decl;
Cast_Result_7 : Internal_Entity_Fun_Decl;
Let_Result_1 : Internal_Entity_Type_Decl;
Scope_Result_3 : Internal_Entity_Type_Decl;
Cast_Result_8 : Internal_Entity_Enum_Lit_Decl;
Let_Result_2 : Internal_Entity_Type_Decl;
Scope_Result_4 : Internal_Entity_Type_Decl;
Cast_Result_9 : Internal_Entity_Val_Decl;
Let_Result_3 : Internal_Entity_Type_Decl;
Scope_Result_5 : Internal_Entity_Type_Decl;
Cast_Result_10 : Internal_Entity_Match_Val_Decl;
Let_Result_4 : Internal_Entity_Type_Decl;
Scope_Result_6 : Internal_Entity_Type_Decl;
Cast_Result_11 : Internal_Entity_Binding_Val_Decl;
Let_Result_5 : Internal_Entity_Type_Decl;
Scope_Result_7 : Internal_Entity_Type_Decl;
Cast_Result_12 : Internal_Entity_Lambda_Param_Decl;
Let_Result_6 : Internal_Entity_Type_Decl;
Scope_Result_8 : Internal_Entity_Type_Decl;
Cast_Result_13 : Internal_Entity_Explicitly_Typed_Decl;
Let_Result_7 : Internal_Entity_Type_Decl;
Scope_Result_9 : Internal_Entity_Type_Decl;
Cast_Result_14 : Internal_Entity_Generic_Decl;
Let_Result_8 : Internal_Entity_Type_Decl;
Scope_Result_10 : Internal_Entity_Type_Decl;
Cast_Result_15 : Internal_Entity_Self_Decl;
Let_Result_9 : Internal_Entity_Type_Decl;
Scope_Result_11 : Internal_Entity_Type_Decl;
Cast_Result_16 : Internal_Entity_Node_Decl;
Let_Result_10 : Internal_Entity_Type_Decl;
Scope_Result_12 : Internal_Entity_Type_Decl;
Cast_Result_17 : Internal_Entity_Decl;
Let_Result_11 : Internal_Entity_Type_Decl;
Scope_Result_13 : Internal_Entity_Type_Decl;
Match_Result : Internal_Entity_Type_Decl;
Scope_Result_14 : Internal_Entity_Type_Decl;
Result_Var_3 : Internal_Entity_Type_Decl;

            procedure Finalizer_Scope_497 is
            begin
                     Dec_Ref (Fld_6);
            end Finalizer_Scope_497;
            procedure Finalizer_Scope_505 is
            begin
                     Dec_Ref (Array_Lit);
            end Finalizer_Scope_505;

      
      Mmz_Handle : Memoization_Handle;
      Mmz_Val    : Mmz_Value;

      Mmz_Stored : Boolean;
      --  Whether the memoization couple was actually stored. Used to determine
      --  whether to inc-ref the memoized value.

      function Create_Mmz_Key return Mmz_Key;
      --  Create a memoization key for this property call and return it

      --------------------
      -- Create_Mmz_Key --
      --------------------

      function Create_Mmz_Key return Mmz_Key is
      begin
         return Mmz_K : Mmz_Key :=
           (Property => Mmz_Bare_Decl_Decl_P_Get_Type,
            Items    => new Mmz_Key_Array (1 ..  2))
         do
            Mmz_K.Items (1) := (Kind => Mmz_Bare_Decl,
                                As_Bare_Decl => Self);
               Mmz_K.Items (2) :=
                 (Kind => Mmz_Internal_Entity_Info,
                  As_Internal_Entity_Info => E_Info);
         end return;
      end Create_Mmz_Key;

begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;

      if Self /= null then

         if not Self.Unit.Context.In_Populate_Lexical_Env
            and then Find_Memoized_Value
                       (Self.Unit, Mmz_Handle, Mmz_Val, Create_Mmz_Key'Access)
         then
            --# memoization-lookup

            if Mmz_Val.Kind = Mmz_Evaluating then
               --# memoization-return
               Raise_Property_Exception
                 (Self,
                  Property_Error'Identity,
                  "Infinite recursion detected");

            elsif Mmz_Val.Kind = Mmz_Error then
               --# memoization-return
               Reraise_Memoized_Error (Mmz_Val);

            else
               Property_Result := Mmz_Val.As_Internal_Entity_Type_Decl;

               --# memoization-return
               return Property_Result;
            end if;
            --# end
         end if;
      end if;

      begin
         
   --# scope-start

         --# expr-start 316 .do Result_Var_3 nodes.lkt:560




Var_Expr := Ent;





if Var_Expr /= No_Entity_Decl then
   



   --# scope-start



--# expr-start 315 Match Match_Result nodes.lkt:562










   if Var_Expr.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Match_Prefix := Var_Expr; 

case Lkt_Decl (Match_Prefix.Node.Kind) is
         when Lkt_Any_Type_Decl .. Lkt_Trait_Decl =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_6 := Create_Internal_Entity_Type_Decl
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Local_Td := Cast_Result_6; 
--# bind td Local_Td

Let_Result := Local_Td; 
--# end
Scope_Result_2 := Let_Result;


   --# end


            Match_Result := Scope_Result_2; 
         when Lkt_Fun_Decl =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_7 := Create_Internal_Entity_Fun_Decl
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Local_Fd := Cast_Result_7; 
--# bind fd Local_Fd
--# expr-start 259 If If_Result nodes.lkt:565



--# expr-start 255 FullDecl.has_annotation Fld_2 nodes.lkt:565



--# expr-start 253 Decl.full_decl Fld_1 nodes.lkt:565







   if Local_Fd.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start Decl.full_decl
Fld_1 := Liblktlang.Implementation.Decl_P_Full_Decl (Node => Local_Fd.Node, E_Info => Local_Fd.Info);
--# end
--# expr-done 253



   if Fld_1.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# expr-start 254 SymbolLiteral Sym nodes.lkt:565
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Property); 
--# expr-done 254
--# property-call-start FullDecl.has_annotation
Fld_2 := Liblktlang.Implementation.Full_Decl_P_Has_Annotation (Node => Fld_1.Node, Name => Sym);
--# end
--# expr-done 255
if Fld_2 then
   --# expr-start 257 TypeRef.referenced_decl Fld_4 nodes.lkt:566



--# expr-start 256 FunDecl.return_type Fld_3 nodes.lkt:566







   if Local_Fd.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_3 := Create_Internal_Entity_Type_Ref (Node => Local_Fd.Node.Fun_Decl_F_Return_Type, Info => Local_Fd.Info);
--# expr-done 256



   if Fld_3.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start TypeRef.referenced_decl
Fld_4 := Liblktlang.Implementation.Type_Ref_P_Referenced_Decl (Node => Fld_3.Node, E_Info => Fld_3.Info);
--# end
--# expr-done 257
   If_Result := Fld_4;
else
   




--# expr-start 258 Decl.function_type Fld nodes.lkt:567







   if Local_Fd.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start Decl.function_type
Fld := Liblktlang.Implementation.Decl_P_Function_Type (Node => Local_Fd.Node, E_Info => Local_Fd.Info);
--# end
--# expr-done 258



   
      Cast_Result := Create_Internal_Entity_Type_Decl
        (Node => Fld.Node,
         Info => Fld.Info);


   If_Result := Cast_Result;
end if;



--# expr-done 259
Let_Result_1 := If_Result; 
--# end
Scope_Result_3 := Let_Result_1;


   --# end


            Match_Result := Scope_Result_3; 
         when Lkt_Enum_Lit_Decl =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_8 := Create_Internal_Entity_Enum_Lit_Decl
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Local_Eld := Cast_Result_8; 
--# bind eld Local_Eld
--# expr-start 260 EnumLitDecl.parent_type Fld_5 nodes.lkt:568







   if Local_Eld.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start EnumLitDecl.parent_type
Fld_5 := Liblktlang.Implementation.Enum_Lit_Decl_P_Parent_Type (Node => Local_Eld.Node, E_Info => Local_Eld.Info);
--# end
--# expr-done 260
Let_Result_2 := Fld_5; 
--# end
Scope_Result_4 := Let_Result_2;


   --# end


            Match_Result := Scope_Result_4; 
         when Lkt_Val_Decl =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_9 := Create_Internal_Entity_Val_Decl
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Local_Vd := Cast_Result_9; 
--# bind vd Local_Vd
--# expr-start 271 If If_Result_2 nodes.lkt:570



--# expr-start 263 Not Not_Val nodes.lkt:570
--# expr-start 262 .is_null Is_Null nodes.lkt:570
--# expr-start 261 ValDecl.decl_type Fld_10 nodes.lkt:570







   if Local_Vd.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_10 := Create_Internal_Entity_Type_Ref (Node => Local_Vd.Node.Val_Decl_F_Decl_Type, Info => Local_Vd.Info);
--# expr-done 261
Is_Null := Fld_10.Node = null; 
--# expr-done 262
Not_Val := not (Is_Null); 
--# expr-done 263
if Not_Val then
   --# expr-start 265 TypeRef.referenced_decl Fld_12 nodes.lkt:571



--# expr-start 264 ValDecl.decl_type Fld_11 nodes.lkt:571







   if Local_Vd.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_11 := Create_Internal_Entity_Type_Ref (Node => Local_Vd.Node.Val_Decl_F_Decl_Type, Info => Local_Vd.Info);
--# expr-done 264



   if Fld_11.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start TypeRef.referenced_decl
Fld_12 := Liblktlang.Implementation.Type_Ref_P_Referenced_Decl (Node => Fld_11.Node, E_Info => Fld_11.Info);
--# end
--# expr-done 265
   If_Result_2 := Fld_12;
else
   --# expr-start 270 If If_Result_1 nodes.lkt:570



--# expr-start 267 SolverResult.success Fld_7 nodes.lkt:572



--# expr-start 266 LktNode.solve_enclosing_context Fld_6 nodes.lkt:572







   if Local_Vd.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start LktNode.solve_enclosing_context
Fld_6 := Liblktlang.Implementation.Lkt_Node_P_Solve_Enclosing_Context (Node => Local_Vd.Node, E_Info => Local_Vd.Info);
--# end
--# expr-done 266





Fld_7 := Fld_6.Success;
--# expr-done 267
if Fld_7 then
   --# expr-start 269 Expr.get_type Fld_9 nodes.lkt:573



--# expr-start 268 ValDecl.expr Fld_8 nodes.lkt:573







   if Local_Vd.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_8 := Create_Internal_Entity_Expr (Node => Local_Vd.Node.Val_Decl_F_Expr, Info => Local_Vd.Info);
--# expr-done 268



   if Fld_8.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start '[dispatcher]Expr.get_type'
Fld_9 := Liblktlang.Implementation.Dispatcher_Expr_P_Get_Type (Node => Fld_8.Node, E_Info => Fld_8.Info);
--# end
--# expr-done 269
   If_Result_1 := Fld_9;
else
   
   If_Result_1 := No_Entity_Type_Decl;
end if;



--# expr-done 270
   If_Result_2 := If_Result_1;
end if;



--# expr-done 271
Let_Result_3 := If_Result_2; 
--# end
Scope_Result_5 := Let_Result_3;


   --# end
      Finalizer_Scope_497;


            Match_Result := Scope_Result_5; 
         when Lkt_Match_Val_Decl =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_10 := Create_Internal_Entity_Match_Val_Decl
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Local_Mvd := Cast_Result_10; 
--# bind mvd Local_Mvd
--# expr-start 281 If If_Result_3 nodes.lkt:576



--# expr-start 273 .is_null Is_Null_1 nodes.lkt:576
--# expr-start 272 MatchValDecl.decl_type Fld_18 nodes.lkt:576







   if Local_Mvd.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_18 := Create_Internal_Entity_Type_Ref (Node => Local_Mvd.Node.Match_Val_Decl_F_Decl_Type, Info => Local_Mvd.Info);
--# expr-done 272
Is_Null_1 := Fld_18.Node = null; 
--# expr-done 273
if Is_Null_1 then
   --# expr-start 275 Expr.get_type Fld_20 nodes.lkt:576



--# expr-start 274 MatchValDecl.match_expr Fld_19 nodes.lkt:576







   if Local_Mvd.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start MatchValDecl.match_expr
Fld_19 := Liblktlang.Implementation.Match_Val_Decl_P_Match_Expr (Node => Local_Mvd.Node, E_Info => Local_Mvd.Info);
--# end
--# expr-done 274



   if Fld_19.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start '[dispatcher]Expr.get_type'
Fld_20 := Liblktlang.Implementation.Dispatcher_Expr_P_Get_Type (Node => Fld_19.Node, E_Info => Fld_19.Info);
--# end
--# expr-done 275
   If_Result_3 := Fld_20;
else
   


--# expr-start 277 Expr.get_type Fld_14 nodes.lkt:578



--# expr-start 276 MatchValDecl.match_expr Fld_13 nodes.lkt:578







   if Local_Mvd.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start MatchValDecl.match_expr
Fld_13 := Liblktlang.Implementation.Match_Val_Decl_P_Match_Expr (Node => Local_Mvd.Node, E_Info => Local_Mvd.Info);
--# end
--# expr-done 276



   if Fld_13.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start '[dispatcher]Expr.get_type'
Fld_14 := Liblktlang.Implementation.Dispatcher_Expr_P_Get_Type (Node => Fld_13.Node, E_Info => Fld_13.Info);
--# end
--# expr-done 277
Var_Expr_1 := Fld_14;





if Var_Expr_1 /= No_Entity_Type_Decl then
   --# expr-start 280 Decl.get_cast_type Fld_17 nodes.lkt:578







   if Var_Expr_1.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# expr-start 279 TypeRef.referenced_decl Fld_16 nodes.lkt:579



--# expr-start 278 MatchValDecl.decl_type Fld_15 nodes.lkt:579







   if Local_Mvd.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_15 := Create_Internal_Entity_Type_Ref (Node => Local_Mvd.Node.Match_Val_Decl_F_Decl_Type, Info => Local_Mvd.Info);
--# expr-done 278



   if Fld_15.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start TypeRef.referenced_decl
Fld_16 := Liblktlang.Implementation.Type_Ref_P_Referenced_Decl (Node => Fld_15.Node, E_Info => Fld_15.Info);
--# end
--# expr-done 279
--# property-call-start Decl.get_cast_type
Fld_17 := Liblktlang.Implementation.Decl_P_Get_Cast_Type (Node => Var_Expr_1.Node, Cast_To => Fld_16, E_Info => Var_Expr_1.Info);
--# end
--# expr-done 280
   Result_Var := Fld_17;
else
   
   Result_Var := No_Entity_Type_Decl;
end if;



   If_Result_3 := Result_Var;
end if;



--# expr-done 281
Let_Result_4 := If_Result_3; 
--# end
Scope_Result_6 := Let_Result_4;


   --# end


            Match_Result := Scope_Result_6; 
         when Lkt_Binding_Val_Decl =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_11 := Create_Internal_Entity_Binding_Val_Decl
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Local_Bvd := Cast_Result_11; 
--# bind bvd Local_Bvd
--# expr-start 290 .do Result_Var_1 nodes.lkt:587



--# expr-start 284 Cast Cast_Result_1 nodes.lkt:587





--# expr-start 283 .parent Fld_22 nodes.lkt:587



--# expr-start 282 .parent Fld_21 nodes.lkt:587







   if Local_Bvd.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_21 := Liblktlang.Implementation.Parent (Node => Local_Bvd.Node, E_Info => Local_Bvd.Info);
--# expr-done 282



   if Fld_21.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_22 := Liblktlang.Implementation.Parent (Node => Fld_21.Node, E_Info => Fld_21.Info);
--# expr-done 283



   if Fld_22.Node = null
      or else Fld_22.Node.Kind in Lkt_Pattern_Match_Branch_Range
   then
      
      Cast_Result_1 := Create_Internal_Entity_Pattern_Match_Branch
        (Node => Fld_22.Node,
         Info => Fld_22.Info);

   else
         Cast_Result_1 := No_Entity_Pattern_Match_Branch;
   end if;


--# expr-done 284
Var_Expr_2 := Cast_Result_1;





if Var_Expr_2 /= No_Entity_Pattern_Match_Branch then
   



   --# scope-start



--# expr-start 289 Expr.get_type Fld_26 nodes.lkt:589



--# expr-start 288 MatchExpr.match_expr Fld_25 nodes.lkt:589



--# expr-start 287 Cast Cast_Result_2 nodes.lkt:589





--# expr-start 286 .parent Fld_24 nodes.lkt:589



--# expr-start 285 .parent Fld_23 nodes.lkt:589







   if Var_Expr_2.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_23 := Liblktlang.Implementation.Parent (Node => Var_Expr_2.Node, E_Info => Var_Expr_2.Info);
--# expr-done 285



   if Fld_23.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_24 := Liblktlang.Implementation.Parent (Node => Fld_23.Node, E_Info => Fld_23.Info);
--# expr-done 286



   if Fld_24.Node = null
      or else Fld_24.Node.Kind in Lkt_Match_Expr_Range
   then
      
      Cast_Result_2 := Create_Internal_Entity_Match_Expr
        (Node => Fld_24.Node,
         Info => Fld_24.Info);

   else
         Raise_Property_Exception
           (Self, Property_Error'Identity, "invalid object cast");
   end if;


--# expr-done 287



   if Cast_Result_2.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_25 := Create_Internal_Entity_Expr (Node => Cast_Result_2.Node.Match_Expr_F_Match_Expr, Info => Cast_Result_2.Info);
--# expr-done 288



   if Fld_25.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start '[dispatcher]Expr.get_type'
Fld_26 := Liblktlang.Implementation.Dispatcher_Expr_P_Get_Type (Node => Fld_25.Node, E_Info => Fld_25.Info);
--# end
--# expr-done 289
Scope_Result := Fld_26;


   --# end


   Result_Var_1 := Scope_Result;
else
   
   Result_Var_1 := raise Property_Error with "Patterns typing not implemented yet";
end if;



--# expr-done 290
Let_Result_5 := Result_Var_1; 
--# end
Scope_Result_7 := Let_Result_5;


   --# end


            Match_Result := Scope_Result_7; 
         when Lkt_Lambda_Param_Decl =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_12 := Create_Internal_Entity_Lambda_Param_Decl
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Local_Lad := Cast_Result_12; 
--# bind lad Local_Lad
--# expr-start 304 If If_Result_6 nodes.lkt:595



--# expr-start 292 .is_null Is_Null_2 nodes.lkt:595
--# expr-start 291 LambdaParamDecl.decl_type Fld_29 nodes.lkt:595







   if Local_Lad.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_29 := Create_Internal_Entity_Type_Ref (Node => Local_Lad.Node.Lambda_Param_Decl_F_Decl_Type, Info => Local_Lad.Info);
--# expr-done 291
Is_Null_2 := Fld_29.Node = null; 
--# expr-done 292
if Is_Null_2 then
   --# expr-start 301 .do Result_Var_2 nodes.lkt:597



--# expr-start 295 Cast Cast_Result_3 nodes.lkt:597





--# expr-start 294 .parent Fld_31 nodes.lkt:597



--# expr-start 293 .parent Fld_30 nodes.lkt:597







   if Local_Lad.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_30 := Liblktlang.Implementation.Parent (Node => Local_Lad.Node, E_Info => Local_Lad.Info);
--# expr-done 293



   if Fld_30.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_31 := Liblktlang.Implementation.Parent (Node => Fld_30.Node, E_Info => Fld_30.Info);
--# expr-done 294



   if Fld_31.Node = null
      or else Fld_31.Node.Kind in Lkt_Lambda_Expr_Range
   then
      
      Cast_Result_3 := Create_Internal_Entity_Lambda_Expr
        (Node => Fld_31.Node,
         Info => Fld_31.Info);

   else
         Cast_Result_3 := No_Entity_Lambda_Expr;
   end if;


--# expr-done 295
Var_Expr_3 := Cast_Result_3;





if Var_Expr_3 /= No_Entity_Lambda_Expr then
   



   --# scope-start



--# expr-start 300 If If_Result_5 nodes.lkt:599



--# expr-start 296 LktNode.solve_generic_types Fld_32 nodes.lkt:599







   if Var_Expr_3.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start LktNode.solve_generic_types
Fld_32 := Liblktlang.Implementation.Lkt_Node_P_Solve_Generic_Types (Node => Var_Expr_3.Node, E_Info => Var_Expr_3.Info);
--# end
--# expr-done 296
if Fld_32 then
   --# expr-start 299 Cast Cast_Result_4 nodes.lkt:600





--# expr-start 298 .get_value If_Result_4 nodes.lkt:600



--# expr-start 297 LambdaParamDecl.type_var Fld_33 nodes.lkt:600







   if Local_Lad.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_33 := Local_Lad.Node.Lambda_Param_Decl_F_Type_Var'Unrestricted_Access;
--# expr-done 297
Is_Logic_Var_Defined := Entity_Vars.Is_Defined (Fld_33); 
if Is_Logic_Var_Defined then
   
Eq_Solution := Entity_Vars.Get_Value (Fld_33); 
   If_Result_4 := Eq_Solution;
else
   
   If_Result_4 := No_Entity;
end if;



--# expr-done 298



   if If_Result_4.Node = null
      or else If_Result_4.Node.Kind in Lkt_Type_Decl
   then
      
      Cast_Result_4 := Create_Internal_Entity_Type_Decl
        (Node => If_Result_4.Node,
         Info => If_Result_4.Info);

   else
         Cast_Result_4 := No_Entity_Type_Decl;
   end if;


--# expr-done 299
   If_Result_5 := Cast_Result_4;
else
   
   If_Result_5 := No_Entity_Type_Decl;
end if;



--# expr-done 300
Scope_Result_1 := If_Result_5;


   --# end


   Result_Var_2 := Scope_Result_1;
else
   
   Result_Var_2 := No_Entity_Type_Decl;
end if;



--# expr-done 301
   If_Result_6 := Result_Var_2;
else
   --# expr-start 303 TypeRef.referenced_decl Fld_28 nodes.lkt:603



--# expr-start 302 LambdaParamDecl.decl_type Fld_27 nodes.lkt:603







   if Local_Lad.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_27 := Create_Internal_Entity_Type_Ref (Node => Local_Lad.Node.Lambda_Param_Decl_F_Decl_Type, Info => Local_Lad.Info);
--# expr-done 302



   if Fld_27.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start TypeRef.referenced_decl
Fld_28 := Liblktlang.Implementation.Type_Ref_P_Referenced_Decl (Node => Fld_27.Node, E_Info => Fld_27.Info);
--# end
--# expr-done 303
   If_Result_6 := Fld_28;
end if;



--# expr-done 304
Let_Result_6 := If_Result_6; 
--# end
Scope_Result_8 := Let_Result_6;


   --# end


            Match_Result := Scope_Result_8; 
         when Lkt_Field_Decl .. Lkt_Fun_Param_Decl | Lkt_Dyn_Var_Decl =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_13 := Create_Internal_Entity_Explicitly_Typed_Decl
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Local_Etd := Cast_Result_13; 
--# bind etd Local_Etd
--# expr-start 306 TypeRef.referenced_decl Fld_35 nodes.lkt:605



--# expr-start 305 ExplicitlyTypedDecl.decl_type Fld_34 nodes.lkt:605







   if Local_Etd.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_34 := Create_Internal_Entity_Type_Ref (Node => Implementation.Explicitly_Typed_Decl_F_Decl_Type (Local_Etd.Node), Info => Local_Etd.Info);
--# expr-done 305



   if Fld_34.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start TypeRef.referenced_decl
Fld_35 := Liblktlang.Implementation.Type_Ref_P_Referenced_Decl (Node => Fld_34.Node, E_Info => Fld_34.Info);
--# end
--# expr-done 306
Let_Result_7 := Fld_35; 
--# end
Scope_Result_9 := Let_Result_7;


   --# end


            Match_Result := Scope_Result_9; 
         when Lkt_Generic_Decl =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_14 := Create_Internal_Entity_Generic_Decl
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Local_Gd := Cast_Result_14; 
--# bind gd Local_Gd
--# expr-start 308 Decl.get_type Fld_37 nodes.lkt:606



--# expr-start 307 GenericDecl.decl Fld_36 nodes.lkt:606







   if Local_Gd.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_36 := Create_Internal_Entity_Decl (Node => Local_Gd.Node.Generic_Decl_F_Decl, Info => Local_Gd.Info);
--# expr-done 307



   if Fld_36.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start Decl.get_type
Fld_37 := Liblktlang.Implementation.Decl_P_Get_Type (Node => Fld_36.Node, E_Info => Fld_36.Info);
--# end
--# expr-done 308
Let_Result_8 := Fld_37; 
--# end
Scope_Result_10 := Let_Result_8;


   --# end


            Match_Result := Scope_Result_10; 
         when Lkt_Self_Decl =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_15 := Create_Internal_Entity_Self_Decl
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Local_Sd := Cast_Result_15; 
--# bind sd Local_Sd
--# expr-start 313 Cast Cast_Result_5 nodes.lkt:608





--# expr-start 312 Decl.instantiate_generic_decl Fld_40 nodes.lkt:608



--# expr-start 309 LktNode.entity_type Fld_38 nodes.lkt:608







   if Ent.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start LktNode.entity_type
Fld_38 := Liblktlang.Implementation.Lkt_Node_P_Entity_Type (Node => Ent.Node);
--# end
--# expr-done 309



   if Fld_38.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# expr-start 311 ArrayLiteral Array_Lit nodes.lkt:609
--# expr-start 310 SelfDecl.owning_type Fld_39 nodes.lkt:609







   if Local_Sd.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start SelfDecl.owning_type
Fld_39 := Liblktlang.Implementation.Self_Decl_P_Owning_Type (Node => Local_Sd.Node, E_Info => Local_Sd.Info);
--# end
--# expr-done 310
Array_Lit := Create_Internal_Entity_Type_Decl_Array (Internal_Internal_Entity_Type_Decl_Array'(1 => Fld_39)); 
--# expr-done 311
--# property-call-start Decl.instantiate_generic_decl
Fld_40 := Liblktlang.Implementation.Decl_P_Instantiate_Generic_Decl (Node => Fld_38.Node, Param_Types => Array_Lit, E_Info => Fld_38.Info);
--# end
--# expr-done 312



   if Fld_40.Node = null
      or else Fld_40.Node.Kind in Lkt_Type_Decl
   then
      
      Cast_Result_5 := Create_Internal_Entity_Type_Decl
        (Node => Fld_40.Node,
         Info => Fld_40.Info);

   else
         Raise_Property_Exception
           (Self, Property_Error'Identity, "invalid object cast");
   end if;


--# expr-done 313
Let_Result_9 := Cast_Result_5; 
--# end
Scope_Result_11 := Let_Result_9;


   --# end
      Finalizer_Scope_505;


            Match_Result := Scope_Result_11; 
         when Lkt_Node_Decl =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_16 := Create_Internal_Entity_Node_Decl
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Local_Nd := Cast_Result_16; 
--# bind nd Local_Nd
--# expr-start 314 NodeDecl.owning_type Fld_41 nodes.lkt:612







   if Local_Nd.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start NodeDecl.owning_type
Fld_41 := Liblktlang.Implementation.Node_Decl_P_Owning_Type (Node => Local_Nd.Node, E_Info => Local_Nd.Info);
--# end
--# expr-done 314
Let_Result_10 := Fld_41; 
--# end
Scope_Result_12 := Let_Result_10;


   --# end


            Match_Result := Scope_Result_12; 
         when Lkt_Grammar_Rule_Decl .. Lkt_Synthetic_Lexer_Decl | Lkt_Env_Spec_Decl .. Lkt_Error_Decl | Lkt_Grammar_Decl .. Lkt_Synth_Param_Decl =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_17 := Create_Internal_Entity_Decl
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Ignored := Cast_Result_17; 
--# bind _ Ignored

Let_Result_11 := No_Entity_Type_Decl; 
--# end
Scope_Result_13 := Let_Result_11;


   --# end


            Match_Result := Scope_Result_13; 
end case;

--# expr-done 315
Scope_Result_14 := Match_Result;


   --# end


   Result_Var_3 := Scope_Result_14;
else
   
   Result_Var_3 := No_Entity_Type_Decl;
end if;



--# expr-done 316

         Property_Result := Result_Var_3;
         
   --# end


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_497;
                     Finalizer_Scope_505;


               if Self /= null
                  and then not Self.Unit.Context.In_Populate_Lexical_Env
               then
                     Add_Memoized_Error
                       (Self.Unit, Mmz_Handle, Exc, Mmz_Stored);
               end if;


            raise;
      end;

      if Self /= null
         and then not Self.Unit.Context.In_Populate_Lexical_Env
      then
         Mmz_Val := (Kind => Mmz_Internal_Entity_Type_Decl,
                     As_Internal_Entity_Type_Decl => Property_Result);
         Add_Memoized_Value (Self.Unit, Mmz_Handle, Mmz_Val, Mmz_Stored);
      end if;



   return Property_Result;
end Decl_P_Get_Type;
--# end

   







--# property-start Decl.get_cast_type nodes.lkt:620
pragma Warnings (Off, "is not referenced");

function Decl_P_Get_Cast_Type
  
  (Node : Bare_Decl
      ; Cast_To : Internal_Entity_Type_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity_Type_Decl
is
   Self : Bare_Decl :=
     Bare_Decl (Node);
     Ent : Internal_Entity_Decl :=
       Internal_Entity_Decl'(Node => Self, Info => E_Info);
      --# bind self Ent

   

   --# bind cast_to Cast_To

   Property_Result : Internal_Entity_Type_Decl;

      
            procedure Finalizer_Scope_95 with Inline_Always;

      Is_Null : Boolean;
Is_Null_1 : Boolean;
If_Result : Boolean;
Not_Val : Boolean;
Fld : Internal_Entity_Named_Type_Decl;
Fld_1 : Bare_Named_Type_Decl;
Cast_Result : Bare_Decl;
Is_Equal : Boolean;
If_Result_1 : Boolean;
Fld_2 : Bare_Type_Decl;
Fld_3 : Internal_Entity_Named_Type_Decl;
Fld_4 : Bare_Named_Type_Decl;
Cast_Result_1 : Bare_Type_Decl;
Is_Equal_1 : Boolean;
Not_Val_1 : Boolean;
If_Result_2 : Boolean;
Fld_5 : Internal_Entity_Named_Type_Decl;
Array_Lit : Internal_Entity_Type_Decl_Array_Access;
Fld_6 : Internal_Entity_Decl;
Cast_Result_2 : Internal_Entity_Decl;
If_Result_3 : Internal_Entity_Decl;
Cast_Result_3 : Internal_Entity_Type_Decl;

            procedure Finalizer_Scope_95 is
            begin
                     Dec_Ref (Array_Lit);
            end Finalizer_Scope_95;


begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 334 Cast Cast_Result_3 nodes.lkt:621





--# expr-start 333 If If_Result_3 nodes.lkt:622



--# expr-start 329 BooleanAnd If_Result_2 nodes.lkt:623



--# expr-start 324 BooleanAnd If_Result_1 nodes.lkt:624



--# expr-start 320 Not Not_Val nodes.lkt:624
--# expr-start 319 BooleanOr If_Result nodes.lkt:624



--# expr-start 317 .is_null Is_Null nodes.lkt:624

Is_Null := Cast_To.Node = null; 
--# expr-done 317
if Is_Null then
   
   If_Result := True;
else
   --# expr-start 318 .is_null Is_Null_1 nodes.lkt:624

Is_Null_1 := Ent.Node = null; 
--# expr-done 318
   If_Result := Is_Null_1;
end if;



--# expr-done 319
Not_Val := not (If_Result); 
--# expr-done 320
if Not_Val then
   --# expr-start 323 Eq Is_Equal nodes.lkt:625






--# expr-start 322 'Entity[NamedTypeDecl].node' Fld_1 nodes.lkt:625



--# expr-start 321 LktNode.entity_type Fld nodes.lkt:625







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start LktNode.entity_type
Fld := Liblktlang.Implementation.Lkt_Node_P_Entity_Type (Node => Self);
--# end
--# expr-done 321





Fld_1 := Fld.Node;
--# expr-done 322



   
      Cast_Result := Fld_1;


Is_Equal := Self = Cast_Result; 
--# expr-done 323
   If_Result_1 := Is_Equal;
else
   
   If_Result_1 := False;
end if;



--# expr-done 324
if If_Result_1 then
   --# expr-start 328 NotEqual Not_Val_1 nodes.lkt:627
--# expr-start 325 'Entity[TypeDecl].node' Fld_2 nodes.lkt:627









Fld_2 := Cast_To.Node;
--# expr-done 325





--# expr-start 327 'Entity[NamedTypeDecl].node' Fld_4 nodes.lkt:627



--# expr-start 326 LktNode.entity_type Fld_3 nodes.lkt:627







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start LktNode.entity_type
Fld_3 := Liblktlang.Implementation.Lkt_Node_P_Entity_Type (Node => Self);
--# end
--# expr-done 326





Fld_4 := Fld_3.Node;
--# expr-done 327



   
      Cast_Result_1 := Fld_4;


Is_Equal_1 := Fld_2 = Cast_Result_1; 
Not_Val_1 := not (Is_Equal_1); 
--# expr-done 328
   If_Result_2 := Not_Val_1;
else
   
   If_Result_2 := False;
end if;



--# expr-done 329
if If_Result_2 then
   --# expr-start 332 Decl.instantiate_generic_decl Fld_6 nodes.lkt:628



--# expr-start 330 LktNode.entity_type Fld_5 nodes.lkt:628







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start LktNode.entity_type
Fld_5 := Liblktlang.Implementation.Lkt_Node_P_Entity_Type (Node => Self);
--# end
--# expr-done 330



   if Fld_5.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# expr-start 331 ArrayLiteral Array_Lit nodes.lkt:628

Array_Lit := Create_Internal_Entity_Type_Decl_Array (Internal_Internal_Entity_Type_Decl_Array'(1 => Cast_To)); 
--# expr-done 331
--# property-call-start Decl.instantiate_generic_decl
Fld_6 := Liblktlang.Implementation.Decl_P_Instantiate_Generic_Decl (Node => Fld_5.Node, Param_Types => Array_Lit, E_Info => Fld_5.Info);
--# end
--# expr-done 332
   If_Result_3 := Fld_6;
else
   








   
      Cast_Result_2 := Create_Internal_Entity_Decl
        (Node => Cast_To.Node,
         Info => Cast_To.Info);


   If_Result_3 := Cast_Result_2;
end if;



--# expr-done 333



   if If_Result_3.Node = null
      or else If_Result_3.Node.Kind in Lkt_Type_Decl
   then
      
      Cast_Result_3 := Create_Internal_Entity_Type_Decl
        (Node => If_Result_3.Node,
         Info => If_Result_3.Info);

   else
         Cast_Result_3 := No_Entity_Type_Decl;
   end if;


--# expr-done 334

         Property_Result := Cast_Result_3;
         
   --# end
      Finalizer_Scope_95;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_95;




            raise;
      end;



   return Property_Result;
end Decl_P_Get_Cast_Type;
--# end

   







--# property-start Decl.get_keep_type nodes.lkt:637
pragma Warnings (Off, "is not referenced");

function Decl_P_Get_Keep_Type
  
  (Node : Bare_Decl
      ; Keep_Type : Internal_Entity_Type_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity_Type_Decl
is
   Self : Bare_Decl :=
     Bare_Decl (Node);
     Ent : Internal_Entity_Decl :=
       Internal_Entity_Decl'(Node => Self, Info => E_Info);
      --# bind self Ent

   

   --# bind keep_type Keep_Type

   Property_Result : Internal_Entity_Type_Decl;

      
            procedure Finalizer_Scope_96 with Inline_Always;

      Is_Null : Boolean;
Is_Null_1 : Boolean;
If_Result : Boolean;
Not_Val : Boolean;
Fld : Internal_Entity_Named_Type_Decl;
Fld_1 : Bare_Named_Type_Decl;
Cast_Result : Bare_Decl;
Is_Equal : Boolean;
Fld_2 : Internal_Entity_Named_Type_Decl;
Fld_3 : Bare_Named_Type_Decl;
Cast_Result_1 : Bare_Decl;
Is_Equal_1 : Boolean;
Cast_Result_2 : Internal_Entity_Type_Decl;
Fld_4 : Internal_Entity_Type_Decl;
Fld_5 : Bare_Type_Decl;
Fld_6 : Internal_Entity_Named_Type_Decl;
Fld_7 : Bare_Named_Type_Decl;
Cast_Result_3 : Bare_Type_Decl;
Is_Equal_2 : Boolean;
If_Result_1 : Boolean;
If_Result_2 : Boolean;
If_Result_3 : Boolean;
Fld_8 : Bare_Type_Decl;
Fld_9 : Internal_Entity_Named_Type_Decl;
Fld_10 : Bare_Named_Type_Decl;
Cast_Result_4 : Bare_Type_Decl;
Is_Equal_3 : Boolean;
Not_Val_1 : Boolean;
If_Result_4 : Boolean;
Fld_11 : Internal_Entity_Named_Type_Decl;
Array_Lit : Internal_Entity_Type_Decl_Array_Access;
Fld_12 : Internal_Entity_Decl;
Cast_Result_5 : Internal_Entity_Decl;
If_Result_5 : Internal_Entity_Decl;
Cast_Result_6 : Internal_Entity_Type_Decl;
Var_Expr : Internal_Entity_Type_Decl;
Fld_13 : Internal_Entity_Type_Decl;
Result_Var : Internal_Entity_Type_Decl;

            procedure Finalizer_Scope_96 is
            begin
                     Dec_Ref (Array_Lit);
            end Finalizer_Scope_96;


begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         


--# expr-start 363 Cast Cast_Result_6 nodes.lkt:638





--# expr-start 362 If If_Result_5 nodes.lkt:639



--# expr-start 358 BooleanAnd If_Result_4 nodes.lkt:640



--# expr-start 353 BooleanAnd If_Result_3 nodes.lkt:641



--# expr-start 338 Not Not_Val nodes.lkt:641
--# expr-start 337 BooleanOr If_Result nodes.lkt:641



--# expr-start 335 .is_null Is_Null nodes.lkt:641

Is_Null := Keep_Type.Node = null; 
--# expr-done 335
if Is_Null then
   
   If_Result := True;
else
   --# expr-start 336 .is_null Is_Null_1 nodes.lkt:641

Is_Null_1 := Ent.Node = null; 
--# expr-done 336
   If_Result := Is_Null_1;
end if;



--# expr-done 337
Not_Val := not (If_Result); 
--# expr-done 338
if Not_Val then
   --# expr-start 352 BooleanOr If_Result_2 nodes.lkt:643



--# expr-start 341 Eq Is_Equal nodes.lkt:643






--# expr-start 340 'Entity[NamedTypeDecl].node' Fld_1 nodes.lkt:643



--# expr-start 339 LktNode.entity_type Fld nodes.lkt:643







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start LktNode.entity_type
Fld := Liblktlang.Implementation.Lkt_Node_P_Entity_Type (Node => Self);
--# end
--# expr-done 339





Fld_1 := Fld.Node;
--# expr-done 340



   
      Cast_Result := Fld_1;


Is_Equal := Self = Cast_Result; 
--# expr-done 341
if Is_Equal then
   
   If_Result_2 := True;
else
   --# expr-start 351 BooleanAnd If_Result_1 nodes.lkt:645



--# expr-start 344 Eq Is_Equal_1 nodes.lkt:645






--# expr-start 343 'Entity[NamedTypeDecl].node' Fld_3 nodes.lkt:645



--# expr-start 342 LktNode.array_type Fld_2 nodes.lkt:645







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start LktNode.array_type
Fld_2 := Liblktlang.Implementation.Lkt_Node_P_Array_Type (Node => Self);
--# end
--# expr-done 342





Fld_3 := Fld_2.Node;
--# expr-done 343



   
      Cast_Result_1 := Fld_3;


Is_Equal_1 := Self = Cast_Result_1; 
--# expr-done 344
if Is_Equal_1 then
   --# expr-start 350 Eq Is_Equal_2 nodes.lkt:646
--# expr-start 347 'Entity[TypeDecl].node' Fld_5 nodes.lkt:646



--# expr-start 346 TypeDecl.get_array_content_type Fld_4 nodes.lkt:646



--# expr-start 345 Cast Cast_Result_2 nodes.lkt:646









   if Ent.Node = null
      or else Ent.Node.Kind in Lkt_Type_Decl
   then
      
      Cast_Result_2 := Create_Internal_Entity_Type_Decl
        (Node => Ent.Node,
         Info => Ent.Info);

   else
         Cast_Result_2 := No_Entity_Type_Decl;
   end if;


--# expr-done 345



   if Cast_Result_2.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start TypeDecl.get_array_content_type
Fld_4 := Liblktlang.Implementation.Type_Decl_P_Get_Array_Content_Type (Node => Cast_Result_2.Node, E_Info => Cast_Result_2.Info);
--# end
--# expr-done 346





Fld_5 := Fld_4.Node;
--# expr-done 347





--# expr-start 349 'Entity[NamedTypeDecl].node' Fld_7 nodes.lkt:647



--# expr-start 348 LktNode.entity_type Fld_6 nodes.lkt:647







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start LktNode.entity_type
Fld_6 := Liblktlang.Implementation.Lkt_Node_P_Entity_Type (Node => Self);
--# end
--# expr-done 348





Fld_7 := Fld_6.Node;
--# expr-done 349



   
      Cast_Result_3 := Fld_7;


Is_Equal_2 := Fld_5 = Cast_Result_3; 
--# expr-done 350
   If_Result_1 := Is_Equal_2;
else
   
   If_Result_1 := False;
end if;



--# expr-done 351
   If_Result_2 := If_Result_1;
end if;



--# expr-done 352
   If_Result_3 := If_Result_2;
else
   
   If_Result_3 := False;
end if;



--# expr-done 353
if If_Result_3 then
   --# expr-start 357 NotEqual Not_Val_1 nodes.lkt:651
--# expr-start 354 'Entity[TypeDecl].node' Fld_8 nodes.lkt:651









Fld_8 := Keep_Type.Node;
--# expr-done 354





--# expr-start 356 'Entity[NamedTypeDecl].node' Fld_10 nodes.lkt:651



--# expr-start 355 LktNode.entity_type Fld_9 nodes.lkt:651







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start LktNode.entity_type
Fld_9 := Liblktlang.Implementation.Lkt_Node_P_Entity_Type (Node => Self);
--# end
--# expr-done 355





Fld_10 := Fld_9.Node;
--# expr-done 356



   
      Cast_Result_4 := Fld_10;


Is_Equal_3 := Fld_8 = Cast_Result_4; 
Not_Val_1 := not (Is_Equal_3); 
--# expr-done 357
   If_Result_4 := Not_Val_1;
else
   
   If_Result_4 := False;
end if;



--# expr-done 358
if If_Result_4 then
   --# expr-start 361 Decl.instantiate_generic_decl Fld_12 nodes.lkt:652



--# expr-start 359 LktNode.entity_type Fld_11 nodes.lkt:652







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start LktNode.entity_type
Fld_11 := Liblktlang.Implementation.Lkt_Node_P_Entity_Type (Node => Self);
--# end
--# expr-done 359



   if Fld_11.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# expr-start 360 ArrayLiteral Array_Lit nodes.lkt:652

Array_Lit := Create_Internal_Entity_Type_Decl_Array (Internal_Internal_Entity_Type_Decl_Array'(1 => Keep_Type)); 
--# expr-done 360
--# property-call-start Decl.instantiate_generic_decl
Fld_12 := Liblktlang.Implementation.Decl_P_Instantiate_Generic_Decl (Node => Fld_11.Node, Param_Types => Array_Lit, E_Info => Fld_11.Info);
--# end
--# expr-done 361
   If_Result_5 := Fld_12;
else
   








   
      Cast_Result_5 := Create_Internal_Entity_Decl
        (Node => Keep_Type.Node,
         Info => Keep_Type.Info);


   If_Result_5 := Cast_Result_5;
end if;



--# expr-done 362



   if If_Result_5.Node = null
      or else If_Result_5.Node.Kind in Lkt_Type_Decl
   then
      
      Cast_Result_6 := Create_Internal_Entity_Type_Decl
        (Node => If_Result_5.Node,
         Info => If_Result_5.Info);

   else
         Cast_Result_6 := No_Entity_Type_Decl;
   end if;


--# expr-done 363
Var_Expr := Cast_Result_6;





if Var_Expr /= No_Entity_Type_Decl then
   --# expr-start 364 TypeDecl.make_array_type Fld_13 nodes.lkt:638







   if Var_Expr.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start TypeDecl.make_array_type
Fld_13 := Liblktlang.Implementation.Type_Decl_P_Make_Array_Type (Node => Var_Expr.Node, E_Info => Var_Expr.Info);
--# end
--# expr-done 364
   Result_Var := Fld_13;
else
   
   Result_Var := No_Entity_Type_Decl;
end if;




         Property_Result := Result_Var;
         
   --# end
      Finalizer_Scope_96;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_96;




            raise;
      end;



   return Property_Result;
end Decl_P_Get_Keep_Type;
--# end

   







--# property-start Decl.get_suffix_type nodes.lkt:661
pragma Warnings (Off, "is not referenced");

function Decl_P_Get_Suffix_Type
  
  (Node : Bare_Decl
      ; Prefix_Type : Internal_Entity_Type_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity_Type_Decl
is
   Self : Bare_Decl :=
     Bare_Decl (Node);
     Ent : Internal_Entity_Decl :=
       Internal_Entity_Decl'(Node => Self, Info => E_Info);
      --# bind self Ent

   

   --# bind prefix_type Prefix_Type

   Property_Result : Internal_Entity_Type_Decl;

      
            procedure Finalizer_Scope_508 with Inline_Always;

      Var_Expr : Internal_Entity_Decl;
Fld : Internal_Entity_Type_Decl;
Result_Var : Internal_Entity_Type_Decl;
Local_Suffix_Type : Internal_Entity_Type_Decl;
Is_Null : Boolean;
Not_Val : Boolean;
Is_Null_1 : Boolean;
If_Result : Boolean;
Is_A : Boolean;
If_Result_1 : Boolean;
Fld_1 : Internal_Entity_Full_Decl;
Sym : Symbol_Type;
Fld_2 : Boolean;
If_Result_2 : Boolean;
Fld_3 : Internal_Entity_Function_Type;
Cast_Result : Internal_Entity_Type_Decl;
If_Result_3 : Internal_Entity_Type_Decl;
Is_Null_2 : Boolean;
Not_Val_1 : Boolean;
Is_Null_3 : Boolean;
If_Result_4 : Boolean;
Is_A_1 : Boolean;
If_Result_5 : Boolean;
Fld_4 : Internal_Entity_Full_Decl;
Sym_1 : Symbol_Type;
Fld_5 : Boolean;
If_Result_6 : Boolean;
Cast_Result_1 : Internal_Entity_Field_Decl;
Fld_6 : Internal_Entity_Function_Type;
Cast_Result_2 : Internal_Entity_Type_Decl;
If_Result_7 : Internal_Entity_Type_Decl;
Is_Null_4 : Boolean;
Not_Val_2 : Boolean;
Is_Null_5 : Boolean;
Not_Val_3 : Boolean;
If_Result_8 : Boolean;
Fld_7 : Bare_Type_Decl;
Fld_8 : Internal_Entity_Named_Type_Decl;
Fld_9 : Bare_Named_Type_Decl;
Cast_Result_3 : Bare_Type_Decl;
Is_Equal : Boolean;
If_Result_9 : Boolean;
Fld_10 : Internal_Entity_Full_Decl;
Var_Expr_1 : Internal_Entity_Full_Decl;
Sym_2 : Symbol_Type;
Fld_11 : Boolean;
Result_Var_1 : Boolean;
If_Result_10 : Boolean;
Fld_12 : Internal_Entity_Named_Type_Decl;
Array_Lit : Internal_Entity_Type_Decl_Array_Access;
Fld_13 : Internal_Entity_Decl;
Cast_Result_4 : Internal_Entity_Decl;
If_Result_11 : Internal_Entity_Decl;
Cast_Result_5 : Internal_Entity_Type_Decl;
Let_Result : Internal_Entity_Type_Decl;
Scope_Result : Internal_Entity_Type_Decl;

            procedure Finalizer_Scope_508 is
            begin
                     Dec_Ref (Array_Lit);
            end Finalizer_Scope_508;


begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         



   --# scope-start



--# expr-start 410 ValDecl Let_Result nodes.lkt:662
--# scope-start




Var_Expr := Ent;





if Var_Expr /= No_Entity_Decl then
   --# expr-start 365 Decl.get_type Fld nodes.lkt:662







   if Var_Expr.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start Decl.get_type
Fld := Liblktlang.Implementation.Decl_P_Get_Type (Node => Var_Expr.Node, E_Info => Var_Expr.Info);
--# end
--# expr-done 365
   Result_Var := Fld;
else
   
   Result_Var := No_Entity_Type_Decl;
end if;



Local_Suffix_Type := Result_Var; 
--# bind suffix_type Local_Suffix_Type
--# expr-start 409 Cast Cast_Result_5 nodes.lkt:664





--# expr-start 408 If If_Result_11 nodes.lkt:665



--# expr-start 379 BooleanAnd If_Result_10 nodes.lkt:666



--# expr-start 375 BooleanAnd If_Result_9 nodes.lkt:667



--# expr-start 370 BooleanAnd If_Result_8 nodes.lkt:667



--# expr-start 367 Not Not_Val_2 nodes.lkt:667
--# expr-start 366 .is_null Is_Null_4 nodes.lkt:667

Is_Null_4 := Prefix_Type.Node = null; 
--# expr-done 366
Not_Val_2 := not (Is_Null_4); 
--# expr-done 367
if Not_Val_2 then
   --# expr-start 369 Not Not_Val_3 nodes.lkt:667
--# expr-start 368 .is_null Is_Null_5 nodes.lkt:667

Is_Null_5 := Ent.Node = null; 
--# expr-done 368
Not_Val_3 := not (Is_Null_5); 
--# expr-done 369
   If_Result_8 := Not_Val_3;
else
   
   If_Result_8 := False;
end if;



--# expr-done 370
if If_Result_8 then
   --# expr-start 374 Eq Is_Equal nodes.lkt:668
--# expr-start 371 'Entity[TypeDecl].node' Fld_7 nodes.lkt:668









Fld_7 := Prefix_Type.Node;
--# expr-done 371





--# expr-start 373 'Entity[NamedTypeDecl].node' Fld_9 nodes.lkt:668



--# expr-start 372 LktNode.entity_type Fld_8 nodes.lkt:668







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start LktNode.entity_type
Fld_8 := Liblktlang.Implementation.Lkt_Node_P_Entity_Type (Node => Self);
--# end
--# expr-done 372





Fld_9 := Fld_8.Node;
--# expr-done 373



   
      Cast_Result_3 := Fld_9;


Is_Equal := Fld_7 = Cast_Result_3; 
--# expr-done 374
   If_Result_9 := Is_Equal;
else
   
   If_Result_9 := False;
end if;



--# expr-done 375
if If_Result_9 then
   


--# expr-start 376 Decl.full_decl Fld_10 nodes.lkt:670







   if Ent.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start Decl.full_decl
Fld_10 := Liblktlang.Implementation.Decl_P_Full_Decl (Node => Ent.Node, E_Info => Ent.Info);
--# end
--# expr-done 376
Var_Expr_1 := Fld_10;





if Var_Expr_1 /= No_Entity_Full_Decl then
   --# expr-start 378 FullDecl.has_annotation Fld_11 nodes.lkt:670







   if Var_Expr_1.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# expr-start 377 SymbolLiteral Sym_2 nodes.lkt:670
Sym_2 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Parse_Field); 
--# expr-done 377
--# property-call-start FullDecl.has_annotation
Fld_11 := Liblktlang.Implementation.Full_Decl_P_Has_Annotation (Node => Var_Expr_1.Node, Name => Sym_2);
--# end
--# expr-done 378
   Result_Var_1 := Fld_11;
else
   
   Result_Var_1 := False;
end if;



   If_Result_10 := Result_Var_1;
else
   
   If_Result_10 := False;
end if;



--# expr-done 379
if If_Result_10 then
   --# expr-start 382 Decl.instantiate_generic_decl Fld_13 nodes.lkt:671



--# expr-start 380 LktNode.entity_type Fld_12 nodes.lkt:671







   if Self = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start LktNode.entity_type
Fld_12 := Liblktlang.Implementation.Lkt_Node_P_Entity_Type (Node => Self);
--# end
--# expr-done 380



   if Fld_12.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# expr-start 381 ArrayLiteral Array_Lit nodes.lkt:671

Array_Lit := Create_Internal_Entity_Type_Decl_Array (Internal_Internal_Entity_Type_Decl_Array'(1 => Local_Suffix_Type)); 
--# expr-done 381
--# property-call-start Decl.instantiate_generic_decl
Fld_13 := Liblktlang.Implementation.Decl_P_Instantiate_Generic_Decl (Node => Fld_12.Node, Param_Types => Array_Lit, E_Info => Fld_12.Info);
--# end
--# expr-done 382
   If_Result_11 := Fld_13;
else
   




--# expr-start 407 If If_Result_7 nodes.lkt:665



--# expr-start 392 BooleanAnd If_Result_6 nodes.lkt:677



--# expr-start 388 BooleanAnd If_Result_5 nodes.lkt:678



--# expr-start 386 BooleanAnd If_Result_4 nodes.lkt:678



--# expr-start 384 Not Not_Val_1 nodes.lkt:678
--# expr-start 383 .is_null Is_Null_2 nodes.lkt:678

Is_Null_2 := Ent.Node = null; 
--# expr-done 383
Not_Val_1 := not (Is_Null_2); 
--# expr-done 384
if Not_Val_1 then
   --# expr-start 385 .is_null Is_Null_3 nodes.lkt:678

Is_Null_3 := Prefix_Type.Node = null; 
--# expr-done 385
   If_Result_4 := Is_Null_3;
else
   
   If_Result_4 := False;
end if;



--# expr-done 386
if If_Result_4 then
   --# expr-start 387 IsA Is_A_1 nodes.lkt:679

Is_A_1 := Self /= null 
and then Self.Kind in Lkt_Field_Decl_Range; 
--# expr-done 387
   If_Result_5 := Is_A_1;
else
   
   If_Result_5 := False;
end if;



--# expr-done 388
if If_Result_5 then
   --# expr-start 391 FullDecl.has_annotation Fld_5 nodes.lkt:681



--# expr-start 389 Decl.full_decl Fld_4 nodes.lkt:681







   if Ent.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start Decl.full_decl
Fld_4 := Liblktlang.Implementation.Decl_P_Full_Decl (Node => Ent.Node, E_Info => Ent.Info);
--# end
--# expr-done 389



   if Fld_4.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# expr-start 390 SymbolLiteral Sym_1 nodes.lkt:681
Sym_1 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Lazy); 
--# expr-done 390
--# property-call-start FullDecl.has_annotation
Fld_5 := Liblktlang.Implementation.Full_Decl_P_Has_Annotation (Node => Fld_4.Node, Name => Sym_1);
--# end
--# expr-done 391
   If_Result_6 := Fld_5;
else
   
   If_Result_6 := False;
end if;



--# expr-done 392
if If_Result_6 then
   




--# expr-start 394 FieldDecl.lazy_field_function_type Fld_6 nodes.lkt:682



--# expr-start 393 Cast Cast_Result_1 nodes.lkt:682









   if Ent.Node = null
      or else Ent.Node.Kind in Lkt_Field_Decl_Range
   then
      
      Cast_Result_1 := Create_Internal_Entity_Field_Decl
        (Node => Ent.Node,
         Info => Ent.Info);

   else
         Cast_Result_1 := No_Entity_Field_Decl;
   end if;


--# expr-done 393



   if Cast_Result_1.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start FieldDecl.lazy_field_function_type
Fld_6 := Liblktlang.Implementation.Field_Decl_P_Lazy_Field_Function_Type (Node => Cast_Result_1.Node, E_Info => Cast_Result_1.Info);
--# end
--# expr-done 394



   
      Cast_Result_2 := Create_Internal_Entity_Type_Decl
        (Node => Fld_6.Node,
         Info => Fld_6.Info);


   If_Result_7 := Cast_Result_2;
else
   --# expr-start 406 If If_Result_3 nodes.lkt:665



--# expr-start 404 BooleanAnd If_Result_2 nodes.lkt:684



--# expr-start 400 BooleanAnd If_Result_1 nodes.lkt:685



--# expr-start 398 BooleanAnd If_Result nodes.lkt:685



--# expr-start 396 Not Not_Val nodes.lkt:685
--# expr-start 395 .is_null Is_Null nodes.lkt:685

Is_Null := Ent.Node = null; 
--# expr-done 395
Not_Val := not (Is_Null); 
--# expr-done 396
if Not_Val then
   --# expr-start 397 .is_null Is_Null_1 nodes.lkt:685

Is_Null_1 := Prefix_Type.Node = null; 
--# expr-done 397
   If_Result := Is_Null_1;
else
   
   If_Result := False;
end if;



--# expr-done 398
if If_Result then
   --# expr-start 399 IsA Is_A nodes.lkt:686

Is_A := Self /= null 
and then Self.Kind in Lkt_Fun_Decl_Range; 
--# expr-done 399
   If_Result_1 := Is_A;
else
   
   If_Result_1 := False;
end if;



--# expr-done 400
if If_Result_1 then
   --# expr-start 403 FullDecl.has_annotation Fld_2 nodes.lkt:688



--# expr-start 401 Decl.full_decl Fld_1 nodes.lkt:688







   if Ent.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start Decl.full_decl
Fld_1 := Liblktlang.Implementation.Decl_P_Full_Decl (Node => Ent.Node, E_Info => Ent.Info);
--# end
--# expr-done 401



   if Fld_1.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# expr-start 402 SymbolLiteral Sym nodes.lkt:688
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Property); 
--# expr-done 402
--# property-call-start FullDecl.has_annotation
Fld_2 := Liblktlang.Implementation.Full_Decl_P_Has_Annotation (Node => Fld_1.Node, Name => Sym);
--# end
--# expr-done 403
   If_Result_2 := Fld_2;
else
   
   If_Result_2 := False;
end if;



--# expr-done 404
if If_Result_2 then
   




--# expr-start 405 Decl.function_type Fld_3 nodes.lkt:689







   if Ent.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start Decl.function_type
Fld_3 := Liblktlang.Implementation.Decl_P_Function_Type (Node => Ent.Node, E_Info => Ent.Info);
--# end
--# expr-done 405



   
      Cast_Result := Create_Internal_Entity_Type_Decl
        (Node => Fld_3.Node,
         Info => Fld_3.Info);


   If_Result_3 := Cast_Result;
else
   
   If_Result_3 := Local_Suffix_Type;
end if;



--# expr-done 406
   If_Result_7 := If_Result_3;
end if;



--# expr-done 407



   
      Cast_Result_4 := Create_Internal_Entity_Decl
        (Node => If_Result_7.Node,
         Info => If_Result_7.Info);


   If_Result_11 := Cast_Result_4;
end if;



--# expr-done 408



   if If_Result_11.Node = null
      or else If_Result_11.Node.Kind in Lkt_Type_Decl
   then
      
      Cast_Result_5 := Create_Internal_Entity_Type_Decl
        (Node => If_Result_11.Node,
         Info => If_Result_11.Info);

   else
         Cast_Result_5 := No_Entity_Type_Decl;
   end if;


--# expr-done 409
Let_Result := Cast_Result_5; 
--# end
--# expr-done 410
Scope_Result := Let_Result;


   --# end
      Finalizer_Scope_508;



         Property_Result := Scope_Result;
         
   --# end


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_508;




            raise;
      end;



   return Property_Result;
end Decl_P_Get_Suffix_Type;
--# end

   







--# property-start Decl.type_var_suffix_ref nodes.lkt:704
pragma Warnings (Off, "is not referenced");

function Decl_P_Type_Var_Suffix_Ref
  
  (Node : Bare_Decl
      ; Current_Name : Internal_Entity_Ref_Id
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity_Decl
is
   Self : Bare_Decl :=
     Bare_Decl (Node);
     Ent : Internal_Entity_Decl :=
       Internal_Entity_Decl'(Node => Self, Info => E_Info);
      --# bind self Ent

   

   --# bind current_name Current_Name

   Property_Result : Internal_Entity_Decl;

      
            procedure Finalizer_Scope_511 with Inline_Always;
            procedure Finalizer_Scope_513 with Inline_Always;

      Var_Expr : Internal_Entity_Decl;
Local_E : Internal_Entity_Enum_Type_Decl;
Origin : Internal_Entity;
Fld : Internal_Entity_Type_Decl;
Fld_1 : Lexical_Env := Empty_Env;
Fld_2 : Bare_Ref_Id;
Sym : Symbol_Type;
Env_Get_Result : Internal_Entity;
Cast_Result : Internal_Entity_Decl;
Var_Expr_1 : Internal_Entity_Decl;
Fld_3 : Internal_Entity_Decl;
Result_Var : Internal_Entity_Decl;
Cast_Result_1 : Internal_Entity;
Dyn_Var_Bind_Result : Internal_Entity_Decl;
Scope_Result : Internal_Entity_Decl;
Ignored : Internal_Entity_Decl;
Origin_1 : Internal_Entity;
Fld_4 : Lexical_Env := Empty_Env;
Fld_5 : Bare_Ref_Id;
Sym_1 : Symbol_Type;
Env_Get_Result_1 : Internal_Entity;
Cast_Result_2 : Internal_Entity_Decl;
Var_Expr_2 : Internal_Entity_Decl;
Fld_6 : Internal_Entity_Decl;
Result_Var_1 : Internal_Entity_Decl;
Cast_Result_3 : Internal_Entity;
Dyn_Var_Bind_Result_1 : Internal_Entity_Decl;
Scope_Result_1 : Internal_Entity_Decl;
Match_Prefix : Internal_Entity_Decl;
Cast_Result_4 : Internal_Entity_Enum_Type_Decl;
Let_Result : Internal_Entity_Decl;
Scope_Result_2 : Internal_Entity_Decl;
Cast_Result_5 : Internal_Entity_Decl;
Let_Result_1 : Internal_Entity_Decl;
Scope_Result_3 : Internal_Entity_Decl;
Match_Result : Internal_Entity_Decl;
Scope_Result_4 : Internal_Entity_Decl;
Result_Var_2 : Internal_Entity_Decl;

            procedure Finalizer_Scope_511 is
            begin
                     Dec_Ref (Fld_1);
            end Finalizer_Scope_511;
            procedure Finalizer_Scope_513 is
            begin
                     Dec_Ref (Fld_4);
            end Finalizer_Scope_513;


begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      begin
         
   --# scope-start

         --# expr-start 425 .do Result_Var_2 nodes.lkt:705




Var_Expr := Ent;





if Var_Expr /= No_Entity_Decl then
   



   --# scope-start



--# expr-start 424 Match Match_Result nodes.lkt:707










   if Var_Expr.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Match_Prefix := Var_Expr; 

case Lkt_Decl (Match_Prefix.Node.Kind) is
         when Lkt_Enum_Type_Decl =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_4 := Create_Internal_Entity_Enum_Type_Decl
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Local_E := Cast_Result_4; 
--# bind e Local_E




   --# scope-start



--# expr-start 417 bind Dyn_Var_Bind_Result nodes.lkt:710









   
      Cast_Result_1 := Create_Internal_Entity
        (Node => Ent.Node,
         Info => Ent.Info);


Origin := Cast_Result_1; 



--# expr-start 415 Cast Cast_Result nodes.lkt:712





--# expr-start 414 .get_first Env_Get_Result nodes.lkt:712
--# expr-start 412 Decl.defined_scope Fld_1 nodes.lkt:712



--# expr-start 411 TypeDecl.basic_trait_from_self Fld nodes.lkt:712







   if Local_E.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start TypeDecl.basic_trait_from_self
Fld := Liblktlang.Implementation.Type_Decl_P_Basic_Trait_From_Self (Node => Local_E.Node, E_Info => Local_E.Info);
--# end
--# expr-done 411



   if Fld.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start '[dispatcher]Decl.defined_scope'
Fld_1 := Liblktlang.Implementation.Dispatcher_Decl_P_Defined_Scope (Node => Fld.Node, Origin => Origin, E_Info => Fld.Info);
--# end
--# expr-done 412
--# expr-start 413 .symbol Sym nodes.lkt:713









Fld_2 := Current_Name.Node; 
Sym := Get_Symbol (Fld_2); 
--# expr-done 413


Env_Get_Result := AST_Envs.Get_First (Self => Fld_1, Key => Thin (Sym), Lookup_Kind => To_Lookup_Kind_Type (Flat), Categories => (Nocat => True)); 
--# expr-done 414



   if Env_Get_Result.Node = null
      or else Env_Get_Result.Node.Kind in Lkt_Decl
   then
      
      Cast_Result := Create_Internal_Entity_Decl
        (Node => Env_Get_Result.Node,
         Info => Env_Get_Result.Info);

   else
         Cast_Result := No_Entity_Decl;
   end if;


--# expr-done 415
Var_Expr_1 := Cast_Result;





if Var_Expr_1 /= No_Entity_Decl then
   --# expr-start 416 Decl.subdecl_if_generic Fld_3 nodes.lkt:712







   if Var_Expr_1.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start Decl.subdecl_if_generic
Fld_3 := Liblktlang.Implementation.Decl_P_Subdecl_If_Generic (Node => Var_Expr_1.Node, E_Info => Var_Expr_1.Info);
--# end
--# expr-done 416
   Result_Var := Fld_3;
else
   
   Result_Var := No_Entity_Decl;
end if;



Dyn_Var_Bind_Result := Result_Var; 
--# expr-done 417
Scope_Result := Dyn_Var_Bind_Result;


   --# end
      Finalizer_Scope_511;


Let_Result := Scope_Result; 
--# end
Scope_Result_2 := Let_Result;


   --# end


            Match_Result := Scope_Result_2; 
         when Lkt_Grammar_Rule_Decl .. Lkt_Enum_Class_Decl | Lkt_Struct_Decl .. Lkt_Trait_Decl =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_5 := Create_Internal_Entity_Decl
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Ignored := Cast_Result_5; 
--# bind _ Ignored




   --# scope-start



--# expr-start 423 bind Dyn_Var_Bind_Result_1 nodes.lkt:721









   
      Cast_Result_3 := Create_Internal_Entity
        (Node => Ent.Node,
         Info => Ent.Info);


Origin_1 := Cast_Result_3; 



--# expr-start 421 Cast Cast_Result_2 nodes.lkt:723





--# expr-start 420 .get_first Env_Get_Result_1 nodes.lkt:723
--# expr-start 418 Decl.defined_scope Fld_4 nodes.lkt:723







   if Ent.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start '[dispatcher]Decl.defined_scope'
Fld_4 := Liblktlang.Implementation.Dispatcher_Decl_P_Defined_Scope (Node => Ent.Node, Origin => Origin_1, E_Info => Ent.Info);
--# end
--# expr-done 418
--# expr-start 419 .symbol Sym_1 nodes.lkt:724









Fld_5 := Current_Name.Node; 
Sym_1 := Get_Symbol (Fld_5); 
--# expr-done 419


Env_Get_Result_1 := AST_Envs.Get_First (Self => Fld_4, Key => Thin (Sym_1), Lookup_Kind => To_Lookup_Kind_Type (Flat), Categories => (Nocat => True)); 
--# expr-done 420



   if Env_Get_Result_1.Node = null
      or else Env_Get_Result_1.Node.Kind in Lkt_Decl
   then
      
      Cast_Result_2 := Create_Internal_Entity_Decl
        (Node => Env_Get_Result_1.Node,
         Info => Env_Get_Result_1.Info);

   else
         Cast_Result_2 := No_Entity_Decl;
   end if;


--# expr-done 421
Var_Expr_2 := Cast_Result_2;





if Var_Expr_2 /= No_Entity_Decl then
   --# expr-start 422 Decl.subdecl_if_generic Fld_6 nodes.lkt:723







   if Var_Expr_2.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start Decl.subdecl_if_generic
Fld_6 := Liblktlang.Implementation.Decl_P_Subdecl_If_Generic (Node => Var_Expr_2.Node, E_Info => Var_Expr_2.Info);
--# end
--# expr-done 422
   Result_Var_1 := Fld_6;
else
   
   Result_Var_1 := No_Entity_Decl;
end if;



Dyn_Var_Bind_Result_1 := Result_Var_1; 
--# expr-done 423
Scope_Result_1 := Dyn_Var_Bind_Result_1;


   --# end
      Finalizer_Scope_513;


Let_Result_1 := Scope_Result_1; 
--# end
Scope_Result_3 := Let_Result_1;


   --# end


            Match_Result := Scope_Result_3; 
end case;

--# expr-done 424
Scope_Result_4 := Match_Result;


   --# end


   Result_Var_2 := Scope_Result_4;
else
   
   Result_Var_2 := No_Entity_Decl;
end if;



--# expr-done 425

         Property_Result := Result_Var_2;
         
   --# end


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_511;
                     Finalizer_Scope_513;




            raise;
      end;



   return Property_Result;
end Decl_P_Type_Var_Suffix_Ref;
--# end

   







--# property-start Decl.ref_var_suffix_ref nodes.lkt:735
pragma Warnings (Off, "is not referenced");

function Decl_P_Ref_Var_Suffix_Ref
  
  (Node : Bare_Decl
      ; Type_Var : Internal_Entity_Type_Decl
      ; Current_Name : Internal_Entity_Ref_Id
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity_Decl
is
   Self : Bare_Decl :=
     Bare_Decl (Node);
     Ent : Internal_Entity_Decl :=
       Internal_Entity_Decl'(Node => Self, Info => E_Info);
      --# bind self Ent

   

   --# bind type_var Type_Var
   --# bind current_name Current_Name

   Property_Result : Internal_Entity_Decl;

      
            procedure Finalizer_Scope_516 with Inline_Always;
            procedure Finalizer_Scope_518 with Inline_Always;
            procedure Finalizer_Scope_520 with Inline_Always;
            procedure Finalizer_Scope_524 with Inline_Always;
            procedure Finalizer_Scope_528 with Inline_Always;
            procedure Finalizer_Scope_531 with Inline_Always;

      Var_Expr : Internal_Entity_Decl;
Local_E : Internal_Entity_Enum_Type_Decl;
Origin : Internal_Entity;
Fld : Lexical_Env := Empty_Env;
Fld_1 : Bare_Ref_Id;
Sym : Symbol_Type;
Env_Get_Result : Internal_Entity;
Cast_Result : Internal_Entity_Decl;
Cast_Expr : Internal_Entity_Type_Decl;
Cast_Result_1 : Internal_Entity;
Dyn_Var_Bind_Result : Internal_Entity_Decl;
Scope_Result : Internal_Entity_Decl;
Local_E_1 : Internal_Entity_Enum_Lit_Decl;
Origin_1 : Internal_Entity;
Fld_2 : Lexical_Env := Empty_Env;
Fld_3 : Bare_Ref_Id;
Sym_1 : Symbol_Type;
Env_Get_Result_1 : Internal_Entity;
Cast_Result_2 : Internal_Entity_Decl;
Var_Expr_1 : Internal_Entity_Decl;
Fld_4 : Internal_Entity_Decl;
Result_Var : Internal_Entity_Decl;
Cast_Expr_1 : Internal_Entity_Type_Decl;
Cast_Result_3 : Internal_Entity;
Dyn_Var_Bind_Result_1 : Internal_Entity_Decl;
Scope_Result_1 : Internal_Entity_Decl;
Local_Td : Internal_Entity_Type_Decl;
Origin_2 : Internal_Entity;
Fld_5 : Lexical_Env := Empty_Env;
Fld_6 : Lexical_Env := Empty_Env;
Array_Lit : Lexical_Env_Array_Access;
Group_Env : Lexical_Env := Empty_Env;
Fld_7 : Bare_Ref_Id;
Sym_2 : Symbol_Type;
Env_Get_Result_2 : Internal_Entity_Array_Access;
Is_A : Boolean;
Cast_Result_4 : Internal_Entity_Field_Decl;
Var_Expr_2 : Internal_Entity_Field_Decl;
Fld_8 : Internal_Entity_Full_Decl;
Sym_3 : Symbol_Type;
Fld_9 : Boolean;
Result_Var_1 : Boolean;
If_Result : Boolean;
Find_Result : Internal_Entity;
Cast_Result_5 : Internal_Entity_Decl;
Var_Expr_3 : Internal_Entity_Decl;
Fld_10 : Internal_Entity_Decl;
Result_Var_2 : Internal_Entity_Decl;
Cast_Expr_2 : Internal_Entity_Type_Decl;
Cast_Result_6 : Internal_Entity;
Dyn_Var_Bind_Result_2 : Internal_Entity_Decl;
Scope_Result_2 : Internal_Entity_Decl;
Ignored : Internal_Entity_Match_Val_Decl;
Var_Expr_4 : Internal_Entity_Type_Decl;
Origin_3 : Internal_Entity;
Fld_11 : Lexical_Env := Empty_Env;
Fld_12 : Bare_Ref_Id;
Sym_4 : Symbol_Type;
Env_Get_Result_3 : Internal_Entity;
Cast_Result_7 : Internal_Entity_Decl;
Var_Expr_5 : Internal_Entity_Decl;
Fld_13 : Internal_Entity_Decl;
Result_Var_3 : Internal_Entity_Decl;
Cast_Result_8 : Internal_Entity;
Dyn_Var_Bind_Result_3 : Internal_Entity_Decl;
Scope_Result_3 : Internal_Entity_Decl;
Scope_Result_4 : Internal_Entity_Decl;
Result_Var_4 : Internal_Entity_Decl;
Local_Mvd : Internal_Entity_Lambda_Param_Decl;
Fld_14 : Internal_Entity_Type_Ref;
Var_Expr_6 : Internal_Entity_Type_Ref;
Fld_15 : Internal_Entity_Type_Decl;
Scope_Result_5 : Internal_Entity_Type_Decl;
Result_Var_5 : Internal_Entity_Type_Decl;
Var_Expr_7 : Internal_Entity_Type_Decl;
Origin_4 : Internal_Entity;
Fld_16 : Lexical_Env := Empty_Env;
Fld_17 : Bare_Ref_Id;
Sym_5 : Symbol_Type;
Env_Get_Result_4 : Internal_Entity;
Cast_Result_9 : Internal_Entity_Decl;
Var_Expr_8 : Internal_Entity_Decl;
Fld_18 : Internal_Entity_Decl;
Result_Var_6 : Internal_Entity_Decl;
Cast_Result_10 : Internal_Entity;
Dyn_Var_Bind_Result_4 : Internal_Entity_Decl;
Scope_Result_6 : Internal_Entity_Decl;
Scope_Result_7 : Internal_Entity_Decl;
Result_Var_7 : Internal_Entity_Decl;
Ignored_1 : Internal_Entity_Decl;
Fld_19 : Boolean;
Is_Null : Boolean;
If_Result_1 : Boolean;
Cast_Result_11 : Internal_Entity_Decl;
If_Result_2 : Internal_Entity_Decl;
Var_Expr_9 : Internal_Entity_Decl;
Origin_5 : Internal_Entity;
Fld_20 : Lexical_Env := Empty_Env;
Fld_21 : Bare_Ref_Id;
Sym_6 : Symbol_Type;
Env_Get_Result_5 : Internal_Entity;
Cast_Result_12 : Internal_Entity_Decl;
Var_Expr_10 : Internal_Entity_Decl;
Fld_22 : Internal_Entity_Decl;
Result_Var_8 : Internal_Entity_Decl;
Cast_Result_13 : Internal_Entity;
Dyn_Var_Bind_Result_5 : Internal_Entity_Decl;
Scope_Result_8 : Internal_Entity_Decl;
Scope_Result_9 : Internal_Entity_Decl;
Result_Var_9 : Internal_Entity_Decl;
Match_Prefix : Internal_Entity_Decl;
Cast_Result_14 : Internal_Entity_Enum_Type_Decl;
Let_Result : Internal_Entity_Decl;
Scope_Result_10 : Internal_Entity_Decl;
Cast_Result_15 : Internal_Entity_Enum_Lit_Decl;
Let_Result_1 : Internal_Entity_Decl;
Scope_Result_11 : Internal_Entity_Decl;
Cast_Result_16 : Internal_Entity_Type_Decl;
Let_Result_2 : Internal_Entity_Decl;
Scope_Result_12 : Internal_Entity_Decl;
Cast_Result_17 : Internal_Entity_Match_Val_Decl;
Let_Result_3 : Internal_Entity_Decl;
Scope_Result_13 : Internal_Entity_Decl;
Cast_Result_18 : Internal_Entity_Lambda_Param_Decl;
Let_Result_4 : Internal_Entity_Decl;
Scope_Result_14 : Internal_Entity_Decl;
Cast_Result_19 : Internal_Entity_Decl;
Let_Result_5 : Internal_Entity_Decl;
Scope_Result_15 : Internal_Entity_Decl;
Match_Result : Internal_Entity_Decl;
Scope_Result_16 : Internal_Entity_Decl;
Result_Var_10 : Internal_Entity_Decl;

            procedure Finalizer_Scope_516 is
            begin
                     Dec_Ref (Fld);
            end Finalizer_Scope_516;
            procedure Finalizer_Scope_518 is
            begin
                     Dec_Ref (Fld_2);
            end Finalizer_Scope_518;
            procedure Finalizer_Scope_520 is
            begin
                     Dec_Ref (Fld_5);
                     Dec_Ref (Fld_6);
                     Dec_Ref (Array_Lit);
                     Dec_Ref (Group_Env);
                     Dec_Ref (Env_Get_Result_2);
            end Finalizer_Scope_520;
            procedure Finalizer_Scope_524 is
            begin
                     Dec_Ref (Fld_11);
            end Finalizer_Scope_524;
            procedure Finalizer_Scope_528 is
            begin
                     Dec_Ref (Fld_16);
            end Finalizer_Scope_528;
            procedure Finalizer_Scope_531 is
            begin
                     Dec_Ref (Fld_20);
            end Finalizer_Scope_531;


begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      begin
         
   --# scope-start

         --# expr-start 482 .do Result_Var_10 nodes.lkt:739




Var_Expr := Ent;





if Var_Expr /= No_Entity_Decl then
   



   --# scope-start



--# expr-start 481 Match Match_Result nodes.lkt:741










   if Var_Expr.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Match_Prefix := Var_Expr; 

case Lkt_Decl (Match_Prefix.Node.Kind) is
         when Lkt_Enum_Type_Decl =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_14 := Create_Internal_Entity_Enum_Type_Decl
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Local_E := Cast_Result_14; 
--# bind e Local_E




   --# scope-start



--# expr-start 430 bind Dyn_Var_Bind_Result nodes.lkt:744






Cast_Expr := No_Entity_Type_Decl; 



   
      Cast_Result_1 := Create_Internal_Entity
        (Node => Cast_Expr.Node,
         Info => Cast_Expr.Info);


Origin := Cast_Result_1; 
--# expr-start 429 Cast Cast_Result nodes.lkt:746





--# expr-start 428 .get_first Env_Get_Result nodes.lkt:746
--# expr-start 426 NamedTypeDecl.defined_scope Fld nodes.lkt:746







   if Local_E.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start '[dispatcher]Decl.defined_scope'
Fld := Liblktlang.Implementation.Dispatcher_Decl_P_Defined_Scope (Node => Local_E.Node, Origin => Origin, E_Info => Local_E.Info);
--# end
--# expr-done 426
--# expr-start 427 .symbol Sym nodes.lkt:747









Fld_1 := Current_Name.Node; 
Sym := Get_Symbol (Fld_1); 
--# expr-done 427


Env_Get_Result := AST_Envs.Get_First (Self => Fld, Key => Thin (Sym), Lookup_Kind => To_Lookup_Kind_Type (Flat), Categories => (Nocat => True)); 
--# expr-done 428



   if Env_Get_Result.Node = null
      or else Env_Get_Result.Node.Kind in Lkt_Decl
   then
      
      Cast_Result := Create_Internal_Entity_Decl
        (Node => Env_Get_Result.Node,
         Info => Env_Get_Result.Info);

   else
         Cast_Result := No_Entity_Decl;
   end if;


--# expr-done 429
Dyn_Var_Bind_Result := Cast_Result; 
--# expr-done 430
Scope_Result := Dyn_Var_Bind_Result;


   --# end
      Finalizer_Scope_516;


Let_Result := Scope_Result; 
--# end
Scope_Result_10 := Let_Result;


   --# end


            Match_Result := Scope_Result_10; 
         when Lkt_Enum_Lit_Decl =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_15 := Create_Internal_Entity_Enum_Lit_Decl
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Local_E_1 := Cast_Result_15; 
--# bind e Local_E_1




   --# scope-start



--# expr-start 436 bind Dyn_Var_Bind_Result_1 nodes.lkt:754






Cast_Expr_1 := No_Entity_Type_Decl; 



   
      Cast_Result_3 := Create_Internal_Entity
        (Node => Cast_Expr_1.Node,
         Info => Cast_Expr_1.Info);


Origin_1 := Cast_Result_3; 



--# expr-start 434 Cast Cast_Result_2 nodes.lkt:756





--# expr-start 433 .get_first Env_Get_Result_1 nodes.lkt:756
--# expr-start 431 EnumLitDecl.defined_scope Fld_2 nodes.lkt:756







   if Local_E_1.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start '[dispatcher]Decl.defined_scope'
Fld_2 := Liblktlang.Implementation.Dispatcher_Decl_P_Defined_Scope (Node => Local_E_1.Node, Origin => Origin_1, E_Info => Local_E_1.Info);
--# end
--# expr-done 431
--# expr-start 432 .symbol Sym_1 nodes.lkt:757









Fld_3 := Current_Name.Node; 
Sym_1 := Get_Symbol (Fld_3); 
--# expr-done 432


Env_Get_Result_1 := AST_Envs.Get_First (Self => Fld_2, Key => Thin (Sym_1), Lookup_Kind => To_Lookup_Kind_Type (Flat), Categories => (Nocat => True)); 
--# expr-done 433



   if Env_Get_Result_1.Node = null
      or else Env_Get_Result_1.Node.Kind in Lkt_Decl
   then
      
      Cast_Result_2 := Create_Internal_Entity_Decl
        (Node => Env_Get_Result_1.Node,
         Info => Env_Get_Result_1.Info);

   else
         Cast_Result_2 := No_Entity_Decl;
   end if;


--# expr-done 434
Var_Expr_1 := Cast_Result_2;





if Var_Expr_1 /= No_Entity_Decl then
   --# expr-start 435 Decl.subdecl_if_generic Fld_4 nodes.lkt:756







   if Var_Expr_1.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start Decl.subdecl_if_generic
Fld_4 := Liblktlang.Implementation.Decl_P_Subdecl_If_Generic (Node => Var_Expr_1.Node, E_Info => Var_Expr_1.Info);
--# end
--# expr-done 435
   Result_Var := Fld_4;
else
   
   Result_Var := No_Entity_Decl;
end if;



Dyn_Var_Bind_Result_1 := Result_Var; 
--# expr-done 436
Scope_Result_1 := Dyn_Var_Bind_Result_1;


   --# end
      Finalizer_Scope_518;


Let_Result_1 := Scope_Result_1; 
--# end
Scope_Result_11 := Let_Result_1;


   --# end


            Match_Result := Scope_Result_11; 
         when Lkt_Any_Type_Decl .. Lkt_Enum_Class_Decl | Lkt_Struct_Decl .. Lkt_Trait_Decl =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_16 := Create_Internal_Entity_Type_Decl
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Local_Td := Cast_Result_16; 
--# bind td Local_Td




   --# scope-start



--# expr-start 452 bind Dyn_Var_Bind_Result_2 nodes.lkt:765






Cast_Expr_2 := No_Entity_Type_Decl; 



   
      Cast_Result_6 := Create_Internal_Entity
        (Node => Cast_Expr_2.Node,
         Info => Cast_Expr_2.Info);


Origin_2 := Cast_Result_6; 



--# expr-start 450 Cast Cast_Result_5 nodes.lkt:767





--# expr-start 449 .find Find_Result nodes.lkt:767








   

   --# expr-start 442 .get Env_Get_Result_2 nodes.lkt:767
--# expr-start 440 .env_group Group_Env nodes.lkt:767
--# expr-start 439 ArrayLiteral Array_Lit nodes.lkt:767
--# expr-start 437 Decl.defined_scope Fld_5 nodes.lkt:767







   if Local_Td.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start '[dispatcher]Decl.defined_scope'
Fld_5 := Liblktlang.Implementation.Dispatcher_Decl_P_Defined_Scope (Node => Local_Td.Node, Origin => Origin_2, E_Info => Local_Td.Info);
--# end
--# expr-done 437
--# expr-start 438 TypeDecl.node_builder_scope Fld_6 nodes.lkt:767







   if Local_Td.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start TypeDecl.node_builder_scope
Fld_6 := Liblktlang.Implementation.Type_Decl_P_Node_Builder_Scope (Node => Local_Td.Node, E_Info => Local_Td.Info);
--# end
--# expr-done 438
Array_Lit := Create_Lexical_Env_Array (Internal_Lexical_Env_Array'(1 => Fld_5, 2 => Fld_6)); 
--# expr-done 439

Group_Env := Group (Array_Lit, No_Metadata); 
--# expr-done 440
--# expr-start 441 .symbol Sym_2 nodes.lkt:769









Fld_7 := Current_Name.Node; 
Sym_2 := Get_Symbol (Fld_7); 
--# expr-done 441


Env_Get_Result_2 := Construct_Entity_Array (AST_Envs.Get (Self => Group_Env, Key => Thin (Sym_2), Lookup_Kind => To_Lookup_Kind_Type (Flat), Categories => (Nocat => True))); 
--# expr-done 442

   
      Find_Result := No_Entity;
   

   

      

      declare
         
         Collection : constant Internal_Entity_Array_Access := Env_Get_Result_2;
      begin
         for Item of
               Collection.Items
         loop

            
   --# scope-start


               --# bind d Item

            
      --# expr-start 448 BooleanOr If_Result nodes.lkt:772



--# expr-start 443 IsA Is_A nodes.lkt:772

Is_A := Item.Node /= null 
and then Item.Node.Kind in Lkt_Type_Decl | Lkt_Fun_Decl_Range; 
--# expr-done 443
if Is_A then
   
   If_Result := True;
else
   


--# expr-start 444 Cast Cast_Result_4 nodes.lkt:773









   if Item.Node = null
      or else Item.Node.Kind in Lkt_Field_Decl_Range
   then
      
      Cast_Result_4 := Create_Internal_Entity_Field_Decl
        (Node => Item.Node,
         Info => Item.Info);

   else
         Cast_Result_4 := No_Entity_Field_Decl;
   end if;


--# expr-done 444
Var_Expr_2 := Cast_Result_4;





if Var_Expr_2 /= No_Entity_Field_Decl then
   --# expr-start 447 FullDecl.has_annotation Fld_9 nodes.lkt:773



--# expr-start 445 Decl.full_decl Fld_8 nodes.lkt:773







   if Var_Expr_2.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start Decl.full_decl
Fld_8 := Liblktlang.Implementation.Decl_P_Full_Decl (Node => Var_Expr_2.Node, E_Info => Var_Expr_2.Info);
--# end
--# expr-done 445



   if Fld_8.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# expr-start 446 SymbolLiteral Sym_3 nodes.lkt:774
Sym_3 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Lazy); 
--# expr-done 446
--# property-call-start FullDecl.has_annotation
Fld_9 := Liblktlang.Implementation.Full_Decl_P_Has_Annotation (Node => Fld_8.Node, Name => Sym_3);
--# end
--# expr-done 447
   Result_Var_1 := Fld_9;
else
   
   Result_Var_1 := False;
end if;



   If_Result := Result_Var_1;
end if;



--# expr-done 448
      if If_Result then
         Find_Result := Item; 
      end if;
   

            
   --# end


               exit when If_Result;

         end loop;
      end;
   

   
   



--# expr-done 449



   if Find_Result.Node = null
      or else Find_Result.Node.Kind in Lkt_Decl
   then
      
      Cast_Result_5 := Create_Internal_Entity_Decl
        (Node => Find_Result.Node,
         Info => Find_Result.Info);

   else
         Cast_Result_5 := No_Entity_Decl;
   end if;


--# expr-done 450
Var_Expr_3 := Cast_Result_5;





if Var_Expr_3 /= No_Entity_Decl then
   --# expr-start 451 Decl.subdecl_if_generic Fld_10 nodes.lkt:767







   if Var_Expr_3.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start Decl.subdecl_if_generic
Fld_10 := Liblktlang.Implementation.Decl_P_Subdecl_If_Generic (Node => Var_Expr_3.Node, E_Info => Var_Expr_3.Info);
--# end
--# expr-done 451
   Result_Var_2 := Fld_10;
else
   
   Result_Var_2 := No_Entity_Decl;
end if;



Dyn_Var_Bind_Result_2 := Result_Var_2; 
--# expr-done 452
Scope_Result_2 := Dyn_Var_Bind_Result_2;


   --# end
      Finalizer_Scope_520;


Let_Result_2 := Scope_Result_2; 
--# end
Scope_Result_12 := Let_Result_2;


   --# end


            Match_Result := Scope_Result_12; 
         when Lkt_Match_Val_Decl =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_17 := Create_Internal_Entity_Match_Val_Decl
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Ignored := Cast_Result_17; 
--# bind _ Ignored
--# expr-start 459 .do Result_Var_4 nodes.lkt:781




Var_Expr_4 := Type_Var;





if Var_Expr_4 /= No_Entity_Type_Decl then
   



   --# scope-start







   --# scope-start



--# expr-start 458 bind Dyn_Var_Bind_Result_3 nodes.lkt:783









   
      Cast_Result_8 := Create_Internal_Entity
        (Node => Var_Expr_4.Node,
         Info => Var_Expr_4.Info);


Origin_3 := Cast_Result_8; 



--# expr-start 456 Cast Cast_Result_7 nodes.lkt:785





--# expr-start 455 .get_first Env_Get_Result_3 nodes.lkt:785
--# expr-start 453 Decl.defined_scope Fld_11 nodes.lkt:785







   if Var_Expr_4.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start '[dispatcher]Decl.defined_scope'
Fld_11 := Liblktlang.Implementation.Dispatcher_Decl_P_Defined_Scope (Node => Var_Expr_4.Node, Origin => Origin_3, E_Info => Var_Expr_4.Info);
--# end
--# expr-done 453
--# expr-start 454 .symbol Sym_4 nodes.lkt:786









Fld_12 := Current_Name.Node; 
Sym_4 := Get_Symbol (Fld_12); 
--# expr-done 454


Env_Get_Result_3 := AST_Envs.Get_First (Self => Fld_11, Key => Thin (Sym_4), Lookup_Kind => To_Lookup_Kind_Type (Flat), Categories => (Nocat => True)); 
--# expr-done 455



   if Env_Get_Result_3.Node = null
      or else Env_Get_Result_3.Node.Kind in Lkt_Decl
   then
      
      Cast_Result_7 := Create_Internal_Entity_Decl
        (Node => Env_Get_Result_3.Node,
         Info => Env_Get_Result_3.Info);

   else
         Cast_Result_7 := No_Entity_Decl;
   end if;


--# expr-done 456
Var_Expr_5 := Cast_Result_7;





if Var_Expr_5 /= No_Entity_Decl then
   --# expr-start 457 Decl.subdecl_if_generic Fld_13 nodes.lkt:785







   if Var_Expr_5.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start Decl.subdecl_if_generic
Fld_13 := Liblktlang.Implementation.Decl_P_Subdecl_If_Generic (Node => Var_Expr_5.Node, E_Info => Var_Expr_5.Info);
--# end
--# expr-done 457
   Result_Var_3 := Fld_13;
else
   
   Result_Var_3 := No_Entity_Decl;
end if;



Dyn_Var_Bind_Result_3 := Result_Var_3; 
--# expr-done 458
Scope_Result_3 := Dyn_Var_Bind_Result_3;


   --# end
      Finalizer_Scope_524;


Scope_Result_4 := Scope_Result_3;


   --# end


   Result_Var_4 := Scope_Result_4;
else
   
   Result_Var_4 := No_Entity_Decl;
end if;



--# expr-done 459
Let_Result_3 := Result_Var_4; 
--# end
Scope_Result_13 := Let_Result_3;


   --# end


            Match_Result := Scope_Result_13; 
         when Lkt_Lambda_Param_Decl =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_18 := Create_Internal_Entity_Lambda_Param_Decl
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Local_Mvd := Cast_Result_18; 
--# bind mvd Local_Mvd
--# expr-start 469 .do Result_Var_7 nodes.lkt:798



--# expr-start 462 .do Result_Var_5 nodes.lkt:798



--# expr-start 460 LambdaParamDecl.decl_type Fld_14 nodes.lkt:798







   if Local_Mvd.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_14 := Create_Internal_Entity_Type_Ref (Node => Local_Mvd.Node.Lambda_Param_Decl_F_Decl_Type, Info => Local_Mvd.Info);
--# expr-done 460
Var_Expr_6 := Fld_14;





if Var_Expr_6 /= No_Entity_Type_Ref then
   



   --# scope-start



--# expr-start 461 Decl.get_type Fld_15 nodes.lkt:799







   if Local_Mvd.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start Decl.get_type
Fld_15 := Liblktlang.Implementation.Decl_P_Get_Type (Node => Local_Mvd.Node, E_Info => Local_Mvd.Info);
--# end
--# expr-done 461
Scope_Result_5 := Fld_15;


   --# end


   Result_Var_5 := Scope_Result_5;
else
   
   Result_Var_5 := Type_Var;
end if;



--# expr-done 462
Var_Expr_7 := Result_Var_5;





if Var_Expr_7 /= No_Entity_Type_Decl then
   



   --# scope-start







   --# scope-start



--# expr-start 468 bind Dyn_Var_Bind_Result_4 nodes.lkt:804









   
      Cast_Result_10 := Create_Internal_Entity
        (Node => Var_Expr_7.Node,
         Info => Var_Expr_7.Info);


Origin_4 := Cast_Result_10; 



--# expr-start 466 Cast Cast_Result_9 nodes.lkt:806





--# expr-start 465 .get_first Env_Get_Result_4 nodes.lkt:806
--# expr-start 463 Decl.defined_scope Fld_16 nodes.lkt:806







   if Var_Expr_7.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start '[dispatcher]Decl.defined_scope'
Fld_16 := Liblktlang.Implementation.Dispatcher_Decl_P_Defined_Scope (Node => Var_Expr_7.Node, Origin => Origin_4, E_Info => Var_Expr_7.Info);
--# end
--# expr-done 463
--# expr-start 464 .symbol Sym_5 nodes.lkt:807









Fld_17 := Current_Name.Node; 
Sym_5 := Get_Symbol (Fld_17); 
--# expr-done 464


Env_Get_Result_4 := AST_Envs.Get_First (Self => Fld_16, Key => Thin (Sym_5), Lookup_Kind => To_Lookup_Kind_Type (Flat), Categories => (Nocat => True)); 
--# expr-done 465



   if Env_Get_Result_4.Node = null
      or else Env_Get_Result_4.Node.Kind in Lkt_Decl
   then
      
      Cast_Result_9 := Create_Internal_Entity_Decl
        (Node => Env_Get_Result_4.Node,
         Info => Env_Get_Result_4.Info);

   else
         Cast_Result_9 := No_Entity_Decl;
   end if;


--# expr-done 466
Var_Expr_8 := Cast_Result_9;





if Var_Expr_8 /= No_Entity_Decl then
   --# expr-start 467 Decl.subdecl_if_generic Fld_18 nodes.lkt:806







   if Var_Expr_8.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start Decl.subdecl_if_generic
Fld_18 := Liblktlang.Implementation.Decl_P_Subdecl_If_Generic (Node => Var_Expr_8.Node, E_Info => Var_Expr_8.Info);
--# end
--# expr-done 467
   Result_Var_6 := Fld_18;
else
   
   Result_Var_6 := No_Entity_Decl;
end if;



Dyn_Var_Bind_Result_4 := Result_Var_6; 
--# expr-done 468
Scope_Result_6 := Dyn_Var_Bind_Result_4;


   --# end
      Finalizer_Scope_528;


Scope_Result_7 := Scope_Result_6;


   --# end


   Result_Var_7 := Scope_Result_7;
else
   
   Result_Var_7 := No_Entity_Decl;
end if;



--# expr-done 469
Let_Result_4 := Result_Var_7; 
--# end
Scope_Result_14 := Let_Result_4;


   --# end


            Match_Result := Scope_Result_14; 
         when Lkt_Grammar_Rule_Decl .. Lkt_Binding_Val_Decl | Lkt_Field_Decl .. Lkt_Fun_Param_Decl | Lkt_Dyn_Var_Decl | Lkt_Val_Decl .. Lkt_Synth_Param_Decl =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_19 := Create_Internal_Entity_Decl
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Ignored_1 := Cast_Result_19; 
--# bind _ Ignored_1
--# expr-start 480 .do Result_Var_9 nodes.lkt:815



--# expr-start 473 If If_Result_2 nodes.lkt:816



--# expr-start 472 BooleanAnd If_Result_1 nodes.lkt:816



--# expr-start 470 LktNode.xref_entry_point Fld_19 nodes.lkt:816







   if Ent.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start '[dispatcher]LktNode.xref_entry_point'
Fld_19 := Liblktlang.Implementation.Dispatcher_Lkt_Node_P_Xref_Entry_Point (Node => Ent.Node, E_Info => Ent.Info);
--# end
--# expr-done 470
if Fld_19 then
   --# expr-start 471 .is_null Is_Null nodes.lkt:816

Is_Null := Type_Var.Node = null; 
--# expr-done 471
   If_Result_1 := Is_Null;
else
   
   If_Result_1 := False;
end if;



--# expr-done 472
if If_Result_1 then
   
   If_Result_2 := Ent;
else
   








   
      Cast_Result_11 := Create_Internal_Entity_Decl
        (Node => Type_Var.Node,
         Info => Type_Var.Info);


   If_Result_2 := Cast_Result_11;
end if;



--# expr-done 473
Var_Expr_9 := If_Result_2;





if Var_Expr_9 /= No_Entity_Decl then
   



   --# scope-start







   --# scope-start



--# expr-start 479 bind Dyn_Var_Bind_Result_5 nodes.lkt:822









   
      Cast_Result_13 := Create_Internal_Entity
        (Node => Var_Expr_9.Node,
         Info => Var_Expr_9.Info);


Origin_5 := Cast_Result_13; 



--# expr-start 477 Cast Cast_Result_12 nodes.lkt:824





--# expr-start 476 .get_first Env_Get_Result_5 nodes.lkt:824
--# expr-start 474 Decl.defined_scope Fld_20 nodes.lkt:824







   if Var_Expr_9.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start '[dispatcher]Decl.defined_scope'
Fld_20 := Liblktlang.Implementation.Dispatcher_Decl_P_Defined_Scope (Node => Var_Expr_9.Node, Origin => Origin_5, E_Info => Var_Expr_9.Info);
--# end
--# expr-done 474
--# expr-start 475 .symbol Sym_6 nodes.lkt:825









Fld_21 := Current_Name.Node; 
Sym_6 := Get_Symbol (Fld_21); 
--# expr-done 475


Env_Get_Result_5 := AST_Envs.Get_First (Self => Fld_20, Key => Thin (Sym_6), Lookup_Kind => To_Lookup_Kind_Type (Flat), Categories => (Nocat => True)); 
--# expr-done 476



   if Env_Get_Result_5.Node = null
      or else Env_Get_Result_5.Node.Kind in Lkt_Decl
   then
      
      Cast_Result_12 := Create_Internal_Entity_Decl
        (Node => Env_Get_Result_5.Node,
         Info => Env_Get_Result_5.Info);

   else
         Cast_Result_12 := No_Entity_Decl;
   end if;


--# expr-done 477
Var_Expr_10 := Cast_Result_12;





if Var_Expr_10 /= No_Entity_Decl then
   --# expr-start 478 Decl.subdecl_if_generic Fld_22 nodes.lkt:824







   if Var_Expr_10.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start Decl.subdecl_if_generic
Fld_22 := Liblktlang.Implementation.Decl_P_Subdecl_If_Generic (Node => Var_Expr_10.Node, E_Info => Var_Expr_10.Info);
--# end
--# expr-done 478
   Result_Var_8 := Fld_22;
else
   
   Result_Var_8 := No_Entity_Decl;
end if;



Dyn_Var_Bind_Result_5 := Result_Var_8; 
--# expr-done 479
Scope_Result_8 := Dyn_Var_Bind_Result_5;


   --# end
      Finalizer_Scope_531;


Scope_Result_9 := Scope_Result_8;


   --# end


   Result_Var_9 := Scope_Result_9;
else
   
   Result_Var_9 := No_Entity_Decl;
end if;



--# expr-done 480
Let_Result_5 := Result_Var_9; 
--# end
Scope_Result_15 := Let_Result_5;


   --# end


            Match_Result := Scope_Result_15; 
end case;

--# expr-done 481
Scope_Result_16 := Match_Result;


   --# end


   Result_Var_10 := Scope_Result_16;
else
   
   Result_Var_10 := No_Entity_Decl;
end if;



--# expr-done 482

         Property_Result := Result_Var_10;
         
   --# end


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_516;
                     Finalizer_Scope_518;
                     Finalizer_Scope_520;
                     Finalizer_Scope_524;
                     Finalizer_Scope_528;
                     Finalizer_Scope_531;




            raise;
      end;



   return Property_Result;
end Decl_P_Ref_Var_Suffix_Ref;
--# end

   







--# property-start Decl.get_params nodes.lkt:838
pragma Warnings (Off, "is not referenced");

function Decl_P_Get_Params
  
  (Node : Bare_Decl
      ; Is_Logic : Boolean
         := False
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Resolved_Param_Array_Access
is
   Self : Bare_Decl :=
     Bare_Decl (Node);
     Ent : Internal_Entity_Decl :=
       Internal_Entity_Decl'(Node => Self, Info => E_Info);
      --# bind self Ent

   

   --# bind is_logic Is_Logic

   Property_Result : Internal_Resolved_Param_Array_Access;

      
            procedure Finalizer_Scope_101 with Inline_Always;
            procedure Finalizer_Scope_532 with Inline_Always;
            procedure Finalizer_Scope_533 with Inline_Always;
            procedure Finalizer_Scope_536 with Inline_Always;
            procedure Finalizer_Scope_537 with Inline_Always;
            procedure Finalizer_Scope_539 with Inline_Always;
            procedure Finalizer_Scope_543 with Inline_Always;

      Var_Expr : Internal_Entity_Decl;
Local_Fd : Internal_Entity_Fun_Decl;
Sym : Symbol_Type;
Fld : Internal_Entity_Named_Type_Decl;
New_Node : Bare_Synth_Param_Decl;
As_Entity : Internal_Entity_Synth_Param_Decl;
Is_Null : Boolean;
If_Result : Internal_Entity_Synth_Param_Decl;
Cast_Result : Internal_Entity_Type_Decl;
Cast_Result_1 : Internal_Entity_Decl;
New_Struct : Internal_Resolved_Param;
Array_Lit : Internal_Resolved_Param_Array_Access;
Fld_1 : Internal_Entity_Fun_Param_Decl_List;
Item : Internal_Entity_Fun_Param_Decl;
Fld_2 : Internal_Resolved_Param;
Fld_3 : Bare_Fun_Param_Decl_List;
Fld_4 : Internal_Entity_Info;
Bare_Item : Bare_Fun_Param_Decl;
As_Entity_1 : Internal_Entity_Fun_Param_Decl;
Is_Null_1 : Boolean;
If_Result_1 : Internal_Entity_Fun_Param_Decl;
Map_Result : Internal_Resolved_Param_Array_Access;
Concat_Result : Internal_Resolved_Param_Array_Access;
Fld_5 : Boolean;
Fld_6 : Internal_Entity_Fun_Param_Decl_List;
Fld_7 : Bare_Fun_Param_Decl_List;
Fld_8 : Internal_Entity_Info;
Get_Result : Bare_Fun_Param_Decl;
As_Entity_2 : Internal_Entity_Fun_Param_Decl;
Is_Null_2 : Boolean;
If_Result_2 : Internal_Entity_Fun_Param_Decl;
Fld_9 : Symbol_Type;
Fld_10 : Internal_Entity_Named_Type_Decl;
Fld_11 : Internal_Entity_Type_Decl;
Fld_12 : Internal_Entity_Fun_Param_Decl_List;
Fld_13 : Bare_Fun_Param_Decl_List;
Fld_14 : Internal_Entity_Info;
Get_Result_1 : Bare_Fun_Param_Decl;
As_Entity_3 : Internal_Entity_Fun_Param_Decl;
Is_Null_3 : Boolean;
If_Result_3 : Internal_Entity_Fun_Param_Decl;
Cast_Expr : Internal_Entity_Fun_Param_Decl;
Cast_Result_2 : Internal_Entity_Decl;
New_Struct_1 : Internal_Resolved_Param;
Array_Lit_1 : Internal_Resolved_Param_Array_Access;
If_Result_4 : Internal_Resolved_Param_Array_Access;
Not_Val : Boolean;
Fld_15 : Internal_Entity_Fun_Param_Decl_List;
Item_1 : Internal_Entity_Fun_Param_Decl;
Fld_16 : Internal_Resolved_Param;
Fld_17 : Bare_Fun_Param_Decl_List;
Fld_18 : Internal_Entity_Info;
Bare_Item_1 : Bare_Fun_Param_Decl;
As_Entity_4 : Internal_Entity_Fun_Param_Decl;
Is_Null_4 : Boolean;
If_Result_5 : Internal_Entity_Fun_Param_Decl;
Map_Result_1 : Internal_Resolved_Param_Array_Access;
If_Result_6 : Internal_Resolved_Param_Array_Access;
Local_Lfd : Internal_Entity_Synth_Fun_Decl;
Fld_19 : Internal_Resolved_Param_Array_Access;
Local_Cd : Internal_Entity_Class_Decl;
Fld_20 : Internal_Entity_Field_Decl_Array_Access;
Fld_21 : Internal_Resolved_Param;
Map_Result_2 : Internal_Resolved_Param_Array_Access;
Local_Td : Internal_Entity_Named_Type_Decl;
Fld_22 : Internal_Entity_Decl_Block;
Item_3 : Internal_Entity_Full_Decl;
Fld_23 : Internal_Entity_Decl;
Is_A : Boolean;
Fld_24 : Bare_Decl_Block;
Fld_25 : Internal_Entity_Info;
Bare_Item_3 : Bare_Full_Decl;
As_Entity_5 : Internal_Entity_Full_Decl;
Is_Null_5 : Boolean;
If_Result_7 : Internal_Entity_Full_Decl;
Map_Result_3 : Internal_Entity_Full_Decl_Array_Access;
Fld_26 : Internal_Entity_Decl;
Cast_Result_3 : Internal_Entity_Field_Decl;
Fld_27 : Internal_Resolved_Param;
Map_Result_4 : Internal_Resolved_Param_Array_Access;
Ignored : Internal_Entity_Decl;
Match_Prefix : Internal_Entity_Decl;
Cast_Result_4 : Internal_Entity_Fun_Decl;
Let_Result : Internal_Resolved_Param_Array_Access;
Scope_Result : Internal_Resolved_Param_Array_Access;
Cast_Result_5 : Internal_Entity_Synth_Fun_Decl;
Let_Result_1 : Internal_Resolved_Param_Array_Access;
Scope_Result_1 : Internal_Resolved_Param_Array_Access;
Cast_Result_6 : Internal_Entity_Class_Decl;
Let_Result_2 : Internal_Resolved_Param_Array_Access;
Scope_Result_2 : Internal_Resolved_Param_Array_Access;
Cast_Result_7 : Internal_Entity_Named_Type_Decl;
Let_Result_3 : Internal_Resolved_Param_Array_Access;
Scope_Result_3 : Internal_Resolved_Param_Array_Access;
Cast_Result_8 : Internal_Entity_Decl;
Let_Result_4 : Internal_Resolved_Param_Array_Access;
Scope_Result_4 : Internal_Resolved_Param_Array_Access;
Match_Result : Internal_Resolved_Param_Array_Access;
Scope_Result_5 : Internal_Resolved_Param_Array_Access;
Result_Var : Internal_Resolved_Param_Array_Access;

            procedure Finalizer_Scope_101 is
            begin
                     Dec_Ref (Scope_Result_5);
                     Dec_Ref (Result_Var);
            end Finalizer_Scope_101;
            procedure Finalizer_Scope_532 is
            begin
                     Dec_Ref (Scope_Result);
                     Dec_Ref (Scope_Result_1);
                     Dec_Ref (Scope_Result_2);
                     Dec_Ref (Scope_Result_3);
                     Dec_Ref (Scope_Result_4);
                     Dec_Ref (Match_Result);
            end Finalizer_Scope_532;
            procedure Finalizer_Scope_533 is
            begin
                     Dec_Ref (Array_Lit);
                     Dec_Ref (Map_Result);
                     Dec_Ref (Concat_Result);
                     Dec_Ref (Array_Lit_1);
                     Dec_Ref (If_Result_4);
                     Dec_Ref (Map_Result_1);
                     Dec_Ref (If_Result_6);
                     Dec_Ref (Let_Result);
            end Finalizer_Scope_533;
            procedure Finalizer_Scope_536 is
            begin
                     Dec_Ref (Fld_19);
                     Dec_Ref (Let_Result_1);
            end Finalizer_Scope_536;
            procedure Finalizer_Scope_537 is
            begin
                     Dec_Ref (Fld_20);
                     Dec_Ref (Map_Result_2);
                     Dec_Ref (Let_Result_2);
            end Finalizer_Scope_537;
            procedure Finalizer_Scope_539 is
            begin
                     Dec_Ref (Map_Result_3);
                     Dec_Ref (Map_Result_4);
                     Dec_Ref (Let_Result_3);
            end Finalizer_Scope_539;
            procedure Finalizer_Scope_543 is
            begin
                     Dec_Ref (Let_Result_4);
            end Finalizer_Scope_543;

      
      Mmz_Handle : Memoization_Handle;
      Mmz_Val    : Mmz_Value;

      Mmz_Stored : Boolean;
      --  Whether the memoization couple was actually stored. Used to determine
      --  whether to inc-ref the memoized value.

      function Create_Mmz_Key return Mmz_Key;
      --  Create a memoization key for this property call and return it

      --------------------
      -- Create_Mmz_Key --
      --------------------

      function Create_Mmz_Key return Mmz_Key is
      begin
         return Mmz_K : Mmz_Key :=
           (Property => Mmz_Bare_Decl_Decl_P_Get_Params,
            Items    => new Mmz_Key_Array (1 ..  3))
         do
            Mmz_K.Items (1) := (Kind => Mmz_Bare_Decl,
                                As_Bare_Decl => Self);
               Mmz_K.Items (2) := (Kind => Mmz_Boolean,
                                      As_Boolean => Is_Logic);
               Mmz_K.Items (3) :=
                 (Kind => Mmz_Internal_Entity_Info,
                  As_Internal_Entity_Info => E_Info);
         end return;
      end Create_Mmz_Key;

begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;

      if Self /= null then

         if not Self.Unit.Context.In_Populate_Lexical_Env
            and then Find_Memoized_Value
                       (Self.Unit, Mmz_Handle, Mmz_Val, Create_Mmz_Key'Access)
         then
            --# memoization-lookup

            if Mmz_Val.Kind = Mmz_Evaluating then
               --# memoization-return
               Raise_Property_Exception
                 (Self,
                  Property_Error'Identity,
                  "Infinite recursion detected");

            elsif Mmz_Val.Kind = Mmz_Error then
               --# memoization-return
               Reraise_Memoized_Error (Mmz_Val);

            else
               Property_Result := Mmz_Val.As_Internal_Resolved_Param_Array_Access;
                  Inc_Ref (Property_Result);

               --# memoization-return
               return Property_Result;
            end if;
            --# end
         end if;
      end if;

      begin
         
   --# scope-start

         --# expr-start 520 .do Result_Var nodes.lkt:839




Var_Expr := Ent;





if Var_Expr /= No_Entity_Decl then
   



   --# scope-start



--# expr-start 519 Match Match_Result nodes.lkt:841










   if Var_Expr.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Match_Prefix := Var_Expr; 

case Lkt_Decl (Match_Prefix.Node.Kind) is
         when Lkt_Fun_Decl =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_4 := Create_Internal_Entity_Fun_Decl
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Local_Fd := Cast_Result_4; 
--# bind fd Local_Fd
--# expr-start 506 If If_Result_6 nodes.lkt:843



--# expr-start 483 Not Not_Val nodes.lkt:843

Not_Val := not (Is_Logic); 
--# expr-done 483
if Not_Val then
   --# expr-start 486 .map Map_Result_1 nodes.lkt:844








   

   --# expr-start 484 FunDecl.params Fld_15 nodes.lkt:844







   if Local_Fd.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_15 := Create_Internal_Entity_Fun_Param_Decl_List (Node => Local_Fd.Node.Fun_Decl_F_Params, Info => Local_Fd.Info);
--# expr-done 484









Fld_17 := Fld_15.Node;

   
      declare
         Map_Result_1_Vec : Internal_Resolved_Param_Vectors.Vector;
      begin
   

   

      if Fld_17 = null then
         
      Map_Result_1 := Create_Internal_Resolved_Param_Array (0);
   
      else
         

      declare
         
         Collection : constant Bare_Fun_Param_Decl_List := Fld_17;
      begin
         for Untyped_Item_1 of
               Collection.Nodes (1 .. Children_Count (Collection))
         loop
                  
                  Bare_Item_1 := Untyped_Item_1; 
                  



Is_Null_4 := Bare_Item_1 = null; 
if Is_Null_4 then
   
   If_Result_5 := No_Entity_Fun_Param_Decl;
else
   








Fld_18 := Fld_15.Info;

As_Entity_4 := (Info => Fld_18, Node => Bare_Item_1); 
   If_Result_5 := As_Entity_4;
end if;



                  Item_1 := If_Result_5; 

            
   --# scope-start


               --# bind p Item_1

            
         
         
      --# expr-start 485 ComponentDecl.to_generic_param Fld_16 nodes.lkt:844







   if Item_1.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start ComponentDecl.to_generic_param
Fld_16 := Liblktlang.Implementation.Component_Decl_P_To_Generic_Param (Node => Item_1.Node, E_Info => Item_1.Info);
--# end
--# expr-done 485
      

         declare
            Item_To_Append : constant Internal_Resolved_Param := Fld_16;
         begin
            Internal_Resolved_Param_Vectors.Append (Map_Result_1_Vec, Item_To_Append);
         end;
   
   
   

            
   --# end



         end loop;
      end;
   
      end if;

   

         Map_Result_1 := Create_Internal_Resolved_Param_Array
           (Items_Count => Natural (Internal_Resolved_Param_Vectors.Length (Map_Result_1_Vec)));
         for I in Map_Result_1.Items'Range loop
            Map_Result_1.Items (I) := Internal_Resolved_Param_Vectors.Get
              (Map_Result_1_Vec,
               I + Internal_Resolved_Param_Vectors.Index_Type'First - Map_Result_1.Items'First);
         end loop;
         Internal_Resolved_Param_Vectors.Destroy (Map_Result_1_Vec);

      end;
   



--# expr-done 486
   If_Result_6 := Map_Result_1;
else
   --# expr-start 505 If If_Result_4 nodes.lkt:843



--# expr-start 487 FunDecl.is_dynamic_combiner Fld_5 nodes.lkt:845







   if Local_Fd.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start FunDecl.is_dynamic_combiner
Fld_5 := Liblktlang.Implementation.Fun_Decl_P_Is_Dynamic_Combiner (Node => Local_Fd.Node, E_Info => Local_Fd.Info);
--# end
--# expr-done 487
if Fld_5 then
   --# expr-start 494 ArrayLiteral Array_Lit_1 nodes.lkt:847
--# expr-start 493 'New[ResolvedParam]' New_Struct_1 nodes.lkt:848






--# expr-start 488 FunDecl.params Fld_12 nodes.lkt:853







   if Local_Fd.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_12 := Create_Internal_Entity_Fun_Param_Decl_List (Node => Local_Fd.Node.Fun_Decl_F_Params, Info => Local_Fd.Info);
--# expr-done 488
















Fld_13 := Fld_12.Node;



   if Fld_13 = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;




Get_Result_1 := Get (Self, Fld_13, 0, True); 
Is_Null_3 := Get_Result_1 = null; 
if Is_Null_3 then
   
   If_Result_3 := No_Entity_Fun_Param_Decl;
else
   








Fld_14 := Fld_12.Info;

As_Entity_3 := (Info => Fld_14, Node => Get_Result_1); 
   If_Result_3 := As_Entity_3;
end if;



Cast_Expr := If_Result_3; 



   
      Cast_Result_2 := Create_Internal_Entity_Decl
        (Node => Cast_Expr.Node,
         Info => Cast_Expr.Info);



--# expr-start 490 Decl.name Fld_9 nodes.lkt:849



--# expr-start 489 FunDecl.params Fld_6 nodes.lkt:849







   if Local_Fd.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_6 := Create_Internal_Entity_Fun_Param_Decl_List (Node => Local_Fd.Node.Fun_Decl_F_Params, Info => Local_Fd.Info);
--# expr-done 489
















Fld_7 := Fld_6.Node;



   if Fld_7 = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;




Get_Result := Get (Self, Fld_7, 0, True); 
Is_Null_2 := Get_Result = null; 
if Is_Null_2 then
   
   If_Result_2 := No_Entity_Fun_Param_Decl;
else
   








Fld_8 := Fld_6.Info;

As_Entity_2 := (Info => Fld_8, Node => Get_Result); 
   If_Result_2 := As_Entity_2;
end if;






   if If_Result_2.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start '[dispatcher]Decl.name'
Fld_9 := Liblktlang.Implementation.Dispatcher_Decl_P_Name (Node => If_Result_2.Node);
--# end
--# expr-done 490
--# expr-start 492 TypeDecl.make_array_type Fld_11 nodes.lkt:850



--# expr-start 491 LktNode.logicvar_type Fld_10 nodes.lkt:850







   if Ent.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start LktNode.logicvar_type
Fld_10 := Liblktlang.Implementation.Lkt_Node_P_Logicvar_Type (Node => Ent.Node);
--# end
--# expr-done 491



   if Fld_10.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start TypeDecl.make_array_type
Fld_11 := Liblktlang.Implementation.Type_Decl_P_Make_Array_Type (Node => Fld_10.Node, E_Info => Fld_10.Info);
--# end
--# expr-done 492
New_Struct_1 := (Accept_Logical_Var => False, Decl => Cast_Result_2, Has_Default_Value => False, Name => Fld_9, Param_Type => Fld_11); 
--# expr-done 493
Array_Lit_1 := Create_Internal_Resolved_Param_Array (Internal_Internal_Resolved_Param_Array'(1 => New_Struct_1)); 
--# expr-done 494
   If_Result_4 := Array_Lit_1;
else
   --# expr-start 504 ArrayConcat Concat_Result nodes.lkt:857
--# expr-start 500 ArrayLiteral Array_Lit nodes.lkt:857
--# expr-start 499 'New[ResolvedParam]' New_Struct nodes.lkt:858






--# expr-start 496 .as_entity If_Result nodes.lkt:862



--# expr-start 495 'New[SynthParamDecl]' New_Node nodes.lkt:862



if Is_Foreign_Strict (Self.Self_Env, Self) then
   Raise_Property_Exception
     (Self,
      Property_Error'Identity,
      "synthetic nodes cannot have foreign lexical envs");
end if;


New_Node := new Root_Node_Record
  (Lkt_Synth_Param_Decl);
Initialize
  (Self => New_Node,
   Kind => Lkt_Synth_Param_Decl,
   Unit => Self.Unit,

   Token_Start_Index => No_Token_Index,
   Token_End_Index   => No_Token_Index,

   Parent => Self,

   Self_Env => Self.Self_Env);
Register_Destroyable (Self.Unit, New_Node);



--# expr-done 495
Is_Null := New_Node = null; 
if Is_Null then
   
   If_Result := No_Entity_Synth_Param_Decl;
else
   

As_Entity := (Info => E_Info, Node => New_Node); 
   If_Result := As_Entity;
end if;



--# expr-done 496



   
      Cast_Result_1 := Create_Internal_Entity_Decl
        (Node => If_Result.Node,
         Info => If_Result.Info);



--# expr-start 497 SymbolLiteral Sym nodes.lkt:859
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Var); 
--# expr-done 497





--# expr-start 498 LktNode.logicvar_type Fld nodes.lkt:860







   if Ent.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start LktNode.logicvar_type
Fld := Liblktlang.Implementation.Lkt_Node_P_Logicvar_Type (Node => Ent.Node);
--# end
--# expr-done 498



   
      Cast_Result := Create_Internal_Entity_Type_Decl
        (Node => Fld.Node,
         Info => Fld.Info);


New_Struct := (Accept_Logical_Var => False, Decl => Cast_Result_1, Has_Default_Value => False, Name => Sym, Param_Type => Cast_Result); 
--# expr-done 499
Array_Lit := Create_Internal_Resolved_Param_Array (Internal_Internal_Resolved_Param_Array'(1 => New_Struct)); 
--# expr-done 500
--# expr-start 503 .map Map_Result nodes.lkt:865








   

   --# expr-start 501 FunDecl.params Fld_1 nodes.lkt:865







   if Local_Fd.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_1 := Create_Internal_Entity_Fun_Param_Decl_List (Node => Local_Fd.Node.Fun_Decl_F_Params, Info => Local_Fd.Info);
--# expr-done 501









Fld_3 := Fld_1.Node;

   
      declare
         Map_Result_Vec : Internal_Resolved_Param_Vectors.Vector;
      begin
   

   

      if Fld_3 = null then
         
      Map_Result := Create_Internal_Resolved_Param_Array (0);
   
      else
         

      declare
         
         Collection : constant Bare_Fun_Param_Decl_List := Fld_3;
      begin
         for Untyped_Item of
               Collection.Nodes (1 .. Children_Count (Collection))
         loop
                  
                  Bare_Item := Untyped_Item; 
                  



Is_Null_1 := Bare_Item = null; 
if Is_Null_1 then
   
   If_Result_1 := No_Entity_Fun_Param_Decl;
else
   








Fld_4 := Fld_1.Info;

As_Entity_1 := (Info => Fld_4, Node => Bare_Item); 
   If_Result_1 := As_Entity_1;
end if;



                  Item := If_Result_1; 

            
   --# scope-start


               --# bind p Item

            
         
         
      --# expr-start 502 ComponentDecl.to_generic_param Fld_2 nodes.lkt:865







   if Item.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start ComponentDecl.to_generic_param
Fld_2 := Liblktlang.Implementation.Component_Decl_P_To_Generic_Param (Node => Item.Node, E_Info => Item.Info);
--# end
--# expr-done 502
      

         declare
            Item_To_Append : constant Internal_Resolved_Param := Fld_2;
         begin
            Internal_Resolved_Param_Vectors.Append (Map_Result_Vec, Item_To_Append);
         end;
   
   
   

            
   --# end



         end loop;
      end;
   
      end if;

   

         Map_Result := Create_Internal_Resolved_Param_Array
           (Items_Count => Natural (Internal_Resolved_Param_Vectors.Length (Map_Result_Vec)));
         for I in Map_Result.Items'Range loop
            Map_Result.Items (I) := Internal_Resolved_Param_Vectors.Get
              (Map_Result_Vec,
               I + Internal_Resolved_Param_Vectors.Index_Type'First - Map_Result.Items'First);
         end loop;
         Internal_Resolved_Param_Vectors.Destroy (Map_Result_Vec);

      end;
   



--# expr-done 503
Concat_Result := Concat (Array_Lit, Map_Result); 
--# expr-done 504
   If_Result_4 := Concat_Result;
end if;

      Inc_Ref (If_Result_4);


--# expr-done 505
   If_Result_6 := If_Result_4;
end if;

      Inc_Ref (If_Result_6);


--# expr-done 506
Let_Result := If_Result_6; Inc_Ref (Let_Result);
--# end
Scope_Result := Let_Result;
   Inc_Ref (Scope_Result);


   --# end
      Finalizer_Scope_533;


            Match_Result := Scope_Result; Inc_Ref (Match_Result);
         when Lkt_Synth_Fun_Decl =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_5 := Create_Internal_Entity_Synth_Fun_Decl
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Local_Lfd := Cast_Result_5; 
--# bind lfd Local_Lfd
--# expr-start 507 SynthFunDecl.params Fld_19 nodes.lkt:866







   if Local_Lfd.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_19 := Local_Lfd.Node.Synth_Fun_Decl_F_Params;
Inc_Ref (Fld_19);
--# expr-done 507
Let_Result_1 := Fld_19; Inc_Ref (Let_Result_1);
--# end
Scope_Result_1 := Let_Result_1;
   Inc_Ref (Scope_Result_1);


   --# end
      Finalizer_Scope_536;


            Match_Result := Scope_Result_1; Inc_Ref (Match_Result);
         when Lkt_Class_Decl =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_6 := Create_Internal_Entity_Class_Decl
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Local_Cd := Cast_Result_6; 
--# bind cd Local_Cd
--# expr-start 510 .map Map_Result_2 nodes.lkt:868








   

   --# expr-start 508 ClassDecl.constructor_fields Fld_20 nodes.lkt:868







   if Local_Cd.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start ClassDecl.constructor_fields
Fld_20 := Liblktlang.Implementation.Class_Decl_P_Constructor_Fields (Node => Local_Cd.Node, E_Info => Local_Cd.Info);
--# end
--# expr-done 508

   
      declare
         Map_Result_2_Vec : Internal_Resolved_Param_Vectors.Vector;
      begin
   

   

      

      declare
         
         Collection : constant Internal_Entity_Field_Decl_Array_Access := Fld_20;
      begin
         for Item_2 of
               Collection.Items
         loop

            
   --# scope-start


               --# bind d Item_2

            
         
         
      --# expr-start 509 ComponentDecl.to_generic_param Fld_21 nodes.lkt:868







   if Item_2.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start ComponentDecl.to_generic_param
Fld_21 := Liblktlang.Implementation.Component_Decl_P_To_Generic_Param (Node => Item_2.Node, E_Info => Item_2.Info);
--# end
--# expr-done 509
      

         declare
            Item_To_Append : constant Internal_Resolved_Param := Fld_21;
         begin
            Internal_Resolved_Param_Vectors.Append (Map_Result_2_Vec, Item_To_Append);
         end;
   
   
   

            
   --# end



         end loop;
      end;
   

   

         Map_Result_2 := Create_Internal_Resolved_Param_Array
           (Items_Count => Natural (Internal_Resolved_Param_Vectors.Length (Map_Result_2_Vec)));
         for I in Map_Result_2.Items'Range loop
            Map_Result_2.Items (I) := Internal_Resolved_Param_Vectors.Get
              (Map_Result_2_Vec,
               I + Internal_Resolved_Param_Vectors.Index_Type'First - Map_Result_2.Items'First);
         end loop;
         Internal_Resolved_Param_Vectors.Destroy (Map_Result_2_Vec);

      end;
   



--# expr-done 510
Let_Result_2 := Map_Result_2; Inc_Ref (Let_Result_2);
--# end
Scope_Result_2 := Let_Result_2;
   Inc_Ref (Scope_Result_2);


   --# end
      Finalizer_Scope_537;


            Match_Result := Scope_Result_2; Inc_Ref (Match_Result);
         when Lkt_Enum_Class_Decl .. Lkt_Trait_Decl =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_7 := Create_Internal_Entity_Named_Type_Decl
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Local_Td := Cast_Result_7; 
--# bind td Local_Td
--# expr-start 518 .map Map_Result_4 nodes.lkt:870








   

   --# expr-start 514 .filter Map_Result_3 nodes.lkt:870








   

   --# expr-start 511 NamedTypeDecl.decls Fld_22 nodes.lkt:870







   if Local_Td.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_22 := Create_Internal_Entity_Decl_Block (Node => Implementation.Named_Type_Decl_F_Decls (Local_Td.Node), Info => Local_Td.Info);
--# expr-done 511









Fld_24 := Fld_22.Node;

   
      declare
         Map_Result_3_Vec : Internal_Entity_Full_Decl_Vectors.Vector;
      begin
   

   

      if Fld_24 = null then
         
      Map_Result_3 := Create_Internal_Entity_Full_Decl_Array (0);
   
      else
         

      declare
         
         Collection : constant Bare_Decl_Block := Fld_24;
      begin
         for Untyped_Item_3 of
               Collection.Nodes (1 .. Children_Count (Collection))
         loop
                  
                  Bare_Item_3 := Untyped_Item_3; 
                  



Is_Null_5 := Bare_Item_3 = null; 
if Is_Null_5 then
   
   If_Result_7 := No_Entity_Full_Decl;
else
   








Fld_25 := Fld_22.Info;

As_Entity_5 := (Info => Fld_25, Node => Bare_Item_3); 
   If_Result_7 := As_Entity_5;
end if;



                  Item_3 := If_Result_7; 

            
   --# scope-start


               --# bind d Item_3

            
         --# expr-start 513 IsA Is_A nodes.lkt:870
--# expr-start 512 FullDecl.decl Fld_23 nodes.lkt:870







   if Item_3.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_23 := Create_Internal_Entity_Decl (Node => Item_3.Node.Full_Decl_F_Decl, Info => Item_3.Info);
--# expr-done 512
Is_A := Fld_23.Node /= null 
and then Fld_23.Node.Kind in Lkt_Field_Decl_Range; 
--# expr-done 513
         if Is_A then
            
         
      
      

         declare
            Item_To_Append : constant Internal_Entity_Full_Decl := Item_3;
         begin
            Internal_Entity_Full_Decl_Vectors.Append (Map_Result_3_Vec, Item_To_Append);
         end;
   
   
         end if;
   

            
   --# end



         end loop;
      end;
   
      end if;

   

         Map_Result_3 := Create_Internal_Entity_Full_Decl_Array
           (Items_Count => Natural (Internal_Entity_Full_Decl_Vectors.Length (Map_Result_3_Vec)));
         for I in Map_Result_3.Items'Range loop
            Map_Result_3.Items (I) := Internal_Entity_Full_Decl_Vectors.Get
              (Map_Result_3_Vec,
               I + Internal_Entity_Full_Decl_Vectors.Index_Type'First - Map_Result_3.Items'First);
         end loop;
         Internal_Entity_Full_Decl_Vectors.Destroy (Map_Result_3_Vec);

      end;
   



--# expr-done 514

   
      declare
         Map_Result_4_Vec : Internal_Resolved_Param_Vectors.Vector;
      begin
   

   

      

      declare
         
         Collection : constant Internal_Entity_Full_Decl_Array_Access := Map_Result_3;
      begin
         for Item_4 of
               Collection.Items
         loop

            
   --# scope-start


               --# bind d Item_4

            
         
         
      --# expr-start 517 ComponentDecl.to_generic_param Fld_27 nodes.lkt:871



--# expr-start 516 Cast Cast_Result_3 nodes.lkt:871





--# expr-start 515 FullDecl.decl Fld_26 nodes.lkt:871







   if Item_4.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_26 := Create_Internal_Entity_Decl (Node => Item_4.Node.Full_Decl_F_Decl, Info => Item_4.Info);
--# expr-done 515



   if Fld_26.Node = null
      or else Fld_26.Node.Kind in Lkt_Field_Decl_Range
   then
      
      Cast_Result_3 := Create_Internal_Entity_Field_Decl
        (Node => Fld_26.Node,
         Info => Fld_26.Info);

   else
         Cast_Result_3 := No_Entity_Field_Decl;
   end if;


--# expr-done 516



   if Cast_Result_3.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start ComponentDecl.to_generic_param
Fld_27 := Liblktlang.Implementation.Component_Decl_P_To_Generic_Param (Node => Cast_Result_3.Node, E_Info => Cast_Result_3.Info);
--# end
--# expr-done 517
      

         declare
            Item_To_Append : constant Internal_Resolved_Param := Fld_27;
         begin
            Internal_Resolved_Param_Vectors.Append (Map_Result_4_Vec, Item_To_Append);
         end;
   
   
   

            
   --# end



         end loop;
      end;
   

   

         Map_Result_4 := Create_Internal_Resolved_Param_Array
           (Items_Count => Natural (Internal_Resolved_Param_Vectors.Length (Map_Result_4_Vec)));
         for I in Map_Result_4.Items'Range loop
            Map_Result_4.Items (I) := Internal_Resolved_Param_Vectors.Get
              (Map_Result_4_Vec,
               I + Internal_Resolved_Param_Vectors.Index_Type'First - Map_Result_4.Items'First);
         end loop;
         Internal_Resolved_Param_Vectors.Destroy (Map_Result_4_Vec);

      end;
   



--# expr-done 518
Let_Result_3 := Map_Result_4; Inc_Ref (Let_Result_3);
--# end
Scope_Result_3 := Let_Result_3;
   Inc_Ref (Scope_Result_3);


   --# end
      Finalizer_Scope_539;


            Match_Result := Scope_Result_3; Inc_Ref (Match_Result);
         when Lkt_Grammar_Rule_Decl .. Lkt_Val_Decl | Lkt_Env_Spec_Decl .. Lkt_Lexer_Family_Decl | Lkt_Synth_Param_Decl .. Lkt_Generic_Param_Type_Decl =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_8 := Create_Internal_Entity_Decl
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Ignored := Cast_Result_8; 
--# bind _ Ignored

Let_Result_4 := No_Internal_Resolved_Param_Array_Type; Inc_Ref (Let_Result_4);
--# end
Scope_Result_4 := Let_Result_4;
   Inc_Ref (Scope_Result_4);


   --# end
      Finalizer_Scope_543;


            Match_Result := Scope_Result_4; Inc_Ref (Match_Result);
end case;

--# expr-done 519
Scope_Result_5 := Match_Result;
   Inc_Ref (Scope_Result_5);


   --# end
      Finalizer_Scope_532;


   Result_Var := Scope_Result_5;
else
   
   Result_Var := No_Internal_Resolved_Param_Array_Type;
end if;

      Inc_Ref (Result_Var);


--# expr-done 520

         Property_Result := Result_Var;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_101;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_101;
                     Finalizer_Scope_532;
                     Finalizer_Scope_533;
                     Finalizer_Scope_536;
                     Finalizer_Scope_537;
                     Finalizer_Scope_539;
                     Finalizer_Scope_543;


               if Self /= null
                  and then not Self.Unit.Context.In_Populate_Lexical_Env
               then
                     Add_Memoized_Error
                       (Self.Unit, Mmz_Handle, Exc, Mmz_Stored);
               end if;


            raise;
      end;

      if Self /= null
         and then not Self.Unit.Context.In_Populate_Lexical_Env
      then
         Mmz_Val := (Kind => Mmz_Internal_Resolved_Param_Array_Access,
                     As_Internal_Resolved_Param_Array_Access => Property_Result);
         Add_Memoized_Value (Self.Unit, Mmz_Handle, Mmz_Val, Mmz_Stored);
            if Mmz_Stored then
               Inc_Ref (Property_Result);
            end if;
      end if;



   return Property_Result;
end Decl_P_Get_Params;
--# end

   







--# property-start Decl.subdecl_if_generic nodes.lkt:881
pragma Warnings (Off, "is not referenced");

function Decl_P_Subdecl_If_Generic
  
  (Node : Bare_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity_Decl
is
   Self : Bare_Decl :=
     Bare_Decl (Node);
     Ent : Internal_Entity_Decl :=
       Internal_Entity_Decl'(Node => Self, Info => E_Info);
      --# bind self Ent

   


   Property_Result : Internal_Entity_Decl;

      

      Local_Gc : Internal_Entity_Generic_Decl;
Fld : Internal_Entity_Decl;
Ignored : Internal_Entity_Decl;
Match_Prefix : Internal_Entity_Decl;
Cast_Result : Internal_Entity_Generic_Decl;
Let_Result : Internal_Entity_Decl;
Scope_Result : Internal_Entity_Decl;
Cast_Result_1 : Internal_Entity_Decl;
Let_Result_1 : Internal_Entity_Decl;
Scope_Result_1 : Internal_Entity_Decl;
Match_Result : Internal_Entity_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 522 Match Match_Result nodes.lkt:881










   if Ent.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Match_Prefix := Ent; 

case Lkt_Decl (Match_Prefix.Node.Kind) is
         when Lkt_Generic_Decl =>
            



   --# scope-start



--# scope-start









   
      Cast_Result := Create_Internal_Entity_Generic_Decl
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Local_Gc := Cast_Result; 
--# bind gc Local_Gc
--# expr-start 521 GenericDecl.decl Fld nodes.lkt:882







   if Local_Gc.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld := Create_Internal_Entity_Decl (Node => Local_Gc.Node.Generic_Decl_F_Decl, Info => Local_Gc.Info);
--# expr-done 521
Let_Result := Fld; 
--# end
Scope_Result := Let_Result;


   --# end


            Match_Result := Scope_Result; 
         when Lkt_Grammar_Rule_Decl .. Lkt_Error_Decl | Lkt_Grammar_Decl .. Lkt_Trait_Decl =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_1 := Create_Internal_Entity_Decl
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Ignored := Cast_Result_1; 
--# bind _ Ignored

Let_Result_1 := Ent; 
--# end
Scope_Result_1 := Let_Result_1;


   --# end


            Match_Result := Scope_Result_1; 
end case;

--# expr-done 522

         Property_Result := Match_Result;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Decl_P_Subdecl_If_Generic;
--# end

   







--# property-start Decl.is_generic nodes.lkt:889
pragma Warnings (Off, "is not referenced");

function Decl_P_Is_Generic
  
  (Node : Bare_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Boolean
is
   Self : Bare_Decl :=
     Bare_Decl (Node);
     Ent : Internal_Entity_Decl :=
       Internal_Entity_Decl'(Node => Self, Info => E_Info);
      --# bind self Ent

   


   Property_Result : Boolean;

      

      Is_Null : Boolean;
Fld : Internal_Entity;
Is_A : Boolean;
If_Result : Boolean;
Cast_Result : Internal_Entity_Function_Type;
Var_Expr : Internal_Entity_Function_Type;
Fld_1 : Internal_Entity_Decl;
Var_Expr_1 : Internal_Entity_Decl;
Fld_2 : Boolean;
Result_Var : Boolean;
Scope_Result : Boolean;
Result_Var_1 : Boolean;
If_Result_1 : Boolean;



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 531 BooleanOr If_Result_1 nodes.lkt:890



--# expr-start 526 BooleanOr If_Result nodes.lkt:890



--# expr-start 523 .is_null Is_Null nodes.lkt:890

Is_Null := Ent.Node = null; 
--# expr-done 523
if Is_Null then
   
   If_Result := True;
else
   --# expr-start 525 IsA Is_A nodes.lkt:890
--# expr-start 524 .parent Fld nodes.lkt:890







   if Ent.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld := Liblktlang.Implementation.Parent (Node => Ent.Node, E_Info => Ent.Info);
--# expr-done 524
Is_A := Fld.Node /= null 
and then Fld.Node.Kind in Lkt_Generic_Decl_Range; 
--# expr-done 525
   If_Result := Is_A;
end if;



--# expr-done 526
if If_Result then
   
   If_Result_1 := True;
else
   --# expr-start 530 .do Result_Var_1 nodes.lkt:891



--# expr-start 527 Cast Cast_Result nodes.lkt:891









   if Ent.Node = null
      or else Ent.Node.Kind in Lkt_Function_Type_Range
   then
      
      Cast_Result := Create_Internal_Entity_Function_Type
        (Node => Ent.Node,
         Info => Ent.Info);

   else
         Cast_Result := No_Entity_Function_Type;
   end if;


--# expr-done 527
Var_Expr := Cast_Result;





if Var_Expr /= No_Entity_Function_Type then
   



   --# scope-start






--# expr-start 528 FunctionType.origin Fld_1 nodes.lkt:892







   if Var_Expr.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_1 := Var_Expr.Node.Function_Type_F_Origin;
--# expr-done 528
Var_Expr_1 := Fld_1;





if Var_Expr_1 /= No_Entity_Decl then
   --# expr-start 529 Decl.is_generic Fld_2 nodes.lkt:892







   if Var_Expr_1.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start Decl.is_generic
Fld_2 := Liblktlang.Implementation.Decl_P_Is_Generic (Node => Var_Expr_1.Node, E_Info => Var_Expr_1.Info);
--# end
--# expr-done 529
   Result_Var := Fld_2;
else
   
   Result_Var := False;
end if;



Scope_Result := Result_Var;


   --# end


   Result_Var_1 := Scope_Result;
else
   
   Result_Var_1 := False;
end if;



--# expr-done 530
   If_Result_1 := Result_Var_1;
end if;



--# expr-done 531

         Property_Result := If_Result_1;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Decl_P_Is_Generic;
--# end

   







--# property-start Decl.return_type_is_instantiated nodes.lkt:898
pragma Warnings (Off, "is not referenced");

function Decl_P_Return_Type_Is_Instantiated
  
  (Node : Bare_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Boolean
is
   Self : Bare_Decl :=
     Bare_Decl (Node);
     Ent : Internal_Entity_Decl :=
       Internal_Entity_Decl'(Node => Self, Info => E_Info);
      --# bind self Ent

   


   Property_Result : Boolean;

      

      Cast_Result : Internal_Entity_Function_Type;
Var_Expr : Internal_Entity_Function_Type;
Fld : Internal_Entity_Type_Decl;
Fld_1 : Boolean;
Result_Var : Boolean;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         


--# expr-start 532 Cast Cast_Result nodes.lkt:899









   if Ent.Node = null
      or else Ent.Node.Kind in Lkt_Function_Type_Range
   then
      
      Cast_Result := Create_Internal_Entity_Function_Type
        (Node => Ent.Node,
         Info => Ent.Info);

   else
         Cast_Result := No_Entity_Function_Type;
   end if;


--# expr-done 532
Var_Expr := Cast_Result;





if Var_Expr /= No_Entity_Function_Type then
   --# expr-start 534 Decl.is_instantiated Fld_1 nodes.lkt:899



--# expr-start 533 FunctionType.return_type Fld nodes.lkt:899







   if Var_Expr.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld := Var_Expr.Node.Function_Type_F_Return_Type;
--# expr-done 533



   if Fld.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start Decl.is_instantiated
Fld_1 := Liblktlang.Implementation.Decl_P_Is_Instantiated (Node => Fld.Node, E_Info => Fld.Info);
--# end
--# expr-done 534
   Result_Var := Fld_1;
else
   
   Result_Var := False;
end if;




         Property_Result := Result_Var;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Decl_P_Return_Type_Is_Instantiated;
--# end

   







--# property-start Decl.is_instantiated nodes.lkt:904
pragma Warnings (Off, "is not referenced");

function Decl_P_Is_Instantiated
  
  (Node : Bare_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Boolean
is
   Self : Bare_Decl :=
     Bare_Decl (Node);
     Ent : Internal_Entity_Decl :=
       Internal_Entity_Decl'(Node => Self, Info => E_Info);
      --# bind self Ent

   


   Property_Result : Boolean;

      
            procedure Finalizer_Scope_547 with Inline_Always;
            procedure Finalizer_Scope_548 with Inline_Always;

      Fld : Internal_Entity;
Cast_Result : Internal_Entity_Generic_Decl;
Var_Expr : Internal_Entity_Generic_Decl;
Fld_1 : Internal_Entity_Generic_Param_Type_Decl_Array_Access;
Fld_2 : Lexical_Env := Empty_Env;
Fld_3 : Symbol_Type;
Env_Get_Result : Internal_Entity;
Cast_Result_1 : Internal_Entity_Type_Decl;
Cast_Result_2 : Internal_Entity_Type_Decl;
Fld_4 : Boolean;
Not_Val : Boolean;
Quantifier_Result : Boolean;
Scope_Result : Boolean;
Result_Var : Boolean;
Is_A : Boolean;
If_Result : Boolean;
Is_Null : Boolean;
If_Result_1 : Boolean;

            procedure Finalizer_Scope_547 is
            begin
                     Dec_Ref (Fld_1);
            end Finalizer_Scope_547;
            procedure Finalizer_Scope_548 is
            begin
                     Dec_Ref (Fld_2);
            end Finalizer_Scope_548;


begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 549 If If_Result_1 nodes.lkt:905



--# expr-start 535 .is_null Is_Null nodes.lkt:905

Is_Null := Ent.Node = null; 
--# expr-done 535
if Is_Null then
   
   If_Result_1 := True;
else
   --# expr-start 548 If If_Result nodes.lkt:905



--# expr-start 536 IsA Is_A nodes.lkt:906

Is_A := Ent.Node /= null 
and then Ent.Node.Kind in Lkt_Generic_Param_Type_Decl_Range; 
--# expr-done 536
if Is_A then
   
   If_Result := False;
else
   --# expr-start 547 .do Result_Var nodes.lkt:908



--# expr-start 538 Cast Cast_Result nodes.lkt:908





--# expr-start 537 .parent Fld nodes.lkt:908







   if Ent.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld := Liblktlang.Implementation.Parent (Node => Ent.Node, E_Info => Ent.Info);
--# expr-done 537



   if Fld.Node = null
      or else Fld.Node.Kind in Lkt_Generic_Decl_Range
   then
      
      Cast_Result := Create_Internal_Entity_Generic_Decl
        (Node => Fld.Node,
         Info => Fld.Info);

   else
         Cast_Result := No_Entity_Generic_Decl;
   end if;


--# expr-done 538
Var_Expr := Cast_Result;





if Var_Expr /= No_Entity_Generic_Decl then
   



   --# scope-start



--# expr-start 546 .all Quantifier_Result nodes.lkt:910








   

   --# expr-start 539 GenericDecl.generic_params Fld_1 nodes.lkt:910







   if Var_Expr.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start GenericDecl.generic_params
Fld_1 := Liblktlang.Implementation.Generic_Decl_P_Generic_Params (Node => Var_Expr.Node, E_Info => Var_Expr.Info);
--# end
--# expr-done 539

   
      Quantifier_Result := True;
   

   

      

      declare
         
         Collection : constant Internal_Entity_Generic_Param_Type_Decl_Array_Access := Fld_1;
      begin
         for Item of
               Collection.Items
         loop

            
   --# scope-start


               --# bind gf Item

            
      --# expr-start 545 Not Not_Val nodes.lkt:912
--# expr-start 544 TypeDecl.matching_type Fld_4 nodes.lkt:912



--# expr-start 543 Cast Cast_Result_1 nodes.lkt:912





--# expr-start 542 .get_first Env_Get_Result nodes.lkt:912
--# expr-start 540 LktNode.children_env Fld_2 nodes.lkt:912







   if Var_Expr.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_2 := Liblktlang.Implementation.Children_Env (Node => Var_Expr.Node, E_Info => Var_Expr.Info);
--# expr-done 540
--# expr-start 541 Decl.name Fld_3 nodes.lkt:912







   if Item.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


--# property-call-start '[dispatcher]Decl.name'
Fld_3 := Liblktlang.Implementation.Dispatcher_Decl_P_Name (Node => Item.Node);
--# end
--# expr-done 541


Env_Get_Result := AST_Envs.Get_First (Self => Fld_2, Key => Thin (Fld_3), Lookup_Kind => To_Lookup_Kind_Type (Recursive), Categories => (Nocat => True)); 
--# expr-done 542



   if Env_Get_Result.Node = null
      or else Env_Get_Result.Node.Kind in Lkt_Type_Decl
   then
      
      Cast_Result_1 := Create_Internal_Entity_Type_Decl
        (Node => Env_Get_Result.Node,
         Info => Env_Get_Result.Info);

   else
         Raise_Property_Exception
           (Self, Property_Error'Identity, "invalid object cast");
   end if;


--# expr-done 543



   if Cast_Result_1.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;











   
      Cast_Result_2 := Create_Internal_Entity_Type_Decl
        (Node => Item.Node,
         Info => Item.Info);


--# property-call-start TypeDecl.matching_type
Fld_4 := Liblktlang.Implementation.Type_Decl_P_Matching_Type (Node => Cast_Result_1.Node, Other => Cast_Result_2, E_Info => Cast_Result_1.Info);
--# end
--# expr-done 544
Not_Val := not (Fld_4); 
--# expr-done 545
      Quantifier_Result := Not_Val;
   

            
   --# end
      Finalizer_Scope_548;


               exit when not Quantifier_Result;

         end loop;
      end;
   

   
   



--# expr-done 546
Scope_Result := Quantifier_Result;


   --# end
      Finalizer_Scope_547;


   Result_Var := Scope_Result;
else
   
   Result_Var := True;
end if;



--# expr-done 547
   If_Result := Result_Var;
end if;



--# expr-done 548
   If_Result_1 := If_Result;
end if;



--# expr-done 549

         Property_Result := If_Result_1;
         
   --# end


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_547;
                     Finalizer_Scope_548;




            raise;
      end;



   return Property_Result;
end Decl_P_Is_Instantiated;
--# end

   







--# property-start Decl.has_correct_type_arg_number nodes.lkt:921
pragma Warnings (Off, "is not referenced");

function Decl_P_Has_Correct_Type_Arg_Number
  
  (Node : Bare_Decl
      ; Nb_Types : Integer
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Boolean
is
   Self : Bare_Decl :=
     Bare_Decl (Node);
     Ent : Internal_Entity_Decl :=
       Internal_Entity_Decl'(Node => Self, Info => E_Info);
      --# bind self Ent

   

   --# bind nb_types Nb_Types

   Property_Result : Boolean;

      

      Is_Null : Boolean;
Fld : Internal_Entity;
Local_Gd : Internal_Entity_Generic_Decl;
Fld_1 : Internal_Entity_Generic_Param_Decl_List;
Fld_2 : Bare_Generic_Param_Decl_List;
Len : Integer;
Is_Equal : Boolean;
Ignored : Internal_Entity;
Match_Prefix : Internal_Entity;
Cast_Result : Internal_Entity_Generic_Decl;
Let_Result : Boolean;
Scope_Result : Boolean;
Cast_Result_1 : Internal_Entity;
Let_Result_1 : Boolean;
Scope_Result_1 : Boolean;
Match_Result : Boolean;
If_Result : Boolean;



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 556 BooleanOr If_Result nodes.lkt:922



--# expr-start 550 .is_null Is_Null nodes.lkt:922

Is_Null := Ent.Node = null; 
--# expr-done 550
if Is_Null then
   
   If_Result := True;
else
   --# expr-start 555 Match Match_Result nodes.lkt:923






--# expr-start 551 .parent Fld nodes.lkt:923







   if Ent.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld := Liblktlang.Implementation.Parent (Node => Ent.Node, E_Info => Ent.Info);
--# expr-done 551



   if Fld.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Match_Prefix := Fld; 

case Lkt_Lkt_Node (Match_Prefix.Node.Kind) is
         when Lkt_Generic_Decl =>
            



   --# scope-start



--# scope-start









   
      Cast_Result := Create_Internal_Entity_Generic_Decl
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Local_Gd := Cast_Result; 
--# bind gd Local_Gd
--# expr-start 554 Eq Is_Equal nodes.lkt:925
--# expr-start 553 .length Len nodes.lkt:925



--# expr-start 552 GenericDecl.generic_param_decls Fld_1 nodes.lkt:925







   if Local_Gd.Node = null then
      Raise_Property_Exception
        (Self, Property_Error'Identity, "dereferencing a null access");
   end if;


Fld_1 := Create_Internal_Entity_Generic_Param_Decl_List (Node => Local_Gd.Node.Generic_Decl_F_Generic_Param_Decls, Info => Local_Gd.Info);
--# expr-done 552





Fld_2 := Fld_1.Node; 
Len := Length (Fld_2); 
--# expr-done 553

Is_Equal := Len = Nb_Types; 
--# expr-done 554
Let_Result := Is_Equal; 
--# end
Scope_Result := Let_Result;


   --# end


            Match_Result := Scope_Result; 
         when Lkt_Argument .. Lkt_Error_Decl | Lkt_Grammar_Decl .. Lkt_Var_Bind =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_1 := Create_Internal_Entity
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Ignored := Cast_Result_1; 
--# bind _ Ignored

Let_Result_1 := True; 
--# end
Scope_Result_1 := Let_Result_1;


   --# end


            Match_Result := Scope_Result_1; 
end case;

--# expr-done 555
   If_Result := Match_Result;
end if;



--# expr-done 556

         Property_Result := If_Result;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Decl_P_Has_Correct_Type_Arg_Number;
--# end

   







--# property-start Decl.could_infer nodes.lkt:932
pragma Warnings (Off, "is not referenced");

function Decl_P_Could_Infer
  
  (Node : Bare_Decl
      ; Generic_Type : Internal_Entity_Function_Type
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Boolean
is
   Self : Bare_Decl :=
     Bare_Decl (Node);
     Ent : Internal_Entity_Decl :=
       Internal_Entity_Decl'(Node => Self, Info => E_Info);
      --# bind self Ent

   

   --# bind generic_type Generic_Type

   Property_Result : Boolean;

      

      Is_Null : Boolean;
Is_Null_1 : Boolean;
Not_Val : Boolean;
If_Result : Boolean;



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 560 BooleanOr If_Result nodes.lkt:933



--# expr-start 557 .is_null Is_Null nodes.lkt:933

Is_Null := Generic_Type.Node = null; 
--# expr-done 557
if Is_Null then
   
   If_Result := True;
else
   --# expr-start 559 Not Not_Val nodes.lkt:933
--# expr-start 558 .is_null Is_Null_1 nodes.lkt:933

Is_Null_1 := Ent.Node = null; 
--# expr-done 558
Not_Val := not (Is_Null_1); 
--# expr-done 559
   If_Result := Not_Val;
end if;



--# expr-done 560

         Property_Result := If_Result;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Decl_P_Could_Infer;
--# end

   







--# property-start Decl.instantiate_generic_decl nodes.lkt:938
pragma Warnings (Off, "is not referenced");

function Decl_P_Instantiate_Generic_Decl
  
  (Node : Bare_Decl
      ; Param_Types : Internal_Entity_Type_Decl_Array_Access
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity_Decl
is
   Self : Bare_Decl :=
     Bare_Decl (Node);
     Ent : Internal_Entity_Decl :=
       Internal_Entity_Decl'(Node => Self, Info => E_Info);
      --# bind self Ent

   

   --# bind param_types Param_Types

   Property_Result : Internal_Entity_Decl;

      
            procedure Finalizer_Scope_108 with Inline_Always;
            procedure Finalizer_Scope_551 with Inline_Always;

      Fld : Internal_Entity;
Cast_Result : Internal_Entity_Generic_Decl;
Fld_1 : Symbol_Type_Array_Access;
New_Node : Bare_Dyn_Env_Wrapper;
Var_Expr : Bare_Dyn_Env_Wrapper;
Fld_2 : Lexical_Env := Empty_Env;
Fld_3 : Internal_Entity_Decl;
Scope_Result : Internal_Entity_Decl;
Result_Var : Internal_Entity_Decl;
Is_Null : Boolean;
Is_Null_1 : Boolean;
Quantifier_Result : Boolean;
If_Result : Boolean;
Cast_Expr : Internal_Entity_Type_Decl;
Cast_Result_1 : Internal_Entity_Decl;
If_Result_1 : Internal_Entity_Decl;

            procedure Finalizer_Scope_108 is
            begin
                     Dec_Ref (Fld_1);
            end Finalizer_Scope_108;
            procedure Finalizer_Scope_551 is
            begin
                     Dec_Ref (Fld_2);
            end Finalizer_Scope_551;

      
      Mmz_Handle : Memoization_Handle;
      Mmz_Val    : Mmz_Value;

      Mmz_Stored : Boolean;
      --  Whether the memoization couple was actually stored. Used to determine
      --  whether to inc-ref the memoized value.

      function Create_Mmz_Key return Mmz_Key;
      --  Create a memoization key for this property call and return it

      --------------------
      -- Create_Mmz_Key --
      --------------------

      function Create_Mmz_Key return Mmz_Key is
      begin
         return Mmz_K : Mmz_Key :=
           (Property => Mmz_Bare_Decl_Decl_P_Instantiate_Generic_Decl,
            Items    => new Mmz_Key_Array (1 ..  3))
         do
            Mmz_K.Items (1) := (Kind => Mmz_Bare_Decl,
                                As_Bare_Decl => Self);
               Mmz_K.Items (2) := (Kind => Mmz_Internal_Entity_Type_Decl_Array_Access,
                                      As_Internal_Entity_Type_Decl_Array_Access => Param_Types);
            