/*
** Name:        aegis128x4_avx512.c
** Purpose:     Implementation of AEGIS-128x4 - AES-NI AVX512
** Copyright:   (c) 2023-2024 Frank Denis
** SPDX-License-Identifier: MIT
*/

#if defined(__i386__) || defined(_M_IX86) || defined(__x86_64__) || defined(_M_AMD64)

#include <errno.h>
#include <stddef.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>

#include "../common/common.h"
#include "aegis128x4.h"
#include "aegis128x4_avx512.h"

#ifdef HAVE_VAESINTRIN_H

#ifdef __clang__
#  if __clang_major__ >= 18
#    pragma clang attribute push(__attribute__((target("vaes,avx512f,evex512"))), \
                                             apply_to = function)
#  else
#    pragma clang attribute push(__attribute__((target("vaes,avx512f"))), \
                                             apply_to = function)
#  endif
#elif defined(__GNUC__)
#  pragma GCC target("vaes,avx512f")
#endif

#include <immintrin.h>

#define AES_BLOCK_LENGTH 64

typedef __m512i aegis128x4_avx512_aes_block_t;

#define AEGIS_AES_BLOCK_T aegis128x4_avx512_aes_block_t
#define AEGIS_BLOCKS      aegis128x4_avx512_blocks
#define AEGIS_STATE      _aegis128x4_avx512_state
#define AEGIS_MAC_STATE  _aegis128x4_avx512_mac_state

#define AEGIS_FUNC_PREFIX  aegis128x4_avx512_impl

#include "../common/func_names_define.h"

static inline AEGIS_AES_BLOCK_T
AEGIS_AES_BLOCK_XOR(const AEGIS_AES_BLOCK_T a, const AEGIS_AES_BLOCK_T b)
{
    return _mm512_xor_si512(a, b);
}

static inline AEGIS_AES_BLOCK_T
AEGIS_AES_BLOCK_AND(const AEGIS_AES_BLOCK_T a, const AEGIS_AES_BLOCK_T b)
{
    return _mm512_and_si512(a, b);
}

static inline AEGIS_AES_BLOCK_T
AEGIS_AES_BLOCK_LOAD(const uint8_t *a)
{
    return _mm512_loadu_si512((const AEGIS_AES_BLOCK_T *) (const void *) a);
}

static inline AEGIS_AES_BLOCK_T
AEGIS_AES_BLOCK_LOAD_64x2(uint64_t a, uint64_t b)
{
#if defined(_MSC_VER) && (_MSC_VER >= 1910 && _MSC_VER < 1920)
    /*
    ** Visual Studio 2017 produces an internal compiler error in release mode
    ** for the function call _mm512_broadcast_i32x4(_mm_set_epi64x(a, b)).
    ** Therefore perform the operation "manually".
    */
    __m128i x128 = _mm_set_epi64x(a, b);
    __m256i x256 = _mm256_broadcastsi128_si256(x128);
    __m512i x512 = _mm512_castsi256_si512(x256);
    x512 = _mm512_inserti64x4(x512, x256, 1);
    return x512;
#else
    return _mm512_broadcast_i32x4(_mm_set_epi64x(a, b));
#endif
}

static inline void
AEGIS_AES_BLOCK_STORE(uint8_t *a, const AEGIS_AES_BLOCK_T b)
{
    _mm512_storeu_si512((AEGIS_AES_BLOCK_T *) (void *) a, b);
}

static inline AEGIS_AES_BLOCK_T
AEGIS_AES_ENC(const AEGIS_AES_BLOCK_T a, const AEGIS_AES_BLOCK_T b)
{
    return _mm512_aesenc_epi128(a, b);
}

static inline void
AEGIS_update(AEGIS_AES_BLOCK_T *const state, const AEGIS_AES_BLOCK_T d1, const AEGIS_AES_BLOCK_T d2)
{
    AEGIS_AES_BLOCK_T tmp;

    tmp      = state[7];
    state[7] = AEGIS_AES_ENC(state[6], state[7]);
    state[6] = AEGIS_AES_ENC(state[5], state[6]);
    state[5] = AEGIS_AES_ENC(state[4], state[5]);
    state[4] = AEGIS_AES_ENC(state[3], state[4]);
    state[3] = AEGIS_AES_ENC(state[2], state[3]);
    state[2] = AEGIS_AES_ENC(state[1], state[2]);
    state[1] = AEGIS_AES_ENC(state[0], state[1]);
    state[0] = AEGIS_AES_ENC(tmp, state[0]);

    state[0] = AEGIS_AES_BLOCK_XOR(state[0], d1);
    state[4] = AEGIS_AES_BLOCK_XOR(state[4], d2);
}

#include "aegis128x4_common.h"

struct aegis128x4_implementation aegis128x4_avx512_implementation = {
#include "../common/func_table.h"
};

#include "../common/type_names_undefine.h"
#include "../common/func_names_undefine.h"

#ifdef __clang__
#  pragma clang attribute pop
#endif

#endif /* HAVE_VAESINTRIN_H */

#endif
