/*
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 */

#ifndef _SH4LT_C_SHTYPE_H_
#define _SH4LT_C_SHTYPE_H_

#include <stdlib.h>

#ifdef __cplusplus
extern "C" {
#endif

typedef void *Sh4ltShType;

/**
 * Construct a Sh4ltShtype.
 *
 * \param   media                 The media mime type.
 * \param   label                 The media label.
 * \param   group_label           The group label.
 *
 * \return the created Sh4ltShtype
 */

Sh4ltShType sh4lt_make_shtype(const char *media, const char *label,
                              const char *group_label);

/**
 * Delete the Sh4ltShType and release associated ressources.
 *
 * \param shtype The ShType to delete.
 */
void sh4lt_delete_shtype(Sh4ltShType shtype);

/**
 * Provide the absolute file path for the Sh4lt Unix socket.
 *
 * \param label The Sh4lt label.
 * \param group_label The Sh4lt group label.
 *
 * \return The absolute file path. Free after use.
 **/
char *sh4lt_get_path(const char *label, const char *group_label);

/**
 * Provide the log for issue with the socket path directory.
 *
 * \return The log message (empty if no issue found). Free after use.
 **/
char *sh4lt_get_log_for_path_issue();

/**
 * Get the media specification.
 *
 * \param shtype The ShType.
 *
 * \return The media specification. Free after use.
 **/
char *sh4lt_shtype_media(const Sh4ltShType shtype);

/**
 * Get the media label.
 *
 * \param shtype The ShType.
 *
 * \return The media label. Free after use.
 **/
char *sh4lt_shtype_label(const Sh4ltShType shtype);

/**
 * Get the media group label.
 *
 * \param shtype The ShType.
 *
 * \return The media group label. Free after use.
 **/
char *sh4lt_shtype_group(const Sh4ltShType shtype);

/**
 * Get default group label
 *
 * \return The default group label. Free after use.
 **/
char *sh4lt_shtype_default_group();

  /**
   * Set a string property.
   *
   * \param shtype The ShType.
   * \param key The property key.
   * \param value The value for the string property.
   **/
  void sh4lt_shtype_set_str_prop(const Sh4ltShType shtype, const char *key,
                                 const char *value);

  /**
   * Set an integer property.
   *
   * \param shtype The ShType.
   * \param key The property key.
   * \param value The value for the integer property.
   **/
  void sh4lt_shtype_set_int_prop(const Sh4ltShType shtype, const char *key,
                                 int64_t value);

  /**
   * Set an float property.
   *
   * \param shtype The ShType.
   * \param key The property key.
   * \param value The value for the float property.
   **/
  void sh4lt_shtype_set_float_prop(const Sh4ltShType shtype, const char *key,
                                   double value);
/**
   * Set a boolean property.
   *
   * \param shtype The ShType.
   * \param key The property key.
   * \param value The value for the boolean property.
   **/
  void sh4lt_shtype_set_bool_prop(const Sh4ltShType shtype, const char *key,
                                  int value);

  /**
   * Set the custom type of a property.
   *
   * \param shtype The ShType.
   * \param key The property key.
   * \param custom_type The property custom type.
   **/
  void sh4lt_shtype_set_custom_type(const Sh4ltShType shtype, const char *key,
                                    const char *custom_type);

  /**
   * Get the value for a given key.
   *
   * \param shtype The ShType.
   * \param key The key.
   *
   * \return The value, or NULL if key is not found. Free after use.
   **/
  char *sh4lt_shtype_get_str_prop(const Sh4ltShType shtype, const char *key);

  /**
   * Get the integer value for a given key.
   *
   * \param shtype The ShType.
   * \param key The key.
   *
   * \return The value, or 0 if the value is not found.
   **/
  int64_t sh4lt_shtype_get_int_prop(const Sh4ltShType shtype, const char *key);

  /**
   * Get the float value for a given key.
   *
   * \param shtype The ShType.
   * \param key The key.
   *
   * \return The value, or 0 if the value is not found.
   **/
  double sh4lt_shtype_get_float_prop(const Sh4ltShType shtype, const char *key);

  /**
   * Get the value for a given key.
   *
   * \param shtype The ShType.
   * \param key The key.
   *
   * \return The value, or NULL if key is not found. Free after use.
   **/
  int sh4lt_shtype_get_bool_prop(const Sh4ltShType shtype, const char *key);


  /**
   * Get the custom type for a given key.
   *
   * \param shtype The ShType.
   * \param key The key.
   *
   * \return The value, or NULL if no custom type is specified for this key.
   **/
  char *sh4lt_shtype_get_custom_type(const Sh4ltShType shtype, const char *key);
  
  /**
   * Get the properties.
   *
   * \param shtype The ShType.
   * \param properties A previously allocated array of properties. Free elements after use.
   * \param size A previously allocated pointer to store size of the properties array.
   **/
  void sh4lt_shtype_get_properties(const Sh4ltShType shtype, char *properties[],
                                   size_t *size);
  
  /**
   * Get the custom property types.
   *
   * \param shtype The ShType.
   * \param types The array of types. Free after use array and elements after use.
   * \param size The pointer to store size of the types array.
   **/
  void sh4lt_shtype_get_custom_types(const Sh4ltShType shtype, char *types[],
                                     size_t *size);

/**
 * Construct a Shtype from a serialized GStreamer Caps.
 *
 * \param   gst_caps The serialized GStreamer Caps.
 * \param   label The ShType label.
 * \param   group_label The ShType group label.
 *
 * \return  The ShType created. Free after use.
 */
Sh4ltShType sh4lt_shtype_from_gst_caps(const char *gst_caps, const char *label,
                                 const char *group_label);

/**
 * Produce a serialized GStreamer caps from a ShType.
 * Note the ShType label and group label are not embedded in the GStreamer caps.
 *
 * \param   shtype the ShType instance to convert.
 *
 * \return  The string in the GStreamer caps format. Free after use.
 */
char *sh4lt_shtype_to_gst_caps(const Sh4ltShType shtype);

/**
 * Obtain a ShType from serialization.
 *
 * \param serialized A string description (JSON) of a Shtype.
 *
 * \return The corresponding Shtype.
 **/
Sh4ltShType sh4lt_shtype_deserialize(const char *serialized);

/**
 * Obtain a string representation of a ShType.
 *
 * \param shtype The ShType.
 *
 * \return The resulting serialization. Free after use.
 **/
char *sh4lt_shtype_serialize(const Sh4ltShType shtype);

#ifdef __cplusplus
}
#endif

#endif
