package root

import (
	"testing"

	"github.com/cli/cli/v2/pkg/iostreams"
	"github.com/stretchr/testify/require"
)

func TestCmdHelpTopic(t *testing.T) {
	t.Parallel()

	tests := []struct {
		name                string
		topic               helpTopic
		args                []string
		flags               []string
		errorAssertion      require.ErrorAssertionFunc
		expectedAnnotations map[string]string
	}{
		{
			name:           "when there are no args or flags, prints the long message to stdout",
			topic:          helpTopic{name: "test-topic", long: "test-topic-long"},
			args:           []string{},
			flags:          []string{},
			errorAssertion: require.NoError,
		},
		{
			name:           "when an arg is provided, it is ignored and help is printed",
			topic:          helpTopic{name: "test-topic"},
			args:           []string{"anything"},
			flags:          []string{},
			errorAssertion: require.NoError,
		},
		{
			name:           "when a flag is provided, returns an error",
			topic:          helpTopic{name: "test-topic"},
			args:           []string{},
			flags:          []string{"--anything"},
			errorAssertion: require.Error,
		},
		{
			name:           "when there is an example, include it in the stdout",
			topic:          helpTopic{name: "test-topic", example: "test-topic-example"},
			args:           []string{},
			flags:          []string{},
			errorAssertion: require.NoError,
		},
		{
			name:           "sets important markdown annotations on the command",
			topic:          helpTopic{name: "test-topic"},
			args:           []string{},
			flags:          []string{},
			errorAssertion: require.NoError,
			expectedAnnotations: map[string]string{
				"markdown:generate": "true",
				"markdown:basename": "gh_help_test-topic",
			},
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()

			ios, _, stdout, _ := iostreams.Test()

			cmd := NewCmdHelpTopic(ios, tt.topic)
			cmd.SetArgs(append(tt.args, tt.flags...))

			_, err := cmd.ExecuteC()
			tt.errorAssertion(t, err)

			if tt.topic.long != "" {
				require.Contains(t, stdout.String(), tt.topic.long)
			}

			if tt.topic.example != "" {
				require.Contains(t, stdout.String(), tt.topic.example)
			}

			if len(tt.expectedAnnotations) > 0 {
				require.Equal(t, cmd.Annotations, tt.expectedAnnotations)
			}
		})
	}
}
