/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { StreamingState } from '../types.js';
import { useTimer } from './useTimer.js';
import { usePhraseCycler } from './usePhraseCycler.js';
import { useState, useEffect, useRef } from 'react'; // Added useRef
export const useLoadingIndicator = (streamingState, customWittyPhrases) => {
    const [timerResetKey, setTimerResetKey] = useState(0);
    const isTimerActive = streamingState === StreamingState.Responding;
    const elapsedTimeFromTimer = useTimer(isTimerActive, timerResetKey);
    const isPhraseCyclingActive = streamingState === StreamingState.Responding;
    const isWaiting = streamingState === StreamingState.WaitingForConfirmation;
    const currentLoadingPhrase = usePhraseCycler(isPhraseCyclingActive, isWaiting, customWittyPhrases);
    const [retainedElapsedTime, setRetainedElapsedTime] = useState(0);
    const prevStreamingStateRef = useRef(null);
    useEffect(() => {
        if (prevStreamingStateRef.current === StreamingState.WaitingForConfirmation &&
            streamingState === StreamingState.Responding) {
            setTimerResetKey((prevKey) => prevKey + 1);
            setRetainedElapsedTime(0); // Clear retained time when going back to responding
        }
        else if (streamingState === StreamingState.Idle &&
            prevStreamingStateRef.current === StreamingState.Responding) {
            setTimerResetKey((prevKey) => prevKey + 1); // Reset timer when becoming idle from responding
            setRetainedElapsedTime(0);
        }
        else if (streamingState === StreamingState.WaitingForConfirmation) {
            // Capture the time when entering WaitingForConfirmation
            // elapsedTimeFromTimer will hold the last value from when isTimerActive was true.
            setRetainedElapsedTime(elapsedTimeFromTimer);
        }
        prevStreamingStateRef.current = streamingState;
    }, [streamingState, elapsedTimeFromTimer]);
    return {
        elapsedTime: streamingState === StreamingState.WaitingForConfirmation
            ? retainedElapsedTime
            : elapsedTimeFromTimer,
        currentLoadingPhrase,
    };
};
//# sourceMappingURL=useLoadingIndicator.js.map