/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2016, Hewlett Packard Enterprise Development LP
   Copyright (C) 2018 - 2019, Advanced Micro Devices, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_H
#define BLIS_H


// Allow C++ users to include this header file in their source code. However,
// we make the extern "C" conditional on whether we're using a C++ compiler,
// since regular C compilers don't understand the extern "C" construct.
#ifdef __cplusplus
extern "C" {
#endif

// NOTE: PLEASE DON'T CHANGE THE ORDER IN WHICH HEADERS ARE INCLUDED UNLESS
// YOU ARE SURE THAT IT DOESN'T BREAK INTER-HEADER MACRO DEPENDENCIES.

// -- configure definitions --

// NOTE: bli_config.h must be included before any other BLIS header. It is
// bootstrapped by ./configure and does not depend on later headers. Moreover
// these configuration variables are necessary to change some default behaviors
// (e.g. disable OS detection in bli_system.h in case of --disable-system).
// begin bli_config.h
#line 1 ".//bli_config.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2018 - 2019, Advanced Micro Devices, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_CONFIG_H
#define BLIS_CONFIG_H

// Enabled configuration "family" (config_name)
#define BLIS_FAMILY_ARM64


// Enabled sub-configurations (config_list)
#define BLIS_CONFIG_FIRESTORM
#define BLIS_CONFIG_THUNDERX2
#define BLIS_CONFIG_CORTEXA57
#define BLIS_CONFIG_CORTEXA53
#define BLIS_CONFIG_GENERIC


// Enabled kernel sets (kernel_list)
#define BLIS_KERNELS_ARMV8A
#define BLIS_KERNELS_GENERIC


#define BLIS_VERSION_STRING "2.0"

#define BLIS_VERSION_MAJOR 2
#define BLIS_VERSION_MINOR 0
#define BLIS_VERSION_REVISION 0

#if 1
#define BLIS_ENABLE_SYSTEM
#else
#define BLIS_DISABLE_SYSTEM
#endif

#if 1
#define BLIS_ENABLE_TLS
#else
#define BLIS_DISABLE_TLS
#endif

#if 1
#define BLIS_ENABLE_OPENMP
#if 1
#define BLIS_ENABLE_OPENMP_AS_DEFAULT
#endif
#endif

#if 1
#define BLIS_ENABLE_PTHREADS
#if 0
#define BLIS_ENABLE_PTHREADS_AS_DEFAULT
#endif
#endif

#if 0
#define BLIS_ENABLE_HPX
#if 0
#define BLIS_ENABLE_HPX_AS_DEFAULT
#endif
#endif

#if 1
#define BLIS_ENABLE_JRIR_SLAB
#endif

#if 0
#define BLIS_ENABLE_JRIR_RR
#endif

#if 0
#define BLIS_ENABLE_JRIR_TLB
#endif

#if 1
#define BLIS_ENABLE_PBA_POOLS
#else
#define BLIS_DISABLE_PBA_POOLS
#endif

#if 1
#define BLIS_ENABLE_SBA_POOLS
#else
#define BLIS_DISABLE_SBA_POOLS
#endif

#if 0
#define BLIS_ENABLE_MEM_TRACING
#else
#define BLIS_DISABLE_MEM_TRACING
#endif

#if 0
#define BLIS_ENABLE_SCALAPACK_COMPAT
#else
#define BLIS_DISABLE_SCALAPACK_COMPAT
#endif

#if 0 == 64
#define BLIS_INT_TYPE_SIZE 64
#elif 0 == 32
#define BLIS_INT_TYPE_SIZE 32
#else
// determine automatically
#endif

#if 32 == 64
#define BLIS_BLAS_INT_TYPE_SIZE 64
#elif 32 == 32
#define BLIS_BLAS_INT_TYPE_SIZE 32
#else
// determine automatically
#endif

#ifndef BLIS_ENABLE_BLAS
#ifndef BLIS_DISABLE_BLAS
#if 1
#define BLIS_ENABLE_BLAS
#else
#define BLIS_DISABLE_BLAS
#endif
#endif
#endif

#ifndef BLIS_ENABLE_CBLAS
#ifndef BLIS_DISABLE_CBLAS
#if 1
#define BLIS_ENABLE_CBLAS
#else
#define BLIS_DISABLE_CBLAS
#endif
#endif
#endif

#if 1
#define BLIS_ENABLE_SUP_HANDLING
#else
#define BLIS_DISABLE_SUP_HANDLING
#endif

#if 0
#define BLIS_ENABLE_MEMKIND
#else
#define BLIS_DISABLE_MEMKIND
#endif

#if 1
#define BLIS_ENABLE_TRSM_PREINVERSION
#else
#define BLIS_DISABLE_TRSM_PREINVERSION
#endif

#if 1
#define BLIS_ENABLE_PRAGMA_OMP_SIMD
#else
#define BLIS_DISABLE_PRAGMA_OMP_SIMD
#endif

#if 0
#define BLIS_ENABLE_SANDBOX
#else
#define BLIS_DISABLE_SANDBOX
#endif

#if 1
#define BLIS_ENABLE_SHARED
#else
#define BLIS_DISABLE_SHARED
#endif

#if 0
#define BLIS_ENABLE_COMPLEX_RETURN_INTEL
#else
#define BLIS_DISABLE_COMPLEX_RETURN_INTEL
#endif


#endif
// end bli_config.h
#line 58 "../blis-2.0/frame/include/blis.h"


// -- System and language-related headers --

// NOTE: bli_system.h header must be included before bli_config_macro_defs.h.
// begin bli_system.h
#line 1 "../blis-2.0/frame/include//bli_system.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2018 - 2019, Advanced Micro Devices, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SYSTEM_H
#define BLIS_SYSTEM_H

// NOTE: If not yet defined, we define _POSIX_C_SOURCE to make sure that
// various parts of POSIX are defined and made available.
#ifndef _POSIX_C_SOURCE
#define _POSIX_C_SOURCE 200809L
#endif

#include <stdio.h> // skipped
#include <stdlib.h> // skipped
#include <math.h> // skipped
#include <string.h> // skipped
#include <stdarg.h> // skipped
#include <float.h> // skipped
#include <errno.h> // skipped
#include <ctype.h> // skipped

#ifdef _MSC_VER
  #define strncasecmp _strnicmp
  #define strcasecmp _stricmp
#else
#include <strings.h> // skipped
#endif

// Determine the compiler (hopefully) and define conveniently named macros
// accordingly.
#if   defined(__ICC) || defined(__INTEL_COMPILER)
  #define BLIS_ICC
#elif defined(__clang__)
  #define BLIS_CLANG
#elif defined(__GNUC__)
  #define BLIS_GCC
#endif

// Determine if we are on a 64-bit or 32-bit architecture.
#if defined(_M_X64) || defined(__x86_64) || defined(__aarch64__) || \
    defined(_ARCH_PPC64) || defined(__s390x__) || defined(_LP64)
  #define BLIS_ARCH_64
#else
  #define BLIS_ARCH_32
#endif

// Determine the target operating system.
#if defined(BLIS_ENABLE_SYSTEM)
  #if defined(_WIN32) || defined(__CYGWIN__)
    #define BLIS_OS_WINDOWS 1
  #elif defined(__gnu_hurd__)
    #define BLIS_OS_GNU 1
  #elif defined(__APPLE__) || defined(__MACH__)
    #define BLIS_OS_OSX 1
  #elif defined(__ANDROID__)
    #define BLIS_OS_ANDROID 1
  #elif defined(__linux__)
    #define BLIS_OS_LINUX 1
  #elif defined(__bgq__)
    #define BLIS_OS_BGQ 1
  #elif defined(__bg__)
    #define BLIS_OS_BGP 1
  #elif defined(__FreeBSD__) || defined(__NetBSD__) || defined(__OpenBSD__) || \
        defined(__bsdi__) || defined(__DragonFly__) || \
        defined(__FreeBSD_kernel__) || defined(__HAIKU__)
    #define BLIS_OS_BSD 1
  #elif defined(EMSCRIPTEN)
    #define BLIS_OS_EMSCRIPTEN
  #else
    #error "Cannot determine operating system"
  #endif
#else // #if defined(BLIS_DISABLE_SYSTEM)
  #define BLIS_OS_NONE
#endif

// A few changes that may be necessary in Windows environments.
#if BLIS_OS_WINDOWS

  // Include Windows header file.
  #define WIN32_LEAN_AND_MEAN
  #define VC_EXTRALEAN
#include <windows.h> // skipped

  #if !defined(__clang__) && !defined(__GNUC__)
    // Undefine attribute specifiers in Windows.
    #define __attribute__(x)

    // Undefine restrict.
    #define restrict
  #endif

#endif

// time.h provides clock_gettime().
#if BLIS_OS_WINDOWS
#include <time.h> // skipped
#elif BLIS_OS_OSX
#include <mach/mach_time.h> // skipped
#else
  //#include <sys/time.h>

#include <time.h> // skipped
#endif


#endif
// end bli_system.h
#line 63 "../blis-2.0/frame/include/blis.h"

// begin bli_lang_defs.h
#line 1 "../blis-2.0/frame/include//bli_lang_defs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2018 - 2019, Advanced Micro Devices, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_LANG_DEFS_H
#define BLIS_LANG_DEFS_H


// -- Undefine restrict for C++ and C89/90 --

#ifdef __cplusplus
  // Language is C++; define restrict as nothing.
  #ifndef restrict
  #define restrict
  #endif
#elif __STDC_VERSION__ >= 199901L
  // Language is C99 (or later); do nothing since restrict is recognized.
#else
  // Language is pre-C99; define restrict as nothing.
  #ifndef restrict
  #define restrict
  #endif
#endif


// -- Define typeof() operator if using non-GNU compiler --

#ifndef __GNUC__
  #define typeof __typeof__
#else
  #ifndef typeof
  #define typeof __typeof__
  #endif
#endif


// -- BLIS Thread Local Storage Keyword --

// __thread for TLS is supported by GCC, CLANG, ICC, and IBMC.
// There is a small risk here as __GNUC__ can also be defined by some other
// compiler (other than ICC and CLANG which we know define it) that
// doesn't support __thread, as __GNUC__ is not quite unique to GCC.
// But the possibility of someone using such non-main-stream compiler
// for building BLIS is low.
#if defined(BLIS_ENABLE_TLS) && ( defined(__GNUC__)  || \
                                  defined(__clang__) || \
                                  defined(__ICC)     || \
                                  defined(__IBMC__) )
  #define BLIS_THREAD_LOCAL __thread
#else
  #define BLIS_THREAD_LOCAL
#endif


// -- BLIS constructor/destructor function attribute --

// __attribute__((constructor/destructor)) is supported by GCC only.
// There is a small risk here as __GNUC__ can also be defined by some other
// compiler (other than ICC and CLANG which we know define it) that
// doesn't support this, as __GNUC__ is not quite unique to GCC.
// But the possibility of someone using such non-main-stream compiler
// for building BLIS is low.

#if defined(__ICC) || defined(__INTEL_COMPILER)
  // ICC defines __GNUC__ but doesn't support this
  #define BLIS_ATTRIB_CTOR
  #define BLIS_ATTRIB_DTOR
#elif defined(__clang__)
  // CLANG supports __attribute__, but its documentation doesn't
  // mention support for constructor/destructor. Compiling with
  // clang and testing shows that it does support.
  #define BLIS_ATTRIB_CTOR __attribute__((constructor))
  #define BLIS_ATTRIB_DTOR __attribute__((destructor))
#elif defined(__GNUC__)
  #define BLIS_ATTRIB_CTOR __attribute__((constructor))
  #define BLIS_ATTRIB_DTOR __attribute__((destructor))
#else
  #define BLIS_ATTRIB_CTOR
  #define BLIS_ATTRIB_DTOR
#endif


#endif
// end bli_lang_defs.h
#line 64 "../blis-2.0/frame/include/blis.h"



// -- configure default definitions --

// begin bli_config_macro_defs.h
#line 1 "../blis-2.0/frame/include//bli_config_macro_defs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2019, Advanced Micro Devices, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_CONFIG_MACRO_DEFS_H
#define BLIS_CONFIG_MACRO_DEFS_H

// NOTE: This file should ONLY contain processing of macros that are set by
// configure and output into bli_config.h. Any other macro processing --
// especially such as for those macros that are expected to be optionally
// set within a configuration's bli_family_<conf>.h header -- MUST be placed
// in bli_kernel_macro_defs.h instead. The reason: bli_arch_config.h (which
// #includes the configuration's bli_family_<conf>.h header) is #included
// much later in blis.h than this file (bli_config_macro_defs.h), and so any
// macros set in bli_family_<conf>.h would have no effect on the processing
// that happens below.


// -- INTEGER PROPERTIES -------------------------------------------------------

// The bit size of the integer type used to track values such as dimensions,
// strides, diagonal offsets. A value of 32 results in BLIS using 32-bit signed
// integers while 64 results in 64-bit integers. Any other value results in use
// of the C99 type "long int". Note that this ONLY affects integers used
// internally within BLIS as well as those exposed in the native BLAS-like BLIS
// interface.
#ifndef BLIS_INT_TYPE_SIZE
  #ifdef BLIS_ARCH_64
    #define BLIS_INT_TYPE_SIZE 64
  #else
    #define BLIS_INT_TYPE_SIZE 32
  #endif
#endif


// -- FLOATING-POINT PROPERTIES ------------------------------------------------

// Enable use of built-in C99 "float complex" and "double complex" types and
// associated overloaded operations and functions? Disabling results in
// scomplex and dcomplex being defined in terms of simple structs.
// NOTE: AVOID USING THIS FEATURE. IT IS PROBABLY BROKEN.
#ifdef BLIS_ENABLE_C99_COMPLEX
  // No additional definitions needed.
#else
  // Default behavior is disabled.
#endif


// -- MEMORY SUBSYSTEM PROPERTIES ----------------------------------------------

// Size of a cache line (in bytes).
#ifndef BLIS_CACHE_LINE_SIZE
#define BLIS_CACHE_LINE_SIZE 64
#endif


// -- MULTITHREADING -----------------------------------------------------------

// Enable caching of queried cntx_t pointers in the gks?
#ifdef BLIS_DISABLE_GKS_CACHING
  #undef BLIS_ENABLE_GKS_CACHING
#else
  // Default behavior is enabled.
  #define BLIS_ENABLE_GKS_CACHING
#endif


// -- MULTITHREADING -----------------------------------------------------------

// Enable multithreading via POSIX threads.
#ifdef BLIS_ENABLE_PTHREADS
  // No additional definitions needed.
#else
  // Default behavior is disabled.
#endif

// Enable multithreading via OpenMP.
#ifdef BLIS_ENABLE_OPENMP
  // No additional definitions needed.
#else
  // Default behavior is disabled.
#endif

// Enable multithreading via HPX.
#ifdef BLIS_ENABLE_HPX
  // No additional definitions needed.
#else
  // Default behavior is disabled.
#endif

// Here, we define BLIS_ENABLE_MULTITHREADING if either OpenMP
// or pthreads are enabled. This macro is useful in situations when
// we want to detect use of either OpenMP or pthreads, or both (as
// opposed to neither being used).
#if defined ( BLIS_ENABLE_OPENMP )   || \
    defined ( BLIS_ENABLE_PTHREADS ) || \
    defined ( BLIS_ENABLE_HPX )
  #define BLIS_ENABLE_MULTITHREADING
#endif

// Enable the use of prime numbers of threads when requesting automatic thread
// factorization. When disabled, requesting a prime number of threads will
// result in a reduction (by one) of the number of threads, provided that the
// prime number exceeds a minimum threshold (see below).
#ifdef BLIS_ENABLE_AUTO_PRIME_NUM_THREADS
  #undef BLIS_DISABLE_AUTO_PRIME_NUM_THREADS
#else
  // Default behavior is disabled.
  #undef  BLIS_DISABLE_AUTO_PRIME_NUM_THREADS // In case user explicitly disabled.
  #define BLIS_DISABLE_AUTO_PRIME_NUM_THREADS
#endif

// Set the maximum requested number of threads that BLIS will accept from the
// user that may be prime. If a larger prime number of threads is requested,
// it will be reduced by one to allow for more efficient thread factorizations.
// This value will only be used if BLIS_ENABLE_AUTO_PRIME_NUM_THREADS is defined.
#ifndef BLIS_NT_MAX_PRIME
  #define BLIS_NT_MAX_PRIME 11
#endif


// -- MISCELLANEOUS OPTIONS ----------------------------------------------------

// Do NOT require the cross-blocksize constraints. That is, do not enforce
// MC % NR = 0 and NC % MR = 0 in bli_kernel_macro_defs.h. These are ONLY
// needed when implementing trsm_r by allowing the right-hand matrix B to
// be triangular.
#ifndef BLIS_RELAX_MCNR_NCMR_CONSTRAINTS
  #define BLIS_RELAX_MCNR_NCMR_CONSTRAINTS
#endif


// -- BLAS COMPATIBILITY LAYER -------------------------------------------------

// Enable the BLAS compatibility layer?
#ifdef BLIS_DISABLE_BLAS
  #undef BLIS_ENABLE_BLAS
#else
  // Default behavior is enabled.
  #undef  BLIS_ENABLE_BLAS // In case user explicitly enabled.
  #define BLIS_ENABLE_BLAS
#endif

// The bit size of the integer type used to track values such as dimensions and
// leading dimensions (ie: column strides) within the BLAS compatibility layer.
// A value of 32 results in the compatibility layer using 32-bit signed integers
// while 64 results in 64-bit integers. Any other value results in use of the
// C99 type "long int". Note that this ONLY affects integers used within the
// BLAS compatibility layer.
#ifndef BLIS_BLAS_INT_TYPE_SIZE
  #define BLIS_BLAS_INT_TYPE_SIZE 32
#endif

// By default, the level-3 BLAS routines are implemented by directly calling
// the BLIS object API. Alternatively, they may first call the typed BLIS
// API, which will then call the object API.
//#define BLIS_BLAS3_CALLS_TAPI
#ifdef BLIS_BLAS3_CALLS_TAPI
  #undef  BLIS_BLAS3_CALLS_OAPI
#else
  // Default behavior is to call object API directly.
  #undef  BLIS_BLAS3_CALLS_OAPI // In case user explicitly enabled.
  #define BLIS_BLAS3_CALLS_OAPI
#endif


// -- CBLAS COMPATIBILITY LAYER ------------------------------------------------

// Enable the CBLAS compatibility layer?
// NOTE: Enabling CBLAS will automatically enable the BLAS compatibility layer
// regardless of whether or not it was explicitly enabled above. Furthermore,
// the CBLAS compatibility layer will use the integer type size definition
// specified above when defining the size of its own integers (regardless of
// whether the BLAS layer was enabled directly or indirectly).
#ifdef BLIS_ENABLE_CBLAS
  // No additional definitions needed.
#else
  // Default behavior is disabled.
#endif


// -- SHARED LIBRARY SYMBOL EXPORT ---------------------------------------------

// When building shared libraries, we can control which symbols are exported for
// linking by external applications. BLIS annotates all function prototypes that
// are meant to be "public" with BLIS_EXPORT_BLIS (with BLIS_EXPORT_BLAS playing
// a similar role for BLAS compatibility routines). Which symbols are exported
// is controlled by the default symbol visibility, as specifed by the gcc option
// -fvisibility=[default|hidden]. The default for this option is 'default', or,
// "public", which, if allowed to stand, causes all symbols in BLIS to be
// linkable from the outside. But when compiling with -fvisibility=hidden, all
// symbols start out hidden (that is, restricted only for internal use by BLIS),
// with that setting overridden only for function prototypes or variable
// declarations that are annotated with BLIS_EXPORT_BLIS.

#ifndef BLIS_EXPORT
  #if !defined(BLIS_ENABLE_SHARED)
    #define BLIS_EXPORT
  #else
    #if defined(_WIN32) || defined(__CYGWIN__)
      #ifdef BLIS_IS_BUILDING_LIBRARY
        #define BLIS_EXPORT __declspec(dllexport)
      #else
        #define BLIS_EXPORT __declspec(dllimport)
      #endif
    #elif defined(__GNUC__) && __GNUC__ >= 4
      #define BLIS_EXPORT __attribute__ ((visibility ("default")))
    #else
      #define BLIS_EXPORT
    #endif
  #endif
#endif

#define BLIS_EXPORT_BLIS  BLIS_EXPORT
#define BLIS_EXPORT_BLAS  BLIS_EXPORT
#define BLIS_EXPORT_ADDON BLIS_EXPORT


// -- OVERRIDABLE (WEAK) SYMBOLS -----------------------------------------------

// On Linux, functions called from a shared library can be overriden by the main
// program simply by providing a new definition. However, macOS uses a "two-level
// namespace" which causes calls to shared library functions to be tied to the
// library and not overridable. As a workaround, certain symbols can be defined
// as "weak" and are given lower preference during linking.
#ifndef BLIS_OVERRIDABLE
#if BLIS_OS_OSX
#define BLIS_OVERRIDABLE __attribute__((weak))
#else
#define BLIS_OVERRIDABLE
#endif
#endif


// -- STATIC INLINE FUNCTIONS --------------------------------------------------

// C and C++ have different semantics for defining "inline" functions. In C,
// the keyword phrase "static inline" accomplishes this, though the "inline"
// is optional. In C++, the "inline" keyword is required and obviates "static"
// altogether. Why does this matter? While BLIS is compiled in C99, blis.h may
// be #included by a source file that is compiled with C++.
#ifdef __cplusplus
  #define BLIS_INLINE inline
#else
  //#define BLIS_INLINE static inline
  #define BLIS_INLINE static
#endif


#endif

// end bli_config_macro_defs.h
#line 69 "../blis-2.0/frame/include/blis.h"



// -- Common BLIS definitions --

// begin bli_type_defs.h
#line 1 "../blis-2.0/frame/include//bli_type_defs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2016, Hewlett Packard Enterprise Development LP
   Copyright (C) 2020, Advanced Micro Devices, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_TYPE_DEFS_H
#define BLIS_TYPE_DEFS_H


//
// -- BLIS basic types ---------------------------------------------------------
//

#ifdef __cplusplus
  // For C++, include stdint.h.
#include <cstdint> // skipped
#elif __STDC_VERSION__ >= 199901L
  // For C99 (or later), include stdint.h.
#include <stddef.h> // skipped
#include <stdint.h> // skipped
#include <stdbool.h> // skipped
#else
  // When stdint.h is not available, manually typedef the types we will use.
  #ifdef _WIN32
    typedef          __int32  int32_t;
    typedef unsigned __int32 uint32_t;
    typedef          __int64  int64_t;
    typedef unsigned __int64 uint64_t;
  #else
    #error "Attempting to compile on pre-C99 system without stdint.h."
  #endif
#endif

// -- General-purpose integers --

// If BLAS integers are 64 bits, mandate that BLIS integers also be 64 bits.
// NOTE: This cpp guard will only meaningfully change BLIS's behavior on
// systems where the BLIS integer size would have been automatically selected
// to be 32 bits, since explicit selection of 32 bits is prohibited at
// configure-time (and explicit or automatic selection of 64 bits is fine
// and would have had the same result).
#if BLIS_BLAS_INT_TYPE_SIZE == 64
  #undef  BLIS_INT_TYPE_SIZE
  #define BLIS_INT_TYPE_SIZE 64
#endif

// Define integer types depending on what size integer was requested.
#if   BLIS_INT_TYPE_SIZE == 32
typedef           int32_t  gint_t;
typedef          uint32_t guint_t;
#elif BLIS_INT_TYPE_SIZE == 64
typedef           int64_t  gint_t;
typedef          uint64_t guint_t;
#else
typedef   signed long int  gint_t;
typedef unsigned long int guint_t;
#endif

// -- Boolean type --

// NOTE: bool_t is no longer used and has been replaced with C99's bool type.
//typedef bool bool_t;

// BLIS uses TRUE and FALSE macro constants as possible boolean values, but we
// define these macros in terms of true and false, respectively, which are
// defined by C99 in stdbool.h.
#ifndef TRUE
  #define TRUE  true
#endif

#ifndef FALSE
  #define FALSE false
#endif

// -- Special-purpose integers --

// This cpp guard provides a temporary hack to allow libflame
// interoperability with BLIS.
#ifndef _DEFINED_DIM_T
#define _DEFINED_DIM_T
typedef   gint_t dim_t;      // dimension type
#endif
typedef   gint_t inc_t;      // increment/stride type
typedef   gint_t doff_t;     // diagonal offset type
typedef  guint_t siz_t;      // byte size type
typedef uint32_t objbits_t;  // object information bit field

// -- Real types --

// Define the number of floating-point types supported, and the size of the
// largest type.
#define BLIS_NUM_FP_TYPES   4
#define BLIS_MAX_TYPE_SIZE  sizeof(dcomplex)

// There are some places where we need to use sizeof() inside of a C
// preprocessor #if conditional, and so here we define the various sizes
// for those purposes.
#define BLIS_SIZEOF_S      4  // sizeof(float)
#define BLIS_SIZEOF_D      8  // sizeof(double)
#define BLIS_SIZEOF_C      8  // sizeof(scomplex)
#define BLIS_SIZEOF_Z      16 // sizeof(dcomplex)

// -- Complex types --

#if defined(__cplusplus) && defined(BLIS_ENABLE_STD_COMPLEX)

	} //extern "C"

#include <complex> // skipped

	// Typedef official C++ complex types to BLIS complex type names.

	// This cpp guard provides a temporary hack to allow libflame
	// interoperability with BLIS.
	#ifndef _DEFINED_SCOMPLEX
	#define _DEFINED_SCOMPLEX
	typedef std::complex<float> scomplex;
	#endif

	// This cpp guard provides a temporary hack to allow libflame
	// interoperability with BLIS.
	#ifndef _DEFINED_DCOMPLEX
	#define _DEFINED_DCOMPLEX
	typedef std::complex<double> dcomplex;
	#endif

	extern "C"
	{

#elif defined(BLIS_ENABLE_C99_COMPLEX)

	#if __STDC_VERSION__ >= 199901L
#include <complex.h> // skipped

		// Typedef official C99 complex types to BLIS complex type names.

		// This cpp guard provides a temporary hack to allow libflame
		// interoperability with BLIS.
		#ifndef _DEFINED_SCOMPLEX
		#define _DEFINED_SCOMPLEX
		typedef float complex scomplex;
		#endif

		// This cpp guard provides a temporary hack to allow libflame
		// interoperability with BLIS.
		#ifndef _DEFINED_DCOMPLEX
		#define _DEFINED_DCOMPLEX
		typedef double complex dcomplex;
		#endif
	#else
		#error "Configuration requested C99 complex types, but C99 does not appear to be supported."
	#endif

#else // ifndef BLIS_ENABLE_C99_COMPLEX

	// This cpp guard provides a temporary hack to allow libflame
	// interoperability with BLIS.
	#ifndef _DEFINED_SCOMPLEX
	#define _DEFINED_SCOMPLEX
	typedef struct scomplex
	{
		float  real;
		float  imag;
	} scomplex;
	#endif

	// This cpp guard provides a temporary hack to allow libflame
	// interoperability with BLIS.
	#ifndef _DEFINED_DCOMPLEX
	#define _DEFINED_DCOMPLEX
	typedef struct dcomplex
	{
		double real;
		double imag;
	} dcomplex;
	#endif

#endif // BLIS_ENABLE_C99_COMPLEX

// -- Atom type --

// Note: atom types are used to hold "bufferless" scalar object values. Note
// that it needs to be as large as the largest possible scalar value we might
// want to hold. Thus, for now, it is a dcomplex.
typedef dcomplex atom_t;

// -- Fortran-77 types --

// Note: These types are typically only used by BLAS compatibility layer, but
// we must define them even when the compatibility layer isn't being built
// because they also occur in bli_slamch() and bli_dlamch().

// Define f77_int depending on what size of integer was requested.
#if   BLIS_BLAS_INT_TYPE_SIZE == 32
typedef int32_t   f77_int;
#elif BLIS_BLAS_INT_TYPE_SIZE == 64
typedef int64_t   f77_int;
#else
typedef long int  f77_int;
#endif

typedef char      f77_char;
typedef float     f77_float;
typedef double    f77_double;
typedef scomplex  f77_scomplex;
typedef dcomplex  f77_dcomplex;

// -- Misc. function pointer types --

// Note: This type should be used in any situation where the address of a
// *function* will be conveyed or stored prior to it being typecast back
// to the correct function type. It does not need to be used when conveying
// or storing the address of *data* (such as an array of float or double).
//typedef void (*void_fp)( void );
typedef void* void_fp;

// Typedef function pointer types for malloc() and free() substitutes.
typedef void* (*malloc_ft)( size_t size );
typedef void  (*free_ft)  ( void*  p    );


//
// -- BLIS info bit field sizes ------------------------------------------------
//

#define BLIS_DATATYPE_NUM_BITS             ( BLIS_DOMAIN_NUM_BITS + BLIS_PRECISION_NUM_BITS )
#define   BLIS_DOMAIN_NUM_BITS               1
#define   BLIS_PRECISION_NUM_BITS            2
#define BLIS_CONJTRANS_NUM_BITS            ( BLIS_TRANS_NUM_BITS + BLIS_CONJ_NUM_BITS )
#define   BLIS_TRANS_NUM_BITS                1
#define   BLIS_CONJ_NUM_BITS                 1
#define BLIS_UPLO_NUM_BITS                 ( BLIS_UPPER_NUM_BITS + BLIS_DIAG_NUM_BITS + BLIS_LOWER_NUM_BITS )
#define   BLIS_UPPER_NUM_BITS                1
#define   BLIS_DIAG_NUM_BITS                 1
#define   BLIS_LOWER_NUM_BITS                1
#define BLIS_UNIT_DIAG_NUM_BITS            1
#define BLIS_INVERT_DIAG_NUM_BITS          1
#define BLIS_PACK_SCHEMA_NUM_BITS          ( BLIS_PACK_PANEL_NUM_BITS + BLIS_PACK_FORMAT_NUM_BITS + BLIS_PACK_NUM_BITS )
#define   BLIS_PACK_PANEL_NUM_BITS           1
#define   BLIS_PACK_FORMAT_NUM_BITS          4
#define   BLIS_PACK_NUM_BITS                 1
#define BLIS_PACK_REV_IF_UPPER_NUM_BITS    1
#define BLIS_PACK_REV_IF_LOWER_NUM_BITS    1
#define BLIS_PACK_BUFFER_NUM_BITS          2
#define BLIS_STRUC_NUM_BITS                2


//
// -- BLIS info bit field offsets ----------------------------------------------
//

#define BLIS_DATATYPE_SHIFT                0
#define   BLIS_DOMAIN_SHIFT              (   BLIS_DATATYPE_SHIFT )
#define   BLIS_PRECISION_SHIFT           (   BLIS_DOMAIN_SHIFT + BLIS_DOMAIN_NUM_BITS )
#define BLIS_CONJTRANS_SHIFT             ( BLIS_DATATYPE_SHIFT + BLIS_DATATYPE_NUM_BITS )
#define   BLIS_TRANS_SHIFT               (   BLIS_CONJTRANS_SHIFT )
#define   BLIS_CONJ_SHIFT                (   BLIS_TRANS_SHIFT + BLIS_TRANS_NUM_BITS )
#define BLIS_UPLO_SHIFT                  ( BLIS_CONJTRANS_SHIFT + BLIS_CONJTRANS_NUM_BITS )
#define   BLIS_UPPER_SHIFT               (   BLIS_UPLO_SHIFT )
#define   BLIS_DIAG_SHIFT                (   BLIS_UPPER_SHIFT + BLIS_UPPER_NUM_BITS )
#define   BLIS_LOWER_SHIFT               (   BLIS_DIAG_SHIFT + BLIS_DIAG_NUM_BITS )
#define BLIS_UNIT_DIAG_SHIFT             ( BLIS_UPLO_SHIFT + BLIS_UPLO_NUM_BITS )
#define BLIS_INVERT_DIAG_SHIFT           ( BLIS_UNIT_DIAG_SHIFT + BLIS_UNIT_DIAG_NUM_BITS )
#define BLIS_PACK_SCHEMA_SHIFT           ( BLIS_INVERT_DIAG_SHIFT + BLIS_INVERT_DIAG_NUM_BITS )
#define   BLIS_PACK_PANEL_SHIFT          (   BLIS_PACK_SCHEMA_SHIFT )
#define   BLIS_PACK_FORMAT_SHIFT         (   BLIS_PACK_PANEL_SHIFT + BLIS_PACK_PANEL_NUM_BITS )
#define   BLIS_PACK_SHIFT                (   BLIS_PACK_FORMAT_SHIFT + BLIS_PACK_FORMAT_NUM_BITS )
#define BLIS_PACK_REV_IF_UPPER_SHIFT     ( BLIS_PACK_SCHEMA_SHIFT + BLIS_PACK_SCHEMA_NUM_BITS )
#define BLIS_PACK_REV_IF_LOWER_SHIFT     ( BLIS_PACK_REV_IF_UPPER_SHIFT + BLIS_PACK_REV_IF_UPPER_NUM_BITS )
#define BLIS_PACK_BUFFER_SHIFT           ( BLIS_PACK_REV_IF_LOWER_SHIFT + BLIS_PACK_REV_IF_LOWER_NUM_BITS )
#define BLIS_STRUC_SHIFT                 ( BLIS_PACK_BUFFER_SHIFT + BLIS_PACK_BUFFER_NUM_BITS )
#define BLIS_COMP_PREC_SHIFT             ( BLIS_STRUC_SHIFT + BLIS_STRUC_NUM_BITS )
#define BLIS_SCALAR_DT_SHIFT             ( BLIS_COMP_PREC_SHIFT + BLIS_PRECISION_NUM_BITS )
#define   BLIS_SCALAR_DOMAIN_SHIFT       (   BLIS_SCALAR_DT_SHIFT )
#define   BLIS_SCALAR_PREC_SHIFT         (   BLIS_SCALAR_DOMAIN_SHIFT + BLIS_DOMAIN_NUM_BITS )
// This is the total number of bits, which should always be <= 32
#define BLIS_INFO_NUM_BITS               ( BLIS_SCALAR_DT_SHIFT + BLIS_DATATYPE_NUM_BITS )

//
// -- BLIS info bit field masks ------------------------------------------------
//

#define BLIS_DATATYPE_BITS                 ( ( ( 1 << BLIS_DATATYPE_NUM_BITS          ) - 1 ) << BLIS_DATATYPE_SHIFT )
#define   BLIS_DOMAIN_BIT                  ( ( ( 1 << BLIS_DOMAIN_NUM_BITS            ) - 1 ) << BLIS_DOMAIN_SHIFT )
#define   BLIS_PRECISION_BIT               ( ( ( 1 << BLIS_PRECISION_NUM_BITS         ) - 1 ) << BLIS_PRECISION_SHIFT )
#define BLIS_CONJTRANS_BITS                ( ( ( 1 << BLIS_CONJTRANS_NUM_BITS         ) - 1 ) << BLIS_CONJTRANS_SHIFT )
#define   BLIS_TRANS_BIT                   ( ( ( 1 << BLIS_TRANS_NUM_BITS             ) - 1 ) << BLIS_TRANS_SHIFT )
#define   BLIS_CONJ_BIT                    ( ( ( 1 << BLIS_CONJ_NUM_BITS              ) - 1 ) << BLIS_CONJ_SHIFT )
#define BLIS_UPLO_BITS                     ( ( ( 1 << BLIS_UPLO_NUM_BITS              ) - 1 ) << BLIS_UPLO_SHIFT )
#define   BLIS_UPPER_BIT                   ( ( ( 1 << BLIS_UPPER_NUM_BITS             ) - 1 ) << BLIS_UPPER_SHIFT )
#define   BLIS_DIAG_BIT                    ( ( ( 1 << BLIS_DIAG_NUM_BITS              ) - 1 ) << BLIS_DIAG_SHIFT )
#define   BLIS_LOWER_BIT                   ( ( ( 1 << BLIS_LOWER_NUM_BITS             ) - 1 ) << BLIS_LOWER_SHIFT )
#define BLIS_UNIT_DIAG_BIT                 ( ( ( 1 << BLIS_UNIT_DIAG_NUM_BITS         ) - 1 ) << BLIS_UNIT_DIAG_SHIFT )
#define BLIS_INVERT_DIAG_BIT               ( ( ( 1 << BLIS_INVERT_DIAG_NUM_BITS       ) - 1 ) << BLIS_INVERT_DIAG_SHIFT )
#define BLIS_PACK_SCHEMA_BITS              ( ( ( 1 << BLIS_PACK_SCHEMA_NUM_BITS       ) - 1 ) << BLIS_PACK_SCHEMA_SHIFT )
#define   BLIS_PACK_PANEL_BIT              ( ( ( 1 << BLIS_PACK_PANEL_NUM_BITS        ) - 1 ) << BLIS_PACK_PANEL_SHIFT )
#define   BLIS_PACK_FORMAT_BITS            ( ( ( 1 << BLIS_PACK_FORMAT_NUM_BITS       ) - 1 ) << BLIS_PACK_FORMAT_SHIFT )
#define   BLIS_PACK_BIT                    ( ( ( 1 << BLIS_PACK_NUM_BITS              ) - 1 ) << BLIS_PACK_SHIFT )
#define BLIS_PACK_REV_IF_UPPER_BIT         ( ( ( 1 << BLIS_PACK_REV_IF_UPPER_NUM_BITS ) - 1 ) << BLIS_PACK_REV_IF_UPPER_SHIFT )
#define BLIS_PACK_REV_IF_LOWER_BIT         ( ( ( 1 << BLIS_PACK_REV_IF_LOWER_NUM_BITS ) - 1 ) << BLIS_PACK_REV_IF_LOWER_SHIFT )
#define BLIS_PACK_BUFFER_BITS              ( ( ( 1 << BLIS_PACK_BUFFER_NUM_BITS       ) - 1 ) << BLIS_PACK_BUFFER_SHIFT )
#define BLIS_STRUC_BITS                    ( ( ( 1 << BLIS_STRUC_NUM_BITS             ) - 1 ) << BLIS_STRUC_SHIFT )
#define BLIS_COMP_PREC_BIT                 ( ( ( 1 << BLIS_PRECISION_NUM_BITS         ) - 1 ) << BLIS_COMP_PREC_SHIFT )
#define BLIS_SCALAR_DT_BITS                ( ( ( 1 << BLIS_DATATYPE_NUM_BITS          ) - 1 ) << BLIS_SCALAR_DT_SHIFT )
#define   BLIS_SCALAR_DOMAIN_BIT           ( ( ( 1 << BLIS_DOMAIN_NUM_BITS            ) - 1 ) << BLIS_SCALAR_DOMAIN_SHIFT )
#define   BLIS_SCALAR_PREC_BIT             ( ( ( 1 << BLIS_PRECISION_NUM_BITS         ) - 1 ) << BLIS_SCALAR_PREC_SHIFT )


//
// -- BLIS enumerated type value definitions -----------------------------------
//

#define BLIS_BITVAL_REAL                  0x0
#define BLIS_BITVAL_COMPLEX               BLIS_DOMAIN_BIT
#define BLIS_BITVAL_SINGLE_PREC           0x0
#define BLIS_BITVAL_DOUBLE_PREC         ( 0x1 << BLIS_PRECISION_SHIFT )
#define   BLIS_BITVAL_FLOAT_TYPE          0x0
#define   BLIS_BITVAL_SCOMPLEX_TYPE       BLIS_DOMAIN_BIT
#define   BLIS_BITVAL_DOUBLE_TYPE         BLIS_BITVAL_DOUBLE_PREC
#define   BLIS_BITVAL_DCOMPLEX_TYPE     ( BLIS_DOMAIN_BIT | BLIS_BITVAL_DOUBLE_PREC )
#define   BLIS_BITVAL_INT_TYPE            0x04
#define   BLIS_BITVAL_CONST_TYPE          0x05
#define BLIS_BITVAL_NO_TRANS              0x0
#define BLIS_BITVAL_TRANS                 BLIS_TRANS_BIT
#define BLIS_BITVAL_NO_CONJ               0x0
#define BLIS_BITVAL_CONJ                  BLIS_CONJ_BIT
#define BLIS_BITVAL_CONJ_TRANS          ( BLIS_CONJ_BIT | BLIS_TRANS_BIT )
#define BLIS_BITVAL_ZEROS                 0x0
#define BLIS_BITVAL_UPPER               ( BLIS_UPPER_BIT | BLIS_DIAG_BIT )
#define BLIS_BITVAL_LOWER               ( BLIS_LOWER_BIT | BLIS_DIAG_BIT )
#define BLIS_BITVAL_DENSE                 BLIS_UPLO_BITS
#define BLIS_BITVAL_NONUNIT_DIAG          0x0
#define BLIS_BITVAL_UNIT_DIAG             BLIS_UNIT_DIAG_BIT
#define BLIS_BITVAL_INVERT_DIAG           BLIS_INVERT_DIAG_BIT
#define BLIS_BITVAL_NOT_PACKED            0x0
#define   BLIS_BITVAL_1E                ( 0x1  << BLIS_PACK_FORMAT_SHIFT )
#define   BLIS_BITVAL_1R                ( 0x2  << BLIS_PACK_FORMAT_SHIFT )
#define   BLIS_BITVAL_RO                ( 0x3  << BLIS_PACK_FORMAT_SHIFT )
#define   BLIS_BITVAL_PACKED_UNSPEC     ( BLIS_PACK_BIT                                         )
#define   BLIS_BITVAL_PACKED_PANELS     ( BLIS_PACK_BIT                   | BLIS_PACK_PANEL_BIT )
#define   BLIS_BITVAL_PACKED_PANELS_1E  ( BLIS_PACK_BIT | BLIS_BITVAL_1E  | BLIS_PACK_PANEL_BIT )
#define   BLIS_BITVAL_PACKED_PANELS_1R  ( BLIS_PACK_BIT | BLIS_BITVAL_1R  | BLIS_PACK_PANEL_BIT )
#define   BLIS_BITVAL_PACKED_PANELS_RO  ( BLIS_PACK_BIT | BLIS_BITVAL_RO  | BLIS_PACK_PANEL_BIT )
#define BLIS_BITVAL_PACK_FWD_IF_UPPER     0x0
#define BLIS_BITVAL_PACK_REV_IF_UPPER     BLIS_PACK_REV_IF_UPPER_BIT
#define BLIS_BITVAL_PACK_FWD_IF_LOWER     0x0
#define BLIS_BITVAL_PACK_REV_IF_LOWER     BLIS_PACK_REV_IF_LOWER_BIT
#define BLIS_BITVAL_BUFFER_FOR_A_BLOCK    0x0
#define BLIS_BITVAL_BUFFER_FOR_B_PANEL  ( 0x1 << BLIS_PACK_BUFFER_SHIFT )
#define BLIS_BITVAL_BUFFER_FOR_C_PANEL  ( 0x2 << BLIS_PACK_BUFFER_SHIFT )
#define BLIS_BITVAL_BUFFER_FOR_GEN_USE  ( 0x3 << BLIS_PACK_BUFFER_SHIFT )
#define BLIS_BITVAL_GENERAL               0x0
#define BLIS_BITVAL_HERMITIAN           ( 0x1 << BLIS_STRUC_SHIFT )
#define BLIS_BITVAL_SYMMETRIC           ( 0x2 << BLIS_STRUC_SHIFT )
#define BLIS_BITVAL_TRIANGULAR          ( 0x3 << BLIS_STRUC_SHIFT )


//
// -- BLIS enumerated type definitions -----------------------------------------
//

// -- Operational parameter types --

typedef enum
{
	BLIS_NO_TRANSPOSE      = 0x0,
	BLIS_TRANSPOSE         = BLIS_BITVAL_TRANS,
	BLIS_CONJ_NO_TRANSPOSE = BLIS_BITVAL_CONJ,
	BLIS_CONJ_TRANSPOSE    = BLIS_BITVAL_CONJ_TRANS
} trans_t;

typedef enum
{
	BLIS_NO_CONJUGATE      = 0x0,
	BLIS_CONJUGATE         = BLIS_BITVAL_CONJ
} conj_t;

typedef enum
{
	BLIS_ZEROS             = BLIS_BITVAL_ZEROS,
	BLIS_LOWER             = BLIS_BITVAL_LOWER,
	BLIS_UPPER             = BLIS_BITVAL_UPPER,
	BLIS_DENSE             = BLIS_BITVAL_DENSE
} uplo_t;

typedef enum
{
	BLIS_LEFT              = 0x0,
	BLIS_RIGHT
} side_t;

typedef enum
{
	BLIS_NONUNIT_DIAG      = 0x0,
	BLIS_UNIT_DIAG         = BLIS_BITVAL_UNIT_DIAG
} diag_t;

typedef enum
{
	BLIS_NO_INVERT_DIAG    = 0x0,
	BLIS_INVERT_DIAG       = BLIS_BITVAL_INVERT_DIAG
} invdiag_t;

typedef enum
{
	BLIS_GENERAL           = BLIS_BITVAL_GENERAL,
	BLIS_HERMITIAN         = BLIS_BITVAL_HERMITIAN,
	BLIS_SYMMETRIC         = BLIS_BITVAL_SYMMETRIC,
	BLIS_TRIANGULAR        = BLIS_BITVAL_TRIANGULAR
} struc_t;


// -- Data type --

typedef enum
{
	BLIS_FLOAT             = BLIS_BITVAL_FLOAT_TYPE,
	BLIS_DOUBLE            = BLIS_BITVAL_DOUBLE_TYPE,
	BLIS_SCOMPLEX          = BLIS_BITVAL_SCOMPLEX_TYPE,
	BLIS_DCOMPLEX          = BLIS_BITVAL_DCOMPLEX_TYPE,
	BLIS_INT               = BLIS_BITVAL_INT_TYPE,
	BLIS_CONSTANT          = BLIS_BITVAL_CONST_TYPE,
	BLIS_DT_LO             = BLIS_FLOAT,
	BLIS_DT_HI             = BLIS_DCOMPLEX
} num_t;

typedef enum
{
	BLIS_REAL              = BLIS_BITVAL_REAL,
	BLIS_COMPLEX           = BLIS_BITVAL_COMPLEX
} dom_t;

typedef enum
{
	BLIS_SINGLE_PREC       = BLIS_BITVAL_SINGLE_PREC,
	BLIS_DOUBLE_PREC       = BLIS_BITVAL_DOUBLE_PREC
} prec_t;


// -- Pack schema type --

typedef enum
{
	BLIS_NOT_PACKED       = BLIS_BITVAL_NOT_PACKED,
	BLIS_PACKED_UNSPEC    = BLIS_BITVAL_PACKED_UNSPEC,
	BLIS_PACKED_VECTOR    = BLIS_BITVAL_PACKED_UNSPEC,
	BLIS_PACKED_MATRIX    = BLIS_BITVAL_PACKED_UNSPEC,
	BLIS_PACKED_PANELS    = BLIS_BITVAL_PACKED_PANELS,
	BLIS_PACKED_PANELS_1E = BLIS_BITVAL_PACKED_PANELS_1E,
	BLIS_PACKED_PANELS_1R = BLIS_BITVAL_PACKED_PANELS_1R,
	BLIS_PACKED_PANELS_RO = BLIS_BITVAL_PACKED_PANELS_RO,

	// BLIS_NUM_PACK_SCHEMA_TYPES must be last!
	// We start with BLIS_PACKED_PANELS.
	BLIS_NUM_PACK_SCHEMA_TYPES_,
	BLIS_NUM_PACK_SCHEMA_TYPES = ((( BLIS_NUM_PACK_SCHEMA_TYPES_ - BLIS_PACKED_PANELS - 1 ) >> BLIS_PACK_FORMAT_SHIFT ) + 1)
} pack_t;


// -- Pack order type --

typedef enum
{
	BLIS_PACK_FWD_IF_UPPER = BLIS_BITVAL_PACK_FWD_IF_UPPER,
	BLIS_PACK_REV_IF_UPPER = BLIS_BITVAL_PACK_REV_IF_UPPER,

	BLIS_PACK_FWD_IF_LOWER = BLIS_BITVAL_PACK_FWD_IF_LOWER,
	BLIS_PACK_REV_IF_LOWER = BLIS_BITVAL_PACK_REV_IF_LOWER
} packord_t;


// -- Pack buffer type --

typedef enum
{
	BLIS_BUFFER_FOR_A_BLOCK = BLIS_BITVAL_BUFFER_FOR_A_BLOCK,
	BLIS_BUFFER_FOR_B_PANEL = BLIS_BITVAL_BUFFER_FOR_B_PANEL,
	BLIS_BUFFER_FOR_C_PANEL = BLIS_BITVAL_BUFFER_FOR_C_PANEL,
	BLIS_BUFFER_FOR_GEN_USE = BLIS_BITVAL_BUFFER_FOR_GEN_USE
} packbuf_t;


// -- Partitioning direction --

typedef enum
{
	BLIS_FWD,
	BLIS_BWD
} dir_t;


// -- Subpartition type --

typedef enum
{
	BLIS_SUBPART0,
	BLIS_SUBPART1,
	BLIS_SUBPART2,
	BLIS_SUBPART1AND0,
	BLIS_SUBPART1AND2,
	BLIS_SUBPART1A,
	BLIS_SUBPART1B,
	BLIS_SUBPART00,
	BLIS_SUBPART10,
	BLIS_SUBPART20,
	BLIS_SUBPART01,
	BLIS_SUBPART11,
	BLIS_SUBPART21,
	BLIS_SUBPART02,
	BLIS_SUBPART12,
	BLIS_SUBPART22
} subpart_t;


// -- Matrix dimension type --

typedef enum
{
	BLIS_M = 0,
	BLIS_N = 1
} mdim_t;


// -- Machine parameter types --

typedef enum
{
	BLIS_MACH_EPS = 0,
	BLIS_MACH_SFMIN,
	BLIS_MACH_BASE,
	BLIS_MACH_PREC,
	BLIS_MACH_NDIGMANT,
	BLIS_MACH_RND,
	BLIS_MACH_EMIN,
	BLIS_MACH_RMIN,
	BLIS_MACH_EMAX,
	BLIS_MACH_RMAX,
	BLIS_MACH_EPS2,

	// BLIS_NUM_MACH_PARAMS must be last!
	BLIS_NUM_MACH_PARAMS
} machval_t;

#define BLIS_MACH_PARAM_FIRST  BLIS_MACH_EPS
#define BLIS_MACH_PARAM_LAST   BLIS_MACH_EPS2


// -- Induced method types --

typedef enum
{
	BLIS_1M        = 0,
	BLIS_NAT,

	BLIS_IND_FIRST = 0,
	BLIS_IND_LAST  = BLIS_NAT,

	// BLIS_NUM_IND_METHODS must be last!
	BLIS_NUM_IND_METHODS
} ind_t;

// These are used in bli_l3_*_oapi.c to construct the ind_t values from
// the induced method substrings that go into function names.
#define bli_1m   BLIS_1M
#define bli_nat  BLIS_NAT


// -- Threading implementation type --

typedef enum
{
	BLIS_SINGLE = 0,
	BLIS_OPENMP,
	BLIS_POSIX,
	BLIS_HPX,

	// BLIS_NUM_THREAD_IMPLS must be last!
	BLIS_NUM_THREAD_IMPLS

} timpl_t;


// -- Kernel ID types --

// Encode the number of independent type parameters in the high
// bits of the kernel ID. This lets us identify kernel IDs as the
// appropriate type while also using them as linear indices after
// masking out these bits.
#define BLIS_NTYPE_KER_SHIFT 28
#define BLIS_NTYPE_KER_BITS  (0xFu << BLIS_NTYPE_KER_SHIFT)
#define BLIS_1TYPE_KER       (  0u << BLIS_NTYPE_KER_SHIFT)
#define BLIS_2TYPE_KER       (  1u << BLIS_NTYPE_KER_SHIFT)
#define BLIS_3TYPE_KER       (  2u << BLIS_NTYPE_KER_SHIFT)

#define bli_ker_idx( ker )   ( kerid_t )((ker) & ~BLIS_NTYPE_KER_BITS)
#define bli_ker_ntype( ker ) ((((ker) & BLIS_NTYPE_KER_BITS) >> BLIS_NTYPE_KER_SHIFT) + 1)

// We have to use a 32-bit type here to avoid problems with passing small enum
// constants to variadic functions. See https://github.com/flame/blis/issues/839.
typedef uint32_t kerid_t;

// Sentinel constant used to indicate the end of a variable argument function
// (See bli_cntx.c)

#define BLIS_VA_END  ((kerid_t)-1)

typedef enum
{
	// -- Single-type kernels --

	// l1v kernels
	BLIS_ADDV_KER = BLIS_1TYPE_KER,
	BLIS_AMAXV_KER,
	BLIS_AXPBYV_KER,
	BLIS_AXPYV_KER,
	BLIS_COPYV_KER,
	BLIS_DOTV_KER,
	BLIS_DOTXV_KER,
	BLIS_INVERTV_KER,
	BLIS_INVSCALV_KER,
	BLIS_SCALV_KER,
	BLIS_SCAL2V_KER,
	BLIS_SETV_KER,
	BLIS_SUBV_KER,
	BLIS_SWAPV_KER,
	BLIS_XPBYV_KER,
	BLIS_AXPY2V_KER,
	BLIS_DOTAXPYV_KER,

	// l1f kernels
	BLIS_AXPYF_KER,
	BLIS_DOTXF_KER,
	BLIS_DOTXAXPYF_KER,

	// l3 native kernels
	BLIS_GEMMTRSM_L_UKR,
	BLIS_GEMMTRSM_U_UKR,
	BLIS_TRSM_L_UKR,
	BLIS_TRSM_U_UKR,

	// l3 1m kernels
	BLIS_GEMMTRSM1M_L_UKR,
	BLIS_GEMMTRSM1M_U_UKR,

	// gemmsup kernels
	BLIS_GEMMSUP_RRR_UKR,
	BLIS_GEMMSUP_RRC_UKR,
	BLIS_GEMMSUP_RCR_UKR,
	BLIS_GEMMSUP_RCC_UKR,
	BLIS_GEMMSUP_CRR_UKR,
	BLIS_GEMMSUP_CRC_UKR,
	BLIS_GEMMSUP_CCR_UKR,
	BLIS_GEMMSUP_CCC_UKR,
	BLIS_GEMMSUP_XXX_UKR,

	// BLIS_NUM_UKRS must after all 1-type kernels and before 2-type kernels!
	BLIS_NUM_UKRS_, BLIS_NUM_UKRS = bli_ker_idx( BLIS_NUM_UKRS_ ),

	// -- Two-type kernels --

	// pack kernels
	BLIS_PACKM_KER = BLIS_2TYPE_KER,
	BLIS_PACKM_1ER_KER,
	BLIS_PACKM_RO_KER,
	BLIS_PACKM_DIAG_KER,
	BLIS_PACKM_DIAG_1ER_KER,
	BLIS_PACKM_DIAG_RO_KER,

	// unpack kernels
	BLIS_UNPACKM_KER,

	// l3 native kernels
	BLIS_GEMM_UKR,

	// l3 1m kernels
	BLIS_GEMM1M_UKR,

	// mixed-domain kernels
	BLIS_GEMM_CCR_UKR,
	BLIS_GEMM_RCC_UKR,
	BLIS_GEMM_CRR_UKR,

	// BLIS_NUM_UKR2S must come after all kernels!
	BLIS_NUM_UKR2S_, BLIS_NUM_UKR2S = bli_ker_idx( BLIS_NUM_UKR2S_ ),

	// Force the size of ukr_t values to be as large as kerid_t
	BLIS_UKRS_END_ = BLIS_VA_END
} ukr_t;


typedef enum
{
    // l3 kernel row preferences
	BLIS_GEMM_UKR_ROW_PREF,
	BLIS_GEMMTRSM_L_UKR_ROW_PREF,
	BLIS_GEMMTRSM_U_UKR_ROW_PREF,
	BLIS_TRSM_L_UKR_ROW_PREF,
	BLIS_TRSM_U_UKR_ROW_PREF,

    // gemmsup kernel row preferences
	BLIS_GEMMSUP_RRR_UKR_ROW_PREF,
	BLIS_GEMMSUP_RRC_UKR_ROW_PREF,
	BLIS_GEMMSUP_RCR_UKR_ROW_PREF,
	BLIS_GEMMSUP_RCC_UKR_ROW_PREF,
	BLIS_GEMMSUP_CRR_UKR_ROW_PREF,
	BLIS_GEMMSUP_CRC_UKR_ROW_PREF,
	BLIS_GEMMSUP_CCR_UKR_ROW_PREF,
	BLIS_GEMMSUP_CCC_UKR_ROW_PREF,
	BLIS_GEMMSUP_XXX_UKR_ROW_PREF,

    // BLIS_NUM_UKR_PREFS must be last!
    BLIS_NUM_UKR_PREFS,

	// Force the size of ukr_pref_t values to be as large as kerid_t
	BLIS_UKR_PREFS_END_ = BLIS_VA_END
} ukr_pref_t;

typedef enum
{
	BLIS_REFERENCE_UKERNEL = 0,
	BLIS_VIRTUAL_UKERNEL,
	BLIS_OPTIMIZED_UKERNEL,
	BLIS_NOTAPPLIC_UKERNEL,

    // BLIS_NUM_UKR_IMPL_TYPES must be last!
	BLIS_NUM_UKR_IMPL_TYPES
} kimpl_t;


#if 0
typedef enum
{
	// RV = row-stored, contiguous vector-loading
	// RG = row-stored, non-contiguous gather-loading
	// CV = column-stored, contiguous vector-loading
	// CG = column-stored, non-contiguous gather-loading

	// RD = row-stored, dot-based
	// CD = col-stored, dot-based

	// RC = row-stored, column-times-column
	// CR = column-stored, row-times-row

	// GX = general-stored generic implementation

	BLIS_GEMMSUP_RV_UKR = 0,
	BLIS_GEMMSUP_RG_UKR,
	BLIS_GEMMSUP_CV_UKR,
	BLIS_GEMMSUP_CG_UKR,

	BLIS_GEMMSUP_RD_UKR,
	BLIS_GEMMSUP_CD_UKR,

	BLIS_GEMMSUP_RC_UKR,
	BLIS_GEMMSUP_CR_UKR,

	BLIS_GEMMSUP_GX_UKR,

	// BLIS_NUM_LEVEL3_SUP_UKRS must be last!
	BLIS_NUM_LEVEL3_SUP_UKRS
} l3sup_t;
#endif


typedef enum
{
	// 3-operand storage combinations
	BLIS_RRR = 0,
	BLIS_RRC, // 1
	BLIS_RCR, // 2
	BLIS_RCC, // 3
	BLIS_CRR, // 4
	BLIS_CRC, // 5
	BLIS_CCR, // 6
	BLIS_CCC, // 7
	BLIS_XXX, // 8

#if 0
	BLIS_RRG,
	BLIS_RCG,
	BLIS_RGR,
	BLIS_RGC,
	BLIS_RGG,
	BLIS_CRG,
	BLIS_CCG,
	BLIS_CGR,
	BLIS_CGC,
	BLIS_CGG,
	BLIS_GRR,
	BLIS_GRC,
	BLIS_GRG,
	BLIS_GCR,
	BLIS_GCC,
	BLIS_GCG,
	BLIS_GGR,
	BLIS_GGC,
	BLIS_GGG,
#endif

	// BLIS_NUM_3OP_RC_COMBOS must be last!
	BLIS_NUM_3OP_RC_COMBOS
} stor3_t;


#if 0
typedef enum
{
	BLIS_JC_IDX = 0,
	BLIS_PC_IDX,
	BLIS_IC_IDX,
	BLIS_JR_IDX,
	BLIS_IR_IDX,
	BLIS_PR_IDX
} thridx_t;
#endif

#define BLIS_NUM_LOOPS 6


// -- Operation ID type --

typedef enum
{
//
// NOTE: If/when additional type values are added to this enum,
// you must either:
// - keep the level-3 values (starting with _GEMM) beginning at
//   index 0; or
// - if the value range is moved such that it does not begin at
//   index 0, implement something like a BLIS_OPID_LEVEL3_RANGE_START
//   value that can be subtracted from the opid_t value to map it
//   to a zero-based range.
// This is needed because these level-3 opid_t values are used in
// bli_l3_ind.c to index into arrays.
//
	BLIS_GEMM = 0,
	BLIS_GEMMT,
	BLIS_HEMM,
	BLIS_HERK,
	BLIS_HER2K,
	BLIS_SYMM,
	BLIS_SYRK,
	BLIS_SYR2K,
	BLIS_TRMM3,
	BLIS_TRMM,
	BLIS_TRSM,

	// BLIS_NOID (= BLIS_NUM_LEVEL3_OPS) must be last!
	BLIS_NOID,
	BLIS_NUM_LEVEL3_OPS = BLIS_NOID,

	// Force the size of opid_t values to be as large as kerid_t
	BLIS_LEVEL3_OPS_END_ = BLIS_VA_END
} opid_t;


// -- Blocksize ID type --

typedef enum
{
	// NOTE: the level-3 blocksizes MUST be indexed starting at zero.
	// At one point, we made this assumption in bli_cntx_set_blkszs()
	// and friends.
	BLIS_KR,
	BLIS_MR,
	BLIS_NR,
	BLIS_MC,
	BLIS_KC,
	BLIS_NC,

	// broadcast factors for packing
	BLIS_BBM,
	BLIS_BBN,

	// level-2 blocksizes
	BLIS_M2, // level-2 blocksize in m dimension
	BLIS_N2, // level-2 blocksize in n dimension

	// level-1f blocksizes
	BLIS_AF, // level-1f axpyf fusing factor
	BLIS_DF, // level-1f dotxf fusing factor
	BLIS_XF, // level-1f dotxaxpyf fusing factor

	// gemmsup thresholds
	BLIS_MT, // level-3 small/unpacked matrix threshold in m dimension
	BLIS_NT, // level-3 small/unpacked matrix threshold in n dimension
	BLIS_KT, // level-3 small/unpacked matrix threshold in k dimension

	// gemmsup block sizes
	BLIS_KR_SUP,
	BLIS_MR_SUP,
	BLIS_NR_SUP,
	BLIS_MC_SUP,
	BLIS_KC_SUP,
	BLIS_NC_SUP,

	// BLIS_NO_PART (= BLIS_NUM_BLKSZS) must be last!
	BLIS_NO_PART, // used as a placeholder when blocksizes are not applicable,
	              // such as when characterizing a packm operation.
	BLIS_NUM_BLKSZS = BLIS_NO_PART,

	// Force the size of bszid_t values to be as large as kerid_t
	BLIS_BLKSZS_END_ = BLIS_VA_END
} bszid_t;


// A convenient version of the BLIS_XX block size IDs which can be used in bitfields.
enum
{
	BLIS_THREAD_NONE = 0,
	BLIS_THREAD_KR   = 1 << BLIS_KR,
	BLIS_THREAD_MR   = 1 << BLIS_MR,
	BLIS_THREAD_NR   = 1 << BLIS_NR,
	BLIS_THREAD_MC   = 1 << BLIS_MC,
	BLIS_THREAD_KC   = 1 << BLIS_KC,
	BLIS_THREAD_NC   = 1 << BLIS_NC,
};

// -- Architecture ID type --

// NOTE: This typedef enum must be kept up-to-date with the arch_t
// string array in bli_arch.c. Whenever values are added/inserted
// OR if values are rearranged, be sure to update the string array
// in bli_arch.c.

typedef enum
{
	// NOTE: The C language standard guarantees that the first enum value
	// starts at 0.

	// Intel
	BLIS_ARCH_SKX,
	BLIS_ARCH_KNL,
	BLIS_ARCH_KNC,
	BLIS_ARCH_HASWELL,
	BLIS_ARCH_SANDYBRIDGE,
	BLIS_ARCH_PENRYN,

	// AMD
	BLIS_ARCH_ZEN3,
	BLIS_ARCH_ZEN2,
	BLIS_ARCH_ZEN,
	BLIS_ARCH_EXCAVATOR,
	BLIS_ARCH_STEAMROLLER,
	BLIS_ARCH_PILEDRIVER,
	BLIS_ARCH_BULLDOZER,

	// ARM-SVE
	BLIS_ARCH_ARMSVE,
	BLIS_ARCH_A64FX,

	// ARM-NEON (4 pipes x 128-bit vectors)
	BLIS_ARCH_ALTRAMAX,
	BLIS_ARCH_ALTRA,
	BLIS_ARCH_FIRESTORM,

	// ARM (2 pipes x 128-bit vectors)
	BLIS_ARCH_THUNDERX2,
	BLIS_ARCH_CORTEXA57,
	BLIS_ARCH_CORTEXA53,

	// ARM 32-bit (vintage)
	BLIS_ARCH_CORTEXA15,
	BLIS_ARCH_CORTEXA9,

	// IBM/Power
	BLIS_ARCH_POWER10,
	BLIS_ARCH_POWER9,
	BLIS_ARCH_POWER7,
	BLIS_ARCH_BGQ,

	// RISC-V
	BLIS_ARCH_RV32I,
	BLIS_ARCH_RV64I,
	BLIS_ARCH_RV32IV,
	BLIS_ARCH_RV64IV,

	// SiFive
	BLIS_ARCH_SIFIVE_X280,

	// Generic architecture/configuration
	BLIS_ARCH_GENERIC,

	// The total number of defined architectures. This must be last in the
	// list of enums since its definition assumes that the previous enum
	// value (BLIS_ARCH_GENERIC) is given index num_archs-1.
	BLIS_NUM_ARCHS

} arch_t;


//
// -- BLIS misc. structure types -----------------------------------------------
//

// This header must be included here (or earlier) because definitions it
// provides are needed in the pool_t and related structs.
// begin bli_pthread.h
#line 1 "../blis-2.0/frame/thread//bli_pthread.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2018, Southern Methodist University
   Copyright (C) 2018, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_PTHREAD_H
#define BLIS_PTHREAD_H

// -- Type and macro definitions -----------------------------------------------

#if defined(BLIS_DISABLE_SYSTEM)

// This branch defines a pthread-like API, bli_pthread_*(), and implements it
// in terms of "dummy" code that doesn't depend on POSIX threads or any other
// threading mechanism. See issue #454 to see the use case that prompted this
// feature.
// NOTE: THIS CODE DOES NOT IMPLEMENT THREADING AND IS NOT THREAD-SAFE!

// -- pthread types --

typedef int bli_pthread_t;
typedef int bli_pthread_attr_t;
typedef int bli_pthread_mutex_t;
typedef int bli_pthread_mutexattr_t;
typedef int bli_pthread_cond_t;
typedef int bli_pthread_condattr_t;
typedef int bli_pthread_once_t;

typedef int bli_pthread_barrier_t;
typedef int bli_pthread_barrierattr_t;

// -- pthreads macros --

#define BLIS_PTHREAD_MUTEX_INITIALIZER 0
#define BLIS_PTHREAD_COND_INITIALIZER  0
#define BLIS_PTHREAD_ONCE_INIT         0

#elif defined(_MSC_VER) // !defined(BLIS_DISABLE_SYSTEM)

// This branch defines a pthread-like API, bli_pthread_*(), and implements it
// in terms of Windows API calls.

// -- pthread types --

typedef struct
{
    HANDLE handle;
    void* retval;
} bli_pthread_t;
typedef void bli_pthread_attr_t;
typedef SRWLOCK bli_pthread_mutex_t;
typedef void bli_pthread_mutexattr_t;
typedef CONDITION_VARIABLE bli_pthread_cond_t;
typedef void bli_pthread_condattr_t;
typedef INIT_ONCE bli_pthread_once_t;
typedef struct
{
    bli_pthread_mutex_t mutex;
    bli_pthread_cond_t  cond;
    int                 count;
    int                 tripCount;
} bli_pthread_barrier_t;
typedef void bli_pthread_barrierattr_t;

// -- pthreads macros --

#define BLIS_PTHREAD_MUTEX_INITIALIZER SRWLOCK_INIT
#define BLIS_PTHREAD_ONCE_INIT         INIT_ONCE_STATIC_INIT
#define BLIS_PTHREAD_COND_INITIALIZER  CONDITION_VARIABLE_INIT

#else // !defined(BLIS_DISABLE_SYSTEM) && !defined(_MSC_VER)

#include <pthread.h> // skipped

// This branch defines a pthreads-like API, bli_pthreads_*(), and implements it
// in terms of the corresponding pthreads_*() types, macros, and function calls. 

// -- pthread types --

typedef pthread_t           bli_pthread_t;
typedef pthread_attr_t      bli_pthread_attr_t;
typedef pthread_mutex_t     bli_pthread_mutex_t;
typedef pthread_mutexattr_t bli_pthread_mutexattr_t;
typedef pthread_cond_t      bli_pthread_cond_t;
typedef pthread_condattr_t  bli_pthread_condattr_t;
typedef pthread_once_t      bli_pthread_once_t;

#if defined(__APPLE__)

// For OS X, we must define the barrier types ourselves since Apple does
// not implement barriers in their variant of pthreads.

typedef void bli_pthread_barrierattr_t;

typedef struct
{
	bli_pthread_mutex_t mutex;
	bli_pthread_cond_t  cond;
	int                 count;
	int                 tripCount;
} bli_pthread_barrier_t;

#else

// For other non-Windows OSes (primarily Linux), we can define the barrier
// types in terms of existing pthreads barrier types since we expect they
// will be provided by the pthreads implementation.

typedef pthread_barrier_t      bli_pthread_barrier_t;
typedef pthread_barrierattr_t  bli_pthread_barrierattr_t;

#endif

// -- pthreads macros --

#define BLIS_PTHREAD_MUTEX_INITIALIZER PTHREAD_MUTEX_INITIALIZER
#define BLIS_PTHREAD_COND_INITIALIZER  PTHREAD_COND_INITIALIZER
#define BLIS_PTHREAD_ONCE_INIT         PTHREAD_ONCE_INIT

#endif

// -- Function definitions -----------------------------------------------------

// -- pthread_create(), pthread_join() --

BLIS_EXPORT_BLIS int bli_pthread_create
     (
       bli_pthread_t*            thread,
       const bli_pthread_attr_t* attr,
       void*                   (*start_routine)(void*),
       void*                     arg
     );

BLIS_EXPORT_BLIS int bli_pthread_join
     (
       bli_pthread_t thread,
       void**        retval
     );

// -- pthread_mutex_*() --

BLIS_EXPORT_BLIS int bli_pthread_mutex_init
     (
       bli_pthread_mutex_t*           mutex,
       const bli_pthread_mutexattr_t* attr
     );

BLIS_EXPORT_BLIS int bli_pthread_mutex_destroy
     (
       bli_pthread_mutex_t* mutex
     );

BLIS_EXPORT_BLIS int bli_pthread_mutex_lock
     (
       bli_pthread_mutex_t* mutex
     );

BLIS_EXPORT_BLIS int bli_pthread_mutex_trylock
     (
       bli_pthread_mutex_t* mutex
     );

BLIS_EXPORT_BLIS int bli_pthread_mutex_unlock
     (
       bli_pthread_mutex_t* mutex
     );

// -- pthread_cond_*() --

BLIS_EXPORT_BLIS int bli_pthread_cond_init
     (
       bli_pthread_cond_t*           cond,
       const bli_pthread_condattr_t* attr
     );

BLIS_EXPORT_BLIS int bli_pthread_cond_destroy
     (
       bli_pthread_cond_t* cond
     );

BLIS_EXPORT_BLIS int bli_pthread_cond_wait
     (
       bli_pthread_cond_t*  cond,
       bli_pthread_mutex_t* mutex
     );

BLIS_EXPORT_BLIS int bli_pthread_cond_broadcast
     (
       bli_pthread_cond_t* cond
     );

// -- pthread_once() --

BLIS_EXPORT_BLIS void bli_pthread_once
     (
       bli_pthread_once_t* once,
       void              (*init)(void)
     );

#if 0
// NOTE: This part of the API is disabled because (1) we don't actually need
// _self() or _equal() yet, and (2) when we do try to include these functions,
// AppVeyor for some reason fails on all the Windows/clang builds with the
// error:
//    libblis.a(bli_pthread.o) : error LNK2019: unresolved external symbol
//     __imp_CompareObjectHandles referenced in function bli_pthread_equal

// -- pthread_self() --

BLIS_EXPORT_BLIS bli_pthread_t bli_pthread_self
     (
       void
     );

// -- pthread_equal() --

BLIS_EXPORT_BLIS int bli_pthread_equal
     (
       bli_pthread_t t1,
       bli_pthread_t t2
     );
#endif

// -- pthread_barrier_*() --

BLIS_EXPORT_BLIS int bli_pthread_barrier_init
     (
       bli_pthread_barrier_t*           barrier,
       const bli_pthread_barrierattr_t* attr,
       unsigned int                     count
     );

BLIS_EXPORT_BLIS int bli_pthread_barrier_destroy
     (
       bli_pthread_barrier_t* barrier
     );

BLIS_EXPORT_BLIS int bli_pthread_barrier_wait
     (
       bli_pthread_barrier_t* barrier
     );

// -- Non-standard extensions --------------------------------------------------

// -- pthread_switch --

typedef struct
{
    int                 status;
    bli_pthread_mutex_t mutex;
} bli_pthread_switch_t;

#define BLIS_PTHREAD_SWITCH_INIT { .status = 0, \
                                   .mutex  = BLIS_PTHREAD_MUTEX_INITIALIZER }

int bli_pthread_switch_on
     (
       bli_pthread_switch_t* sw,
       int                 (*init)(void)
     );

int bli_pthread_switch_off
     (
       bli_pthread_switch_t* sw,
       int                 (*deinit)(void)
     );

#endif // BLIS_PTHREAD_H
// end bli_pthread.h
#line 1030 "../blis-2.0/frame/include//bli_type_defs.h"


// -- Pool block type --

typedef struct
{
	void*     buf;
	siz_t     block_size;

} pblk_t;


// -- Pool type --

typedef struct
{
	void*     block_ptrs;
	dim_t     block_ptrs_len;

	dim_t     top_index;
	dim_t     num_blocks;

	siz_t     block_size;
	siz_t     align_size;
	siz_t     offset_size;

	malloc_ft malloc_fp;
	free_ft   free_fp;

} pool_t;


// -- Array type --

typedef struct
{
	void*     buf;

	siz_t     num_elem;
	siz_t     elem_size;

} array_t;


// -- Locked pool-of-arrays-of-pools type --

typedef struct
{
	bli_pthread_mutex_t mutex;
	pool_t              pool;

	siz_t               def_array_len;

} apool_t;


// -- packing block allocator: Locked set of pools type --

typedef struct pba_s
{
	pool_t              pools[3];
	bli_pthread_mutex_t mutex;

	// These fields are used for general-purpose allocation.
	siz_t               align_size;
	malloc_ft           malloc_fp;
	free_ft             free_fp;

} pba_t;


// -- Memory object type --

typedef struct mem_s
{
	pblk_t    pblk;
	packbuf_t buf_type;
	pool_t*   pool;
	siz_t     size;
} mem_t;


// -- Control tree node type --

#define BLIS_MAX_SUB_NODES 2

struct cntl_s
{
	// Actually this is a l3_var_oft, but that type hasn't been defined yet
	void_fp var_func;
	struct
	{
		dim_t          ways;
		struct cntl_s* sub_node;
	} sub_nodes[ BLIS_MAX_SUB_NODES ];
};
typedef struct cntl_s cntl_t;


// -- Blocksize object type --

typedef struct blksz_s
{
	// Primary blocksize values.
	dim_t  v[BLIS_NUM_FP_TYPES];

	// Blocksize extensions.
	dim_t  e[BLIS_NUM_FP_TYPES];

} blksz_t;


// -- Function pointer object type --

typedef struct func_s
{
	// Kernel function address.
	void_fp ptr[BLIS_NUM_FP_TYPES];

} func_t;

typedef struct func2_s
{
	// Kernel function address.
	// A func2_t* can be cast to a func_t* in order to access
	// only the "diagonal" elements (dt,dt) (but note that to accomplish
	// this those elements are not stored in ptr[dt][dt]...see bli_func.c
	// for more details).
	void_fp ptr[BLIS_NUM_FP_TYPES][BLIS_NUM_FP_TYPES];

} func2_t;


// -- Multi-boolean object type --

typedef struct mbool_s
{
	bool v[BLIS_NUM_FP_TYPES];

} mbool_t;


// -- Auxiliary kernel info type --

// Note: This struct is used by macro-kernels to package together extra
// parameter values that may be of use to the micro-kernel without
// cluttering up the micro-kernel interface itself.

typedef struct
{
	// The pack schemas of A and B.
	pack_t schema_a;
	pack_t schema_b;

	// Pointers to the micro-panels of A and B which will be used by the
	// next call to the micro-kernel.
	const void* a_next;
	const void* b_next;

	// The imaginary strides of A and B.
	inc_t is_a;
	inc_t is_b;

	// The panel strides of A and B.
	// NOTE: These are only used in situations where iteration over the
	// micropanels takes place in part within the kernel code (e.g. sup
	// millikernels).
	inc_t ps_a;
	inc_t ps_b;

	// The row and column offset of the current micro-tile in C.
	dim_t off_m;
	dim_t off_n;

	// The type to convert to on output.
	//num_t  dt_on_output;

	// (Virtual) microkernel address and additional parameters.
	void_fp ukr;
	const void* params;

} auxinfo_t;


// -- Global scalar constant data struct --

// Note: This struct is used only when statically initializing the
// global scalar constants in bli_const.c.
typedef struct constdata_s
{
	float    s;
	double   d;
	scomplex c;
	dcomplex z;
	gint_t   i;

} constdata_t;


//
// -- BLIS object type definitions ---------------------------------------------
//

typedef struct obj_s
{
	// Basic fields
	struct obj_s* root;

	dim_t         off[2];
	dim_t         dim[2];
	doff_t        diag_off;

	objbits_t     info;
	objbits_t     info2;
	siz_t         elem_size;

	void*         buffer;
	inc_t         rs;
	inc_t         cs;
	inc_t         is;

	// Bufferless scalar storage
	atom_t        scalar;

	// Pack-related fields
	dim_t         m_padded; // m dimension of matrix, including any padding
	dim_t         n_padded; // n dimension of matrix, including any padding
	inc_t         ps;       // panel stride (distance to next panel)
	inc_t         pd;       // panel dimension (the "width" of a panel:
	                        // usually MR or NR)
	dim_t         m_panel;  // m dimension of a "full" panel
	dim_t         n_panel;  // n dimension of a "full" panel

} obj_t;

// Pre-initializors. Things that must be set afterwards:
// - root object pointer
// - info bitfields: dt, target_dt, exec_dt, comp_dt
// - info2 bitfields: scalar_dt
// - elem_size
// - dims, strides
// - buffer
// - internal scalar buffer (must always set imaginary component)

#define BLIS_OBJECT_INITIALIZER \
{ \
	/* .root        = */ NULL, \
\
	/* .off         = */ { 0, 0 }, \
	/* .dim         = */ { 0, 0 }, \
	/* .diag_off    = */ 0, \
\
	/* .info        = */ 0x0 | BLIS_BITVAL_DENSE      | \
	/*                */       BLIS_BITVAL_GENERAL, \
	/* .info2       = */ 0x0, \
	/* .elem_size   = */ sizeof( float ), /* this is changed later. */ \
\
	/* .buffer      = */ NULL, \
	/* .rs          = */ 0, \
	/* .cs          = */ 0, \
	/* .is          = */ 1,  \
\
	/* .scalar      = */ { 0.0, 0.0 }, \
\
	/* .m_padded    = */ 0, \
	/* .n_padded    = */ 0, \
	/* .ps          = */ 0, \
	/* .pd          = */ 0, \
	/* .m_panel     = */ 0, \
	/* .n_panel     = */ 0, \
}

#define BLIS_OBJECT_INITIALIZER_1X1 \
{ \
	/* .root        = */ NULL, \
\
	/* .off         = */ { 0, 0 }, \
	/* .dim         = */ { 1, 1 }, \
	/* .diag_off    = */ 0, \
\
	/* .info        = */ 0x0 | BLIS_BITVAL_DENSE      | \
	/*                */       BLIS_BITVAL_GENERAL, \
	/* .info2       = */ 0x0, \
	/* .elem_size   = */ sizeof( float ), /* this is changed later. */ \
\
	/* .buffer      = */ NULL, \
	/* .rs          = */ 0, \
	/* .cs          = */ 0, \
	/* .is          = */ 1,  \
\
	/* .scalar      = */ { 0.0, 0.0 }, \
\
	/* .m_padded    = */ 0, \
	/* .n_padded    = */ 0, \
	/* .ps          = */ 0, \
	/* .pd          = */ 0, \
	/* .m_panel     = */ 0, \
	/* .n_panel     = */ 0, \
}

// Define these macros here since they must be updated if contents of
// obj_t changes.

BLIS_INLINE void bli_obj_init_full_shallow_copy_of( const obj_t* a, obj_t* b )
{
	b->root        = a->root;

	b->off[0]      = a->off[0];
	b->off[1]      = a->off[1];
	b->dim[0]      = a->dim[0];
	b->dim[1]      = a->dim[1];
	b->diag_off    = a->diag_off;

	b->info        = a->info;
	b->info2       = a->info2;
	b->elem_size   = a->elem_size;

	b->buffer      = a->buffer;
	b->rs          = a->rs;
	b->cs          = a->cs;
	b->is          = a->is;

	b->scalar      = a->scalar;

	//b->pack_mem    = a->pack_mem;
	b->m_padded    = a->m_padded;
	b->n_padded    = a->n_padded;
	b->ps          = a->ps;
	b->pd          = a->pd;
	b->m_panel     = a->m_panel;
	b->n_panel     = a->n_panel;
}

BLIS_INLINE void bli_obj_init_subpart_from( const obj_t* a, obj_t* b )
{
	b->root        = a->root;

	b->off[0]      = a->off[0];
	b->off[1]      = a->off[1];
	// Avoid copying m and n since they will be overwritten.
	//b->dim[0]      = a->dim[0];
	//b->dim[1]      = a->dim[1];
	b->diag_off    = a->diag_off;

	b->info        = a->info;
	b->info2       = a->info2;
	b->elem_size   = a->elem_size;

	b->buffer      = a->buffer;
	b->rs          = a->rs;
	b->cs          = a->cs;
	b->is          = a->is;

	b->scalar      = a->scalar;

	// Avoid copying pack_mem entry.
	// FGVZ: You should probably make sure this is right.
	//b->pack_mem    = a->pack_mem;
	b->m_padded    = a->m_padded;
	b->n_padded    = a->n_padded;
	b->ps          = a->ps;
	b->pd          = a->pd;
	b->m_panel     = a->m_panel;
	b->n_panel     = a->n_panel;
}

// Initializors for global scalar constants.
// NOTE: These must remain cpp macros since they are initializor
// expressions, not functions.

#define bli_obj_init_const( buffer0 ) \
{ \
	.root      = NULL, \
\
	.off       = { 0, 0 }, \
	.dim       = { 1, 1 }, \
	.diag_off  = 0, \
\
	.info      = 0x0 | BLIS_BITVAL_CONST_TYPE | \
	                   BLIS_BITVAL_DENSE      | \
	                   BLIS_BITVAL_GENERAL, \
	.info2     = 0x0, \
	.elem_size = sizeof( constdata_t ), \
\
	.buffer    = buffer0, \
	.rs        = 1, \
	.cs        = 1, \
	.is        = 1  \
}

#define bli_obj_init_constdata( val ) \
{ \
	.s =           ( float  )val, \
	.d =           ( double )val, \
	.c = { .real = ( float  )val, .imag = 0.0f }, \
	.z = { .real = ( double )val, .imag = 0.0 }, \
	.i =           ( gint_t )val, \
}

#define bli_obj_init_constdata_ri( valr, vali ) \
{ \
	.s =           ( float  )valr, \
	.d =           ( double )valr, \
	.c = { .real = ( float  )valr, .imag = ( float  )vali }, \
	.z = { .real = ( double )valr, .imag = ( double )vali }, \
	.i =           ( gint_t )valr, \
}


// -- Stack type --

// NB: stack_t is already taken by <signal.h>
typedef struct
{
	siz_t elem_size;
	siz_t block_len;
	siz_t max_blocks;
	siz_t size;
	siz_t capacity;

	void** blocks;

	bli_pthread_mutex_t lock;
} stck_t;


// -- Context type --

typedef struct cntx_s
{
	stck_t blkszs;
	stck_t bmults;

	stck_t ukrs;
	stck_t ukr2s;
	stck_t ukr_prefs;

	stck_t l3_sup_handlers;
} cntx_t;


// -- Runtime type --

// NOTE: The order of these fields must be kept consistent with the definition
// of the BLIS_RNTM_INITIALIZER macro in bli_rntm.h.

typedef struct rntm_s
{
	// "External" fields: these may be queried by the end-user.
	timpl_t   thread_impl;

	bool      auto_factor;

	dim_t     num_threads;
	dim_t     thrloop[ BLIS_NUM_LOOPS ];
	bool      pack_a; // enable/disable packing of left-hand matrix A.
	bool      pack_b; // enable/disable packing of right-hand matrix B.
	bool      l3_sup; // enable/disable small matrix handling in level-3 ops.
} rntm_t;


// -- Error types --

typedef enum
{
	BLIS_NO_ERROR_CHECKING = 0,
	BLIS_FULL_ERROR_CHECKING
} errlev_t;

typedef enum
{
	// Generic error codes
	BLIS_SUCCESS                               = (  -1),
	BLIS_FAILURE                               = (  -2),

	BLIS_ERROR_CODE_MIN                        = (  -9),

	// General errors
	BLIS_INVALID_ERROR_CHECKING_LEVEL          = ( -10),
	BLIS_UNDEFINED_ERROR_CODE                  = ( -11),
	BLIS_NULL_POINTER                          = ( -12),
	BLIS_NOT_YET_IMPLEMENTED                   = ( -13),
	BLIS_OUT_OF_BOUNDS                         = ( -14),
	BLIS_LOCK_FAILURE                          = ( -15),

	// Parameter-specific errors
	BLIS_INVALID_SIDE                          = ( -20),
	BLIS_INVALID_UPLO                          = ( -21),
	BLIS_INVALID_TRANS                         = ( -22),
	BLIS_INVALID_CONJ                          = ( -23),
	BLIS_INVALID_DIAG                          = ( -24),
	BLIS_INVALID_MACHVAL                       = ( -25),
	BLIS_EXPECTED_NONUNIT_DIAG                 = ( -26),

	// Datatype-specific errors
	BLIS_INVALID_DATATYPE                      = ( -30),
	BLIS_EXPECTED_FLOATING_POINT_DATATYPE      = ( -31),
	BLIS_EXPECTED_NONINTEGER_DATATYPE          = ( -32),
	BLIS_EXPECTED_NONCONSTANT_DATATYPE         = ( -33),
	BLIS_EXPECTED_REAL_DATATYPE                = ( -34),
	BLIS_EXPECTED_INTEGER_DATATYPE             = ( -35),
	BLIS_INCONSISTENT_DATATYPES                = ( -36),
	BLIS_EXPECTED_REAL_PROJ_OF                 = ( -37),
	BLIS_EXPECTED_REAL_VALUED_OBJECT           = ( -38),
	BLIS_INCONSISTENT_PRECISIONS               = ( -39),

	// Dimension-specific errors
	BLIS_NONCONFORMAL_DIMENSIONS               = ( -40),
	BLIS_EXPECTED_SCALAR_OBJECT                = ( -41),
	BLIS_EXPECTED_VECTOR_OBJECT                = ( -42),
	BLIS_UNEQUAL_VECTOR_LENGTHS                = ( -43),
	BLIS_EXPECTED_SQUARE_OBJECT                = ( -44),
	BLIS_UNEXPECTED_OBJECT_LENGTH              = ( -45),
	BLIS_UNEXPECTED_OBJECT_WIDTH               = ( -46),
	BLIS_UNEXPECTED_VECTOR_DIM                 = ( -47),
	BLIS_UNEXPECTED_DIAG_OFFSET                = ( -48),
	BLIS_NEGATIVE_DIMENSION                    = ( -49),

	// Stride-specific errors
	BLIS_INVALID_ROW_STRIDE                    = ( -50),
	BLIS_INVALID_COL_STRIDE                    = ( -51),
	BLIS_INVALID_DIM_STRIDE_COMBINATION        = ( -52),

	// Structure-specific errors
	BLIS_EXPECTED_GENERAL_OBJECT               = ( -60),
	BLIS_EXPECTED_HERMITIAN_OBJECT             = ( -61),
	BLIS_EXPECTED_SYMMETRIC_OBJECT             = ( -62),
	BLIS_EXPECTED_TRIANGULAR_OBJECT            = ( -63),

	// Storage-specific errors
	BLIS_EXPECTED_UPPER_OR_LOWER_OBJECT        = ( -70),

	// Partitioning-specific errors
	BLIS_INVALID_3x1_SUBPART                   = ( -80),
	BLIS_INVALID_1x3_SUBPART                   = ( -81),
	BLIS_INVALID_3x3_SUBPART                   = ( -82),

	// Control tree-specific errors
	BLIS_UNEXPECTED_NULL_CONTROL_TREE          = ( -90),

	// Packing-specific errors
	BLIS_PACK_SCHEMA_NOT_SUPPORTED_FOR_UNPACK  = (-100),
	BLIS_PACK_SCHEMA_NOT_SUPPORTED_FOR_PART    = (-101),

	// Buffer-specific errors
	BLIS_EXPECTED_NONNULL_OBJECT_BUFFER        = (-110),

	// Memory errors
	BLIS_MALLOC_RETURNED_NULL                  = (-120),

	// Internal memory pool errors
	BLIS_INVALID_PACKBUF                       = (-130),
	BLIS_EXHAUSTED_CONTIG_MEMORY_POOL          = (-131),
	BLIS_INSUFFICIENT_STACK_BUF_SIZE           = (-132),
	BLIS_ALIGNMENT_NOT_POWER_OF_TWO            = (-133),
	BLIS_ALIGNMENT_NOT_MULT_OF_PTR_SIZE        = (-134),

	// Object-related errors
	BLIS_EXPECTED_OBJECT_ALIAS                 = (-140),

	// Architecture-related errors
	BLIS_INVALID_ARCH_ID                       = (-150),
	BLIS_UNINITIALIZED_GKS_CNTX                = (-151),
	BLIS_INVALID_UKR_ID                        = (-152),

	// Blocksize-related errors
	BLIS_MC_DEF_NONMULTIPLE_OF_MR              = (-160),
	BLIS_MC_MAX_NONMULTIPLE_OF_MR              = (-161),
	BLIS_NC_DEF_NONMULTIPLE_OF_NR              = (-162),
	BLIS_NC_MAX_NONMULTIPLE_OF_NR              = (-163),
	BLIS_KC_DEF_NONMULTIPLE_OF_KR              = (-164),
	BLIS_KC_MAX_NONMULTIPLE_OF_KR              = (-165),
	BLIS_MR_NOT_EVEN_FOR_REAL_TYPE             = (-166),
	BLIS_PACKMR_NOT_EVEN_FOR_REAL_TYPE         = (-167),
	BLIS_NR_NOT_EVEN_FOR_REAL_TYPE             = (-168),
	BLIS_PACKNR_NOT_EVEN_FOR_REAL_TYPE         = (-169),

	BLIS_ERROR_CODE_MAX                        = (-170)
} err_t;

#endif
// end bli_type_defs.h
#line 74 "../blis-2.0/frame/include/blis.h"

// begin bli_macro_defs.h
#line 1 "../blis-2.0/frame/include//bli_macro_defs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2018 - 2019, Advanced Micro Devices, Inc.
   Copyright (C) 2024, Southern Methodist University

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_MACRO_DEFS_H
#define BLIS_MACRO_DEFS_H


// -- Concatenation macros --

#define BLIS_FUNC_PREFIX_STR       "bli"

// We add an extra layer the definitions of these string-pasting macros
// because sometimes it is needed if, for example, one of the PASTE
// macros is invoked with an "op" argument that is itself a macro.

#define PASTEMAC0_(op)                         bli_ ## op
#define PASTEMAC1_(ch,op)                      bli_ ## ch  ## op
#define PASTEMAC2_(ch1,ch2,op)                 bli_ ## ch1 ## ch2 ## op
#define PASTEMAC3_(ch1,ch2,ch3,op)             bli_ ## ch1 ## ch2 ## ch3 ## op
#define PASTEMAC4_(ch1,ch2,ch3,ch4,op)         bli_ ## ch1 ## ch2 ## ch3 ## ch4 ## op
#define PASTEMAC5_(ch1,ch2,ch3,ch4,ch5,op)     bli_ ## ch1 ## ch2 ## ch3 ## ch4 ## ch5 ## op
#define PASTEMAC6_(ch1,ch2,ch3,ch4,ch5,ch6,op) bli_ ## ch1 ## ch2 ## ch3 ## ch4 ## ch5 ## ch6 ## op

#define PASTEMAC__(arg1,arg2,arg3,arg4,arg5,arg6,arg7,arg8,...) PASTEMAC ## arg8 ## _
#define PASTEMAC_(...) PASTEMAC__(__VA_ARGS__, 6, 5, 4, 3, 2, 1, 0, XXX)
#define PASTEMAC(...) PASTEMAC_(__VA_ARGS__)(__VA_ARGS__)

#define PASTECH0_(op)                         op
#define PASTECH1_(ch,op)                      ch  ## op
#define PASTECH2_(ch1,ch2,op)                 ch1 ## ch2 ## op
#define PASTECH3_(ch1,ch2,ch3,op)             ch1 ## ch2 ## ch3 ## op
#define PASTECH4_(ch1,ch2,ch3,ch4,op)         ch1 ## ch2 ## ch3 ## ch4 ## op
#define PASTECH5_(ch1,ch2,ch3,ch4,ch5,op)     ch1 ## ch2 ## ch3 ## ch4 ## ch5 ## op
#define PASTECH6_(ch1,ch2,ch3,ch4,ch5,ch6,op) ch1 ## ch2 ## ch3 ## ch4 ## ch5 ## ch6 ## op

#define PASTECH__(arg1,arg2,arg3,arg4,arg5,arg6,arg7,arg8,...) PASTECH ## arg8 ## _
#define PASTECH_(...) PASTECH__(__VA_ARGS__, 6, 5, 4, 3, 2, 1, 0, XXX)
#define PASTECH(...) PASTECH_(__VA_ARGS__)(__VA_ARGS__)

// Fortran-77 name-mangling macros.
#define PASTEF770_(op)                         op  ## _
#define PASTEF771_(ch,op)                      ch  ## op ## _
#define PASTEF772_(ch1,ch2,op)                 ch1 ## ch2 ## op ## _
#define PASTEF773_(ch1,ch2,ch3,op)             ch1 ## ch2 ## ch3 ## op ## _
#define PASTEF774_(ch1,ch2,ch3,ch4,op)         ch1 ## ch2 ## ch3 ## ch4 ## op ## _
#define PASTEF775_(ch1,ch2,ch3,ch4,ch5,op)     ch1 ## ch2 ## ch3 ## ch4 ## ch5 ## op ## _
#define PASTEF776_(ch1,ch2,ch3,ch4,ch5,ch6,op) ch1 ## ch2 ## ch3 ## ch4 ## ch5 ## ch6 ## op ## _

#define PASTEF77__(arg1,arg2,arg3,arg4,arg5,arg6,arg7,arg8,...) PASTEF77 ## arg8 ## _
#define PASTEF77_(...) PASTEF77__(__VA_ARGS__, 6, 5, 4, 3, 2, 1, 0, XXX)
#define PASTEF77(...) PASTEF77_(__VA_ARGS__)(__VA_ARGS__)

#define PASTEBLACHK_(op)           bla_ ## op ## _check
#define PASTEBLACHK(op)            PASTEBLACHK_(op)

#define MKSTR(s1)                  #s1
#define STRINGIFY_INT( s )         MKSTR( s )

// -- Include other groups of macros

// begin bli_genarray_macro_defs.h
#line 1 "../blis-2.0/frame/include//bli_genarray_macro_defs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_GENARRAY_MACRO_DEFS_H
#define BLIS_GENARRAY_MACRO_DEFS_H


// -- Macros to generate function arrays ---------------------------------------

// -- "Smart" one-operand macro --

#define GENARRAY_FPA(tname,opname) \
\
static tname PASTECH(opname,_fpa)[BLIS_NUM_FP_TYPES] = \
{ \
	( tname )PASTEMAC(s,opname), \
	( tname )PASTEMAC(c,opname), \
	( tname )PASTEMAC(d,opname), \
	( tname )PASTEMAC(z,opname)  \
}

#define GENARRAYRO_FPA(tname,opname) \
\
static tname PASTECH(opname,_fpa)[BLIS_NUM_FP_TYPES] = \
{ \
	( tname )PASTEMAC(s,opname), \
	( tname )PASTEMAC(d,opname), \
	NULL, \
	NULL \
}

// -- "Smart" one-operand macro (with integer support) --

#define GENARRAY_FPA_I(tname,opname) \
\
static tname PASTECH(opname,_fpa)[BLIS_NUM_FP_TYPES+1] = \
{ \
	( tname )PASTEMAC(s,opname), \
	( tname )PASTEMAC(c,opname), \
	( tname )PASTEMAC(d,opname), \
	( tname )PASTEMAC(z,opname), \
	( tname )PASTEMAC(i,opname)  \
}

// -- "Smart" two-operand macro --

#define GENARRAY_FPA2(tname,op) \
\
static tname PASTECH(op,_fpa2)[BLIS_NUM_FP_TYPES][BLIS_NUM_FP_TYPES] = \
{ \
	{ ( tname )PASTEMAC(s,s,op), ( tname )PASTEMAC(s,c,op), ( tname )PASTEMAC(s,d,op), ( tname )PASTEMAC(s,z,op) }, \
	{ ( tname )PASTEMAC(c,s,op), ( tname )PASTEMAC(c,c,op), ( tname )PASTEMAC(c,d,op), ( tname )PASTEMAC(c,z,op) }, \
	{ ( tname )PASTEMAC(d,s,op), ( tname )PASTEMAC(d,c,op), ( tname )PASTEMAC(d,d,op), ( tname )PASTEMAC(d,z,op) }, \
	{ ( tname )PASTEMAC(z,s,op), ( tname )PASTEMAC(z,c,op), ( tname )PASTEMAC(z,d,op), ( tname )PASTEMAC(z,z,op) }  \
}

// -- "Smart" two-operand macro --

/*
#define GENARRAY2_VFP(arrayname,op) \
\
arrayname[BLIS_NUM_FP_TYPES][BLIS_NUM_FP_TYPES] = \
{ \
	{ PASTEMAC(s,s,op), PASTEMAC(s,c,op), PASTEMAC(s,d,op), PASTEMAC(s,z,op) }, \
	{ PASTEMAC(c,s,op), PASTEMAC(c,c,op), PASTEMAC(c,d,op), PASTEMAC(c,z,op) }, \
	{ PASTEMAC(d,s,op), PASTEMAC(d,c,op), PASTEMAC(d,d,op), PASTEMAC(d,z,op) }, \
	{ PASTEMAC(z,s,op), PASTEMAC(z,c,op), PASTEMAC(z,d,op), PASTEMAC(z,z,op) }  \
}
*/



// -- One-operand macro --

#define GENARRAY(arrayname,op) \
\
arrayname[BLIS_NUM_FP_TYPES] = \
{ \
	PASTEMAC(s,op), \
	PASTEMAC(c,op), \
	PASTEMAC(d,op), \
	PASTEMAC(z,op)  \
}

#define GENARRAY_I(arrayname,op) \
\
arrayname[BLIS_NUM_FP_TYPES+1] = \
{ \
	PASTEMAC(s,op), \
	PASTEMAC(c,op), \
	PASTEMAC(d,op), \
	PASTEMAC(z,op), \
	PASTEMAC(i,op)  \
}

/*
#define GENARRAYR(arrayname,op) \
\
arrayname[BLIS_NUM_FP_TYPES][BLIS_NUM_FP_TYPES] = \
{ \
	{ PASTEMAC(s,s,op), NULL,              PASTEMAC(s,d,op), NULL,             }, \
	{ PASTEMAC(c,s,op), NULL,              PASTEMAC(c,d,op), NULL,             }, \
	{ PASTEMAC(d,s,op), NULL,              PASTEMAC(d,d,op), NULL,             }, \
	{ PASTEMAC(z,s,op), NULL,              PASTEMAC(z,d,op), NULL,             }  \
}
*/



// -- One-operand macro (with custom prefix) --

#define GENARRAY_PREF(arrayname,prefix,op) \
\
arrayname[BLIS_NUM_FP_TYPES] = \
{ \
	PASTECH(prefix,s,op), \
	PASTECH(prefix,c,op), \
	PASTECH(prefix,d,op), \
	PASTECH(prefix,z,op)  \
}



// -- Two-operand macros --


#define GENARRAY2_ALL(arrayname,op) \
\
arrayname[BLIS_NUM_FP_TYPES][BLIS_NUM_FP_TYPES] = \
{ \
	{ PASTEMAC(s,s,op), PASTEMAC(s,c,op), PASTEMAC(s,d,op), PASTEMAC(s,z,op) }, \
	{ PASTEMAC(c,s,op), PASTEMAC(c,c,op), PASTEMAC(c,d,op), PASTEMAC(c,z,op) }, \
	{ PASTEMAC(d,s,op), PASTEMAC(d,c,op), PASTEMAC(d,d,op), PASTEMAC(d,z,op) }, \
	{ PASTEMAC(z,s,op), PASTEMAC(z,c,op), PASTEMAC(z,d,op), PASTEMAC(z,z,op) }  \
}


#define GENARRAY2_MIXP(arrayname,op) \
\
arrayname[BLIS_NUM_FP_TYPES][BLIS_NUM_FP_TYPES] = \
{ \
	{ PASTEMAC(s,s,op), NULL,              PASTEMAC(s,d,op), NULL,             }, \
	{ NULL,              PASTEMAC(c,c,op), NULL,              PASTEMAC(c,z,op) }, \
	{ PASTEMAC(d,s,op), NULL,              PASTEMAC(d,d,op), NULL,             }, \
	{ NULL,              PASTEMAC(z,c,op), NULL,              PASTEMAC(z,z,op) }  \
}


#define GENARRAY2_EXT(arrayname,op) \
\
arrayname[BLIS_NUM_FP_TYPES][BLIS_NUM_FP_TYPES] = \
{ \
	{ PASTEMAC(s,s,op), PASTEMAC(s,c,op), NULL,              NULL,             }, \
	{ PASTEMAC(c,s,op), PASTEMAC(c,c,op), NULL,              NULL,             }, \
	{ NULL,              NULL,              PASTEMAC(d,d,op), PASTEMAC(d,z,op) }, \
	{ NULL,              NULL,              PASTEMAC(z,d,op), PASTEMAC(z,z,op) }  \
}


#define GENARRAY2_MIN(arrayname,op) \
\
arrayname[BLIS_NUM_FP_TYPES][BLIS_NUM_FP_TYPES] = \
{ \
	{ PASTEMAC(s,s,op), NULL,              NULL,              NULL,             }, \
	{ NULL,              PASTEMAC(c,c,op), NULL,              NULL,             }, \
	{ NULL,              NULL,              PASTEMAC(d,d,op), NULL,             }, \
	{ NULL,              NULL,              NULL,              PASTEMAC(z,z,op) }  \
}


// -- Three-operand macros --


#define GENARRAY3_ALL(arrayname,op) \
\
arrayname[BLIS_NUM_FP_TYPES][BLIS_NUM_FP_TYPES][BLIS_NUM_FP_TYPES] = \
{ \
	{ \
	{ PASTEMAC(s,s,s,op), PASTEMAC(s,s,c,op), PASTEMAC(s,s,d,op), PASTEMAC(s,s,z,op) }, \
	{ PASTEMAC(s,c,s,op), PASTEMAC(s,c,c,op), PASTEMAC(s,c,d,op), PASTEMAC(s,c,z,op) }, \
	{ PASTEMAC(s,d,s,op), PASTEMAC(s,d,c,op), PASTEMAC(s,d,d,op), PASTEMAC(s,d,z,op) }, \
	{ PASTEMAC(s,z,s,op), PASTEMAC(s,z,c,op), PASTEMAC(s,z,d,op), PASTEMAC(s,z,z,op) }  \
	}, \
	{ \
	{ PASTEMAC(c,s,s,op), PASTEMAC(c,s,c,op), PASTEMAC(c,s,d,op), PASTEMAC(c,s,z,op) }, \
	{ PASTEMAC(c,c,s,op), PASTEMAC(c,c,c,op), PASTEMAC(c,c,d,op), PASTEMAC(c,c,z,op) }, \
	{ PASTEMAC(c,d,s,op), PASTEMAC(c,d,c,op), PASTEMAC(c,d,d,op), PASTEMAC(c,d,z,op) }, \
	{ PASTEMAC(c,z,s,op), PASTEMAC(c,z,c,op), PASTEMAC(c,z,d,op), PASTEMAC(c,z,z,op) }  \
	}, \
	{ \
	{ PASTEMAC(d,s,s,op), PASTEMAC(d,s,c,op), PASTEMAC(d,s,d,op), PASTEMAC(d,s,z,op) }, \
	{ PASTEMAC(d,c,s,op), PASTEMAC(d,c,c,op), PASTEMAC(d,c,d,op), PASTEMAC(d,c,z,op) }, \
	{ PASTEMAC(d,d,s,op), PASTEMAC(d,d,c,op), PASTEMAC(d,d,d,op), PASTEMAC(d,d,z,op) }, \
	{ PASTEMAC(d,z,s,op), PASTEMAC(d,z,c,op), PASTEMAC(d,z,d,op), PASTEMAC(d,z,z,op) }  \
	}, \
	{ \
	{ PASTEMAC(z,s,s,op), PASTEMAC(z,s,c,op), PASTEMAC(z,s,d,op), PASTEMAC(z,s,z,op) }, \
	{ PASTEMAC(z,c,s,op), PASTEMAC(z,c,c,op), PASTEMAC(z,c,d,op), PASTEMAC(z,c,z,op) }, \
	{ PASTEMAC(z,d,s,op), PASTEMAC(z,d,c,op), PASTEMAC(z,d,d,op), PASTEMAC(z,d,z,op) }, \
	{ PASTEMAC(z,z,s,op), PASTEMAC(z,z,c,op), PASTEMAC(z,z,d,op), PASTEMAC(z,z,z,op) }  \
	} \
}


#define GENARRAY3_EXT(arrayname,op) \
\
arrayname[BLIS_NUM_FP_TYPES][BLIS_NUM_FP_TYPES][BLIS_NUM_FP_TYPES] = \
{ \
	{ \
	{ PASTEMAC(s,s,s,op), PASTEMAC(s,s,c,op), NULL,                NULL,               }, \
	{ PASTEMAC(s,c,s,op), PASTEMAC(s,c,c,op), NULL,                NULL,               }, \
	{ NULL,                NULL,                NULL,                NULL,               }, \
	{ NULL,                NULL,                NULL,                NULL,               }  \
	}, \
	{ \
	{ PASTEMAC(c,s,s,op), PASTEMAC(c,s,c,op), NULL,                NULL,               }, \
	{ PASTEMAC(c,c,s,op), PASTEMAC(c,c,c,op), NULL,                NULL,               }, \
	{ NULL,                NULL,                NULL,                NULL,               }, \
	{ NULL,                NULL,                NULL,                NULL,               }  \
	}, \
	{ \
	{ NULL,                NULL,                NULL,                NULL,               }, \
	{ NULL,                NULL,                NULL,                NULL,               }, \
	{ NULL,                NULL,                PASTEMAC(d,d,d,op), PASTEMAC(d,d,z,op) }, \
	{ NULL,                NULL,                PASTEMAC(d,z,d,op), PASTEMAC(d,z,z,op) }  \
	}, \
	{ \
	{ NULL,                NULL,                NULL,                NULL,               }, \
	{ NULL,                NULL,                NULL,                NULL,               }, \
	{ NULL,                NULL,                PASTEMAC(z,d,d,op), PASTEMAC(z,d,z,op) }, \
	{ NULL,                NULL,                PASTEMAC(z,z,d,op), PASTEMAC(z,z,z,op) }  \
	} \
}


#define GENARRAY3_MIN(arrayname,op) \
\
arrayname[BLIS_NUM_FP_TYPES][BLIS_NUM_FP_TYPES][BLIS_NUM_FP_TYPES] = \
{ \
	{ \
	{ PASTEMAC(s,s,s,op), NULL,                NULL,                NULL,               }, \
	{ NULL,                NULL,                NULL,                NULL,               }, \
	{ NULL,                NULL,                NULL,                NULL,               }, \
	{ NULL,                NULL,                NULL,                NULL,               }  \
	}, \
	{ \
	{ NULL,                NULL,                NULL,                NULL,               }, \
	{ NULL,                PASTEMAC(c,c,c,op), NULL,                NULL,               }, \
	{ NULL,                NULL,                NULL,                NULL,               }, \
	{ NULL,                NULL,                NULL,                NULL,               }  \
	}, \
	{ \
	{ NULL,                NULL,                NULL,                NULL,               }, \
	{ NULL,                NULL,                NULL,                NULL,               }, \
	{ NULL,                NULL,                PASTEMAC(d,d,d,op), NULL,               }, \
	{ NULL,                NULL,                NULL,                NULL,               }  \
	}, \
	{ \
	{ NULL,                NULL,                NULL,                NULL,               }, \
	{ NULL,                NULL,                NULL,                NULL,               }, \
	{ NULL,                NULL,                NULL,                NULL,               }, \
	{ NULL,                NULL,                NULL,                PASTEMAC(z,z,z,op) }  \
	} \
}


#endif
// end bli_genarray_macro_defs.h
#line 95 "../blis-2.0/frame/include//bli_macro_defs.h"

// begin bli_gentdef_macro_defs.h
#line 1 "../blis-2.0/frame/include//bli_gentdef_macro_defs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_GENTDEF_MACRO_DEFS_H
#define BLIS_GENTDEF_MACRO_DEFS_H

//
// -- MACROS TO INSERT TYPEDEF-GENERATING MACROS -------------------------------
//


// -- function typedef macro (both typed and void) --

#define INSERT_GENTDEF( opname ) \
\
GENTDEF( float,    s, opname, _ft ) \
GENTDEF( double,   d, opname, _ft ) \
GENTDEF( scomplex, c, opname, _ft ) \
GENTDEF( dcomplex, z, opname, _ft ) \
\
GENTDEF( void,     s, opname, _vft ) \
GENTDEF( void,     d, opname, _vft ) \
GENTDEF( void,     c, opname, _vft ) \
GENTDEF( void,     z, opname, _vft ) \
\
GENTDEF( void,      , opname, _vft )

// -- function typedef macro (both typed and void) with real projection --

#define INSERT_GENTDEFR( opname ) \
\
GENTDEFR( float,    float,    s, s, opname, _ft ) \
GENTDEFR( double,   double,   d, d, opname, _ft ) \
GENTDEFR( scomplex, float,    c, s, opname, _ft ) \
GENTDEFR( dcomplex, double,   z, d, opname, _ft ) \
\
GENTDEFR( void,     void,     s, s, opname, _vft ) \
GENTDEFR( void,     void,     d, d, opname, _vft ) \
GENTDEFR( void,     void,     c, s, opname, _vft ) \
GENTDEFR( void,     void,     z, d, opname, _vft ) \
\
GENTDEFR( void,     void,      ,  , opname, _vft )


#endif
// end bli_gentdef_macro_defs.h
#line 96 "../blis-2.0/frame/include//bli_macro_defs.h"

// begin bli_gentfunc_macro_defs.h
#line 1 "../blis-2.0/frame/include//bli_gentfunc_macro_defs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2023, Southern Methodist University

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/


#ifndef BLIS_GENTFUNC_MACRO_DEFS_H
#define BLIS_GENTFUNC_MACRO_DEFS_H

//
// -- MACROS TO INSERT FUNCTION-GENERATING MACROS ------------------------------
//



// -- Macros for generating BLAS routines --------------------------------------


// -- Basic one-operand macro --


#define INSERT_GENTFUNC_BLAS( blasname, blisname ) \
\
GENTFUNC( float,    s, blasname, blisname ) \
GENTFUNC( double,   d, blasname, blisname ) \
GENTFUNC( scomplex, c, blasname, blisname ) \
GENTFUNC( dcomplex, z, blasname, blisname )


// -- Basic one-operand macro with real domain only --


#define INSERT_GENTFUNCRO_BLAS( blasname, blisname ) \
\
GENTFUNCRO( float,    s, blasname, blisname ) \
GENTFUNCRO( double,   d, blasname, blisname )


// -- Basic one-operand macro with complex domain only and real projection --


#define INSERT_GENTFUNCCO_BLAS( blasname, blisname ) \
\
GENTFUNCCO( scomplex, float,  c, s, blasname, blisname ) \
GENTFUNCCO( dcomplex, double, z, d, blasname, blisname )


// -- Basic one-operand macro with conjugation (real funcs only, used only for dot, ger) --


#define INSERT_GENTFUNCDOTR_BLAS( blasname, blisname ) \
\
GENTFUNCDOT( float,    s,  , BLIS_NO_CONJUGATE, blasname, blisname ) \
GENTFUNCDOT( double,   d,  , BLIS_NO_CONJUGATE, blasname, blisname )


// -- Basic one-operand macro with conjugation (complex funcs only, used only for dot, ger) --


#define INSERT_GENTFUNCDOTC_BLAS( blasname, blisname ) \
\
GENTFUNCDOT( scomplex, c, c, BLIS_CONJUGATE,    blasname, blisname ) \
GENTFUNCDOT( scomplex, c, u, BLIS_NO_CONJUGATE, blasname, blisname ) \
GENTFUNCDOT( dcomplex, z, c, BLIS_CONJUGATE,    blasname, blisname ) \
GENTFUNCDOT( dcomplex, z, u, BLIS_NO_CONJUGATE, blasname, blisname )


// -- Basic one-operand macro with conjugation (used only for dot, ger) --


#define INSERT_GENTFUNCDOT_BLAS( blasname, blisname ) \
\
INSERT_GENTFUNCDOTR_BLAS( blasname, blisname ) \
INSERT_GENTFUNCDOTC_BLAS( blasname, blisname )


// -- Basic one-operand macro with real projection --


#define INSERT_GENTFUNCR_BLAS( rblasname, cblasname, blisname ) \
\
GENTFUNCR( float,    float,  s, s, rblasname, blisname ) \
GENTFUNCR( double,   double, d, d, rblasname, blisname ) \
GENTFUNCR( scomplex, float,  c, s, cblasname, blisname ) \
GENTFUNCR( dcomplex, double, z, d, cblasname, blisname )


// -- Alternate two-operand macro (one char for complex, one for real proj) --


#define INSERT_GENTFUNCR2_BLAS( blasname, blisname ) \
\
GENTFUNCR2( float,    float,  s,  , blasname, blisname ) \
GENTFUNCR2( double,   double, d,  , blasname, blisname ) \
GENTFUNCR2( scomplex, float,  c, s, blasname, blisname ) \
GENTFUNCR2( dcomplex, double, z, d, blasname, blisname )


// -- Extended two-operand macro (used only for scal) --


#define INSERT_GENTFUNCSCAL_BLAS( blasname, blisname ) \
\
GENTFUNCSCAL( float,    float,    s,  , s, blasname, blisname ) \
GENTFUNCSCAL( double,   double,   d,  , d, blasname, blisname ) \
GENTFUNCSCAL( scomplex, scomplex, c,  , c, blasname, blisname ) \
GENTFUNCSCAL( dcomplex, dcomplex, z,  , z, blasname, blisname ) \
GENTFUNCSCAL( scomplex, float,    c, s, s, blasname, blisname ) \
GENTFUNCSCAL( dcomplex, double,   z, d, d, blasname, blisname )




// -- Macros for functions with one operand ------------------------------------


// -- Basic one-operand macro --

#define INSERT_GENTFUNC_BASIC( ... ) \
\
GENTFUNC( float,    s, __VA_ARGS__ ) \
GENTFUNC( double,   d, __VA_ARGS__ ) \
GENTFUNC( scomplex, c, __VA_ARGS__ ) \
GENTFUNC( dcomplex, z, __VA_ARGS__ )



// -- Basic one-operand with real projection --

#define INSERT_GENTFUNCR_BASIC( ... ) \
\
GENTFUNCR( float,    float,  s, s, __VA_ARGS__ ) \
GENTFUNCR( double,   double, d, d, __VA_ARGS__ ) \
GENTFUNCR( scomplex, float,  c, s, __VA_ARGS__ ) \
GENTFUNCR( dcomplex, double, z, d, __VA_ARGS__ )



// -- Basic one-operand macro with real domain only --

#define INSERT_GENTFUNCRO_BASIC( ... ) \
\
GENTFUNCRO( float,  s, __VA_ARGS__ ) \
GENTFUNCRO( double, d, __VA_ARGS__ )

// -- Basic one-operand macro with complex domain only --

#define INSERT_GENTFUNCCO_BASIC( ... ) \
\
GENTFUNCCO( scomplex, c, __VA_ARGS__ ) \
GENTFUNCCO( dcomplex, z, __VA_ARGS__ )

// -- Basic one-operand macro with real domain only and complex projection --

#define INSERT_GENTFUNCRO( ... ) \
\
GENTFUNCRO( float,  scomplex, s, c, __VA_ARGS__ ) \
GENTFUNCRO( double, dcomplex, d, z, __VA_ARGS__ )

// -- Basic one-operand macro with complex domain only and real projection --

#define INSERT_GENTFUNCCO( ... ) \
\
GENTFUNCCO( scomplex, float,  c, s, __VA_ARGS__ ) \
GENTFUNCCO( dcomplex, double, z, d, __VA_ARGS__ )



// -- Basic one-operand macro with integer instance --

#define INSERT_GENTFUNC_BASIC_I( ... ) \
\
GENTFUNC( float,    s, __VA_ARGS__ ) \
GENTFUNC( double,   d, __VA_ARGS__ ) \
GENTFUNC( scomplex, c, __VA_ARGS__ ) \
GENTFUNC( dcomplex, z, __VA_ARGS__ ) \
GENTFUNC( gint_t,   i, __VA_ARGS__ )



// -- Basic one-operand with integer projection --

#define INSERT_GENTFUNCI_BASIC( ... ) \
\
GENTFUNCI( float,    gint_t, s, i, __VA_ARGS__ ) \
GENTFUNCI( double,   gint_t, d, i, __VA_ARGS__ ) \
GENTFUNCI( scomplex, gint_t, c, i, __VA_ARGS__ ) \
GENTFUNCI( dcomplex, gint_t, z, i, __VA_ARGS__ )



// -- Basic one-operand with real and integer projections --

#define INSERT_GENTFUNCRI_BASIC( ... ) \
\
GENTFUNCRI( float,    float,  gint_t, s, s, i, __VA_ARGS__ ) \
GENTFUNCRI( double,   double, gint_t, d, d, i, __VA_ARGS__ ) \
GENTFUNCRI( scomplex, float,  gint_t, c, s, i, __VA_ARGS__ ) \
GENTFUNCRI( dcomplex, double, gint_t, z, d, i, __VA_ARGS__ )




// -- Macros for functions with two primary operands ---------------------------


// -- Basic two-operand macro --

#define INSERT_GENTFUNC2_BASIC( ... ) \
\
GENTFUNC2( float,    float,    s, s, __VA_ARGS__ ) \
GENTFUNC2( double,   double,   d, d, __VA_ARGS__ ) \
GENTFUNC2( scomplex, scomplex, c, c, __VA_ARGS__ ) \
GENTFUNC2( dcomplex, dcomplex, z, z, __VA_ARGS__ )



// -- Mixed domain two-operand macro --

#define INSERT_GENTFUNC2_MIX_D( ... ) \
\
GENTFUNC2( float,    scomplex, s, c, __VA_ARGS__ ) \
GENTFUNC2( scomplex, float,    c, s, __VA_ARGS__ ) \
\
GENTFUNC2( double,   dcomplex, d, z, __VA_ARGS__ ) \
GENTFUNC2( dcomplex, double,   z, d, __VA_ARGS__ )



// -- Mixed precision two-operand macro --

#define INSERT_GENTFUNC2_MIX_P( ... ) \
\
GENTFUNC2( float,    double,   s, d, __VA_ARGS__ ) \
GENTFUNC2( double,   float,    d, s, __VA_ARGS__ ) \
\
GENTFUNC2( scomplex, dcomplex, c, z, __VA_ARGS__ ) \
GENTFUNC2( dcomplex, scomplex, z, c, __VA_ARGS__ )



// -- Mixed domain/precision (all) two-operand macro --

#define INSERT_GENTFUNC2_MIX_DP( ... ) \
\
GENTFUNC2( float,    double,   s, d, __VA_ARGS__ ) \
GENTFUNC2( float,    scomplex, s, c, __VA_ARGS__ ) \
GENTFUNC2( float,    dcomplex, s, z, __VA_ARGS__ ) \
\
GENTFUNC2( double,   float,    d, s, __VA_ARGS__ ) \
GENTFUNC2( double,   scomplex, d, c, __VA_ARGS__ ) \
GENTFUNC2( double,   dcomplex, d, z, __VA_ARGS__ ) \
\
GENTFUNC2( scomplex, float,    c, s, __VA_ARGS__ ) \
GENTFUNC2( scomplex, double,   c, d, __VA_ARGS__ ) \
GENTFUNC2( scomplex, dcomplex, c, z, __VA_ARGS__ ) \
\
GENTFUNC2( dcomplex, float,    z, s, __VA_ARGS__ ) \
GENTFUNC2( dcomplex, double,   z, d, __VA_ARGS__ ) \
GENTFUNC2( dcomplex, scomplex, z, c, __VA_ARGS__ )



// -- Basic two-operand with real projection of second operand --

#define INSERT_GENTFUNC2R_BASIC( ... ) \
\
GENTFUNC2R( float,    float,    float,    s, s, s, __VA_ARGS__ ) \
GENTFUNC2R( double,   double,   double,   d, d, d, __VA_ARGS__ ) \
GENTFUNC2R( scomplex, scomplex, float,    c, c, s, __VA_ARGS__ ) \
GENTFUNC2R( dcomplex, dcomplex, double,   z, z, d, __VA_ARGS__ )



// -- Mixed domain two-operand with real projection of second operand --

#define INSERT_GENTFUNC2R_MIX_D( ... ) \
\
GENTFUNC2R( float,    scomplex, float,    s, c, s, __VA_ARGS__ ) \
GENTFUNC2R( scomplex, float,    float,    c, s, s, __VA_ARGS__ ) \
\
GENTFUNC2R( double,   dcomplex, double,   d, z, d, __VA_ARGS__ ) \
GENTFUNC2R( dcomplex, double,   double,   z, d, d, __VA_ARGS__ )



// -- Mixed precision two-operand with real projection of second operand --

#define INSERT_GENTFUNC2R_MIX_P( ... ) \
\
GENTFUNC2R( float,    double,   double,   s, d, d, __VA_ARGS__ ) \
GENTFUNC2R( double,   float,    float,    d, s, s, __VA_ARGS__ ) \
\
GENTFUNC2R( scomplex, dcomplex, double,   c, z, d, __VA_ARGS__ ) \
GENTFUNC2R( dcomplex, scomplex, float,    z, c, s, __VA_ARGS__ )



// -- Real-only two-operand with complex projection of both operands --

#define INSERT_GENTFUNC2RO( ... ) \
\
GENTFUNC2RO( float,  scomplex, float,  scomplex, s, c, s, c, __VA_ARGS__ ) \
GENTFUNC2RO( double, dcomplex, double, dcomplex, d, z, d, z, __VA_ARGS__ )



// -- Mixed precision real-only two-operand with complex projection of both operands --

#define INSERT_GENTFUNC2RO_MIX_P( ... ) \
\
GENTFUNC2RO( float,  scomplex, double, dcomplex, s, c, d, z, __VA_ARGS__ ) \
GENTFUNC2RO( double, dcomplex, float,  scomplex, d, z, s, c, __VA_ARGS__ )



// -- Mixed domain/precision (all) two-operand macro with real projection of second operand --

#define INSERT_GENTFUNC2R_MIX_DP( ... ) \
\
GENTFUNC2R( float,    double,   double,   s, d, d, __VA_ARGS__ ) \
GENTFUNC2R( float,    scomplex, float,    s, c, s, __VA_ARGS__ ) \
GENTFUNC2R( float,    dcomplex, double,   s, z, d, __VA_ARGS__ ) \
\
GENTFUNC2R( double,   float,    float,    d, s, s, __VA_ARGS__ ) \
GENTFUNC2R( double,   scomplex, float,    d, c, s, __VA_ARGS__ ) \
GENTFUNC2R( double,   dcomplex, double,   d, z, d, __VA_ARGS__ ) \
\
GENTFUNC2R( scomplex, float,    float,    c, s, s, __VA_ARGS__ ) \
GENTFUNC2R( scomplex, double,   double,   c, d, d, __VA_ARGS__ ) \
GENTFUNC2R( scomplex, dcomplex, double,   c, z, d, __VA_ARGS__ ) \
\
GENTFUNC2R( dcomplex, float,    float,    z, s, s, __VA_ARGS__ ) \
GENTFUNC2R( dcomplex, double,   double,   z, d, d, __VA_ARGS__ ) \
GENTFUNC2R( dcomplex, scomplex, float,    z, c, s, __VA_ARGS__ )




// -- Macros for functions with three primary operands -------------------------


// -- Basic three-operand macro --

#define INSERT_GENTFUNC3_BASIC( ... ) \
\
GENTFUNC3( float,    float,    float,    s, s, s, __VA_ARGS__ ) \
GENTFUNC3( double,   double,   double,   d, d, d, __VA_ARGS__ ) \
GENTFUNC3( scomplex, scomplex, scomplex, c, c, c, __VA_ARGS__ ) \
GENTFUNC3( dcomplex, dcomplex, dcomplex, z, z, z, __VA_ARGS__ )



// -- Mixed domain three-operand macro --

#define INSERT_GENTFUNC3_MIX_D( ... ) \
\
GENTFUNC3( float,    float,    scomplex, s, s, c, __VA_ARGS__ ) \
GENTFUNC3( float,    scomplex, float,    s, c, s, __VA_ARGS__ ) \
GENTFUNC3( float,    scomplex, scomplex, s, c, c, __VA_ARGS__ ) \
\
GENTFUNC3( double,   double,   dcomplex, d, d, z, __VA_ARGS__ ) \
GENTFUNC3( double,   dcomplex, double,   d, z, d, __VA_ARGS__ ) \
GENTFUNC3( double,   dcomplex, dcomplex, d, z, z, __VA_ARGS__ ) \
\
GENTFUNC3( scomplex, float,    float,    c, s, s, __VA_ARGS__ ) \
GENTFUNC3( scomplex, float,    scomplex, c, s, c, __VA_ARGS__ ) \
GENTFUNC3( scomplex, scomplex, float,    c, c, s, __VA_ARGS__ ) \
\
GENTFUNC3( dcomplex, double,   double,   z, d, d, __VA_ARGS__ ) \
GENTFUNC3( dcomplex, double,   dcomplex, z, d, z, __VA_ARGS__ ) \
GENTFUNC3( dcomplex, dcomplex, double,   z, z, d, __VA_ARGS__ )



// -- Mixed precision three-operand macro --

#define INSERT_GENTFUNC3_MIX_P( ... ) \
\
GENTFUNC3( float,    float,    double,   s, s, d, __VA_ARGS__ ) \
GENTFUNC3( float,    float,    dcomplex, s, s, z, __VA_ARGS__ ) \
\
GENTFUNC3( float,    double,   float,    s, d, s, __VA_ARGS__ ) \
GENTFUNC3( float,    double,   double,   s, d, d, __VA_ARGS__ ) \
GENTFUNC3( float,    double,   scomplex, s, d, c, __VA_ARGS__ ) \
GENTFUNC3( float,    double,   dcomplex, s, d, z, __VA_ARGS__ ) \
\
GENTFUNC3( float,    scomplex, double,   s, c, d, __VA_ARGS__ ) \
GENTFUNC3( float,    scomplex, dcomplex, s, c, z, __VA_ARGS__ ) \
\
GENTFUNC3( float,    dcomplex, float,    s, z, s, __VA_ARGS__ ) \
GENTFUNC3( float,    dcomplex, double,   s, z, d, __VA_ARGS__ ) \
GENTFUNC3( float,    dcomplex, scomplex, s, z, c, __VA_ARGS__ ) \
GENTFUNC3( float,    dcomplex, dcomplex, s, z, z, __VA_ARGS__ ) \
\
\
GENTFUNC3( double,   float,    float,    d, s, s, __VA_ARGS__ ) \
GENTFUNC3( double,   float,    double,   d, s, d, __VA_ARGS__ ) \
GENTFUNC3( double,   float,    scomplex, d, s, c, __VA_ARGS__ ) \
GENTFUNC3( double,   float,    dcomplex, d, s, z, __VA_ARGS__ ) \
\
GENTFUNC3( double,   double,   float,    d, d, s, __VA_ARGS__ ) \
GENTFUNC3( double,   double,   scomplex, d, d, c, __VA_ARGS__ ) \
\
GENTFUNC3( double,   scomplex, float,    d, c, s, __VA_ARGS__ ) \
GENTFUNC3( double,   scomplex, double,   d, c, d, __VA_ARGS__ ) \
GENTFUNC3( double,   scomplex, scomplex, d, c, c, __VA_ARGS__ ) \
GENTFUNC3( double,   scomplex, dcomplex, d, c, z, __VA_ARGS__ ) \
\
GENTFUNC3( double,   dcomplex, float,    d, z, s, __VA_ARGS__ ) \
GENTFUNC3( double,   dcomplex, scomplex, d, z, c, __VA_ARGS__ ) \
\
\
GENTFUNC3( scomplex, float,    double,   c, s, d, __VA_ARGS__ ) \
GENTFUNC3( scomplex, float,    dcomplex, c, s, z, __VA_ARGS__ ) \
\
GENTFUNC3( scomplex, double,   float,    c, d, s, __VA_ARGS__ ) \
GENTFUNC3( scomplex, double,   double,   c, d, d, __VA_ARGS__ ) \
GENTFUNC3( scomplex, double,   scomplex, c, d, c, __VA_ARGS__ ) \
GENTFUNC3( scomplex, double,   dcomplex, c, d, z, __VA_ARGS__ ) \
\
GENTFUNC3( scomplex, scomplex, double,   c, c, d, __VA_ARGS__ ) \
GENTFUNC3( scomplex, scomplex, dcomplex, c, c, z, __VA_ARGS__ ) \
\
GENTFUNC3( scomplex, dcomplex, float,    c, z, s, __VA_ARGS__ ) \
GENTFUNC3( scomplex, dcomplex, double,   c, z, d, __VA_ARGS__ ) \
GENTFUNC3( scomplex, dcomplex, scomplex, c, z, c, __VA_ARGS__ ) \
GENTFUNC3( scomplex, dcomplex, dcomplex, c, z, z, __VA_ARGS__ ) \
\
\
GENTFUNC3( dcomplex, float,    float,    z, s, s, __VA_ARGS__ ) \
GENTFUNC3( dcomplex, float,    double,   z, s, d, __VA_ARGS__ ) \
GENTFUNC3( dcomplex, float,    scomplex, z, s, c, __VA_ARGS__ ) \
GENTFUNC3( dcomplex, float,    dcomplex, z, s, z, __VA_ARGS__ ) \
\
GENTFUNC3( dcomplex, double,   float,    z, d, s, __VA_ARGS__ ) \
GENTFUNC3( dcomplex, double,   scomplex, z, d, c, __VA_ARGS__ ) \
\
GENTFUNC3( dcomplex, scomplex, float,    z, c, s, __VA_ARGS__ ) \
GENTFUNC3( dcomplex, scomplex, double,   z, c, d, __VA_ARGS__ ) \
GENTFUNC3( dcomplex, scomplex, scomplex, z, c, c, __VA_ARGS__ ) \
GENTFUNC3( dcomplex, scomplex, dcomplex, z, c, z, __VA_ARGS__ ) \
\
GENTFUNC3( dcomplex, dcomplex, float,    z, z, s, __VA_ARGS__ ) \
GENTFUNC3( dcomplex, dcomplex, scomplex, z, z, c, __VA_ARGS__ )



// -- Basic three-operand with union of operands 1 and 2 --

#define INSERT_GENTFUNC3U12_BASIC( ... ) \
\
GENTFUNC3U12( float,    float,    float,    float,    s, s, s, s, __VA_ARGS__ ) \
GENTFUNC3U12( double,   double,   double,   double,   d, d, d, d, __VA_ARGS__ ) \
GENTFUNC3U12( scomplex, scomplex, scomplex, scomplex, c, c, c, c, __VA_ARGS__ ) \
GENTFUNC3U12( dcomplex, dcomplex, dcomplex, dcomplex, z, z, z, z, __VA_ARGS__ )



// -- Mixed domain three-operand with union of operands 1 and 2 --

#define INSERT_GENTFUNC3U12_MIX_D( ... ) \
\
GENTFUNC3U12( float,    float,    scomplex, float,    s, s, c, s, __VA_ARGS__ ) \
GENTFUNC3U12( float,    scomplex, float,    scomplex, s, c, s, c, __VA_ARGS__ ) \
GENTFUNC3U12( float,    scomplex, scomplex, scomplex, s, c, c, c, __VA_ARGS__ ) \
\
GENTFUNC3U12( double,   double,   dcomplex, double,   d, d, z, d, __VA_ARGS__ ) \
GENTFUNC3U12( double,   dcomplex, double,   dcomplex, d, z, d, z, __VA_ARGS__ ) \
GENTFUNC3U12( double,   dcomplex, dcomplex, dcomplex, d, z, z, z, __VA_ARGS__ ) \
\
GENTFUNC3U12( scomplex, float,    float,    scomplex, c, s, s, c, __VA_ARGS__ ) \
GENTFUNC3U12( scomplex, float,    scomplex, scomplex, c, s, c, c, __VA_ARGS__ ) \
GENTFUNC3U12( scomplex, scomplex, float,    scomplex, c, c, s, c, __VA_ARGS__ ) \
\
GENTFUNC3U12( dcomplex, double,   double,   dcomplex, z, d, d, z, __VA_ARGS__ ) \
GENTFUNC3U12( dcomplex, double,   dcomplex, dcomplex, z, d, z, z, __VA_ARGS__ ) \
GENTFUNC3U12( dcomplex, dcomplex, double,   dcomplex, z, z, d, z, __VA_ARGS__ )



// -- Mixed precision three-operand with union of operands 1 and 2 --

#define INSERT_GENTFUNC3U12_MIX_P( ... ) \
\
GENTFUNC3U12( float,    float,    double,   float,    s, s, d, s, __VA_ARGS__ ) \
GENTFUNC3U12( float,    float,    dcomplex, float,    s, s, z, s, __VA_ARGS__ ) \
\
GENTFUNC3U12( float,    double,   float,    double,   s, d, s, d, __VA_ARGS__ ) \
GENTFUNC3U12( float,    double,   double,   double,   s, d, d, d, __VA_ARGS__ ) \
GENTFUNC3U12( float,    double,   scomplex, double,   s, d, c, d, __VA_ARGS__ ) \
GENTFUNC3U12( float,    double,   dcomplex, double,   s, d, z, d, __VA_ARGS__ ) \
\
GENTFUNC3U12( float,    scomplex, double,   scomplex, s, c, d, c, __VA_ARGS__ ) \
GENTFUNC3U12( float,    scomplex, dcomplex, scomplex, s, c, z, c, __VA_ARGS__ ) \
\
GENTFUNC3U12( float,    dcomplex, float,    dcomplex, s, z, s, z, __VA_ARGS__ ) \
GENTFUNC3U12( float,    dcomplex, double,   dcomplex, s, z, d, z, __VA_ARGS__ ) \
GENTFUNC3U12( float,    dcomplex, scomplex, dcomplex, s, z, c, z, __VA_ARGS__ ) \
GENTFUNC3U12( float,    dcomplex, dcomplex, dcomplex, s, z, z, z, __VA_ARGS__ ) \
\
\
GENTFUNC3U12( double,   float,    float,    double,   d, s, s, d, __VA_ARGS__ ) \
GENTFUNC3U12( double,   float,    double,   double,   d, s, d, d, __VA_ARGS__ ) \
GENTFUNC3U12( double,   float,    scomplex, double,   d, s, c, d, __VA_ARGS__ ) \
GENTFUNC3U12( double,   float,    dcomplex, double,   d, s, z, d, __VA_ARGS__ ) \
\
GENTFUNC3U12( double,   double,   float,    double,   d, d, s, d, __VA_ARGS__ ) \
GENTFUNC3U12( double,   double,   scomplex, double,   d, d, c, d, __VA_ARGS__ ) \
\
GENTFUNC3U12( double,   scomplex, float,    dcomplex, d, c, s, z, __VA_ARGS__ ) \
GENTFUNC3U12( double,   scomplex, double,   dcomplex, d, c, d, z, __VA_ARGS__ ) \
GENTFUNC3U12( double,   scomplex, scomplex, dcomplex, d, c, c, z, __VA_ARGS__ ) \
GENTFUNC3U12( double,   scomplex, dcomplex, dcomplex, d, c, z, z, __VA_ARGS__ ) \
\
GENTFUNC3U12( double,   dcomplex, float,    dcomplex, d, z, s, z, __VA_ARGS__ ) \
GENTFUNC3U12( double,   dcomplex, scomplex, dcomplex, d, z, c, z, __VA_ARGS__ ) \
\
\
GENTFUNC3U12( scomplex, float,    double,   scomplex, c, s, d, c, __VA_ARGS__ ) \
GENTFUNC3U12( scomplex, float,    dcomplex, scomplex, c, s, z, c, __VA_ARGS__ ) \
\
GENTFUNC3U12( scomplex, double,   float,    dcomplex, c, d, s, z, __VA_ARGS__ ) \
GENTFUNC3U12( scomplex, double,   double,   dcomplex, c, d, d, z, __VA_ARGS__ ) \
GENTFUNC3U12( scomplex, double,   scomplex, dcomplex, c, d, c, z, __VA_ARGS__ ) \
GENTFUNC3U12( scomplex, double,   dcomplex, dcomplex, c, d, z, z, __VA_ARGS__ ) \
\
GENTFUNC3U12( scomplex, scomplex, double,   scomplex, c, c, d, c, __VA_ARGS__ ) \
GENTFUNC3U12( scomplex, scomplex, dcomplex, scomplex, c, c, z, c, __VA_ARGS__ ) \
\
GENTFUNC3U12( scomplex, dcomplex, float,    dcomplex, c, z, s, z, __VA_ARGS__ ) \
GENTFUNC3U12( scomplex, dcomplex, double,   dcomplex, c, z, d, z, __VA_ARGS__ ) \
GENTFUNC3U12( scomplex, dcomplex, scomplex, dcomplex, c, z, c, z, __VA_ARGS__ ) \
GENTFUNC3U12( scomplex, dcomplex, dcomplex, dcomplex, c, z, z, z, __VA_ARGS__ ) \
\
\
GENTFUNC3U12( dcomplex, float,    float,    dcomplex, z, s, s, z, __VA_ARGS__ ) \
GENTFUNC3U12( dcomplex, float,    double,   dcomplex, z, s, d, z, __VA_ARGS__ ) \
GENTFUNC3U12( dcomplex, float,    scomplex, dcomplex, z, s, c, z, __VA_ARGS__ ) \
GENTFUNC3U12( dcomplex, float,    dcomplex, dcomplex, z, s, z, z, __VA_ARGS__ ) \
\
GENTFUNC3U12( dcomplex, double,   float,    dcomplex, z, d, s, z, __VA_ARGS__ ) \
GENTFUNC3U12( dcomplex, double,   scomplex, dcomplex, z, d, c, z, __VA_ARGS__ ) \
\
GENTFUNC3U12( dcomplex, scomplex, float,    dcomplex, z, c, s, z, __VA_ARGS__ ) \
GENTFUNC3U12( dcomplex, scomplex, double,   dcomplex, z, c, d, z, __VA_ARGS__ ) \
GENTFUNC3U12( dcomplex, scomplex, scomplex, dcomplex, z, c, c, z, __VA_ARGS__ ) \
GENTFUNC3U12( dcomplex, scomplex, dcomplex, dcomplex, z, c, z, z, __VA_ARGS__ ) \
\
GENTFUNC3U12( dcomplex, dcomplex, float,    dcomplex, z, z, s, z, __VA_ARGS__ ) \
GENTFUNC3U12( dcomplex, dcomplex, scomplex, dcomplex, z, z, c, z, __VA_ARGS__ )


#endif
// end bli_gentfunc_macro_defs.h
#line 97 "../blis-2.0/frame/include//bli_macro_defs.h"

// begin bli_gentprot_macro_defs.h
#line 1 "../blis-2.0/frame/include//bli_gentprot_macro_defs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/


#ifndef BLIS_GENTPROT_MACRO_DEFS_H
#define BLIS_GENTPROT_MACRO_DEFS_H

//
// -- MACROS TO INSERT PROTOTYPE-GENERATING MACROS -----------------------------
//



// -- Macros for generating BLAS routines --------------------------------------


// -- Basic one-operand macro --


#define INSERT_GENTPROT_BLAS( blasname ) \
\
GENTPROT( float,    s, blasname ) \
GENTPROT( double,   d, blasname ) \
GENTPROT( scomplex, c, blasname ) \
GENTPROT( dcomplex, z, blasname )


// -- Basic one-operand macro with real domain only --


#define INSERT_GENTPROTRO_BLAS( blasname ) \
\
GENTPROTRO( float,    s, blasname ) \
GENTPROTRO( double,   d, blasname )


// -- Basic one-operand macro with complex domain only and real projection --


#define INSERT_GENTPROTCO_BLAS( blasname ) \
\
GENTPROTCO( scomplex, float,  c, s, blasname ) \
GENTPROTCO( dcomplex, double, z, d, blasname )


// -- Basic one-operand macro with conjugation (real funcs only, used only for dot, ger) --


#define INSERT_GENTPROTDOTR_BLAS( blasname ) \
\
GENTPROTDOT( float,    s,  , blasname ) \
GENTPROTDOT( double,   d,  , blasname )


// -- Basic one-operand macro with conjugation (complex funcs only, used only for dot, ger) --


#define INSERT_GENTPROTDOTC_BLAS( blasname ) \
\
GENTPROTDOT( scomplex, c, c, blasname ) \
GENTPROTDOT( scomplex, c, u, blasname ) \
GENTPROTDOT( dcomplex, z, c, blasname ) \
GENTPROTDOT( dcomplex, z, u, blasname )


// -- Basic one-operand macro with conjugation (used only for dot, ger) --


#define INSERT_GENTPROTDOT_BLAS( blasname ) \
\
INSERT_GENTPROTDOTR_BLAS( blasname ) \
INSERT_GENTPROTDOTC_BLAS( blasname )


// -- Basic one-operand macro with real projection --


#define INSERT_GENTPROTR_BLAS( rblasname, cblasname ) \
\
GENTPROTR( float,    float,  s, s, rblasname ) \
GENTPROTR( double,   double, d, d, rblasname ) \
GENTPROTR( scomplex, float,  c, s, cblasname ) \
GENTPROTR( dcomplex, double, z, d, cblasname )


// -- Alternate two-operand macro (one char for complex, one for real proj) --


#define INSERT_GENTPROTR2_BLAS( blasname ) \
\
GENTPROTR2( float,    float,   , s, blasname ) \
GENTPROTR2( double,   double,  , d, blasname ) \
GENTPROTR2( scomplex, float,  c, s, blasname ) \
GENTPROTR2( dcomplex, double, z, d, blasname )


// -- Extended two-operand macro (used only for scal) --


#define INSERT_GENTPROTSCAL_BLAS( blasname ) \
\
GENTPROTSCAL( float,    float,     , s, blasname ) \
GENTPROTSCAL( double,   double,    , d, blasname ) \
GENTPROTSCAL( scomplex, scomplex,  , c, blasname ) \
GENTPROTSCAL( dcomplex, dcomplex,  , z, blasname ) \
GENTPROTSCAL( float,    scomplex, s, c, blasname ) \
GENTPROTSCAL( double,   dcomplex, d, z, blasname )




// -- Macros for functions with one operand ------------------------------------


// -- Basic one-operand macro --

#define INSERT_GENTPROT_BASIC( ... ) \
\
GENTPROT( float,    s, __VA_ARGS__ ) \
GENTPROT( double,   d, __VA_ARGS__ ) \
GENTPROT( scomplex, c, __VA_ARGS__ ) \
GENTPROT( dcomplex, z, __VA_ARGS__ )



// -- Basic one-operand with real projection --

#define INSERT_GENTPROTR_BASIC( ... ) \
\
GENTPROTR( float,    float,  s, s, __VA_ARGS__ ) \
GENTPROTR( double,   double, d, d, __VA_ARGS__ ) \
GENTPROTR( scomplex, float,  c, s, __VA_ARGS__ ) \
GENTPROTR( dcomplex, double, z, d, __VA_ARGS__ )



// -- Basic one-operand macro with real domain only --

#define INSERT_GENTPROTRO_BASIC( ... ) \
\
GENTPROTRO( float,  s, __VA_ARGS__ ) \
GENTPROTRO( double, d, __VA_ARGS__ )



// -- Basic one-operand macro with complex domain only and real projection --

#define INSERT_GENTPROTCO_BASIC( ... ) \
\
GENTPROTCO( scomplex, float,  c, s, __VA_ARGS__ ) \
GENTPROTCO( dcomplex, double, z, d, __VA_ARGS__ )



// -- Basic one-operand macro with integer instance --

#define INSERT_GENTPROT_BASIC_I( ... ) \
\
GENTPROT( float,    s, __VA_ARGS__ ) \
GENTPROT( double,   d, __VA_ARGS__ ) \
GENTPROT( scomplex, c, __VA_ARGS__ ) \
GENTPROT( dcomplex, z, __VA_ARGS__ ) \
GENTPROT( gint_t,   i, __VA_ARGS__ )



// -- Basic one-operand with integer projection --

#define INSERT_GENTPROTI_BASIC( ... ) \
\
GENTPROTI( float,    gint_t, s, i, __VA_ARGS__ ) \
GENTPROTI( double,   gint_t, d, i, __VA_ARGS__ ) \
GENTPROTI( scomplex, gint_t, c, i, __VA_ARGS__ ) \
GENTPROTI( dcomplex, gint_t, z, i, __VA_ARGS__ )



// -- Basic one-operand with real and integer projections --

#define INSERT_GENTPROTRI_BASIC( funcname ) \
\
GENTPROTRI( float,    float,  gint_t, s, s, i, funcname ) \
GENTPROTRI( double,   double, gint_t, d, d, i, funcname ) \
GENTPROTRI( scomplex, float,  gint_t, c, s, i, funcname ) \
GENTPROTRI( dcomplex, double, gint_t, z, d, i, funcname )




// -- Macros for functions with two primary operands ---------------------------


// -- Basic two-operand macro --

#define INSERT_GENTPROT2_BASIC( ... ) \
\
GENTPROT2( float,    float,    s, s, __VA_ARGS__ ) \
GENTPROT2( double,   double,   d, d, __VA_ARGS__ ) \
GENTPROT2( scomplex, scomplex, c, c, __VA_ARGS__ ) \
GENTPROT2( dcomplex, dcomplex, z, z, __VA_ARGS__ )



// -- Mixed domain two-operand macro --

#define INSERT_GENTPROT2_MIX_D( ... ) \
\
GENTPROT2( float,    scomplex, s, c, __VA_ARGS__ ) \
GENTPROT2( scomplex, float,    c, s, __VA_ARGS__ ) \
\
GENTPROT2( double,   dcomplex, d, z, __VA_ARGS__ ) \
GENTPROT2( dcomplex, double,   z, d, __VA_ARGS__ )



// -- Mixed precision two-operand macro --

#define INSERT_GENTPROT2_MIX_P( ... ) \
\
GENTPROT2( float,    double,   s, d, __VA_ARGS__ ) \
GENTPROT2( float,    dcomplex, s, z, __VA_ARGS__ ) \
\
GENTPROT2( double,   float,    d, s, __VA_ARGS__ ) \
GENTPROT2( double,   scomplex, d, c, __VA_ARGS__ ) \
\
GENTPROT2( scomplex, double,   c, d, __VA_ARGS__ ) \
GENTPROT2( scomplex, dcomplex, c, z, __VA_ARGS__ ) \
\
GENTPROT2( dcomplex, float,    z, s, __VA_ARGS__ ) \
GENTPROT2( dcomplex, scomplex, z, c, __VA_ARGS__ ) \



// -- Mixed domain/precision (all) two-operand macro --

#define INSERT_GENTPROT2_MIX_DP( ... ) \
\
GENTPROT2( float,    double,   s, d, __VA_ARGS__ ) \
GENTPROT2( float,    scomplex, s, c, __VA_ARGS__ ) \
GENTPROT2( float,    dcomplex, s, z, __VA_ARGS__ ) \
\
GENTPROT2( double,   float,    d, s, __VA_ARGS__ ) \
GENTPROT2( double,   scomplex, d, c, __VA_ARGS__ ) \
GENTPROT2( double,   dcomplex, d, z, __VA_ARGS__ ) \
\
GENTPROT2( scomplex, float,    c, s, __VA_ARGS__ ) \
GENTPROT2( scomplex, double,   c, d, __VA_ARGS__ ) \
GENTPROT2( scomplex, dcomplex, c, z, __VA_ARGS__ ) \
\
GENTPROT2( dcomplex, float,    z, s, __VA_ARGS__ ) \
GENTPROT2( dcomplex, double,   z, d, __VA_ARGS__ ) \
GENTPROT2( dcomplex, scomplex, z, c, __VA_ARGS__ )



// -- Basic two-operand with real projection of first operand --

#define INSERT_GENTPROT2R_BASIC( ... ) \
\
GENTPROT2R( float,    float,    float,    s, s, s, __VA_ARGS__ ) \
GENTPROT2R( double,   double,   double,   d, d, d, __VA_ARGS__ ) \
GENTPROT2R( scomplex, scomplex, float,    c, c, s, __VA_ARGS__ ) \
GENTPROT2R( dcomplex, dcomplex, double,   z, z, d, __VA_ARGS__ )



// -- Mixed domain two-operand with real projection of first operand --

#define INSERT_GENTPROT2R_MIX_D( ... ) \
\
GENTPROT2R( float,    scomplex, float,    s, c, s, __VA_ARGS__ ) \
GENTPROT2R( scomplex, float,    float,    c, s, s, __VA_ARGS__ ) \
\
GENTPROT2R( double,   dcomplex, double,   d, z, d, __VA_ARGS__ ) \
GENTPROT2R( dcomplex, double,   double,   z, d, d, __VA_ARGS__ )



// -- Mixed precision two-operand with real projection of first operand --

#define INSERT_GENTPROT2R_MIX_P( ... ) \
\
GENTPROT2R( float,    double,   float,    s, d, s, __VA_ARGS__ ) \
GENTPROT2R( float,    dcomplex, float,    s, z, s, __VA_ARGS__ ) \
\
GENTPROT2R( double,   float,    double,   d, s, d, __VA_ARGS__ ) \
GENTPROT2R( double,   scomplex, double,   d, c, d, __VA_ARGS__ ) \
\
GENTPROT2R( scomplex, double,   float,    c, d, s, __VA_ARGS__ ) \
GENTPROT2R( scomplex, dcomplex, float,    c, z, s, __VA_ARGS__ ) \
\
GENTPROT2R( dcomplex, float,    double,   z, s, d, __VA_ARGS__ ) \
GENTPROT2R( dcomplex, scomplex, double,   z, c, d, __VA_ARGS__ )



// -- Macros for functions with three primary operands -------------------------


// -- Basic three-operand macro --


#define INSERT_GENTPROT3_BASIC( funcname ) \
\
GENTPROT3( float,    float,    float,    s, s, s, funcname ) \
GENTPROT3( double,   double,   double,   d, d, d, funcname ) \
GENTPROT3( scomplex, scomplex, scomplex, c, c, c, funcname ) \
GENTPROT3( dcomplex, dcomplex, dcomplex, z, z, z, funcname )


// -- Mixed domain three-operand macro --


#define INSERT_GENTPROT3_MIX_D( funcname ) \
\
GENTPROT3( float,    float,    scomplex, s, s, c, funcname ) \
GENTPROT3( float,    scomplex, float,    s, c, s, funcname ) \
GENTPROT3( float,    scomplex, scomplex, s, c, c, funcname ) \
\
GENTPROT3( double,   double,   dcomplex, d, d, z, funcname ) \
GENTPROT3( double,   dcomplex, double,   d, z, d, funcname ) \
GENTPROT3( double,   dcomplex, dcomplex, d, z, z, funcname ) \
\
GENTPROT3( scomplex, float,    float,    c, s, s, funcname ) \
GENTPROT3( scomplex, float,    scomplex, c, s, c, funcname ) \
GENTPROT3( scomplex, scomplex, float,    c, c, s, funcname ) \
\
GENTPROT3( dcomplex, double,   double,   z, d, d, funcname ) \
GENTPROT3( dcomplex, double,   dcomplex, z, d, z, funcname ) \
GENTPROT3( dcomplex, dcomplex, double,   z, z, d, funcname )


// -- Mixed precision three-operand macro --


#define INSERT_GENTPROT3_MIX_P( funcname ) \
\
GENTPROT3( float,    float,    double,   s, s, d, funcname ) \
GENTPROT3( float,    float,    dcomplex, s, s, z, funcname ) \
\
GENTPROT3( float,    double,   float,    s, d, s, funcname ) \
GENTPROT3( float,    double,   double,   s, d, d, funcname ) \
GENTPROT3( float,    double,   scomplex, s, d, c, funcname ) \
GENTPROT3( float,    double,   dcomplex, s, d, z, funcname ) \
\
GENTPROT3( float,    scomplex, double,   s, c, d, funcname ) \
GENTPROT3( float,    scomplex, dcomplex, s, c, z, funcname ) \
\
GENTPROT3( float,    dcomplex, float,    s, z, s, funcname ) \
GENTPROT3( float,    dcomplex, double,   s, z, d, funcname ) \
GENTPROT3( float,    dcomplex, scomplex, s, z, c, funcname ) \
GENTPROT3( float,    dcomplex, dcomplex, s, z, z, funcname ) \
\
\
GENTPROT3( double,   float,    float,    d, s, s, funcname ) \
GENTPROT3( double,   float,    double,   d, s, d, funcname ) \
GENTPROT3( double,   float,    scomplex, d, s, c, funcname ) \
GENTPROT3( double,   float,    dcomplex, d, s, z, funcname ) \
\
GENTPROT3( double,   double,   float,    d, d, s, funcname ) \
GENTPROT3( double,   double,   scomplex, d, d, c, funcname ) \
\
GENTPROT3( double,   scomplex, float,    d, c, s, funcname ) \
GENTPROT3( double,   scomplex, double,   d, c, d, funcname ) \
GENTPROT3( double,   scomplex, scomplex, d, c, c, funcname ) \
GENTPROT3( double,   scomplex, dcomplex, d, c, z, funcname ) \
\
GENTPROT3( double,   dcomplex, float,    d, z, s, funcname ) \
GENTPROT3( double,   dcomplex, scomplex, d, z, c, funcname ) \
\
\
GENTPROT3( scomplex, float,    double,   c, s, d, funcname ) \
GENTPROT3( scomplex, float,    dcomplex, c, s, z, funcname ) \
\
GENTPROT3( scomplex, double,   float,    c, d, s, funcname ) \
GENTPROT3( scomplex, double,   double,   c, d, d, funcname ) \
GENTPROT3( scomplex, double,   scomplex, c, d, c, funcname ) \
GENTPROT3( scomplex, double,   dcomplex, c, d, z, funcname ) \
\
GENTPROT3( scomplex, scomplex, double,   c, c, d, funcname ) \
GENTPROT3( scomplex, scomplex, dcomplex, c, c, z, funcname ) \
\
GENTPROT3( scomplex, dcomplex, float,    c, z, s, funcname ) \
GENTPROT3( scomplex, dcomplex, double,   c, z, d, funcname ) \
GENTPROT3( scomplex, dcomplex, scomplex, c, z, c, funcname ) \
GENTPROT3( scomplex, dcomplex, dcomplex, c, z, z, funcname ) \
\
\
GENTPROT3( dcomplex, float,    float,    z, s, s, funcname ) \
GENTPROT3( dcomplex, float,    double,   z, s, d, funcname ) \
GENTPROT3( dcomplex, float,    scomplex, z, s, c, funcname ) \
GENTPROT3( dcomplex, float,    dcomplex, z, s, z, funcname ) \
\
GENTPROT3( dcomplex, double,   float,    z, d, s, funcname ) \
GENTPROT3( dcomplex, double,   scomplex, z, d, c, funcname ) \
\
GENTPROT3( dcomplex, scomplex, float,    z, c, s, funcname ) \
GENTPROT3( dcomplex, scomplex, double,   z, c, d, funcname ) \
GENTPROT3( dcomplex, scomplex, scomplex, z, c, c, funcname ) \
GENTPROT3( dcomplex, scomplex, dcomplex, z, c, z, funcname ) \
\
GENTPROT3( dcomplex, dcomplex, float,    z, z, s, funcname ) \
GENTPROT3( dcomplex, dcomplex, scomplex, z, z, c, funcname ) \



// -- Basic three-operand with union of operands 1 and 2 --


#define INSERT_GENTPROT3U12_BASIC( funcname ) \
\
GENTPROT3U12( float,    float,    float,    float,    s, s, s, s, funcname ) \
GENTPROT3U12( double,   double,   double,   double,   d, d, d, d, funcname ) \
GENTPROT3U12( scomplex, scomplex, scomplex, scomplex, c, c, c, c, funcname ) \
GENTPROT3U12( dcomplex, dcomplex, dcomplex, dcomplex, z, z, z, z, funcname )


// -- Mixed domain three-operand with union of operands 1 and 2 --


#define INSERT_GENTPROT3U12_MIX_D( funcname ) \
\
GENTPROT3U12( float,    float,    scomplex, float,    s, s, c, s, funcname ) \
GENTPROT3U12( float,    scomplex, float,    scomplex, s, c, s, c, funcname ) \
GENTPROT3U12( float,    scomplex, scomplex, scomplex, s, c, c, c, funcname ) \
\
GENTPROT3U12( double,   double,   dcomplex, double,   d, d, z, d, funcname ) \
GENTPROT3U12( double,   dcomplex, double,   dcomplex, d, z, d, z, funcname ) \
GENTPROT3U12( double,   dcomplex, dcomplex, dcomplex, d, z, z, z, funcname ) \
\
GENTPROT3U12( scomplex, float,    float,    scomplex, c, s, s, c, funcname ) \
GENTPROT3U12( scomplex, float,    scomplex, scomplex, c, s, c, c, funcname ) \
GENTPROT3U12( scomplex, scomplex, float,    scomplex, c, c, s, c, funcname ) \
\
GENTPROT3U12( dcomplex, double,   double,   dcomplex, z, d, d, z, funcname ) \
GENTPROT3U12( dcomplex, double,   dcomplex, dcomplex, z, d, z, z, funcname ) \
GENTPROT3U12( dcomplex, dcomplex, double,   dcomplex, z, z, d, z, funcname )


// -- Mixed precision three-operand with union of operands 1 and 2 --


#define INSERT_GENTPROT3U12_MIX_P( funcname ) \
\
GENTPROT3U12( float,    float,    double,   float,    s, s, d, s, funcname ) \
GENTPROT3U12( float,    float,    dcomplex, float,    s, s, z, s, funcname ) \
\
GENTPROT3U12( float,    double,   float,    double,   s, d, s, d, funcname ) \
GENTPROT3U12( float,    double,   double,   double,   s, d, d, d, funcname ) \
GENTPROT3U12( float,    double,   scomplex, double,   s, d, c, d, funcname ) \
GENTPROT3U12( float,    double,   dcomplex, double,   s, d, z, d, funcname ) \
\
GENTPROT3U12( float,    scomplex, double,   scomplex, s, c, d, c, funcname ) \
GENTPROT3U12( float,    scomplex, dcomplex, scomplex, s, c, z, c, funcname ) \
\
GENTPROT3U12( float,    dcomplex, float,    dcomplex, s, z, s, z, funcname ) \
GENTPROT3U12( float,    dcomplex, double,   dcomplex, s, z, d, z, funcname ) \
GENTPROT3U12( float,    dcomplex, scomplex, dcomplex, s, z, c, z, funcname ) \
GENTPROT3U12( float,    dcomplex, dcomplex, dcomplex, s, z, z, z, funcname ) \
\
\
GENTPROT3U12( double,   float,    float,    double,   d, s, s, d, funcname ) \
GENTPROT3U12( double,   float,    double,   double,   d, s, d, d, funcname ) \
GENTPROT3U12( double,   float,    scomplex, double,   d, s, c, d, funcname ) \
GENTPROT3U12( double,   float,    dcomplex, double,   d, s, z, d, funcname ) \
\
GENTPROT3U12( double,   double,   float,    double,   d, d, s, d, funcname ) \
GENTPROT3U12( double,   double,   scomplex, double,   d, d, c, d, funcname ) \
\
GENTPROT3U12( double,   scomplex, float,    dcomplex, d, c, s, z, funcname ) \
GENTPROT3U12( double,   scomplex, double,   dcomplex, d, c, d, z, funcname ) \
GENTPROT3U12( double,   scomplex, scomplex, dcomplex, d, c, c, z, funcname ) \
GENTPROT3U12( double,   scomplex, dcomplex, dcomplex, d, c, z, z, funcname ) \
\
GENTPROT3U12( double,   dcomplex, float,    dcomplex, d, z, s, z, funcname ) \
GENTPROT3U12( double,   dcomplex, scomplex, dcomplex, d, z, c, z, funcname ) \
\
\
GENTPROT3U12( scomplex, float,    double,   scomplex, c, s, d, c, funcname ) \
GENTPROT3U12( scomplex, float,    dcomplex, scomplex, c, s, z, c, funcname ) \
\
GENTPROT3U12( scomplex, double,   float,    dcomplex, c, d, s, z, funcname ) \
GENTPROT3U12( scomplex, double,   double,   dcomplex, c, d, d, z, funcname ) \
GENTPROT3U12( scomplex, double,   scomplex, dcomplex, c, d, c, z, funcname ) \
GENTPROT3U12( scomplex, double,   dcomplex, dcomplex, c, d, z, z, funcname ) \
\
GENTPROT3U12( scomplex, scomplex, double,   scomplex, c, c, d, c, funcname ) \
GENTPROT3U12( scomplex, scomplex, dcomplex, scomplex, c, c, z, c, funcname ) \
\
GENTPROT3U12( scomplex, dcomplex, float,    dcomplex, c, z, s, z, funcname ) \
GENTPROT3U12( scomplex, dcomplex, double,   dcomplex, c, z, d, z, funcname ) \
GENTPROT3U12( scomplex, dcomplex, scomplex, dcomplex, c, z, c, z, funcname ) \
GENTPROT3U12( scomplex, dcomplex, dcomplex, dcomplex, c, z, z, z, funcname ) \
\
\
GENTPROT3U12( dcomplex, float,    float,    dcomplex, z, s, s, z, funcname ) \
GENTPROT3U12( dcomplex, float,    double,   dcomplex, z, s, d, z, funcname ) \
GENTPROT3U12( dcomplex, float,    scomplex, dcomplex, z, s, c, z, funcname ) \
GENTPROT3U12( dcomplex, float,    dcomplex, dcomplex, z, s, z, z, funcname ) \
\
GENTPROT3U12( dcomplex, double,   float,    dcomplex, z, d, s, z, funcname ) \
GENTPROT3U12( dcomplex, double,   scomplex, dcomplex, z, d, c, z, funcname ) \
\
GENTPROT3U12( dcomplex, scomplex, float,    dcomplex, z, c, s, z, funcname ) \
GENTPROT3U12( dcomplex, scomplex, double,   dcomplex, z, c, d, z, funcname ) \
GENTPROT3U12( dcomplex, scomplex, scomplex, dcomplex, z, c, c, z, funcname ) \
GENTPROT3U12( dcomplex, scomplex, dcomplex, dcomplex, z, c, z, z, funcname ) \
\
GENTPROT3U12( dcomplex, dcomplex, float,    dcomplex, z, z, s, z, funcname ) \
GENTPROT3U12( dcomplex, dcomplex, scomplex, dcomplex, z, z, c, z, funcname )


#endif
// end bli_gentprot_macro_defs.h
#line 98 "../blis-2.0/frame/include//bli_macro_defs.h"

// begin bli_gentconf_macro_defs.h
#line 1 "../blis-2.0/frame/include//bli_gentconf_macro_defs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2023, Southern Methodist University

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_GENTCONF_MACRO_DEFS_H
#define BLIS_GENTCONF_MACRO_DEFS_H


//
// -- MACROS TO INSERT CONFIGURATION-SPECIFIC MACROS ---------------------------
//


// -- configuration-specific macros which are conditionally-enabled --

// -- Intel architectures ------------------------------------------------------

#ifdef BLIS_CONFIG_SKX
#define INSERT_GENTCONF_SKX GENTCONF( SKX, skx )
#else
#define INSERT_GENTCONF_SKX
#endif
#ifdef BLIS_CONFIG_KNL
#define INSERT_GENTCONF_KNL GENTCONF( KNL, knl )
#else
#define INSERT_GENTCONF_KNL
#endif
#ifdef BLIS_CONFIG_KNC
#define INSERT_GENTCONF_KNC GENTCONF( KNC, knc )
#else
#define INSERT_GENTCONF_KNC
#endif
#ifdef BLIS_CONFIG_HASWELL
#define INSERT_GENTCONF_HASWELL GENTCONF( HASWELL, haswell )
#else
#define INSERT_GENTCONF_HASWELL
#endif
#ifdef BLIS_CONFIG_SANDYBRIDGE
#define INSERT_GENTCONF_SANDYBRIDGE GENTCONF( SANDYBRIDGE, sandybridge )
#else
#define INSERT_GENTCONF_SANDYBRIDGE
#endif
#ifdef BLIS_CONFIG_PENRYN
#define INSERT_GENTCONF_PENRYN GENTCONF( PENRYN, penryn )
#else
#define INSERT_GENTCONF_PENRYN
#endif

// -- AMD architectures --------------------------------------------------------

#ifdef BLIS_CONFIG_ZEN3
#define INSERT_GENTCONF_ZEN3 GENTCONF( ZEN3, zen3 )
#else
#define INSERT_GENTCONF_ZEN3
#endif
#ifdef BLIS_CONFIG_ZEN2
#define INSERT_GENTCONF_ZEN2 GENTCONF( ZEN2, zen2 )
#else
#define INSERT_GENTCONF_ZEN2
#endif
#ifdef BLIS_CONFIG_ZEN
#define INSERT_GENTCONF_ZEN GENTCONF( ZEN, zen )
#else
#define INSERT_GENTCONF_ZEN
#endif
#ifdef BLIS_CONFIG_EXCAVATOR
#define INSERT_GENTCONF_EXCAVATOR GENTCONF( EXCAVATOR, excavator )
#else
#define INSERT_GENTCONF_EXCAVATOR
#endif
#ifdef BLIS_CONFIG_STEAMROLLER
#define INSERT_GENTCONF_STEAMROLLER GENTCONF( STEAMROLLER, steamroller )
#else
#define INSERT_GENTCONF_STEAMROLLER
#endif
#ifdef BLIS_CONFIG_PILEDRIVER
#define INSERT_GENTCONF_PILEDRIVER GENTCONF( PILEDRIVER, piledriver )
#else
#define INSERT_GENTCONF_PILEDRIVER
#endif
#ifdef BLIS_CONFIG_BULLDOZER
#define INSERT_GENTCONF_BULLDOZER GENTCONF( BULLDOZER, bulldozer )
#else
#define INSERT_GENTCONF_BULLDOZER
#endif

// -- ARM architectures --------------------------------------------------------

// -- ARM-SVE --
#ifdef BLIS_CONFIG_ARMSVE
#define INSERT_GENTCONF_ARMSVE GENTCONF( ARMSVE, armsve )
#else
#define INSERT_GENTCONF_ARMSVE
#endif
#ifdef BLIS_CONFIG_A64FX
#define INSERT_GENTCONF_A64FX GENTCONF( A64FX, a64fx )
#else
#define INSERT_GENTCONF_A64FX
#endif

// -- ARM-NEON (4 pipes x 128-bit vectors) --
#ifdef BLIS_CONFIG_ALTRA
#define INSERT_GENTCONF_ALTRA GENTCONF( ALTRA, altra )
#else
#define INSERT_GENTCONF_ALTRA
#endif
#ifdef BLIS_CONFIG_ALTRAMAX
#define INSERT_GENTCONF_ALTRAMAX GENTCONF( ALTRAMAX, altramax )
#else
#define INSERT_GENTCONF_ALTRAMAX
#endif
#ifdef BLIS_CONFIG_FIRESTORM
#define INSERT_GENTCONF_FIRESTORM GENTCONF( FIRESTORM, firestorm )
#else
#define INSERT_GENTCONF_FIRESTORM
#endif

// -- ARM (2 pipes x 128-bit vectors) --
#ifdef BLIS_CONFIG_THUNDERX2
#define INSERT_GENTCONF_THUNDERX2 GENTCONF( THUNDERX2, thunderx2 )
#else
#define INSERT_GENTCONF_THUNDERX2
#endif
#ifdef BLIS_CONFIG_CORTEXA57
#define INSERT_GENTCONF_CORTEXA57 GENTCONF( CORTEXA57, cortexa57 )
#else
#define INSERT_GENTCONF_CORTEXA57
#endif
#ifdef BLIS_CONFIG_CORTEXA53
#define INSERT_GENTCONF_CORTEXA53 GENTCONF( CORTEXA53, cortexa53 )
#else
#define INSERT_GENTCONF_CORTEXA53
#endif

		// -- ARM (older 32-bit microarchitectures) --
#ifdef BLIS_CONFIG_CORTEXA15
#define INSERT_GENTCONF_CORTEXA15 GENTCONF( CORTEXA15, cortexa15 )
#else
#define INSERT_GENTCONF_CORTEXA15
#endif
#ifdef BLIS_CONFIG_CORTEXA9
#define INSERT_GENTCONF_CORTEXA9 GENTCONF( CORTEXA9, cortexa9 )
#else
#define INSERT_GENTCONF_CORTEXA9
#endif

		// -- IBM architectures ------------------------------------------------

#ifdef BLIS_CONFIG_POWER10
#define INSERT_GENTCONF_POWER10 GENTCONF( POWER10, power10 )
#else
#define INSERT_GENTCONF_POWER10
#endif
#ifdef BLIS_CONFIG_POWER9
#define INSERT_GENTCONF_POWER9 GENTCONF( POWER9, power9 )
#else
#define INSERT_GENTCONF_POWER9
#endif
#ifdef BLIS_CONFIG_POWER7
#define INSERT_GENTCONF_POWER7 GENTCONF( POWER7, power7 )
#else
#define INSERT_GENTCONF_POWER7
#endif
#ifdef BLIS_CONFIG_BGQ
#define INSERT_GENTCONF_BGQ GENTCONF( BGQ, bgq )
#else
#define INSERT_GENTCONF_BGQ
#endif

// -- RISC-V architectures ----------------------------------------------------

#ifdef BLIS_CONFIG_RV32I
#define INSERT_GENTCONF_RV32I GENTCONF( RV32I, rv32i )
#else
#define INSERT_GENTCONF_RV32I
#endif
#ifdef BLIS_CONFIG_RV64I
#define INSERT_GENTCONF_RV64I GENTCONF( RV64I, rv64i )
#else
#define INSERT_GENTCONF_RV64I
#endif
#ifdef BLIS_CONFIG_RV32IV
#define INSERT_GENTCONF_RV32IV GENTCONF( RV32IV, rv32iv )
#else
#define INSERT_GENTCONF_RV32IV
#endif
#ifdef BLIS_CONFIG_RV64IV
#define INSERT_GENTCONF_RV64IV GENTCONF( RV64IV, rv64iv )
#else
#define INSERT_GENTCONF_RV64IV
#endif

// -- SiFive architectures ----------------------------------------------------

#ifdef BLIS_CONFIG_SIFIVE_X280
#define INSERT_GENTCONF_SIFIVE_X280 GENTCONF( SIFIVE_X280, sifive_x280 )
#else
#define INSERT_GENTCONF_SIFIVE_X280
#endif

// -- Generic architectures ----------------------------------------------------

#ifdef BLIS_CONFIG_GENERIC
#define INSERT_GENTCONF_GENERIC GENTCONF( GENERIC, generic )
#else
#define INSERT_GENTCONF_GENERIC
#endif


// -- configuration-specific macro --

#define INSERT_GENTCONF \
\
INSERT_GENTCONF_SKX \
INSERT_GENTCONF_KNL \
INSERT_GENTCONF_KNC \
INSERT_GENTCONF_HASWELL \
INSERT_GENTCONF_SANDYBRIDGE \
INSERT_GENTCONF_PENRYN \
\
INSERT_GENTCONF_ZEN3 \
INSERT_GENTCONF_ZEN2 \
INSERT_GENTCONF_ZEN \
INSERT_GENTCONF_EXCAVATOR \
INSERT_GENTCONF_STEAMROLLER \
INSERT_GENTCONF_PILEDRIVER \
INSERT_GENTCONF_BULLDOZER \
\
INSERT_GENTCONF_ARMSVE \
INSERT_GENTCONF_A64FX \
\
INSERT_GENTCONF_ALTRAMAX \
INSERT_GENTCONF_ALTRA \
INSERT_GENTCONF_FIRESTORM \
\
INSERT_GENTCONF_THUNDERX2 \
INSERT_GENTCONF_CORTEXA57 \
INSERT_GENTCONF_CORTEXA53 \
\
INSERT_GENTCONF_CORTEXA15 \
INSERT_GENTCONF_CORTEXA9 \
\
INSERT_GENTCONF_POWER10 \
INSERT_GENTCONF_POWER9 \
INSERT_GENTCONF_POWER7 \
INSERT_GENTCONF_BGQ \
\
INSERT_GENTCONF_RV32I \
INSERT_GENTCONF_RV64I \
INSERT_GENTCONF_RV32IV \
INSERT_GENTCONF_RV64IV \
\
INSERT_GENTCONF_SIFIVE_X280 \
\
INSERT_GENTCONF_GENERIC


#endif
// end bli_gentconf_macro_defs.h
#line 99 "../blis-2.0/frame/include//bli_macro_defs.h"


// begin bli_misc_macro_defs.h
#line 1 "../blis-2.0/frame/include//bli_misc_macro_defs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_MISC_MACRO_DEFS_H
#define BLIS_MISC_MACRO_DEFS_H


// -- Miscellaneous macros --

// min, max, abs
// NOTE: These must remain macros since we don't know the types of a and b.

#define bli_min( a, b )  ( (a) < (b) ? (a) : (b) )
#define bli_max( a, b )  ( (a) > (b) ? (a) : (b) )
#define bli_abs( a )     ( (a) <= 0 ? -(a) : (a) )

// fmin, fmax, fabs
// NOTE: These must remain macros since we don't know the types of a and b.

#define bli_fmin( a, b ) bli_min( a, b )
#define bli_fmax( a, b ) bli_max( a, b )
#define bli_fabs( a )    ( (a) <= 0.0 ? -(a) : (a) )

// fminabs, fmaxabs
// NOTE: These must remain macros since we don't know the types of a and b.

#define bli_fminabs( a, b ) \
\
	bli_fmin( bli_fabs( a ), \
	          bli_fabs( b ) )

#define bli_fmaxabs( a, b ) \
\
	bli_fmax( bli_fabs( a ), \
	          bli_fabs( b ) )

// round

BLIS_INLINE double bli_round( double a )
{
	return round( a );
}

// round_to_mult

BLIS_INLINE guint_t bli_round_to_mult( guint_t val, guint_t mult )
{
	return ( guint_t )
	       ( ( ( ( guint_t )val +
	             ( guint_t )mult / 2
	           ) / mult
	         ) * mult
	       );
}

// is_odd, is_even

BLIS_INLINE bool bli_is_odd( gint_t a )
{
	return ( bool )( a % 2 == 1 );
}

BLIS_INLINE bool bli_is_even( gint_t a )
{
	return ( bool )( a % 2 == 0 );
}

// swap_dims

BLIS_INLINE void bli_swap_dims( dim_t* dim1, dim_t* dim2 )
{
	dim_t temp = *dim1;
	*dim1 = *dim2;
	*dim2 = temp;
}

// swap_incs

BLIS_INLINE void bli_swap_incs( inc_t* inc1, inc_t* inc2 )
{
	inc_t temp = *inc1;
	*inc1 = *inc2;
	*inc2 = temp;
}

// toggle_bool

BLIS_INLINE void bli_toggle_bool( bool* b )
{
	if ( *b == TRUE ) *b = FALSE;
	else              *b = TRUE;
}

// return datatype for datatype char

#define bli_stype ( BLIS_FLOAT    )
#define bli_dtype ( BLIS_DOUBLE   )
#define bli_ctype ( BLIS_SCOMPLEX )
#define bli_ztype ( BLIS_DCOMPLEX )

// return C type for datatype char

#define bli_sctype  float
#define bli_dctype  double
#define bli_cctype  scomplex
#define bli_zctype  dcomplex

// return C type for domain and precision chars

#define bli_rsctype float
#define bli_rdctype double
#define bli_csctype scomplex
#define bli_cdctype dcomplex

// return real proj of C type for datatype char

#define bli_sctyper  float
#define bli_dctyper  double
#define bli_cctyper  float
#define bli_zctyper  double

// return precision component of dt char

#define bli_sprec  s
#define bli_dprec  d
#define bli_cprec  s
#define bli_zprec  d
#define bli_iprec  i

// return domain component of dt char

#define bli_sdom   r
#define bli_ddom   r
#define bli_cdom   c
#define bli_zdom   c
#define bli_idom   r

// return whether or not two types are the same

#define bli_sssame 1
#define bli_sdsame 0
#define bli_scsame 0
#define bli_szsame 0

#define bli_dssame 0
#define bli_ddsame 1
#define bli_dcsame 0
#define bli_dzsame 0

#define bli_cssame 0
#define bli_cdsame 0
#define bli_ccsame 1
#define bli_czsame 0

#define bli_zssame 0
#define bli_zdsame 0
#define bli_zcsame 0
#define bli_zzsame 1


// return default format specifier for char

// NOTE: These must remain macros due to the way they are used to initialize
// local char arrays.

#define bli_sformatspec() "%9.2e"
#define bli_dformatspec() "%9.2e"
#define bli_cformatspec() "%9.2e + %9.2e "
#define bli_zformatspec() "%9.2e + %9.2e "
#define bli_iformatspec() "%6d"


// Static assertion compatible with any version of C/C++
#define bli_static_assert(cond) while(0){struct s {int STATIC_ASSERT_FAILED : !!(cond);};}

#endif
// end bli_misc_macro_defs.h
#line 101 "../blis-2.0/frame/include//bli_macro_defs.h"

// begin bli_cast_macro_defs.h
#line 1 "../blis-2.0/frame/include//bli_cast_macro_defs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2024, Southern Methodist University

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_CAST_MACRO_DEFS_H
#define BLIS_CAST_MACRO_DEFS_H

// -- Typecast { bfloat16 | float | double } to bfloat16 -----------------------

#ifdef BFLOAT
BLIS_INLINE bfloat bli_bbcast( bfloat b )
{
	return b;
}
#endif

#ifdef BFLOAT
BLIS_INLINE bfloat bli_sbcast( float s )
{
	bfloat b;

	// View the float as a char array.
	char* s_ch = ( char* )&s;

	// Copy upper two bytes of float to a local bfloat16.
	memcpy( &b, &s_ch[2], 2 );

	return b;
}
#endif

#ifdef BFLOAT
BLIS_INLINE bfloat bli_dbcast( double d )
{
	bfloat b;

	// Typecast double input argument to a local float.
	float s = ( float )d;

	// View the float as a char array.
	char* s_ch = ( char* )&s;

	// Copy upper two bytes of float to a local bfloat16.
	memcpy( &b, &s_ch[2], 2 );

	return b;
}
#endif

// -- Typecast { bfloat16 | float | double | int } to float --------------------------

#ifdef BFLOAT
BLIS_INLINE float bli_bscast( bfloat b )
{
	// Initialize all bits in a local float to zero.
	float s = 0.0F;

	// View the float as a char array.
	char* s_ch = ( char* )&s;

	// Copy bfloat16 to the upper two bytes of a local float.
	memcpy( &s_ch[2], &b, 2 );

	return s;
}
#endif

BLIS_INLINE float bli_sscast( float s )
{
	return s;
}

BLIS_INLINE float bli_dscast( double d )
{
	return ( float )d;
}

BLIS_INLINE float bli_iscast( dim_t i )
{
	return ( float )i;
}

// -- Typecast { bfloat16 | float | double | int } to double -------------------------

#ifdef BFLOAT
BLIS_INLINE double bli_bdcast( bfloat b )
{
	// Initialize all bits in a local float to zero.
	float s = 0.0F;

	// View the float as a char array.
	char* s_ch = ( char* )&s;

	// Copy bfloat16 to the upper two bytes of a local float.
	memcpy( &s_ch[2], &b, 2 );

	return ( double )s;
}
#endif

BLIS_INLINE double bli_sdcast( float s )
{
	return ( double )s;
}

//#if 1
BLIS_INLINE double bli_ddcast( double d )
{
	return d;
}
//#else
//#define bli_ddcast( d )  ( d )
//#endif

BLIS_INLINE double bli_idcast( dim_t i )
{
	return ( double )i;
}

// -- Typecast { float | double | int } to int -------------------------

BLIS_INLINE dim_t bli_sicast( float s )
{
	return ( dim_t )s;
}

BLIS_INLINE dim_t bli_dicast( double d )
{
	return ( dim_t )d;
}

BLIS_INLINE dim_t bli_iicast( dim_t i )
{
	return i;
}

#if 0
// -- Fused real/imag accessor + typecast --------------------------------------

// Generate static functions that fuse two operations:
// - accessing the real and imaginary components of all datatypes (real
//   and complex)
// - typecasting a real (or imaginary) component to any real datatype
// Examples:
// static float  bli_dreals( double   a ) { return bli_dscast( bli_dreal( a ) ); }
// static double bli_sreald( float    a ) { return bli_sdcast( bli_sreal( a ) ); }
// static float  bli_creals( scomplex a ) { return bli_sscast( bli_creal( a ) ); }
// static double bli_cimagd( scomplex a ) { return bli_sdcast( bli_cimag( a ) ); }

#undef  GENTFUNC
#define GENTFUNC( chi, cho ) \
\
BLIS_INLINE PASTEMAC(cho,ctype) PASTEMAC2(chi,real,cho)( PASTEMAC(chi,ctype) a ) \
{ \
	return PASTEMAC2(chi,cho,cast)( PASTEMAC(chi,real)( a ) ); \
} \
BLIS_INLINE PASTEMAC(cho,ctype) PASTEMAC2(chi,imag,cho)( PASTEMAC(chi,ctype) a ) \
{ \
	return PASTEMAC2(chi,cho,cast)( PASTEMAC(chi,imag)( a ) ); \
}

// NOTE: We only have to generate functions that output to types [bsd] because
// these macros only need to output real types. The composition that allows
// complex types will be handled by the consumers to these bli_?[real|imag]?()
// functions.

// [bsdkcz][bsd]

GENTFUNC( b, b )
GENTFUNC( s, b )
GENTFUNC( d, b )
GENTFUNC( k, b )
GENTFUNC( c, b )
GENTFUNC( z, b )

GENTFUNC( b, s )
GENTFUNC( s, s )
GENTFUNC( d, s )
GENTFUNC( k, s )
GENTFUNC( c, s )
GENTFUNC( z, s )

GENTFUNC( b, d )
GENTFUNC( s, d )
GENTFUNC( d, d )
GENTFUNC( k, d )
GENTFUNC( c, d )
GENTFUNC( z, d )
#endif

// bli_xytcast() macros are only used in the definitions of level0 scalar
// macros. There, we use a different name from the actual cast functions--
// which are named using the format bli_xycast()--so that we can optionally
// replace them as part of the optimization below without distrubing any
// other uses of bli_xycast() that should not be changed.

#define bli_bbtcast  bli_bbcast
#define bli_sbtcast  bli_sbcast
#define bli_dbtcast  bli_dbcast
#define bli_kbtcast  bli_kbcast
#define bli_cbtcast  bli_cbcast
#define bli_zbtcast  bli_zbcast

#define bli_bstcast  bli_bscast
#define bli_sstcast  bli_sscast
#define bli_dstcast  bli_dscast
#define bli_kstcast  bli_kscast
#define bli_cstcast  bli_cscast
#define bli_zstcast  bli_zscast
#define bli_istcast  bli_iscast

#define bli_bdtcast  bli_bdcast
#define bli_sdtcast  bli_sdcast
#define bli_ddtcast  bli_ddcast
#define bli_kdtcast  bli_kdcast
#define bli_cdtcast  bli_cdcast
#define bli_zdtcast  bli_zdcast
#define bli_idtcast  bli_idcast

#define bli_sitcast  bli_sicast
#define bli_ditcast  bli_dicast
#define bli_iitcast  bli_iicast

// An optimization. In situations where computations would normally occur
// in bfloat, redundant typecasting may occur. For example, in the case of
// performing ssbbaxpy (a and x stored in type s; y stored in type b;
// compute in b), a and x would normally be typecast to b so that all
// operands are in the computation precision (namely, bfloat), but since
// our reference implementation implements bfloat flops in terms of float
// flops, all operands would need to be typecast back to s anyway just so
// the computation can take place. This means that a and x were truncated
// down to bfloat (and thus lost precision) somewhat unnecessarily. Instead,
// what could happen is that a and x remain in s, y is typecast to s,
// computation would take place in s, and then the result is truncated to
// bfloat on output to y. These macros substitute certain static function
// calls to be the equivalent calls that would cast to float instead of
// bfloat.
#ifdef BLIS_OPTIMIZE_BFLOAT_AS_FLOAT

#undef  bli_bbcast
#define bli_bbcast  bli_bscast
#undef  bli_sbcast
#define bli_sbcast  bli_sscast
#undef  bli_dbcast
#define bli_dbcast  bli_dscast
#undef  bli_kbcast
#define bli_kbcast  bli_kscast
#undef  bli_cbcast
#define bli_cbcast  bli_cscast
#undef  bli_zbcast
#define bli_zbcast  bli_zscast

#endif


// -- Basic constants (per precision) ------------------------------------------

#ifdef BLIS_OPTIMIZE_BFLOAT_AS_FLOAT

#define bli_btwo                bli_stwo
#define bli_bone                bli_sone
#define bli_bzero               bli_szero
#define bli_bmone               bli_smone
#define bli_bmtwo               bli_smtwo

#else

#define bli_btwo    bli_sbcast( bli_stwo )
#define bli_bone    bli_sbcast( bli_sone )
#define bli_bzero   bli_sbcast( bli_szero )
#define bli_bmone   bli_sbcast( bli_smone )
#define bli_bmtwo   bli_sbcast( bli_smtwo )

#endif

#define bli_stwo    2.0F
#define bli_sone    1.0F
#define bli_szero   0.0F
#define bli_smone  -1.0F
#define bli_smtwo  -2.0F

#define bli_dtwo    2.0
#define bli_done    1.0
#define bli_dzero   0.0
#define bli_dmone  -1.0
#define bli_dmtwo  -2.0

// -- Basic arithmetic operations (per precision) ------------------------------

#ifdef BLIS_OPTIMIZE_BFLOAT_AS_FLOAT

#define bli_bmul( a, b )                  bli_smul(             a,             b  )
#define bli_bdiv( a, b )                  bli_sdiv(             a,             b  )
#define bli_badd( a, b )                  bli_sadd(             a,             b  )
#define bli_bsub( a, b )                  bli_ssub(             a,             b  )
#define bli_bneg( a )                     bli_sneg(             a                 )
#define bli_bsqrt( a )                    bli_ssqrt(            a                 )
#define bli_bhypot( a, b )                bli_shypot(            a,             b  )

#else

#define bli_bmul( a, b )      bli_sbcast( bli_smul(  bli_bscast(a), bli_bscast(b) ) )
#define bli_bdiv( a, b )      bli_sbcast( bli_sdiv(  bli_bscast(a), bli_bscast(b) ) )
#define bli_badd( a, b )      bli_sbcast( bli_sadd(  bli_bscast(a), bli_bscast(b) ) )
#define bli_bsub( a, b )      bli_sbcast( bli_ssub(  bli_bscast(a), bli_bscast(b) ) )
#define bli_bneg( a )         bli_sbcast( bli_sneg(  bli_bscast(a)                ) )
#define bli_bsqrt( a )        bli_sbcast( bli_ssqrt( bli_bscast(a)                ) )
#define bli_bhypot( a, b )    bli_sbcast( bli_shypot( bli_bscast(a), bli_bscast(b) ) )

#endif

#define bli_smul( a, b )       (a) * (b)
#define bli_sdiv( a, b )       (a) / (b)
#define bli_sadd( a, b )       (a) + (b)
#define bli_ssub( a, b )       (a) - (b)
#define bli_sneg( a )          -(a)
#define bli_ssqrt( a )         sqrtf(a)
#define bli_shypot( a, b )     hypotf(a,b)

#define bli_dmul( a, b )       (a) * (b)
#define bli_ddiv( a, b )       (a) / (b)
#define bli_dadd( a, b )       (a) + (b)
#define bli_dsub( a, b )       (a) - (b)
#define bli_dneg( a )          -(a)
#define bli_dsqrt( a )         sqrt(a)
#define bli_dhypot( a, b )     hypot(a,b)

// -- Basic compare operations (per precision) ---------------------------------

#ifdef BLIS_OPTIMIZE_BFLOAT_AS_FLOAT

#define bli_beq( a, b )                  bli_seq(            a,             b  )
#define bli_blt( a, b )                  bli_slt(            a,             b  )
#define bli_ble( a, b )                  bli_sle(            a,             b  )
#define bli_bgt( a, b )                  bli_sgt(            a,             b  )
#define bli_bge( a, b )                  bli_sge(            a,             b  )

#else

#define bli_beq( a, b )      bli_sbcast( bli_seq( bli_bscast(a), bli_bscast(b) ) )
#define bli_blt( a, b )      bli_sbcast( bli_slt( bli_bscast(a), bli_bscast(b) ) )
#define bli_ble( a, b )      bli_sbcast( bli_sle( bli_bscast(a), bli_bscast(b) ) )
#define bli_bgt( a, b )      bli_sbcast( bli_sgt( bli_bscast(a), bli_bscast(b) ) )
#define bli_bge( a, b )      bli_sbcast( bli_sge( bli_bscast(a), bli_bscast(b) ) )

#endif

#define bli_seq( a, b )  ( a == b )
#define bli_slt( a, b )  ( a <  b )
#define bli_sle( a, b )  ( a <= b )
#define bli_sgt( a, b )  ( a >  b )
#define bli_sge( a, b )  ( a >= b )

#define bli_deq( a, b )  ( a == b )
#define bli_dlt( a, b )  ( a <  b )
#define bli_dle( a, b )  ( a <= b )
#define bli_dgt( a, b )  ( a >  b )
#define bli_dge( a, b )  ( a >= b )

#define bli_ieq( a, b )  ( a == b )
#define bli_ilt( a, b )  ( a <  b )
#define bli_ile( a, b )  ( a <= b )
#define bli_igt( a, b )  ( a >  b )
#define bli_ige( a, b )  ( a >= b )

// -- Min/max/abs/etc. operations (per precision) ------------------------------

#ifdef BLIS_OPTIMIZE_BFLOAT_AS_FLOAT

#define bli_bmin( a, b )                   bli_smin(               a,             b  )
#define bli_bmax( a, b )                   bli_smax(               a,             b  )
#define bli_babs( a )                      bli_sabs(               a                 )
#define bli_bminabs( a, b )                bli_sminabs(            a              b  )
#define bli_bmaxabs( a, b )                bli_smaxabs(            a              b  )
#define bli_bcopysign( a, b )            ( bli_slt(            b , bli_szero ) \
                                           ? bli_sneg( bli_sabs(            a  ) ) \
                                           :           bli_sabs(            a  )   )

#else

#define bli_bmin( a, b )       bli_sbcast(    bli_smin( bli_bscast(a), bli_bscast(b) ) )
#define bli_bmax( a, b )       bli_sbcast(    bli_smax( bli_bscast(a), bli_bscast(b) ) )
#define bli_babs( a )          bli_sbcast(    bli_sabs( bli_bscast(a)                ) )
#define bli_bminabs( a, b )    bli_sbcast( bli_sminabs( bli_bscast(a), bli_bscast(b) ) )
#define bli_bmaxabs( a, b )    bli_sbcast( bli_smaxabs( bli_bscast(a), bli_bscast(b) ) )
#define bli_bcopysign( a, b )  bli_sbcast( bli_slt( bli_bscast(b), bli_szero ) \
                                           ? bli_sneg( bli_sabs( bli_bscast(a) ) ) \
                                           :           bli_sabs( bli_bscast(a) )   )

#endif

#define bli_smin( a, b )       ( bli_slt( a, b ) ? a : b )
#define bli_smax( a, b )       ( bli_sgt( a, b ) ? a : b )
//#define bli_sabs( a )          ( bli_slt( a, PASTEMAC(s,zero) ) ? -(a) : a )
#define bli_sabs( a )          ( fabsf(a) )
#define bli_sminabs( a, b )    bli_smin( bli_sabs( a ), bli_sabs( b ) )
#define bli_smaxabs( a, b )    bli_smax( bli_sabs( a ), bli_sabs( b ) )
#define bli_scopysign( a, b )  ( copysignf( a, b ) ) \

#define bli_dmin( a, b )       ( bli_dlt( a, b ) ? a : b )
#define bli_dmax( a, b )       ( bli_dgt( a, b ) ? a : b )
//#define bli_dabs( a )          ( bli_dlt( a, PASTEMAC(d,zero) ) ? -(a) : a )
#define bli_dabs( a )          ( fabs(a) )
#define bli_dminabs( a, b )    bli_dmin( bli_dabs( a ), bli_dabs( b ) )
#define bli_dmaxabs( a, b )    bli_dmax( bli_dabs( a ), bli_dabs( b ) )
#define bli_dcopysign( a, b )  ( copysign( a, b ) ) \

// -- Infinity/NaN check (per precision) ---------------------------------------

#ifdef BLIS_OPTIMIZE_BFLOAT_AS_FLOAT

#define bli_bisinf( a )        bli_sisinf(            a  )
#define bli_bisnan( a )        bli_sisnan(            a  )

#else

#define bli_bisinf( a )        bli_sisinf( bli_bscast(a) )
#define bli_bisnan( a )        bli_sisnan( bli_bscast(a) )

#endif

#define bli_sisinf( a )        isinf( a )
#define bli_sisnan( a )        isnan( a )

#define bli_disinf( a )        isinf( a )
#define bli_disnan( a )        isnan( a )

// -- Randomization operations (per precision) ---------------------------------

#define bli_brand              bli_dbcast( bli_rand() )
#define bli_srand              bli_dscast( bli_rand() )
#define bli_drand              bli_ddcast( bli_rand() )

// Randomize a real number on the interval [-1.0,1.0] and return it as a double.
BLIS_INLINE double bli_rand( void )
{
	return ( ( ( double ) rand()         ) /
             ( ( double ) RAND_MAX / 2.0 )
           ) - 1.0;
}

#define bli_brandnp2           bli_dbcast( bli_randnp2s() )
#define bli_srandnp2           bli_dscast( bli_randnp2s() )
#define bli_drandnp2           bli_ddcast( bli_randnp2s() )

// Randomize a power of two on a narrow range and return it as a double.
BLIS_INLINE double bli_randnp2s( void )
{
	const double m_max  = 6.0;
	const double m_max2 = m_max + 2.0;
	double       t;
	double       r_val;

	// Compute a narrow-range power of two.
	//
	// For the purposes of commentary, we'll assume that m_max = 4. This
	// represents the largest power of two we will use to generate the
	// random numbers.

	do
	{
		// Generate a random real number t on the interval: [0.0, 6.0].
		t = ( ( double ) rand() / ( double ) RAND_MAX ) * m_max2;

		// Transform the interval into the set of integers, {0,1,2,3,4,5}.
		// Note that 6 is prohibited by the loop guard below.
		t = floor( t );
	}
	// If t is ever equal to m_max2, we re-randomize. The guard against
	// m_max2 < t is for sanity and shouldn't happen, unless perhaps there
	// is weirdness in the typecasting to double when computing t above.
	while ( m_max2 <= t );

	// Map values of t == 0 to a final value of 0.
	if ( t == 0.0 ) r_val = 0.0;
	else
	{
		// This case handles values of t = {1,2,3,4,5}.

		// Compute r_val = 2^s where s = -(t-1) = {-4,-3,-2,-1,0}.
		r_val = pow( 2.0, -(t - 1.0) );

		// Compute a random number to determine the sign of the final
		// result.
		const double s_val = PASTEMAC(d,rand);

		// If our sign value is negative, our random power of two will
		// be negative.
		if ( s_val < 0.0 ) r_val = -r_val;
	}

	// r_val = 0, or +/-{2^0, 2^-1, 2^-2, 2^-3, 2^-4}.
	return r_val;
}



#endif

// end bli_cast_macro_defs.h
#line 102 "../blis-2.0/frame/include//bli_macro_defs.h"

// begin bli_edge_case_macro_defs.h
#line 1 "../blis-2.0/frame/include//bli_edge_case_macro_defs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2021, Southern Methodist University

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_EDGE_CASE_MACRO_DEFS_H
#define BLIS_EDGE_CASE_MACRO_DEFS_H

//
// Macros for edge-case handling within gemm microkernels.
//

// -- Setup helper macros --

#define GEMM_UKR_SETUP_CT_PRE(ch,mr,nr,row_major,alignment) \
\
	const PASTEMAC(ch,ctype)* restrict _beta   = beta; \
	      PASTEMAC(ch,ctype)* restrict _c      = c; \
	const inc_t                        _rs_c   = rs_c; \
	const inc_t                        _cs_c   = cs_c; \
	      PASTEMAC(ch,ctype)           _ct[ BLIS_STACK_BUF_MAX_SIZE \
	                                        / sizeof( PASTEMAC(ch,ctype) ) \
	                                      ] __attribute__((aligned(alignment))); \
	const inc_t                        _rs_ct  = row_major ? nr :  1; \
	const inc_t                        _cs_ct  = row_major ?  1 : mr;

#define GEMM_UKR_SETUP_CT_POST(ch) \
\
	PASTEMAC(ch,ctype) _zero; \
	bli_tset0s( ch, _zero ); \
	\
	if ( _use_ct ) \
	{ \
		c = _ct; \
		rs_c = _rs_ct; \
		cs_c = _cs_ct; \
		beta = &_zero; \
	}

// -- Setup macros --

#define GEMM_UKR_SETUP_CT(ch,mr,nr,row_major) \
\
	/* Scenario 1: the ukernel contains assembly-level support only for its
	   IO preference (e.g. only row-oriented or only column-oriented IO).
	   Use a temporary microtile for the other two cases as well as edge
	   cases. */ \
	GEMM_UKR_SETUP_CT_PRE(ch,mr,nr,row_major,1); \
	const bool _use_ct = ( row_major ? cs_c != 1 : rs_c != 1 ) || \
	                     m != mr || n != nr; \
	GEMM_UKR_SETUP_CT_POST(ch);

#define GEMM_UKR_SETUP_CT_AMBI(ch,mr,nr,row_major) \
\
	/* Scenario 2: the ukernel contains assembly-level support for its IO
	   preference as well as its opposite via in-register transpose
	   (e.g. both row- and column-oriented IO). Use a temporary microtile
	   for the general stride case as well as edge cases. */ \
	GEMM_UKR_SETUP_CT_PRE(ch,mr,nr,row_major,1); \
	const bool _use_ct = ( cs_c != 1 && rs_c != 1 ) || \
	                     m != mr || n != nr; \
	GEMM_UKR_SETUP_CT_POST(ch);

#define GEMM_UKR_SETUP_CT_ANY(ch,mr,nr,row_major) \
\
	/* Scenario 3: Similar to (2) where the assembly region also supports
	   general stride I0. Use a temporary microtile only for edge cases. */ \
	GEMM_UKR_SETUP_CT_PRE(ch,mr,nr,row_major,1); \
	const bool _use_ct = ( m != mr || n != nr ); \
	GEMM_UKR_SETUP_CT_POST(ch);

#define GEMM_UKR_SETUP_CT_ALIGNED(ch,mr,nr,row_major,alignment) \
\
	/* Scenario 4: Similar to (1), but uses temporary microtile to handle
	   cases where the pointer to the C microtile is not aligned. */ \
	GEMM_UKR_SETUP_CT_PRE(ch,mr,nr,row_major,alignment); \
	const bool _use_ct = ( row_major ? cs_c != 1 : rs_c != 1 ) || \
	                     m != mr || n != nr || \
	                     ( (uintptr_t)_c % alignment ) || \
	                     ( ( ( row_major ? _rs_c : _cs_c )*sizeof( PASTEMAC(ch,ctype) ) ) % alignment ); \
	GEMM_UKR_SETUP_CT_POST(ch);

// -- Flush macros --

#define GEMM_UKR_FLUSH_CT(ch) \
\
	/* If we actually used the temporary microtile, accumulate it to the output
	   microtile. */ \
	if ( _use_ct ) \
	{ \
		bli_txpbys_mxn \
		( \
		  ch,ch,ch,ch, \
		  m, n, \
		  _ct, _rs_ct, _cs_ct, \
		  _beta, \
		  _c,  _rs_c,  _cs_c \
		); \
	} \


//
// Macros for edge-case handling within gemmtrsm microkernels.
//

// -- Setup helper macros --

#define GEMMTRSM_UKR_SETUP_CT_PRE(ch,mr,nr,row_major,alignment) \
\
	      PASTEMAC(ch,ctype)* restrict _c      = c11; \
	const inc_t                        _rs_c   = rs_c; \
	const inc_t                        _cs_c   = cs_c; \
	      PASTEMAC(ch,ctype)           _ct[ BLIS_STACK_BUF_MAX_SIZE \
	                                        / sizeof( PASTEMAC(ch,ctype) ) \
	                                      ] __attribute__((aligned(alignment))); \
	const inc_t                        _rs_ct  = row_major ? nr :  1; \
	const inc_t                        _cs_ct  = row_major ?  1 : mr;

#define GEMMTRSM_UKR_SETUP_CT_POST(ch) \
\
	if ( _use_ct ) \
	{ \
		c11 = _ct; \
		rs_c = _rs_ct; \
		cs_c = _cs_ct; \
	}

// -- Setup macros --

#define GEMMTRSM_UKR_SETUP_CT(ch,mr,nr,row_major) \
\
	/* Scenario 1: the ukernel contains assembly-level support only for its
	   IO preference (e.g. only row-oriented or only column-oriented IO).
	   Use a temporary microtile for the other two cases as well as edge
	   cases. */ \
	GEMMTRSM_UKR_SETUP_CT_PRE(ch,mr,nr,row_major,1); \
	const bool _use_ct = ( row_major ? cs_c != 1 : rs_c != 1 ) || \
	                     m != mr || n != nr; \
	GEMMTRSM_UKR_SETUP_CT_POST(ch);

#define GEMMTRSM_UKR_SETUP_CT_AMBI(ch,mr,nr,row_major) \
\
	/* Scenario 2: the ukernel contains assembly-level support for its IO
	   preference as well as its opposite via in-register transpose
	   (e.g. both row- and column-oriented IO). Use a temporary microtile
	   for the general stride case as well as edge cases. */ \
	GEMMTRSM_UKR_SETUP_CT_PRE(ch,mr,nr,row_major,1); \
	const bool _use_ct = ( cs_c != 1 && rs_c != 1 ) || \
	                     m != mr || n != nr; \
	GEMMTRSM_UKR_SETUP_CT_POST(ch);

#define GEMMTRSM_UKR_SETUP_CT_ANY(ch,mr,nr,row_major) \
\
	/* Scenario 3: Similar to (2) where the assembly region also supports
	   general stride I0. Use a temporary microtile only for edge cases. */ \
	GEMMTRSM_UKR_SETUP_CT_PRE(ch,mr,nr,row_major,1); \
	const bool _use_ct = ( m != mr || n != nr ); \
	GEMMTRSM_UKR_SETUP_CT_POST(ch);

#define GEMMTRSM_UKR_SETUP_CT_ALIGNED(ch,mr,nr,row_major,alignment) \
\
	/* Scenario 4: Similar to (1), but uses temporary microtile to handle
	   cases where the pointer to the C microtile is not aligned. */ \
	GEMMTRSM_UKR_SETUP_CT_PRE(ch,mr,nr,row_major,alignment); \
	const bool _use_ct = ( row_major ? cs_c != 1 : rs_c != 1 ) || \
	                     m != mr || n != nr || \
	                     ( (uintptr_t)_c % alignment ) || \
	                     ( ( ( row_major ? _rs_c : _cs_c )*sizeof( PASTEMAC(ch,ctype) ) ) % alignment ); \
	GEMMTRSM_UKR_SETUP_CT_POST(ch);

// -- Flush macros --

#define GEMMTRSM_UKR_FLUSH_CT(ch) \
\
	/* If we actually used the temporary microtile, use it to overwrite the
	   output microtile. Used by trsm. */ \
	if ( _use_ct ) \
	{ \
		PASTEMAC(t,copys_mxn) \
		( \
		  ch,ch, \
		  m, n, \
		  _ct, _rs_ct, _cs_ct, \
		  _c,  _rs_c,  _cs_c \
		); \
	} \


#endif

// end bli_edge_case_macro_defs.h
#line 103 "../blis-2.0/frame/include//bli_macro_defs.h"

// begin bli_param_macro_defs.h
#line 1 "../blis-2.0/frame/include//bli_param_macro_defs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2018 - 2019, Advanced Micro Devices, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_PARAM_MACRO_DEFS_H
#define BLIS_PARAM_MACRO_DEFS_H


// -- Parameter query macros --

// buffer

BLIS_INLINE bool bli_is_aligned_to( siz_t p, siz_t size )
{
	return ( bool )
	       ( p % size == 0 );
}

BLIS_INLINE bool bli_is_unaligned_to( siz_t p, siz_t size )
{
	return ( bool )
	       ( p % size != 0 );
}

BLIS_INLINE siz_t bli_offset_past_alignment( siz_t p, siz_t size )
{
	return ( siz_t )
	       ( p % size );
}


// datatype

BLIS_INLINE bool bli_is_float( num_t dt )
{
	return ( bool )
	       ( dt == BLIS_FLOAT );
}

BLIS_INLINE bool bli_is_double( num_t dt )
{
	return ( bool )
	       ( dt == BLIS_DOUBLE );
}

BLIS_INLINE bool bli_is_scomplex( num_t dt )
{
	return ( bool )
	       ( dt == BLIS_SCOMPLEX );
}

BLIS_INLINE bool bli_is_dcomplex( num_t dt )
{
	return ( bool )
	       ( dt == BLIS_DCOMPLEX );
}

BLIS_INLINE bool bli_is_constant( num_t dt )
{
	return ( bool )
	       ( dt == BLIS_CONSTANT );
}

BLIS_INLINE bool bli_is_int( num_t dt )
{
	return ( bool )
	       ( dt == BLIS_INT );
}

BLIS_INLINE bool bli_is_real( num_t dt )
{
	return ( bool )
	       ( bli_is_float( dt ) ||
	                   bli_is_double( dt ) );
}

BLIS_INLINE bool bli_is_complex( num_t dt )
{
	return ( bool )
	       ( bli_is_scomplex( dt ) ||
	                   bli_is_dcomplex( dt ) );
}

BLIS_INLINE bool bli_is_single_prec( num_t dt )
{
	return ( bool )
	       ( bli_is_float( dt ) ||
	                   bli_is_scomplex( dt ) );
}

BLIS_INLINE bool bli_is_double_prec( num_t dt )
{
	return ( bool )
	       ( bli_is_double( dt ) ||
	                   bli_is_dcomplex( dt ) );
}

BLIS_INLINE dom_t bli_dt_domain( num_t dt )
{
	return ( dom_t )
	       ( dt & BLIS_DOMAIN_BIT );
}

BLIS_INLINE bool bli_dt_dom_is_real( num_t dt )
{
	return ( bool )
	       ( ( dt & BLIS_DOMAIN_BIT ) == BLIS_REAL );
}

BLIS_INLINE bool bli_dt_dom_is_complex( num_t dt )
{
	return ( bool )
	       ( ( dt & BLIS_DOMAIN_BIT ) == BLIS_COMPLEX );
}

BLIS_INLINE prec_t bli_dt_prec( num_t dt )
{
	return ( prec_t )
	       ( dt & BLIS_PRECISION_BIT );
}

BLIS_INLINE bool bli_dt_prec_is_single( num_t dt )
{
	return ( bool )
	       ( ( dt & BLIS_PRECISION_BIT ) == BLIS_SINGLE_PREC );
}

BLIS_INLINE bool bli_dt_prec_is_double( num_t dt )
{
	return ( bool )
	       ( ( dt & BLIS_PRECISION_BIT ) == BLIS_DOUBLE_PREC );
}

BLIS_INLINE num_t bli_dt_proj_to_real( num_t dt )
{
	return ( num_t )
	       ( dt & ~BLIS_BITVAL_COMPLEX );
}

BLIS_INLINE num_t bli_dt_proj_to_complex( num_t dt )
{
	return ( num_t )
	       ( dt | BLIS_BITVAL_COMPLEX );
}

BLIS_INLINE num_t bli_dt_proj_to_single_prec( num_t dt )
{
	return ( num_t )
	       ( dt & ~BLIS_BITVAL_DOUBLE_PREC );
}

BLIS_INLINE num_t bli_dt_proj_to_double_prec( num_t dt )
{
	return ( num_t )
	       ( dt | BLIS_BITVAL_DOUBLE_PREC );
}


// trans

BLIS_INLINE bool bli_is_notrans( trans_t trans )
{
	return ( bool )
	       ( trans == BLIS_NO_TRANSPOSE );
}

BLIS_INLINE bool bli_is_trans( trans_t trans )
{
	return ( bool )
	       ( trans == BLIS_TRANSPOSE );
}

BLIS_INLINE bool bli_is_conjnotrans( trans_t trans )
{
	return ( bool )
	       ( trans == BLIS_CONJ_NO_TRANSPOSE );
}

BLIS_INLINE bool bli_is_conjtrans( trans_t trans )
{
	return ( bool )
	       ( trans == BLIS_CONJ_TRANSPOSE );
}

BLIS_INLINE bool bli_does_notrans( trans_t trans )
{
	return ( bool )
	       ( (~trans & BLIS_TRANS_BIT ) == BLIS_BITVAL_TRANS );
}

BLIS_INLINE bool bli_does_trans( trans_t trans )
{
	return ( bool )
	       ( ( trans & BLIS_TRANS_BIT ) == BLIS_BITVAL_TRANS );
}

BLIS_INLINE bool bli_does_noconj( trans_t trans )
{
	return ( bool )
	       ( (~trans & BLIS_CONJ_BIT ) == BLIS_BITVAL_CONJ );
}

BLIS_INLINE bool bli_does_conj( trans_t trans )
{
	return ( bool )
	       ( ( trans & BLIS_CONJ_BIT ) == BLIS_BITVAL_CONJ );
}

BLIS_INLINE trans_t bli_extract_trans( trans_t trans )
{
	return ( trans_t )
	       ( trans & BLIS_TRANS_BIT );
}

BLIS_INLINE conj_t bli_extract_conj( trans_t trans )
{
	return ( conj_t )
	       ( trans & BLIS_CONJ_BIT );
}

BLIS_INLINE trans_t bli_trans_toggled( trans_t trans )
{
	return ( trans_t )
	       ( trans ^ BLIS_TRANS_BIT );
}

BLIS_INLINE trans_t bli_trans_toggled_conj( trans_t trans )
{
	return ( trans_t )
	       ( trans ^ BLIS_CONJ_BIT );
}

BLIS_INLINE trans_t bli_apply_trans( trans_t transapp, trans_t trans )
{
	return ( trans_t )
	       ( trans ^ transapp );
}

BLIS_INLINE void bli_toggle_trans( trans_t* trans )
{
	*trans = bli_trans_toggled( *trans );
}


// side

BLIS_INLINE bool bli_is_left( side_t side )
{
	return ( bool )
	       ( side == BLIS_LEFT );
}

BLIS_INLINE bool bli_is_right( side_t side )
{
	return ( bool )
	       ( side == BLIS_RIGHT );
}

BLIS_INLINE side_t bli_side_toggled( side_t side )
{
	return ( bli_is_left( side ) ? BLIS_RIGHT : BLIS_LEFT );
}

BLIS_INLINE void bli_toggle_side( side_t* side )
{
	*side = bli_side_toggled( *side );
}


// uplo

BLIS_INLINE bool bli_is_lower( uplo_t uplo )
{
	return ( bool )
	       ( uplo == BLIS_LOWER );
}

BLIS_INLINE bool bli_is_upper( uplo_t uplo )
{
	return ( bool )
	       ( uplo == BLIS_UPPER );
}

BLIS_INLINE bool bli_is_upper_or_lower( uplo_t uplo )
{
	return ( bool )
	       ( bli_is_upper( uplo ) ||
	         bli_is_lower( uplo ) );
}

BLIS_INLINE bool bli_is_dense( uplo_t uplo )
{
	return ( bool )
	       ( uplo == BLIS_DENSE );
}

BLIS_INLINE bool bli_is_zeros( uplo_t uplo )
{
	return ( bool )
	       ( uplo == BLIS_ZEROS );
}

BLIS_INLINE uplo_t bli_uplo_toggled( uplo_t uplo )
{
	return ( uplo_t )
	       ( bli_is_upper_or_lower( uplo )
	         ? ( ( uplo ^ BLIS_LOWER_BIT ) ^ BLIS_UPPER_BIT )
	         :     uplo
	       );
}

BLIS_INLINE void bli_toggle_uplo( uplo_t* uplo )
{
	*uplo = bli_uplo_toggled( *uplo );
}


// structure

BLIS_INLINE bool bli_is_general( struc_t struc )
{
	return ( bool )
	       ( struc == BLIS_GENERAL );
}

BLIS_INLINE bool bli_is_hermitian( struc_t struc )
{
	return ( bool )
	       ( struc == BLIS_HERMITIAN );
}

BLIS_INLINE bool bli_is_symmetric( struc_t struc )
{
	return ( bool )
	       ( struc == BLIS_SYMMETRIC );
}

BLIS_INLINE bool bli_is_triangular( struc_t struc )
{
	return ( bool )
	       ( struc == BLIS_TRIANGULAR );
}

BLIS_INLINE bool bli_is_herm_or_symm( struc_t struc )
{
	return ( bool )
	       ( bli_is_hermitian( struc ) ||
	         bli_is_symmetric( struc ) );
}


// conj

BLIS_INLINE bool bli_is_noconj( conj_t conj )
{
	return ( bool )
	       ( conj == BLIS_NO_CONJUGATE );
}

BLIS_INLINE bool bli_is_conj( conj_t conj )
{
	return ( bool )
	       ( conj == BLIS_CONJUGATE );
}

BLIS_INLINE conj_t bli_conj_toggled( conj_t conj )
{
	return ( conj_t )
	       ( conj ^ BLIS_CONJ_BIT );
}

BLIS_INLINE conj_t bli_apply_conj( conj_t conjapp, conj_t conj )
{
	return ( conj_t )
	       ( conj ^ conjapp );
}

BLIS_INLINE void bli_toggle_conj( conj_t* conj )
{
	*conj = bli_conj_toggled( *conj );
}


// diag

BLIS_INLINE bool bli_is_nonunit_diag( diag_t diag )
{
	return ( bool )
	       ( diag == BLIS_NONUNIT_DIAG );
}

BLIS_INLINE bool bli_is_unit_diag( diag_t diag )
{
	return ( bool )
	       ( diag == BLIS_UNIT_DIAG );
}


// err_t-related

BLIS_INLINE bool bli_is_success( err_t err )
{
	return ( bool )
	       ( err == BLIS_SUCCESS );
}

BLIS_INLINE bool bli_is_failure( err_t err )
{
	return ( bool )
	       ( err != BLIS_SUCCESS );
}


// dimension-related

BLIS_INLINE bool bli_zero_dim1( dim_t m )
{
	return ( bool )
	       ( m == 0 );
}

BLIS_INLINE bool bli_zero_dim2( dim_t m, dim_t n )
{
	return ( bool )
	       ( m == 0 || n == 0 );
}

BLIS_INLINE bool bli_zero_dim3( dim_t m, dim_t n, dim_t k )
{
	return ( bool )
	       ( m == 0 || n == 0 || k == 0 );
}

BLIS_INLINE bool bli_nonzero_dim( dim_t m )
{
	return ( bool )
	       ( m > 0 );
}

BLIS_INLINE bool bli_vector_dim( dim_t m, dim_t n )
{
	return ( bool )
	       ( m == 1 ? n : m );
}

BLIS_INLINE bool bli_is_vector( dim_t m, dim_t n )
{
	return ( bool )
	       ( m == 1 || n == 1 );
}

BLIS_INLINE bool bli_is_row_vector( dim_t m, dim_t n )
{
	return ( bool )
	       ( m == 1 );
}

BLIS_INLINE bool bli_is_col_vector( dim_t m, dim_t n )
{
	return ( bool )
	       ( n == 1 );
}

BLIS_INLINE void bli_set_dim_with_side( side_t side, dim_t m, dim_t n, dim_t* dim )
{
	if ( bli_is_left( side ) ) *dim = m;
	else                       *dim = n;
}

BLIS_INLINE void bli_set_dims_with_trans( trans_t trans, dim_t m, dim_t n, dim_t* mt, dim_t* nt )
{
	if ( bli_does_notrans( trans ) ) { *mt = m; *nt = n; }
	else                             { *mt = n; *nt = m; }
}

BLIS_INLINE void bli_set_dims_incs_with_trans( trans_t trans,
                                          dim_t  m,  dim_t  n,  inc_t  rs,  inc_t  cs,
                                          dim_t* mt, dim_t* nt, inc_t* rst, inc_t* cst )
{
	if ( bli_does_notrans( trans ) ) { *mt = m; *nt = n; *rst = rs; *cst = cs; }
	else                             { *mt = n; *nt = m; *rst = cs; *cst = rs; }
}


// blocksize-related

BLIS_INLINE dim_t bli_determine_blocksize_dim_f( dim_t i, dim_t dim, dim_t b_alg )
{
	return ( dim_t )
	       ( bli_min( b_alg, dim - i ) );
}

BLIS_INLINE dim_t bli_determine_blocksize_dim_b( dim_t i, dim_t dim, dim_t b_alg )
{
	return ( dim_t )
	       ( i == 0 && dim % b_alg != 0 ? dim % b_alg
	                                    : b_alg );
}


// stride-related

BLIS_INLINE inc_t bli_vector_inc( trans_t trans, dim_t m, dim_t n, inc_t rs, inc_t cs )
{
	return ( inc_t )
	       ( bli_does_notrans( trans ) ? ( m == 1 ? cs : rs )
	                                   : ( m == 1 ? rs : cs ) );
}

BLIS_INLINE bool bli_is_row_stored( inc_t rs, inc_t cs )
{
	return ( bool )
	       ( bli_abs( cs ) == 1 );
}

BLIS_INLINE bool bli_is_col_stored( inc_t rs, inc_t cs )
{
	return ( bool )
	       ( bli_abs( rs ) == 1 );
}

BLIS_INLINE bool bli_is_row_stored_f( dim_t m, dim_t n, inc_t rs, inc_t cs )
{
	return ( bool )
	       ( cs == 1 && ( rs > 1 || n == 1 ) );
}

BLIS_INLINE bool bli_is_col_stored_f( dim_t m, dim_t n, inc_t rs, inc_t cs )
{
	return ( bool )
	       ( rs == 1 && ( cs > 1 || m == 1 ) );
}

BLIS_INLINE bool bli_is_gen_stored( inc_t rs, inc_t cs )
{
	return ( bool )
	       ( bli_abs( rs ) != 1 &&
	         bli_abs( cs ) != 1 );
}

BLIS_INLINE bool bli_is_preferentially_stored( inc_t rs, inc_t cs, bool row_pref )
{
	return ( bli_is_row_stored( rs, cs ) &&  row_pref ) ||
	       ( bli_is_col_stored( rs, cs ) && !row_pref );
}

BLIS_INLINE bool bli_is_row_tilted( dim_t m, dim_t n, inc_t rs, inc_t cs )
{
	return ( bool )
	       ( bli_abs( cs ) == bli_abs( rs )
	         ? n < m
	         : bli_abs( cs ) < bli_abs( rs ) );
}

BLIS_INLINE bool bli_is_col_tilted( dim_t m, dim_t n, inc_t rs, inc_t cs )
{
	return ( bool )
	       ( bli_abs( rs ) == bli_abs( cs )
	         ? m < n
	         : bli_abs( rs ) < bli_abs( cs ) );
}

BLIS_INLINE bool bli_has_nonunit_inc1( inc_t s1 )
{
	return ( bool )
	       ( s1 != 1 );
}

BLIS_INLINE bool bli_has_nonunit_inc2( inc_t s1, inc_t s2 )
{
	return ( bool )
	       ( s1 != 1 || s2 != 1 );
}

BLIS_INLINE bool bli_has_nonunit_inc3( inc_t s1, inc_t s2, inc_t s3 )
{
	return ( bool )
	       ( s1 != 1 || s2 != 1 || s3 != 1 );
}


// diag offset-related

BLIS_INLINE void bli_negate_diag_offset( doff_t* diagoff )
{
	*diagoff = -(*diagoff);
}

BLIS_INLINE void bli_shift_diag_offset_to_grow_uplo( uplo_t uplo, doff_t* diagoff )
{
	if      ( bli_is_upper( uplo ) ) *diagoff -= 1;
	else if ( bli_is_lower( uplo ) ) *diagoff += 1;
}

BLIS_INLINE void bli_shift_diag_offset_to_shrink_uplo( uplo_t uplo, doff_t* diagoff )
{
	if      ( bli_is_upper( uplo ) ) *diagoff += 1;
	else if ( bli_is_lower( uplo ) ) *diagoff -= 1;
}

BLIS_INLINE doff_t bli_diag_offset_with_trans( trans_t trans, doff_t diagoff )
{
	return ( doff_t )
	       ( bli_does_trans( trans ) ? -diagoff : diagoff );
}

BLIS_INLINE bool bli_is_strictly_above_diag( doff_t diagoff, trans_t trans, dim_t m, dim_t n )
{
	return ( bool )
	       ( bli_does_trans( trans )
	         ? ( ( doff_t )n <= -diagoff )
	         : ( ( doff_t )m <= -diagoff ) );
}

BLIS_INLINE bool bli_is_strictly_below_diag( doff_t diagoff, trans_t trans, dim_t m, dim_t n )
{
	return ( bool )
	       ( bli_does_trans( trans )
	         ? ( ( doff_t )m <=  diagoff )
	         : ( ( doff_t )n <=  diagoff ) );
}

BLIS_INLINE bool bli_is_outside_diag( doff_t diagoff, trans_t trans, dim_t m, dim_t n )
{
	return ( bool )
	       ( bli_is_strictly_above_diag( diagoff, trans, m, n ) ||
	         bli_is_strictly_below_diag( diagoff, trans, m, n ) );
}

BLIS_INLINE bool bli_is_stored_subpart( doff_t diagoff, trans_t trans, uplo_t uplo, dim_t m, dim_t n )
{
	return ( bool )
	( ( bli_is_upper( uplo ) && bli_is_strictly_above_diag( diagoff, trans, m, n ) ) ||
	  ( bli_is_lower( uplo ) && bli_is_strictly_below_diag( diagoff, trans, m, n ) ) );
}

BLIS_INLINE bool bli_is_unstored_subpart( doff_t diagoff, trans_t trans, uplo_t uplo, dim_t m, dim_t n )
{
	return ( bool )
	( ( bli_is_upper( uplo ) && bli_is_strictly_below_diag( diagoff, trans, m, n ) ) ||
	  ( bli_is_lower( uplo ) && bli_is_strictly_above_diag( diagoff, trans, m, n ) ) );
}

BLIS_INLINE bool bli_is_strictly_above_diag_n( doff_t diagoff, dim_t m, dim_t n )
{
	return ( bool )
	       ( ( doff_t )m <= -diagoff );
}

BLIS_INLINE bool bli_is_strictly_below_diag_n( doff_t diagoff, dim_t m, dim_t n )
{
	return ( bool )
	       ( ( doff_t )n <=  diagoff );
}

BLIS_INLINE bool bli_intersects_diag_n( doff_t diagoff, dim_t m, dim_t n )
{
	return ( bool )
	       ( !bli_is_strictly_above_diag_n( diagoff, m, n ) &&
	         !bli_is_strictly_below_diag_n( diagoff, m, n ) );
}

BLIS_INLINE bool bli_is_outside_diag_n( doff_t diagoff, dim_t m, dim_t n )
{
	return ( bool )
	       ( bli_is_strictly_above_diag_n( diagoff, m, n ) ||
	         bli_is_strictly_below_diag_n( diagoff, m, n ) );
}

BLIS_INLINE bool bli_is_stored_subpart_n( doff_t diagoff, uplo_t uplo, dim_t m, dim_t n )
{
	return ( bool )
	( ( bli_is_upper( uplo ) && bli_is_strictly_above_diag_n( diagoff, m, n ) ) ||
	  ( bli_is_lower( uplo ) && bli_is_strictly_below_diag_n( diagoff, m, n ) ) );
}

BLIS_INLINE bool bli_is_unstored_subpart_n( doff_t diagoff, uplo_t uplo, dim_t m, dim_t n )
{
	return ( bool )
	( ( bli_is_upper( uplo ) && bli_is_strictly_below_diag_n( diagoff, m, n ) ) ||
	  ( bli_is_lower( uplo ) && bli_is_strictly_above_diag_n( diagoff, m, n ) ) );
}


// pruning-related

BLIS_INLINE void bli_prune_unstored_region_top_l( doff_t* diagoff, dim_t* m, dim_t* n, dim_t* offm_inc )
{
	*offm_inc = 0;

	// If the diagonal intersects the left side of the matrix,
	// ignore the area above that intersection.
	if ( *diagoff < 0 )
	{
		*m        = *m + *diagoff;
		*offm_inc =    - *diagoff;
		*diagoff  = 0;
	}
}

BLIS_INLINE void bli_prune_unstored_region_right_l( doff_t* diagoff, dim_t* m, dim_t* n, dim_t* offn_inc )
{
	*offn_inc = 0;

	// If the diagonal intersects the bottom side of the matrix,
	// ignore the area to the right of that intersection.
	if ( *n > *diagoff + *m )
	{
		*n = *diagoff + *m;
	}
}

BLIS_INLINE void bli_prune_unstored_region_left_u( doff_t* diagoff, dim_t* m, dim_t* n, dim_t* offn_inc )
{
	*offn_inc = 0;

	// If the diagonal intersects the top side of the matrix,
	// ignore the area to the left of that intersection.
	if ( *diagoff > 0 )
	{
		*n        = *n - *diagoff;
		*offn_inc =    + *diagoff;
		*diagoff  = 0;
	}
}

BLIS_INLINE void bli_prune_unstored_region_bottom_u( doff_t* diagoff, dim_t* m, dim_t* n, dim_t* offm_inc )
{
	*offm_inc = 0;

	// If the diagonal intersects the right side of the matrix,
	// ignore the area below that intersection.
	if ( *m > -(*diagoff) + *n )
	{
		*m = -(*diagoff) + *n;
	}
}


// thread range-related

BLIS_INLINE void bli_rotate180_trapezoid( doff_t* diagoff, uplo_t* uplo, dim_t* m, dim_t* n )
{
	*diagoff = *n - *diagoff - *m;
	bli_toggle_uplo( uplo );
}

BLIS_INLINE void bli_reflect_about_diag( doff_t* diagoff, uplo_t* uplo, dim_t* m, dim_t* n )
{
	bli_swap_dims( m, n );
	bli_negate_diag_offset( diagoff );
	bli_toggle_uplo( uplo );
}

// we don't know the type of a, so this must be a macro
// rs_a and cs_a must be variables and not expressions
#define bli_reflect_to_stored_part( diagoff, a, rs_a, cs_a ) \
do { \
	a += ( diagoff ) * ( cs_a - rs_a ); \
	bli_swap_incs( &rs_a, &cs_a ); \
} while (0) \

BLIS_INLINE void bli_reverse_index_direction( dim_t n, dim_t* start, dim_t* end )
{
	dim_t start2 = n - *start;
	dim_t end2   = n - *end;
	*start = end2;
	*end   = start2;
}


// mdim_t-related

BLIS_INLINE bool bli_is_m_dim( mdim_t mdim )
{
	return ( bool )
	       ( mdim == BLIS_M );
}

BLIS_INLINE bool bli_is_n_dim( mdim_t mdim )
{
	return ( bool )
	       ( mdim == BLIS_N );
}

BLIS_INLINE mdim_t bli_dim_toggled( mdim_t mdim )
{
	return ( mdim_t )
	       ( mdim == BLIS_M ? BLIS_N : BLIS_M );
}

BLIS_INLINE void bli_toggle_dim( mdim_t* mdim )
{
	*mdim = bli_dim_toggled( *mdim );
}


// stor3_t-related

BLIS_INLINE stor3_t bli_stor3_from_strides( inc_t rs_c, inc_t cs_c,
                                       inc_t rs_a, inc_t cs_a,
                                       inc_t rs_b, inc_t cs_b  )
{
	// If any matrix is general-stored, return the stor3_t id for the
	// general-purpose sup microkernel.
	if ( bli_is_gen_stored( rs_c, cs_c ) ||
	     bli_is_gen_stored( rs_a, cs_a ) ||
	     bli_is_gen_stored( rs_b, cs_b ) ) return BLIS_XXX;

	// Otherwise, compute and return the stor3_t id as follows.
	const bool c_is_col = bli_is_col_stored( rs_c, cs_c );
	const bool a_is_col = bli_is_col_stored( rs_a, cs_a );
	const bool b_is_col = bli_is_col_stored( rs_b, cs_b );

	return ( stor3_t )( 4 * c_is_col +
	                    2 * a_is_col +
	                    1 * b_is_col );
}

BLIS_INLINE stor3_t bli_stor3_trans( stor3_t id )
{
#if 1
	stor3_t map[ BLIS_NUM_3OP_RC_COMBOS ]
	=
	{
	  ( stor3_t )7,  // BLIS_RRR = 0  ->  BLIS_CCC = 7
	  ( stor3_t )5,  // BLIS_RRC = 1  ->  BLIS_CRC = 5
	  ( stor3_t )6,  // BLIS_RCR = 2  ->  BLIS_CCR = 6
	  ( stor3_t )4,  // BLIS_RCC = 3  ->  BLIS_CRR = 4
	  ( stor3_t )3,  // BLIS_CRR = 4  ->  BLIS_RCC = 3
	  ( stor3_t )1,  // BLIS_CRC = 5  ->  BLIS_RRC = 1
	  ( stor3_t )2,  // BLIS_CCR = 6  ->  BLIS_RCR = 2
	  ( stor3_t )0,  // BLIS_CCC = 7  ->  BLIS_RRR = 0
	};

	return map[id];
#else
	return   ( ( id & 0x4 ) ^ 0x4 )        | // flip c bit
	       ( ( ( id & 0x1 ) ^ 0x1 ) << 1 ) | // flip b bit and move to a position
	       ( ( ( id & 0x2 ) ^ 0x2 ) >> 1 );  // flip a bit and move to b position
#endif
}

BLIS_INLINE ukr_t bli_stor3_ukr( stor3_t id )
{
	switch ( id )
	{
		case BLIS_RRR: return BLIS_GEMMSUP_RRR_UKR;
		case BLIS_RRC: return BLIS_GEMMSUP_RRC_UKR;
		case BLIS_RCR: return BLIS_GEMMSUP_RCR_UKR;
		case BLIS_RCC: return BLIS_GEMMSUP_RCC_UKR;
		case BLIS_CRR: return BLIS_GEMMSUP_CRR_UKR;
		case BLIS_CRC: return BLIS_GEMMSUP_CRC_UKR;
		case BLIS_CCR: return BLIS_GEMMSUP_CCR_UKR;
		case BLIS_CCC: return BLIS_GEMMSUP_CCC_UKR;
		default: return BLIS_GEMMSUP_XXX_UKR;
	}
}

BLIS_INLINE stor3_t bli_stor3_transa( stor3_t id )
{
#if 0
	stor3_t map[ BLIS_NUM_3OP_RC_COMBOS ]
	=
	{
	  ( stor3_t )1,  // BLIS_RRR = 0  ->  BLIS_RRC = 1
	  ( stor3_t )0,  // BLIS_RRC = 1  ->  BLIS_RRR = 0
	  ( stor3_t )3,  // BLIS_RCR = 2  ->  BLIS_RCC = 3
	  ( stor3_t )2,  // BLIS_RCC = 3  ->  BLIS_RCR = 2
	  ( stor3_t )5,  // BLIS_CRR = 4  ->  BLIS_CRC = 5
	  ( stor3_t )4,  // BLIS_CRC = 5  ->  BLIS_CRR = 4
	  ( stor3_t )7,  // BLIS_CCR = 6  ->  BLIS_CCC = 7
	  ( stor3_t )6,  // BLIS_CCC = 7  ->  BLIS_CCR = 6
	};

	return map[id];
#else
	return ( stor3_t )( id ^ 0x1 );
#endif
}

BLIS_INLINE stor3_t bli_stor3_transb( stor3_t id )
{
#if 0
	stor3_t map[ BLIS_NUM_3OP_RC_COMBOS ]
	=
	{
	  ( stor3_t )2,  // BLIS_RRR = 0  ->  BLIS_RCR = 2
	  ( stor3_t )3,  // BLIS_RRC = 1  ->  BLIS_RCC = 3
	  ( stor3_t )0,  // BLIS_RCR = 2  ->  BLIS_RRR = 0
	  ( stor3_t )1,  // BLIS_RCC = 3  ->  BLIS_RRC = 1
	  ( stor3_t )6,  // BLIS_CRR = 4  ->  BLIS_CCR = 6
	  ( stor3_t )7,  // BLIS_CRC = 5  ->  BLIS_CCC = 7
	  ( stor3_t )4,  // BLIS_CCR = 6  ->  BLIS_CRR = 4
	  ( stor3_t )5,  // BLIS_CCC = 7  ->  BLIS_CRC = 5
	};

	return map[id];
#else
	return ( stor3_t )( id ^ 0x2 );
#endif
}


// index-related

BLIS_INLINE bool bli_is_edge_f( dim_t i, dim_t n_iter, dim_t n_left )
{
	return ( bool )
	       ( i == n_iter - 1 && n_left != 0 );
}

BLIS_INLINE bool bli_is_not_edge_f( dim_t i, dim_t n_iter, dim_t n_left )
{
	return ( bool )
	       ( i != n_iter - 1 || n_left == 0 );
}

BLIS_INLINE bool bli_is_edge_b( dim_t i, dim_t n_iter, dim_t n_left )
{
	return ( bool )
	       ( i == 0 && n_left != 0 );
}

BLIS_INLINE bool bli_is_not_edge_b( dim_t i, dim_t n_iter, dim_t n_left )
{
	return ( bool )
	       ( i != 0 || n_left == 0 );
}

BLIS_INLINE bool bli_is_last_iter_sl( dim_t i, dim_t end_iter )
{
	return ( bool )
	       ( i == end_iter - 1 );
}

BLIS_INLINE bool bli_is_last_iter_rr( dim_t i, dim_t end_iter, dim_t tid, dim_t nth )
{
	return ( bool )
	       ( i == end_iter - 1 - ( ( end_iter - tid - 1 ) % nth ) );
}

BLIS_INLINE bool bli_is_last_iter_slrr( dim_t i, dim_t end_iter, dim_t tid, dim_t nth )
{
#ifdef BLIS_ENABLE_JRIR_RR
	return bli_is_last_iter_rr( i, end_iter, tid, nth );
#else // ifdef ( _SLAB || _TLB )
	return bli_is_last_iter_sl( i, end_iter );
#endif
}

BLIS_INLINE bool bli_is_last_iter_l( dim_t i, dim_t end_iter, dim_t tid, dim_t nth )
{
	return bli_is_last_iter_slrr( i, end_iter, tid, nth );
}

BLIS_INLINE bool bli_is_last_iter_u( doff_t diagoff, dim_t mr, dim_t nr, inc_t inc )
{
	return bli_is_strictly_below_diag_n( diagoff + inc*mr, mr, nr );
}

BLIS_INLINE bool bli_is_last_iter_tlb_l( dim_t i, dim_t end_iter )
{
	return bli_is_last_iter_sl( i, end_iter );
}

BLIS_INLINE bool bli_is_last_iter_tlb_u( doff_t diagoff, dim_t mr, dim_t nr )
{
	return bli_is_strictly_below_diag_n( diagoff + 1*mr, mr, nr );
}

BLIS_INLINE bool bli_is_my_iter_sl( dim_t i, dim_t st, dim_t en )
{
	return ( st <= i && i < en );
}

BLIS_INLINE bool bli_is_my_iter_rr( dim_t i, dim_t work_id, dim_t n_way )
{
	return ( i % n_way == work_id % n_way );
}

BLIS_INLINE bool bli_is_my_iter( dim_t i, dim_t st, dim_t en, dim_t work_id, dim_t n_way )
{
	// NOTE: This function is (as of this writing) only called from packm.
	// If the structure of the cpp macros below is ever changed, make sure
	// it is still consistent with that of bli_thread_range_slrr() since
	// these functions are used together in packm.

#ifdef BLIS_ENABLE_JRIR_RR
	return bli_is_my_iter_rr( i, work_id, n_way );
#else // ifdef ( _SLAB || _TLB )
	return bli_is_my_iter_sl( i, st, en );
#endif
}


// packbuf_t-related

BLIS_INLINE guint_t bli_packbuf_index( packbuf_t buf_type )
{
	return ( guint_t )
	       ( ( buf_type & BLIS_PACK_BUFFER_BITS ) >> BLIS_PACK_BUFFER_SHIFT );
}

// pack_t-related

BLIS_INLINE bool bli_is_packed( pack_t schema )
{
	return ( bool )
	       ( schema & BLIS_PACK_BIT );
}

BLIS_INLINE bool bli_is_panel_packed( pack_t schema )
{
	return ( bool )
	       ( schema & BLIS_PACK_PANEL_BIT );
}

BLIS_INLINE bool bli_is_1r_packed( pack_t schema )
{
	return ( bool )
	       ( ( schema & BLIS_PACK_FORMAT_BITS ) == BLIS_BITVAL_1R );
}

BLIS_INLINE bool bli_is_1e_packed( pack_t schema )
{
	return ( bool )
	       ( ( schema & BLIS_PACK_FORMAT_BITS ) == BLIS_BITVAL_1E );
}

BLIS_INLINE bool bli_is_1m_packed( pack_t schema )
{
	return ( bool )
	       ( bli_is_1r_packed( schema ) ||
	         bli_is_1e_packed( schema ) );
}

BLIS_INLINE bool bli_is_ro_packed( pack_t schema )
{
	return ( bool )
	       ( ( schema & BLIS_PACK_FORMAT_BITS ) == BLIS_BITVAL_RO );
}

BLIS_INLINE bool bli_is_nat_packed( pack_t schema )
{
	return ( bool )
	       ( ( schema & BLIS_PACK_FORMAT_BITS ) == 0 );
}

BLIS_INLINE bool bli_is_ind_packed( pack_t schema )
{
	return ( bool )
	       ( ( schema & BLIS_PACK_FORMAT_BITS ) != 0 );
}

BLIS_INLINE guint_t bli_pack_schema_index( pack_t schema )
{
	return ( guint_t )
	       ( ( schema & BLIS_PACK_FORMAT_BITS ) >> BLIS_PACK_FORMAT_SHIFT );
}


// Set dimensions, increments, effective uplo/diagoff, etc for ONE matrix
// argument.

BLIS_INLINE void bli_set_dims_incs_uplo_1m
     (
       doff_t  diagoffa, diag_t diaga,
       uplo_t  uploa,    dim_t  m,          dim_t  n,      inc_t  rs_a, inc_t  cs_a,
       uplo_t* uplo_eff, dim_t* n_elem_max, dim_t* n_iter, inc_t* inca, inc_t* lda,
       dim_t*  ij0,      dim_t* n_shift
     )
{
	// This is to prevent the compiler from warning about uninitialized
	// variables.
	*ij0     = 0;
	*n_shift = 0;

	// If matrix A is entirely "unstored", that is, if either:
	// - A is lower-stored and entirely above the diagonal, or
	// - A is upper-stored and entirely below the diagonal
	// then we mark the storage as implicitly zero.
	if ( bli_is_unstored_subpart( diagoffa, BLIS_NO_TRANSPOSE, uploa, m, n ) )
	{
		*uplo_eff = BLIS_ZEROS;
	}
	else
	{
		doff_t diagoffa_use_ = diagoffa;
		doff_t diagoff_eff_;
		dim_t  n_iter_max_;

		if ( bli_is_unit_diag( diaga ) )
			bli_shift_diag_offset_to_shrink_uplo( uploa, &diagoffa_use_ );

		// If matrix A is entirely "stored", that is, if either:
		// - A is upper-stored and entirely above the diagonal, or
		// - A is lower-stored and entirely below the diagonal
		// then we mark the storage as dense.
		if ( bli_is_stored_subpart( diagoffa_use_, BLIS_NO_TRANSPOSE, uploa, m, n ) )
			uploa = BLIS_DENSE;

		n_iter_max_  = n;
		*n_elem_max   = m;
		*inca         = rs_a;
		*lda          = cs_a;
		*uplo_eff     = uploa;
		diagoff_eff_ = diagoffa_use_;

		if ( bli_is_row_tilted( *n_elem_max, n_iter_max_, *inca, *lda ) )
		{
			bli_swap_dims( &n_iter_max_, n_elem_max );
			bli_swap_incs( inca, lda );
			bli_toggle_uplo( uplo_eff );
			bli_negate_diag_offset( &diagoff_eff_ );
		}

		if ( bli_is_dense( *uplo_eff ) )
		{
			*n_iter = n_iter_max_;
		}
		else if ( bli_is_upper( *uplo_eff ) )
		{
			if ( diagoff_eff_ < 0 )
			{
				*ij0        = 0;
				*n_shift    = -diagoff_eff_;
				*n_elem_max = bli_min( *n_elem_max, *n_shift + bli_min( m, n ) );
				*n_iter     = n_iter_max_;
			}
			else
			{
				*ij0        = diagoff_eff_;
				*n_shift    = 0;
				*n_iter     = n_iter_max_ - diagoff_eff_;
			}
		}
		else // if ( bli_is_lower( *uplo_eff ) )
		{
			if ( diagoff_eff_ < 0 )
			{
				*ij0        = -diagoff_eff_;
				*n_shift    = 0;
				*n_elem_max = *n_elem_max + diagoff_eff_;
				*n_iter     = bli_min( *n_elem_max, bli_min( m, n ) );
			}
			else
			{
				*ij0        = 0;
				*n_shift    = diagoff_eff_;
				*n_iter     = bli_min( n_iter_max_, *n_shift + bli_min( m, n ) );
			}
		}
	}
}

// Set dimensions, increments, effective uplo/diagoff, etc for ONE matrix
// argument (without column-wise stride optimization).

BLIS_INLINE void bli_set_dims_incs_uplo_1m_noswap
     (
       doff_t  diagoffa, diag_t diaga,
       uplo_t  uploa,    dim_t  m,          dim_t  n,      inc_t  rs_a, inc_t  cs_a,
       uplo_t* uplo_eff, dim_t* n_elem_max, dim_t* n_iter, inc_t* inca, inc_t* lda,
       dim_t*  ij0,      dim_t* n_shift
     )
{
	// This is to prevent the compiler from warning about uninitialized
	// variables.
	*ij0     = 0;
	*n_shift = 0;

	// If matrix A is entirely "unstored", that is, if either:
	// - A is lower-stored and entirely above the diagonal, or
	// - A is upper-stored and entirely below the diagonal
	// then we mark the storage as implicitly zero.
	if ( bli_is_unstored_subpart( diagoffa, BLIS_NO_TRANSPOSE, uploa, m, n ) )
	{
		*uplo_eff = BLIS_ZEROS;
	}
	else
	{
		doff_t diagoffa_use_ = diagoffa;
		doff_t diagoff_eff_;
		dim_t  n_iter_max_;

		if ( bli_is_unit_diag( diaga ) )
			bli_shift_diag_offset_to_shrink_uplo( uploa, &diagoffa_use_ );

		// If matrix A is entirely "stored", that is, if either:
		// - A is upper-stored and entirely above the diagonal, or
		// - A is lower-stored and entirely below the diagonal
		// then we mark the storage as dense.
		if ( bli_is_stored_subpart( diagoffa_use_, BLIS_NO_TRANSPOSE, uploa, m, n ) )
			uploa = BLIS_DENSE;

		n_iter_max_  = n;
		*n_elem_max   = m;
		*inca         = rs_a;
		*lda          = cs_a;
		*uplo_eff     = uploa;
		diagoff_eff_ = diagoffa_use_;

		if ( bli_is_dense( *uplo_eff ) )
		{
			*n_iter = n_iter_max_;
		}
		else if ( bli_is_upper( *uplo_eff ) )
		{
			if ( diagoff_eff_ < 0 )
			{
				*ij0        = 0;
				*n_shift    = -diagoff_eff_;
				*n_elem_max = bli_min( *n_elem_max, *n_shift + bli_min( m, n ) );
				*n_iter     = n_iter_max_;
			}
			else
			{
				*ij0        = diagoff_eff_;
				*n_shift    = 0;
				*n_iter     = n_iter_max_ - diagoff_eff_;
			}
		}
		else // if ( bli_is_lower( *uplo_eff ) )
		{
			if ( diagoff_eff_ < 0 )
			{
				*ij0        = -diagoff_eff_;
				*n_shift    = 0;
				*n_elem_max = *n_elem_max + diagoff_eff_;
				*n_iter     = bli_min( *n_elem_max, bli_min( m, n ) );
			}
			else
			{
				*ij0        = 0;
				*n_shift    = diagoff_eff_;
				*n_iter     = bli_min( n_iter_max_, *n_shift + bli_min( m, n ) );
			}
		}
	}
}

// Set dimensions and increments for TWO matrix arguments.

BLIS_INLINE void bli_set_dims_incs_2m
     (
       trans_t transa,
       dim_t  m,      dim_t  n,      inc_t  rs_a, inc_t  cs_a,
                                     inc_t  rs_b, inc_t  cs_b,
       dim_t* n_elem, dim_t* n_iter, inc_t* inca, inc_t* lda,
                                     inc_t* incb, inc_t* ldb
     )
{
	{
		*n_iter = n;
		*n_elem = m;
		*inca   = rs_a;
		*lda    = cs_a;
		*incb   = rs_b;
		*ldb    = cs_b;

		if ( bli_does_trans( transa ) )
		{
			bli_swap_incs( inca, lda );
		}

		if ( bli_is_row_tilted( *n_elem, *n_iter, *incb, *ldb ) &&
		     bli_is_row_tilted( *n_elem, *n_iter, *inca, *lda ) )
		{
			bli_swap_dims( n_iter, n_elem );
			bli_swap_incs( inca, lda );
			bli_swap_incs( incb, ldb );
		}
	}
}

// Set dimensions, increments, effective uplo/diagoff, etc for TWO matrix
// arguments.

BLIS_INLINE void bli_set_dims_incs_uplo_2m
     (
       doff_t  diagoffa, diag_t diaga, trans_t transa,
       uplo_t  uploa,    dim_t  m,          dim_t  n,      inc_t  rs_a, inc_t  cs_a,
                                                           inc_t  rs_b, inc_t  cs_b,
       uplo_t* uplo_eff, dim_t* n_elem_max, dim_t* n_iter, inc_t* inca, inc_t* lda,
                                                           inc_t* incb, inc_t* ldb,
       dim_t*  ij0,      dim_t* n_shift
     )
{
	// This is to prevent the compiler from warning about uninitialized
	// variables.
	*ij0     = 0;
	*n_shift = 0;

	// If matrix A is entirely "unstored", that is, if either:
	// - A is lower-stored and entirely above the diagonal, or
	// - A is upper-stored and entirely below the diagonal
	// then we mark the storage as implicitly zero.
	if ( bli_is_unstored_subpart( diagoffa, transa, uploa, m, n ) )
	{
		*uplo_eff = BLIS_ZEROS;
	}
	else
	{
		doff_t diagoffa_use_ = diagoffa;
		doff_t diagoff_eff_;
		dim_t  n_iter_max_;

		if ( bli_is_unit_diag( diaga ) )
			bli_shift_diag_offset_to_shrink_uplo( uploa, &diagoffa_use_ );

		// If matrix A is entirely "stored", that is, if either:
		// - A is upper-stored and entirely above the diagonal, or
		// - A is lower-stored and entirely below the diagonal
		// then we mark the storage as dense.
		if ( bli_is_stored_subpart( diagoffa_use_, transa, uploa, m, n ) )
			uploa = BLIS_DENSE;

		n_iter_max_   = n;
		*n_elem_max   = m;
		*inca         = rs_a;
		*lda          = cs_a;
		*incb         = rs_b;
		*ldb          = cs_b;
		*uplo_eff     = uploa;
		diagoff_eff_ = diagoffa_use_;

		if ( bli_does_trans( transa ) )
		{
			bli_swap_incs( inca, lda );
			bli_toggle_uplo( uplo_eff );
			bli_negate_diag_offset( &diagoff_eff_ );
		}

		if ( bli_is_row_tilted( *n_elem_max, n_iter_max_, *incb, *ldb ) &&
		     bli_is_row_tilted( *n_elem_max, n_iter_max_, *inca, *lda ) )
		{
			bli_swap_dims( &n_iter_max_, n_elem_max );
			bli_swap_incs( inca, lda );
			bli_swap_incs( incb, ldb );
			bli_toggle_uplo( uplo_eff );
			bli_negate_diag_offset( &diagoff_eff_ );
		}

		if ( bli_is_dense( *uplo_eff ) )
		{
			*n_iter = n_iter_max_;
		}
		else if ( bli_is_upper( *uplo_eff ) )
		{
			if ( diagoff_eff_ < 0 )
			{
				*ij0        = 0;
				*n_shift    = -diagoff_eff_;
				*n_elem_max = bli_min( *n_elem_max, *n_shift + bli_min( m, n ) );
				*n_iter     = n_iter_max_;
			}
			else
			{
				*ij0        = diagoff_eff_;
				*n_shift    = 0;
				*n_iter     = n_iter_max_ - diagoff_eff_;
			}
		}
		else // if ( bli_is_lower( *uplo_eff ) )
		{
			if ( diagoff_eff_ < 0 )
			{
				*ij0        = -diagoff_eff_;
				*n_shift    = 0;
				*n_elem_max = *n_elem_max + diagoff_eff_;
				*n_iter     = bli_min( *n_elem_max, bli_min( m, n ) );
			}
			else
			{
				*ij0        = 0;
				*n_shift    = diagoff_eff_;
				*n_iter     = bli_min( n_iter_max_, *n_shift + bli_min( m, n ) );
			}
		}
	}
}

// Set dimensions, increments, etc for ONE matrix argument when operating
// on the diagonal.

BLIS_INLINE void bli_set_dims_incs_1d
     (
       doff_t diagoffx,
       dim_t  m,    dim_t  n,      inc_t  rs_x, inc_t  cs_x,
       dim_t* offx, dim_t* n_elem, inc_t* incx
     )
{
	if ( diagoffx < 0 )
	{
		*n_elem = bli_min( m - ( dim_t )(-diagoffx), n );
		*offx   = ( dim_t )(-diagoffx) * rs_x;
	}
	else
	{
		*n_elem = bli_min( n - ( dim_t )( diagoffx), m );
		*offx   = ( dim_t )( diagoffx) * cs_x;
	}

	*incx = rs_x + cs_x; \
}

// Set dimensions, increments, etc for TWO matrix arguments when operating
// on diagonals.
BLIS_INLINE void bli_set_dims_incs_2d
     (
       doff_t diagoffx, trans_t transx,
       dim_t  m, dim_t  n, inc_t  rs_x, inc_t  cs_x,
                           inc_t  rs_y, inc_t  cs_y,
       dim_t* offx, dim_t* offy, dim_t* n_elem,
       inc_t* incx, inc_t* incy
     )
{
	doff_t diagoffy_ = bli_diag_offset_with_trans( transx, diagoffx );

	if ( diagoffx < 0 ) *offx = -diagoffx * rs_x;
	else                *offx =  diagoffx * cs_x;

	if ( diagoffy_ < 0 )
	{
		*n_elem = bli_min( m - ( dim_t )(-diagoffy_), n );
		*offy   = -diagoffy_ * rs_y;
	}
	else
	{
		*n_elem = bli_min( n - ( dim_t )( diagoffy_), m );
		*offy   = diagoffy_ * cs_y;
	}

	*incx = rs_x + cs_x;
	*incy = rs_y + cs_y;
}


#endif
// end bli_param_macro_defs.h
#line 104 "../blis-2.0/frame/include//bli_macro_defs.h"

// begin bli_complex_macro_defs.h
#line 1 "../blis-2.0/frame/include//bli_complex_macro_defs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2023, Southern Methodist University

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_COMPLEX_MACRO_DEFS_H
#define BLIS_COMPLEX_MACRO_DEFS_H


// -- Real and imaginary accessor macros --


#define bli_sreal( x )  ( x )
#define bli_simag( x )  ( 0.0F )
#define bli_dreal( x )  ( x )
#define bli_dimag( x )  ( 0.0 )
#define bli_ireal( x )  ( x )
#define bli_iimag( x )  ( 0 )


#if defined(__cplusplus) && defined(BLIS_ENABLE_STD_COMPLEX)

} // extern "C"

// Create functions bli_[cz]{real,imag} for std::complex<T> which mimic those
// for the simple struct version. Since normally x.real/x.imag are
// lvalues, we have to create a wrapper since x.real()/x.imag() in std::complex
// are rvalues. These will only be used if the user has typedef'd scomplex as
// std::complex<float> and dcomplex as std::complex<double> themselves.

#include <complex> // skipped

template <typename T, bool Imag>
struct bli_complex_wrapper
{
	std::complex<T>& ref;

	bli_complex_wrapper(std::complex<T>& ref) : ref(ref) {}

	operator T() const { return Imag ? ref.imag() : ref.real(); }

	bli_complex_wrapper& operator=(const bli_complex_wrapper& other)
	{
		return *this = static_cast<T>( other );
	}

	bli_complex_wrapper& operator=(T other)
	{
		if (Imag)
			ref.imag(other);
		else
			ref.real(other);
		return *this;
	}
};

inline bli_complex_wrapper<float,false> bli_creal( std::complex<float>& x )
{
	return x;
}

inline float bli_creal( const std::complex<float>& x )
{
	return x.real();
}

inline bli_complex_wrapper<float,true> bli_cimag( std::complex<float>& x )
{
	return x;
}

inline float bli_cimag( const std::complex<float>& x )
{
	return x.imag();
}

inline bli_complex_wrapper<double,false> bli_zreal( std::complex<double>& x )
{
	return x;
}

inline double bli_zreal( const std::complex<double>& x )
{
	return x.real();
}

inline bli_complex_wrapper<double,true> bli_zimag( std::complex<double>& x )
{
	return x;
}

inline double bli_zimag( const std::complex<double>& x )
{
	return x.imag();
}

#define __typeof__(x) auto

extern "C"
{

#elif !defined(BLIS_ENABLE_C99_COMPLEX)


#define bli_creal( x )  ( (x).real )
#define bli_cimag( x )  ( (x).imag )
#define bli_zreal( x )  ( (x).real )
#define bli_zimag( x )  ( (x).imag )


#else // ifdef BLIS_ENABLE_C99_COMPLEX

// Note that these definitions probably don't work because of constructs
// like `bli_zreal( x ) = yr`.

#define bli_creal( x )  ( crealf(x) )
#define bli_cimag( x )  ( cimagf(x) )
#define bli_zreal( x )  ( creal(x) )
#define bli_zimag( x )  ( cimag(x) )


#endif // BLIS_ENABLE_C99_COMPLEX


#endif

// end bli_complex_macro_defs.h
#line 105 "../blis-2.0/frame/include//bli_macro_defs.h"

// begin bli_obj_macro_defs.h
#line 1 "../blis-2.0/frame/include//bli_obj_macro_defs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2016, Hewlett Packard Enterprise Development LP
   Copyright (C) 2019, Advanced Micro Devices, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_OBJ_MACRO_DEFS_H
#define BLIS_OBJ_MACRO_DEFS_H


// -- Object query/modification macros --

// Info query

BLIS_INLINE num_t bli_obj_dt( const obj_t* obj )
{
	return ( num_t )
	       ( obj->info & BLIS_DATATYPE_BITS );
}

BLIS_INLINE bool bli_obj_is_float( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_dt( obj ) == BLIS_BITVAL_FLOAT_TYPE );
}

BLIS_INLINE bool bli_obj_is_double( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_dt( obj ) == BLIS_BITVAL_DOUBLE_TYPE );
}

BLIS_INLINE bool bli_obj_is_scomplex( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_dt( obj ) == BLIS_BITVAL_SCOMPLEX_TYPE );
}

BLIS_INLINE bool bli_obj_is_dcomplex( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_dt( obj ) == BLIS_BITVAL_DCOMPLEX_TYPE );
}

BLIS_INLINE bool bli_obj_is_int( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_dt( obj ) == BLIS_BITVAL_INT_TYPE );
}

BLIS_INLINE bool bli_obj_is_const( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_dt( obj ) == BLIS_BITVAL_CONST_TYPE );
}

BLIS_INLINE dom_t bli_obj_domain( const obj_t* obj )
{
	return ( dom_t )
	       ( obj->info & BLIS_DOMAIN_BIT );
}

BLIS_INLINE prec_t bli_obj_prec( const obj_t* obj )
{
	return ( prec_t )
	       ( obj->info & BLIS_PRECISION_BIT );
}

BLIS_INLINE bool bli_obj_is_single_prec( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_prec( obj ) == BLIS_BITVAL_SINGLE_PREC );
}

BLIS_INLINE bool bli_obj_is_double_prec( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_prec( obj ) == BLIS_BITVAL_DOUBLE_PREC );
}

BLIS_INLINE num_t bli_obj_dt_proj_to_single_prec( const obj_t* obj )
{
	return ( num_t )
	       ( bli_obj_dt( obj ) & ~BLIS_BITVAL_SINGLE_PREC );
}

BLIS_INLINE num_t bli_obj_dt_proj_to_double_prec( const obj_t* obj )
{
	return ( num_t )
	       ( bli_obj_dt( obj ) | BLIS_BITVAL_DOUBLE_PREC );
}

BLIS_INLINE bool bli_obj_is_real( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_domain( obj ) == BLIS_BITVAL_REAL &&
	         !bli_obj_is_const( obj ) );
}

BLIS_INLINE bool bli_obj_is_complex( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_domain( obj ) == BLIS_BITVAL_COMPLEX &&
	         !bli_obj_is_const( obj ) );
}

BLIS_INLINE num_t bli_obj_dt_proj_to_real( const obj_t* obj )
{
	return ( num_t )
	       ( bli_obj_dt( obj ) & ~BLIS_BITVAL_COMPLEX );
}

BLIS_INLINE num_t bli_obj_dt_proj_to_complex( const obj_t* obj )
{
	return ( num_t )
	       ( bli_obj_dt( obj ) | BLIS_BITVAL_COMPLEX );
}

BLIS_INLINE prec_t bli_obj_comp_prec( const obj_t* obj )
{
	return ( prec_t )
	       ( ( obj->info & BLIS_COMP_PREC_BIT ) >> BLIS_COMP_PREC_SHIFT );
}

// NOTE: This function queries info2.
BLIS_INLINE num_t bli_obj_scalar_dt( const obj_t* obj )
{
	return ( num_t )
	       ( ( obj->info2 & BLIS_SCALAR_DT_BITS ) >> BLIS_SCALAR_DT_SHIFT );
}

// NOTE: This function queries info2.
BLIS_INLINE dom_t bli_obj_scalar_domain( const obj_t* obj )
{
	return ( dom_t )
	       ( ( obj->info2 & BLIS_SCALAR_DOMAIN_BIT ) >> BLIS_SCALAR_DT_SHIFT );
}

// NOTE: This function queries info2.
BLIS_INLINE prec_t bli_obj_scalar_prec( const obj_t* obj )
{
	return ( prec_t )
	       ( ( obj->info2 & BLIS_SCALAR_PREC_BIT ) >> BLIS_SCALAR_DT_SHIFT );
}

BLIS_INLINE trans_t bli_obj_conjtrans_status( const obj_t* obj )
{
	return ( trans_t )
	       ( obj->info & BLIS_CONJTRANS_BITS );
}

BLIS_INLINE trans_t bli_obj_onlytrans_status( const obj_t* obj )
{
	return ( trans_t )
	       ( obj->info & BLIS_TRANS_BIT );
}

BLIS_INLINE bool bli_obj_has_trans( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_onlytrans_status( obj ) == BLIS_BITVAL_TRANS );
}

BLIS_INLINE bool bli_obj_has_notrans( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_onlytrans_status( obj ) == BLIS_BITVAL_NO_TRANS );
}

BLIS_INLINE conj_t bli_obj_conj_status( const obj_t* obj )
{
	return ( conj_t )
	       ( obj->info & BLIS_CONJ_BIT );
}

BLIS_INLINE bool bli_obj_has_conj( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_conj_status( obj ) == BLIS_BITVAL_CONJ );
}

BLIS_INLINE bool bli_obj_has_noconj( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_conj_status( obj ) == BLIS_BITVAL_NO_CONJ );
}

BLIS_INLINE uplo_t bli_obj_uplo( const obj_t* obj )
{
	return ( uplo_t )
	       ( obj->info & BLIS_UPLO_BITS );
}

BLIS_INLINE bool bli_obj_is_upper( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_uplo( obj ) == BLIS_BITVAL_UPPER );
}

BLIS_INLINE bool bli_obj_is_lower( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_uplo( obj ) == BLIS_BITVAL_LOWER );
}

BLIS_INLINE bool bli_obj_is_upper_or_lower( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_is_upper( obj ) ||
	         bli_obj_is_lower( obj ) );
}

BLIS_INLINE bool bli_obj_is_dense( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_uplo( obj ) == BLIS_BITVAL_DENSE );
}

BLIS_INLINE bool bli_obj_is_zeros( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_uplo( obj ) == BLIS_BITVAL_ZEROS );
}

BLIS_INLINE diag_t bli_obj_diag( const obj_t* obj )
{
	return ( diag_t )
	       ( obj->info & BLIS_UNIT_DIAG_BIT );
}

BLIS_INLINE bool bli_obj_has_nonunit_diag( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_diag( obj ) == BLIS_BITVAL_NONUNIT_DIAG );
}

BLIS_INLINE bool bli_obj_has_unit_diag( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_diag( obj ) == BLIS_BITVAL_UNIT_DIAG );
}

BLIS_INLINE bool bli_obj_has_inverted_diag( const obj_t* obj )
{
	return ( bool )
	       ( ( obj->info & BLIS_INVERT_DIAG_BIT ) == BLIS_BITVAL_INVERT_DIAG );
}

BLIS_INLINE bool bli_obj_is_pack_rev_if_upper( const obj_t* obj )
{
	return ( bool )
	       ( ( obj->info & BLIS_PACK_REV_IF_UPPER_BIT ) == BLIS_BITVAL_PACK_REV_IF_UPPER );
}

BLIS_INLINE bool bli_obj_is_pack_rev_if_lower( const obj_t* obj )
{
	return ( bool )
	       ( ( obj->info & BLIS_PACK_REV_IF_LOWER_BIT ) == BLIS_BITVAL_PACK_REV_IF_LOWER );
}

BLIS_INLINE pack_t bli_obj_pack_schema( const obj_t* obj )
{
	return ( pack_t )
	       ( obj->info & BLIS_PACK_SCHEMA_BITS );
}

BLIS_INLINE bool bli_obj_is_packed( const obj_t* obj )
{
	return ( bool )
	       ( obj->info & BLIS_PACK_BIT );
}

BLIS_INLINE bool bli_obj_is_panel_packed( const obj_t* obj )
{
	return ( bool )
	       ( obj->info & BLIS_PACK_PANEL_BIT );
}

BLIS_INLINE packbuf_t bli_obj_pack_buffer_type( const obj_t* obj )
{
	return ( packbuf_t )
	       ( obj->info & BLIS_PACK_BUFFER_BITS );
}

BLIS_INLINE struc_t bli_obj_struc( const obj_t* obj )
{
	return ( struc_t )
	       ( obj->info & BLIS_STRUC_BITS );
}

BLIS_INLINE bool bli_obj_is_general( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_struc( obj ) == BLIS_BITVAL_GENERAL );
}

BLIS_INLINE bool bli_obj_is_hermitian( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_struc( obj ) == BLIS_BITVAL_HERMITIAN );
}

BLIS_INLINE bool bli_obj_is_symmetric( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_struc( obj ) == BLIS_BITVAL_SYMMETRIC );
}

BLIS_INLINE bool bli_obj_is_triangular( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_struc( obj ) == BLIS_BITVAL_TRIANGULAR );
}

// Info modification

BLIS_INLINE void bli_obj_apply_trans( trans_t trans, obj_t* obj )
{
	obj->info = ( objbits_t )
	            ( obj->info ^ trans );
}

BLIS_INLINE void bli_obj_apply_conj( conj_t conj, obj_t* obj )
{
	obj->info = ( objbits_t )
	            ( obj->info ^ conj );
}

BLIS_INLINE void bli_obj_set_conjtrans( trans_t trans, obj_t* obj )
{
	obj->info = ( objbits_t )
	            ( ( obj->info & ~BLIS_CONJTRANS_BITS ) | trans );
}

BLIS_INLINE void bli_obj_set_onlytrans( trans_t trans, obj_t* obj )
{
	obj->info = ( objbits_t )
	            ( ( obj->info & ~BLIS_TRANS_BIT ) | trans );
}

BLIS_INLINE void bli_obj_set_conj( conj_t conj, obj_t* obj )
{
	obj->info = ( objbits_t )
	            ( ( obj->info & ~BLIS_CONJ_BIT ) | conj );
}

BLIS_INLINE void bli_obj_set_uplo( uplo_t uplo, obj_t* obj )
{
	obj->info = ( objbits_t )
	            ( ( obj->info & ~BLIS_UPLO_BITS ) | uplo );
}

BLIS_INLINE void bli_obj_set_diag( diag_t diag, obj_t* obj )
{
	obj->info = ( objbits_t )
	            ( ( obj->info & ~BLIS_UNIT_DIAG_BIT ) | diag );
}

BLIS_INLINE void bli_obj_set_invert_diag( invdiag_t invdiag, obj_t* obj )
{
	obj->info = ( objbits_t )
	            ( ( obj->info & ~BLIS_INVERT_DIAG_BIT ) | invdiag );
}

BLIS_INLINE void bli_obj_set_dt( num_t dt, obj_t* obj )
{
	obj->info = ( objbits_t )
	            ( ( obj->info & ~BLIS_DATATYPE_BITS ) | dt );
}

BLIS_INLINE void bli_obj_set_comp_prec( prec_t dt, obj_t* obj )
{
	obj->info = ( objbits_t )
	            ( ( obj->info & ~BLIS_COMP_PREC_BIT ) |
	              ( dt << BLIS_COMP_PREC_SHIFT ) );
}

// NOTE: This function queries and modifies info2.
BLIS_INLINE void bli_obj_set_scalar_dt( num_t dt, obj_t* obj )
{
	obj->info2 = ( objbits_t )
	             ( ( obj->info2 & ~BLIS_SCALAR_DT_BITS ) |
	               ( dt << BLIS_SCALAR_DT_SHIFT ) );
}

// NOTE: This function queries and modifies info2.
BLIS_INLINE void bli_obj_set_scalar_domain( dom_t dt, obj_t* obj )
{
	obj->info2 = ( objbits_t )
	             ( ( obj->info2 & ~BLIS_SCALAR_DOMAIN_BIT ) |
	               ( dt << BLIS_SCALAR_DT_SHIFT ) );
}

// NOTE: This function queries and modifies info2.
BLIS_INLINE void bli_obj_set_scalar_prec( prec_t dt, obj_t* obj )
{
	obj->info2 = ( objbits_t )
	             ( ( obj->info2 & ~BLIS_SCALAR_PREC_BIT ) |
	               ( dt << BLIS_SCALAR_DT_SHIFT ) );
}

BLIS_INLINE void bli_obj_set_pack_schema( pack_t schema, obj_t* obj )
{
	obj->info = ( objbits_t )
	            ( ( obj->info & ~BLIS_PACK_SCHEMA_BITS ) | schema );
}

BLIS_INLINE void bli_obj_set_pack_order_if_upper( packord_t ordif, obj_t* obj )
{
	obj->info = ( objbits_t )
	            ( ( obj->info & ~BLIS_PACK_REV_IF_UPPER_BIT ) | ordif );
}

BLIS_INLINE void bli_obj_set_pack_order_if_lower( packord_t ordif, obj_t* obj )
{
	obj->info = ( objbits_t )
	            ( ( obj->info & ~BLIS_PACK_REV_IF_LOWER_BIT ) | ordif );
}

// NOTE: The packbuf_t bitfield in the obj_t is currently unused. Instead,
// packbuf_t is stored/used from the context in order to support various
// induced methods. (Though ideally the packbuf_t field would only be
// present in the control tree).
BLIS_INLINE void bli_obj_set_pack_buffer_type( packbuf_t buf_type, obj_t* obj )
{
	obj->info = ( objbits_t )
	            ( ( obj->info & ~BLIS_PACK_BUFFER_BITS ) | buf_type );
}

BLIS_INLINE void bli_obj_set_struc( struc_t struc, obj_t* obj )
{
	obj->info = ( objbits_t )
	            ( ( obj->info & ~BLIS_STRUC_BITS ) | struc );
}

BLIS_INLINE void bli_obj_toggle_trans( obj_t* obj )
{
	bli_obj_apply_trans( BLIS_TRANSPOSE, obj );
}

BLIS_INLINE void bli_obj_toggle_conj( obj_t* obj )
{
	bli_obj_apply_conj( BLIS_CONJUGATE, obj );
}

BLIS_INLINE void bli_obj_toggle_uplo( obj_t* obj )
{
	obj->info = ( objbits_t )
	            ( obj->info ^ BLIS_LOWER_BIT ) ^ BLIS_UPPER_BIT;
}

// Root matrix query

BLIS_INLINE obj_t* bli_obj_root( const obj_t* obj )
{
	return ( obj_t* )( obj->root );
}

BLIS_INLINE bool bli_obj_root_is_general( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_is_general( bli_obj_root( obj ) ) );
}

BLIS_INLINE bool bli_obj_root_is_hermitian( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_is_hermitian( bli_obj_root( obj ) ) );
}

BLIS_INLINE bool bli_obj_root_is_symmetric( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_is_symmetric( bli_obj_root( obj ) ) );
}

BLIS_INLINE bool bli_obj_root_is_triangular( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_is_triangular( bli_obj_root( obj ) ) );
}

BLIS_INLINE bool bli_obj_root_is_herm_or_symm( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_is_hermitian( bli_obj_root( obj ) ) ||
	         bli_obj_is_symmetric( bli_obj_root( obj ) ) );
}

BLIS_INLINE bool bli_obj_root_is_upper( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_is_upper( bli_obj_root( obj ) ) );
}

BLIS_INLINE bool bli_obj_root_is_lower( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_is_lower( bli_obj_root( obj ) ) );
}

// Root matrix modification

BLIS_INLINE void bli_obj_set_as_root( obj_t* obj )
{
	obj->root = obj;
}

// Diagonal offset query

BLIS_INLINE doff_t bli_obj_diag_offset( const obj_t* obj )
{
	return ( doff_t )
	       ( obj->diag_off );
}

BLIS_INLINE doff_t bli_obj_diag_offset_after_trans( const obj_t* obj )
{
	return ( doff_t )
	       ( bli_obj_has_trans( obj ) ? -bli_obj_diag_offset( obj )
	                                  :  bli_obj_diag_offset( obj ) );
}

// Diagonal offset modification

BLIS_INLINE void bli_obj_set_diag_offset( doff_t offset, obj_t* obj )
{
	obj->diag_off = ( doff_t )offset;
}

BLIS_INLINE void bli_obj_negate_diag_offset( obj_t* obj )
{
	obj->diag_off = -(obj->diag_off);
}

BLIS_INLINE void bli_obj_inc_diag_offset( doff_t offset, obj_t* obj )
{
	obj->diag_off += ( doff_t )offset;
}

// Dimension query

BLIS_INLINE dim_t bli_obj_length( const obj_t* obj )
{
	return ( obj->dim[ BLIS_M ] );
}

BLIS_INLINE dim_t bli_obj_width( const obj_t* obj )
{
	return ( obj->dim[ BLIS_N ] );
}

BLIS_INLINE dim_t bli_obj_dim( mdim_t mdim, const obj_t* obj )
{
	return ( obj->dim[ mdim ] );
}

BLIS_INLINE dim_t bli_obj_min_dim( const obj_t* obj )
{
	return bli_min( bli_obj_length( obj ),
	                bli_obj_width( obj ) );
}

BLIS_INLINE dim_t bli_obj_max_dim( const obj_t* obj )
{
	return bli_max( bli_obj_length( obj ),
	                bli_obj_width( obj ) );
}

BLIS_INLINE dim_t bli_obj_length_after_trans( const obj_t* obj )
{
	return ( bli_obj_has_trans( obj ) ? bli_obj_width( obj )
	                                  : bli_obj_length( obj ) );
}

BLIS_INLINE dim_t bli_obj_width_after_trans( const obj_t* obj )
{
	return ( bli_obj_has_trans( obj ) ? bli_obj_length( obj )
	                                  : bli_obj_width( obj ) );
}

BLIS_INLINE bool bli_obj_is_1x1( const obj_t* x )
{
	return ( bool )
	       ( bli_obj_length( x ) == 1 &&
	         bli_obj_width( x ) == 1 );
}

// Stride/increment query

BLIS_INLINE inc_t bli_obj_row_stride( const obj_t* obj )
{
	return ( obj->rs );
}

BLIS_INLINE inc_t bli_obj_col_stride( const obj_t* obj )
{
	return ( obj->cs );
}

BLIS_INLINE inc_t bli_obj_imag_stride( const obj_t* obj )
{
	return ( obj->is );
}

BLIS_INLINE inc_t bli_obj_row_stride_mag( const obj_t* obj )
{
	return ( inc_t )
	       ( bli_abs( obj->rs ) );
}

BLIS_INLINE inc_t bli_obj_col_stride_mag( const obj_t* obj )
{
	return ( inc_t )
	       ( bli_abs( obj->cs ) );
}

BLIS_INLINE inc_t bli_obj_imag_stride_mag( const obj_t* obj )
{
	return ( inc_t )
	       ( bli_abs( obj->is ) );
}

// Note: The purpose of these functions is to obtain the length and width
// of the smallest submatrices of an object that could still encompass
// the stored data above (if obj is upper) or below (if obj is lower)
// the diagonal.
BLIS_INLINE dim_t bli_obj_length_stored( const obj_t* obj )
{
	return ( dim_t )
	       ( bli_obj_is_upper( obj )
	         ? bli_min( bli_obj_length( obj ),
	                    bli_obj_width( obj )  - bli_obj_diag_offset( obj ) )
	         : bli_min( bli_obj_length( obj ),
	                    bli_obj_length( obj ) + bli_obj_diag_offset( obj ) )
	       );
}

BLIS_INLINE dim_t bli_obj_width_stored( const obj_t* obj )
{
	return ( dim_t )
	       ( bli_obj_is_lower( obj )
	         ? bli_min( bli_obj_width( obj ),
	                    bli_obj_length( obj ) + bli_obj_diag_offset( obj ) )
	         : bli_min( bli_obj_width( obj ),
	                    bli_obj_width( obj )  - bli_obj_diag_offset( obj ) )
	       );
}

BLIS_INLINE dim_t bli_obj_length_stored_after_trans( const obj_t* obj )
{
	return ( bli_obj_has_trans( obj ) ? bli_obj_width_stored( obj )
	                                  : bli_obj_length_stored( obj ) );
}

BLIS_INLINE dim_t bli_obj_width_stored_after_trans( const obj_t* obj )
{
	return ( bli_obj_has_trans( obj ) ? bli_obj_length_stored( obj )
	                                  : bli_obj_width_stored( obj ) );
}

BLIS_INLINE dim_t bli_obj_vector_dim( const obj_t* x )
{
	return ( bli_obj_length( x ) == 1 ? bli_obj_width( x )
	                                  : bli_obj_length( x ) );
}

BLIS_INLINE inc_t bli_obj_vector_inc( const obj_t* x )
{
	return ( bli_obj_is_1x1( x ) ? 1 :
	         ( bli_obj_length( x ) == 1 ? bli_obj_col_stride( x )
	                                    : bli_obj_row_stride( x ) )
	       );
}

BLIS_INLINE bool bli_obj_is_vector( const obj_t* x )
{
	return ( bool )
	       ( bli_obj_length( x ) == 1 ||
	         bli_obj_width(  x ) == 1 );
}

BLIS_INLINE bool bli_obj_is_row_vector( const obj_t* x )
{
	return ( bool )
	       ( bli_obj_length( x ) == 1 );
}

BLIS_INLINE bool bli_obj_is_col_vector( const obj_t* x )
{
	return ( bool )
	       ( bli_obj_width( x ) == 1 );
}

BLIS_INLINE bool bli_obj_has_zero_dim( const obj_t* x )
{
	return ( bool )
	       ( bli_obj_length( x ) == 0 ||
	         bli_obj_width(  x ) == 0 );
}

// Dimension modification

BLIS_INLINE void bli_obj_set_length( dim_t m, obj_t* obj )
{
	obj->dim[ BLIS_M ] = m;
}

BLIS_INLINE void bli_obj_set_width( dim_t n, obj_t* obj )
{
	obj->dim[ BLIS_N ] = n;
}

BLIS_INLINE void bli_obj_set_dim( mdim_t mdim, dim_t dim_val, obj_t* obj )
{
	obj->dim[ mdim ] = dim_val;
}

BLIS_INLINE void bli_obj_set_dims( dim_t m, dim_t n, obj_t* obj )
{
	bli_obj_set_length( m, obj );
	bli_obj_set_width( n, obj );
}

BLIS_INLINE void bli_obj_set_dims_with_trans( trans_t trans, dim_t m, dim_t n, obj_t* obj )
{
	if ( bli_does_notrans( trans ) )
	{
		bli_obj_set_length( m, obj );
		bli_obj_set_width( n, obj );
	}
	else // if ( bli_does_trans( trans ) )
	{
		bli_obj_set_length( n, obj );
		bli_obj_set_width( m, obj );
	}
}

// Stride/increment predicates

//
// NOTE: The following two macros differ from their non-obj counterparts
// in that they do not identify m x 1 and 1 x n objects as row-stored and
// column-stored, respectively, which is needed when considering packed
// objects. But this is okay, since none of the invocations of these
// "obj" macros are used on packed matrices.
//

BLIS_INLINE bool bli_obj_is_row_stored( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_col_stride_mag( obj ) == 1 );
}

BLIS_INLINE bool bli_obj_is_col_stored( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_row_stride_mag( obj ) == 1 );
}

BLIS_INLINE bool bli_obj_is_gen_stored( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_row_stride_mag( obj ) != 1 &&
	         bli_obj_col_stride_mag( obj ) != 1 );
}

BLIS_INLINE bool bli_obj_is_row_tilted( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_col_stride_mag( obj ) < bli_obj_row_stride_mag( obj ) );
}

BLIS_INLINE bool bli_obj_is_col_tilted( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_row_stride_mag( obj ) < bli_obj_col_stride_mag( obj ) );
}

// Stride/increment modification

BLIS_INLINE void bli_obj_set_row_stride( inc_t rs, obj_t* obj )
{
	obj->rs = rs;
}

BLIS_INLINE void bli_obj_set_col_stride( inc_t cs, obj_t* obj )
{
	obj->cs = cs;
}

BLIS_INLINE void bli_obj_set_strides( inc_t rs, inc_t cs, obj_t* obj )
{
	bli_obj_set_row_stride( rs, obj );
	bli_obj_set_col_stride( cs, obj );
}

BLIS_INLINE void bli_obj_set_imag_stride( inc_t is, obj_t* obj )
{
	obj->is = is;
}

// Offset query

BLIS_INLINE dim_t bli_obj_row_off( const obj_t* obj )
{
	return ( obj->off[ BLIS_M ] );
}

BLIS_INLINE dim_t bli_obj_col_off( const obj_t* obj )
{
	return ( obj->off[ BLIS_N ] );
}

BLIS_INLINE dim_t bli_obj_off( mdim_t mdim, const obj_t* obj )
{
	return ( obj->off[ mdim ] );
}

// Offset modification

BLIS_INLINE void bli_obj_set_off( mdim_t mdim, dim_t offset, obj_t* obj )
{
	obj->off[ mdim ] = offset;
}

BLIS_INLINE void bli_obj_set_offs( dim_t offm, dim_t offn, obj_t* obj )
{
	bli_obj_set_off( BLIS_M, offm, obj );
	bli_obj_set_off( BLIS_N, offn, obj );
}

BLIS_INLINE void bli_obj_inc_off( mdim_t mdim, dim_t offset, obj_t* obj )
{
	obj->off[ mdim ] += offset;
}

BLIS_INLINE void bli_obj_inc_offs( dim_t offm, dim_t offn, obj_t* obj )
{
	bli_obj_inc_off( BLIS_M, offm, obj );
	bli_obj_inc_off( BLIS_N, offn, obj );
}

// Diagonal offset predicates

BLIS_INLINE bool bli_obj_is_strictly_above_diag( const obj_t* obj )
{
	return ( bool )
	       ( ( doff_t )bli_obj_length( obj ) <= -bli_obj_diag_offset( obj ) );
}

BLIS_INLINE bool bli_obj_is_strictly_below_diag( const obj_t* obj )
{
	return ( bool )
	       ( ( doff_t )bli_obj_width( obj ) <= bli_obj_diag_offset( obj ) );
}

BLIS_INLINE bool bli_obj_is_outside_diag( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_is_strictly_above_diag( obj ) ||
	         bli_obj_is_strictly_below_diag( obj ) );
}

BLIS_INLINE bool bli_obj_intersects_diag( const obj_t* obj )
{
	return ( bool )
	       ( !bli_obj_is_strictly_above_diag( obj ) &&
	         !bli_obj_is_strictly_below_diag( obj ) );
}

BLIS_INLINE bool bli_obj_is_unstored_subpart( const obj_t* obj )
{
	return ( bool )
	       ( ( bli_obj_root_is_lower( obj ) && bli_obj_is_strictly_above_diag( obj ) ) ||
	         ( bli_obj_root_is_upper( obj ) && bli_obj_is_strictly_below_diag( obj ) ) );
}

// Buffer address query

BLIS_INLINE void* bli_obj_buffer( const obj_t* obj )
{
	return ( void* )
	       ( obj->buffer );
}

// Buffer address modification

BLIS_INLINE void bli_obj_set_buffer( void* p, obj_t* obj )
{
	obj->buffer = p;
}

// Bufferless scalar field query

BLIS_INLINE void* bli_obj_internal_scalar_buffer( const obj_t* obj )
{
	return ( void* )
	       ( &( obj->scalar ) );
}

// Bufferless scalar field modification

BLIS_INLINE void bli_obj_copy_internal_scalar( const obj_t* a, obj_t* b )
{
	b->scalar = a->scalar;
}

// Element size query

BLIS_INLINE siz_t bli_obj_elem_size( const obj_t* obj )
{
	return ( siz_t )
	       ( obj->elem_size );
}

// Element size modification

BLIS_INLINE void bli_obj_set_elem_size( siz_t size, obj_t* obj )
{
	obj->elem_size = size;
}

// Packed matrix info query

BLIS_INLINE dim_t bli_obj_padded_length( const obj_t* obj )
{
	return ( obj->m_padded );
}

BLIS_INLINE dim_t bli_obj_padded_width( const obj_t* obj )
{
	return ( obj->n_padded );
}

// Packed matrix info modification

BLIS_INLINE void bli_obj_set_padded_length( dim_t m, obj_t* obj )
{
	obj->m_padded = m;
}

BLIS_INLINE void bli_obj_set_padded_width( dim_t n, obj_t* obj )
{
	obj->n_padded = n;
}

BLIS_INLINE void bli_obj_set_padded_dims( dim_t m, dim_t n, obj_t* obj )
{
	bli_obj_set_padded_length( m, obj );
	bli_obj_set_padded_width( n, obj );
}

// Packed panel info query

BLIS_INLINE dim_t bli_obj_panel_length( const obj_t* obj )
{
	return ( obj->m_panel );
}

BLIS_INLINE dim_t bli_obj_panel_width( const obj_t* obj )
{
	return ( obj->n_panel );
}

BLIS_INLINE inc_t bli_obj_panel_dim( const obj_t* obj )
{
	return ( obj->pd );
}

BLIS_INLINE inc_t bli_obj_panel_stride( const obj_t* obj )
{
	return ( obj->ps );
}

// Packed panel info modification

BLIS_INLINE void bli_obj_set_panel_length( dim_t m, obj_t* obj )
{
	obj->m_panel = m;
}

BLIS_INLINE void bli_obj_set_panel_width( dim_t n, obj_t* obj )
{
	obj->n_panel = n;
}

BLIS_INLINE void bli_obj_set_panel_dims( dim_t m, dim_t n, obj_t* obj )
{
	bli_obj_set_panel_length( m, obj );
	bli_obj_set_panel_width( n, obj );
}

BLIS_INLINE void bli_obj_set_panel_dim( inc_t pd, obj_t* obj )
{
	obj->pd = pd;
}

BLIS_INLINE void bli_obj_set_panel_stride( inc_t ps, obj_t* obj )
{
	obj->ps = ps;
}

// stor3_t-related

BLIS_INLINE stor3_t bli_obj_stor3_from_strides( const obj_t* c, const obj_t* a, const obj_t* b )
{
	const inc_t rs_c = bli_obj_row_stride( c );
	const inc_t cs_c = bli_obj_col_stride( c );

	inc_t rs_a, cs_a;
	inc_t rs_b, cs_b;

	if ( bli_obj_has_notrans( a ) )
	{
		rs_a = bli_obj_row_stride( a );
		cs_a = bli_obj_col_stride( a );
	}
	else
	{
		rs_a = bli_obj_col_stride( a );
		cs_a = bli_obj_row_stride( a );
	}

	if ( bli_obj_has_notrans( b ) )
	{
		rs_b = bli_obj_row_stride( b );
		cs_b = bli_obj_col_stride( b );
	}
	else
	{
		rs_b = bli_obj_col_stride( b );
		cs_b = bli_obj_row_stride( b );
	}

	return bli_stor3_from_strides( rs_c, cs_c,
	                               rs_a, cs_a,
	                               rs_b, cs_b  );
}


// -- Initialization-related macros --

// Finish the initialization started by the matrix-specific static initializer
// (e.g. BLIS_OBJECT_INITIALIZER)
// NOTE: This is intended only for use in the BLAS compatibility API and typed
// BLIS API.

BLIS_INLINE void bli_obj_init_finish( num_t dt, dim_t m, dim_t n, void* p, inc_t rs, inc_t cs, obj_t* obj )
{
	bli_obj_set_as_root( obj );

	bli_obj_set_dt( dt, obj );
	bli_obj_set_comp_prec( bli_dt_prec( dt ), obj );

	bli_obj_set_dims( m, n, obj );
	bli_obj_set_strides( rs, cs, obj );

	siz_t elem_size = sizeof( float );
	if ( bli_dt_prec_is_double( dt ) ) elem_size *= 2;
	if ( bli_dt_dom_is_complex( dt ) ) elem_size *= 2;
	bli_obj_set_elem_size( elem_size, obj );

	bli_obj_set_buffer( p, obj );

	bli_obj_set_scalar_dt( dt, obj );
	void* s = bli_obj_internal_scalar_buffer( obj );

	if      ( bli_dt_prec_is_single( dt ) ) { bli_creal( *( scomplex* )s ) = 1.0F;
	                                          bli_cimag( *( scomplex* )s ) = 0.0F; }
	else if ( bli_dt_prec_is_double( dt ) ) { bli_zreal( *( dcomplex* )s ) = 1.0;
	                                          bli_zimag( *( dcomplex* )s ) = 0.0; }
}

// Finish the initialization started by the 1x1-specific static initializer
// (e.g. BLIS_OBJECT_INITIALIZER_1X1)
// NOTE: This is intended only for use in the BLAS compatibility API and typed
// BLIS API.

BLIS_INLINE void bli_obj_init_finish_1x1( num_t dt, void* p, obj_t* obj )
{
	bli_obj_set_as_root( obj );

	bli_obj_set_dt( dt, obj );

	bli_obj_set_buffer( p, obj );
}

// -- Miscellaneous object macros --

// Toggle the region referenced (or "stored").

BLIS_INLINE void bli_obj_toggle_region_ref( obj_t* obj )
{
	if      ( bli_obj_is_upper( obj ) ) bli_obj_inc_diag_offset( -1, obj );
	else if ( bli_obj_is_lower( obj ) ) bli_obj_inc_diag_offset(  1, obj );

	bli_obj_toggle_uplo( obj );
}

BLIS_INLINE void bli_obj_toggle_uplo_if_trans( trans_t trans, obj_t* obj )
{
	if ( bli_does_trans( trans ) &&
	     bli_obj_is_upper_or_lower( obj ) )
	{
		bli_obj_toggle_uplo( obj );
		bli_obj_negate_diag_offset( obj );
	}
}

// Initialize object with default properties (info field).

BLIS_INLINE void bli_obj_set_defaults( obj_t* obj )
{
	obj->info = 0x0;
	obj->info = obj->info | BLIS_BITVAL_DENSE | BLIS_BITVAL_GENERAL;
}

// Acquire buffer at object's submatrix offset (offset-aware buffer query).

BLIS_INLINE void* bli_obj_buffer_at_off( const obj_t* obj )
{
	return ( void* )
	       (
	         ( ( char* )( bli_obj_buffer   ( obj ) ) +
	           ( dim_t )( bli_obj_elem_size( obj ) ) *
	                      ( bli_obj_col_off( obj ) * bli_obj_col_stride( obj ) +
	                        bli_obj_row_off( obj ) * bli_obj_row_stride( obj )
	                      )
	         )
	       );
}

// Acquire buffer from BLIS_CONSTANT object.

BLIS_INLINE const void* bli_obj_buffer_for_const( num_t dt, const obj_t* obj )
{
	void* p;

	if      ( dt == BLIS_FLOAT    ) p = &((( constdata_t* )bli_obj_buffer( obj ))->s);
	else if ( dt == BLIS_DOUBLE   ) p = &((( constdata_t* )bli_obj_buffer( obj ))->d);
	else if ( dt == BLIS_SCOMPLEX ) p = &((( constdata_t* )bli_obj_buffer( obj ))->c);
	else if ( dt == BLIS_DCOMPLEX ) p = &((( constdata_t* )bli_obj_buffer( obj ))->z);
	else                            p = &((( constdata_t* )bli_obj_buffer( obj ))->i);

	return p;
}

// Acquire buffer from scalar (1x1) object, including BLIS_CONSTANT objects.

BLIS_INLINE void* bli_obj_buffer_for_1x1( num_t dt, const obj_t* obj )
{
	return ( void* )
	       ( bli_obj_is_const( obj ) ? bli_obj_buffer_for_const( dt, obj )
	                                 : bli_obj_buffer_at_off( obj )
	       );
}

// Adjust the pointer based on current offsets, zero the offsets, and then
// set the current object as the root. For obj_t's with at least one non-zero
// offset, this effectively makes the obj_t "forget" that it was ever a view
// into a larger matrix.

BLIS_INLINE void bli_obj_reset_origin( obj_t* obj )
{
	bli_obj_set_buffer( bli_obj_buffer_at_off( obj ), obj );
	bli_obj_set_offs( 0, 0, obj );
	bli_obj_set_as_root( obj );
}

// Make a full alias (shallow copy).

BLIS_INLINE void bli_obj_alias_to( const obj_t* a, obj_t* b )
{
	bli_obj_init_full_shallow_copy_of( a, b );
}

// Check if two objects are aliases of one another.

BLIS_INLINE bool bli_obj_is_alias_of( const obj_t* a, const obj_t* b )
{
	return ( bool )
	       ( bli_obj_buffer( a ) == bli_obj_buffer( b ) );
}


// Create an alias with a trans value applied.
// (Note: trans may include a conj component.)

BLIS_INLINE void bli_obj_alias_with_trans( trans_t trans, const obj_t* a, obj_t* b )
{
	bli_obj_alias_to( a, b );
	bli_obj_apply_trans( trans, b );
}

// Create an alias with a conj value applied.

BLIS_INLINE void bli_obj_alias_with_conj( conj_t conja, const obj_t* a, obj_t* b )
{
	bli_obj_alias_to( a, b );
	bli_obj_apply_conj( conja, b );
}

// Alias only the real part.

BLIS_INLINE void bli_obj_real_part( const obj_t* c, obj_t* r )
{
	bli_obj_alias_to( c, r );

	if ( bli_obj_is_complex( c ) )
	{
		// Change the datatypes.
		bli_obj_set_dt( bli_obj_dt_proj_to_real( c ), r );

		// Don't touch the attached scalar datatype.

		// Update the element size.
		siz_t es_c = bli_obj_elem_size( c );
		bli_obj_set_elem_size( es_c/2, r );

		// Update the strides.
		inc_t rs_c = bli_obj_row_stride( c );
		inc_t cs_c = bli_obj_col_stride( c );
		bli_obj_set_strides( 2*rs_c, 2*cs_c, r );

		// Buffer is left unchanged.
	}
}

// Alias only the imaginary part.

BLIS_INLINE void bli_obj_imag_part( const obj_t* c, obj_t* i )
{
	if ( bli_obj_is_complex( c ) )
	{
		bli_obj_alias_to( c, i );

		// Change the datatype.
		bli_obj_set_dt( bli_obj_dt_proj_to_real( c ), i );

		// Don't touch the attached scalar datatype.

		// Update the element size.
		siz_t es_c = bli_obj_elem_size( c );
		bli_obj_set_elem_size( es_c/2, i );

		// Update the strides.
		inc_t rs_c = bli_obj_row_stride( c );
		inc_t cs_c = bli_obj_col_stride( c );
		bli_obj_set_strides( 2*rs_c, 2*cs_c, i );

		// Update the buffer.
		inc_t is_c = bli_obj_imag_stride( c );
		char* p    = ( char* )bli_obj_buffer( c );
		bli_obj_set_buffer( p + is_c * es_c/2, i );
	}
}

// Given a 1x1 object, acquire an address to the buffer depending on whether
// the object is a BLIS_CONSTANT, and also set a datatype associated with the
// chosen buffer (possibly using an auxiliary datatype if the object is
// BLIS_CONSTANT).

BLIS_INLINE void bli_obj_scalar_set_dt_buffer( const obj_t* obj, num_t dt_aux, num_t* dt, void** buf )
{
	if ( bli_obj_is_const( obj ) )
	{
		*dt  = dt_aux;
		*buf = bli_obj_buffer_for_1x1( dt_aux, obj );
	}
	else
	{
		*dt  = bli_obj_dt( obj );
		*buf = bli_obj_buffer_at_off( obj );
	}
}

// Swap all object fields (metadata/properties).

BLIS_INLINE void bli_obj_swap( obj_t* a, obj_t* b )
{
	bool a_root_is_self = ( bli_obj_root( a ) == a );
	bool b_root_is_self = ( bli_obj_root( b ) == b );

	obj_t t = *b; *b = *a; *a = t;

	if ( a_root_is_self ) bli_obj_set_as_root( b );
	if ( b_root_is_self ) bli_obj_set_as_root( a );
}

// Induce a transposition on an object: swap dimensions, increments, and
// offsets, then clear the trans bit.

BLIS_INLINE void bli_obj_induce_trans( obj_t* obj )
{
	// Induce transposition among basic fields.
	dim_t  m        = bli_obj_length( obj );
	dim_t  n        = bli_obj_width( obj );
	inc_t  rs       = bli_obj_row_stride( obj );
	inc_t  cs       = bli_obj_col_stride( obj );
	dim_t  offm     = bli_obj_row_off( obj );
	dim_t  offn     = bli_obj_col_off( obj );
	doff_t diag_off = bli_obj_diag_offset( obj );

	bli_obj_set_dims( n, m, obj );
	bli_obj_set_strides( cs, rs, obj );
	bli_obj_set_offs( offn, offm, obj );
	bli_obj_set_diag_offset( -diag_off, obj );

	if ( bli_obj_is_upper_or_lower( obj ) )
		bli_obj_toggle_uplo( obj );

	// Induce transposition among packed fields.
	dim_t  m_padded = bli_obj_padded_length( obj );
	dim_t  n_padded = bli_obj_padded_width( obj );
	dim_t  m_panel  = bli_obj_panel_length( obj );
	dim_t  n_panel  = bli_obj_panel_width( obj );

	bli_obj_set_padded_dims( n_padded, m_padded, obj );
	bli_obj_set_panel_dims( n_panel, m_panel, obj );

	// Note that this macro DOES NOT touch the transposition bit! If
	// the calling code is using this function to handle an object whose
	// transposition bit is set prior to computation, that code needs
	// to manually clear or toggle the bit, via
	// bli_obj_set_onlytrans() or bli_obj_toggle_trans(),
	// respectively.
}

BLIS_INLINE void bli_obj_induce_fast_trans( obj_t* obj )
{
	// NOTE: This function is only used in situations where the matrices
	// are guaranteed to not have structure or be packed.

	// Induce transposition among basic fields.
	dim_t  m        = bli_obj_length( obj );
	dim_t  n        = bli_obj_width( obj );
	inc_t  rs       = bli_obj_row_stride( obj );
	inc_t  cs       = bli_obj_col_stride( obj );
	dim_t  offm     = bli_obj_row_off( obj );
	dim_t  offn     = bli_obj_col_off( obj );

	bli_obj_set_dims( n, m, obj );
	bli_obj_set_strides( cs, rs, obj );
	bli_obj_set_offs( offn, offm, obj );

	// Note that this macro DOES NOT touch the transposition bit! If
	// the calling code is using this function to handle an object whose
	// transposition bit is set prior to computation, that code needs
	// to manually clear or toggle the bit, via
	// bli_obj_set_onlytrans() or bli_obj_toggle_trans(),
	// respectively.
}

// Sometimes we need to "reflect" a partition because the data we want is
// actually stored on the other side of the diagonal. The nuts and bolts of
// this macro look a lot like an induced transposition, except that the row
// and column strides are left unchanged (which, of course, drastically
// changes the effect of the macro).

BLIS_INLINE void bli_obj_reflect_about_diag( obj_t* obj )
{
	dim_t  m        = bli_obj_length( obj );
	dim_t  n        = bli_obj_width( obj );
	dim_t  offm     = bli_obj_row_off( obj );
	dim_t  offn     = bli_obj_col_off( obj );
	doff_t diag_off = bli_obj_diag_offset( obj );

	bli_obj_set_dims( n, m, obj );
	bli_obj_set_offs( offn, offm, obj );
	bli_obj_set_diag_offset( -diag_off, obj );

	bli_obj_toggle_trans( obj );
}

// Create an alias which refers to only a portion of the original matrix
// without any "historical baggage", stripping out all offsets, transposes,
// and references to the original root object.

BLIS_INLINE void bli_obj_alias_submatrix( const obj_t* a, obj_t* b )
{
	bli_obj_alias_to( a, b );
    bli_obj_reset_origin( b );
    if ( bli_obj_has_trans( b ) )
    {
        bli_obj_induce_trans( b );
        bli_obj_set_onlytrans( BLIS_NO_TRANSPOSE, b );
    }
}

#endif
// end bli_obj_macro_defs.h
#line 106 "../blis-2.0/frame/include//bli_macro_defs.h"

// begin bli_scalar_macro_defs.h
#line 1 "../blis-2.0/frame/include//bli_scalar_macro_defs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2024, Southern Methodist University

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SCALAR_MACRO_DEFS_H
#define BLIS_SCALAR_MACRO_DEFS_H


// begin bli_assigns.h
#line 1 "../blis-2.0/frame/include/level0//bli_assigns.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2024, Southern Methodist University

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_ASSIGNS_H
#define BLIS_ASSIGNS_H

// assigns

// Notes:
// - The first char encodes the domain of output y.
// - These macros are used to avoid updates on an output's imaginary part
//   when that output exists only in the real domain (i.e. has no imaginary
//   part to begin with).

#define bli_rassigns( xr, xi, yr, yi ) { yr = xr;          (void)(xi); (void)(yi); }
#define bli_cassigns( xr, xi, yr, yi ) { yr = xr; yi = xi;                         }
#define bli_jassigns( xr, xi, yr, yi ) {          yi = xi; (void)(xr); (void)(yr); }


#endif

// end bli_assigns.h
#line 41 "../blis-2.0/frame/include//bli_scalar_macro_defs.h"

// begin bli_complex_terms.h
#line 1 "../blis-2.0/frame/include/level0//bli_complex_terms.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2024, Southern Methodist University

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_COMPLEX_TERMS_H
#define BLIS_COMPLEX_TERMS_H


// -- Complex term-zeroing macros ----------------------------------------------

// Note:
// - pab is the precision of the A_[ri] * B_[ri] product. It is only used in
//   certain cases where we need to decide which precision of zero to substitute
//   into the calling expression.

// ar * br term
#define bli_rrtermrr( pab, ab )  ab
#define bli_rctermrr( pab, ab )  ab
#define bli_crtermrr( pab, ab )  ab
#define bli_cctermrr( pab, ab )  ab

// ai * bi term
#define bli_rrtermii( pab, ab )  ( (void)(ab), PASTEMAC(pab,zero) )
#define bli_rctermii( pab, ab )  ( (void)(ab), PASTEMAC(pab,zero) )
#define bli_crtermii( pab, ab )  ( (void)(ab), PASTEMAC(pab,zero) )
#define bli_cctermii( pab, ab )  ab

// ai * br term
#define bli_rrtermir( pab, ab )  ( (void)(ab), PASTEMAC(pab,zero) )
#define bli_rctermir( pab, ab )  ( (void)(ab), PASTEMAC(pab,zero) )
#define bli_crtermir( pab, ab )  ab
#define bli_cctermir( pab, ab )  ab

// ar * bi term
#define bli_rrtermri( pab, ab )  ( (void)(ab), PASTEMAC(pab,zero) )
#define bli_rctermri( pab, ab )  ab
#define bli_crtermri( pab, ab )  ( (void)(ab), PASTEMAC(pab,zero) )
#define bli_cctermri( pab, ab )  ab



#endif
// end bli_complex_terms.h
#line 42 "../blis-2.0/frame/include//bli_scalar_macro_defs.h"

// begin bli_constants.h
#line 1 "../blis-2.0/frame/include/level0//bli_constants.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2024, Southern Methodist University

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_CONSTANTS_H
#define BLIS_CONSTANTS_H

// return pointers to constants

// 1

#define bli_s1 \
\
	( ( float*    ) bli_obj_buffer_for_const( BLIS_FLOAT,    &BLIS_ONE ) )

#define bli_d1 \
\
	( ( double*   ) bli_obj_buffer_for_const( BLIS_DOUBLE,   &BLIS_ONE ) )

#define bli_c1 \
\
	( ( scomplex* ) bli_obj_buffer_for_const( BLIS_SCOMPLEX, &BLIS_ONE ) )

#define bli_z1 \
\
	( ( dcomplex* ) bli_obj_buffer_for_const( BLIS_DCOMPLEX, &BLIS_ONE ) )

#define bli_i1 \
\
	( ( gint_t*   ) bli_obj_buffer_for_const( BLIS_INT,      &BLIS_ONE ) )

// 0

#define bli_s0 \
\
	( ( float*    ) bli_obj_buffer_for_const( BLIS_FLOAT,    &BLIS_ZERO ) )

#define bli_d0 \
\
	( ( double*   ) bli_obj_buffer_for_const( BLIS_DOUBLE,   &BLIS_ZERO ) )

#define bli_c0 \
\
	( ( scomplex* ) bli_obj_buffer_for_const( BLIS_SCOMPLEX, &BLIS_ZERO ) )

#define bli_z0 \
\
	( ( dcomplex* ) bli_obj_buffer_for_const( BLIS_DCOMPLEX, &BLIS_ZERO ) )

#define bli_i0 \
\
	( ( gint_t*   ) bli_obj_buffer_for_const( BLIS_INT,      &BLIS_ZERO ) )

// -1

#define bli_sm1 \
\
	( ( float*    ) bli_obj_buffer_for_const( BLIS_FLOAT,    &BLIS_MINUS_ONE ) )

#define bli_dm1 \
\
	( ( double*   ) bli_obj_buffer_for_const( BLIS_DOUBLE,   &BLIS_MINUS_ONE ) )

#define bli_cm1 \
\
	( ( scomplex* ) bli_obj_buffer_for_const( BLIS_SCOMPLEX, &BLIS_MINUS_ONE ) )

#define bli_zm1 \
\
	( ( dcomplex* ) bli_obj_buffer_for_const( BLIS_DCOMPLEX, &BLIS_MINUS_ONE ) )

#define bli_im1 \
\
	( ( gint_t*   ) bli_obj_buffer_for_const( BLIS_INT,      &BLIS_MINUS_ONE ) )


#endif

// end bli_constants.h
#line 43 "../blis-2.0/frame/include//bli_scalar_macro_defs.h"

// begin bli_declinits.h
#line 1 "../blis-2.0/frame/include/level0//bli_declinits.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2024, Southern Methodist University

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_DECLINITS_H
#define BLIS_DECLINITS_H

// declinits

// Notes:
// - The first char encodes the domain of output yr + yi.
// - The pxy precision character encodes the precision of x AND y (they
//   are assumed to be the same).
// - These macros are used to declare AND initialize variables corresponding
//   to the real and imaginary parts of (presumably) temporary variables.
//   If the domain is real, only the real part is declared and initialized.

#define bli_rdeclinits( pxy, xr, xi, yr, yi ) PASTEMAC(pxy,ctype) yr = xr; (void)yr; \
                                              PASTEMAC(pxy,ctype) yi;      (void)yi;
#define bli_cdeclinits( pxy, xr, xi, yr, yi ) PASTEMAC(pxy,ctype) yr = xr; (void)yr; \
                                              PASTEMAC(pxy,ctype) yi = xi; (void)yi;

// An extra definition for situations where we only need a real value declared
// and initialized (e.g. when explicitly implementing in the complex domain).
#define bli_rodeclinits( pxy, xr, yr ) PASTEMAC(pxy,ctype) yr = xr; (void)yr;


#endif

// end bli_declinits.h
#line 44 "../blis-2.0/frame/include//bli_scalar_macro_defs.h"


// -- Assignment/Accessor macros --

// NOTE: This macro is defined first since some of the other scalar macros
// use it to abstract away the method used to assign complex values (ie:
// whether fields of a struct are set directly or whether native C99
// assignment is used).

// begin bli_tsets.h
#line 1 "../blis-2.0/frame/include/level0//bli_tsets.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2024, Southern Methodist University

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_TSETS_H
#define BLIS_TSETS_H

// -- Implementation macros ----------------------------------------------------

#define bli_tsetims( \
          dx, px, xr, xi, \
          dy, py, yr, yi \
        ) \
{ \
	PASTEMAC(dy,assigns) \
	( \
	  PASTEMAC(px,py,tcast)(xr), \
	  PASTEMAC(px,py,tcast)(xi), \
	  yr, \
	  yi \
	); \
}

#define bli_tsetrims( \
              px, xr, \
          dy, py, yr, yi \
        ) \
{ \
	PASTEMAC(dy,assigns) \
	( \
	  PASTEMAC(px,py,tcast)(xr), \
	  yi, \
	  yr, \
	  yi \
	); \
}

#define bli_tsetiims( \
              px,     xi, \
          dy, py, yr, yi \
        ) \
{ \
	PASTEMAC(dy,assigns) \
	( \
	  yr, \
	  PASTEMAC(px,py,tcast)(xi), \
	  yr, \
	  yi \
	); \
}

// -- API macros ---------------------------------------------------------------

// -- Consolidated --

// tsets
#define bli_tsets( chx,chy, xr, xi, y ) \
        bli_tsetims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
                     xr, \
                     xi, \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
          PASTEMAC(chy,real)(y), \
          PASTEMAC(chy,imag)(y) \
        )

// tsetrs
#define bli_tsetrs( chx,chy, xr, y ) \
        bli_tsetrims \
        ( \
          PASTEMAC(chx,prec), \
                    xr, \
          PASTEMAC(chy,dom), \
          PASTEMAC(chy,prec), \
          PASTEMAC(chy,real)(y), \
          PASTEMAC(chy,imag)(y) \
        )

// tsetis
#define bli_tsetis( chx,chy, xi, y ) \
        bli_tsetiims \
        ( \
          PASTEMAC(chx,prec), \
                    xi, \
          PASTEMAC(chy,dom), \
          PASTEMAC(chy,prec), \
          PASTEMAC(chy,real)(y), \
          PASTEMAC(chy,imag)(y) \
        )

// -- Exposed real/imaginary --

// tsetris
#define bli_tsetris( chx,chy, xr, xi, yr, yi ) \
        bli_tsetims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
                     xr, \
                     xi, \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
                     yr, \
                     yi \
        )

// -- Set to constant --

// tset0s
#define bli_tset0s( chy, y ) \
    bli_tsets \
    ( \
      chy,chy, \
      PASTEMAC \
      ( \
        PASTEMAC(chy,prec), \
        zero \
      ), \
      PASTEMAC \
      ( \
        PASTEMAC(chy,prec), \
        zero \
      ), \
      y \
    )

// tset1s
#define bli_tset1s( chy, y ) \
    bli_tsets \
    ( \
      chy,chy, \
      PASTEMAC \
      ( \
       PASTEMAC(chy,prec), \
       one \
      ), \
      PASTEMAC \
      ( \
        PASTEMAC(chy,prec), \
        zero \
      ), \
      y \
    )

// tsetr0s
#define bli_tsetr0s( chy, y ) \
    bli_tsetrs( chy,chy, PASTEMAC(PASTEMAC(chy,prec),zero), y )

// tseti0s
#define bli_tseti0s( chy, y ) \
    bli_tsetis( chy,chy, PASTEMAC(PASTEMAC(chy,prec),zero), y )

// tset0ris
#define bli_tset0ris( chy, yr, yi ) \
    bli_tsetris \
    ( \
      chy,chy, \
      PASTEMAC \
      ( \
       PASTEMAC(chy,prec), \
       zero \
      ), \
      PASTEMAC \
      ( \
        PASTEMAC(chy,prec), \
        zero \
      ), \
      yr, \
      yi \
    )

// -- Micro-tile --

// set0s_mxn
#define bli_tset0s_mxn( chy, m, n, y, rs_y, cs_y ) \
{ \
	for ( dim_t _j = 0; _j < (n); ++_j ) \
	for ( dim_t _i = 0; _i < (m); ++_i ) \
	bli_tset0s( chy, *((y) + _i*(rs_y) + _j*(cs_y)) ); \
}

// set0bbs_mxn
#define bli_tset0bbs_mxn( chy, m, n, y, incy, ldy ) \
{ \
	/* Assume that the duplication factor is the row stride of y. */ \
	const dim_t _d    = incy; \
	const dim_t _ds_y = 1; \
\
	for ( dim_t _j = 0; _j < (n); ++_j ) \
	{ \
		PASTEMAC(chy,ctype)* restrict _yj = (PASTEMAC(chy,ctype)*)(y) + _j*(ldy); \
\
		for ( dim_t _i = 0; _i < (m); ++_i ) \
		{ \
			PASTEMAC(chy,ctype)* restrict _yij = _yj + _i*(incy); \
\
			for ( dim_t _p = 0; _p < _d; ++_p ) \
			{ \
				PASTEMAC(chy,ctype)* restrict _yijd = _yij + _p*_ds_y; \
\
				bli_tset0s( chy, *_yijd ); \
			} \
		} \
	} \
}

// bcastbbs_mxn
#define bli_tbcastbbs_mxn( chy, m, n, y, incy, ldy ) \
{ \
	/* Assume that the duplication factor is the row stride of y. */ \
	const dim_t _d = incy; \
\
	for ( dim_t _j = 0; _j < (n); ++_j ) \
	{ \
		PASTEMAC(chy,ctype)* _yj = (PASTEMAC(chy,ctype)*)(y) + _j*(ldy); \
\
		for ( dim_t _i = 0; _i < (m); ++_i ) \
		{ \
			PASTEMAC(chy,ctype)* _yij = _yj + _i*(incy); \
			PASTEMAC(chy,ctyper) _yij_r, _yij_i; \
\
			bli_tgets( chy,chy, *_yij, _yij_r, _yij_i ); \
\
			for ( dim_t _p = 0; _p < _d; ++_p ) \
			{ \
				PASTEMAC(chy,ctyper)* _yijd_r = (PASTEMAC(chy,ctyper)*)_yij      + _p; \
				PASTEMAC(chy,ctyper)* _yijd_i = (PASTEMAC(chy,ctyper)*)_yij + _d + _p; \
\
				bli_tcopyris( chy,chy, _yij_r, _yij_i, *_yijd_r, *_yijd_i ); \
			} \
		} \
	} \
}

// bcastbbs_mxn
#define bli_tcompressbbs_mxn( chy, m, n, y, incy, ldy ) \
{ \
	/* Assume that the duplication factor is the row stride of y. */ \
	const dim_t _d = incy; \
\
	for ( dim_t _j = 0; _j < (n); ++_j ) \
	{ \
		PASTEMAC(chy,ctype)* _yj = (PASTEMAC(chy,ctype)*)(y) + _j*(ldy); \
\
		for ( dim_t _i = 0; _i < (m); ++_i ) \
		{ \
			PASTEMAC(chy,ctype)* _yij = _yj + _i*(incy); \
			PASTEMAC(chy,ctyper)* _yij_r = (PASTEMAC(chy,ctyper)*)_yij; \
			PASTEMAC(chy,ctyper)* _yij_i = (PASTEMAC(chy,ctyper)*)_yij + _d; \
\
			bli_tsets( chy,chy, *_yij_r, *_yij_i, *_yij ); \
		} \
	} \
}

#define bli_tset0s_edge( chp, i, m, j, n, p, ldp ) \
{ \
	if ( (i) < (m) ) \
	{ \
		bli_tset0s_mxn \
		( \
		  chp, \
		  (m) - (i), \
		  j, \
		  (p) + (i)*1, 1, ldp \
		); \
	} \
\
	if ( (j) < (n) ) \
	{ \
		bli_tset0s_mxn \
		( \
		  chp, \
		  m, \
		  (n) - (j), \
		  (p) + (j)*(ldp), 1, ldp \
		); \
	} \
}

#endif

// -- Legacy macros ------------------------------------------------------------

#define bli_sset0s( x ) bli_tset0s( s, x )
#define bli_dset0s( x ) bli_tset0s( d, x )
#define bli_cset0s( x ) bli_tset0s( c, x )
#define bli_zset0s( x ) bli_tset0s( z, x )

#define bli_sset0s_edge( i, m, j, n, p, ldp ) bli_tset0s_edge( s, i, m, j, n, (float   *)(p), ldp )
#define bli_dset0s_edge( i, m, j, n, p, ldp ) bli_tset0s_edge( d, i, m, j, n, (double  *)(p), ldp )
#define bli_cset0s_edge( i, m, j, n, p, ldp ) bli_tset0s_edge( c, i, m, j, n, (scomplex*)(p), ldp )
#define bli_zset0s_edge( i, m, j, n, p, ldp ) bli_tset0s_edge( z, i, m, j, n, (dcomplex*)(p), ldp )

// -- Notes --------------------------------------------------------------------
// end bli_tsets.h
#line 53 "../blis-2.0/frame/include//bli_scalar_macro_defs.h"


// NOTE: This macro also needs to be defined early on since it determines
// how real and imaginary components are accessed (ie: whether the fields
// of a struct are read directly or whether native C99 functions are used.)

// begin bli_tgets.h
#line 1 "../blis-2.0/frame/include/level0//bli_tgets.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2024, Southern Methodist University

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_TGETS_H
#define BLIS_TGETS_H

// -- Implementation macro -----------------------------------------------------

// (yr) := (xr)
// (yi) := (xi)

// NOTE: always assign the imaginary component, even for real y

#define bli_tgetims( \
          \
          dx, px, xr, xi, \
          dy, py, yr, yi \
        ) \
{ \
	PASTEMAC(c,assigns) \
	( \
	  PASTEMAC(px,py,tcast)(xr), \
	  PASTEMAC(px,py,tcast)(xi), \
	  yr, \
	  yi \
	); \
}

// -- API macros ---------------------------------------------------------------

// -- Hybrid --

// tgets
#define bli_tgets( chx, chy, x, yr, yi ) \
        bli_tgetims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(chx,real)(x), \
          PASTEMAC(chx,imag)(x), \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
                     yr, \
                     yi \
        )

// -- Higher-level static functions --------------------------------------------

// -- Notes --------------------------------------------------------------------

// -- Domain cases --

//   r       r
// (yr) := (xr);
// (yi) :=   0 ;

//   r       c
// (yr) := (xr);
// (yi) := (xi);

//   c       r
// (yr) := (xr);
// (yi) :=   0 ;

//   c       c
// (yr) := (xr);
// (yi) := (xi);

#endif

// end bli_tgets.h
#line 59 "../blis-2.0/frame/include//bli_scalar_macro_defs.h"


// -- Scalar macros --

// begin bli_tabsq2s.h
#line 1 "../blis-2.0/frame/include/level0//bli_tabsq2s.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2024, Southern Methodist University

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_TABSQ2S_H
#define BLIS_TABSQ2S_H

// -- Implementation macro -----------------------------------------------------

// (yr) = (xr) * (xr) + (xi) * (xi);
// (yi) = 0;

#define bli_tabsq2ims( \
          \
          dx, px, xr, xi, \
          dy, py, yr, yi, \
          chc  \
        ) \
{ \
	PASTEMAC(dy,assigns) \
	( \
	  PASTEMAC(chc,py,tcast)( \
	    PASTEMAC(chc,add)( \
	      PASTEMAC(dx,dx,termrr)( \
	        chc, \
	        PASTEMAC(chc,mul)( \
	          PASTEMAC(px,chc,tcast)(xr), \
	          PASTEMAC(px,chc,tcast)(xr) \
	        )  \
	      ), \
	      PASTEMAC(dx,dx,termii)( \
	        chc, \
	        PASTEMAC(chc,mul)( \
	          PASTEMAC(px,chc,tcast)(xi), \
	          PASTEMAC(px,chc,tcast)(xi) \
	        ) \
	      ) \
	    ) \
	  ),\
	  PASTEMAC(py,zero), \
	  yr, \
	  yi \
	); \
}

// -- API macros ---------------------------------------------------------------

// -- Consolidated --

// tabsq2s
#define bli_tabsq2s( chx, chy, chc, x, y ) \
        bli_tabsq2ims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(chx,real)(x), \
          PASTEMAC(chx,imag)(x), \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
          PASTEMAC(chy,real)(y), \
          PASTEMAC(chy,imag)(y), \
          PASTEMAC(chc,prec)  \
        )

// -- Exposed real/imaginary --

// tabsq2ris
#define bli_tabsq2ris( chx, chy, chc, xr, xi, yr, yi ) \
        bli_tabsq2ims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
                     xr, \
                     xi, \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
                     yr, \
                     yi, \
          PASTEMAC(chc,prec)  \
        )

// -- Higher-level static functions --------------------------------------------

// -- Notes --------------------------------------------------------------------

// -- Domain cases --

//   r       r
// (yr) := (xr) * (xr);
// (yi) xx   0 ;

//   r       c
// (yr) := (xr) * (xr) + (xi) * (xi);
// (yi) xx   0 ;

//   c       r
// (yr) := (xr) * (xr);
// (yi) :=   0 ;

//   c       c
// (yr) := (xr) * (xr) + (xi) * (xi);
// (yi) :=   0 ;

#endif

// end bli_tabsq2s.h
#line 63 "../blis-2.0/frame/include//bli_scalar_macro_defs.h"

// begin bli_tabval2s.h
#line 1 "../blis-2.0/frame/include/level0//bli_tabval2s.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2024, Southern Methodist University

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_TABVAL2S_H
#define BLIS_TABVAL2S_H

// -- Implementation macro -----------------------------------------------------

#define bli_tabval2ims( \
          \
          dx, px, xr, xi, \
          dy, py, yr, yi, \
          chc  \
        ) \
{ \
	PASTEMAC(dx,abval2ims) \
	( \
	  dx, px, xr, xi, \
	  dy, py, yr, yi, \
	  chc  \
	); \
}

// -- real-to-real domain implementation --
// -- real-to-complex domain implementation --

// (yr) = abs( xr );
// (yi) = 0;

#define bli_rabval2ims( \
          \
          dx, px, xr, xi, \
          dy, py, yr, yi, \
          chc  \
        ) \
{ \
	PASTEMAC(dy,assigns) \
	( \
	  PASTEMAC(chc,py,tcast)( \
	    PASTEMAC(chc,abs)( \
	      PASTEMAC(px,chc,tcast)(xr) \
	    ) \
	  ), \
	  PASTEMAC(py,zero), \
	  yr, \
	  yi \
	) \
} \

// -- complex-to-real domain implementation --
// -- complex-to-complex domain implementation --

// NOTE: Instead of defining abval2 in terms of bli_?hypot(), we use an
// alternate definition that can avoid overflow in the final result due
// to overflow in the intermediate results (e.g. xr * xr and xi * xi).

// xmaxr = maxabs( xr, xi );
// if ( s == 0.0 ) mag = 0.0;
// else            mag = sqrt( xmaxr ) *
//                       sqrt( ( xr / xmaxr ) * xr +
//                             ( xi / xmaxr ) * xi );
// yr = mag;
// yi = 0.0;

#define bli_cabval2ims( \
          \
          dx, px, xr, xi, \
          dy, py, yr, yi, \
          chc  \
        ) \
{ \
	PASTEMAC(ro,declinits) \
	( \
	  px, \
	  PASTEMAC(px,maxabs)(xr,xi), \
	  xmaxr  \
	) \
	PASTEMAC(dy,assigns) \
	( \
	  ( PASTEMAC(teq0s)(px,xmaxr) && \
	    !PASTEMAC(px,isnan)(xi) && \
	    !PASTEMAC(px,isnan)(xr) \
	    ? PASTEMAC(py,zero) \
	    : PASTEMAC(chc,py,tcast)( \
	        PASTEMAC(chc,mul)( \
	          PASTEMAC(chc,sqrt)( \
	            PASTEMAC(px,chc,tcast)(xmaxr) \
	          ), \
	          PASTEMAC(chc,sqrt)( \
	            PASTEMAC(chc,add)( \
	              PASTEMAC(chc,mul)( \
	                PASTEMAC(px,chc,tcast)(xr), \
	                PASTEMAC(chc,div)( \
	                  PASTEMAC(px,chc,tcast)(xr), \
	                  PASTEMAC(px,chc,tcast)(xmaxr) \
	                ) \
	              ), \
	              PASTEMAC(chc,mul)( \
	                PASTEMAC(px,chc,tcast)(xi), \
	                PASTEMAC(chc,div)( \
	                  PASTEMAC(px,chc,tcast)(xi), \
	                  PASTEMAC(px,chc,tcast)(xmaxr) \
                    ) \
	              ) \
	            ) \
	          ) \
	        ) \
	      ) \
	  ), \
	  PASTEMAC(py,zero), \
	  yr, \
	  yi \
	) \
}

// -- API macros ---------------------------------------------------------------

// -- Consolidated --

// tabval2s
#define bli_tabval2s( chx, chy, chc, x, y ) \
        bli_tabval2ims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(chx,real)(x), \
          PASTEMAC(chx,imag)(x), \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
          PASTEMAC(chy,real)(y), \
          PASTEMAC(chy,imag)(y), \
          PASTEMAC(chc,prec)  \
        )

// -- Exposed real/imaginary --

// tabval2ris
#define bli_tabval2ris( chx, chy, chc, xr, xi, yr, yi ) \
        bli_tabval2ims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
                     xr, \
                     xi, \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
                     yr, \
                     yi, \
          PASTEMAC(chc,prec)  \
        )

// -- Higher-level static functions --------------------------------------------

// -- Notes --------------------------------------------------------------------

// -- Domain cases --

//   r       r
// (yr) := abs(xr);
// (yi) xx   0 ;

//   r       c
// (yr) := sqrt(s) * sqrt( ( xr / s ) * xr + ( xi / s ) * xi );
// (yi) xx   0 ;

//   c       r
// (yr) := abs(xr);
// (yi) :=   0 ;

//   c       c
// (yr) := sqrt(s) * sqrt( ( xr / s ) * xr + ( xi / s ) * xi );
// (yi) :=   0 ;

#endif

// end bli_tabval2s.h
#line 64 "../blis-2.0/frame/include//bli_scalar_macro_defs.h"

// begin bli_tadd3s.h
#line 1 "../blis-2.0/frame/include/level0//bli_tadd3s.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2024, Southern Methodist University

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_TADD3S_H
#define BLIS_TADD3S_H

// -- Implementation macro -----------------------------------------------------

// (zr) = (yr) + (xr);
// (zi) = (yi) + (xi);

#define bli_tadd3ims( \
          \
          dx, px, xr, xi, \
          dy, py, yr, yi, \
          dz, pz, zr, zi, \
          chc  \
        ) \
{ \
	PASTEMAC(dz,assigns) \
	( \
	  PASTEMAC(chc,pz,tcast)( \
	    PASTEMAC(chc,add)( \
	      PASTEMAC(py,chc,tcast)(yr), \
	      PASTEMAC(px,chc,tcast)(xr)  \
	    ) \
	  ),\
	  PASTEMAC(chc,pz,tcast)( \
	    PASTEMAC(chc,add)( \
	      PASTEMAC(py,chc,tcast)(yi), \
	      PASTEMAC(px,chc,tcast)(xi) \
	    ) \
	  ), \
	  zr, \
	  zi \
	); \
}

// -- API macros ---------------------------------------------------------------

// -- Consolidated --

// tadd3s
#define bli_tadd3s( chx, chy, chz, chc, x, y, z ) \
        bli_tadd3ims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(chx,real)(x), \
          PASTEMAC(chx,imag)(x), \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
          PASTEMAC(chy,real)(y), \
          PASTEMAC(chy,imag)(y), \
          PASTEMAC(chz,dom),  \
          PASTEMAC(chz,prec), \
          PASTEMAC(chz,real)(z), \
          PASTEMAC(chz,imag)(z), \
          PASTEMAC(chc,prec)  \
        )

#undef GENTFUNC
#define GENTFUNC( ctypex, chx, ctypey, chy, ctypez, chz, ctypec, chc, opname ) \
UNIT_TEST(chx,chy,chz,chc,opname) \
( \
	for ( auto x : test_values<ctypex>() ) \
	for ( auto y : test_values<ctypey>() ) \
	{ \
		auto z0 = convert<ctypez>( convert_prec<ctypec>( x ) + \
		                           convert_prec<ctypec>( y ) ); \
\
		INFO( "x: " << x ); \
		INFO( "y: " << y ); \
\
		ctypez z; \
		bli_tadds( chx,chy,chz,chc, x, y, z ); \
\
		INFO( "z (C++):  " << z0 ); \
		INFO( "z (BLIS): " << z ); \
\
		check<ctypec>( z, z0 ); \
	} \
)

// tadd3js
#define bli_tadd3js( chx, chy, chz, chc, x, y, z ) \
        bli_tadd3ims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(chx,real)(x), \
          PASTEMAC(PASTEMAC(chx,prec),neg)( \
            PASTEMAC(chx,imag)(x)  \
          ), \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
          PASTEMAC(chy,real)(y), \
          PASTEMAC(chy,imag)(y), \
          PASTEMAC(chz,dom),  \
          PASTEMAC(chz,prec), \
          PASTEMAC(chz,real)(z), \
          PASTEMAC(chz,imag)(z), \
          PASTEMAC(chc,prec)  \
        )

// -- Exposed real/imaginary --

// tadd3ris
#define bli_tadd3ris( chx, chy, chz, chc, xr, xi, yr, yi, zr, zi ) \
        bli_tadd3ims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
                     xr, \
                     xi, \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
                     yr, \
                     yi, \
          PASTEMAC(chz,dom),  \
          PASTEMAC(chz,prec), \
                     zr, \
                     zi, \
          PASTEMAC(chc,prec)  \
        )

// tadd3jris
#define bli_tadd3jris( chx, chy, chz, chc, xr, xi, yr, yi, zr, zi ) \
        bli_tadd3ims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
                     xr, \
          PASTEMAC(PASTEMAC(chx,prec),neg)( \
                     xi ), \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
                     yr, \
                     yi, \
          PASTEMAC(chz,dom),  \
          PASTEMAC(chz,prec), \
                     zr, \
                     zi, \
          PASTEMAC(chc,prec)  \
        )

// -- Notes --------------------------------------------------------------------

// -- Domain cases --

//   r       r
// (yr) += (xr);
// (yi) xx   0 ;

//   r       c
// (yr) += (xr);
// (yi) xx (xi);

//   c       r
// (yr) += (xr);
// (yi) +=   0 ;

//   c       c
// (yr) += (xr);
// (yi) += (xi);

#endif

// end bli_tadd3s.h
#line 65 "../blis-2.0/frame/include//bli_scalar_macro_defs.h"

// begin bli_tadds.h
#line 1 "../blis-2.0/frame/include/level0//bli_tadds.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2024, Southern Methodist University

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_TADDS_H
#define BLIS_TADDS_H

// -- Implementation macro -----------------------------------------------------

// (yr) = (yr) + (xr);
// (yi) = (yi) + (xi);

#define bli_taddims( \
          \
          dx, px, xr, xi, \
          dy, py, yr, yi, \
          chc  \
        ) \
{ \
	PASTEMAC(dy,assigns) \
	( \
	  PASTEMAC(chc,py,tcast)( \
	    PASTEMAC(chc,add)( \
	      PASTEMAC(py,chc,tcast)(yr), \
	      PASTEMAC(px,chc,tcast)(xr)  \
	    ) \
	  ),\
	  PASTEMAC(chc,py,tcast)( \
	    PASTEMAC(chc,add)( \
	      PASTEMAC(py,chc,tcast)(yi), \
	      PASTEMAC(px,chc,tcast)(xi) \
	    ) \
	  ), \
	  yr, \
	  yi \
	); \
}

// -- API macros ---------------------------------------------------------------

// -- Consolidated --

// tadds
#define bli_tadds( chx, chy, chc, x, y ) \
        bli_taddims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(chx,real)(x), \
          PASTEMAC(chx,imag)(x), \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
          PASTEMAC(chy,real)(y), \
          PASTEMAC(chy,imag)(y), \
          PASTEMAC(chc,prec)  \
        )

// taddjs
#define bli_taddjs( chx, chy, chc, x, y ) \
        bli_taddims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(chx,real)(x), \
          PASTEMAC(PASTEMAC(chx,prec),neg)( \
            PASTEMAC(chx,imag)(x)  \
          ), \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
          PASTEMAC(chy,real)(y), \
          PASTEMAC(chy,imag)(y), \
          PASTEMAC(chc,prec)  \
        )

// -- Exposed real/imaginary --

// taddris
#define bli_taddris( chx, chy, chc, xr, xi, yr, yi ) \
        bli_taddims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
                     xr, \
                     xi, \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
                     yr, \
                     yi, \
          PASTEMAC(chc,prec)  \
        )

// taddjris
#define bli_taddjris( chx, chy, chc, xr, xi, yr, yi ) \
        bli_taddims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
                     xr, \
          PASTEMAC(PASTEMAC(chx,prec),neg)( \
                     xi ), \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
                     yr, \
                     yi, \
          PASTEMAC(chc,prec)  \
        )

// -- Higher-level static functions --------------------------------------------

// -- mxn --

// tadds_mxn
#define bli_tadds_mxn( chx, chy, chc, m, n, x, rs_x, cs_x, y, rs_y, cs_y ) \
{ \
	for ( dim_t jj = 0; jj < (n); ++jj ) \
	for ( dim_t ii = 0; ii < (m); ++ii ) \
	{ \
		PASTEMAC(chx,ctype)* restrict xij = ( PASTEMAC(chx,ctype)* )(x) + ii*(rs_x) + jj*(cs_x); \
		PASTEMAC(chy,ctype)* restrict yij = ( PASTEMAC(chy,ctype)* )(y) + ii*(rs_y) + jj*(cs_y); \
\
		bli_tadds( chx,chy,chc, *xij, *yij ); \
	} \
}

// -- Notes --------------------------------------------------------------------

// -- Domain cases --

//   r       r
// (yr) += (xr);
// (yi) xx   0 ;

//   r       c
// (yr) += (xr);
// (yi) xx (xi);

//   c       r
// (yr) += (xr);
// (yi) +=   0 ;

//   c       c
// (yr) += (xr);
// (yi) += (xi);

#endif

// end bli_tadds.h
#line 66 "../blis-2.0/frame/include//bli_scalar_macro_defs.h"

// begin bli_taxpbys.h
#line 1 "../blis-2.0/frame/include/level0//bli_taxpbys.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2024, Southern Methodist University

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_TAXPBYS_H
#define BLIS_TAXPBYS_H

// -- Implementation macro -----------------------------------------------------

// (yorigr) := (yr)
// (yorigi) := (yi)
// (yr) := (ar) * (xr) - (ai) * (xi) + (br) * (yorigr) - (bi) * (yorigi);
// (yi) := (ai) * (xr) + (ar) * (xi) + (bi) * (yorigr) + (br) * (yorigi);

#define bli_taxpbyims( \
          \
          da, pa, ar, ai, \
          dx, px, xr, xi, \
          db, pb, br, bi, \
          dy, py, yr, yi, \
          chc  \
        ) \
{ \
	PASTEMAC(c,declinits) \
	( \
	  py, \
	  PASTEMAC(chc,py,tcast)( \
	    PASTEMAC(chc,add)( \
	      PASTEMAC(chc,sub)( \
	        PASTEMAC(da,dx,termrr)( \
	          chc, \
	          PASTEMAC(chc,mul)( \
	            PASTEMAC(pa,chc,tcast)(ar), \
	            PASTEMAC(px,chc,tcast)(xr) \
	          )  \
	        ), \
	        PASTEMAC(da,dx,termii)( \
	          chc, \
	          PASTEMAC(chc,mul)( \
	            PASTEMAC(pa,chc,tcast)(ai), \
	            PASTEMAC(px,chc,tcast)(xi) \
	          )  \
	        )  \
	      ), \
	      PASTEMAC(chc,sub)( \
	        PASTEMAC(db,dy,termrr)( \
	          chc, \
	          PASTEMAC(chc,mul)( \
	            PASTEMAC(pb,chc,tcast)(br), \
	            PASTEMAC(py,chc,tcast)(yr) \
	          )  \
	        ), \
	        PASTEMAC(db,dy,termii)( \
	          chc, \
	          PASTEMAC(chc,mul)( \
	            PASTEMAC(pb,chc,tcast)(bi), \
	            PASTEMAC(py,chc,tcast)(yi) \
	          ) \
	        ) \
	      ) \
	    ) \
	  ),\
	  PASTEMAC(chc,py,tcast)( \
	    PASTEMAC(chc,add)( \
	      PASTEMAC(chc,add)( \
	        PASTEMAC(da,dx,termir)( \
	          chc, \
	          PASTEMAC(chc,mul)( \
	            PASTEMAC(pa,chc,tcast)(ai), \
	            PASTEMAC(px,chc,tcast)(xr) \
	          )  \
	        ), \
	        PASTEMAC(da,dx,termri)( \
	          chc, \
	          PASTEMAC(chc,mul)( \
	            PASTEMAC(pa,chc,tcast)(ar), \
	            PASTEMAC(px,chc,tcast)(xi) \
	          ) \
	        ) \
	      ), \
	      PASTEMAC(chc,add)( \
	        PASTEMAC(db,dy,termir)( \
	          chc, \
	          PASTEMAC(chc,mul)( \
	            PASTEMAC(pb,chc,tcast)(bi), \
	            PASTEMAC(py,chc,tcast)(yr) \
	          )  \
	        ), \
	        PASTEMAC(db,dy,termri)( \
	          chc, \
	          PASTEMAC(chc,mul)( \
	            PASTEMAC(pb,chc,tcast)(br), \
	            PASTEMAC(py,chc,tcast)(yi) \
	          ) \
	        ) \
	      ) \
	    ) \
	  ), \
	  tr, \
	  ti \
	); \
	PASTEMAC(dy,assigns) \
	( \
	  tr, \
	  ti, \
	  yr, \
	  yi \
	); \
}

// -- API macros ---------------------------------------------------------------

// -- Consolidated --

// taxpbys
#define bli_taxpbys( cha, chx, chb, chy, chc, a, x, b, y ) \
        bli_taxpbyims \
        ( \
          PASTEMAC(cha,dom),  \
          PASTEMAC(cha,prec), \
          PASTEMAC(cha,real)(a), \
          PASTEMAC(cha,imag)(a), \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(chx,real)(x), \
          PASTEMAC(chx,imag)(x), \
          PASTEMAC(chb,dom),  \
          PASTEMAC(chb,prec), \
          PASTEMAC(chb,real)(b), \
          PASTEMAC(chb,imag)(b), \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
          PASTEMAC(chy,real)(y), \
          PASTEMAC(chy,imag)(y), \
          PASTEMAC(chc,prec)  \
        )

// taxpbyjs
#define bli_taxpbyjs( cha, chx, chb, chy, chc, a, x, b, y ) \
        bli_taxpbyims \
        ( \
          PASTEMAC(cha,dom),  \
          PASTEMAC(cha,prec), \
          PASTEMAC(cha,real)(a), \
          PASTEMAC(cha,imag)(a), \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(chx,real)(x), \
          PASTEMAC(PASTEMAC(chx,prec),neg)( \
            PASTEMAC(chx,imag)(x)  \
          ), \
          PASTEMAC(chb,dom),  \
          PASTEMAC(chb,prec), \
          PASTEMAC(chb,real)(b), \
          PASTEMAC(chb,imag)(b), \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
          PASTEMAC(chy,real)(y), \
          PASTEMAC(chy,imag)(y), \
          PASTEMAC(chc,prec)  \
        )

// -- Exposed real/imaginary --

// taxpbyris
#define bli_taxpbyris( cha, chx, chb, chy, chc, ar, ai, xr, xi, br, bi, yr, yi ) \
        bli_taxpbyims \
        ( \
          PASTEMAC(cha,dom),  \
          PASTEMAC(cha,prec), \
                     ar, \
                     ai, \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
                     xr, \
                     xi, \
          PASTEMAC(chb,dom),  \
          PASTEMAC(chb,prec), \
                     br, \
                     bi, \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
                     yr, \
                     yi, \
          PASTEMAC(chc,prec)  \
        )

// taxpbyjris
#define bli_taxpbyjris( cha, chx, chb, chy, chc, ar, ai, xr, xi, br, bi, yr, yi ) \
        bli_taxpbyims \
        ( \
          PASTEMAC(cha,dom),  \
          PASTEMAC(cha,prec), \
                     ar, \
                     ai, \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
                     xr, \
          PASTEMAC(PASTEMAC(chx,prec),neg)( \
                     xi ), \
          PASTEMAC(chb,dom),  \
          PASTEMAC(chb,prec), \
                     br, \
                     bi, \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
                     yr, \
                     yi, \
          PASTEMAC(chc,prec)  \
        )

// -- Higher-level static functions --------------------------------------------

// -- mxn --

// axpbys_mxn
#define bli_taxpbys_mxn( cha, chx, chb, chy, chc, m, n, alpha, x, rs_x, cs_x, beta, y, rs_y, cs_y ) \
{ \
\
	/* If beta is zero, overwrite y with x (in case y has infs or NaNs). */ \
	if ( bli_teq0s( chb, *(beta) ) ) \
	{ \
		bli_tscal2s_mxn( cha, chx, chy, chc, BLIS_NO_CONJUGATE, m, n, alpha, x, rs_x, cs_x, y, rs_y, cs_y ); \
	} \
	else \
	{ \
		for ( dim_t jj = 0; jj < n; ++jj ) \
		for ( dim_t ii = 0; ii < m; ++ii ) \
		{ \
			PASTEMAC(chx,ctype)* restrict xij = ( PASTEMAC(chx,ctype)* )(x) + ii*(rs_x) + jj*(cs_x); \
			PASTEMAC(chy,ctype)* restrict yij = ( PASTEMAC(chy,ctype)* )(y) + ii*(rs_y) + jj*(cs_y); \
\
			bli_taxpbys( cha,chx,chb,chy,chc, *(alpha), *xij, *(beta), *yij ); \
		} \
	} \
}

// -- Legacy macros ------------------------------------------------------------

#define bli_saxpbys( a, x, b, y ) bli_taxpbys( s,s,s,s,s, a, x, b, y )
#define bli_daxpbys( a, x, b, y ) bli_taxpbys( d,d,d,d,d, a, x, b, y )
#define bli_caxpbys( a, x, b, y ) bli_taxpbys( c,c,c,c,s, a, x, b, y )
#define bli_zaxpbys( a, x, b, y ) bli_taxpbys( z,z,z,z,d, a, x, b, y )

// -- Notes --------------------------------------------------------------------

#endif

// end bli_taxpbys.h
#line 67 "../blis-2.0/frame/include//bli_scalar_macro_defs.h"

// begin bli_taxpys.h
#line 1 "../blis-2.0/frame/include/level0//bli_taxpys.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2024, Southern Methodist University

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_TAXPYS_H
#define BLIS_TAXPYS_H

// -- Implementation macro -----------------------------------------------------

// (tr) += (ar) * (xr) - (ai) * (xi);
// (ti) += (ai) * (xr) + (ar) * (xi);
// (yr) += (tr);
// (yi) += (ti);

#define bli_taxpyims( \
          \
          da, pa, ar, ai, \
          dx, px, xr, xi, \
          dy, py, yr, yi, \
          chc  \
        ) \
{ \
	PASTEMAC(c,declinits) \
	( \
	  py, \
	  PASTEMAC(chc,py,tcast)( \
	    PASTEMAC(chc,add)( \
	      PASTEMAC(py,chc,tcast)(yr), \
	      PASTEMAC(chc,sub)( \
	        PASTEMAC(da,dx,termrr)( \
	          chc, \
	          PASTEMAC(chc,mul)( \
	            PASTEMAC(pa,chc,tcast)(ar), \
	            PASTEMAC(px,chc,tcast)(xr) \
	          )  \
	        ), \
	        PASTEMAC(da,dx,termii)( \
	          chc, \
	          PASTEMAC(chc,mul)( \
	            PASTEMAC(pa,chc,tcast)(ai), \
	            PASTEMAC(px,chc,tcast)(xi) \
	          ) \
	        ) \
	      ) \
	    ) \
	  ),\
	  PASTEMAC(chc,py,tcast)( \
	    PASTEMAC(chc,add)( \
	      PASTEMAC(py,chc,tcast)(yi), \
	      PASTEMAC(chc,add)( \
	        PASTEMAC(da,dx,termir)( \
	          chc, \
	          PASTEMAC(chc,mul)( \
	            PASTEMAC(pa,chc,tcast)(ai), \
	            PASTEMAC(px,chc,tcast)(xr) \
	          )  \
	        ), \
	        PASTEMAC(da,dx,termri)( \
	          chc, \
	          PASTEMAC(chc,mul)( \
	            PASTEMAC(pa,chc,tcast)(ar), \
	            PASTEMAC(px,chc,tcast)(xi) \
	          ) \
	        ) \
	      ) \
	    ) \
	  ), \
	  tr, \
	  ti \
	); \
	PASTEMAC(dy,assigns) \
	( \
	  tr, \
	  ti, \
	  yr, \
	  yi \
	); \
}

// -- API macros ---------------------------------------------------------------

// -- Consolidated --

// taxpys
#define bli_taxpys( cha, chx, chy, chc, a, x, y ) \
        bli_taxpyims \
        ( \
          PASTEMAC(cha,dom),  \
          PASTEMAC(cha,prec), \
          PASTEMAC(cha,real)(a), \
          PASTEMAC(cha,imag)(a), \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(chx,real)(x), \
          PASTEMAC(chx,imag)(x), \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
          PASTEMAC(chy,real)(y), \
          PASTEMAC(chy,imag)(y), \
          PASTEMAC(chc,prec)  \
        )

// taxpyjs
#define bli_taxpyjs( cha, chx, chy, chc, a, x, y ) \
        bli_taxpyims \
        ( \
          PASTEMAC(cha,dom),  \
          PASTEMAC(cha,prec), \
          PASTEMAC(cha,real)(a), \
          PASTEMAC(cha,imag)(a), \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(chx,real)(x), \
          PASTEMAC(PASTEMAC(chx,prec),neg)( \
            PASTEMAC(chx,imag)(x)  \
          ), \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
          PASTEMAC(chy,real)(y), \
          PASTEMAC(chy,imag)(y), \
          PASTEMAC(chc,prec)  \
        )

// -- Exposed real/imaginary --

// taxpyris
#define bli_taxpyris( cha, chx, chy, chc, ar, ai, xr, xi, yr, yi ) \
        bli_taxpyims \
        ( \
          PASTEMAC(cha,dom),  \
          PASTEMAC(cha,prec), \
                     ar, \
                     ai, \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
                     xr, \
                     xi, \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
                     yr, \
                     yi, \
          PASTEMAC(chc,prec)  \
        )

// taxpyjris
#define bli_taxpyjris( cha, chx, chy, chc, ar, ai, xr, xi, yr, yi ) \
        bli_taxpyims \
        ( \
          PASTEMAC(cha,dom),  \
          PASTEMAC(cha,prec), \
                     ar, \
                     ai, \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
                     xr, \
          PASTEMAC(PASTEMAC(chx,prec),neg)( \
                     xi ), \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
                     yr, \
                     yi, \
          PASTEMAC(chc,prec)  \
        )

// -- Higher-level static functions --------------------------------------------

// -- Legacy macros ------------------------------------------------------------

#define bli_saxpys( a, x, y ) bli_taxpys( s,s,s,s, a, x, y )
#define bli_daxpys( a, x, y ) bli_taxpys( d,d,d,d, a, x, y )
#define bli_caxpys( a, x, y ) bli_taxpys( c,c,c,s, a, x, y )
#define bli_zaxpys( a, x, y ) bli_taxpys( z,z,z,d, a, x, y )

// -- Notes --------------------------------------------------------------------

// -- Domain cases --

//   r       r      r
// (yr) += (ar) * (xr) -   0  *   0 ;
// (yi) xx   0  * (xr) + (ar) *   0 ;

//   r       r      c
// (yr) += (ar) * (xr) -   0  * (xi);
// (yi) xx   0  * (xr) + (ar) * (xi);

//   r       c      r
// (yr) += (ar) * (xr) - (ai) *   0 ;
// (yi) xx (ai) * (xr) + (ar) *   0 ;

//   r       c      c
// (yr) += (ar) * (xr) - (ai) * (xi);
// (yi) xx (ai) * (xr) + (ar) * (xi);

//   c       r      r
// (yr) += (ar) * (xr) -   0  *   0 ;
// (yi) +=   0  * (xr) + (ar) *   0 ;

//   c       r      c
// (yr) += (ar) * (xr) -   0  * (xi);
// (yi) +=   0  * (xr) + (ar) * (xi);

//   c       c      r
// (yr) += (ar) * (xr) - (ai) *   0 ;
// (yi) += (ai) * (xr) + (ar) *   0 ;

//   c       c      c
// (yr) += (ar) * (xr) - (ai) * (xi);
// (yi) += (ai) * (xr) + (ar) * (xi);

#endif

// end bli_taxpys.h
#line 68 "../blis-2.0/frame/include//bli_scalar_macro_defs.h"

// begin bli_tconjs.h
#line 1 "../blis-2.0/frame/include/level0//bli_tconjs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2024, Southern Methodist University

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_TCONJS_H
#define BLIS_TCONJS_H

// -- Implementation macro -----------------------------------------------------

// (xr) :=  (xr)
// (xi) := -(xi)

#define bli_tconjims( \
          \
          dx, px, xr, xi \
        ) \
{ \
	PASTEMAC(dx,assigns) \
	( \
	                   xr, \
	  PASTEMAC(px,neg)(xi),\
	                   xr, \
	                   xi \
	); \
}

// -- API macros ---------------------------------------------------------------

// -- Consolidated --

// tconjs
#define bli_tconjs( chx, x ) \
        bli_tconjims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(chx,real)(x), \
          PASTEMAC(chx,imag)(x) \
        )

// -- Exposed real/imaginary --

// tconjris
#define bli_tconjris( chx, xr, xi ) \
        bli_tconjims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
                     xr, \
                     xi \
        )

// -- Higher-level static functions --------------------------------------------

// -- Notes --------------------------------------------------------------------

#endif

// end bli_tconjs.h
#line 69 "../blis-2.0/frame/include//bli_scalar_macro_defs.h"

// begin bli_tcopycjs.h
#line 1 "../blis-2.0/frame/include/level0//bli_tcopycjs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2024, Southern Methodist University

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_TCOPYCJS_H
#define BLIS_TCOPYCJS_H

// -- Implementation macro -----------------------------------------------------

// (yr) := (xr);
// (yi) := ( is_conj( conj ) ? -(xi) : (xi) );

#define bli_tcopycjims( \
          \
          conj, \
          dx, px, xr, xi, \
          dy, py, yr, yi \
        ) \
{ \
	PASTEMAC(dy,assigns) \
	( \
	  PASTEMAC(px,py,tcast)(xr), \
	  ( bli_is_conj( conj ) ? PASTEMAC(px,neg)( \
	                            PASTEMAC(px,py,tcast)(xi) \
	                          ) \
	                        :   PASTEMAC(px,py,tcast)(xi) \
	  ), \
	  yr, \
	  yi \
	); \
}

// -- API macros ---------------------------------------------------------------

// -- Consolidated --

// tcopycjs
#define bli_tcopycjs( chx, chy, conj, x, y ) \
        bli_tcopycjims \
        ( \
          conj, \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(chx,real)(x), \
          PASTEMAC(chx,imag)(x), \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
          PASTEMAC(chy,real)(y), \
          PASTEMAC(chy,imag)(y) \
        )

// -- Exposed real/imaginary --

// tcopyris
#define bli_tcopycjris( chx, chy, conj, xr, xi, yr, yi ) \
        bli_tcopycjims \
        ( \
          conj, \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
                     xr, \
                     xi, \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
                     yr, \
                     yi \
        )

// -- Higher-level static functions --------------------------------------------

// -- Legacy macros ------------------------------------------------------------

#define bli_scopycjs( conj, x, y ) bli_tcopycjs( s,s, conj, x, y )
#define bli_dcopycjs( conj, x, y ) bli_tcopycjs( d,d, conj, x, y )
#define bli_ccopycjs( conj, x, y ) bli_tcopycjs( c,c, conj, x, y )
#define bli_zcopycjs( conj, x, y ) bli_tcopycjs( z,z, conj, x, y )

// -- Notes --------------------------------------------------------------------

// -- Domain cases --

//   r       r
// (yr) := (xr);
// (yi) xx   0 ;

//   r       c
// (yr) := (xr);
// (yi) xx (xi);

//   c       r
// (yr) := (xr);
// (yi) :=   0 ;

//   c       c
// (yr) := (xr);
// (yi) := (xi);

#endif

// end bli_tcopycjs.h
#line 70 "../blis-2.0/frame/include//bli_scalar_macro_defs.h"

// begin bli_tcopynzs.h
#line 1 "../blis-2.0/frame/include/level0//bli_tcopynzs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2024, Southern Methodist University

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_TCOPYNZS_H
#define BLIS_TCOPYNZS_H

// -- Implementation macro -----------------------------------------------------

// (yr) := (xr);
// if ( is_complex( x ) )
//     (yi) := (xi);

#define bli_tcopynzims( \
          \
          dx, px, xr, xi, \
          dy, py, yr, yi \
        ) \
{ \
	PASTEMAC(dx,dy,copynzims) \
	( \
	  dx, px, xr, xi, \
	  dy, py, yr, yi \
	); \
}

// -- real-to-real domain implementation --

#define bli_rrcopynzims( \
          \
          dx, px, xr, xi, \
          dy, py, yr, yi \
        ) \
{ \
	PASTEMAC(r,assigns) \
	( \
	  PASTEMAC(px,py,tcast)(xr), \
	  PASTEMAC(px,py,tcast)(xi), \
	  yr, \
	  yi \
	); \
}

// -- complex-to-real domain implementation --
// -- real-to-complex domain implementation --

// NOTE: Normally, the real-to-complex case would take place in the complex
// domain (in that an implicit zero would be copied to y.imag), but since
// this is copynz, we avoid updating the imaginary parts of complex y when
// x is real. Thus, real-to-complex ends up getting implemented the same as
// real-to-real (and complex-to-real).

#define bli_rccopynzims bli_rrcopynzims
#define bli_crcopynzims bli_rrcopynzims

// -- complex-to-complex domain implementation --

#define bli_cccopynzims( \
          \
          dx, px, xr, xi, \
          dy, py, yr, yi \
        ) \
{ \
	PASTEMAC(c,assigns) \
	( \
	  PASTEMAC(px,py,tcast)(xr), \
	  PASTEMAC(px,py,tcast)(xi), \
	  yr, \
	  yi \
	); \
}

// -- API macros ---------------------------------------------------------------

// -- Consolidated --

// tcopynzs
#define bli_tcopynzs( chx, chy, x, y ) \
        bli_tcopynzims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(chx,real)(x), \
          PASTEMAC(chx,imag)(x), \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
          PASTEMAC(chy,real)(y), \
          PASTEMAC(chy,imag)(y) \
        )

// tcopyjnzs
#define bli_tcopyjnzs( chx, chy, x, y ) \
        bli_tcopynzims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(chx,real)(x), \
          PASTEMAC(PASTEMAC(chx,prec),neg)( \
            PASTEMAC(chx,imag)(x)  \
          ), \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
          PASTEMAC(chy,real)(y), \
          PASTEMAC(chy,imag)(y) \
        )

// -- Exposed real/imaginary --

// tcopynzris
#define bli_tcopynzris( chx, chy, xr, xi, yr, yi ) \
        bli_tcopynzims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
                     xr, \
                     xi, \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
                     yr, \
                     yi \
        )

// tcopyjnzris
#define bli_tcopyjnzris( chx, chy, xr, xi, yr, yi ) \
        bli_tcopynzims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
                     xr, \
          PASTEMAC(PASTEMAC(chx,prec),neg)( \
                     xi ), \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
                     yr, \
                     yi \
        )

// -- Higher-level static functions --------------------------------------------

// -- Notes --------------------------------------------------------------------

// -- Domain cases --

//   r       r
// (yr) := (xr);
// (yi) xx   0 ;

//   r       c
// (yr) := (xr);
// (yi) xx (xi);

//   c       r
// (yr) := (xr);
// (yi) xx   0 ;    // NOTE: This is what copynzs does differently from copys.

//   c       c
// (yr) := (xr);
// (yi) := (xi);

#endif

// end bli_tcopynzs.h
#line 71 "../blis-2.0/frame/include//bli_scalar_macro_defs.h"

// begin bli_tcopys.h
#line 1 "../blis-2.0/frame/include/level0//bli_tcopys.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2024, Southern Methodist University

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_TCOPYS_H
#define BLIS_TCOPYS_H

// -- Implementation macro -----------------------------------------------------

// (yr) := (xr);
// (yi) := (xi);

#define bli_tcopyims( \
          \
          dx, px, xr, xi, \
          dy, py, yr, yi \
        ) \
{ \
	PASTEMAC(dy,assigns) \
	( \
	  PASTEMAC(px,py,tcast)(xr), \
	  PASTEMAC(px,py,tcast)(xi), \
	  yr, \
	  yi \
	); \
}

// -- API macros ---------------------------------------------------------------

// -- Consolidated --

// tcopys
#define bli_tcopys( chx, chy, x, y ) \
        bli_tcopyims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(chx,real)(x), \
          PASTEMAC(chx,imag)(x), \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
          PASTEMAC(chy,real)(y), \
          PASTEMAC(chy,imag)(y) \
        )

// tcopyjs
#define bli_tcopyjs( chx, chy, x, y ) \
        bli_tcopyims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(chx,real)(x), \
          PASTEMAC(PASTEMAC(chx,prec),neg)( \
            PASTEMAC(chx,imag)(x)  \
          ), \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
          PASTEMAC(chy,real)(y), \
          PASTEMAC(chy,imag)(y) \
        )

// -- Exposed real/imaginary --

// tcopyris
#define bli_tcopyris( chx, chy, xr, xi, yr, yi ) \
        bli_tcopyims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
                     xr, \
                     xi, \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
                     yr, \
                     yi \
        )

// tcopyjris
#define bli_tcopyjris( chx, chy, xr, xi, yr, yi ) \
        bli_tcopyims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
                     xr, \
          PASTEMAC(PASTEMAC(chx,prec),neg)( \
                     xi ), \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
                     yr, \
                     yi \
        )

// -- 1e / 1r --

// tcopy1es
#define bli_tcopy1es( chx, chy, x, yri, yir ) \
        bli_tcopyims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(chx,real)(x), \
          PASTEMAC(chx,imag)(x), \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
          PASTEMAC(chy,real)(yri), \
          PASTEMAC(chy,imag)(yri) \
        ); \
        bli_tcopyims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(PASTEMAC(chx,prec),neg)( \
            PASTEMAC(chx,imag)(x)  \
          ), \
          PASTEMAC(chx,real)(x), \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
          PASTEMAC(chy,real)(yir), \
          PASTEMAC(chy,imag)(yir) \
        )

// tcopyj1es
#define bli_tcopyj1es( chx, chy, x, yri, yir ) \
        bli_tcopyims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(chx,real)(x), \
          PASTEMAC(PASTEMAC(chx,prec),neg)( \
            PASTEMAC(chx,imag)(x)  \
          ), \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
          PASTEMAC(chy,real)(yri), \
          PASTEMAC(chy,imag)(yri) \
        ); \
        bli_tcopyims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(chx,imag)(x), \
          PASTEMAC(chx,real)(x), \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
          PASTEMAC(chy,real)(yir), \
          PASTEMAC(chy,imag)(yir) \
        )

// tcopy1rs
#define bli_tcopy1rs( chx, chy, x, yr, yi ) \
        bli_tcopyims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(chx,real)(x), \
          PASTEMAC(chx,imag)(x), \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
          yr, \
          yi \
        )

// tcopyj1rs
#define bli_tcopyj1rs( chx, chy, x, yr, yi ) \
        bli_tcopyims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(chx,real)(x), \
          PASTEMAC(PASTEMAC(chx,prec),neg)( \
            PASTEMAC(chx,imag)(x)  \
          ), \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
          yr, \
          yi \
        )

// -- Higher-level static functions --------------------------------------------

// -- mxn --

#define bli_tcopys_mxn( chx, chy, m, n, x, rs_x, cs_x, y, rs_y, cs_y ) \
{ \
	for ( dim_t jj = 0; jj < (n); ++jj ) \
	for ( dim_t ii = 0; ii < (m); ++ii ) \
	{ \
		PASTEMAC(chx,ctype)* restrict xij = ( PASTEMAC(chx,ctype)* )(x) + ii*(rs_x) + jj*(cs_x); \
		PASTEMAC(chy,ctype)* restrict yij = ( PASTEMAC(chy,ctype)* )(y) + ii*(rs_y) + jj*(cs_y); \
\
		bli_tcopys( chx,chy, *xij, *yij ); \
	} \
}

// -- Legacy macros ------------------------------------------------------------

#define bli_scopys( x, y ) bli_tcopys( s,s, x, y )
#define bli_dcopys( x, y ) bli_tcopys( d,d, x, y )
#define bli_ccopys( x, y ) bli_tcopys( c,c, x, y )
#define bli_zcopys( x, y ) bli_tcopys( z,z, x, y )
#define bli_icopys( x, y ) bli_tcopys( i,i, x, y )

// -- Notes --------------------------------------------------------------------

// -- Domain cases --

//   r       r
// (yr) := (xr);
// (yi) xx   0 ;

//   r       c
// (yr) := (xr);
// (yi) xx (xi);

//   c       r
// (yr) := (xr);
// (yi) :=   0 ;

//   c       c
// (yr) := (xr);
// (yi) := (xi);

#endif

// end bli_tcopys.h
#line 72 "../blis-2.0/frame/include//bli_scalar_macro_defs.h"

// begin bli_tdots.h
#line 1 "../blis-2.0/frame/include/level0//bli_tdots.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2024, Southern Methodist University

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_TDOTS_H
#define BLIS_TDOTS_H

// -- Implementation macro -----------------------------------------------------

// (cr) += (ar) * (br) - (ai) * (bi);
// (ci) += (ai) * (br) + (ar) * (bi);

#define bli_tdotims bli_taxpyims

// -- API macros ---------------------------------------------------------------

// NOTE: When defining the tdots macros, we can recycle taxpys since they both
// perform c += a * b. However, when invoking taxpys, the first two operands
// passed in must be swapped because in BLIS axpy is set up to conjugate its
// second operand (ie: the second operand to the a*x product) while dot
// is set up to conjugate its first operand (ie: the first operand to the x*y
// product).

// -- Consolidated --

// tdots
#define  bli_tdots( chx,chy,cha,chc, x, y, a ) \
        bli_taxpys( chy,chx,cha,chc, y, x, a )

// tdotjs
#define  bli_tdotjs( chx,chy,cha,chc, x, y, a ) \
        bli_taxpyjs( chy,chx,cha,chc, y, x, a )

// -- Exposed real/imaginary --

// tdotris
#define  bli_tdotris( chx,chy,cha,chc, xr, xi, yr, yi, ar, ai ) \
        bli_taxpyris( chy,chx,cha,chc, yr, yi, xr, yx, ar, ai )

// tdotjris
#define  bli_tdotjris( chx,chy,cha,chc, xr, xi, yr, yi, ar, ai ) \
        bli_taxpyjris( chy,chx,cha,chc, yr, yi, xr, yx, ar, ai )

// -- Higher-level static functions --------------------------------------------

// -- Legacy macros ------------------------------------------------------------

#define bli_sdots( x, y, a ) bli_tdots( s,s,s,s, x, y, a )
#define bli_ddots( x, y, a ) bli_tdots( d,d,d,d, x, y, a )
#define bli_cdots( x, y, a ) bli_tdots( c,c,c,s, x, y, a )
#define bli_zdots( x, y, a ) bli_tdots( z,z,z,d, x, y, a )

// -- Notes --------------------------------------------------------------------

// -- Domain cases --

//   r       r      r
// (yr) += (ar) * (xr) -   0  *   0 ;
// (yi) xx   0  * (xr) + (ar) *   0 ;

//   r       r      c
// (yr) += (ar) * (xr) -   0  * (xi);
// (yi) xx   0  * (xr) + (ar) * (xi);

//   r       c      r
// (yr) += (ar) * (xr) - (ai) *   0 ;
// (yi) xx (ai) * (xr) + (ar) *   0 ;

//   r       c      c
// (yr) += (ar) * (xr) - (ai) * (xi);
// (yi) xx (ai) * (xr) + (ar) * (xi);

//   c       r      r
// (yr) += (ar) * (xr) -   0  *   0 ;
// (yi) +=   0  * (xr) + (ar) *   0 ;

//   c       r      c
// (yr) += (ar) * (xr) -   0  * (xi);
// (yi) +=   0  * (xr) + (ar) * (xi);

//   c       c      r
// (yr) += (ar) * (xr) - (ai) *   0 ;
// (yi) += (ai) * (xr) + (ar) *   0 ;

//   c       c      c
// (yr) += (ar) * (xr) - (ai) * (xi);
// (yi) += (ai) * (xr) + (ar) * (xi);

#endif

// end bli_tdots.h
#line 73 "../blis-2.0/frame/include//bli_scalar_macro_defs.h"

// begin bli_teqs.h
#line 1 "../blis-2.0/frame/include/level0//bli_teqs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2024, Southern Methodist University

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_TEQS_H
#define BLIS_TEQS_H

// -- Implementation macro -----------------------------------------------------

// (xr) == (yr) && (xi) == (yi)

#define bli_teqims( \
          \
          dx, px, xr, xi, \
          dy, py, yr, yi, \
          chc  \
        ) \
    ( PASTEMAC(PASTEMAC(chc,prec),eq)( PASTEMAC(px,chc,tcast)(xr), \
                                       PASTEMAC(py,chc,tcast)(yr) ) && \
      PASTEMAC(PASTEMAC(chc,prec),eq)( PASTEMAC(px,chc,tcast)(xi), \
                                       PASTEMAC(py,chc,tcast)(yi) ) )

// -- API macros ---------------------------------------------------------------

// -- Consolidated --

// teqs
#define bli_teqs( chx, chy, chc, x, y ) \
        bli_teqims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(chx,real)(x), \
          PASTEMAC(chx,imag)(x), \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
          PASTEMAC(chy,real)(y), \
          PASTEMAC(chy,imag)(y), \
          PASTEMAC(chc,prec)  \
        )

// -- Exposed real/imaginary --

// teqris
#define bli_teqris( chx, chy, chc, xr, xi, yr, yi ) \
        bli_teqims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
                     xr, \
                     xi, \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
                     yr, \
                     yi, \
          PASTEMAC(chc,prec)  \
        )

// -- Convenience macros -------------------------------------------------------

// -- Exposed real/imaginary --

#define bli_teq1ris( chx, xr, xi ) \
        bli_teqris \
        ( \
          chx, chx, chx, \
                     xr, \
                     xi, \
          PASTEMAC(PASTEMAC(chx,prec),one), \
          PASTEMAC(PASTEMAC(chx,prec),zero) \
        )

#define bli_teq0ris( chx, xr, xi ) \
        bli_teqris \
        ( \
          chx, chx, chx, \
                     xr, \
                     xi, \
          PASTEMAC(PASTEMAC(chx,prec),zero), \
          PASTEMAC(PASTEMAC(chx,prec),zero) \
        )

#define bli_teqm1ris( chx, xr, xi ) \
        bli_teqris \
        ( \
          chx, chx, chx, \
                     xr, \
                     xi, \
          PASTEMAC(PASTEMAC(chx,prec),mone), \
          PASTEMAC(PASTEMAC(chx,prec),zero) \
        )

// -- Consolidated --

#define bli_teq1s( chx, x ) \
        bli_teq1ris \
        ( \
          chx, \
          PASTEMAC(chx,real)(x), \
          PASTEMAC(chx,imag)(x) \
        )

#define bli_teq0s( chx, x ) \
        bli_teq0ris \
        ( \
          chx, \
          PASTEMAC(chx,real)(x), \
          PASTEMAC(chx,imag)(x) \
        )

#define bli_teqm1s( chx, x ) \
        bli_teqm1ris \
        ( \
          chx, \
          PASTEMAC(chx,real)(x), \
          PASTEMAC(chx,imag)(x)  \
        )

// -- Higher-level static functions --------------------------------------------

// -- Legacy macros ------------------------------------------------------------

#define bli_seqs( x, y ) bli_teqs( s,s,s, x, y )
#define bli_deqs( x, y ) bli_teqs( d,d,d, x, y )
#define bli_ceqs( x, y ) bli_teqs( c,c,c, x, y )
#define bli_zeqs( x, y ) bli_teqs( z,z,z, x, y )

#define bli_seq1( x ) bli_teq1s( s, x )
#define bli_deq1( x ) bli_teq1s( d, x )
#define bli_ceq1( x ) bli_teq1s( c, x )
#define bli_zeq1( x ) bli_teq1s( z, x )

#define bli_seq0( x ) bli_teq0s( s, x )
#define bli_deq0( x ) bli_teq0s( d, x )
#define bli_ceq0( x ) bli_teq0s( c, x )
#define bli_zeq0( x ) bli_teq0s( z, x )

// -- Notes --------------------------------------------------------------------

#endif

// end bli_teqs.h
#line 74 "../blis-2.0/frame/include//bli_scalar_macro_defs.h"

// begin bli_tfprints.h
#line 1 "../blis-2.0/frame/include/level0//bli_tfprints.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2024, Southern Methodist University

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_TFPRINTS_H
#define BLIS_TFPRINTS_H

// -- Implementation macro -----------------------------------------------------

// -- real domain implementation --

#define bli_rfprintims( \
          \
          file, spec, \
          dx, px, xr, xi \
        ) \
{ \
	fprintf( file, spec, xr ); \
}

// -- complex domain implementation --

#define bli_cfprintims( \
          \
          file, spec, \
          dx, px, xr, xi \
        ) \
{ \
	fprintf( file, spec, xr ); \
	fprintf( file, " + " ); \
	fprintf( file, spec, xi ); \
	fprintf( file, "i" ); \
}

// -- general implementation --

#define bli_tfprintims( \
          \
          file, spec, \
          dx, px, xr, xi \
        ) \
{ \
	PASTEMAC(dx,fprintims) \
	( \
	  file, spec, \
	  dx, px, xr, xi \
	); \
}

// -- API macros ---------------------------------------------------------------

// -- Consolidated --

// tfprints
#define bli_tfprints( chx, file, spec, x ) \
        bli_tfprintims \
        ( \
          file, spec, \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(chx,real)(x), \
          PASTEMAC(chx,imag)(x) \
        )

// -- Exposed real/imaginary --

// tfprintris
#define bli_tfprintris( chx, file, spec, xr, xi ) \
        bli_tfprintims \
        ( \
          file, spec, \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
                     xr, \
                     xi \
        )

// -- Higher-level static functions --------------------------------------------

// -- Notes --------------------------------------------------------------------

#endif

// end bli_tfprints.h
#line 75 "../blis-2.0/frame/include//bli_scalar_macro_defs.h"

// begin bli_tinverts.h
#line 1 "../blis-2.0/frame/include/level0//bli_tinverts.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2024, Southern Methodist University

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_TINVERTS_H
#define BLIS_TINVERTS_H

// -- Implementation macro -----------------------------------------------------

#define bli_tinvertims( \
          \
          dx, px, xr, xi, \
          chc  \
        ) \
{ \
	PASTEMAC(dx,invertims) \
	( \
	  dx, px, xr, xi, \
	  chc  \
	); \
}

// -- real domain implementation --

// (xr) = 1.0 / (xr);

#define bli_rinvertims( \
          \
          dx, px, xr, xi, \
          chc  \
        ) \
{ \
	PASTEMAC(r,assigns) \
	( \
	  PASTEMAC(chc,px,tcast)( \
	    PASTEMAC(chc,div)( \
	      PASTEMAC(chc,one), \
	      PASTEMAC(px,chc,tcast)(xr)  \
	    ) \
	  ),\
	  PASTEMAC(px,zero), \
	  xr, \
	  xi \
	); \
}

// -- complex domain implementation --

// sr    = maxabs( xr, xi );
// xsr   = xr / sr;
// xsi   = xi / sr;
// tempr = xr * xsr + xi * xsi
// xr    =  xsr / tempr;
// xi    = -xsi / tempr;

#define bli_cinvertims( \
          \
          dx, px, xr, xi, \
          chc  \
        ) \
{ \
	PASTEMAC(ro,declinits) \
	( \
	  chc, \
	  PASTEMAC(chc,maxabs)( \
	    PASTEMAC(px,chc,tcast)(xr), \
	    PASTEMAC(px,chc,tcast)(xi)  \
	  ), \
	  sr  \
	) \
	PASTEMAC(c,declinits) \
	( \
	  chc, \
	  PASTEMAC(chc,div)( \
	    PASTEMAC(px,chc,tcast)(xr), \
	    sr  \
	  ), \
	  PASTEMAC(chc,div)( \
	    PASTEMAC(px,chc,tcast)(xi), \
	    sr  \
	  ), \
	  xsr, \
	  xsi \
	) \
	PASTEMAC(ro,declinits) \
	( \
	  chc, \
	  PASTEMAC(chc,add)( \
	    PASTEMAC(chc,mul)( \
	      PASTEMAC(px,chc,tcast)(xr), \
	      xsr  \
	    ), \
	    PASTEMAC(chc,mul)( \
	      PASTEMAC(px,chc,tcast)(xi), \
	      xsi  \
	    ) \
	  ), \
	  tempr  \
	) \
	PASTEMAC(c,assigns) \
	( \
	  PASTEMAC(chc,px,tcast)( \
	    PASTEMAC(chc,div)( \
	      xsr, \
	      tempr  \
	    ) \
	  ),\
	  PASTEMAC(chc,px,tcast)( \
	    PASTEMAC(chc,div)( \
	      PASTEMAC(PASTEMAC(chc,prec),neg)(xsi), \
	      tempr  \
	    ) \
	  ),\
	  xr, \
	  xi \
	); \
}

// -- API macros ---------------------------------------------------------------

// -- Consolidated --

// tinverts
#define bli_tinverts( chx, chc, x ) \
        bli_tinvertims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(chx,real)(x), \
          PASTEMAC(chx,imag)(x), \
          PASTEMAC(chc,prec)  \
        )

// -- Exposed real/imaginary --

// tinvertris
#define bli_tinvertris( chx, chc, xr, xi ) \
        bli_tinvertims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
                     xr, \
                     xi, \
          PASTEMAC(chc,prec)  \
        )

// -- 1e / 1r --

// invert1es
#define bli_tinvert1es( chx, chc, xri, xir ) \
        bli_tinvertims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(chx,real)(xri), \
          PASTEMAC(chx,imag)(xri), \
          PASTEMAC(chc,prec)  \
        ); \
        bli_tcopyims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(PASTEMAC(chx,prec),neg)( \
            PASTEMAC(chx,imag)(xri)  \
          ), \
          PASTEMAC(chx,real)(xri), \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(chx,real)(xir), \
          PASTEMAC(chx,imag)(xir) \
        )

// invert1rs
#define bli_tinvert1rs( chx, chc, xr, xi ) \
        bli_tinvertims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          xr, \
          xi, \
          PASTEMAC(chc,prec)  \
        )

// -- Higher-level static functions --------------------------------------------

// -- Legacy macros ------------------------------------------------------------

#define bli_sinverts( x ) bli_tinverts( s,s, x )
#define bli_dinverts( x ) bli_tinverts( d,d, x )
#define bli_cinverts( x ) bli_tinverts( c,c, x )
#define bli_zinverts( x ) bli_tinverts( z,z, x )

// -- Notes --------------------------------------------------------------------

#endif

// end bli_tinverts.h
#line 76 "../blis-2.0/frame/include//bli_scalar_macro_defs.h"

// begin bli_tinvscals.h
#line 1 "../blis-2.0/frame/include/level0//bli_tinvscals.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2024, Southern Methodist University

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_TINVSCALS_H
#define BLIS_TINVSCALS_H

// -- Implementation macro -----------------------------------------------------

#define bli_tinvscalims( \
          \
          da, pa, ar, ai, \
          dx, px, xr, xi, \
          chc  \
        ) \
{ \
	PASTEMAC(da,dx,invscalims) \
	( \
	  da, pa, ar, ai, \
	  dx, px, xr, xi, \
	  chc  \
	); \
}

// -- real-real domain implementation --
// -- real-complex domain implementation --

// (xr) = (xr) / (ar);
// (xi) = (xi) / (ar);

#define bli_rrinvscalims bli_rcinvscalims

#define bli_rcinvscalims( \
          \
          da, pa, ar, ai, \
          dx, px, xr, xi, \
          chc  \
        ) \
{ \
	PASTEMAC(dx,assigns) \
	( \
	  PASTEMAC(chc,px,tcast)( \
	    PASTEMAC(chc,div)( \
	      PASTEMAC(px,chc,tcast)(xr),  \
	      PASTEMAC(pa,chc,tcast)(ar)  \
	    ) \
	  ),\
	  PASTEMAC(chc,px,tcast)( \
	    PASTEMAC(chc,div)( \
	      PASTEMAC(px,chc,tcast)(xi),  \
	      PASTEMAC(pa,chc,tcast)(ar)  \
	    ) \
	  ),\
	  xr, \
	  xi \
	); \
}

// -- complex-real domain implementation --
// -- complex-complex domain implementation --

// sr    = maxabs( ar, ai );
// asr   = ar / sr;
// asi   = ai / sr;
// xrt   = xr;
// tempr = ar * asr + ai * asi
// xr    = ( asr * xrt + asi * xi  ) / tempr;
// xi    = ( asr * xi  - asi * xrt ) / tempr;

#define bli_crinvscalims bli_ccinvscalims

#define bli_ccinvscalims( \
          \
          da, pa, ar, ai, \
          dx, px, xr, xi, \
          chc  \
        ) \
{ \
	PASTEMAC(ro,declinits) \
	( \
	  chc, \
	  PASTEMAC(chc,maxabs)( \
	    PASTEMAC(pa,chc,tcast)(ar), \
	    PASTEMAC(pa,chc,tcast)(ai)  \
	  ), \
	  sr  \
	) \
	PASTEMAC(c,declinits) \
	( \
	  chc, \
	  PASTEMAC(chc,div)( \
	    PASTEMAC(pa,chc,tcast)(ar), \
	    sr  \
	  ), \
	  PASTEMAC(chc,div)( \
	    PASTEMAC(pa,chc,tcast)(ai), \
	    sr  \
	  ), \
	  asr, \
	  asi \
	) \
	PASTEMAC(ro,declinits) \
	( \
	  chc, \
	  PASTEMAC(px,chc,tcast)(xr), \
	  xrt  \
	) \
	PASTEMAC(ro,declinits) \
	( \
	  chc, \
	  PASTEMAC(chc,add)( \
	    PASTEMAC(chc,mul)( \
	      PASTEMAC(pa,chc,tcast)(ar), \
	      asr  \
	    ), \
	    PASTEMAC(chc,mul)( \
	      PASTEMAC(pa,chc,tcast)(ai), \
	      asi  \
	    ) \
	  ), \
	  tempr  \
	) \
	PASTEMAC(dx,assigns) \
	( \
	  PASTEMAC(chc,px,tcast)( \
	    PASTEMAC(chc,div)( \
	      PASTEMAC(chc,add)( \
	        PASTEMAC(chc,mul)( \
	          asr, \
              xrt  \
            ), \
	        PASTEMAC(chc,mul)( \
	          asi, \
	          PASTEMAC(px,chc,tcast)(xi)  \
            )  \
          ), \
	      tempr  \
	    ) \
	  ),\
	  PASTEMAC(chc,px,tcast)( \
	    PASTEMAC(chc,div)( \
	      PASTEMAC(chc,sub)( \
	        PASTEMAC(chc,mul)( \
	          asr, \
	          PASTEMAC(px,chc,tcast)(xi)  \
            ), \
	        PASTEMAC(chc,mul)( \
	          asi, \
              xrt  \
            )  \
          ), \
	      tempr  \
	    ) \
	  ),\
	  xr, \
	  xi \
	); \
}

// -- API macros ---------------------------------------------------------------

// -- Consolidated --

// tinvscals
#define bli_tinvscals( cha, chx, chc, a, x ) \
        bli_tinvscalims \
        ( \
          PASTEMAC(cha,dom),  \
          PASTEMAC(cha,prec), \
          PASTEMAC(cha,real)(a), \
          PASTEMAC(cha,imag)(a), \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(chx,real)(x), \
          PASTEMAC(chx,imag)(x), \
          PASTEMAC(chc,prec)  \
        )

// tinvscaljs
#define bli_tinvscaljs( cha, chx, chc, a, x ) \
        bli_tinvscalims \
        ( \
          PASTEMAC(cha,dom),  \
          PASTEMAC(cha,prec), \
          PASTEMAC(cha,real)(a), \
          PASTEMAC(PASTEMAC(cha,prec),neg)( \
            PASTEMAC(cha,imag)(a)  \
          ), \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(chx,real)(x), \
          PASTEMAC(chx,imag)(x), \
          PASTEMAC(chc,prec)  \
        )

// -- Exposed real/imaginary --

// tinvscalris
#define bli_tinvscalris( cha, chx, chc, ar, ai, xr, xi ) \
        bli_tinvscalims \
        ( \
          PASTEMAC(cha,dom),  \
          PASTEMAC(cha,prec), \
                     ar, \
                     ai, \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
                     xr, \
                     xi, \
          PASTEMAC(chc,prec)  \
        )

// tinvscaljris
#define bli_tinvscaljris( cha, chx, chc, ar, ai, xr, xi ) \
        bli_tinvscalims \
        ( \
          PASTEMAC(cha,dom),  \
          PASTEMAC(cha,prec), \
                     ar, \
          PASTEMAC(PASTEMAC(cha,prec),neg)( \
                     ai ), \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
                     xr, \
                     xi, \
          PASTEMAC(chc,prec)  \
        )

// -- Higher-level static functions --------------------------------------------

// -- Notes --------------------------------------------------------------------

#endif

// end bli_tinvscals.h
#line 77 "../blis-2.0/frame/include//bli_scalar_macro_defs.h"

// begin bli_tneg2s.h
#line 1 "../blis-2.0/frame/include/level0//bli_tneg2s.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2024, Southern Methodist University

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_TNEG2S_H
#define BLIS_TNEG2S_H

// -- Implementation macro -----------------------------------------------------

// (yr) = -(xr);
// (yi) = -(xi);

#define bli_tneg2ims( \
          \
          dx, px, xr, xi, \
          dy, py, yr, yi \
        ) \
{ \
	PASTEMAC(dy,assigns) \
	( \
	  PASTEMAC(py,neg)( \
	    PASTEMAC(px,py,tcast)(xr) \
	  ), \
	  PASTEMAC(py,neg)( \
	    PASTEMAC(px,py,tcast)(xi) \
	  ), \
	  yr, \
	  yi \
	); \
}

// -- API macros ---------------------------------------------------------------

// -- Consolidated --

// tneg2s
#define bli_tneg2s( chx, chy, x, y ) \
        bli_tneg2ims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(chx,real)(x), \
          PASTEMAC(chx,imag)(x), \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
          PASTEMAC(chy,real)(y), \
          PASTEMAC(chy,imag)(y) \
        )

// -- Exposed real/imaginary --

// tneg2ris
#define bli_tneg2ris( chx, chy, xr, xi, yr, yi ) \
        bli_tneg2ims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
                     xr, \
                     xi, \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
                     yr, \
                     yi \
        )

// -- Higher-level static functions --------------------------------------------

// -- Notes --------------------------------------------------------------------

#endif

// end bli_tneg2s.h
#line 78 "../blis-2.0/frame/include//bli_scalar_macro_defs.h"

// begin bli_trandnp2s.h
#line 1 "../blis-2.0/frame/include/level0//bli_trandnp2s.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2024, Southern Methodist University

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_TRANDNP2S_H
#define BLIS_TRANDNP2S_H

// -- Implementation macro -----------------------------------------------------

// (xr) = randnp2();
// (xi) = randnp2();

#define bli_trandnp2ims( \
          \
          dx, px, xr, xi \
        ) \
{ \
	PASTEMAC(dx,assigns) \
	( \
	  PASTEMAC(px,randnp2),\
	  PASTEMAC(px,randnp2),\
	            xr, \
	            xi \
	); \
}

// -- API macros ---------------------------------------------------------------

// -- Consolidated --

// trandnp2s
#define bli_trandnp2s( chx, x ) \
        bli_trandnp2ims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(chx,real)(x), \
          PASTEMAC(chx,imag)(x) \
        )

// -- Exposed real/imaginary --

// trandnp2ris
#define bli_trandnp2ris( chx, xr, xi ) \
        bli_trandnp2ims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
                     xr, \
                     xi \
        )

// -- Higher-level static functions --------------------------------------------

// -- Notes --------------------------------------------------------------------

#endif

// end bli_trandnp2s.h
#line 79 "../blis-2.0/frame/include//bli_scalar_macro_defs.h"

// begin bli_trands.h
#line 1 "../blis-2.0/frame/include/level0//bli_trands.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2024, Southern Methodist University

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_TRANDS_H
#define BLIS_TRANDS_H

// -- Implementation macro -----------------------------------------------------

// (xr) = rand();
// (xi) = rand();

#define bli_trandims( \
          \
          dx, px, xr, xi \
        ) \
{ \
	PASTEMAC(dx,assigns) \
	( \
	  PASTEMAC(px,rand),\
	  PASTEMAC(px,rand),\
	            xr, \
	            xi \
	); \
}

// -- API macros ---------------------------------------------------------------

// -- Consolidated --

// trands
#define bli_trands( chx, x ) \
        bli_trandims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(chx,real)(x), \
          PASTEMAC(chx,imag)(x) \
        )

// -- Exposed real/imaginary --

// trandris
#define bli_trandris( chx, xr, xi ) \
        bli_trandims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
                     xr, \
                     xi \
        )

// -- Higher-level static functions --------------------------------------------

// -- Notes --------------------------------------------------------------------

#endif

// end bli_trands.h
#line 80 "../blis-2.0/frame/include//bli_scalar_macro_defs.h"

// begin bli_tscalcjs.h
#line 1 "../blis-2.0/frame/include/level0//bli_tscalcjs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2024, Southern Methodist University

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_TSCALCJS_H
#define BLIS_TSCALCJS_H

// -- Implementation macro -----------------------------------------------------

// (tr) := (ar) * (xr) - ( is_conj( conj ) ? -(ai) : (ai) ) * (xi);
// (ti) := ( is_conj( conj ) ? -(ai) : (ai) ) * (xr) + (ar) * (xi);
// (xr) := (tr);
// (xi) := (ti);

#define bli_tscalcjims( \
        \
          conj, \
          da, pa, ar, ai, \
          dx, px, xr, xi, \
          chc  \
        ) \
{ \
	PASTEMAC(c,declinits) \
	( \
	  chc, \
	  PASTEMAC(chc,sub)( \
	    PASTEMAC(da,dx,termrr)( \
	      chc, \
	      PASTEMAC(chc,mul)( \
	        PASTEMAC(pa,chc,tcast)(ar), \
	        PASTEMAC(px,chc,tcast)(xr) \
	      )  \
	    ), \
	    PASTEMAC(da,dx,termii)( \
	      chc, \
	      PASTEMAC(chc,mul)( \
	        ( bli_is_conj( conj ) ? PASTEMAC(chc,neg)( \
	                                  PASTEMAC(pa,chc,tcast)(ai) \
	                                ) \
	                              :   PASTEMAC(pa,chc,tcast)(ai) \
	        ), \
	        PASTEMAC(px,chc,tcast)(xi) \
	      ) \
	    ) \
	  ),\
	  PASTEMAC(chc,add)( \
	    PASTEMAC(da,dx,termir)( \
	      chc, \
	      PASTEMAC(chc,mul)( \
	        ( bli_is_conj( conj ) ? PASTEMAC(chc,neg)( \
	                                  PASTEMAC(pa,chc,tcast)(ai) \
	                                ) \
	                              :   PASTEMAC(pa,chc,tcast)(ai) \
	        ), \
	        PASTEMAC(px,chc,tcast)(xr) \
	      )  \
	    ), \
	    PASTEMAC(da,dx,termri)( \
	      chc, \
	      PASTEMAC(chc,mul)( \
	        PASTEMAC(pa,chc,tcast)(ar), \
	        PASTEMAC(px,chc,tcast)(xi) \
	      ) \
	    ) \
	  ), \
	  tr, \
	  ti \
	) \
	PASTEMAC(dx,assigns) \
	( \
	  PASTEMAC(chc,px,tcast)(tr), \
	  PASTEMAC(chc,px,tcast)(ti), \
	  xr, \
	  xi \
	); \
}

// -- API macros ---------------------------------------------------------------

// -- Consolidated --

// tscals
#define bli_tscalcjs( cha, chx, chc, conj, a, x ) \
        bli_tscalcjims \
        ( \
          conj, \
          PASTEMAC(cha,dom),  \
          PASTEMAC(cha,prec), \
          PASTEMAC(cha,real)(a), \
          PASTEMAC(cha,imag)(a), \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(chx,real)(x), \
          PASTEMAC(chx,imag)(x), \
          PASTEMAC(chc,prec)  \
        )

#endif

// end bli_tscalcjs.h
#line 81 "../blis-2.0/frame/include//bli_scalar_macro_defs.h"

// begin bli_tscal2s.h
#line 1 "../blis-2.0/frame/include/level0//bli_tscal2s.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2024, Southern Methodist University

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_TSCAL2S_H
#define BLIS_TSCAL2S_H

// -- Implementation macro -----------------------------------------------------

// (tr) := (ar) * (xr) - (ai) * (xi);
// (ti) := (ai) * (xr) + (ar) * (xi);
// (yr) := (tr);
// (yi) := (ti);

#define bli_tscal2ims( \
          \
          da, pa, ar, ai, \
          dx, px, xr, xi, \
          dy, py, yr, yi, \
          chc  \
        ) \
{ \
	PASTEMAC(c,declinits) \
	( \
	  py, \
	  PASTEMAC(chc,py,tcast)( \
	    PASTEMAC(chc,sub)( \
	      PASTEMAC(da,dx,termrr)( \
	        chc, \
	        PASTEMAC(chc,mul)( \
	          PASTEMAC(pa,chc,tcast)(ar), \
	          PASTEMAC(px,chc,tcast)(xr) \
	        )  \
	      ), \
	      PASTEMAC(da,dx,termii)( \
	        chc, \
	        PASTEMAC(chc,mul)( \
	          PASTEMAC(pa,chc,tcast)(ai), \
	          PASTEMAC(px,chc,tcast)(xi) \
	        ) \
	      ) \
	    ) \
	  ), \
	  PASTEMAC(chc,py,tcast)( \
	    PASTEMAC(chc,add)( \
	      PASTEMAC(da,dx,termir)( \
	        chc, \
	        PASTEMAC(chc,mul)( \
	          PASTEMAC(pa,chc,tcast)(ai), \
	          PASTEMAC(px,chc,tcast)(xr) \
	        )  \
	      ), \
	      PASTEMAC(da,dx,termri)( \
	        chc, \
	        PASTEMAC(chc,mul)( \
	          PASTEMAC(pa,chc,tcast)(ar), \
	          PASTEMAC(px,chc,tcast)(xi) \
	        ) \
	      ) \
	    ) \
	  ), \
	  tr, \
	  ti \
	); \
	PASTEMAC(dy,assigns) \
	( \
	  tr, \
	  ti, \
	  yr, \
	  yi \
	); \
}

// -- API macros ---------------------------------------------------------------

// -- Consolidated --

// tscal2s
#define bli_tscal2s( cha, chx, chy, chc, a, x, y ) \
        bli_tscal2ims \
        ( \
          PASTEMAC(cha,dom),  \
          PASTEMAC(cha,prec), \
          PASTEMAC(cha,real)(a), \
          PASTEMAC(cha,imag)(a), \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(chx,real)(x), \
          PASTEMAC(chx,imag)(x), \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
          PASTEMAC(chy,real)(y), \
          PASTEMAC(chy,imag)(y), \
          PASTEMAC(chc,prec)  \
        )

// tscal2js
#define bli_tscal2js( cha, chx, chy, chc, a, x, y ) \
        bli_tscal2ims \
        ( \
          PASTEMAC(cha,dom),  \
          PASTEMAC(cha,prec), \
          PASTEMAC(cha,real)(a), \
          PASTEMAC(cha,imag)(a), \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(chx,real)(x), \
          PASTEMAC(PASTEMAC(chx,prec),neg)( \
            PASTEMAC(chx,imag)(x)  \
          ), \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
          PASTEMAC(chy,real)(y), \
          PASTEMAC(chy,imag)(y), \
          PASTEMAC(chc,prec)  \
        )

// -- Exposed real/imaginary --

// tscal2ris
#define bli_tscal2ris( cha, chx, chy, chc, ar, ai, xr, xi, yr, yi ) \
        bli_tscal2ims \
        ( \
          PASTEMAC(cha,dom),  \
          PASTEMAC(cha,prec), \
                     ar, \
                     ai, \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
                     xr, \
                     xi, \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
                     yr, \
                     yi, \
          PASTEMAC(chc,prec)  \
        )

// tscal2jris
#define bli_tscal2jris( cha, chx, chy, chc, ar, ai, xr, xi, yr, yi ) \
        bli_tscal2ims \
        ( \
          PASTEMAC(cha,dom),  \
          PASTEMAC(cha,prec), \
                     ar, \
                     ai, \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
                     xr, \
          PASTEMAC(PASTEMAC(chx,prec),neg)( \
                     xi ), \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
                     yr, \
                     yi, \
          PASTEMAC(chc,prec)  \
        )

// -- 1e / 1r --

// tscal21es
#define bli_tscal21es( cha, chx, chy, chc, a, x, yri, yir ) \
        bli_tscal2ims \
        ( \
          PASTEMAC(cha,dom),  \
          PASTEMAC(cha,prec), \
          PASTEMAC(cha,real)(a), \
          PASTEMAC(cha,imag)(a), \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(chx,real)(x), \
          PASTEMAC(chx,imag)(x), \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
          PASTEMAC(chy,real)(yri), \
          PASTEMAC(chy,imag)(yri), \
          PASTEMAC(chc,prec)  \
        ); \
        bli_tcopyims \
        ( \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
          PASTEMAC(PASTEMAC(chy,prec),neg)( \
            PASTEMAC(chy,imag)(yri)  \
          ), \
          PASTEMAC(chy,real)(yri), \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
          PASTEMAC(chy,real)(yir), \
          PASTEMAC(chy,imag)(yir) \
        )

// tscal2j1es
#define bli_tscal2j1es( cha, chx, chy, chc, a, x, yri, yir ) \
        bli_tscal2ims \
        ( \
          PASTEMAC(cha,dom),  \
          PASTEMAC(cha,prec), \
          PASTEMAC(cha,real)(a), \
          PASTEMAC(cha,imag)(a), \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(chx,real)(x), \
          PASTEMAC(PASTEMAC(chx,prec),neg)( \
            PASTEMAC(chx,imag)(x)  \
          ), \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
          PASTEMAC(chy,real)(yri), \
          PASTEMAC(chy,imag)(yri), \
          PASTEMAC(chc,prec)  \
        ); \
        bli_tcopyims \
        ( \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
          PASTEMAC(PASTEMAC(chy,prec),neg)( \
            PASTEMAC(chy,imag)(yri)  \
          ), \
          PASTEMAC(chy,real)(yri), \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
          PASTEMAC(chy,real)(yir), \
          PASTEMAC(chy,imag)(yir) \
        )

// tscal21rs
#define bli_tscal21rs( cha, chx, chy, chc, a, x, yr, yi ) \
        bli_tscal2ims \
        ( \
          PASTEMAC(cha,dom),  \
          PASTEMAC(cha,prec), \
          PASTEMAC(cha,real)(a), \
          PASTEMAC(cha,imag)(a), \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(chx,real)(x), \
          PASTEMAC(chx,imag)(x), \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
          yr, \
          yi, \
          PASTEMAC(chc,prec)  \
        )

// tscal2j1rs
#define bli_tscal2j1rs( cha, chx, chy, chc, a, x, yr, yi ) \
        bli_tscal2ims \
        ( \
          PASTEMAC(cha,dom),  \
          PASTEMAC(cha,prec), \
          PASTEMAC(cha,real)(a), \
          PASTEMAC(cha,imag)(a), \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(chx,real)(x), \
          PASTEMAC(PASTEMAC(chx,prec),neg)( \
            PASTEMAC(chx,imag)(x)  \
          ), \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
          yr, \
          yi, \
          PASTEMAC(chc,prec)  \
        )

// -- Higher-level static functions --------------------------------------------

// -- mxn --

// scal2bbs_mxn

#define bli_tscal2bbs_mxn_r( \
                             cha,chx,chy,chc, \
                             ctypea, ctypea_r, \
                             ctypex, ctypex_r, \
                             ctypey, ctypey_r, \
                             conjx, \
                             m, \
                             n, \
                             alpha, \
                             x, incx, ldx, \
                             y, incy, ldy  \
                           ) \
{ \
	/* Assume that the duplication factor is the row stride of y. */ \
	const dim_t d    = incy; \
	const dim_t ds_y = 1; \
\
	if ( bli_is_conj( conjx ) ) \
	{ \
		for ( dim_t j = 0; j < (n); ++j ) \
		{ \
			ctypex* restrict xj = (ctypex*)(x) + j*(ldx); \
			ctypey* restrict yj = (ctypey*)(y) + j*(ldy); \
	\
			for ( dim_t i = 0; i < (m); ++i ) \
			{ \
				ctypex* restrict xij = xj + i*(incx); \
				ctypey* restrict yij = yj + i*(incy); \
	\
				bli_tscal2js( cha,chx,chy,chc, *(const ctypea* restrict)(alpha), *xij, *yij ); \
	\
				for ( dim_t p = 1; p < d; ++p ) \
				{ \
					ctypey* restrict yijd = yij + p*ds_y; \
	\
					bli_tcopys( chy,chy, *yij, *yijd ); \
				} \
			} \
		} \
	} \
	else \
	{ \
		for ( dim_t j = 0; j < (n); ++j ) \
		{ \
			ctypex* restrict xj = (ctypex*)(x) + j*(ldx); \
			ctypey* restrict yj = (ctypey*)(y) + j*(ldy); \
	\
			for ( dim_t i = 0; i < (m); ++i ) \
			{ \
				ctypex* restrict xij = xj + i*(incx); \
				ctypey* restrict yij = yj + i*(incy); \
	\
				bli_tscal2s( cha,chx,chy,chc, *(const ctypea* restrict)(alpha), *xij, *yij ); \
	\
				for ( dim_t p = 1; p < d; ++p ) \
				{ \
					ctypey* restrict yijd = yij + p*ds_y; \
	\
					bli_tcopys( chy,chy, *yij, *yijd ); \
				} \
			} \
		} \
	} \
}

#define bli_tscal2bbs_mxn_c( \
                             cha,chx,chy,chc, \
                             ctypea, ctypea_r, \
                             ctypex, ctypex_r, \
                             ctypey, ctypey_r, \
                             conjx, \
                             m, \
                             n, \
                             alpha, \
                             x, incx, ldx, \
                             y, incy, ldy  \
                           ) \
{ \
	/* Assume that the duplication factor is the row stride of y. */ \
	const dim_t       d          = incy; \
	const dim_t       ds_y       = 1; \
\
	const inc_t       incx2      = sizeof(ctypex) / sizeof(ctypex_r) * (incx); \
	const inc_t       ldx2       = sizeof(ctypex) / sizeof(ctypex_r) * (ldx); \
\
	const inc_t       incy2      = sizeof(ctypey) / sizeof(ctypey_r) * (incy); \
	const inc_t       ldy2       = sizeof(ctypey) / sizeof(ctypey_r) * (ldy); \
\
	ctypea_r* restrict alpha_r    = ( ctypea_r* )(alpha); \
	ctypea_r* restrict alpha_i    = ( ctypea_r* )(alpha) + 1; (void)alpha_i; \
	ctypex_r* restrict chi_r      = ( ctypex_r* )(x); \
	ctypex_r* restrict chi_i      = ( ctypex_r* )(x) + 1; (void)chi_i; \
	ctypey_r* restrict psi_r      = ( ctypey_r* )(y); \
	ctypey_r* restrict psi_i      = ( ctypey_r* )(y) + 1*d; (void)psi_i; \
\
	if ( bli_is_conj( conjx ) ) \
	{ \
		for ( dim_t j = 0; j < (n); ++j ) \
		{ \
			ctypex_r* restrict chij_r = chi_r + j*ldx2; \
			ctypex_r* restrict chij_i = chi_i + j*ldx2; \
			ctypey_r* restrict psij_r = psi_r + j*ldy2; \
			ctypey_r* restrict psij_i = psi_i + j*ldy2; \
\
			for ( dim_t i = 0; i < (m); ++i ) \
			{ \
				ctypex_r* restrict chiij_r = chij_r + i*incx2; \
				ctypex_r* restrict chiij_i = chij_i + i*incx2; (void)chiij_i; \
				ctypey_r* restrict psiij_r = psij_r + i*incy2; \
				ctypey_r* restrict psiij_i = psij_i + i*incy2; (void)psiij_i; \
\
				bli_tscal2jris( cha,chx,chy,chc, \
				                *alpha_r, *alpha_i, \
				                *chiij_r, *chiij_i, \
				                *psiij_r, *psiij_i ); \
\
				for ( dim_t p = 1; p < d; ++p ) \
				{ \
					ctypey_r* restrict psiijd_r = psiij_r + p*ds_y; \
					ctypey_r* restrict psiijd_i = psiij_i + p*ds_y; (void)psiijd_i; \
\
					bli_tcopyris( chy,chy, *psiij_r,  *psiij_i, \
					                       *psiijd_r, *psiijd_i ); \
				} \
			} \
		} \
	} \
	else /* if ( bli_is_noconj( conjx ) ) */ \
	{ \
		for ( dim_t j = 0; j < (n); ++j ) \
		{ \
			ctypex_r* restrict chij_r = chi_r + j*ldx2; \
			ctypex_r* restrict chij_i = chi_i + j*ldx2; \
			ctypey_r* restrict psij_r = psi_r + j*ldy2; \
			ctypey_r* restrict psij_i = psi_i + j*ldy2; \
\
			for ( dim_t i = 0; i < (m); ++i ) \
			{ \
				ctypex_r* restrict chiij_r = chij_r + i*incx2; \
				ctypex_r* restrict chiij_i = chij_i + i*incx2; (void)chiij_i; \
				ctypey_r* restrict psiij_r = psij_r + i*incy2; \
				ctypey_r* restrict psiij_i = psij_i + i*incy2; (void)psiij_i; \
\
				bli_tscal2ris( cha,chx,chy,chc, \
				               *alpha_r, *alpha_i, \
				               *chiij_r, *chiij_i, \
				               *psiij_r, *psiij_i ); \
\
				for ( dim_t p = 1; p < d; ++p ) \
				{ \
					ctypey_r* restrict psiijd_r = psiij_r + p*ds_y; \
					ctypey_r* restrict psiijd_i = psiij_i + p*ds_y; (void)psiijd_i; \
\
					bli_tcopyris( chy,chy, *psiij_r,  *psiij_i, \
					                       *psiijd_r, *psiijd_i ); \
				} \
			} \
		} \
	} \
}

#define bli_tscal2bbs_mxn( \
                           cha,chx,chy,chc, \
                           conjx, \
                           m, \
                           n, \
                           alpha, \
                           x, incx, ldx, \
                           y, incy, ldy  \
                         ) \
PASTECH(bli_tscal2bbs_mxn_,PASTEMAC(chy,dom)) \
( \
  cha,chx,chy,chc, \
  PASTEMAC(cha,ctype),PASTEMAC(cha,ctyper), \
  PASTEMAC(chx,ctype),PASTEMAC(chx,ctyper), \
  PASTEMAC(chy,ctype),PASTEMAC(chy,ctyper), \
  conjx, \
  m, \
  n, \
  alpha, \
  x, incx, ldx, \
  y, incy, ldy \
)

#define bli_tscal2s_mxn( cha, chx, chy, chc, conjx, m, n, alpha, x, rs_x, cs_x, y, rs_y, cs_y ) \
{ \
	if ( bli_is_conj( conjx ) ) \
	{ \
		for ( dim_t jj = 0; jj < (n); ++jj ) \
		for ( dim_t ii = 0; ii < (m); ++ii ) \
		{ \
			PASTEMAC(chx,ctype)* restrict xij = ( PASTEMAC(chx,ctype)* )(x) + ii*(rs_x) + jj*(cs_x); \
			PASTEMAC(chy,ctype)* restrict yij = ( PASTEMAC(chy,ctype)* )(y) + ii*(rs_y) + jj*(cs_y); \
\
			bli_tscal2js( cha,chx,chy,chc, *(alpha), *xij, *yij ); \
		} \
	} \
	else \
	{ \
		for ( dim_t jj = 0; jj < (n); ++jj ) \
		for ( dim_t ii = 0; ii < (m); ++ii ) \
		{ \
			PASTEMAC(chx,ctype)* restrict xij = ( PASTEMAC(chx,ctype)* )(x) + ii*(rs_x) + jj*(cs_x); \
			PASTEMAC(chy,ctype)* restrict yij = ( PASTEMAC(chy,ctype)* )(y) + ii*(rs_y) + jj*(cs_y); \
\
			bli_tscal2s( cha,chx,chy,chc, *(alpha), *xij, *yij ); \
		} \
	} \
}

#define bli_tscal2ris_mxn( cha, chx, chy, chc, conjx, m, n, alpha, x, rs_x, cs_x, y, rs_y, cs_y, is_y ) \
{ \
	PASTEMAC(cha,ctyper)* restrict alpha_r = ( PASTEMAC(cha,ctyper)* )(alpha);     (void)alpha_r; \
	PASTEMAC(cha,ctyper)* restrict alpha_i = ( PASTEMAC(cha,ctyper)* )(alpha) + 1; (void)alpha_i; \
	PASTEMAC(chx,ctyper)* restrict x_r     = ( PASTEMAC(chx,ctyper)* )(x); \
	PASTEMAC(chx,ctyper)* restrict x_i     = ( PASTEMAC(chx,ctyper)* )(x) + 1; \
	PASTEMAC(chy,ctyper)* restrict y_r     = ( PASTEMAC(chy,ctyper)* )(y); \
	PASTEMAC(chy,ctyper)* restrict y_i     = ( PASTEMAC(chy,ctyper)* )(y) + (is_y); \
	const dim_t incx2                      = 2*(rs_x); \
	const dim_t ldx2                       = 2*(cs_x); \
\
	if ( bli_is_conj( conjx ) ) \
	{ \
		for ( dim_t jj = 0; jj < (n); ++jj ) \
		for ( dim_t ii = 0; ii < (m); ++ii ) \
		{ \
			PASTEMAC(chx,ctyper)* restrict chi11_r = x_r + ii*incx2  + jj*ldx2;   (void)chi11_r; \
			PASTEMAC(chx,ctyper)* restrict chi11_i = x_i + ii*incx2  + jj*ldx2;   (void)chi11_i; \
			PASTEMAC(chy,ctyper)* restrict psi11_r = y_r + ii*(rs_y) + jj*(cs_y); (void)psi11_r; \
			PASTEMAC(chy,ctyper)* restrict psi11_i = y_i + ii*(rs_y) + jj*(cs_y); (void)psi11_i; \
\
			bli_tscal2jris \
			( \
			  cha,chx,chy,chc, \
			  *alpha_r, *alpha_i, \
			  *chi11_r, *chi11_i, \
			  *psi11_r, *psi11_i  \
			); \
		} \
	} \
	else \
	{ \
		for ( dim_t jj = 0; jj < (n); ++jj ) \
		for ( dim_t ii = 0; ii < (m); ++ii ) \
		{ \
			PASTEMAC(chx,ctyper)* restrict chi11_r = x_r + ii*incx2  + jj*ldx2;   (void)chi11_r; \
			PASTEMAC(chx,ctyper)* restrict chi11_i = x_i + ii*incx2  + jj*ldx2;   (void)chi11_i; \
			PASTEMAC(chy,ctyper)* restrict psi11_r = y_r + ii*(rs_y) + jj*(cs_y); (void)psi11_r; \
			PASTEMAC(chy,ctyper)* restrict psi11_i = y_i + ii*(rs_y) + jj*(cs_y); (void)psi11_i; \
\
			bli_tscal2ris \
			( \
			  cha,chx,chy,chc, \
			  *alpha_r, *alpha_i, \
			  *chi11_r, *chi11_i, \
			  *psi11_r, *psi11_i  \
			); \
		} \
	} \
}

// -- Legacy macros ------------------------------------------------------------

#define bli_sscal2bbs_mxn( conjx, m, n, alpha, x, incx, ldx, y, incy, ldy ) \
  bli_tscal2bbs_mxn( \
                     s,s,s,s, \
                     conjx, \
                     m, \
                     n, \
                     alpha, \
                     x, incx, ldx, \
                     y, incy, ldy  \
                   ) \

#define bli_dscal2bbs_mxn( conjx, m, n, alpha, x, incx, ldx, y, incy, ldy ) \
  bli_tscal2bbs_mxn( \
                     d,d,d,d, \
                     conjx, \
                     m, \
                     n, \
                     alpha, \
                     x, incx, ldx, \
                     y, incy, ldy  \
                   ) \

#define bli_cscal2bbs_mxn( conjx, m, n, alpha, x, incx, ldx, y, incy, ldy ) \
  bli_tscal2bbs_mxn( \
                     c,c,c,c, \
                     conjx, \
                     m, \
                     n, \
                     alpha, \
                     x, incx, ldx, \
                     y, incy, ldy  \
                   ) \

#define bli_zscal2bbs_mxn( conjx, m, n, alpha, x, incx, ldx, y, incy, ldy ) \
  bli_tscal2bbs_mxn( \
                     z,z,z,z, \
                     conjx, \
                     m, \
                     n, \
                     alpha, \
                     x, incx, ldx, \
                     y, incy, ldy  \
                   ) \

// -- Legacy macros ------------------------------------------------------------

#define bli_sscal2s( a, x, y ) bli_tscal2s( s,s,s,s, a, x, y )
#define bli_dscal2s( a, x, y ) bli_tscal2s( d,d,d,d, a, x, y )
#define bli_cscal2s( a, x, y ) bli_tscal2s( c,c,c,s, a, x, y )
#define bli_zscal2s( a, x, y ) bli_tscal2s( z,z,z,d, a, x, y )

// -- Notes --------------------------------------------------------------------

// -- Domain cases --

//   r       r      r
// (yr) := (ar) * (xr) -   0  *   0 ;
// (yi) xx   0  * (xr) + (ar) *   0 ;

//   r       r      c
// (yr) := (ar) * (xr) -   0  * (xi);
// (yi) xx   0  * (xr) + (ar) * (xi);

//   r       c      r
// (yr) := (ar) * (xr) - (ai) *   0 ;
// (yi) xx (ai) * (xr) + (ar) *   0 ;

//   r       c      c
// (yr) := (ar) * (xr) - (ai) * (xi);
// (yi) xx (ai) * (xr) + (ar) * (xi);

//   c       r      r
// (yr) := (ar) * (xr) -   0  *   0 ;
// (yi) :=   0  * (xr) + (ar) *   0 ;

//   c       r      c
// (yr) := (ar) * (xr) -   0  * (xi);
// (yi) :=   0  * (xr) + (ar) * (xi);

//   c       c      r
// (yr) := (ar) * (xr) - (ai) *   0 ;
// (yi) := (ai) * (xr) + (ar) *   0 ;

//   c       c      c
// (yr) := (ar) * (xr) - (ai) * (xi);
// (yi) := (ai) * (xr) + (ar) * (xi);

#endif

// end bli_tscal2s.h
#line 82 "../blis-2.0/frame/include//bli_scalar_macro_defs.h"

// begin bli_tscals.h
#line 1 "../blis-2.0/frame/include/level0//bli_tscals.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2024, Southern Methodist University

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_TSCALS_H
#define BLIS_TSCALS_H

// -- Implementation macro -----------------------------------------------------

// (tr) := (ar) * (xr) - (ai) * (xi);
// (ti) := (ai) * (xr) + (ar) * (xi);
// (xr) := (tr);
// (xi) := (ti);

#define bli_tscalims( \
        \
          da, pa, ar, ai, \
          dx, px, xr, xi, \
          chc  \
        ) \
{ \
	PASTEMAC(c,declinits) \
	( \
	  chc, \
	  PASTEMAC(chc,sub)( \
	    PASTEMAC(da,dx,termrr)( \
	      chc, \
	      PASTEMAC(chc,mul)( \
	        PASTEMAC(pa,chc,tcast)(ar), \
	        PASTEMAC(px,chc,tcast)(xr) \
	      )  \
	    ), \
	    PASTEMAC(da,dx,termii)( \
	      chc, \
	      PASTEMAC(chc,mul)( \
	        PASTEMAC(pa,chc,tcast)(ai), \
	        PASTEMAC(px,chc,tcast)(xi) \
	      ) \
	    ) \
	  ),\
	  PASTEMAC(chc,add)( \
	    PASTEMAC(da,dx,termir)( \
	      chc, \
	      PASTEMAC(chc,mul)( \
	        PASTEMAC(pa,chc,tcast)(ai), \
	        PASTEMAC(px,chc,tcast)(xr) \
	      )  \
	    ), \
	    PASTEMAC(da,dx,termri)( \
	      chc, \
	      PASTEMAC(chc,mul)( \
	        PASTEMAC(pa,chc,tcast)(ar), \
	        PASTEMAC(px,chc,tcast)(xi) \
	      ) \
	    ) \
	  ), \
	  tr, \
	  ti \
	) \
	PASTEMAC(dx,assigns) \
	( \
	  PASTEMAC(chc,px,tcast)(tr), \
	  PASTEMAC(chc,px,tcast)(ti), \
	  xr, \
	  xi \
	); \
}

// -- API macros ---------------------------------------------------------------

// -- Consolidated --

// tscals
#define bli_tscals( cha, chx, chc, a, x ) \
        bli_tscalims \
        ( \
          PASTEMAC(cha,dom),  \
          PASTEMAC(cha,prec), \
          PASTEMAC(cha,real)(a), \
          PASTEMAC(cha,imag)(a), \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(chx,real)(x), \
          PASTEMAC(chx,imag)(x), \
          PASTEMAC(chc,prec)  \
        )

// tscaljs
#define bli_tscaljs( cha, chx, chc, a, x ) \
        bli_tscalims \
        ( \
          PASTEMAC(cha,dom),  \
          PASTEMAC(cha,prec), \
          PASTEMAC(cha,real)(a), \
          PASTEMAC(PASTEMAC(cha,prec),neg)( \
            PASTEMAC(cha,imag)(a)  \
          ), \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(chx,real)(x), \
          PASTEMAC(chx,imag)(x), \
          PASTEMAC(chc,prec)  \
        )

// -- Exposed real/imaginary --

// tscalris
#define bli_tscalris( cha, chx, chc, ar, ai, xr, xi ) \
        bli_tscalims \
        ( \
          PASTEMAC(cha,dom),  \
          PASTEMAC(cha,prec), \
                     ar, \
                     ai, \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
                     xr, \
                     xi, \
          PASTEMAC(chc,prec)  \
        )

// tscaljris
#define bli_tscaljris( cha, chx, chc, ar, ai, xr, xi ) \
        bli_tscalims \
        ( \
          PASTEMAC(cha,dom),  \
          PASTEMAC(cha,prec), \
                     ar, \
          PASTEMAC(PASTEMAC(cha,prec),neg)( \
                     ai ), \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
                     xr, \
                     xi, \
          PASTEMAC(chc,prec)  \
        )

// -- 1e / 1r --

// scal1es
#define bli_tscal1es( cha, chx, chc, a, xri, xir ) \
        bli_tscalims \
        ( \
          PASTEMAC(cha,dom),  \
          PASTEMAC(cha,prec), \
          PASTEMAC(cha,real)(a), \
          PASTEMAC(cha,imag)(a), \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(chx,real)(xri), \
          PASTEMAC(chx,imag)(xri), \
          PASTEMAC(chc,prec)  \
        ); \
        bli_tcopyims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(PASTEMAC(chx,prec),neg)( \
            PASTEMAC(chx,imag)(xri)  \
          ), \
          PASTEMAC(chx,real)(xri), \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(chx,real)(xir), \
          PASTEMAC(chx,imag)(xir) \
        )

// scal1rs
#define bli_tscal1rs( cha, chx, chc, a, xr, xi ) \
        bli_tscalims \
        ( \
          PASTEMAC(cha,dom),  \
          PASTEMAC(cha,prec), \
          PASTEMAC(cha,real)(a), \
          PASTEMAC(cha,imag)(a), \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          xr, \
          xi, \
          PASTEMAC(chc,prec)  \
        )

// -- Higher-level static functions --------------------------------------------

// -- mxn_uplo --

#define bli_tscalris_mxn_uplo( cha, chx, chc, uplo, diagoff, m, n, ar, ai, xr, xi, rs_x, cs_x ) \
{ \
	if ( bli_is_upper( uplo ) ) \
	{ \
		for ( dim_t jj = 0; jj < n; ++jj ) \
		for ( dim_t ii = 0; ii < m; ++ii ) \
		{ \
			if ( (doff_t)jj - (doff_t)ii >= diagoff ) \
			{ \
				PASTEMAC(chx,ctyper)* restrict xij_r = (xr) + ii*(rs_x) + jj*(cs_x); \
				PASTEMAC(chx,ctyper)* restrict xij_i = (xi) + ii*(rs_x) + jj*(cs_x); \
				(void)xij_i; \
\
				bli_tscalris( cha,chx,chc, *(ar), *(ai), *xij_r, *xij_i ); \
			} \
		} \
	} \
	else \
	{ \
		for ( dim_t jj = 0; jj < n; ++jj ) \
		for ( dim_t ii = 0; ii < m; ++ii ) \
		{ \
			if ( (doff_t)jj - (doff_t)ii <= diagoff ) \
			{ \
				PASTEMAC(chx,ctyper)* restrict xij_r = (xr) + ii*(rs_x) + jj*(cs_x); \
				PASTEMAC(chx,ctyper)* restrict xij_i = (xi) + ii*(rs_x) + jj*(cs_x); \
				(void)xij_i; \
\
				bli_tscalris( cha,chx,chc, *(ar), *(ai), *xij_r, *xij_i ); \
			} \
		} \
	} \
}

// -- Legacy macros ------------------------------------------------------------

#define bli_sscals( a, x ) bli_tscals( s,s,s, a, x )
#define bli_dscals( a, x ) bli_tscals( d,d,d, a, x )
#define bli_cscals( a, x ) bli_tscals( c,c,s, a, x )
#define bli_zscals( a, x ) bli_tscals( z,z,d, a, x )

#define bli_ssscals( a, x ) bli_tscals( s,s,s, a, x )
#define bli_ddscals( a, x ) bli_tscals( d,d,d, a, x )
#define bli_ccscals( a, x ) bli_tscals( c,c,s, a, x )
#define bli_zzscals( a, x ) bli_tscals( z,z,d, a, x )

// -- Notes --------------------------------------------------------------------

// -- Domain cases --

//   r       r
// (xr) := (ar) * (xr) -   0  *   0 ;
// (xi) xx   0  * (xr) + (ar) *   0 ;

//   r       c
// (xr) := (ar) * (xr) - (ai) *   0 ;
// (xi) xx (ai) * (xr) + (ar) *   0 ;

//   c       r
// (xr) := (ar) * (xr) -   0  * (xi);
// (xi) :=   0  * (xr) + (ar) * (xi);

//   c       c
// (xr) := (ar) * (xr) - (ai) * (xi);
// (xi) := (ai) * (xr) + (ar) * (xi);

#endif

// end bli_tscals.h
#line 83 "../blis-2.0/frame/include//bli_scalar_macro_defs.h"

// begin bli_tsets.h
#line 1 "../blis-2.0/frame/include/level0//bli_tsets.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2024, Southern Methodist University

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_TSETS_H
#define BLIS_TSETS_H

// -- Implementation macros ----------------------------------------------------

#define bli_tsetims( \
          dx, px, xr, xi, \
          dy, py, yr, yi \
        ) \
{ \
	PASTEMAC(dy,assigns) \
	( \
	  PASTEMAC(px,py,tcast)(xr), \
	  PASTEMAC(px,py,tcast)(xi), \
	  yr, \
	  yi \
	); \
}

#define bli_tsetrims( \
              px, xr, \
          dy, py, yr, yi \
        ) \
{ \
	PASTEMAC(dy,assigns) \
	( \
	  PASTEMAC(px,py,tcast)(xr), \
	  yi, \
	  yr, \
	  yi \
	); \
}

#define bli_tsetiims( \
              px,     xi, \
          dy, py, yr, yi \
        ) \
{ \
	PASTEMAC(dy,assigns) \
	( \
	  yr, \
	  PASTEMAC(px,py,tcast)(xi), \
	  yr, \
	  yi \
	); \
}

// -- API macros ---------------------------------------------------------------

// -- Consolidated --

// tsets
#define bli_tsets( chx,chy, xr, xi, y ) \
        bli_tsetims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
                     xr, \
                     xi, \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
          PASTEMAC(chy,real)(y), \
          PASTEMAC(chy,imag)(y) \
        )

// tsetrs
#define bli_tsetrs( chx,chy, xr, y ) \
        bli_tsetrims \
        ( \
          PASTEMAC(chx,prec), \
                    xr, \
          PASTEMAC(chy,dom), \
          PASTEMAC(chy,prec), \
          PASTEMAC(chy,real)(y), \
          PASTEMAC(chy,imag)(y) \
        )

// tsetis
#define bli_tsetis( chx,chy, xi, y ) \
        bli_tsetiims \
        ( \
          PASTEMAC(chx,prec), \
                    xi, \
          PASTEMAC(chy,dom), \
          PASTEMAC(chy,prec), \
          PASTEMAC(chy,real)(y), \
          PASTEMAC(chy,imag)(y) \
        )

// -- Exposed real/imaginary --

// tsetris
#define bli_tsetris( chx,chy, xr, xi, yr, yi ) \
        bli_tsetims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
                     xr, \
                     xi, \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
                     yr, \
                     yi \
        )

// -- Set to constant --

// tset0s
#define bli_tset0s( chy, y ) \
    bli_tsets \
    ( \
      chy,chy, \
      PASTEMAC \
      ( \
        PASTEMAC(chy,prec), \
        zero \
      ), \
      PASTEMAC \
      ( \
        PASTEMAC(chy,prec), \
        zero \
      ), \
      y \
    )

// tset1s
#define bli_tset1s( chy, y ) \
    bli_tsets \
    ( \
      chy,chy, \
      PASTEMAC \
      ( \
       PASTEMAC(chy,prec), \
       one \
      ), \
      PASTEMAC \
      ( \
        PASTEMAC(chy,prec), \
        zero \
      ), \
      y \
    )

// tsetr0s
#define bli_tsetr0s( chy, y ) \
    bli_tsetrs( chy,chy, PASTEMAC(PASTEMAC(chy,prec),zero), y )

// tseti0s
#define bli_tseti0s( chy, y ) \
    bli_tsetis( chy,chy, PASTEMAC(PASTEMAC(chy,prec),zero), y )

// tset0ris
#define bli_tset0ris( chy, yr, yi ) \
    bli_tsetris \
    ( \
      chy,chy, \
      PASTEMAC \
      ( \
       PASTEMAC(chy,prec), \
       zero \
      ), \
      PASTEMAC \
      ( \
        PASTEMAC(chy,prec), \
        zero \
      ), \
      yr, \
      yi \
    )

// -- Micro-tile --

// set0s_mxn
#define bli_tset0s_mxn( chy, m, n, y, rs_y, cs_y ) \
{ \
	for ( dim_t _j = 0; _j < (n); ++_j ) \
	for ( dim_t _i = 0; _i < (m); ++_i ) \
	bli_tset0s( chy, *((y) + _i*(rs_y) + _j*(cs_y)) ); \
}

// set0bbs_mxn
#define bli_tset0bbs_mxn( chy, m, n, y, incy, ldy ) \
{ \
	/* Assume that the duplication factor is the row stride of y. */ \
	const dim_t _d    = incy; \
	const dim_t _ds_y = 1; \
\
	for ( dim_t _j = 0; _j < (n); ++_j ) \
	{ \
		PASTEMAC(chy,ctype)* restrict _yj = (PASTEMAC(chy,ctype)*)(y) + _j*(ldy); \
\
		for ( dim_t _i = 0; _i < (m); ++_i ) \
		{ \
			PASTEMAC(chy,ctype)* restrict _yij = _yj + _i*(incy); \
\
			for ( dim_t _p = 0; _p < _d; ++_p ) \
			{ \
				PASTEMAC(chy,ctype)* restrict _yijd = _yij + _p*_ds_y; \
\
				bli_tset0s( chy, *_yijd ); \
			} \
		} \
	} \
}

// bcastbbs_mxn
#define bli_tbcastbbs_mxn( chy, m, n, y, incy, ldy ) \
{ \
	/* Assume that the duplication factor is the row stride of y. */ \
	const dim_t _d = incy; \
\
	for ( dim_t _j = 0; _j < (n); ++_j ) \
	{ \
		PASTEMAC(chy,ctype)* _yj = (PASTEMAC(chy,ctype)*)(y) + _j*(ldy); \
\
		for ( dim_t _i = 0; _i < (m); ++_i ) \
		{ \
			PASTEMAC(chy,ctype)* _yij = _yj + _i*(incy); \
			PASTEMAC(chy,ctyper) _yij_r, _yij_i; \
\
			bli_tgets( chy,chy, *_yij, _yij_r, _yij_i ); \
\
			for ( dim_t _p = 0; _p < _d; ++_p ) \
			{ \
				PASTEMAC(chy,ctyper)* _yijd_r = (PASTEMAC(chy,ctyper)*)_yij      + _p; \
				PASTEMAC(chy,ctyper)* _yijd_i = (PASTEMAC(chy,ctyper)*)_yij + _d + _p; \
\
				bli_tcopyris( chy,chy, _yij_r, _yij_i, *_yijd_r, *_yijd_i ); \
			} \
		} \
	} \
}

// bcastbbs_mxn
#define bli_tcompressbbs_mxn( chy, m, n, y, incy, ldy ) \
{ \
	/* Assume that the duplication factor is the row stride of y. */ \
	const dim_t _d = incy; \
\
	for ( dim_t _j = 0; _j < (n); ++_j ) \
	{ \
		PASTEMAC(chy,ctype)* _yj = (PASTEMAC(chy,ctype)*)(y) + _j*(ldy); \
\
		for ( dim_t _i = 0; _i < (m); ++_i ) \
		{ \
			PASTEMAC(chy,ctype)* _yij = _yj + _i*(incy); \
			PASTEMAC(chy,ctyper)* _yij_r = (PASTEMAC(chy,ctyper)*)_yij; \
			PASTEMAC(chy,ctyper)* _yij_i = (PASTEMAC(chy,ctyper)*)_yij + _d; \
\
			bli_tsets( chy,chy, *_yij_r, *_yij_i, *_yij ); \
		} \
	} \
}

#define bli_tset0s_edge( chp, i, m, j, n, p, ldp ) \
{ \
	if ( (i) < (m) ) \
	{ \
		bli_tset0s_mxn \
		( \
		  chp, \
		  (m) - (i), \
		  j, \
		  (p) + (i)*1, 1, ldp \
		); \
	} \
\
	if ( (j) < (n) ) \
	{ \
		bli_tset0s_mxn \
		( \
		  chp, \
		  m, \
		  (n) - (j), \
		  (p) + (j)*(ldp), 1, ldp \
		); \
	} \
}

#endif

// -- Legacy macros ------------------------------------------------------------

#define bli_sset0s( x ) bli_tset0s( s, x )
#define bli_dset0s( x ) bli_tset0s( d, x )
#define bli_cset0s( x ) bli_tset0s( c, x )
#define bli_zset0s( x ) bli_tset0s( z, x )

#define bli_sset0s_edge( i, m, j, n, p, ldp ) bli_tset0s_edge( s, i, m, j, n, (float   *)(p), ldp )
#define bli_dset0s_edge( i, m, j, n, p, ldp ) bli_tset0s_edge( d, i, m, j, n, (double  *)(p), ldp )
#define bli_cset0s_edge( i, m, j, n, p, ldp ) bli_tset0s_edge( c, i, m, j, n, (scomplex*)(p), ldp )
#define bli_zset0s_edge( i, m, j, n, p, ldp ) bli_tset0s_edge( z, i, m, j, n, (dcomplex*)(p), ldp )

// -- Notes --------------------------------------------------------------------
// end bli_tsets.h
#line 84 "../blis-2.0/frame/include//bli_scalar_macro_defs.h"

// begin bli_tsqrt2s.h
#line 1 "../blis-2.0/frame/include/level0//bli_tsqrt2s.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2024, Southern Methodist University

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_TSQRT2S_H
#define BLIS_TSQRT2S_H

// -- Implementation macro -----------------------------------------------------

#define bli_tsqrt2ims( \
          \
          dx, px, xr, xi, \
          dy, py, yr, yi, \
          chc  \
        ) \
{ \
	PASTEMAC(dx,sqrtims) \
	( \
	  dx, px, xr, xi, \
	  dy, py, yr, yi, \
	  chc  \
	); \
}

// -- real domain implementation --

// yr = sqrt( xr );

#define bli_rsqrtims( \
          \
          dx, px, xr, xi, \
          dy, py, yr, yi, \
          chc  \
        ) \
{ \
	PASTEMAC(dy,assigns) \
	( \
	  PASTEMAC(chc,py,tcast)( \
	    PASTEMAC(chc,sqrt)( \
	      PASTEMAC(px,chc,tcast)( xr )  \
	    ) \
	  ), \
	  PASTEMAC(py,zero), \
	  yr, \
	  yi \
	); \
}

// -- complex domain implementation --

// mag = hypot( xr, xi );
// tr  = sqrt( ( mag + xr ) / 2.0 );
// ti  = sqrt( ( mag - xr ) / 2.0 );
// yr  =           tr;
// yi  = copysign( ti, xi );

#define bli_csqrtims( \
          \
          dx, px, xr, xi, \
          dy, py, yr, yi, \
          chc  \
        ) \
{ \
	PASTEMAC(ro,declinits) \
	( \
	  chc, \
	  PASTEMAC(chc,hypot)( \
	    PASTEMAC(px,chc,tcast)( xr ), \
	    PASTEMAC(px,chc,tcast)( xi )  \
	  ), \
	  mag  \
	) \
	PASTEMAC(c,declinits) \
	( \
	  chc, \
	  PASTEMAC(chc,sqrt)( \
	    PASTEMAC(chc,div)( \
	      PASTEMAC(chc,add)( \
	        mag, \
	        PASTEMAC(px,chc,tcast)( xr ) \
	      ), \
	      PASTEMAC(chc,two) \
	    ) \
	  ), \
	  PASTEMAC(chc,sqrt)( \
	    PASTEMAC(chc,div)( \
	      PASTEMAC(chc,sub)( \
	        mag, \
	        PASTEMAC(px,chc,tcast)( xr ) \
	      ), \
	      PASTEMAC(chc,two) \
	    ) \
	  ), \
	  tr, \
	  ti \
	) \
	PASTEMAC(dy,assigns) \
	( \
	  PASTEMAC(chc,py,tcast)( tr ), \
	  PASTEMAC(chc,py,tcast)( \
	    PASTEMAC(chc,copysign)( ti, xi ) \
	  ), \
	  yr, \
	  yi \
	); \
}

// -- API macros ---------------------------------------------------------------

// -- Consolidated --

// tsqrt2s
#define bli_tsqrt2s( chx, chy, chc, x, y ) \
        bli_tsqrt2ims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(chx,real)(x), \
          PASTEMAC(chx,imag)(x), \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
          PASTEMAC(chy,real)(y), \
          PASTEMAC(chy,imag)(y), \
          PASTEMAC(chc,prec)  \
        )

// -- Exposed real/imaginary --

// tsqrt2ris
#define bli_tsqrt2ris( chx, chy, chc, xr, xi, yr, yi ) \
        bli_tsqrt2ims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
                     xr, \
                     xi, \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
                     yr, \
                     yi, \
          PASTEMAC(chc,prec)  \
        )

// -- Higher-level static functions --------------------------------------------

// -- Notes --------------------------------------------------------------------

// -- Domain cases --

//   r       r
// (yr) := sqrt(xr);
// (yi) xx       0 ;

//   r       c
// (yr) := sqrt(xr ...);
// (yi) xx copysign(xi ...);

//   c       r
// (yr) := sqrt(xr);
// (yi) :=       0 ;

//   c       c
// (yr) := sqrt(xr ... );
// (yi) := copysign(xi ...);

#endif

// end bli_tsqrt2s.h
#line 85 "../blis-2.0/frame/include//bli_scalar_macro_defs.h"

// begin bli_tsubs.h
#line 1 "../blis-2.0/frame/include/level0//bli_tsubs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2024, Southern Methodist University

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_TSUBS_H
#define BLIS_TSUBS_H

// -- Implementation macro -----------------------------------------------------

// (yr) = (yr) - (xr);
// (yi) = (yi) - (xi);

#define bli_tsubims( \
          \
          dx, px, xr, xi, \
          dy, py, yr, yi, \
          chc  \
        ) \
{ \
	PASTEMAC(dy,assigns) \
	( \
	  PASTEMAC(chc,py,tcast)( \
	    PASTEMAC(chc,sub)( \
	      PASTEMAC(py,chc,tcast)(yr), \
	      PASTEMAC(px,chc,tcast)(xr)  \
	    ) \
	  ),\
	  PASTEMAC(chc,py,tcast)( \
	    PASTEMAC(chc,sub)( \
	      PASTEMAC(py,chc,tcast)(yi), \
	      PASTEMAC(px,chc,tcast)(xi) \
	    ) \
	  ), \
	  yr, \
	  yi \
	); \
}

// -- API macros ---------------------------------------------------------------

// -- Consolidated --

// tsubs
#define bli_tsubs( chx, chy, chc, x, y ) \
        bli_tsubims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(chx,real)(x), \
          PASTEMAC(chx,imag)(x), \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
          PASTEMAC(chy,real)(y), \
          PASTEMAC(chy,imag)(y), \
          PASTEMAC(chc,prec)  \
        )

// tsubjs
#define bli_tsubjs( chx, chy, chc, x, y ) \
        bli_tsubims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(chx,real)(x), \
          PASTEMAC(PASTEMAC(chx,prec),neg)( \
            PASTEMAC(chx,imag)(x)  \
          ), \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
          PASTEMAC(chy,real)(y), \
          PASTEMAC(chy,imag)(y), \
          PASTEMAC(chc,prec)  \
        )

// -- Exposed real/imaginary --

// tsubris
#define bli_tsubris( chx, chy, chc, xr, xi, yr, yi ) \
        bli_tsubims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
                     xr, \
                     xi, \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
                     yr, \
                     yi, \
          PASTEMAC(chc,prec)  \
        )

// tsubjris
#define bli_tsubjris( chx, chy, chc, xr, xi, yr, yi ) \
        bli_tsubims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
                     xr, \
          PASTEMAC(PASTEMAC(chx,prec),neg)( \
                     xi ), \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
                     yr, \
                     yi, \
          PASTEMAC(chc,prec)  \
        )

// -- Higher-level static functions --------------------------------------------

// -- Notes --------------------------------------------------------------------

// -- Domain cases --

//   r       r
// (yr) -= (xr);
// (yi) xx   0 ;

//   r       c
// (yr) -= (xr);
// (yi) xx (xi);

//   c       r
// (yr) -= (xr);
// (yi) -=   0 ;

//   c       c
// (yr) -= (xr);
// (yi) -= (xi);

#endif

// end bli_tsubs.h
#line 86 "../blis-2.0/frame/include//bli_scalar_macro_defs.h"

// begin bli_tswaps.h
#line 1 "../blis-2.0/frame/include/level0//bli_tswaps.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2024, Southern Methodist University

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_TSWAPS_H
#define BLIS_TSWAPS_H

// -- Implementation macro -----------------------------------------------------

// (tr) := (yr);
// (ti) := (yi);
// (yr) := (xr);
// (yi) := (xi);
// (xr) := (tr);
// (xi) := (ti);

#define bli_tswapims( \
          \
          dx, px, xr, xi, \
          dy, py, yr, yi \
        ) \
{ \
	/* It is important to use dx (or, alternatively, 'c') in the declinits macro
	   so that in the case of chy chx = r c, ti gets set to zero. The c r case
	   also works since ti, while unset by declinits, is unused by assigns. */ \
	PASTEMAC(dx,declinits)( py, yr, yi, tr, ti ) \
	PASTEMAC(dy,assigns) \
	( \
	  PASTEMAC(px,py,tcast)(xr),\
	  PASTEMAC(px,py,tcast)(xi), \
	  yr, \
	  yi \
	); \
	PASTEMAC(dx,assigns) \
	( \
	  PASTEMAC(py,px,tcast)(tr),\
	  PASTEMAC(py,px,tcast)(ti), \
	  xr, \
	  xi \
	); \
}

// -- API macros ---------------------------------------------------------------

// -- Consolidated --

// tswaps
#define bli_tswaps( chx, chy, x, y ) \
        bli_tswapims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
          PASTEMAC(chx,real)(x), \
          PASTEMAC(chx,imag)(x), \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
          PASTEMAC(chy,real)(y), \
          PASTEMAC(chy,imag)(y) \
        )

// -- Exposed real/imaginary --

// tswapris
#define bli_tswapris( chx, chy, xr, xi, yr, yi ) \
        bli_tswapims \
        ( \
          PASTEMAC(chx,dom),  \
          PASTEMAC(chx,prec), \
                     xr, \
                     xi, \
          PASTEMAC(chy,dom),  \
          PASTEMAC(chy,prec), \
                     yr, \
                     yi \
        )

// -- Higher-level static functions --------------------------------------------

// -- Legacy macros ------------------------------------------------------------


#define bli_sswaps( x, y ) bli_tswaps( s,s, x, y )
#define bli_dswaps( x, y ) bli_tswaps( d,d, x, y )
#define bli_cswaps( x, y ) bli_tswaps( c,c, x, y )
#define bli_zswaps( x, y ) bli_tswaps( z,z, x, y )

// -- Notes --------------------------------------------------------------------

// -- Domain cases --

// chy chx: r  r
// (tr) := (yr);
// (ti) :=   0 ;
// (yr) := (xr);
// (yi) xx (xi);
// (xr) := (tr);
// (xi) xx (ti);

// chy chx: r  c
// (tr) := (yr);
// (ti) :=   0 ;
// (yr) := (xr);
// (yi) xx (xi);
// (xr) := (tr);
// (xi) := (ti);

// chy chx: c  r
// (tr) := (yr);
// (ti) xx (yi);
// (yr) := (xr);
// (yi) :=   0 ;
// (xr) := (tr);
// (xi) xx (ti);

// chy chx: c  c
// (tr) := (yr);
// (ti) := (yi);
// (yr) := (xr);
// (yi) := (xi);
// (xr) := (tr);
// (xi) := (ti);

#endif

// end bli_tswaps.h
#line 87 "../blis-2.0/frame/include//bli_scalar_macro_defs.h"

// begin bli_txpbys.h
#line 1 "../blis-2.0/frame/include/level0//bli_txpbys.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2024, Southern Methodist University

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
	- Redistributions of source code must retain the above copyright
	  notice, this list of conditions and the following disclaimer.
	- Redistributions in binary form must reproduce the above copyright
	  notice, this list of conditions and the following disclaimer in the
	  documentation and/or other materials provided with the distribution.
	- Neither the name(s) of the copyright holder(s) nor the names of its
	  contributors may be used to endorse or promote products derived
	  from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_TXPBYS_H
#define BLIS_TXPBYS_H

// -- Implementation macro -----------------------------------------------------

// (yr) := (xr) + (br) * (yr) - (bi) * (yi);
// (yi) := (xi) + (bi) * (yr) + (br) * (yi);

#define bli_txpbyims( \
	      \
	      dx, px, xr, xi, \
	      db, pb, br, bi, \
	      dy, py, yr, yi, \
	      chc  \
	    ) \
{ \
	PASTEMAC(c,declinits) \
	( \
	  py, \
	  PASTEMAC(chc,py,tcast)( \
	    PASTEMAC(chc,add)( \
	      PASTEMAC(px,chc,tcast)(xr), \
	      PASTEMAC(chc,sub)( \
	        PASTEMAC(db,dy,termrr)( \
	          chc, \
	          PASTEMAC(chc,mul)( \
	            PASTEMAC(pb,chc,tcast)(br), \
	            PASTEMAC(py,chc,tcast)(yr) \
	          )  \
	        ), \
	        PASTEMAC(db,dy,termii)( \
	          chc, \
	          PASTEMAC(chc,mul)( \
	            PASTEMAC(pb,chc,tcast)(bi), \
	            PASTEMAC(py,chc,tcast)(yi) \
	          ) \
	        ) \
	      ) \
	    ) \
	  ),\
	  PASTEMAC(chc,py,tcast)( \
	    PASTEMAC(chc,add)( \
	      PASTEMAC(px,chc,tcast)(xi), \
	      PASTEMAC(chc,add)( \
	        PASTEMAC(db,dy,termir)( \
	          chc, \
	          PASTEMAC(chc,mul)( \
	            PASTEMAC(pb,chc,tcast)(bi), \
	            PASTEMAC(py,chc,tcast)(yr) \
	          )  \
	        ), \
	        PASTEMAC(db,dy,termri)( \
	          chc, \
	          PASTEMAC(chc,mul)( \
	            PASTEMAC(pb,chc,tcast)(br), \
	            PASTEMAC(py,chc,tcast)(yi) \
	          ) \
	        ) \
	      ) \
	    ) \
	  ), \
	  tr, \
	  ti \
	); \
	PASTEMAC(dy,assigns) \
	( \
	  tr, \
	  ti, \
	  yr, \
	  yi \
	); \
}

// -- API macros ---------------------------------------------------------------

// -- Consolidated --

// txpbys
#define bli_txpbys( chx, chb, chy, chc, x, b, y ) \
	    bli_txpbyims \
	    ( \
	      PASTEMAC(chx,dom),  \
	      PASTEMAC(chx,prec), \
	      PASTEMAC(chx,real)(x), \
	      PASTEMAC(chx,imag)(x), \
	      PASTEMAC(chb,dom),  \
	      PASTEMAC(chb,prec), \
	      PASTEMAC(chb,real)(b), \
	      PASTEMAC(chb,imag)(b), \
	      PASTEMAC(chy,dom),  \
	      PASTEMAC(chy,prec), \
	      PASTEMAC(chy,real)(y), \
	      PASTEMAC(chy,imag)(y), \
	      PASTEMAC(chc,prec)  \
	    )

// txpbyjs
#define bli_txpbyjs( chx, chb, chy, chc, x, b, y ) \
	    bli_txpbyims \
	    ( \
	      PASTEMAC(chx,dom),  \
	      PASTEMAC(chx,prec), \
	      PASTEMAC(chx,real)(x), \
	      PASTEMAC(PASTEMAC(chx,prec),neg)( \
	        PASTEMAC(chx,imag)(x)  \
	      ), \
	      PASTEMAC(chb,dom),  \
	      PASTEMAC(chb,prec), \
	      PASTEMAC(chb,real)(b), \
	      PASTEMAC(chb,imag)(b), \
	      PASTEMAC(chy,dom),  \
	      PASTEMAC(chy,prec), \
	      PASTEMAC(chy,real)(y), \
	      PASTEMAC(chy,imag)(y), \
	      PASTEMAC(chc,prec)  \
	    )

// -- Exposed real/imaginary --

// txpbyris
#define bli_txpbyris( chx, chb, chy, chc, xr, xi, br, bi, yr, yi ) \
	    bli_txpbyims \
	    ( \
	      PASTEMAC(chx,dom),  \
	      PASTEMAC(chx,prec), \
	                 xr, \
	                 xi, \
	      PASTEMAC(chb,dom),  \
	      PASTEMAC(chb,prec), \
	                 br, \
	                 bi, \
	      PASTEMAC(chy,dom),  \
	      PASTEMAC(chy,prec), \
	                 yr, \
	                 yi, \
	      PASTEMAC(chc,prec)  \
	    )

// txpbyjris
#define bli_txpbyjris( chx, chb, chy, chc, xr, xi, br, bi, yr, yi ) \
	    bli_txpbyims \
	    ( \
	      PASTEMAC(chx,dom),  \
	      PASTEMAC(chx,prec), \
	                 xr, \
	      PASTEMAC(PASTEMAC(chx,prec),neg)( \
	                 xi ), \
	      PASTEMAC(chb,dom),  \
	      PASTEMAC(chb,prec), \
	                 br, \
	                 bi, \
	      PASTEMAC(chy,dom),  \
	      PASTEMAC(chy,prec), \
	                 yr, \
	                 yi, \
	      PASTEMAC(chc,prec)  \
	    )

// -- Higher-level static functions --------------------------------------------

// -- mxn --

// xpbys_mxn
#define bli_txpbys_mxn( chx, chb, chy, chc, m, n, x, rs_x, cs_x, beta, y, rs_y, cs_y ) \
{ \
\
	/* If beta is zero, overwrite y with x (in case y has infs or NaNs). */ \
	if ( bli_teq0s( chb, *(beta) ) ) \
	{ \
		bli_tcopys_mxn( chx, chy, m, n, x, rs_x, cs_x, y, rs_y, cs_y ); \
	} \
	else \
	{ \
		for ( dim_t jj = 0; jj < n; ++jj ) \
		for ( dim_t ii = 0; ii < m; ++ii ) \
		{ \
			PASTEMAC(chx,ctype)* restrict xij = ( PASTEMAC(chx,ctype)* )(x) + ii*(rs_x) + jj*(cs_x); \
			PASTEMAC(chy,ctype)* restrict yij = ( PASTEMAC(chy,ctype)* )(y) + ii*(rs_y) + jj*(cs_y); \
\
			bli_txpbys( chx,chb,chy,chc, *xij, *(beta), *yij ); \
		} \
	} \
}

// xpbys_mxn_uplo
#define bli_txpbys_mxn_uplo( chx, chb, chy, chc, diagoff, uplo, m, n, x, rs_x, cs_x, beta, y, rs_y, cs_y ) \
{ \
	if ( bli_is_upper( uplo ) ) \
	{ \
		/* If beta is zero, overwrite y with x (in case y has infs or NaNs). */ \
		if ( bli_teq0s( chb, *(beta) ) ) \
		{ \
			for ( dim_t jj = 0; jj < n; ++jj ) \
			for ( dim_t ii = 0; ii < m; ++ii ) \
			{ \
				if ( (doff_t)jj - (doff_t)ii >= (diagoff) ) \
				{ \
					PASTEMAC(chx,ctype)* restrict xij = ( PASTEMAC(chx,ctype)* )(x) + ii*(rs_x) + jj*(cs_x); \
					PASTEMAC(chy,ctype)* restrict yij = ( PASTEMAC(chy,ctype)* )(y) + ii*(rs_y) + jj*(cs_y); \
\
					bli_tcopys( chx,chy, *xij, *yij ); \
				} \
			} \
		} \
		else \
		{ \
			for ( dim_t jj = 0; jj < n; ++jj ) \
			for ( dim_t ii = 0; ii < m; ++ii ) \
			{ \
				if ( (doff_t)jj - (doff_t)ii >= (diagoff) ) \
				{ \
					PASTEMAC(chx,ctype)* restrict xij = ( PASTEMAC(chx,ctype)* )(x) + ii*(rs_x) + jj*(cs_x); \
					PASTEMAC(chy,ctype)* restrict yij = ( PASTEMAC(chy,ctype)* )(y) + ii*(rs_y) + jj*(cs_y); \
\
					bli_txpbys( chx,chb,chy,chc, *xij, *(beta), *yij ); \
				} \
			} \
		} \
	} \
	else /* if ( bli_is_lower( uplo ) ) */ \
	{ \
		/* If beta is zero, overwrite y with x (in case y has infs or NaNs). */ \
		if ( bli_teq0s( chb, *(beta) ) ) \
		{ \
			for ( dim_t jj = 0; jj < n; ++jj ) \
			for ( dim_t ii = 0; ii < m; ++ii ) \
			{ \
				if ( (doff_t)jj - (doff_t)ii <= (diagoff) ) \
				{ \
					PASTEMAC(chx,ctype)* restrict xij = ( PASTEMAC(chx,ctype)* )(x) + ii*(rs_x) + jj*(cs_x); \
					PASTEMAC(chy,ctype)* restrict yij = ( PASTEMAC(chy,ctype)* )(y) + ii*(rs_y) + jj*(cs_y); \
\
					bli_tcopys( chx,chy, *xij, *yij ); \
				} \
			} \
		} \
		else \
		{ \
			for ( dim_t jj = 0; jj < n; ++jj ) \
			for ( dim_t ii = 0; ii < m; ++ii ) \
			{ \
				if ( (doff_t)jj - (doff_t)ii <= (diagoff) ) \
				{ \
					PASTEMAC(chx,ctype)* restrict xij = ( PASTEMAC(chx,ctype)* )(x) + ii*(rs_x) + jj*(cs_x); \
					PASTEMAC(chy,ctype)* restrict yij = ( PASTEMAC(chy,ctype)* )(y) + ii*(rs_y) + jj*(cs_y); \
\
					bli_txpbys( chx,chb,chy,chc, *xij, *(beta), *yij ); \
				} \
			} \
		} \
	} \
}

// -- Notes --------------------------------------------------------------------

// -- Domain cases --

//   r       r      r
// (yr) := (xr) + (br) * (yr) -   0  *   0 ;
// (yi) xx   0  +   0  * (yr) + (br) *   0 ;

//   r       r      c
// (yr) := (xr) + (br) * (yr) - (bi) *   0 ;
// (yi) xx   0  + (bi) * (yr) + (br) *   0 ;

//   r       c      r
// (yr) := (xr) + (br) * (yr) -   0  *   0 ;
// (yi) xx (xi) +   0  * (yr) + (br) *   0 ;

//   r       c      c
// (yr) := (xr) + (br) * (yr) - (bi) *   0 ;
// (yi) xx (xi) + (bi) * (yr) + (br) *   0 ;

//   c       r      r
// (yr) := (xr) + (br) * (yr) -   0  * (yi);
// (yi) :=   0  +   0  * (yr) + (br) * (yi);

//   c       r      c
// (yr) := (xr) + (br) * (yr) - (bi) * (yi);
// (yi) :=   0  + (bi) * (yr) + (br) * (yi);

//   c       c      r
// (yr) := (xr) + (br) * (yr) -   0  * (yi);
// (yi) := (xi) +   0  * (yr) + (br) * (yi);

//   c       c      c
// (yr) := (xr) + (br) * (yr) - (bi) * (yi);
// (yi) := (xi) + (bi) * (yr) + (br) * (yi);

#endif

// end bli_txpbys.h
#line 88 "../blis-2.0/frame/include//bli_scalar_macro_defs.h"



#endif
// end bli_scalar_macro_defs.h
#line 107 "../blis-2.0/frame/include//bli_macro_defs.h"

// begin bli_error_macro_defs.h
#line 1 "../blis-2.0/frame/include//bli_error_macro_defs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_ERROR_MACRO_DEFS_H
#define BLIS_ERROR_MACRO_DEFS_H

// Used to insert filenames and line numbers into error-checking code.
#define bli_check_error_code( code ) \
        bli_check_error_code_helper( code, __FILE__, __LINE__ )


#endif

// end bli_error_macro_defs.h
#line 108 "../blis-2.0/frame/include//bli_macro_defs.h"

// begin bli_blas_macro_defs.h
#line 1 "../blis-2.0/frame/include//bli_blas_macro_defs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_BLAS_MACRO_DEFS_H
#define BLIS_BLAS_MACRO_DEFS_H

// -- Various Fortran compatibility macros --

// Macro to treat negative dimensions as zero.

#define bli_convert_blas_dim1( n_blas, n_blis )\
{ \
	if ( n_blas < 0 ) n_blis = ( dim_t )0; \
	else              n_blis = ( dim_t )n_blas; \
}

// Macro to flip signs of increments if input increments are negative.

#define bli_convert_blas_incv( n, x_blas, incx_blas, \
                                  x_blis, incx_blis ) \
{ \
	if ( incx_blas < 0 ) \
	{ \
		/* The semantics of negative stride in BLAS are that the vector
		   operand be traversed in reverse order. (Another way to think
		   of this is that negative strides effectively reverse the order
		   of the vector, but without any explicit data movements.) This
		   is also how BLIS interprets negative strides. The differences
		   is that with BLAS, the caller *always* passes in the 0th (i.e.,
		   top-most or left-most) element of the vector, even when the
		   stride is negative. By contrast, in BLIS, negative strides are
		   used *relative* to the vector address as it is given. Thus, in
		   BLIS, if this backwards traversal is desired, the caller *must*
		   pass in the address to the (n-1)th (i.e., the bottom-most or
		   right-most) element along with a negative stride. */ \
		x_blis    = (x_blas) + (n-1)*(-incx_blas); \
		incx_blis = ( inc_t )(incx_blas); \
	} \
	else \
	{ \
		x_blis    = (x_blas); \
		incx_blis = ( inc_t )(incx_blas); \
	} \
}



#endif

// end bli_blas_macro_defs.h
#line 109 "../blis-2.0/frame/include//bli_macro_defs.h"

// begin bli_builtin_macro_defs.h
#line 1 "../blis-2.0/frame/include//bli_builtin_macro_defs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2019, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_BUILTIN_MACRO_DEFS_H
#define BLIS_BUILTIN_MACRO_DEFS_H

#if   defined(__ICC) || defined(__INTEL_COMPILER)

  // icc

  #define bli_prefetch( addr, rw, loc )

#elif defined(__clang__)

  // clang

  #define bli_prefetch( addr, rw, loc )

#elif defined(__GNUC__)

  // gcc

  #define bli_prefetch( addr, rw, loc ) __builtin_prefetch( addr, rw, loc );

#endif


#endif
// end bli_builtin_macro_defs.h
#line 110 "../blis-2.0/frame/include//bli_macro_defs.h"


// begin bli_oapi_macro_defs.h
#line 1 "../blis-2.0/frame/include//bli_oapi_macro_defs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

// Define the suffix to add to object API function names that include
// additional "expert" parameters.
#define BLIS_OAPI_EX_SUF _ex

// end bli_oapi_macro_defs.h
#line 112 "../blis-2.0/frame/include//bli_macro_defs.h"

// begin bli_tapi_macro_defs.h
#line 1 "../blis-2.0/frame/include//bli_tapi_macro_defs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

// Define the suffix to add to typed API function names that include
// additional "expert" parameters.
#define BLIS_TAPI_EX_SUF _ex

// end bli_tapi_macro_defs.h
#line 113 "../blis-2.0/frame/include//bli_macro_defs.h"



#endif
// end bli_macro_defs.h
#line 75 "../blis-2.0/frame/include/blis.h"



// -- pragma definitions --

// begin bli_pragma_macro_defs.h
#line 1 "../blis-2.0/frame/include//bli_pragma_macro_defs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2019, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

/*
  NOTE: The following code is based on [1].

  [1] https://github.com/jeffhammond/nwchem-tce-triples-kernels/blob/master/src/pragma_vendor.h
*/

#ifndef BLIS_PRAGMA_MACRO_DEFS_H
#define BLIS_PRAGMA_MACRO_DEFS_H

// Generally speaking, if BLIS_ENABLE_PRAGMA_OMP_SIMD is set, then we define
// all instances of PRAGMA_SIMD as _Pragma("omp simd").

#ifdef BLIS_ENABLE_PRAGMA_OMP_SIMD
  #define PRAGMA_OMP_SIMD _Pragma("omp simd")
#else
  #define PRAGMA_OMP_SIMD
#endif

// Require ISO C99 or later for SIMD-related pragmas.
#if (( __STDC_VERSION__ >= 199901L ))

  #define GEN_PRAGMA(x) _Pragma(#x)

  #if   defined(__ICC) || defined(__INTEL_COMPILER)

    // Intel icc.
    //#define PRAGMA_SIMD  GEN_PRAGMA(simd)
    #define PRAGMA_SIMD  PRAGMA_OMP_SIMD

  #elif defined(__clang__)

    // clang/llvm.
    #define PRAGMA_SIMD  PRAGMA_OMP_SIMD

  #elif defined(__GNUC__)

    // GNU gcc.
    #define PRAGMA_SIMD  PRAGMA_OMP_SIMD

  #else

    // Unknown compiler.
    #define PRAGMA_SIMD

  #endif
#endif

#endif
// end bli_pragma_macro_defs.h
#line 80 "../blis-2.0/frame/include/blis.h"



// -- BLIS architecture/kernel definitions --

// begin bli_pre_ker_params.h
#line 1 "../blis-2.0/frame/include//bli_pre_ker_params.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2023, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_PRE_KER_PARAMS_H
#define BLIS_PRE_KER_PARAMS_H

// These macros are used in bli_*_ker_prot.h and bli_*_ker_ft.h to make it
// easy to update them in the future, if needed.

#define BLIS_AUXINFO_PARAM  const auxinfo_t* data
#define BLIS_CNTX_PARAM     const cntx_t*    cntx


#endif
// end bli_pre_ker_params.h
#line 85 "../blis-2.0/frame/include/blis.h"

// begin bli_l1v_ker_params.h
#line 1 "../blis-2.0/frame/1//bli_l1v_ker_params.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_L1V_KER_PARAMS_H
#define BLIS_L1V_KER_PARAMS_H


#define addv_params \
\
             conj_t  conjx, \
             dim_t   n, \
       const void*   x, inc_t incx, \
             void*   y, inc_t incy

#define amaxv_params \
\
             dim_t   n, \
       const void*   x, inc_t incx, \
             dim_t*  index

#define axpbyv_params \
\
             conj_t  conjx, \
             dim_t   n, \
       const void*   alpha, \
       const void*   x, inc_t incx, \
       const void*   beta, \
             void*   y, inc_t incy

#define axpyv_params \
\
             conj_t  conjx, \
             dim_t   n, \
       const void*   alpha, \
       const void*   x, inc_t incx, \
             void*   y, inc_t incy

#define copyv_params \
\
             conj_t  conjx, \
             dim_t   n, \
       const void*   x, inc_t incx, \
             void*   y, inc_t incy

#define dotv_params \
\
             conj_t  conjx, \
             conj_t  conjy, \
             dim_t   n, \
       const void*   x, inc_t incx, \
       const void*   y, inc_t incy, \
             void*   rho

#define dotxv_params \
\
             conj_t  conjx, \
             conj_t  conjy, \
             dim_t   n, \
       const void*   alpha, \
       const void*   x, inc_t incx, \
       const void*   y, inc_t incy, \
       const void*   beta, \
             void*   rho

#define invertv_params \
\
             dim_t   n, \
             void*   x, inc_t incx

#define invscalv_params \
\
             conj_t  conjalpha, \
             dim_t   n, \
       const void*   alpha, \
             void*   x, inc_t incx

#define scalv_params \
\
             conj_t  conjalpha, \
             dim_t   n, \
       const void*   alpha, \
             void*   x, inc_t incx

#define scal2v_params \
\
             conj_t  conjx, \
             dim_t   n, \
       const void*   alpha, \
       const void*   x, inc_t incx, \
             void*   y, inc_t incy

#define setv_params \
\
             conj_t  conjalpha, \
             dim_t   n, \
       const void*   alpha, \
             void*   x, inc_t incx

#define subv_params \
\
             conj_t  conjx, \
             dim_t   n, \
       const void*   x, inc_t incx, \
             void*   y, inc_t incy

#define swapv_params \
\
             dim_t   n, \
             void*   x, inc_t incx, \
             void*   y, inc_t incy

#define xpbyv_params \
\
             conj_t  conjx, \
             dim_t   n, \
       const void*   x, inc_t incx, \
       const void*   beta, \
             void*   y, inc_t incy

#endif

// end bli_l1v_ker_params.h
#line 86 "../blis-2.0/frame/include/blis.h"

// begin bli_l1f_ker_params.h
#line 1 "../blis-2.0/frame/1f//bli_l1f_ker_params.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_L1F_KER_PARAMS_H
#define BLIS_L1F_KER_PARAMS_H


#define axpy2v_params \
\
             conj_t  conjx, \
             conj_t  conjy, \
             dim_t   n, \
       const void*   alphax, \
       const void*   alphay, \
       const void*   x, inc_t incx, \
       const void*   y, inc_t incy, \
             void*   z, inc_t incz

#define axpyf_params \
\
             conj_t  conja, \
             conj_t  conjx, \
             dim_t   m, \
             dim_t   b_n, \
       const void*   alpha, \
       const void*   a, inc_t inca, inc_t lda, \
       const void*   x, inc_t incx, \
             void*   y, inc_t incy

#define dotaxpyv_params \
\
             conj_t  conjxt, \
             conj_t  conjx, \
             conj_t  conjy, \
             dim_t   m, \
       const void*   alpha, \
       const void*   x, inc_t incx, \
       const void*   y, inc_t incy, \
             void*   rho, \
             void*   z, inc_t incz

#define dotxaxpyf_params \
\
             conj_t  conjat, \
             conj_t  conja, \
             conj_t  conjw, \
             conj_t  conjx, \
             dim_t   m, \
             dim_t   b_n, \
       const void*   alpha, \
       const void*   a, inc_t inca, inc_t lda, \
       const void*   w, inc_t incw, \
       const void*   x, inc_t incx, \
       const void*   beta, \
             void*   y, inc_t incy, \
             void*   z, inc_t incz

#define dotxf_params \
\
             conj_t  conjat, \
             conj_t  conjx, \
             dim_t   m, \
             dim_t   b_n, \
       const void*   alpha, \
       const void*   a, inc_t inca, inc_t lda, \
       const void*   x, inc_t incx, \
       const void*   beta, \
             void*   y, inc_t incy

#endif

// end bli_l1f_ker_params.h
#line 87 "../blis-2.0/frame/include/blis.h"

// begin bli_l1m_ker_params.h
#line 1 "../blis-2.0/frame/1m//bli_l1m_ker_params.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_L1M_KER_PARAMS_H
#define BLIS_L1M_KER_PARAMS_H


//
// -- Level-1m kernel function types -------------------------------------------
//

// packm

// NOTE: This is the function type for the structure-aware "kernel".

#define packm_params \
\
             struc_t strucc, \
             diag_t  diagc, \
             uplo_t  uploc, \
             conj_t  conjc, \
             pack_t  schema, \
             bool    invdiag, \
             dim_t   panel_dim, \
             dim_t   panel_len, \
             dim_t   panel_dim_max, \
             dim_t   panel_len_max, \
             dim_t   panel_dim_off, \
             dim_t   panel_len_off, \
             dim_t   panel_bcast, \
       const void*   kappa, \
       const void*   c, inc_t incc, inc_t ldc, \
             void*   p,             inc_t ldp, \
       const void*   params  \


// packm_cxk (packm microkernel)

#define packm_cxk_params \
\
             conj_t  conja, \
             pack_t  schema, \
             dim_t   cdim, \
             dim_t   cdim_max, \
             dim_t   cdim_bcast, \
             dim_t   n, \
             dim_t   n_max, \
       const void*   kappa, \
       const void*   a, inc_t inca, inc_t lda, \
             void*   p,             inc_t ldp, \
       const void*   params  \


// unpackm_cxk kernel

#define unpackm_cxk_params \
\
             conj_t  conja, \
             pack_t  schema, \
             dim_t   cdim, \
             dim_t   cdim_bcast, \
             dim_t   n, \
       const void*   kappa, \
       const void*   p,             inc_t ldp, \
             void*   a, inc_t inca, inc_t lda, \
       const void*   params  \


// packm_cxc_diag kernel

#define packm_cxc_diag_params \
\
             struc_t struca, \
             diag_t  diaga, \
             uplo_t  uploa, \
             conj_t  conja, \
             pack_t  schema, \
             bool    invdiag, \
             dim_t   cdim, \
             dim_t   cdim_max, \
             dim_t   cdim_bcast, \
             dim_t   n_max, \
       const void*   kappa, \
       const void*   a, inc_t inca, inc_t lda, \
             void*   p,             inc_t ldp, \
       const void*   params  \


#endif

// end bli_l1m_ker_params.h
#line 88 "../blis-2.0/frame/include/blis.h"

// begin bli_l3_ukr_params.h
#line 1 "../blis-2.0/frame/3//bli_l3_ukr_params.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_L3_UKR_PARAMS_H
#define BLIS_L3_UKR_PARAMS_H


#define gemm_params \
\
             dim_t  m, \
             dim_t  n, \
             dim_t  k, \
       const void*  alpha, \
       const void*  a, \
       const void*  b, \
       const void*  beta, \
             void*  c, inc_t rs_c, inc_t cs_c

#define gemmtrsm_params \
\
             dim_t  m, \
             dim_t  n, \
             dim_t  k, \
       const void*  alpha, \
       const void*  a1x, \
       const void*  a11, \
       const void*  bx1, \
             void*  b11, \
             void*  c11, inc_t rs_c, inc_t cs_c

#define trsm_params \
\
       const void*  a, \
             void*  b, \
             void*  c, inc_t rs_c, inc_t cs_c


#endif

// end bli_l3_ukr_params.h
#line 89 "../blis-2.0/frame/include/blis.h"

// begin bli_l3_sup_ker_params.h
#line 1 "../blis-2.0/frame/3//bli_l3_sup_ker_params.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_L3_SUP_KER_PARAMS_H
#define BLIS_L3_SUP_KER_PARAMS_H


#define gemmsup_params \
\
             conj_t conja, \
             conj_t conjb, \
             dim_t  m, \
             dim_t  n, \
             dim_t  k, \
       const void*  alpha, \
       const void*  a, inc_t rs_a, inc_t cs_a, \
       const void*  b, inc_t rs_b, inc_t cs_b, \
       const void*  beta, \
             void*  c, inc_t rs_c, inc_t cs_c


#endif

// end bli_l3_sup_ker_params.h
#line 90 "../blis-2.0/frame/include/blis.h"


// begin bli_l1v_ker_prot.h
#line 1 "../blis-2.0/frame/1//bli_l1v_ker_prot.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_L1V_KER_PROT_H
#define BLIS_L1V_KER_PROT_H

//
// Define template prototypes for level-1v kernels.
//

#undef  L1VTPROT
#define L1VTPROT( ctype, ch, funcname, opname ) \
\
void PASTEMAC(ch,funcname) \
     ( \
       PASTECH(opname,_params), \
       BLIS_CNTX_PARAM  \
     );

#define ADDV_KER_PROT(     ctype, ch, fn )  L1VTPROT( ctype, ch, fn, addv );
#define AMAXV_KER_PROT(    ctype, ch, fn )  L1VTPROT( ctype, ch, fn, amaxv );
#define AXPBYV_KER_PROT(   ctype, ch, fn )  L1VTPROT( ctype, ch, fn, axpbyv );
#define AXPYV_KER_PROT(    ctype, ch, fn )  L1VTPROT( ctype, ch, fn, axpyv );
#define COPYV_KER_PROT(    ctype, ch, fn )  L1VTPROT( ctype, ch, fn, copyv );
#define DOTV_KER_PROT(     ctype, ch, fn )  L1VTPROT( ctype, ch, fn, dotv );
#define DOTXV_KER_PROT(    ctype, ch, fn )  L1VTPROT( ctype, ch, fn, dotxv );
#define INVERTV_KER_PROT(  ctype, ch, fn )  L1VTPROT( ctype, ch, fn, invertv );
#define INVSCALV_KER_PROT( ctype, ch, fn )  L1VTPROT( ctype, ch, fn, invscalv );
#define SCALV_KER_PROT(    ctype, ch, fn )  L1VTPROT( ctype, ch, fn, scalv );
#define SCAL2V_KER_PROT(   ctype, ch, fn )  L1VTPROT( ctype, ch, fn, scal2v );
#define SETV_KER_PROT(     ctype, ch, fn )  L1VTPROT( ctype, ch, fn, setv );
#define SUBV_KER_PROT(     ctype, ch, fn )  L1VTPROT( ctype, ch, fn, subv );
#define SWAPV_KER_PROT(    ctype, ch, fn )  L1VTPROT( ctype, ch, fn, swapv );
#define XPBYV_KER_PROT(    ctype, ch, fn )  L1VTPROT( ctype, ch, fn, xpbyv );


#endif

// end bli_l1v_ker_prot.h
#line 92 "../blis-2.0/frame/include/blis.h"

// begin bli_l1f_ker_prot.h
#line 1 "../blis-2.0/frame/1f//bli_l1f_ker_prot.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_L1F_KER_PROT_H
#define BLIS_L1F_KER_PROT_H

//
// Define template prototypes for level-1f kernels.
//

#undef  L1FTPROT
#define L1FTPROT( ctype, ch, funcname, opname ) \
\
void PASTEMAC(ch,funcname) \
     ( \
       PASTECH(opname,_params), \
       BLIS_CNTX_PARAM  \
     );

#define AXPY2V_KER_PROT(    ctype, ch, fn )  L1FTPROT( ctype, ch, fn, axpy2v );
#define AXPYF_KER_PROT(     ctype, ch, fn )  L1FTPROT( ctype, ch, fn, axpyf );
#define DOTAXPYV_KER_PROT(  ctype, ch, fn )  L1FTPROT( ctype, ch, fn, dotaxpyv );
#define DOTXAXPYF_KER_PROT( ctype, ch, fn )  L1FTPROT( ctype, ch, fn, dotxaxpyf );
#define DOTXF_KER_PROT(     ctype, ch, fn )  L1FTPROT( ctype, ch, fn, dotxf );


#endif

// end bli_l1f_ker_prot.h
#line 93 "../blis-2.0/frame/include/blis.h"

// begin bli_l1m_ker_prot.h
#line 1 "../blis-2.0/frame/1m//bli_l1m_ker_prot.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_L1M_KER_PROT_H
#define BLIS_L1M_KER_PROT_H

//
// Define template prototypes for level-1m kernels.
//

#undef  L1MTPROT
#define L1MTPROT( ctype, ch, funcname, opname ) \
\
void PASTEMAC(ch,funcname) \
     ( \
       PASTECH(opname,_params), \
       BLIS_CNTX_PARAM  \
     );

#undef  L1MTPROT2
#define L1MTPROT2( ctypex, ctypey, chx, chy, funcname, opname ) \
\
void PASTEMAC(chx,chy,funcname) \
     ( \
       PASTECH(opname,_params), \
       BLIS_CNTX_PARAM  \
     );

#define PACKM_KER_PROT(      ctype, ch, fn )  L1MTPROT( ctype, ch, fn, packm_cxk );
#define UNPACKM_KER_PROT(    ctype, ch, fn )  L1MTPROT( ctype, ch, fn, unpackm_cxk );
#define PACKM_DIAG_KER_PROT( ctype, ch, fn )  L1MTPROT( ctype, ch, fn, packm_cxc_diag );

#define PACKM_KER_PROT2(      ctypex, ctypey, chx, chy, fn )  L1MTPROT2( ctypex, ctypey, chx, chy, fn, packm_cxk );
#define UNPACKM_KER_PROT2(    ctypex, ctypey, chx, chy, fn )  L1MTPROT2( ctypex, ctypey, chx, chy, fn, unpackm_cxk );
#define PACKM_DIAG_KER_PROT2( ctypex, ctypey, chx, chy, fn )  L1MTPROT2( ctypex, ctypey, chx, chy, fn, packm_cxc_diag );


#endif

// end bli_l1m_ker_prot.h
#line 94 "../blis-2.0/frame/include/blis.h"

// begin bli_l3_ukr_prot.h
#line 1 "../blis-2.0/frame/3//bli_l3_ukr_prot.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_L3_UKR_PROT_H
#define BLIS_L3_UKR_PROT_H

//
// Define template prototypes for level-3 micro-kernels.
//

#undef  L3TPROT
#define L3TPROT( ctype, ch, funcname, opname ) \
\
void PASTEMAC(ch,funcname) \
     ( \
       PASTECH(opname,_params), \
       BLIS_AUXINFO_PARAM, \
       BLIS_CNTX_PARAM  \
     );

#define GEMM_UKR2_PROT( ctypeab, ctypec, chab, chc, fn ) L3TPROT( /* not used */, PASTECH(chab,chc), fn, gemm );

#define GEMM_UKR_PROT(     ctype, ch, fn )  L3TPROT( ctype, ch, fn, gemm );
#define GEMMTRSM_UKR_PROT( ctype, ch, fn )  L3TPROT( ctype, ch, fn, gemmtrsm );
#define TRSM_UKR_PROT(     ctype, ch, fn )  L3TPROT( ctype, ch, fn, trsm );


#endif

// end bli_l3_ukr_prot.h
#line 95 "../blis-2.0/frame/include/blis.h"

// begin bli_l3_sup_ker_prot.h
#line 1 "../blis-2.0/frame/3//bli_l3_sup_ker_prot.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2019, Advanced Micro Devices, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_L3_SUP_KER_PROT_H
#define BLIS_L3_SUP_KER_PROT_H

//
// Define template prototypes for level-3 kernels on small/unpacked matrices.
//

#undef  SUPTPROT
#define SUPTPROT( ctype, ch, funcname, opname ) \
\
void PASTEMAC(ch,funcname) \
     ( \
       PASTECH(opname,_params), \
       BLIS_AUXINFO_PARAM, \
       BLIS_CNTX_PARAM  \
     );

#define GEMMSUP_KER_PROT( ctype, ch, fn )  SUPTPROT( ctype, ch, fn, gemmsup );


#endif

// end bli_l3_sup_ker_prot.h
#line 96 "../blis-2.0/frame/include/blis.h"


// begin bli_arch_config_pre.h
#line 1 "../blis-2.0/frame/include//bli_arch_config_pre.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_ARCH_CONFIG_PRE_H
#define BLIS_ARCH_CONFIG_PRE_H


// -- Naming-related kernel definitions ----------------------------------------

// The default suffix appended to reference kernels.
#define BLIS_REF_SUFFIX  _ref

// A suffix used for labeling certain induced method aware functions.
#define BLIS_IND_SUFFIX  _ind

// Add an underscore to the BLIS kernel set string, if it was defined.
#ifdef  BLIS_CNAME
#define BLIS_CNAME_INFIX  PASTECH(_,BLIS_CNAME)
#endif

// Add an underscore to the BLIS kernel set string, if it was defined.
#ifdef  BLIS_CNAME_UPPER
#define BLIS_CNAME_UPPER_INFIX  PASTECH(_,BLIS_CNAME_UPPER)
#endif

// Add an underscore to the plugin name, if it was defined.
#ifdef  BLIS_PNAME
#define BLIS_PNAME_INFIX  PASTECH(_,BLIS_PNAME)
#endif

// Combine the CNAME and _ref for convenience to the code that defines
// reference kernels.
//#define BLIS_CNAME_REF_SUFFIX  PASTECH(_,BLIS_CNAME,BLIS_REF_SUFFIX)

// -- Prototype-generating macro definitions -----------------------------------

// Prototype-generating macro for bli_cntx_init_<arch>*() functions.
#define CNTX_INIT_PROTS( archname ) \
\
void PASTEMAC(cntx_init_,archname) \
     ( \
       cntx_t* cntx \
     ); \
void PASTEMAC(cntx_init_,archname,BLIS_REF_SUFFIX) \
     ( \
       cntx_t* cntx \
     ); \
void PASTEMAC(cntx_init_,archname,BLIS_IND_SUFFIX) \
     ( \
       ind_t   method, \
       cntx_t* cntx \
     );


#endif

// end bli_arch_config_pre.h
#line 98 "../blis-2.0/frame/include/blis.h"

// begin bli_arch_config.h
#line 1 "../blis-2.0/frame/include//bli_arch_config.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2016, Hewlett Packard Enterprise Development LP
   Copyright (C) 2019 - 2020, Advanced Micro Devices, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_ARCH_CONFIG_H
#define BLIS_ARCH_CONFIG_H

//
// -- Context initialization prototypes ----------------------------------------
//

#undef GENTCONF
#define GENTCONF( CONFIG, config ) CNTX_INIT_PROTS( config )

INSERT_GENTCONF

//
// -- Architecture family-specific headers -------------------------------------
//

// -- x86_64 families --

#ifdef BLIS_FAMILY_INTEL64
#include "bli_family_intel64.h" // skipped
#endif
#ifdef BLIS_FAMILY_AMD64
#include "bli_family_amd64.h" // skipped
#endif
#ifdef BLIS_FAMILY_AMD64_LEGACY
#include "bli_family_amd64_legacy.h" // skipped
#endif
#ifdef BLIS_FAMILY_X86_64
#include "bli_family_x86_64.h" // skipped
#endif

// -- Intel64 architectures --

#ifdef BLIS_FAMILY_SKX
#include "bli_family_skx.h" // skipped
#endif
#ifdef BLIS_FAMILY_KNL
#include "bli_family_knl.h" // skipped
#endif
#ifdef BLIS_FAMILY_KNC
#include "bli_family_knc.h" // skipped
#endif
#ifdef BLIS_FAMILY_HASWELL
#include "bli_family_haswell.h" // skipped
#endif
#ifdef BLIS_FAMILY_SANDYBRIDGE
#include "bli_family_sandybridge.h" // skipped
#endif
#ifdef BLIS_FAMILY_PENRYN
#include "bli_family_penryn.h" // skipped
#endif

// -- AMD64 architectures --

#ifdef BLIS_FAMILY_ZEN3
#include "bli_family_zen3.h" // skipped
#endif
#ifdef BLIS_FAMILY_ZEN2
#include "bli_family_zen2.h" // skipped
#endif
#ifdef BLIS_FAMILY_ZEN
#include "bli_family_zen.h" // skipped
#endif
#ifdef BLIS_FAMILY_EXCAVATOR
#include "bli_family_excavator.h" // skipped
#endif
#ifdef BLIS_FAMILY_STEAMROLLER
#include "bli_family_steamroller.h" // skipped
#endif
#ifdef BLIS_FAMILY_PILEDRIVER
#include "bli_family_piledriver.h" // skipped
#endif
#ifdef BLIS_FAMILY_BULLDOZER
#include "bli_family_bulldozer.h" // skipped
#endif

// -- ARM families --
#ifdef BLIS_FAMILY_ARM64
// begin bli_family_arm64.h
#line 1 "../blis-2.0/config/arm64//bli_family_arm64.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

//#ifndef BLIS_FAMILY_H
//#define BLIS_FAMILY_H


// -- MEMORY ALLOCATION --------------------------------------------------------

#define BLIS_SIMD_ALIGN_SIZE 16

#define BLIS_SIMD_MAX_SIZE 128 // Note: The default is 64.
#define BLIS_SIMD_MAX_NUM_REGISTERS 32

// SVE-specific configs.
#define N_L1_SVE_DEFAULT 64
#define W_L1_SVE_DEFAULT 4
#define C_L1_SVE_DEFAULT 256
#define N_L2_SVE_DEFAULT 2048
#define W_L2_SVE_DEFAULT 16
#define C_L2_SVE_DEFAULT 256
#define N_L3_SVE_DEFAULT 8192
#define W_L3_SVE_DEFAULT 16
#define C_L3_SVE_DEFAULT 256


//#endif

// end bli_family_arm64.h
#line 116 "../blis-2.0/frame/include//bli_arch_config.h"

#endif
#ifdef BLIS_FAMILY_ARM32
#include "bli_family_arm32.h" // skipped
#endif

// -- ARM architectures --

// ARM-SVE
#ifdef BLIS_FAMILY_ARMSVE
#include "bli_family_armsve.h" // skipped
#endif
#ifdef BLIS_FAMILY_A64FX
#include "bli_family_a64fx.h" // skipped
#endif
// ARM-NEON (4x128)
#ifdef BLIS_FAMILY_ALTRAMAX
#include "bli_family_altramax.h" // skipped
#endif
#ifdef BLIS_FAMILY_ALTRA
#include "bli_family_altra.h" // skipped
#endif
#ifdef BLIS_FAMILY_FIRESTORM
// begin bli_family_firestorm.h
#line 1 "../blis-2.0/config/firestorm//bli_family_firestorm.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

//#ifndef BLIS_FAMILY_H
//#define BLIS_FAMILY_H


// -- MEMORY ALLOCATION --------------------------------------------------------

#define BLIS_SIMD_ALIGN_SIZE           16


#if 0
// -- LEVEL-3 MICRO-KERNEL CONSTANTS -------------------------------------------

#define BLIS_SGEMM_UKERNEL             bli_sgemm_opt_8x12
#define BLIS_DEFAULT_MR_S              8
#define BLIS_DEFAULT_NR_S              12
#define BLIS_DEFAULT_MC_S              120 //1536 //336 //416 // 1280 //160 // 160 // 160 //2048 //336 
#define BLIS_DEFAULT_KC_S              640 //1536 //336 //704 //1280 //672 //528 // 856 //2048 //528 
#define BLIS_DEFAULT_NC_S              3072

#define BLIS_DGEMM_UKERNEL             bli_dgemm_opt_6x8
#define BLIS_DEFAULT_MR_D              6
#define BLIS_DEFAULT_NR_D              8
#define BLIS_DEFAULT_MC_D              120 //1536 //160 //80 //176 
#define BLIS_DEFAULT_KC_D              240 //1536 //304 //336 //368 
#define BLIS_DEFAULT_NC_D              3072

#define BLIS_DEFAULT_MR_C              8
#define BLIS_DEFAULT_NR_C              4
#define BLIS_DEFAULT_MC_C              64
#define BLIS_DEFAULT_KC_C              128
#define BLIS_DEFAULT_NC_C              4096

#define BLIS_DEFAULT_MR_Z              8
#define BLIS_DEFAULT_NR_Z              4
#define BLIS_DEFAULT_MC_Z              64
#define BLIS_DEFAULT_KC_Z              128
#define BLIS_DEFAULT_NC_Z              4096
#endif


//#endif

// end bli_family_firestorm.h
#line 139 "../blis-2.0/frame/include//bli_arch_config.h"

#endif
// ARM-NEON (2x128)
#ifdef BLIS_FAMILY_THUNDERX2
// begin bli_family_thunderx2.h
#line 1 "../blis-2.0/config/thunderx2//bli_family_thunderx2.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

//#ifndef BLIS_FAMILY_H
//#define BLIS_FAMILY_H


// -- MEMORY ALLOCATION --------------------------------------------------------

#define BLIS_SIMD_ALIGN_SIZE           16

// end bli_family_thunderx2.h
#line 143 "../blis-2.0/frame/include//bli_arch_config.h"

#endif
#ifdef BLIS_FAMILY_CORTEXA57
// begin bli_family_cortexa57.h
#line 1 "../blis-2.0/config/cortexa57//bli_family_cortexa57.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

//#ifndef BLIS_FAMILY_H
//#define BLIS_FAMILY_H


// -- MEMORY ALLOCATION --------------------------------------------------------

#define BLIS_SIMD_ALIGN_SIZE           16


#if 0
// -- LEVEL-3 MICRO-KERNEL CONSTANTS -------------------------------------------

#define BLIS_SGEMM_UKERNEL             bli_sgemm_opt_8x12
#define BLIS_DEFAULT_MR_S              8
#define BLIS_DEFAULT_NR_S              12
#define BLIS_DEFAULT_MC_S              120 //1536 //336 //416 // 1280 //160 // 160 // 160 //2048 //336 
#define BLIS_DEFAULT_KC_S              640 //1536 //336 //704 //1280 //672 //528 // 856 //2048 //528 
#define BLIS_DEFAULT_NC_S              3072

#define BLIS_DGEMM_UKERNEL             bli_dgemm_opt_6x8
#define BLIS_DEFAULT_MR_D              6
#define BLIS_DEFAULT_NR_D              8
#define BLIS_DEFAULT_MC_D              120 //1536 //160 //80 //176 
#define BLIS_DEFAULT_KC_D              240 //1536 //304 //336 //368 
#define BLIS_DEFAULT_NC_D              3072

#define BLIS_DEFAULT_MR_C              8
#define BLIS_DEFAULT_NR_C              4
#define BLIS_DEFAULT_MC_C              64
#define BLIS_DEFAULT_KC_C              128
#define BLIS_DEFAULT_NC_C              4096

#define BLIS_DEFAULT_MR_Z              8
#define BLIS_DEFAULT_NR_Z              4
#define BLIS_DEFAULT_MC_Z              64
#define BLIS_DEFAULT_KC_Z              128
#define BLIS_DEFAULT_NC_Z              4096
#endif


//#endif

// end bli_family_cortexa57.h
#line 146 "../blis-2.0/frame/include//bli_arch_config.h"

#endif
#ifdef BLIS_FAMILY_CORTEXA53
// begin bli_family_cortexa53.h
#line 1 "../blis-2.0/config/cortexa53//bli_family_cortexa53.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/


// -- MEMORY ALLOCATION --------------------------------------------------------

#define BLIS_SIMD_ALIGN_SIZE           16


// end bli_family_cortexa53.h
#line 149 "../blis-2.0/frame/include//bli_arch_config.h"

#endif
// ARM 32-bit (vintage)
#ifdef BLIS_FAMILY_CORTEXA15
#include "bli_family_cortexa15.h" // skipped
#endif
#ifdef BLIS_FAMILY_CORTEXA9
#include "bli_family_cortexa9.h" // skipped
#endif

// -- IBM Power families --
#ifdef BLIS_FAMILY_POWER
#include "bli_family_power.h" // skipped
#endif

// -- IBM Power architectures --

#ifdef BLIS_FAMILY_POWER10
#include "bli_family_power10.h" // skipped
#endif
#ifdef BLIS_FAMILY_POWER9
#include "bli_family_power9.h" // skipped
#endif
#ifdef BLIS_FAMILY_POWER7
#include "bli_family_power7.h" // skipped
#endif

// -- IBM BG/Q --

#ifdef BLIS_FAMILY_BGQ
#include "bli_family_bgq.h" // skipped
#endif

// -- SiFive families --

#ifdef BLIS_FAMILY_SIFIVE_X280
#include "bli_family_sifive_x280.h" // skipped
#endif

// -- Generic --

#ifdef BLIS_FAMILY_GENERIC
// begin bli_family_generic.h
#line 1 "../blis-2.0/config/generic//bli_family_generic.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

//#ifndef BLIS_FAMILY_H
//#define BLIS_FAMILY_H




//#endif

// end bli_family_generic.h
#line 191 "../blis-2.0/frame/include//bli_arch_config.h"

#endif


//
// -- kernel set prototypes ----------------------------------------------------
//

// -- Intel64 architectures --
#ifdef BLIS_KERNELS_SKX
#include "bli_kernels_skx.h" // skipped
#endif
#ifdef BLIS_KERNELS_KNL
#include "bli_kernels_knl.h" // skipped
#endif
#ifdef BLIS_KERNELS_KNC
#include "bli_kernels_knc.h" // skipped
#endif
#ifdef BLIS_KERNELS_HASWELL
#include "bli_kernels_haswell.h" // skipped
#endif
#ifdef BLIS_KERNELS_SANDYBRIDGE
#include "bli_kernels_sandybridge.h" // skipped
#endif
#ifdef BLIS_KERNELS_PENRYN
#include "bli_kernels_penryn.h" // skipped
#endif

// -- AMD64 architectures --

#ifdef BLIS_KERNELS_ZEN2
#include "bli_kernels_zen2.h" // skipped
#endif
#ifdef BLIS_KERNELS_ZEN
#include "bli_kernels_zen.h" // skipped
#endif
//#ifdef BLIS_KERNELS_EXCAVATOR
//#include "bli_kernels_excavator.h"
//#endif
//#ifdef BLIS_KERNELS_STEAMROLLER
//#include "bli_kernels_steamroller.h"
//#endif
#ifdef BLIS_KERNELS_PILEDRIVER
#include "bli_kernels_piledriver.h" // skipped
#endif
#ifdef BLIS_KERNELS_BULLDOZER
#include "bli_kernels_bulldozer.h" // skipped
#endif

// -- ARM architectures --

#ifdef BLIS_KERNELS_ARMSVE
#include "bli_kernels_armsve.h" // skipped
#endif
#ifdef BLIS_KERNELS_ARMV8A
// begin bli_kernels_armv8a.h
#line 1 "../blis-2.0/kernels/armv8a//bli_kernels_armv8a.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

PACKM_KER_PROT( float,    s, packm_armv8a_int_8x12 )
PACKM_KER_PROT( double,   d, packm_armv8a_int_6x8 )

GEMM_UKR_PROT( float,    s, gemm_armv8a_asm_8x12 )
GEMM_UKR_PROT( double,   d, gemm_armv8a_asm_6x8 )
GEMM_UKR_PROT( float,    s, gemm_armv8a_asm_12x8r )
GEMM_UKR_PROT( double,   d, gemm_armv8a_asm_8x6r )
// GEMM_UKR_PROT( double,   d, gemm_armv8a_asm_6x8r )
// GEMM_UKR_PROT( double,   d, gemm_armv8a_asm_8x4 )
// GEMM_UKR_PROT( double,   d, gemm_armv8a_asm_4x4 )

GEMMSUP_KER_PROT( double,   d, gemmsup_rd_armv8a_asm_6x8n )
GEMMSUP_KER_PROT( double,   d, gemmsup_rd_armv8a_asm_6x8m )
GEMMSUP_KER_PROT( double,   d, gemmsup_rv_armv8a_asm_6x8n )
GEMMSUP_KER_PROT( double,   d, gemmsup_rv_armv8a_asm_6x8m )
GEMMSUP_KER_PROT( double,   d, gemmsup_rv_armv8a_asm_6x7m )
GEMMSUP_KER_PROT( double,   d, gemmsup_rv_armv8a_asm_6x6m )
GEMMSUP_KER_PROT( double,   d, gemmsup_rv_armv8a_asm_6x5m )
GEMMSUP_KER_PROT( double,   d, gemmsup_rv_armv8a_asm_5x8n )
GEMMSUP_KER_PROT( double,   d, gemmsup_rv_armv8a_asm_4x8n )
GEMMSUP_KER_PROT( double,   d, gemmsup_rv_armv8a_asm_4x8m )
GEMMSUP_KER_PROT( double,   d, gemmsup_rv_armv8a_asm_8x4m )

GEMMSUP_KER_PROT( double,   d, gemmsup_rd_armv8a_int_2x8 )
GEMMSUP_KER_PROT( double,   d, gemmsup_rd_armv8a_int_3x4 )
GEMMSUP_KER_PROT( double,   d, gemmsup_rd_armv8a_asm_3x4 )
GEMMSUP_KER_PROT( double,   d, gemmsup_rd_armv8a_asm_6x3 )

GEMMSUP_KER_PROT( double,   d, gemmsup_rv_armv8a_int_6x4mn )
GEMMSUP_KER_PROT( double,   d, gemmsup_rv_armv8a_int_3x8mn )

// end bli_kernels_armv8a.h
#line 246 "../blis-2.0/frame/include//bli_arch_config.h"

#endif
#ifdef BLIS_KERNELS_ARMV7A
#include "bli_kernels_armv7a.h" // skipped
#endif

// -- IBM Power --

#ifdef BLIS_KERNELS_POWER10
#include "bli_kernels_power10.h" // skipped
#endif
#ifdef BLIS_KERNELS_POWER9
#include "bli_kernels_power9.h" // skipped
#endif
#ifdef BLIS_KERNELS_POWER7
#include "bli_kernels_power7.h" // skipped
#endif

// -- IBM BG/Q --

#ifdef BLIS_KERNELS_BGQ
#include "bli_kernels_bgq.h" // skipped
#endif

// -- RISC-V --

#ifdef BLIS_KERNELS_RVI
#include "bli_kernels_rvi.h" // skipped
#endif
#ifdef BLIS_KERNELS_RVIV
#include "bli_kernels_rviv.h" // skipped
#endif

// -- SiFive RISC-V architectures --

#ifdef BLIS_KERNELS_SIFIVE_X280
#include "bli_kernels_sifive_x280.h" // skipped
#endif


#endif

// end bli_arch_config.h
#line 99 "../blis-2.0/frame/include/blis.h"


// begin bli_kernel_macro_defs.h
#line 1 "../blis-2.0/frame/include//bli_kernel_macro_defs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_KERNEL_MACRO_DEFS_H
#define BLIS_KERNEL_MACRO_DEFS_H


// -- Define default threading parameters --------------------------------------

// -- Conventional (large code path) values --

// These BLIS_THREAD_RATIO_? macros distort the amount of work in the m and n
// dimensions for the purposes of factorizing the total number of threads into
// ways of parallelism in the ic and jc loops. See bli_rntm.c to see how these
// macros are used.
#ifndef BLIS_THREAD_RATIO_M
#define BLIS_THREAD_RATIO_M     1
#endif

#ifndef BLIS_THREAD_RATIO_N
#define BLIS_THREAD_RATIO_N     1
#endif

// These BLIS_THREAD_MAX_?R macros place a ceiling on the maximum amount of
// parallelism allowed when performing automatic factorization. See bli_rntm.c
// to see how these macros are used.
#ifndef BLIS_THREAD_MAX_IR
#define BLIS_THREAD_MAX_IR      1
#endif

#ifndef BLIS_THREAD_MAX_JR
#define BLIS_THREAD_MAX_JR      4
#endif

#if 0
// -- Skinny/small possibly-unpacked (sup code path) values --

#ifndef BLIS_THREAD_SUP_RATIO_M
#define BLIS_THREAD_SUP_RATIO_M   1
#endif

#ifndef BLIS_THREAD_SUP_RATIO_N
#define BLIS_THREAD_SUP_RATIO_N   2
#endif

#ifndef BLIS_THREAD_SUP_MAX_IR
#define BLIS_THREAD_SUP_MAX_IR    1
#endif

#ifndef BLIS_THREAD_SUP_MAX_JR
#define BLIS_THREAD_SUP_MAX_JR    8
#endif
#endif


// -- Memory allocation --------------------------------------------------------

// hbwmalloc.h provides hbw_malloc() and hbw_free() on systems with
// libmemkind. But disable use of libmemkind if BLIS_DISABLE_MEMKIND
// was explicitly defined.
#ifdef BLIS_DISABLE_MEMKIND
  #undef BLIS_ENABLE_MEMKIND
#endif
#ifdef BLIS_ENABLE_MEMKIND
#include <hbwmalloc.h> // skipped
#endif

// Memory allocation functions. These macros define the three types of
// malloc()-style functions, and their free() counterparts: one for each
// type of memory to be allocated.
// NOTE: ANY ALTERNATIVE TO malloc()/free() USED FOR ANY OF THE FOLLOWING
// THREE PAIRS OF MACROS MUST USE THE SAME FUNCTION PROTOTYPE AS malloc()
// and free():
//
//   void* malloc( size_t size );
//   void  free( void* p );
//

// This allocation function is called to allocate memory for blocks within
// BLIS's internal memory pools.
#ifndef BLIS_MALLOC_POOL
  // If use of libmemkind was enabled at configure-time, the default
  // memory allocation function for memory pools should be hbw_malloc()
  // instead of malloc().
  #ifdef  BLIS_ENABLE_MEMKIND
  #define BLIS_MALLOC_POOL               hbw_malloc
  #else
  #define BLIS_MALLOC_POOL               malloc
  #endif
#endif

#ifndef BLIS_FREE_POOL
  // If use of libmemkind was enabled at configure-time, the default
  // memory deallocation function for memory pools should be hbw_free()
  // instead of free().
  #ifdef  BLIS_ENABLE_MEMKIND
  #define BLIS_FREE_POOL                 hbw_free
  #else
  #define BLIS_FREE_POOL                 free
  #endif
#endif

// This allocation function is called to allocate memory for internally-
// used objects and structures, such as control tree nodes.
#ifndef BLIS_MALLOC_INTL
#define BLIS_MALLOC_INTL                 malloc
#endif

#ifndef BLIS_FREE_INTL
#define BLIS_FREE_INTL                   free
#endif

// This allocation function is called to allocate memory for objects
// created by user-level API functions, such as bli_obj_create().
#ifndef BLIS_MALLOC_USER
#define BLIS_MALLOC_USER                 malloc
#endif

#ifndef BLIS_FREE_USER
#define BLIS_FREE_USER                   free
#endif


// -- Other system-related definitions -----------------------------------------

// Size of a virtual memory page. This is used to align blocks within the
// memory pools.
#ifndef BLIS_PAGE_SIZE
#define BLIS_PAGE_SIZE                   4096
#endif

// The maximum number of named SIMD vector registers available for use.
// When configuring with umbrella configuration families, this should be
// set to the maximum number of registers across all sub-configurations in
// the family.
#ifndef BLIS_SIMD_MAX_NUM_REGISTERS
#define BLIS_SIMD_MAX_NUM_REGISTERS      32
#endif

// The maximum size (in bytes) of each SIMD vector.
// When configuring with umbrella configuration families, this should be
// set to the maximum SIMD size across all sub-configurations in the family.
#ifndef BLIS_SIMD_MAX_SIZE
#define BLIS_SIMD_MAX_SIZE               64
#endif

// Alignment size (in bytes) needed by the instruction set for aligned
// SIMD/vector instructions.
#ifndef BLIS_SIMD_ALIGN_SIZE
#define BLIS_SIMD_ALIGN_SIZE             BLIS_SIMD_MAX_SIZE
#endif

// The maximum size in bytes of local stack buffers within macro-kernel
// functions. These buffers are usually used to store a temporary copy
// of a single microtile. The reason we multiply by 4 is to handle induced
// methods, where we use real domain register blocksizes in units of
// complex elements, as well as mixed-precision, where data may be
// converted to a wider type. Specifically, the macro-kernels will need this larger
// micro-tile footprint, even though the virtual micro-kernels will only
// ever be writing to half (real or imaginary part) at a time.
#ifndef BLIS_STACK_BUF_MAX_SIZE
#define BLIS_STACK_BUF_MAX_SIZE          ( BLIS_SIMD_MAX_NUM_REGISTERS * \
                                           BLIS_SIMD_MAX_SIZE * 4 )
#endif

// Alignment size used to align local stack buffers within macro-kernel
// functions.
#ifndef BLIS_STACK_BUF_ALIGN_SIZE
#define BLIS_STACK_BUF_ALIGN_SIZE        BLIS_SIMD_ALIGN_SIZE
#endif

// Alignment size used when allocating memory via BLIS_MALLOC_USER.
// To disable heap alignment, set this to 1.
#ifndef BLIS_HEAP_ADDR_ALIGN_SIZE
#define BLIS_HEAP_ADDR_ALIGN_SIZE        BLIS_SIMD_ALIGN_SIZE
#endif

// Alignment size used when sizing leading dimensions of memory allocated
// via BLIS_MALLOC_USER.
#ifndef BLIS_HEAP_STRIDE_ALIGN_SIZE
#define BLIS_HEAP_STRIDE_ALIGN_SIZE      BLIS_SIMD_ALIGN_SIZE
#endif

// Alignment sizes used when allocating blocks to the internal memory
// pool, via BLIS_MALLOC_POOL.
#ifndef BLIS_POOL_ADDR_ALIGN_SIZE_A
#define BLIS_POOL_ADDR_ALIGN_SIZE_A      BLIS_PAGE_SIZE
#endif

#ifndef BLIS_POOL_ADDR_ALIGN_SIZE_B
#define BLIS_POOL_ADDR_ALIGN_SIZE_B      BLIS_PAGE_SIZE
#endif

#ifndef BLIS_POOL_ADDR_ALIGN_SIZE_C
#define BLIS_POOL_ADDR_ALIGN_SIZE_C      BLIS_PAGE_SIZE
#endif

#ifndef BLIS_POOL_ADDR_ALIGN_SIZE_GEN
#define BLIS_POOL_ADDR_ALIGN_SIZE_GEN    BLIS_PAGE_SIZE
#endif

// Offsets from alignment specified by BLIS_POOL_ADDR_ALIGN_SIZE_*.
#ifndef BLIS_POOL_ADDR_OFFSET_SIZE_A
#define BLIS_POOL_ADDR_OFFSET_SIZE_A     0
#endif

#ifndef BLIS_POOL_ADDR_OFFSET_SIZE_B
#define BLIS_POOL_ADDR_OFFSET_SIZE_B     0
#endif

#ifndef BLIS_POOL_ADDR_OFFSET_SIZE_C
#define BLIS_POOL_ADDR_OFFSET_SIZE_C     0
#endif

#ifndef BLIS_POOL_ADDR_OFFSET_SIZE_GEN
#define BLIS_POOL_ADDR_OFFSET_SIZE_GEN   0
#endif


// -- MR and NR blocksizes (only for reference kernels) ------------------------

// The build system defines BLIS_IN_REF_KERNEL, but only when compiling
// reference kernels. By using compile-time constants for MR and NR, the
// compiler can perform certain optimizations, such as unrolling and
// vectorization, that would not be otherwise be possible.
#ifdef BLIS_IN_REF_KERNEL

#ifndef BLIS_MR_s
#define BLIS_MR_s 4
#endif

#ifndef BLIS_MR_d
#define BLIS_MR_d 4
#endif

#ifndef BLIS_MR_c
#define BLIS_MR_c 4
#endif

#ifndef BLIS_MR_z
#define BLIS_MR_z 4
#endif

#ifndef BLIS_NR_s
#define BLIS_NR_s 16
#endif

#ifndef BLIS_NR_d
#define BLIS_NR_d 8
#endif

#ifndef BLIS_NR_c
#define BLIS_NR_c 8
#endif

#ifndef BLIS_NR_z
#define BLIS_NR_z 4
#endif

#ifndef BLIS_BBM_s
#define BLIS_BBM_s 1
#endif

#ifndef BLIS_BBM_d
#define BLIS_BBM_d 1
#endif

#ifndef BLIS_BBM_c
#define BLIS_BBM_c 1
#endif

#ifndef BLIS_BBM_z
#define BLIS_BBM_z 1
#endif

#ifndef BLIS_BBN_s
#define BLIS_BBN_s 1
#endif

#ifndef BLIS_BBN_d
#define BLIS_BBN_d 1
#endif

#ifndef BLIS_BBN_c
#define BLIS_BBN_c 1
#endif

#ifndef BLIS_BBN_z
#define BLIS_BBN_z 1
#endif

#ifndef BLIS_PACKMR_s
#define BLIS_PACKMR_s (BLIS_MR_s*BLIS_BBM_s)
#endif

#ifndef BLIS_PACKMR_d
#define BLIS_PACKMR_d (BLIS_MR_d*BLIS_BBM_d)
#endif

#ifndef BLIS_PACKMR_c
#define BLIS_PACKMR_c (BLIS_MR_c*BLIS_BBM_c)
#endif

#ifndef BLIS_PACKMR_z
#define BLIS_PACKMR_z (BLIS_MR_z*BLIS_BBM_z)
#endif

#ifndef BLIS_PACKNR_s
#define BLIS_PACKNR_s (BLIS_NR_s*BLIS_BBN_s)
#endif

#ifndef BLIS_PACKNR_d
#define BLIS_PACKNR_d (BLIS_NR_d*BLIS_BBN_d)
#endif

#ifndef BLIS_PACKNR_c
#define BLIS_PACKNR_c (BLIS_NR_c*BLIS_BBN_c)
#endif

#ifndef BLIS_PACKNR_z
#define BLIS_PACKNR_z (BLIS_NR_z*BLIS_BBN_z)
#endif

#endif

#endif

// end bli_kernel_macro_defs.h
#line 101 "../blis-2.0/frame/include/blis.h"



// -- Threading definitions --

// begin bli_thread.h
#line 1 "../blis-2.0/frame/thread//bli_thread.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2016, Hewlett Packard Enterprise Development LP
   Copyright (C) 2018 - 2019, Advanced Micro Devices, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_THREAD_H
#define BLIS_THREAD_H

// Include thread communicator (thrcomm_t) object definitions and prototypes.
// begin bli_thrcomm.h
#line 1 "../blis-2.0/frame/thread//bli_thrcomm.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2018 - 2019, Advanced Micro Devices, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_THRCOMM_H
#define BLIS_THRCOMM_H

// Define barrier_t, which is specific to the tree barrier in the OpenMP
// implementation. This needs to be done first since it is (potentially)
// used within the definition of thrcomm_t below.

#ifdef BLIS_ENABLE_OPENMP
#ifdef BLIS_TREE_BARRIER
struct barrier_s
{
	int               arity;
	struct barrier_s* dad;

	// We insert a cache line of padding here to eliminate false sharing between
	// the fields above and fields below.
	char   padding1[ BLIS_CACHE_LINE_SIZE ];

	dim_t             count;

	// We insert a cache line of padding here to eliminate false sharing between
	// the fields above and fields below.
	char   padding2[ BLIS_CACHE_LINE_SIZE ];

	gint_t            signal;

	// We insert a cache line of padding here to eliminate false sharing between
	// this struct and the next one.
	char   padding3[ BLIS_CACHE_LINE_SIZE ];
};
typedef struct barrier_s barrier_t;
#endif
#endif

// Define hpx_barrier_t, which is specific to the barrier used in HPX
// implementation. This needs to be done first since it is (potentially)
// used within the definition of thrcomm_t below.

#ifdef BLIS_ENABLE_HPX
typedef struct hpx_barrier_t
{
	void* handle;
} hpx_barrier_t;
#endif

// Define the thrcomm_t structure, which will be common to all threading
// implementations.

typedef struct thrcomm_s
{
	// -- Fields common to all threading implementations --

	void*       sent_object;
	dim_t       n_threads;
	timpl_t     ti;

	// We insert a cache line of padding here to eliminate false sharing between
	// the fields above and fields below.
	char   padding1[ BLIS_CACHE_LINE_SIZE ];

	// NOTE: barrier_sense was originally a gint_t-based bool_t, but upon
	// redefining bool_t as bool we discovered that some gcc __atomic built-ins
	// don't allow the use of bool for the variables being operated upon.
	// (Specifically, this was observed of __atomic_fetch_xor(), but it likely
	// applies to all other related built-ins.) Thus, we get around this by
	// redefining barrier_sense as a gint_t.
	//volatile gint_t  barrier_sense;
	gint_t barrier_sense;

	// We insert a cache line of padding here to eliminate false sharing between
	// the fields above and fields below.
	char   padding2[ BLIS_CACHE_LINE_SIZE ];

	dim_t  barrier_threads_arrived;

	// We insert a cache line of padding here to eliminate false sharing between
	// the fields above and whatever data structures follow.
	char   padding3[ BLIS_CACHE_LINE_SIZE ];

	// -- Fields specific to OpenMP --

	#ifdef BLIS_ENABLE_OPENMP
	#ifdef BLIS_TREE_BARRIER
	// This field is only needed if the tree barrier implementation is being
	// compiled. The non-tree barrier code does not use it.
	barrier_t** barriers;
	#endif
	#endif

	// -- Fields specific to pthreads --

	#ifdef BLIS_ENABLE_PTHREADS
	#ifdef BLIS_USE_PTHREAD_BARRIER
	// This field is only needed if the pthread_barrier_t implementation is
	// being compiled. The non-pthread_barrier_t code does not use it.
	bli_pthread_barrier_t barrier;
	#endif
	#endif

	// -- Fields specific to HPX --

	#ifdef BLIS_ENABLE_HPX
	hpx_barrier_t barrier;
	#endif

} thrcomm_t;





// Include definitions (mostly thrcomm_t) specific to the method of
// multithreading.
// begin bli_thrcomm_single.h
#line 1 "../blis-2.0/frame/thread//bli_thrcomm_single.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_THRCOMM_SINGLE_H
#define BLIS_THRCOMM_SINGLE_H

// Always define these prototypes since disabling multithreading is always
// an option.

// Sequential-specific function prototypes.
void bli_thrcomm_init_single( dim_t nt, thrcomm_t* comm );
void bli_thrcomm_cleanup_single( thrcomm_t* comm );
void bli_thrcomm_barrier_single( dim_t tid, thrcomm_t* comm );

#endif

// end bli_thrcomm_single.h
#line 150 "../blis-2.0/frame/thread//bli_thrcomm.h"

// begin bli_thrcomm_openmp.h
#line 1 "../blis-2.0/frame/thread//bli_thrcomm_openmp.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2018 - 2019, Advanced Micro Devices, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_THRCOMM_OPENMP_H
#define BLIS_THRCOMM_OPENMP_H

// Define these prototypes for situations when OpenMP multithreading is
// enabled.
#ifdef BLIS_ENABLE_OPENMP

// OpenMP-specific function prototypes.
void bli_thrcomm_init_openmp( dim_t nt, thrcomm_t* comm );
void bli_thrcomm_cleanup_openmp( thrcomm_t* comm );
void bli_thrcomm_barrier_openmp( dim_t tid, thrcomm_t* comm );

// Prototypes specific to the OpenMP tree barrier implementation.
#ifdef BLIS_TREE_BARRIER
barrier_t* bli_thrcomm_tree_barrier_create( int num_threads, int arity, barrier_t** leaves, int leaf_index );
void       bli_thrcomm_tree_barrier_free( barrier_t* barrier );
void       bli_thrcomm_tree_barrier( barrier_t* barack );
#endif

#endif

#endif

// end bli_thrcomm_openmp.h
#line 151 "../blis-2.0/frame/thread//bli_thrcomm.h"

// begin bli_thrcomm_pthreads.h
#line 1 "../blis-2.0/frame/thread//bli_thrcomm_pthreads.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_THRCOMM_PTHREADS_H
#define BLIS_THRCOMM_PTHREADS_H

// Define these prototypes for situations when POSIX multithreading is enabled.
#ifdef BLIS_ENABLE_PTHREADS 

// pthreads-specific function prototypes.
void bli_thrcomm_init_pthreads( dim_t nt, thrcomm_t* comm );
void bli_thrcomm_cleanup_pthreads( thrcomm_t* comm );
void bli_thrcomm_barrier_pthreads( dim_t tid, thrcomm_t* comm );

#endif

#endif

// end bli_thrcomm_pthreads.h
#line 152 "../blis-2.0/frame/thread//bli_thrcomm.h"

// begin bli_thrcomm_hpx.h
#line 1 "../blis-2.0/frame/thread//bli_thrcomm_hpx.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2022 Tactical Computing Laboratories, LLC

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_THRCOMM_HPX_H
#define BLIS_THRCOMM_HPX_H

// Define these prototypes for situations when HPX multithreading is enabled.
#ifdef BLIS_ENABLE_HPX

void bli_thrcomm_init_hpx( dim_t nt, thrcomm_t* comm );
void bli_thrcomm_cleanup_hpx( thrcomm_t* comm );
void bli_thrcomm_barrier_hpx( dim_t tid, thrcomm_t* comm );

#endif

#endif

// end bli_thrcomm_hpx.h
#line 153 "../blis-2.0/frame/thread//bli_thrcomm.h"


// Define a function pointer type for each of the functions that are
// "overloaded" by each method of multithreading.
typedef void (*thrcomm_init_ft)( dim_t nt, thrcomm_t* comm );
typedef void (*thrcomm_cleanup_ft)( thrcomm_t* comm );
typedef void (*thrcomm_barrier_ft)( dim_t tid, thrcomm_t* comm );


// thrcomm_t query (field only)

BLIS_INLINE dim_t bli_thrcomm_num_threads( thrcomm_t* comm )
{
	return comm->n_threads;
}

BLIS_INLINE timpl_t bli_thrcomm_thread_impl( thrcomm_t* comm )
{
	return comm->ti;
}


// Threading method-agnostic function prototypes.
thrcomm_t* bli_thrcomm_create( timpl_t ti, pool_t* sba_pool, dim_t n_threads );
void       bli_thrcomm_free( pool_t* sba_pool, thrcomm_t* comm );

// Threading method-specific function prototypes.
// NOTE: These are the prototypes to the dispatcher functions and thus they
// require the timpl_t as an argument. The threading-specific functions can
// (and do) omit the timpl_t from their function signatures since their
// threading implementation is intrinsically known.
void                   bli_thrcomm_init( timpl_t ti, dim_t n_threads, thrcomm_t* comm );
void                   bli_thrcomm_cleanup( thrcomm_t* comm );
BLIS_EXPORT_BLIS void  bli_thrcomm_barrier( dim_t thread_id, thrcomm_t* comm );

// Other function prototypes.
BLIS_EXPORT_BLIS void* bli_thrcomm_bcast( dim_t inside_id, void* to_send, thrcomm_t* comm );
void                   bli_thrcomm_barrier_atomic( dim_t thread_id, thrcomm_t* comm );

#endif

// end bli_thrcomm.h
#line 42 "../blis-2.0/frame/thread//bli_thread.h"


// Include thread info (thrinfo_t) object definitions and prototypes.
// begin bli_thrinfo.h
#line 1 "../blis-2.0/frame/thread//bli_thrinfo.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2018 - 2019, Advanced Micro Devices, Inc.
   Copyright (C) 2022, Southern Methodist University

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_THRINFO_H
#define BLIS_THRINFO_H

// Thread info structure definition
struct thrinfo_s
{
	// The thread communicator for the other threads sharing the same work
	// at this level.
	thrcomm_t*         comm;

	// Our thread id within the thread communicator.
	dim_t              thread_id;

	// The number of communicators which are "siblings" of our communicator.
	dim_t              n_way;

	// An id to identify what we're working on. This is the same for all threads
	// in the same communicator, and 0 <= work_id < n_way.
	dim_t              work_id;

	// When freeing, should the communicators in this node be freed? Usually,
	// this is field is true, but when nodes are created that share the same
	// communicators as other nodes (such as with packm nodes), this is set
	// to false.
	bool               free_comm;

	// The small block pool.
	pool_t*            sba_pool;

	// The packing block allocator.
	pba_t*             pba;

	// Storage for allocated memory obtained from the packing block allocator.
	mem_t              mem;

	// Child thread info nodes.
	struct thrinfo_s*  sub_nodes[ BLIS_MAX_SUB_NODES ];
};
typedef struct thrinfo_s thrinfo_t;

//
// thrinfo_t functions
//

// thrinfo_t query (field only)

BLIS_INLINE dim_t bli_thrinfo_num_threads( const thrinfo_t* t )
{
	return (t->comm)->n_threads;
}

BLIS_INLINE dim_t bli_thrinfo_thread_id( const thrinfo_t* t )
{
	return t->thread_id;
}

BLIS_INLINE dim_t bli_thrinfo_n_way( const thrinfo_t* t )
{
	return t->n_way;
}

BLIS_INLINE dim_t bli_thrinfo_work_id( const thrinfo_t* t )
{
	return t->work_id;
}

BLIS_INLINE thrcomm_t* bli_thrinfo_comm( const thrinfo_t* t )
{
	return t->comm;
}

BLIS_INLINE bool bli_thrinfo_needs_free_comm( const thrinfo_t* t )
{
	return t->free_comm;
}

BLIS_INLINE pool_t* bli_thrinfo_sba_pool( const thrinfo_t* t )
{
	return t->sba_pool;
}

BLIS_INLINE pba_t* bli_thrinfo_pba( const thrinfo_t* t )
{
	return t->pba;
}

BLIS_INLINE mem_t* bli_thrinfo_mem( thrinfo_t* t )
{
	return &t->mem;
}

BLIS_INLINE thrinfo_t* bli_thrinfo_sub_node( dim_t which, const thrinfo_t* t )
{
	return t->sub_nodes[ which ];
}

// thrinfo_t query (complex)

BLIS_INLINE bool bli_thrinfo_am_chief( const thrinfo_t* t )
{
	return t->thread_id == 0;
}

// thrinfo_t modification

BLIS_INLINE void bli_thrinfo_set_comm( thrcomm_t* comm, thrinfo_t* t )
{
	t->comm = comm;
}

BLIS_INLINE void bli_thrinfo_set_thread_id( dim_t thread_id, thrinfo_t* t )
{
	t->thread_id = thread_id;
}

BLIS_INLINE void bli_thrinfo_set_n_way( dim_t n_way, thrinfo_t* t )
{
	t->n_way = n_way;
}

BLIS_INLINE void bli_thrinfo_set_work_id( dim_t work_id, thrinfo_t* t )
{
	t->work_id = work_id;
}

BLIS_INLINE void bli_thrinfo_set_free_comm( bool free_comm, thrinfo_t* t )
{
	t->free_comm = free_comm;
}

BLIS_INLINE void bli_thrinfo_set_sba_pool( pool_t* sba_pool, thrinfo_t* t )
{
	t->sba_pool = sba_pool;
}

BLIS_INLINE void bli_thrinfo_set_pba( pba_t* pba, thrinfo_t* t )
{
	t->pba = pba;
}

BLIS_INLINE void bli_thrinfo_set_sub_node( dim_t which, thrinfo_t* sub_node, thrinfo_t* t )
{
	t->sub_nodes[ which ] = sub_node;
}

void bli_thrinfo_attach_sub_node( thrinfo_t* sub_node, thrinfo_t* t );

// other thrinfo_t-related functions

BLIS_INLINE void* bli_thrinfo_broadcast( const thrinfo_t* t, void* p )
{
	return bli_thrcomm_bcast( t->thread_id, p, t->comm );
}

BLIS_INLINE void bli_thrinfo_barrier( const thrinfo_t* t )
{
	bli_thrcomm_barrier( t->thread_id, t->comm );
}


//
// Prototypes for level-3 thrinfo functions not specific to any operation.
//

thrinfo_t* bli_thrinfo_create_root
     (
       thrcomm_t* comm,
       dim_t      thread_id,
       pool_t*    sba_pool,
       pba_t*     pba
     );

thrinfo_t* bli_thrinfo_create
     (
       thrcomm_t* comm,
       dim_t      thread_id,
       dim_t      n_way,
       dim_t      work_id,
       bool       free_comm,
       pool_t*    sba_pool,
       pba_t*     pba
     );

BLIS_EXPORT_BLIS void bli_thrinfo_free
     (
       thrinfo_t* thread
     );

// -----------------------------------------------------------------------------

thrinfo_t* bli_thrinfo_split
     (
       dim_t      n_way,
       thrinfo_t* thread_par
     );

void bli_thrinfo_print
     (
       thrinfo_t* thread
     );

void bli_thrinfo_print_sub
     (
       thrinfo_t* thread,
       gint_t     level
     );

#endif
// end bli_thrinfo.h
#line 45 "../blis-2.0/frame/thread//bli_thread.h"


// Thread lanuch prototypes. Must go before including implementation headers.
typedef void (*thread_func_t)( thrcomm_t* gl_comm, dim_t tid, const void* params );

// Include threading implementations.
// begin bli_thread_openmp.h
#line 1 "../blis-2.0/frame/thread//bli_thread_openmp.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_THREAD_OPENMP_H
#define BLIS_THREAD_OPENMP_H

// Definitions specific to situations when OpenMP multithreading is enabled.
#ifdef BLIS_ENABLE_OPENMP

void bli_thread_launch_openmp
     (
             dim_t         nt,
             thread_func_t func,
       const void*         params
     );

#endif

#endif

// end bli_thread_openmp.h
#line 51 "../blis-2.0/frame/thread//bli_thread.h"

// begin bli_thread_pthreads.h
#line 1 "../blis-2.0/frame/thread//bli_thread_pthreads.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_THREAD_PTHREADS_H
#define BLIS_THREAD_PTHREADS_H

// Definitions specific to situations when POSIX multithreading is enabled.
#ifdef BLIS_ENABLE_PTHREADS

void bli_thread_launch_pthreads
     (
             dim_t         nt,
             thread_func_t func,
       const void*         params
     );

#endif

#endif

// end bli_thread_pthreads.h
#line 52 "../blis-2.0/frame/thread//bli_thread.h"

// begin bli_thread_hpx.h
#line 1 "../blis-2.0/frame/thread//bli_thread_hpx.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2022 Tactical Computing Laboratories, LLC

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_THREAD_HPX_H
#define BLIS_THREAD_HPX_H

// Definitions specific to situations when HPX multithreading is enabled.
#ifdef BLIS_ENABLE_HPX

void bli_thread_launch_hpx
     (
             dim_t         nt,
             thread_func_t func,
       const void*         params
     );

void bli_thread_initialize_hpx( int argc, char** argv );

int bli_thread_finalize_hpx();

#endif

#endif
// end bli_thread_hpx.h
#line 53 "../blis-2.0/frame/thread//bli_thread.h"

// begin bli_thread_single.h
#line 1 "../blis-2.0/frame/thread//bli_thread_single.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2023, Southern Methodist University

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_THREAD_SINGLE_H
#define BLIS_THREAD_SINGLE_H

void bli_thread_launch_single
     (
             dim_t         nt,
             thread_func_t func,
       const void*         params
     );

#endif

// end bli_thread_single.h
#line 54 "../blis-2.0/frame/thread//bli_thread.h"


// Initialization-related prototypes.
int bli_thread_init( void );
int bli_thread_finalize( void );

// -----------------------------------------------------------------------------

BLIS_EXPORT_BLIS void bli_thread_launch
     (
             timpl_t       ti,
             dim_t         nt,
             thread_func_t func,
       const void*         params
     );

// -----------------------------------------------------------------------------

// Factorization and partitioning prototypes
typedef struct
{
    dim_t n;
    dim_t sqrt_n;
    dim_t f;
} bli_prime_factors_t;

void bli_prime_factorization( dim_t n, bli_prime_factors_t* factors );

dim_t bli_next_prime_factor( bli_prime_factors_t* factors );
bool  bli_is_prime( dim_t n );

void bli_thread_partition_2x2
     (
       dim_t           n_thread,
       dim_t           work1,
       dim_t           work2,
       dim_t* restrict nt1,
       dim_t* restrict nt2
     );
void bli_thread_partition_2x2_slow
     (
       dim_t           n_thread,
       dim_t           work1,
       dim_t           work2,
       dim_t* restrict nt1,
       dim_t* restrict nt2
     );
void bli_thread_partition_2x2_fast
     (
       dim_t           n_thread,
       dim_t           work1,
       dim_t           work2,
       dim_t* restrict nt1,
       dim_t* restrict nt2
     );

// -----------------------------------------------------------------------------

dim_t bli_gcd( dim_t x, dim_t y );
dim_t bli_lcm( dim_t x, dim_t y );
dim_t bli_ipow( dim_t base, dim_t power );

// -----------------------------------------------------------------------------

BLIS_EXPORT_BLIS dim_t   bli_thread_get_jc_nt( void );
BLIS_EXPORT_BLIS dim_t   bli_thread_get_pc_nt( void );
BLIS_EXPORT_BLIS dim_t   bli_thread_get_ic_nt( void );
BLIS_EXPORT_BLIS dim_t   bli_thread_get_jr_nt( void );
BLIS_EXPORT_BLIS dim_t   bli_thread_get_ir_nt( void );
BLIS_EXPORT_BLIS dim_t   bli_thread_get_num_threads( void );
BLIS_EXPORT_BLIS timpl_t bli_thread_get_thread_impl( void );
BLIS_EXPORT_BLIS const char* bli_thread_get_thread_impl_str( timpl_t ti );

BLIS_EXPORT_BLIS void    bli_thread_set_ways( dim_t jc, dim_t pc, dim_t ic, dim_t jr, dim_t ir );
BLIS_EXPORT_BLIS void    bli_thread_set_num_threads( dim_t value );
BLIS_EXPORT_BLIS void    bli_thread_set_thread_impl( timpl_t ti );
BLIS_EXPORT_BLIS void    bli_thread_reset( void );


#endif
// end bli_thread.h
#line 106 "../blis-2.0/frame/include/blis.h"

// begin bli_thread_range.h
#line 1 "../blis-2.0/frame/thread//bli_thread_range.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2016, Hewlett Packard Enterprise Development LP
   Copyright (C) 2018 - 2019, Advanced Micro Devices, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_THREAD_RANGE_H
#define BLIS_THREAD_RANGE_H

// Thread range-related prototypes.

BLIS_EXPORT_BLIS void bli_thread_range_sub
     (
       dim_t  work_id,
       dim_t  n_way,
       dim_t  n,
       dim_t  bf,
       bool   handle_edge_low,
       dim_t* start,
       dim_t* end
     );

#undef  GENPROT
#define GENPROT( opname ) \
\
siz_t PASTEMAC( opname ) \
     ( \
             dir_t      direct, \
             dim_t      bmult, \
             bool       use_weighted, \
       const thrinfo_t* thr, \
       const obj_t*     a, \
       const obj_t*     b, \
       const obj_t*     c, \
             dim_t*     start, \
             dim_t*     end \
     );

GENPROT( thread_range_mdim )
GENPROT( thread_range_ndim )


BLIS_EXPORT_BLIS siz_t bli_thread_range
     (
       const thrinfo_t* thr,
       const obj_t*     a,
             dim_t      bf,
             dir_t      direct,
             mdim_t     dim,
             bool       use_weighted,
             dim_t*     start,
             dim_t*     end
     );


dim_t bli_thread_range_width_l
     (
       doff_t diagoff_j,
       dim_t  m,
       dim_t  n_j,
       dim_t  j,
       dim_t  n_way,
       dim_t  bf,
       dim_t  bf_left,
       double area_per_thr,
       bool   handle_edge_low
     );
siz_t bli_find_area_trap_l
     (
       doff_t diagoff,
       dim_t  m,
       dim_t  n,
       dim_t  bf
     );

siz_t bli_thread_range_weighted_sub
     (
       const thrinfo_t* thread,
             doff_t     diagoff,
             uplo_t     uplo,
             uplo_t     uplo_orig,
             dim_t      m,
             dim_t      n,
             dim_t      bf,
             bool       handle_edge_low,
             dim_t*     j_start_thr,
             dim_t*     j_end_thr
     );

#endif
// end bli_thread_range.h
#line 107 "../blis-2.0/frame/include/blis.h"

// begin bli_thread_range_slab_rr.h
#line 1 "../blis-2.0/frame/thread//bli_thread_range_slab_rr.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_THREAD_RANGE_SLAB_RR_H
#define BLIS_THREAD_RANGE_SLAB_RR_H

BLIS_INLINE void bli_thread_range_rr
     (
       dim_t  tid,
       dim_t  nt,
       dim_t  n,
       dim_t  bf,
       bool   handle_edge_low,
       dim_t* start,
       dim_t* end,
       dim_t* inc
     )
{
	const dim_t n_iter = n / bf + ( n % bf ? 1 : 0 );

	// Use round-robin (interleaved) partitioning of jr/ir loops.
	*start = tid;
	*end   = n_iter;
	*inc   = nt;
}

BLIS_INLINE void bli_thread_range_sl
     (
       dim_t  work_id,
       dim_t  n_way,
       dim_t  n,
       dim_t  bf,
       bool   handle_edge_low,
       dim_t* start,
       dim_t* end,
       dim_t* inc
     )
{
	// Use contiguous slab partitioning of jr/ir loops.
	bli_thread_range_sub( work_id, n_way, n, bf, handle_edge_low, start, end );
	*inc = 1;
}

BLIS_INLINE void bli_thread_range_slrr
     (
       dim_t  work_id,
       dim_t  n_way,
       dim_t  n,
       dim_t  bf,
       bool   handle_edge_low,
       dim_t* start,
       dim_t* end,
       dim_t* inc
     )
{
	// Define a general-purpose slab/rr function whose definition depends on
	// whether slab or round-robin partitioning was requested at configure-time.
	// Note that this function also uses the slab code path when tlb is enabled.
	// If this is ever changed, make sure to change bli_is_my_iter() since they
	// are used together by packm.

#ifdef BLIS_ENABLE_JRIR_RR
	bli_thread_range_rr( work_id, n_way, bf, handle_edge_low, start, end, inc );
#else // ifdef ( _SLAB || _TLB )
	bli_thread_range_sl( work_id, n_way, n, bf, handle_edge_low, start, end, inc );
#endif
}

// -----------------------------------------------------------------------------

void bli_thread_range_quad
     (
       const thrinfo_t* thread,
             doff_t     diagoff,
             uplo_t     uplo,
             dim_t      m,
             dim_t      n,
             dim_t      bf,
             bool       handle_edge_low,
             dim_t*     start,
             dim_t*     end,
             dim_t*     inc
     );

#endif

// end bli_thread_range_slab_rr.h
#line 108 "../blis-2.0/frame/include/blis.h"

// begin bli_thread_range_tlb.h
#line 1 "../blis-2.0/frame/thread//bli_thread_range_tlb.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2022, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_THREAD_RANGE_TLB_H
#define BLIS_THREAD_RANGE_TLB_H

#if 0
dim_t bli_thread_range_tlb
     (
       const dim_t  nt,
       const dim_t  tid,
       const doff_t diagoff,
       const uplo_t uplo,
       const dim_t  m_iter,
       const dim_t  n_iter,
       const dim_t  mr,
       const dim_t  nr,
             inc_t* j_st_p,
             inc_t* i_st_p
     );
#endif
dim_t bli_thread_range_tlb_l
     (
       const dim_t  nt,
       const dim_t  tid,
       const doff_t diagoff,
       const dim_t  m_iter,
       const dim_t  n_iter,
       const dim_t  mr,
       const dim_t  nr,
             inc_t* j_st_p,
             inc_t* i_st_p
     );
dim_t bli_thread_range_tlb_u
     (
       const dim_t  nt,
       const dim_t  tid,
       const doff_t diagoff,
       const dim_t  m_iter,
       const dim_t  n_iter,
       const dim_t  mr,
       const dim_t  nr,
             inc_t* j_st_p,
             inc_t* i_st_p
     );
dim_t bli_thread_range_tlb_d
     (
       const dim_t  nt,
       const dim_t  tid,
       const dim_t  m_iter,
       const dim_t  n_iter,
       const dim_t  mr,
       const dim_t  nr,
             inc_t* j_st_p,
             inc_t* i_st_p
     );

// ---

dim_t bli_thread_range_tlb_trmm_ll
     (
       const dim_t  nt,
       const dim_t  tid,
       const doff_t diagoff,
       const dim_t  m_iter,
       const dim_t  n_iter,
       const dim_t  k_iter,
       const dim_t  mr,
       const dim_t  nr,
             inc_t* j_st_p,
             inc_t* i_st_p
     );
dim_t bli_thread_range_tlb_trmm_lu
     (
       const dim_t  nt,
       const dim_t  tid,
       const doff_t diagoff,
       const dim_t  m_iter,
       const dim_t  n_iter,
       const dim_t  k_iter,
       const dim_t  mr,
       const dim_t  nr,
             inc_t* j_st_p,
             inc_t* i_st_p
     );
dim_t bli_thread_range_tlb_trmm_lx_impl
     (
       const dim_t  nt,
       const dim_t  tid,
       const doff_t diagoff,
       const uplo_t uplo,
       const dim_t  m_iter,
       const dim_t  n_iter,
       const dim_t  k_iter,
       const dim_t  mr,
       const dim_t  nr,
             inc_t* j_st_p,
             inc_t* i_st_p
     );
#if 0
dim_t bli_thread_range_tlb_trmm_r
     (
       const dim_t  nt,
       const dim_t  tid,
       const doff_t diagoff,
       const uplo_t uplo,
       const dim_t  m_iter,
       const dim_t  n_iter,
       const dim_t  k_iter,
       const dim_t  mr,
       const dim_t  nr,
             inc_t* j_st_p,
             inc_t* i_st_p
     );
#endif

// ---

dim_t bli_thread_range_tlb_trmm_rl
     (
       const dim_t  nt,
       const dim_t  tid,
       const doff_t diagoff,
       const dim_t  m_iter,
       const dim_t  n_iter,
       const dim_t  k_iter,
       const dim_t  mr,
       const dim_t  nr,
             inc_t* j_st_p,
             inc_t* i_st_p
     );
dim_t bli_thread_range_tlb_trmm_ru
     (
       const dim_t  nt,
       const dim_t  tid,
       const doff_t diagoff,
       const dim_t  m_iter,
       const dim_t  n_iter,
       const dim_t  k_iter,
       const dim_t  mr,
       const dim_t  nr,
             inc_t* j_st_p,
             inc_t* i_st_p
     );
dim_t bli_thread_range_tlb_trmm_rl_impl
     (
       const dim_t  nt,
       const dim_t  tid,
       const doff_t diagoff,
       const dim_t  m_iter,
       const dim_t  n_iter,
       const dim_t  k_iter,
       const dim_t  mr,
       const dim_t  nr,
             inc_t* j_st_p,
             inc_t* i_st_p,
             inc_t* j_en_p,
             inc_t* i_en_p
     );

#endif
// end bli_thread_range_tlb.h
#line 109 "../blis-2.0/frame/include/blis.h"


// begin bli_pthread.h
#line 1 "../blis-2.0/frame/thread//bli_pthread.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2018, Southern Methodist University
   Copyright (C) 2018, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_PTHREAD_H
#define BLIS_PTHREAD_H

// -- Type and macro definitions -----------------------------------------------

#if defined(BLIS_DISABLE_SYSTEM)

// This branch defines a pthread-like API, bli_pthread_*(), and implements it
// in terms of "dummy" code that doesn't depend on POSIX threads or any other
// threading mechanism. See issue #454 to see the use case that prompted this
// feature.
// NOTE: THIS CODE DOES NOT IMPLEMENT THREADING AND IS NOT THREAD-SAFE!

// -- pthread types --

typedef int bli_pthread_t;
typedef int bli_pthread_attr_t;
typedef int bli_pthread_mutex_t;
typedef int bli_pthread_mutexattr_t;
typedef int bli_pthread_cond_t;
typedef int bli_pthread_condattr_t;
typedef int bli_pthread_once_t;

typedef int bli_pthread_barrier_t;
typedef int bli_pthread_barrierattr_t;

// -- pthreads macros --

#define BLIS_PTHREAD_MUTEX_INITIALIZER 0
#define BLIS_PTHREAD_COND_INITIALIZER  0
#define BLIS_PTHREAD_ONCE_INIT         0

#elif defined(_MSC_VER) // !defined(BLIS_DISABLE_SYSTEM)

// This branch defines a pthread-like API, bli_pthread_*(), and implements it
// in terms of Windows API calls.

// -- pthread types --

typedef struct
{
    HANDLE handle;
    void* retval;
} bli_pthread_t;
typedef void bli_pthread_attr_t;
typedef SRWLOCK bli_pthread_mutex_t;
typedef void bli_pthread_mutexattr_t;
typedef CONDITION_VARIABLE bli_pthread_cond_t;
typedef void bli_pthread_condattr_t;
typedef INIT_ONCE bli_pthread_once_t;
typedef struct
{
    bli_pthread_mutex_t mutex;
    bli_pthread_cond_t  cond;
    int                 count;
    int                 tripCount;
} bli_pthread_barrier_t;
typedef void bli_pthread_barrierattr_t;

// -- pthreads macros --

#define BLIS_PTHREAD_MUTEX_INITIALIZER SRWLOCK_INIT
#define BLIS_PTHREAD_ONCE_INIT         INIT_ONCE_STATIC_INIT
#define BLIS_PTHREAD_COND_INITIALIZER  CONDITION_VARIABLE_INIT

#else // !defined(BLIS_DISABLE_SYSTEM) && !defined(_MSC_VER)

#include <pthread.h> // skipped

// This branch defines a pthreads-like API, bli_pthreads_*(), and implements it
// in terms of the corresponding pthreads_*() types, macros, and function calls. 

// -- pthread types --

typedef pthread_t           bli_pthread_t;
typedef pthread_attr_t      bli_pthread_attr_t;
typedef pthread_mutex_t     bli_pthread_mutex_t;
typedef pthread_mutexattr_t bli_pthread_mutexattr_t;
typedef pthread_cond_t      bli_pthread_cond_t;
typedef pthread_condattr_t  bli_pthread_condattr_t;
typedef pthread_once_t      bli_pthread_once_t;

#if defined(__APPLE__)

// For OS X, we must define the barrier types ourselves since Apple does
// not implement barriers in their variant of pthreads.

typedef void bli_pthread_barrierattr_t;

typedef struct
{
	bli_pthread_mutex_t mutex;
	bli_pthread_cond_t  cond;
	int                 count;
	int                 tripCount;
} bli_pthread_barrier_t;

#else

// For other non-Windows OSes (primarily Linux), we can define the barrier
// types in terms of existing pthreads barrier types since we expect they
// will be provided by the pthreads implementation.

typedef pthread_barrier_t      bli_pthread_barrier_t;
typedef pthread_barrierattr_t  bli_pthread_barrierattr_t;

#endif

// -- pthreads macros --

#define BLIS_PTHREAD_MUTEX_INITIALIZER PTHREAD_MUTEX_INITIALIZER
#define BLIS_PTHREAD_COND_INITIALIZER  PTHREAD_COND_INITIALIZER
#define BLIS_PTHREAD_ONCE_INIT         PTHREAD_ONCE_INIT

#endif

// -- Function definitions -----------------------------------------------------

// -- pthread_create(), pthread_join() --

BLIS_EXPORT_BLIS int bli_pthread_create
     (
       bli_pthread_t*            thread,
       const bli_pthread_attr_t* attr,
       void*                   (*start_routine)(void*),
       void*                     arg
     );

BLIS_EXPORT_BLIS int bli_pthread_join
     (
       bli_pthread_t thread,
       void**        retval
     );

// -- pthread_mutex_*() --

BLIS_EXPORT_BLIS int bli_pthread_mutex_init
     (
       bli_pthread_mutex_t*           mutex,
       const bli_pthread_mutexattr_t* attr
     );

BLIS_EXPORT_BLIS int bli_pthread_mutex_destroy
     (
       bli_pthread_mutex_t* mutex
     );

BLIS_EXPORT_BLIS int bli_pthread_mutex_lock
     (
       bli_pthread_mutex_t* mutex
     );

BLIS_EXPORT_BLIS int bli_pthread_mutex_trylock
     (
       bli_pthread_mutex_t* mutex
     );

BLIS_EXPORT_BLIS int bli_pthread_mutex_unlock
     (
       bli_pthread_mutex_t* mutex
     );

// -- pthread_cond_*() --

BLIS_EXPORT_BLIS int bli_pthread_cond_init
     (
       bli_pthread_cond_t*           cond,
       const bli_pthread_condattr_t* attr
     );

BLIS_EXPORT_BLIS int bli_pthread_cond_destroy
     (
       bli_pthread_cond_t* cond
     );

BLIS_EXPORT_BLIS int bli_pthread_cond_wait
     (
       bli_pthread_cond_t*  cond,
       bli_pthread_mutex_t* mutex
     );

BLIS_EXPORT_BLIS int bli_pthread_cond_broadcast
     (
       bli_pthread_cond_t* cond
     );

// -- pthread_once() --

BLIS_EXPORT_BLIS void bli_pthread_once
     (
       bli_pthread_once_t* once,
       void              (*init)(void)
     );

#if 0
// NOTE: This part of the API is disabled because (1) we don't actually need
// _self() or _equal() yet, and (2) when we do try to include these functions,
// AppVeyor for some reason fails on all the Windows/clang builds with the
// error:
//    libblis.a(bli_pthread.o) : error LNK2019: unresolved external symbol
//     __imp_CompareObjectHandles referenced in function bli_pthread_equal

// -- pthread_self() --

BLIS_EXPORT_BLIS bli_pthread_t bli_pthread_self
     (
       void
     );

// -- pthread_equal() --

BLIS_EXPORT_BLIS int bli_pthread_equal
     (
       bli_pthread_t t1,
       bli_pthread_t t2
     );
#endif

// -- pthread_barrier_*() --

BLIS_EXPORT_BLIS int bli_pthread_barrier_init
     (
       bli_pthread_barrier_t*           barrier,
       const bli_pthread_barrierattr_t* attr,
       unsigned int                     count
     );

BLIS_EXPORT_BLIS int bli_pthread_barrier_destroy
     (
       bli_pthread_barrier_t* barrier
     );

BLIS_EXPORT_BLIS int bli_pthread_barrier_wait
     (
       bli_pthread_barrier_t* barrier
     );

// -- Non-standard extensions --------------------------------------------------

// -- pthread_switch --

typedef struct
{
    int                 status;
    bli_pthread_mutex_t mutex;
} bli_pthread_switch_t;

#define BLIS_PTHREAD_SWITCH_INIT { .status = 0, \
                                   .mutex  = BLIS_PTHREAD_MUTEX_INITIALIZER }

int bli_pthread_switch_on
     (
       bli_pthread_switch_t* sw,
       int                 (*init)(void)
     );

int bli_pthread_switch_off
     (
       bli_pthread_switch_t* sw,
       int                 (*deinit)(void)
     );

#endif // BLIS_PTHREAD_H
// end bli_pthread.h
#line 111 "../blis-2.0/frame/include/blis.h"



// -- Constant definitions --

// begin bli_extern_defs.h
#line 1 "../blis-2.0/frame/include//bli_extern_defs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_EXTERN_DEFS_H
#define BLIS_EXTERN_DEFS_H

BLIS_EXPORT_BLIS extern const obj_t BLIS_TWO;
BLIS_EXPORT_BLIS extern const obj_t BLIS_ONE;
//BLIS_EXPORT_BLIS extern obj_t BLIS_ONE_HALF;
BLIS_EXPORT_BLIS extern const obj_t BLIS_ZERO;
//BLIS_EXPORT_BLIS extern obj_t BLIS_MINUS_ONE_HALF;
BLIS_EXPORT_BLIS extern const obj_t BLIS_MINUS_ONE;
BLIS_EXPORT_BLIS extern const obj_t BLIS_MINUS_TWO;
BLIS_EXPORT_BLIS extern const obj_t BLIS_ONE_I;
BLIS_EXPORT_BLIS extern const obj_t BLIS_MINUS_ONE_I;
BLIS_EXPORT_BLIS extern const obj_t BLIS_NAN;

BLIS_EXPORT_BLIS extern thrcomm_t BLIS_SINGLE_COMM;

#endif
// end bli_extern_defs.h
#line 116 "../blis-2.0/frame/include/blis.h"



// -- Base operation prototypes --

// begin bli_init.h
#line 1 "../blis-2.0/frame/base//bli_init.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

BLIS_EXPORT_BLIS void bli_init( void );
BLIS_EXPORT_BLIS void bli_finalize( void );

void bli_init_auto( void );
void bli_finalize_auto( void );

void bli_init_once( void );
void bli_finalize_once( void );

int  bli_init_apis( void );
int  bli_finalize_apis( void );

// end bli_init.h
#line 121 "../blis-2.0/frame/include/blis.h"

// begin bli_malloc.h
#line 1 "../blis-2.0/frame/base//bli_malloc.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2018 - 2019, Advanced Micro Devices, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

// Typedef function pointer types for malloc() and free() substitutes.
//typedef void* (*malloc_ft) ( size_t size );
//typedef void  (*free_ft)   ( void*  p    );

// -----------------------------------------------------------------------------

#if 0
BLIS_EXPORT_BLIS void* bli_malloc_pool( size_t size );
BLIS_EXPORT_BLIS void   bli_free_pool( void* p );
#endif

void* bli_malloc_intl( size_t size, err_t* r_val );
void* bli_calloc_intl( size_t size, err_t* r_val );
void  bli_free_intl( void* p );

BLIS_EXPORT_BLIS void* bli_malloc_user( size_t size, err_t* r_val );
BLIS_EXPORT_BLIS void  bli_free_user( void* p );

// -----------------------------------------------------------------------------

void* bli_fmalloc_align( malloc_ft f, size_t size, size_t align_size, err_t* r_val );
void  bli_ffree_align( free_ft f, void* p );

void* bli_fmalloc_noalign( malloc_ft f, size_t size, err_t* r_val );
void  bli_ffree_noalign( free_ft f, void* p );

void  bli_fmalloc_align_check( malloc_ft f, size_t size, size_t align_size );
void  bli_fmalloc_post_check( void* p );

// end bli_malloc.h
#line 122 "../blis-2.0/frame/include/blis.h"

// begin bli_const.h
#line 1 "../blis-2.0/frame/base//bli_const.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

void bli_const_init( void );
void bli_const_finalize( void );

// end bli_const.h
#line 123 "../blis-2.0/frame/include/blis.h"

// begin bli_obj.h
#line 1 "../blis-2.0/frame/base//bli_obj.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

// begin bli_obj_check.h
#line 1 "../blis-2.0/frame/base/check//bli_obj_check.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

void bli_obj_create_check(       num_t  dt,
                                 dim_t  m,
                                 dim_t  n,
                                 inc_t  rs,
                                 inc_t  cs,
                           const obj_t* obj );

void bli_obj_create_without_buffer_check(       num_t  dt,
                                                dim_t  m,
                                                dim_t  n,
                                          const obj_t* obj );

void bli_obj_alloc_buffer_check(       inc_t  rs,
                                       inc_t  cs,
                                       inc_t  is,
                                 const obj_t* obj );

void bli_obj_attach_buffer_check( const void*  p,
                                        inc_t  rs,
                                        inc_t  cs,
                                        inc_t  is,
                                  const obj_t* obj );

void bli_obj_create_scalar_check( num_t dt, const obj_t* obj );

void bli_obj_free_check( const obj_t* obj );

void bli_obj_create_const_check( double value, const obj_t* obj );

void bli_obj_create_const_copy_of_check( const obj_t* a, const obj_t* b );

void bli_dt_size_check( num_t dt );

void bli_dt_string_check( num_t dt );

void bli_dt_union_check( num_t dt1, num_t dt2 );

void bli_obj_print_check( const char* label, const obj_t* obj );

// end bli_obj_check.h
#line 36 "../blis-2.0/frame/base//bli_obj.h"


BLIS_EXPORT_BLIS void bli_obj_create
     (
       num_t  dt,
       dim_t  m,
       dim_t  n,
       inc_t  rs,
       inc_t  cs,
       obj_t* obj
     );

BLIS_EXPORT_BLIS void bli_obj_create_with_attached_buffer
     (
       num_t  dt,
       dim_t  m,
       dim_t  n,
       void*  p,
       inc_t  rs,
       inc_t  cs,
       obj_t* obj
     );

BLIS_EXPORT_BLIS void bli_obj_create_without_buffer
     (
       num_t  dt,
       dim_t  m,
       dim_t  n,
       obj_t* obj
     );

BLIS_EXPORT_BLIS void bli_obj_alloc_buffer
     (
       inc_t  rs,
       inc_t  cs,
       inc_t  is,
       obj_t* obj
     );

BLIS_EXPORT_BLIS void bli_obj_attach_buffer
     (
       void*  p,
       inc_t  rs,
       inc_t  cs,
       inc_t  is,
       obj_t* obj
     );

BLIS_EXPORT_BLIS void bli_obj_create_1x1
     (
       num_t  dt,
       obj_t* obj
     );

BLIS_EXPORT_BLIS void bli_obj_create_1x1_with_attached_buffer
     (
       num_t  dt,
       void*  p,
       obj_t* obj
     );

BLIS_EXPORT_BLIS void bli_obj_create_conf_to
     (
       const obj_t* s,
             obj_t* d
     );

BLIS_EXPORT_BLIS void bli_obj_free
     (
       obj_t* obj
     );

void bli_adjust_strides
     (
       dim_t  m,
       dim_t  n,
       siz_t  elem_size,
       inc_t* rs,
       inc_t* cs,
       inc_t* is
     );

BLIS_EXPORT_BLIS siz_t bli_dt_size
     (
       num_t dt
     );

BLIS_EXPORT_BLIS const char* bli_dt_string
     (
       num_t dt
     );

BLIS_EXPORT_BLIS dim_t bli_align_dim_to_mult
     (
       dim_t dim,
       dim_t dim_mult,
       bool  round_up
     );

BLIS_EXPORT_BLIS dim_t bli_align_dim_to_size
     (
       dim_t dim,
       siz_t elem_size,
       siz_t align_size
     );

BLIS_EXPORT_BLIS dim_t bli_align_ptr_to_size
     (
       const void*  p,
             size_t align_size
     );

BLIS_EXPORT_BLIS void bli_obj_print
     (
       const char*  label,
       const obj_t* obj
     );

// end bli_obj.h
#line 124 "../blis-2.0/frame/include/blis.h"

// begin bli_obj_scalar.h
#line 1 "../blis-2.0/frame/base//bli_obj_scalar.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

BLIS_EXPORT_BLIS void bli_obj_scalar_init_detached
     (
       num_t  dt,
       obj_t* beta
     );

BLIS_EXPORT_BLIS void bli_obj_scalar_init_detached_copy_of
     (
             num_t  dt,
             conj_t conj,
       const obj_t* alpha,
             obj_t* beta
     );

BLIS_EXPORT_BLIS void bli_obj_scalar_detach
     (
       const obj_t* a,
             obj_t* alpha
     );

BLIS_EXPORT_BLIS void bli_obj_scalar_attach
     (
             conj_t conj,
       const obj_t* alpha,
             obj_t* a
     );

BLIS_EXPORT_BLIS void bli_obj_scalar_cast_to
     (
       num_t  dt,
       obj_t* a
     );

BLIS_EXPORT_BLIS void bli_obj_scalar_apply_scalar
     (
       const obj_t* alpha,
             obj_t* a
     );

BLIS_EXPORT_BLIS void bli_obj_scalar_reset
     (
       obj_t* a
     );

BLIS_EXPORT_BLIS bool bli_obj_scalar_has_nonzero_imag
     (
       obj_t* a
     );

BLIS_EXPORT_BLIS bool bli_obj_scalar_equals
     (
       const obj_t* a,
       const obj_t* beta
     );

// end bli_obj_scalar.h
#line 125 "../blis-2.0/frame/include/blis.h"

// begin bli_blksz.h
#line 1 "../blis-2.0/frame/base//bli_blksz.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

// blksz_t query

BLIS_INLINE dim_t bli_blksz_get_def
     (
             num_t    dt,
       const blksz_t* b
     )
{
	return b->v[ dt ];
}

BLIS_INLINE dim_t bli_blksz_get_max
     (
             num_t    dt,
       const blksz_t* b
     )
{
	return b->e[ dt ];
}


// blksz_t modification

BLIS_INLINE void bli_blksz_set_def
     (
       dim_t    val,
       num_t    dt,
       blksz_t* b
     )
{
	b->v[ dt ] = val;
}

BLIS_INLINE void bli_blksz_set_max
     (
       dim_t    val,
       num_t    dt,
       blksz_t* b
     )
{
	b->e[ dt ] = val;
}

BLIS_INLINE void bli_blksz_copy
     (
       const blksz_t* b_src,
             blksz_t* b_dst
     )
{
	*b_dst = *b_src;
}

BLIS_INLINE void bli_blksz_copy_if_nonneg
     (
       const blksz_t* b_src,
             blksz_t* b_dst
     )
{
	// Copy the blocksize values over to b_dst one-by-one. Note that we
	// only copy valuse that are zero or positive (and skip copying any
	// values that are negative).

	const dim_t v_s = bli_blksz_get_def( BLIS_FLOAT,    b_src );
	const dim_t v_d = bli_blksz_get_def( BLIS_DOUBLE,   b_src );
	const dim_t v_c = bli_blksz_get_def( BLIS_SCOMPLEX, b_src );
	const dim_t v_z = bli_blksz_get_def( BLIS_DCOMPLEX, b_src );

	const dim_t e_s = bli_blksz_get_max( BLIS_FLOAT,    b_src );
	const dim_t e_d = bli_blksz_get_max( BLIS_DOUBLE,   b_src );
	const dim_t e_c = bli_blksz_get_max( BLIS_SCOMPLEX, b_src );
	const dim_t e_z = bli_blksz_get_max( BLIS_DCOMPLEX, b_src );

	if ( v_s >= 0 ) bli_blksz_set_def( v_s, BLIS_FLOAT,    b_dst );
	if ( v_d >= 0 ) bli_blksz_set_def( v_d, BLIS_DOUBLE,   b_dst );
	if ( v_c >= 0 ) bli_blksz_set_def( v_c, BLIS_SCOMPLEX, b_dst );
	if ( v_z >= 0 ) bli_blksz_set_def( v_z, BLIS_DCOMPLEX, b_dst );

	if ( e_s >= 0 ) bli_blksz_set_max( e_s, BLIS_FLOAT,    b_dst );
	if ( e_d >= 0 ) bli_blksz_set_max( e_d, BLIS_DOUBLE,   b_dst );
	if ( e_c >= 0 ) bli_blksz_set_max( e_c, BLIS_SCOMPLEX, b_dst );
	if ( e_z >= 0 ) bli_blksz_set_max( e_z, BLIS_DCOMPLEX, b_dst );
}

BLIS_INLINE void bli_blksz_copy_def_dt
     (
       num_t dt_src, const blksz_t* b_src,
       num_t dt_dst,       blksz_t* b_dst
     )
{
	const dim_t val = bli_blksz_get_def( dt_src, b_src );

	bli_blksz_set_def( val, dt_dst, b_dst );
}

BLIS_INLINE void bli_blksz_copy_max_dt
     (
       num_t dt_src, const blksz_t* b_src,
       num_t dt_dst,       blksz_t* b_dst
     )
{
	const dim_t val = bli_blksz_get_max( dt_src, b_src );

	bli_blksz_set_max( val, dt_dst, b_dst );
}

BLIS_INLINE void bli_blksz_copy_dt
     (
       num_t dt_src, const blksz_t* b_src,
       num_t dt_dst,       blksz_t* b_dst
     )
{
	bli_blksz_copy_def_dt( dt_src, b_src, dt_dst, b_dst );
	bli_blksz_copy_max_dt( dt_src, b_src, dt_dst, b_dst );
}

BLIS_INLINE void bli_blksz_scale_def
     (
       dim_t    num,
       dim_t    den,
       num_t    dt,
       blksz_t* b
     )
{
	const dim_t val = bli_blksz_get_def( dt, b );

	bli_blksz_set_def( ( val * num ) / den, dt, b );
}

BLIS_INLINE void bli_blksz_scale_max
     (
       dim_t    num,
       dim_t    den,
       num_t    dt,
       blksz_t* b
     )
{
	const dim_t val = bli_blksz_get_max( dt, b );

	bli_blksz_set_max( ( val * num ) / den, dt, b );
}

BLIS_INLINE void bli_blksz_scale_def_max
     (
       dim_t    num,
       dim_t    den,
       num_t    dt,
       blksz_t* b
     )
{
	bli_blksz_scale_def( num, den, dt, b );
	bli_blksz_scale_max( num, den, dt, b );
}

// -----------------------------------------------------------------------------

BLIS_EXPORT_BLIS blksz_t* bli_blksz_create_ed
     (
       dim_t b_s, dim_t be_s,
       dim_t b_d, dim_t be_d,
       dim_t b_c, dim_t be_c,
       dim_t b_z, dim_t be_z
     );

BLIS_EXPORT_BLIS blksz_t* bli_blksz_create
     (
       dim_t b_s,  dim_t b_d,  dim_t b_c,  dim_t b_z,
       dim_t be_s, dim_t be_d, dim_t be_c, dim_t be_z
     );

BLIS_EXPORT_BLIS void bli_blksz_init_ed
     (
       blksz_t* b,
       dim_t    b_s, dim_t be_s,
       dim_t    b_d, dim_t be_d,
       dim_t    b_c, dim_t be_c,
       dim_t    b_z, dim_t be_z
     );

BLIS_EXPORT_BLIS void bli_blksz_init
     (
       blksz_t* b,
       dim_t b_s,  dim_t b_d,  dim_t b_c,  dim_t b_z,
       dim_t be_s, dim_t be_d, dim_t be_c, dim_t be_z
     );

BLIS_EXPORT_BLIS void bli_blksz_init_easy
     (
       blksz_t* b,
       dim_t b_s,  dim_t b_d,  dim_t b_c,  dim_t b_z
     );

BLIS_EXPORT_BLIS void bli_blksz_free
     (
       blksz_t* b
     );

// -----------------------------------------------------------------------------

#if 0
BLIS_EXPORT_BLIS void bli_blksz_reduce_dt_to
     (
       num_t dt_bm, blksz_t* bmult,
       num_t dt_bs, blksz_t* blksz
     );
#endif

void bli_blksz_reduce_def_to
     (
       num_t dt_bm, blksz_t* bmult,
       num_t dt_bs, blksz_t* blksz
     );

void bli_blksz_reduce_max_to
     (
       num_t dt_bm, blksz_t* bmult,
       num_t dt_bs, blksz_t* blksz
     );
// -----------------------------------------------------------------------------

dim_t bli_determine_blocksize
     (
       dir_t direct,
       dim_t i,
       dim_t dim,
       dim_t b_alg,
       dim_t b_max
     );

// end bli_blksz.h
#line 126 "../blis-2.0/frame/include/blis.h"

// begin bli_func.h
#line 1 "../blis-2.0/frame/base//bli_func.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2024, Southern Methodist University

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

// -----------------------------------------------------------------------------

// func_t query

BLIS_INLINE void_fp bli_func_get_dt
     (
             num_t   dt,
       const func_t* func
     )
{
	return func->ptr[ dt ];
}

BLIS_INLINE void_fp bli_func2_get_dt
     (
             num_t    dt1,
             num_t    dt2,
       const func2_t* func
     )
{
	// Arrange the pointer elements such that ((func2_t*)x)->ptr[0][dt]
	// is equivalent to ((func_t*)x)->ptr[dt] and encodes the (dt,dt)
	// "diagonal" value.
	gint_t off = dt2 < dt1 ? dt2 + BLIS_NUM_FP_TYPES - dt1 : dt2 - dt1;
	return func->ptr[ off ][ dt1 ];
}

// func_t modification

BLIS_INLINE void bli_func_set_dt
     (
       void_fp fp,
       num_t   dt,
       func_t* func
     )
{
	func->ptr[ dt ] = fp;
}

BLIS_INLINE void bli_func2_set_dt
     (
       void_fp  fp,
       num_t    dt1,
       num_t    dt2,
       func2_t* func
     )
{
	// Arrange the pointer elements such that ((func2_t*)x)->ptr[0][dt]
	// is equivalent to ((func_t*)x)->ptr[dt] and encodes the (dt,dt)
	// "diagonal" value.
	gint_t off = dt2 < dt1 ? dt2 + BLIS_NUM_FP_TYPES - dt1 : dt2 - dt1;
	func->ptr[ off ][ dt1 ] = fp;
}

BLIS_INLINE void bli_func_copy_dt
     (
       num_t dt_src, const func_t* func_src,
       num_t dt_dst,       func_t* func_dst
     )
{
	void_fp fp = bli_func_get_dt( dt_src, func_src );

	bli_func_set_dt( fp, dt_dst, func_dst );
}

// -----------------------------------------------------------------------------

BLIS_EXPORT_BLIS func_t* bli_func_create
     (
       void_fp ptr_s,
       void_fp ptr_d,
       void_fp ptr_c,
       void_fp ptr_z
     );

BLIS_EXPORT_BLIS void bli_func_init
     (
       func_t* f,
       void_fp ptr_s,
       void_fp ptr_d,
       void_fp ptr_c,
       void_fp ptr_z
     );

BLIS_EXPORT_BLIS void bli_func_init_null
     (
       func_t* f
     );

BLIS_EXPORT_BLIS void bli_func_free( func_t* f );

BLIS_EXPORT_BLIS func2_t* bli_func2_create
     (
       void_fp ptr_ss, void_fp ptr_sd, void_fp ptr_sc, void_fp ptr_sz,
       void_fp ptr_ds, void_fp ptr_dd, void_fp ptr_dc, void_fp ptr_dz,
       void_fp ptr_cs, void_fp ptr_cd, void_fp ptr_cc, void_fp ptr_cz,
       void_fp ptr_zs, void_fp ptr_zd, void_fp ptr_zc, void_fp ptr_zz
     );

BLIS_EXPORT_BLIS void bli_func2_init
     (
       func2_t* f,
       void_fp ptr_ss, void_fp ptr_sd, void_fp ptr_sc, void_fp ptr_sz,
       void_fp ptr_ds, void_fp ptr_dd, void_fp ptr_dc, void_fp ptr_dz,
       void_fp ptr_cs, void_fp ptr_cd, void_fp ptr_cc, void_fp ptr_cz,
       void_fp ptr_zs, void_fp ptr_zd, void_fp ptr_zc, void_fp ptr_zz
     );

BLIS_EXPORT_BLIS void bli_func2_init_null
     (
       func2_t* f
     );

BLIS_EXPORT_BLIS void bli_func2_free( func2_t* f );

// -----------------------------------------------------------------------------

BLIS_EXPORT_BLIS bool bli_func_is_null_dt(       num_t   dt,
                                           const func_t* f );
BLIS_EXPORT_BLIS bool bli_func_is_null( const func_t* f );

BLIS_EXPORT_BLIS bool bli_func2_is_null_dt(       num_t    dt1,
                                                  num_t    dt2,
                                            const func2_t* f );
BLIS_EXPORT_BLIS bool bli_func2_is_null( const func2_t* f );

// end bli_func.h
#line 127 "../blis-2.0/frame/include/blis.h"

// begin bli_mbool.h
#line 1 "../blis-2.0/frame/base//bli_mbool.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

// -----------------------------------------------------------------------------

// mbool_t query

BLIS_INLINE bool bli_mbool_get_dt( num_t dt, const mbool_t* mb )
{
	return ( bool )( mb->v[ dt ] );
}

// mbool_t modification

BLIS_INLINE void bli_mbool_set_dt( bool val, num_t dt, mbool_t* mb )
{
	mb->v[ dt ] = val;
}

// -----------------------------------------------------------------------------

BLIS_EXPORT_BLIS mbool_t* bli_mbool_create
     (
       bool b_s,
       bool b_d,
       bool b_c,
       bool b_z
     );

BLIS_EXPORT_BLIS void bli_mbool_init
     (
       mbool_t* b,
       bool     b_s,
       bool     b_d,
       bool     b_c,
       bool     b_z
     );

BLIS_EXPORT_BLIS void bli_mbool_free( mbool_t* b );

// end bli_mbool.h
#line 128 "../blis-2.0/frame/include/blis.h"

// begin bli_stack.h
#line 1 "../blis-2.0/frame/base//bli_stack.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2023, Southern Methodist University

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_STACK_H
#define BLIS_STACK_H

// -- Stack type based on a dynamic block array --

/*
typedef struct
{
	siz_t elem_size;
	siz_t block_len;
	siz_t max_blocks;
	siz_t size;
	siz_t capacity;

	void** blocks;

	bli_pthread_mutex_t lock;
} stck_t;
*/


BLIS_EXPORT_BLIS err_t bli_stack_init
     (
       siz_t   elem_size,
       siz_t   block_len,
       siz_t   max_blocks,
       siz_t   initial_size,
       stck_t* stack
     );

BLIS_EXPORT_BLIS err_t bli_stack_finalize( stck_t* stack );

BLIS_INLINE siz_t bli_stack_size( const stck_t* stack )
{
	return stack->size;
}

BLIS_EXPORT_BLIS err_t bli_stack_get( siz_t i, void** elem, const stck_t* stack );

BLIS_EXPORT_BLIS err_t bli_stack_push( siz_t* i, stck_t* stack );


#endif

// end bli_stack.h
#line 129 "../blis-2.0/frame/include/blis.h"

// begin bli_check.h
#line 1 "../blis-2.0/frame/base//bli_check.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2018 - 2019, Advanced Micro Devices, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/


BLIS_EXPORT_BLIS err_t bli_check_error_code_helper( gint_t code, const char* file, guint_t line );

err_t bli_check_valid_error_level( errlev_t level );

err_t bli_check_null_pointer( const void* ptr );

err_t bli_check_valid_side( side_t side );
err_t bli_check_valid_uplo( uplo_t uplo );
err_t bli_check_valid_trans( trans_t trans );
err_t bli_check_valid_diag( diag_t diag );
err_t bli_check_nonunit_diag( const obj_t* a );

err_t bli_check_valid_datatype( num_t dt );
err_t bli_check_object_valid_datatype( const obj_t* a );
err_t bli_check_noninteger_datatype( num_t dt );
err_t bli_check_noninteger_object( const obj_t* a );
err_t bli_check_nonconstant_datatype( num_t dt );
err_t bli_check_nonconstant_object( const obj_t* a );
err_t bli_check_floating_datatype( num_t dt );
err_t bli_check_floating_object( const obj_t* a );
err_t bli_check_real_datatype( num_t dt );
err_t bli_check_real_object( const obj_t* a );
err_t bli_check_integer_datatype( num_t dt );
err_t bli_check_integer_object( const obj_t* a );
err_t bli_check_consistent_datatypes( num_t dt_a, num_t dt_b );
err_t bli_check_consistent_object_datatypes( const obj_t* a, const obj_t* b );
err_t bli_check_datatype_real_proj_of( num_t dt_c, num_t dt_r );
err_t bli_check_object_real_proj_of( const obj_t* c, const obj_t* r );
err_t bli_check_real_valued_object( const obj_t* a );
err_t bli_check_consistent_precisions( num_t dt_a, num_t dt_b );
err_t bli_check_consistent_object_precisions( const obj_t* a, const obj_t* b );

err_t bli_check_conformal_dims( const obj_t* a, const obj_t* b );
err_t bli_check_level3_dims( const obj_t* a, const obj_t* b, const obj_t* c );
err_t bli_check_scalar_object( const obj_t* a );
err_t bli_check_vector_object( const obj_t* a );
err_t bli_check_matrix_object( const obj_t* a );
err_t bli_check_equal_vector_lengths( const obj_t* x, const obj_t* y );
err_t bli_check_square_object( const obj_t* a );
err_t bli_check_object_length_equals( const obj_t* a, dim_t m );
err_t bli_check_object_width_equals( const obj_t* a, dim_t n );
err_t bli_check_vector_dim_equals( const obj_t* a, dim_t n );
err_t bli_check_object_diag_offset_equals( const obj_t* a, doff_t offset );

err_t bli_check_matrix_strides( dim_t m, dim_t n, inc_t rs, inc_t cs, inc_t is );

err_t bli_check_general_object( const obj_t* a );
err_t bli_check_hermitian_object( const obj_t* a );
err_t bli_check_symmetric_object( const obj_t* a );
err_t bli_check_triangular_object( const obj_t* a );
err_t bli_check_object_struc( const obj_t* a, struc_t struc );

err_t bli_check_upper_or_lower_object( const obj_t* a );

err_t bli_check_valid_3x1_subpart( subpart_t part );
err_t bli_check_valid_1x3_subpart( subpart_t part );
err_t bli_check_valid_3x3_subpart( subpart_t part );

err_t bli_check_valid_cntl( const void* cntl );

err_t bli_check_packm_schema_on_unpack( const obj_t* a );
err_t bli_check_packv_schema_on_unpack( const obj_t* a );

err_t bli_check_object_buffer( const obj_t* a );

err_t bli_check_valid_malloc_buf( const void* ptr );

err_t bli_check_valid_packbuf( packbuf_t buf_type );
err_t bli_check_if_exhausted_pool( const pool_t* pool );
err_t bli_check_sufficient_stack_buf_size( const cntx_t* cntx );
err_t bli_check_alignment_is_power_of_two( size_t align_size );
err_t bli_check_alignment_is_mult_of_ptr_size( size_t align_size );

err_t bli_check_object_alias_of( const obj_t* a, const obj_t* b );

err_t bli_check_valid_arch_id( arch_t id );
err_t bli_check_initialized_gks_cntx( const cntx_t* cntx );

err_t bli_check_valid_mc_mod_mult( const blksz_t* mc, const blksz_t* mr );
err_t bli_check_valid_nc_mod_mult( const blksz_t* nc, const blksz_t* nr );
err_t bli_check_valid_kc_mod_mult( const blksz_t* kc, const blksz_t* kr );

err_t bli_check_valid_mr_even( const blksz_t* mr, const mbool_t* row_pref );
err_t bli_check_valid_nr_even( const blksz_t* mr, const mbool_t* row_pref );

// end bli_check.h
#line 130 "../blis-2.0/frame/include/blis.h"

// begin bli_cntx.h
#line 1 "../blis-2.0/frame/base//bli_cntx.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2016, Hewlett Packard Enterprise Development LP
   Copyright (C) 2019, Advanced Micro Devices, Inc.
   Copyright (C) 2024, Southern Methodist University

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_CNTX_H
#define BLIS_CNTX_H


// Context object type (defined in bli_type_defs.h)

/*
typedef struct cntx_s
{
	stck_t blkszs; // blksz_t
	stck_t bmults; // kerid_t

	stck_t ukrs; // func_t
	stck_t ukr2s; // func2_t
	stck_t ukr_prefs; // mbool_t

	stck_t l3_sup_handlers; // void_fp
} cntx_t;
*/

// -----------------------------------------------------------------------------

//
// -- cntx_t query (complex) ---------------------------------------------------
//

BLIS_INLINE const blksz_t* bli_cntx_get_blksz( kerid_t bs_id, const cntx_t* cntx )
{
	const blksz_t* blksz;
	err_t error = bli_stack_get( bs_id, ( void** )&blksz, &cntx->blkszs );
	if ( error != BLIS_SUCCESS )
		bli_check_error_code( error );
	return blksz;
}

BLIS_INLINE dim_t bli_cntx_get_blksz_def_dt( num_t dt, kerid_t bs_id, const cntx_t* cntx )
{
	const blksz_t* blksz  = bli_cntx_get_blksz( bs_id, cntx );
	dim_t          bs_dt  = bli_blksz_get_def( dt, blksz );

	// Return the main (default) blocksize value for the datatype given.
	return bs_dt;
}

BLIS_INLINE dim_t bli_cntx_get_blksz_max_dt( num_t dt, kerid_t bs_id, const cntx_t* cntx )
{
	const blksz_t* blksz  = bli_cntx_get_blksz( bs_id, cntx );
	dim_t          bs_dt  = bli_blksz_get_max( dt, blksz );

	// Return the auxiliary (maximum) blocksize value for the datatype given.
	return bs_dt;
}

BLIS_INLINE kerid_t bli_cntx_get_bmult_id( kerid_t bs_id, const cntx_t* cntx )
{
	const kerid_t* bsz;
	err_t error = bli_stack_get( bs_id, ( void** )&bsz, &cntx->bmults );
	if ( error != BLIS_SUCCESS )
		bli_check_error_code( error );
	return *bsz;
}

BLIS_INLINE const blksz_t* bli_cntx_get_bmult( kerid_t bs_id, const cntx_t* cntx )
{
	kerid_t        bm_id  = bli_cntx_get_bmult_id( bs_id, cntx );
	const blksz_t* bmult  = bli_cntx_get_blksz( bm_id, cntx );

	return bmult;
}

BLIS_INLINE dim_t bli_cntx_get_bmult_dt( num_t dt, kerid_t bs_id, const cntx_t* cntx )
{
	const blksz_t* bmult  = bli_cntx_get_bmult( bs_id, cntx );
	dim_t          bm_dt  = bli_blksz_get_def( dt, bmult );

	return bm_dt;
}

// -----------------------------------------------------------------------------

BLIS_INLINE const func2_t* bli_cntx_get_ukr2s( kerid_t ukr_id, const cntx_t* cntx )
{
	const func2_t* ukr;
	err_t error = bli_stack_get( bli_ker_idx( ukr_id ), ( void** )&ukr, &cntx->ukr2s );
	if ( error != BLIS_SUCCESS )
		bli_check_error_code( error );
	return ukr;
}

BLIS_INLINE void_fp bli_cntx_get_ukr2_dt( num_t dt1, num_t dt2, kerid_t ukr_id, const cntx_t* cntx )
{
	const func2_t* func = bli_cntx_get_ukr2s( ukr_id, cntx );

	return bli_func2_get_dt( dt1, dt2, func );
}

// -----------------------------------------------------------------------------

BLIS_INLINE const func_t* bli_cntx_get_ukrs( kerid_t ukr_id, const cntx_t* cntx )
{
	if ( bli_ker_ntype( ukr_id ) == 2 )
	{
		return ( const func_t* )bli_cntx_get_ukr2s( ukr_id, cntx );
	}
	else
	{
		const func_t* ukr;
		err_t error = bli_stack_get( bli_ker_idx( ukr_id ), ( void** )&ukr, &cntx->ukrs );
		if ( error != BLIS_SUCCESS )
			bli_check_error_code( error );
		return ukr;
	}
}

BLIS_INLINE void_fp bli_cntx_get_ukr_dt( num_t dt, kerid_t ukr_id, const cntx_t* cntx )
{
	if ( bli_ker_ntype( ukr_id ) == 2 )
	{
		return bli_cntx_get_ukr2_dt( dt, dt, ukr_id, cntx );
	}
	else
	{
		const func_t* func = bli_cntx_get_ukrs( ukr_id, cntx );

		return bli_func_get_dt( dt, func );
	}
}

// -----------------------------------------------------------------------------

BLIS_INLINE const mbool_t* bli_cntx_get_ukr_prefs( kerid_t pref_id, const cntx_t* cntx )
{
	const mbool_t* ukr_prefs;
	err_t error = bli_stack_get( pref_id, ( void** )&ukr_prefs, &cntx->ukr_prefs );
	if ( error != BLIS_SUCCESS )
		bli_check_error_code( error );
	return ukr_prefs;
}

BLIS_INLINE bool bli_cntx_get_ukr_prefs_dt( num_t dt, kerid_t ukr_id, const cntx_t* cntx )
{
	const mbool_t* mbool = bli_cntx_get_ukr_prefs( ukr_id, cntx );

	return ( bool )bli_mbool_get_dt( dt, mbool );
}

// -----------------------------------------------------------------------------

BLIS_INLINE bool bli_cntx_l3_sup_thresh_is_met( num_t dt, dim_t m, dim_t n, dim_t k, const cntx_t* cntx )
{
	if ( m < bli_cntx_get_blksz_def_dt( dt, BLIS_MT, cntx ) ) return TRUE;
	if ( n < bli_cntx_get_blksz_def_dt( dt, BLIS_NT, cntx ) ) return TRUE;
	if ( k < bli_cntx_get_blksz_def_dt( dt, BLIS_KT, cntx ) ) return TRUE;

	return FALSE;
}

// -----------------------------------------------------------------------------

BLIS_INLINE void_fp bli_cntx_get_l3_sup_handler( opid_t op, const cntx_t* cntx )
{
	const void_fp* l3_handler;
	err_t error = bli_stack_get( op, ( void** )&l3_handler, &cntx->l3_sup_handlers );
	if ( error != BLIS_SUCCESS )
		bli_check_error_code( error );
	return *l3_handler;
}

// -----------------------------------------------------------------------------

BLIS_INLINE bool bli_cntx_ukr_prefers_rows_dt( num_t dt, ukr_t ukr_id, const cntx_t* cntx )
{
	// This initial value will get overwritten during the switch statement below.
	ukr_pref_t ukr_pref_id = BLIS_GEMM_UKR_ROW_PREF;

	// Get the correct preference from the kernel ID.
	switch ( ukr_id )
	{
		case BLIS_GEMM_UKR: ukr_pref_id = BLIS_GEMM_UKR_ROW_PREF; break;
		case BLIS_TRSM_L_UKR: ukr_pref_id = BLIS_TRSM_L_UKR_ROW_PREF; break;
		case BLIS_TRSM_U_UKR: ukr_pref_id = BLIS_TRSM_U_UKR_ROW_PREF; break;
		case BLIS_GEMMTRSM_L_UKR: ukr_pref_id = BLIS_GEMMTRSM_L_UKR_ROW_PREF; break;
		case BLIS_GEMMTRSM_U_UKR: ukr_pref_id = BLIS_GEMMTRSM_U_UKR_ROW_PREF; break;
		case BLIS_GEMMSUP_RRR_UKR: ukr_pref_id = BLIS_GEMMSUP_RRR_UKR_ROW_PREF; break;
		case BLIS_GEMMSUP_RRC_UKR: ukr_pref_id = BLIS_GEMMSUP_RRC_UKR_ROW_PREF; break;
		case BLIS_GEMMSUP_RCR_UKR: ukr_pref_id = BLIS_GEMMSUP_RCR_UKR_ROW_PREF; break;
		case BLIS_GEMMSUP_RCC_UKR: ukr_pref_id = BLIS_GEMMSUP_RCC_UKR_ROW_PREF; break;
		case BLIS_GEMMSUP_CRR_UKR: ukr_pref_id = BLIS_GEMMSUP_CRR_UKR_ROW_PREF; break;
		case BLIS_GEMMSUP_CRC_UKR: ukr_pref_id = BLIS_GEMMSUP_CRC_UKR_ROW_PREF; break;
		case BLIS_GEMMSUP_CCR_UKR: ukr_pref_id = BLIS_GEMMSUP_CCR_UKR_ROW_PREF; break;
		case BLIS_GEMMSUP_CCC_UKR: ukr_pref_id = BLIS_GEMMSUP_CCC_UKR_ROW_PREF; break;
		case BLIS_GEMMSUP_XXX_UKR: ukr_pref_id = BLIS_GEMMSUP_XXX_UKR_ROW_PREF; break;
		default: break; // TODO: should be an error condition
	}

	return bli_cntx_get_ukr_prefs_dt( dt, ukr_pref_id, cntx );
}

BLIS_INLINE bool bli_cntx_ukr_prefers_cols_dt( num_t dt, ukr_t ukr_id, const cntx_t* cntx )
{
	return ! bli_cntx_ukr_prefers_rows_dt( dt, ukr_id, cntx );
}

BLIS_INLINE bool bli_cntx_prefers_storage_of( const obj_t* obj, ukr_t ukr_id, const cntx_t* cntx )
{
	const bool ukr_prefers_rows
		= bli_cntx_ukr_prefers_rows_dt( bli_obj_dt( obj ), ukr_id, cntx );

	if      ( bli_obj_is_row_stored( obj ) &&  ukr_prefers_rows ) return TRUE;
	else if ( bli_obj_is_col_stored( obj ) && !ukr_prefers_rows ) return TRUE;

	return FALSE;
}

BLIS_INLINE bool bli_cntx_dislikes_storage_of( const obj_t* obj, ukr_t ukr_id, const cntx_t* cntx )
{
	return ! bli_cntx_prefers_storage_of( obj, ukr_id, cntx );
}

// -----------------------------------------------------------------------------

//
// -- cntx_t modification (complex) --------------------------------------------
//

// NOTE: The framework does not use any of the following functions. We provide
// them in order to facilitate creating/modifying custom contexts.

BLIS_INLINE err_t bli_cntx_set_blksz( kerid_t bs_id, const blksz_t* blksz, kerid_t mult_id, cntx_t* cntx )
{
	blksz_t* cntx_blksz;
	err_t error = bli_stack_get( bs_id, ( void** )&cntx_blksz, &cntx->blkszs );
	if ( error != BLIS_SUCCESS )
		return error;

	kerid_t* cntx_mult_id;
	error = bli_stack_get( bs_id, ( void** )&cntx_mult_id, &cntx->bmults );
	if ( error != BLIS_SUCCESS )
		return error;

	bli_blksz_copy_if_nonneg( blksz, cntx_blksz );
	*cntx_mult_id = mult_id;

	return BLIS_SUCCESS;
}

BLIS_INLINE void bli_cntx_set_blksz_def_dt( num_t dt, kerid_t bs_id, dim_t bs, cntx_t* cntx )
{
	bli_blksz_set_def( bs, dt, ( blksz_t* )bli_cntx_get_blksz( bs_id, cntx ) );
}

BLIS_INLINE void bli_cntx_set_blksz_max_dt( num_t dt, kerid_t bs_id, dim_t bs, cntx_t* cntx )
{
	bli_blksz_set_max( bs, dt, ( blksz_t* )bli_cntx_get_blksz( bs_id, cntx ) );
}

BLIS_INLINE err_t bli_cntx_set_ukr2( kerid_t ukr_id, const func2_t* func, cntx_t* cntx )
{
	*( func2_t* )bli_cntx_get_ukr2s( ukr_id, cntx ) = *func;
	return BLIS_SUCCESS;
}

BLIS_INLINE void bli_cntx_set_ukr2_dt( void_fp fp, num_t dt1, num_t dt2, kerid_t ker_id, cntx_t* cntx )
{
	bli_func2_set_dt( fp, dt1, dt2, ( func2_t* )bli_cntx_get_ukr2s( ker_id, cntx ) );
}

BLIS_INLINE err_t bli_cntx_set_ukr( kerid_t ukr_id, const func_t* func, cntx_t* cntx )
{
	*( func_t* )bli_cntx_get_ukrs( ukr_id, cntx ) = *func;
	return BLIS_SUCCESS;
}

BLIS_INLINE void bli_cntx_set_ukr_dt( void_fp fp, num_t dt, kerid_t ker_id, cntx_t* cntx )
{
	if ( bli_ker_ntype( ker_id ) == 2 )
	{
		bli_cntx_set_ukr2_dt( fp, dt, dt, ker_id, cntx );
	}
	else
	{
		bli_func_set_dt( fp, dt, ( func_t* )bli_cntx_get_ukrs( ker_id, cntx ) );
	}
}

BLIS_INLINE err_t bli_cntx_set_ukr_pref( kerid_t ukr_id, const mbool_t* prefs, cntx_t* cntx )
{
	*( mbool_t* )bli_cntx_get_ukr_prefs( ukr_id, cntx ) = *prefs;
	return BLIS_SUCCESS;
}

BLIS_INLINE err_t bli_cntx_set_ukr_pref_dt( bool pref, num_t dt, kerid_t ukr_id, cntx_t* cntx )
{
	bli_mbool_set_dt( pref, dt, ( mbool_t* )bli_cntx_get_ukr_prefs( ukr_id, cntx ));
	return BLIS_SUCCESS;
}

BLIS_INLINE err_t bli_cntx_set_l3_sup_handler( opid_t opid, void_fp handler, cntx_t* cntx )
{
	void_fp* cntx_handler;
	err_t error = bli_stack_get( opid, ( void** )&cntx_handler, &cntx->l3_sup_handlers );
	if ( error != BLIS_SUCCESS )
		return error;

	*cntx_handler = handler;

	return BLIS_SUCCESS;
}

BLIS_INLINE void_fp bli_cntx_get_l3_sup_ker_dt( num_t dt, stor3_t stor_id, const cntx_t* cntx )
{
	ukr_t ukr_id = bli_stor3_ukr( stor_id );

	return bli_cntx_get_ukr_dt( dt, ukr_id, cntx );
}

BLIS_INLINE dim_t bli_cntx_get_l3_sup_blksz_def_dt( num_t dt, bszid_t bs_id, const cntx_t* cntx )
{
	switch ( bs_id )
	{
		case BLIS_MR: bs_id = BLIS_MR_SUP; break;
		case BLIS_NR: bs_id = BLIS_NR_SUP; break;
		case BLIS_KR: bs_id = BLIS_KR_SUP; break;
		case BLIS_MC: bs_id = BLIS_MC_SUP; break;
		case BLIS_NC: bs_id = BLIS_NC_SUP; break;
		case BLIS_KC: bs_id = BLIS_KC_SUP; break;
		default: break;
	}

	return bli_cntx_get_blksz_def_dt( dt, bs_id, cntx );
}

BLIS_INLINE dim_t bli_cntx_get_l3_sup_blksz_max_dt( num_t dt, bszid_t bs_id, const cntx_t* cntx )
{
	switch ( bs_id )
	{
		case BLIS_MR: bs_id = BLIS_MR_SUP; break;
		case BLIS_NR: bs_id = BLIS_NR_SUP; break;
		case BLIS_KR: bs_id = BLIS_KR_SUP; break;
		case BLIS_MC: bs_id = BLIS_MC_SUP; break;
		case BLIS_NC: bs_id = BLIS_NC_SUP; break;
		case BLIS_KC: bs_id = BLIS_KC_SUP; break;
		default: break;
	}

	return bli_cntx_get_blksz_max_dt( dt, bs_id, cntx );
}

// -----------------------------------------------------------------------------

// Function prototypes

BLIS_EXPORT_BLIS err_t bli_cntx_init( cntx_t* cntx );

BLIS_EXPORT_BLIS err_t bli_cntx_free( cntx_t* cntx );

BLIS_EXPORT_BLIS void bli_cntx_set_blkszs( cntx_t* cntx, ... );

BLIS_EXPORT_BLIS void bli_cntx_set_ukrs( cntx_t* cntx, ... );
BLIS_EXPORT_BLIS void bli_cntx_set_ukr2s( cntx_t* cntx, ... );
BLIS_EXPORT_BLIS void bli_cntx_set_ukr_prefs( cntx_t* cntx, ... );

BLIS_EXPORT_BLIS void bli_cntx_print( const cntx_t* cntx );

BLIS_EXPORT_BLIS void bli_cntx_set_l3_sup_handlers( cntx_t* cntx, ... );

BLIS_EXPORT_BLIS err_t bli_cntx_register_blksz( kerid_t* bs_id, const blksz_t* blksz, kerid_t bmult_id, cntx_t* cntx );

BLIS_EXPORT_BLIS err_t bli_cntx_register_ukr( kerid_t* ukr_id, const func_t* ukr, cntx_t* cntx );

BLIS_EXPORT_BLIS err_t bli_cntx_register_ukr2( kerid_t* ukr_id, const func2_t* ukr, cntx_t* cntx );

BLIS_EXPORT_BLIS err_t bli_cntx_register_ukr_pref( kerid_t* ukr_pref_id, const mbool_t* ukr_pref, cntx_t* cntx );


#endif

// end bli_cntx.h
#line 131 "../blis-2.0/frame/include/blis.h"

// begin bli_rntm.h
#line 1 "../blis-2.0/frame/base//bli_rntm.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2016, Hewlett Packard Enterprise Development LP
   Copyright (C) 2018 - 2019, Advanced Micro Devices, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_RNTM_H
#define BLIS_RNTM_H


// Runtime object type (defined in bli_type_defs.h)

/*
typedef struct rntm_s
{
	timpl_t   thread_impl;

	dim_t     num_threads;
	dim_t     thrloop[ BLIS_NUM_LOOPS ];

	bool      auto_factor;
	bool      pack_a;
	bool      pack_b;
	bool      l3_sup;
} rntm_t;
*/

//
// -- rntm_t query (public API) ------------------------------------------------
//

BLIS_INLINE timpl_t bli_rntm_thread_impl( const rntm_t* rntm )
{
	return rntm->thread_impl;
}

BLIS_INLINE bool bli_rntm_auto_factor( const rntm_t* rntm )
{
	return rntm->auto_factor;
}

BLIS_INLINE dim_t bli_rntm_num_threads( const rntm_t* rntm )
{
	return rntm->num_threads;
}

BLIS_INLINE dim_t bli_rntm_ways_for( bszid_t bszid, const rntm_t* rntm )
{
	return ( bszid == BLIS_NO_PART ? 1 : rntm->thrloop[ bszid ] );
}

BLIS_INLINE dim_t bli_rntm_total_ways_for( dim_t thread_by, const rntm_t* rntm )
{
    dim_t n_way = 1;
    for ( dim_t i = 0; i < BLIS_NUM_LOOPS; i++ )
	{
        if ( thread_by & (1 << i) )
            n_way *= bli_rntm_ways_for( (bszid_t)i, rntm );
    }
    return n_way;
}

BLIS_INLINE dim_t bli_rntm_jc_ways( const rntm_t* rntm )
{
	return bli_rntm_ways_for( BLIS_NC, rntm );
}
BLIS_INLINE dim_t bli_rntm_pc_ways( const rntm_t* rntm )
{
	return bli_rntm_ways_for( BLIS_KC, rntm );
}
BLIS_INLINE dim_t bli_rntm_ic_ways( const rntm_t* rntm )
{
	return bli_rntm_ways_for( BLIS_MC, rntm );
}
BLIS_INLINE dim_t bli_rntm_jr_ways( const rntm_t* rntm )
{
	return bli_rntm_ways_for( BLIS_NR, rntm );
}
BLIS_INLINE dim_t bli_rntm_ir_ways( const rntm_t* rntm )
{
	return bli_rntm_ways_for( BLIS_MR, rntm );
}
BLIS_INLINE dim_t bli_rntm_pr_ways( const rntm_t* rntm )
{
	return bli_rntm_ways_for( BLIS_KR, rntm );
}

BLIS_INLINE bool bli_rntm_pack_a( const rntm_t* rntm )
{
	return ( bool )( rntm->pack_a );
}
BLIS_INLINE bool bli_rntm_pack_b( const rntm_t* rntm )
{
	return ( bool )( rntm->pack_b );
}

BLIS_INLINE bool bli_rntm_l3_sup( const rntm_t* rntm )
{
	return rntm->l3_sup;
}

//
// -- rntm_t modification (internal use only) ----------------------------------
//

BLIS_INLINE void bli_rntm_set_thread_impl_only( timpl_t thread_impl, rntm_t* rntm )
{
	rntm->thread_impl = thread_impl;
}

BLIS_INLINE void bli_rntm_set_auto_factor_only( bool auto_factor, rntm_t* rntm )
{
	rntm->auto_factor = auto_factor;
}

BLIS_INLINE void bli_rntm_set_num_threads_only( dim_t nt, rntm_t* rntm )
{
	rntm->num_threads = nt;
}

BLIS_INLINE void bli_rntm_set_ways_for_only( bszid_t loop, dim_t n_ways, rntm_t* rntm )
{
	rntm->thrloop[ loop ] = n_ways;
}

BLIS_INLINE void bli_rntm_set_jc_ways_only( dim_t ways, rntm_t* rntm )
{
	bli_rntm_set_ways_for_only( BLIS_NC, ways, rntm );
}
BLIS_INLINE void bli_rntm_set_pc_ways_only( dim_t ways, rntm_t* rntm )
{
	bli_rntm_set_ways_for_only( BLIS_KC, 1, rntm );
}
BLIS_INLINE void bli_rntm_set_ic_ways_only( dim_t ways, rntm_t* rntm )
{
	bli_rntm_set_ways_for_only( BLIS_MC, ways, rntm );
}
BLIS_INLINE void bli_rntm_set_jr_ways_only( dim_t ways, rntm_t* rntm )
{
	bli_rntm_set_ways_for_only( BLIS_NR, ways, rntm );
}
BLIS_INLINE void bli_rntm_set_ir_ways_only( dim_t ways, rntm_t* rntm )
{
	bli_rntm_set_ways_for_only( BLIS_MR, ways, rntm );
}
BLIS_INLINE void bli_rntm_set_pr_ways_only( dim_t ways, rntm_t* rntm )
{
	bli_rntm_set_ways_for_only( BLIS_KR, ways, rntm );
}

BLIS_INLINE void bli_rntm_set_ways_only( dim_t jc, dim_t pc, dim_t ic, dim_t jr, dim_t ir, rntm_t* rntm )
{
	// Record the number of ways of parallelism per loop.
	bli_rntm_set_jc_ways_only( jc, rntm );
	bli_rntm_set_pc_ways_only(  1, rntm );
	bli_rntm_set_ic_ways_only( ic, rntm );
	bli_rntm_set_jr_ways_only( jr, rntm );
	bli_rntm_set_ir_ways_only( ir, rntm );
	bli_rntm_set_pr_ways_only(  1, rntm );
}

BLIS_INLINE void bli_rntm_clear_num_threads_only( rntm_t* rntm )
{
	bli_rntm_set_num_threads_only( 1, rntm );
}
BLIS_INLINE void bli_rntm_clear_ways_only( rntm_t* rntm )
{
	bli_rntm_set_ways_only( 1, 1, 1, 1, 1, rntm );
}

//
// -- rntm_t modification (public API) -----------------------------------------
//

BLIS_INLINE void bli_rntm_set_thread_impl( timpl_t thread_impl, rntm_t* rntm )
{
	// Set the threading implementation to use.
	bli_rntm_set_thread_impl_only( thread_impl, rntm );
}

BLIS_INLINE void bli_rntm_set_pack_a( bool pack_a, rntm_t* rntm )
{
	// Set the bool indicating whether matrix A should be packed.
	rntm->pack_a = pack_a;
}
BLIS_INLINE void bli_rntm_set_pack_b( bool pack_b, rntm_t* rntm )
{
	// Set the bool indicating whether matrix B should be packed.
	rntm->pack_b = pack_b;
}

BLIS_INLINE void bli_rntm_set_l3_sup( bool l3_sup, rntm_t* rntm )
{
	// Set the bool indicating whether level-3 sup handling is enabled.
	rntm->l3_sup = l3_sup;
}
BLIS_INLINE void bli_rntm_enable_l3_sup( rntm_t* rntm )
{
	bli_rntm_set_l3_sup( TRUE, rntm );
}
BLIS_INLINE void bli_rntm_disable_l3_sup( rntm_t* rntm )
{
	bli_rntm_set_l3_sup( FALSE, rntm );
}

//
// -- rntm_t modification (internal use only) ----------------------------------
//

BLIS_INLINE void bli_rntm_clear_thread_impl( rntm_t* rntm )
{
	bli_rntm_set_thread_impl_only( BLIS_SINGLE, rntm );
}

BLIS_INLINE void bli_rntm_clear_auto_factor( rntm_t* rntm )
{
	bli_rntm_set_auto_factor_only( FALSE, rntm );
}
BLIS_INLINE void bli_rntm_clear_pack_a( rntm_t* rntm )
{
	bli_rntm_set_pack_a( FALSE, rntm );
}
BLIS_INLINE void bli_rntm_clear_pack_b( rntm_t* rntm )
{
	bli_rntm_set_pack_b( FALSE, rntm );
}
BLIS_INLINE void bli_rntm_clear_l3_sup( rntm_t* rntm )
{
	bli_rntm_set_l3_sup( TRUE, rntm );
}

//
// -- rntm_t initialization ----------------------------------------------------
//

// NOTE: Initialization is not necessary as long the user calls at least ONE
// of the public "set" accessors, each of which guarantees that the rntm_t
// will be in a good state upon return.

#define BLIS_RNTM_INITIALIZER \
        { \
          /* .thread_impl = */ BLIS_SINGLE, \
\
          /* .auto_factor = */ FALSE, \
\
          /* .num_threads = */ 1, \
          /* .thrloop     = */ { 1, 1, 1, 1, 1, 1 }, \
          /* .pack_a      = */ FALSE, \
          /* .pack_b      = */ FALSE, \
          /* .l3_sup      = */ TRUE, \
        }  \

#if 0
//BLIS_INLINE void bli_rntm_clear( rntm_t* rntm )
{
	bli_rntm_clear_thread_impl( rntm );

	bli_rntm_clear_num_threads_only( rntm );
	bli_rntm_clear_ways_only( rntm );

	bli_rntm_clear_auto_factor( rntm );
	bli_rntm_clear_pack_a( rntm );
	bli_rntm_clear_pack_b( rntm );
	bli_rntm_clear_l3_sup( rntm );
}
#endif

//
// -- rntm_t total thread calculation ------------------------------------------
//

BLIS_INLINE dim_t bli_rntm_calc_num_threads
     (
       const rntm_t* rntm
     )
{
	dim_t n_threads;

	n_threads  = bli_rntm_ways_for( BLIS_NC, rntm );
	n_threads *= bli_rntm_ways_for( BLIS_KC, rntm );
	n_threads *= bli_rntm_ways_for( BLIS_MC, rntm );
	n_threads *= bli_rntm_ways_for( BLIS_NR, rntm );
	n_threads *= bli_rntm_ways_for( BLIS_MR, rntm );

	return n_threads;
}

//
// -- Function prototypes ------------------------------------------------------
//

rntm_t*              bli_global_rntm( void );
rntm_t*              bli_global_rntm_at_init( void );
bli_pthread_mutex_t* bli_global_rntm_mutex( void );

int bli_rntm_init( void );
int bli_rntm_finalize( void );

void bli_rntm_init_from_env( rntm_t* rntm );

BLIS_EXPORT_BLIS void bli_rntm_init_from_global( rntm_t* rntm );

BLIS_EXPORT_BLIS void bli_rntm_set_num_threads
     (
       dim_t   nt,
       rntm_t* rntm
     );

BLIS_EXPORT_BLIS void bli_rntm_set_ways
     (
       dim_t   jc,
       dim_t   pc,
       dim_t   ic,
       dim_t   jr,
       dim_t   ir,
       rntm_t* rntm
     );

void bli_rntm_sanitize
     (
       rntm_t* rntm
     );

void bli_rntm_factorize
     (
       dim_t   m,
       dim_t   n,
       dim_t   k,
       rntm_t* rntm
     );

void bli_rntm_factorize_sup
     (
       dim_t   m,
       dim_t   n,
       dim_t   k,
       rntm_t* rntm
     );

void bli_rntm_print
     (
       const rntm_t* rntm
     );

dim_t bli_rntm_calc_num_threads_in
     (
       const bszid_t* bszid_cur,
       const rntm_t*  rntm
     );

#endif

// end bli_rntm.h
#line 132 "../blis-2.0/frame/include/blis.h"

// begin bli_gks.h
#line 1 "../blis-2.0/frame/base//bli_gks.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2024, Southern Methodist University

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_GKS_H
#define BLIS_GKS_H

int                            bli_gks_init( void );
int                            bli_gks_finalize( void );

void                           bli_gks_init_index( void );

BLIS_EXPORT_BLIS const cntx_t* bli_gks_lookup_id( arch_t id );
void                           bli_gks_register_cntx( arch_t id, void_fp nat_fp, void_fp ref_fp );

BLIS_EXPORT_BLIS const cntx_t* bli_gks_query_cntx( void );
const cntx_t*                  bli_gks_query_cntx_noinit( void );
const cntx_t*                  bli_gks_query_cntx_impl( void );

BLIS_EXPORT_BLIS void          bli_gks_init_ref_cntx( cntx_t* cntx );

bool                           bli_gks_cntx_ukr_is_ref( num_t dt, ukr_t ukr_id, const cntx_t* cntx );
bool                           bli_gks_cntx_ukr2_is_ref( num_t dt1, num_t dt2, ukr_t ukr_id, const cntx_t* cntx );

BLIS_EXPORT_BLIS const char*   bli_gks_l3_ukr_impl_string( ukr_t ukr, ind_t method, num_t dt );
BLIS_EXPORT_BLIS kimpl_t       bli_gks_l3_ukr_impl_type( ukr_t ukr, ind_t method, num_t dt );

//char*                          bli_gks_l3_ukr_avail_impl_string( ukr_t ukr, num_t dt );

BLIS_EXPORT_BLIS err_t bli_gks_register_blksz( kerid_t* bs_id );

BLIS_EXPORT_BLIS err_t bli_gks_register_ukr( kerid_t* ukr_id );

BLIS_EXPORT_BLIS err_t bli_gks_register_ukr2( kerid_t* ukr_id );

BLIS_EXPORT_BLIS err_t bli_gks_register_ukr_pref( kerid_t* ukr_pref_id );

#endif

// end bli_gks.h
#line 133 "../blis-2.0/frame/include/blis.h"

// begin bli_ind.h
#line 1 "../blis-2.0/frame/base//bli_ind.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_IND_H
#define BLIS_IND_H

// level-3 induced method management
// begin bli_l3_ind.h
#line 1 "../blis-2.0/frame/3//bli_l3_ind.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2020, Advanced Micro Devices, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_L3_IND_H
#define BLIS_L3_IND_H

// -----------------------------------------------------------------------------

#undef  GENPROT
#define GENPROT( opname ) \
\
ind_t   PASTEMAC(opname,ind_find_avail)( num_t dt );
/*bool PASTEMAC(opname,ind_has_avail)( num_t dt ); */

GENPROT( gemm )
GENPROT( gemmt )
GENPROT( hemm )
GENPROT( symm )
GENPROT( trmm3 )
GENPROT( trmm )
GENPROT( trsm )

// -----------------------------------------------------------------------------

//bool bli_l3_ind_oper_is_avail( opid_t oper, ind_t method, num_t dt );

ind_t   bli_l3_ind_oper_find_avail( opid_t oper, num_t dt );

void    bli_l3_ind_set_enable_dt( ind_t method, num_t dt, bool status );

void    bli_l3_ind_oper_enable_only( opid_t oper, ind_t method, num_t dt );
void    bli_l3_ind_oper_set_enable_all( opid_t oper, num_t dt, bool status );

void    bli_l3_ind_oper_set_enable( opid_t oper, ind_t method, num_t dt, bool status );
bool    bli_l3_ind_oper_get_enable( opid_t oper, ind_t method, num_t dt );

bool    bli_l3_ind_oper_is_impl( opid_t oper, ind_t method );


#endif

// end bli_l3_ind.h
#line 40 "../blis-2.0/frame/base//bli_ind.h"


int                          bli_ind_init( void );
int                          bli_ind_finalize( void );

BLIS_EXPORT_BLIS void        bli_ind_enable( ind_t method );
BLIS_EXPORT_BLIS void        bli_ind_disable( ind_t method );
BLIS_EXPORT_BLIS void        bli_ind_disable_all( void );

BLIS_EXPORT_BLIS void        bli_ind_enable_dt( ind_t method, num_t dt );
BLIS_EXPORT_BLIS void        bli_ind_disable_dt( ind_t method, num_t dt );
BLIS_EXPORT_BLIS void        bli_ind_disable_all_dt( num_t dt );

BLIS_EXPORT_BLIS void        bli_ind_oper_enable_only( opid_t oper, ind_t method, num_t dt );

BLIS_EXPORT_BLIS bool        bli_ind_oper_is_impl( opid_t oper, ind_t method );
BLIS_EXPORT_BLIS ind_t       bli_ind_oper_find_avail( opid_t oper, num_t dt );
BLIS_EXPORT_BLIS const char* bli_ind_oper_get_avail_impl_string( opid_t oper, num_t dt );

const char*                  bli_ind_get_impl_string( ind_t method );
num_t                        bli_ind_map_cdt_to_index( num_t dt );


#endif

// end bli_ind.h
#line 134 "../blis-2.0/frame/include/blis.h"

// begin bli_pba.h
#line 1 "../blis-2.0/frame/base//bli_pba.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2016, Hewlett Packard Enterprise Development LP
   Copyright (C) 2018 - 2019, Advanced Micro Devices, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_PBA_H
#define BLIS_PBA_H

// Packing block allocator (formerly memory broker)

/*
typedef struct pba_s
{
	pool_t              pools[3];
	bli_pthread_mutex_t mutex;

	// These fields are used for general-purpose allocation.
	siz_t               align_size;
	malloc_ft           malloc_fp;
	free_ft             free_fp;

} pba_t;
*/


// pba init

//BLIS_INLINE void bli_pba_init_mutex( pba_t* pba )
//{
//	bli_pthread_mutex_init( &(pba->mutex), NULL );
//}

//BLIS_INLINE void bli_pba_finalize_mutex( pba_t* pba )
//{
//	bli_pthread_mutex_destroy( &(pba->mutex) );
//}

// pba query

BLIS_INLINE pool_t* bli_pba_pool( dim_t pool_index, pba_t* pba )
{
	return &(pba->pools[ pool_index ]);
}

BLIS_INLINE siz_t bli_pba_align_size( const pba_t* pba )
{
	return pba->align_size;
}

BLIS_INLINE malloc_ft bli_pba_malloc_fp( const pba_t* pba )
{
	return pba->malloc_fp;
}

BLIS_INLINE free_ft bli_pba_free_fp( const pba_t* pba )
{
	return pba->free_fp;
}

// pba modification

BLIS_INLINE void bli_pba_set_align_size( siz_t align_size, pba_t* pba )
{
	pba->align_size = align_size;
}

BLIS_INLINE void bli_pba_set_malloc_fp( malloc_ft malloc_fp, pba_t* pba )
{
	pba->malloc_fp = malloc_fp;
}

BLIS_INLINE void bli_pba_set_free_fp( free_ft free_fp, pba_t* pba )
{
	pba->free_fp = free_fp;
}

// pba action

BLIS_INLINE void bli_pba_lock( pba_t* pba )
{
	bli_pthread_mutex_lock( &(pba->mutex) );
}

BLIS_INLINE void bli_pba_unlock( pba_t* pba )
{
	bli_pthread_mutex_unlock( &(pba->mutex) );
}

// -----------------------------------------------------------------------------

BLIS_EXPORT_BLIS pba_t* bli_pba_query( void );

void bli_pba_init
     (
       const cntx_t* cntx
     );
void bli_pba_finalize
     (
       void
     );

void bli_pba_acquire_m
     (
       pba_t*    pba,
       siz_t     req_size,
       packbuf_t buf_type,
       mem_t*    mem
     );

void bli_pba_release
     (
       pba_t* pba,
       mem_t* mem
     );

siz_t bli_pba_pool_size
     (
       const pba_t*    pba,
             packbuf_t buf_type
     );

// ----------------------------------------------------------------------------

void bli_pba_init_pools
     (
       const cntx_t* cntx,
             pba_t*  pba
     );
void bli_pba_finalize_pools
     (
       pba_t* pba
     );

void bli_pba_compute_pool_block_sizes
     (
             siz_t*  bs_a,
             siz_t*  bs_b,
             siz_t*  bs_c,
       const cntx_t* cntx
     );
void bli_pba_compute_pool_block_sizes_dt
     (
             num_t   dt,
             siz_t*  bs_a,
             siz_t*  bs_b,
             siz_t*  bs_c,
       const cntx_t* cntx
     );

#endif

// end bli_pba.h
#line 135 "../blis-2.0/frame/include/blis.h"

// begin bli_pool.h
#line 1 "../blis-2.0/frame/base//bli_pool.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2018 - 2019, Advanced Micro Devices, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_POOL_H
#define BLIS_POOL_H

// -- Pool block type --

/*
typedef struct
{
	void*     buf;
	siz_t     block_size;

} pblk_t;
*/

// -- Pool type --

/*
typedef struct
{
	void*     block_ptrs;
	siz_t     block_ptrs_len;

	siz_t     top_index;
	siz_t     num_blocks;

	siz_t     block_size;
	siz_t     align_size;

	malloc_ft malloc_fp;
	free_ft   free_fp;

} pool_t;
*/


// Pool block query

BLIS_INLINE void* bli_pblk_buf( const pblk_t* pblk )
{
	return pblk->buf;
}

BLIS_INLINE siz_t bli_pblk_block_size( const pblk_t* pblk )
{
	return pblk->block_size;
}

// Pool block modification

BLIS_INLINE void bli_pblk_set_buf( void* buf, pblk_t* pblk )
{
	pblk->buf = buf;
}

BLIS_INLINE void bli_pblk_set_block_size( siz_t block_size, pblk_t* pblk )
{
	pblk->block_size = block_size;
}

//
// -- pool block initialization ------------------------------------------------
//

// NOTE: This initializer macro must be updated whenever fields are added or
// removed from the pblk_t type definition. An alternative to the initializer is
// calling bli_pblk_clear() at runtime.

#define BLIS_PBLK_INITIALIZER \
        { \
          /* .buf        = */ NULL, \
          /* .block_size = */ 0, \
        }  \

BLIS_INLINE void bli_pblk_clear( pblk_t* pblk )
{
	bli_pblk_set_buf( NULL, pblk );
	bli_pblk_set_block_size( 0, pblk );
}


// Pool entry query

BLIS_INLINE void* bli_pool_block_ptrs( const pool_t* pool )
{
	return pool->block_ptrs;
}

BLIS_INLINE siz_t bli_pool_block_ptrs_len( const pool_t* pool )
{
	return pool->block_ptrs_len;
}

BLIS_INLINE siz_t bli_pool_num_blocks( const pool_t* pool )
{
	return pool->num_blocks;
}

BLIS_INLINE siz_t bli_pool_block_size( const pool_t* pool )
{
	return pool->block_size;
}

BLIS_INLINE siz_t bli_pool_align_size( const pool_t* pool )
{
	return pool->align_size;
}

BLIS_INLINE siz_t bli_pool_offset_size( const pool_t* pool )
{
	return pool->offset_size;
}

BLIS_INLINE malloc_ft bli_pool_malloc_fp( const pool_t* pool )
{
	return pool->malloc_fp;
}

BLIS_INLINE free_ft bli_pool_free_fp( const pool_t* pool )
{
	return pool->free_fp;
}

BLIS_INLINE siz_t bli_pool_top_index( const pool_t* pool )
{
	return pool->top_index;
}

BLIS_INLINE bool bli_pool_is_exhausted( const pool_t* pool )
{
	return ( bool )
	       ( bli_pool_top_index( pool ) == bli_pool_num_blocks( pool ) );
}

// Pool entry modification

BLIS_INLINE void bli_pool_set_block_ptrs( void* block_ptrs, pool_t* pool ) \
{
	pool->block_ptrs = block_ptrs;
}

BLIS_INLINE void bli_pool_set_block_ptrs_len( siz_t block_ptrs_len, pool_t* pool ) \
{
	pool->block_ptrs_len = block_ptrs_len;
}

BLIS_INLINE void bli_pool_set_num_blocks( siz_t num_blocks, pool_t* pool ) \
{
	pool->num_blocks = num_blocks;
}

BLIS_INLINE void bli_pool_set_block_size( siz_t block_size, pool_t* pool ) \
{
	pool->block_size = block_size;
}

BLIS_INLINE void bli_pool_set_align_size( siz_t align_size, pool_t* pool ) \
{
	pool->align_size = align_size;
}

BLIS_INLINE void bli_pool_set_offset_size( siz_t offset_size, pool_t* pool ) \
{
	pool->offset_size = offset_size;
}

BLIS_INLINE void bli_pool_set_malloc_fp( malloc_ft malloc_fp, pool_t* pool ) \
{
	pool->malloc_fp = malloc_fp;
}

BLIS_INLINE void bli_pool_set_free_fp( free_ft free_fp, pool_t* pool ) \
{
	pool->free_fp = free_fp;
}

BLIS_INLINE void bli_pool_set_top_index( siz_t top_index, pool_t* pool ) \
{
	pool->top_index = top_index;
}

// -----------------------------------------------------------------------------

void bli_pool_init
     (
       siz_t     num_blocks,
       siz_t     block_ptrs_len,
       siz_t     block_size,
       siz_t     align_size,
       siz_t     offset_size,
       malloc_ft malloc_fp,
       free_ft   free_fp,
       pool_t*   pool
     );
void bli_pool_finalize
     (
       pool_t* pool,
       bool    reinit
     );
void bli_pool_reinit
     (
       siz_t   num_blocks_new,
       siz_t   block_ptrs_len_new,
       siz_t   block_size_new,
       siz_t   align_size_new,
       siz_t   offset_size_new,
       pool_t* pool
     );

void bli_pool_checkout_block
     (
       siz_t   req_size,
       pblk_t* block,
       pool_t* pool
     );
void bli_pool_checkin_block
     (
       pblk_t* block,
       pool_t* pool
     );

void bli_pool_grow
     (
       siz_t   num_blocks_add,
       pool_t* pool
     );
void bli_pool_shrink
     (
       siz_t   num_blocks_sub,
       pool_t* pool
     );

void bli_pool_alloc_block
     (
       siz_t     block_size,
       siz_t     align_size,
       siz_t     offset_size,
       malloc_ft malloc_fp,
       pblk_t*   block
     );
void bli_pool_free_block
     (
       siz_t   offset_size,
       free_ft free_fp,
       pblk_t* block
     );

void bli_pool_print
     (
       const pool_t* pool
     );
void bli_pblk_print
     (
       const pblk_t* pblk
     );

#endif

// end bli_pool.h
#line 136 "../blis-2.0/frame/include/blis.h"

// begin bli_array.h
#line 1 "../blis-2.0/frame/base//bli_array.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2018 - 2019, Advanced Micro Devices, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_ARRAY_H
#define BLIS_ARRAY_H

// -- Array type --

/*
typedef struct
{
	void*     buf;

	siz_t     num_elem;
	siz_t     elem_size;

} array_t;
*/


// Array entry query

BLIS_INLINE void* bli_array_buf( const array_t* array )
{
	return array->buf;
}

BLIS_INLINE siz_t bli_array_num_elem( const array_t* array )
{
	return array->num_elem;
}

BLIS_INLINE siz_t bli_array_elem_size( const array_t* array )
{
	return array->elem_size;
}

// Array entry modification

BLIS_INLINE void bli_array_set_buf( void* buf, array_t* array ) \
{
	array->buf = buf;
}

BLIS_INLINE void bli_array_set_num_elem( siz_t num_elem, array_t* array ) \
{
	array->num_elem = num_elem;
}

BLIS_INLINE void bli_array_set_elem_size( siz_t elem_size, array_t* array ) \
{
	array->elem_size = elem_size;
}

// -----------------------------------------------------------------------------

void bli_array_init
     (
       siz_t    num_elem,
       siz_t    elem_size,
       array_t* array
     );
void bli_array_resize
     (
       siz_t    num_elem_new,
       array_t* array
     );
void bli_array_finalize
     (
       array_t* array
     );

void* bli_array_elem
     (
             siz_t    index,
       const array_t* array
     );
void bli_array_set_elem
     (
       void*    elem,
       siz_t    index,
       array_t* array
     );

#endif

// end bli_array.h
#line 137 "../blis-2.0/frame/include/blis.h"

// begin bli_apool.h
#line 1 "../blis-2.0/frame/base//bli_apool.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2018 - 2019, Advanced Micro Devices, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_APOOL_H
#define BLIS_APOOL_H

// -- Locked pool-of-arrays type --

/*
typedef struct
{
	bli_pthread_mutex_t mutex;
	pool_t              pool;

	siz_t               def_array_len;

} apool_t;
*/


// apool entry query

BLIS_INLINE pool_t* bli_apool_pool( apool_t* apool )
{
	return &(apool->pool);
}

BLIS_INLINE bli_pthread_mutex_t* bli_apool_mutex( apool_t* apool )
{
	return &(apool->mutex);
}

BLIS_INLINE siz_t bli_apool_def_array_len( const apool_t* pool )
{
	return pool->def_array_len;
}

BLIS_INLINE bool bli_apool_is_exhausted( const apool_t* apool )
{
	return bli_pool_is_exhausted( &apool->pool );
}

// apool action

BLIS_INLINE void bli_apool_lock( apool_t* apool )
{
	bli_pthread_mutex_lock( bli_apool_mutex( apool ) );
}

BLIS_INLINE void bli_apool_unlock( apool_t* apool )
{
	bli_pthread_mutex_unlock( bli_apool_mutex( apool ) );
}

// apool entry modification

BLIS_INLINE void bli_apool_set_def_array_len( siz_t def_array_len, apool_t* pool ) \
{
	pool->def_array_len = def_array_len;
}

// -----------------------------------------------------------------------------

void bli_apool_init
     (
       apool_t* apool
     );
void bli_apool_finalize
     (
       apool_t* apool
     );

array_t* bli_apool_checkout_array
     (
       siz_t    n_threads,
       apool_t* apool
     );
void bli_apool_checkin_array
     (
       array_t* array,
       apool_t* apool
     );

pool_t* bli_apool_array_elem
     (
       siz_t    index,
       array_t* array
     );

void bli_apool_grow
     (
       siz_t    num_blocks_add,
       apool_t* apool
     );

void bli_apool_alloc_block
     (
       siz_t     num_elem,
       array_t** array_p
     );
void bli_apool_free_block
     (
       array_t* array
     );


#endif

// end bli_apool.h
#line 138 "../blis-2.0/frame/include/blis.h"

// begin bli_sba.h
#line 1 "../blis-2.0/frame/base//bli_sba.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2018 - 2019, Advanced Micro Devices, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SBA_H
#define BLIS_SBA_H

apool_t* bli_sba_query( void );

// -----------------------------------------------------------------------------

void bli_sba_init( void );
void bli_sba_finalize( void );

void* bli_sba_acquire
     (
       pool_t* sba_pool,
       siz_t   req_size
     );

void bli_sba_release
     (
       pool_t* sba_pool,
       void*   block
     );

array_t* bli_sba_checkout_array
     (
       siz_t n_threads
     );

void bli_sba_checkin_array
     (
       array_t* array
     );

pool_t* bli_sba_array_elem
     (
       siz_t    index,
       array_t* array
     );

#endif

// end bli_sba.h
#line 139 "../blis-2.0/frame/include/blis.h"

// begin bli_memsys.h
#line 1 "../blis-2.0/frame/base//bli_memsys.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2016, Hewlett Packard Enterprise Development LP
   Copyright (C) 2018 - 2019, Advanced Micro Devices, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_MEMSYS_H
#define BLIS_MEMSYS_H

int bli_memsys_init( void );
int bli_memsys_finalize( void );


#endif

// end bli_memsys.h
#line 140 "../blis-2.0/frame/include/blis.h"

// begin bli_mem.h
#line 1 "../blis-2.0/frame/base//bli_mem.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2016, Hewlett Packard Enterprise Development LP
   Copyright (C) 2018 - 2019, Advanced Micro Devices, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/


#ifndef BLIS_MEM_H
#define BLIS_MEM_H


// mem_t object type (defined in bli_type_defs.h)

/*
typedef struct mem_s
{
	pblk_t    pblk;
	packbuf_t buf_type;
	pool_t*   pool;
	siz_t     size;
} mem_t;

typedef struct
{
	void*     buf;
	siz_t     block_size;
} pblk_t;
*/

//
// -- mem_t query --------------------------------------------------------------
//

BLIS_INLINE pblk_t* bli_mem_pblk( mem_t* mem )
{
	return &(mem->pblk);
}

BLIS_INLINE void* bli_mem_buffer( const mem_t* mem )
{
	return bli_pblk_buf( bli_mem_pblk( ( mem_t* )mem ) );
}

BLIS_INLINE packbuf_t bli_mem_buf_type( const mem_t* mem )
{
	return mem->buf_type;
}

BLIS_INLINE pool_t* bli_mem_pool( const mem_t* mem )
{
	return mem->pool;
}

BLIS_INLINE siz_t bli_mem_size( const mem_t* mem )
{
	return mem->size;
}

BLIS_INLINE bool bli_mem_is_alloc( const mem_t* mem )
{
	return ( bool )
	       ( bli_mem_buffer( mem ) != NULL );
}

BLIS_INLINE bool bli_mem_is_unalloc( const mem_t* mem )
{
	return ( bool )
	       ( bli_mem_buffer( mem ) == NULL );
}


//
// -- mem_t modification -------------------------------------------------------
//

BLIS_INLINE void bli_mem_set_pblk( pblk_t* pblk, mem_t* mem )
{
	mem->pblk = *pblk;
}

BLIS_INLINE void bli_mem_set_buffer( void* buf, mem_t* mem )
{
	bli_pblk_set_buf( buf, &(mem->pblk) );
}

BLIS_INLINE void bli_mem_set_buf_type( packbuf_t buf_type, mem_t* mem )
{
	mem->buf_type = buf_type;
}

BLIS_INLINE void bli_mem_set_pool( pool_t* pool, mem_t* mem )
{
	mem->pool = pool;
}

BLIS_INLINE void bli_mem_set_size( siz_t size, mem_t* mem )
{
	mem->size = size;
}

//
// -- mem_t initialization -----------------------------------------------------
//

// NOTE: This initializer macro must be updated whenever fields are added or
// removed from the mem_t type definition. An alternative to the initializer is
// calling bli_mem_clear() at runtime.

#define BLIS_MEM_INITIALIZER \
        { \
          /* .pblk     = */ BLIS_PBLK_INITIALIZER, \
          /* .buf_type = */ BLIS_BUFFER_FOR_GEN_USE, \
          /* .pool     = */ NULL, \
          /* .size     = */ 0, \
        }


BLIS_INLINE void bli_mem_clear( mem_t* mem )
{
	bli_mem_set_buffer( NULL, mem );
	bli_mem_set_buf_type( BLIS_BUFFER_FOR_GEN_USE, mem );
	bli_mem_set_pool( NULL, mem );
	bli_mem_set_size( 0, mem );
}


#endif
// end bli_mem.h
#line 141 "../blis-2.0/frame/include/blis.h"

// begin bli_part.h
#line 1 "../blis-2.0/frame/base//bli_part.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

// begin bli_part_check.h
#line 1 "../blis-2.0/frame/base/check//bli_part_check.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

void bli_acquire_mpart_t2b_check(       subpart_t requested_part,
                                        dim_t     i,
                                        dim_t     b,
                                  const obj_t*    obj,
                                  const obj_t*    sub_obj );

void bli_acquire_mpart_l2r_check(       subpart_t requested_part,
                                        dim_t     j,
                                        dim_t     b,
                                  const obj_t*    obj,
                                  const obj_t*    sub_obj );

void bli_acquire_mpart_tl2br_check(       subpart_t requested_part,
                                          dim_t     ij,
                                          dim_t     b,
                                    const obj_t*    obj,
                                    const obj_t*    sub_obj );

// end bli_part_check.h
#line 36 "../blis-2.0/frame/base//bli_part.h"


// -- Matrix partitioning ------------------------------------------------------

BLIS_EXPORT_BLIS void bli_acquire_mpart
     (
             dim_t  i,
             dim_t  j,
             dim_t  m,
             dim_t  n,
       const obj_t* obj,
             obj_t* sub_obj
     );

#undef  GENPROT
#define GENPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC( opname ) \
     ( \
             subpart_t req_part, \
             dim_t     i, \
             dim_t     b, \
       const obj_t*    obj, \
             obj_t*    sub_obj \
     );

GENPROT( acquire_mpart_t2b )
GENPROT( acquire_mpart_b2t )
GENPROT( acquire_mpart_l2r )
GENPROT( acquire_mpart_r2l )
GENPROT( acquire_mpart_tl2br )
GENPROT( acquire_mpart_br2tl )


#undef  GENPROT
#define GENPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC( opname ) \
     ( \
             dir_t     direct, \
             subpart_t req_part, \
             dim_t     i, \
             dim_t     b, \
       const obj_t*    obj, \
             obj_t*    sub_obj \
     );

GENPROT( acquire_mpart_mdim )
GENPROT( acquire_mpart_ndim )
GENPROT( acquire_mpart_mndim )


// -- Vector partitioning ------------------------------------------------------

#undef  GENPROT
#define GENPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC( opname ) \
     ( \
             subpart_t req_part, \
             dim_t     i, \
             dim_t     b, \
       const obj_t*    obj, \
             obj_t*    sub_obj \
     );

GENPROT( acquire_vpart_f2b )
GENPROT( acquire_vpart_b2f )

// -- Scalar acquisition -------------------------------------------------------

BLIS_EXPORT_BLIS void bli_acquire_mij
     (
             dim_t  i,
             dim_t  j,
       const obj_t* obj,
             obj_t* sub_obj
     );

BLIS_EXPORT_BLIS void bli_acquire_vi
     (
             dim_t  i,
       const obj_t* obj,
             obj_t* sub_obj
     );

// end bli_part.h
#line 142 "../blis-2.0/frame/include/blis.h"

// begin bli_part_cntl.h
#line 1 "../blis-2.0/frame/base//bli_part_cntl.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2024, Southern Methodist University

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/


struct part_cntl_s
{
	cntl_t cntl; // cntl field must be present and come first.
	num_t  b_dt;
	dim_t  b_alg;
	dim_t  b_max;
	dim_t  b_scale;
	dim_t  b_mult;
	dim_t  b_mult_scale;
	dir_t  direct;
	bool   use_weighted;
};
typedef struct part_cntl_s part_cntl_t;

// -----------------------------------------------------------------------------

BLIS_INLINE dim_t bli_part_cntl_blksz_alg( const cntl_t* cntl )
{
	return ( ( const part_cntl_t* )cntl )->b_alg;
}

BLIS_INLINE dim_t bli_part_cntl_blksz_max( const cntl_t* cntl )
{
	return ( ( const part_cntl_t* )cntl )->b_max;
}

BLIS_INLINE dim_t bli_part_cntl_blksz_mult( const cntl_t* cntl )
{
	return ( ( const part_cntl_t* )cntl )->b_mult;
}

BLIS_INLINE dir_t bli_part_cntl_direct( const cntl_t* cntl )
{
	return ( ( const part_cntl_t* )cntl )->direct;
}

BLIS_INLINE bool bli_part_cntl_use_weighted( const cntl_t* cntl )
{
	return ( ( const part_cntl_t* )cntl )->use_weighted;
}

// -----------------------------------------------------------------------------

BLIS_INLINE void bli_part_cntl_set_blksz( const blksz_t* blksz, cntl_t* cntl_ )
{
	part_cntl_t* cntl = ( part_cntl_t* )cntl_;
	num_t dt = cntl->b_dt;
	cntl->b_alg = bli_blksz_get_def( dt, blksz ) / cntl->b_scale;
	cntl->b_max = bli_blksz_get_max( dt, blksz ) / cntl->b_scale;
}

BLIS_INLINE void bli_part_cntl_set_blksz_mult( const blksz_t* blksz, cntl_t* cntl_ )
{
	part_cntl_t* cntl = ( part_cntl_t* )cntl_;
	num_t dt = cntl->b_dt;
	cntl->b_mult = bli_blksz_get_def( dt, blksz ) / cntl->b_mult_scale;
}

BLIS_INLINE void bli_part_cntl_set_direct( dir_t direct, cntl_t* cntl )
{
	( ( part_cntl_t* )cntl )->direct = direct;
}

BLIS_INLINE void bli_part_cntl_set_use_weighted( bool use_weighted, cntl_t* cntl )
{
	( ( part_cntl_t* )cntl )->use_weighted = use_weighted;
}

BLIS_INLINE void bli_part_cntl_align_blksz_to_mult( dim_t mult, bool round_up, cntl_t* cntl_ )
{
	part_cntl_t* cntl = ( part_cntl_t* )cntl_;
	cntl->b_alg = bli_align_dim_to_mult( cntl->b_alg, mult, round_up );
	cntl->b_max = bli_align_dim_to_mult( cntl->b_max, mult, round_up );
}

BLIS_INLINE void bli_part_cntl_align_blksz( bool round_up, cntl_t* cntl )
{
	bli_part_cntl_align_blksz_to_mult( ( ( part_cntl_t* )cntl )->b_mult, round_up, cntl );
}

// -----------------------------------------------------------------------------

void bli_part_cntl_init_node
     (
       void_fp      var_func,
       num_t        b_dt,
       dim_t        b_alg,
       dim_t        b_max,
       dim_t        b_scale,
       dim_t        b_mult,
       dim_t        b_mult_scale,
       dir_t        direct,
       bool         use_weighted,
       part_cntl_t* cntl
     );

// end bli_part_cntl.h
#line 143 "../blis-2.0/frame/include/blis.h"

// begin bli_prune.h
#line 1 "../blis-2.0/frame/base//bli_prune.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

void bli_prune_unref_mparts( obj_t* p, mdim_t mdim_p,
                             obj_t* s, mdim_t mdim_s );
// end bli_prune.h
#line 144 "../blis-2.0/frame/include/blis.h"

// begin bli_query.h
#line 1 "../blis-2.0/frame/base//bli_query.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

BLIS_EXPORT_BLIS bool bli_obj_equals( const obj_t* a, const obj_t* b );

BLIS_EXPORT_BLIS bool bli_obj_imag_equals( const obj_t* a, const obj_t* b );

BLIS_EXPORT_BLIS bool bli_obj_imag_is_zero( const obj_t* a );
// end bli_query.h
#line 145 "../blis-2.0/frame/include/blis.h"

// begin bli_auxinfo.h
#line 1 "../blis-2.0/frame/base//bli_auxinfo.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_AUXINFO_MACRO_DEFS_H
#define BLIS_AUXINFO_MACRO_DEFS_H


// auxinfo_t field query

BLIS_INLINE pack_t bli_auxinfo_schema_a( const auxinfo_t* ai )
{
	return ai->schema_a;
}
BLIS_INLINE pack_t bli_auxinfo_schema_b( const auxinfo_t* ai )
{
	return ai->schema_b;
}

BLIS_INLINE const void* bli_auxinfo_next_a( const auxinfo_t* ai )
{
	return ai->a_next;
}
BLIS_INLINE const void* bli_auxinfo_next_b( const auxinfo_t* ai )
{
	return ai->b_next;
}

BLIS_INLINE inc_t bli_auxinfo_is_a( const auxinfo_t* ai )
{
	return ai->is_a;
}
BLIS_INLINE inc_t bli_auxinfo_is_b( const auxinfo_t* ai )
{
	return ai->is_b;
}

BLIS_INLINE inc_t bli_auxinfo_ps_a( const auxinfo_t* ai )
{
	return ai->ps_a;
}
BLIS_INLINE inc_t bli_auxinfo_ps_b( const auxinfo_t* ai )
{
	return ai->ps_b;
}

BLIS_INLINE inc_t bli_auxinfo_off_m( const auxinfo_t* ai )
{
	return ai->off_m;
}
BLIS_INLINE inc_t bli_auxinfo_off_n( const auxinfo_t* ai )
{
	return ai->off_n;
}

BLIS_INLINE void_fp bli_auxinfo_ukr( const auxinfo_t* ai )
{
	return ai->ukr;
}
BLIS_INLINE const void* bli_auxinfo_params( const auxinfo_t* ai )
{
	return ai->params;
}


// auxinfo_t field modification

BLIS_INLINE void bli_auxinfo_set_schema_a( pack_t schema, auxinfo_t* ai )
{
	ai->schema_a = schema;
}
BLIS_INLINE void bli_auxinfo_set_schema_b( pack_t schema, auxinfo_t* ai )
{
	ai->schema_b = schema;
}

BLIS_INLINE void bli_auxinfo_set_next_a( const void* p, auxinfo_t* ai )
{
	ai->a_next = p;
}
BLIS_INLINE void bli_auxinfo_set_next_b( const void* p, auxinfo_t* ai )
{
	ai->b_next = p;
}
BLIS_INLINE void bli_auxinfo_set_next_ab( const void* ap, const void* bp, auxinfo_t* ai )
{
	ai->a_next = ap;
	ai->b_next = bp;
}

BLIS_INLINE void bli_auxinfo_set_is_a( inc_t is, auxinfo_t* ai )
{
	ai->is_a = is;
}
BLIS_INLINE void bli_auxinfo_set_is_b( inc_t is, auxinfo_t* ai )
{
	ai->is_b = is;
}

BLIS_INLINE void bli_auxinfo_set_ps_a( inc_t ps, auxinfo_t* ai )
{
	ai->ps_a = ps;
}
BLIS_INLINE void bli_auxinfo_set_ps_b( inc_t ps, auxinfo_t* ai )
{
	ai->ps_b = ps;
}

BLIS_INLINE void bli_auxinfo_set_off_m( dim_t off, auxinfo_t* ai )
{
	ai->off_m = off;
}
BLIS_INLINE void bli_auxinfo_set_off_n( dim_t off, auxinfo_t* ai )
{
	ai->off_n = off;
}

BLIS_INLINE void bli_auxinfo_set_ukr( void_fp ukr, auxinfo_t* ai )
{
	ai->ukr = ukr;
}
BLIS_INLINE void bli_auxinfo_set_params( const void* params, auxinfo_t* ai )
{
	ai->params = params;
}

#endif

// end bli_auxinfo.h
#line 146 "../blis-2.0/frame/include/blis.h"

// begin bli_param_map.h
#line 1 "../blis-2.0/frame/base//bli_param_map.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2019, Advanced Micro Devices, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/


// --- BLIS to BLAS/LAPACK mappings --------------------------------------------

BLIS_EXPORT_BLIS void bli_param_map_blis_to_netlib_side( side_t side, char* blas_side );
BLIS_EXPORT_BLIS void bli_param_map_blis_to_netlib_uplo( uplo_t uplo, char* blas_uplo );
BLIS_EXPORT_BLIS void bli_param_map_blis_to_netlib_trans( trans_t trans, char* blas_trans );
BLIS_EXPORT_BLIS void bli_param_map_blis_to_netlib_diag( diag_t diag, char* blas_diag );
BLIS_EXPORT_BLIS void bli_param_map_blis_to_netlib_machval( machval_t machval, char* blas_machval );


// --- BLAS/LAPACK to BLIS mappings --------------------------------------------

// NOTE: These static functions were converted from regular functions in order
// to reduce function call overhead within the BLAS compatibility layer.

BLIS_INLINE void bli_param_map_netlib_to_blis_side( char side, side_t* blis_side )
{
	if      ( side == 'l' || side == 'L' ) *blis_side = BLIS_LEFT;
	else if ( side == 'r' || side == 'R' ) *blis_side = BLIS_RIGHT;
	else
	{
		// Instead of reporting an error to the framework, default to
		// an arbitrary value. This is needed because this function is
		// called by the BLAS compatibility layer AFTER it has already
		// checked errors and called xerbla(). If the application wants
		// to override the BLAS compatibility layer's xerbla--which
		// responds to errors with abort()--we need to also NOT call
		// abort() here, since either way it has already been dealt
		// with.
		//bli_check_error_code( BLIS_INVALID_SIDE );
		*blis_side = BLIS_LEFT;
	}
}

BLIS_INLINE void bli_param_map_netlib_to_blis_uplo( char uplo, uplo_t* blis_uplo )
{
	if      ( uplo == 'l' || uplo == 'L' ) *blis_uplo = BLIS_LOWER;
	else if ( uplo == 'u' || uplo == 'U' ) *blis_uplo = BLIS_UPPER;
	else
	{
		// See comment for bli_param_map_netlib_to_blis_side() above.
		//bli_check_error_code( BLIS_INVALID_UPLO );
		*blis_uplo = BLIS_LOWER;
	}
}

BLIS_INLINE void bli_param_map_netlib_to_blis_trans( char trans, trans_t* blis_trans )
{
	if      ( trans == 'n' || trans == 'N' ) *blis_trans = BLIS_NO_TRANSPOSE;
	else if ( trans == 't' || trans == 'T' ) *blis_trans = BLIS_TRANSPOSE;
	else if ( trans == 'c' || trans == 'C' ) *blis_trans = BLIS_CONJ_TRANSPOSE;
	else
	{
		// See comment for bli_param_map_netlib_to_blis_side() above.
		//bli_check_error_code( BLIS_INVALID_TRANS );
		*blis_trans = BLIS_NO_TRANSPOSE;
	}
}

BLIS_INLINE void bli_param_map_netlib_to_blis_diag( char diag, diag_t* blis_diag )
{
	if      ( diag == 'n' || diag == 'N' ) *blis_diag = BLIS_NONUNIT_DIAG;
	else if ( diag == 'u' || diag == 'U' ) *blis_diag = BLIS_UNIT_DIAG;
	else
	{
		// See comment for bli_param_map_netlib_to_blis_side() above.
		//bli_check_error_code( BLIS_INVALID_DIAG );
		*blis_diag = BLIS_NONUNIT_DIAG;
	}
}


// --- BLIS char to BLIS mappings ----------------------------------------------

BLIS_EXPORT_BLIS void bli_param_map_char_to_blis_side( char side, side_t* blis_side );
BLIS_EXPORT_BLIS void bli_param_map_char_to_blis_uplo( char uplo, uplo_t* blis_uplo );
BLIS_EXPORT_BLIS void bli_param_map_char_to_blis_trans( char trans, trans_t* blis_trans );
BLIS_EXPORT_BLIS void bli_param_map_char_to_blis_conj( char conj, conj_t* blis_conj );
BLIS_EXPORT_BLIS void bli_param_map_char_to_blis_diag( char diag, diag_t* blis_diag );
BLIS_EXPORT_BLIS void bli_param_map_char_to_blis_dt( char dt, num_t* blis_dt );


// --- BLIS to BLIS char mappings ----------------------------------------------

BLIS_EXPORT_BLIS void bli_param_map_blis_to_char_side( side_t blis_side, char* side );
BLIS_EXPORT_BLIS void bli_param_map_blis_to_char_uplo( uplo_t blis_uplo, char* uplo );
BLIS_EXPORT_BLIS void bli_param_map_blis_to_char_trans( trans_t blis_trans, char* trans );
BLIS_EXPORT_BLIS void bli_param_map_blis_to_char_conj( conj_t blis_conj, char* conj );
BLIS_EXPORT_BLIS void bli_param_map_blis_to_char_diag( diag_t blis_diag, char* diag );
BLIS_EXPORT_BLIS void bli_param_map_blis_to_char_dt( num_t blis_dt, char* dt );

// end bli_param_map.h
#line 147 "../blis-2.0/frame/include/blis.h"

// begin bli_clock.h
#line 1 "../blis-2.0/frame/base//bli_clock.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

BLIS_EXPORT_BLIS double bli_clock( void );
BLIS_EXPORT_BLIS double bli_clock_min_diff( double time_min, double time_start );

double bli_clock_helper( void );

// end bli_clock.h
#line 148 "../blis-2.0/frame/include/blis.h"

// begin bli_error.h
#line 1 "../blis-2.0/frame/base//bli_error.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2020, Advanced Micro Devices, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/


BLIS_EXPORT_BLIS errlev_t bli_error_checking_level( void );
BLIS_EXPORT_BLIS void     bli_error_checking_level_set( errlev_t new_level );

BLIS_EXPORT_BLIS bool     bli_error_checking_is_enabled( void );

void                      bli_print_msg( const char* str, const char* file, guint_t line );
BLIS_EXPORT_BLIS void     bli_abort( void );

const char*               bli_error_string_for_code( gint_t code );

// end bli_error.h
#line 149 "../blis-2.0/frame/include/blis.h"

// begin bli_f2c.h
#line 1 "../blis-2.0/frame/include//bli_f2c.h"

// f2c.h  --  Standard Fortran to C header file
//  barf  [ba:rf]  2.  "He suggested using FORTRAN, and everybody barfed."
//  - From The Shogakukan DICTIONARY OF NEW ENGLISH (Second edition)

#ifndef BLIS_F2C_H
#define BLIS_F2C_H

typedef f77_int bla_integer;
typedef f77_char bla_character;
//typedef char *address;
//typedef short int shortint;
typedef float bla_real;
typedef double bla_double;
typedef scomplex bla_scomplex;
typedef dcomplex bla_dcomplex;
typedef f77_int bla_logical;
//typedef short int shortlogical;
//typedef char logical1;
//typedef char integer1;
#ifdef INTEGER_STAR_8                // Adjust for integer*8.
typedef long long longint;           // system-dependent
typedef unsigned long long ulongint; // system-dependent
#define qbit_clear(a,b)	((a) & ~((ulongint)1 << (b)))
#define qbit_set(a,b)	((a) |  ((ulongint)1 << (b)))
#endif

#ifndef TRUE_
#define TRUE_ (1)
#endif

#ifndef FALSE_
#define FALSE_ (0)
#endif

// Extern is for use with -E
#ifndef Extern
#define Extern extern
#endif

// I/O stuff

#ifdef f2c_i2
// for -i2
//typedef short flag;
//typedef short ftnlen;
typedef bla_integer ftnlen;
//typedef short ftnint;
#else
//typedef long int flag;
//typedef long int ftnlen;
typedef bla_integer ftnlen;
//typedef long int ftnint;
#endif

#ifndef VOID
#define VOID void
#endif

#ifndef f2c_abs
  #define f2c_abs(x) ((x) >= 0 ? (x) : -(x))
#endif
#ifndef f2c_dabs
  #define f2c_dabs(x) (doublereal)f2c_abs(x)
#endif
#ifndef f2c_min
  #define f2c_min(a,b) ((a) <= (b) ? (a) : (b))
#endif
#ifndef f2c_max
  #define f2c_max(a,b) ((a) >= (b) ? (a) : (b))
#endif
#ifndef f2c_dmin
  #define f2c_dmin(a,b) (doublereal)f2c_min(a,b)
#endif
#ifndef f2c_dmax
  #define f2c_dmax(a,b) (doublereal)f2c_max(a,b)
#endif

#ifndef bit_test
  #define bit_test(a,b)  ((a) >> (b) & 1)
#endif

#ifndef bit_clear
  #define bit_clear(a,b) ((a) & ~((uinteger)1 << (b)))
#endif

#ifndef bit_set
  #define bit_set(a,b)   ((a) |  ((uinteger)1 << (b)))
#endif

// undef any lower-case symbols that your C compiler predefines, e.g.:

#ifndef Skip_f2c_Undefs
#undef cray
#undef gcos
#undef mc68010
#undef mc68020
#undef mips
#undef pdp11
#undef sgi
#undef sparc
#undef sun
#undef sun2
#undef sun3
#undef sun4
#undef u370
#undef u3b
#undef u3b2
#undef u3b5
#undef unix
#undef vax
#endif

#endif
// end bli_f2c.h
#line 150 "../blis-2.0/frame/include/blis.h"

// begin bli_machval.h
#line 1 "../blis-2.0/frame/base//bli_machval.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

// begin bli_lsame.h
#line 1 "../blis-2.0/frame/base/noopt//bli_lsame.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

bla_logical bli_lsame( bla_character* ca, bla_character* cb, ftnlen ca_len, ftnlen cb_len );
// end bli_lsame.h
#line 36 "../blis-2.0/frame/base//bli_machval.h"

// begin bli_slamch.h
#line 1 "../blis-2.0/frame/base/noopt//bli_slamch.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

bla_real bli_slamch( bla_character* cmach, ftnlen cmach_len );
// end bli_slamch.h
#line 37 "../blis-2.0/frame/base//bli_machval.h"

// begin bli_dlamch.h
#line 1 "../blis-2.0/frame/base/noopt//bli_dlamch.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

bla_double bli_dlamch( bla_character* cmach, ftnlen cmach_len );
// end bli_dlamch.h
#line 38 "../blis-2.0/frame/base//bli_machval.h"


//
// Prototype object-based interface.
//
BLIS_EXPORT_BLIS void bli_machval( machval_t mval, obj_t* v );


//
// Prototype BLAS-like interfaces.
//
#undef  GENTPROTR
#define GENTPROTR( ctype_v, ctype_vr, chv, chvr, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(chv,opname) \
     ( \
       machval_t mval, \
       void*     v     \
     );

INSERT_GENTPROTR_BASIC( machval )

// end bli_machval.h
#line 151 "../blis-2.0/frame/include/blis.h"

// begin bli_getopt.h
#line 1 "../blis-2.0/frame/base//bli_getopt.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

typedef struct getopt_s
{
	const char* nextchar;
	const char* optarg;
	      int   optind;
	      int   opterr;
	      int   optopt;
} getopt_t;

BLIS_EXPORT_BLIS void bli_getopt_init_state( int opterr, getopt_t* state );

BLIS_EXPORT_BLIS int bli_getopt( int argc, const char* const * argv, const char* optstring, getopt_t* state );

// end bli_getopt.h
#line 152 "../blis-2.0/frame/include/blis.h"

// begin bli_opid.h
#line 1 "../blis-2.0/frame/base//bli_opid.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

BLIS_INLINE bool bli_opid_is_level3( opid_t opid )
{
	return ( bool )
	       ( BLIS_GEMM <= opid && opid <= BLIS_TRSM );
}

// end bli_opid.h
#line 153 "../blis-2.0/frame/include/blis.h"

// begin bli_cntl.h
#line 1 "../blis-2.0/frame/base//bli_cntl.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2018 - 2019, Advanced Micro Devices, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/


/*
// -- Control tree node definition --

struct cntl_s
{
	l3_var_oft var_func;
	struct
	{
		dim_t          ways;
		struct cntl_s* sub_node;
	} sub_nodes[ BLIS_MAX_SUB_NODES ];
};
typedef struct cntl_s cntl_t;
*/


// -- Control tree prototypes --

BLIS_EXPORT_BLIS void bli_cntl_init_node
     (
       void_fp var_func,
       cntl_t* cntl
     );

BLIS_EXPORT_BLIS void bli_cntl_attach_sub_node
     (
       dim_t   ways,
       cntl_t* sub_node,
       cntl_t* cntl
     );

BLIS_EXPORT_BLIS void bli_cntl_clear_node
     (
       cntl_t* cntl
     );

// -----------------------------------------------------------------------------

// cntl_t query (fields only)

BLIS_INLINE void_fp bli_cntl_var_func( const cntl_t* cntl )
{
	return cntl->var_func;
}

BLIS_INLINE dim_t bli_cntl_ways( dim_t which, const cntl_t* cntl )
{
	return cntl->sub_nodes[ which ].ways;
}

BLIS_INLINE cntl_t* bli_cntl_sub_node( dim_t which, const cntl_t* cntl )
{
	return cntl->sub_nodes[ which ].sub_node;
}

// cntl_t query (complex)

BLIS_INLINE bool bli_cntl_is_leaf( const cntl_t* cntl )
{
	return ( bool )
	       ( bli_cntl_sub_node( 0, cntl ) == NULL );
}

// cntl_t modification

BLIS_INLINE void bli_cntl_set_var_func( void_fp var_func, cntl_t* cntl )
{
	cntl->var_func = var_func;
}

BLIS_INLINE void bli_cntl_set_ways( dim_t which, dim_t ways, cntl_t* cntl )
{
	cntl->sub_nodes[ which ].ways = ways;
}

BLIS_INLINE void bli_cntl_set_sub_node( dim_t which, cntl_t* sub_node, cntl_t* cntl )
{
	cntl->sub_nodes[ which ].sub_node = sub_node;
}

// end bli_cntl.h
#line 154 "../blis-2.0/frame/include/blis.h"

// begin bli_env.h
#line 1 "../blis-2.0/frame/base//bli_env.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2016, Hewlett Packard Enterprise Development LP
   Copyright (C) 2018, Advanced Micro Devices, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_ENV_H
#define BLIS_ENV_H

gint_t bli_env_get_var( const char* env, gint_t fallback );
char*  bli_env_get_str( const char* env );
//void  bli_env_set_var( const char* env, dim_t value );

#endif

// end bli_env.h
#line 155 "../blis-2.0/frame/include/blis.h"

// begin bli_pack.h
#line 1 "../blis-2.0/frame/base//bli_pack.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_PACK_H
#define BLIS_PACK_H

BLIS_EXPORT_BLIS void bli_pack_get_pack_a( bool* pack_a );
BLIS_EXPORT_BLIS void bli_pack_get_pack_b( bool* pack_b );
BLIS_EXPORT_BLIS void bli_pack_set_pack_a( bool pack_a );
BLIS_EXPORT_BLIS void bli_pack_set_pack_b( bool pack_b );

#endif

// end bli_pack.h
#line 156 "../blis-2.0/frame/include/blis.h"

// begin bli_info.h
#line 1 "../blis-2.0/frame/base//bli_info.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2018 - 2019, Advanced Micro Devices, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/


// -- General library information ----------------------------------------------

BLIS_EXPORT_BLIS const char* bli_info_get_version_str( void );
BLIS_EXPORT_BLIS const char* bli_info_get_int_type_size_str( void );


// -- General configuration-related --------------------------------------------

BLIS_EXPORT_BLIS gint_t bli_info_get_int_type_size( void );
BLIS_EXPORT_BLIS gint_t bli_info_get_num_fp_types( void );
BLIS_EXPORT_BLIS gint_t bli_info_get_max_type_size( void );
BLIS_EXPORT_BLIS gint_t bli_info_get_page_size( void );
BLIS_EXPORT_BLIS gint_t bli_info_get_simd_num_registers( void );
BLIS_EXPORT_BLIS gint_t bli_info_get_simd_size( void );
BLIS_EXPORT_BLIS gint_t bli_info_get_simd_align_size( void );
BLIS_EXPORT_BLIS gint_t bli_info_get_stack_buf_max_size( void );
BLIS_EXPORT_BLIS gint_t bli_info_get_stack_buf_align_size( void );
BLIS_EXPORT_BLIS gint_t bli_info_get_heap_addr_align_size( void );
BLIS_EXPORT_BLIS gint_t bli_info_get_heap_stride_align_size( void );
BLIS_EXPORT_BLIS gint_t bli_info_get_pool_addr_align_size_a( void );
BLIS_EXPORT_BLIS gint_t bli_info_get_pool_addr_align_size_b( void );
BLIS_EXPORT_BLIS gint_t bli_info_get_pool_addr_align_size_c( void );
BLIS_EXPORT_BLIS gint_t bli_info_get_pool_addr_align_size_gen( void );
BLIS_EXPORT_BLIS gint_t bli_info_get_pool_addr_offset_size_a( void );
BLIS_EXPORT_BLIS gint_t bli_info_get_pool_addr_offset_size_b( void );
BLIS_EXPORT_BLIS gint_t bli_info_get_pool_addr_offset_size_c( void );
BLIS_EXPORT_BLIS gint_t bli_info_get_pool_addr_offset_size_gen( void );
BLIS_EXPORT_BLIS gint_t bli_info_get_enable_stay_auto_init( void );
BLIS_EXPORT_BLIS gint_t bli_info_get_enable_blas( void );
BLIS_EXPORT_BLIS gint_t bli_info_get_enable_cblas( void );
BLIS_EXPORT_BLIS gint_t bli_info_get_blas_int_type_size( void );
BLIS_EXPORT_BLIS gint_t bli_info_get_enable_pba_pools( void );
BLIS_EXPORT_BLIS gint_t bli_info_get_enable_sba_pools( void );
BLIS_EXPORT_BLIS gint_t bli_info_get_enable_threading( void );
BLIS_EXPORT_BLIS gint_t bli_info_get_enable_openmp( void );
BLIS_EXPORT_BLIS gint_t bli_info_get_enable_pthreads( void );
BLIS_EXPORT_BLIS gint_t bli_info_get_enable_hpx( void );
BLIS_EXPORT_BLIS gint_t bli_info_get_enable_openmp_as_default( void );
BLIS_EXPORT_BLIS gint_t bli_info_get_enable_pthreads_as_default( void );
BLIS_EXPORT_BLIS gint_t bli_info_get_enable_hpx_as_default( void );
BLIS_EXPORT_BLIS gint_t bli_info_get_thread_jrir_slab( void );
BLIS_EXPORT_BLIS gint_t bli_info_get_thread_jrir_rr( void );
BLIS_EXPORT_BLIS gint_t bli_info_get_thread_jrir_tlb( void );
BLIS_EXPORT_BLIS gint_t bli_info_get_enable_tls( void );
BLIS_EXPORT_BLIS gint_t bli_info_get_enable_memkind( void );
BLIS_EXPORT_BLIS gint_t bli_info_get_enable_sandbox( void );


// -- Kernel implementation-related --------------------------------------------


// -- Level-3 kernel definitions --

BLIS_EXPORT_BLIS const char* bli_info_get_gemm_ukr_impl_string( ind_t method, num_t dt );
BLIS_EXPORT_BLIS const char* bli_info_get_gemmtrsm_l_ukr_impl_string( ind_t method, num_t dt );
BLIS_EXPORT_BLIS const char* bli_info_get_gemmtrsm_u_ukr_impl_string( ind_t method, num_t dt );
BLIS_EXPORT_BLIS const char* bli_info_get_trsm_l_ukr_impl_string( ind_t method, num_t dt );
BLIS_EXPORT_BLIS const char* bli_info_get_trsm_u_ukr_impl_string( ind_t method, num_t dt );


// -- BLIS implementation query (level-3) --------------------------------------

BLIS_EXPORT_BLIS const char* bli_info_get_gemm_impl_string( num_t dt );
BLIS_EXPORT_BLIS const char* bli_info_get_gemmt_impl_string( num_t dt );
BLIS_EXPORT_BLIS const char* bli_info_get_hemm_impl_string( num_t dt );
BLIS_EXPORT_BLIS const char* bli_info_get_herk_impl_string( num_t dt );
BLIS_EXPORT_BLIS const char* bli_info_get_her2k_impl_string( num_t dt );
BLIS_EXPORT_BLIS const char* bli_info_get_symm_impl_string( num_t dt );
BLIS_EXPORT_BLIS const char* bli_info_get_syrk_impl_string( num_t dt );
BLIS_EXPORT_BLIS const char* bli_info_get_syr2k_impl_string( num_t dt );
BLIS_EXPORT_BLIS const char* bli_info_get_trmm_impl_string( num_t dt );
BLIS_EXPORT_BLIS const char* bli_info_get_trmm3_impl_string( num_t dt );
BLIS_EXPORT_BLIS const char* bli_info_get_trsm_impl_string( num_t dt );

// end bli_info.h
#line 157 "../blis-2.0/frame/include/blis.h"

// begin bli_arch.h
#line 1 "../blis-2.0/frame/base//bli_arch.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_ARCH_H
#define BLIS_ARCH_H

BLIS_EXPORT_BLIS arch_t bli_arch_query_id( void );

void   bli_arch_set_id_once( void );
void   bli_arch_set_id( void );
arch_t bli_arch_query_id_impl( void );

BLIS_EXPORT_BLIS const char*  bli_arch_string( arch_t id );

void bli_arch_set_logging( bool dolog );
bool bli_arch_get_logging( void );
void bli_arch_log( const char*, ... );

#endif

// end bli_arch.h
#line 158 "../blis-2.0/frame/include/blis.h"

// begin bli_cpuid.h
#line 1 "../blis-2.0/frame/base//bli_cpuid.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2018-2020, Advanced Micro Devices, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#if 0
  // Used only during standalone testing of ARM support.
  #define FALSE 0
  #define TRUE  1
  typedef enum
  {
	BLIS_ARCH_CORTEXA57 = 10,
	BLIS_ARCH_CORTEXA15 = 11,
	BLIS_ARCH_CORTEXA9  = 12,
	BLIS_ARCH_GENERIC   = 13
  } arch_t;
  typedef uint64_t bool;
  #define bli_abort abort
#endif

#ifndef BLIS_CPUID_H
#define BLIS_CPUID_H

arch_t bli_cpuid_query_id( void );

// Intel
bool bli_cpuid_is_skx( uint32_t family, uint32_t model, uint32_t features );
bool bli_cpuid_is_knl( uint32_t family, uint32_t model, uint32_t features );
bool bli_cpuid_is_haswell( uint32_t family, uint32_t model, uint32_t features );
bool bli_cpuid_is_sandybridge( uint32_t family, uint32_t model, uint32_t features );
bool bli_cpuid_is_penryn( uint32_t family, uint32_t model, uint32_t features );

// AMD
bool bli_cpuid_is_zen3( uint32_t family, uint32_t model, uint32_t features );
bool bli_cpuid_is_zen2( uint32_t family, uint32_t model, uint32_t features );
bool bli_cpuid_is_zen( uint32_t family, uint32_t model, uint32_t features );
bool bli_cpuid_is_excavator( uint32_t family, uint32_t model, uint32_t features );
bool bli_cpuid_is_steamroller( uint32_t family, uint32_t model, uint32_t features );
bool bli_cpuid_is_piledriver( uint32_t family, uint32_t model, uint32_t features );
bool bli_cpuid_is_bulldozer( uint32_t family, uint32_t model, uint32_t features );

// ARM
bool bli_cpuid_is_thunderx2( uint32_t model, uint32_t part, uint32_t features );
bool bli_cpuid_is_cortexa57( uint32_t model, uint32_t part, uint32_t features );
bool bli_cpuid_is_cortexa53( uint32_t model, uint32_t part, uint32_t features );
bool bli_cpuid_is_armsve( uint32_t model, uint32_t part, uint32_t features );
bool bli_cpuid_is_a64fx( uint32_t model, uint32_t part, uint32_t features );
bool bli_cpuid_is_cortexa15( uint32_t model, uint32_t part, uint32_t features );
bool bli_cpuid_is_cortexa9( uint32_t model, uint32_t part, uint32_t features );

uint32_t bli_cpuid_query( uint32_t* family, uint32_t* model, uint32_t* features );

// -----------------------------------------------------------------------------

//
// This section of the file was based off of cpuid.hpp from TBLIS [1].
//
// [1] https://github.com/devinamatthews/tblis
//

/*

   Copyright (C) 2017, The University of Texas at Austin
   Copyright (C) 2017, Devin Matthews

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

BLIS_INLINE bool bli_cpuid_has_features( uint32_t have, uint32_t want )
{
    return ( have & want ) == want;
}

// -----------------------------------------------------------------------------

#if defined(__x86_64__) || defined(_M_X64) || defined(__i386) || defined(_M_IX86)

// cpuid.h is now #included in bli_cpuid.c instead of here. See issue #393
// for more information why this move was made.
//#include "cpuid.h"

void get_cpu_name( char *cpu_name );
int  vpu_count( void );


enum
{
	VENDOR_INTEL = 0,
	VENDOR_AMD,
	VENDOR_UNKNOWN
};
enum
{
	FEATURE_SSE3     = 0x0001,
	FEATURE_SSSE3    = 0x0002,
	FEATURE_SSE41    = 0x0004,
	FEATURE_SSE42    = 0x0008,
	FEATURE_AVX      = 0x0010,
	FEATURE_AVX2     = 0x0020,
	FEATURE_FMA3     = 0x0040,
	FEATURE_FMA4     = 0x0080,
	FEATURE_AVX512F  = 0x0100,
	FEATURE_AVX512DQ = 0x0200,
	FEATURE_AVX512PF = 0x0400,
	FEATURE_AVX512ER = 0x0800,
	FEATURE_AVX512CD = 0x1000,
	FEATURE_AVX512BW = 0x2000,
	FEATURE_AVX512VL = 0x4000
};

#elif defined(__aarch64__) || defined(__arm__) || defined(_M_ARM) || defined(_ARCH_PPC)

char* find_string_in( char* target, char* buffer, size_t buf_len, char* filepath );

enum
{
	VENDOR_ARM = 0,
	VENDOR_IBM,
	VENDOR_UNKNOWN
};
enum
{
	MODEL_ARMV7 = 0,
	MODEL_ARMV8,
	MODEL_POWER7,
	MODEL_POWER9,
	MODEL_POWER10,
	MODEL_UNKNOWN
};
enum
{
	FEATURE_NEON = 0x01,
	FEATURE_SVE  = 0x02
};

#endif



#endif

// end bli_cpuid.h
#line 159 "../blis-2.0/frame/include/blis.h"

// begin bli_string.h
#line 1 "../blis-2.0/frame/base//bli_string.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

void bli_string_mkupper( char* s );
// end bli_string.h
#line 160 "../blis-2.0/frame/include/blis.h"

// begin bli_setgetijm.h
#line 1 "../blis-2.0/frame/base//bli_setgetijm.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

BLIS_EXPORT_BLIS err_t bli_setijm
     (
             double ar,
             double ai,
             dim_t  i,
             dim_t  j,
       const obj_t* b
     );

#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname) \
     ( \
       double ar, \
       double ai, \
       dim_t  i, \
       dim_t  j, \
       void*  b, inc_t rs, inc_t cs  \
     );

INSERT_GENTPROT_BASIC( setijm )

// -----------------------------------------------------------------------------

BLIS_EXPORT_BLIS err_t bli_getijm
      (
              dim_t   i,
              dim_t   j,
        const obj_t*  b,
              double* ar,
              double* ai
      );

#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname) \
     ( \
             dim_t   i, \
             dim_t   j, \
       const void*   b, inc_t rs, inc_t cs, \
             double* ar, \
             double* ai  \
     );

INSERT_GENTPROT_BASIC( getijm )

// end bli_setgetijm.h
#line 161 "../blis-2.0/frame/include/blis.h"

// begin bli_setgetijv.h
#line 1 "../blis-2.0/frame/base//bli_setgetijv.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

BLIS_EXPORT_BLIS err_t bli_setijv
     (
             double  ar,
             double  ai,
             dim_t   i,
       const obj_t*  x
     );

#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname) \
     ( \
       double ar, \
       double ai, \
       dim_t  i, \
       void*  x, inc_t incx  \
     );

INSERT_GENTPROT_BASIC( setijv )

// -----------------------------------------------------------------------------

BLIS_EXPORT_BLIS err_t bli_getijv
      (
              dim_t   i,
        const obj_t*  x,
              double* ar,
              double* ai
      );

#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname) \
     ( \
             dim_t   i, \
       const void*   b, inc_t incx, \
             double* ar, \
             double* ai  \
     );

INSERT_GENTPROT_BASIC( getijv )

// end bli_setgetijv.h
#line 162 "../blis-2.0/frame/include/blis.h"

// begin bli_setri.h
#line 1 "../blis-2.0/frame/base//bli_setri.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

// -- setr ---------------------------------------------------------------------

BLIS_EXPORT_BLIS void bli_setrm
     (
       const obj_t* alpha,
       const obj_t* b
     );

BLIS_EXPORT_BLIS void bli_setrv
     (
       const obj_t* alpha,
       const obj_t* x
     );

// -- seti ---------------------------------------------------------------------

BLIS_EXPORT_BLIS void bli_setim
     (
       const obj_t* alpha,
       const obj_t* b
     );

BLIS_EXPORT_BLIS void bli_setiv
     (
       const obj_t* alpha,
       const obj_t* x
     );

// end bli_setri.h
#line 163 "../blis-2.0/frame/include/blis.h"


// begin bli_castm.h
#line 1 "../blis-2.0/frame/base/cast//bli_castm.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

//
// Prototype object-based interface.
//

BLIS_EXPORT_BLIS void bli_castm
     (
       const obj_t* a,
       const obj_t* b
     );

//
// Prototype BLAS-like interfaces with heterogeneous-typed operands.
//

#undef  GENTPROT2
#define GENTPROT2( ctype_a, ctype_b, cha, chb, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(cha,chb,opname) \
     ( \
             trans_t transa, \
             dim_t   m, \
             dim_t   n, \
       const void*   a, inc_t rs_a, inc_t cs_a, \
             void*   b, inc_t rs_b, inc_t cs_b  \
     );

INSERT_GENTPROT2_BASIC( castm )
INSERT_GENTPROT2_MIX_DP( castm )

//
// Prototype object-based _check() function.
//

void bli_castm_check
     (
       const obj_t* a,
       const obj_t* b
     );

// end bli_castm.h
#line 165 "../blis-2.0/frame/include/blis.h"

// begin bli_castnzm.h
#line 1 "../blis-2.0/frame/base/cast//bli_castnzm.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

//
// Prototype object-based interface.
//

BLIS_EXPORT_BLIS void bli_castnzm
     (
       const obj_t* a,
       const obj_t* b
     );

//
// Prototype BLAS-like interfaces with heterogeneous-typed operands.
//

#undef  GENTPROT2
#define GENTPROT2( ctype_a, ctype_b, cha, chb, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(cha,chb,opname) \
     ( \
             trans_t transa, \
             dim_t   m, \
             dim_t   n, \
       const void*   a, inc_t rs_a, inc_t cs_a, \
             void*   b, inc_t rs_b, inc_t cs_b  \
     );

INSERT_GENTPROT2_BASIC( castnzm )
INSERT_GENTPROT2_MIX_DP( castnzm )

//
// Prototype object-based _check() function.
//

void bli_castnzm_check
     (
       const obj_t* a,
       const obj_t* b
     );

// end bli_castnzm.h
#line 166 "../blis-2.0/frame/include/blis.h"

// begin bli_castv.h
#line 1 "../blis-2.0/frame/base/cast//bli_castv.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

//
// Prototype object-based interface.
//

BLIS_EXPORT_BLIS void bli_castv
     (
       const obj_t* x,
       const obj_t* y
     );

//
// Prototype BLAS-like interfaces with heterogeneous-typed operands.
//

#undef  GENTPROT2
#define GENTPROT2( ctype_x, ctype_y, chx, chy, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(chx,chy,opname) \
     ( \
             conj_t conjx, \
             dim_t  n, \
       const void*  x, inc_t incx, \
             void*  y, inc_t incy  \
     );

INSERT_GENTPROT2_BASIC( castv )
INSERT_GENTPROT2_MIX_DP( castv )

//
// Prototype object-based _check() function.
//

void bli_castv_check
     (
       const obj_t* x,
       const obj_t* y
     );

// end bli_castv.h
#line 167 "../blis-2.0/frame/include/blis.h"

// begin bli_projm.h
#line 1 "../blis-2.0/frame/base/proj//bli_projm.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

BLIS_EXPORT_BLIS void bli_projm
     (
       const obj_t* a,
       const obj_t* b
     );

void bli_projm_check
     (
       const obj_t* a,
       const obj_t* b
     );

// end bli_projm.h
#line 168 "../blis-2.0/frame/include/blis.h"

// begin bli_projv.h
#line 1 "../blis-2.0/frame/base/proj//bli_projv.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

BLIS_EXPORT_BLIS void bli_projv
     (
       const obj_t* x,
       const obj_t* y
     );

void bli_projv_check
     (
       const obj_t* x,
       const obj_t* y
     );

// end bli_projv.h
#line 169 "../blis-2.0/frame/include/blis.h"



// -- Level-0 operations --

// begin bli_l0.h
#line 1 "../blis-2.0/frame/0//bli_l0.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

// begin bli_l0_check.h
#line 1 "../blis-2.0/frame/0//bli_l0_check.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/


//
// Prototype object-based check functions.
//

#undef  GENTPROT
#define GENTPROT( opname ) \
\
void PASTEMAC(opname,_check) \
     ( \
       const obj_t* chi, \
       const obj_t* psi  \
     );

GENTPROT( addsc )
GENTPROT( copysc )
GENTPROT( divsc )
GENTPROT( mulsc )
GENTPROT( sqrtsc )
GENTPROT( sqrtrsc )
GENTPROT( subsc )
GENTPROT( invertsc )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
void PASTEMAC(opname,_check) \
     ( \
       const obj_t* chi, \
       const obj_t* absq  \
     );

GENTPROT( absqsc )
GENTPROT( normfsc )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
void PASTEMAC(opname,_check) \
     ( \
       const obj_t*  chi, \
       const double* zeta_r, \
       const double* zeta_i  \
     );

GENTPROT( getsc )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
void PASTEMAC(opname,_check) \
     ( \
             double zeta_r, \
             double zeta_i, \
       const obj_t* chi  \
     );

GENTPROT( setsc )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
void PASTEMAC(opname,_check) \
     ( \
       const obj_t* chi, \
       const obj_t* zeta_r, \
       const obj_t* zeta_i  \
     );

GENTPROT( unzipsc )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
void PASTEMAC(opname,_check) \
     ( \
       const obj_t* zeta_r, \
       const obj_t* zeta_i, \
       const obj_t* chi  \
     );

GENTPROT( zipsc )

// -----------------------------------------------------------------------------

void bli_l0_xsc_check
     (
       const obj_t* chi
     );

void bli_l0_xxsc_check
     (
       const obj_t* chi,
       const obj_t* psi
     );

void bli_l0_xx2sc_check
     (
       const obj_t* chi,
       const obj_t* norm
     );

void bli_l0_xxbsc_check
     (
       const obj_t* chi,
       const obj_t* psi,
       const bool*  is
     );
// end bli_l0_check.h
#line 36 "../blis-2.0/frame/0//bli_l0.h"


// begin bli_l0_oapi.h
#line 1 "../blis-2.0/frame/0//bli_l0_oapi.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/


//
// Prototype object-based interfaces.
//

#undef  GENPROT
#define GENPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname) \
     ( \
       const obj_t* chi, \
       const obj_t* absq  \
     );

GENPROT( absqsc )
GENPROT( normfsc )


#undef  GENPROT
#define GENPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname) \
     ( \
       const obj_t* chi, \
       const obj_t* psi  \
     );

GENPROT( addsc )
GENPROT( divsc )
GENPROT( mulsc )
GENPROT( sqrtsc )
GENPROT( sqrtrsc )
GENPROT( subsc )
GENPROT( invertsc )


#undef  GENPROT
#define GENPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname) \
     ( \
       const obj_t*  chi, \
             double* zeta_r, \
             double* zeta_i  \
     );

GENPROT( getsc )


#undef  GENPROT
#define GENPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname) \
     ( \
             double zeta_r, \
             double zeta_i, \
       const obj_t* chi  \
     );

GENPROT( setsc )


#undef  GENPROT
#define GENPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname) \
     ( \
       const obj_t* chi, \
       const obj_t* zeta_r, \
       const obj_t* zeta_i  \
     );

GENPROT( unzipsc )


#undef  GENPROT
#define GENPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname) \
     ( \
       const obj_t* zeta_r, \
       const obj_t* zeta_i, \
       const obj_t* chi  \
     );

GENPROT( zipsc )

// end bli_l0_oapi.h
#line 38 "../blis-2.0/frame/0//bli_l0.h"

// begin bli_l0_tapi.h
#line 1 "../blis-2.0/frame/0//bli_l0_tapi.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/


//
// Prototype BLAS-like interfaces with typed operands.
//

#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname) \
     ( \
             conj_t conjchi, \
       const ctype* chi, \
             ctype* psi  \
     );

INSERT_GENTPROT_BASIC( addsc )
INSERT_GENTPROT_BASIC( divsc )
INSERT_GENTPROT_BASIC( mulsc )
INSERT_GENTPROT_BASIC( subsc )
INSERT_GENTPROT_BASIC( invertsc )


#undef  GENTPROTR
#define GENTPROTR( ctype, ctype_r, ch, chr, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname) \
     ( \
       const ctype*   chi, \
             ctype_r* absq  \
     );

INSERT_GENTPROTR_BASIC( absqsc )
INSERT_GENTPROTR_BASIC( normfsc )


#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname) \
     ( \
       const ctype* chi, \
             ctype* psi  \
     );

INSERT_GENTPROT_BASIC( sqrtsc )
INSERT_GENTPROT_BASIC( sqrtrsc )


#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname) \
     ( \
       const ctype*  chi, \
             double* zeta_r, \
             double* zeta_i  \
     );

INSERT_GENTPROT_BASIC( getsc )


#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname) \
     ( \
       double zeta_r, \
       double zeta_i, \
       ctype* chi  \
     );

INSERT_GENTPROT_BASIC( setsc )


#undef  GENTPROTR
#define GENTPROTR( ctype, ctype_r, ch, chr, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname) \
     ( \
       const ctype*   chi, \
             ctype_r* zeta_r, \
             ctype_r* zeta_i  \
     );

INSERT_GENTPROTR_BASIC( unzipsc )


#undef  GENTPROTR
#define GENTPROTR( ctype, ctype_r, ch, chr, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname) \
     ( \
       const ctype_r* zeta_r, \
       const ctype_r* zeta_i, \
             ctype*   chi  \
     );

INSERT_GENTPROTR_BASIC( zipsc )

// -----------------------------------------------------------------------------

BLIS_EXPORT_BLIS void bli_igetsc
     (
       const dim_t*  chi,
             double* zeta_r,
             double* zeta_i
     );

BLIS_EXPORT_BLIS void bli_isetsc
     (
       double zeta_r,
       double zeta_i,
       dim_t* chi
     );

// end bli_l0_tapi.h
#line 39 "../blis-2.0/frame/0//bli_l0.h"

// begin bli_l0_ft.h
#line 1 "../blis-2.0/frame/0//bli_l0_ft.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/


//
// -- Level-0 function types ---------------------------------------------------
//

// addsc, divsc, subsc, invertsc

#undef  GENTDEF
#define GENTDEF( ctype, ch, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,tsuf)) \
     ( \
             conj_t conjchi, \
       const ctype* chi, \
             ctype* psi  \
     );

INSERT_GENTDEF( addsc )
INSERT_GENTDEF( divsc )
INSERT_GENTDEF( subsc )
INSERT_GENTDEF( invertsc )

// mulsc

#undef  GENTDEF
#define GENTDEF( ctype, ch, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,tsuf)) \
     ( \
             conj_t conjchi, \
       const ctype* chi, \
             ctype* psi  \
     );

INSERT_GENTDEF( mulsc )

// absqsc

#undef  GENTDEFR
#define GENTDEFR( ctype, ctype_r, ch, chr, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,tsuf)) \
     ( \
       const ctype*   chi, \
             ctype_r* absq  \
     );

INSERT_GENTDEFR( absqsc )

// normfsc

#undef  GENTDEFR
#define GENTDEFR( ctype, ctype_r, ch, chr, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,tsuf)) \
     ( \
       const ctype*   chi, \
             ctype_r* norm  \
     );

INSERT_GENTDEFR( normfsc )

// sqrtsc

#undef  GENTDEF
#define GENTDEF( ctype, ch, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,tsuf)) \
     ( \
       const ctype* chi, \
             ctype* psi  \
     );

INSERT_GENTDEF( sqrtsc )

// sqrtrsc

#undef  GENTDEF
#define GENTDEF( ctype, ch, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,tsuf)) \
     ( \
       const ctype* chi, \
             ctype* psi  \
     );

INSERT_GENTDEF( sqrtrsc )

// getsc

#undef  GENTDEF
#define GENTDEF( ctype, ch, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,tsuf)) \
     ( \
       const ctype*  chi, \
             double* zeta_r, \
             double* zeta_i  \
     );

INSERT_GENTDEF( getsc )

// setsc

#undef  GENTDEF
#define GENTDEF( ctype, ch, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,tsuf)) \
     ( \
       double  zeta_r, \
       double  zeta_i, \
       ctype*  chi  \
     );

INSERT_GENTDEF( setsc )

// unzipsc

#undef  GENTDEFR
#define GENTDEFR( ctype, ctype_r, ch, chr, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,tsuf)) \
     ( \
       const ctype*   chi, \
             ctype_r* zeta_r, \
             ctype_r* zeta_i  \
     );

INSERT_GENTDEFR( unzipsc )

// zipsc

#undef  GENTDEFR
#define GENTDEFR( ctype, ctype_r, ch, chr, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,tsuf)) \
     ( \
       const ctype_r* zeta_r, \
       const ctype_r* zeta_i, \
             ctype*   chi  \
     );

INSERT_GENTDEFR( zipsc )

// end bli_l0_ft.h
#line 40 "../blis-2.0/frame/0//bli_l0.h"


// Generate function pointer arrays for tapi functions.
// begin bli_l0_fpa.h
#line 1 "../blis-2.0/frame/0//bli_l0_fpa.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

//
// Prototype function pointer query interface.
//

#undef  GENPROT
#define GENPROT( opname ) \
\
PASTECH(opname,_vft) \
PASTEMAC(opname,_qfp)( num_t dt );

GENPROT( absqsc )
GENPROT( normfsc )
GENPROT( addsc )
GENPROT( divsc )
GENPROT( mulsc )
GENPROT( subsc )
GENPROT( invertsc )
GENPROT( sqrtsc )
GENPROT( sqrtrsc )
GENPROT( unzipsc )
GENPROT( zipsc )

GENPROT( getsc )
GENPROT( setsc )

// end bli_l0_fpa.h
#line 43 "../blis-2.0/frame/0//bli_l0.h"


// copysc
// begin bli_copysc.h
#line 1 "../blis-2.0/frame/0/copysc//bli_copysc.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/


//
// Prototype object-based interfaces.
//

#undef  GENFRONT
#define GENFRONT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname) \
     ( \
       const obj_t* chi, \
       const obj_t* psi  \
     );
GENFRONT( copysc )


//
// Prototype BLAS-like interfaces with heterogeneous-typed operands.
//

#undef  GENTPROT2
#define GENTPROT2( ctype_x, ctype_y, chx, chy, varname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(chx,chy,varname) \
     ( \
             conj_t conjchi, \
       const void*  chi, \
             void*  psi \
     );

INSERT_GENTPROT2_BASIC( copysc )
INSERT_GENTPROT2_MIX_D( copysc )
INSERT_GENTPROT2_MIX_P( copysc )

// end bli_copysc.h
#line 46 "../blis-2.0/frame/0//bli_l0.h"

// end bli_l0.h
#line 174 "../blis-2.0/frame/include/blis.h"



// -- Level-1v operations --

// begin bli_l1v.h
#line 1 "../blis-2.0/frame/1//bli_l1v.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

// begin bli_l1v_check.h
#line 1 "../blis-2.0/frame/1//bli_l1v_check.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/


//
// Prototype object-based check functions.
//

#undef  GENTPROT
#define GENTPROT( opname ) \
\
void PASTEMAC(opname,_check) \
     ( \
       const obj_t* x, \
       const obj_t* y  \
     );

GENTPROT( addv )
GENTPROT( copyv )
GENTPROT( subv )
GENTPROT( swapv )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
void PASTEMAC(opname,_check) \
     ( \
       const obj_t* x, \
       const obj_t* index  \
     );

GENTPROT( amaxv )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
void PASTEMAC(opname,_check) \
     ( \
       const obj_t* alpha, \
       const obj_t* x, \
       const obj_t* beta, \
       const obj_t* y  \
     );

GENTPROT( axpbyv )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
void PASTEMAC(opname,_check) \
     ( \
       const obj_t* alpha, \
       const obj_t* x, \
       const obj_t* y  \
     );

GENTPROT( axpyv )
GENTPROT( scal2v )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
void PASTEMAC(opname,_check) \
     ( \
       const obj_t* x, \
       const obj_t* y, \
       const obj_t* rho  \
     );

GENTPROT( dotv )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
void PASTEMAC(opname,_check) \
     ( \
       const obj_t* alpha, \
       const obj_t* x, \
       const obj_t* y, \
       const obj_t* beta, \
       const obj_t* rho  \
     );

GENTPROT( dotxv )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
void PASTEMAC(opname,_check) \
     ( \
       const obj_t* x  \
     );

GENTPROT( invertv )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
void PASTEMAC(opname,_check) \
     ( \
       const obj_t* alpha, \
       const obj_t* x  \
     );

GENTPROT( invscalv )
GENTPROT( scalv )
GENTPROT( setv )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
void PASTEMAC(opname,_check) \
     ( \
       const obj_t* x, \
       const obj_t* beta, \
       const obj_t* y  \
     );

GENTPROT( xpbyv )



// -----------------------------------------------------------------------------

void bli_l1v_xy_check
     (
       const obj_t* x,
       const obj_t* y
     );

void bli_l1v_axy_check
     (
       const obj_t* alpha,
       const obj_t* x,
       const obj_t* y
     );

void bli_l1v_xby_check
     (
       const obj_t* x,
       const obj_t* beta,
       const obj_t* y
     );

void bli_l1v_axby_check
     (
       const obj_t* alpha,
       const obj_t* x,
       const obj_t* beta,
       const obj_t* y
     );

void bli_l1v_dot_check
     (
       const obj_t* alpha,
       const obj_t* x,
       const obj_t* y,
       const obj_t* beta,
       const obj_t* rho
     );

void bli_l1v_x_check
     (
       const obj_t* x
     );

void bli_l1v_ax_check
     (
       const obj_t* alpha,
       const obj_t* x
     );

void bli_l1v_xi_check
     (
       const obj_t* x,
       const obj_t* index
     );

// end bli_l1v_check.h
#line 36 "../blis-2.0/frame/1//bli_l1v.h"


// Define kernel function types.
// begin bli_l1v_ker_ft.h
#line 1 "../blis-2.0/frame/1//bli_l1v_ker_ft.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_L1V_KER_FT_H
#define BLIS_L1V_KER_FT_H


//
// -- Level-1v kernel function types -------------------------------------------
//

#undef  GENTDEF
#define GENTDEF( opname ) \
\
typedef void (*PASTECH(opname,_ker_ft)) \
     ( \
       PASTECH(opname,_params), \
       BLIS_CNTX_PARAM  \
     );

GENTDEF( addv )
GENTDEF( amaxv )
GENTDEF( axpbyv )
GENTDEF( axpyv )
GENTDEF( copyv )
GENTDEF( dotv )
GENTDEF( dotxv )
GENTDEF( invertv )
GENTDEF( invscalv )
GENTDEF( scalv )
GENTDEF( scal2v )
GENTDEF( setv )
GENTDEF( subv )
GENTDEF( swapv )
GENTDEF( xpbyv )

#endif

// end bli_l1v_ker_ft.h
#line 39 "../blis-2.0/frame/1//bli_l1v.h"


// Prototype object APIs (expert and non-expert).
// begin bli_oapi_ex.h
#line 1 "../blis-2.0/frame/include//bli_oapi_ex.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

// This file defines macros used to allow the _oapi.c files to produce
// object APIs that contain context parameters.

// Define a macro that allows the source code to determine which interface
// (basic or expert) we are compiling.
#undef  BLIS_OAPI_EXPERT
#define BLIS_OAPI_EXPERT

// Define the macro to add a suffix to the function names (in function
// definitions).
#undef  EX_SUF
#define EX_SUF BLIS_OAPI_EX_SUF

// Define the macro to add expert arguments to function signatures
// and prototypes.
#undef  BLIS_OAPI_EX_PARAMS
#define BLIS_OAPI_EX_PARAMS   , const cntx_t* cntx, const rntm_t* rntm

// Define the macro to omit the expert variable declaration block, since
// it is not needed when expert parameters are passed in through the API.
#undef  BLIS_OAPI_EX_DECLS
#define BLIS_OAPI_EX_DECLS

// end bli_oapi_ex.h
#line 42 "../blis-2.0/frame/1//bli_l1v.h"

// begin bli_l1v_oapi.h
#line 1 "../blis-2.0/frame/1//bli_l1v_oapi.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/


//
// Prototype object-based interfaces.
//

#undef  GENTPROT
#define GENTPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname,EX_SUF) \
     ( \
       const obj_t* x, \
       const obj_t* y  \
       BLIS_OAPI_EX_PARAMS  \
     );

GENTPROT( addv )
GENTPROT( copyv )
GENTPROT( subv )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname,EX_SUF) \
     ( \
       const obj_t* x, \
       const obj_t* index  \
       BLIS_OAPI_EX_PARAMS  \
     );

GENTPROT( amaxv )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname,EX_SUF) \
     ( \
       const obj_t* alpha, \
       const obj_t* x, \
       const obj_t* beta, \
       const obj_t* y  \
       BLIS_OAPI_EX_PARAMS  \
     );

GENTPROT( axpbyv )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname,EX_SUF) \
     ( \
       const obj_t* alpha, \
       const obj_t* x, \
       const obj_t* y  \
       BLIS_OAPI_EX_PARAMS  \
     );

GENTPROT( axpyv )
GENTPROT( scal2v )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname,EX_SUF) \
     ( \
       const obj_t* x, \
       const obj_t* y, \
       const obj_t* rho  \
       BLIS_OAPI_EX_PARAMS  \
     );

GENTPROT( dotv )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname,EX_SUF) \
     ( \
       const obj_t* alpha, \
       const obj_t* x, \
       const obj_t* y, \
       const obj_t* beta, \
       const obj_t* rho  \
       BLIS_OAPI_EX_PARAMS  \
     );

GENTPROT( dotxv )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname,EX_SUF) \
     ( \
       const obj_t* x  \
       BLIS_OAPI_EX_PARAMS  \
     );

GENTPROT( invertv )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname,EX_SUF) \
     ( \
       const obj_t* alpha, \
       const obj_t* x  \
       BLIS_OAPI_EX_PARAMS  \
     );

GENTPROT( invscalv )
GENTPROT( scalv )
GENTPROT( setv )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname,EX_SUF) \
     ( \
       const obj_t* x, \
       const obj_t* y  \
       BLIS_OAPI_EX_PARAMS  \
     );

GENTPROT( swapv )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname,EX_SUF) \
     ( \
       const obj_t* x, \
       const obj_t* beta, \
       const obj_t* y  \
       BLIS_OAPI_EX_PARAMS  \
     );

GENTPROT( xpbyv )

// end bli_l1v_oapi.h
#line 43 "../blis-2.0/frame/1//bli_l1v.h"

// begin bli_xapi_undef.h
#line 1 "../blis-2.0/frame/include//bli_xapi_undef.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

// This file un-defines macros used to allow the _oapi.c and _tapi.c files to
// produce object and typed APIs that omit or contain expert parameters.

// Un-define all macros that allow the source code to determine which interface
// (basic or expert) we are compiling.
#undef  BLIS_OAPI_BASIC
#undef  BLIS_OAPI_EXPERT
#undef  BLIS_TAPI_BASIC
#undef  BLIS_TAPI_EXPERT

// Un-define the macro to omit or add the function name suffix (in function
// definitions).
#undef  EX_SUF

// Un-define the macro to omit or add expert arguments from function signatures
// and prototypes.
#undef  BLIS_OAPI_EX_PARAMS
#undef  BLIS_TAPI_EX_PARAMS

// Un-define the macro to omit or add local expert variables.
#undef  BLIS_OAPI_EX_DECLS
#undef  BLIS_TAPI_EX_DECLS

// end bli_xapi_undef.h
#line 44 "../blis-2.0/frame/1//bli_l1v.h"


// begin bli_oapi_ba.h
#line 1 "../blis-2.0/frame/include//bli_oapi_ba.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

// This file defines macros used to allow the _oapi.c files to produce
// object APIs that omit expert parameters.

// Define a macro that allows the source code to determine which interface
// (basic or expert) we are compiling.
#undef  BLIS_OAPI_BASIC
#define BLIS_OAPI_BASIC

// Define the macro to omit a suffix from the function names (in function
// definitions).
#undef  EX_SUF
#define EX_SUF

// Define the macro to omit expert arguments from function signatures
// and prototypes.
#undef  BLIS_OAPI_EX_PARAMS
#define BLIS_OAPI_EX_PARAMS

// Define the macro to add local expert variables that are initialized
// to NULL. The "( void )" statements are to prevent unused variable
// warnings by the compiler.
#undef  BLIS_OAPI_EX_DECLS
#define BLIS_OAPI_EX_DECLS   const cntx_t* cntx = NULL; ( void )cntx; \
                                   rntm_t* rntm = NULL; ( void )rntm;

// end bli_oapi_ba.h
#line 46 "../blis-2.0/frame/1//bli_l1v.h"

// begin bli_l1v_oapi.h
#line 1 "../blis-2.0/frame/1//bli_l1v_oapi.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/


//
// Prototype object-based interfaces.
//

#undef  GENTPROT
#define GENTPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname,EX_SUF) \
     ( \
       const obj_t* x, \
       const obj_t* y  \
       BLIS_OAPI_EX_PARAMS  \
     );

GENTPROT( addv )
GENTPROT( copyv )
GENTPROT( subv )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname,EX_SUF) \
     ( \
       const obj_t* x, \
       const obj_t* index  \
       BLIS_OAPI_EX_PARAMS  \
     );

GENTPROT( amaxv )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname,EX_SUF) \
     ( \
       const obj_t* alpha, \
       const obj_t* x, \
       const obj_t* beta, \
       const obj_t* y  \
       BLIS_OAPI_EX_PARAMS  \
     );

GENTPROT( axpbyv )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname,EX_SUF) \
     ( \
       const obj_t* alpha, \
       const obj_t* x, \
       const obj_t* y  \
       BLIS_OAPI_EX_PARAMS  \
     );

GENTPROT( axpyv )
GENTPROT( scal2v )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname,EX_SUF) \
     ( \
       const obj_t* x, \
       const obj_t* y, \
       const obj_t* rho  \
       BLIS_OAPI_EX_PARAMS  \
     );

GENTPROT( dotv )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname,EX_SUF) \
     ( \
       const obj_t* alpha, \
       const obj_t* x, \
       const obj_t* y, \
       const obj_t* beta, \
       const obj_t* rho  \
       BLIS_OAPI_EX_PARAMS  \
     );

GENTPROT( dotxv )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname,EX_SUF) \
     ( \
       const obj_t* x  \
       BLIS_OAPI_EX_PARAMS  \
     );

GENTPROT( invertv )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname,EX_SUF) \
     ( \
       const obj_t* alpha, \
       const obj_t* x  \
       BLIS_OAPI_EX_PARAMS  \
     );

GENTPROT( invscalv )
GENTPROT( scalv )
GENTPROT( setv )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname,EX_SUF) \
     ( \
       const obj_t* x, \
       const obj_t* y  \
       BLIS_OAPI_EX_PARAMS  \
     );

GENTPROT( swapv )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname,EX_SUF) \
     ( \
       const obj_t* x, \
       const obj_t* beta, \
       const obj_t* y  \
       BLIS_OAPI_EX_PARAMS  \
     );

GENTPROT( xpbyv )

// end bli_l1v_oapi.h
#line 47 "../blis-2.0/frame/1//bli_l1v.h"

// begin bli_xapi_undef.h
#line 1 "../blis-2.0/frame/include//bli_xapi_undef.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

// This file un-defines macros used to allow the _oapi.c and _tapi.c files to
// produce object and typed APIs that omit or contain expert parameters.

// Un-define all macros that allow the source code to determine which interface
// (basic or expert) we are compiling.
#undef  BLIS_OAPI_BASIC
#undef  BLIS_OAPI_EXPERT
#undef  BLIS_TAPI_BASIC
#undef  BLIS_TAPI_EXPERT

// Un-define the macro to omit or add the function name suffix (in function
// definitions).
#undef  EX_SUF

// Un-define the macro to omit or add expert arguments from function signatures
// and prototypes.
#undef  BLIS_OAPI_EX_PARAMS
#undef  BLIS_TAPI_EX_PARAMS

// Un-define the macro to omit or add local expert variables.
#undef  BLIS_OAPI_EX_DECLS
#undef  BLIS_TAPI_EX_DECLS

// end bli_xapi_undef.h
#line 48 "../blis-2.0/frame/1//bli_l1v.h"


// Prototype typed APIs (expert and non-expert).
// begin bli_tapi_ex.h
#line 1 "../blis-2.0/frame/include//bli_tapi_ex.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

// This file defines macros used to allow the _tapi.c files to produce
// typed APIs that contain context parameters.

// Define a macro that allows the source code to determine which interface
// (basic or expert) we are compiling.
#undef  BLIS_TAPI_EXPERT
#define BLIS_TAPI_EXPERT

// Define the macro to add a suffix to the function names (in function
// definitions).
#undef  EX_SUF
#define EX_SUF BLIS_TAPI_EX_SUF

// Define the macro to add expert arguments to function signatures
// and prototypes.
#undef  BLIS_TAPI_EX_PARAMS
#define BLIS_TAPI_EX_PARAMS   , const cntx_t* cntx, const rntm_t* rntm

// Define the macro to omit the expert variable declaration block, since
// it is not needed when expert parameters are passed in through the API.
#undef  BLIS_TAPI_EX_DECLS
#define BLIS_TAPI_EX_DECLS

// end bli_tapi_ex.h
#line 51 "../blis-2.0/frame/1//bli_l1v.h"

// begin bli_l1v_tapi.h
#line 1 "../blis-2.0/frame/1//bli_l1v_tapi.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/


//
// Prototype BLAS-like interfaces with typed operands.
//

#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname,EX_SUF) \
      ( \
              conj_t conjx, \
              dim_t  n, \
        const ctype* x, inc_t incx, \
              ctype* y, inc_t incy  \
        BLIS_TAPI_EX_PARAMS  \
      );

INSERT_GENTPROT_BASIC( addv )
INSERT_GENTPROT_BASIC( copyv )
INSERT_GENTPROT_BASIC( subv )


#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname,EX_SUF) \
     ( \
             dim_t  n, \
       const ctype* x, inc_t incx, \
             dim_t* index  \
       BLIS_TAPI_EX_PARAMS  \
     ); \

INSERT_GENTPROT_BASIC( amaxv )


#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname,EX_SUF) \
     ( \
             conj_t conjx, \
             dim_t  n, \
       const ctype* alpha, \
       const ctype* x, inc_t incx, \
       const ctype* beta, \
             ctype* y, inc_t incy  \
       BLIS_TAPI_EX_PARAMS  \
     ); \

INSERT_GENTPROT_BASIC( axpbyv )


#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname,EX_SUF) \
     ( \
             conj_t conjx, \
             dim_t  n, \
       const ctype* alpha, \
       const ctype* x, inc_t incx, \
             ctype* y, inc_t incy  \
       BLIS_TAPI_EX_PARAMS  \
     ); \

INSERT_GENTPROT_BASIC( axpyv )
INSERT_GENTPROT_BASIC( scal2v )


#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname,EX_SUF) \
     ( \
             conj_t conjx, \
             conj_t conjy, \
             dim_t  n, \
       const ctype* x, inc_t incx, \
       const ctype* y, inc_t incy, \
             ctype* rho  \
       BLIS_TAPI_EX_PARAMS  \
     ); \

INSERT_GENTPROT_BASIC( dotv )


#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname,EX_SUF) \
     ( \
             conj_t conjx, \
             conj_t conjy, \
             dim_t  n, \
       const ctype* alpha, \
       const ctype* x, inc_t incx, \
       const ctype* y, inc_t incy, \
       const ctype* beta, \
             ctype* rho  \
       BLIS_TAPI_EX_PARAMS  \
     ); \

INSERT_GENTPROT_BASIC( dotxv )


#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname,EX_SUF) \
     ( \
       dim_t  n, \
       ctype* x, inc_t incx  \
       BLIS_TAPI_EX_PARAMS  \
     ); \

INSERT_GENTPROT_BASIC( invertv )


#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname,EX_SUF) \
     ( \
             conj_t conjalpha, \
             dim_t  n, \
       const ctype* alpha, \
             ctype* x, inc_t incx  \
       BLIS_TAPI_EX_PARAMS  \
     ); \

INSERT_GENTPROT_BASIC( invscalv )
INSERT_GENTPROT_BASIC( scalv )
INSERT_GENTPROT_BASIC( setv )


#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname,EX_SUF) \
     ( \
       dim_t  n, \
       ctype* x, inc_t incx, \
       ctype* y, inc_t incy  \
       BLIS_TAPI_EX_PARAMS  \
     ); \

INSERT_GENTPROT_BASIC( swapv )


#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname,EX_SUF) \
     ( \
             conj_t conjx, \
             dim_t  n, \
       const ctype* x, inc_t incx, \
       const ctype* beta, \
             ctype* y, inc_t incy  \
       BLIS_TAPI_EX_PARAMS  \
     ); \

INSERT_GENTPROT_BASIC( xpbyv )
// end bli_l1v_tapi.h
#line 52 "../blis-2.0/frame/1//bli_l1v.h"

// begin bli_l1v_ft.h
#line 1 "../blis-2.0/frame/1//bli_l1v_ft.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/


//
// -- Level-1v function types --------------------------------------------------
//

// addv, copyv, subv

#undef  GENTDEF
#define GENTDEF( ctype, ch, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,EX_SUF,tsuf)) \
     ( \
             conj_t conjx, \
             dim_t  n, \
       const ctype* x, inc_t incx, \
             ctype* y, inc_t incy  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTDEF( addv )
INSERT_GENTDEF( copyv )
INSERT_GENTDEF( subv )

// amaxv

#undef  GENTDEF
#define GENTDEF( ctype, ch, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,EX_SUF,tsuf)) \
     ( \
             dim_t  n, \
       const ctype* x, inc_t incx, \
             dim_t* index  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTDEF( amaxv )

// axpbyv

#undef  GENTDEF
#define GENTDEF( ctype, ch, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,EX_SUF,tsuf)) \
     ( \
             conj_t conjx, \
             dim_t  n, \
       const ctype* alpha, \
       const ctype* x, inc_t incx, \
       const ctype* beta, \
             ctype* y, inc_t incy  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTDEF( axpbyv )

// axpyv, scal2v

#undef  GENTDEF
#define GENTDEF( ctype, ch, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,EX_SUF,tsuf)) \
     ( \
             conj_t conjx, \
             dim_t  n, \
       const ctype* alpha, \
       const ctype* x, inc_t incx, \
             ctype* y, inc_t incy  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTDEF( axpyv )
INSERT_GENTDEF( scal2v )

// dotv

#undef  GENTDEF
#define GENTDEF( ctype, ch, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,EX_SUF,tsuf)) \
     ( \
             conj_t conjx, \
             conj_t conjy, \
             dim_t  n, \
       const ctype* x, inc_t incx, \
       const ctype* y, inc_t incy, \
             ctype* rho  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTDEF( dotv )

// dotxv

#undef  GENTDEF
#define GENTDEF( ctype, ch, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,EX_SUF,tsuf)) \
     ( \
             conj_t conjx, \
             conj_t conjy, \
             dim_t  n, \
       const ctype* alpha, \
       const ctype* x, inc_t incx, \
       const ctype* y, inc_t incy, \
       const ctype* beta, \
             ctype* rho  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTDEF( dotxv )

// invertv

#undef  GENTDEF
#define GENTDEF( ctype, ch, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,EX_SUF,tsuf)) \
     ( \
       dim_t  n, \
       ctype* x, inc_t incx  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTDEF( invertv )

// invscalv, scalv, setv

#undef  GENTDEF
#define GENTDEF( ctype, ch, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,EX_SUF,tsuf)) \
     ( \
             conj_t conjalpha, \
             dim_t  n, \
       const ctype* alpha, \
             ctype* x, inc_t incx  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTDEF( invscalv )
INSERT_GENTDEF( scalv )
INSERT_GENTDEF( setv )

// swapv

#undef  GENTDEF
#define GENTDEF( ctype, ch, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,EX_SUF,tsuf)) \
     ( \
       dim_t  n, \
       ctype* x, inc_t incx, \
       ctype* y, inc_t incy  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTDEF( swapv )

// xpybv

#undef  GENTDEF
#define GENTDEF( ctype, ch, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,EX_SUF,tsuf)) \
     ( \
             conj_t conjx, \
             dim_t  n, \
       const ctype* x, inc_t incx, \
       const ctype* beta, \
             ctype* y, inc_t incy  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTDEF( xpbyv )


// end bli_l1v_ft.h
#line 53 "../blis-2.0/frame/1//bli_l1v.h"

// begin bli_xapi_undef.h
#line 1 "../blis-2.0/frame/include//bli_xapi_undef.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

// This file un-defines macros used to allow the _oapi.c and _tapi.c files to
// produce object and typed APIs that omit or contain expert parameters.

// Un-define all macros that allow the source code to determine which interface
// (basic or expert) we are compiling.
#undef  BLIS_OAPI_BASIC
#undef  BLIS_OAPI_EXPERT
#undef  BLIS_TAPI_BASIC
#undef  BLIS_TAPI_EXPERT

// Un-define the macro to omit or add the function name suffix (in function
// definitions).
#undef  EX_SUF

// Un-define the macro to omit or add expert arguments from function signatures
// and prototypes.
#undef  BLIS_OAPI_EX_PARAMS
#undef  BLIS_TAPI_EX_PARAMS

// Un-define the macro to omit or add local expert variables.
#undef  BLIS_OAPI_EX_DECLS
#undef  BLIS_TAPI_EX_DECLS

// end bli_xapi_undef.h
#line 54 "../blis-2.0/frame/1//bli_l1v.h"


// begin bli_tapi_ba.h
#line 1 "../blis-2.0/frame/include//bli_tapi_ba.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

// This file defines macros used to allow the _tapi.c files to produce
// typed APIs that omit expert parameters.

// Define a macro that allows the source code to determine which interface
// (basic or expert) we are compiling.
#undef  BLIS_TAPI_BASIC
#define BLIS_TAPI_BASIC

// Define the macro to omit a suffix from the function names (in function
// definitions).
#undef  EX_SUF
#define EX_SUF

// Define the macro to omit expert arguments from function signatures
// and prototypes.
#undef  BLIS_TAPI_EX_PARAMS
#define BLIS_TAPI_EX_PARAMS

// Define the macro to add local expert variables that are initialized
// to NULL. The "( void )" statements are to prevent unused variable
// warnings by the compiler.
#undef  BLIS_TAPI_EX_DECLS
#define BLIS_TAPI_EX_DECLS   const cntx_t* cntx = NULL; ( void )cntx; \
                                   rntm_t* rntm = NULL; ( void )rntm;

// end bli_tapi_ba.h
#line 56 "../blis-2.0/frame/1//bli_l1v.h"

// begin bli_l1v_tapi.h
#line 1 "../blis-2.0/frame/1//bli_l1v_tapi.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/


//
// Prototype BLAS-like interfaces with typed operands.
//

#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname,EX_SUF) \
      ( \
              conj_t conjx, \
              dim_t  n, \
        const ctype* x, inc_t incx, \
              ctype* y, inc_t incy  \
        BLIS_TAPI_EX_PARAMS  \
      );

INSERT_GENTPROT_BASIC( addv )
INSERT_GENTPROT_BASIC( copyv )
INSERT_GENTPROT_BASIC( subv )


#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname,EX_SUF) \
     ( \
             dim_t  n, \
       const ctype* x, inc_t incx, \
             dim_t* index  \
       BLIS_TAPI_EX_PARAMS  \
     ); \

INSERT_GENTPROT_BASIC( amaxv )


#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname,EX_SUF) \
     ( \
             conj_t conjx, \
             dim_t  n, \
       const ctype* alpha, \
       const ctype* x, inc_t incx, \
       const ctype* beta, \
             ctype* y, inc_t incy  \
       BLIS_TAPI_EX_PARAMS  \
     ); \

INSERT_GENTPROT_BASIC( axpbyv )


#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname,EX_SUF) \
     ( \
             conj_t conjx, \
             dim_t  n, \
       const ctype* alpha, \
       const ctype* x, inc_t incx, \
             ctype* y, inc_t incy  \
       BLIS_TAPI_EX_PARAMS  \
     ); \

INSERT_GENTPROT_BASIC( axpyv )
INSERT_GENTPROT_BASIC( scal2v )


#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname,EX_SUF) \
     ( \
             conj_t conjx, \
             conj_t conjy, \
             dim_t  n, \
       const ctype* x, inc_t incx, \
       const ctype* y, inc_t incy, \
             ctype* rho  \
       BLIS_TAPI_EX_PARAMS  \
     ); \

INSERT_GENTPROT_BASIC( dotv )


#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname,EX_SUF) \
     ( \
             conj_t conjx, \
             conj_t conjy, \
             dim_t  n, \
       const ctype* alpha, \
       const ctype* x, inc_t incx, \
       const ctype* y, inc_t incy, \
       const ctype* beta, \
             ctype* rho  \
       BLIS_TAPI_EX_PARAMS  \
     ); \

INSERT_GENTPROT_BASIC( dotxv )


#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname,EX_SUF) \
     ( \
       dim_t  n, \
       ctype* x, inc_t incx  \
       BLIS_TAPI_EX_PARAMS  \
     ); \

INSERT_GENTPROT_BASIC( invertv )


#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname,EX_SUF) \
     ( \
             conj_t conjalpha, \
             dim_t  n, \
       const ctype* alpha, \
             ctype* x, inc_t incx  \
       BLIS_TAPI_EX_PARAMS  \
     ); \

INSERT_GENTPROT_BASIC( invscalv )
INSERT_GENTPROT_BASIC( scalv )
INSERT_GENTPROT_BASIC( setv )


#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname,EX_SUF) \
     ( \
       dim_t  n, \
       ctype* x, inc_t incx, \
       ctype* y, inc_t incy  \
       BLIS_TAPI_EX_PARAMS  \
     ); \

INSERT_GENTPROT_BASIC( swapv )


#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname,EX_SUF) \
     ( \
             conj_t conjx, \
             dim_t  n, \
       const ctype* x, inc_t incx, \
       const ctype* beta, \
             ctype* y, inc_t incy  \
       BLIS_TAPI_EX_PARAMS  \
     ); \

INSERT_GENTPROT_BASIC( xpbyv )
// end bli_l1v_tapi.h
#line 57 "../blis-2.0/frame/1//bli_l1v.h"

// begin bli_l1v_ft.h
#line 1 "../blis-2.0/frame/1//bli_l1v_ft.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/


//
// -- Level-1v function types --------------------------------------------------
//

// addv, copyv, subv

#undef  GENTDEF
#define GENTDEF( ctype, ch, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,EX_SUF,tsuf)) \
     ( \
             conj_t conjx, \
             dim_t  n, \
       const ctype* x, inc_t incx, \
             ctype* y, inc_t incy  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTDEF( addv )
INSERT_GENTDEF( copyv )
INSERT_GENTDEF( subv )

// amaxv

#undef  GENTDEF
#define GENTDEF( ctype, ch, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,EX_SUF,tsuf)) \
     ( \
             dim_t  n, \
       const ctype* x, inc_t incx, \
             dim_t* index  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTDEF( amaxv )

// axpbyv

#undef  GENTDEF
#define GENTDEF( ctype, ch, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,EX_SUF,tsuf)) \
     ( \
             conj_t conjx, \
             dim_t  n, \
       const ctype* alpha, \
       const ctype* x, inc_t incx, \
       const ctype* beta, \
             ctype* y, inc_t incy  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTDEF( axpbyv )

// axpyv, scal2v

#undef  GENTDEF
#define GENTDEF( ctype, ch, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,EX_SUF,tsuf)) \
     ( \
             conj_t conjx, \
             dim_t  n, \
       const ctype* alpha, \
       const ctype* x, inc_t incx, \
             ctype* y, inc_t incy  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTDEF( axpyv )
INSERT_GENTDEF( scal2v )

// dotv

#undef  GENTDEF
#define GENTDEF( ctype, ch, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,EX_SUF,tsuf)) \
     ( \
             conj_t conjx, \
             conj_t conjy, \
             dim_t  n, \
       const ctype* x, inc_t incx, \
       const ctype* y, inc_t incy, \
             ctype* rho  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTDEF( dotv )

// dotxv

#undef  GENTDEF
#define GENTDEF( ctype, ch, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,EX_SUF,tsuf)) \
     ( \
             conj_t conjx, \
             conj_t conjy, \
             dim_t  n, \
       const ctype* alpha, \
       const ctype* x, inc_t incx, \
       const ctype* y, inc_t incy, \
       const ctype* beta, \
             ctype* rho  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTDEF( dotxv )

// invertv

#undef  GENTDEF
#define GENTDEF( ctype, ch, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,EX_SUF,tsuf)) \
     ( \
       dim_t  n, \
       ctype* x, inc_t incx  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTDEF( invertv )

// invscalv, scalv, setv

#undef  GENTDEF
#define GENTDEF( ctype, ch, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,EX_SUF,tsuf)) \
     ( \
             conj_t conjalpha, \
             dim_t  n, \
       const ctype* alpha, \
             ctype* x, inc_t incx  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTDEF( invscalv )
INSERT_GENTDEF( scalv )
INSERT_GENTDEF( setv )

// swapv

#undef  GENTDEF
#define GENTDEF( ctype, ch, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,EX_SUF,tsuf)) \
     ( \
       dim_t  n, \
       ctype* x, inc_t incx, \
       ctype* y, inc_t incy  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTDEF( swapv )

// xpybv

#undef  GENTDEF
#define GENTDEF( ctype, ch, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,EX_SUF,tsuf)) \
     ( \
             conj_t conjx, \
             dim_t  n, \
       const ctype* x, inc_t incx, \
       const ctype* beta, \
             ctype* y, inc_t incy  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTDEF( xpbyv )


// end bli_l1v_ft.h
#line 58 "../blis-2.0/frame/1//bli_l1v.h"

// begin bli_xapi_undef.h
#line 1 "../blis-2.0/frame/include//bli_xapi_undef.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

// This file un-defines macros used to allow the _oapi.c and _tapi.c files to
// produce object and typed APIs that omit or contain expert parameters.

// Un-define all macros that allow the source code to determine which interface
// (basic or expert) we are compiling.
#undef  BLIS_OAPI_BASIC
#undef  BLIS_OAPI_EXPERT
#undef  BLIS_TAPI_BASIC
#undef  BLIS_TAPI_EXPERT

// Un-define the macro to omit or add the function name suffix (in function
// definitions).
#undef  EX_SUF

// Un-define the macro to omit or add expert arguments from function signatures
// and prototypes.
#undef  BLIS_OAPI_EX_PARAMS
#undef  BLIS_TAPI_EX_PARAMS

// Un-define the macro to omit or add local expert variables.
#undef  BLIS_OAPI_EX_DECLS
#undef  BLIS_TAPI_EX_DECLS

// end bli_xapi_undef.h
#line 59 "../blis-2.0/frame/1//bli_l1v.h"


// Generate function pointer arrays for tapi functions (expert only).
// begin bli_l1v_fpa.h
#line 1 "../blis-2.0/frame/1//bli_l1v_fpa.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

//
// Prototype function pointer query interface.
//

#undef  GENPROT
#define GENPROT( opname ) \
\
PASTECH(opname,BLIS_TAPI_EX_SUF,_vft) \
PASTEMAC(opname,BLIS_TAPI_EX_SUF,_qfp)( num_t dt );

GENPROT( addv )
GENPROT( copyv )
GENPROT( subv )
GENPROT( amaxv )
GENPROT( axpbyv )
GENPROT( axpyv )
GENPROT( scal2v )
GENPROT( dotv )
GENPROT( dotxv )
GENPROT( invertv )
GENPROT( invscalv )
GENPROT( scalv )
GENPROT( setv )
GENPROT( swapv )
GENPROT( xpbyv )

// end bli_l1v_fpa.h
#line 62 "../blis-2.0/frame/1//bli_l1v.h"


// Pack-related
// NOTE: packv and unpackv are temporarily disabled.
//#include "bli_packv.h"
//#include "bli_unpackv.h"

// Other
// NOTE: scalv control tree code is temporarily disabled.
//#include "bli_scalv_cntl.h"
//#include "bli_scalv_int.h"

// end bli_l1v.h
#line 179 "../blis-2.0/frame/include/blis.h"



// -- Level-1d operations --

// begin bli_l1d.h
#line 1 "../blis-2.0/frame/1d//bli_l1d.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

// begin bli_l1d_check.h
#line 1 "../blis-2.0/frame/1d//bli_l1d_check.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/


//
// Prototype object-based check functions.
//

#undef  GENTPROT
#define GENTPROT( opname ) \
\
void PASTEMAC(opname,_check) \
     ( \
       const obj_t*  x, \
       const obj_t*  y  \
    );

GENTPROT( addd )
GENTPROT( copyd )
GENTPROT( subd )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
void PASTEMAC(opname,_check) \
     ( \
       const obj_t*  alpha, \
       const obj_t*  x, \
       const obj_t*  y  \
    );

GENTPROT( axpyd )
GENTPROT( scal2d )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
void PASTEMAC(opname,_check) \
     ( \
       const obj_t*  x  \
    );

GENTPROT( invertd )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
void PASTEMAC(opname,_check) \
     ( \
       const obj_t*  alpha, \
       const obj_t*  x  \
    );

GENTPROT( invscald )
GENTPROT( scald )
GENTPROT( setd )
GENTPROT( setid )
GENTPROT( shiftd )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
void PASTEMAC(opname,_check) \
     ( \
       const obj_t*  x, \
       const obj_t*  beta, \
       const obj_t*  y  \
    );

GENTPROT( xpbyd )


// -----------------------------------------------------------------------------

void bli_l1d_xy_check
     (
       const obj_t*  x,
       const obj_t*  y
     );

void bli_l1d_axy_check
     (
       const obj_t*  alpha,
       const obj_t*  x,
       const obj_t*  y
     );

void bli_l1d_x_check
     (
       const obj_t*  x
     );

void bli_l1d_ax_check
     (
       const obj_t*  alpha,
       const obj_t*  x
     );

// end bli_l1d_check.h
#line 36 "../blis-2.0/frame/1d//bli_l1d.h"


// Prototype object APIs (expert and non-expert).
// begin bli_oapi_ex.h
#line 1 "../blis-2.0/frame/include//bli_oapi_ex.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

// This file defines macros used to allow the _oapi.c files to produce
// object APIs that contain context parameters.

// Define a macro that allows the source code to determine which interface
// (basic or expert) we are compiling.
#undef  BLIS_OAPI_EXPERT
#define BLIS_OAPI_EXPERT

// Define the macro to add a suffix to the function names (in function
// definitions).
#undef  EX_SUF
#define EX_SUF BLIS_OAPI_EX_SUF

// Define the macro to add expert arguments to function signatures
// and prototypes.
#undef  BLIS_OAPI_EX_PARAMS
#define BLIS_OAPI_EX_PARAMS   , const cntx_t* cntx, const rntm_t* rntm

// Define the macro to omit the expert variable declaration block, since
// it is not needed when expert parameters are passed in through the API.
#undef  BLIS_OAPI_EX_DECLS
#define BLIS_OAPI_EX_DECLS

// end bli_oapi_ex.h
#line 39 "../blis-2.0/frame/1d//bli_l1d.h"

// begin bli_l1d_oapi.h
#line 1 "../blis-2.0/frame/1d//bli_l1d_oapi.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/


//
// Prototype object-based interfaces.
//

#undef  GENTPROT
#define GENTPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname,EX_SUF) \
     ( \
       const obj_t* x, \
       const obj_t* y  \
       BLIS_OAPI_EX_PARAMS  \
     );

GENTPROT( addd )
GENTPROT( copyd )
GENTPROT( subd )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname,EX_SUF) \
     ( \
       const obj_t* alpha, \
       const obj_t* x, \
       const obj_t* y  \
       BLIS_OAPI_EX_PARAMS  \
     );

GENTPROT( axpyd )
GENTPROT( scal2d )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname,EX_SUF) \
     ( \
       const obj_t* x  \
       BLIS_OAPI_EX_PARAMS  \
     );

GENTPROT( invertd )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname,EX_SUF) \
     ( \
       const obj_t* alpha, \
       const obj_t* x  \
       BLIS_OAPI_EX_PARAMS  \
     );

GENTPROT( invscald )
GENTPROT( scald )
GENTPROT( setd )
GENTPROT( setid )
GENTPROT( shiftd )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname,EX_SUF) \
     ( \
       const obj_t* x, \
       const obj_t* beta, \
       const obj_t* y  \
       BLIS_OAPI_EX_PARAMS  \
     );

GENTPROT( xpbyd )

// end bli_l1d_oapi.h
#line 40 "../blis-2.0/frame/1d//bli_l1d.h"

// begin bli_xapi_undef.h
#line 1 "../blis-2.0/frame/include//bli_xapi_undef.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

// This file un-defines macros used to allow the _oapi.c and _tapi.c files to
// produce object and typed APIs that omit or contain expert parameters.

// Un-define all macros that allow the source code to determine which interface
// (basic or expert) we are compiling.
#undef  BLIS_OAPI_BASIC
#undef  BLIS_OAPI_EXPERT
#undef  BLIS_TAPI_BASIC
#undef  BLIS_TAPI_EXPERT

// Un-define the macro to omit or add the function name suffix (in function
// definitions).
#undef  EX_SUF

// Un-define the macro to omit or add expert arguments from function signatures
// and prototypes.
#undef  BLIS_OAPI_EX_PARAMS
#undef  BLIS_TAPI_EX_PARAMS

// Un-define the macro to omit or add local expert variables.
#undef  BLIS_OAPI_EX_DECLS
#undef  BLIS_TAPI_EX_DECLS

// end bli_xapi_undef.h
#line 41 "../blis-2.0/frame/1d//bli_l1d.h"


// begin bli_oapi_ba.h
#line 1 "../blis-2.0/frame/include//bli_oapi_ba.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

// This file defines macros used to allow the _oapi.c files to produce
// object APIs that omit expert parameters.

// Define a macro that allows the source code to determine which interface
// (basic or expert) we are compiling.
#undef  BLIS_OAPI_BASIC
#define BLIS_OAPI_BASIC

// Define the macro to omit a suffix from the function names (in function
// definitions).
#undef  EX_SUF
#define EX_SUF

// Define the macro to omit expert arguments from function signatures
// and prototypes.
#undef  BLIS_OAPI_EX_PARAMS
#define BLIS_OAPI_EX_PARAMS

// Define the macro to add local expert variables that are initialized
// to NULL. The "( void )" statements are to prevent unused variable
// warnings by the compiler.
#undef  BLIS_OAPI_EX_DECLS
#define BLIS_OAPI_EX_DECLS   const cntx_t* cntx = NULL; ( void )cntx; \
                                   rntm_t* rntm = NULL; ( void )rntm;

// end bli_oapi_ba.h
#line 43 "../blis-2.0/frame/1d//bli_l1d.h"

// begin bli_l1d_oapi.h
#line 1 "../blis-2.0/frame/1d//bli_l1d_oapi.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/


//
// Prototype object-based interfaces.
//

#undef  GENTPROT
#define GENTPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname,EX_SUF) \
     ( \
       const obj_t* x, \
       const obj_t* y  \
       BLIS_OAPI_EX_PARAMS  \
     );

GENTPROT( addd )
GENTPROT( copyd )
GENTPROT( subd )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname,EX_SUF) \
     ( \
       const obj_t* alpha, \
       const obj_t* x, \
       const obj_t* y  \
       BLIS_OAPI_EX_PARAMS  \
     );

GENTPROT( axpyd )
GENTPROT( scal2d )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname,EX_SUF) \
     ( \
       const obj_t* x  \
       BLIS_OAPI_EX_PARAMS  \
     );

GENTPROT( invertd )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname,EX_SUF) \
     ( \
       const obj_t* alpha, \
       const obj_t* x  \
       BLIS_OAPI_EX_PARAMS  \
     );

GENTPROT( invscald )
GENTPROT( scald )
GENTPROT( setd )
GENTPROT( setid )
GENTPROT( shiftd )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname,EX_SUF) \
     ( \
       const obj_t* x, \
       const obj_t* beta, \
       const obj_t* y  \
       BLIS_OAPI_EX_PARAMS  \
     );

GENTPROT( xpbyd )

// end bli_l1d_oapi.h
#line 44 "../blis-2.0/frame/1d//bli_l1d.h"

// begin bli_xapi_undef.h
#line 1 "../blis-2.0/frame/include//bli_xapi_undef.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

// This file un-defines macros used to allow the _oapi.c and _tapi.c files to
// produce object and typed APIs that omit or contain expert parameters.

// Un-define all macros that allow the source code to determine which interface
// (basic or expert) we are compiling.
#undef  BLIS_OAPI_BASIC
#undef  BLIS_OAPI_EXPERT
#undef  BLIS_TAPI_BASIC
#undef  BLIS_TAPI_EXPERT

// Un-define the macro to omit or add the function name suffix (in function
// definitions).
#undef  EX_SUF

// Un-define the macro to omit or add expert arguments from function signatures
// and prototypes.
#undef  BLIS_OAPI_EX_PARAMS
#undef  BLIS_TAPI_EX_PARAMS

// Un-define the macro to omit or add local expert variables.
#undef  BLIS_OAPI_EX_DECLS
#undef  BLIS_TAPI_EX_DECLS

// end bli_xapi_undef.h
#line 45 "../blis-2.0/frame/1d//bli_l1d.h"


// Prototype typed APIs (expert and non-expert).
// begin bli_tapi_ex.h
#line 1 "../blis-2.0/frame/include//bli_tapi_ex.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

// This file defines macros used to allow the _tapi.c files to produce
// typed APIs that contain context parameters.

// Define a macro that allows the source code to determine which interface
// (basic or expert) we are compiling.
#undef  BLIS_TAPI_EXPERT
#define BLIS_TAPI_EXPERT

// Define the macro to add a suffix to the function names (in function
// definitions).
#undef  EX_SUF
#define EX_SUF BLIS_TAPI_EX_SUF

// Define the macro to add expert arguments to function signatures
// and prototypes.
#undef  BLIS_TAPI_EX_PARAMS
#define BLIS_TAPI_EX_PARAMS   , const cntx_t* cntx, const rntm_t* rntm

// Define the macro to omit the expert variable declaration block, since
// it is not needed when expert parameters are passed in through the API.
#undef  BLIS_TAPI_EX_DECLS
#define BLIS_TAPI_EX_DECLS

// end bli_tapi_ex.h
#line 48 "../blis-2.0/frame/1d//bli_l1d.h"

// begin bli_l1d_tapi.h
#line 1 "../blis-2.0/frame/1d//bli_l1d_tapi.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/


//
// Prototype BLAS-like interfaces with typed operands.
//

#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname,EX_SUF) \
     ( \
             doff_t  diagoffx, \
             diag_t  diagx, \
             trans_t transx, \
             dim_t   m, \
             dim_t   n, \
       const ctype*  x, inc_t rs_x, inc_t cs_x, \
             ctype*  y, inc_t rs_y, inc_t cs_y  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTPROT_BASIC( addd )
INSERT_GENTPROT_BASIC( copyd )
INSERT_GENTPROT_BASIC( subd )


#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname,EX_SUF) \
     ( \
             doff_t  diagoffx, \
             diag_t  diagx, \
             trans_t transx, \
             dim_t   m, \
             dim_t   n, \
       const ctype*  alpha, \
       const ctype*  x, inc_t rs_x, inc_t cs_x, \
             ctype*  y, inc_t rs_y, inc_t cs_y  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTPROT_BASIC( axpyd )
INSERT_GENTPROT_BASIC( scal2d )


#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname,EX_SUF) \
     ( \
       doff_t diagoffx, \
       dim_t  m, \
       dim_t  n, \
       ctype* x, inc_t rs_x, inc_t cs_x  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTPROT_BASIC( invertd )


#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname,EX_SUF) \
     ( \
             conj_t conjalpha, \
             doff_t diagoffx, \
             dim_t  m, \
             dim_t  n, \
       const ctype* alpha, \
             ctype* x, inc_t rs_x, inc_t cs_x  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTPROT_BASIC( invscald )
INSERT_GENTPROT_BASIC( scald )
INSERT_GENTPROT_BASIC( setd )


#undef  GENTPROTR
#define GENTPROTR( ctype, ctype_r, ch, chr, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname,EX_SUF) \
     ( \
             doff_t   diagoffx, \
             dim_t    m, \
             dim_t    n, \
       const ctype_r* alpha, \
             ctype*   x, inc_t rs_x, inc_t cs_x  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTPROTR_BASIC( setid )


#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname,EX_SUF) \
     ( \
             doff_t diagoffx, \
             dim_t  m, \
             dim_t  n, \
       const ctype* alpha, \
             ctype* x, inc_t rs_x, inc_t cs_x  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTPROT_BASIC( shiftd )


#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname,EX_SUF) \
     ( \
             doff_t  diagoffx, \
             diag_t  diagx, \
             trans_t transx, \
             dim_t   m, \
             dim_t   n, \
       const ctype*  x, inc_t rs_x, inc_t cs_x, \
       const ctype*  beta, \
             ctype*  y, inc_t rs_y, inc_t cs_y  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTPROT_BASIC( xpbyd )

// end bli_l1d_tapi.h
#line 49 "../blis-2.0/frame/1d//bli_l1d.h"

// begin bli_l1d_ft.h
#line 1 "../blis-2.0/frame/1d//bli_l1d_ft.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/


//
// -- Level-1d function types --------------------------------------------------
//

// addd, copyd, subd

#undef  GENTDEF
#define GENTDEF( ctype, ch, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,EX_SUF,tsuf)) \
     ( \
             doff_t  diagoffx, \
             diag_t  diagx, \
             trans_t transx, \
             dim_t   m, \
             dim_t   n, \
       const ctype*  x, inc_t rs_x, inc_t cs_x, \
             ctype*  y, inc_t rs_y, inc_t cs_y  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTDEF( addd )
INSERT_GENTDEF( copyd )
INSERT_GENTDEF( subd )

// axpyd, scal2d

#undef  GENTDEF
#define GENTDEF( ctype, ch, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,EX_SUF,tsuf)) \
     ( \
             doff_t  diagoffx, \
             diag_t  diagx, \
             trans_t transx, \
             dim_t   m, \
             dim_t   n, \
       const ctype*  alpha, \
       const ctype*  x, inc_t rs_x, inc_t cs_x, \
             ctype*  y, inc_t rs_y, inc_t cs_y  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTDEF( axpyd )
INSERT_GENTDEF( scal2d )

// invertd

#undef  GENTDEF
#define GENTDEF( ctype, ch, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,EX_SUF,tsuf)) \
     ( \
       doff_t diagoffx, \
       dim_t  m, \
       dim_t  n, \
       ctype* x, inc_t rs_x, inc_t cs_x  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTDEF( invertd )

// invscald, scald, setd

#undef  GENTDEF
#define GENTDEF( ctype, ch, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,EX_SUF,tsuf)) \
     ( \
             conj_t conjalpha, \
             doff_t diagoffx, \
             dim_t  m, \
             dim_t  n, \
       const ctype* alpha, \
             ctype* x, inc_t rs_x, inc_t cs_x  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTDEF( invscald )
INSERT_GENTDEF( scald )
INSERT_GENTDEF( setd )

// setid

#undef  GENTDEFR
#define GENTDEFR( ctype, ctype_r, ch, chr, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,EX_SUF,tsuf)) \
     ( \
             doff_t   diagoffx, \
             dim_t    m, \
             dim_t    n, \
       const ctype_r* alpha, \
             ctype*   x, inc_t rs_x, inc_t cs_x  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTDEFR( setid )

// shiftd

#undef  GENTDEF
#define GENTDEF( ctype, ch, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,EX_SUF,tsuf)) \
     ( \
             doff_t diagoffx, \
             dim_t  m, \
             dim_t  n, \
       const ctype* alpha, \
             ctype* x, inc_t rs_x, inc_t cs_x  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTDEF( shiftd )

// xpbyd

#undef  GENTDEF
#define GENTDEF( ctype, ch, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,EX_SUF,tsuf)) \
     ( \
             doff_t  diagoffx, \
             diag_t  diagx, \
             trans_t transx, \
             dim_t   m, \
             dim_t   n, \
       const ctype*  x, inc_t rs_x, inc_t cs_x, \
       const ctype*  beta, \
             ctype*  y, inc_t rs_y, inc_t cs_y  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTDEF( xpbyd )

// end bli_l1d_ft.h
#line 50 "../blis-2.0/frame/1d//bli_l1d.h"

// begin bli_xapi_undef.h
#line 1 "../blis-2.0/frame/include//bli_xapi_undef.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

// This file un-defines macros used to allow the _oapi.c and _tapi.c files to
// produce object and typed APIs that omit or contain expert parameters.

// Un-define all macros that allow the source code to determine which interface
// (basic or expert) we are compiling.
#undef  BLIS_OAPI_BASIC
#undef  BLIS_OAPI_EXPERT
#undef  BLIS_TAPI_BASIC
#undef  BLIS_TAPI_EXPERT

// Un-define the macro to omit or add the function name suffix (in function
// definitions).
#undef  EX_SUF

// Un-define the macro to omit or add expert arguments from function signatures
// and prototypes.
#undef  BLIS_OAPI_EX_PARAMS
#undef  BLIS_TAPI_EX_PARAMS

// Un-define the macro to omit or add local expert variables.
#undef  BLIS_OAPI_EX_DECLS
#undef  BLIS_TAPI_EX_DECLS

// end bli_xapi_undef.h
#line 51 "../blis-2.0/frame/1d//bli_l1d.h"


// begin bli_tapi_ba.h
#line 1 "../blis-2.0/frame/include//bli_tapi_ba.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

// This file defines macros used to allow the _tapi.c files to produce
// typed APIs that omit expert parameters.

// Define a macro that allows the source code to determine which interface
// (basic or expert) we are compiling.
#undef  BLIS_TAPI_BASIC
#define BLIS_TAPI_BASIC

// Define the macro to omit a suffix from the function names (in function
// definitions).
#undef  EX_SUF
#define EX_SUF

// Define the macro to omit expert arguments from function signatures
// and prototypes.
#undef  BLIS_TAPI_EX_PARAMS
#define BLIS_TAPI_EX_PARAMS

// Define the macro to add local expert variables that are initialized
// to NULL. The "( void )" statements are to prevent unused variable
// warnings by the compiler.
#undef  BLIS_TAPI_EX_DECLS
#define BLIS_TAPI_EX_DECLS   const cntx_t* cntx = NULL; ( void )cntx; \
                                   rntm_t* rntm = NULL; ( void )rntm;

// end bli_tapi_ba.h
#line 53 "../blis-2.0/frame/1d//bli_l1d.h"

// begin bli_l1d_tapi.h
#line 1 "../blis-2.0/frame/1d//bli_l1d_tapi.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/


//
// Prototype BLAS-like interfaces with typed operands.
//

#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname,EX_SUF) \
     ( \
             doff_t  diagoffx, \
             diag_t  diagx, \
             trans_t transx, \
             dim_t   m, \
             dim_t   n, \
       const ctype*  x, inc_t rs_x, inc_t cs_x, \
             ctype*  y, inc_t rs_y, inc_t cs_y  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTPROT_BASIC( addd )
INSERT_GENTPROT_BASIC( copyd )
INSERT_GENTPROT_BASIC( subd )


#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname,EX_SUF) \
     ( \
             doff_t  diagoffx, \
             diag_t  diagx, \
             trans_t transx, \
             dim_t   m, \
             dim_t   n, \
       const ctype*  alpha, \
       const ctype*  x, inc_t rs_x, inc_t cs_x, \
             ctype*  y, inc_t rs_y, inc_t cs_y  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTPROT_BASIC( axpyd )
INSERT_GENTPROT_BASIC( scal2d )


#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname,EX_SUF) \
     ( \
       doff_t diagoffx, \
       dim_t  m, \
       dim_t  n, \
       ctype* x, inc_t rs_x, inc_t cs_x  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTPROT_BASIC( invertd )


#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname,EX_SUF) \
     ( \
             conj_t conjalpha, \
             doff_t diagoffx, \
             dim_t  m, \
             dim_t  n, \
       const ctype* alpha, \
             ctype* x, inc_t rs_x, inc_t cs_x  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTPROT_BASIC( invscald )
INSERT_GENTPROT_BASIC( scald )
INSERT_GENTPROT_BASIC( setd )


#undef  GENTPROTR
#define GENTPROTR( ctype, ctype_r, ch, chr, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname,EX_SUF) \
     ( \
             doff_t   diagoffx, \
             dim_t    m, \
             dim_t    n, \
       const ctype_r* alpha, \
             ctype*   x, inc_t rs_x, inc_t cs_x  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTPROTR_BASIC( setid )


#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname,EX_SUF) \
     ( \
             doff_t diagoffx, \
             dim_t  m, \
             dim_t  n, \
       const ctype* alpha, \
             ctype* x, inc_t rs_x, inc_t cs_x  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTPROT_BASIC( shiftd )


#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname,EX_SUF) \
     ( \
             doff_t  diagoffx, \
             diag_t  diagx, \
             trans_t transx, \
             dim_t   m, \
             dim_t   n, \
       const ctype*  x, inc_t rs_x, inc_t cs_x, \
       const ctype*  beta, \
             ctype*  y, inc_t rs_y, inc_t cs_y  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTPROT_BASIC( xpbyd )

// end bli_l1d_tapi.h
#line 54 "../blis-2.0/frame/1d//bli_l1d.h"

// begin bli_l1d_ft.h
#line 1 "../blis-2.0/frame/1d//bli_l1d_ft.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/


//
// -- Level-1d function types --------------------------------------------------
//

// addd, copyd, subd

#undef  GENTDEF
#define GENTDEF( ctype, ch, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,EX_SUF,tsuf)) \
     ( \
             doff_t  diagoffx, \
             diag_t  diagx, \
             trans_t transx, \
             dim_t   m, \
             dim_t   n, \
       const ctype*  x, inc_t rs_x, inc_t cs_x, \
             ctype*  y, inc_t rs_y, inc_t cs_y  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTDEF( addd )
INSERT_GENTDEF( copyd )
INSERT_GENTDEF( subd )

// axpyd, scal2d

#undef  GENTDEF
#define GENTDEF( ctype, ch, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,EX_SUF,tsuf)) \
     ( \
             doff_t  diagoffx, \
             diag_t  diagx, \
             trans_t transx, \
             dim_t   m, \
             dim_t   n, \
       const ctype*  alpha, \
       const ctype*  x, inc_t rs_x, inc_t cs_x, \
             ctype*  y, inc_t rs_y, inc_t cs_y  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTDEF( axpyd )
INSERT_GENTDEF( scal2d )

// invertd

#undef  GENTDEF
#define GENTDEF( ctype, ch, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,EX_SUF,tsuf)) \
     ( \
       doff_t diagoffx, \
       dim_t  m, \
       dim_t  n, \
       ctype* x, inc_t rs_x, inc_t cs_x  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTDEF( invertd )

// invscald, scald, setd

#undef  GENTDEF
#define GENTDEF( ctype, ch, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,EX_SUF,tsuf)) \
     ( \
             conj_t conjalpha, \
             doff_t diagoffx, \
             dim_t  m, \
             dim_t  n, \
       const ctype* alpha, \
             ctype* x, inc_t rs_x, inc_t cs_x  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTDEF( invscald )
INSERT_GENTDEF( scald )
INSERT_GENTDEF( setd )

// setid

#undef  GENTDEFR
#define GENTDEFR( ctype, ctype_r, ch, chr, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,EX_SUF,tsuf)) \
     ( \
             doff_t   diagoffx, \
             dim_t    m, \
             dim_t    n, \
       const ctype_r* alpha, \
             ctype*   x, inc_t rs_x, inc_t cs_x  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTDEFR( setid )

// shiftd

#undef  GENTDEF
#define GENTDEF( ctype, ch, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,EX_SUF,tsuf)) \
     ( \
             doff_t diagoffx, \
             dim_t  m, \
             dim_t  n, \
       const ctype* alpha, \
             ctype* x, inc_t rs_x, inc_t cs_x  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTDEF( shiftd )

// xpbyd

#undef  GENTDEF
#define GENTDEF( ctype, ch, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,EX_SUF,tsuf)) \
     ( \
             doff_t  diagoffx, \
             diag_t  diagx, \
             trans_t transx, \
             dim_t   m, \
             dim_t   n, \
       const ctype*  x, inc_t rs_x, inc_t cs_x, \
       const ctype*  beta, \
             ctype*  y, inc_t rs_y, inc_t cs_y  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTDEF( xpbyd )

// end bli_l1d_ft.h
#line 55 "../blis-2.0/frame/1d//bli_l1d.h"

// begin bli_xapi_undef.h
#line 1 "../blis-2.0/frame/include//bli_xapi_undef.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

// This file un-defines macros used to allow the _oapi.c and _tapi.c files to
// produce object and typed APIs that omit or contain expert parameters.

// Un-define all macros that allow the source code to determine which interface
// (basic or expert) we are compiling.
#undef  BLIS_OAPI_BASIC
#undef  BLIS_OAPI_EXPERT
#undef  BLIS_TAPI_BASIC
#undef  BLIS_TAPI_EXPERT

// Un-define the macro to omit or add the function name suffix (in function
// definitions).
#undef  EX_SUF

// Un-define the macro to omit or add expert arguments from function signatures
// and prototypes.
#undef  BLIS_OAPI_EX_PARAMS
#undef  BLIS_TAPI_EX_PARAMS

// Un-define the macro to omit or add local expert variables.
#undef  BLIS_OAPI_EX_DECLS
#undef  BLIS_TAPI_EX_DECLS

// end bli_xapi_undef.h
#line 56 "../blis-2.0/frame/1d//bli_l1d.h"


// Generate function pointer arrays for tapi functions (expert only).
// begin bli_l1d_fpa.h
#line 1 "../blis-2.0/frame/1d//bli_l1d_fpa.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

//
// Prototype function pointer query interface.
//

#undef  GENPROT
#define GENPROT( opname ) \
\
PASTECH(opname,BLIS_TAPI_EX_SUF,_vft) \
PASTEMAC(opname,BLIS_TAPI_EX_SUF,_qfp)( num_t dt );

GENPROT( addd )
GENPROT( copyd )
GENPROT( subd )
GENPROT( axpyd )
GENPROT( scal2d )
GENPROT( invertd )
GENPROT( invscald )
GENPROT( scald )
GENPROT( setd )
GENPROT( setid )
GENPROT( shiftd )
GENPROT( xpbyd )

// end bli_l1d_fpa.h
#line 59 "../blis-2.0/frame/1d//bli_l1d.h"


// end bli_l1d.h
#line 184 "../blis-2.0/frame/include/blis.h"



// -- Level-1f operations --

// begin bli_l1f.h
#line 1 "../blis-2.0/frame/1f//bli_l1f.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

// begin bli_l1f_check.h
#line 1 "../blis-2.0/frame/1f//bli_l1f_check.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/


//
// Prototype object-based check functions.
//

#undef  GENTPROT
#define GENTPROT( opname ) \
\
void PASTEMAC(opname,_check) \
     ( \
       const obj_t* alphax, \
       const obj_t* alphay, \
       const obj_t* x, \
       const obj_t* y, \
       const obj_t* z  \
    );

GENTPROT( axpy2v )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
void PASTEMAC(opname,_check) \
     ( \
       const obj_t* alpha, \
       const obj_t* a, \
       const obj_t* x, \
       const obj_t* y  \
    );

GENTPROT( axpyf )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
void PASTEMAC(opname,_check) \
     ( \
       const obj_t* alpha, \
       const obj_t* xt, \
       const obj_t* x, \
       const obj_t* y, \
       const obj_t* rho, \
       const obj_t* z  \
    );

GENTPROT( dotaxpyv )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
void PASTEMAC(opname,_check) \
     ( \
       const obj_t* alpha, \
       const obj_t* at, \
       const obj_t* a, \
       const obj_t* w, \
       const obj_t* x, \
       const obj_t* beta, \
       const obj_t* y, \
       const obj_t* z  \
    );

GENTPROT( dotxaxpyf )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
void PASTEMAC(opname,_check) \
     ( \
       const obj_t* alpha, \
       const obj_t* a, \
       const obj_t* x, \
       const obj_t* beta, \
       const obj_t* y  \
    );

GENTPROT( dotxf )

// end bli_l1f_check.h
#line 36 "../blis-2.0/frame/1f//bli_l1f.h"


// Define kernel function types.
// begin bli_l1f_ker_ft.h
#line 1 "../blis-2.0/frame/1f//bli_l1f_ker_ft.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_L1F_KER_FT_H
#define BLIS_L1F_KER_FT_H


//
// -- Level-1f kernel function types -------------------------------------------
//

#undef  GENTDEF
#define GENTDEF( opname ) \
\
typedef void (*PASTECH(opname,_ker_ft)) \
     ( \
       PASTECH(opname,_params), \
       BLIS_CNTX_PARAM  \
     );

GENTDEF( axpy2v )
GENTDEF( axpyf )
GENTDEF( dotaxpyv )
GENTDEF( dotxaxpyf )
GENTDEF( dotxf )

#endif

// end bli_l1f_ker_ft.h
#line 39 "../blis-2.0/frame/1f//bli_l1f.h"


// Prototype object APIs (expert and non-expert).
// begin bli_oapi_ex.h
#line 1 "../blis-2.0/frame/include//bli_oapi_ex.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

// This file defines macros used to allow the _oapi.c files to produce
// object APIs that contain context parameters.

// Define a macro that allows the source code to determine which interface
// (basic or expert) we are compiling.
#undef  BLIS_OAPI_EXPERT
#define BLIS_OAPI_EXPERT

// Define the macro to add a suffix to the function names (in function
// definitions).
#undef  EX_SUF
#define EX_SUF BLIS_OAPI_EX_SUF

// Define the macro to add expert arguments to function signatures
// and prototypes.
#undef  BLIS_OAPI_EX_PARAMS
#define BLIS_OAPI_EX_PARAMS   , const cntx_t* cntx, const rntm_t* rntm

// Define the macro to omit the expert variable declaration block, since
// it is not needed when expert parameters are passed in through the API.
#undef  BLIS_OAPI_EX_DECLS
#define BLIS_OAPI_EX_DECLS

// end bli_oapi_ex.h
#line 42 "../blis-2.0/frame/1f//bli_l1f.h"

// begin bli_l1f_oapi.h
#line 1 "../blis-2.0/frame/1f//bli_l1f_oapi.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/


//
// Prototype object-based interfaces.
//

#undef  GENTPROT
#define GENTPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname,EX_SUF) \
     ( \
       const obj_t* alphax, \
       const obj_t* alphay, \
       const obj_t* x, \
       const obj_t* y, \
       const obj_t* z  \
       BLIS_OAPI_EX_PARAMS  \
     );

GENTPROT( axpy2v )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname,EX_SUF) \
     ( \
       const obj_t* alpha, \
       const obj_t* a, \
       const obj_t* x, \
       const obj_t* y  \
       BLIS_OAPI_EX_PARAMS  \
     );

GENTPROT( axpyf )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname,EX_SUF) \
     ( \
       const obj_t* alpha, \
       const obj_t* xt, \
       const obj_t* x, \
       const obj_t* y, \
       const obj_t* rho, \
       const obj_t* z  \
       BLIS_OAPI_EX_PARAMS  \
     );

GENTPROT( dotaxpyv )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname,EX_SUF) \
     ( \
       const obj_t* alpha, \
       const obj_t* at, \
       const obj_t* a, \
       const obj_t* w, \
       const obj_t* x, \
       const obj_t* beta, \
       const obj_t* y, \
       const obj_t* z  \
       BLIS_OAPI_EX_PARAMS  \
     );

GENTPROT( dotxaxpyf )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname,EX_SUF) \
     ( \
       const obj_t* alpha, \
       const obj_t* a, \
       const obj_t* x, \
       const obj_t* beta, \
       const obj_t* y  \
       BLIS_OAPI_EX_PARAMS  \
     );

GENTPROT( dotxf )

// end bli_l1f_oapi.h
#line 43 "../blis-2.0/frame/1f//bli_l1f.h"

// begin bli_xapi_undef.h
#line 1 "../blis-2.0/frame/include//bli_xapi_undef.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

// This file un-defines macros used to allow the _oapi.c and _tapi.c files to
// produce object and typed APIs that omit or contain expert parameters.

// Un-define all macros that allow the source code to determine which interface
// (basic or expert) we are compiling.
#undef  BLIS_OAPI_BASIC
#undef  BLIS_OAPI_EXPERT
#undef  BLIS_TAPI_BASIC
#undef  BLIS_TAPI_EXPERT

// Un-define the macro to omit or add the function name suffix (in function
// definitions).
#undef  EX_SUF

// Un-define the macro to omit or add expert arguments from function signatures
// and prototypes.
#undef  BLIS_OAPI_EX_PARAMS
#undef  BLIS_TAPI_EX_PARAMS

// Un-define the macro to omit or add local expert variables.
#undef  BLIS_OAPI_EX_DECLS
#undef  BLIS_TAPI_EX_DECLS

// end bli_xapi_undef.h
#line 44 "../blis-2.0/frame/1f//bli_l1f.h"


// begin bli_oapi_ba.h
#line 1 "../blis-2.0/frame/include//bli_oapi_ba.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

// This file defines macros used to allow the _oapi.c files to produce
// object APIs that omit expert parameters.

// Define a macro that allows the source code to determine which interface
// (basic or expert) we are compiling.
#undef  BLIS_OAPI_BASIC
#define BLIS_OAPI_BASIC

// Define the macro to omit a suffix from the function names (in function
// definitions).
#undef  EX_SUF
#define EX_SUF

// Define the macro to omit expert arguments from function signatures
// and prototypes.
#undef  BLIS_OAPI_EX_PARAMS
#define BLIS_OAPI_EX_PARAMS

// Define the macro to add local expert variables that are initialized
// to NULL. The "( void )" statements are to prevent unused variable
// warnings by the compiler.
#undef  BLIS_OAPI_EX_DECLS
#define BLIS_OAPI_EX_DECLS   const cntx_t* cntx = NULL; ( void )cntx; \
                                   rntm_t* rntm = NULL; ( void )rntm;

// end bli_oapi_ba.h
#line 46 "../blis-2.0/frame/1f//bli_l1f.h"

// begin bli_l1f_oapi.h
#line 1 "../blis-2.0/frame/1f//bli_l1f_oapi.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/


//
// Prototype object-based interfaces.
//

#undef  GENTPROT
#define GENTPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname,EX_SUF) \
     ( \
       const obj_t* alphax, \
       const obj_t* alphay, \
       const obj_t* x, \
       const obj_t* y, \
       const obj_t* z  \
       BLIS_OAPI_EX_PARAMS  \
     );

GENTPROT( axpy2v )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname,EX_SUF) \
     ( \
       const obj_t* alpha, \
       const obj_t* a, \
       const obj_t* x, \
       const obj_t* y  \
       BLIS_OAPI_EX_PARAMS  \
     );

GENTPROT( axpyf )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname,EX_SUF) \
     ( \
       const obj_t* alpha, \
       const obj_t* xt, \
       const obj_t* x, \
       const obj_t* y, \
       const obj_t* rho, \
       const obj_t* z  \
       BLIS_OAPI_EX_PARAMS  \
     );

GENTPROT( dotaxpyv )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname,EX_SUF) \
     ( \
       const obj_t* alpha, \
       const obj_t* at, \
       const obj_t* a, \
       const obj_t* w, \
       const obj_t* x, \
       const obj_t* beta, \
       const obj_t* y, \
       const obj_t* z  \
       BLIS_OAPI_EX_PARAMS  \
     );

GENTPROT( dotxaxpyf )


#undef  GENTPROT
#define GENTPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname,EX_SUF) \
     ( \
       const obj_t* alpha, \
       const obj_t* a, \
       const obj_t* x, \
       const obj_t* beta, \
       const obj_t* y  \
       BLIS_OAPI_EX_PARAMS  \
     );

GENTPROT( dotxf )

// end bli_l1f_oapi.h
#line 47 "../blis-2.0/frame/1f//bli_l1f.h"

// begin bli_xapi_undef.h
#line 1 "../blis-2.0/frame/include//bli_xapi_undef.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

// This file un-defines macros used to allow the _oapi.c and _tapi.c files to
// produce object and typed APIs that omit or contain expert parameters.

// Un-define all macros that allow the source code to determine which interface
// (basic or expert) we are compiling.
#undef  BLIS_OAPI_BASIC
#undef  BLIS_OAPI_EXPERT
#undef  BLIS_TAPI_BASIC
#undef  BLIS_TAPI_EXPERT

// Un-define the macro to omit or add the function name suffix (in function
// definitions).
#undef  EX_SUF

// Un-define the macro to omit or add expert arguments from function signatures
// and prototypes.
#undef  BLIS_OAPI_EX_PARAMS
#undef  BLIS_TAPI_EX_PARAMS

// Un-define the macro to omit or add local expert variables.
#undef  BLIS_OAPI_EX_DECLS
#undef  BLIS_TAPI_EX_DECLS

// end bli_xapi_undef.h
#line 48 "../blis-2.0/frame/1f//bli_l1f.h"


// Prototype typed APIs (expert and non-expert).
// begin bli_tapi_ex.h
#line 1 "../blis-2.0/frame/include//bli_tapi_ex.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

// This file defines macros used to allow the _tapi.c files to produce
// typed APIs that contain context parameters.

// Define a macro that allows the source code to determine which interface
// (basic or expert) we are compiling.
#undef  BLIS_TAPI_EXPERT
#define BLIS_TAPI_EXPERT

// Define the macro to add a suffix to the function names (in function
// definitions).
#undef  EX_SUF
#define EX_SUF BLIS_TAPI_EX_SUF

// Define the macro to add expert arguments to function signatures
// and prototypes.
#undef  BLIS_TAPI_EX_PARAMS
#define BLIS_TAPI_EX_PARAMS   , const cntx_t* cntx, const rntm_t* rntm

// Define the macro to omit the expert variable declaration block, since
// it is not needed when expert parameters are passed in through the API.
#undef  BLIS_TAPI_EX_DECLS
#define BLIS_TAPI_EX_DECLS

// end bli_tapi_ex.h
#line 51 "../blis-2.0/frame/1f//bli_l1f.h"

// begin bli_l1f_tapi.h
#line 1 "../blis-2.0/frame/1f//bli_l1f_tapi.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/


//
// Prototype BLAS-like interfaces with typed operands.
//

#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname,EX_SUF) \
     ( \
             conj_t conjx, \
             conj_t conjy, \
             dim_t  n, \
       const ctype* alphax, \
       const ctype* alphay, \
       const ctype* x, inc_t incx, \
       const ctype* y, inc_t incy, \
             ctype* z, inc_t incz  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTPROT_BASIC( axpy2v )


#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname,EX_SUF) \
     ( \
             conj_t conja, \
             conj_t conjx, \
             dim_t  m, \
             dim_t  b_n, \
       const ctype* alpha, \
       const ctype* a, inc_t inca, inc_t lda, \
       const ctype* x, inc_t incx, \
             ctype* y, inc_t incy  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTPROT_BASIC( axpyf )


#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname,EX_SUF) \
     ( \
             conj_t conjxt, \
             conj_t conjx, \
             conj_t conjy, \
             dim_t  n, \
       const ctype* alpha, \
       const ctype* x, inc_t incx, \
       const ctype* y, inc_t incy, \
             ctype* rho, \
             ctype* z, inc_t incz  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTPROT_BASIC( dotaxpyv )


#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname,EX_SUF) \
     ( \
             conj_t conjat, \
             conj_t conja, \
             conj_t conjw, \
             conj_t conjx, \
             dim_t  m, \
             dim_t  b_n, \
       const ctype* alpha, \
       const ctype* a, inc_t inca, inc_t lda, \
       const ctype* w, inc_t incw, \
       const ctype* x, inc_t incx, \
       const ctype* beta, \
             ctype* y, inc_t incy, \
             ctype* z, inc_t incz  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTPROT_BASIC( dotxaxpyf )


#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname,EX_SUF) \
     ( \
             conj_t conjat, \
             conj_t conjx, \
             dim_t  m, \
             dim_t  b_n, \
       const ctype* alpha, \
       const ctype* a, inc_t inca, inc_t lda, \
       const ctype* x, inc_t incx, \
       const ctype* beta, \
             ctype* y, inc_t incy  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTPROT_BASIC( dotxf )

// end bli_l1f_tapi.h
#line 52 "../blis-2.0/frame/1f//bli_l1f.h"

// begin bli_l1f_ft.h
#line 1 "../blis-2.0/frame/1f//bli_l1f_ft.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/


//
// -- Level-1f function types --------------------------------------------------
//

// axpy2v

#undef  GENTDEF
#define GENTDEF( ctype, ch, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,EX_SUF,tsuf)) \
     ( \
             conj_t conjx, \
             conj_t conjy, \
             dim_t  n, \
       const ctype* alpha1, \
       const ctype* alpha2, \
       const ctype* x, inc_t incx, \
       const ctype* y, inc_t incy, \
             ctype* z, inc_t incz  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTDEF( axpy2v )

// axpyf

#undef  GENTDEF
#define GENTDEF( ctype, ch, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,EX_SUF,tsuf)) \
     ( \
             conj_t conja, \
             conj_t conjx, \
             dim_t  m, \
             dim_t  b_n, \
       const ctype* alpha, \
       const ctype* a, inc_t inca, inc_t lda, \
       const ctype* x, inc_t incx, \
             ctype* y, inc_t incy  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTDEF( axpyf )

// dotaxpyv

#undef  GENTDEF
#define GENTDEF( ctype, ch, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,EX_SUF,tsuf)) \
     ( \
             conj_t conjxt, \
             conj_t conjx, \
             conj_t conjy, \
             dim_t  m, \
       const ctype* alpha, \
       const ctype* x, inc_t incx, \
       const ctype* y, inc_t incy, \
             ctype* rho, \
             ctype* z, inc_t incz  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTDEF( dotaxpyv )

// dotxf

#undef  GENTDEF
#define GENTDEF( ctype, ch, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,EX_SUF,tsuf)) \
     ( \
             conj_t conjat, \
             conj_t conjx, \
             dim_t  m, \
             dim_t  b_n, \
       const ctype* alpha, \
       const ctype* a, inc_t inca, inc_t lda, \
       const ctype* x, inc_t incx, \
       const ctype* beta, \
             ctype* y, inc_t incy  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTDEF( dotxf )

// dotxaxpyf

#undef  GENTDEF
#define GENTDEF( ctype, ch, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,EX_SUF,tsuf)) \
     ( \
             conj_t conjat, \
             conj_t conja, \
             conj_t conjw, \
             conj_t conjx, \
             dim_t  m, \
             dim_t  b_n, \
       const ctype* alpha, \
       const ctype* a, inc_t inca, inc_t lda, \
       const ctype* w, inc_t incw, \
       const ctype* x, inc_t incx, \
       const ctype* beta, \
             ctype* y, inc_t incy, \
             ctype* z, inc_t incz  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTDEF( dotxaxpyf )


// end bli_l1f_ft.h
#line 53 "../blis-2.0/frame/1f//bli_l1f.h"

// begin bli_xapi_undef.h
#line 1 "../blis-2.0/frame/include//bli_xapi_undef.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

// This file un-defines macros used to allow the _oapi.c and _tapi.c files to
// produce object and typed APIs that omit or contain expert parameters.

// Un-define all macros that allow the source code to determine which interface
// (basic or expert) we are compiling.
#undef  BLIS_OAPI_BASIC
#undef  BLIS_OAPI_EXPERT
#undef  BLIS_TAPI_BASIC
#undef  BLIS_TAPI_EXPERT

// Un-define the macro to omit or add the function name suffix (in function
// definitions).
#undef  EX_SUF

// Un-define the macro to omit or add expert arguments from function signatures
// and prototypes.
#undef  BLIS_OAPI_EX_PARAMS
#undef  BLIS_TAPI_EX_PARAMS

// Un-define the macro to omit or add local expert variables.
#undef  BLIS_OAPI_EX_DECLS
#undef  BLIS_TAPI_EX_DECLS

// end bli_xapi_undef.h
#line 54 "../blis-2.0/frame/1f//bli_l1f.h"


// begin bli_tapi_ba.h
#line 1 "../blis-2.0/frame/include//bli_tapi_ba.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

// This file defines macros used to allow the _tapi.c files to produce
// typed APIs that omit expert parameters.

// Define a macro that allows the source code to determine which interface
// (basic or expert) we are compiling.
#undef  BLIS_TAPI_BASIC
#define BLIS_TAPI_BASIC

// Define the macro to omit a suffix from the function names (in function
// definitions).
#undef  EX_SUF
#define EX_SUF

// Define the macro to omit expert arguments from function signatures
// and prototypes.
#undef  BLIS_TAPI_EX_PARAMS
#define BLIS_TAPI_EX_PARAMS

// Define the macro to add local expert variables that are initialized
// to NULL. The "( void )" statements are to prevent unused variable
// warnings by the compiler.
#undef  BLIS_TAPI_EX_DECLS
#define BLIS_TAPI_EX_DECLS   const cntx_t* cntx = NULL; ( void )cntx; \
                                   rntm_t* rntm = NULL; ( void )rntm;

// end bli_tapi_ba.h
#line 56 "../blis-2.0/frame/1f//bli_l1f.h"

// begin bli_l1f_tapi.h
#line 1 "../blis-2.0/frame/1f//bli_l1f_tapi.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/


//
// Prototype BLAS-like interfaces with typed operands.
//

#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname,EX_SUF) \
     ( \
             conj_t conjx, \
             conj_t conjy, \
             dim_t  n, \
       const ctype* alphax, \
       const ctype* alphay, \
       const ctype* x, inc_t incx, \
       const ctype* y, inc_t incy, \
             ctype* z, inc_t incz  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTPROT_BASIC( axpy2v )


#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname,EX_SUF) \
     ( \
             conj_t conja, \
             conj_t conjx, \
             dim_t  m, \
             dim_t  b_n, \
       const ctype* alpha, \
       const ctype* a, inc_t inca, inc_t lda, \
       const ctype* x, inc_t incx, \
             ctype* y, inc_t incy  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTPROT_BASIC( axpyf )


#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname,EX_SUF) \
     ( \
             conj_t conjxt, \
             conj_t conjx, \
             conj_t conjy, \
             dim_t  n, \
       const ctype* alpha, \
       const ctype* x, inc_t incx, \
       const ctype* y, inc_t incy, \
             ctype* rho, \
             ctype* z, inc_t incz  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTPROT_BASIC( dotaxpyv )


#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname,EX_SUF) \
     ( \
             conj_t conjat, \
             conj_t conja, \
             conj_t conjw, \
             conj_t conjx, \
             dim_t  m, \
             dim_t  b_n, \
       const ctype* alpha, \
       const ctype* a, inc_t inca, inc_t lda, \
       const ctype* w, inc_t incw, \
       const ctype* x, inc_t incx, \
       const ctype* beta, \
             ctype* y, inc_t incy, \
             ctype* z, inc_t incz  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTPROT_BASIC( dotxaxpyf )


#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname,EX_SUF) \
     ( \
             conj_t conjat, \
             conj_t conjx, \
             dim_t  m, \
             dim_t  b_n, \
       const ctype* alpha, \
       const ctype* a, inc_t inca, inc_t lda, \
       const ctype* x, inc_t incx, \
       const ctype* beta, \
             ctype* y, inc_t incy  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTPROT_BASIC( dotxf )

// end bli_l1f_tapi.h
#line 57 "../blis-2.0/frame/1f//bli_l1f.h"

// begin bli_l1f_ft.h
#line 1 "../blis-2.0/frame/1f//bli_l1f_ft.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/


//
// -- Level-1f function types --------------------------------------------------
//

// axpy2v

#undef  GENTDEF
#define GENTDEF( ctype, ch, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,EX_SUF,tsuf)) \
     ( \
             conj_t conjx, \
             conj_t conjy, \
             dim_t  n, \
       const ctype* alpha1, \
       const ctype* alpha2, \
       const ctype* x, inc_t incx, \
       const ctype* y, inc_t incy, \
             ctype* z, inc_t incz  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTDEF( axpy2v )

// axpyf

#undef  GENTDEF
#define GENTDEF( ctype, ch, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,EX_SUF,tsuf)) \
     ( \
             conj_t conja, \
             conj_t conjx, \
             dim_t  m, \
             dim_t  b_n, \
       const ctype* alpha, \
       const ctype* a, inc_t inca, inc_t lda, \
       const ctype* x, inc_t incx, \
             ctype* y, inc_t incy  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTDEF( axpyf )

// dotaxpyv

#undef  GENTDEF
#define GENTDEF( ctype, ch, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,EX_SUF,tsuf)) \
     ( \
             conj_t conjxt, \
             conj_t conjx, \
             conj_t conjy, \
             dim_t  m, \
       const ctype* alpha, \
       const ctype* x, inc_t incx, \
       const ctype* y, inc_t incy, \
             ctype* rho, \
             ctype* z, inc_t incz  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTDEF( dotaxpyv )

// dotxf

#undef  GENTDEF
#define GENTDEF( ctype, ch, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,EX_SUF,tsuf)) \
     ( \
             conj_t conjat, \
             conj_t conjx, \
             dim_t  m, \
             dim_t  b_n, \
       const ctype* alpha, \
       const ctype* a, inc_t inca, inc_t lda, \
       const ctype* x, inc_t incx, \
       const ctype* beta, \
             ctype* y, inc_t incy  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTDEF( dotxf )

// dotxaxpyf

#undef  GENTDEF
#define GENTDEF( ctype, ch, opname, tsuf ) \
\
typedef void (*PASTECH(ch,opname,EX_SUF,tsuf)) \
     ( \
             conj_t conjat, \
             conj_t conja, \
             conj_t conjw, \
             conj_t conjx, \
             dim_t  m, \
             dim_t  b_n, \
       const ctype* alpha, \
       const ctype* a, inc_t inca, inc_t lda, \
       const ctype* w, inc_t incw, \
       const ctype* x, inc_t incx, \
       const ctype* beta, \
             ctype* y, inc_t incy, \
             ctype* z, inc_t incz  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTDEF( dotxaxpyf )


// end bli_l1f_ft.h
#line 58 "../blis-2.0/frame/1f//bli_l1f.h"

// begin bli_xapi_undef.h
#line 1 "../blis-2.0/frame/include//bli_xapi_undef.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

// This file un-defines macros used to allow the _oapi.c and _tapi.c files to
// produce object and typed APIs that omit or contain expert parameters.

// Un-define all macros that allow the source code to determine which interface
// (basic or expert) we are compiling.
#undef  BLIS_OAPI_BASIC
#undef  BLIS_OAPI_EXPERT
#undef  BLIS_TAPI_BASIC
#undef  BLIS_TAPI_EXPERT

// Un-define the macro to omit or add the function name suffix (in function
// definitions).
#undef  EX_SUF

// Un-define the macro to omit or add expert arguments from function signatures
// and prototypes.
#undef  BLIS_OAPI_EX_PARAMS
#undef  BLIS_TAPI_EX_PARAMS

// Un-define the macro to omit or add local expert variables.
#undef  BLIS_OAPI_EX_DECLS
#undef  BLIS_TAPI_EX_DECLS

// end bli_xapi_undef.h
#line 59 "../blis-2.0/frame/1f//bli_l1f.h"


// Generate function pointer arrays for tapi functions (expert only).
// begin bli_l1f_fpa.h
#line 1 "../blis-2.0/frame/1f//bli_l1f_fpa.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

//
// Prototype function pointer query interface.
//

#undef  GENPROT
#define GENPROT( opname ) \
\
PASTECH(opname,BLIS_TAPI_EX_SUF,_vft) \
PASTEMAC(opname,BLIS_TAPI_EX_SUF,_qfp)( num_t dt );

GENPROT( axpy2v )
GENPROT( axpyf )
GENPROT( dotaxpyv )
GENPROT( dotxaxpyf )
GENPROT( dotxf )

// end bli_l1f_fpa.h
#line 62 "../blis-2.0/frame/1f//bli_l1f.h"


// end bli_l1f.h
#line 189 "../blis-2.0/frame/include/blis.h"



// -- Level-1m operations --

// begin bli_l1m.h
#line 1 "../blis-2.0/frame/1m//bli_l1m.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

// begin bli_l1m_check.h
#line 1 "../blis-2.0/frame/1m//bli_l1m_check.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/


//
// Prototype object-based check functions.
//

#undef  GENPROT
#define GENPROT( opname ) \
\
void PASTEMAC(opname,_check) \
     ( \
       const obj_t* x, \
       const obj_t* y  \
    );

GENPROT( addm )
GENPROT( copym )
GENPROT( subm )


#undef  GENPROT
#define GENPROT( opname ) \
\
void PASTEMAC(opname,_check) \
     ( \
       const obj_t* alpha, \
       const obj_t* x, \
       const obj_t* y  \
    );

GENPROT( axpym )
GENPROT( scal2m )


#undef  GENPROT
#define GENPROT( opname ) \
\
void PASTEMAC(opname,_check) \
     ( \
       const obj_t* alpha, \
       const obj_t* x  \
    );

GENPROT( invscalm )
GENPROT( scalm )
GENPROT( setm )


#undef  GENPROT
#define GENPROT( opname ) \
\
void PASTEMAC(opname,_check) \
     ( \
       const obj_t* x, \
       const obj_t* beta, \
       const obj_t* y  \
    );

GENPROT( xpbym )


// -----------------------------------------------------------------------------

void bli_l1m_xy_check
     (
       const obj_t*  x,
       const obj_t*  y
     );

void bli_l1m_axy_check
     (
       const obj_t*  alpha,
       const obj_t*  x,
       const obj_t*  y
     );

void bli_l1m_ax_check
     (
       const obj_t*  alpha,
       const obj_t*  x
     );

// end bli_l1m_check.h
#line 36 "../blis-2.0/frame/1m//bli_l1m.h"


// Define kernel function types.
// begin bli_l1m_ker_ft.h
#line 1 "../blis-2.0/frame/1m//bli_l1m_ker_ft.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_L1M_KER_FT_H
#define BLIS_L1M_KER_FT_H


//
// -- Level-1m kernel function types -------------------------------------------
//

#undef  GENTDEF
#define GENTDEF( opname ) \
\
typedef void (*PASTECH(opname,_ker_ft)) \
     ( \
       PASTECH(opname,_params), \
       BLIS_CNTX_PARAM  \
     );

GENTDEF( packm )
GENTDEF( packm_cxk )
GENTDEF( unpackm_cxk )
GENTDEF( packm_cxc_diag )


#endif

// end bli_l1m_ker_ft.h
#line 39 "../blis-2.0/frame/1m//bli_l1m.h"


// Define object function types for variants.
// begin bli_l1m_oft_var.h
#line 1 "../blis-2.0/frame/1m//bli_l1m_oft_var.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_L1M_OFT_VAR_H
#define BLIS_L1M_OFT_VAR_H


//
// -- Level-3 variant function types -------------------------------------------
//

#undef  GENTDEF
#define GENTDEF( opname ) \
\
typedef void (*PASTECH(opname,_var_oft)) \
( \
  const obj_t*  a, \
        obj_t*  p, \
  const cntx_t* cntx, \
  const cntl_t* cntl, \
        thrinfo_t* thread  \
);

GENTDEF( packm )


#undef  GENTDEF
#define GENTDEF( opname ) \
\
typedef void (*PASTECH(opname,_var_oft)) \
( \
  const obj_t*  p, \
  const obj_t*  a, \
  const cntx_t* cntx, \
  const cntl_t* cntl, \
  const thrinfo_t* thread  \
);

GENTDEF( unpackm )



#endif

// end bli_l1m_oft_var.h
#line 42 "../blis-2.0/frame/1m//bli_l1m.h"


// Prototype object APIs (expert and non-expert).
// begin bli_oapi_ex.h
#line 1 "../blis-2.0/frame/include//bli_oapi_ex.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

// This file defines macros used to allow the _oapi.c files to produce
// object APIs that contain context parameters.

// Define a macro that allows the source code to determine which interface
// (basic or expert) we are compiling.
#undef  BLIS_OAPI_EXPERT
#define BLIS_OAPI_EXPERT

// Define the macro to add a suffix to the function names (in function
// definitions).
#undef  EX_SUF
#define EX_SUF BLIS_OAPI_EX_SUF

// Define the macro to add expert arguments to function signatures
// and prototypes.
#undef  BLIS_OAPI_EX_PARAMS
#define BLIS_OAPI_EX_PARAMS   , const cntx_t* cntx, const rntm_t* rntm

// Define the macro to omit the expert variable declaration block, since
// it is not needed when expert parameters are passed in through the API.
#undef  BLIS_OAPI_EX_DECLS
#define BLIS_OAPI_EX_DECLS

// end bli_oapi_ex.h
#line 45 "../blis-2.0/frame/1m//bli_l1m.h"

// begin bli_l1m_oapi.h
#line 1 "../blis-2.0/frame/1m//bli_l1m_oapi.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/


//
// Prototype object-based interfaces.
//

#undef  GENPROT
#define GENPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname,EX_SUF) \
     ( \
       const obj_t* x, \
       const obj_t* y  \
       BLIS_OAPI_EX_PARAMS  \
     );

GENPROT( addm )
GENPROT( copym )
GENPROT( subm )


#undef  GENPROT
#define GENPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname,EX_SUF) \
     ( \
       const obj_t* alpha, \
       const obj_t* x, \
       const obj_t* y  \
       BLIS_OAPI_EX_PARAMS  \
     );

GENPROT( axpym )
GENPROT( scal2m )


#undef  GENPROT
#define GENPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname,EX_SUF) \
     ( \
       const obj_t* alpha, \
       const obj_t* x  \
       BLIS_OAPI_EX_PARAMS  \
     );

GENPROT( invscalm )
GENPROT( scalm )
GENPROT( setm )


#undef  GENPROT
#define GENPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname,EX_SUF) \
     ( \
       const obj_t* x, \
       const obj_t* beta, \
       const obj_t* y  \
       BLIS_OAPI_EX_PARAMS  \
     );

GENPROT( xpbym )
GENPROT( xpbym_md )

// end bli_l1m_oapi.h
#line 46 "../blis-2.0/frame/1m//bli_l1m.h"

// begin bli_xapi_undef.h
#line 1 "../blis-2.0/frame/include//bli_xapi_undef.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

// This file un-defines macros used to allow the _oapi.c and _tapi.c files to
// produce object and typed APIs that omit or contain expert parameters.

// Un-define all macros that allow the source code to determine which interface
// (basic or expert) we are compiling.
#undef  BLIS_OAPI_BASIC
#undef  BLIS_OAPI_EXPERT
#undef  BLIS_TAPI_BASIC
#undef  BLIS_TAPI_EXPERT

// Un-define the macro to omit or add the function name suffix (in function
// definitions).
#undef  EX_SUF

// Un-define the macro to omit or add expert arguments from function signatures
// and prototypes.
#undef  BLIS_OAPI_EX_PARAMS
#undef  BLIS_TAPI_EX_PARAMS

// Un-define the macro to omit or add local expert variables.
#undef  BLIS_OAPI_EX_DECLS
#undef  BLIS_TAPI_EX_DECLS

// end bli_xapi_undef.h
#line 47 "../blis-2.0/frame/1m//bli_l1m.h"


// begin bli_oapi_ba.h
#line 1 "../blis-2.0/frame/include//bli_oapi_ba.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

// This file defines macros used to allow the _oapi.c files to produce
// object APIs that omit expert parameters.

// Define a macro that allows the source code to determine which interface
// (basic or expert) we are compiling.
#undef  BLIS_OAPI_BASIC
#define BLIS_OAPI_BASIC

// Define the macro to omit a suffix from the function names (in function
// definitions).
#undef  EX_SUF
#define EX_SUF

// Define the macro to omit expert arguments from function signatures
// and prototypes.
#undef  BLIS_OAPI_EX_PARAMS
#define BLIS_OAPI_EX_PARAMS

// Define the macro to add local expert variables that are initialized
// to NULL. The "( void )" statements are to prevent unused variable
// warnings by the compiler.
#undef  BLIS_OAPI_EX_DECLS
#define BLIS_OAPI_EX_DECLS   const cntx_t* cntx = NULL; ( void )cntx; \
                                   rntm_t* rntm = NULL; ( void )rntm;

// end bli_oapi_ba.h
#line 49 "../blis-2.0/frame/1m//bli_l1m.h"

// begin bli_l1m_oapi.h
#line 1 "../blis-2.0/frame/1m//bli_l1m_oapi.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/


//
// Prototype object-based interfaces.
//

#undef  GENPROT
#define GENPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname,EX_SUF) \
     ( \
       const obj_t* x, \
       const obj_t* y  \
       BLIS_OAPI_EX_PARAMS  \
     );

GENPROT( addm )
GENPROT( copym )
GENPROT( subm )


#undef  GENPROT
#define GENPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname,EX_SUF) \
     ( \
       const obj_t* alpha, \
       const obj_t* x, \
       const obj_t* y  \
       BLIS_OAPI_EX_PARAMS  \
     );

GENPROT( axpym )
GENPROT( scal2m )


#undef  GENPROT
#define GENPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname,EX_SUF) \
     ( \
       const obj_t* alpha, \
       const obj_t* x  \
       BLIS_OAPI_EX_PARAMS  \
     );

GENPROT( invscalm )
GENPROT( scalm )
GENPROT( setm )


#undef  GENPROT
#define GENPROT( opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(opname,EX_SUF) \
     ( \
       const obj_t* x, \
       const obj_t* beta, \
       const obj_t* y  \
       BLIS_OAPI_EX_PARAMS  \
     );

GENPROT( xpbym )
GENPROT( xpbym_md )

// end bli_l1m_oapi.h
#line 50 "../blis-2.0/frame/1m//bli_l1m.h"

// begin bli_xapi_undef.h
#line 1 "../blis-2.0/frame/include//bli_xapi_undef.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

// This file un-defines macros used to allow the _oapi.c and _tapi.c files to
// produce object and typed APIs that omit or contain expert parameters.

// Un-define all macros that allow the source code to determine which interface
// (basic or expert) we are compiling.
#undef  BLIS_OAPI_BASIC
#undef  BLIS_OAPI_EXPERT
#undef  BLIS_TAPI_BASIC
#undef  BLIS_TAPI_EXPERT

// Un-define the macro to omit or add the function name suffix (in function
// definitions).
#undef  EX_SUF

// Un-define the macro to omit or add expert arguments from function signatures
// and prototypes.
#undef  BLIS_OAPI_EX_PARAMS
#undef  BLIS_TAPI_EX_PARAMS

// Un-define the macro to omit or add local expert variables.
#undef  BLIS_OAPI_EX_DECLS
#undef  BLIS_TAPI_EX_DECLS

// end bli_xapi_undef.h
#line 51 "../blis-2.0/frame/1m//bli_l1m.h"


// Prototype typed APIs (expert and non-expert).
// begin bli_tapi_ex.h
#line 1 "../blis-2.0/frame/include//bli_tapi_ex.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

// This file defines macros used to allow the _tapi.c files to produce
// typed APIs that contain context parameters.

// Define a macro that allows the source code to determine which interface
// (basic or expert) we are compiling.
#undef  BLIS_TAPI_EXPERT
#define BLIS_TAPI_EXPERT

// Define the macro to add a suffix to the function names (in function
// definitions).
#undef  EX_SUF
#define EX_SUF BLIS_TAPI_EX_SUF

// Define the macro to add expert arguments to function signatures
// and prototypes.
#undef  BLIS_TAPI_EX_PARAMS
#define BLIS_TAPI_EX_PARAMS   , const cntx_t* cntx, const rntm_t* rntm

// Define the macro to omit the expert variable declaration block, since
// it is not needed when expert parameters are passed in through the API.
#undef  BLIS_TAPI_EX_DECLS
#define BLIS_TAPI_EX_DECLS

// end bli_tapi_ex.h
#line 54 "../blis-2.0/frame/1m//bli_l1m.h"

// begin bli_l1m_tapi.h
#line 1 "../blis-2.0/frame/1m//bli_l1m_tapi.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/


//
// Prototype BLAS-like interfaces with typed operands.
//

#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname,EX_SUF) \
     ( \
             doff_t  diagoffx, \
             diag_t  diagx, \
             uplo_t  uplox, \
             trans_t transx, \
             dim_t   m, \
             dim_t   n, \
       const ctype*  x, inc_t rs_x, inc_t cs_x, \
             ctype*  y, inc_t rs_y, inc_t cs_y  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTPROT_BASIC( addm )
INSERT_GENTPROT_BASIC( copym )
INSERT_GENTPROT_BASIC( subm )


#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname,EX_SUF) \
     ( \
             doff_t  diagoffx, \
             diag_t  diagx, \
             uplo_t  uplox, \
             trans_t transx, \
             dim_t   m, \
             dim_t   n, \
       const ctype*  alpha, \
       const ctype*  x, inc_t rs_x, inc_t cs_x, \
             ctype*  y, inc_t rs_y, inc_t cs_y  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTPROT_BASIC( axpym )
INSERT_GENTPROT_BASIC( scal2m )


#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname,EX_SUF) \
     ( \
             conj_t conjalpha, \
             doff_t diagoffx, \
             diag_t diagx, \
             uplo_t uplox, \
             dim_t  m, \
             dim_t  n, \
       const ctype* alpha, \
             ctype* x, inc_t rs_x, inc_t cs_x  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTPROT_BASIC( invscalm )
INSERT_GENTPROT_BASIC( scalm )
INSERT_GENTPROT_BASIC( setm )


#undef  GENTPROT
#define GENTPROT( ctype, ch, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(ch,opname,EX_SUF) \
     ( \
             doff_t  diagoffx, \
             diag_t  diagx, \
             uplo_t  uplox, \
             trans_t transx, \
             dim_t   m, \
             dim_t   n, \
       const ctype*  x, inc_t rs_x, inc_t cs_x, \
       const ctype*  beta, \
             ctype*  y, inc_t rs_y, inc_t cs_y  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTPROT_BASIC( xpbym )


#undef  GENTPROT2
#define GENTPROT2( ctype_x, ctype_y, chx, chy, opname ) \
\
BLIS_EXPORT_BLIS void PASTEMAC(chx,chy,opname,EX_SUF) \
     ( \
             doff_t   diagoffx, \
             diag_t   diagx, \
             uplo_t   uplox, \
             trans_t  transx, \
             dim_t    m, \
             dim_t    n, \
       const ctype_x* x, inc_t rs_x, inc_t cs_x, \
       const ctype_y* beta, \
             ctype_y* y, inc_t rs_y, inc_t cs_y  \
       BLIS_TAPI_EX_PARAMS  \
     );

INSERT_GENTPROT2_BASIC( xpbym_md )
INSERT_GENTPROT2_MIX_DP( xpbym_md )

// end bli_l