%feature("docstring") OT::MultiStart
"Multi-start optimization algorithm.

The algorithm runs an optimization solver for multiple starting points and returns the
best result of each local search.
The algorithm succeeds when at least one local search succeeds.

Parameters
----------
solver : :class:`~openturns.OptimizationAlgorithm`
    The internal solver
startingSample : 2-d sequence of float
    Starting points set

Notes
-----
The starting point of the internal solver is ignored.
If you want to use it, add it to *startingSample*.

Stopping criteria used are the ones set in the internal solver.

A global number of evaluations can be set,
in that case all starting points might not be used depending on the number
of evaluations allocated to the internal solver.

Starting points provided through the *startingSample* parameter
should be within the bounds of the
:class:`~openturns.OptimizationProblem`, but this is not enforced.

Examples
--------
First define the :class:`~openturns.OptimizationAlgorithm` to be run from multiple starting points.

>>> import openturns as ot
>>> dim = 2
>>> model = ot.SymbolicFunction(['x', 'y'], ['x^2+y^2*(1-x)^3'])
>>> bounds = ot.Interval([-2.0] * dim, [3.0] * dim)
>>> problem = ot.OptimizationProblem(model)
>>> problem.setBounds(bounds)
>>> solver = ot.TNC(problem)

Starting points must be manually specified.

>>> uniform = ot.JointDistribution([ot.Uniform(-2.0, 3.0)] * dim)
>>> ot.RandomGenerator.SetSeed(0)
>>> startingSample = uniform.getSample(5)
>>> print(startingSample)
    [ X0        X1        ]
0 : [  1.14938   2.84712  ]
1 : [  2.41403   2.6034   ]
2 : [ -1.32362   0.515201 ]
3 : [ -1.83749  -1.68397  ]
4 : [ -0.264715 -0.536216 ]
>>> algo = ot.MultiStart(solver, startingSample)
>>> algo.run()
>>> result = algo.getResult()
>>> print(result.getOptimalPoint())
[3,3]"

// ---------------------------------------------------------------------

%feature("docstring") OT::MultiStart::getOptimizationAlgorithm
"Internal solver accessor.

Returns
-------
solver : :class:`~openturns.OptimizationAlgorithm`
    The internal solver"

// ---------------------------------------------------------------------

%feature("docstring") OT::MultiStart::setOptimizationAlgorithm
"Internal solver accessor.

Parameters
----------
solver : :class:`~openturns.OptimizationAlgorithm`
    The internal solver"

// ---------------------------------------------------------------------

%feature("docstring") OT::MultiStart::setStartingPoint
"Inherited but raises an Exception.

Notes
-----
This method is inherited from :class:`OptimizationAlgorithm` but makes
no sense in a multi-start context."

// ---------------------------------------------------------------------

%feature("docstring") OT::MultiStart::getStartingPoint
"Inherited but raises an Exception.

Notes
-----
This method is inherited from :class:`OptimizationAlgorithm` but makes
no sense in a multi-start context."

// ---------------------------------------------------------------------

%feature("docstring") OT::MultiStart::getStartingSample
"Accessor to the sample of starting points."

// ---------------------------------------------------------------------

%feature("docstring") OT::MultiStart::setStartingSample
"Accessor to the sample of starting points.

Parameters
----------
startingSample : 2-d sequence of float
    A new sample of starting points to overwrite the existing sample"

// ---------------------------------------------------------------------

%feature("docstring") OT::MultiStart::setKeepResults
"Flag to keep intermediate results accessor.

Parameters
----------
keepResults : bool
    If *True* all the intermediate results are stored, otherwise they are ignored.
    Default value is *MultiStart-KeepResults* in :class:`~openturns.ResourceMap`"

// ---------------------------------------------------------------------

%feature("docstring") OT::MultiStart::getKeepResults
"Flag to keep intermediate results accessor.

Returns
-------
keepResults : bool
    If *True* all the intermediate results are stored, otherwise they are ignored.
    Default value is *MultiStart-KeepResults* in :class:`~openturns.ResourceMap`"

// ---------------------------------------------------------------------

%feature("docstring") OT::MultiStart::getResultCollection
"Intermediate optimization results accessor.

Returns
-------
results : :class:`~openturns.OptimizationResultCollection`
    Intermediate optimization results"

// ---------------------------------------------------------------------

%feature("docstring") OT::MultiStart::setProblem
"Sets the optimization problem.

Parameters
----------
problem : :class:`~openturns.OptimizationProblem`
    Optimization problem."
