%feature("docstring") OT::LevelSet
"Level set.

Parameters
----------
function : :class:`~openturns.Function`
    A function such that: :math:`f: \\Rset^{dim} \\mapsto \\Rset` defining the
    LevelSet.
operator : :class:`~openturns.ComparisonOperator`, optional
    Comparison operator against the level.
    The default value is :class:`~openturns.LessOrEqual`.
level : float, optional
    Level :math:`s` defining the LevelSet.
    The default value is 0.0.

Notes
-----
A LevelSet is a :class:`~openturns.Domain` defined as follows:

.. math::

    \\{ \\vect{x} \\in \\Rset^{dim} \\, | \\, f(\\vect{x}) \\leq s \\}

Examples
--------
>>> import openturns as ot
>>> function = ot.SymbolicFunction(['x1', 'x2'], ['x1^4 + x2^4'])
>>> s = 1.0
>>> op = ot.LessOrEqual()
>>> levelSet = ot.LevelSet(function, op, s)"

// ---------------------------------------------------------------------

%feature("docstring") OT::LevelSet::intersect
"Return the intersection with another LevelSet.

Parameters
----------
other : :class:`~openturns.LevelSet`
    A LevelSet defined by :math:`(f_2, s_2)`.

Returns
-------
levelSet : :class:`~openturns.LevelSet`
    The intersection between this LevelSet and the LevelSet *other*,
    i.e. the LevelSet defined as:
    :math:`\\{\\vect{x} \\in \\Rset^{dim} | f(\\vect{x}) \\leq s \\, \\mbox{and} \\, f_2(\\vect{x}) \\leq s_2\\}`.

Examples
--------
>>> import openturns as ot
>>> # First level set
>>> function = ot.SymbolicFunction(['x'], ['3*x-1'])
>>> levelSet1 = ot.LevelSet(function, ot.LessOrEqual(), 0.5)
>>> # Second level set
>>> function = ot.SymbolicFunction(['x'], ['x'])
>>> levelSet2 = ot.LevelSet(function, ot.LessOrEqual(), 0.5)
>>> # Intersection between levelSet1 and levelSet2
>>> intersection = levelSet1.intersect(levelSet2)
>>> # Tests
>>> print([1.0] in intersection)
False
>>> print([0.25] in intersection)
True"

// ---------------------------------------------------------------------

%feature("docstring") OT::LevelSet::join
"Return the union with another LevelSet.

Parameters
----------
other : :class:`~openturns.LevelSet`
    A LevelSet defined by :math:`(f_2, s_2)`.

Returns
-------
levelSet : :class:`~openturns.LevelSet`
    The union between this LevelSet and the LevelSet *other*,
    i.e. the LevelSet defined as:
    :math:`\\{\\vect{x} \\in \\Rset^{dim} | f(\\vect{x}) \\leq s \\, \\mbox{or} \\, f_2(\\vect{x}) \\leq s_2\\}`.

Examples
--------
>>> import openturns as ot
>>> # First level set
>>> function = ot.SymbolicFunction(['x'], ['3*x-1'])
>>> levelSet1 = ot.LevelSet(function, ot.LessOrEqual(), 0.0)
>>> # Second level set
>>> function = ot.SymbolicFunction(['x'], ['x'])
>>> levelSet2 = ot.LevelSet(function, ot.LessOrEqual(), 0.0)
>>> # Union between levelSet1 and levelSet2
>>> union = levelSet1.join(levelSet2)
>>> # Tests
>>> print([0.5] in union)
False
>>> print([0.25] in union)
True"

// ---------------------------------------------------------------------

%feature("docstring") OT::LevelSet::getFunction
"Get the function defining the level set.

Returns
-------
function : :class:`~openturns.Function`
    A function such that: :math:`f: \\Rset^{dim} \\mapsto \\Rset` defining the
    LevelSet.

Examples
--------
>>> import openturns as ot
>>> function = ot.SymbolicFunction(['x'], ['3*x-1'])
>>> levelSet = ot.LevelSet(function, ot.LessOrEqual(), 0.0)
>>> print(levelSet.getFunction().getEvaluation())
[x]->[3*x-1]"

// ---------------------------------------------------------------------

%feature("docstring") OT::LevelSet::setFunction
"Set the function defining the level set.

Parameters
----------
function : :class:`~openturns.Function`
    A function such that: :math:`f: \\Rset^{dim} \\mapsto \\Rset` defining the
    LevelSet.

Examples
--------
>>> import openturns as ot
>>> levelSet = ot.LevelSet()
>>> function = ot.SymbolicFunction(['x'], ['3*x-1'])
>>> levelSet.setFunction(function)"

// ---------------------------------------------------------------------

%feature("docstring") OT::LevelSet::getOperator
"Operator accessor.

Returns
-------
op : :class:`~openturns.ComparisonOperator`
    Comparison operator against the level."

// ---------------------------------------------------------------------

%feature("docstring") OT::LevelSet::setOperator
"Operator accessor.

Parameters
----------
op : :class:`~openturns.ComparisonOperator`
    Comparison operator against the level."

// ---------------------------------------------------------------------

%feature("docstring") OT::LevelSet::getLevel
"Get the level defining the level set.

Returns
-------
level : float
    Level :math:`s` defining the LevelSet.

Examples
--------
>>> import openturns as ot
>>> function = ot.SymbolicFunction(['x'], ['3*x-1'])
>>> levelSet = ot.LevelSet(function, ot.LessOrEqual(), 0.0)
>>> print(levelSet.getLevel())
0.0"

// ---------------------------------------------------------------------

%feature("docstring") OT::LevelSet::setLevel
"Set the level defining the level set.

Parameters
----------
level : float
    Level :math:`s` defining the LevelSet.

Examples
--------
>>> import openturns as ot
>>> levelSet = ot.LevelSet()
>>> levelSet.setLevel(3.0)"

// ---------------------------------------------------------------------

%feature("docstring") OT::LevelSet::setLowerBound
"Set the lower bound of the bounding box.

Parameters
----------
bound : sequence of floats
    Lower bound of the bounding box of the level set. It allows one to clip the level set."

// ---------------------------------------------------------------------

%feature("docstring") OT::LevelSet::setUpperBound
"Set the upper bound of the bounding box.

Parameters
----------
bound : sequence of floats
    Upper bound of the bounding box of the level set. It allows one to clip the level set."

// ---------------------------------------------------------------------

%feature("docstring") OT::LevelSet::getLowerBound
"Get the lower bound of the bounding box.

Returns
-------
bound : :class:`~openturns.Point`
    Lower bound of the bounding box of the level set. It allows one to clip the level set."

// ---------------------------------------------------------------------

%feature("docstring") OT::LevelSet::getUpperBound
"Get the upper bound of the bounding box.

Returns
-------
bound : :class:`~openturns.Point`
    Upper bound of the bounding box of the level set. It allows one to clip the level set."
