/*
 * Copyright (C) 2021-2023 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1.  Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 * 2.  Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#pragma once

#include "ArgumentCoders.h"
#include "Connection.h"
#include "GamepadData.h"
#include "IdentifierTypes.h"
#include "MediaDeviceSandboxExtensions.h"
#include "MessageNames.h"
#include "SandboxExtension.h"
#include "WebPageCreationParameters.h"
#include "WebPageProxyIdentifier.h"
#include "WebProcessCreationParameters.h"
#include "WebProcessDataStoreParameters.h"
#include "WebsiteDataType.h"
#include <WebCore/BackForwardItemIdentifier.h>
#include <WebCore/FrameIdentifier.h>
#include <WebCore/PageIdentifier.h>
#include <WebCore/ProcessIdentity.h>
#include <WebCore/ProcessQualified.h>
#include <WebCore/RegistrableDomain.h>
#include <WebCore/ScriptExecutionContextIdentifier.h>
#include <WebCore/SecurityOriginData.h>
#include <optional>
#include <span>
#include <wtf/Forward.h>
#include <wtf/HashMap.h>
#include <wtf/HashSet.h>
#include <wtf/ObjectIdentifier.h>
#include <wtf/OptionSet.h>
#include <wtf/ThreadSafeRefCounted.h>
#include <wtf/Vector.h>
#include <wtf/text/WTFString.h>

namespace IPC {
class SharedBufferReference;
}

namespace PAL {
enum class UserInterfaceIdiom : uint8_t;
}

namespace WebCore {
class RegistrableDomain;
enum class EventMakesGamepadsVisible : bool;
enum class CaptionUserPreferencesDisplayMode : uint8_t;
enum class ThirdPartyCookieBlockingMode : uint8_t;
struct ClientOrigin;
struct DisplayUpdate;
struct MockMediaDevice;
struct PrewarmInformation;
struct ScreenProperties;
}

namespace WebKit {
class GamepadData;
class UserData;
enum class CacheModel : uint8_t;
enum class RemoteWorkerType : uint8_t;
struct AccessibilityPreferences;
struct RemoteWorkerInitializationData;
struct TextCheckerState;
struct UserMessage;
struct WebPreferencesStore;
struct WebsiteData;
}

namespace Messages {
namespace WebProcess {

static inline IPC::ReceiverName messageReceiverName()
{
    return IPC::ReceiverName::WebProcess;
}

class InitializeWebProcess {
public:
    using Arguments = std::tuple<WebKit::WebProcessCreationParameters>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_InitializeWebProcess; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::WebProcess_InitializeWebProcessReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<WebCore::ProcessIdentity>;
    using Reply = CompletionHandler<void(WebCore::ProcessIdentity&&)>;
    using Promise = WTF::NativePromise<WebCore::ProcessIdentity, IPC::Error>;
    explicit InitializeWebProcess(WebKit::WebProcessCreationParameters&& processCreationParameters)
        : m_arguments(WTFMove(processCreationParameters))
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<WebKit::WebProcessCreationParameters&&> m_arguments;
};

class SetWebsiteDataStoreParameters {
public:
    using Arguments = std::tuple<WebKit::WebProcessDataStoreParameters>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_SetWebsiteDataStoreParameters; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit SetWebsiteDataStoreParameters(WebKit::WebProcessDataStoreParameters&& parameters)
        : m_arguments(WTFMove(parameters))
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<WebKit::WebProcessDataStoreParameters&&> m_arguments;
};

class CreateWebPage {
public:
    using Arguments = std::tuple<WebCore::PageIdentifier, WebKit::WebPageCreationParameters>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_CreateWebPage; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    CreateWebPage(const WebCore::PageIdentifier& newPageID, WebKit::WebPageCreationParameters&& pageCreationParameters)
        : m_arguments(newPageID, WTFMove(pageCreationParameters))
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const WebCore::PageIdentifier&, WebKit::WebPageCreationParameters&&> m_arguments;
};

class PrewarmWithDomainInformation {
public:
    using Arguments = std::tuple<WebCore::PrewarmInformation>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_PrewarmWithDomainInformation; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit PrewarmWithDomainInformation(const WebCore::PrewarmInformation& prewarmInformation)
        : m_arguments(prewarmInformation)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const WebCore::PrewarmInformation&> m_arguments;
};

class SetCacheModel {
public:
    using Arguments = std::tuple<WebKit::CacheModel>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_SetCacheModel; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit SetCacheModel(WebKit::CacheModel cacheModel)
        : m_arguments(cacheModel)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<WebKit::CacheModel> m_arguments;
};

class RegisterURLSchemeAsEmptyDocument {
public:
    using Arguments = std::tuple<String>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_RegisterURLSchemeAsEmptyDocument; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit RegisterURLSchemeAsEmptyDocument(const String& scheme)
        : m_arguments(scheme)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const String&> m_arguments;
};

class RegisterURLSchemeAsSecure {
public:
    using Arguments = std::tuple<String>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_RegisterURLSchemeAsSecure; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit RegisterURLSchemeAsSecure(const String& scheme)
        : m_arguments(scheme)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const String&> m_arguments;
};

class RegisterURLSchemeAsBypassingContentSecurityPolicy {
public:
    using Arguments = std::tuple<String>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_RegisterURLSchemeAsBypassingContentSecurityPolicy; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit RegisterURLSchemeAsBypassingContentSecurityPolicy(const String& scheme)
        : m_arguments(scheme)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const String&> m_arguments;
};

class SetDomainRelaxationForbiddenForURLScheme {
public:
    using Arguments = std::tuple<String>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_SetDomainRelaxationForbiddenForURLScheme; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit SetDomainRelaxationForbiddenForURLScheme(const String& scheme)
        : m_arguments(scheme)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const String&> m_arguments;
};

class RegisterURLSchemeAsLocal {
public:
    using Arguments = std::tuple<String>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_RegisterURLSchemeAsLocal; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit RegisterURLSchemeAsLocal(const String& scheme)
        : m_arguments(scheme)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const String&> m_arguments;
};

class RegisterURLSchemeAsNoAccess {
public:
    using Arguments = std::tuple<String>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_RegisterURLSchemeAsNoAccess; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit RegisterURLSchemeAsNoAccess(const String& scheme)
        : m_arguments(scheme)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const String&> m_arguments;
};

class RegisterURLSchemeAsDisplayIsolated {
public:
    using Arguments = std::tuple<String>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_RegisterURLSchemeAsDisplayIsolated; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit RegisterURLSchemeAsDisplayIsolated(const String& scheme)
        : m_arguments(scheme)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const String&> m_arguments;
};

class RegisterURLSchemeAsCORSEnabled {
public:
    using Arguments = std::tuple<String>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_RegisterURLSchemeAsCORSEnabled; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit RegisterURLSchemeAsCORSEnabled(const String& scheme)
        : m_arguments(scheme)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const String&> m_arguments;
};

class RegisterURLSchemeAsCachePartitioned {
public:
    using Arguments = std::tuple<String>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_RegisterURLSchemeAsCachePartitioned; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit RegisterURLSchemeAsCachePartitioned(const String& scheme)
        : m_arguments(scheme)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const String&> m_arguments;
};

class RegisterURLSchemeAsCanDisplayOnlyIfCanRequest {
public:
    using Arguments = std::tuple<String>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_RegisterURLSchemeAsCanDisplayOnlyIfCanRequest; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit RegisterURLSchemeAsCanDisplayOnlyIfCanRequest(const String& scheme)
        : m_arguments(scheme)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const String&> m_arguments;
};

#if ENABLE(WK_WEB_EXTENSIONS)
class RegisterURLSchemeAsWebExtension {
public:
    using Arguments = std::tuple<String>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_RegisterURLSchemeAsWebExtension; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit RegisterURLSchemeAsWebExtension(const String& scheme)
        : m_arguments(scheme)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const String&> m_arguments;
};
#endif

class SetDefaultRequestTimeoutInterval {
public:
    using Arguments = std::tuple<double>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_SetDefaultRequestTimeoutInterval; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit SetDefaultRequestTimeoutInterval(double timeoutInterval)
        : m_arguments(timeoutInterval)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<double> m_arguments;
};

class SetAlwaysUsesComplexTextCodePath {
public:
    using Arguments = std::tuple<bool>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_SetAlwaysUsesComplexTextCodePath; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit SetAlwaysUsesComplexTextCodePath(bool alwaysUseComplexText)
        : m_arguments(alwaysUseComplexText)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<bool> m_arguments;
};

class SetDisableFontSubpixelAntialiasingForTesting {
public:
    using Arguments = std::tuple<bool>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_SetDisableFontSubpixelAntialiasingForTesting; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit SetDisableFontSubpixelAntialiasingForTesting(bool disable)
        : m_arguments(disable)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<bool> m_arguments;
};

class SetTrackingPreventionEnabled {
public:
    using Arguments = std::tuple<bool>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_SetTrackingPreventionEnabled; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit SetTrackingPreventionEnabled(bool enabled)
        : m_arguments(enabled)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<bool> m_arguments;
};

class ClearResourceLoadStatistics {
public:
    using Arguments = std::tuple<>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_ClearResourceLoadStatistics; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<> m_arguments;
};

class UserPreferredLanguagesChanged {
public:
    using Arguments = std::tuple<Vector<String>>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_UserPreferredLanguagesChanged; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit UserPreferredLanguagesChanged(const Vector<String>& languages)
        : m_arguments(languages)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const Vector<String>&> m_arguments;
};

class FullKeyboardAccessModeChanged {
public:
    using Arguments = std::tuple<bool>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_FullKeyboardAccessModeChanged; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit FullKeyboardAccessModeChanged(bool fullKeyboardAccessEnabled)
        : m_arguments(fullKeyboardAccessEnabled)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<bool> m_arguments;
};

class UpdateStorageAccessUserAgentStringQuirks {
public:
    using Arguments = std::tuple<HashMap<WebCore::RegistrableDomain, String>>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_UpdateStorageAccessUserAgentStringQuirks; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit UpdateStorageAccessUserAgentStringQuirks(const HashMap<WebCore::RegistrableDomain, String>& userAgentStringQuirks)
        : m_arguments(userAgentStringQuirks)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const HashMap<WebCore::RegistrableDomain, String>&> m_arguments;
};

#if HAVE(MOUSE_DEVICE_OBSERVATION)
class SetHasMouseDevice {
public:
    using Arguments = std::tuple<bool>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_SetHasMouseDevice; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit SetHasMouseDevice(bool hasMouseDevice)
        : m_arguments(hasMouseDevice)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<bool> m_arguments;
};
#endif

#if HAVE(STYLUS_DEVICE_OBSERVATION)
class SetHasStylusDevice {
public:
    using Arguments = std::tuple<bool>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_SetHasStylusDevice; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit SetHasStylusDevice(bool hasStylusDevice)
        : m_arguments(hasStylusDevice)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<bool> m_arguments;
};
#endif

class StartMemorySampler {
public:
    using Arguments = std::tuple<WebKit::SandboxExtensionHandle, String, double>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_StartMemorySampler; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    StartMemorySampler(WebKit::SandboxExtensionHandle&& sampleLogFileHandle, const String& sampleLogFilePath, double interval)
        : m_arguments(WTFMove(sampleLogFileHandle), sampleLogFilePath, interval)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<WebKit::SandboxExtensionHandle&&, const String&, double> m_arguments;
};

class StopMemorySampler {
public:
    using Arguments = std::tuple<>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_StopMemorySampler; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<> m_arguments;
};

class SetTextCheckerState {
public:
    using Arguments = std::tuple<WebKit::TextCheckerState>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_SetTextCheckerState; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit SetTextCheckerState(const WebKit::TextCheckerState& textCheckerState)
        : m_arguments(textCheckerState)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const WebKit::TextCheckerState&> m_arguments;
};

class SetEnhancedAccessibility {
public:
    using Arguments = std::tuple<bool>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_SetEnhancedAccessibility; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit SetEnhancedAccessibility(bool flag)
        : m_arguments(flag)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<bool> m_arguments;
};

class BindAccessibilityFrameWithData {
public:
    using Arguments = std::tuple<WebCore::FrameIdentifier, std::span<const uint8_t>>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_BindAccessibilityFrameWithData; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    BindAccessibilityFrameWithData(const WebCore::FrameIdentifier& frameID, const std::span<const uint8_t>& data)
        : m_arguments(frameID, data)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const WebCore::FrameIdentifier&, const std::span<const uint8_t>&> m_arguments;
};

class GarbageCollectJavaScriptObjects {
public:
    using Arguments = std::tuple<>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_GarbageCollectJavaScriptObjects; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<> m_arguments;
};

class SetJavaScriptGarbageCollectorTimerEnabled {
public:
    using Arguments = std::tuple<bool>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_SetJavaScriptGarbageCollectorTimerEnabled; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit SetJavaScriptGarbageCollectorTimerEnabled(bool enable)
        : m_arguments(enable)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<bool> m_arguments;
};

class SetInjectedBundleParameter {
public:
    using Arguments = std::tuple<String, std::span<const uint8_t>>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_SetInjectedBundleParameter; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    SetInjectedBundleParameter(const String& parameter, const std::span<const uint8_t>& value)
        : m_arguments(parameter, value)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const String&, const std::span<const uint8_t>&> m_arguments;
};

class SetInjectedBundleParameters {
public:
    using Arguments = std::tuple<std::span<const uint8_t>>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_SetInjectedBundleParameters; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit SetInjectedBundleParameters(const std::span<const uint8_t>& parameters)
        : m_arguments(parameters)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const std::span<const uint8_t>&> m_arguments;
};

class HandleInjectedBundleMessage {
public:
    using Arguments = std::tuple<String, WebKit::UserData>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_HandleInjectedBundleMessage; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    HandleInjectedBundleMessage(const String& messageName, const WebKit::UserData& messageBody)
        : m_arguments(messageName, messageBody)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const String&, const WebKit::UserData&> m_arguments;
};

class FetchWebsiteData {
public:
    using Arguments = std::tuple<OptionSet<WebKit::WebsiteDataType>>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_FetchWebsiteData; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::WebProcess_FetchWebsiteDataReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<WebKit::WebsiteData>;
    using Reply = CompletionHandler<void(WebKit::WebsiteData&&)>;
    using Promise = WTF::NativePromise<WebKit::WebsiteData, IPC::Error>;
    explicit FetchWebsiteData(const OptionSet<WebKit::WebsiteDataType>& websiteDataTypes)
        : m_arguments(websiteDataTypes)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const OptionSet<WebKit::WebsiteDataType>&> m_arguments;
};

class DeleteWebsiteData {
public:
    using Arguments = std::tuple<OptionSet<WebKit::WebsiteDataType>, WallTime>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_DeleteWebsiteData; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::WebProcess_DeleteWebsiteDataReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    DeleteWebsiteData(const OptionSet<WebKit::WebsiteDataType>& websiteDataTypes, const WallTime& modifiedSince)
        : m_arguments(websiteDataTypes, modifiedSince)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const OptionSet<WebKit::WebsiteDataType>&, const WallTime&> m_arguments;
};

class DeleteWebsiteDataForOrigins {
public:
    using Arguments = std::tuple<OptionSet<WebKit::WebsiteDataType>, Vector<WebCore::SecurityOriginData>>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_DeleteWebsiteDataForOrigins; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::WebProcess_DeleteWebsiteDataForOriginsReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    DeleteWebsiteDataForOrigins(const OptionSet<WebKit::WebsiteDataType>& websiteDataTypes, const Vector<WebCore::SecurityOriginData>& origins)
        : m_arguments(websiteDataTypes, origins)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const OptionSet<WebKit::WebsiteDataType>&, const Vector<WebCore::SecurityOriginData>&> m_arguments;
};

class DeleteWebsiteDataForOrigin {
public:
    using Arguments = std::tuple<OptionSet<WebKit::WebsiteDataType>, WebCore::ClientOrigin>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_DeleteWebsiteDataForOrigin; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::WebProcess_DeleteWebsiteDataForOriginReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    DeleteWebsiteDataForOrigin(const OptionSet<WebKit::WebsiteDataType>& websiteDataTypes, const WebCore::ClientOrigin& origin)
        : m_arguments(websiteDataTypes, origin)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const OptionSet<WebKit::WebsiteDataType>&, const WebCore::ClientOrigin&> m_arguments;
};

class ReloadExecutionContextsForOrigin {
public:
    using Arguments = std::tuple<WebCore::ClientOrigin, std::optional<WebCore::FrameIdentifier>>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_ReloadExecutionContextsForOrigin; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::WebProcess_ReloadExecutionContextsForOriginReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    ReloadExecutionContextsForOrigin(const WebCore::ClientOrigin& origin, const std::optional<WebCore::FrameIdentifier>& triggeringFrame)
        : m_arguments(origin, triggeringFrame)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const WebCore::ClientOrigin&, const std::optional<WebCore::FrameIdentifier>&> m_arguments;
};

class DeleteAllCookies {
public:
    using Arguments = std::tuple<>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_DeleteAllCookies; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::WebProcess_DeleteAllCookiesReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<> m_arguments;
};

class SetHiddenPageDOMTimerThrottlingIncreaseLimit {
public:
    using Arguments = std::tuple<Seconds>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_SetHiddenPageDOMTimerThrottlingIncreaseLimit; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit SetHiddenPageDOMTimerThrottlingIncreaseLimit(const Seconds& seconds)
        : m_arguments(seconds)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const Seconds&> m_arguments;
};

class SetMemoryCacheDisabled {
public:
    using Arguments = std::tuple<bool>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_SetMemoryCacheDisabled; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit SetMemoryCacheDisabled(bool disabled)
        : m_arguments(disabled)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<bool> m_arguments;
};

#if ENABLE(SERVICE_CONTROLS)
class SetEnabledServices {
public:
    using Arguments = std::tuple<bool, bool, bool>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_SetEnabledServices; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    SetEnabledServices(bool hasImageServices, bool hasSelectionServices, bool hasRichContentServices)
        : m_arguments(hasImageServices, hasSelectionServices, hasRichContentServices)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<bool, bool, bool> m_arguments;
};
#endif

class EnsureAutomationSessionProxy {
public:
    using Arguments = std::tuple<String>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_EnsureAutomationSessionProxy; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit EnsureAutomationSessionProxy(const String& sessionIdentifier)
        : m_arguments(sessionIdentifier)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const String&> m_arguments;
};

class DestroyAutomationSessionProxy {
public:
    using Arguments = std::tuple<>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_DestroyAutomationSessionProxy; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<> m_arguments;
};

class PrepareToSuspend {
public:
    using Arguments = std::tuple<bool, MonotonicTime>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_PrepareToSuspend; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::WebProcess_PrepareToSuspendReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    PrepareToSuspend(bool isSuspensionImminent, const MonotonicTime& estimatedSuspendTime)
        : m_arguments(isSuspensionImminent, estimatedSuspendTime)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<bool, const MonotonicTime&> m_arguments;
};

class ProcessDidResume {
public:
    using Arguments = std::tuple<>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_ProcessDidResume; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<> m_arguments;
};

class BackgroundResponsivenessPing {
public:
    using Arguments = std::tuple<>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_BackgroundResponsivenessPing; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<> m_arguments;
};

#if ENABLE(GAMEPAD)
class SetInitialGamepads {
public:
    using Arguments = std::tuple<Vector<std::optional<WebKit::GamepadData>>>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_SetInitialGamepads; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit SetInitialGamepads(const Vector<std::optional<WebKit::GamepadData>>& gamepadDatas)
        : m_arguments(gamepadDatas)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const Vector<std::optional<WebKit::GamepadData>>&> m_arguments;
};
#endif

#if ENABLE(GAMEPAD)
class GamepadConnected {
public:
    using Arguments = std::tuple<WebKit::GamepadData, WebCore::EventMakesGamepadsVisible>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_GamepadConnected; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    GamepadConnected(const WebKit::GamepadData& gamepadData, WebCore::EventMakesGamepadsVisible eventVisibility)
        : m_arguments(gamepadData, eventVisibility)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const WebKit::GamepadData&, WebCore::EventMakesGamepadsVisible> m_arguments;
};
#endif

#if ENABLE(GAMEPAD)
class GamepadDisconnected {
public:
    using Arguments = std::tuple<unsigned>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_GamepadDisconnected; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit GamepadDisconnected(const unsigned& index)
        : m_arguments(index)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const unsigned&> m_arguments;
};
#endif

class EstablishRemoteWorkerContextConnectionToNetworkProcess {
public:
    using Arguments = std::tuple<WebKit::RemoteWorkerType, WebKit::PageGroupIdentifier, WebKit::WebPageProxyIdentifier, WebCore::PageIdentifier, WebKit::WebPreferencesStore, WebCore::RegistrableDomain, std::optional<WebCore::ScriptExecutionContextIdentifier>, WebKit::RemoteWorkerInitializationData>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_EstablishRemoteWorkerContextConnectionToNetworkProcess; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::WebProcess_EstablishRemoteWorkerContextConnectionToNetworkProcessReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    EstablishRemoteWorkerContextConnectionToNetworkProcess(WebKit::RemoteWorkerType workerType, const WebKit::PageGroupIdentifier& pageGroupID, const WebKit::WebPageProxyIdentifier& webPageProxyID, const WebCore::PageIdentifier& pageID, const WebKit::WebPreferencesStore& store, const WebCore::RegistrableDomain& domain, const std::optional<WebCore::ScriptExecutionContextIdentifier>& serviceWorkerPageIdentifier, const WebKit::RemoteWorkerInitializationData& initializationData)
        : m_arguments(workerType, pageGroupID, webPageProxyID, pageID, store, domain, serviceWorkerPageIdentifier, initializationData)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<WebKit::RemoteWorkerType, const WebKit::PageGroupIdentifier&, const WebKit::WebPageProxyIdentifier&, const WebCore::PageIdentifier&, const WebKit::WebPreferencesStore&, const WebCore::RegistrableDomain&, const std::optional<WebCore::ScriptExecutionContextIdentifier>&, const WebKit::RemoteWorkerInitializationData&> m_arguments;
};

class SetHasSuspendedPageProxy {
public:
    using Arguments = std::tuple<bool>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_SetHasSuspendedPageProxy; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit SetHasSuspendedPageProxy(bool hasSuspendedPageProxy)
        : m_arguments(hasSuspendedPageProxy)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<bool> m_arguments;
};

class SetIsInProcessCache {
public:
    using Arguments = std::tuple<bool>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_SetIsInProcessCache; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::WebProcess_SetIsInProcessCacheReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    explicit SetIsInProcessCache(bool isInProcessCache)
        : m_arguments(isInProcessCache)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<bool> m_arguments;
};

class MarkIsNoLongerPrewarmed {
public:
    using Arguments = std::tuple<>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_MarkIsNoLongerPrewarmed; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<> m_arguments;
};

class GetActivePagesOriginsForTesting {
public:
    using Arguments = std::tuple<>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_GetActivePagesOriginsForTesting; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::WebProcess_GetActivePagesOriginsForTestingReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<Vector<String>>;
    using Reply = CompletionHandler<void(Vector<String>&&)>;
    using Promise = WTF::NativePromise<Vector<String>, IPC::Error>;
    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<> m_arguments;
};

#if PLATFORM(COCOA) || PLATFORM(GTK) || PLATFORM(WPE)
class SetScreenProperties {
public:
    using Arguments = std::tuple<WebCore::ScreenProperties>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_SetScreenProperties; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit SetScreenProperties(const WebCore::ScreenProperties& screenProperties)
        : m_arguments(screenProperties)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const WebCore::ScreenProperties&> m_arguments;
};
#endif

#if PLATFORM(MAC)
class ScrollerStylePreferenceChanged {
public:
    using Arguments = std::tuple<bool>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_ScrollerStylePreferenceChanged; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit ScrollerStylePreferenceChanged(bool useOvelayScrollbars)
        : m_arguments(useOvelayScrollbars)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<bool> m_arguments;
};
#endif

#if PLATFORM(IOS_FAMILY)
class UserInterfaceIdiomDidChange {
public:
    using Arguments = std::tuple<PAL::UserInterfaceIdiom>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_UserInterfaceIdiomDidChange; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit UserInterfaceIdiomDidChange(PAL::UserInterfaceIdiom idiom)
        : m_arguments(idiom)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<PAL::UserInterfaceIdiom> m_arguments;
};
#endif

#if PLATFORM(IOS_FAMILY) && !PLATFORM(MACCATALYST)
class BacklightLevelDidChange {
public:
    using Arguments = std::tuple<float>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_BacklightLevelDidChange; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit BacklightLevelDidChange(float backlightLevel)
        : m_arguments(backlightLevel)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<float> m_arguments;
};
#endif

#if PLATFORM(MAC) || PLATFORM(MACCATALYST)
class ColorPreferencesDidChange {
public:
    using Arguments = std::tuple<>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_ColorPreferencesDidChange; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<> m_arguments;
};
#endif

class IsJITEnabled {
public:
    using Arguments = std::tuple<>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_IsJITEnabled; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::WebProcess_IsJITEnabledReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<bool>;
    using Reply = CompletionHandler<void(bool)>;
    using Promise = WTF::NativePromise<bool, IPC::Error>;
    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<> m_arguments;
};

#if PLATFORM(COCOA)
class SetMediaMIMETypes {
public:
    using Arguments = std::tuple<Vector<String>>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_SetMediaMIMETypes; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit SetMediaMIMETypes(const Vector<String>& types)
        : m_arguments(types)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const Vector<String>&> m_arguments;
};
#endif

#if (PLATFORM(COCOA) && ENABLE(REMOTE_INSPECTOR))
class EnableRemoteWebInspector {
public:
    using Arguments = std::tuple<>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_EnableRemoteWebInspector; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<> m_arguments;
};
#endif

#if ENABLE(MEDIA_STREAM)
class AddMockMediaDevice {
public:
    using Arguments = std::tuple<WebCore::MockMediaDevice>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_AddMockMediaDevice; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit AddMockMediaDevice(const WebCore::MockMediaDevice& device)
        : m_arguments(device)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const WebCore::MockMediaDevice&> m_arguments;
};
#endif

#if ENABLE(MEDIA_STREAM)
class ClearMockMediaDevices {
public:
    using Arguments = std::tuple<>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_ClearMockMediaDevices; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<> m_arguments;
};
#endif

#if ENABLE(MEDIA_STREAM)
class RemoveMockMediaDevice {
public:
    using Arguments = std::tuple<String>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_RemoveMockMediaDevice; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit RemoveMockMediaDevice(const String& persistentId)
        : m_arguments(persistentId)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const String&> m_arguments;
};
#endif

#if ENABLE(MEDIA_STREAM)
class SetMockMediaDeviceIsEphemeral {
public:
    using Arguments = std::tuple<String, bool>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_SetMockMediaDeviceIsEphemeral; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    SetMockMediaDeviceIsEphemeral(const String& persistentId, bool isEphemeral)
        : m_arguments(persistentId, isEphemeral)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const String&, bool> m_arguments;
};
#endif

#if ENABLE(MEDIA_STREAM)
class ResetMockMediaDevices {
public:
    using Arguments = std::tuple<>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_ResetMockMediaDevices; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<> m_arguments;
};
#endif

#if (ENABLE(MEDIA_STREAM) && ENABLE(SANDBOX_EXTENSIONS))
class GrantUserMediaDeviceSandboxExtensions {
public:
    using Arguments = std::tuple<WebKit::MediaDeviceSandboxExtensions>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_GrantUserMediaDeviceSandboxExtensions; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit GrantUserMediaDeviceSandboxExtensions(WebKit::MediaDeviceSandboxExtensions&& sandboxExtensions)
        : m_arguments(WTFMove(sandboxExtensions))
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<WebKit::MediaDeviceSandboxExtensions&&> m_arguments;
};
#endif

#if (ENABLE(MEDIA_STREAM) && ENABLE(SANDBOX_EXTENSIONS))
class RevokeUserMediaDeviceSandboxExtensions {
public:
    using Arguments = std::tuple<Vector<String>>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_RevokeUserMediaDeviceSandboxExtensions; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit RevokeUserMediaDeviceSandboxExtensions(const Vector<String>& sandboxExtensionIDs)
        : m_arguments(sandboxExtensionIDs)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const Vector<String>&> m_arguments;
};
#endif

class ClearCurrentModifierStateForTesting {
public:
    using Arguments = std::tuple<>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_ClearCurrentModifierStateForTesting; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<> m_arguments;
};

class SetBackForwardCacheCapacity {
public:
    using Arguments = std::tuple<unsigned>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_SetBackForwardCacheCapacity; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit SetBackForwardCacheCapacity(const unsigned& capacity)
        : m_arguments(capacity)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const unsigned&> m_arguments;
};

class ClearCachedPage {
public:
    using Arguments = std::tuple<WebCore::BackForwardItemIdentifier>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_ClearCachedPage; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::WebProcess_ClearCachedPageReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    explicit ClearCachedPage(const WebCore::BackForwardItemIdentifier& backForwardItemID)
        : m_arguments(backForwardItemID)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const WebCore::BackForwardItemIdentifier&> m_arguments;
};

#if PLATFORM(GTK) || PLATFORM(WPE)
class SendMessageToWebProcessExtension {
public:
    using Arguments = std::tuple<WebKit::UserMessage>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_SendMessageToWebProcessExtension; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit SendMessageToWebProcessExtension(const WebKit::UserMessage& userMessage)
        : m_arguments(userMessage)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const WebKit::UserMessage&> m_arguments;
};
#endif

class SeedResourceLoadStatisticsForTesting {
public:
    using Arguments = std::tuple<WebCore::RegistrableDomain, WebCore::RegistrableDomain, bool>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_SeedResourceLoadStatisticsForTesting; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::WebProcess_SeedResourceLoadStatisticsForTestingReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    SeedResourceLoadStatisticsForTesting(const WebCore::RegistrableDomain& firstPartyDomain, const WebCore::RegistrableDomain& thirdPartyDomain, bool shouldScheduleNotification)
        : m_arguments(firstPartyDomain, thirdPartyDomain, shouldScheduleNotification)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const WebCore::RegistrableDomain&, const WebCore::RegistrableDomain&, bool> m_arguments;
};

class SetThirdPartyCookieBlockingMode {
public:
    using Arguments = std::tuple<WebCore::ThirdPartyCookieBlockingMode>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_SetThirdPartyCookieBlockingMode; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::WebProcess_SetThirdPartyCookieBlockingModeReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    explicit SetThirdPartyCookieBlockingMode(WebCore::ThirdPartyCookieBlockingMode blockingMode)
        : m_arguments(blockingMode)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<WebCore::ThirdPartyCookieBlockingMode> m_arguments;
};

class SetDomainsWithUserInteraction {
public:
    using Arguments = std::tuple<HashSet<WebCore::RegistrableDomain>>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_SetDomainsWithUserInteraction; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit SetDomainsWithUserInteraction(const HashSet<WebCore::RegistrableDomain>& domains)
        : m_arguments(domains)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const HashSet<WebCore::RegistrableDomain>&> m_arguments;
};

class SetDomainsWithCrossPageStorageAccess {
public:
    using Arguments = std::tuple<HashMap<WebCore::RegistrableDomain, Vector<WebCore::RegistrableDomain>>>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_SetDomainsWithCrossPageStorageAccess; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::WebProcess_SetDomainsWithCrossPageStorageAccessReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    explicit SetDomainsWithCrossPageStorageAccess(const HashMap<WebCore::RegistrableDomain, Vector<WebCore::RegistrableDomain>>& domains)
        : m_arguments(domains)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const HashMap<WebCore::RegistrableDomain, Vector<WebCore::RegistrableDomain>>&> m_arguments;
};

class SendResourceLoadStatisticsDataImmediately {
public:
    using Arguments = std::tuple<>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_SendResourceLoadStatisticsDataImmediately; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::WebProcess_SendResourceLoadStatisticsDataImmediatelyReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<> m_arguments;
};

class UpdateDomainsWithStorageAccessQuirks {
public:
    using Arguments = std::tuple<HashSet<WebCore::RegistrableDomain>>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_UpdateDomainsWithStorageAccessQuirks; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit UpdateDomainsWithStorageAccessQuirks(const HashSet<WebCore::RegistrableDomain>& domainsWithStorageAccessQuirks)
        : m_arguments(domainsWithStorageAccessQuirks)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const HashSet<WebCore::RegistrableDomain>&> m_arguments;
};

class GrantAccessToAssetServices {
public:
    using Arguments = std::tuple<Vector<WebKit::SandboxExtensionHandle>>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_GrantAccessToAssetServices; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit GrantAccessToAssetServices(Vector<WebKit::SandboxExtensionHandle>&& assetServicesHandles)
        : m_arguments(WTFMove(assetServicesHandles))
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<Vector<WebKit::SandboxExtensionHandle>&&> m_arguments;
};

class RevokeAccessToAssetServices {
public:
    using Arguments = std::tuple<>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_RevokeAccessToAssetServices; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<> m_arguments;
};

class SwitchFromStaticFontRegistryToUserFontRegistry {
public:
    using Arguments = std::tuple<Vector<WebKit::SandboxExtensionHandle>>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_SwitchFromStaticFontRegistryToUserFontRegistry; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit SwitchFromStaticFontRegistryToUserFontRegistry(Vector<WebKit::SandboxExtensionHandle>&& fontMachExtensionHandles)
        : m_arguments(WTFMove(fontMachExtensionHandles))
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<Vector<WebKit::SandboxExtensionHandle>&&> m_arguments;
};

#if PLATFORM(COCOA)
class DisableURLSchemeCheckInDataDetectors {
public:
    using Arguments = std::tuple<>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_DisableURLSchemeCheckInDataDetectors; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<> m_arguments;
};
#endif

#if PLATFORM(COCOA)
class UnblockServicesRequiredByAccessibility {
public:
    using Arguments = std::tuple<Vector<WebKit::SandboxExtensionHandle>>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_UnblockServicesRequiredByAccessibility; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit UnblockServicesRequiredByAccessibility(Vector<WebKit::SandboxExtensionHandle>&& handleArray)
        : m_arguments(WTFMove(handleArray))
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<Vector<WebKit::SandboxExtensionHandle>&&> m_arguments;
};
#endif

#if PLATFORM(COCOA)
class PowerSourceDidChange {
public:
    using Arguments = std::tuple<bool>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_PowerSourceDidChange; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit PowerSourceDidChange(bool hasAC)
        : m_arguments(hasAC)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<bool> m_arguments;
};
#endif

#if PLATFORM(GTK) && !USE(GTK4) && USE(CAIRO)
class SetUseSystemAppearanceForScrollbars {
public:
    using Arguments = std::tuple<bool>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_SetUseSystemAppearanceForScrollbars; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit SetUseSystemAppearanceForScrollbars(bool useSystemAppearanceForScrollbars)
        : m_arguments(useSystemAppearanceForScrollbars)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<bool> m_arguments;
};
#endif

#if PLATFORM(COCOA)
class DidWriteToPasteboardAsynchronously {
public:
    using Arguments = std::tuple<String>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_DidWriteToPasteboardAsynchronously; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = true;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit DidWriteToPasteboardAsynchronously(const String& pasteboardName)
        : m_arguments(pasteboardName)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const String&> m_arguments;
};
#endif

#if HAVE(DISPLAY_LINK)
class DisplayDidRefresh {
public:
    using Arguments = std::tuple<uint32_t, WebCore::DisplayUpdate>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_DisplayDidRefresh; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    DisplayDidRefresh(uint32_t displayID, const WebCore::DisplayUpdate& update)
        : m_arguments(displayID, update)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<uint32_t, const WebCore::DisplayUpdate&> m_arguments;
};
#endif

#if PLATFORM(MAC)
class SystemWillPowerOn {
public:
    using Arguments = std::tuple<>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_SystemWillPowerOn; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<> m_arguments;
};
#endif

#if PLATFORM(MAC)
class SystemWillSleep {
public:
    using Arguments = std::tuple<>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_SystemWillSleep; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<> m_arguments;
};
#endif

#if PLATFORM(MAC)
class SystemDidWake {
public:
    using Arguments = std::tuple<>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_SystemDidWake; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<> m_arguments;
};
#endif

#if HAVE(AUDIO_COMPONENT_SERVER_REGISTRATIONS)
class ConsumeAudioComponentRegistrations {
public:
    using Arguments = std::tuple<IPC::SharedBufferReference>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_ConsumeAudioComponentRegistrations; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit ConsumeAudioComponentRegistrations(const IPC::SharedBufferReference& registrationData)
        : m_arguments(registrationData)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const IPC::SharedBufferReference&> m_arguments;
};
#endif

#if PLATFORM(COCOA)
class AccessibilityPreferencesDidChange {
public:
    using Arguments = std::tuple<WebKit::AccessibilityPreferences>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_AccessibilityPreferencesDidChange; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit AccessibilityPreferencesDidChange(const WebKit::AccessibilityPreferences& accessibilityPreferences)
        : m_arguments(accessibilityPreferences)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const WebKit::AccessibilityPreferences&> m_arguments;
};
#endif

#if HAVE(MEDIA_ACCESSIBILITY_FRAMEWORK)
class SetMediaAccessibilityPreferences {
public:
    using Arguments = std::tuple<WebCore::CaptionUserPreferencesDisplayMode, Vector<String>>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_SetMediaAccessibilityPreferences; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    SetMediaAccessibilityPreferences(WebCore::CaptionUserPreferencesDisplayMode displayMode, const Vector<String>& languages)
        : m_arguments(displayMode, languages)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<WebCore::CaptionUserPreferencesDisplayMode, const Vector<String>&> m_arguments;
};
#endif

#if PLATFORM(MAC)
class OpenDirectoryCacheInvalidated {
public:
    using Arguments = std::tuple<WebKit::SandboxExtensionHandle, WebKit::SandboxExtensionHandle>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_OpenDirectoryCacheInvalidated; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    OpenDirectoryCacheInvalidated(WebKit::SandboxExtensionHandle&& handle, WebKit::SandboxExtensionHandle&& machBootstrapHandle)
        : m_arguments(WTFMove(handle), WTFMove(machBootstrapHandle))
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<WebKit::SandboxExtensionHandle&&, WebKit::SandboxExtensionHandle&&> m_arguments;
};
#endif

class ReleaseMemory {
public:
    using Arguments = std::tuple<>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_ReleaseMemory; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::WebProcess_ReleaseMemoryReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<> m_arguments;
};

#if ENABLE(NOTIFY_BLOCKING)
class PostNotification {
public:
    using Arguments = std::tuple<String, std::optional<uint64_t>>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_PostNotification; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    PostNotification(const String& message, const std::optional<uint64_t>& state)
        : m_arguments(message, state)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const String&, const std::optional<uint64_t>&> m_arguments;
};
#endif

#if ENABLE(NOTIFY_BLOCKING)
class PostObserverNotification {
public:
    using Arguments = std::tuple<String>;

    static IPC::MessageName name() { return IPC::MessageName::WebProcess_PostObserverNotification; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit PostObserverNotification(const String& message)
        : m_arguments(message)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const String&> m_arguments;
};
#endif

} // namespace WebProcess
} // namespace Messages
