





(**
 * OCaml binding of the Liblktlang API.
 *)

(**
 * Subprograms may raise this when they cannot open a source file. Note that
 * this does *not* concern analysis unit getters, which create diagnostic
 * vectors for such errors.
 *)
exception FileReadError of string

(**
 * Raised when introspection functions (``Liblktlang.Introspection``) are
 * provided mismatching types/values.
 *)
exception BadTypeError of string

(**
 * Raised when introspection functions (``Liblktlang.Introspection``) are
 * passed an out of bounds index.
 *)
exception OutOfBoundsError of string

(**
 * Raised by lexing functions (``Liblktlang.Lexer``) when the input contains an
 * invalid byte sequence.
 *)
exception InvalidInput of string

(**
 * Exception raise when an invalid symbol is passed to a subprogram.
 *)
exception InvalidSymbolError of string

(**
 * Raised when an invalid unit name is provided.
 *)
exception InvalidUnitNameError of string

(**
 * Exception raised in language bindings when the underlying C API reports an
 * unexpected error that occurred in the library.
 *
 * This kind of exception is raised for internal errors: they should never
 * happen in normal situations and if they are raised at some point, it means
 * the library state is potentially corrupted.
 *
 * Nevertheless, the library does its best not to crash the program,
 * materializing internal errors using this kind of exception.
 *)
exception NativeException of string

(**
 * Exception raised when an API is called while its preconditions are not
 * satisfied.
 *)
exception PreconditionFailure of string

(**
 * Exception that is raised when an error occurs while evaluating any AST node
 * method whose name starts with ``p_``. This is the only exceptions that such
 * functions can raise.
 *)
exception PropertyError of string

(**
 * Exception raised when the provided arguments for a template don't match what
 * the template expects.
 *)
exception TemplateArgsError of string

(**
 * Exception raised when a template has an invalid syntax, such as badly
 * formatted placeholders.
 *)
exception TemplateFormatError of string

(**
 * Exception raised when the instantiation of a template cannot be parsed.
 *)
exception TemplateInstantiationError of string

(**
 * Exception raised while trying to access data that was deallocated. This
 * happens when one tries to use a node whose unit has been reparsed, for
 * instance.
 *)
exception StaleReferenceError of string

(**
 * Subprograms may raise this when they try to parse invalid syntax. Also
 * raised if a field in a parsing node is null due to a syntax error.
 *)
exception SyntaxError of string

(**
 * Raised by lexing functions (``Liblktlang.Lexer``) when the input charset is
 * not supported.
 *)
exception UnknownCharset of string

(**
 * Raised when unparsing functions working on rewritten trees
 * (``Liblktlang.Rewriting``) are called on malformed trees.
 *)
exception MalformedTreeError of string





module AnalysisUnitKind : sig
  (**
  * Specify a kind of analysis unit. Specification units provide an interface
  * to the outer world while body units provide an implementation for the
  * corresponding interface.
  *)

  type t =
  | UnitSpecification
  | UnitBody

  val name : unit -> string
end

module CompletionItemKind : sig
  (**
  * Type of completion item. Refer to the official LSP specification.
  *)

  type t =
  | TextKind
  | MethodKind
  | FunctionKind
  | ConstructorKind
  | FieldKind
  | VariableKind
  | ClassKind
  | InterfaceKind
  | ModuleKind
  | PropertyKind
  | UnitKind
  | ValueKind
  | EnumKind
  | KeywordKind
  | SnippetKind
  | ColorKind
  | FileKind
  | ReferenceKind
  | FolderKind
  | EnumMemberKind
  | ConstantKind
  | StructKind
  | EventKind
  | OperatorKind
  | TypeParameterKind

  val name : unit -> string
end

module DesignatedEnvKind : sig
  (**
  * Discriminant for DesignatedEnv structures.
  *)

  type t =
  | None
  | CurrentEnv
  | NamedEnv
  | DirectEnv

  val name : unit -> string
end

module GrammarRule : sig
  (**
  * Gramar rule to use for parsing.
  *)

  type t =
  | MainRuleRule
  | IdRule
  | RefIdRule
  | TypeRefIdRule
  | DefIdRule
  | DocRule
  | ImportStmtRule
  | ImportsRule
  | LexerDeclRule
  | GrammarDeclRule
  | GrammarRuleRule
  | LexerRuleRule
  | LexerFamilyDeclRule
  | LexerCaseRuleRule
  | LexerCaseAltRule
  | LexerCaseSendRule
  | GrammarPrimaryRule
  | GrammarExprRule
  | GrammarPickRule
  | GrammarImplicitPickRule
  | GrammarOptRule
  | GrammarOptErrorRule
  | GrammarCutRule
  | GrammarStopcutRule
  | GrammarOrExprRule
  | GrammarDiscardExprRule
  | TokenLiteralRule
  | TokenNoCaseLiteralRule
  | TokenPatternRule
  | TokenPatternLiteralRule
  | ParseNodeExprRule
  | GrammarRuleRefRule
  | GrammarListExprRule
  | GrammarListSepRule
  | GrammarSkipRule
  | GrammarNullRule
  | GrammarTokenRule
  | TypeDeclRule
  | GenericDeclRule
  | GenericParamTypeRule
  | EnumLitDeclRule
  | FunDeclRule
  | LambdaParamDeclRule
  | FunParamDeclRule
  | FunParamListRule
  | LambdaParamListRule
  | FieldDeclRule
  | BareDeclRule
  | DeclRule
  | TypeMemberRefRule
  | TypeExprRule
  | TypeRefRule
  | TypeListRule
  | DeclsRule
  | DeclBlockRule
  | ValDeclRule
  | DynvarDeclRule
  | VarBindRule
  | EnvSpecActionRule
  | EnvSpecDeclRule
  | BlockRule
  | PatternRule
  | FilPatternRule
  | ValuePatternRule
  | RegexPatternRule
  | BoolPatternRule
  | EllipsisPatternRule
  | IntegerPatternRule
  | ListPatternRule
  | TuplePatternRule
  | PatternArgRule
  | SelectorCallRule
  | ExprRule
  | RelRule
  | EqRule
  | Arith1Rule
  | Arith2Rule
  | Arith3Rule
  | IsaOrPrimaryRule
  | LogicPropagateCallRule
  | PrimaryRule
  | MatchExprRule
  | NumLitRule
  | BigNumLitRule
  | StringLitRule
  | BlockStringLitRule
  | CharLitRule
  | IfExprRule
  | RaiseExprRule
  | TryExprRule
  | ArrayLiteralRule
  | CallableRefRule
  | NullCondQualRule
  | BasicExprRule
  | TermRule
  | BasicNameRule
  | LambdaExprRule
  | NullLitRule
  | ArgumentRule
  | ArgsRule
  | DeclAnnotationArgsRule
  | DeclAnnotationRule

  val name : unit -> string
end

module LookupKind : sig
  (**

  *)

  type t =
  | Recursive
  | Flat
  | Minimal

  val name : unit -> string
end

val default_grammar_rule : GrammarRule.t

module Sloc : sig
  (**
  * Location in a source file. Line and column numbers are one-based.
  *)

  type t = {
    line : int;
    column : int;
  }
end

module SlocRange : sig
  (**
  * Location of a span of text in a source file.
  *)

  type t = {
    loc_start : Sloc.t;
    loc_end : Sloc.t;
  }

  val pp : Format.formatter -> t -> unit
end

module Diagnostic : sig
  (**
  * Diagnostic for an analysis unit: cannot open the source file, parsing
  * error, ...
  *)

  type t = {
    sloc_range : SlocRange.t;
    message : string;
  }
end

module TokenData : sig
  type t
end

module Token : sig
  (**
  * Reference to a token in an analysis unit.
  *)

  type dummy_context

  type t = {
    context : dummy_context;
    token_data : TokenData.t;
    token_index : int;
    trivia_index : int;
    kind : int;
    text : string;
    sloc_range : SlocRange.t;
  }

  val kind_name : t -> string
  (**
  * Return a human-readable name for a token kind.
  *)

  val text_range : t -> t -> string
  (**
  * Compute the source buffer slice corresponding to the text that spans
  * between the ``First`` and ``Last`` tokens (both included). This yields an
  * empty slice if ``Last`` actually appears before ``First``.
  *
  * if ``First`` and ``Last`` don't belong to the same analysis unit.
  *)

  val is_trivia : t -> bool
  (**
  * Return whether this token is a trivia. If it's not, it's a regular token.
  *)

  val index : t -> int
  (**
  * Zero-based index for this token/trivia. Tokens and trivias get their own
  * index space.
  *)

  val next : t -> t option
  (**
  * Return a reference to the first token scanned in this unit.
  *)

  val previous : t -> t option
  (**
  * Return a reference to the last token scanned in this unit.
  *)

  val compare : t -> t -> int

  val equal : t -> t -> bool

  val hash : t -> int

  val is_equivalent : t -> t -> bool
  (**
  * Return whether ``L`` and ``R`` are structurally equivalent tokens. This
  * means that their position in the stream won't be taken into account, only
  * the kind and text of the token.
  *)

  val pp : Format.formatter -> t -> unit
end

module BigInteger : sig
  type t = Z.t
end

module Character : sig
  (** Represents an utf8 character *)

  type t = private string

  val code : t -> int
  (** Get the codepoint representing the given utf8 character *)

  val chr : int -> t
  (** Get the character representing the given integer codepoint *)
end

module BareNode : sig
  type t
end

module Rebindings : sig
  type t
end

module FileReader : sig
  type t
end

module UnitProvider : sig
  (**
  * Interface to fetch analysis units from a name and a unit kind.
  *
  * The unit provider mechanism provides an abstraction which assumes that to
  * any couple (unit name, unit kind) we can associate at most one source file.
  * This means that several couples can be associated to the same source file,
  * but on the other hand, only one one source file can be associated to a
  * couple.
  *
  * This is used to make the semantic analysis able to switch from one analysis
  * units to another.
  *
  * See the documentation of each unit provider for the exact semantics of the
  * unit name/kind information.
  *)

  type t

  

end

type analysis_context

and analysis_unit

and entity


  and entity_info = {
      rebindings :
         Rebindings.t;
      from_rebound :
         bool;
  }





  
   
  (**
    * argument
    * base_lexer_case_rule_alt
    * base_match_branch
    * block_expr_clause
    * block_string_line
    * class_qualifier
    * decl
    * decl_annotation
    * decl_annotation_args
    * dyn_env_wrapper
    * elsif_branch
    * enum_class_case
    * excludes_null
    * expr
    * full_decl
    * grammar_list_sep
    * import
    * langkit_root
    * lexer_case_rule
    * lexer_case_rule_send
    * list_kind
    * lkt_node_base_list
    * null_cond_qualifier
    * op
    * pattern
    * pattern_detail
    * selector_call
    * type_ref
    * var_bind
    *)
  and lkt_node =
    [
    | `Argument
        of argument_fields
    | `LexerCaseRuleCondAlt
        of lexer_case_rule_cond_alt_fields
    | `LexerCaseRuleDefaultAlt
        of lexer_case_rule_default_alt_fields
    | `MatchBranch
        of match_branch_fields
    | `PatternMatchBranch
        of pattern_match_branch_fields
    | `BlockExprClause
        of block_expr_clause_fields
    | `BlockStringLine
        of block_string_line_fields
    | `ClassQualifierAbsent
        of class_qualifier_absent_fields
    | `ClassQualifierPresent
        of class_qualifier_present_fields
    | `GrammarRuleDecl
        of grammar_rule_decl_fields
    | `SyntheticLexerDecl
        of synthetic_lexer_decl_fields
    | `NodeDecl
        of node_decl_fields
    | `SelfDecl
        of self_decl_fields
    | `BindingValDecl
        of binding_val_decl_fields
    | `EnumLitDecl
        of enum_lit_decl_fields
    | `FieldDecl
        of field_decl_fields
    | `FunParamDecl
        of fun_param_decl_fields
    | `LambdaParamDecl
        of lambda_param_decl_fields
    | `DynVarDecl
        of dyn_var_decl_fields
    | `MatchValDecl
        of match_val_decl_fields
    | `ValDecl
        of val_decl_fields
    | `FunDecl
        of fun_decl_fields
    | `EnvSpecDecl
        of env_spec_decl_fields
    | `ErrorDecl
        of error_decl_fields
    | `GenericDecl
        of generic_decl_fields
    | `GrammarDecl
        of grammar_decl_fields
    | `LexerDecl
        of lexer_decl_fields
    | `LexerFamilyDecl
        of lexer_family_decl_fields
    | `SynthFunDecl
        of synth_fun_decl_fields
    | `SynthParamDecl
        of synth_param_decl_fields
    | `AnyTypeDecl
        of any_type_decl_fields
    | `EnumClassAltDecl
        of enum_class_alt_decl_fields
    | `FunctionType
        of function_type_fields
    | `GenericParamTypeDecl
        of generic_param_type_decl_fields
    | `ClassDecl
        of class_decl_fields
    | `EnumClassDecl
        of enum_class_decl_fields
    | `EnumTypeDecl
        of enum_type_decl_fields
    | `StructDecl
        of struct_decl_fields
    | `TraitDecl
        of trait_decl_fields
    | `DeclAnnotation
        of decl_annotation_fields
    | `DeclAnnotationArgs
        of decl_annotation_args_fields
    | `DynEnvWrapper
        of dyn_env_wrapper_fields
    | `ElsifBranch
        of elsif_branch_fields
    | `EnumClassCase
        of enum_class_case_fields
    | `ExcludesNullAbsent
        of excludes_null_absent_fields
    | `ExcludesNullPresent
        of excludes_null_present_fields
    | `AnyOf
        of any_of_fields
    | `ArrayLiteral
        of array_literal_fields
    | `CallExpr
        of call_expr_fields
    | `LogicPredicate
        of logic_predicate_fields
    | `LogicPropagateCall
        of logic_propagate_call_fields
    | `BinOp
        of bin_op_fields
    | `BlockExpr
        of block_expr_fields
    | `CastExpr
        of cast_expr_fields
    | `DotExpr
        of dot_expr_fields
    | `ErrorOnNull
        of error_on_null_fields
    | `GenericInstantiation
        of generic_instantiation_fields
    | `GrammarCut
        of grammar_cut_fields
    | `GrammarDiscard
        of grammar_discard_fields
    | `GrammarDontSkip
        of grammar_dont_skip_fields
    | `GrammarList
        of grammar_list_fields
    | `GrammarNull
        of grammar_null_fields
    | `GrammarOpt
        of grammar_opt_fields
    | `GrammarOptError
        of grammar_opt_error_fields
    | `GrammarOptErrorGroup
        of grammar_opt_error_group_fields
    | `GrammarOptGroup
        of grammar_opt_group_fields
    | `GrammarOrExpr
        of grammar_or_expr_fields
    | `GrammarPick
        of grammar_pick_fields
    | `GrammarImplicitPick
        of grammar_implicit_pick_fields
    | `GrammarPredicate
        of grammar_predicate_fields
    | `GrammarRuleRef
        of grammar_rule_ref_fields
    | `GrammarSkip
        of grammar_skip_fields
    | `GrammarStopCut
        of grammar_stop_cut_fields
    | `ParseNodeExpr
        of parse_node_expr_fields
    | `TokenLit
        of token_lit_fields
    | `TokenNoCaseLit
        of token_no_case_lit_fields
    | `TokenPatternConcat
        of token_pattern_concat_fields
    | `TokenPatternLit
        of token_pattern_lit_fields
    | `TokenRef
        of token_ref_fields
    | `Id
        of id_fields
    | `DefId
        of def_id_fields
    | `ModuleRefId
        of module_ref_id_fields
    | `RefId
        of ref_id_fields
    | `IfExpr
        of if_expr_fields
    | `Isa
        of isa_fields
    | `KeepExpr
        of keep_expr_fields
    | `LambdaExpr
        of lambda_expr_fields
    | `BigNumLit
        of big_num_lit_fields
    | `CharLit
        of char_lit_fields
    | `NullLit
        of null_lit_fields
    | `NumLit
        of num_lit_fields
    | `BlockStringLit
        of block_string_lit_fields
    | `SingleLineStringLit
        of single_line_string_lit_fields
    | `PatternSingleLineStringLit
        of pattern_single_line_string_lit_fields
    | `LogicAssign
        of logic_assign_fields
    | `LogicExpr
        of logic_expr_fields
    | `LogicPropagate
        of logic_propagate_fields
    | `LogicUnify
        of logic_unify_fields
    | `MatchExpr
        of match_expr_fields
    | `NotExpr
        of not_expr_fields
    | `ParenExpr
        of paren_expr_fields
    | `RaiseExpr
        of raise_expr_fields
    | `SubscriptExpr
        of subscript_expr_fields
    | `TryExpr
        of try_expr_fields
    | `UnOp
        of un_op_fields
    | `FullDecl
        of full_decl_fields
    | `GrammarListSep
        of grammar_list_sep_fields
    | `Import
        of import_fields
    | `LangkitRoot
        of langkit_root_fields
    | `LexerCaseRule
        of lexer_case_rule_fields
    | `LexerCaseRuleSend
        of lexer_case_rule_send_fields
    | `ListKindOne
        of list_kind_one_fields
    | `ListKindZero
        of list_kind_zero_fields
    | `ArgumentList
        of argument_list_fields
    | `BaseLexerCaseRuleAltList
        of base_lexer_case_rule_alt_list_fields
    | `BaseMatchBranchList
        of base_match_branch_list_fields
    | `BlockStringLineList
        of block_string_line_list_fields
    | `CallExprList
        of call_expr_list_fields
    | `DeclAnnotationList
        of decl_annotation_list_fields
    | `ElsifBranchList
        of elsif_branch_list_fields
    | `EnumClassAltDeclList
        of enum_class_alt_decl_list_fields
    | `EnumClassCaseList
        of enum_class_case_list_fields
    | `EnumLitDeclList
        of enum_lit_decl_list_fields
    | `ExprList
        of expr_list_fields
    | `AnyOfList
        of any_of_list_fields
    | `FullDeclList
        of full_decl_list_fields
    | `DeclBlock
        of decl_block_fields
    | `GenericParamDeclList
        of generic_param_decl_list_fields
    | `FunParamDeclList
        of fun_param_decl_list_fields
    | `GrammarExprList
        of grammar_expr_list_fields
    | `GrammarExprListList
        of grammar_expr_list_list_fields
    | `ImportList
        of import_list_fields
    | `LambdaParamDeclList
        of lambda_param_decl_list_fields
    | `LktNodeList
        of lkt_node_list_fields
    | `PatternDetailList
        of pattern_detail_list_fields
    | `PatternList
        of pattern_list_fields
    | `RefIdList
        of ref_id_list_fields
    | `TypeRefList
        of type_ref_list_fields
    | `SyntheticTypeRefList
        of synthetic_type_ref_list_fields
    | `NullCondQualifierAbsent
        of null_cond_qualifier_absent_fields
    | `NullCondQualifierPresent
        of null_cond_qualifier_present_fields
    | `OpAmp
        of op_amp_fields
    | `OpAnd
        of op_and_fields
    | `OpDiv
        of op_div_fields
    | `OpEq
        of op_eq_fields
    | `OpGt
        of op_gt_fields
    | `OpGte
        of op_gte_fields
    | `OpLogicAnd
        of op_logic_and_fields
    | `OpLogicOr
        of op_logic_or_fields
    | `OpLt
        of op_lt_fields
    | `OpLte
        of op_lte_fields
    | `OpMinus
        of op_minus_fields
    | `OpMult
        of op_mult_fields
    | `OpNe
        of op_ne_fields
    | `OpOr
        of op_or_fields
    | `OpOrInt
        of op_or_int_fields
    | `OpPlus
        of op_plus_fields
    | `AnyTypePattern
        of any_type_pattern_fields
    | `BindingPattern
        of binding_pattern_fields
    | `BoolPatternFalse
        of bool_pattern_false_fields
    | `BoolPatternTrue
        of bool_pattern_true_fields
    | `EllipsisPattern
        of ellipsis_pattern_fields
    | `ExtendedPattern
        of extended_pattern_fields
    | `FilteredPattern
        of filtered_pattern_fields
    | `IntegerPattern
        of integer_pattern_fields
    | `ListPattern
        of list_pattern_fields
    | `NotPattern
        of not_pattern_fields
    | `NullPattern
        of null_pattern_fields
    | `OrPattern
        of or_pattern_fields
    | `ParenPattern
        of paren_pattern_fields
    | `RegexPattern
        of regex_pattern_fields
    | `TuplePattern
        of tuple_pattern_fields
    | `TypePattern
        of type_pattern_fields
    | `FieldPatternDetail
        of field_pattern_detail_fields
    | `PropertyPatternDetail
        of property_pattern_detail_fields
    | `SelectorPatternDetail
        of selector_pattern_detail_fields
    | `SelectorCall
        of selector_call_fields
    | `DefaultListTypeRef
        of default_list_type_ref_fields
    | `FunctionTypeRef
        of function_type_ref_fields
    | `GenericTypeRef
        of generic_type_ref_fields
    | `SimpleTypeRef
        of simple_type_ref_fields
    | `VarBind
        of var_bind_fields
    ]

  
   
  and argument =
    [
    | `Argument
        of argument_fields
    ]
  and argument_fields = 
  {
         
    f_name: ref_id
    option
    Lazy.t;
         
    f_value: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and base_lexer_case_rule_alt =
    [
    | `LexerCaseRuleCondAlt
        of lexer_case_rule_cond_alt_fields
    | `LexerCaseRuleDefaultAlt
        of lexer_case_rule_default_alt_fields
    ]

  
   
  and lexer_case_rule_cond_alt =
    [
    | `LexerCaseRuleCondAlt
        of lexer_case_rule_cond_alt_fields
    ]
  and lexer_case_rule_cond_alt_fields = 
  {
         
    f_cond_exprs: ref_id_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and lexer_case_rule_default_alt =
    [
    | `LexerCaseRuleDefaultAlt
        of lexer_case_rule_default_alt_fields
    ]
  and lexer_case_rule_default_alt_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and base_match_branch =
    [
    | `MatchBranch
        of match_branch_fields
    | `PatternMatchBranch
        of pattern_match_branch_fields
    ]

  
   
  and match_branch =
    [
    | `MatchBranch
        of match_branch_fields
    ]
  and match_branch_fields = 
  {
         
    f_decl: match_val_decl
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and pattern_match_branch =
    [
    | `PatternMatchBranch
        of pattern_match_branch_fields
    ]
  and pattern_match_branch_fields = 
  {
         
    f_pattern: [
      | `BindingPattern
          of binding_pattern_fields
      | `BoolPatternFalse
          of bool_pattern_false_fields
      | `BoolPatternTrue
          of bool_pattern_true_fields
      | `ExtendedPattern
          of extended_pattern_fields
      | `FilteredPattern
          of filtered_pattern_fields
      | `IntegerPattern
          of integer_pattern_fields
      | `ListPattern
          of list_pattern_fields
      | `NotPattern
          of not_pattern_fields
      | `NullPattern
          of null_pattern_fields
      | `OrPattern
          of or_pattern_fields
      | `ParenPattern
          of paren_pattern_fields
      | `RegexPattern
          of regex_pattern_fields
      | `TuplePattern
          of tuple_pattern_fields
      | `TypePattern
          of type_pattern_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and block_expr_clause =
    [
    | `BlockExprClause
        of block_expr_clause_fields
    ]
  and block_expr_clause_fields = 
  {
         
    f_clause: [
      | `ValDecl
          of val_decl_fields
      | `VarBind
          of var_bind_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and block_string_line =
    [
    | `BlockStringLine
        of block_string_line_fields
    ]
  and block_string_line_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and class_qualifier =
    [
    | `ClassQualifierAbsent
        of class_qualifier_absent_fields
    | `ClassQualifierPresent
        of class_qualifier_present_fields
    ]

  
   
  and class_qualifier_absent =
    [
    | `ClassQualifierAbsent
        of class_qualifier_absent_fields
    ]
  and class_qualifier_absent_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and class_qualifier_present =
    [
    | `ClassQualifierPresent
        of class_qualifier_present_fields
    ]
  and class_qualifier_present_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  (**
    * base_grammar_rule_decl
    * base_val_decl
    * env_spec_decl
    * error_decl
    * generic_decl
    * grammar_decl
    * lexer_decl
    * lexer_family_decl
    * synth_fun_decl
    * synth_param_decl
    * type_decl
    *)
  and decl =
    [
    | `GrammarRuleDecl
        of grammar_rule_decl_fields
    | `SyntheticLexerDecl
        of synthetic_lexer_decl_fields
    | `NodeDecl
        of node_decl_fields
    | `SelfDecl
        of self_decl_fields
    | `BindingValDecl
        of binding_val_decl_fields
    | `EnumLitDecl
        of enum_lit_decl_fields
    | `FieldDecl
        of field_decl_fields
    | `FunParamDecl
        of fun_param_decl_fields
    | `LambdaParamDecl
        of lambda_param_decl_fields
    | `DynVarDecl
        of dyn_var_decl_fields
    | `MatchValDecl
        of match_val_decl_fields
    | `ValDecl
        of val_decl_fields
    | `FunDecl
        of fun_decl_fields
    | `EnvSpecDecl
        of env_spec_decl_fields
    | `ErrorDecl
        of error_decl_fields
    | `GenericDecl
        of generic_decl_fields
    | `GrammarDecl
        of grammar_decl_fields
    | `LexerDecl
        of lexer_decl_fields
    | `LexerFamilyDecl
        of lexer_family_decl_fields
    | `SynthFunDecl
        of synth_fun_decl_fields
    | `SynthParamDecl
        of synth_param_decl_fields
    | `AnyTypeDecl
        of any_type_decl_fields
    | `EnumClassAltDecl
        of enum_class_alt_decl_fields
    | `FunctionType
        of function_type_fields
    | `GenericParamTypeDecl
        of generic_param_type_decl_fields
    | `ClassDecl
        of class_decl_fields
    | `EnumClassDecl
        of enum_class_decl_fields
    | `EnumTypeDecl
        of enum_type_decl_fields
    | `StructDecl
        of struct_decl_fields
    | `TraitDecl
        of trait_decl_fields
    ]

  
   
  and base_grammar_rule_decl =
    [
    | `GrammarRuleDecl
        of grammar_rule_decl_fields
    | `SyntheticLexerDecl
        of synthetic_lexer_decl_fields
    ]

  
   
  and grammar_rule_decl =
    [
    | `GrammarRuleDecl
        of grammar_rule_decl_fields
    ]
  and grammar_rule_decl_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and synthetic_lexer_decl =
    [
    | `SyntheticLexerDecl
        of synthetic_lexer_decl_fields
    ]
  and synthetic_lexer_decl_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  (**
    * node_decl
    * self_decl
    * user_val_decl
    *)
  and base_val_decl =
    [
    | `NodeDecl
        of node_decl_fields
    | `SelfDecl
        of self_decl_fields
    | `BindingValDecl
        of binding_val_decl_fields
    | `EnumLitDecl
        of enum_lit_decl_fields
    | `FieldDecl
        of field_decl_fields
    | `FunParamDecl
        of fun_param_decl_fields
    | `LambdaParamDecl
        of lambda_param_decl_fields
    | `DynVarDecl
        of dyn_var_decl_fields
    | `MatchValDecl
        of match_val_decl_fields
    | `ValDecl
        of val_decl_fields
    | `FunDecl
        of fun_decl_fields
    ]

  
   
  and node_decl =
    [
    | `NodeDecl
        of node_decl_fields
    ]
  and node_decl_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and self_decl =
    [
    | `SelfDecl
        of self_decl_fields
    ]
  and self_decl_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  (**
    * binding_val_decl
    * enum_lit_decl
    * explicitly_typed_decl
    * fun_decl
    *)
  and user_val_decl =
    [
    | `BindingValDecl
        of binding_val_decl_fields
    | `EnumLitDecl
        of enum_lit_decl_fields
    | `FieldDecl
        of field_decl_fields
    | `FunParamDecl
        of fun_param_decl_fields
    | `LambdaParamDecl
        of lambda_param_decl_fields
    | `DynVarDecl
        of dyn_var_decl_fields
    | `MatchValDecl
        of match_val_decl_fields
    | `ValDecl
        of val_decl_fields
    | `FunDecl
        of fun_decl_fields
    ]

  
   
  and binding_val_decl =
    [
    | `BindingValDecl
        of binding_val_decl_fields
    ]
  and binding_val_decl_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and enum_lit_decl =
    [
    | `EnumLitDecl
        of enum_lit_decl_fields
    ]
  and enum_lit_decl_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  (**
    * component_decl
    * dyn_var_decl
    * match_val_decl
    * val_decl
    *)
  and explicitly_typed_decl =
    [
    | `FieldDecl
        of field_decl_fields
    | `FunParamDecl
        of fun_param_decl_fields
    | `LambdaParamDecl
        of lambda_param_decl_fields
    | `DynVarDecl
        of dyn_var_decl_fields
    | `MatchValDecl
        of match_val_decl_fields
    | `ValDecl
        of val_decl_fields
    ]

  
   
  and component_decl =
    [
    | `FieldDecl
        of field_decl_fields
    | `FunParamDecl
        of fun_param_decl_fields
    | `LambdaParamDecl
        of lambda_param_decl_fields
    ]

  
   
  and field_decl =
    [
    | `FieldDecl
        of field_decl_fields
    ]
  and field_decl_fields = 
  {
         
    f_trait_ref: dot_expr
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and fun_param_decl =
    [
    | `FunParamDecl
        of fun_param_decl_fields
    ]
  and fun_param_decl_fields = 
  {
         
    f_decl_annotations: decl_annotation_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and lambda_param_decl =
    [
    | `LambdaParamDecl
        of lambda_param_decl_fields
    ]
  and lambda_param_decl_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and dyn_var_decl =
    [
    | `DynVarDecl
        of dyn_var_decl_fields
    ]
  and dyn_var_decl_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and match_val_decl =
    [
    | `MatchValDecl
        of match_val_decl_fields
    ]
  and match_val_decl_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and val_decl =
    [
    | `ValDecl
        of val_decl_fields
    ]
  and val_decl_fields = 
  {
         
    f_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and fun_decl =
    [
    | `FunDecl
        of fun_decl_fields
    ]
  and fun_decl_fields = 
  {
         
    f_params: fun_param_decl_list
    Lazy.t;
         
    f_return_type: [
      | `FunctionTypeRef
          of function_type_ref_fields
      | `GenericTypeRef
          of generic_type_ref_fields
      | `SimpleTypeRef
          of simple_type_ref_fields
    ]
    Lazy.t;
         
    f_trait_ref: dot_expr
    option
    Lazy.t;
         
    f_body: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and env_spec_decl =
    [
    | `EnvSpecDecl
        of env_spec_decl_fields
    ]
  and env_spec_decl_fields = 
  {
         
    f_actions: call_expr_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and error_decl =
    [
    | `ErrorDecl
        of error_decl_fields
    ]
  and error_decl_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and generic_decl =
    [
    | `GenericDecl
        of generic_decl_fields
    ]
  and generic_decl_fields = 
  {
         
    f_generic_param_decls: generic_param_decl_list
    Lazy.t;
         
    f_decl: [
      | `ClassDecl
          of class_decl_fields
      | `DynVarDecl
          of dyn_var_decl_fields
      | `EnumClassDecl
          of enum_class_decl_fields
      | `EnumTypeDecl
          of enum_type_decl_fields
      | `EnvSpecDecl
          of env_spec_decl_fields
      | `FieldDecl
          of field_decl_fields
      | `FunDecl
          of fun_decl_fields
      | `GenericDecl
          of generic_decl_fields
      | `GrammarDecl
          of grammar_decl_fields
      | `GrammarRuleDecl
          of grammar_rule_decl_fields
      | `LexerDecl
          of lexer_decl_fields
      | `StructDecl
          of struct_decl_fields
      | `TraitDecl
          of trait_decl_fields
      | `ValDecl
          of val_decl_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and grammar_decl =
    [
    | `GrammarDecl
        of grammar_decl_fields
    ]
  and grammar_decl_fields = 
  {
         
    f_rules: full_decl_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and lexer_decl =
    [
    | `LexerDecl
        of lexer_decl_fields
    ]
  and lexer_decl_fields = 
  {
         
    f_rules: lkt_node_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and lexer_family_decl =
    [
    | `LexerFamilyDecl
        of lexer_family_decl_fields
    ]
  and lexer_family_decl_fields = 
  {
         
    f_rules: full_decl_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and synth_fun_decl =
    [
    | `SynthFunDecl
        of synth_fun_decl_fields
    ]
  and synth_fun_decl_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and synth_param_decl =
    [
    | `SynthParamDecl
        of synth_param_decl_fields
    ]
  and synth_param_decl_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  (**
    * any_type_decl
    * enum_class_alt_decl
    * function_type
    * generic_param_type_decl
    * named_type_decl
    *)
  and type_decl =
    [
    | `AnyTypeDecl
        of any_type_decl_fields
    | `EnumClassAltDecl
        of enum_class_alt_decl_fields
    | `FunctionType
        of function_type_fields
    | `GenericParamTypeDecl
        of generic_param_type_decl_fields
    | `ClassDecl
        of class_decl_fields
    | `EnumClassDecl
        of enum_class_decl_fields
    | `EnumTypeDecl
        of enum_type_decl_fields
    | `StructDecl
        of struct_decl_fields
    | `TraitDecl
        of trait_decl_fields
    ]

  
   
  and any_type_decl =
    [
    | `AnyTypeDecl
        of any_type_decl_fields
    ]
  and any_type_decl_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and enum_class_alt_decl =
    [
    | `EnumClassAltDecl
        of enum_class_alt_decl_fields
    ]
  and enum_class_alt_decl_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and function_type =
    [
    | `FunctionType
        of function_type_fields
    ]
  and function_type_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and generic_param_type_decl =
    [
    | `GenericParamTypeDecl
        of generic_param_type_decl_fields
    ]
  and generic_param_type_decl_fields = 
  {
         
    f_has_class: class_qualifier
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  (**
    * basic_class_decl
    * enum_type_decl
    * struct_decl
    * trait_decl
    *)
  and named_type_decl =
    [
    | `ClassDecl
        of class_decl_fields
    | `EnumClassDecl
        of enum_class_decl_fields
    | `EnumTypeDecl
        of enum_type_decl_fields
    | `StructDecl
        of struct_decl_fields
    | `TraitDecl
        of trait_decl_fields
    ]

  
   
  and basic_class_decl =
    [
    | `ClassDecl
        of class_decl_fields
    | `EnumClassDecl
        of enum_class_decl_fields
    ]

  
   
  and class_decl =
    [
    | `ClassDecl
        of class_decl_fields
    ]
  and class_decl_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and enum_class_decl =
    [
    | `EnumClassDecl
        of enum_class_decl_fields
    ]
  and enum_class_decl_fields = 
  {
         
    f_branches: enum_class_case_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and enum_type_decl =
    [
    | `EnumTypeDecl
        of enum_type_decl_fields
    ]
  and enum_type_decl_fields = 
  {
         
    f_literals: enum_lit_decl_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and struct_decl =
    [
    | `StructDecl
        of struct_decl_fields
    ]
  and struct_decl_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and trait_decl =
    [
    | `TraitDecl
        of trait_decl_fields
    ]
  and trait_decl_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and decl_annotation =
    [
    | `DeclAnnotation
        of decl_annotation_fields
    ]
  and decl_annotation_fields = 
  {
         
    f_name: id
    Lazy.t;
         
    f_args: decl_annotation_args
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and decl_annotation_args =
    [
    | `DeclAnnotationArgs
        of decl_annotation_args_fields
    ]
  and decl_annotation_args_fields = 
  {
         
    f_args: argument_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and dyn_env_wrapper =
    [
    | `DynEnvWrapper
        of dyn_env_wrapper_fields
    ]
  and dyn_env_wrapper_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and elsif_branch =
    [
    | `ElsifBranch
        of elsif_branch_fields
    ]
  and elsif_branch_fields = 
  {
         
    f_cond_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
         
    f_then_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and enum_class_case =
    [
    | `EnumClassCase
        of enum_class_case_fields
    ]
  and enum_class_case_fields = 
  {
         
    f_decls: enum_class_alt_decl_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and excludes_null =
    [
    | `ExcludesNullAbsent
        of excludes_null_absent_fields
    | `ExcludesNullPresent
        of excludes_null_present_fields
    ]

  
   
  and excludes_null_absent =
    [
    | `ExcludesNullAbsent
        of excludes_null_absent_fields
    ]
  and excludes_null_absent_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and excludes_null_present =
    [
    | `ExcludesNullPresent
        of excludes_null_present_fields
    ]
  and excludes_null_present_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  (**
    * any_of
    * array_literal
    * base_call_expr
    * bin_op
    * block_expr
    * cast_expr
    * dot_expr
    * error_on_null
    * generic_instantiation
    * grammar_expr
    * id
    * if_expr
    * isa
    * keep_expr
    * lambda_expr
    * lit
    * logic_assign
    * logic_expr
    * logic_propagate
    * logic_unify
    * match_expr
    * not_expr
    * paren_expr
    * raise_expr
    * subscript_expr
    * try_expr
    * un_op
    *)
  and expr =
    [
    | `AnyOf
        of any_of_fields
    | `ArrayLiteral
        of array_literal_fields
    | `CallExpr
        of call_expr_fields
    | `LogicPredicate
        of logic_predicate_fields
    | `LogicPropagateCall
        of logic_propagate_call_fields
    | `BinOp
        of bin_op_fields
    | `BlockExpr
        of block_expr_fields
    | `CastExpr
        of cast_expr_fields
    | `DotExpr
        of dot_expr_fields
    | `ErrorOnNull
        of error_on_null_fields
    | `GenericInstantiation
        of generic_instantiation_fields
    | `GrammarCut
        of grammar_cut_fields
    | `GrammarDiscard
        of grammar_discard_fields
    | `GrammarDontSkip
        of grammar_dont_skip_fields
    | `GrammarList
        of grammar_list_fields
    | `GrammarNull
        of grammar_null_fields
    | `GrammarOpt
        of grammar_opt_fields
    | `GrammarOptError
        of grammar_opt_error_fields
    | `GrammarOptErrorGroup
        of grammar_opt_error_group_fields
    | `GrammarOptGroup
        of grammar_opt_group_fields
    | `GrammarOrExpr
        of grammar_or_expr_fields
    | `GrammarPick
        of grammar_pick_fields
    | `GrammarImplicitPick
        of grammar_implicit_pick_fields
    | `GrammarPredicate
        of grammar_predicate_fields
    | `GrammarRuleRef
        of grammar_rule_ref_fields
    | `GrammarSkip
        of grammar_skip_fields
    | `GrammarStopCut
        of grammar_stop_cut_fields
    | `ParseNodeExpr
        of parse_node_expr_fields
    | `TokenLit
        of token_lit_fields
    | `TokenNoCaseLit
        of token_no_case_lit_fields
    | `TokenPatternConcat
        of token_pattern_concat_fields
    | `TokenPatternLit
        of token_pattern_lit_fields
    | `TokenRef
        of token_ref_fields
    | `Id
        of id_fields
    | `DefId
        of def_id_fields
    | `ModuleRefId
        of module_ref_id_fields
    | `RefId
        of ref_id_fields
    | `IfExpr
        of if_expr_fields
    | `Isa
        of isa_fields
    | `KeepExpr
        of keep_expr_fields
    | `LambdaExpr
        of lambda_expr_fields
    | `BigNumLit
        of big_num_lit_fields
    | `CharLit
        of char_lit_fields
    | `NullLit
        of null_lit_fields
    | `NumLit
        of num_lit_fields
    | `BlockStringLit
        of block_string_lit_fields
    | `SingleLineStringLit
        of single_line_string_lit_fields
    | `PatternSingleLineStringLit
        of pattern_single_line_string_lit_fields
    | `LogicAssign
        of logic_assign_fields
    | `LogicExpr
        of logic_expr_fields
    | `LogicPropagate
        of logic_propagate_fields
    | `LogicUnify
        of logic_unify_fields
    | `MatchExpr
        of match_expr_fields
    | `NotExpr
        of not_expr_fields
    | `ParenExpr
        of paren_expr_fields
    | `RaiseExpr
        of raise_expr_fields
    | `SubscriptExpr
        of subscript_expr_fields
    | `TryExpr
        of try_expr_fields
    | `UnOp
        of un_op_fields
    ]

  
   
  and any_of =
    [
    | `AnyOf
        of any_of_fields
    ]
  and any_of_fields = 
  {
         
    f_expr: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
    ]
    Lazy.t;
         
    f_values: any_of_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and array_literal =
    [
    | `ArrayLiteral
        of array_literal_fields
    ]
  and array_literal_fields = 
  {
         
    f_exprs: expr_list
    Lazy.t;
         
    f_element_type: [
      | `FunctionTypeRef
          of function_type_ref_fields
      | `GenericTypeRef
          of generic_type_ref_fields
      | `SimpleTypeRef
          of simple_type_ref_fields
    ]
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  (**
    * call_expr
    * logic_call_expr
    *)
  and base_call_expr =
    [
    | `CallExpr
        of call_expr_fields
    | `LogicPredicate
        of logic_predicate_fields
    | `LogicPropagateCall
        of logic_propagate_call_fields
    ]

  
   
  and call_expr =
    [
    | `CallExpr
        of call_expr_fields
    ]
  and call_expr_fields = 
  {
         
    f_name: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `KeepExpr
          of keep_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
    ]
    Lazy.t;
         
    f_args: argument_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and logic_call_expr =
    [
    | `LogicPredicate
        of logic_predicate_fields
    | `LogicPropagateCall
        of logic_propagate_call_fields
    ]

  
   
  and logic_predicate =
    [
    | `LogicPredicate
        of logic_predicate_fields
    ]
  and logic_predicate_fields = 
  {
         
    f_name: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `KeepExpr
          of keep_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
    ]
    Lazy.t;
         
    f_args: argument_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and logic_propagate_call =
    [
    | `LogicPropagateCall
        of logic_propagate_call_fields
    ]
  and logic_propagate_call_fields = 
  {
         
    f_name: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `KeepExpr
          of keep_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
    ]
    Lazy.t;
         
    f_args: argument_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and bin_op =
    [
    | `BinOp
        of bin_op_fields
    ]
  and bin_op_fields = 
  {
         
    f_left: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
         
    f_op: op
    Lazy.t;
         
    f_right: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and block_expr =
    [
    | `BlockExpr
        of block_expr_fields
    ]
  and block_expr_fields = 
  {
         
    f_clauses: lkt_node_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and cast_expr =
    [
    | `CastExpr
        of cast_expr_fields
    ]
  and cast_expr_fields = 
  {
         
    f_expr: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `KeepExpr
          of keep_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
    ]
    Lazy.t;
         
    f_null_cond: null_cond_qualifier
    Lazy.t;
         
    f_excludes_null: excludes_null
    Lazy.t;
         
    f_dest_type: [
      | `FunctionTypeRef
          of function_type_ref_fields
      | `GenericTypeRef
          of generic_type_ref_fields
      | `SimpleTypeRef
          of simple_type_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and dot_expr =
    [
    | `DotExpr
        of dot_expr_fields
    ]
  and dot_expr_fields = 
  {
         
    f_prefix: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `KeepExpr
          of keep_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
    ]
    Lazy.t;
         
    f_null_cond: null_cond_qualifier
    Lazy.t;
         
    f_suffix: ref_id
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and error_on_null =
    [
    | `ErrorOnNull
        of error_on_null_fields
    ]
  and error_on_null_fields = 
  {
         
    f_expr: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `KeepExpr
          of keep_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and generic_instantiation =
    [
    | `GenericInstantiation
        of generic_instantiation_fields
    ]
  and generic_instantiation_fields = 
  {
         
    f_name: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `KeepExpr
          of keep_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
    ]
    Lazy.t;
         
    f_args: type_ref_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  (**
    * grammar_cut
    * grammar_discard
    * grammar_dont_skip
    * grammar_list
    * grammar_null
    * grammar_opt
    * grammar_opt_error
    * grammar_opt_error_group
    * grammar_opt_group
    * grammar_or_expr
    * grammar_pick
    * grammar_predicate
    * grammar_rule_ref
    * grammar_skip
    * grammar_stop_cut
    * parse_node_expr
    * token_lit
    * token_no_case_lit
    * token_pattern_concat
    * token_pattern_lit
    * token_ref
    *)
  and grammar_expr =
    [
    | `GrammarCut
        of grammar_cut_fields
    | `GrammarDiscard
        of grammar_discard_fields
    | `GrammarDontSkip
        of grammar_dont_skip_fields
    | `GrammarList
        of grammar_list_fields
    | `GrammarNull
        of grammar_null_fields
    | `GrammarOpt
        of grammar_opt_fields
    | `GrammarOptError
        of grammar_opt_error_fields
    | `GrammarOptErrorGroup
        of grammar_opt_error_group_fields
    | `GrammarOptGroup
        of grammar_opt_group_fields
    | `GrammarOrExpr
        of grammar_or_expr_fields
    | `GrammarPick
        of grammar_pick_fields
    | `GrammarImplicitPick
        of grammar_implicit_pick_fields
    | `GrammarPredicate
        of grammar_predicate_fields
    | `GrammarRuleRef
        of grammar_rule_ref_fields
    | `GrammarSkip
        of grammar_skip_fields
    | `GrammarStopCut
        of grammar_stop_cut_fields
    | `ParseNodeExpr
        of parse_node_expr_fields
    | `TokenLit
        of token_lit_fields
    | `TokenNoCaseLit
        of token_no_case_lit_fields
    | `TokenPatternConcat
        of token_pattern_concat_fields
    | `TokenPatternLit
        of token_pattern_lit_fields
    | `TokenRef
        of token_ref_fields
    ]

  
   
  and grammar_cut =
    [
    | `GrammarCut
        of grammar_cut_fields
    ]
  and grammar_cut_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and grammar_discard =
    [
    | `GrammarDiscard
        of grammar_discard_fields
    ]
  and grammar_discard_fields = 
  {
         
    f_expr: grammar_expr
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and grammar_dont_skip =
    [
    | `GrammarDontSkip
        of grammar_dont_skip_fields
    ]
  and grammar_dont_skip_fields = 
  {
         
    f_expr: grammar_expr
    Lazy.t;
         
    f_dont_skip: grammar_expr
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and grammar_list =
    [
    | `GrammarList
        of grammar_list_fields
    ]
  and grammar_list_fields = 
  {
         
    f_list_type: type_ref
    Lazy.t;
         
    f_kind: list_kind
    Lazy.t;
         
    f_expr: grammar_expr
    Lazy.t;
         
    f_sep: grammar_list_sep
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and grammar_null =
    [
    | `GrammarNull
        of grammar_null_fields
    ]
  and grammar_null_fields = 
  {
         
    f_name: [
      | `FunctionTypeRef
          of function_type_ref_fields
      | `GenericTypeRef
          of generic_type_ref_fields
      | `SimpleTypeRef
          of simple_type_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and grammar_opt =
    [
    | `GrammarOpt
        of grammar_opt_fields
    ]
  and grammar_opt_fields = 
  {
         
    f_expr: grammar_expr
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and grammar_opt_error =
    [
    | `GrammarOptError
        of grammar_opt_error_fields
    ]
  and grammar_opt_error_fields = 
  {
         
    f_expr: grammar_expr
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and grammar_opt_error_group =
    [
    | `GrammarOptErrorGroup
        of grammar_opt_error_group_fields
    ]
  and grammar_opt_error_group_fields = 
  {
         
    f_expr: grammar_expr_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and grammar_opt_group =
    [
    | `GrammarOptGroup
        of grammar_opt_group_fields
    ]
  and grammar_opt_group_fields = 
  {
         
    f_expr: grammar_expr_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and grammar_or_expr =
    [
    | `GrammarOrExpr
        of grammar_or_expr_fields
    ]
  and grammar_or_expr_fields = 
  {
         
    f_sub_exprs: grammar_expr_list_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and grammar_pick =
    [
    | `GrammarPick
        of grammar_pick_fields
    | `GrammarImplicitPick
        of grammar_implicit_pick_fields
    ]
  and grammar_pick_fields = 
  {
         
    f_exprs: grammar_expr_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and grammar_implicit_pick =
    [
    | `GrammarImplicitPick
        of grammar_implicit_pick_fields
    ]
  and grammar_implicit_pick_fields = 
  {
         
    f_exprs: grammar_expr_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and grammar_predicate =
    [
    | `GrammarPredicate
        of grammar_predicate_fields
    ]
  and grammar_predicate_fields = 
  {
         
    f_expr: grammar_expr
    Lazy.t;
         
    f_prop_ref: [
      | `DotExpr
          of dot_expr_fields
      | `RefId
          of ref_id_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and grammar_rule_ref =
    [
    | `GrammarRuleRef
        of grammar_rule_ref_fields
    ]
  and grammar_rule_ref_fields = 
  {
         
    f_node_name: ref_id
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and grammar_skip =
    [
    | `GrammarSkip
        of grammar_skip_fields
    ]
  and grammar_skip_fields = 
  {
         
    f_name: [
      | `FunctionTypeRef
          of function_type_ref_fields
      | `GenericTypeRef
          of generic_type_ref_fields
      | `SimpleTypeRef
          of simple_type_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and grammar_stop_cut =
    [
    | `GrammarStopCut
        of grammar_stop_cut_fields
    ]
  and grammar_stop_cut_fields = 
  {
         
    f_expr: grammar_expr
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and parse_node_expr =
    [
    | `ParseNodeExpr
        of parse_node_expr_fields
    ]
  and parse_node_expr_fields = 
  {
         
    f_node_name: [
      | `FunctionTypeRef
          of function_type_ref_fields
      | `GenericTypeRef
          of generic_type_ref_fields
      | `SimpleTypeRef
          of simple_type_ref_fields
    ]
    Lazy.t;
         
    f_sub_exprs: grammar_expr_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and token_lit =
    [
    | `TokenLit
        of token_lit_fields
    ]
  and token_lit_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and token_no_case_lit =
    [
    | `TokenNoCaseLit
        of token_no_case_lit_fields
    ]
  and token_no_case_lit_fields = 
  {
         
    f_lit: token_lit
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and token_pattern_concat =
    [
    | `TokenPatternConcat
        of token_pattern_concat_fields
    ]
  and token_pattern_concat_fields = 
  {
         
    f_left: [
      | `TokenPatternConcat
          of token_pattern_concat_fields
      | `TokenPatternLit
          of token_pattern_lit_fields
    ]
    Lazy.t;
         
    f_right: token_pattern_lit
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and token_pattern_lit =
    [
    | `TokenPatternLit
        of token_pattern_lit_fields
    ]
  and token_pattern_lit_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and token_ref =
    [
    | `TokenRef
        of token_ref_fields
    ]
  and token_ref_fields = 
  {
         
    f_token_name: ref_id
    Lazy.t;
         
    f_expr: token_lit
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and id =
    [
    | `Id
        of id_fields
    | `DefId
        of def_id_fields
    | `ModuleRefId
        of module_ref_id_fields
    | `RefId
        of ref_id_fields
    ]
  and id_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and def_id =
    [
    | `DefId
        of def_id_fields
    ]
  and def_id_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and module_ref_id =
    [
    | `ModuleRefId
        of module_ref_id_fields
    ]
  and module_ref_id_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and ref_id =
    [
    | `RefId
        of ref_id_fields
    ]
  and ref_id_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and if_expr =
    [
    | `IfExpr
        of if_expr_fields
    ]
  and if_expr_fields = 
  {
         
    f_cond_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
         
    f_then_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
         
    f_alternatives: elsif_branch_list
    Lazy.t;
         
    f_else_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and isa =
    [
    | `Isa
        of isa_fields
    ]
  and isa_fields = 
  {
         
    f_expr: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
    ]
    Lazy.t;
         
    f_pattern: [
      | `BindingPattern
          of binding_pattern_fields
      | `BoolPatternFalse
          of bool_pattern_false_fields
      | `BoolPatternTrue
          of bool_pattern_true_fields
      | `ExtendedPattern
          of extended_pattern_fields
      | `FilteredPattern
          of filtered_pattern_fields
      | `IntegerPattern
          of integer_pattern_fields
      | `ListPattern
          of list_pattern_fields
      | `NotPattern
          of not_pattern_fields
      | `NullPattern
          of null_pattern_fields
      | `OrPattern
          of or_pattern_fields
      | `ParenPattern
          of paren_pattern_fields
      | `RegexPattern
          of regex_pattern_fields
      | `TuplePattern
          of tuple_pattern_fields
      | `TypePattern
          of type_pattern_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and keep_expr =
    [
    | `KeepExpr
        of keep_expr_fields
    ]
  and keep_expr_fields = 
  {
         
    f_expr: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `KeepExpr
          of keep_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
    ]
    Lazy.t;
         
    f_null_cond: null_cond_qualifier
    Lazy.t;
         
    f_keep_type: [
      | `FunctionTypeRef
          of function_type_ref_fields
      | `GenericTypeRef
          of generic_type_ref_fields
      | `SimpleTypeRef
          of simple_type_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and lambda_expr =
    [
    | `LambdaExpr
        of lambda_expr_fields
    ]
  and lambda_expr_fields = 
  {
         
    f_params: lambda_param_decl_list
    Lazy.t;
         
    f_return_type: [
      | `FunctionTypeRef
          of function_type_ref_fields
      | `GenericTypeRef
          of generic_type_ref_fields
      | `SimpleTypeRef
          of simple_type_ref_fields
    ]
    option
    Lazy.t;
         
    f_body: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  (**
    * big_num_lit
    * char_lit
    * null_lit
    * num_lit
    * string_lit
    *)
  and lit =
    [
    | `BigNumLit
        of big_num_lit_fields
    | `CharLit
        of char_lit_fields
    | `NullLit
        of null_lit_fields
    | `NumLit
        of num_lit_fields
    | `BlockStringLit
        of block_string_lit_fields
    | `SingleLineStringLit
        of single_line_string_lit_fields
    | `PatternSingleLineStringLit
        of pattern_single_line_string_lit_fields
    ]

  
   
  and big_num_lit =
    [
    | `BigNumLit
        of big_num_lit_fields
    ]
  and big_num_lit_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and char_lit =
    [
    | `CharLit
        of char_lit_fields
    ]
  and char_lit_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and null_lit =
    [
    | `NullLit
        of null_lit_fields
    ]
  and null_lit_fields = 
  {
         
    f_dest_type: [
      | `FunctionTypeRef
          of function_type_ref_fields
      | `GenericTypeRef
          of generic_type_ref_fields
      | `SimpleTypeRef
          of simple_type_ref_fields
    ]
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and num_lit =
    [
    | `NumLit
        of num_lit_fields
    ]
  and num_lit_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  (**
    * block_string_lit
    * single_line_string_lit
    *)
  and string_lit =
    [
    | `BlockStringLit
        of block_string_lit_fields
    | `SingleLineStringLit
        of single_line_string_lit_fields
    | `PatternSingleLineStringLit
        of pattern_single_line_string_lit_fields
    ]

  
   
  and block_string_lit =
    [
    | `BlockStringLit
        of block_string_lit_fields
    ]
  and block_string_lit_fields = 
  {
         
    f_lines: block_string_line_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and single_line_string_lit =
    [
    | `SingleLineStringLit
        of single_line_string_lit_fields
    | `PatternSingleLineStringLit
        of pattern_single_line_string_lit_fields
    ]
  and single_line_string_lit_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and pattern_single_line_string_lit =
    [
    | `PatternSingleLineStringLit
        of pattern_single_line_string_lit_fields
    ]
  and pattern_single_line_string_lit_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and logic_assign =
    [
    | `LogicAssign
        of logic_assign_fields
    ]
  and logic_assign_fields = 
  {
         
    f_dest_var: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
    ]
    Lazy.t;
         
    f_value: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and logic_expr =
    [
    | `LogicExpr
        of logic_expr_fields
    ]
  and logic_expr_fields = 
  {
         
    f_expr: [
      | `CallExpr
          of call_expr_fields
      | `RefId
          of ref_id_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and logic_propagate =
    [
    | `LogicPropagate
        of logic_propagate_fields
    ]
  and logic_propagate_fields = 
  {
         
    f_dest_var: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
    ]
    Lazy.t;
         
    f_call: logic_propagate_call
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and logic_unify =
    [
    | `LogicUnify
        of logic_unify_fields
    ]
  and logic_unify_fields = 
  {
         
    f_lhs: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
    ]
    Lazy.t;
         
    f_rhs: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and match_expr =
    [
    | `MatchExpr
        of match_expr_fields
    ]
  and match_expr_fields = 
  {
         
    f_match_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
         
    f_branches: base_match_branch_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and not_expr =
    [
    | `NotExpr
        of not_expr_fields
    ]
  and not_expr_fields = 
  {
         
    f_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and paren_expr =
    [
    | `ParenExpr
        of paren_expr_fields
    ]
  and paren_expr_fields = 
  {
         
    f_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and raise_expr =
    [
    | `RaiseExpr
        of raise_expr_fields
    ]
  and raise_expr_fields = 
  {
         
    f_dest_type: [
      | `FunctionTypeRef
          of function_type_ref_fields
      | `GenericTypeRef
          of generic_type_ref_fields
      | `SimpleTypeRef
          of simple_type_ref_fields
    ]
    option
    Lazy.t;
         
    f_except_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and subscript_expr =
    [
    | `SubscriptExpr
        of subscript_expr_fields
    ]
  and subscript_expr_fields = 
  {
         
    f_prefix: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `KeepExpr
          of keep_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
    ]
    Lazy.t;
         
    f_null_cond: null_cond_qualifier
    Lazy.t;
         
    f_index: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and try_expr =
    [
    | `TryExpr
        of try_expr_fields
    ]
  and try_expr_fields = 
  {
         
    f_try_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
         
    f_or_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and un_op =
    [
    | `UnOp
        of un_op_fields
    ]
  and un_op_fields = 
  {
         
    f_op: [
      | `OpMinus
          of op_minus_fields
      | `OpPlus
          of op_plus_fields
    ]
    Lazy.t;
         
    f_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and full_decl =
    [
    | `FullDecl
        of full_decl_fields
    ]
  and full_decl_fields = 
  {
         
    f_doc: string_lit
    option
    Lazy.t;
         
    f_decl_annotations: decl_annotation_list
    Lazy.t;
         
    f_decl: [
      | `ClassDecl
          of class_decl_fields
      | `DynVarDecl
          of dyn_var_decl_fields
      | `EnumClassDecl
          of enum_class_decl_fields
      | `EnumTypeDecl
          of enum_type_decl_fields
      | `EnvSpecDecl
          of env_spec_decl_fields
      | `FieldDecl
          of field_decl_fields
      | `FunDecl
          of fun_decl_fields
      | `GenericDecl
          of generic_decl_fields
      | `GenericParamTypeDecl
          of generic_param_type_decl_fields
      | `GrammarDecl
          of grammar_decl_fields
      | `GrammarRuleDecl
          of grammar_rule_decl_fields
      | `LexerDecl
          of lexer_decl_fields
      | `LexerFamilyDecl
          of lexer_family_decl_fields
      | `StructDecl
          of struct_decl_fields
      | `TraitDecl
          of trait_decl_fields
      | `ValDecl
          of val_decl_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and grammar_list_sep =
    [
    | `GrammarListSep
        of grammar_list_sep_fields
    ]
  and grammar_list_sep_fields = 
  {
         
    f_token: grammar_expr
    Lazy.t;
         
    f_extra: id
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and import =
    [
    | `Import
        of import_fields
    ]
  and import_fields = 
  {
         
    f_name: module_ref_id
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and langkit_root =
    [
    | `LangkitRoot
        of langkit_root_fields
    ]
  and langkit_root_fields = 
  {
         
    f_imports: import_list
    Lazy.t;
         
    f_decls: full_decl_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and lexer_case_rule =
    [
    | `LexerCaseRule
        of lexer_case_rule_fields
    ]
  and lexer_case_rule_fields = 
  {
         
    f_expr: [
      | `GrammarCut
          of grammar_cut_fields
      | `GrammarDiscard
          of grammar_discard_fields
      | `GrammarImplicitPick
          of grammar_implicit_pick_fields
      | `GrammarList
          of grammar_list_fields
      | `GrammarNull
          of grammar_null_fields
      | `GrammarOpt
          of grammar_opt_fields
      | `GrammarOptError
          of grammar_opt_error_fields
      | `GrammarOptErrorGroup
          of grammar_opt_error_group_fields
      | `GrammarOptGroup
          of grammar_opt_group_fields
      | `GrammarOrExpr
          of grammar_or_expr_fields
      | `GrammarPick
          of grammar_pick_fields
      | `GrammarRuleRef
          of grammar_rule_ref_fields
      | `GrammarSkip
          of grammar_skip_fields
      | `GrammarStopCut
          of grammar_stop_cut_fields
      | `ParseNodeExpr
          of parse_node_expr_fields
      | `TokenLit
          of token_lit_fields
      | `TokenNoCaseLit
          of token_no_case_lit_fields
      | `TokenPatternConcat
          of token_pattern_concat_fields
      | `TokenPatternLit
          of token_pattern_lit_fields
      | `TokenRef
          of token_ref_fields
    ]
    Lazy.t;
         
    f_alts: base_lexer_case_rule_alt_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and lexer_case_rule_send =
    [
    | `LexerCaseRuleSend
        of lexer_case_rule_send_fields
    ]
  and lexer_case_rule_send_fields = 
  {
         
    f_sent: ref_id
    Lazy.t;
         
    f_match_size: num_lit
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and list_kind =
    [
    | `ListKindOne
        of list_kind_one_fields
    | `ListKindZero
        of list_kind_zero_fields
    ]

  
   
  and list_kind_one =
    [
    | `ListKindOne
        of list_kind_one_fields
    ]
  and list_kind_one_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and list_kind_zero =
    [
    | `ListKindZero
        of list_kind_zero_fields
    ]
  and list_kind_zero_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  (**
    * argument_list
    * base_lexer_case_rule_alt_list
    * base_match_branch_list
    * block_string_line_list
    * call_expr_list
    * decl_annotation_list
    * elsif_branch_list
    * enum_class_alt_decl_list
    * enum_class_case_list
    * enum_lit_decl_list
    * expr_list
    * full_decl_list
    * fun_param_decl_list
    * grammar_expr_list
    * grammar_expr_list_list
    * import_list
    * lambda_param_decl_list
    * lkt_node_list
    * pattern_detail_list
    * pattern_list
    * ref_id_list
    * type_ref_list
    *)
  and lkt_node_base_list =
    [
    | `ArgumentList
        of argument_list_fields
    | `BaseLexerCaseRuleAltList
        of base_lexer_case_rule_alt_list_fields
    | `BaseMatchBranchList
        of base_match_branch_list_fields
    | `BlockStringLineList
        of block_string_line_list_fields
    | `CallExprList
        of call_expr_list_fields
    | `DeclAnnotationList
        of decl_annotation_list_fields
    | `ElsifBranchList
        of elsif_branch_list_fields
    | `EnumClassAltDeclList
        of enum_class_alt_decl_list_fields
    | `EnumClassCaseList
        of enum_class_case_list_fields
    | `EnumLitDeclList
        of enum_lit_decl_list_fields
    | `ExprList
        of expr_list_fields
    | `AnyOfList
        of any_of_list_fields
    | `FullDeclList
        of full_decl_list_fields
    | `DeclBlock
        of decl_block_fields
    | `GenericParamDeclList
        of generic_param_decl_list_fields
    | `FunParamDeclList
        of fun_param_decl_list_fields
    | `GrammarExprList
        of grammar_expr_list_fields
    | `GrammarExprListList
        of grammar_expr_list_list_fields
    | `ImportList
        of import_list_fields
    | `LambdaParamDeclList
        of lambda_param_decl_list_fields
    | `LktNodeList
        of lkt_node_list_fields
    | `PatternDetailList
        of pattern_detail_list_fields
    | `PatternList
        of pattern_list_fields
    | `RefIdList
        of ref_id_list_fields
    | `TypeRefList
        of type_ref_list_fields
    | `SyntheticTypeRefList
        of synthetic_type_ref_list_fields
    ]

  
   
  and argument_list =
    [
    | `ArgumentList
        of argument_list_fields
    ]
  and argument_list_fields = 
  {
    list : argument list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and base_lexer_case_rule_alt_list =
    [
    | `BaseLexerCaseRuleAltList
        of base_lexer_case_rule_alt_list_fields
    ]
  and base_lexer_case_rule_alt_list_fields = 
  {
    list : base_lexer_case_rule_alt list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and base_match_branch_list =
    [
    | `BaseMatchBranchList
        of base_match_branch_list_fields
    ]
  and base_match_branch_list_fields = 
  {
    list : base_match_branch list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and block_string_line_list =
    [
    | `BlockStringLineList
        of block_string_line_list_fields
    ]
  and block_string_line_list_fields = 
  {
    list : block_string_line list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and call_expr_list =
    [
    | `CallExprList
        of call_expr_list_fields
    ]
  and call_expr_list_fields = 
  {
    list : call_expr list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and decl_annotation_list =
    [
    | `DeclAnnotationList
        of decl_annotation_list_fields
    ]
  and decl_annotation_list_fields = 
  {
    list : decl_annotation list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and elsif_branch_list =
    [
    | `ElsifBranchList
        of elsif_branch_list_fields
    ]
  and elsif_branch_list_fields = 
  {
    list : elsif_branch list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and enum_class_alt_decl_list =
    [
    | `EnumClassAltDeclList
        of enum_class_alt_decl_list_fields
    ]
  and enum_class_alt_decl_list_fields = 
  {
    list : enum_class_alt_decl list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and enum_class_case_list =
    [
    | `EnumClassCaseList
        of enum_class_case_list_fields
    ]
  and enum_class_case_list_fields = 
  {
    list : enum_class_case list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and enum_lit_decl_list =
    [
    | `EnumLitDeclList
        of enum_lit_decl_list_fields
    ]
  and enum_lit_decl_list_fields = 
  {
    list : enum_lit_decl list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and expr_list =
    [
    | `ExprList
        of expr_list_fields
    | `AnyOfList
        of any_of_list_fields
    ]
  and expr_list_fields = 
  {
    list : expr list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and any_of_list =
    [
    | `AnyOfList
        of any_of_list_fields
    ]
  and any_of_list_fields = 
  {
    list : expr list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and full_decl_list =
    [
    | `FullDeclList
        of full_decl_list_fields
    | `DeclBlock
        of decl_block_fields
    | `GenericParamDeclList
        of generic_param_decl_list_fields
    ]
  and full_decl_list_fields = 
  {
    list : full_decl list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and decl_block =
    [
    | `DeclBlock
        of decl_block_fields
    ]
  and decl_block_fields = 
  {
    list : full_decl list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and generic_param_decl_list =
    [
    | `GenericParamDeclList
        of generic_param_decl_list_fields
    ]
  and generic_param_decl_list_fields = 
  {
    list : full_decl list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and fun_param_decl_list =
    [
    | `FunParamDeclList
        of fun_param_decl_list_fields
    ]
  and fun_param_decl_list_fields = 
  {
    list : fun_param_decl list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and grammar_expr_list =
    [
    | `GrammarExprList
        of grammar_expr_list_fields
    ]
  and grammar_expr_list_fields = 
  {
    list : grammar_expr list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and grammar_expr_list_list =
    [
    | `GrammarExprListList
        of grammar_expr_list_list_fields
    ]
  and grammar_expr_list_list_fields = 
  {
    list : grammar_expr_list list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and import_list =
    [
    | `ImportList
        of import_list_fields
    ]
  and import_list_fields = 
  {
    list : import list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and lambda_param_decl_list =
    [
    | `LambdaParamDeclList
        of lambda_param_decl_list_fields
    ]
  and lambda_param_decl_list_fields = 
  {
    list : lambda_param_decl list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and lkt_node_list =
    [
    | `LktNodeList
        of lkt_node_list_fields
    ]
  and lkt_node_list_fields = 
  {
    list : lkt_node list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and pattern_detail_list =
    [
    | `PatternDetailList
        of pattern_detail_list_fields
    ]
  and pattern_detail_list_fields = 
  {
    list : pattern_detail list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and pattern_list =
    [
    | `PatternList
        of pattern_list_fields
    ]
  and pattern_list_fields = 
  {
    list : pattern list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and ref_id_list =
    [
    | `RefIdList
        of ref_id_list_fields
    ]
  and ref_id_list_fields = 
  {
    list : ref_id list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and type_ref_list =
    [
    | `TypeRefList
        of type_ref_list_fields
    | `SyntheticTypeRefList
        of synthetic_type_ref_list_fields
    ]
  and type_ref_list_fields = 
  {
    list : type_ref list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and synthetic_type_ref_list =
    [
    | `SyntheticTypeRefList
        of synthetic_type_ref_list_fields
    ]
  and synthetic_type_ref_list_fields = 
  {
    list : type_ref list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and null_cond_qualifier =
    [
    | `NullCondQualifierAbsent
        of null_cond_qualifier_absent_fields
    | `NullCondQualifierPresent
        of null_cond_qualifier_present_fields
    ]

  
   
  and null_cond_qualifier_absent =
    [
    | `NullCondQualifierAbsent
        of null_cond_qualifier_absent_fields
    ]
  and null_cond_qualifier_absent_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and null_cond_qualifier_present =
    [
    | `NullCondQualifierPresent
        of null_cond_qualifier_present_fields
    ]
  and null_cond_qualifier_present_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op =
    [
    | `OpAmp
        of op_amp_fields
    | `OpAnd
        of op_and_fields
    | `OpDiv
        of op_div_fields
    | `OpEq
        of op_eq_fields
    | `OpGt
        of op_gt_fields
    | `OpGte
        of op_gte_fields
    | `OpLogicAnd
        of op_logic_and_fields
    | `OpLogicOr
        of op_logic_or_fields
    | `OpLt
        of op_lt_fields
    | `OpLte
        of op_lte_fields
    | `OpMinus
        of op_minus_fields
    | `OpMult
        of op_mult_fields
    | `OpNe
        of op_ne_fields
    | `OpOr
        of op_or_fields
    | `OpOrInt
        of op_or_int_fields
    | `OpPlus
        of op_plus_fields
    ]

  
   
  and op_amp =
    [
    | `OpAmp
        of op_amp_fields
    ]
  and op_amp_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_and =
    [
    | `OpAnd
        of op_and_fields
    ]
  and op_and_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_div =
    [
    | `OpDiv
        of op_div_fields
    ]
  and op_div_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_eq =
    [
    | `OpEq
        of op_eq_fields
    ]
  and op_eq_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_gt =
    [
    | `OpGt
        of op_gt_fields
    ]
  and op_gt_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_gte =
    [
    | `OpGte
        of op_gte_fields
    ]
  and op_gte_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_logic_and =
    [
    | `OpLogicAnd
        of op_logic_and_fields
    ]
  and op_logic_and_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_logic_or =
    [
    | `OpLogicOr
        of op_logic_or_fields
    ]
  and op_logic_or_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_lt =
    [
    | `OpLt
        of op_lt_fields
    ]
  and op_lt_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_lte =
    [
    | `OpLte
        of op_lte_fields
    ]
  and op_lte_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_minus =
    [
    | `OpMinus
        of op_minus_fields
    ]
  and op_minus_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_mult =
    [
    | `OpMult
        of op_mult_fields
    ]
  and op_mult_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_ne =
    [
    | `OpNe
        of op_ne_fields
    ]
  and op_ne_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_or =
    [
    | `OpOr
        of op_or_fields
    ]
  and op_or_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_or_int =
    [
    | `OpOrInt
        of op_or_int_fields
    ]
  and op_or_int_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_plus =
    [
    | `OpPlus
        of op_plus_fields
    ]
  and op_plus_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  (**
    * any_type_pattern
    * binding_pattern
    * bool_pattern
    * ellipsis_pattern
    * extended_pattern
    * filtered_pattern
    * integer_pattern
    * list_pattern
    * not_pattern
    * null_pattern
    * or_pattern
    * paren_pattern
    * regex_pattern
    * tuple_pattern
    * type_pattern
    *)
  and pattern =
    [
    | `AnyTypePattern
        of any_type_pattern_fields
    | `BindingPattern
        of binding_pattern_fields
    | `BoolPatternFalse
        of bool_pattern_false_fields
    | `BoolPatternTrue
        of bool_pattern_true_fields
    | `EllipsisPattern
        of ellipsis_pattern_fields
    | `ExtendedPattern
        of extended_pattern_fields
    | `FilteredPattern
        of filtered_pattern_fields
    | `IntegerPattern
        of integer_pattern_fields
    | `ListPattern
        of list_pattern_fields
    | `NotPattern
        of not_pattern_fields
    | `NullPattern
        of null_pattern_fields
    | `OrPattern
        of or_pattern_fields
    | `ParenPattern
        of paren_pattern_fields
    | `RegexPattern
        of regex_pattern_fields
    | `TuplePattern
        of tuple_pattern_fields
    | `TypePattern
        of type_pattern_fields
    ]

  
   
  and any_type_pattern =
    [
    | `AnyTypePattern
        of any_type_pattern_fields
    ]
  and any_type_pattern_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and binding_pattern =
    [
    | `BindingPattern
        of binding_pattern_fields
    ]
  and binding_pattern_fields = 
  {
         
    f_decl: binding_val_decl
    Lazy.t;
         
    f_sub_pattern: [
      | `BindingPattern
          of binding_pattern_fields
      | `BoolPatternFalse
          of bool_pattern_false_fields
      | `BoolPatternTrue
          of bool_pattern_true_fields
      | `ExtendedPattern
          of extended_pattern_fields
      | `IntegerPattern
          of integer_pattern_fields
      | `ListPattern
          of list_pattern_fields
      | `NotPattern
          of not_pattern_fields
      | `NullPattern
          of null_pattern_fields
      | `ParenPattern
          of paren_pattern_fields
      | `RegexPattern
          of regex_pattern_fields
      | `TuplePattern
          of tuple_pattern_fields
      | `TypePattern
          of type_pattern_fields
    ]
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and bool_pattern =
    [
    | `BoolPatternFalse
        of bool_pattern_false_fields
    | `BoolPatternTrue
        of bool_pattern_true_fields
    ]

  
   
  and bool_pattern_false =
    [
    | `BoolPatternFalse
        of bool_pattern_false_fields
    ]
  and bool_pattern_false_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and bool_pattern_true =
    [
    | `BoolPatternTrue
        of bool_pattern_true_fields
    ]
  and bool_pattern_true_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and ellipsis_pattern =
    [
    | `EllipsisPattern
        of ellipsis_pattern_fields
    ]
  and ellipsis_pattern_fields = 
  {
         
    f_binding: id
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and extended_pattern =
    [
    | `ExtendedPattern
        of extended_pattern_fields
    ]
  and extended_pattern_fields = 
  {
         
    f_sub_pattern: [
      | `AnyTypePattern
          of any_type_pattern_fields
      | `ParenPattern
          of paren_pattern_fields
      | `TypePattern
          of type_pattern_fields
    ]
    Lazy.t;
         
    f_details: pattern_detail_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and filtered_pattern =
    [
    | `FilteredPattern
        of filtered_pattern_fields
    ]
  and filtered_pattern_fields = 
  {
         
    f_sub_pattern: [
      | `BindingPattern
          of binding_pattern_fields
      | `BoolPatternFalse
          of bool_pattern_false_fields
      | `BoolPatternTrue
          of bool_pattern_true_fields
      | `ExtendedPattern
          of extended_pattern_fields
      | `IntegerPattern
          of integer_pattern_fields
      | `ListPattern
          of list_pattern_fields
      | `NotPattern
          of not_pattern_fields
      | `NullPattern
          of null_pattern_fields
      | `ParenPattern
          of paren_pattern_fields
      | `RegexPattern
          of regex_pattern_fields
      | `TuplePattern
          of tuple_pattern_fields
      | `TypePattern
          of type_pattern_fields
    ]
    Lazy.t;
         
    f_predicate: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and integer_pattern =
    [
    | `IntegerPattern
        of integer_pattern_fields
    ]
  and integer_pattern_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and list_pattern =
    [
    | `ListPattern
        of list_pattern_fields
    ]
  and list_pattern_fields = 
  {
         
    f_sub_patterns: pattern_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and not_pattern =
    [
    | `NotPattern
        of not_pattern_fields
    ]
  and not_pattern_fields = 
  {
         
    f_sub_pattern: [
      | `BindingPattern
          of binding_pattern_fields
      | `BoolPatternFalse
          of bool_pattern_false_fields
      | `BoolPatternTrue
          of bool_pattern_true_fields
      | `ExtendedPattern
          of extended_pattern_fields
      | `IntegerPattern
          of integer_pattern_fields
      | `ListPattern
          of list_pattern_fields
      | `NotPattern
          of not_pattern_fields
      | `NullPattern
          of null_pattern_fields
      | `ParenPattern
          of paren_pattern_fields
      | `RegexPattern
          of regex_pattern_fields
      | `TuplePattern
          of tuple_pattern_fields
      | `TypePattern
          of type_pattern_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and null_pattern =
    [
    | `NullPattern
        of null_pattern_fields
    ]
  and null_pattern_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and or_pattern =
    [
    | `OrPattern
        of or_pattern_fields
    ]
  and or_pattern_fields = 
  {
         
    f_left_sub_pattern: [
      | `BindingPattern
          of binding_pattern_fields
      | `BoolPatternFalse
          of bool_pattern_false_fields
      | `BoolPatternTrue
          of bool_pattern_true_fields
      | `ExtendedPattern
          of extended_pattern_fields
      | `FilteredPattern
          of filtered_pattern_fields
      | `IntegerPattern
          of integer_pattern_fields
      | `ListPattern
          of list_pattern_fields
      | `NotPattern
          of not_pattern_fields
      | `NullPattern
          of null_pattern_fields
      | `ParenPattern
          of paren_pattern_fields
      | `RegexPattern
          of regex_pattern_fields
      | `TuplePattern
          of tuple_pattern_fields
      | `TypePattern
          of type_pattern_fields
    ]
    Lazy.t;
         
    f_right_sub_pattern: [
      | `BindingPattern
          of binding_pattern_fields
      | `BoolPatternFalse
          of bool_pattern_false_fields
      | `BoolPatternTrue
          of bool_pattern_true_fields
      | `ExtendedPattern
          of extended_pattern_fields
      | `FilteredPattern
          of filtered_pattern_fields
      | `IntegerPattern
          of integer_pattern_fields
      | `ListPattern
          of list_pattern_fields
      | `NotPattern
          of not_pattern_fields
      | `NullPattern
          of null_pattern_fields
      | `OrPattern
          of or_pattern_fields
      | `ParenPattern
          of paren_pattern_fields
      | `RegexPattern
          of regex_pattern_fields
      | `TuplePattern
          of tuple_pattern_fields
      | `TypePattern
          of type_pattern_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and paren_pattern =
    [
    | `ParenPattern
        of paren_pattern_fields
    ]
  and paren_pattern_fields = 
  {
         
    f_sub_pattern: [
      | `BindingPattern
          of binding_pattern_fields
      | `BoolPatternFalse
          of bool_pattern_false_fields
      | `BoolPatternTrue
          of bool_pattern_true_fields
      | `ExtendedPattern
          of extended_pattern_fields
      | `FilteredPattern
          of filtered_pattern_fields
      | `IntegerPattern
          of integer_pattern_fields
      | `ListPattern
          of list_pattern_fields
      | `NotPattern
          of not_pattern_fields
      | `NullPattern
          of null_pattern_fields
      | `OrPattern
          of or_pattern_fields
      | `ParenPattern
          of paren_pattern_fields
      | `RegexPattern
          of regex_pattern_fields
      | `TuplePattern
          of tuple_pattern_fields
      | `TypePattern
          of type_pattern_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and regex_pattern =
    [
    | `RegexPattern
        of regex_pattern_fields
    ]
  and regex_pattern_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and tuple_pattern =
    [
    | `TuplePattern
        of tuple_pattern_fields
    ]
  and tuple_pattern_fields = 
  {
         
    f_sub_patterns: pattern_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and type_pattern =
    [
    | `TypePattern
        of type_pattern_fields
    ]
  and type_pattern_fields = 
  {
         
    f_type_name: [
      | `FunctionTypeRef
          of function_type_ref_fields
      | `GenericTypeRef
          of generic_type_ref_fields
      | `SimpleTypeRef
          of simple_type_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and pattern_detail =
    [
    | `FieldPatternDetail
        of field_pattern_detail_fields
    | `PropertyPatternDetail
        of property_pattern_detail_fields
    | `SelectorPatternDetail
        of selector_pattern_detail_fields
    ]

  
   
  and field_pattern_detail =
    [
    | `FieldPatternDetail
        of field_pattern_detail_fields
    ]
  and field_pattern_detail_fields = 
  {
         
    f_id: id
    Lazy.t;
         
    f_expected_value: [
      | `BindingPattern
          of binding_pattern_fields
      | `BoolPatternFalse
          of bool_pattern_false_fields
      | `BoolPatternTrue
          of bool_pattern_true_fields
      | `ExtendedPattern
          of extended_pattern_fields
      | `FilteredPattern
          of filtered_pattern_fields
      | `IntegerPattern
          of integer_pattern_fields
      | `ListPattern
          of list_pattern_fields
      | `NotPattern
          of not_pattern_fields
      | `NullPattern
          of null_pattern_fields
      | `OrPattern
          of or_pattern_fields
      | `ParenPattern
          of paren_pattern_fields
      | `RegexPattern
          of regex_pattern_fields
      | `TuplePattern
          of tuple_pattern_fields
      | `TypePattern
          of type_pattern_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and property_pattern_detail =
    [
    | `PropertyPatternDetail
        of property_pattern_detail_fields
    ]
  and property_pattern_detail_fields = 
  {
         
    f_call: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `KeepExpr
          of keep_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
    ]
    Lazy.t;
         
    f_expected_value: [
      | `BindingPattern
          of binding_pattern_fields
      | `BoolPatternFalse
          of bool_pattern_false_fields
      | `BoolPatternTrue
          of bool_pattern_true_fields
      | `ExtendedPattern
          of extended_pattern_fields
      | `FilteredPattern
          of filtered_pattern_fields
      | `IntegerPattern
          of integer_pattern_fields
      | `ListPattern
          of list_pattern_fields
      | `NotPattern
          of not_pattern_fields
      | `NullPattern
          of null_pattern_fields
      | `OrPattern
          of or_pattern_fields
      | `ParenPattern
          of paren_pattern_fields
      | `RegexPattern
          of regex_pattern_fields
      | `TuplePattern
          of tuple_pattern_fields
      | `TypePattern
          of type_pattern_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and selector_pattern_detail =
    [
    | `SelectorPatternDetail
        of selector_pattern_detail_fields
    ]
  and selector_pattern_detail_fields = 
  {
         
    f_call: selector_call
    Lazy.t;
         
    f_sub_pattern: [
      | `BindingPattern
          of binding_pattern_fields
      | `BoolPatternFalse
          of bool_pattern_false_fields
      | `BoolPatternTrue
          of bool_pattern_true_fields
      | `ExtendedPattern
          of extended_pattern_fields
      | `FilteredPattern
          of filtered_pattern_fields
      | `IntegerPattern
          of integer_pattern_fields
      | `ListPattern
          of list_pattern_fields
      | `NotPattern
          of not_pattern_fields
      | `NullPattern
          of null_pattern_fields
      | `OrPattern
          of or_pattern_fields
      | `ParenPattern
          of paren_pattern_fields
      | `RegexPattern
          of regex_pattern_fields
      | `TuplePattern
          of tuple_pattern_fields
      | `TypePattern
          of type_pattern_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and selector_call =
    [
    | `SelectorCall
        of selector_call_fields
    ]
  and selector_call_fields = 
  {
         
    f_quantifier: id
    Lazy.t;
         
    f_binding: id
    option
    Lazy.t;
         
    f_selector_call: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `KeepExpr
          of keep_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and type_ref =
    [
    | `DefaultListTypeRef
        of default_list_type_ref_fields
    | `FunctionTypeRef
        of function_type_ref_fields
    | `GenericTypeRef
        of generic_type_ref_fields
    | `SimpleTypeRef
        of simple_type_ref_fields
    ]

  
   
  and default_list_type_ref =
    [
    | `DefaultListTypeRef
        of default_list_type_ref_fields
    ]
  and default_list_type_ref_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and function_type_ref =
    [
    | `FunctionTypeRef
        of function_type_ref_fields
    ]
  and function_type_ref_fields = 
  {
         
    f_param_types: type_ref_list
    Lazy.t;
         
    f_return_type: [
      | `FunctionTypeRef
          of function_type_ref_fields
      | `GenericTypeRef
          of generic_type_ref_fields
      | `SimpleTypeRef
          of simple_type_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and generic_type_ref =
    [
    | `GenericTypeRef
        of generic_type_ref_fields
    ]
  and generic_type_ref_fields = 
  {
         
    f_type_name: [
      | `DotExpr
          of dot_expr_fields
      | `RefId
          of ref_id_fields
    ]
    Lazy.t;
         
    f_args: type_ref_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and simple_type_ref =
    [
    | `SimpleTypeRef
        of simple_type_ref_fields
    ]
  and simple_type_ref_fields = 
  {
         
    f_type_name: [
      | `DotExpr
          of dot_expr_fields
      | `RefId
          of ref_id_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and var_bind =
    [
    | `VarBind
        of var_bind_fields
    ]
  and var_bind_fields = 
  {
         
    f_name: ref_id
    Lazy.t;
         
    f_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }



module Entity : sig
  type t = entity

  val info : t -> entity_info
end

module AnalysisUnit : sig
  (**
  * This type represents the analysis of a single file.
  *
  * This type has strong-reference semantics and is ref-counted. Furthermore, a
  * reference to a unit contains an implicit reference to the context that owns
  * it. This means that keeping a reference to a unit will keep the context and
  * all the unit it contains allocated.
  *)

  type t = analysis_unit

  val root : t -> lkt_node option
  (**
  * Return the root node for this unit, or ``None`` if there is none.
  *)

  val diagnostics : t -> Diagnostic.t list
  (**
   * Diagnostics for this unit.
   *)

  val filename : t -> string
  (**
  * Return the filename this unit is associated to.
  *)

  val reparse : ?charset:string -> ?buffer:string -> t -> unit
  (**
  * Reparse an analysis unit from a buffer, if provided, or from the original
  * file otherwise. If ``Charset`` is empty or ``None``, use the last charset
  * successfuly used for this unit, otherwise use it to decode the content of
  * the source file.
  *
  * If any failure occurs, such as decoding, lexing or parsing failure,
  * diagnostic are emitted to explain what happened.
  *)

  val first_token : t -> Token.t option
  (**
  * Return a reference to the first token scanned in this unit.
  *)

  val last_token : t -> Token.t option
  (**
  * Return a reference to the last token scanned in this unit.
  *)

  val token_count : t -> int
  (**
  * Return the number of tokens in this unit.
  *)

  val trivia_count : t -> int
  (**
  * Return the number of trivias in this unit. This is 0 for units that were
  * parsed with trivia analysis disabled.
  *)

  
  val fold_tokens : ('a -> Token.t -> 'a) -> 'a -> t -> 'a
  (**
   * Fold all the token this node contains by calling f on each token.
   *)

  val iter_tokens : (Token.t -> unit) -> t-> unit
  (**
   * Iterate over all token this node contains by calling f on each token.
   *)

  val map_tokens : (Token.t -> 'a) -> t -> 'a list
  (**
   * Map each token calling the given function
   *)

  val tokens : t -> Token.t list
  (**
   * Return a list of tokens for the given node
   *)

end

module AnalysisContext : sig
  (**
  * This type represents a context for all source analysis. This is the first
  * type you need to create to use Liblktlang. It will contain the results of
  * all analysis, and is the main holder for all the data.
  *
  * You can create several analysis contexts if you need to, which enables you,
  * for example to:
  *
  * * analyze several different projects at the same time;
  *
  * * analyze different parts of the same projects in parallel.
  *
  * In the current design, contexts always keep all of their analysis units
  * allocated. If you need to get this memory released, the only option at your
  * disposal is to destroy your analysis context instance.
  *)

  type t = analysis_context

  val create :
    ?charset:string
    -> ?with_trivia:bool
    -> ?tab_stop:int
    -> ?unit_provider:UnitProvider.t
    -> ?file_reader:FileReader.t
    -> unit
    -> t
  (**
  * Create a new analysis context.
  *
  * ``Charset`` will be used as a default charset to decode input sources in
  * analysis units. Please see ``GNATCOLL.Iconv`` for several supported
  * charsets. Be careful: passing an unsupported charset is not guaranteed to
  * raise an error here. If no charset is provided, ``"utf-8"`` is the default.
  *
  * .. TODO: Passing an unsupported charset here is not guaranteed to raise an
  *    error right here, but this would be really helpful for users.
  *
  * When ``With_Trivia`` is true, the parsed analysis units will contain
  * trivias.
  *
  * If provided, ``File_Reader`` will be used to fetch the contents of source
  * files instead of the default, which is to just read it from the filesystem
  * and decode it using the regular charset rules. Note that if provided, all
  * parsing APIs that provide a buffer are forbidden, and any use of the
  * rewriting API with the returned context is rejected.
  *
  * If provided, ``Unit_Provider`` will be used to query the file name that
  * corresponds to a unit reference during semantic analysis. If it is
  * ``None``, the default one is used instead.
  *
  * If provided, ``Event_Handler`` will be notified when various events happen.
  *
  * ``Tab_Stop`` is a positive number to describe the effect of tabulation
  * characters on the column number in source files.
  *)

  val get_from_file :
    ?charset:string
    -> ?reparse:bool
    -> ?grammar_rule:GrammarRule.t
    -> t
    -> string
    -> AnalysisUnit.t
  (**
  * Create a new analysis unit for ``Filename`` or return the existing one if
  * any. If ``Reparse`` is true and the analysis unit already exists, reparse
  * it from ``Filename``.
  *
  * ``Rule`` controls which grammar rule is used to parse the unit.
  *
  * Use ``Charset`` in order to decode the source. If ``Charset`` is empty then
  * use the context's default charset.
  *
  * If any failure occurs, such as file opening, decoding, lexing or parsing
  * failure, return an analysis unit anyway: errors are described as
  * diagnostics of the returned analysis unit.
  *)

  val get_from_buffer :
    ?charset:string
    -> ?grammar_rule:GrammarRule.t
    -> t
    -> string
    -> string
    -> AnalysisUnit.t
  (**
  * Create a new analysis unit for ``Filename`` or return the existing one if
  * any. Whether the analysis unit already exists or not, (re)parse it from the
  * source code in ``Buffer``.
  *
  * ``Rule`` controls which grammar rule is used to parse the unit.
  *
  * Use ``Charset`` in order to decode the source. If ``Charset`` is empty then
  * use the context's default charset.
  *
  * If any failure occurs, such as file opening, decoding, lexing or parsing
  * failure, return an analysis unit anyway: errors are described as
  * diagnostics of the returned analysis unit.
  *)
end





module CompleteItem : sig
  (**
  * Completion item for language servers
  *)

  
  type t = {
      declaration :
         decl option;
  }

end


module DecodedCharValue : sig
  (**
  * Result for ``CharLit.p_denoted_value``.
  *
  * If that property is successful, set ``has_error`` to false and ``value`` to
  * the decoded character value. Otherwise, set ``has_error`` to true and
  * ``error_sloc`` and ``error_message`` to give information about the decoding
  * failure.
  *)

  
  type t = {
      value :
         string;
      has_error :
         bool;
      error_sloc :
         Sloc.t;
      error_message :
         string;
  }

end


module DecodedStringValue : sig
  (**
  * Result for ``StringLit.p_denoted_value``.
  *
  * If that property is successful, set ``has_error`` to false and ``value`` to
  * the decoded string value. Otherwise, set ``has_error`` to true and
  * ``error_sloc`` and ``error_message`` to give information about the decoding
  * failure.
  *)

  
  type t = {
      value :
         string;
      has_error :
         bool;
      error_sloc :
         Sloc.t;
      error_message :
         string;
  }

end


module LogicContext : sig
  (**
  * Describes an interpretation of a reference. Can be attached to logic atoms
  * (e.g. Binds) to indicate under which interpretation this particular atom
  * was produced, which can in turn be used to produce informative diagnostics
  * for resolution failures.
  *)

  
  type t = {
      ref_node :
         lkt_node option;
      decl_node :
         lkt_node option;
  }

end


module RefResult : sig
  (**
  * Reference result struct
  *)

  
  type t = {
      ref_ :
         ref_id option;
  }

end


module SolverDiagnostic : sig
  (**
  * A raw diagnostic produced by a solver resolution failure. This contains as
  * much information as possible to allow formatters down the chain to
  * filter/choose which diagnostics to show among a set of diagnostics produced
  * for a single equation.
  *
  * * ``Message_Template`` is a string explaining the error, which may contain
  *   holes represented by the ``{}`` characters. Literal opening braces are
  *   encoded as ``{{``.
  *
  * * ``Args`` is an array of nodes, which are to be plugged in the holes of
  *   the template in the same order (i.e. the first argument goes into the
  *   first hole of the template, etc.).
  *
  * * ``Location`` is a node which indicates the location of the error.
  *
  * * ``Contexts`` is the array of contexts that were deemed relevant for this
  *   error.
  *
  * * ``Round`` is the solver round during which this diagnostic was emitted.
  *)

  
  type t = {
      message_template :
         string;
      args :
         lkt_node list;
      location :
         BareNode.t;
      contexts :
         LogicContext.t list;
      round :
         int;
  }

end


module SolverResult : sig
  (**
  * A pair returned by the ``Solve_With_Diagnostic`` primitive, consisting of:
  *
  * * A ``Success`` field indicating whether resolution was successful or not.
  *
  * * A ``Diagnostics`` field containing an array of diagnostics which may be
  *   non-empty if ``Success`` is ``False``.
  *)

  
  type t = {
      success :
         bool;
      diagnostics :
         SolverDiagnostic.t list;
  }

end


type _ node =
  | LktNode :
      lkt_node node
  | Argument :
      argument node
  | BaseLexerCaseRuleAlt :
      base_lexer_case_rule_alt node
  | LexerCaseRuleCondAlt :
      lexer_case_rule_cond_alt node
  | LexerCaseRuleDefaultAlt :
      lexer_case_rule_default_alt node
  | BaseMatchBranch :
      base_match_branch node
  | MatchBranch :
      match_branch node
  | PatternMatchBranch :
      pattern_match_branch node
  | BlockExprClause :
      block_expr_clause node
  | BlockStringLine :
      block_string_line node
  | ClassQualifier :
      class_qualifier node
  | ClassQualifierAbsent :
      class_qualifier_absent node
  | ClassQualifierPresent :
      class_qualifier_present node
  | Decl :
      decl node
  | BaseGrammarRuleDecl :
      base_grammar_rule_decl node
  | GrammarRuleDecl :
      grammar_rule_decl node
  | SyntheticLexerDecl :
      synthetic_lexer_decl node
  | BaseValDecl :
      base_val_decl node
  | NodeDecl :
      node_decl node
  | SelfDecl :
      self_decl node
  | UserValDecl :
      user_val_decl node
  | BindingValDecl :
      binding_val_decl node
  | EnumLitDecl :
      enum_lit_decl node
  | ExplicitlyTypedDecl :
      explicitly_typed_decl node
  | ComponentDecl :
      component_decl node
  | FieldDecl :
      field_decl node
  | FunParamDecl :
      fun_param_decl node
  | LambdaParamDecl :
      lambda_param_decl node
  | DynVarDecl :
      dyn_var_decl node
  | MatchValDecl :
      match_val_decl node
  | ValDecl :
      val_decl node
  | FunDecl :
      fun_decl node
  | EnvSpecDecl :
      env_spec_decl node
  | ErrorDecl :
      error_decl node
  | GenericDecl :
      generic_decl node
  | GrammarDecl :
      grammar_decl node
  | LexerDecl :
      lexer_decl node
  | LexerFamilyDecl :
      lexer_family_decl node
  | SynthFunDecl :
      synth_fun_decl node
  | SynthParamDecl :
      synth_param_decl node
  | TypeDecl :
      type_decl node
  | AnyTypeDecl :
      any_type_decl node
  | EnumClassAltDecl :
      enum_class_alt_decl node
  | FunctionType :
      function_type node
  | GenericParamTypeDecl :
      generic_param_type_decl node
  | NamedTypeDecl :
      named_type_decl node
  | BasicClassDecl :
      basic_class_decl node
  | ClassDecl :
      class_decl node
  | EnumClassDecl :
      enum_class_decl node
  | EnumTypeDecl :
      enum_type_decl node
  | StructDecl :
      struct_decl node
  | TraitDecl :
      trait_decl node
  | DeclAnnotation :
      decl_annotation node
  | DeclAnnotationArgs :
      decl_annotation_args node
  | DynEnvWrapper :
      dyn_env_wrapper node
  | ElsifBranch :
      elsif_branch node
  | EnumClassCase :
      enum_class_case node
  | ExcludesNull :
      excludes_null node
  | ExcludesNullAbsent :
      excludes_null_absent node
  | ExcludesNullPresent :
      excludes_null_present node
  | Expr :
      expr node
  | AnyOf :
      any_of node
  | ArrayLiteral :
      array_literal node
  | BaseCallExpr :
      base_call_expr node
  | CallExpr :
      call_expr node
  | LogicCallExpr :
      logic_call_expr node
  | LogicPredicate :
      logic_predicate node
  | LogicPropagateCall :
      logic_propagate_call node
  | BinOp :
      bin_op node
  | BlockExpr :
      block_expr node
  | CastExpr :
      cast_expr node
  | DotExpr :
      dot_expr node
  | ErrorOnNull :
      error_on_null node
  | GenericInstantiation :
      generic_instantiation node
  | GrammarExpr :
      grammar_expr node
  | GrammarCut :
      grammar_cut node
  | GrammarDiscard :
      grammar_discard node
  | GrammarDontSkip :
      grammar_dont_skip node
  | GrammarList :
      grammar_list node
  | GrammarNull :
      grammar_null node
  | GrammarOpt :
      grammar_opt node
  | GrammarOptError :
      grammar_opt_error node
  | GrammarOptErrorGroup :
      grammar_opt_error_group node
  | GrammarOptGroup :
      grammar_opt_group node
  | GrammarOrExpr :
      grammar_or_expr node
  | GrammarPick :
      grammar_pick node
  | GrammarImplicitPick :
      grammar_implicit_pick node
  | GrammarPredicate :
      grammar_predicate node
  | GrammarRuleRef :
      grammar_rule_ref node
  | GrammarSkip :
      grammar_skip node
  | GrammarStopCut :
      grammar_stop_cut node
  | ParseNodeExpr :
      parse_node_expr node
  | TokenLit :
      token_lit node
  | TokenNoCaseLit :
      token_no_case_lit node
  | TokenPatternConcat :
      token_pattern_concat node
  | TokenPatternLit :
      token_pattern_lit node
  | TokenRef :
      token_ref node
  | Id :
      id node
  | DefId :
      def_id node
  | ModuleRefId :
      module_ref_id node
  | RefId :
      ref_id node
  | IfExpr :
      if_expr node
  | Isa :
      isa node
  | KeepExpr :
      keep_expr node
  | LambdaExpr :
      lambda_expr node
  | Lit :
      lit node
  | BigNumLit :
      big_num_lit node
  | CharLit :
      char_lit node
  | NullLit :
      null_lit node
  | NumLit :
      num_lit node
  | StringLit :
      string_lit node
  | BlockStringLit :
      block_string_lit node
  | SingleLineStringLit :
      single_line_string_lit node
  | PatternSingleLineStringLit :
      pattern_single_line_string_lit node
  | LogicAssign :
      logic_assign node
  | LogicExpr :
      logic_expr node
  | LogicPropagate :
      logic_propagate node
  | LogicUnify :
      logic_unify node
  | MatchExpr :
      match_expr node
  | NotExpr :
      not_expr node
  | ParenExpr :
      paren_expr node
  | RaiseExpr :
      raise_expr node
  | SubscriptExpr :
      subscript_expr node
  | TryExpr :
      try_expr node
  | UnOp :
      un_op node
  | FullDecl :
      full_decl node
  | GrammarListSep :
      grammar_list_sep node
  | Import :
      import node
  | LangkitRoot :
      langkit_root node
  | LexerCaseRule :
      lexer_case_rule node
  | LexerCaseRuleSend :
      lexer_case_rule_send node
  | ListKind :
      list_kind node
  | ListKindOne :
      list_kind_one node
  | ListKindZero :
      list_kind_zero node
  | LktNodeBaseList :
      lkt_node_base_list node
  | ArgumentList :
      argument_list node
  | BaseLexerCaseRuleAltList :
      base_lexer_case_rule_alt_list node
  | BaseMatchBranchList :
      base_match_branch_list node
  | BlockStringLineList :
      block_string_line_list node
  | CallExprList :
      call_expr_list node
  | DeclAnnotationList :
      decl_annotation_list node
  | ElsifBranchList :
      elsif_branch_list node
  | EnumClassAltDeclList :
      enum_class_alt_decl_list node
  | EnumClassCaseList :
      enum_class_case_list node
  | EnumLitDeclList :
      enum_lit_decl_list node
  | ExprList :
      expr_list node
  | AnyOfList :
      any_of_list node
  | FullDeclList :
      full_decl_list node
  | DeclBlock :
      decl_block node
  | GenericParamDeclList :
      generic_param_decl_list node
  | FunParamDeclList :
      fun_param_decl_list node
  | GrammarExprList :
      grammar_expr_list node
  | GrammarExprListList :
      grammar_expr_list_list node
  | ImportList :
      import_list node
  | LambdaParamDeclList :
      lambda_param_decl_list node
  | LktNodeList :
      lkt_node_list node
  | PatternDetailList :
      pattern_detail_list node
  | PatternList :
      pattern_list node
  | RefIdList :
      ref_id_list node
  | TypeRefList :
      type_ref_list node
  | SyntheticTypeRefList :
      synthetic_type_ref_list node
  | NullCondQualifier :
      null_cond_qualifier node
  | NullCondQualifierAbsent :
      null_cond_qualifier_absent node
  | NullCondQualifierPresent :
      null_cond_qualifier_present node
  | Op :
      op node
  | OpAmp :
      op_amp node
  | OpAnd :
      op_and node
  | OpDiv :
      op_div node
  | OpEq :
      op_eq node
  | OpGt :
      op_gt node
  | OpGte :
      op_gte node
  | OpLogicAnd :
      op_logic_and node
  | OpLogicOr :
      op_logic_or node
  | OpLt :
      op_lt node
  | OpLte :
      op_lte node
  | OpMinus :
      op_minus node
  | OpMult :
      op_mult node
  | OpNe :
      op_ne node
  | OpOr :
      op_or node
  | OpOrInt :
      op_or_int node
  | OpPlus :
      op_plus node
  | Pattern :
      pattern node
  | AnyTypePattern :
      any_type_pattern node
  | BindingPattern :
      binding_pattern node
  | BoolPattern :
      bool_pattern node
  | BoolPatternFalse :
      bool_pattern_false node
  | BoolPatternTrue :
      bool_pattern_true node
  | EllipsisPattern :
      ellipsis_pattern node
  | ExtendedPattern :
      extended_pattern node
  | FilteredPattern :
      filtered_pattern node
  | IntegerPattern :
      integer_pattern node
  | ListPattern :
      list_pattern node
  | NotPattern :
      not_pattern node
  | NullPattern :
      null_pattern node
  | OrPattern :
      or_pattern node
  | ParenPattern :
      paren_pattern node
  | RegexPattern :
      regex_pattern node
  | TuplePattern :
      tuple_pattern node
  | TypePattern :
      type_pattern node
  | PatternDetail :
      pattern_detail node
  | FieldPatternDetail :
      field_pattern_detail node
  | PropertyPatternDetail :
      property_pattern_detail node
  | SelectorPatternDetail :
      selector_pattern_detail node
  | SelectorCall :
      selector_call node
  | TypeRef :
      type_ref node
  | DefaultListTypeRef :
      default_list_type_ref node
  | FunctionTypeRef :
      function_type_ref node
  | GenericTypeRef :
      generic_type_ref node
  | SimpleTypeRef :
      simple_type_ref node
  | VarBind :
      var_bind node

module VarBind : sig
  (**
  * Dynamic var bind expression.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `VarBind of
          var_bind_fields
    ]

  type fields = var_bind_fields =
    
  {
         
    f_name: ref_id
    Lazy.t;
         
    f_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_name :
    [< var_bind]
    -> ref_id

      
  val f_expr :
    [< var_bind]
    -> [any_of | array_literal | bin_op | block_expr | call_expr | cast_expr | dot_expr | error_on_null | generic_instantiation | if_expr | isa | keep_expr | lambda_expr | lit | logic_assign | logic_expr | logic_predicate | logic_propagate | logic_unify | match_expr | not_expr | paren_expr | raise_expr | ref_id | subscript_expr | try_expr | un_op]


end

module SimpleTypeRef : sig
  (**
  * Simple reference to a type.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `SimpleTypeRef of
          simple_type_ref_fields
    ]

  type fields = simple_type_ref_fields =
    
  {
         
    f_type_name: [
      | `DotExpr
          of dot_expr_fields
      | `RefId
          of ref_id_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_type_name :
    [< simple_type_ref]
    -> [dot_expr | ref_id]


end

module GenericTypeRef : sig
  (**
  * Reference to a generic type.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `GenericTypeRef of
          generic_type_ref_fields
    ]

  type fields = generic_type_ref_fields =
    
  {
         
    f_type_name: [
      | `DotExpr
          of dot_expr_fields
      | `RefId
          of ref_id_fields
    ]
    Lazy.t;
         
    f_args: type_ref_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_type_name :
    [< generic_type_ref]
    -> [dot_expr | ref_id]

      
  val f_args :
    [< generic_type_ref]
    -> type_ref_list


end

module FunctionTypeRef : sig
  (**
  * Reference to a function type.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `FunctionTypeRef of
          function_type_ref_fields
    ]

  type fields = function_type_ref_fields =
    
  {
         
    f_param_types: type_ref_list
    Lazy.t;
         
    f_return_type: [
      | `FunctionTypeRef
          of function_type_ref_fields
      | `GenericTypeRef
          of generic_type_ref_fields
      | `SimpleTypeRef
          of simple_type_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_param_types :
    [< function_type_ref]
    -> type_ref_list

      
  val f_return_type :
    [< function_type_ref]
    -> [function_type_ref | generic_type_ref | simple_type_ref]


end

module DefaultListTypeRef : sig
  (**
  * "list" type reference in parsers.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `DefaultListTypeRef of
          default_list_type_ref_fields
    ]

  type fields = default_list_type_ref_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module TypeRef : sig
  (**
  * Base class for a reference to a type.
  *
  * Derived nodes: ``default_list_type_ref``, ``function_type_ref``,
  * ``generic_type_ref``, ``simple_type_ref``
  *)

  type t =
    [
      | DefaultListTypeRef.t
      | FunctionTypeRef.t
      | GenericTypeRef.t
      | SimpleTypeRef.t
    ]


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int


      
  val p_referenced_decl :
    [< type_ref ]
    -> type_decl option
  (**
  * Returns the referenced type declaration.
  *)



end

module SelectorCall : sig
  (**
  * Root node for selector patterns
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `SelectorCall of
          selector_call_fields
    ]

  type fields = selector_call_fields =
    
  {
         
    f_quantifier: id
    Lazy.t;
         
    f_binding: id
    option
    Lazy.t;
         
    f_selector_call: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `KeepExpr
          of keep_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_quantifier :
    [< selector_call]
    -> id

      
  val f_binding :
    [< selector_call]
    -> id option

      
  val f_selector_call :
    [< selector_call]
    -> [array_literal | block_expr | call_expr | cast_expr | dot_expr | error_on_null | generic_instantiation | keep_expr | lit | logic_expr | logic_predicate | match_expr | paren_expr | ref_id | subscript_expr]


end

module SelectorPatternDetail : sig
  (**
  * Pattern detail denoting the use of a selector in a node pattern
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `SelectorPatternDetail of
          selector_pattern_detail_fields
    ]

  type fields = selector_pattern_detail_fields =
    
  {
         
    f_call: selector_call
    Lazy.t;
         
    f_sub_pattern: [
      | `BindingPattern
          of binding_pattern_fields
      | `BoolPatternFalse
          of bool_pattern_false_fields
      | `BoolPatternTrue
          of bool_pattern_true_fields
      | `ExtendedPattern
          of extended_pattern_fields
      | `FilteredPattern
          of filtered_pattern_fields
      | `IntegerPattern
          of integer_pattern_fields
      | `ListPattern
          of list_pattern_fields
      | `NotPattern
          of not_pattern_fields
      | `NullPattern
          of null_pattern_fields
      | `OrPattern
          of or_pattern_fields
      | `ParenPattern
          of paren_pattern_fields
      | `RegexPattern
          of regex_pattern_fields
      | `TuplePattern
          of tuple_pattern_fields
      | `TypePattern
          of type_pattern_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_call :
    [< selector_pattern_detail]
    -> selector_call

      
  val f_sub_pattern :
    [< selector_pattern_detail]
    -> [binding_pattern | bool_pattern | extended_pattern | filtered_pattern | integer_pattern | list_pattern | not_pattern | null_pattern | or_pattern | paren_pattern | regex_pattern | tuple_pattern | type_pattern]


end

module PropertyPatternDetail : sig
  (**
  * Pattern detail denoting an access to a property in a node pattern.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `PropertyPatternDetail of
          property_pattern_detail_fields
    ]

  type fields = property_pattern_detail_fields =
    
  {
         
    f_call: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `KeepExpr
          of keep_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
    ]
    Lazy.t;
         
    f_expected_value: [
      | `BindingPattern
          of binding_pattern_fields
      | `BoolPatternFalse
          of bool_pattern_false_fields
      | `BoolPatternTrue
          of bool_pattern_true_fields
      | `ExtendedPattern
          of extended_pattern_fields
      | `FilteredPattern
          of filtered_pattern_fields
      | `IntegerPattern
          of integer_pattern_fields
      | `ListPattern
          of list_pattern_fields
      | `NotPattern
          of not_pattern_fields
      | `NullPattern
          of null_pattern_fields
      | `OrPattern
          of or_pattern_fields
      | `ParenPattern
          of paren_pattern_fields
      | `RegexPattern
          of regex_pattern_fields
      | `TuplePattern
          of tuple_pattern_fields
      | `TypePattern
          of type_pattern_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_call :
    [< property_pattern_detail]
    -> [array_literal | block_expr | call_expr | cast_expr | dot_expr | error_on_null | generic_instantiation | keep_expr | lit | logic_expr | logic_predicate | match_expr | paren_expr | ref_id | subscript_expr]

      
  val f_expected_value :
    [< property_pattern_detail]
    -> [binding_pattern | bool_pattern | extended_pattern | filtered_pattern | integer_pattern | list_pattern | not_pattern | null_pattern | or_pattern | paren_pattern | regex_pattern | tuple_pattern | type_pattern]


end

module FieldPatternDetail : sig
  (**
  * Pattern detail denoting an access to a field.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `FieldPatternDetail of
          field_pattern_detail_fields
    ]

  type fields = field_pattern_detail_fields =
    
  {
         
    f_id: id
    Lazy.t;
         
    f_expected_value: [
      | `BindingPattern
          of binding_pattern_fields
      | `BoolPatternFalse
          of bool_pattern_false_fields
      | `BoolPatternTrue
          of bool_pattern_true_fields
      | `ExtendedPattern
          of extended_pattern_fields
      | `FilteredPattern
          of filtered_pattern_fields
      | `IntegerPattern
          of integer_pattern_fields
      | `ListPattern
          of list_pattern_fields
      | `NotPattern
          of not_pattern_fields
      | `NullPattern
          of null_pattern_fields
      | `OrPattern
          of or_pattern_fields
      | `ParenPattern
          of paren_pattern_fields
      | `RegexPattern
          of regex_pattern_fields
      | `TuplePattern
          of tuple_pattern_fields
      | `TypePattern
          of type_pattern_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_id :
    [< field_pattern_detail]
    -> id

      
  val f_expected_value :
    [< field_pattern_detail]
    -> [binding_pattern | bool_pattern | extended_pattern | filtered_pattern | integer_pattern | list_pattern | not_pattern | null_pattern | or_pattern | paren_pattern | regex_pattern | tuple_pattern | type_pattern]


end

module PatternDetail : sig
  (**
  * Base class for a detail in an ExtendedPattern.
  *
  * Derived nodes: ``field_pattern_detail``, ``property_pattern_detail``,
  * ``selector_pattern_detail``
  *)

  type t =
    [
      | FieldPatternDetail.t
      | PropertyPatternDetail.t
      | SelectorPatternDetail.t
    ]


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module TypePattern : sig
  (**
  * Pattern matching on a specific type.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `TypePattern of
          type_pattern_fields
    ]

  type fields = type_pattern_fields =
    
  {
         
    f_type_name: [
      | `FunctionTypeRef
          of function_type_ref_fields
      | `GenericTypeRef
          of generic_type_ref_fields
      | `SimpleTypeRef
          of simple_type_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_type_name :
    [< type_pattern]
    -> [function_type_ref | generic_type_ref | simple_type_ref]


end

module TuplePattern : sig
  (**
  * Pattern to match on tuples.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `TuplePattern of
          tuple_pattern_fields
    ]

  type fields = tuple_pattern_fields =
    
  {
         
    f_sub_patterns: pattern_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_sub_patterns :
    [< tuple_pattern]
    -> pattern_list


end

module RegexPattern : sig
  (**
  * Pattern that considers the value as text and matches it against the given
  * regular expression.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `RegexPattern of
          regex_pattern_fields
    ]

  type fields = regex_pattern_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module ParenPattern : sig
  (**
  * A syntactically parenthesized pattern. Has no effect, only used to
  * disambiguate syntax.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `ParenPattern of
          paren_pattern_fields
    ]

  type fields = paren_pattern_fields =
    
  {
         
    f_sub_pattern: [
      | `BindingPattern
          of binding_pattern_fields
      | `BoolPatternFalse
          of bool_pattern_false_fields
      | `BoolPatternTrue
          of bool_pattern_true_fields
      | `ExtendedPattern
          of extended_pattern_fields
      | `FilteredPattern
          of filtered_pattern_fields
      | `IntegerPattern
          of integer_pattern_fields
      | `ListPattern
          of list_pattern_fields
      | `NotPattern
          of not_pattern_fields
      | `NullPattern
          of null_pattern_fields
      | `OrPattern
          of or_pattern_fields
      | `ParenPattern
          of paren_pattern_fields
      | `RegexPattern
          of regex_pattern_fields
      | `TuplePattern
          of tuple_pattern_fields
      | `TypePattern
          of type_pattern_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_sub_pattern :
    [< paren_pattern]
    -> [binding_pattern | bool_pattern | extended_pattern | filtered_pattern | integer_pattern | list_pattern | not_pattern | null_pattern | or_pattern | paren_pattern | regex_pattern | tuple_pattern | type_pattern]


end

module OrPattern : sig
  (**
  * Pattern that matches if any of its subpatterns matches.
  *
  * For instance:
  *
  * .. code::
  *
  *    val value_decls = select ObjectDecl | ParamSpec
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `OrPattern of
          or_pattern_fields
    ]

  type fields = or_pattern_fields =
    
  {
         
    f_left_sub_pattern: [
      | `BindingPattern
          of binding_pattern_fields
      | `BoolPatternFalse
          of bool_pattern_false_fields
      | `BoolPatternTrue
          of bool_pattern_true_fields
      | `ExtendedPattern
          of extended_pattern_fields
      | `FilteredPattern
          of filtered_pattern_fields
      | `IntegerPattern
          of integer_pattern_fields
      | `ListPattern
          of list_pattern_fields
      | `NotPattern
          of not_pattern_fields
      | `NullPattern
          of null_pattern_fields
      | `ParenPattern
          of paren_pattern_fields
      | `RegexPattern
          of regex_pattern_fields
      | `TuplePattern
          of tuple_pattern_fields
      | `TypePattern
          of type_pattern_fields
    ]
    Lazy.t;
         
    f_right_sub_pattern: [
      | `BindingPattern
          of binding_pattern_fields
      | `BoolPatternFalse
          of bool_pattern_false_fields
      | `BoolPatternTrue
          of bool_pattern_true_fields
      | `ExtendedPattern
          of extended_pattern_fields
      | `FilteredPattern
          of filtered_pattern_fields
      | `IntegerPattern
          of integer_pattern_fields
      | `ListPattern
          of list_pattern_fields
      | `NotPattern
          of not_pattern_fields
      | `NullPattern
          of null_pattern_fields
      | `OrPattern
          of or_pattern_fields
      | `ParenPattern
          of paren_pattern_fields
      | `RegexPattern
          of regex_pattern_fields
      | `TuplePattern
          of tuple_pattern_fields
      | `TypePattern
          of type_pattern_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_left_sub_pattern :
    [< or_pattern]
    -> [binding_pattern | bool_pattern | extended_pattern | filtered_pattern | integer_pattern | list_pattern | not_pattern | null_pattern | paren_pattern | regex_pattern | tuple_pattern | type_pattern]

      
  val f_right_sub_pattern :
    [< or_pattern]
    -> [binding_pattern | bool_pattern | extended_pattern | filtered_pattern | integer_pattern | list_pattern | not_pattern | null_pattern | or_pattern | paren_pattern | regex_pattern | tuple_pattern | type_pattern]


end

module NullPattern : sig
  (**
  * Null pattern. Will only match the null value.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `NullPattern of
          null_pattern_fields
    ]

  type fields = null_pattern_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module NotPattern : sig
  (**
  * Pattern that matches if its inner pattern doesn't match.
  *
  * For instance:
  *
  * .. code::
  *
  *    val non_objects = select not ObjectDecl
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `NotPattern of
          not_pattern_fields
    ]

  type fields = not_pattern_fields =
    
  {
         
    f_sub_pattern: [
      | `BindingPattern
          of binding_pattern_fields
      | `BoolPatternFalse
          of bool_pattern_false_fields
      | `BoolPatternTrue
          of bool_pattern_true_fields
      | `ExtendedPattern
          of extended_pattern_fields
      | `IntegerPattern
          of integer_pattern_fields
      | `ListPattern
          of list_pattern_fields
      | `NotPattern
          of not_pattern_fields
      | `NullPattern
          of null_pattern_fields
      | `ParenPattern
          of paren_pattern_fields
      | `RegexPattern
          of regex_pattern_fields
      | `TuplePattern
          of tuple_pattern_fields
      | `TypePattern
          of type_pattern_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_sub_pattern :
    [< not_pattern]
    -> [binding_pattern | bool_pattern | extended_pattern | integer_pattern | list_pattern | not_pattern | null_pattern | paren_pattern | regex_pattern | tuple_pattern | type_pattern]


end

module ListPattern : sig
  (**
  * Pattern to match on lists.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `ListPattern of
          list_pattern_fields
    ]

  type fields = list_pattern_fields =
    
  {
         
    f_sub_patterns: pattern_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_sub_patterns :
    [< list_pattern]
    -> pattern_list


end

module IntegerPattern : sig
  (**
  * Pattern to match on integers.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `IntegerPattern of
          integer_pattern_fields
    ]

  type fields = integer_pattern_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module FilteredPattern : sig
  (**
  * Pattern with a filtering predicate, of the form: ``<pattern> when
  * <predicate>``
  *
  * For instance:
  *
  * .. code::
  *
  *    o@ObjectDecl when o.children.length == 3
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `FilteredPattern of
          filtered_pattern_fields
    ]

  type fields = filtered_pattern_fields =
    
  {
         
    f_sub_pattern: [
      | `BindingPattern
          of binding_pattern_fields
      | `BoolPatternFalse
          of bool_pattern_false_fields
      | `BoolPatternTrue
          of bool_pattern_true_fields
      | `ExtendedPattern
          of extended_pattern_fields
      | `IntegerPattern
          of integer_pattern_fields
      | `ListPattern
          of list_pattern_fields
      | `NotPattern
          of not_pattern_fields
      | `NullPattern
          of null_pattern_fields
      | `ParenPattern
          of paren_pattern_fields
      | `RegexPattern
          of regex_pattern_fields
      | `TuplePattern
          of tuple_pattern_fields
      | `TypePattern
          of type_pattern_fields
    ]
    Lazy.t;
         
    f_predicate: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_sub_pattern :
    [< filtered_pattern]
    -> [binding_pattern | bool_pattern | extended_pattern | integer_pattern | list_pattern | not_pattern | null_pattern | paren_pattern | regex_pattern | tuple_pattern | type_pattern]

      
  val f_predicate :
    [< filtered_pattern]
    -> [any_of | array_literal | bin_op | block_expr | call_expr | cast_expr | dot_expr | error_on_null | generic_instantiation | if_expr | isa | keep_expr | lambda_expr | lit | logic_assign | logic_expr | logic_predicate | logic_propagate | logic_unify | match_expr | not_expr | paren_expr | raise_expr | ref_id | subscript_expr | try_expr | un_op]


end

module ExtendedPattern : sig
  (**
  * Pattern that takes a base pattern, and adds details to match on the shape
  * of what is being matched. The syntactic form is:
  *
  * .. code::
  *
  *    <sub_pattern>(<detail>, <detail>, ...)
  *
  * For instance:
  *
  * .. code::
  *
  *    ObjectDecl(any children: AspectAssoc)
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `ExtendedPattern of
          extended_pattern_fields
    ]

  type fields = extended_pattern_fields =
    
  {
         
    f_sub_pattern: [
      | `AnyTypePattern
          of any_type_pattern_fields
      | `ParenPattern
          of paren_pattern_fields
      | `TypePattern
          of type_pattern_fields
    ]
    Lazy.t;
         
    f_details: pattern_detail_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_sub_pattern :
    [< extended_pattern]
    -> [any_type_pattern | paren_pattern | type_pattern]

      
  val f_details :
    [< extended_pattern]
    -> pattern_detail_list


end

module EllipsisPattern : sig
  (**
  * Pattern to match any remaining number of elements in a list pattern.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `EllipsisPattern of
          ellipsis_pattern_fields
    ]

  type fields = ellipsis_pattern_fields =
    
  {
         
    f_binding: id
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_binding :
    [< ellipsis_pattern]
    -> id option


end

module BoolPatternTrue : sig
  (**
  * This node type has no derivation.
  *)

  type t =
    [
      | `BoolPatternTrue of
          bool_pattern_true_fields
    ]

  type fields = bool_pattern_true_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module BoolPatternFalse : sig
  (**
  * This node type has no derivation.
  *)

  type t =
    [
      | `BoolPatternFalse of
          bool_pattern_false_fields
    ]

  type fields = bool_pattern_false_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module BoolPattern : sig
  (**
  * Pattern to match on booleans.
  *
  * Derived nodes: ``bool_pattern_false``, ``bool_pattern_true``
  *)

  type t =
    [
      | BoolPatternFalse.t
      | BoolPatternTrue.t
    ]


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module BindingPattern : sig
  (**
  * Pattern comprising a binding name and a value pattern.
  *
  * For instance:
  *
  * .. code::
  *
  *    o@ObjectDecl
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `BindingPattern of
          binding_pattern_fields
    ]

  type fields = binding_pattern_fields =
    
  {
         
    f_decl: binding_val_decl
    Lazy.t;
         
    f_sub_pattern: [
      | `BindingPattern
          of binding_pattern_fields
      | `BoolPatternFalse
          of bool_pattern_false_fields
      | `BoolPatternTrue
          of bool_pattern_true_fields
      | `ExtendedPattern
          of extended_pattern_fields
      | `IntegerPattern
          of integer_pattern_fields
      | `ListPattern
          of list_pattern_fields
      | `NotPattern
          of not_pattern_fields
      | `NullPattern
          of null_pattern_fields
      | `ParenPattern
          of paren_pattern_fields
      | `RegexPattern
          of regex_pattern_fields
      | `TuplePattern
          of tuple_pattern_fields
      | `TypePattern
          of type_pattern_fields
    ]
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_decl :
    [< binding_pattern]
    -> binding_val_decl

      
  val f_sub_pattern :
    [< binding_pattern]
    -> [binding_pattern | bool_pattern | extended_pattern | integer_pattern | list_pattern | not_pattern | null_pattern | paren_pattern | regex_pattern | tuple_pattern | type_pattern] option


end

module AnyTypePattern : sig
  (**
  * Pattern that allows to match any type. Only usable as an ExtendedPattern's
  * left side pattern:
  *
  * .. code::
  *
  *    *(f_field: BasicDecl(...))
  *
  * For the general case of matching any value, the idiom is to use a binding
  * pattern with no right hand side:
  *
  * .. code::
  *
  *    _ => true
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `AnyTypePattern of
          any_type_pattern_fields
    ]

  type fields = any_type_pattern_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module Pattern : sig
  (**
  * Root node class for patterns.
  *
  * This is a mostly LKQL specific node for the moment, as are every node
  * derived from it.
  *
  * The only patterns that are currently used and implemented in Lkt's IsA are
  * ``OrPattern`` and ``TypePattern``.
  *
  * Derived nodes: ``any_type_pattern``, ``binding_pattern``, ``bool_pattern``,
  * ``ellipsis_pattern``, ``extended_pattern``, ``filtered_pattern``,
  * ``integer_pattern``, ``list_pattern``, ``not_pattern``, ``null_pattern``,
  * ``or_pattern``, ``paren_pattern``, ``regex_pattern``, ``tuple_pattern``,
  * ``type_pattern``
  *)

  type t =
    [
      | AnyTypePattern.t
      | BindingPattern.t
      | BoolPattern.t
      | EllipsisPattern.t
      | ExtendedPattern.t
      | FilteredPattern.t
      | IntegerPattern.t
      | ListPattern.t
      | NotPattern.t
      | NullPattern.t
      | OrPattern.t
      | ParenPattern.t
      | RegexPattern.t
      | TuplePattern.t
      | TypePattern.t
    ]


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module OpPlus : sig
  (**
  * This node type has no derivation.
  *)

  type t =
    [
      | `OpPlus of
          op_plus_fields
    ]

  type fields = op_plus_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module OpOrInt : sig
  (**
  * This node type has no derivation.
  *)

  type t =
    [
      | `OpOrInt of
          op_or_int_fields
    ]

  type fields = op_or_int_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module OpOr : sig
  (**
  * This node type has no derivation.
  *)

  type t =
    [
      | `OpOr of
          op_or_fields
    ]

  type fields = op_or_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module OpNe : sig
  (**
  * This node type has no derivation.
  *)

  type t =
    [
      | `OpNe of
          op_ne_fields
    ]

  type fields = op_ne_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module OpMult : sig
  (**
  * This node type has no derivation.
  *)

  type t =
    [
      | `OpMult of
          op_mult_fields
    ]

  type fields = op_mult_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module OpMinus : sig
  (**
  * This node type has no derivation.
  *)

  type t =
    [
      | `OpMinus of
          op_minus_fields
    ]

  type fields = op_minus_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module OpLte : sig
  (**
  * This node type has no derivation.
  *)

  type t =
    [
      | `OpLte of
          op_lte_fields
    ]

  type fields = op_lte_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module OpLt : sig
  (**
  * This node type has no derivation.
  *)

  type t =
    [
      | `OpLt of
          op_lt_fields
    ]

  type fields = op_lt_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module OpLogicOr : sig
  (**
  * This node type has no derivation.
  *)

  type t =
    [
      | `OpLogicOr of
          op_logic_or_fields
    ]

  type fields = op_logic_or_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module OpLogicAnd : sig
  (**
  * This node type has no derivation.
  *)

  type t =
    [
      | `OpLogicAnd of
          op_logic_and_fields
    ]

  type fields = op_logic_and_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module OpGte : sig
  (**
  * This node type has no derivation.
  *)

  type t =
    [
      | `OpGte of
          op_gte_fields
    ]

  type fields = op_gte_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module OpGt : sig
  (**
  * This node type has no derivation.
  *)

  type t =
    [
      | `OpGt of
          op_gt_fields
    ]

  type fields = op_gt_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module OpEq : sig
  (**
  * This node type has no derivation.
  *)

  type t =
    [
      | `OpEq of
          op_eq_fields
    ]

  type fields = op_eq_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module OpDiv : sig
  (**
  * This node type has no derivation.
  *)

  type t =
    [
      | `OpDiv of
          op_div_fields
    ]

  type fields = op_div_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module OpAnd : sig
  (**
  * This node type has no derivation.
  *)

  type t =
    [
      | `OpAnd of
          op_and_fields
    ]

  type fields = op_and_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module OpAmp : sig
  (**
  * This node type has no derivation.
  *)

  type t =
    [
      | `OpAmp of
          op_amp_fields
    ]

  type fields = op_amp_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module Op : sig
  (**
  * Operator in a binary operator expression.
  *
  * Derived nodes: ``op_amp``, ``op_and``, ``op_div``, ``op_eq``, ``op_gt``,
  * ``op_gte``, ``op_logic_and``, ``op_logic_or``, ``op_lt``, ``op_lte``,
  * ``op_minus``, ``op_mult``, ``op_ne``, ``op_or_int``, ``op_or``, ``op_plus``
  *)

  type t =
    [
      | OpAmp.t
      | OpAnd.t
      | OpDiv.t
      | OpEq.t
      | OpGt.t
      | OpGte.t
      | OpLogicAnd.t
      | OpLogicOr.t
      | OpLt.t
      | OpLte.t
      | OpMinus.t
      | OpMult.t
      | OpNe.t
      | OpOr.t
      | OpOrInt.t
      | OpPlus.t
    ]


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module NullCondQualifierPresent : sig
  (**
  * This node type has no derivation.
  *)

  type t =
    [
      | `NullCondQualifierPresent of
          null_cond_qualifier_present_fields
    ]

  type fields = null_cond_qualifier_present_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module NullCondQualifierAbsent : sig
  (**
  * This node type has no derivation.
  *)

  type t =
    [
      | `NullCondQualifierAbsent of
          null_cond_qualifier_absent_fields
    ]

  type fields = null_cond_qualifier_absent_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module NullCondQualifier : sig
  (**
  * Whether the "?" operation qualifier (to denote the null-conditional
  * behavior) is present.
  *
  * Derived nodes: ``null_cond_qualifier_absent``,
  * ``null_cond_qualifier_present``
  *)

  type t =
    [
      | NullCondQualifierAbsent.t
      | NullCondQualifierPresent.t
    ]


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int


      
  val p_as_bool :
    [< null_cond_qualifier ]
    -> bool
  (**
  * Return whether this node is present
  *)



end

module SyntheticTypeRefList : sig
  (**
  * Synthetic list of type references, used to create synthetic type
  * declarations.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `SyntheticTypeRefList of
          synthetic_type_ref_list_fields
    ]

  type fields = synthetic_type_ref_list_fields =
    
  {
    list : type_ref list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




  val f_list :
    [< synthetic_type_ref_list]
    -> type_ref list

end

module TypeRefList : sig
  (**
  * List of TypeRef.
  *
  * This list node can contain one of the following nodes:
  * ``function_type_ref``, ``generic_type_ref``, ``simple_type_ref``
  *
  * Derived nodes: ``synthetic_type_ref_list``
  *)

  type t =
    [
      | `TypeRefList of
          type_ref_list_fields
      | `SyntheticTypeRefList of
          synthetic_type_ref_list_fields
    ]

  type fields = type_ref_list_fields =
    
  {
    list : type_ref list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




  val f_list :
    [< type_ref_list]
    -> type_ref list

end

module RefIdList : sig
  (**
  * List of RefId.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `RefIdList of
          ref_id_list_fields
    ]

  type fields = ref_id_list_fields =
    
  {
    list : ref_id list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




  val f_list :
    [< ref_id_list]
    -> ref_id list

end

module PatternList : sig
  (**
  * List of Pattern.
  *
  * This list node can contain one of the following nodes: ``binding_pattern``,
  * ``bool_pattern``, ``ellipsis_pattern``, ``extended_pattern``,
  * ``integer_pattern``, ``list_pattern``, ``not_pattern``, ``null_pattern``,
  * ``paren_pattern``, ``regex_pattern``, ``tuple_pattern``, ``type_pattern``
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `PatternList of
          pattern_list_fields
    ]

  type fields = pattern_list_fields =
    
  {
    list : pattern list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




  val f_list :
    [< pattern_list]
    -> pattern list

end

module PatternDetailList : sig
  (**
  * List of PatternDetail.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `PatternDetailList of
          pattern_detail_list_fields
    ]

  type fields = pattern_detail_list_fields =
    
  {
    list : pattern_detail list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




  val f_list :
    [< pattern_detail_list]
    -> pattern_detail list

end

module LktNodeList : sig
  (**
  * List of LktNode.
  *
  * This list node can contain one of the following nodes: ``any_of``,
  * ``array_literal``, ``bin_op``, ``block_expr_clause``, ``block_expr``,
  * ``call_expr``, ``cast_expr``, ``dot_expr``, ``error_decl``,
  * ``error_on_null``, ``full_decl``, ``generic_instantiation``, ``if_expr``,
  * ``isa``, ``keep_expr``, ``lambda_expr``, ``lexer_case_rule``, ``lit``,
  * ``logic_assign``, ``logic_expr``, ``logic_predicate``, ``logic_propagate``,
  * ``logic_unify``, ``match_expr``, ``not_expr``, ``paren_expr``,
  * ``raise_expr``, ``ref_id``, ``subscript_expr``, ``try_expr``, ``un_op``
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `LktNodeList of
          lkt_node_list_fields
    ]

  type fields = lkt_node_list_fields =
    
  {
    list : lkt_node list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




  val f_list :
    [< lkt_node_list]
    -> lkt_node list

end

module LambdaParamDeclList : sig
  (**
  * List of LambdaParamDecl.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `LambdaParamDeclList of
          lambda_param_decl_list_fields
    ]

  type fields = lambda_param_decl_list_fields =
    
  {
    list : lambda_param_decl list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




  val f_list :
    [< lambda_param_decl_list]
    -> lambda_param_decl list

end

module ImportList : sig
  (**
  * List of Import.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `ImportList of
          import_list_fields
    ]

  type fields = import_list_fields =
    
  {
    list : import list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




  val f_list :
    [< import_list]
    -> import list

end

module GrammarExprListList : sig
  (**
  * List of ASTList[GrammarExpr].
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `GrammarExprListList of
          grammar_expr_list_list_fields
    ]

  type fields = grammar_expr_list_list_fields =
    
  {
    list : grammar_expr_list list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




  val f_list :
    [< grammar_expr_list_list]
    -> grammar_expr_list list

end

module GrammarExprList : sig
  (**
  * List of GrammarExpr.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `GrammarExprList of
          grammar_expr_list_fields
    ]

  type fields = grammar_expr_list_fields =
    
  {
    list : grammar_expr list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




  val f_list :
    [< grammar_expr_list]
    -> grammar_expr list

end

module FunParamDeclList : sig
  (**
  * List of FunParamDecl.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `FunParamDeclList of
          fun_param_decl_list_fields
    ]

  type fields = fun_param_decl_list_fields =
    
  {
    list : fun_param_decl list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




  val f_list :
    [< fun_param_decl_list]
    -> fun_param_decl list

end

module GenericParamDeclList : sig
  (**
  * Comma-separated list of generic parameter types.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `GenericParamDeclList of
          generic_param_decl_list_fields
    ]

  type fields = generic_param_decl_list_fields =
    
  {
    list : full_decl list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




  val f_list :
    [< generic_param_decl_list]
    -> full_decl list

end

module DeclBlock : sig
  (**
  * List of declarations that also introduces a containing lexical scope.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `DeclBlock of
          decl_block_fields
    ]

  type fields = decl_block_fields =
    
  {
    list : full_decl list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




  val f_list :
    [< decl_block]
    -> full_decl list

end

module FullDeclList : sig
  (**
  * List of FullDecl.
  *
  * Derived nodes: ``decl_block``, ``generic_param_decl_list``
  *)

  type t =
    [
      | `FullDeclList of
          full_decl_list_fields
      | `DeclBlock of
          decl_block_fields
      | `GenericParamDeclList of
          generic_param_decl_list_fields
    ]

  type fields = full_decl_list_fields =
    
  {
    list : full_decl list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




  val f_list :
    [< full_decl_list]
    -> full_decl list

end

module AnyOfList : sig
  (**
  * Pipe-separated list of expressions.
  *
  * This is used to represent the "values" operand of an ``AnyOf`` expression.
  *
  * This list node can contain one of the following nodes: ``array_literal``,
  * ``block_expr``, ``call_expr``, ``cast_expr``, ``dot_expr``,
  * ``error_on_null``, ``generic_instantiation``, ``if_expr``, ``keep_expr``,
  * ``lambda_expr``, ``lit``, ``logic_expr``, ``logic_predicate``,
  * ``match_expr``, ``paren_expr``, ``raise_expr``, ``ref_id``,
  * ``subscript_expr``, ``try_expr``
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `AnyOfList of
          any_of_list_fields
    ]

  type fields = any_of_list_fields =
    
  {
    list : expr list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




  val f_list :
    [< any_of_list]
    -> expr list

end

module ExprList : sig
  (**
  * List of Expr.
  *
  * This list node can contain one of the following nodes: ``any_of``,
  * ``array_literal``, ``bin_op``, ``block_expr``, ``call_expr``,
  * ``cast_expr``, ``dot_expr``, ``error_on_null``, ``generic_instantiation``,
  * ``if_expr``, ``isa``, ``keep_expr``, ``lambda_expr``, ``lit``,
  * ``logic_assign``, ``logic_expr``, ``logic_predicate``, ``logic_propagate``,
  * ``logic_unify``, ``match_expr``, ``not_expr``, ``paren_expr``,
  * ``raise_expr``, ``ref_id``, ``subscript_expr``, ``try_expr``, ``un_op``
  *
  * Derived nodes: ``any_of_list``
  *)

  type t =
    [
      | `ExprList of
          expr_list_fields
      | `AnyOfList of
          any_of_list_fields
    ]

  type fields = expr_list_fields =
    
  {
    list : expr list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




  val f_list :
    [< expr_list]
    -> expr list

end

module EnumLitDeclList : sig
  (**
  * List of EnumLitDecl.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `EnumLitDeclList of
          enum_lit_decl_list_fields
    ]

  type fields = enum_lit_decl_list_fields =
    
  {
    list : enum_lit_decl list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




  val f_list :
    [< enum_lit_decl_list]
    -> enum_lit_decl list

end

module EnumClassCaseList : sig
  (**
  * List of EnumClassCase.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `EnumClassCaseList of
          enum_class_case_list_fields
    ]

  type fields = enum_class_case_list_fields =
    
  {
    list : enum_class_case list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




  val f_list :
    [< enum_class_case_list]
    -> enum_class_case list

end

module EnumClassAltDeclList : sig
  (**
  * List of EnumClassAltDecl.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `EnumClassAltDeclList of
          enum_class_alt_decl_list_fields
    ]

  type fields = enum_class_alt_decl_list_fields =
    
  {
    list : enum_class_alt_decl list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




  val f_list :
    [< enum_class_alt_decl_list]
    -> enum_class_alt_decl list

end

module ElsifBranchList : sig
  (**
  * List of ElsifBranch.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `ElsifBranchList of
          elsif_branch_list_fields
    ]

  type fields = elsif_branch_list_fields =
    
  {
    list : elsif_branch list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




  val f_list :
    [< elsif_branch_list]
    -> elsif_branch list

end

module DeclAnnotationList : sig
  (**
  * List of DeclAnnotation.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `DeclAnnotationList of
          decl_annotation_list_fields
    ]

  type fields = decl_annotation_list_fields =
    
  {
    list : decl_annotation list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




  val f_list :
    [< decl_annotation_list]
    -> decl_annotation list

end

module CallExprList : sig
  (**
  * List of CallExpr.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `CallExprList of
          call_expr_list_fields
    ]

  type fields = call_expr_list_fields =
    
  {
    list : call_expr list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




  val f_list :
    [< call_expr_list]
    -> call_expr list

end

module BlockStringLineList : sig
  (**
  * List of BlockStringLine.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `BlockStringLineList of
          block_string_line_list_fields
    ]

  type fields = block_string_line_list_fields =
    
  {
    list : block_string_line list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




  val f_list :
    [< block_string_line_list]
    -> block_string_line list

end

module BaseMatchBranchList : sig
  (**
  * List of BaseMatchBranch.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `BaseMatchBranchList of
          base_match_branch_list_fields
    ]

  type fields = base_match_branch_list_fields =
    
  {
    list : base_match_branch list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




  val f_list :
    [< base_match_branch_list]
    -> base_match_branch list

end

module BaseLexerCaseRuleAltList : sig
  (**
  * List of BaseLexerCaseRuleAlt.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `BaseLexerCaseRuleAltList of
          base_lexer_case_rule_alt_list_fields
    ]

  type fields = base_lexer_case_rule_alt_list_fields =
    
  {
    list : base_lexer_case_rule_alt list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




  val f_list :
    [< base_lexer_case_rule_alt_list]
    -> base_lexer_case_rule_alt list

end

module ArgumentList : sig
  (**
  * List of Argument.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `ArgumentList of
          argument_list_fields
    ]

  type fields = argument_list_fields =
    
  {
    list : argument list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




  val f_list :
    [< argument_list]
    -> argument list

end

module LktNodeBaseList : sig
  (**
  * Derived nodes: ``argument_list``, ``base_lexer_case_rule_alt_list``,
  * ``base_match_branch_list``, ``block_string_line_list``, ``call_expr_list``,
  * ``decl_annotation_list``, ``elsif_branch_list``,
  * ``enum_class_alt_decl_list``, ``enum_class_case_list``,
  * ``enum_lit_decl_list``, ``expr_list``, ``full_decl_list``,
  * ``fun_param_decl_list``, ``grammar_expr_list_list``, ``grammar_expr_list``,
  * ``import_list``, ``lambda_param_decl_list``, ``lkt_node_list``,
  * ``pattern_detail_list``, ``pattern_list``, ``ref_id_list``,
  * ``type_ref_list``
  *)

  type t =
    [
      | ArgumentList.t
      | BaseLexerCaseRuleAltList.t
      | BaseMatchBranchList.t
      | BlockStringLineList.t
      | CallExprList.t
      | DeclAnnotationList.t
      | ElsifBranchList.t
      | EnumClassAltDeclList.t
      | EnumClassCaseList.t
      | EnumLitDeclList.t
      | ExprList.t
      | FullDeclList.t
      | FunParamDeclList.t
      | GrammarExprList.t
      | GrammarExprListList.t
      | ImportList.t
      | LambdaParamDeclList.t
      | LktNodeList.t
      | PatternDetailList.t
      | PatternList.t
      | RefIdList.t
      | TypeRefList.t
    ]


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module ListKindZero : sig
  (**
  * This node type has no derivation.
  *)

  type t =
    [
      | `ListKindZero of
          list_kind_zero_fields
    ]

  type fields = list_kind_zero_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module ListKindOne : sig
  (**
  * This node type has no derivation.
  *)

  type t =
    [
      | `ListKindOne of
          list_kind_one_fields
    ]

  type fields = list_kind_one_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module ListKind : sig
  (**
  * Kind for list parser expressions.
  *
  * Derived nodes: ``list_kind_one``, ``list_kind_zero``
  *)

  type t =
    [
      | ListKindOne.t
      | ListKindZero.t
    ]


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module LexerCaseRuleSend : sig
  (**
  * Lexer construction used by case alternatives to represent the token to send
  * if that alternative was chosen.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `LexerCaseRuleSend of
          lexer_case_rule_send_fields
    ]

  type fields = lexer_case_rule_send_fields =
    
  {
         
    f_sent: ref_id
    Lazy.t;
         
    f_match_size: num_lit
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_sent :
    [< lexer_case_rule_send]
    -> ref_id

      
  val f_match_size :
    [< lexer_case_rule_send]
    -> num_lit


end

module LexerCaseRule : sig
  (**
  * Lexer construct to introduce a conditional lexing action.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `LexerCaseRule of
          lexer_case_rule_fields
    ]

  type fields = lexer_case_rule_fields =
    
  {
         
    f_expr: [
      | `GrammarCut
          of grammar_cut_fields
      | `GrammarDiscard
          of grammar_discard_fields
      | `GrammarImplicitPick
          of grammar_implicit_pick_fields
      | `GrammarList
          of grammar_list_fields
      | `GrammarNull
          of grammar_null_fields
      | `GrammarOpt
          of grammar_opt_fields
      | `GrammarOptError
          of grammar_opt_error_fields
      | `GrammarOptErrorGroup
          of grammar_opt_error_group_fields
      | `GrammarOptGroup
          of grammar_opt_group_fields
      | `GrammarOrExpr
          of grammar_or_expr_fields
      | `GrammarPick
          of grammar_pick_fields
      | `GrammarRuleRef
          of grammar_rule_ref_fields
      | `GrammarSkip
          of grammar_skip_fields
      | `GrammarStopCut
          of grammar_stop_cut_fields
      | `ParseNodeExpr
          of parse_node_expr_fields
      | `TokenLit
          of token_lit_fields
      | `TokenNoCaseLit
          of token_no_case_lit_fields
      | `TokenPatternConcat
          of token_pattern_concat_fields
      | `TokenPatternLit
          of token_pattern_lit_fields
      | `TokenRef
          of token_ref_fields
    ]
    Lazy.t;
         
    f_alts: base_lexer_case_rule_alt_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_expr :
    [< lexer_case_rule]
    -> [grammar_cut | grammar_discard | grammar_list | grammar_null | grammar_opt | grammar_opt_error | grammar_opt_error_group | grammar_opt_group | grammar_or_expr | grammar_pick | grammar_rule_ref | grammar_skip | grammar_stop_cut | parse_node_expr | token_lit | token_no_case_lit | token_pattern_concat | token_pattern_lit | token_ref]

      
  val f_alts :
    [< lexer_case_rule]
    -> base_lexer_case_rule_alt_list


end

module LangkitRoot : sig
  (**
  * For the moment, root node of a lkt compilation unit.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `LangkitRoot of
          langkit_root_fields
    ]

  type fields = langkit_root_fields =
    
  {
         
    f_imports: import_list
    Lazy.t;
         
    f_decls: full_decl_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int


      
  val p_fetch_prelude :
    [< langkit_root ]
    -> analysis_unit
  (**
  * External property that will fetch the prelude unit, containing predefined
  * types and values.
  *)


      
  val f_imports :
    [< langkit_root]
    -> import_list

      
  val f_decls :
    [< langkit_root]
    -> full_decl_list


end

module Import : sig
  (**
  * Statement to import another source file.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `Import of
          import_fields
    ]

  type fields = import_fields =
    
  {
         
    f_name: module_ref_id
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int


      
  val p_referenced_unit :
    [< import ]
    -> analysis_unit
  (**
  * Return the unit that this import statements designates. Load it if needed.
  *)


      
  val f_name :
    [< import]
    -> module_ref_id


end

module GrammarListSep : sig
  (**
  * Specification for the separator of a list parser.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `GrammarListSep of
          grammar_list_sep_fields
    ]

  type fields = grammar_list_sep_fields =
    
  {
         
    f_token: grammar_expr
    Lazy.t;
         
    f_extra: id
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_token :
    [< grammar_list_sep]
    -> grammar_expr

      
  val f_extra :
    [< grammar_list_sep]
    -> id option


end

module FullDecl : sig
  (**
  * Container for an lkt declaration. Contains the decl node plus the
  * documentation and annotations.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `FullDecl of
          full_decl_fields
    ]

  type fields = full_decl_fields =
    
  {
         
    f_doc: string_lit
    option
    Lazy.t;
         
    f_decl_annotations: decl_annotation_list
    Lazy.t;
         
    f_decl: [
      | `ClassDecl
          of class_decl_fields
      | `DynVarDecl
          of dyn_var_decl_fields
      | `EnumClassDecl
          of enum_class_decl_fields
      | `EnumTypeDecl
          of enum_type_decl_fields
      | `EnvSpecDecl
          of env_spec_decl_fields
      | `FieldDecl
          of field_decl_fields
      | `FunDecl
          of fun_decl_fields
      | `GenericDecl
          of generic_decl_fields
      | `GenericParamTypeDecl
          of generic_param_type_decl_fields
      | `GrammarDecl
          of grammar_decl_fields
      | `GrammarRuleDecl
          of grammar_rule_decl_fields
      | `LexerDecl
          of lexer_decl_fields
      | `LexerFamilyDecl
          of lexer_family_decl_fields
      | `StructDecl
          of struct_decl_fields
      | `TraitDecl
          of trait_decl_fields
      | `ValDecl
          of val_decl_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int


      
  val p_has_annotation :
    [< full_decl ]
    -> string
    -> bool
  (**
  * Return whether this node has an annotation with name ``name``.
  *)


      
  val f_doc :
    [< full_decl]
    -> string_lit option

      
  val f_decl_annotations :
    [< full_decl]
    -> decl_annotation_list

      
  val f_decl :
    [< full_decl]
    -> [dyn_var_decl | env_spec_decl | field_decl | fun_decl | generic_decl | generic_param_type_decl | grammar_decl | grammar_rule_decl | lexer_decl | lexer_family_decl | named_type_decl | val_decl]


end

module UnOp : sig
  (**
  * Unary operator expression.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `UnOp of
          un_op_fields
    ]

  type fields = un_op_fields =
    
  {
         
    f_op: [
      | `OpMinus
          of op_minus_fields
      | `OpPlus
          of op_plus_fields
    ]
    Lazy.t;
         
    f_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_op :
    [< un_op]
    -> [op_minus | op_plus]

      
  val f_expr :
    [< un_op]
    -> [any_of | array_literal | block_expr | call_expr | cast_expr | dot_expr | error_on_null | generic_instantiation | if_expr | isa | keep_expr | lambda_expr | lit | logic_assign | logic_expr | logic_predicate | logic_propagate | logic_unify | match_expr | paren_expr | raise_expr | ref_id | subscript_expr | try_expr]


end

module TryExpr : sig
  (**
  * Try expression.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `TryExpr of
          try_expr_fields
    ]

  type fields = try_expr_fields =
    
  {
         
    f_try_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
         
    f_or_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_try_expr :
    [< try_expr]
    -> [any_of | array_literal | bin_op | block_expr | call_expr | cast_expr | dot_expr | error_on_null | generic_instantiation | if_expr | isa | keep_expr | lambda_expr | lit | logic_assign | logic_expr | logic_predicate | logic_propagate | logic_unify | match_expr | not_expr | paren_expr | raise_expr | ref_id | subscript_expr | try_expr | un_op]

      
  val f_or_expr :
    [< try_expr]
    -> [any_of | array_literal | bin_op | block_expr | call_expr | cast_expr | dot_expr | error_on_null | generic_instantiation | if_expr | isa | keep_expr | lambda_expr | lit | logic_assign | logic_expr | logic_predicate | logic_propagate | logic_unify | match_expr | not_expr | paren_expr | raise_expr | ref_id | subscript_expr | try_expr | un_op] option


end

module SubscriptExpr : sig
  (**
  * Array subscript expression.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `SubscriptExpr of
          subscript_expr_fields
    ]

  type fields = subscript_expr_fields =
    
  {
         
    f_prefix: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `KeepExpr
          of keep_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
    ]
    Lazy.t;
         
    f_null_cond: null_cond_qualifier
    Lazy.t;
         
    f_index: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_prefix :
    [< subscript_expr]
    -> [array_literal | block_expr | call_expr | cast_expr | dot_expr | error_on_null | generic_instantiation | keep_expr | lit | logic_expr | logic_predicate | match_expr | paren_expr | ref_id | subscript_expr]

      
  val f_null_cond :
    [< subscript_expr]
    -> null_cond_qualifier

      
  val f_index :
    [< subscript_expr]
    -> [any_of | array_literal | bin_op | block_expr | call_expr | cast_expr | dot_expr | error_on_null | generic_instantiation | if_expr | isa | keep_expr | lambda_expr | lit | logic_assign | logic_expr | logic_predicate | logic_propagate | logic_unify | match_expr | not_expr | paren_expr | raise_expr | ref_id | subscript_expr | try_expr | un_op]


end

module RaiseExpr : sig
  (**
  * Raise expression.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `RaiseExpr of
          raise_expr_fields
    ]

  type fields = raise_expr_fields =
    
  {
         
    f_dest_type: [
      | `FunctionTypeRef
          of function_type_ref_fields
      | `GenericTypeRef
          of generic_type_ref_fields
      | `SimpleTypeRef
          of simple_type_ref_fields
    ]
    option
    Lazy.t;
         
    f_except_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_dest_type :
    [< raise_expr]
    -> [function_type_ref | generic_type_ref | simple_type_ref] option

      
  val f_except_expr :
    [< raise_expr]
    -> [any_of | array_literal | bin_op | block_expr | call_expr | cast_expr | dot_expr | error_on_null | generic_instantiation | if_expr | isa | keep_expr | lambda_expr | lit | logic_assign | logic_expr | logic_predicate | logic_propagate | logic_unify | match_expr | not_expr | paren_expr | raise_expr | ref_id | subscript_expr | try_expr | un_op]


end

module ParenExpr : sig
  (**
  * Parenthesized expression.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `ParenExpr of
          paren_expr_fields
    ]

  type fields = paren_expr_fields =
    
  {
         
    f_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_expr :
    [< paren_expr]
    -> [any_of | array_literal | bin_op | block_expr | call_expr | cast_expr | dot_expr | error_on_null | generic_instantiation | if_expr | isa | keep_expr | lambda_expr | lit | logic_assign | logic_expr | logic_predicate | logic_propagate | logic_unify | match_expr | not_expr | paren_expr | raise_expr | ref_id | subscript_expr | try_expr | un_op]


end

module NotExpr : sig
  (**
  * Boolean negation expression.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `NotExpr of
          not_expr_fields
    ]

  type fields = not_expr_fields =
    
  {
         
    f_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_expr :
    [< not_expr]
    -> [any_of | array_literal | bin_op | block_expr | call_expr | cast_expr | dot_expr | error_on_null | generic_instantiation | if_expr | isa | keep_expr | lambda_expr | lit | logic_assign | logic_expr | logic_predicate | logic_propagate | logic_unify | match_expr | paren_expr | raise_expr | ref_id | subscript_expr | try_expr | un_op]


end

module MatchExpr : sig
  (**
  * Binary operator expression.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `MatchExpr of
          match_expr_fields
    ]

  type fields = match_expr_fields =
    
  {
         
    f_match_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
         
    f_branches: base_match_branch_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_match_expr :
    [< match_expr]
    -> [any_of | array_literal | bin_op | block_expr | call_expr | cast_expr | dot_expr | error_on_null | generic_instantiation | if_expr | isa | keep_expr | lambda_expr | lit | logic_assign | logic_expr | logic_predicate | logic_propagate | logic_unify | match_expr | not_expr | paren_expr | raise_expr | ref_id | subscript_expr | try_expr | un_op]

      
  val f_branches :
    [< match_expr]
    -> base_match_branch_list


end

module LogicUnify : sig
  (**
  * Class for "unify" equations.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `LogicUnify of
          logic_unify_fields
    ]

  type fields = logic_unify_fields =
    
  {
         
    f_lhs: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
    ]
    Lazy.t;
         
    f_rhs: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_lhs :
    [< logic_unify]
    -> [any_of | array_literal | block_expr | call_expr | cast_expr | dot_expr | error_on_null | generic_instantiation | if_expr | isa | keep_expr | lambda_expr | lit | logic_assign | logic_expr | logic_predicate | logic_propagate | logic_unify | match_expr | paren_expr | raise_expr | ref_id | subscript_expr | try_expr]

      
  val f_rhs :
    [< logic_unify]
    -> [array_literal | block_expr | call_expr | cast_expr | dot_expr | error_on_null | generic_instantiation | if_expr | keep_expr | lambda_expr | lit | logic_expr | logic_predicate | match_expr | paren_expr | raise_expr | ref_id | subscript_expr | try_expr]


end

module LogicPropagate : sig
  (**
  * Class for "propagate" equations.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `LogicPropagate of
          logic_propagate_fields
    ]

  type fields = logic_propagate_fields =
    
  {
         
    f_dest_var: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
    ]
    Lazy.t;
         
    f_call: logic_propagate_call
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_dest_var :
    [< logic_propagate]
    -> [any_of | array_literal | block_expr | call_expr | cast_expr | dot_expr | error_on_null | generic_instantiation | if_expr | isa | keep_expr | lambda_expr | lit | logic_assign | logic_expr | logic_predicate | logic_propagate | logic_unify | match_expr | paren_expr | raise_expr | ref_id | subscript_expr | try_expr]

      
  val f_call :
    [< logic_propagate]
    -> logic_propagate_call


end

module LogicExpr : sig
  (**
  * Class for logic expressions (any ``basic_expr`` starting with %).
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `LogicExpr of
          logic_expr_fields
    ]

  type fields = logic_expr_fields =
    
  {
         
    f_expr: [
      | `CallExpr
          of call_expr_fields
      | `RefId
          of ref_id_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_expr :
    [< logic_expr]
    -> [call_expr | ref_id]


end

module LogicAssign : sig
  (**
  * Class for "assign to logic var" equations.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `LogicAssign of
          logic_assign_fields
    ]

  type fields = logic_assign_fields =
    
  {
         
    f_dest_var: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
    ]
    Lazy.t;
         
    f_value: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_dest_var :
    [< logic_assign]
    -> [any_of | array_literal | block_expr | call_expr | cast_expr | dot_expr | error_on_null | generic_instantiation | if_expr | isa | keep_expr | lambda_expr | lit | logic_assign | logic_expr | logic_predicate | logic_propagate | logic_unify | match_expr | paren_expr | raise_expr | ref_id | subscript_expr | try_expr]

      
  val f_value :
    [< logic_assign]
    -> [array_literal | block_expr | call_expr | cast_expr | dot_expr | error_on_null | generic_instantiation | if_expr | keep_expr | lambda_expr | lit | logic_expr | logic_predicate | match_expr | paren_expr | raise_expr | ref_id | subscript_expr | try_expr]


end

module PatternSingleLineStringLit : sig
  (**
  * Pattern single line string literal expression.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `PatternSingleLineStringLit of
          pattern_single_line_string_lit_fields
    ]

  type fields = pattern_single_line_string_lit_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module SingleLineStringLit : sig
  (**
  * Single line string literal expression.
  *
  * Note that in order to reduce the size of the node type hierarchy, we define
  * only one node (StringLit) for all our string literals (only regular strings
  * and pattern string literals at the moment). This will also make it easy to
  * add new string prefixes in the future.
  *
  * Derived nodes: ``pattern_single_line_string_lit``
  *)

  type t =
    [
      | `SingleLineStringLit of
          single_line_string_lit_fields
      | `PatternSingleLineStringLit of
          pattern_single_line_string_lit_fields
    ]

  type fields = single_line_string_lit_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module BlockStringLit : sig
  (**
  * String literal expression, made of multiple line strings.
  *
  * The denoted string value is the concatenation of all line string items.
  * Each line string item must be either:
  *
  * * The empty string designator (``|"``), to denote an empty line (``\n``).
  *
  * * ``|" <content>``, to designate a non-empty line. The space before
  *   ``<content>`` is mandatory, and is not included in the denoted string
  *   value. ``<content>`` can be anything that appear in a regular string
  *   literal: escape sequences are interpreted the same way.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `BlockStringLit of
          block_string_lit_fields
    ]

  type fields = block_string_lit_fields =
    
  {
         
    f_lines: block_string_line_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_lines :
    [< block_string_lit]
    -> block_string_line_list


end

module StringLit : sig
  (**
  * Base node type for string literals.
  *
  * Derived nodes: ``block_string_lit``, ``single_line_string_lit``
  *)

  type t =
    [
      | BlockStringLit.t
      | SingleLineStringLit.t
    ]


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int


      
  val p_denoted_value :
    [< string_lit ]
    -> DecodedStringValue.t
  (**
  * Return the content of the given string literal node.
  *)

      
  val p_is_prefixed_string :
    [< string_lit ]
    -> bool
  (**
  * Return whether this string is prefixed or not.
  *)

      
  val p_prefix :
    [< string_lit ]
    -> string
  (**
  * Return the prefix of this string, or the null character if there is no
  * prefix.
  *)

      
  val p_is_regexp_literal :
    [< string_lit ]
    -> bool
  (**
  * Return whether this string literal is actually a regexp literal, by
  * checking that this string is prefixed by 'p'.
  *)



end

module NumLit : sig
  (**
  * Number literal expression.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `NumLit of
          num_lit_fields
    ]

  type fields = num_lit_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module NullLit : sig
  (**
  * Null literal expression.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `NullLit of
          null_lit_fields
    ]

  type fields = null_lit_fields =
    
  {
         
    f_dest_type: [
      | `FunctionTypeRef
          of function_type_ref_fields
      | `GenericTypeRef
          of generic_type_ref_fields
      | `SimpleTypeRef
          of simple_type_ref_fields
    ]
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_dest_type :
    [< null_lit]
    -> [function_type_ref | generic_type_ref | simple_type_ref] option


end

module CharLit : sig
  (**
  * Character literal expression.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `CharLit of
          char_lit_fields
    ]

  type fields = char_lit_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int


      
  val p_denoted_value :
    [< char_lit ]
    -> DecodedCharValue.t
  (**
  * Return the content of the given character literal node.
  *)



end

module BigNumLit : sig
  (**
  * Big number literal expression.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `BigNumLit of
          big_num_lit_fields
    ]

  type fields = big_num_lit_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module Lit : sig
  (**
  * Base class for literals.
  *
  * Derived nodes: ``big_num_lit``, ``char_lit``, ``null_lit``, ``num_lit``,
  * ``string_lit``
  *)

  type t =
    [
      | BigNumLit.t
      | CharLit.t
      | NullLit.t
      | NumLit.t
      | StringLit.t
    ]


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module LambdaExpr : sig
  (**
  * Lambda expression.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `LambdaExpr of
          lambda_expr_fields
    ]

  type fields = lambda_expr_fields =
    
  {
         
    f_params: lambda_param_decl_list
    Lazy.t;
         
    f_return_type: [
      | `FunctionTypeRef
          of function_type_ref_fields
      | `GenericTypeRef
          of generic_type_ref_fields
      | `SimpleTypeRef
          of simple_type_ref_fields
    ]
    option
    Lazy.t;
         
    f_body: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_params :
    [< lambda_expr]
    -> lambda_param_decl_list

      
  val f_return_type :
    [< lambda_expr]
    -> [function_type_ref | generic_type_ref | simple_type_ref] option

      
  val f_body :
    [< lambda_expr]
    -> [any_of | array_literal | bin_op | block_expr | call_expr | cast_expr | dot_expr | error_on_null | generic_instantiation | if_expr | isa | keep_expr | lambda_expr | lit | logic_assign | logic_expr | logic_predicate | logic_propagate | logic_unify | match_expr | not_expr | paren_expr | raise_expr | ref_id | subscript_expr | try_expr | un_op]


end

module KeepExpr : sig
  (**
  * Keep expression.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `KeepExpr of
          keep_expr_fields
    ]

  type fields = keep_expr_fields =
    
  {
         
    f_expr: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `KeepExpr
          of keep_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
    ]
    Lazy.t;
         
    f_null_cond: null_cond_qualifier
    Lazy.t;
         
    f_keep_type: [
      | `FunctionTypeRef
          of function_type_ref_fields
      | `GenericTypeRef
          of generic_type_ref_fields
      | `SimpleTypeRef
          of simple_type_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_expr :
    [< keep_expr]
    -> [array_literal | block_expr | call_expr | cast_expr | dot_expr | error_on_null | generic_instantiation | keep_expr | lit | logic_expr | logic_predicate | match_expr | paren_expr | ref_id | subscript_expr]

      
  val f_null_cond :
    [< keep_expr]
    -> null_cond_qualifier

      
  val f_keep_type :
    [< keep_expr]
    -> [function_type_ref | generic_type_ref | simple_type_ref]


end

module Isa : sig
  (**
  * Isa expression.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `Isa of
          isa_fields
    ]

  type fields = isa_fields =
    
  {
         
    f_expr: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
    ]
    Lazy.t;
         
    f_pattern: [
      | `BindingPattern
          of binding_pattern_fields
      | `BoolPatternFalse
          of bool_pattern_false_fields
      | `BoolPatternTrue
          of bool_pattern_true_fields
      | `ExtendedPattern
          of extended_pattern_fields
      | `FilteredPattern
          of filtered_pattern_fields
      | `IntegerPattern
          of integer_pattern_fields
      | `ListPattern
          of list_pattern_fields
      | `NotPattern
          of not_pattern_fields
      | `NullPattern
          of null_pattern_fields
      | `OrPattern
          of or_pattern_fields
      | `ParenPattern
          of paren_pattern_fields
      | `RegexPattern
          of regex_pattern_fields
      | `TuplePattern
          of tuple_pattern_fields
      | `TypePattern
          of type_pattern_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_expr :
    [< isa]
    -> [array_literal | block_expr | call_expr | cast_expr | dot_expr | error_on_null | generic_instantiation | if_expr | keep_expr | lambda_expr | lit | logic_expr | logic_predicate | match_expr | paren_expr | raise_expr | ref_id | subscript_expr | try_expr]

      
  val f_pattern :
    [< isa]
    -> [binding_pattern | bool_pattern | extended_pattern | filtered_pattern | integer_pattern | list_pattern | not_pattern | null_pattern | or_pattern | paren_pattern | regex_pattern | tuple_pattern | type_pattern]


end

module IfExpr : sig
  (**
  * If expression.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `IfExpr of
          if_expr_fields
    ]

  type fields = if_expr_fields =
    
  {
         
    f_cond_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
         
    f_then_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
         
    f_alternatives: elsif_branch_list
    Lazy.t;
         
    f_else_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_cond_expr :
    [< if_expr]
    -> [any_of | array_literal | bin_op | block_expr | call_expr | cast_expr | dot_expr | error_on_null | generic_instantiation | if_expr | isa | keep_expr | lambda_expr | lit | logic_assign | logic_expr | logic_predicate | logic_propagate | logic_unify | match_expr | not_expr | paren_expr | raise_expr | ref_id | subscript_expr | try_expr | un_op]

      
  val f_then_expr :
    [< if_expr]
    -> [any_of | array_literal | bin_op | block_expr | call_expr | cast_expr | dot_expr | error_on_null | generic_instantiation | if_expr | isa | keep_expr | lambda_expr | lit | logic_assign | logic_expr | logic_predicate | logic_propagate | logic_unify | match_expr | not_expr | paren_expr | raise_expr | ref_id | subscript_expr | try_expr | un_op]

      
  val f_alternatives :
    [< if_expr]
    -> elsif_branch_list

      
  val f_else_expr :
    [< if_expr]
    -> [any_of | array_literal | bin_op | block_expr | call_expr | cast_expr | dot_expr | error_on_null | generic_instantiation | if_expr | isa | keep_expr | lambda_expr | lit | logic_assign | logic_expr | logic_predicate | logic_propagate | logic_unify | match_expr | not_expr | paren_expr | raise_expr | ref_id | subscript_expr | try_expr | un_op]


end

module RefId : sig
  (**
  * Reference identifier.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `RefId of
          ref_id_fields
    ]

  type fields = ref_id_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int


      
  val p_referenced_defining_name :
    [< ref_id ]
    -> def_id option
  (**
  * Return the referenced defining name.
  *)



end

module ModuleRefId : sig
  (**
  * Id referencing a langkit module.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `ModuleRefId of
          module_ref_id_fields
    ]

  type fields = module_ref_id_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module DefId : sig
  (**
  * Defining identifier.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `DefId of
          def_id_fields
    ]

  type fields = def_id_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int


      
  val p_name :
    [< def_id ]
    -> string
  (**
  * Return the name defined by this DefId.
  *)

      
  val p_get_implementatinons :
    [< def_id ]
    -> analysis_unit list
    -> def_id list
  (**
  * Return the implementations of this name.
  *)

      
  val p_decl_detail :
    [< def_id ]
    -> string
  (**
  * Return the details to display in the language server client when it
  * requests for completion or hovering information.
  *)

      
  val p_completion_item_kind :
    [< def_id ]
    -> int
  (**
  * Return the kind of completion item for this DefId.
  *)

      
  val p_doc :
    [< def_id ]
    -> string
  (**
  * Return the documentation associated to this DefId.
  *)

      
  val p_find_all_references :
    [< def_id ]
    -> analysis_unit list
    -> RefResult.t list
  (**
  * Return the list of all RefId that refer to this DefId.
  *)



end

module Id : sig
  (**
  * Identifier.
  *
  * Derived nodes: ``def_id``, ``module_ref_id``, ``ref_id``
  *)

  type t =
    [
      | `Id of
          id_fields
      | `DefId of
          def_id_fields
      | `ModuleRefId of
          module_ref_id_fields
      | `RefId of
          ref_id_fields
    ]

  type fields = id_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int


      
  val p_custom_image :
    [< id ]
    -> string
  (**
  * Returns the image of this RefId using entity information.
  *)



end

module TokenRef : sig
  (**
  * Grammar expression for a token reference.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `TokenRef of
          token_ref_fields
    ]

  type fields = token_ref_fields =
    
  {
         
    f_token_name: ref_id
    Lazy.t;
         
    f_expr: token_lit
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_token_name :
    [< token_ref]
    -> ref_id

      
  val f_expr :
    [< token_ref]
    -> token_lit option


end

module TokenPatternLit : sig
  (**
  * Grammar expression for a pattern literal.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `TokenPatternLit of
          token_pattern_lit_fields
    ]

  type fields = token_pattern_lit_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int


      
  val p_denoted_value :
    [< token_pattern_lit ]
    -> DecodedStringValue.t
  (**
  * Return the content of the given token pattern literal node.
  *)



end

module TokenPatternConcat : sig
  (**
  * Grammar expression for the concatenation of two patterns.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `TokenPatternConcat of
          token_pattern_concat_fields
    ]

  type fields = token_pattern_concat_fields =
    
  {
         
    f_left: [
      | `TokenPatternConcat
          of token_pattern_concat_fields
      | `TokenPatternLit
          of token_pattern_lit_fields
    ]
    Lazy.t;
         
    f_right: token_pattern_lit
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_left :
    [< token_pattern_concat]
    -> [token_pattern_concat | token_pattern_lit]

      
  val f_right :
    [< token_pattern_concat]
    -> token_pattern_lit


end

module TokenNoCaseLit : sig
  (**
  * Grammar expression for a case insensitive token literal.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `TokenNoCaseLit of
          token_no_case_lit_fields
    ]

  type fields = token_no_case_lit_fields =
    
  {
         
    f_lit: token_lit
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_lit :
    [< token_no_case_lit]
    -> token_lit


end

module TokenLit : sig
  (**
  * Grammar expression for a token literal.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `TokenLit of
          token_lit_fields
    ]

  type fields = token_lit_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int


      
  val p_denoted_value :
    [< token_lit ]
    -> DecodedStringValue.t
  (**
  * Return the content of the given token literal node.
  *)



end

module ParseNodeExpr : sig
  (**
  * Expression for the parsing of a Node.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `ParseNodeExpr of
          parse_node_expr_fields
    ]

  type fields = parse_node_expr_fields =
    
  {
         
    f_node_name: [
      | `FunctionTypeRef
          of function_type_ref_fields
      | `GenericTypeRef
          of generic_type_ref_fields
      | `SimpleTypeRef
          of simple_type_ref_fields
    ]
    Lazy.t;
         
    f_sub_exprs: grammar_expr_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_node_name :
    [< parse_node_expr]
    -> [function_type_ref | generic_type_ref | simple_type_ref]

      
  val f_sub_exprs :
    [< parse_node_expr]
    -> grammar_expr_list


end

module GrammarStopCut : sig
  (**
  * Grammar expression for a StopCut.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `GrammarStopCut of
          grammar_stop_cut_fields
    ]

  type fields = grammar_stop_cut_fields =
    
  {
         
    f_expr: grammar_expr
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_expr :
    [< grammar_stop_cut]
    -> grammar_expr


end

module GrammarSkip : sig
  (**
  * Grammar expression (error recovery) to skip a parsing result.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `GrammarSkip of
          grammar_skip_fields
    ]

  type fields = grammar_skip_fields =
    
  {
         
    f_name: [
      | `FunctionTypeRef
          of function_type_ref_fields
      | `GenericTypeRef
          of generic_type_ref_fields
      | `SimpleTypeRef
          of simple_type_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_name :
    [< grammar_skip]
    -> [function_type_ref | generic_type_ref | simple_type_ref]


end

module GrammarRuleRef : sig
  (**
  * Grammar expression for a reference to another grammar rule.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `GrammarRuleRef of
          grammar_rule_ref_fields
    ]

  type fields = grammar_rule_ref_fields =
    
  {
         
    f_node_name: ref_id
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_node_name :
    [< grammar_rule_ref]
    -> ref_id


end

module GrammarPredicate : sig
  (**
  * Grammar expression for a predicate: Only parse something if the predicate
  * (that is a reference to a node property) returns True.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `GrammarPredicate of
          grammar_predicate_fields
    ]

  type fields = grammar_predicate_fields =
    
  {
         
    f_expr: grammar_expr
    Lazy.t;
         
    f_prop_ref: [
      | `DotExpr
          of dot_expr_fields
      | `RefId
          of ref_id_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_expr :
    [< grammar_predicate]
    -> grammar_expr

      
  val f_prop_ref :
    [< grammar_predicate]
    -> [dot_expr | ref_id]


end

module GrammarImplicitPick : sig
  (**
  * Implicit pick operation.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `GrammarImplicitPick of
          grammar_implicit_pick_fields
    ]

  type fields = grammar_implicit_pick_fields =
    
  {
         
    f_exprs: grammar_expr_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_exprs :
    [< grammar_implicit_pick]
    -> grammar_expr_list


end

module GrammarPick : sig
  (**
  * Grammar expression to pick the significant parse out of a list of parses
  * (will automatically discard token results).
  *
  * Derived nodes: ``grammar_implicit_pick``
  *)

  type t =
    [
      | `GrammarPick of
          grammar_pick_fields
      | `GrammarImplicitPick of
          grammar_implicit_pick_fields
    ]

  type fields = grammar_pick_fields =
    
  {
         
    f_exprs: grammar_expr_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_exprs :
    [< grammar_pick]
    -> grammar_expr_list


end

module GrammarOrExpr : sig
  (**
  * Grammar ``Or`` expression (disjunctive choice between several grammar
  * options).
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `GrammarOrExpr of
          grammar_or_expr_fields
    ]

  type fields = grammar_or_expr_fields =
    
  {
         
    f_sub_exprs: grammar_expr_list_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_sub_exprs :
    [< grammar_or_expr]
    -> grammar_expr_list_list


end

module GrammarOptGroup : sig
  (**
  * Grammar expression for a group of optional parsing results.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `GrammarOptGroup of
          grammar_opt_group_fields
    ]

  type fields = grammar_opt_group_fields =
    
  {
         
    f_expr: grammar_expr_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_expr :
    [< grammar_opt_group]
    -> grammar_expr_list


end

module GrammarOptErrorGroup : sig
  (**
  * Grammar expression for a group of optional parsing results. Failure to
  * parse an optional result creates an error, but parsing continues.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `GrammarOptErrorGroup of
          grammar_opt_error_group_fields
    ]

  type fields = grammar_opt_error_group_fields =
    
  {
         
    f_expr: grammar_expr_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_expr :
    [< grammar_opt_error_group]
    -> grammar_expr_list


end

module GrammarOptError : sig
  (**
  * Grammar expression for an optional parsing result. Missing result creates
  * an error, but parsing continues.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `GrammarOptError of
          grammar_opt_error_fields
    ]

  type fields = grammar_opt_error_fields =
    
  {
         
    f_expr: grammar_expr
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_expr :
    [< grammar_opt_error]
    -> grammar_expr


end

module GrammarOpt : sig
  (**
  * Grammar expression for an optional parsing result.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `GrammarOpt of
          grammar_opt_fields
    ]

  type fields = grammar_opt_fields =
    
  {
         
    f_expr: grammar_expr
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_expr :
    [< grammar_opt]
    -> grammar_expr


end

module GrammarNull : sig
  (**
  * Grammar expression to parse a null node.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `GrammarNull of
          grammar_null_fields
    ]

  type fields = grammar_null_fields =
    
  {
         
    f_name: [
      | `FunctionTypeRef
          of function_type_ref_fields
      | `GenericTypeRef
          of generic_type_ref_fields
      | `SimpleTypeRef
          of simple_type_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_name :
    [< grammar_null]
    -> [function_type_ref | generic_type_ref | simple_type_ref]


end

module GrammarList : sig
  (**
  * Grammar expression to parse lists of results. Results can be separated by a
  * separator. List can be empty ('*') or not ('+').
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `GrammarList of
          grammar_list_fields
    ]

  type fields = grammar_list_fields =
    
  {
         
    f_list_type: type_ref
    Lazy.t;
         
    f_kind: list_kind
    Lazy.t;
         
    f_expr: grammar_expr
    Lazy.t;
         
    f_sep: grammar_list_sep
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_list_type :
    [< grammar_list]
    -> type_ref

      
  val f_kind :
    [< grammar_list]
    -> list_kind

      
  val f_expr :
    [< grammar_list]
    -> grammar_expr

      
  val f_sep :
    [< grammar_list]
    -> grammar_list_sep option


end

module GrammarDontSkip : sig
  (**
  * Grammar expression (error recovery) to ensure that any nested skip parser
  * calls won't skip certain parse results.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `GrammarDontSkip of
          grammar_dont_skip_fields
    ]

  type fields = grammar_dont_skip_fields =
    
  {
         
    f_expr: grammar_expr
    Lazy.t;
         
    f_dont_skip: grammar_expr
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_expr :
    [< grammar_dont_skip]
    -> grammar_expr

      
  val f_dont_skip :
    [< grammar_dont_skip]
    -> grammar_expr


end

module GrammarDiscard : sig
  (**
  * Grammar expression to discard the match.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `GrammarDiscard of
          grammar_discard_fields
    ]

  type fields = grammar_discard_fields =
    
  {
         
    f_expr: grammar_expr
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_expr :
    [< grammar_discard]
    -> grammar_expr


end

module GrammarCut : sig
  (**
  * Grammar expression for a cut.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `GrammarCut of
          grammar_cut_fields
    ]

  type fields = grammar_cut_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module GrammarExpr : sig
  (**
  * Base class for expressions related to grammars.
  *
  * Derived nodes: ``grammar_cut``, ``grammar_discard``, ``grammar_dont_skip``,
  * ``grammar_list``, ``grammar_null``, ``grammar_opt_error_group``,
  * ``grammar_opt_error``, ``grammar_opt_group``, ``grammar_opt``,
  * ``grammar_or_expr``, ``grammar_pick``, ``grammar_predicate``,
  * ``grammar_rule_ref``, ``grammar_skip``, ``grammar_stop_cut``,
  * ``parse_node_expr``, ``token_lit``, ``token_no_case_lit``,
  * ``token_pattern_concat``, ``token_pattern_lit``, ``token_ref``
  *)

  type t =
    [
      | GrammarCut.t
      | GrammarDiscard.t
      | GrammarDontSkip.t
      | GrammarList.t
      | GrammarNull.t
      | GrammarOpt.t
      | GrammarOptError.t
      | GrammarOptErrorGroup.t
      | GrammarOptGroup.t
      | GrammarOrExpr.t
      | GrammarPick.t
      | GrammarPredicate.t
      | GrammarRuleRef.t
      | GrammarSkip.t
      | GrammarStopCut.t
      | ParseNodeExpr.t
      | TokenLit.t
      | TokenNoCaseLit.t
      | TokenPatternConcat.t
      | TokenPatternLit.t
      | TokenRef.t
    ]


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module GenericInstantiation : sig
  (**
  * Generic instantiation.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `GenericInstantiation of
          generic_instantiation_fields
    ]

  type fields = generic_instantiation_fields =
    
  {
         
    f_name: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `KeepExpr
          of keep_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
    ]
    Lazy.t;
         
    f_args: type_ref_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_name :
    [< generic_instantiation]
    -> [array_literal | block_expr | call_expr | cast_expr | dot_expr | error_on_null | generic_instantiation | keep_expr | lit | logic_expr | logic_predicate | match_expr | paren_expr | ref_id | subscript_expr]

      
  val f_args :
    [< generic_instantiation]
    -> type_ref_list


end

module ErrorOnNull : sig
  (**
  * Expression that throws an error if LHS is null.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `ErrorOnNull of
          error_on_null_fields
    ]

  type fields = error_on_null_fields =
    
  {
         
    f_expr: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `KeepExpr
          of keep_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_expr :
    [< error_on_null]
    -> [array_literal | block_expr | call_expr | cast_expr | dot_expr | error_on_null | generic_instantiation | keep_expr | lit | logic_expr | logic_predicate | match_expr | paren_expr | ref_id | subscript_expr]


end

module DotExpr : sig
  (**
  * Dotted expression.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `DotExpr of
          dot_expr_fields
    ]

  type fields = dot_expr_fields =
    
  {
         
    f_prefix: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `KeepExpr
          of keep_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
    ]
    Lazy.t;
         
    f_null_cond: null_cond_qualifier
    Lazy.t;
         
    f_suffix: ref_id
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_prefix :
    [< dot_expr]
    -> [array_literal | block_expr | call_expr | cast_expr | dot_expr | error_on_null | generic_instantiation | keep_expr | lit | logic_expr | logic_predicate | match_expr | paren_expr | ref_id | subscript_expr]

      
  val f_null_cond :
    [< dot_expr]
    -> null_cond_qualifier

      
  val f_suffix :
    [< dot_expr]
    -> ref_id


end

module CastExpr : sig
  (**
  * Cast expression.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `CastExpr of
          cast_expr_fields
    ]

  type fields = cast_expr_fields =
    
  {
         
    f_expr: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `KeepExpr
          of keep_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
    ]
    Lazy.t;
         
    f_null_cond: null_cond_qualifier
    Lazy.t;
         
    f_excludes_null: excludes_null
    Lazy.t;
         
    f_dest_type: [
      | `FunctionTypeRef
          of function_type_ref_fields
      | `GenericTypeRef
          of generic_type_ref_fields
      | `SimpleTypeRef
          of simple_type_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_expr :
    [< cast_expr]
    -> [array_literal | block_expr | call_expr | cast_expr | dot_expr | error_on_null | generic_instantiation | keep_expr | lit | logic_expr | logic_predicate | match_expr | paren_expr | ref_id | subscript_expr]

      
  val f_null_cond :
    [< cast_expr]
    -> null_cond_qualifier

      
  val f_excludes_null :
    [< cast_expr]
    -> excludes_null

      
  val f_dest_type :
    [< cast_expr]
    -> [function_type_ref | generic_type_ref | simple_type_ref]


end

module BlockExpr : sig
  (**
  * Block expression.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `BlockExpr of
          block_expr_fields
    ]

  type fields = block_expr_fields =
    
  {
         
    f_clauses: lkt_node_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_clauses :
    [< block_expr]
    -> lkt_node_list


end

module BinOp : sig
  (**
  * Binary operator expression.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `BinOp of
          bin_op_fields
    ]

  type fields = bin_op_fields =
    
  {
         
    f_left: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
         
    f_op: op
    Lazy.t;
         
    f_right: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_left :
    [< bin_op]
    -> [any_of | array_literal | bin_op | block_expr | call_expr | cast_expr | dot_expr | error_on_null | generic_instantiation | if_expr | isa | keep_expr | lambda_expr | lit | logic_assign | logic_expr | logic_predicate | logic_propagate | logic_unify | match_expr | not_expr | paren_expr | raise_expr | ref_id | subscript_expr | try_expr | un_op]

      
  val f_op :
    [< bin_op]
    -> op

      
  val f_right :
    [< bin_op]
    -> [any_of | array_literal | bin_op | block_expr | call_expr | cast_expr | dot_expr | error_on_null | generic_instantiation | if_expr | isa | keep_expr | lambda_expr | lit | logic_assign | logic_expr | logic_predicate | logic_propagate | logic_unify | match_expr | not_expr | paren_expr | raise_expr | ref_id | subscript_expr | try_expr | un_op]


end

module LogicPropagateCall : sig
  (**
  * Class for the call inside "propagate" equations.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `LogicPropagateCall of
          logic_propagate_call_fields
    ]

  type fields = logic_propagate_call_fields =
    
  {
         
    f_name: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `KeepExpr
          of keep_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
    ]
    Lazy.t;
         
    f_args: argument_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_name :
    [< logic_propagate_call]
    -> [array_literal | block_expr | call_expr | cast_expr | dot_expr | error_on_null | generic_instantiation | keep_expr | lit | logic_expr | logic_predicate | match_expr | paren_expr | ref_id | subscript_expr]

      
  val f_args :
    [< logic_propagate_call]
    -> argument_list


end

module LogicPredicate : sig
  (**
  * Class for "predicate" equations.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `LogicPredicate of
          logic_predicate_fields
    ]

  type fields = logic_predicate_fields =
    
  {
         
    f_name: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `KeepExpr
          of keep_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
    ]
    Lazy.t;
         
    f_args: argument_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_name :
    [< logic_predicate]
    -> [array_literal | block_expr | call_expr | cast_expr | dot_expr | error_on_null | generic_instantiation | keep_expr | lit | logic_expr | logic_predicate | match_expr | paren_expr | ref_id | subscript_expr]

      
  val f_args :
    [< logic_predicate]
    -> argument_list


end

module LogicCallExpr : sig
  (**
  * Base class for logic call expresions, of the form:
  *
  * .. code::
  *
  *    name%(args)
  *
  * Derived nodes: ``logic_predicate``, ``logic_propagate_call``
  *)

  type t =
    [
      | LogicPredicate.t
      | LogicPropagateCall.t
    ]


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_name :
    [< logic_call_expr]
    -> [array_literal | block_expr | call_expr | cast_expr | dot_expr | error_on_null | generic_instantiation | keep_expr | lit | logic_expr | logic_predicate | match_expr | paren_expr | ref_id | subscript_expr]

      
  val f_args :
    [< logic_call_expr]
    -> argument_list


end

module CallExpr : sig
  (**
  * Call expression.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `CallExpr of
          call_expr_fields
    ]

  type fields = call_expr_fields =
    
  {
         
    f_name: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `KeepExpr
          of keep_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
    ]
    Lazy.t;
         
    f_args: argument_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_name :
    [< call_expr]
    -> [array_literal | block_expr | call_expr | cast_expr | dot_expr | error_on_null | generic_instantiation | keep_expr | lit | logic_expr | logic_predicate | match_expr | paren_expr | ref_id | subscript_expr]

      
  val f_args :
    [< call_expr]
    -> argument_list


end

module BaseCallExpr : sig
  (**
  * Base class for expressions that are syntactically call-like.
  *
  * Derived nodes: ``call_expr``, ``logic_call_expr``
  *)

  type t =
    [
      | CallExpr.t
      | LogicCallExpr.t
    ]


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_name :
    [< base_call_expr]
    -> [array_literal | block_expr | call_expr | cast_expr | dot_expr | error_on_null | generic_instantiation | keep_expr | lit | logic_expr | logic_predicate | match_expr | paren_expr | ref_id | subscript_expr]

      
  val f_args :
    [< base_call_expr]
    -> argument_list


end

module ArrayLiteral : sig
  (**
  * Literal for an array value.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `ArrayLiteral of
          array_literal_fields
    ]

  type fields = array_literal_fields =
    
  {
         
    f_exprs: expr_list
    Lazy.t;
         
    f_element_type: [
      | `FunctionTypeRef
          of function_type_ref_fields
      | `GenericTypeRef
          of generic_type_ref_fields
      | `SimpleTypeRef
          of simple_type_ref_fields
    ]
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_exprs :
    [< array_literal]
    -> expr_list

      
  val f_element_type :
    [< array_literal]
    -> [function_type_ref | generic_type_ref | simple_type_ref] option


end

module AnyOf : sig
  (**
  * "Any of" expression.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `AnyOf of
          any_of_fields
    ]

  type fields = any_of_fields =
    
  {
         
    f_expr: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
    ]
    Lazy.t;
         
    f_values: any_of_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_expr :
    [< any_of]
    -> [array_literal | block_expr | call_expr | cast_expr | dot_expr | error_on_null | generic_instantiation | if_expr | keep_expr | lambda_expr | lit | logic_expr | logic_predicate | match_expr | paren_expr | raise_expr | ref_id | subscript_expr | try_expr]

      
  val f_values :
    [< any_of]
    -> any_of_list


end

module Expr : sig
  (**
  * Base class for expressions. Encompasses regular expressions as well as
  * special expressions (grammar expressions, etc).
  *
  * Derived nodes: ``any_of``, ``array_literal``, ``base_call_expr``,
  * ``bin_op``, ``block_expr``, ``cast_expr``, ``dot_expr``, ``error_on_null``,
  * ``generic_instantiation``, ``grammar_expr``, ``id``, ``if_expr``, ``isa``,
  * ``keep_expr``, ``lambda_expr``, ``lit``, ``logic_assign``, ``logic_expr``,
  * ``logic_propagate``, ``logic_unify``, ``match_expr``, ``not_expr``,
  * ``paren_expr``, ``raise_expr``, ``subscript_expr``, ``try_expr``, ``un_op``
  *)

  type t =
    [
      | AnyOf.t
      | ArrayLiteral.t
      | BaseCallExpr.t
      | BinOp.t
      | BlockExpr.t
      | CastExpr.t
      | DotExpr.t
      | ErrorOnNull.t
      | GenericInstantiation.t
      | GrammarExpr.t
      | Id.t
      | IfExpr.t
      | Isa.t
      | KeepExpr.t
      | LambdaExpr.t
      | Lit.t
      | LogicAssign.t
      | LogicExpr.t
      | LogicPropagate.t
      | LogicUnify.t
      | MatchExpr.t
      | NotExpr.t
      | ParenExpr.t
      | RaiseExpr.t
      | SubscriptExpr.t
      | TryExpr.t
      | UnOp.t
    ]


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int


      
  val p_get_type :
    [< expr ]
    -> type_decl option
  (**
  * Return the type of this expression.
  *)

      
  val p_get_generic_type :
    [< expr ]
    -> type_decl option
  (**
  * Return the expected type of this expression.
  *)

      
  val p_get_expected_type :
    [< expr ]
    -> type_decl option
  (**
  * Return the expected type of this expression.
  *)

      
  val p_referenced_decl :
    [< expr ]
    -> decl option
  (**
  * Return the declaration referenced by this expression, if applicable, else
  * null.
  *
  * The property is memoized in order to avoid use the value inside logic
  * variables on every redundent call, causing faulty behavior when used with
  * rebindings. TODO: Do like LAL to avoid memoization for more safety.
  *)



end

module ExcludesNullPresent : sig
  (**
  * This node type has no derivation.
  *)

  type t =
    [
      | `ExcludesNullPresent of
          excludes_null_present_fields
    ]

  type fields = excludes_null_present_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module ExcludesNullAbsent : sig
  (**
  * This node type has no derivation.
  *)

  type t =
    [
      | `ExcludesNullAbsent of
          excludes_null_absent_fields
    ]

  type fields = excludes_null_absent_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module ExcludesNull : sig
  (**
  * Whether the containing cast expression will raise on null cast result or
  * not.
  *
  * Derived nodes: ``excludes_null_absent``, ``excludes_null_present``
  *)

  type t =
    [
      | ExcludesNullAbsent.t
      | ExcludesNullPresent.t
    ]


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int


      
  val p_as_bool :
    [< excludes_null ]
    -> bool
  (**
  * Return whether this node is present
  *)



end

module EnumClassCase : sig
  (**
  * Case branch for an enum class declaration.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `EnumClassCase of
          enum_class_case_fields
    ]

  type fields = enum_class_case_fields =
    
  {
         
    f_decls: enum_class_alt_decl_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_decls :
    [< enum_class_case]
    -> enum_class_alt_decl_list


end

module ElsifBranch : sig
  (**
  * Elsif branch of an if expression.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `ElsifBranch of
          elsif_branch_fields
    ]

  type fields = elsif_branch_fields =
    
  {
         
    f_cond_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
         
    f_then_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_cond_expr :
    [< elsif_branch]
    -> [any_of | array_literal | bin_op | block_expr | call_expr | cast_expr | dot_expr | error_on_null | generic_instantiation | if_expr | isa | keep_expr | lambda_expr | lit | logic_assign | logic_expr | logic_predicate | logic_propagate | logic_unify | match_expr | not_expr | paren_expr | raise_expr | ref_id | subscript_expr | try_expr | un_op]

      
  val f_then_expr :
    [< elsif_branch]
    -> [any_of | array_literal | bin_op | block_expr | call_expr | cast_expr | dot_expr | error_on_null | generic_instantiation | if_expr | isa | keep_expr | lambda_expr | lit | logic_assign | logic_expr | logic_predicate | logic_propagate | logic_unify | match_expr | not_expr | paren_expr | raise_expr | ref_id | subscript_expr | try_expr | un_op]


end

module DynEnvWrapper : sig
  (**
  * Synthetic node to instantiate a DynamicEnvironment for generics.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `DynEnvWrapper of
          dyn_env_wrapper_fields
    ]

  type fields = dyn_env_wrapper_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module DeclAnnotationArgs : sig
  (**
  * List of arguments for an annotation with a call syntax. This intermediate
  * node is necessary in order to determine after parsing whether there is no
  * argument list, or if the list is empty.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `DeclAnnotationArgs of
          decl_annotation_args_fields
    ]

  type fields = decl_annotation_args_fields =
    
  {
         
    f_args: argument_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_args :
    [< decl_annotation_args]
    -> argument_list


end

module DeclAnnotation : sig
  (**
  * Compile time annotation attached to a declaration.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `DeclAnnotation of
          decl_annotation_fields
    ]

  type fields = decl_annotation_fields =
    
  {
         
    f_name: id
    Lazy.t;
         
    f_args: decl_annotation_args
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_name :
    [< decl_annotation]
    -> id

      
  val f_args :
    [< decl_annotation]
    -> decl_annotation_args option


end

module TraitDecl : sig
  (**
  * Trait declaration. For the moment, a Trait can just be used to group
  * behavior for built-in types. It's not usable as a type-bound since we don't
  * have generics, and you cannot implement one either.
  *
  * The reason they're added is to lay down the basics of what we want the Lkt
  * type system to be.
  *
  * TODO: Traits are *not* types. They're treated as such in the grammar for
  * convenience for now, but it's probably not a good idea. Migrate away from
  * this.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `TraitDecl of
          trait_decl_fields
    ]

  type fields = trait_decl_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module StructDecl : sig
  (**
  * Declaration for a LK struct.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `StructDecl of
          struct_decl_fields
    ]

  type fields = struct_decl_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module EnumTypeDecl : sig
  (**
  * Enum type declaration.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `EnumTypeDecl of
          enum_type_decl_fields
    ]

  type fields = enum_type_decl_fields =
    
  {
         
    f_literals: enum_lit_decl_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_literals :
    [< enum_type_decl]
    -> enum_lit_decl_list


end

module EnumClassDecl : sig
  (**
  * Declaration for a LK class. This only cover node classes for the moment,
  * but might be extended to support regular classes in the future.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `EnumClassDecl of
          enum_class_decl_fields
    ]

  type fields = enum_class_decl_fields =
    
  {
         
    f_branches: enum_class_case_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_branches :
    [< enum_class_decl]
    -> enum_class_case_list


end

module ClassDecl : sig
  (**
  * Declaration for a LK class. This only cover node classes for the moment,
  * but might be extended to support regular classes in the future.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `ClassDecl of
          class_decl_fields
    ]

  type fields = class_decl_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module BasicClassDecl : sig
  (**
  * Common ancestor for declarations of regular classes and enum classes.
  *
  * Derived nodes: ``class_decl``, ``enum_class_decl``
  *)

  type t =
    [
      | ClassDecl.t
      | EnumClassDecl.t
    ]


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module NamedTypeDecl : sig
  (**
  * Explicit named type declaration.
  *
  * Derived nodes: ``basic_class_decl``, ``enum_type_decl``, ``struct_decl``,
  * ``trait_decl``
  *)

  type t =
    [
      | BasicClassDecl.t
      | EnumTypeDecl.t
      | StructDecl.t
      | TraitDecl.t
    ]


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module GenericParamTypeDecl : sig
  (**
  * Declaration of a parameter type in a generic declaration.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `GenericParamTypeDecl of
          generic_param_type_decl_fields
    ]

  type fields = generic_param_type_decl_fields =
    
  {
         
    f_has_class: class_qualifier
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_has_class :
    [< generic_param_type_decl]
    -> class_qualifier


end

module FunctionType : sig
  (**
  * Function type.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `FunctionType of
          function_type_fields
    ]

  type fields = function_type_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module EnumClassAltDecl : sig
  (**
  * Alternative for an enum class decl.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `EnumClassAltDecl of
          enum_class_alt_decl_fields
    ]

  type fields = enum_class_alt_decl_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module AnyTypeDecl : sig
  (**
  * Internal type to represent a type that can be matched with anything.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `AnyTypeDecl of
          any_type_decl_fields
    ]

  type fields = any_type_decl_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module TypeDecl : sig
  (**
  * Abstract base class for type declarations.
  *
  * Derived nodes: ``any_type_decl``, ``enum_class_alt_decl``,
  * ``function_type``, ``generic_param_type_decl``, ``named_type_decl``
  *)

  type t =
    [
      | AnyTypeDecl.t
      | EnumClassAltDecl.t
      | FunctionType.t
      | GenericParamTypeDecl.t
      | NamedTypeDecl.t
    ]


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int


      
  val p_def_id :
    [< type_decl ]
    -> def_id option
  (**
  * Return the defining name of this type declaration
  *)

      
  val p_base_type :
    [< type_decl ]
    -> type_ref option
  (**
  * Return the base type for this node, if any.
  *)

      
  val p_base_type_if_entity :
    [< type_decl ]
    -> type_decl option
  (**
  * Return the base type for this node, if any.
  *)



end

module SynthParamDecl : sig
  (**
  * Logic function parameter declaration.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `SynthParamDecl of
          synth_param_decl_fields
    ]

  type fields = synth_param_decl_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module SynthFunDecl : sig
  (**
  * Logic function declaration.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `SynthFunDecl of
          synth_fun_decl_fields
    ]

  type fields = synth_fun_decl_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module LexerFamilyDecl : sig
  (**
  * Declaration of a token family.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `LexerFamilyDecl of
          lexer_family_decl_fields
    ]

  type fields = lexer_family_decl_fields =
    
  {
         
    f_rules: full_decl_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_rules :
    [< lexer_family_decl]
    -> full_decl_list


end

module LexerDecl : sig
  (**
  * Declaration of a language's lexer.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `LexerDecl of
          lexer_decl_fields
    ]

  type fields = lexer_decl_fields =
    
  {
         
    f_rules: lkt_node_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_rules :
    [< lexer_decl]
    -> lkt_node_list


end

module GrammarDecl : sig
  (**
  * Declaration of a language's grammar.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `GrammarDecl of
          grammar_decl_fields
    ]

  type fields = grammar_decl_fields =
    
  {
         
    f_rules: full_decl_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_rules :
    [< grammar_decl]
    -> full_decl_list


end

module GenericDecl : sig
  (**
  * Generic entity declaration.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `GenericDecl of
          generic_decl_fields
    ]

  type fields = generic_decl_fields =
    
  {
         
    f_generic_param_decls: generic_param_decl_list
    Lazy.t;
         
    f_decl: [
      | `ClassDecl
          of class_decl_fields
      | `DynVarDecl
          of dyn_var_decl_fields
      | `EnumClassDecl
          of enum_class_decl_fields
      | `EnumTypeDecl
          of enum_type_decl_fields
      | `EnvSpecDecl
          of env_spec_decl_fields
      | `FieldDecl
          of field_decl_fields
      | `FunDecl
          of fun_decl_fields
      | `GenericDecl
          of generic_decl_fields
      | `GrammarDecl
          of grammar_decl_fields
      | `GrammarRuleDecl
          of grammar_rule_decl_fields
      | `LexerDecl
          of lexer_decl_fields
      | `StructDecl
          of struct_decl_fields
      | `TraitDecl
          of trait_decl_fields
      | `ValDecl
          of val_decl_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_generic_param_decls :
    [< generic_decl]
    -> generic_param_decl_list

      
  val f_decl :
    [< generic_decl]
    -> [dyn_var_decl | env_spec_decl | field_decl | fun_decl | generic_decl | grammar_decl | grammar_rule_decl | lexer_decl | named_type_decl | val_decl]


end

module ErrorDecl : sig
  (**
  * Placeholder node for syntax errors in lists of declarations.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `ErrorDecl of
          error_decl_fields
    ]

  type fields = error_decl_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module EnvSpecDecl : sig
  (**
  * Env spec declaration.
  *
  * Each node type can have one or no env spec. Env specs contains only a list
  * of env actions.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `EnvSpecDecl of
          env_spec_decl_fields
    ]

  type fields = env_spec_decl_fields =
    
  {
         
    f_actions: call_expr_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_actions :
    [< env_spec_decl]
    -> call_expr_list


end

module FunDecl : sig
  (**
  * Function declaration.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `FunDecl of
          fun_decl_fields
    ]

  type fields = fun_decl_fields =
    
  {
         
    f_params: fun_param_decl_list
    Lazy.t;
         
    f_return_type: [
      | `FunctionTypeRef
          of function_type_ref_fields
      | `GenericTypeRef
          of generic_type_ref_fields
      | `SimpleTypeRef
          of simple_type_ref_fields
    ]
    Lazy.t;
         
    f_trait_ref: dot_expr
    option
    Lazy.t;
         
    f_body: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int


      
  val p_is_dynamic_combiner :
    [< fun_decl ]
    -> bool
  (**
  * When this property is used as a a combinder inside an NPropagate equation,
  * return whether it expects a dynamic number of arguments.
  *)

      
  val p_find_all_overrides :
    [< fun_decl ]
    -> analysis_unit list
    -> fun_decl list
  (**
  * Return the list of all RefId that refer to this DefId.
  *)


      
  val f_params :
    [< fun_decl]
    -> fun_param_decl_list

      
  val f_return_type :
    [< fun_decl]
    -> [function_type_ref | generic_type_ref | simple_type_ref]

      
  val f_trait_ref :
    [< fun_decl]
    -> dot_expr option

      
  val f_body :
    [< fun_decl]
    -> [any_of | array_literal | bin_op | block_expr | call_expr | cast_expr | dot_expr | error_on_null | generic_instantiation | if_expr | isa | keep_expr | lambda_expr | lit | logic_assign | logic_expr | logic_predicate | logic_propagate | logic_unify | match_expr | not_expr | paren_expr | raise_expr | ref_id | subscript_expr | try_expr | un_op] option


end

module ValDecl : sig
  (**
  * Value declaration.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `ValDecl of
          val_decl_fields
    ]

  type fields = val_decl_fields =
    
  {
         
    f_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_expr :
    [< val_decl]
    -> [any_of | array_literal | bin_op | block_expr | call_expr | cast_expr | dot_expr | error_on_null | generic_instantiation | if_expr | isa | keep_expr | lambda_expr | lit | logic_assign | logic_expr | logic_predicate | logic_propagate | logic_unify | match_expr | not_expr | paren_expr | raise_expr | ref_id | subscript_expr | try_expr | un_op]


end

module MatchValDecl : sig
  (**
  * Value declaration in a match branch.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `MatchValDecl of
          match_val_decl_fields
    ]

  type fields = match_val_decl_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module DynVarDecl : sig
  (**
  * Dynamic variable declaration.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `DynVarDecl of
          dyn_var_decl_fields
    ]

  type fields = dyn_var_decl_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module LambdaParamDecl : sig
  (**
  * Function parameter declaration.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `LambdaParamDecl of
          lambda_param_decl_fields
    ]

  type fields = lambda_param_decl_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module FunParamDecl : sig
  (**
  * Function parameter declaration.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `FunParamDecl of
          fun_param_decl_fields
    ]

  type fields = fun_param_decl_fields =
    
  {
         
    f_decl_annotations: decl_annotation_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_decl_annotations :
    [< fun_param_decl]
    -> decl_annotation_list


end

module FieldDecl : sig
  (**
  * Field declaration.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `FieldDecl of
          field_decl_fields
    ]

  type fields = field_decl_fields =
    
  {
         
    f_trait_ref: dot_expr
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_trait_ref :
    [< field_decl]
    -> dot_expr option


end

module ComponentDecl : sig
  (**
  * Subset of explicitly typed declarations for value declarations that:
  *
  * 1. Have an optional default value.
  *
  * 2. Are part of a bigger declaration that can be referred to via a call
  *    expression (either a type or a function).
  *
  * Derived nodes: ``field_decl``, ``fun_param_decl``, ``lambda_param_decl``
  *)

  type t =
    [
      | FieldDecl.t
      | FunParamDecl.t
      | LambdaParamDecl.t
    ]


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module ExplicitlyTypedDecl : sig
  (**
  * Subset of user declared value declarations for values that have a type that
  * can be syntactically annotated by the user.
  *
  * Derived nodes: ``component_decl``, ``dyn_var_decl``, ``match_val_decl``,
  * ``val_decl``
  *)

  type t =
    [
      | ComponentDecl.t
      | DynVarDecl.t
      | MatchValDecl.t
      | ValDecl.t
    ]


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module EnumLitDecl : sig
  (**
  * Enum literal declaration.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `EnumLitDecl of
          enum_lit_decl_fields
    ]

  type fields = enum_lit_decl_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module BindingValDecl : sig
  (**
  * Variable declaration in pattern
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `BindingValDecl of
          binding_val_decl_fields
    ]

  type fields = binding_val_decl_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module UserValDecl : sig
  (**
  * Class for user declared val declarations (not synthetic).
  *
  * Derived nodes: ``binding_val_decl``, ``enum_lit_decl``,
  * ``explicitly_typed_decl``, ``fun_decl``
  *)

  type t =
    [
      | BindingValDecl.t
      | EnumLitDecl.t
      | ExplicitlyTypedDecl.t
      | FunDecl.t
    ]


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module SelfDecl : sig
  (**
  * Synthetic declaration for the implicit "self" variable available in
  * properties.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `SelfDecl of
          self_decl_fields
    ]

  type fields = self_decl_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module NodeDecl : sig
  (**
  * Synthetic declaration for the implicit "node" variable available in
  * properties.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `NodeDecl of
          node_decl_fields
    ]

  type fields = node_decl_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module BaseValDecl : sig
  (**
  * Abstract class for named values declarations, such as parameters, local
  * value bindings, fields, etc.
  *
  * Derived nodes: ``node_decl``, ``self_decl``, ``user_val_decl``
  *)

  type t =
    [
      | NodeDecl.t
      | SelfDecl.t
      | UserValDecl.t
    ]


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module SyntheticLexerDecl : sig
  (**
  * This node type has no derivation.
  *)

  type t =
    [
      | `SyntheticLexerDecl of
          synthetic_lexer_decl_fields
    ]

  type fields = synthetic_lexer_decl_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module GrammarRuleDecl : sig
  (**
  * Declaration of a grammar rule inside of a grammar.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `GrammarRuleDecl of
          grammar_rule_decl_fields
    ]

  type fields = grammar_rule_decl_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module BaseGrammarRuleDecl : sig
  (**
  * Base class for grammar rules inside of grammars/lexers.
  *
  * Derived nodes: ``grammar_rule_decl``, ``synthetic_lexer_decl``
  *)

  type t =
    [
      | GrammarRuleDecl.t
      | SyntheticLexerDecl.t
    ]


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module Decl : sig
  (**
  * Base class for declarations. Encompasses regular declarations as well as
  * special declarations such as grammars, grammar rules, etc.
  *
  * Derived nodes: ``base_grammar_rule_decl``, ``base_val_decl``,
  * ``env_spec_decl``, ``error_decl``, ``generic_decl``, ``grammar_decl``,
  * ``lexer_decl``, ``lexer_family_decl``, ``synth_fun_decl``,
  * ``synth_param_decl``, ``type_decl``
  *)

  type t =
    [
      | BaseGrammarRuleDecl.t
      | BaseValDecl.t
      | EnvSpecDecl.t
      | ErrorDecl.t
      | GenericDecl.t
      | GrammarDecl.t
      | LexerDecl.t
      | LexerFamilyDecl.t
      | SynthFunDecl.t
      | SynthParamDecl.t
      | TypeDecl.t
    ]


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int


      
  val p_custom_image :
    [< decl ]
    -> string
  (**
  * Return the image string using entity information.
  *)

      
  val p_decl_type_name :
    [< decl ]
    -> string
  (**
  * Return the name of the declaration type, as it should be seen by
  * users/shown in diagnostics.
  *)

      
  val p_def_ids :
    [< decl ]
    -> def_id list
  (**
  * Return all the defining names that this declaration defines.
  *)

      
  val p_as_bare_decl :
    [< decl ]
    -> decl option
  (**
  * Get this declaration without rebindings information.
  *)

      
  val p_get_type :
    [< decl ]
    -> type_decl option
  (**
  * Return the type of the Decl.
  *)

      
  val p_get_cast_type :
    [< decl ]
    -> type_decl
    -> type_decl option
  (**
  * If we are casting an entity (Self) to something that is not an entity, make
  * it an entity.
  *)

      
  val p_get_keep_type :
    [< decl ]
    -> type_decl
    -> type_decl option
  (**
  * Return the type of Entity when we only keep elements of type keep_type. If
  * we are casting an entity (Self) to something that is not an entity, make it
  * an entity.
  *)

      
  val p_get_suffix_type :
    [< decl ]
    -> type_decl
    -> type_decl option
  (**
  * If we are accessing a ParseField of an entity, then that field's type also
  * needs to be an entity.
  *)

      
  val p_is_generic :
    [< decl ]
    -> bool
  (**
  * Returns whether the Decl is generic.
  *)

      
  val p_return_type_is_instantiated :
    [< decl ]
    -> bool
  (**
  * Return True if the return type of this function is instantiated.
  *)

      
  val p_is_instantiated :
    [< decl ]
    -> bool
  (**
  * Return True if Self is an instantiated declaration, meaning that it does
  * not use any of its declared generic types.
  *)

      
  val p_name :
    [< decl ]
    -> string
  (**
  * Return the symbol corresponding to the name of this declaration.
  *)

      
  val p_full_name :
    [< decl ]
    -> string
  (**
  * Return the full name of this decl, as it should be seen by users/shown in
  * diagnostics.
  *)



end

module ClassQualifierPresent : sig
  (**
  * This node type has no derivation.
  *)

  type t =
    [
      | `ClassQualifierPresent of
          class_qualifier_present_fields
    ]

  type fields = class_qualifier_present_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module ClassQualifierAbsent : sig
  (**
  * This node type has no derivation.
  *)

  type t =
    [
      | `ClassQualifierAbsent of
          class_qualifier_absent_fields
    ]

  type fields = class_qualifier_absent_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module ClassQualifier : sig
  (**
  * Whether this generic parameter type must be a class.
  *
  * Derived nodes: ``class_qualifier_absent``, ``class_qualifier_present``
  *)

  type t =
    [
      | ClassQualifierAbsent.t
      | ClassQualifierPresent.t
    ]


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int


      
  val p_as_bool :
    [< class_qualifier ]
    -> bool
  (**
  * Return whether this node is present
  *)



end

module BlockStringLine : sig
  (**
  * A single line in a block string literal.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `BlockStringLine of
          block_string_line_fields
    ]

  type fields = block_string_line_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module BlockExprClause : sig
  (**
  * Clause (value declaration or dynamic variable binding) in a block
  * expression.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `BlockExprClause of
          block_expr_clause_fields
    ]

  type fields = block_expr_clause_fields =
    
  {
         
    f_clause: [
      | `ValDecl
          of val_decl_fields
      | `VarBind
          of var_bind_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_clause :
    [< block_expr_clause]
    -> [val_decl | var_bind]


end

module PatternMatchBranch : sig
  (**
  * Branch inside a match expression. LKQL pattern based syntax ``case
  * <pattern> => <expr>``.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `PatternMatchBranch of
          pattern_match_branch_fields
    ]

  type fields = pattern_match_branch_fields =
    
  {
         
    f_pattern: [
      | `BindingPattern
          of binding_pattern_fields
      | `BoolPatternFalse
          of bool_pattern_false_fields
      | `BoolPatternTrue
          of bool_pattern_true_fields
      | `ExtendedPattern
          of extended_pattern_fields
      | `FilteredPattern
          of filtered_pattern_fields
      | `IntegerPattern
          of integer_pattern_fields
      | `ListPattern
          of list_pattern_fields
      | `NotPattern
          of not_pattern_fields
      | `NullPattern
          of null_pattern_fields
      | `OrPattern
          of or_pattern_fields
      | `ParenPattern
          of paren_pattern_fields
      | `RegexPattern
          of regex_pattern_fields
      | `TuplePattern
          of tuple_pattern_fields
      | `TypePattern
          of type_pattern_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_pattern :
    [< pattern_match_branch]
    -> [binding_pattern | bool_pattern | extended_pattern | filtered_pattern | integer_pattern | list_pattern | not_pattern | null_pattern | or_pattern | paren_pattern | regex_pattern | tuple_pattern | type_pattern]


end

module MatchBranch : sig
  (**
  * Branch inside a match expression. Classic limited Lkt syntax based on
  * ``case <name> : <type> => <expr>``, for the moment, the only supported
  * syntax in Lkt.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `MatchBranch of
          match_branch_fields
    ]

  type fields = match_branch_fields =
    
  {
         
    f_decl: match_val_decl
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_decl :
    [< match_branch]
    -> match_val_decl


end

module BaseMatchBranch : sig
  (**
  * Abstract base class for match branches, exists to accommodate the existence
  * of two different syntaxes.
  *
  * Derived nodes: ``match_branch``, ``pattern_match_branch``
  *)

  type t =
    [
      | MatchBranch.t
      | PatternMatchBranch.t
    ]


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int


      
  val p_match_part :
    [< base_match_branch ]
    -> lkt_node option
  (**
  * Return the "match" part of the branch, either a pattern branch or a legacy
  * match branch with variable declaration.
  *)



end

module LexerCaseRuleDefaultAlt : sig
  (**
  * Default alternative of a case rule which sends the token if all the
  * previous alternatives failed.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `LexerCaseRuleDefaultAlt of
          lexer_case_rule_default_alt_fields
    ]

  type fields = lexer_case_rule_default_alt_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module LexerCaseRuleCondAlt : sig
  (**
  * Alternative of a case rule which sends the token only if the kind of the
  * previous token is among a given set.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `LexerCaseRuleCondAlt of
          lexer_case_rule_cond_alt_fields
    ]

  type fields = lexer_case_rule_cond_alt_fields =
    
  {
         
    f_cond_exprs: ref_id_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_cond_exprs :
    [< lexer_case_rule_cond_alt]
    -> ref_id_list


end

module BaseLexerCaseRuleAlt : sig
  (**
  * Base class for the different kind of alternatives allowed in a case rule.
  *
  * Derived nodes: ``lexer_case_rule_cond_alt``,
  * ``lexer_case_rule_default_alt``
  *)

  type t =
    [
      | LexerCaseRuleCondAlt.t
      | LexerCaseRuleDefaultAlt.t
    ]


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int




end

module Argument : sig
  (**
  * Argument for function calls or for annotations.
  *
  * This node type has no derivation.
  *)

  type t =
    [
      | `Argument of
          argument_fields
    ]

  type fields = argument_fields =
    
  {
         
    f_name: ref_id
    option
    Lazy.t;
         
    f_value: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int



      
  val f_name :
    [< argument]
    -> ref_id option

      
  val f_value :
    [< argument]
    -> [any_of | array_literal | bin_op | block_expr | call_expr | cast_expr | dot_expr | error_on_null | generic_instantiation | if_expr | isa | keep_expr | lambda_expr | lit | logic_assign | logic_expr | logic_predicate | logic_propagate | logic_unify | match_expr | not_expr | paren_expr | raise_expr | ref_id | subscript_expr | try_expr | un_op]


end

module LktNode : sig
  (**
  * Root node class for lkt AST nodes.
  *
  * Derived nodes: ``argument``, ``base_lexer_case_rule_alt``,
  * ``base_match_branch``, ``block_expr_clause``, ``block_string_line``,
  * ``class_qualifier``, ``decl_annotation_args``, ``decl_annotation``,
  * ``decl``, ``dyn_env_wrapper``, ``elsif_branch``, ``enum_class_case``,
  * ``excludes_null``, ``expr``, ``full_decl``, ``grammar_list_sep``,
  * ``import``, ``langkit_root``, ``lexer_case_rule_send``,
  * ``lexer_case_rule``, ``list_kind``, ``lkt_node_base_list``,
  * ``null_cond_qualifier``, ``op``, ``pattern_detail``, ``pattern``,
  * ``selector_call``, ``type_ref``, ``var_bind``
  *)

  type t =
    [
      | Argument.t
      | BaseLexerCaseRuleAlt.t
      | BaseMatchBranch.t
      | BlockExprClause.t
      | BlockStringLine.t
      | ClassQualifier.t
      | Decl.t
      | DeclAnnotation.t
      | DeclAnnotationArgs.t
      | DynEnvWrapper.t
      | ElsifBranch.t
      | EnumClassCase.t
      | ExcludesNull.t
      | Expr.t
      | FullDecl.t
      | GrammarListSep.t
      | Import.t
      | LangkitRoot.t
      | LexerCaseRule.t
      | LexerCaseRuleSend.t
      | ListKind.t
      | LktNodeBaseList.t
      | NullCondQualifier.t
      | Op.t
      | Pattern.t
      | PatternDetail.t
      | SelectorCall.t
      | TypeRef.t
      | VarBind.t
    ]


  val equal : [< t] -> [< t] -> bool

  val hash : [< t] -> int

  val compare : [< t] -> [< t] -> int

  val kind_name : [< lkt_node] -> string
  (**
  * Return the kind of this node.
  *)

  val text : [< lkt_node ] -> string
  (**
   * Return the source buffer slice corresponding to the text that spans
   * between the first and the last tokens of the given node.
   *)

  val image : [< lkt_node ] -> string
  (**
  * Return a representation of this node as a string.
  *)

  val sloc_range : [< lkt_node ] -> SlocRange.t
  (**
  * Return the spanning source location range for this node.
  *
  * Note that this returns the sloc of the parent for synthetic nodes.
  *)

  val lookup : [< lkt_node ] -> Sloc.t -> lkt_node option
  (**
  * Return the bottom-most node from in ``Node`` and its children which
  * contains ``Sloc``, or ``None`` if there is none.
  *)

  
  val fold_tokens : ('a -> Token.t -> 'a) -> 'a -> [< lkt_node] -> 'a
  (**
   * Fold all the token this node contains by calling f on each token.
   *)

  val iter_tokens : (Token.t -> unit) -> [< lkt_node]-> unit
  (**
   * Iterate over all token this node contains by calling f on each token.
   *)

  val map_tokens : (Token.t -> 'a) -> [< lkt_node] -> 'a list
  (**
   * Map each token calling the given function
   *)

  val tokens : [< lkt_node] -> Token.t list
  (**
   * Return a list of tokens for the given node
   *)


  val children_opt : [< lkt_node ] -> lkt_node option list
  (**
   * Return an optional list of nodes which are the children of the given node.
   * Each child is optional because it can either be because of a syntax error,
   * or an optional field evaluated to null.
   *)

  val fold_fields :
    ('a -> lkt_node -> 'a) -> 'a -> [< lkt_node ] -> 'a
  (**
   * Fold all fields of the given node. This skips any child that is None
   * because of a syntax error or because the field is optional
   *)

  val iter_fields :
    (lkt_node -> unit) -> [< lkt_node ] -> unit
  (**
   * Iter all fields of the given node. This skips any child that is None
   * because of a syntax error or because the field is optional
   *)

  val exists_fields :
    (lkt_node -> bool) -> [< lkt_node ] -> bool
  (**
   * Fold all fields of the given node. Return true if the given predicate is
   * evaluated to true for at least one node.
   *)

  val for_all_fields :
    (lkt_node -> bool) -> [< lkt_node ] -> bool
  (**
   * Fold all fields of the given node. Return true if the given predicate is
   * evaluated to true for all nodes.
   *)

  val fold :
    ('a -> lkt_node -> 'a) -> 'a -> [< lkt_node ] -> 'a
  (**
   * Fold the entire AST, below the given node, and call the given function on
   * each node in prefix order.
   *)

  val iter :
    (lkt_node -> unit) -> [< lkt_node ] -> unit
  (**
   * Iterate over the entire AST, below the given node, and call the given
   * function on each node in prefix order.
   *)

  val filter :
    (lkt_node -> bool)
    -> [< lkt_node ]
    -> lkt_node list
  (**
   * Fold the entire AST, below the given node, and return the list of node
   * evaluated to true by the given function
   *)

  val exists :
    (lkt_node -> bool) -> [< lkt_node ] -> bool
  (**
   * Fold the entire AST, below the given node, and return true if the given
   * predicate is evaluated to true by at least one node.
   *)

  val for_all :
    (lkt_node -> bool) -> [< lkt_node ] -> bool
  (**
   * Fold the entire AST, below the given node, and return true if the given
   * predicate is evaluated to true for all nodes.
   *)

  val lookup_with_kind :
    'a node -> [< lkt_node] -> Sloc.t -> 'a option
  (**
   * Given the kind of a node, a source location and a node, return the deepest
   * node containing the source location and of the right kind. Returns None if
   * there is no match.
   *)

  val as_a : 'a node -> [< lkt_node ] -> 'a option
  (**
   * Given the kind of a node, try to cast the given node to this kind. Return
   * None if the node is not of this type and thus, cannot be cast.
   *)

  val find : 'a node -> [< lkt_node ] -> 'a
  (**
   * Given the kind of node, return the first node found by walking the given
   * node. The type of the resulting node depends on the desired kind
   *)


  val findall : 'a node -> [< lkt_node ] -> 'a list
  (**
   * Given the kind of node, return the all nodes of this kind found by walking
   * the given node. The type of the resulting nodes depends on the desired
   * kind
   *)

  val fields_with_names :
    [< lkt_node ] -> (string * lkt_node option) list
  (**
   * Given a node, return the list of it's fields, together with the name of
   * the field. This function does not raise SyntaxError, but instead the
   * returned node is None.
   *)

  val pp_tree : Format.formatter -> [< lkt_node] -> unit
  (**
   * Pretty print the whole tree by completely walking it.
   *)


      
  val parent :
    [< lkt_node ]
    -> lkt_node option
  (**
  * Return the syntactic parent for this node. Return null for the root node.
  *)

      
  val parents :
    ?with_self:
    bool
    -> [< lkt_node ]
    -> lkt_node list
  (**
  * Return an array that contains the lexical parents, this node included iff
  * ``with_self`` is True. Nearer parents are first in the list.
  *)

      
  val children :
    [< lkt_node ]
    -> lkt_node list
  (**
  * Return an array that contains the direct lexical children.
  *
  * .. warning:: This constructs a whole array every-time you call it, and as
  *    such is less efficient than calling the ``Child`` built-in.
  *)

      
  val token_start :
    [< lkt_node ]
    -> Token.t option
  (**
  * Return the first token used to parse this node.
  *)

      
  val token_end :
    [< lkt_node ]
    -> Token.t option
  (**
  * Return the last token used to parse this node.
  *)

      
  val child_index :
    [< lkt_node ]
    -> int
  (**
  * Return the 0-based index for Node in its parent's children.
  *)

      
  val previous_sibling :
    [< lkt_node ]
    -> lkt_node option
  (**
  * Return the node's previous sibling, or null if there is no such sibling.
  *)

      
  val next_sibling :
    [< lkt_node ]
    -> lkt_node option
  (**
  * Return the node's next sibling, or null if there is no such sibling.
  *)

      
  val unit :
    [< lkt_node ]
    -> analysis_unit
  (**
  * Return the analysis unit owning this node.
  *)

      
  val is_ghost :
    [< lkt_node ]
    -> bool
  (**
  * Return whether the node is a ghost.
  *
  * Unlike regular nodes, ghost nodes cover no token in the input source: they
  * are logically located instead between two tokens. Both the ``token_start``
  * and the ``token_end`` of all ghost nodes is the token right after this
  * logical position.
  *)

      
  val full_sloc_image :
    [< lkt_node ]
    -> string
  (**
  * Return a string containing the filename + the sloc in GNU conformant
  * format. Useful to create diagnostics from a node.
  *)

      
  val completion_item_kind_to_int :
    [< lkt_node ]
    -> CompletionItemKind.t
    -> int
  (**
  * Convert a CompletionItemKind enum to its corresponding integer value.
  *)

      
  val p_set_solver_debug_mode :
    [< lkt_node ]
    -> bool
    -> bool
  (**
  * Enable or disable the solver traces for debugging purposes.
  *)

      
  val p_basic_trait_gen :
    [< lkt_node ]
    -> generic_decl option
  (**
  * Unit method. Return the ``BasicTrait`` builtin generic trait.
  *)

      
  val p_basic_trait :
    [< lkt_node ]
    -> trait_decl option
  (**
  * Unit method. Return the ``BasicTrait`` builtin trait.
  *)

      
  val p_node_gen_trait :
    [< lkt_node ]
    -> generic_decl option
  (**
  * Unit method. Return the ``Node`` builtin generic trait.
  *)

      
  val p_node_trait :
    [< lkt_node ]
    -> trait_decl option
  (**
  * Unit method. Return the ``Node`` builtin trait.
  *)

      
  val p_indexable_gen_trait :
    [< lkt_node ]
    -> generic_decl option
  (**
  * Unit method. Return the ``Node`` builtin generic trait.
  *)

      
  val p_indexable_trait :
    [< lkt_node ]
    -> trait_decl option
  (**
  * Unit method. Return the ``Node`` builtin trait.
  *)

      
  val p_token_node_trait :
    [< lkt_node ]
    -> named_type_decl option
  (**
  * Unit method. Return the ``TokenNode`` builtin trait.
  *)

      
  val p_error_node_trait :
    [< lkt_node ]
    -> named_type_decl option
  (**
  * Unit method. Return the ``ErrorNode`` builtin trait.
  *)

      
  val p_char_type :
    [< lkt_node ]
    -> named_type_decl option
  (**
  * Unit method. Return the character builtin type.
  *)

      
  val p_int_type :
    [< lkt_node ]
    -> named_type_decl option
  (**
  * Unit method. Return the integer builtin type.
  *)

      
  val p_bool_type :
    [< lkt_node ]
    -> named_type_decl option
  (**
  * Unit method. Return the boolean builtin type.
  *)

      
  val p_bigint_type :
    [< lkt_node ]
    -> named_type_decl option
  (**
  * Unit method. Return the big integer builtin type.
  *)

      
  val p_string_type :
    [< lkt_node ]
    -> named_type_decl option
  (**
  * Unit method. Return the string builtin type.
  *)

      
  val p_symbol_type :
    [< lkt_node ]
    -> named_type_decl option
  (**
  * Unit method. Return the string builtin type.
  *)

      
  val p_property_error_type :
    [< lkt_node ]
    -> named_type_decl option
  (**
  * Unit method. Return the property error builtin type.
  *)

      
  val p_regexp_type :
    [< lkt_node ]
    -> named_type_decl option
  (**
  * Unit method. Return the regexp builtin type.
  *)

      
  val p_entity_gen_type :
    [< lkt_node ]
    -> generic_decl option
  (**
  * Unit method. Return the logicvar builtin type.
  *)

      
  val p_entity_type :
    [< lkt_node ]
    -> named_type_decl option
  (**
  * Unit method. Return the logicvar builtin type.
  *)

      
  val p_logicvar_type :
    [< lkt_node ]
    -> named_type_decl option
  (**
  * Unit method. Return the logicvar builtin type.
  *)

      
  val p_equation_type :
    [< lkt_node ]
    -> named_type_decl option
  (**
  * Unit method. Return the logicvar builtin type.
  *)

      
  val p_array_gen_type :
    [< lkt_node ]
    -> generic_decl option
  (**
  * Unit method. Return the array builtin generic type.
  *)

      
  val p_array_type :
    [< lkt_node ]
    -> named_type_decl option
  (**
  * Unit method. Return the array builtin type.
  *)

      
  val p_astlist_gen_type :
    [< lkt_node ]
    -> generic_decl option
  (**
  * Unit method. Return the ASTList builtin generic type.
  *)

      
  val p_astlist_type :
    [< lkt_node ]
    -> named_type_decl option
  (**
  * Unit method. Return the ASTList builtin type.
  *)

      
  val p_node_builder_gen_type :
    [< lkt_node ]
    -> generic_decl option
  (**
  * Unit method. Return the NodeBuilder builtin generic type.
  *)

      
  val p_node_builder_type :
    [< lkt_node ]
    -> named_type_decl option
  (**
  * Unit method. Return the NodeBuilder builtin type.
  *)

      
  val p_iterator_gen_trait :
    [< lkt_node ]
    -> generic_decl option
  (**
  * Unit method. Return the Iterator builtin generic trait.
  *)

      
  val p_iterator_trait :
    [< lkt_node ]
    -> trait_decl option
  (**
  * Unit method. Return the Iterator builtin trait.
  *)

      
  val p_analysis_unit_gen_trait :
    [< lkt_node ]
    -> generic_decl option
  (**
  * Unit method. Return the ``AnalysisUnit`` builtin generic trait.
  *)

      
  val p_analysis_unit_trait :
    [< lkt_node ]
    -> trait_decl option
  (**
  * Unit method. Return the ``AnalysisUnit`` builtin trait.
  *)

      
  val p_topmost_invalid_decl :
    [< lkt_node ]
    -> lkt_node option
  (**
  * Return the topmost (from ``Self`` to the root node) FullDecl annotated with
  * ``@invalid``, null otherwise.
  *)

      
  val p_nameres_diagnostics :
    [< lkt_node ]
    -> SolverDiagnostic.t list
  (**
  * If name resolution on this lkt compilation unit fails, this returns all the
  * diagnostics that were produced while resolving it.
  *)

      
  val p_solve_enclosing_context :
    [< lkt_node ]
    -> SolverResult.t
  (**
  * Finds the nearest parent that is an xref_entry_point and solve its
  * equation.
  *)

      
  val p_xref_entry_point :
    [< lkt_node ]
    -> bool
  (**
  * Designates entities that are entry point for the xref solving
  * infrastructure. If this returns true, then nameres_diagnostics can be
  * called on it.
  *)

      
  val p_complete :
    [< lkt_node ]
    -> CompleteItem.t list
  (**
  * Return an array of completion item for language server clients
  *)



end


