[comment {-*- tcl -*- doctools manpage}]
[manpage_begin "Registry" n ""]
[copyright "2020 Ashok P. Nadkarni"]
[moddesc "Registry access in Windows"]
[titledesc "Commands for accessing the Windows registry"]
[require twapi_registry]
[description]
[para]

This module contains commands dealing with the Windows registry and supplants
the [cmd registry] package that is included with the Tcl core distribution. The
commands in the Tcl [cmd registry] package are in most cases easier to use and
may in many cases be more performant as well. This package is useful to fill in
missing capabilities there such as accessing security descriptors, receiving
change notifications and back up / restore. The package also provides some
utility commands for iteration etc and certain bulk operations may also be
more efficient through the explicit ues of registry handles in this package.

[section "Predefined keys"]

The following registry key handles are predefined:
[const HKEY_CLASSES_ROOT] (or [const HKCR]),
[const HKEY_CURRENT_USER] (or [const HKCU]),
[const HKEY_LOCAL_MACHINE] (or [const HKLM]),
[const HKEY_CURRENT_CONFIG] (or [const HKCC]),
[const HKEY_USERS] (or [const HKU]),
[const HKEY_PERFORMANCE_DATA],
[const HKEY_PERFORMANCE_TEXT],
[const HKEY_PERFORMANCE_NLSTEXT] or
[const HKEY_CURRENT_USER_LOCAL_SETTINGS].
These may be directly passed as key handles without an explicit call to
[cmd reg_key_open] or [cmd reg_key_create].

[section "Registry data types"]

The data stored in the registry may be one of several types listed below. When
setting or retrieving registry data, the value is always paired with its type.

[list_begin opt]
[opt_def binary] Binary data represented in Tcl as a binary string.
[opt_def dword] 32-bit integer in little-endian format
represented in Tcl as a decimal string.
[opt_def dword_be] 32-bit integer in big-endian format
represented in Tcl as a decimal string.
[opt_def expand_sz] Same as the [const sz] type but expected to contain
environment variables that should be expanded. 
[opt_def link] Symbolic link and represented in Tcl as a binary string.
[opt_def multi_sz] List of strings represented in Tcl as such.
[opt_def qword] 64-bit little-endian format integer
represented in Tcl as a decimal string.
[opt_def resource_descriptor] Resource descriptor represented in Tcl as a binary
string.
[opt_def resource_list] Resource list represented in Tcl as a binary string.
[opt_def resource_requirements_list] Resource requirements list represented in
Tcl as a binary string.
[opt_def sz] Null-terminated text represented as a Tcl string.
[list_end]

Command may return registry value either in raw or cooked form. The former
is returned as a pair consisting of a data type from the list above and the
corresponding data value. A cooked value returns only the data value which
is also transformed based on the type. In particular, data of type
[const dword_be] are converted to native format, [const expand_sz] have
embedded environment variables expanded.

[section "Registry commands"]

[list_begin definitions]


[call [cmd reg_connect] [arg SYSTEM] [arg HPREDEFINE]]

Returns a handle to a predefined registry key on a remote system.
[arg SYSTEM] specifies the name of the remote system.
[arg HPREDEFINE] must be one of the predefined keys
[const HKEY_USERS], [const HKEY_LOCAL_MACHINE] or [const HKEY_PERFORMANCE_DATA].

[call [cmd reg_disable_current_user_cache]]

Disables the caching of the [const HKEY_CURRENT_USER] handle for the current
process.

[call [cmd reg_iterator] [arg HKEY] [opt [arg SUBKEYPATH]]]

Returns an iterator command that can be invoked to iterate over all keys in the
tree rooted at [arg HKEY]. If a non-empty [arg SUBKEYPATH] argument is
specified, it must a path to a subkey under [arg HKEY] and the iteration is
carried out over its subtree instead.

[nl]
The returned iterator command should be invoked with zero or one argument. If
invoked with no arguments or the argument [cmd next], the iterator returns the
next item in the iteration. If invoked with an argument of [cmd sibling],
the iterator returns the next key at the same level at the current level
skipping over the children of the key that was last returned. The
[cmd parentsibling] is similar except that it skips the remaining keys
at the same level as the current level and returns the parent's sibling.
Finally, the argument [cmd stop] will
end the iteration (even if more items remain). Any further attempt to invoke the
iterator command will result in an error being raised.

[nl]
The return value from the iterator invocation is a list of two elements.
The first element is the handle to the current key. This handle must not
be closed or modified in any way. The second element is the key path
to the key as a list of key names starting with [arg SUBKEYPATH] or [arg HKEY]
if the former was not specified.

[nl]
The iterator will return an empty list when no more items remain. Invoking
it again will raise an error.

[call [cmd reg_key_close] [arg HKEY]]

Releases the passed registry key handle. [arg HKEY] must not be one of
the predefined registry handles.

[call [cmd reg_key_copy] [arg HFROM] [arg HTO] [opt "[cmd -subkey] [arg SUBKEYPATH]"] [opt "[cmd -copysecd] [arg BOOL]"]]

Copies the subtree under a registry key to another registry key. The source key is
specified by [arg HFROM] which must be a handle to a registry key.
If the [cmd -subkey] option is specified it must be a path to subkey under
[arg HFROM] and the source tree is then copied from there. The target key identified
by its handle [arg HTO] must have been opened with the appropriate access
permissions.

[nl]
The [cmd -copysecd] option controls whether the security descriptors in the source
tree are also to be copied. If [const true], the descriptors are copied as well.
The default is [const false]. The option must be [const false] on Windows XP
and Windows 2003 as otherwise the copy will fail.

[call [cmd reg_key_create] [arg HKEY] [arg SUBKEYPATH] [opt [arg options]]]

Returns a handle to a registry key.
[arg HKEY] must be a handle to a registry key as returned by a prior call to
[cmd reg_key_open], [uri \#reg_key_create [cmd reg_key_create]] etc. or a
predefined key handle such as [const HKEY_LOCAL_MACHINE]. [arg SUBKEYPATH] is the
path to the subkey to be created under [arg HKEY]. It is not an error if
the subkey already exists. In that case, it is opened and its handle returned.

The following options may be specified:
[list_begin opt]
[opt_def [cmd -access] [arg ACCESSRIGHTS]] Specifies the desired access
rights and may be
specified as a list of integer access rights flags or symbols as described in
[uri security.html#accessrights "Access Rights"]. Defaults to [const generic_read].
[opt_def [cmd -backup] [arg BOOL]] If [const true], the key opened with access
required for backup and restore. The [cmd -access] option is ignored. Default is
[const false].
[opt_def [cmd -disposition] [arg VARNAME]] If specified, [arg VARNAME] is
treated as a variable name in caller's context and is set to [const created]
if the key was created and [const existed] if it was already present.
[opt_def [cmd -inherit] [arg BOOL]]
If true, the returned handle is inheritable by child processes. Defaults
to [const false].
[opt_def [cmd -secd] [arg SECURITY_DESCRIPTOR]]
Specifies a [uri security.html\#securitydescriptors "security descriptor"]
to be attached to the file or device if it has to be created. Defaults
to a null security descriptor which results in the process' default
security descriptor being applied.
[opt_def [cmd -volatile] [arg BOOL]] If [const true], the key is only stored
in memory and not preserved to disk when the registry hive is unloaded. Default
is [const false].

[list_end]

The returned key handle should be released by calling the
[uri \#reg_key_close [cmd reg_key_close]] command.

[call [cmd reg_key_current_user] [opt [arg options]]]
Returns a handle to the [const HKEY_CURRENT_USER] key for the user that
the current thread is impersonating. Note this is not necessarily the same
as the [const HKEY_CURRENT_USER] handle cached for the process. If the
user profile is not loaded, the default profile is used.

[nl]
The following options may be specified:
[list_begin opt]
[opt_def [cmd -access] [arg ACCESSRIGHTS]] Specifies the desired access
rights and may be
specified as a list of integer access rights flags or symbols as described in
[uri security.html#accessrights "Access Rights"]. Defaults to [const generic_read].
[list_end]

[call [cmd reg_key_delete] [arg HKEY] [arg SUBKEYPATH]]
Deletes the key identified by the key path [arg SUBKEYPATH] under
the key specified by the handle [arg HKEY] and any values it contains.
The key must not have any subkeys. No error is raised if the key does
not exist.

[call [cmd reg_key_export] [arg HKEY] [arg FILEPATH] [opt [arg options]]]
Exports a key and its contents to the file [arg FILEPATH]. The following
options may be specified.
[list_begin opt]
[opt_def [cmd -format] [arg FORMAT]] Specifies the format for the file.
[arg FORMAT] must be one of [const win2k] or [const xp]. Default is [const xp]
which is not readable on earlier (pre-XP) Windows systems.
[opt_def [cmd -secd] [arg SECD]] Specifies a security descriptor to use when
creating the file.
[list_end]

[call [cmd reg_key_import] [arg HKEY] [arg FILEPATH] [opt [arg options]]]
Imports an exported registry key from a file. The following
options may be specified.
[list_begin opt]
[opt_def [cmd -force] [arg BOOL]] If [const true], the file is imported even if
there are open handles under [arg HKEY]. Default is [const false].
[opt_def [cmd -volatile] [arg BOOL]] If [const true], a new volatile hive is
created into which the file is imported. In this case, [arg HKEY] must be
[const HKEY_USERS] or [const HKEY_LOCAL_MACHINE]. Default is [const false].
[list_end]

[call [cmd reg_key_load] [arg HKEY] [arg SUBKEYPATH] [arg REGFILE]]
Creates a key [arg SUBKEYPATH] under [arg HKEY] and loads the contents of
the file [arg REGFILE] under it. [arg HKEY] must be one of the predefined keys
[const HKEY_LOCAL_MACHINE] or [const HKEY_USERS] or a handle returned by
[uri \#reg_connect [cmd reg_connect]]. [arg REGFILE] must be
a file in the format saved by [uri \#reg_key_export [cmd reg_key_export]].
The [uri \#reg_key_unload [cmd reg_key_unload]] command may be used to
undo the operation.

[nl]
Note the caller process must be running with [const SE_BACKUP_NAME] and
[const SE_RESTORE_NAME] privileges.

[call [cmd reg_key_unload] [arg HKEY] [arg SUBKEYPATH]]
Unloads the registry hive at key path [arg SUBKEYPATH] under the key with
handle [arg HKEY]. [arg HKEY] must be one of the predefined keys
[const HKEY_LOCAL_MACHINE] or [const HKEY_USERS] or a handle returned by
[uri \#reg_connect [cmd reg_connect]]. This reverses the effect
of the [uri \#reg_key_load [cmd reg_key_load]] command.

Note the caller process must be running with [const SE_BACKUP_NAME] and
[const SE_RESTORE_NAME] privileges.

[call [cmd reg_key_monitor] [arg HKEY] [arg HEVENT] [opt [arg options]]]

Monitors a registry key for changes. When a change is detected
the event [arg HEVENT] is set. [arg HEVENT] must be a handle to
an event as returned by [uri synch.html#create_event [cmd create_event]]. The
event will be signalled when the key changes. The caller can wait on the
passed event or register a callback using the
[uri base.html#wait_on_handle [cmd wait_on_handle]] command.

[nl]
Note the following important points about asynchronous operation:

[list_begin bullet]
[bullet] The event is signalled only once. To continue to receive notifications,
the command must be called again on the handle.
[bullet] Moreover, the command must not be called
on the key handle again until the previously registered notification request has
been signalled.
[list_end]

[nl]
The following options control the types of changes that are monitored.

[list_begin opt]
[opt_def [cmd -attr] [arg BOOL]] Monitors changes in key attributes like
security descriptors.
[opt_def [cmd -keys] [arg BOOL]] Monitors addition and deletion of immediate
child keys to the specified key.
[opt_def [cmd -secd] [arg BOOL]] Monitors changes to the security attributes.
[opt_def [cmd -subtree] [arg BOOL]] Monitors the entire tree below the specified
key. If [const false] (default), only monitors the specified key.
[opt_def [cmd -values] [arg BOOL]] Monitors changes to the values in the key
including additions and deletions.
[list_end]

If none of the options [cmd -attr], [cmd -keys], [cmd -secd] and [cmd -values]
are specified as [const true], all are enabled.

[call [cmd reg_key_open] [arg HKEY] [arg SUBKEYPATH] [opt [arg options]]]

Returns a handle to a registry key.
[arg HKEY] must be a handle to a registry key as returned by a prior call to
[cmd reg_key_open], [uri \#reg_key_create [cmd reg_key_create]] etc. or a
predefined key handle such as [const HKEY_LOCAL_MACHINE]. [arg SUBKEYPATH] is the
path to the subkey under [arg HKEY] whose handle is to be returned.

The following options may be specified:
[list_begin opt]
[opt_def [cmd -access] [arg ACCESSRIGHTS]] Specifies the desired access
rights and may be
specified as a list of integer access rights flags or symbols as described in
[uri security.html#accessrights "Access Rights"]. Defaults to [const generic_read].
[list_end]

The command will raise an error if the specified key does not exist.

The returned key handle should be released by calling the
[uri \#reg_key_close [cmd reg_key_close]] command.

[call [cmd reg_key_override] [arg HPREDEFINED] [arg HKEY]]
Maps a predefined key with handle [arg HPREDEFINE] to the key with handle
[arg HKEY]. This will cause all access to [arg HPREDEFINED] to be redirected
to [arg HKEY]. The override can be reversed by calling
[uri \#reg_key_override_undo [cmd reg_key_override_undo]].

[arg HPREDEFINED] must be one of the predefined keys
[const HKEY_CLASSES_ROOT],
[const HKEY_CURRENT_CONFIG],
[const HKEY_CURRENT_USER],
[const HKEY_LOCAL_MACHINE],
[const HKEY_USERS] or
[const HKEY_PERFORMANCE_DATA].

[call [cmd reg_key_override_undo] [arg HPREDEFINED]]
Undoes the effect of a previous call to [uri #reg_key_override [cmd reg_key_override]].
[arg HPREDEFINED] must be one of the predefined keys
[const HKEY_CLASSES_ROOT],
[const HKEY_CURRENT_CONFIG],
[const HKEY_CURRENT_USER],
[const HKEY_LOCAL_MACHINE],
[const HKEY_USERS] or
[const HKEY_PERFORMANCE_DATA].


[call [cmd reg_key_prune] [arg HKEY] [arg SUBKEYPATH]]
Deletes the tree rooted at the key identified by the key path [arg SUBKEYPATH] under
the key specified by the handle [arg HKEY]. No error is raised if the key does
not exist.

[call [cmd reg_key_user_classes_root] [arg USERTOKEN] [opt [arg options]]]
Returns a handle to the [emph merged] [const HKEY_CLASSES_ROOT] key for the user
corresponding to the passed [arg USERTOKEN] which must be a user token as
returned by
[uri security.html#open_user_token [cmd open_user_token]],
[uri security.html#open_process_token [cmd open_process_token]],
[uri security.html#open_thread_token [cmd open_thread_token]] etc.

The following options may be specified:
[list_begin opt]
[opt_def [cmd -access] [arg ACCESSRIGHTS]] Specifies the desired access
rights and may be
specified as a list of integer access rights flags or symbols as described in
[uri security.html#accessrights "Access Rights"]. Defaults to [const generic_read].
[list_end]


[call [cmd reg_keys] [arg HKEY] [opt [arg SUBKEYPATH]]]
Returns the list of keys under the key [arg SUBKEYPATH] under the handle [arg HKEY].
If [arg SUBKEYPATH] is not specified or is empty, keys under [arg HKEY] are
returned.

[call [cmd reg_tree] [arg HKEY] [opt [arg SUBKEYPATH]]]
Returns a list of key paths of the keys in the subtree under a registry key.
If [arg SUBKEYPATH] is specified, it identifies the path under [arg HKEY]
for the subtree. If [arg SUBTREE] is not specified or empty the paths
under [arg HKEY] itself are returned.
[nl]
The returned paths are all relative to root of the subtree. Note that the
empty string corresponding to the root is also returned.

[call [cmd reg_tree_values] [arg HKEY] [opt [arg SUBKEYPATH]]]
Returns a registry subtree in the form of a nested dictionary mapping keys to their
values. [arg SUBKEYPATH] is the registry path relative to [arg HKEY]
iterate over. If unspecified, the root of the  tree is [arg HKEY]
itself.
[nl]
The keys of the returned dictionary are the relative key paths in the subtree.
Thus the tree includes an empty string as a dictionary key corresponding
to the root of the subtree.
The dictionary content for each key is itself a dictionary mapping
value names contained in the key to their content in cooked form.

[call [cmd reg_tree_values_raw] [arg HKEY] [opt [arg SUBKEYPATH]]]
Returns a registry subtree in the form of a nested dictionary mapping keys to their
values. [arg SUBKEYPATH] is the registry path relative to [arg HKEY]
iterate over. If unspecified, the root of the  tree is [arg HKEY]
itself.
[nl]
The keys of the returned dictionary are the relative key paths in the subtree.
The corresponding dictionary element is itself a dictionary mapping
value names contained in the key to their content in raw form as a type and data
pair.

[call [cmd reg_value] [arg HKEY] [opt [arg SUBKEYPATH]] [arg VALUENAME]]
Returns the cooked data value for a registry value. If
[arg SUBKEYPATH] is not specified, the returned data is from the value
[arg VALUENAME] under [arg HKEY] which is a handle to a registry key.
If [arg SUBKEYPATH] is specified, it must be a registry path under [arg HKEY]
and [arg VALUENAME] is retrieved from that key.

[call [cmd reg_value_delete] [arg HKEY] [opt [arg SUBKEYPATH]] [arg VALUENAME]]
If the optional [arg SUBKEYPATH] argument is not specified, the command deletes the
specified registry value under the key specified by the handle [arg HKEY].
If [arg SUBKEYPATH] is specified, it should be a key path under [arg HKEY].
In this case the value is deleted from that key.

[call [cmd reg_value_names] [arg HKEY] [opt [arg SUBKEYPATH]]]
Returns the list of names of values under the key specified by the handle [arg HKEY]
or if [arg SUBKEYPATH] is specified and non-empty, the value names under that subkey.

[call [cmd reg_value_raw] [arg HKEY] [opt [arg SUBKEYPATH]] [arg VALUENAME]]

Returns a pair consisting of the type and data for a registry value. If
[arg SUBKEYPATH] is not specified, the returned data is from the value
[arg VALUENAME] under [arg HKEY] which is a handle to a registry key.
If [arg SUBKEYPATH] is specified, it must be a registry path under [arg HKEY]
and [arg VALUENAME] is retrieved from that key.

[call [cmd reg_value_set] [arg HKEY] [opt [arg SUBKEYPATH]] [arg VALUENAME] [arg TYPE] [arg DATA]]

Stores the data [arg DATA] as a registry value of type [arg TYPE]. If
[arg SUBKEYPATH] is not specified, the data is stored as
[arg VALUENAME] under [arg HKEY] which is a handle to a registry key.
If [arg SUBKEYPATH] is specified, it must be a registry path under [arg HKEY]
and the data is stored in [arg VALUENAME] under that key.

[call [cmd reg_values] [arg HKEY] [opt [arg SUBKEYPATH]]]
Returns a dictionary keyed by the value names in the specified registry key
[arg HKEY] or under its subkey [arg SUBKEYPATH] if specified and non-empty.
The value data is in cooked form.

[call [cmd reg_values_raw] [arg HKEY] [opt [arg SUBKEYPATH]]]
Returns a dictionary keyed by the value names in the specified registry key
[arg HKEY] or under its subkey [arg SUBKEYPATH] if specified and non-empty.
The value data is in raw form as a pair containing the data type and data value.

[call [cmd reg_walk] [arg HKEY] [opt "[cmd -subkey] [arg SUBKEYPATH]"] [opt "[cmd -callback] [arg CMDPREFIX]"] [opt "[cmd -cbdata] [arg CBDATA]"]]

The command iterates over all keys in the tree rooted at [arg HKEY]. If the
[cmd -subkey] option is specified, it must be a path to a subkey under [arg HKEY]
and the tree walk begins at that key.
[nl]
The [cmd -callback] option specifies a callback to be invoked for each key
in the tree. If unspecified, the callback defaults to printing the relative path
of the current key to standard output.
[nl]
The [cmd -cbdata] option specifies any context data to be passed
to the first invocation of the callback and defaults to the empty string.
[nl]
The callback is invoked with three arguments. The first argument is the value
returned on the previous invocation of the callback or the value of the
[cmd -cbdata] option for the first invocation. The second argument is the handle
to the current key. The third argument is the path to the current key relative
to the root as a Tcl list. The callback must not close the passed handle or
store it for later use.
[nl]
The iteration will normally terminate when there are no untraversed keys in
the tree. The callback can change this behavior by returning a non-0
return code (not result!) as below.

[list_begin opt]
[opt_def [cmd return]] This is a normal return. The iteration will continue
with the next key.
[opt_def [cmd "return -code continue VALUE"]] The iteration will skip over the key's
children and continue with its siblings.
[opt_def [cmd "return -code break VALUE"]] The iteration will skip over the key's
siblings as well as children and continue with siblings of key's parent.
[opt_def [cmd "return -code return VALUE"]] The iteration will stop without error
even though more keys may remain.
[opt_def [cmd "error MESSAGE"]] The iteration terminates with the same error
exception that was thrown.
[list_end]

The command returns the result of the last callback invocation as the
result of the command.

[list_end]

[keywords "Registry"]

[manpage_end]
