! This file is part of mstore.
! SPDX-Identifier: Apache-2.0
!
! Licensed under the Apache License, Version 2.0 (the "License");
! you may not use this file except in compliance with the License.
! You may obtain a copy of the License at
!
!     http://www.apache.org/licenses/LICENSE-2.0
!
! Unless required by applicable law or agreed to in writing, software
! distributed under the License is distributed on an "AS IS" BASIS,
! WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
! See the License for the specific language governing permissions and
! limitations under the License.

module mstore_upu23
   use mctc_env_accuracy, only : wp
   use mctc_io_structure, only : structure_type, new
   use mstore_data_record, only : record_type, new_record, select_record
   implicit none
   private

   public :: get_upu23_records

contains

subroutine get_upu23_records(records)
   type(record_type), allocatable, intent(out) :: records(:)

   records = [ &
      new_record('0a', upu0a), &
      new_record('0b', upu0b), &
      new_record('1a', upu1a), &
      new_record('1b', upu1b), &
      new_record('1c', upu1c), &
      new_record('1e', upu1e), &
      new_record('1f', upu1f), &
      new_record('1g', upu1g), &
      new_record('1m', upu1m), &
      new_record('1p', upu1p), &
      new_record('2a', upu2a), &
      new_record('2h', upu2h), &
      new_record('2p', upu2p), &
      new_record('3a', upu3a), &
      new_record('3b', upu3b), &
      new_record('3d', upu3d), &
      new_record('4b', upu4b), &
      new_record('5z', upu5z), &
      new_record('6p', upu6p), &
      new_record('7a', upu7a), &
      new_record('7p', upu7p), &
      new_record('8d', upu8d), &
      new_record('9a', upu9a), &
      new_record('aa', upuaa) &
      ]

end subroutine get_upu23_records

subroutine upu0a(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 59
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "O", "C", "H", "H", "C", "H", "O", "C", "H", "N", "C", "H", "C", &
      & "H", "C", "O", "N", "H", "C", "O", "C", "H", "C", "H", "O", "H", "O", &
      & "P", "O", "O", "O", "C", "H", "H", "C", "H", "O", "C", "H", "N", "C", &
      & "H", "C", "H", "C", "O", "N", "H", "C", "O", "C", "H", "C", "H", "O", &
      & "H", "O", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  5.72352554274576_wp, -3.42122719440678_wp, -8.55686315818644_wp, &
      &  4.96295410674576_wp, -3.35977500640678_wp, -6.88939899118644_wp, &
      &  3.75278189674576_wp, -0.94199891340678_wp, -6.60164039918644_wp, &
      &  2.45673619174576_wp, -0.57912371340678_wp, -8.17775974718644_wp, &
      &  5.16576838174576_wp,  0.57771079059322_wp, -6.52663547318644_wp, &
      &  2.27600695574576_wp, -0.93891828240678_wp, -4.14608112518644_wp, &
      &  1.55703084474576_wp,  0.97082255159322_wp, -3.80904887218644_wp, &
      &  0.08768611774576_wp, -2.62448111140678_wp, -4.32813808318644_wp, &
      &  0.31447912574576_wp, -4.62586553040678_wp, -2.53070971118644_wp, &
      & -0.97906269225424_wp, -4.32713178740678_wp, -0.94948205518644_wp, &
      & -0.50054816725424_wp, -6.98869519440678_wp, -3.71777529618644_wp, &
      &  0.87230393874576_wp, -7.98239759040678_wp, -5.69170577618644_wp, &
      &  2.50913886074576_wp, -6.87031236440678_wp, -6.23672202718644_wp, &
      &  0.20845529274576_wp,-10.13876769740678_wp, -6.89814483218644_wp, &
      &  1.33430395974576_wp,-10.87733799940678_wp, -8.43623854718644_wp, &
      & -2.06040887925424_wp,-11.45481520740678_wp, -6.16420152418644_wp, &
      & -2.89377747425424_wp,-13.42925081040678_wp, -7.09714242718644_wp, &
      & -3.42240583225424_wp,-10.23432150340678_wp, -4.24471594018644_wp, &
      & -5.07849498325424_wp,-11.04845306540678_wp, -3.70793965018644_wp, &
      & -2.80043540425424_wp, -8.04483205840678_wp, -2.97241955518644_wp, &
      & -4.17555366925424_wp, -7.10881222540678_wp, -1.35172062618644_wp, &
      &  3.77137396574576_wp, -1.84150678440678_wp, -1.84488638618644_wp, &
      &  5.80164794874576_wp, -1.51562202540678_wp, -2.01879994818644_wp, &
      &  3.07604626374576_wp, -4.64350158740678_wp, -1.61163523918644_wp, &
      &  4.25169828174576_wp, -5.76093643840678_wp, -2.88663563318644_wp, &
      &  3.37680522674576_wp, -5.64509472340678_wp,  0.85396112481356_wp, &
      &  3.08859536774576_wp, -4.25814110440678_wp,  2.05547083081356_wp, &
      &  2.82686945174576_wp, -0.60056624540678_wp,  0.37327813881356_wp, &
      &  4.72143351874576_wp,  0.46549281459322_wp,  2.60422306981356_wp, &
      &  7.29617497574576_wp,  0.92892978359322_wp,  1.51266952681356_wp, &
      &  4.32967102174576_wp, -1.19613776840678_wp,  4.87044691681356_wp, &
      &  3.27746611074576_wp,  3.11206256759322_wp,  3.21731102981356_wp, &
      &  3.54065070574576_wp,  5.17130264859322_wp,  1.44751744481356_wp, &
      &  3.59020480174576_wp,  4.45123087459322_wp, -0.49209712318644_wp, &
      &  5.30220281874576_wp,  6.18688032459322_wp,  1.82973644681356_wp, &
      &  1.33759661874576_wp,  6.96309566859322_wp,  1.75527784081356_wp, &
      &  1.73249303674576_wp,  8.66501936759322_wp,  0.63588032081356_wp, &
      & -0.96314137325424_wp,  5.80781565259322_wp,  0.79236758481356_wp, &
      & -3.06737261725424_wp,  6.53153667659322_wp,  2.26999006781356_wp, &
      & -4.54793381325424_wp,  7.33459463359322_wp,  1.07844224181356_wp, &
      & -4.16081305125424_wp,  4.22278265559322_wp,  3.48801484481356_wp, &
      & -2.91885035925424_wp,  1.94993399659322_wp,  3.51375215681356_wp, &
      & -1.09248253425424_wp,  1.96219208159322_wp,  2.59247331981356_wp, &
      & -3.91033516225424_wp, -0.14553515540678_wp,  4.61182204181356_wp, &
      & -2.88581454725424_wp, -1.91466110140678_wp,  4.60277913581356_wp, &
      & -6.35012391625424_wp, -0.03539946840678_wp,  5.81177517981356_wp, &
      & -7.46790836725424_wp, -1.78935015840678_wp,  6.88133457881356_wp, &
      & -7.48412108225424_wp,  2.36078215859322_wp,  5.68785035081356_wp, &
      & -9.21169705525424_wp,  2.54685024859322_wp,  6.50879154081356_wp, &
      & -6.52998331225424_wp,  4.51866593559322_wp,  4.58666695381356_wp, &
      & -7.66789126625424_wp,  6.54817401359322_wp,  4.56362250081356_wp, &
      &  0.69164084274576_wp,  7.73760799159322_wp,  4.46157106181356_wp, &
      &  0.94516270274576_wp,  6.14381649259322_wp,  5.75263450381356_wp, &
      & -2.12377498225424_wp,  8.43551781459322_wp,  4.24331129981356_wp, &
      & -3.14812098725424_wp,  8.27677554059322_wp,  6.03285649081356_wp, &
      & -2.36991878525424_wp, 10.91280704759322_wp,  3.20395539181356_wp, &
      & -0.90451961625424_wp, 11.85165976959322_wp,  3.83184859881356_wp, &
      &  2.23698846074576_wp,  9.82674164759322_wp,  5.19106916681356_wp, &
      &  2.29959659374576_wp,  9.88216806759322_wp,  7.02583644481356_wp],&
      & shape(xyz))
   real(wp), parameter :: charge = -1.0_wp
   call new(self, sym, xyz, charge=charge)
end subroutine upu0a

subroutine upu0b(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 59
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "O", "C", "H", "H", "C", "H", "O", "C", "H", "N", "C", "H", "C", &
      & "H", "C", "O", "N", "H", "C", "O", "C", "H", "C", "H", "O", "H", "O", &
      & "P", "O", "O", "O", "C", "H", "H", "C", "H", "O", "C", "H", "N", "C", &
      & "H", "C", "H", "C", "O", "N", "H", "C", "O", "C", "H", "C", "H", "O", &
      & "H", "O", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -5.29147560620339_wp, -7.90189136506780_wp,  2.26191603757627_wp, &
      & -4.58442385520339_wp, -6.62765612206780_wp,  1.14764307557627_wp, &
      & -3.46306820020339_wp, -4.68997285406780_wp,  2.68874107757627_wp, &
      & -2.11885583720339_wp, -5.50953114306780_wp,  4.03682038757627_wp, &
      & -4.92355597120339_wp, -3.65530441106780_wp,  3.74255943657627_wp, &
      & -2.07942292420339_wp, -2.85310703506780_wp,  0.98048698557627_wp, &
      & -1.40661055220339_wp, -1.26244908006780_wp,  2.11874888457627_wp, &
      &  0.15011000279661_wp, -4.08276345306780_wp, -0.09607479342373_wp, &
      &  0.00764110379661_wp, -4.10802342106780_wp, -2.78099672542373_wp, &
      &  1.27362939379661_wp, -2.71305226106780_wp, -3.62487824342373_wp, &
      &  0.90304562379661_wp, -6.57600991106780_wp, -3.67248233042373_wp, &
      & -0.19690434920339_wp, -8.73657407606780_wp, -2.73483676642373_wp, &
      & -1.66938488820339_wp, -8.40069172106780_wp, -1.34167615242373_wp, &
      &  0.51928176979661_wp,-11.08290321906780_wp, -3.46321483042373_wp, &
      & -0.37825681720339_wp,-12.74723346406780_wp, -2.68636340442373_wp, &
      &  2.52062329279661_wp,-11.38016958906780_wp, -5.28662372242373_wp, &
      &  3.36105076979661_wp,-13.39737895206780_wp, -6.11750184742373_wp, &
      &  3.55639930379661_wp, -9.08475854606780_wp, -6.11595888542373_wp, &
      &  4.99220063979661_wp, -9.19666339506780_wp, -7.38830459242373_wp, &
      &  2.87609927079661_wp, -6.66377558906780_wp, -5.41834801742373_wp, &
      &  3.92523885779661_wp, -4.77345282306780_wp, -6.26626655742373_wp, &
      & -3.63239728720339_wp, -1.88532733306780_wp, -1.26727253842373_wp, &
      & -5.66555858220339_wp, -1.95755738406780_wp, -0.93041111742373_wp, &
      & -2.77404386920339_wp, -3.56386812906780_wp, -3.46946831142373_wp, &
      & -3.85261310420339_wp, -5.32413806306780_wp, -3.46323353842373_wp, &
      & -3.07767124620339_wp, -2.42285289906780_wp, -5.87039182742373_wp, &
      & -2.76916950920339_wp, -0.61841723406780_wp, -5.59654898942373_wp, &
      & -2.90606466220339_wp,  0.65073672493220_wp, -1.91280161342373_wp, &
      & -4.81748924220339_wp,  3.09923182493220_wp, -1.55674624542373_wp, &
      & -7.36505114820339_wp,  2.17405864093220_wp, -0.72770758042373_wp, &
      & -4.43200573820339_wp,  4.72221826793220_wp, -3.83753458942373_wp, &
      & -3.40403089320339_wp,  4.58329902993220_wp,  0.76122453557627_wp, &
      & -3.97022738220339_wp,  3.86273025793220_wp,  3.33094499057627_wp, &
      & -4.01935987920339_wp,  1.80173802293220_wp,  3.51230974157627_wp, &
      & -5.81723347020339_wp,  4.61512579493220_wp,  3.88260608557627_wp, &
      & -1.93304474920339_wp,  4.93278778893220_wp,  5.03442941457627_wp, &
      & -2.40859788520339_wp,  4.45953010293220_wp,  6.99309714457627_wp, &
      &  0.48634006679661_wp,  3.75144410493220_wp,  4.44754212657627_wp, &
      &  2.30822810679661_wp,  5.63068362793220_wp,  3.82969281257627_wp, &
      &  3.57774810579661_wp,  6.03443095593220_wp,  5.40756790557627_wp, &
      &  3.89921789779661_wp,  4.61524560393220_wp,  1.79305895657627_wp, &
      &  2.77271992079661_wp,  3.46903683893220_wp, -0.24889447742373_wp, &
      &  0.72825447279661_wp,  3.38136923793220_wp, -0.14874939342373_wp, &
      &  4.10937772679661_wp,  2.49503464693220_wp, -2.20171652942373_wp, &
      &  3.16704394079661_wp,  1.60403772493220_wp, -3.78086265142373_wp, &
      &  6.82895779979661_wp,  2.65375197693220_wp, -2.19717495042373_wp, &
      &  8.27841996379661_wp,  1.85369263793220_wp, -3.84552113842373_wp, &
      &  7.81476227479661_wp,  3.87241132393220_wp, -0.05646235742373_wp, &
      &  9.72634634779661_wp,  4.04671397993220_wp,  0.03197266157627_wp, &
      &  6.51510757179661_wp,  4.88225972893220_wp,  1.96683230057627_wp, &
      &  7.58426937279661_wp,  5.92300439693220_wp,  3.74791851057627_wp, &
      & -1.54369089720339_wp,  7.79695967893220_wp,  4.79804566857627_wp, &
      & -3.19653368820339_wp,  8.74875267793220_wp,  4.01168892157627_wp, &
      &  0.77286221079661_wp,  7.99294750793220_wp,  3.05986512057627_wp, &
      &  0.17666140879661_wp,  7.79703545693220_wp,  1.09203127457627_wp, &
      &  2.11131639079661_wp, 10.29383724493220_wp,  3.30482822157627_wp, &
      &  1.88865016579661_wp, 10.79876957293220_wp,  5.07061048557627_wp, &
      & -0.83262499120339_wp,  8.92051856593220_wp,  7.16928688057627_wp, &
      & -2.30223654820339_wp,  9.75212952593220_wp,  7.88255507357627_wp],&
      & shape(xyz))
   real(wp), parameter :: charge = -1.0_wp
   call new(self, sym, xyz, charge=charge)
end subroutine upu0b

subroutine upu1a(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 59
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "O", "C", "H", "H", "C", "H", "O", "C", "H", "N", "C", "H", "C", &
      & "H", "C", "O", "N", "H", "C", "O", "C", "H", "C", "H", "O", "H", "O", &
      & "P", "O", "O", "O", "C", "H", "H", "C", "H", "O", "C", "H", "N", "C", &
      & "H", "C", "H", "C", "O", "N", "H", "C", "O", "C", "H", "C", "H", "O", &
      & "H", "O", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -9.48824464964407_wp,  4.95598422716949_wp,  1.78682265877966_wp, &
      & -8.01251170364407_wp,  4.13641289916949_wp,  1.07062425677966_wp, &
      & -8.49031663664407_wp,  3.71289166816949_wp, -1.56639236522034_wp, &
      & -9.36210543964407_wp,  5.38153218916949_wp, -2.43155956822034_wp, &
      & -9.75541977964407_wp,  2.08868106016949_wp, -1.83606552322034_wp, &
      & -6.03601440864407_wp,  3.13756704616949_wp, -2.90438998722034_wp, &
      & -6.47715979164407_wp,  2.65934750716949_wp, -4.87304869822034_wp, &
      & -4.38794071964407_wp,  5.34991197116949_wp, -2.90329130022034_wp, &
      & -1.83912176764407_wp,  4.60318221116949_wp, -2.61243469122034_wp, &
      & -0.65761235464407_wp,  5.50606286416949_wp, -4.04090710222034_wp, &
      & -0.89291744464407_wp,  5.49768608716949_wp, -0.09766293222034_wp, &
      & -2.50210756464407_wp,  6.28755526816949_wp,  1.77482308777966_wp, &
      & -4.49465090064407_wp,  6.14192825916949_wp,  1.31635024077966_wp, &
      & -1.66334058964407_wp,  7.16704564916949_wp,  4.03237101477966_wp, &
      & -2.98085962864407_wp,  7.78158132916949_wp,  5.46975140477966_wp, &
      &  1.00341143735593_wp,  7.32706840216949_wp,  4.53284793577966_wp, &
      &  2.02267483435593_wp,  8.12541187116949_wp,  6.47627332477966_wp, &
      &  2.50738369235593_wp,  6.43242506416949_wp,  2.53046441877966_wp, &
      &  4.41165452635593_wp,  6.50740202216949_wp,  2.77494053777966_wp, &
      &  1.71276336235593_wp,  5.50733049316949_wp,  0.22604864177966_wp, &
      &  3.18179269435593_wp,  4.79497125716949_wp, -1.43114184122034_wp, &
      & -4.39365770764407_wp,  1.04263174216949_wp, -1.77239517422034_wp, &
      & -4.40546018064407_wp,  1.11933383016949_wp,  0.29450150877966_wp, &
      & -1.78024300764407_wp,  1.71834692316949_wp, -2.82226041622034_wp, &
      & -0.20197031664407_wp,  0.87805569516949_wp, -1.79046624622034_wp, &
      & -1.62960576864407_wp,  1.10412153616949_wp, -5.44937166122034_wp, &
      & -2.73466272564407_wp, -0.34907812783051_wp, -5.70964173222034_wp, &
      & -5.29018037864407_wp, -1.37215007883051_wp, -2.59793085522034_wp, &
      & -4.93756468964407_wp, -3.81886581283051_wp, -0.66907279722034_wp, &
      & -6.38532855764407_wp, -5.90346299383051_wp, -1.91954209022034_wp, &
      & -5.38135323264407_wp, -2.99598463083051_wp,  1.99845590777966_wp, &
      & -1.89240807264407_wp, -4.38999713283051_wp, -0.82859250422034_wp, &
      & -0.84635289564407_wp, -5.28996326183051_wp, -3.17813512722034_wp, &
      & -1.34876584064407_wp, -4.01013650783051_wp, -4.72319097022034_wp, &
      & -1.59438014164407_wp, -7.17131852183051_wp, -3.60832085522034_wp, &
      &  1.99399087235593_wp, -5.44830491383051_wp, -2.95083189222034_wp, &
      &  2.73421128235593_wp, -6.20504304183051_wp, -4.73585610322034_wp, &
      &  3.04622167335593_wp, -2.94362750483051_wp, -2.54366264922034_wp, &
      &  5.22061031735593_wp, -3.09239032483051_wp, -0.98724615122034_wp, &
      &  6.83720910835593_wp, -2.20069628183051_wp, -1.90157325322034_wp, &
      &  4.74778764735593_wp, -1.65202435683051_wp,  1.38840700477966_wp, &
      &  2.35733359135593_wp, -1.34054403383051_wp,  2.34252342077966_wp, &
      &  0.86069517335593_wp, -2.16956588083051_wp,  1.21179831577966_wp, &
      &  1.89769156935593_wp, -0.08672954783051_wp,  4.53006984977966_wp, &
      & -0.01083530464407_wp,  0.12911873416949_wp,  5.23088380377966_wp, &
      &  3.96554842335593_wp,  0.94743037016949_wp,  5.97123328177966_wp, &
      &  3.82153259435593_wp,  2.10564853216949_wp,  7.99156107077966_wp, &
      &  6.34295637835593_wp,  0.52209481316949_wp,  4.86280864477966_wp, &
      &  7.87396210835593_wp,  1.19141043616949_wp,  5.81251311777966_wp, &
      &  6.86990495835593_wp, -0.73226674883051_wp,  2.64123259677966_wp, &
      &  9.03208370535593_wp, -1.05805172983051_wp,  1.84899653377966_wp, &
      &  3.03234541535593_wp, -7.02611934683051_wp, -0.76388960922034_wp, &
      &  1.90244952235593_wp, -6.75025544583051_wp,  0.94620583177966_wp, &
      &  5.69614663535593_wp, -5.90009266683051_wp, -0.43738350922034_wp, &
      &  6.49242822335593_wp, -6.19507435883051_wp,  1.45053835977966_wp, &
      &  7.33461068935593_wp, -6.87703413883051_wp, -2.34352365922034_wp, &
      &  6.75858951535593_wp, -8.60902980383051_wp, -2.64579818122034_wp, &
      &  3.07395604835593_wp, -9.62599812783051_wp, -1.48167991722034_wp, &
      &  3.14114620035593_wp,-10.64636663683051_wp,  0.04421259277966_wp],&
      & shape(xyz))
   real(wp), parameter :: charge = -1.0_wp
   call new(self, sym, xyz, charge=charge)
end subroutine upu1a

subroutine upu1b(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 59
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "O", "C", "H", "H", "C", "H", "O", "C", "H", "N", "C", "H", "C", &
      & "H", "C", "O", "N", "H", "C", "O", "C", "H", "C", "H", "O", "H", "O", &
      & "P", "O", "O", "O", "C", "H", "H", "C", "H", "O", "C", "H", "N", "C", &
      & "H", "C", "H", "C", "O", "N", "H", "C", "O", "C", "H", "C", "H", "O", &
      & "H", "O", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -9.28344351981356_wp, -3.74476671023729_wp,  1.01099432762712_wp, &
      & -7.93373413281356_wp, -2.77884234123729_wp,  0.23067223162712_wp, &
      & -7.81073186781356_wp, -3.48326321523729_wp, -2.38554682237288_wp, &
      & -9.62781527881356_wp, -3.14769757923729_wp, -3.32593305637288_wp, &
      & -7.31277376281356_wp, -5.48765305423729_wp, -2.59863043537288_wp, &
      & -5.77516608481356_wp, -1.94050074323729_wp, -3.66122694937288_wp, &
      & -5.65056227781356_wp, -2.54290872823729_wp, -5.64202809537288_wp, &
      & -6.40787733781356_wp,  0.73070309976271_wp, -3.60530995737288_wp, &
      & -4.21967687481356_wp,  2.21343048876271_wp, -3.23002700537288_wp, &
      & -4.05941979681356_wp,  3.64934974376271_wp, -4.70379388137288_wp, &
      & -4.48923835081356_wp,  3.60973919676271_wp, -0.77988500937288_wp, &
      & -6.41237942081356_wp,  3.10957237976271_wp,  0.88771805262712_wp, &
      & -7.60907278181356_wp,  1.53395626576271_wp,  0.36096316862712_wp, &
      & -6.76315073781356_wp,  4.47204762276271_wp,  3.03091639162712_wp, &
      & -8.29944902381356_wp,  4.03264759076271_wp,  4.30580207862712_wp, &
      & -5.09292057781356_wp,  6.53175011976271_wp,  3.63749141162712_wp, &
      & -5.21936970281356_wp,  7.94683363276271_wp,  5.48944650362712_wp, &
      & -3.12352562981356_wp,  6.83940072276271_wp,  1.87799550962712_wp, &
      & -1.87623636681356_wp,  8.26219167276271_wp,  2.21202139762712_wp, &
      & -2.70857581781356_wp,  5.48236524576271_wp, -0.30874173837288_wp, &
      & -0.97095897381356_wp,  5.95091241176271_wp, -1.78359369437288_wp, &
      & -3.14198182781356_wp, -2.10499099723729_wp, -2.47478894337288_wp, &
      & -3.24561969181356_wp, -2.26577682123729_wp, -0.41727890637288_wp, &
      & -1.94658028381356_wp,  0.40700229776271_wp, -3.29790086137288_wp, &
      & -0.38668177581356_wp,  1.03625157276271_wp, -2.09950145537288_wp, &
      & -1.14946211281356_wp,  0.24615562376271_wp, -5.87913447237288_wp, &
      & -0.91166485281356_wp, -1.55152710423729_wp, -6.23085669837288_wp, &
      & -1.86358673081356_wp, -4.24777189623729_wp, -3.51768827737288_wp, &
      &  0.06021290918644_wp, -5.97443494223729_wp, -1.74934505637288_wp, &
      &  0.66522021218644_wp, -8.20698443923729_wp, -3.37838769237288_wp, &
      & -0.99821297981356_wp, -6.19443174023729_wp,  0.86396363062712_wp, &
      &  2.54724632518644_wp, -4.16670813123729_wp, -1.42419010937288_wp, &
      &  3.98136126918644_wp, -3.35970008023729_wp, -3.59306793537288_wp, &
      &  2.77395448518644_wp, -2.28498361823729_wp, -4.88128809537288_wp, &
      &  4.75232412018644_wp, -5.00159908923729_wp, -4.59299527737288_wp, &
      &  6.11165655918644_wp, -1.68053699723729_wp, -2.66911664737288_wp, &
      &  7.17452519518644_wp, -0.99511410523729_wp, -4.30613241537288_wp, &
      &  5.03392277218644_wp,  0.50806767776271_wp, -1.37151248537288_wp, &
      &  5.96312730918644_wp,  0.64501423076271_wp,  1.15185412562712_wp, &
      &  7.42431956918644_wp,  2.09000001276271_wp,  1.34632922862712_wp, &
      &  3.89544563318644_wp,  1.42315804076271_wp,  2.82690573162712_wp, &
      &  1.74437828618644_wp, -0.03450685823729_wp,  2.95591656862712_wp, &
      &  1.73707619618644_wp, -1.65378792523729_wp,  1.69600336962712_wp, &
      & -0.21490549481356_wp,  0.51300496476271_wp,  4.50654729262712_wp, &
      & -1.87174411081356_wp, -0.68254832423729_wp,  4.57347666362712_wp, &
      & -0.08624935881356_wp,  2.69663323576271_wp,  6.13389919162712_wp, &
      & -1.70342905081356_wp,  3.39465833276271_wp,  7.66047011362712_wp, &
      &  2.14209889918644_wp,  4.11087884476271_wp,  5.83261790962712_wp, &
      &  2.31774420618644_wp,  5.68182374576271_wp,  6.92576997062712_wp, &
      &  4.16567588318644_wp,  3.62188181976271_wp,  4.26040708162712_wp, &
      &  6.04571135818644_wp,  4.98353484276271_wp,  4.18737370662712_wp, &
      &  7.92925888918644_wp, -2.94069189223729_wp, -0.79102040837288_wp, &
      &  7.89565313618644_wp, -4.99862522723729_wp, -0.93104608137288_wp, &
      &  7.00225682818644_wp, -2.00484156723729_wp,  1.78943161162712_wp, &
      &  5.46355178718644_wp, -3.21487148023729_wp,  2.44695969162712_wp, &
      &  8.90679411318644_wp, -1.98866135523729_wp,  3.66765087162712_wp, &
      & 10.48404767518644_wp, -1.69649100923729_wp,  2.74154472962712_wp, &
      & 10.47157340518644_wp, -2.01507046523729_wp, -1.09952913737288_wp, &
      & 11.47705956618644_wp, -3.33267699823729_wp, -1.88164496137288_wp],&
      & shape(xyz))
   real(wp), parameter :: charge = -1.0_wp
   call new(self, sym, xyz, charge=charge)
end subroutine upu1b

subroutine upu1c(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 59
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "O", "C", "H", "H", "C", "H", "O", "C", "H", "N", "C", "H", "C", &
      & "H", "C", "O", "N", "H", "C", "O", "C", "H", "C", "H", "O", "H", "O", &
      & "P", "O", "O", "O", "C", "H", "H", "C", "H", "O", "C", "H", "N", "C", &
      & "H", "C", "H", "C", "O", "N", "H", "C", "O", "C", "H", "C", "H", "O", &
      & "H", "O", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  9.86133722379661_wp, -5.32010262362712_wp,  0.36734836783051_wp, &
      &  8.58311730379661_wp, -4.01078858562712_wp,  0.24749703183051_wp, &
      &  9.79762133279661_wp, -1.72229299462712_wp, -0.57320473016949_wp, &
      & 11.11448463679661_wp, -2.11092723862712_wp, -2.12425327216949_wp, &
      & 10.84692664879661_wp, -0.84521333662712_wp,  0.98807478983051_wp, &
      &  7.81149966279661_wp,  0.13001566537288_wp, -1.45777110416949_wp, &
      &  8.73983908379661_wp,  1.93774417737288_wp, -1.87407301516949_wp, &
      &  6.61426206079661_wp, -0.78960531462712_wp, -3.75825473816949_wp, &
      &  4.02721319679661_wp, -0.10967850162712_wp, -3.79908926016949_wp, &
      &  3.56297007079661_wp,  0.92462805937288_wp, -5.52051285316949_wp, &
      &  2.48371864479661_wp, -2.47035387762712_wp, -3.85046146216949_wp, &
      &  3.37046538479661_wp, -4.69552834162712_wp, -2.85631951516949_wp, &
      &  5.26836878679661_wp, -4.59918935462712_wp, -2.07987456916949_wp, &
      &  1.98601130579661_wp, -6.85319480062712_wp, -2.86036107216949_wp, &
      &  2.73675465579661_wp, -8.58183998062712_wp, -2.06806094716949_wp, &
      & -0.50540532920339_wp, -6.87353713362712_wp, -3.95056667216949_wp, &
      & -1.95642047120339_wp, -8.69753694262712_wp, -4.10359611616949_wp, &
      & -1.27418838120339_wp, -4.50554870762712_wp, -4.87981108216949_wp, &
      & -3.01606428920339_wp, -4.41614633762712_wp, -5.68595722916949_wp, &
      &  0.07714106879661_wp, -2.27549156962712_wp, -4.88713471516949_wp, &
      & -0.77836774520339_wp, -0.28178510762712_wp, -5.72343654316949_wp, &
      &  5.61149351579661_wp,  0.59978264937288_wp,  0.35120065883051_wp, &
      &  5.06225731779661_wp, -1.15491145862712_wp,  1.30256752383051_wp, &
      &  3.47543323779661_wp,  1.46332979037288_wp, -1.42782234816949_wp, &
      &  1.59356812979661_wp,  1.06972178737288_wp, -0.68042060116949_wp, &
      &  3.69584426379661_wp,  4.06885564037288_wp, -2.06755487816949_wp, &
      &  3.31678488279661_wp,  4.97253111237288_wp, -0.50215632416949_wp, &
      &  6.36724841779661_wp,  2.42038794337288_wp,  2.20118268783051_wp, &
      &  4.61315637679661_wp,  2.77054623237288_wp,  4.73180357883051_wp, &
      &  5.99822197079661_wp,  4.62332789137288_wp,  6.36320856183051_wp, &
      &  3.75573931779661_wp,  0.25938687437288_wp,  5.73296566883051_wp, &
      &  2.19007336679661_wp,  4.17118279137288_wp,  3.34002435583051_wp, &
      & -0.27142550420339_wp,  3.93334792537288_wp,  4.48274808683051_wp, &
      & -0.40149251020339_wp,  5.16103260337288_wp,  6.14800675483051_wp, &
      & -0.56253779320339_wp,  1.96896245137288_wp,  5.06724449283051_wp, &
      & -2.25327809420339_wp,  4.74721946137288_wp,  2.59948609283051_wp, &
      & -1.86952047620339_wp,  6.67152233437288_wp,  1.93187538083051_wp, &
      & -2.23609935020339_wp,  3.05913895037288_wp,  0.43193920183051_wp, &
      & -4.71724816020339_wp,  2.63330922937288_wp, -0.45964126816949_wp, &
      & -4.86194693620339_wp,  3.10122692737288_wp, -2.46259699416949_wp, &
      & -5.30354434120339_wp, -0.12600837962712_wp, -0.21691994916949_wp, &
      & -3.83156287920339_wp, -1.74804712462712_wp,  1.16506728183051_wp, &
      & -2.19079921720339_wp, -0.88104310862712_wp,  2.02474539783051_wp, &
      & -4.37172839220339_wp, -4.24041807162712_wp,  1.41415376483051_wp, &
      & -3.16255187920339_wp, -5.46985131362712_wp,  2.51115330883051_wp, &
      & -6.58289960820339_wp, -5.28265278962712_wp,  0.21013998383051_wp, &
      & -7.30338031920339_wp, -7.50094732762712_wp,  0.29026908983051_wp, &
      & -7.97729270120339_wp, -3.49101775062712_wp, -1.16841720516949_wp, &
      & -9.56112043020339_wp, -4.12063287662712_wp, -2.05737360216949_wp, &
      & -7.47097085720339_wp, -0.95232999762712_wp, -1.45831043216949_wp, &
      & -8.81793522920339_wp,  0.48522481437288_wp, -2.69213445216949_wp, &
      & -4.95715926820339_wp,  4.58939880537288_wp,  3.60889343583051_wp, &
      & -5.18786816620339_wp,  2.94387870337288_wp,  4.84510039983051_wp, &
      & -6.50213574320339_wp,  4.24300598437288_wp,  1.16319512983051_wp, &
      & -8.32402775120339_wp,  3.31620272237288_wp,  1.47723640883051_wp, &
      & -6.82054833620339_wp,  6.62518530837288_wp, -0.05569079516949_wp, &
      & -6.96308432020339_wp,  7.86339292637288_wp,  1.31116880083051_wp, &
      & -5.57297599320339_wp,  6.86294628637288_wp,  4.91859732383051_wp, &
      & -6.95597139220339_wp,  6.51018489937288_wp,  6.07488418583051_wp],&
      & shape(xyz))
   real(wp), parameter :: charge = -1.0_wp
   call new(self, sym, xyz, charge=charge)
end subroutine upu1c

subroutine upu1e(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 59
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "O", "C", "H", "H", "C", "H", "O", "C", "H", "N", "C", "H", "C", &
      & "H", "C", "O", "N", "H", "C", "O", "C", "H", "C", "H", "O", "H", "O", &
      & "P", "O", "O", "O", "C", "H", "H", "C", "H", "O", "C", "H", "N", "C", &
      & "H", "C", "H", "C", "O", "N", "H", "C", "O", "C", "H", "C", "H", "O", &
      & "H", "O", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & 10.56433740174576_wp, -0.67251441737288_wp, -5.57504806766102_wp, &
      &  9.08510902974576_wp, -0.71246360237288_wp, -4.49130740866102_wp, &
      &  6.89768902374576_wp, -0.93711630737288_wp, -6.08337925666102_wp, &
      &  7.17339815574576_wp, -2.42633452737288_wp, -7.49858409066102_wp, &
      &  6.51959151374576_wp,  0.85023963562712_wp, -7.06932602866102_wp, &
      &  4.61055649174576_wp, -1.55729792037288_wp, -4.49075504166102_wp, &
      &  2.93501355674576_wp, -1.54122202137288_wp, -5.71026948966102_wp, &
      &  4.88344558574576_wp, -4.08693350837288_wp, -3.41136774466102_wp, &
      &  3.81405928574576_wp, -4.17327433337288_wp, -0.96273696266102_wp, &
      &  2.50577325874576_wp, -5.75798437637288_wp, -0.80199308966102_wp, &
      &  5.86123025374576_wp, -4.61202486637288_wp,  0.94033899733898_wp, &
      &  8.36196607274576_wp, -4.23519158337288_wp,  0.38615738633898_wp, &
      &  8.73416971674576_wp, -3.60375343837288_wp, -1.52964644466102_wp, &
      & 10.22334466174576_wp, -4.60673363337288_wp,  2.11303964133898_wp, &
      & 12.18278264374576_wp, -4.29339533237288_wp,  1.61956661133898_wp, &
      &  9.60611366574576_wp, -5.42511934737288_wp,  4.63256292733898_wp, &
      & 11.10761604974576_wp, -5.84022468437288_wp,  6.37939142933898_wp, &
      &  7.00306675874576_wp, -5.73591256537288_wp,  5.03743691033898_wp, &
      &  6.45858322174576_wp, -6.29285676637288_wp,  6.79402338933898_wp, &
      &  5.06824094574576_wp, -5.36944726337288_wp,  3.33015149133898_wp, &
      &  2.82457305674576_wp, -5.67156720637288_wp,  3.86315494333898_wp, &
      &  4.14324010474576_wp,  0.10965224062712_wp, -2.18301089066102_wp, &
      &  5.92381098674576_wp,  0.53963293362712_wp, -1.22055624766102_wp, &
      &  2.50344492674576_wp, -1.63023624137288_wp, -0.51811013666102_wp, &
      &  2.50581426574576_wp, -1.08788242637288_wp,  1.47602000233898_wp, &
      &  0.00282891674576_wp, -1.87544595237288_wp, -1.46797334666102_wp, &
      & -0.91013947225424_wp, -0.28698178037288_wp, -1.16003928466102_wp, &
      &  2.93797400174576_wp,  2.40896216662712_wp, -2.91000832166102_wp, &
      &  3.13878195474576_wp,  4.84553933262712_wp, -0.97972597666102_wp, &
      &  2.28645884074576_wp,  7.09094841462712_wp, -2.47941724666102_wp, &
      &  5.63652142474576_wp,  4.76164797062712_wp,  0.35755336033898_wp, &
      &  0.97433515874576_wp,  4.08221721062712_wp,  1.11879659433898_wp, &
      & -1.34120373425424_wp,  5.49944499562712_wp,  1.25849383233898_wp, &
      & -0.98704432825424_wp,  7.51756482862712_wp,  0.98750107833898_wp, &
      & -2.08812133325424_wp,  5.17108053862712_wp,  3.15817319833898_wp, &
      & -3.22285379125424_wp,  4.62297118462712_wp, -0.71177473666102_wp, &
      & -2.51656511225424_wp,  5.04437214362712_wp, -2.61115117966102_wp, &
      & -3.55098656825424_wp,  1.89669694662712_wp, -0.49672505266102_wp, &
      & -6.05971252725424_wp,  1.16864523762712_wp, -1.10999593866102_wp, &
      & -6.06928928025424_wp, -0.12635393737288_wp, -2.71644274366102_wp, &
      & -7.13566033425424_wp, -0.20824464637288_wp,  1.11225285133898_wp, &
      & -5.80010499425424_wp, -0.55251832837288_wp,  3.30794532133898_wp, &
      & -3.91336231425424_wp,  0.23371539762712_wp,  3.28934248133898_wp, &
      & -6.76653883325424_wp, -1.78657988737288_wp,  5.33596202733898_wp, &
      & -5.65722526825424_wp, -2.03216848737288_wp,  7.03542794833898_wp, &
      & -9.29348836525424_wp, -2.79476174237288_wp,  5.24217908333898_wp, &
      &-10.40431182125424_wp, -3.94971710337288_wp,  6.94403305133898_wp, &
      &-10.51459868625424_wp, -2.36996343337288_wp,  2.92619024733898_wp, &
      &-12.30361664325424_wp, -3.05001459037288_wp,  2.75004284033898_wp, &
      & -9.58538696925424_wp, -1.13146309737288_wp,  0.83294227933898_wp, &
      &-10.80436719125424_wp, -0.86390284237288_wp, -1.12872255666102_wp, &
      & -5.89210517025424_wp,  5.65545793962712_wp, -0.38342993266102_wp, &
      & -6.39074301025424_wp,  5.79736558962712_wp,  1.62111713233898_wp, &
      & -7.50874194425424_wp,  3.61584171762712_wp, -1.69776648366102_wp, &
      & -9.45362639125424_wp,  3.56123619562712_wp, -1.00778028166102_wp, &
      & -7.43357506925424_wp,  3.99261169562712_wp, -4.36610436966102_wp, &
      & -7.26782210125424_wp,  5.82225101362712_wp, -4.59397943466102_wp, &
      & -6.08706744525424_wp,  8.05402024262712_wp, -1.60209532666102_wp, &
      & -7.51561223225424_wp,  8.95548662562712_wp, -0.88057394366102_wp],&
      & shape(xyz))
   real(wp), parameter :: charge = -1.0_wp
   call new(self, sym, xyz, charge=charge)
end subroutine upu1e

subroutine upu1f(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 59
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "O", "C", "H", "H", "C", "H", "O", "C", "H", "N", "C", "H", "C", &
      & "H", "C", "O", "N", "H", "C", "O", "C", "H", "C", "H", "O", "H", "O", &
      & "P", "O", "O", "O", "C", "H", "H", "C", "H", "O", "C", "H", "N", "C", &
      & "H", "C", "H", "C", "O", "N", "H", "C", "O", "C", "H", "C", "H", "O", &
      & "H", "O", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & 11.41703524779661_wp,  1.49335179335593_wp,  4.30907584694915_wp, &
      &  9.73807493279661_wp,  0.97172505935593_wp,  3.78819177494915_wp, &
      &  9.18032684179661_wp, -1.39211976464407_wp,  5.00528245994915_wp, &
      &  9.83030036379661_wp, -1.37229748464407_wp,  6.97213138094915_wp, &
      & 10.10639783479661_wp, -2.96523104764407_wp,  4.01663980494915_wp, &
      &  6.36267263579661_wp, -1.85117332564407_wp,  4.94339072094915_wp, &
      &  5.99874126779661_wp, -3.74578690264407_wp,  5.70694565694915_wp, &
      &  5.05076816179661_wp, -0.01680761364407_wp,  6.52768937094915_wp, &
      &  2.70774131479661_wp,  0.65911052335593_wp,  5.43382374894915_wp, &
      &  1.17096851779661_wp,  0.40222759735593_wp,  6.78438521394915_wp, &
      &  2.76346725479661_wp,  3.41204879535593_wp,  4.79169750294915_wp, &
      &  4.98618583079661_wp,  4.70493790235593_wp,  4.48276642094915_wp, &
      &  6.68149340979661_wp,  3.58212772535593_wp,  4.75627232094915_wp, &
      &  5.05482086779661_wp,  7.19694621035593_wp,  3.87496969294915_wp, &
      &  6.83854626879661_wp,  8.17048693135593_wp,  3.64839097994915_wp, &
      &  2.74123179379661_wp,  8.58372068435593_wp,  3.52854247894915_wp, &
      &  2.52950292579661_wp, 10.84822960535593_wp,  2.98049945394915_wp, &
      &  0.56184350179661_wp,  7.10512007835593_wp,  3.86160611794915_wp, &
      & -1.12950944820339_wp,  7.99338880335593_wp,  3.65619913894915_wp, &
      &  0.42811704179661_wp,  4.57428243635593_wp,  4.47280831994915_wp, &
      & -1.58178777320339_wp,  3.42795480835593_wp,  4.70051425494915_wp, &
      &  5.07555588579661_wp, -1.61297997864407_wp,  2.37139352394915_wp, &
      &  5.91091625179661_wp, -0.05274018664407_wp,  1.29620141794915_wp, &
      &  2.32743244679661_wp, -0.95180416064407_wp,  3.05262273594915_wp, &
      &  1.38049547779661_wp,  0.12105772335593_wp,  1.56488379294915_wp, &
      &  0.89904148279661_wp, -3.13432032764407_wp,  3.71387716594915_wp, &
      &  0.77469694579661_wp, -4.11967826064407_wp,  2.15977614894915_wp, &
      &  5.39894148279661_wp, -3.91910520064407_wp,  0.99757595694915_wp, &
      &  5.03746656879661_wp, -3.83767123864407_wp, -2.07960074505085_wp, &
      &  5.53883959779661_wp, -6.46771439364407_wp, -3.00205347905085_wp, &
      &  6.37925460279661_wp, -1.61757390264407_wp, -3.22816694205085_wp, &
      &  1.98244595979661_wp, -3.25786362064407_wp, -1.97660822205085_wp, &
      &  0.62790019679661_wp, -2.63139129764407_wp, -4.24427468605085_wp, &
      &  1.66582293679661_wp, -3.24119869464407_wp, -5.93004840805085_wp, &
      &  0.37762526379661_wp, -0.57544160764407_wp, -4.32274858105085_wp, &
      & -1.90075802520339_wp, -3.95987755064407_wp, -4.21497807505085_wp, &
      & -1.61417106320339_wp, -6.01149183164407_wp, -4.14844309005085_wp, &
      & -3.34205732920339_wp, -3.22237777964407_wp, -1.99689405205085_wp, &
      & -5.93975120920339_wp, -3.00630386364407_wp, -2.58637668905085_wp, &
      & -7.06820240720339_wp, -4.19030469164407_wp, -1.32760526305085_wp, &
      & -6.73873076020339_wp, -0.32174986264407_wp, -2.14348727805085_wp, &
      & -5.05357648920339_wp,  1.52547335635593_wp, -1.46924041505085_wp, &
      & -3.12533108020339_wp,  0.87714616335593_wp, -1.26983256005085_wp, &
      & -5.75328098520339_wp,  3.95732299335593_wp, -1.06185952205085_wp, &
      & -4.37377250920339_wp,  5.36396869135593_wp, -0.51941254205085_wp, &
      & -8.36261992420339_wp,  4.68449031035593_wp, -1.32069963705085_wp, &
      & -9.25807244320339_wp,  6.82195352035593_wp, -1.00886007805085_wp, &
      & -9.96184797620339_wp,  2.66991749435593_wp, -1.97190790605085_wp, &
      &-11.83208446620339_wp,  3.07159481635593_wp, -2.15837283905085_wp, &
      & -9.30713917820339_wp,  0.18482029135593_wp, -2.39585489305085_wp, &
      &-10.85537429720339_wp, -1.46354687164407_wp, -2.93981932205085_wp, &
      & -3.60168394820339_wp, -3.28906280764407_wp, -6.44807880505085_wp, &
      & -3.39556435320339_wp, -1.29053981164407_wp, -6.95232535105085_wp, &
      & -6.25810673320339_wp, -3.78052290164407_wp, -5.36597975105085_wp, &
      & -7.73810875920339_wp, -2.69912021164407_wp, -6.31933574205085_wp, &
      & -6.79226480720339_wp, -6.42359930764407_wp, -5.43139658505085_wp, &
      & -5.92760027120339_wp, -7.06387210964407_wp, -6.93589418405085_wp, &
      & -3.00415636920339_wp, -4.86857553764407_wp, -8.54956484405085_wp, &
      & -3.60911850820339_wp, -4.04956016664407_wp,-10.07843291705085_wp],&
      & shape(xyz))
   real(wp), parameter :: charge = -1.0_wp
   call new(self, sym, xyz, charge=charge)
end subroutine upu1f

subroutine upu1g(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 59
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "O", "C", "H", "H", "C", "H", "O", "C", "H", "N", "C", "H", "C", &
      & "H", "C", "O", "N", "H", "C", "O", "C", "H", "C", "H", "O", "H", "O", &
      & "P", "O", "O", "O", "C", "H", "H", "C", "H", "O", "C", "H", "N", "C", &
      & "H", "C", "H", "C", "O", "N", "H", "C", "O", "C", "H", "C", "H", "O", &
      & "H", "O", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -2.59192021381356_wp,  9.34202124932203_wp, -4.98225407311864_wp, &
      & -2.41660277481356_wp,  8.44863987032203_wp, -3.39109496311864_wp, &
      &  0.14922862518644_wp,  8.72188083132203_wp, -2.55239072611864_wp, &
      &  0.46490999718644_wp, 10.61879401732203_wp, -1.77252424311864_wp, &
      &  1.48356679218644_wp,  8.41162902032203_wp, -4.10964305011864_wp, &
      &  0.69539269518644_wp,  6.76248177732203_wp, -0.54903022111864_wp, &
      &  2.66449435818644_wp,  6.99323130532203_wp,  0.06390089188136_wp, &
      & -0.94318455381356_wp,  7.12205600432203_wp,  1.64003535788136_wp, &
      & -1.60022300481356_wp,  4.75807341932203_wp,  2.69963174388136_wp, &
      & -1.15344510481356_wp,  4.71901240532203_wp,  4.71294146788136_wp, &
      & -4.39762407281356_wp,  4.39253483932203_wp,  2.48306744488136_wp, &
      & -5.87287929681356_wp,  5.86188996032203_wp,  0.94102231288136_wp, &
      & -4.84948703681356_wp,  7.26979175932203_wp, -0.13987052911864_wp, &
      & -8.41596633181356_wp,  5.55963395732203_wp,  0.77887512988136_wp, &
      & -9.53266948181356_wp,  6.74722280732203_wp, -0.45567322211864_wp, &
      & -9.66428039281356_wp,  3.65410475532203_wp,  2.26679661988136_wp, &
      &-11.95659417481356_wp,  3.19289936332203_wp,  2.32348405388136_wp, &
      & -8.00469676081356_wp,  2.23183763732203_wp,  3.77201628388136_wp, &
      & -8.78646997781356_wp,  0.85307843732203_wp,  4.85782335788136_wp, &
      & -5.41913548781356_wp,  2.47095846232203_wp,  3.95664988288136_wp, &
      & -4.10703523781356_wp,  1.10331028332203_wp,  5.30746022488136_wp, &
      &  0.24252798818644_wp,  4.01542943532203_wp, -1.31301110111864_wp, &
      & -1.45734384681356_wp,  3.87547859632203_wp, -2.48375890211864_wp, &
      & -0.16344127981356_wp,  2.71075524132203_wp,  1.23495086088136_wp, &
      & -1.25434815781356_wp,  0.96370053932203_wp,  1.09134094488136_wp, &
      &  2.18498016818644_wp,  2.27619290832203_wp,  2.48896150088136_wp, &
      &  3.38330700818644_wp,  1.78033428832203_wp,  1.17726924188136_wp, &
      &  2.36967801818644_wp,  3.04806055932203_wp, -2.67666118611864_wp, &
      &  1.78517802118644_wp,  1.13499607532203_wp, -5.06334542811864_wp, &
      &  4.31219350418644_wp,  0.54159716932203_wp, -6.18998400111864_wp, &
      & -0.34997802181356_wp,  2.10966080532203_wp, -6.64965663511864_wp, &
      &  0.79393415818644_wp, -1.25582213167797_wp, -3.34495938111864_wp, &
      & -0.52570175181356_wp, -3.32317272767797_wp, -4.51953331011864_wp, &
      &  0.29896711118644_wp, -3.75976746067797_wp, -6.36792078911864_wp, &
      & -2.52169705081356_wp, -2.83057558567797_wp, -4.76748728111864_wp, &
      & -0.32172586281356_wp, -5.60805894167797_wp, -2.81149275811864_wp, &
      & -1.53760693481356_wp, -7.10468280867797_wp, -3.57504542611864_wp, &
      &  2.27024290418644_wp, -6.53279333167797_wp, -2.77759182911864_wp, &
      &  2.94751578518644_wp, -7.33729673967797_wp, -0.32420555111864_wp, &
      &  3.76304933518644_wp, -9.22948050767797_wp, -0.39081612511864_wp, &
      &  4.95699881418644_wp, -5.62742390867797_wp,  0.69575364288136_wp, &
      &  5.44750818618644_wp, -3.31019163267797_wp, -0.34845678311864_wp, &
      &  4.25789298318644_wp, -2.81248108167797_wp, -1.93768024711864_wp, &
      &  7.27669850518644_wp, -1.75198896567797_wp,  0.54841510888136_wp, &
      &  7.63307663918644_wp,  0.05655968832203_wp, -0.33844822711864_wp, &
      &  8.80778473918644_wp, -2.52257708367797_wp,  2.66263620088136_wp, &
      & 10.51131413718644_wp, -1.28943789967797_wp,  3.68296918288136_wp, &
      &  8.22356933518644_wp, -4.95221328467797_wp,  3.56149589488136_wp, &
      &  9.27215806718644_wp, -5.60698123067797_wp,  5.03314438788136_wp, &
      &  6.39055874518644_wp, -6.58165257467797_wp,  2.68393851488136_wp, &
      &  6.02428506218644_wp, -8.69511021767797_wp,  3.58308450288136_wp, &
      & -0.99101654181356_wp, -5.17739945867797_wp, -0.03176724211864_wp, &
      & -0.34693612981356_wp, -3.30994899167797_wp,  0.58349635988136_wp, &
      &  0.55368856818644_wp, -7.25080965067797_wp,  1.30201383388136_wp, &
      &  0.97032609518644_wp, -6.82002128867797_wp,  3.28323015288136_wp, &
      & -0.69373807681356_wp, -9.63005800967797_wp,  1.07222164188136_wp, &
      & -2.50352657981356_wp, -9.25599827667797_wp,  1.12243827588136_wp, &
      & -3.65315064581356_wp, -5.43987502967797_wp,  0.30916332588136_wp, &
      & -4.10260156281356_wp, -4.73202864867797_wp,  1.94407488488136_wp],&
      & shape(xyz))
   real(wp), parameter :: charge = -1.0_wp
   call new(self, sym, xyz, charge=charge)
end subroutine upu1g

subroutine upu1m(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 59
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "O", "C", "H", "H", "C", "H", "O", "C", "H", "N", "C", "H", "C", &
      & "H", "C", "O", "N", "H", "C", "O", "C", "H", "C", "H", "O", "H", "O", &
      & "P", "O", "O", "O", "C", "H", "H", "C", "H", "O", "C", "H", "N", "C", &
      & "H", "C", "H", "C", "O", "N", "H", "C", "O", "C", "H", "C", "H", "O", &
      & "H", "O", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -6.62894107159322_wp, -4.44812348691525_wp, -6.97365922083051_wp, &
      & -6.27683749859322_wp, -3.55687863191525_wp, -5.41018591883051_wp, &
      & -7.64515407359322_wp, -1.21376977091525_wp, -5.45628219383051_wp, &
      & -9.64955392759322_wp, -1.54483426791525_wp, -5.86591601783051_wp, &
      & -6.87635722659322_wp,  0.06818198008475_wp, -6.89673357783051_wp, &
      & -7.38266508559322_wp,  0.06868899308475_wp, -2.91833105583051_wp, &
      & -8.33651467359322_wp,  1.90834782808475_wp, -3.02356951883051_wp, &
      & -8.57649040959322_wp, -1.42820321491525_wp, -0.94439622983051_wp, &
      & -7.21052862259322_wp, -1.20009155591525_wp,  1.34915607916949_wp, &
      & -8.43892024759322_wp, -0.54460815991525_wp,  2.86897028516949_wp, &
      & -6.29725296459322_wp, -3.75239686391525_wp,  2.11491197916949_wp, &
      & -5.86953635159322_wp, -5.62264884991525_wp,  0.37030314916949_wp, &
      & -6.26830047259322_wp, -5.09739081791525_wp, -1.57291134583051_wp, &
      & -5.03891693059322_wp, -7.95665301091525_wp,  1.03221237016949_wp, &
      & -4.72879588759322_wp, -9.39919972291525_wp, -0.38375988183051_wp, &
      & -4.56989638859322_wp, -8.54771810491525_wp,  3.64686616316949_wp, &
      & -3.81722381759322_wp,-10.58867792091525_wp,  4.50668733116949_wp, &
      & -5.04558369459322_wp, -6.52307928491525_wp,  5.29443491716949_wp, &
      & -4.72983032359322_wp, -6.83566093991525_wp,  7.16451796216949_wp, &
      & -5.88607221059322_wp, -4.13303832291525_wp,  4.68552476316949_wp, &
      & -6.23863158559322_wp, -2.47020568491525_wp,  6.27380582116949_wp, &
      & -4.67354162259322_wp,  0.47806071308475_wp, -1.98645709183051_wp, &
      & -3.45523244259322_wp, -1.10924524691525_wp, -2.50417152683051_wp, &
      & -5.02812772859322_wp,  0.65659030808475_wp,  0.88806067016949_wp, &
      & -3.33579212159322_wp,  0.14090676208475_wp,  1.96147662216949_wp, &
      & -5.87394678359322_wp,  3.13395002808475_wp,  1.55379393916949_wp, &
      & -5.19034785559322_wp,  4.25750532408475_wp,  0.25335273216949_wp, &
      & -3.69800535159322_wp,  2.78058298408475_wp, -3.01378073783051_wp, &
      & -0.68993177159322_wp,  2.90569021408475_wp, -3.90038329183051_wp, &
      & -0.41035474859322_wp,  5.50439169608475_wp, -4.98929839883051_wp, &
      & -0.01756480759322_wp,  0.55991532808475_wp, -5.32717479183051_wp, &
      &  0.82198999440678_wp,  2.68859339008475_wp, -1.21643683683051_wp, &
      &  0.66914668840678_wp,  4.72904146908475_wp,  0.58589669616949_wp, &
      & -0.63961517159322_wp,  4.18736871008475_wp,  2.09296129916949_wp, &
      & -0.00517122859322_wp,  6.45754132908475_wp, -0.33197968883051_wp, &
      &  3.27840096740678_wp,  5.21939531708475_wp,  1.65254440616949_wp, &
      &  3.15152892140678_wp,  6.77583411408475_wp,  3.02164124816949_wp, &
      &  4.23768049340678_wp,  2.99060278408475_wp,  2.90959089016949_wp, &
      &  6.90200497840678_wp,  2.74159529108475_wp,  2.48457035016949_wp, &
      &  7.89538442040678_wp,  2.78338828208475_wp,  4.28815167016949_wp, &
      &  7.42850550540678_wp,  0.23445197808475_wp,  1.41076228016949_wp, &
      &  6.79321950040678_wp, -0.31903307991525_wp, -1.04940811383051_wp, &
      &  5.89613425840678_wp,  1.20225001008475_wp, -2.08146722383051_wp, &
      &  7.21057587840678_wp, -2.60025020191525_wp, -2.12641681283051_wp, &
      &  6.69941539140678_wp, -2.95439146691525_wp, -4.07352603383051_wp, &
      &  8.31706449240678_wp, -4.60635307791525_wp, -0.65044519483051_wp, &
      &  8.79973943840678_wp, -6.77172961991525_wp, -1.38015627183051_wp, &
      &  8.79271891740678_wp, -3.92812116091525_wp,  1.86849454416949_wp, &
      &  9.52747480740678_wp, -5.29179656791525_wp,  3.00544648516949_wp, &
      &  8.39920766840678_wp, -1.62415343791525_wp,  3.02073682516949_wp, &
      &  8.84912100240678_wp, -1.25882310591525_wp,  5.26902774016949_wp, &
      &  5.24228736440678_wp,  5.88578470008475_wp, -0.36438093083051_wp, &
      &  4.82703614140678_wp,  4.88006270108475_wp, -2.12124009783051_wp, &
      &  7.73763670940678_wp,  4.95689461308475_wp,  0.80774788216949_wp, &
      &  9.14098351240678_wp,  4.37825843608475_wp, -0.60157347883051_wp, &
      &  8.74272319240678_wp,  6.87517417508475_wp,  2.41982909416949_wp, &
      &  8.15540372540678_wp,  8.47191761408475_wp,  1.68685691616949_wp, &
      &  5.21436345040678_wp,  8.54919890408475_wp, -0.80556524183051_wp, &
      &  5.79988767840678_wp,  8.85690960108475_wp, -2.51872638583051_wp],&
      & shape(xyz))
   real(wp), parameter :: charge = -1.0_wp
   call new(self, sym, xyz, charge=charge)
end subroutine upu1m

subroutine upu1p(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 59
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "O", "C", "H", "H", "C", "H", "O", "C", "H", "N", "C", "H", "C", &
      & "H", "C", "O", "N", "H", "C", "O", "C", "H", "C", "H", "O", "H", "O", &
      & "P", "O", "O", "O", "C", "H", "H", "C", "H", "O", "C", "H", "N", "C", &
      & "H", "C", "H", "C", "O", "N", "H", "C", "O", "C", "H", "C", "H", "O", &
      & "H", "O", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -1.88670847279661_wp, -6.32938588466102_wp, -6.37051916213559_wp, &
      & -0.59630208779661_wp, -5.59211779266102_wp, -5.29665818013559_wp, &
      &  0.37513763520339_wp, -7.55140100666102_wp, -3.69115397013559_wp, &
      &  1.18231084820339_wp, -9.09311411366102_wp, -4.81958702713559_wp, &
      & -1.11356015379661_wp, -8.32441477666102_wp, -2.46829773113559_wp, &
      &  2.38291046220339_wp, -6.42893230266102_wp, -1.99797082113559_wp, &
      &  3.12270984120339_wp, -7.93389949966102_wp, -0.77587028313559_wp, &
      &  4.44456655820339_wp, -5.39210201266102_wp, -3.47956796513559_wp, &
      &  5.49637046920339_wp, -3.26205937966102_wp, -2.24477262613559_wp, &
      &  7.50389234020339_wp, -3.54310018466102_wp, -1.86914971213559_wp, &
      &  5.29409079820339_wp, -1.03773453666102_wp, -3.96336351913559_wp, &
      &  3.57286033620339_wp, -0.94879231566102_wp, -5.90450252813559_wp, &
      &  2.41611521520339_wp, -2.63097971566102_wp, -6.10348784113559_wp, &
      &  3.35487004920339_wp,  1.08795927033898_wp, -7.44535543213559_wp, &
      &  1.97074856220339_wp,  1.11934932033898_wp, -8.94821879713559_wp, &
      &  4.94825015520339_wp,  3.26421893233898_wp, -7.07251419513559_wp, &
      &  4.96875160320339_wp,  5.22151698933898_wp, -8.34837195713559_wp, &
      &  6.58705549320339_wp,  3.03612881633898_wp, -4.99469036013559_wp, &
      &  7.72193799620339_wp,  4.54062036933898_wp, -4.61790961113559_wp, &
      &  6.84643758220339_wp,  1.00498026833898_wp, -3.38626463413559_wp, &
      &  8.31357359820339_wp,  0.98492763033898_wp, -1.57999990413559_wp, &
      &  1.51230179320339_wp, -4.22484907166102_wp, -0.34976390613559_wp, &
      &  0.19268289020339_wp, -3.02034906366102_wp, -1.39032111413559_wp, &
      &  4.00128446120339_wp, -2.84543375766102_wp,  0.21374236786441_wp, &
      &  3.75051309820339_wp, -0.83120090266102_wp,  0.61845647986441_wp, &
      &  5.30731245520339_wp, -4.07129390566102_wp,  2.23045548886441_wp, &
      &  3.99130462920339_wp, -4.76009827366102_wp,  3.33047141286441_wp, &
      &  0.35904265020339_wp, -5.13273730966102_wp,  1.92702766686441_wp, &
      & -2.39498693879661_wp, -3.89541805266102_wp,  2.74532947586441_wp, &
      & -3.11305390179661_wp, -5.30303752666102_wp,  5.09450208986441_wp, &
      & -4.10189743479661_wp, -3.68475139966102_wp,  0.50043080886441_wp, &
      & -1.63131238279661_wp, -0.98470674666102_wp,  3.40070288986441_wp, &
      & -0.08783313779661_wp, -0.28695150266102_wp,  5.53036342086441_wp, &
      &  1.83810416020339_wp,  0.09624675533898_wp,  4.88206514086441_wp, &
      & -0.07073810979661_wp, -1.79975633766102_wp,  6.94347235986441_wp, &
      & -1.28409545279661_wp,  2.08001949533898_wp,  6.62606680186441_wp, &
      & -0.24843679279661_wp,  2.66986136933898_wp,  8.31491113986441_wp, &
      & -1.15067758479661_wp,  4.16119872833898_wp,  4.83519804686441_wp, &
      & -3.54415671479661_wp,  4.55524533633898_wp,  3.65630099186441_wp, &
      & -4.13268579579661_wp,  6.49333357233898_wp,  4.04803476386441_wp, &
      & -3.26551283779661_wp,  4.46037939133898_wp,  0.87101986986441_wp, &
      & -3.74065892679661_wp,  2.35389880533898_wp, -0.57939339313559_wp, &
      & -4.38538089979661_wp,  0.68285965533898_wp,  0.40413982086441_wp, &
      & -3.39227433379661_wp,  2.29467101333898_wp, -3.11755693613559_wp, &
      & -3.80113015879661_wp,  0.59310258333898_wp, -4.17431533213559_wp, &
      & -2.47204127179661_wp,  4.49120838033898_wp, -4.42924843913559_wp, &
      & -2.12471020179661_wp,  4.72139080133898_wp, -6.72554122813559_wp, &
      & -1.90652470679661_wp,  6.51329266933898_wp, -2.80219927413559_wp, &
      & -1.19654274679661_wp,  8.10329980233898_wp, -3.61436117513559_wp, &
      & -2.21206996679661_wp,  6.63076880833898_wp, -0.21822782813559_wp, &
      & -1.60210912479661_wp,  8.49798797333898_wp,  1.02292039186441_wp, &
      & -4.08907356579661_wp,  1.71857557333898_wp,  7.22720318986441_wp, &
      & -4.54407234079661_wp, -0.26207892966102_wp,  7.58071064086441_wp, &
      & -5.47020267179661_wp,  2.70452064433898_wp,  4.83266675886441_wp, &
      & -5.88964285079661_wp,  1.11970628933898_wp,  3.59165592186441_wp, &
      & -7.80474823879661_wp,  3.89701144433898_wp,  5.41002264686441_wp, &
      & -7.56586873279661_wp,  4.62734178433898_wp,  7.09574666886441_wp, &
      & -4.78490128879661_wp,  3.23155294033898_wp,  9.37782796286441_wp, &
      & -5.85122585479661_wp,  2.21352088833898_wp, 10.46770966486441_wp],&
      & shape(xyz))
   real(wp), parameter :: charge = -1.0_wp
   call new(self, sym, xyz, charge=charge)
end subroutine upu1p

subroutine upu2a(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 59
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "O", "C", "H", "H", "C", "H", "O", "C", "H", "N", "C", "H", "C", &
      & "H", "C", "O", "N", "H", "C", "O", "C", "H", "C", "H", "O", "H", "O", &
      & "P", "O", "O", "O", "C", "H", "H", "C", "H", "O", "C", "H", "N", "C", &
      & "H", "C", "H", "C", "O", "N", "H", "C", "O", "C", "H", "C", "H", "O", &
      & "H", "O", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  2.90423522461017_wp,  4.13250370294915_wp, -6.19944585283051_wp, &
      &  1.65991528861017_wp,  4.13858030594915_wp, -4.85283920583051_wp, &
      &  2.56789517861017_wp,  5.76950794394915_wp, -2.87874242983051_wp, &
      &  2.96839652461017_wp,  7.66510474594915_wp, -3.61658178583051_wp, &
      &  4.29107678261017_wp,  5.00268434894915_wp, -2.01047532083051_wp, &
      &  0.54327657861017_wp,  5.94512490494915_wp, -0.86265178583051_wp, &
      &  1.28982738261017_wp,  7.01745520894915_wp,  0.74030979716949_wp, &
      & -1.64309160538983_wp,  7.30554828294915_wp, -1.87223165983051_wp, &
      & -3.86703018838983_wp,  5.82965678894915_wp, -1.59205748283051_wp, &
      & -4.98612346338983_wp,  6.40440220894915_wp,  0.04599045316949_wp, &
      & -5.46552842738983_wp,  6.27607318394915_wp, -3.82312619083051_wp, &
      & -4.40055501538983_wp,  6.12952644594915_wp, -6.19005504783051_wp, &
      & -2.37672102938983_wp,  5.78044133194915_wp, -6.18441723983051_wp, &
      & -5.75891556838983_wp,  6.42349996894915_wp, -8.34079018183051_wp, &
      & -4.85737076238983_wp,  6.31473622194915_wp,-10.17300935283051_wp, &
      & -8.44214147138983_wp,  6.89156015194915_wp, -8.19932132483051_wp, &
      & -9.90325209538983_wp,  7.18446786994915_wp, -9.99997980583051_wp, &
      & -9.36765858538983_wp,  7.00314261394915_wp, -5.71796464583051_wp, &
      &-11.24938385438983_wp,  7.33185950494915_wp, -5.51027449983051_wp, &
      & -8.04895790038983_wp,  6.69662244394915_wp, -3.48826050983051_wp, &
      & -9.07018774638983_wp,  6.80711396694915_wp, -1.40522235283051_wp, &
      & -0.44341890238983_wp,  3.38093359894915_wp,  0.06802163516949_wp, &
      &  0.88496171461017_wp,  1.84886716494915_wp, -0.30881504983051_wp, &
      & -2.95373922338983_wp,  3.06680614794915_wp, -1.34688348783051_wp, &
      & -2.58244510438983_wp,  2.29595875994915_wp, -3.22717785583051_wp, &
      & -4.70818483238983_wp,  1.45428878694915_wp, -0.13350611483051_wp, &
      & -4.53722699138983_wp,  1.80970070394915_wp,  1.67122434716949_wp, &
      & -1.05330377738983_wp,  3.48091011894915_wp,  2.72999719016949_wp, &
      &  1.14950577861017_wp,  2.37375348694915_wp,  4.66353666716949_wp, &
      &  0.19399965761017_wp,  2.97162650794915_wp,  7.26119294216949_wp, &
      &  3.69031361561017_wp,  3.18870632394915_wp,  3.72368806016949_wp, &
      &  1.04615080661017_wp, -0.67432445805085_wp,  4.15366043816949_wp, &
      & -1.06030691338983_wp, -2.11346720605085_wp,  5.10132816416949_wp, &
      & -2.75125187138983_wp, -1.70609255005085_wp,  3.97867181116949_wp, &
      & -1.42142429038983_wp, -1.61725690905085_wp,  7.07771509916949_wp, &
      & -0.42982459238983_wp, -4.89395768305085_wp,  4.94075399016949_wp, &
      & -1.98864198738983_wp, -5.96138698305085_wp,  5.79651640416949_wp, &
      & -0.18332023038983_wp, -5.65869256405085_wp,  2.31326442216949_wp, &
      &  1.71621665061017_wp, -7.51696104205085_wp,  2.04107679416949_wp, &
      &  0.97589665161017_wp, -9.17935356205085_wp,  1.07095231916949_wp, &
      &  3.79333467361017_wp, -6.46796942005085_wp,  0.42794149316949_wp, &
      &  3.97874381661017_wp, -3.93740219705085_wp, -0.08714282083051_wp, &
      &  2.47291339261017_wp, -2.79982154405085_wp,  0.70527541116949_wp, &
      &  5.91337253061017_wp, -2.94730880005085_wp, -1.44900314683051_wp, &
      &  6.00732536061017_wp, -0.94094174005085_wp, -1.83259125583051_wp, &
      &  7.88886694761017_wp, -4.56608379905085_wp, -2.38824250683051_wp, &
      &  9.74649518661017_wp, -3.92137751005085_wp, -3.65482908383051_wp, &
      &  7.57361322061017_wp, -7.11664148805085_wp, -1.72343879283051_wp, &
      &  8.93065191061017_wp, -8.34245344805085_wp, -2.31356525983051_wp, &
      &  5.64360299661017_wp, -8.17463349705085_wp, -0.33117088583051_wp, &
      &  5.53511401661017_wp,-10.44099680605085_wp,  0.18447675516949_wp, &
      &  2.05626528361017_wp, -5.70425121205085_wp,  6.16173221916949_wp, &
      &  3.51364596761017_wp, -4.27094941705085_wp,  5.85413755116949_wp, &
      &  2.68557654161017_wp, -8.14128909205085_wp,  4.70237998516949_wp, &
      &  4.70164450961017_wp, -8.60008432705085_wp,  4.71147277916949_wp, &
      &  1.22332433361017_wp,-10.18997089605085_wp,  5.67324749916949_wp, &
      &  1.03642117361017_wp, -9.84483833305085_wp,  7.48148510316949_wp, &
      &  1.66814013161017_wp, -6.13911061105085_wp,  8.79656394016949_wp, &
      &  3.29928660061017_wp, -6.05354665205085_wp,  9.63719966516949_wp],&
      & shape(xyz))
   real(wp), parameter :: charge = -1.0_wp
   call new(self, sym, xyz, charge=charge)
end subroutine upu2a

subroutine upu2h(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 59
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "O", "C", "H", "H", "C", "H", "O", "C", "H", "N", "C", "H", "C", &
      & "H", "C", "O", "N", "H", "C", "O", "C", "H", "C", "H", "O", "H", "O", &
      & "P", "O", "O", "O", "C", "H", "H", "C", "H", "O", "C", "H", "N", "C", &
      & "H", "C", "H", "C", "O", "N", "H", "C", "O", "C", "H", "C", "H", "O", &
      & "H", "O", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -1.20505521862712_wp, 10.16487150437288_wp,  3.20286859691525_wp, &
      & -0.74762001962712_wp,  8.96436108637288_wp,  1.89354907891525_wp, &
      & -2.78926183862712_wp,  8.86091767437288_wp,  0.09802366191525_wp, &
      & -3.19137719962712_wp, 10.74665861037288_wp, -0.66328665808475_wp, &
      & -4.50984317962712_wp,  8.11124617737288_wp,  0.98484410091525_wp, &
      & -2.04245063062712_wp,  7.10913809037288_wp, -2.04201600008475_wp, &
      & -3.67324409862712_wp,  6.85764636337288_wp, -3.28572782308475_wp, &
      & -0.00748729262712_wp,  8.24755438037288_wp, -3.53405539208475_wp, &
      &  2.16180003637288_wp,  6.65627338337288_wp, -3.51114152008475_wp, &
      &  2.35769337837288_wp,  5.62358329337288_wp, -5.28913783908475_wp, &
      &  4.42363386137288_wp,  8.25112577437288_wp, -3.30809065108475_wp, &
      &  4.63461307537288_wp,  9.89382602737288_wp, -1.30293377208475_wp, &
      &  3.03210105737288_wp,  9.87260175837288_wp, -0.01638167308475_wp, &
      &  6.64974355037288_wp, 11.43927323237288_wp, -0.98161081208475_wp, &
      &  6.75156765537288_wp, 12.71041272737288_wp,  0.61635320191525_wp, &
      &  8.67904111137288_wp, 11.42368904037288_wp, -2.80377116108475_wp, &
      & 10.61662592437288_wp, 12.72800494237288_wp, -2.74691478608475_wp, &
      &  8.27604608337288_wp,  9.74804500537288_wp, -4.81927599108475_wp, &
      &  9.62996785937288_wp,  9.69516480237288_wp, -6.18172987908475_wp, &
      &  6.25463756137288_wp,  8.14677453737288_wp, -5.20772995508475_wp, &
      &  6.09203854537288_wp,  6.77050807537288_wp, -7.07195144808475_wp, &
      & -1.06910889862712_wp,  4.52012674537288_wp, -1.17867597308475_wp, &
      & -1.75320822562712_wp,  3.99655321537288_wp,  0.69499055891525_wp, &
      &  1.81281640037288_wp,  4.82115763937288_wp, -1.27455216608475_wp, &
      &  2.46375368337288_wp,  5.70209025937288_wp,  0.47425261091525_wp, &
      &  3.15964376737288_wp,  2.52795190637288_wp, -1.57315419508475_wp, &
      &  2.19046415637288_wp,  1.54040869337288_wp, -2.79538058908475_wp, &
      & -1.70715069062712_wp,  2.58472343237288_wp, -3.00027337308475_wp, &
      & -4.27121784162712_wp,  0.89757096537288_wp, -2.37771146108475_wp, &
      & -4.60873216362712_wp, -0.74960378062712_wp, -4.65545564508475_wp, &
      & -6.30978545362712_wp,  2.59794527837288_wp, -1.39509722008475_wp, &
      & -3.38554654462712_wp, -0.79463141562712_wp,  0.05934361591525_wp, &
      & -1.41030988662712_wp, -2.62467589062712_wp, -0.32597945008475_wp, &
      &  0.29832593037288_wp, -1.70390387862712_wp, -1.04722890108475_wp, &
      & -2.04297540762712_wp, -4.03595651962712_wp, -1.70417116608475_wp, &
      & -0.76441118062712_wp, -3.85786364062712_wp,  2.16309770491525_wp, &
      & -0.14183868462712_wp, -2.44051188862712_wp,  3.54011684491525_wp, &
      & -2.96684411462712_wp, -5.15443072062712_wp,  3.18202869891525_wp, &
      & -2.23630269162712_wp, -7.42564382262712_wp,  4.39447462691525_wp, &
      & -2.86124131262712_wp, -7.43391420862712_wp,  6.35984332491525_wp, &
      & -3.52654051262712_wp, -9.59541775562712_wp,  3.12434783691525_wp, &
      & -4.68869100562712_wp, -9.37593458562712_wp,  0.82205289991525_wp, &
      & -4.64136357362712_wp, -7.49572260962712_wp,  0.01845807191525_wp, &
      & -5.84491720662712_wp,-11.34984522362712_wp, -0.33622422208475_wp, &
      & -6.75536885862712_wp,-11.11098593762712_wp, -2.15083001308475_wp, &
      & -5.89134286162712_wp,-13.79396504162712_wp,  0.86167724091525_wp, &
      & -6.87332309862712_wp,-15.74373216162712_wp,  0.02391994691525_wp, &
      & -4.67837177862712_wp,-13.84169517362712_wp,  3.22084083591525_wp, &
      & -4.65880876762712_wp,-15.52208336562712_wp,  4.15349468991525_wp, &
      & -3.48823879062712_wp,-11.87280667162712_wp,  4.44032939491525_wp, &
      & -2.46250251462712_wp,-12.09074102162712_wp,  6.51636611591525_wp, &
      &  1.23555046437288_wp, -5.92934101962712_wp,  1.90755024791525_wp, &
      &  1.00179570537288_wp, -6.97012896262712_wp,  0.13357469891525_wp, &
      &  0.64537618737288_wp, -7.63780430462712_wp,  4.19294668791525_wp, &
      &  1.26461615037288_wp, -9.59335228562712_wp,  3.92902453291525_wp, &
      &  1.72781255737288_wp, -6.56860244062712_wp,  6.41663223691525_wp, &
      &  3.26307281737288_wp, -5.70720259562712_wp,  5.84346889491525_wp, &
      &  3.69459968337288_wp, -4.83147843862712_wp,  2.06879244091525_wp, &
      &  4.88714434037288_wp, -5.95822926062712_wp,  1.24322633591525_wp],&
      & shape(xyz))
   real(wp), parameter :: charge = -1.0_wp
   call new(self, sym, xyz, charge=charge)
end subroutine upu2h

subroutine upu2p(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 59
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "O", "C", "H", "H", "C", "H", "O", "C", "H", "N", "C", "H", "C", &
      & "H", "C", "O", "N", "H", "C", "O", "C", "H", "C", "H", "O", "H", "O", &
      & "P", "O", "O", "O", "C", "H", "H", "C", "H", "O", "C", "H", "N", "C", &
      & "H", "C", "H", "C", "O", "N", "H", "C", "O", "C", "H", "C", "H", "O", &
      & "H", "O", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -5.29210016016949_wp, -2.72075456042373_wp,  5.27930198159322_wp, &
      & -5.03765838216949_wp, -2.09403279342373_wp,  3.57449531759322_wp, &
      & -4.87447411816949_wp, -4.23151187642373_wp,  1.89908825359322_wp, &
      & -6.55635802916949_wp, -5.43020154442373_wp,  2.07633302359322_wp, &
      & -3.18998994516949_wp, -5.36996274842373_wp,  2.31687473359322_wp, &
      & -4.66961591516949_wp, -3.26421556642373_wp, -0.78867469140678_wp, &
      & -4.27907040216949_wp, -4.86721305442373_wp, -2.03366982740678_wp, &
      & -7.06162030316949_wp, -2.13883800742373_wp, -1.58998540840678_wp, &
      & -6.70046475316949_wp,  0.43774782257627_wp, -2.27582952040678_wp, &
      & -6.69393008116949_wp,  0.68429300257627_wp, -4.32638517640678_wp, &
      & -8.86828338716949_wp,  1.88761665557627_wp, -1.33593461440678_wp, &
      & -9.44602156616949_wp,  1.78096127657627_wp,  1.19663722959322_wp, &
      & -8.20538697416949_wp,  0.57195238257627_wp,  2.30159781059322_wp, &
      &-11.42017068816949_wp,  3.04555740557627_wp,  2.21998621459322_wp, &
      &-11.82447435116949_wp,  2.91573889957627_wp,  4.21935259459322_wp, &
      &-13.02355878216949_wp,  4.57263987657627_wp,  0.63059815559322_wp, &
      &-14.87140976716949_wp,  5.82881217357627_wp,  1.31463474359322_wp, &
      &-12.33454541116949_wp,  4.52800360357627_wp, -1.92894574840678_wp, &
      &-13.41961265716949_wp,  5.55898330257627_wp, -3.13438438240678_wp, &
      &-10.34393753216949_wp,  3.25914387457627_wp, -3.03802574240678_wp, &
      & -9.92918141616949_wp,  3.33939997057627_wp, -5.32211354840678_wp, &
      & -2.64458119816949_wp, -1.22872858542373_wp, -1.16916591740678_wp, &
      & -1.17315890516949_wp, -1.28885928842373_wp,  0.27630590159322_wp, &
      & -4.15099894816949_wp,  1.24801546157627_wp, -1.11601737440678_wp, &
      & -4.42242029216949_wp,  1.83751869757627_wp,  0.84300581259322_wp, &
      & -2.98856367316949_wp,  3.28129977857627_wp, -2.41258105340678_wp, &
      & -2.21730432516949_wp,  2.53953508357627_wp, -3.91739574840678_wp, &
      & -1.54977071316949_wp, -1.42627487142373_wp, -3.65776421640678_wp, &
      &  1.20137137483051_wp, -2.88246483842373_wp, -3.88926604340678_wp, &
      &  1.73590550383051_wp, -2.96473632642373_wp, -6.66817897740678_wp, &
      &  1.19139475483051_wp, -5.18328919042373_wp, -2.24137036640678_wp, &
      &  3.13127029383051_wp, -0.92525673242373_wp, -2.47708401540678_wp, &
      &  3.69181517883051_wp,  1.50662030657627_wp, -3.57837059140678_wp, &
      &  2.35171504783051_wp,  2.90646790357627_wp, -2.85321299740678_wp, &
      &  3.54575088683051_wp,  1.39961665057627_wp, -5.64057735240678_wp, &
      &  6.36900642983051_wp,  2.22100665057627_wp, -2.84582794840678_wp, &
      &  6.85475879583051_wp,  4.02594139157627_wp, -3.73227308840678_wp, &
      &  6.55207779283051_wp,  2.55525136857627_wp, -0.11052157540678_wp, &
      &  8.24696490783051_wp,  0.73055557357627_wp,  0.93465298859322_wp, &
      & 10.04442616083051_wp,  1.61359656057627_wp,  1.43655457359322_wp, &
      &  7.19145807783051_wp, -0.19606589642373_wp,  3.32080376059322_wp, &
      &  5.00706031083051_wp, -1.61128717142373_wp,  3.31027685259322_wp, &
      &  4.21425581783051_wp, -1.95055950542373_wp,  1.44950392059322_wp, &
      &  3.90430560683051_wp, -2.47644794942373_wp,  5.45007179559322_wp, &
      &  2.19550122683051_wp, -3.59726011942373_wp,  5.37910124559322_wp, &
      &  4.99811226983051_wp, -1.88200213542373_wp,  7.87490620359322_wp, &
      &  4.19324527783051_wp, -2.54628711942373_wp,  9.96486963859322_wp, &
      &  7.15440451983051_wp, -0.34703553942373_wp,  7.72138826559322_wp, &
      &  7.96405554183051_wp,  0.17465544557627_wp,  9.38527480359322_wp, &
      &  8.32958826383051_wp,  0.56666436257627_wp,  5.58124485759322_wp, &
      & 10.19000876183051_wp,  1.95349288157627_wp,  5.67848562159322_wp, &
      &  8.32737596183051_wp,  0.20164753557627_wp, -3.54952127840678_wp, &
      &  7.69171012183051_wp, -0.98576840442373_wp, -5.11207938540678_wp, &
      &  8.66168455183051_wp, -1.31584419742373_wp, -1.09445975840678_wp, &
      &  7.20819046783051_wp, -2.77400250342373_wp, -0.97841357440678_wp, &
      & 11.04950111683051_wp, -2.51268079942373_wp, -0.90623813640678_wp, &
      & 12.20701648983051_wp, -1.41416739642373_wp, -1.84695507240678_wp, &
      & 10.73941125483051_wp,  1.32361735157627_wp, -4.13370021040678_wp, &
      & 11.04531990983051_wp,  1.12939547157627_wp, -5.93042295840678_wp],&
      & shape(xyz))
   real(wp), parameter :: charge = -1.0_wp
   call new(self, sym, xyz, charge=charge)
end subroutine upu2p

subroutine upu3a(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 59
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "O", "C", "H", "H", "C", "H", "O", "C", "H", "N", "C", "H", "C", &
      & "H", "C", "O", "N", "H", "C", "O", "C", "H", "C", "H", "O", "H", "O", &
      & "P", "O", "O", "O", "C", "H", "H", "C", "H", "O", "C", "H", "N", "C", &
      & "H", "C", "H", "C", "O", "N", "H", "C", "O", "C", "H", "C", "H", "O", &
      & "H", "O", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -2.05034741444068_wp,  8.93485941145763_wp, -3.98629723962712_wp, &
      & -2.52617342744068_wp,  7.68014220245763_wp, -2.73613408262712_wp, &
      & -0.28700308644068_wp,  6.96766826045763_wp, -1.38176897662712_wp, &
      &  0.63345334255932_wp,  8.63085465545763_wp, -0.55535419562712_wp, &
      &  1.06933338155932_wp,  6.01369055045763_wp, -2.63058438162712_wp, &
      & -0.98915421044068_wp,  5.14371664045763_wp,  0.69993106437288_wp, &
      &  0.75386819355932_wp,  4.52892358045763_wp,  1.64307062937288_wp, &
      & -2.60452600244068_wp,  6.38309362045763_wp,  2.55039516337288_wp, &
      & -4.38625906544068_wp,  4.64966440945763_wp,  3.54888291237288_wp, &
      & -4.23327875444068_wp,  4.55663754545763_wp,  5.60342683537288_wp, &
      & -6.98734916144068_wp,  5.60232422545763_wp,  2.99730137537288_wp, &
      & -7.46833507044068_wp,  7.20337472845763_wp,  1.01526302637288_wp, &
      & -5.82462551444068_wp,  7.71410076745763_wp, -0.10110129462712_wp, &
      & -9.81754056544068_wp,  8.10529540145763_wp,  0.51278658637288_wp, &
      &-10.13553704144068_wp,  9.37779039645763_wp, -1.05599665662712_wp, &
      &-11.92620510744068_wp,  7.39492661445763_wp,  2.08349867237288_wp, &
      &-14.14779287044068_wp,  8.08425783745763_wp,  1.84624036237288_wp, &
      &-11.26129764744068_wp,  5.74877931245763_wp,  4.05627623037288_wp, &
      &-12.67314160344068_wp,  5.18402052745763_wp,  5.23177423637288_wp, &
      & -8.90318965344068_wp,  4.79426397445763_wp,  4.61466323837288_wp, &
      & -8.50757778444068_wp,  3.35820336945763_wp,  6.40165427537288_wp, &
      & -2.47948661244068_wp,  2.80105318745763_wp, -0.10823160862712_wp, &
      & -3.80554645744068_wp,  3.26716789845763_wp, -1.62299231062712_wp, &
      & -3.87058969244068_wp,  2.06333099545763_wp,  2.33277582937288_wp, &
      & -5.61310565044068_wp,  0.99885193545763_wp,  2.00269172137288_wp, &
      & -2.19342499444068_wp,  0.68443157645763_wp,  3.94136501737288_wp, &
      & -1.07246429244068_wp, -0.22734760754237_wp,  2.78388384837288_wp, &
      & -0.76471334444068_wp,  0.87738752945763_wp, -0.90082842062712_wp, &
      & -1.60340302944068_wp, -1.00648768054237_wp, -3.26539209662712_wp, &
      & -2.20627947744068_wp,  0.59626357645763_wp, -5.52502930462712_wp, &
      & -3.41741261244068_wp, -2.96305290154237_wp, -2.32020255762712_wp, &
      &  1.10755573555932_wp, -2.42984120154237_wp, -3.60096472462712_wp, &
      &  3.20507712955932_wp, -1.02477493654237_wp, -4.64821439762712_wp, &
      &  3.29068946455932_wp,  0.85888597845763_wp, -3.80084425562712_wp, &
      &  2.95804855755932_wp, -0.83603307354237_wp, -6.69325696662712_wp, &
      &  5.63213536655932_wp, -2.44058278254237_wp, -4.12579643462712_wp, &
      &  7.15394191155932_wp, -1.55357185854237_wp, -5.22429642062712_wp, &
      &  6.25022392055932_wp, -2.28319222854237_wp, -1.45621851162712_wp, &
      &  7.41420839255932_wp, -4.56425363754237_wp, -0.62960096262712_wp, &
      &  9.28058776255932_wp, -4.18322705254237_wp,  0.15511569137288_wp, &
      &  5.91193651155932_wp, -5.61943354454237_wp,  1.48077718237288_wp, &
      &  3.31900379655932_wp, -5.69048478554237_wp,  1.34206203537288_wp, &
      &  2.51980654955932_wp, -4.87733909354237_wp, -0.36338033362712_wp, &
      &  1.85663348055932_wp, -6.63022095454237_wp,  3.22187868037288_wp, &
      & -0.18037756444068_wp, -6.66206264854237_wp,  3.05080235337288_wp, &
      &  3.01735458455932_wp, -7.56995107554237_wp,  5.50114617237288_wp, &
      &  1.92051869155932_wp, -8.46608899854237_wp,  7.35973647037288_wp, &
      &  5.66119254955932_wp, -7.36254760154237_wp,  5.51043606537288_wp, &
      &  6.57251026255932_wp, -7.98453598254237_wp,  7.08399861437288_wp, &
      &  7.20569942755932_wp, -6.42889238154237_wp,  3.63123679337288_wp, &
      &  9.51798137755932_wp, -6.32328221054237_wp,  3.84530193037288_wp, &
      &  5.57248559955932_wp, -5.27505554954237_wp, -4.71301442462712_wp, &
      &  3.71800791155932_wp, -6.07453417654237_wp, -4.27583463162712_wp, &
      &  7.60000190555932_wp, -6.33680584554237_wp, -2.91858769362712_wp, &
      &  7.27449168955932_wp, -8.31245540854237_wp, -2.39032367462712_wp, &
      & 10.04305912955932_wp, -6.02412006654237_wp, -4.02320812362712_wp, &
      &  9.75785733655932_wp, -6.09706386754237_wp, -5.85145339062712_wp, &
      &  6.18071564155932_wp, -5.65641264854237_wp, -7.31735560462712_wp, &
      &  5.53775810455932_wp, -7.30030890854237_wp, -7.82610933662712_wp],&
      & shape(xyz))
   real(wp), parameter :: charge = -1.0_wp
   call new(self, sym, xyz, charge=charge)
end subroutine upu3a

subroutine upu3b(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 59
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "O", "C", "H", "H", "C", "H", "O", "C", "H", "N", "C", "H", "C", &
      & "H", "C", "O", "N", "H", "C", "O", "C", "H", "C", "H", "O", "H", "O", &
      & "P", "O", "O", "O", "C", "H", "H", "C", "H", "O", "C", "H", "N", "C", &
      & "H", "C", "H", "C", "O", "N", "H", "C", "O", "C", "H", "C", "H", "O", &
      & "H", "O", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -2.33222592579661_wp,  0.69340592440678_wp, -9.39734768925424_wp, &
      & -1.54672824779661_wp, -0.64570266759322_wp, -8.42209487625424_wp, &
      & -3.18938541479661_wp, -1.24429510659322_wp, -6.34910382625424_wp, &
      & -4.83227577479661_wp, -2.32890522859322_wp, -7.00645842925424_wp, &
      & -3.86762073379661_wp,  0.47929448940678_wp, -5.41458632525424_wp, &
      & -1.74687649779661_wp, -2.76066868959322_wp, -4.40551081925424_wp, &
      & -3.04226438879661_wp, -3.21687915759322_wp, -2.85008982825424_wp, &
      & -0.83048014579661_wp, -5.11896527859322_wp, -5.50774906025424_wp, &
      &  1.53055484520339_wp, -5.78902961159322_wp, -4.43341602425424_wp, &
      &  1.44541720920339_wp, -7.68782231759322_wp, -3.63443515125424_wp, &
      &  3.49442177820339_wp, -5.86696625759322_wp, -6.46652667525424_wp, &
      &  3.19667390520339_wp, -4.60826626259322_wp, -8.71190419925424_wp, &
      &  1.43634331220339_wp, -3.57029023159322_wp, -8.88987897125424_wp, &
      &  4.96558725720339_wp, -4.67442991559322_wp,-10.57050054425424_wp, &
      &  4.66509077820339_wp, -3.67139756459322_wp,-12.32704431625424_wp, &
      &  7.26098488320339_wp, -6.10438349359322_wp,-10.23382337125424_wp, &
      &  9.01145488620339_wp, -6.32909572459322_wp,-11.76728158925424_wp, &
      &  7.39597330220339_wp, -7.33776815259322_wp, -7.88784420625424_wp, &
      &  8.96422968920339_wp, -8.39577924759322_wp, -7.55253878525424_wp, &
      &  5.64264414320339_wp, -7.30804465259322_wp, -5.96291186425424_wp, &
      &  5.92967405620339_wp, -8.45544148759322_wp, -3.95980703825424_wp, &
      &  0.63268007620339_wp, -1.51808408859322_wp, -3.32415045825424_wp, &
      &  1.62500316120339_wp, -0.46108116259322_wp, -4.79750839725424_wp, &
      &  2.16592305320339_wp, -3.80580621459322_wp, -2.42032853225424_wp, &
      &  4.19747392420339_wp, -3.44514558459322_wp, -2.29566463125424_wp, &
      &  1.18961634020339_wp, -4.65161506459322_wp, -0.04536928425424_wp, &
      &  0.52944869120339_wp, -3.12047327359322_wp,  0.75521086374576_wp, &
      &  0.00394745220339_wp,  0.09093633340678_wp, -1.25114504625424_wp, &
      &  1.31768779520339_wp,  2.93968598140678_wp, -1.10963934125424_wp, &
      &  0.88962252820339_wp,  4.23433876940678_wp, -3.59476356725424_wp, &
      &  3.89885602120339_wp,  2.74420138640678_wp,  0.02956704374576_wp, &
      & -0.50879454679661_wp,  4.09717792140678_wp,  1.08346420474576_wp, &
      & -3.13966370079661_wp,  4.56213101740678_wp,  0.50700612574576_wp, &
      & -4.04523532379661_wp,  2.80537729740678_wp, -0.10674210425424_wp, &
      & -3.28374510279661_wp,  5.96890502740678_wp, -1.00471589625424_wp, &
      & -4.42021610879661_wp,  5.53897157740678_wp,  2.87499511974576_wp, &
      & -6.34170858279661_wp,  6.13538043840678_wp,  2.39417399474576_wp, &
      & -4.62120755479661_wp,  3.48703339840678_wp,  4.71454245974576_wp, &
      & -3.39608062079661_wp,  4.14193173540678_wp,  7.00897197974576_wp, &
      & -4.72867608279661_wp,  4.68814474940678_wp,  8.48907264974576_wp, &
      & -2.09074369979661_wp,  1.85721087140678_wp,  7.92858105474576_wp, &
      & -0.81069584779661_wp,  0.34993575340678_wp,  6.24593388374576_wp, &
      & -0.90783267779661_wp,  0.99384193240678_wp,  4.30325966074576_wp, &
      &  0.44755930520339_wp, -1.76286407559322_wp,  6.95896038074576_wp, &
      &  1.43012327920339_wp, -2.92674121059322_wp,  5.59542330174576_wp, &
      &  0.51112080620339_wp, -2.48047391459322_wp,  9.58611319974576_wp, &
      &  1.58009306720339_wp, -4.35001290759322_wp, 10.49497048974576_wp, &
      & -0.76380966779661_wp, -0.78802624659322_wp, 11.17846850674576_wp, &
      & -0.73993751479661_wp, -1.18654659259322_wp, 13.05825945174576_wp, &
      & -2.06630122779661_wp,  1.37437529840678_wp, 10.51808788574576_wp, &
      & -3.09578879879661_wp,  2.74576192240678_wp, 12.08454412074576_wp, &
      & -3.02291529479661_wp,  7.69314374440678_wp,  4.22151124774576_wp, &
      & -1.76704913379661_wp,  8.72918356240678_wp,  2.95542601474576_wp, &
      & -1.57227110679661_wp,  6.32605624740678_wp,  6.33157853274576_wp, &
      &  0.17108521220339_wp,  5.52645384340678_wp,  5.56569167574576_wp, &
      & -0.93608540379661_wp,  7.91244663440678_wp,  8.38547295774576_wp, &
      & -2.31030283679661_wp,  9.14822822240678_wp,  8.48338230674576_wp, &
      & -4.75701498079661_wp,  9.39244148040678_wp,  5.45232512374576_wp, &
      & -4.81535781379661_wp, 10.95500582440678_wp,  4.49588661174576_wp],&
      & shape(xyz))
   real(wp), parameter :: charge = -1.0_wp
   call new(self, sym, xyz, charge=charge)
end subroutine upu3b

subroutine upu3d(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 59
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "O", "C", "H", "H", "C", "H", "O", "C", "H", "N", "C", "H", "C", &
      & "H", "C", "O", "N", "H", "C", "O", "C", "H", "C", "H", "O", "H", "O", &
      & "P", "O", "O", "O", "C", "H", "H", "C", "H", "O", "C", "H", "N", "C", &
      & "H", "C", "H", "C", "O", "N", "H", "C", "O", "C", "H", "C", "H", "O", &
      & "H", "O", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -5.83974690701695_wp,  7.97735003030509_wp,  2.47172362161017_wp, &
      & -6.10441455901695_wp,  6.17423358330509_wp,  2.26623576261017_wp, &
      & -4.69848865801695_wp,  4.90585913630508_wp,  4.21245889061017_wp, &
      & -5.41215879601695_wp,  5.42234694930509_wp,  6.09048634361017_wp, &
      & -2.68166763401695_wp,  5.38461082230508_wp,  4.10200440661017_wp, &
      & -4.93499353501695_wp,  2.08693786830508_wp,  3.83383981661017_wp, &
      & -3.79516165101695_wp,  1.14150001830508_wp,  5.28610349961017_wp, &
      & -7.54865333201695_wp,  1.28241310630508_wp,  4.11192924761017_wp, &
      & -8.10468328401695_wp, -0.72895775869492_wp,  2.44421374661017_wp, &
      & -8.84286014501695_wp, -2.35529468569492_wp,  3.47630611561017_wp, &
      &-10.15482414501695_wp,  0.11520608530508_wp,  0.67676965761017_wp, &
      &-10.87780609701695_wp,  2.59675721730508_wp,  0.49682597961017_wp, &
      & -9.83348361101695_wp,  3.87995883230508_wp,  1.70477662761017_wp, &
      &-12.74309906401695_wp,  3.37564765730508_wp, -1.08391941738983_wp, &
      &-13.27314188201695_wp,  5.34789420230508_wp, -1.17572892038983_wp, &
      &-14.02243808001695_wp,  1.58411806530508_wp, -2.68236455538983_wp, &
      &-15.74084018901695_wp,  2.03641283130508_wp, -4.20365085838983_wp, &
      &-13.11316372701695_wp, -0.89567637769492_wp, -2.42354239238983_wp, &
      &-13.92096130601695_wp, -2.23763429869492_wp, -3.53693946638983_wp, &
      &-11.23430668501695_wp, -1.74765045869492_wp, -0.83314049538983_wp, &
      &-10.56386894201695_wp, -3.97396140369492_wp, -0.73488778338983_wp, &
      & -4.15254115001695_wp,  1.09501994030508_wp,  1.23565876461017_wp, &
      & -4.69496318501695_wp,  2.40339875230508_wp, -0.26759350538983_wp, &
      & -5.64835867101695_wp, -1.37977390969492_wp,  1.02737505561017_wp, &
      & -6.00146587801695_wp, -1.95803119669492_wp, -0.92444581938983_wp, &
      & -4.38212566101695_wp, -3.34031794869492_wp,  2.39153007561017_wp, &
      & -2.58343665401695_wp, -2.93270310769492_wp,  2.27383850761017_wp, &
      & -1.49831725101695_wp,  0.62209295630508_wp,  1.16937171261017_wp, &
      &  0.33740396198305_wp,  2.27654617130508_wp, -0.77031542138983_wp, &
      &  1.40169291498305_wp,  4.51060952830509_wp,  0.59736431561017_wp, &
      & -1.01333192501695_wp,  2.57038023230508_wp, -3.24800886638983_wp, &
      &  2.63119872198305_wp,  0.23039395430508_wp, -0.96791669738983_wp, &
      &  2.15206001998305_wp, -2.16574326369492_wp, -2.16758732238983_wp, &
      &  1.54934476598305_wp, -1.85572237669492_wp, -4.12183838438983_wp, &
      &  0.65747762398305_wp, -3.18542599069492_wp, -1.15838860538983_wp, &
      &  4.53139171598305_wp, -3.74253138569492_wp, -2.08708915238983_wp, &
      &  4.09949162798305_wp, -5.60055589969491_wp, -2.90610376638983_wp, &
      &  6.49791031198305_wp, -2.55086431769492_wp, -3.58306975638983_wp, &
      &  8.90227294098305_wp, -2.97067280469492_wp, -2.49338723238983_wp, &
      & 10.17044518698305_wp, -3.87600537769492_wp, -3.84446100238983_wp, &
      & 10.05473991098305_wp, -0.47005320369492_wp, -1.85470690138983_wp, &
      &  8.67580156398305_wp,  1.72235171030508_wp, -1.80903335838983_wp, &
      &  6.69024038798305_wp,  1.49347509030508_wp, -2.25232962738983_wp, &
      &  9.72555871398305_wp,  3.99391781330508_wp, -1.24313147738983_wp, &
      &  8.59230477698305_wp,  5.69578122930509_wp, -1.23268790738983_wp, &
      & 12.37424401498305_wp,  4.16537888830509_wp, -0.64339046338983_wp, &
      & 13.56609117398305_wp,  6.10299223630509_wp, -0.09754235438983_wp, &
      & 13.62529326598305_wp,  1.82562939330508_wp, -0.69594544338983_wp, &
      & 15.50036518698305_wp,  1.83296682130508_wp, -0.27397322038983_wp, &
      & 12.61609209198305_wp, -0.50716420969492_wp, -1.26340880438983_wp, &
      & 13.86580077098305_wp, -2.46989957469492_wp, -1.26416941938983_wp, &
      &  5.71155375398305_wp, -4.12081805769491_wp,  0.52146857361017_wp, &
      &  5.51556819298305_wp, -2.41618108969492_wp,  1.67284895161017_wp, &
      &  8.50832629898305_wp, -4.61871569169491_wp, -0.13605837238983_wp, &
      &  9.81059411598305_wp, -4.11498466169491_wp,  1.38988062561017_wp, &
      &  8.81429051098305_wp, -7.20103079269491_wp, -0.85488028138983_wp, &
      &  7.54492074498305_wp, -8.12770120669492_wp,  0.12497231361017_wp, &
      &  4.54691883798305_wp, -6.24247087469491_wp,  1.72023326361017_wp, &
      &  4.74590849698305_wp, -6.06563919769491_wp,  3.53742117661017_wp],&
      & shape(xyz))
   real(wp), parameter :: charge = -1.0_wp
   call new(self, sym, xyz, charge=charge)
end subroutine upu3d

subroutine upu4b(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 59
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "O", "C", "H", "H", "C", "H", "O", "C", "H", "N", "C", "H", "C", &
      & "H", "C", "O", "N", "H", "C", "O", "C", "H", "C", "H", "O", "H", "O", &
      & "P", "O", "O", "O", "C", "H", "H", "C", "H", "O", "C", "H", "N", "C", &
      & "H", "C", "H", "C", "O", "N", "H", "C", "O", "C", "H", "C", "H", "O", &
      & "H", "O", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -5.84108728674576_wp, -5.79015737779661_wp,  8.09384381281356_wp, &
      & -4.70066391874576_wp, -4.87855941779661_wp,  6.98420105681356_wp, &
      & -5.93451987474576_wp, -4.56791888979661_wp,  4.57954930081356_wp, &
      & -7.82214448974576_wp, -3.74817162879661_wp,  4.82582746281356_wp, &
      & -6.11071206774576_wp, -6.38995640679661_wp,  3.60015194081356_wp, &
      & -4.34127752974576_wp, -2.81844836179661_wp,  2.96635938681356_wp, &
      & -5.13966276174576_wp, -2.77361385679661_wp,  1.05883085481356_wp, &
      & -4.40691470574576_wp, -0.24872450479661_wp,  3.97138029681356_wp, &
      & -1.90673314274576_wp,  0.58638396020339_wp,  4.56118532181356_wp, &
      & -1.17663826274576_wp,  1.86057630720339_wp,  3.10942128181356_wp, &
      & -2.04037966774576_wp,  2.08054249120339_wp,  6.88364117881356_wp, &
      & -2.80414852074576_wp,  0.94441106920339_wp,  9.09581489381356_wp, &
      & -3.27250501274576_wp, -1.04578825079661_wp,  8.91740642981356_wp, &
      & -3.00561371974576_wp,  2.21252189920339_wp, 11.30824013181356_wp, &
      & -3.59667106574576_wp,  1.26161234820339_wp, 13.01839094181356_wp, &
      & -2.46799536874576_wp,  4.88286818520339_wp, 11.39014917181356_wp, &
      & -2.58147511474576_wp,  6.26854680620339_wp, 13.26821555281356_wp, &
      & -1.80585087674576_wp,  5.90863053920339_wp,  9.03677664181356_wp, &
      & -1.45477380274576_wp,  7.79722250520339_wp,  8.98493711181356_wp, &
      & -1.56978139274576_wp,  4.67557620420339_wp,  6.75086959681356_wp, &
      & -1.02345896374576_wp,  5.77917259220339_wp,  4.78335492581356_wp, &
      & -1.54582061274576_wp, -3.54366491479661_wp,  2.82609768781356_wp, &
      & -1.23818172474576_wp, -5.54466156179661_wp,  3.22408399681356_wp, &
      & -0.26854498874576_wp, -1.81393163079661_wp,  4.75969933581356_wp, &
      & -0.43737027374576_wp, -2.64234233979661_wp,  6.64239101081356_wp, &
      &  2.33234460725424_wp, -1.40629297879661_wp,  4.25571980781356_wp, &
      &  2.52860795825424_wp, -1.76901000879661_wp,  2.45093586681356_wp, &
      & -0.52287130474576_wp, -2.89319874079661_wp,  0.39002205581356_wp, &
      &  0.49878807625424_wp, -5.18642752879661_wp, -1.47535046918644_wp, &
      & -1.44186848674576_wp, -7.25126630079661_wp, -1.53206983818644_wp, &
      &  3.21264775925424_wp, -5.72013376979661_wp, -0.86519706418644_wp, &
      &  0.51922735225424_wp, -3.56048158679661_wp, -4.08698844818644_wp, &
      & -1.84317504374576_wp, -2.68385073879661_wp, -5.14645992318644_wp, &
      & -2.92912441474576_wp, -1.67384926679661_wp, -3.70319549318644_wp, &
      & -2.93935350174576_wp, -4.30069538279661_wp, -5.83221786418644_wp, &
      & -1.22488655774576_wp, -0.92716996179661_wp, -7.32289155018644_wp, &
      & -2.96248960574576_wp, -0.50799415479661_wp, -8.36348863118644_wp, &
      & -0.24652760274576_wp,  1.46128627620339_wp, -6.33230077718644_wp, &
      &  2.23205760425424_wp,  1.92629209520339_wp, -7.26114021818644_wp, &
      &  2.24509860425424_wp,  3.34957645320339_wp, -8.75678331818644_wp, &
      &  3.72823663025424_wp,  2.99505932120339_wp, -5.17447700118644_wp, &
      &  3.67237198325424_wp,  1.84342515420339_wp, -2.84576350518644_wp, &
      &  2.46383287525424_wp,  0.19025619720339_wp, -2.75302709218644_wp, &
      &  4.98677788825424_wp,  2.72320825420339_wp, -0.83363712818644_wp, &
      &  4.89145954225424_wp,  1.80305361520339_wp,  0.98882349881356_wp, &
      &  6.53787178425424_wp,  4.94749738120339_wp, -1.09534301318644_wp, &
      &  7.83635259025424_wp,  5.94472978520339_wp,  0.57232467781356_wp, &
      &  6.48628302025424_wp,  5.99718612220339_wp, -3.52820875618644_wp, &
      &  7.55342035825424_wp,  7.56993457720339_wp, -3.81261648618644_wp, &
      &  5.16507637025424_wp,  5.15998743120339_wp, -5.61581967118644_wp, &
      &  5.25806203825424_wp,  6.24251148420339_wp, -7.66909577718644_wp, &
      &  0.79997373825424_wp, -1.94647644479661_wp, -9.13058340218644_wp, &
      &  0.95242020125424_wp, -4.00150224779661_wp, -9.04603422618644_wp, &
      &  3.23193430325424_wp, -0.63855418979661_wp, -8.23007794518644_wp, &
      &  4.04329817225424_wp, -1.69633662779661_wp, -6.65348918918644_wp, &
      &  5.08901036325424_wp, -0.42720251079661_wp,-10.14314526418644_wp, &
      &  4.13267087325424_wp, -0.28443994779661_wp,-11.72270655918644_wp, &
      &  0.34718065125424_wp, -1.13406245479661_wp,-11.68884852718644_wp, &
      & -0.14178368074576_wp, -2.59718506979661_wp,-12.67768809218644_wp],&
      & shape(xyz))
   real(wp), parameter :: charge = -1.0_wp
   call new(self, sym, xyz, charge=charge)
end subroutine upu4b

subroutine upu5z(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 59
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "O", "C", "H", "H", "C", "H", "O", "C", "H", "N", "C", "H", "C", &
      & "H", "C", "O", "N", "H", "C", "O", "C", "H", "C", "H", "O", "H", "O", &
      & "P", "O", "O", "O", "C", "H", "H", "C", "H", "O", "C", "H", "N", "C", &
      & "H", "C", "H", "C", "O", "N", "H", "C", "O", "C", "H", "C", "H", "O", &
      & "H", "O", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -0.64581216571186_wp,  5.89583091513559_wp,  3.35707213755932_wp, &
      & -0.80805251271186_wp,  4.09640756013559_wp,  3.66389655255932_wp, &
      &  0.54953741028814_wp,  3.47493847613559_wp,  5.92984809055932_wp, &
      & -0.40280530871186_wp,  4.25686627313559_wp,  7.59744850755932_wp, &
      &  2.48851173928814_wp,  4.20814025513559_wp,  5.86241567555932_wp, &
      &  0.70415611428814_wp,  0.62963512213559_wp,  6.13366203055932_wp, &
      &  1.80597105728814_wp,  0.13724528313559_wp,  7.81988739755932_wp, &
      & -1.80720431171186_wp, -0.45162640586441_wp,  6.34890314355932_wp, &
      & -1.90359526571186_wp, -2.84135064986441_wp,  5.12940230255932_wp, &
      & -2.47463361071186_wp, -4.32076251386441_wp,  6.44501593055932_wp, &
      & -3.89956108471186_wp, -2.73781369586441_wp,  3.14612863955932_wp, &
      & -4.41838214271186_wp, -0.53938093386441_wp,  1.86766607855932_wp, &
      & -3.31589332371186_wp,  1.08795773613559_wp,  2.45769239055932_wp, &
      & -6.21265656171186_wp, -0.39486735186441_wp,  0.04177748855932_wp, &
      & -6.56500712171186_wp,  1.36778136913559_wp, -0.93119820444068_wp, &
      & -7.67125158271186_wp, -2.59857188186441_wp, -0.62338149344068_wp, &
      & -9.35807183571186_wp, -2.72985908386441_wp, -2.23418808544068_wp, &
      & -6.97064908971186_wp, -4.77010892986441_wp,  0.73147834255932_wp, &
      & -7.89465064371186_wp, -6.39834427586441_wp,  0.29956729455932_wp, &
      & -5.15246566771186_wp, -4.98785166286441_wp,  2.58403077755932_wp, &
      & -4.66170382871186_wp, -6.99531684186441_wp,  3.64768912855932_wp, &
      &  1.85447843328814_wp, -0.69056902886441_wp,  3.83952995455932_wp, &
      &  1.25468771928814_wp,  0.22843703413559_wp,  2.09070439055932_wp, &
      &  0.74522571828814_wp, -3.36701232686441_wp,  4.04899190755932_wp, &
      &  0.66746406028814_wp, -4.36162500386441_wp,  2.23966393155932_wp, &
      &  2.11245324328814_wp, -4.79798333386441_wp,  5.88262553955932_wp, &
      &  3.87769372328814_wp, -4.28337163986441_wp,  5.71158360555932_wp, &
      &  4.53997370828814_wp, -0.63883494586441_wp,  4.06608939255932_wp, &
      &  6.35312819028814_wp, -0.70968190686441_wp,  1.53688900855932_wp, &
      &  5.82737675128814_wp, -3.06072799486441_wp,  0.03384819855932_wp, &
      &  8.96998351428813_wp, -0.11822733986441_wp,  2.43721191755932_wp, &
      &  5.09746139428814_wp,  1.72123345013559_wp,  0.07434068055932_wp, &
      &  6.49421135328814_wp,  2.83425843713559_wp, -1.99031913444068_wp, &
      &  7.69704216628814_wp,  1.41565153013559_wp, -2.90060675844068_wp, &
      &  7.68846337728814_wp,  4.36089606613559_wp, -1.26121239244068_wp, &
      &  4.65875923828814_wp,  3.89496560413559_wp, -3.91913146144068_wp, &
      &  5.74176914028814_wp,  4.94408062513559_wp, -5.33740218744068_wp, &
      &  3.36434191028814_wp,  1.83205095113559_wp, -5.21954904744068_wp, &
      &  0.70767402828814_wp,  2.06084536813559_wp, -4.93320359244068_wp, &
      & -0.19649882071186_wp,  2.84462547413559_wp, -6.61702787344068_wp, &
      & -0.32827357071186_wp, -0.51098307686441_wp, -4.62161536744068_wp, &
      &  0.98667922028814_wp, -2.29290322286441_wp, -3.27088023644068_wp, &
      &  2.79345304228814_wp, -1.69111393486441_wp, -2.51052441744068_wp, &
      &  0.12747221328814_wp, -4.68311917386441_wp, -2.92647049344068_wp, &
      &  1.24944939828814_wp, -6.04435570086441_wp, -1.89365284744068_wp, &
      & -2.26060494571186_wp, -5.44055139686441_wp, -3.98466205744068_wp, &
      & -3.26496974371186_wp, -7.54401805886441_wp, -3.82089519044068_wp, &
      & -3.49325676871186_wp, -3.49067731786441_wp, -5.29682145544068_wp, &
      & -5.20338754771186_wp, -3.91269432786441_wp, -6.06381531444068_wp, &
      & -2.68002112571186_wp, -1.04126815786441_wp, -5.68785716344068_wp, &
      & -3.92620962271186_wp,  0.51589534913559_wp, -6.87495884244068_wp, &
      &  2.57666518928814_wp,  5.58569815613559_wp, -2.81779235144068_wp, &
      &  3.10302171828814_wp,  6.40005062813559_wp, -0.99558003444068_wp, &
      &  0.31218215728814_wp,  3.78696417313559_wp, -2.60940357344068_wp, &
      &  0.49382129628814_wp,  2.66631603213559_wp, -0.88734673444068_wp, &
      & -2.04431201171186_wp,  5.03669137113559_wp, -2.53477470444068_wp, &
      & -1.82728699671186_wp,  6.54470915213559_wp, -3.58243350144068_wp, &
      &  1.83205437828814_wp,  7.53586552113559_wp, -4.56139055544068_wp, &
      &  2.71205460728814_wp,  9.08153420313559_wp, -4.11696536444068_wp],&
      & shape(xyz))
   real(wp), parameter :: charge = -1.0_wp
   call new(self, sym, xyz, charge=charge)
end subroutine upu5z

subroutine upu6p(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 59
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "O", "C", "H", "H", "C", "H", "O", "C", "H", "N", "C", "H", "C", &
      & "H", "C", "O", "N", "H", "C", "O", "C", "H", "C", "H", "O", "H", "O", &
      & "P", "O", "O", "O", "C", "H", "H", "C", "H", "O", "C", "H", "N", "C", &
      & "H", "C", "H", "C", "O", "N", "H", "C", "O", "C", "H", "C", "H", "O", &
      & "H", "O", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -2.08459395835593_wp, -6.13900585271186_wp, -5.76592323933898_wp, &
      & -0.70327965935593_wp, -5.44343040771186_wp, -4.77944999033898_wp, &
      & -1.76785887435593_wp, -4.15250510471186_wp, -2.63883471733898_wp, &
      & -3.06277282235593_wp, -5.39552358571186_wp, -1.60313051533898_wp, &
      & -2.80172040535593_wp, -2.45583919071186_wp, -3.24295895133898_wp, &
      &  0.36837733564407_wp, -3.35667553871186_wp, -0.90485862033898_wp, &
      & -0.42076486735593_wp, -2.22585235671186_wp,  0.63741535766102_wp, &
      &  1.53885018164407_wp, -5.59334974071186_wp,  0.21597717066102_wp, &
      &  4.17650250764407_wp, -5.62710062471186_wp, -0.31602718233898_wp, &
      &  5.30230241964407_wp, -4.98588938371186_wp,  1.29202229866102_wp, &
      &  4.91892992864407_wp, -8.25657610571186_wp, -0.79067230533898_wp, &
      &  3.58450615664407_wp, -9.64890750671186_wp, -2.53524938733898_wp, &
      &  2.00494996364407_wp, -8.64913039771186_wp, -3.38717981533898_wp, &
      &  4.17169976864407_wp,-12.07219290671186_wp, -3.11269465833898_wp, &
      &  3.08134998164407_wp,-13.12152524571187_wp, -4.48768102133898_wp, &
      &  6.27364671264407_wp,-13.28714139871186_wp, -1.87369184433898_wp, &
      &  7.01745155464407_wp,-15.47641295771187_wp, -2.21642162133898_wp, &
      &  7.51980818664407_wp,-11.73489344671186_wp, -0.12391329233898_wp, &
      &  9.01049510164407_wp,-12.51064637471186_wp,  0.80762354466102_wp, &
      &  6.97890794764407_wp, -9.26397429171186_wp,  0.51163500666102_wp, &
      &  8.22250506364407_wp, -8.07265323271186_wp,  2.07041026066102_wp, &
      &  2.50266788464407_wp, -1.88344610071186_wp, -2.20160314533898_wp, &
      &  1.89199121464407_wp, -0.94946736071186_wp, -3.93596474733898_wp, &
      &  4.53716899564407_wp, -3.89314947171186_wp, -2.63411607133898_wp, &
      &  4.09637151164407_wp, -4.94772825771186_wp, -4.35445515033898_wp, &
      &  7.00968081264407_wp, -2.90896713671186_wp, -2.88186311733898_wp, &
      &  7.07931816064407_wp, -1.45714092471186_wp, -1.73961106133898_wp, &
      &  3.59292129564407_wp, -0.04333562271186_wp, -0.51556316133898_wp, &
      &  2.84548345464407_wp,  2.97524540028814_wp, -0.77964877733898_wp, &
      &  1.90700376464407_wp,  3.53395838528814_wp, -3.38920257833898_wp, &
      &  5.00769791764407_wp,  4.36044214128814_wp,  0.41208291866102_wp, &
      &  0.34636977664407_wp,  3.17423657128814_wp,  1.03351590966102_wp, &
      &  0.57804715864407_wp,  2.86718198628814_wp,  3.72884928566102_wp, &
      &  2.04927934464407_wp,  4.13062802828814_wp,  4.44846922666102_wp, &
      &  1.08207015064407_wp,  0.90851992628814_wp,  4.18558056166102_wp, &
      & -1.96130747835593_wp,  3.50518881928814_wp,  4.89840448066102_wp, &
      & -1.84295677435593_wp,  3.20444062828814_wp,  6.94218471366102_wp, &
      & -2.54966402835593_wp,  6.16242002828814_wp,  4.47149950566102_wp, &
      & -4.82157292735593_wp,  6.39640042028814_wp,  3.03193534866102_wp, &
      & -6.40756325935593_wp,  6.94773326928814_wp,  4.23415351166102_wp, &
      & -4.49055075935593_wp,  8.46967625228814_wp,  1.22672054966102_wp, &
      & -2.67788496035593_wp,  8.28751195828814_wp, -0.63285108333898_wp, &
      & -1.60079405735593_wp,  6.54257129228814_wp, -0.61500299933898_wp, &
      & -2.23092432335593_wp, 10.15322746828814_wp, -2.32498014333898_wp, &
      & -0.79279390035593_wp,  9.94675279328814_wp, -3.76328083033898_wp, &
      & -3.66515680835593_wp, 12.46637925828814_wp, -2.20070136833898_wp, &
      & -3.45417305835593_wp, 14.31680178328814_wp, -3.61187333833898_wp, &
      & -5.42491216935593_wp, 12.51884617828814_wp, -0.21886276333898_wp, &
      & -6.46756073635593_wp, 14.12200429228814_wp, -0.03320247633898_wp, &
      & -5.93410809735593_wp, 10.65935107928814_wp,  1.53506147066102_wp, &
      & -7.49454442035593_wp, 10.94239404928814_wp,  3.23256944666102_wp, &
      & -4.18473640235593_wp,  1.99258297228814_wp,  3.81294624966102_wp, &
      & -3.57680966135593_wp,  0.19961699928814_wp,  2.99380804766102_wp, &
      & -5.31066576035593_wp,  3.78868948028814_wp,  1.82243682566102_wp, &
      & -4.25893025735593_wp,  3.60862504928814_wp,  0.05720182666102_wp, &
      & -7.89391155235593_wp,  3.32306061828814_wp,  1.30225138566102_wp, &
      & -8.59496474535593_wp,  2.67877271528814_wp,  2.89031247966102_wp, &
      & -6.11650747335593_wp,  1.57158981628814_wp,  5.67940706766102_wp, &
      & -6.10237005535593_wp, -0.20238913471186_wp,  6.13699552366102_wp],&
      & shape(xyz))
   real(wp), parameter :: charge = -1.0_wp
   call new(self, sym, xyz, charge=charge)
end subroutine upu6p

subroutine upu7a(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 59
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "O", "C", "H", "H", "C", "H", "O", "C", "H", "N", "C", "H", "C", &
      & "H", "C", "O", "N", "H", "C", "O", "C", "H", "C", "H", "O", "H", "O", &
      & "P", "O", "O", "O", "C", "H", "H", "C", "H", "O", "C", "H", "N", "C", &
      & "H", "C", "H", "C", "O", "N", "H", "C", "O", "C", "H", "C", "H", "O", &
      & "H", "O", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &-10.11323205122034_wp, -2.64249749930508_wp, -4.48407978318644_wp, &
      & -8.54120547122034_wp, -2.69964149030508_wp, -3.54109120718644_wp, &
      & -8.82527645222034_wp, -1.12018016230508_wp, -1.35317609318644_wp, &
      &-10.54980940022034_wp, -1.60841846130508_wp, -0.31192101018644_wp, &
      & -8.91998158222034_wp,  0.88333473369492_wp, -1.89160948318644_wp, &
      & -6.55849286822034_wp, -1.45507985830508_wp,  0.34938198681356_wp, &
      & -6.73360508322034_wp, -0.12603292430508_wp,  1.93107149081356_wp, &
      & -6.47856841922034_wp, -4.01521694630508_wp,  1.37131991281356_wp, &
      & -3.95183110422034_wp, -4.85820197930508_wp,  1.58510593681356_wp, &
      & -3.58702158022034_wp, -5.57631183930508_wp,  3.48312839181356_wp, &
      & -3.53466161922034_wp, -7.01157649230508_wp, -0.21062073618644_wp, &
      & -5.22473994922034_wp, -7.59522176630508_wp, -2.08503692218644_wp, &
      & -6.87361393722034_wp, -6.38146323530508_wp, -2.16235770718644_wp, &
      & -4.83996225722034_wp, -9.52577339730508_wp, -3.73063544818644_wp, &
      & -6.20987262722034_wp, -9.94459563630508_wp, -5.18952224318644_wp, &
      & -2.58790077222034_wp,-11.04084287930508_wp, -3.55805622218644_wp, &
      & -2.00114349822034_wp,-12.84383309230508_wp, -4.92885060618644_wp, &
      & -0.95608949822034_wp,-10.29414316630508_wp, -1.60265167118644_wp, &
      &  0.67597102977966_wp,-11.28989478130508_wp, -1.40519363518644_wp, &
      & -1.29196127022034_wp, -8.34906275430508_wp,  0.09782676481356_wp, &
      &  0.25924638677966_wp, -7.81694010430508_wp,  1.74773386581356_wp, &
      & -3.97090826622034_wp, -1.12680799830508_wp, -0.88975197418644_wp, &
      & -3.94192931822034_wp, -1.96488975830508_wp, -2.77704135218644_wp, &
      & -2.24546541522034_wp, -2.62106177030508_wp,  0.89300796581356_wp, &
      & -0.48364893022034_wp, -3.22624179530508_wp,  0.00093465481356_wp, &
      & -1.75292780022034_wp, -1.19377062730508_wp,  3.13027327481356_wp, &
      & -1.73080326622034_wp,  0.57669947969492_wp,  2.60341521181356_wp, &
      & -3.30238330422034_wp,  1.48068448969492_wp, -0.98854023518644_wp, &
      & -1.73517873822034_wp,  2.53403892169492_wp, -3.50621347918644_wp, &
      & -3.53236938222034_wp,  4.09966424369492_wp, -5.03866088218644_wp, &
      & -0.24694978922034_wp,  0.41388768069492_wp, -4.64372100018644_wp, &
      &  0.28998599277966_wp,  4.35370766769492_wp, -2.05658879818644_wp, &
      & -0.62714527322034_wp,  6.36176678769492_wp, -0.45756823918644_wp, &
      & -2.25503083022034_wp,  5.71678667769492_wp,  0.64286002381356_wp, &
      & -1.19448542422034_wp,  7.98130599169492_wp, -1.61622651818644_wp, &
      &  1.46876739677966_wp,  7.15577427969492_wp,  1.30807021681356_wp, &
      &  0.82321564477966_wp,  8.80677798869492_wp,  2.38523916481356_wp, &
      &  1.99753899677966_wp,  5.10807305469492_wp,  3.07094324781356_wp, &
      &  4.59893107077966_wp,  5.02117513869492_wp,  3.67751675581356_wp, &
      &  4.86901222077966_wp,  5.02463636069491_wp,  5.72170365781356_wp, &
      &  5.70355263477966_wp,  2.60411374069492_wp,  2.68028869781356_wp, &
      &  4.37895732777966_wp,  1.03141555269492_wp,  1.10561026581356_wp, &
      &  2.49301645877966_wp,  1.67584915369492_wp,  0.65602744481356_wp, &
      &  5.37538017377966_wp, -1.12887006730508_wp,  0.14940834781356_wp, &
      &  4.27949254477966_wp, -2.31886926230508_wp, -1.09971357118644_wp, &
      &  7.93424887777966_wp, -1.83053458730508_wp,  0.74569135281356_wp, &
      &  9.07843865877966_wp, -3.73202827930508_wp,  0.00715638881356_wp, &
      &  9.17378705077966_wp, -0.07491262330508_wp,  2.30357484581356_wp, &
      & 11.00014712777966_wp, -0.45918103330508_wp,  2.76169488081356_wp, &
      &  8.21270415977966_wp,  2.12760988369492_wp,  3.30695698481356_wp, &
      &  9.46559702677966_wp,  3.57076546569492_wp,  4.63460741281356_wp, &
      &  4.03580198477966_wp,  7.72685627669492_wp,  0.12782446481356_wp, &
      &  4.45293461977966_wp,  6.38568032469492_wp, -1.39035872018644_wp, &
      &  5.82708080977966_wp,  7.30765892769492_wp,  2.37700827481356_wp, &
      &  7.77896631377966_wp,  6.93779854369492_wp,  1.81164987881356_wp, &
      &  5.69090318677966_wp,  9.40870088069492_wp,  4.06549072981356_wp, &
      &  5.45379321977966_wp, 10.89280002269492_wp,  2.98861128781356_wp, &
      &  4.05831750277966_wp, 10.26367773669492_wp, -0.79106500318644_wp, &
      &  5.42240676077966_wp, 10.42085626269492_wp, -2.01088124018644_wp],&
      & shape(xyz))
   real(wp), parameter :: charge = -1.0_wp
   call new(self, sym, xyz, charge=charge)
end subroutine upu7a

subroutine upu7p(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 59
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "O", "C", "H", "H", "C", "H", "O", "C", "H", "N", "C", "H", "C", &
      & "H", "C", "O", "N", "H", "C", "O", "C", "H", "C", "H", "O", "H", "O", &
      & "P", "O", "O", "O", "C", "H", "H", "C", "H", "O", "C", "H", "N", "C", &
      & "H", "C", "H", "C", "O", "N", "H", "C", "O", "C", "H", "C", "H", "O", &
      & "H", "O", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  3.71294799123729_wp, -8.79942894345763_wp, -4.03008671374576_wp, &
      &  3.48726822023729_wp, -7.81528950545763_wp, -2.49940393674576_wp, &
      &  0.85901841623729_wp, -7.89717341145763_wp, -1.82490951974576_wp, &
      &  0.20508761823729_wp, -9.85583358245763_wp, -1.64202750774576_wp, &
      & -0.30215584476271_wp, -6.93665082345763_wp, -3.25343975574576_wp, &
      &  0.49718237623729_wp, -6.52321354845763_wp,  0.65194450925424_wp, &
      & -1.53496338076271_wp, -6.44370635045763_wp,  1.06014573225424_wp, &
      &  1.75792837723729_wp, -7.85429760745763_wp,  2.70190168925424_wp, &
      &  2.73183740623729_wp, -6.09566417645763_wp,  4.48746296025424_wp, &
      &  1.98392788923729_wp, -6.47856310245763_wp,  6.36897034325424_wp, &
      &  5.50963464523729_wp, -6.45663396645763_wp,  4.68038810925424_wp, &
      &  6.93031269223729_wp, -7.01360311145763_wp,  2.57935012425424_wp, &
      &  5.86378989423729_wp, -7.20658703245763_wp,  0.83648267725424_wp, &
      &  9.46632667223729_wp, -7.36865427945763_wp,  2.67878429325424_wp, &
      & 10.52821510723729_wp, -7.81329641145763_wp,  0.98895729825424_wp, &
      & 10.77588902023729_wp, -7.19958673145763_wp,  5.06100670325424_wp, &
      & 13.06664264523729_wp, -7.48839393245763_wp,  5.42398489325424_wp, &
      &  9.17437099923729_wp, -6.67501788245763_wp,  7.11030304625424_wp, &
      &  9.99489285823729_wp, -6.55688260745763_wp,  8.84418994825424_wp, &
      &  6.59721636123729_wp, -6.29320403745763_wp,  7.07575828825424_wp, &
      &  5.36522613123729_wp, -5.85181998145763_wp,  8.99799493525424_wp, &
      &  1.58845604923729_wp, -3.84220712845763_wp,  0.75375538725424_wp, &
      &  3.38082656923729_wp, -3.71418953145763_wp, -0.26262220974576_wp, &
      &  1.98574183723729_wp, -3.43586766445763_wp,  3.59117083625424_wp, &
      &  3.44332132023729_wp, -2.03450356145763_wp,  4.02990417225424_wp, &
      & -0.35962619076271_wp, -2.77283745845763_wp,  4.76579049625424_wp, &
      & -1.36154133676271_wp, -1.96094408745763_wp,  3.44251743025424_wp, &
      & -0.16924839176271_wp, -2.06466528845763_wp, -0.25637987774576_wp, &
      &  0.67200735223729_wp, -0.20079585245763_wp, -2.64456758574576_wp, &
      & -0.71894473776271_wp, -1.00712568045763_wp, -4.96764289474576_wp, &
      &  3.49238956723729_wp,  0.08613069354237_wp, -2.59827269974576_wp, &
      & -0.72805170576271_wp,  2.37825833754237_wp, -1.70468571874576_wp, &
      &  0.15130129223729_wp,  3.67804626754237_wp,  0.52396962025424_wp, &
      &  1.82855405123729_wp,  4.79412013754237_wp,  0.05921777925424_wp, &
      &  0.63278665623729_wp,  2.32211363354237_wp,  2.01467316425424_wp, &
      & -1.96954933576271_wp,  5.37700366254237_wp,  1.43405920025424_wp, &
      & -1.38634228976271_wp,  6.29624310754237_wp,  3.19337897925424_wp, &
      & -2.43078477376271_wp,  7.34996093354237_wp, -0.43825901274576_wp, &
      & -4.97835650676271_wp,  7.24643720854237_wp, -1.30679200674576_wp, &
      & -6.14000848976271_wp,  8.72414076154237_wp, -0.45205136674576_wp, &
      & -4.96764043776271_wp,  7.75783523454237_wp, -4.03274253474576_wp, &
      & -3.45940609376271_wp,  6.31606746754237_wp, -5.58602076474576_wp, &
      & -2.35575830576271_wp,  4.88824493254237_wp, -4.60870323774576_wp, &
      & -3.33692085876271_wp,  6.67164227754237_wp, -8.11705985174576_wp, &
      & -2.12843069376271_wp,  5.50963105754237_wp, -9.28717648474576_wp, &
      & -4.82535692076271_wp,  8.63403465754237_wp, -9.28215793974576_wp, &
      & -4.89507269176271_wp,  9.16733970954237_wp,-11.55675969774576_wp, &
      & -6.29945619876271_wp, 10.01231084454237_wp, -7.56294570474576_wp, &
      & -7.37830397976271_wp, 11.42560176354237_wp, -8.29198876374576_wp, &
      & -6.47412243876271_wp,  9.70366397654237_wp, -4.97889791374576_wp, &
      & -7.83546970376271_wp, 11.03877107354237_wp, -3.65148691474576_wp, &
      & -4.49911840576271_wp,  4.01363760354237_wp,  1.83001310925424_wp, &
      & -4.25762691976271_wp,  1.98463842954237_wp,  2.11455179725424_wp, &
      & -5.96568882576271_wp,  4.59079677054237_wp, -0.61183563574576_wp, &
      & -5.42285842176271_wp,  3.24304173754237_wp, -2.07794212874576_wp, &
      & -8.62055749976271_wp,  4.47317776854237_wp, -0.31549768774576_wp, &
      & -8.93352749376271_wp,  5.01440190654237_wp,  1.42706286725424_wp, &
      & -5.89436811076271_wp,  5.10646954554237_wp,  3.89396389925424_wp, &
      & -6.05384303176271_wp,  3.85287577854237_wp,  5.22070177725424_wp],&
      & shape(xyz))
   real(wp), parameter :: charge = -1.0_wp
   call new(self, sym, xyz, charge=charge)
end subroutine upu7p

subroutine upu8d(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 59
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "O", "C", "H", "H", "C", "H", "O", "C", "H", "N", "C", "H", "C", &
      & "H", "C", "O", "N", "H", "C", "O", "C", "H", "C", "H", "O", "H", "O", &
      & "P", "O", "O", "O", "C", "H", "H", "C", "H", "O", "C", "H", "N", "C", &
      & "H", "C", "H", "C", "O", "N", "H", "C", "O", "C", "H", "C", "H", "O", &
      & "H", "O", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -6.36178445842373_wp,  1.93408954967797_wp, -9.11854018137288_wp, &
      & -5.57294876842373_wp,  1.11560935067797_wp, -7.67949550937288_wp, &
      & -7.02620228942373_wp,  1.72853479467797_wp, -5.46315305837288_wp, &
      & -9.02866234042373_wp,  1.27986244367797_wp, -5.75297353037288_wp, &
      & -6.84410338042373_wp,  3.74503532067797_wp, -5.00569801737288_wp, &
      & -6.02482514942373_wp,  0.19860771967797_wp, -3.25710791337288_wp, &
      & -6.95753418242373_wp,  0.84600516667797_wp, -1.52924205537288_wp, &
      & -6.65669188342373_wp, -2.47467312232203_wp, -3.59304865937288_wp, &
      & -4.40956881942373_wp, -3.95093424632203_wp, -3.75200806337288_wp, &
      & -4.09520817642373_wp, -5.01194112432203_wp, -2.00854592037288_wp, &
      & -4.73430084542373_wp, -5.85279624632203_wp, -5.74098378637288_wp, &
      & -5.10395392642373_wp, -5.07421477632203_wp, -8.19536122537288_wp, &
      & -5.15435991042373_wp, -3.03625565532203_wp, -8.44477160737288_wp, &
      & -5.42600216342373_wp, -6.70786607632203_wp,-10.13728314837288_wp, &
      & -5.70008443042373_wp, -6.03166863832203_wp,-12.04691513437288_wp, &
      & -5.43866276042373_wp, -9.39006453432203_wp, -9.65968476137288_wp, &
      & -5.71479500242373_wp,-11.08602841532203_wp,-11.24338399037288_wp, &
      & -5.15061730742373_wp,-10.01466224932203_wp, -7.09928008837288_wp, &
      & -5.19542535642373_wp,-11.88461169032203_wp, -6.65906010437288_wp, &
      & -4.81025724842373_wp, -8.40508265632204_wp, -5.07533441937288_wp, &
      & -4.60483193842373_wp, -9.17221484432203_wp, -2.89276080537288_wp, &
      & -3.14756558342373_wp,  0.31435834967797_wp, -2.93635357137288_wp, &
      & -2.34864423642373_wp,  2.03711004167797_wp, -3.73508103237288_wp, &
      & -2.21510826142373_wp, -2.09947728632203_wp, -4.24284171337288_wp, &
      & -2.00312031142373_wp, -1.74770970632203_wp, -6.26356634337288_wp, &
      &  0.13896365557627_wp, -2.99947384032203_wp, -3.33067702437288_wp, &
      &  0.10205541757627_wp, -2.68896143532203_wp, -1.50925310137288_wp, &
      & -2.41702170442373_wp,  0.07549566167797_wp, -0.32835331437288_wp, &
      & -2.17563150742373_wp,  2.62914626467797_wp,  1.47262698662712_wp, &
      & -3.56662706142373_wp,  2.13619347667797_wp,  3.88148924162712_wp, &
      & -2.70323544542373_wp,  4.90272605967797_wp, -0.14019348637288_wp, &
      &  0.82137016757627_wp,  2.47532804967797_wp,  2.20557762062712_wp, &
      &  2.65376621757627_wp,  2.66681625167797_wp,  0.20401348862712_wp, &
      &  2.33941124357627_wp,  4.40702901167797_wp, -0.87188121737288_wp, &
      &  2.49157160257627_wp,  1.05001223667797_wp, -1.08249628037288_wp, &
      &  5.27249766157627_wp,  2.65188666067797_wp,  1.34034522162712_wp, &
      &  6.65227580257627_wp,  2.68034744567797_wp, -0.20802104337288_wp, &
      &  5.65254763657627_wp,  4.89541395467797_wp,  2.88004558162712_wp, &
      &  7.01885902257627_wp,  4.32931762067797_wp,  5.10103449162712_wp, &
      &  8.59648882857627_wp,  5.63908122467797_wp,  5.32583856262712_wp, &
      &  5.30657414557627_wp,  4.66636688067797_wp,  7.34898336562712_wp, &
      &  2.81936630757627_wp,  5.34073355267797_wp,  7.07552981062712_wp, &
      &  2.21899582457627_wp,  5.59302085467797_wp,  5.13870013162712_wp, &
      &  1.23390188657627_wp,  5.62639776167797_wp,  9.07140511162712_wp, &
      & -0.72067931142373_wp,  6.15057770467797_wp,  8.78129872362712_wp, &
      &  2.13839844457627_wp,  5.21264830367797_wp, 11.60112989462712_wp, &
      &  0.91096150957627_wp,  5.40525295567797_wp, 13.58439146262712_wp, &
      &  4.69623592557627_wp,  4.51608679967797_wp, 11.70530236262712_wp, &
      &  5.43963825657627_wp,  4.19698464567797_wp, 13.44847707862712_wp, &
      &  6.35296304757627_wp,  4.21173830267797_wp,  9.71855821762712_wp, &
      &  8.57835974257627_wp,  3.59802295367797_wp, 10.01554093962712_wp, &
      &  5.86485173757627_wp,  0.44111644467797_wp,  3.09567295562712_wp, &
      &  4.18997700957627_wp, -0.11204282432203_wp,  4.17296700262712_wp, &
      &  7.89157360257627_wp,  1.57118978367797_wp,  4.84915121362712_wp, &
      &  8.02077568057627_wp,  0.61090983667797_wp,  6.67089072062712_wp, &
      & 10.27889335857627_wp,  1.57672479067797_wp,  3.58783281462712_wp, &
      & 10.25480426557627_wp,  0.07969349867797_wp,  2.50033708062712_wp, &
      &  6.79914700657627_wp, -1.62589456732203_wp,  1.62813083762712_wp, &
      &  6.57322875257627_wp, -3.16889779032203_wp,  2.59807918862712_wp],&
      & shape(xyz))
   real(wp), parameter :: charge = -1.0_wp
   call new(self, sym, xyz, charge=charge)
end subroutine upu8d

subroutine upu9a(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 59
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "O", "C", "H", "H", "C", "H", "O", "C", "H", "N", "C", "H", "C", &
      & "H", "C", "O", "N", "H", "C", "O", "C", "H", "C", "H", "O", "H", "O", &
      & "P", "O", "O", "O", "C", "H", "H", "C", "H", "O", "C", "H", "N", "C", &
      & "H", "C", "H", "C", "O", "N", "H", "C", "O", "C", "H", "C", "H", "O", &
      & "H", "O", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -1.20504291955932_wp,  3.26913835406780_wp, -5.94749681820339_wp, &
      & -1.50089048355932_wp,  1.67705478906780_wp, -5.08643599020339_wp, &
      & -2.75550980455932_wp,  2.21372204206780_wp, -2.73664033320339_wp, &
      & -4.33768213455932_wp,  3.51864662406780_wp, -3.02952178320339_wp, &
      & -1.44300061755932_wp,  3.06634788906780_wp, -1.37328542320339_wp, &
      & -3.73100072255932_wp, -0.22704899193220_wp, -1.62166666120339_wp, &
      & -4.38462840855932_wp,  0.15080531006780_wp,  0.31009193779661_wp, &
      & -5.84813841755932_wp, -1.17159300193220_wp, -3.09726203520339_wp, &
      & -5.86962120055932_wp, -3.85122540093220_wp, -3.08192936520339_wp, &
      & -7.66208431755932_wp, -4.56847883793220_wp, -2.36134755420339_wp, &
      & -5.66336006555932_wp, -4.75743631893220_wp, -5.74011394220339_wp, &
      & -4.48940653355932_wp, -3.34540338993220_wp, -7.57147631220339_wp, &
      & -3.76740383755932_wp, -1.53319603893220_wp, -6.93399387920339_wp, &
      & -4.27538901755932_wp, -4.14732070793220_wp, -9.99752562520339_wp, &
      & -3.34699819755932_wp, -2.99050340093220_wp,-11.40506875320339_wp, &
      & -5.29562354555932_wp, -6.56262541693220_wp,-10.73188826220339_wp, &
      & -5.23808308955932_wp, -7.51320596693220_wp,-12.86802356120339_wp, &
      & -6.41927181555932_wp, -7.89295471993220_wp, -8.73158249920339_wp, &
      & -7.14210258955932_wp, -9.62586066493220_wp, -9.14150148320339_wp, &
      & -6.66219136755932_wp, -7.13758683293220_wp, -6.25030046420339_wp, &
      & -7.66631295755932_wp, -8.45233874593220_wp, -4.61559091720339_wp, &
      & -1.86529220455932_wp, -2.43553547493220_wp, -1.57335638320339_wp, &
      & -0.70819219355932_wp, -2.49471583493220_wp, -3.28231307520339_wp, &
      & -3.63561642555932_wp, -4.73231694693220_wp, -1.45075568520339_wp, &
      & -2.77666547655932_wp, -6.46693479493220_wp, -2.18425301720339_wp, &
      & -4.51267472955932_wp, -5.08152942893220_wp,  1.07807998179661_wp, &
      & -3.10340401355932_wp, -4.56835581693220_wp,  2.16025481379661_wp, &
      & -0.30094792855932_wp, -2.23397843393220_wp,  0.60700322879661_wp, &
      &  2.64685813244068_wp, -3.33133910393220_wp,  0.61243467879661_wp, &
      &  3.26722002544068_wp, -4.26127723293220_wp, -1.99299460620339_wp, &
      &  2.93927546544068_wp, -4.93707386093220_wp,  2.91760426679661_wp, &
      &  4.14423411044068_wp, -0.70505820493220_wp,  1.23067214279661_wp, &
      &  4.30231819244068_wp,  1.18723403306780_wp, -0.72806191620339_wp, &
      &  2.59373377744068_wp,  1.17986655806780_wp, -1.89478271120339_wp, &
      &  5.92981919044068_wp,  0.78875998606780_wp, -1.94310017020339_wp, &
      &  4.65742737544068_wp,  3.74729761106780_wp,  0.48281875079661_wp, &
      &  5.05622985744068_wp,  5.11929575006780_wp, -1.02249237520339_wp, &
      &  2.35194692744068_wp,  4.52938357806780_wp,  1.75855198079661_wp, &
      &  2.90765373544068_wp,  5.80298444006780_wp,  4.03733041179661_wp, &
      &  1.86105583044068_wp,  7.57751367906780_wp,  4.13947633679661_wp, &
      &  2.06030627044068_wp,  4.18334829306780_wp,  6.21597641979661_wp, &
      &  1.03549728344068_wp,  1.83193087306780_wp,  5.84217841379661_wp, &
      &  0.88975934744068_wp,  1.26656794306780_wp,  3.88435463579661_wp, &
      &  0.28785138444068_wp,  0.32450627706780_wp,  7.77716636579661_wp, &
      & -0.49607819955932_wp, -1.52737832793220_wp,  7.40926449379661_wp, &
      &  0.55298693244068_wp,  1.17236472406780_wp, 10.34667649479661_wp, &
      & -0.04660668355932_wp,  0.00269576806780_wp, 12.28232169279661_wp, &
      &  1.61051954844068_wp,  3.59753172406780_wp, 10.55366422979661_wp, &
      &  1.83797055944068_wp,  4.29612959606780_wp, 12.32931785579661_wp, &
      &  2.38805240944068_wp,  5.17234357106780_wp,  8.62991636779661_wp, &
      &  3.29455491244068_wp,  7.28175927806780_wp,  9.01109923379661_wp, &
      &  6.70713530044068_wp,  3.94710778806780_wp,  2.49927524679661_wp, &
      &  6.74368789244068_wp,  2.23622754406780_wp,  3.66106253379661_wp, &
      &  5.78472754144068_wp,  6.19945481106780_wp,  4.08088632779661_wp, &
      &  6.55138880844068_wp,  6.22196617206780_wp,  5.99712403979661_wp, &
      &  6.31660169244068_wp,  8.51436480306780_wp,  2.79668844879661_wp, &
      &  7.92172701244068_wp,  8.23742054606780_wp,  1.92347535979661_wp, &
      &  9.09509076044068_wp,  4.34364297706780_wp,  1.31140407179661_wp, &
      & 10.41358962244068_wp,  3.89485856606780_wp,  2.50859083979661_wp],&
      & shape(xyz))
   real(wp), parameter :: charge = -1.0_wp
   call new(self, sym, xyz, charge=charge)
end subroutine upu9a

subroutine upuaa(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 59
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "O", "C", "H", "H", "C", "H", "O", "C", "H", "N", "C", "H", "C", &
      & "H", "C", "O", "N", "H", "C", "O", "C", "H", "C", "H", "O", "H", "O", &
      & "P", "O", "O", "O", "C", "H", "H", "C", "H", "O", "C", "H", "N", "C", &
      & "H", "C", "H", "C", "O", "N", "H", "C", "O", "C", "H", "C", "H", "O", &
      & "H", "O", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  5.86215640562712_wp, -9.59316011298305_wp,  3.49742246781356_wp, &
      &  5.03615566262712_wp, -8.02347641598305_wp,  3.03109062781356_wp, &
      &  3.14253192362712_wp, -7.49075774598305_wp,  4.90071787081356_wp, &
      &  3.91587790762712_wp, -7.69881563198305_wp,  6.81192248681356_wp, &
      &  1.52887732562712_wp, -8.78147687998305_wp,  4.69718833381356_wp, &
      &  2.18148077962712_wp, -4.82305329698305_wp,  4.55458998881356_wp, &
      &  0.58441797662712_wp, -4.51681223998305_wp,  5.83910912081356_wp, &
      &  4.18425150162712_wp, -3.02815205598305_wp,  5.19541912681356_wp, &
      &  4.07720041362712_wp, -0.89089690498305_wp,  3.58969249681356_wp, &
      &  4.03251746462712_wp,  0.84739419201695_wp,  4.69572908781356_wp, &
      &  6.42939696762712_wp, -0.78093601698305_wp,  2.03055290381356_wp, &
      &  7.84065189562712_wp, -2.89619494698305_wp,  1.53252279981356_wp, &
      &  7.12785575662712_wp, -4.62977816998305_wp,  2.36885826381356_wp, &
      &  9.97560800662712_wp, -2.82478881498305_wp,  0.11227220881356_wp, &
      & 11.04805415062712_wp, -4.52812168298305_wp, -0.24623753618644_wp, &
      & 10.84570134662712_wp, -0.46898968798305_wp, -0.93510237518644_wp, &
      & 12.75120598262712_wp, -0.13787265598305_wp, -2.25048999218644_wp, &
      &  9.28197066462712_wp,  1.59189974501695_wp, -0.34055016918644_wp, &
      &  9.79955962162712_wp,  3.30553048601695_wp, -1.03951219218644_wp, &
      &  7.10136349862712_wp,  1.58517080901695_wp,  1.08975829881356_wp, &
      &  5.84444097962712_wp,  3.49608629701695_wp,  1.49824165881356_wp, &
      &  1.42635024262712_wp, -4.06501201498305_wp,  1.87560255281356_wp, &
      &  2.76817739362712_wp, -4.82148123398305_wp,  0.49512804881356_wp, &
      &  1.69540753962712_wp, -1.16498767598305_wp,  1.95518798481356_wp, &
      &  1.94152394062712_wp, -0.33624135098305_wp,  0.08032620181356_wp, &
      & -0.30683708337288_wp,  0.04017203501695_wp,  3.30289917681356_wp, &
      & -1.76510102437288_wp,  0.19558064401695_wp,  2.18777357081356_wp, &
      & -1.04452261537288_wp, -4.99832198098305_wp,  1.35029765581356_wp, &
      & -2.13091758337288_wp, -4.84877738098305_wp, -1.56561568718644_wp, &
      & -3.93712543237288_wp, -7.01169784498305_wp, -1.83192305518644_wp, &
      & -0.01501709237288_wp, -4.36200569798305_wp, -3.37746266818644_wp, &
      & -3.73241519537288_wp, -2.18534692498305_wp, -1.42463891618644_wp, &
      & -5.90376516037288_wp, -2.09206022398305_wp,  0.22453156981356_wp, &
      & -5.32819695337288_wp, -2.47093233198305_wp,  2.17777578581356_wp, &
      & -7.29006833637288_wp, -3.51693081698305_wp, -0.35090284418644_wp, &
      & -7.06437042437288_wp,  0.50156167501695_wp,  0.02166664981356_wp, &
      & -8.77378065037288_wp,  0.56272020001695_wp,  1.19377070781356_wp, &
      & -5.25202606737288_wp,  2.34077529001695_wp,  0.98726239581356_wp, &
      & -5.49634628437288_wp,  4.67138569201695_wp, -0.32941382518644_wp, &
      & -5.83902295937288_wp,  6.21944865701695_wp,  0.98535811881356_wp, &
      & -3.07287966037288_wp,  5.24132724001695_wp, -1.63689841918644_wp, &
      & -1.47170405637288_wp,  3.36355187601695_wp, -2.43604332118644_wp, &
      & -2.09507685137288_wp,  1.45648179201695_wp, -2.01317669018644_wp, &
      &  0.72254674162712_wp,  3.85040538501695_wp, -3.66373499118644_wp, &
      &  1.94955111862712_wp,  2.32923378801695_wp, -4.26290190718644_wp, &
      &  1.43617927462712_wp,  6.42023374401695_wp, -4.21396121618644_wp, &
      &  3.36811588462712_wp,  7.13142664001695_wp, -5.31608607518644_wp, &
      & -0.35495442037288_wp,  8.20909560901695_wp, -3.41028664218644_wp, &
      &  0.03090098162712_wp, 10.05244300001695_wp, -3.79391802518644_wp, &
      & -2.59732406737288_wp,  7.77656510901695_wp, -2.15881654918644_wp, &
      & -4.07608246537288_wp,  9.47316696201695_wp, -1.57226563318644_wp, &
      & -7.67958035837288_wp,  1.46641230201695_wp, -2.62744421818644_wp, &
      & -6.20221072037288_wp,  0.91191038301695_wp, -3.96344969018644_wp, &
      & -7.67099930137288_wp,  4.34698153401695_wp, -2.22156849118644_wp, &
      & -7.34262936437288_wp,  5.42290099501695_wp, -3.95901620418644_wp, &
      & -9.96433580337288_wp,  5.10568667701695_wp, -1.02425076518644_wp, &
      &-11.26889831437288_wp,  3.97984708001695_wp, -1.69530814918644_wp, &
      &-10.07774827537288_wp,  0.50362090901695_wp, -3.39467070218644_wp, &
      &-10.17609282837288_wp,  0.58806199201695_wp, -5.22702121018644_wp],&
      & shape(xyz))
   real(wp), parameter :: charge = -1.0_wp
   call new(self, sym, xyz, charge=charge)
end subroutine upuaa

end module mstore_upu23
