// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

import '//resources/cr_components/omnibox/realbox_dropdown.js';
import './strings.m.js';

import {startColorChangeUpdater} from '//resources/cr_components/color_change_listener/colors_css_updater.js';
import {AutocompleteResult, PageCallbackRouter} from '//resources/cr_components/omnibox/omnibox.mojom-webui.js';
import {RealboxBrowserProxy} from '//resources/cr_components/omnibox/realbox_browser_proxy.js';
import {RealboxDropdownElement} from '//resources/cr_components/omnibox/realbox_dropdown.js';
import {assert} from '//resources/js/assert_ts.js';
import {loadTimeData} from '//resources/js/load_time_data.js';
import {MetricsReporterImpl} from '//resources/js/metrics_reporter/metrics_reporter.js';
import {PolymerElement} from '//resources/polymer/v3_0/polymer/polymer_bundled.min.js';

import {getTemplate} from './app.html.js';

// 675px ~= 449px (--ntp-realbox-primary-side-min-width) * 1.5 + some margin.
const showSecondaryMatchesMediaQueryList =
    window.matchMedia('(min-width: 675px)');

export interface OmniboxPopupAppElement {
  $: {
    matches: RealboxDropdownElement,
  };
}

// Displays the autocomplete matches in the autocomplete result.
export class OmniboxPopupAppElement extends PolymerElement {
  static get is() {
    return 'omnibox-popup-app';
  }

  static get template() {
    return getTemplate();
  }

  static get properties() {
    return {
      /** Whether secondary matches can be shown. */
      canShowSecondaryMatches: {
        type: Boolean,
        value: () => showSecondaryMatchesMediaQueryList.matches &&
            loadTimeData.getBoolean('showSecondarySide'),
        reflectToAttribute: true,
      },

      hasSecondaryMatches: {
        reflectToAttribute: true,
        type: Boolean,
      },

      result_: Object,
    };
  }

  canShowSecondaryMatches: boolean;
  private result_: AutocompleteResult;

  private callbackRouter_: PageCallbackRouter;
  private omniboxAutocompleteResultChangedListenerId_: number|null = null;
  private selectMatchAtLineListenerId_: number|null = null;

  constructor() {
    super();
    this.callbackRouter_ = RealboxBrowserProxy.getInstance().callbackRouter;
    startColorChangeUpdater();
  }

  override connectedCallback() {
    super.connectedCallback();
    this.omniboxAutocompleteResultChangedListenerId_ =
        this.callbackRouter_.omniboxAutocompleteResultChanged.addListener(
            this.onOmniboxAutocompleteResultChanged_.bind(this));
    this.selectMatchAtLineListenerId_ =
        this.callbackRouter_.selectMatchAtLine.addListener(
            this.onSelectMatchAtLine_.bind(this));
    showSecondaryMatchesMediaQueryList.addEventListener(
        'change', this.onCanShowSecondaryMatchesChanged_.bind(this));
  }

  override disconnectedCallback() {
    super.disconnectedCallback();
    assert(this.omniboxAutocompleteResultChangedListenerId_);
    this.callbackRouter_.removeListener(
        this.omniboxAutocompleteResultChangedListenerId_);
    assert(this.selectMatchAtLineListenerId_);
    this.callbackRouter_.removeListener(this.selectMatchAtLineListenerId_);
    showSecondaryMatchesMediaQueryList.removeEventListener(
        'change', this.onCanShowSecondaryMatchesChanged_.bind(this));
  }

  private onCanShowSecondaryMatchesChanged_(e: MediaQueryListEvent) {
    this.canShowSecondaryMatches =
        e.matches && loadTimeData.getBoolean('showSecondarySide');
  }

  private onOmniboxAutocompleteResultChanged_(result: AutocompleteResult) {
    this.result_ = result;

    if (result.matches[0]?.allowedToBeDefaultMatch) {
      this.$.matches.selectFirst();
    } else if (this.$.matches.selectedMatchIndex >= result.matches.length) {
      this.$.matches.unselect();
    }
  }

  private onResultRepaint_() {
    const metricsReporter = MetricsReporterImpl.getInstance();
    metricsReporter.measure('ResultChanged')
        .then(
            duration => metricsReporter.umaReportTime(
                'WebUIOmnibox.ResultChangedToRepaintLatency.ToPaint', duration))
        .then(() => metricsReporter.clearMark('ResultChanged'))
        // Ignore silently if mark 'ResultChanged' is missing.
        .catch(() => {});
  }

  private onSelectMatchAtLine_(line: number) {
    this.$.matches.selectIndex(line);
  }
}

declare global {
  interface HTMLElementTagNameMap {
    'omnibox-popup-app': OmniboxPopupAppElement;
  }
}

customElements.define(OmniboxPopupAppElement.is, OmniboxPopupAppElement);
