#!/usr/bin/env python3

# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public License
# as published by the Free Software Foundation; either version 2.1
# of the License, or (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.

from sh4lt import ShType

# checking type parsing robustness with a GStreamer simple caps:
test_type_str = "video/x-raw, format=(string)I420, width=(int)320, height=(int)240, framerate=(fraction)30/1, multiview-mode=(string)mono, pixel-aspect-ratio=(fraction)1/1, interlace-mode=(string)progressive  , label=(string)\"\\(unexpected\\) \\= chars\\,\""
type = ShType.from_gst_caps(test_type_str, "a label", ShType.default_group())
assert type.media() == "video/x-raw"
assert len(type.get_property_keys()) == 8
assert type.get_prop("label").as_str() == "\\(unexpected\\) \\= chars\\,"
assert type.get_prop("pixel-aspect-ratio").as_str() == "1/1"
assert type.get_prop("width").as_int() == 320

# check building a Type and consistency when serializing/deserializing it
type = ShType.from_gst_caps("video/x-raw", "a label", ShType.default_group())
type.set_prop("format", "I420")
type.set_prop("height", 4096)
type.set_prop("interlace-mode", "progressive")
type.set_prop("label", "unexpected \\= comma\\,")
type.set_prop("multiview-mode", "mono")
type.set_prop("width", 4096)
type.set_custom_prop("framerate", "fraction", "30/1")
type.set_custom_prop("pixel-aspect-ratio", "fraction", "1/1")

# building an other Type from 'type' serialization and check it has the right name and properties
type2 = ShType.from_gst_caps(ShType.to_gst_caps(type), "a label", ShType.default_group())
assert type2.media() == "video/x-raw"
type2.get_prop("format").as_str() == "I420"
type2.get_prop("framerate").as_str() == "30/1"
type2.get_prop("height").as_int() == 4096
type2.get_prop("interlace-mode").as_str() == "progressive"
type2.get_prop("label").as_str() == "unexpected \\= comma\\,"
type2.get_prop("multiview-mode").as_str() == "mono"
type2.get_prop("pixel-aspect-ratio") == "1/1"
type2.get_prop("width").as_int() == 4096

# check serializations are the same
assert ShType.to_gst_caps(type) == ShType.to_gst_caps(type2)

# check int/float parsing
type = ShType.from_gst_caps("audio/x-raw, channels=2, rate=(int)44100, id=-435, id2=(int)-1, label=-label-, end-label-char=-, pi=(float)3.14, dur-ns=123456789", "a label", ShType.default_group())
assert type. get_prop("channels").as_int() == 2
assert type.get_prop("rate").as_int() == 44100
assert type.get_prop("id").as_int() == -435
assert type.get_prop("id2").as_int() == -1
assert type.get_prop("label").as_str() == "-label-"
assert type.get_prop("end-label-char").as_str() == "-"
assert type.get_prop('pi').as_float() == 3.14
assert type.get_prop("dur-ns").as_int() == 123456789
