/*
Copyright (c) 2019-2023,
Lawrence Livermore National Security, LLC;
See the top-level NOTICE for additional details. All rights reserved.
SPDX-License-Identifier: BSD-3-Clause
*/
#pragma once

#include "units.hpp"
#include <array>
#include <limits>
#include <utility>

namespace UNITS_NAMESPACE {

UNITS_CPP14_CONSTEXPR_OBJECT std::array<std::pair<unit, const char*>, 113>
    defined_unit_names_si{
        {{m, "m"},
         {m * m, "m^2"},
         {m * m * m, "m^3"},
         {kg, "kg"},
         {mol, "mol"},
         {unit(0.1, m), "dm"},  // don't want to use deci in most conversion
                                // contexts but for meter and liter it is fine
         {unit(0.1, L), "dL"},
         {A, "A"},
         {A * h, "Ah"},
         {V, "V"},
         {s, "s"},
         // this is so Gs doesn't get used which can cause issues
         {giga * s, "Bs"},  // B is for billion
         {cd, "cd"},
         {K, "K"},
         {N, "N"},
         {Pa, "Pa"},
         {J, "J"},
         {C, "C"},
         {F, "F"},
         // because GF is gram force not giga Farad which is a ridiculous unit
         // otherwise generates confusion
         {giga * F, "(1000MF)"},

         // some of these are just dumb units but here to avoid round trip
         // errors
         {unit_cast(precise::ronna) * T, "(1000YT)"},
         {unit_cast(precise::quetta) * T, "(1000000YT)"},
         {S, "S"},
         {Wb, "Wb"},
         {T, "T"},
         {H, "H"},
         // deal with pico henry which is interpreted as acidity (pH)
         {pico * H, "(A^-2*pJ)"},
         {lm, "lm"},
         {lx, "lux"},
         // remove Bq since it is often confusing in other contexts 1/s
         // {Bq, "Bq"},
         {unit(2.58e-4, C / kg), "R"},

         {unit_cast(precise::cgs::curie), "Ci"},
         {(mega * m).pow(3),
          "ZL"},  // another one of those units that can be confused
         {bar, "bar"},
         {min, "min"},
         {ms, "ms"},
         {h, "h"},
         {unit_cast(precise::time::day), "day"},
         {unit_cast(precise::time::week), "week"},
         {unit_cast(precise::time::yr), "yr"},
         {deg, "deg"},
         {rad, "rad"},
         {degC, u8"\u00B0C"},
         {degC.pow(2), "degC^2"},
         {cm, "cm"},
         {km, "km"},
         {km * km, "km^2"},
         {mm, "mm"},
         {nm, "nm"},
         {pu * m, "strain"},
         {unit_cast(precise::distance::ly), "ly"},
         {unit_cast(precise::distance::au), "au"},
         {percent, "%"},
         {unit_cast(precise::special::ASD), "ASD"},
         {unit_cast(precise::special::rootHertz), "rootHertz"},
         {currency, "$"},
         {count, "count"},
         {ratio, ""},
         {error, "ERROR"},
         {defunit, "defunit"},
         {iflag, "flag"},
         {eflag, "eflag"},
         {pu, "pu"},
         {Gy, "Gy"},
         {Sv, "Sv"},
         {Hz, "Hz"},
         {kilo * Hz, "kHz"},
         {rpm, "rpm"},
         {kat, "kat"},
         {sr, "sr"},
         {W, "W"},
         {VAR, "VAR"},
         {MVAR, "MVAR"},
         {MW, "MW"},
         {kW, "kW"},
         {mW, "mW"},
         {puMW, "puMW"},
         {puMW / mega, "puW"},
         {puV, "puV"},
         {puA, "puA"},
         {mA, "mA"},
         {kV, "kV"},
         {unit_cast(precise::area::are), "are"},
         {unit_cast(precise::area::hectare), "hectare"},
         {unit_cast(precise::area::barn), "barn"},
         {pu * ohm, "puOhm"},
         {puHz, "puHz"},
         {unit_cast(precise::energy::eV), "eV"},
         {unit_cast(precise::pressure::atm), "atm"},
         {unit_cast(precise::pressure::mmHg), "mmHg"},
         {unit_cast(precise::pressure::mmH2O), "mmH2O"},
         {unit_cast(precise::laboratory::arbU), "arb. unit"},
         {unit_cast(precise::laboratory::IU), "[IU]"},
         {kWh, "kWh"},
         {MWh, "MWh"},
         {unit_cast(precise::other::MegaBuck), "M$"},
         {unit_cast(precise::other::GigaBuck), "B$"},
         {L, "L"},
         {unit_cast(precise::mL), "mL"},
         {unit_cast(precise::micro * precise::L), "uL"},
         {tonne, "t"},  // metric ton
         {unit_cast(precise::mass::u), "u"},
         {kB, "kB"},
         {MB, "MB"},
         {GB, "GB"},
         {unit_cast(precise::data::KiB), "KiB"},
         {unit_cast(precise::data::MiB), "MiB"},
         {unit_cast(precise::distance::angstrom), u8"\u00C5"},
         {g, "g"},
         {mg, "mg"},
         {unit_cast(precise::energy::eV / (constants::c.as_unit().pow(2))),
          "(eV*[c]^-2)"},
         {unit_cast(precise::data::GiB), "GiB"},
         {unit_cast(precise::other::ppm), "ppm"},
         {unit_cast(precise::other::ppb), "ppb"}}};

UNITS_CPP14_CONSTEXPR_OBJECT std::array<std::pair<unit, const char*>, 56>
    defined_unit_names_customary{{
        {in, "in"},
        {unit_cast(precise::in.pow(2)), "in^2"},
        {unit_cast(precise::in.pow(3)), "in^3"},
        {ft, "ft"},
        {unit_cast(precise::imp::foot), "ft_br"},
        {unit_cast(precise::imp::inch), "in_br"},
        {unit_cast(precise::imp::yard), "yd_br"},
        {unit_cast(precise::imp::rod), "rd_br"},
        {unit_cast(precise::imp::mile), "mi_br"},
        {unit_cast(precise::imp::pace), "pc_br"},
        {unit_cast(precise::imp::link), "lk_br"},
        {unit_cast(precise::imp::chain), "ch_br"},
        {unit_cast(precise::imp::nautical_mile), "nmi_br"},
        {unit_cast(precise::imp::knot), "kn_br"},
        {unit_cast(precise::nautical::knot), "knot"},
        {unit_cast(precise::ft.pow(2)), "ft^2"},
        {unit_cast(precise::ft.pow(3)), "ft^3"},
        {degF, u8"\u00B0F"},
        {yd, "yd"},
        {unit_cast(precise::us::rod), "rd"},
        {unit_cast(precise::yd.pow(2)), "yd^2"},
        {unit_cast(precise::yd.pow(3)), "yd^3"},
        {unit_cast(precise::time::syr), "syr"},
        {unit_cast(precise::time::ag), "a_g"},
        {unit_cast(precise::time::at), "a_t"},
        {unit_cast(precise::time::aj), "a_j"},
        {unit_cast(precise::angle::grad), "grad"},
        {mile, "mi"},
        {unit_cast(precise::mile.pow(2)), "mi^2"},
        {acre, "acre"},
        {unit_cast(precise::energy::therm_ec), "therm"},
        {unit_cast(precise::energy::tonc), "tonc"},
        {hp, "hp"},
        {mph, "mph"},
        {kcal, "kcal"},
        {btu, "btu"},
        {unit_cast(precise::other::CFM), "CFM"},
        {unit_cast(precise::pressure::psi), "psi"},
        {unit_cast(precise::pressure::psig), "psig"},
        {unit_cast(precise::pressure::inHg), "inHg"},
        {unit_cast(precise::pressure::inH2O), "inH2O"},
        {unit_cast(precise::pressure::torr), "torr"},
        {unit_cast(precise::energy::EER), "EER"},
        {unit_cast(precise::energy::quad), "quad"},
        {gal, "gal"},
        {unit_cast(precise::us::barrel), "bbl"},
        {lb, "lb"},
        {ton, "ton"},
        {unit_cast(precise::us::dry::bushel), "bu"},
        {unit_cast(precise::us::floz), "floz"},
        {oz, "oz"},
        {unit_cast(precise::us::cup), "cup"},
        {unit_cast(precise::us::tsp), "tsp"},
        {unit_cast(precise::us::tbsp), "tbsp"},
        {unit_cast(precise::us::quart), "qt"},
        {unit_cast(precise::lbf), "lbf"},
    }};

/// definitions for the default units for specific types of measurmeents
UNITS_CPP14_CONSTEXPR_OBJECT std::array<
    std::pair<const char*, precise_unit>,
    1155>
    defined_unit_strings_si{{
        {"", precise::defunit},
        {"[]", precise::defunit},
        {"def", precise::defunit},
        {"default", precise::defunit},
        {"defunit", precise::defunit},
        {"*", precise::defunit},
        {"**", precise::defunit},
        {"***", precise::defunit},
        {"****", precise::defunit},
        {"*****", precise::defunit},
        {"******", precise::defunit},
        {"^^^", precise::defunit},
        {"1", precise::one},
        {"one", precise::one},
        {"inf", precise::infinite},
        {".inf", precise::infinite},
        {"INF", precise::infinite},
        {"infinity", precise::infinite},
        {"INFINITY", precise::infinite},
        {"-inf", precise::neginfinite},
        {"-.inf", precise::neginfinite},
        {"-.Inf", precise::neginfinite},
        {"-*inf",
         precise::neginfinite},  // this is to handle internal string cleaning
        {"-*Inf", precise::neginfinite},
        {"-INF", precise::neginfinite},
        {"-infinity", precise::neginfinite},
        {"-INFINITY", precise::neginfinite},
        {"infinite", precise::infinite},
        {u8"\u221e", precise::infinite},
        {u8"-\u221e", precise::neginfinite},
        {"nan", precise::nan},
        {"NaN", precise::nan},
        {"NaN%", precise::nan},
        {"NAN", precise::nan},
        {".NAN", precise::nan},
        {"NaNQ",
         precise_unit(
             detail::unit_data(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0),
             std::numeric_limits<double>::quiet_NaN())},
        {"NaNS", precise::nan},
        {"qNaN",
         precise_unit(
             detail::unit_data(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0),
             std::numeric_limits<double>::quiet_NaN())},
        {"sNaN", precise::nan},
        {"1.#SNAN", precise::nan},
        {"#SNAN", precise::nan},
        {"1.#QNAN",
         precise_unit(
             detail::unit_data(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0),
             std::numeric_limits<double>::quiet_NaN())},
        {"#QNAN",
         precise_unit(
             detail::unit_data(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0),
             std::numeric_limits<double>::quiet_NaN())},
        {"1.#IND", precise::nan},
        {"#IND", precise::nan},
        {"0.1", precise_unit(0.1, precise::one)},
        {".1", precise_unit(0.1, precise::one)},
        {"deci", precise_unit(0.1, precise::one)},
        {"0.01", precise_unit(0.01, precise::one)},
        {".01", precise_unit(0.01, precise::one)},
        {"centi", precise_unit(0.01, precise::one)},
        {"0.001", precise::milli},
        {".001", precise::milli},
        {"milli", precise::milli},
        {"1e-3", precise::milli},
        {"1e-6", precise::micro},
        {"micro", precise::micro},
        {"1e-9", precise::nano},
        {"1e-12", precise::pico},
        {"1e-15", precise::femto},
        {"1e-18", precise::atto},
        {"nano", precise::nano},
        {"pico", precise::pico},
        {"femto", precise::femto},
        {"atto", precise::atto},
        {"yocto", precise::yocto},
        {"zepto", precise::zepto},
        {"ronto", precise::ronto},
        {"quecto", precise::quecto},
        {"10", precise_unit(10.0, precise::one)},
        {"ten", precise_unit(10.0, precise::one)},
        {"two", precise_unit(2.0, precise::one)},
        {"twenty", precise_unit(20.0, precise::one)},
        {"twelve", precise_unit(12.0, precise::one)},
        {"eight", precise_unit(8.0, precise::one)},
        {"100", precise_unit(100, precise::one)},
        {"hundred", precise_unit(100, precise::one)},
        {"fifty", precise_unit(50, precise::one)},
        {"centum", precise_unit(100, precise::one)},
        {"1000", precise::kilo},
        {"thousand", precise::kilo},
        {"1000000", precise::mega},
        {"million", precise::mega},
        {"1000000000", precise::giga},
        {"billion", precise::giga},
        {"trillion", precise::tera},
        {"quadrillion", precise::peta},
        {"1e3", precise::kilo},
        {"1e6", precise::mega},
        {"1e9", precise::giga},
        {"1e12", precise::tera},
        {"1e15", precise::peta},
        {"1e18", precise::exa},
        {"kilo", precise::kilo},
        {"k", precise::kilo},
        {"mega", precise::mega},
        {"giga", precise::giga},
        {"tera", precise::tera},
        {"peta", precise::peta},
        {"exa", precise::exa},
        {"yotta", precise::yotta},
        {"zetta", precise::zetta},
        {"ronna", precise::ronna},
        {"quetta", precise::quetta},
        {"%", precise::percent},
        {"percent", precise::percent},
        {"percentage", precise::percent},
        {"permille", precise::milli},
        {u8"\u2030", precise::milli},  // per mille symbol
        {"bp", precise_unit(0.1, precise::milli)},
        {"basispoint", precise_unit(0.1, precise::milli)},
        {u8"\u2031",
         precise_unit(0.1, precise::milli)},  // per ten thousand symbol
        {"pct", precise::percent},
        {"pi", precise_unit(constants::pi, one)},
        {"PI", precise_unit(constants::pi, one)},
        {"[pi]", precise_unit(constants::pi, one)},
        {"thenumberpi", precise_unit(constants::pi, one)},
        {"[PI]", precise_unit(constants::pi, one)},
        {"Pi", precise_unit(constants::pi, one)},
        {u8"\u03C0", precise_unit(constants::pi, one)},
        {u8"\U0001D70B", precise_unit(constants::pi, one)},
        {u8"\U0001D745", precise_unit(constants::pi, one)},
        {u8"\U0001D6D1", precise_unit(constants::pi, one)},
        {u8"\u03B5", precise::pu* precise::m},  // strain
        {u8"\u0190", precise::pu* precise::m},  // strain
        {"strain", precise::pu* precise::m},  // strain
        {"m", precise::m},
        {"Sm", precise::m},  // standard meter used in oil and gas usually Sm^3
        {"meter", precise::m},
        {"squaremeter",
         precise::m.pow(2)},  // to simplify some things later in the chain
        {"cubicmeter", precise::m.pow(3)},
        {"micron", precise::micro* precise::m},
        {"fermi", precise::femto* precise::m},
        {"xunit", precise::distance::xu},
        {"xu", precise::distance::xu},
        {"X", precise::distance::xu},
        {"cc", precise::cm.pow(3)},
        {"cubiccentimeter", precise::cm.pow(3)},
        {"m/s^2", precise::m / precise::s.pow(2)},
        {"kg/m^3", precise::kg / precise::m.pow(3)},
        {"kg", precise::kg},
        {"KG", precise::kg},
        {"kilogram", precise::kg},
        {"mol", precise::mol},
        {"einstein", precise::mol},  // technically of photons
        {"gmol", precise::mol},
        {"lbmol", precise_unit(424.0, precise::mol)},
        {"atom", constants::Na.as_unit().inv()},
        {"avogadroconstant", constants::Na.as_unit()},
        {"molecule", constants::Na.as_unit().inv()},
        {"molec", constants::Na.as_unit().inv()},
        {"nucleon", constants::Na.as_unit().inv()},
        {"nuc", constants::Na.as_unit().inv()},
        {"MOL", precise::mol},
        {"mOL", precise::mol},
        {"mole", precise::mol},
        {"M", precise::laboratory::molarity},
        {"molar", precise::laboratory::molarity},
        {"Molar", precise::laboratory::molarity},
        {"eq", precise::mol},
        {"equivalent", precise::mol},
        {"EQ", precise::mol},
        {"osm", precise_unit(1.0, precise::mol, commodities::particles)},
        {"osmole", precise_unit(1.0, precise::mol, commodities::particles)},
        {"osmol", precise_unit(1.0, precise::mol, commodities::particles)},
        {"OSM", precise_unit(1.0, precise::mol, commodities::particles)},
        {"g%", precise_unit(10.0, precise::g / precise::L)},
        {"gram%", precise_unit(10.0, precise::g / precise::L)},
        {"grampercent", precise_unit(10.0, precise::g / precise::L)},
        {"G%", precise_unit(10.0, precise::g / precise::L)},
        {"U", precise::laboratory::enzyme_unit},
        // this may not be best but it doesn't actually conflict with anything
        // else right now
        {"units", precise::laboratory::enzyme_unit},
        {"enzymeunit", precise::laboratory::enzyme_unit},
        {"internationalunitofenzymeactivity", precise::laboratory::enzyme_unit},
        {"eu", precise::laboratory::enzyme_unit},
        {"A", precise::A},
        {"amp", precise::A},
        {"amps", precise::A},
        {"ampturn", precise::A* precise::count},
        {"At", precise::A* precise::count},
        {"V", precise::V},
        {"volt", precise::V},
        {"W", precise::W},
        {"W/m^2", precise::W / precise::m.pow(2)},
        {"watt", precise::W},
        {"kW", precise::electrical::kW},
        {"kilowatt", precise::electrical::kW},
        {"MW", precise::MW},
        {"megawatt", precise::MW},
        {"MVA", precise::electrical::MVAR},
        {"mvar", precise::electrical::MVAR},
        {"MVAR", precise::electrical::MVAR},
        {"VA", precise::electrical::VAR},
        {"voltampreactive", precise::electrical::VAR},
        {"VAR", precise::electrical::VAR},
        {"var", precise::electrical::VAR},
        {"s", precise::s},
        {"sec", precise::s},
        {"MAS", precise::mega* precise::s},
        {"mAS", precise::mega* precise::s},
        {"second", precise::s},
        {"second-time", precise::s},
        {"shake", precise_unit(10.0, precise::ns)},
        {"jiffy", precise_unit(0.01, precise::s)},  // use the computer science
                                                    // definition for playback
                                                    // and clock rate
        {"cd", precise::cd},
        {"mcd",
         precise::milli* precise::cd},  // prefer milli candela to micro-day
        {"CD", precise::cd},
        {"candela", precise::cd},
        {"metercandle", precise::lumen / precise::m.pow(2)},
        {"centimetercandle", precise::lumen / precise::cm.pow(2)},
        {"K", precise::K},
        {"kelvin", precise::K},
        {"kelvins", precise::K},
        {"degKelvin", precise::K},
        {"degsKelvin", precise::K},
        {"degkelvin", precise::K},
        {"degskelvin", precise::K},
        {"degK", precise::K},
        {"degsK", precise::K},
        {"N", precise::N},
        {"Ns", precise::N* precise::s},  // this would not pass through to the
                                         // separation functions
        {"Nm", precise::N* precise::m},  // this would not pass through to the
                                         // separation functions
        {"As", precise::A* precise::s},  // this would not pass through to the
                                         // separation functions
        {"Ah", precise::A* precise::hr},  // this would not pass through to the
                                          // separation functions
        {"Ahr", precise::A* precise::hr},  // this would not pass through to
                                           // the separation functions
        {"newton", precise::N},
        {"Pa", precise::Pa},
        {"pa", precise::Pa},
        {"pascal", precise::Pa},
        {"PAL", precise::Pa},
        {"pAL", precise::Pa},
        {"J", precise::J},
        {"joule", precise::J},
        {"Joule", precise::J},
        {"Joules", precise::J},
        {"C", precise::C},
        {"coulomb", precise::C},
        {"faraday", precise::other::faraday},
        {"farady", precise::other::faraday},
        {"F", precise::F},
        {"farad", precise::F},
        {"abF", precise::cgs::abFarad},
        {"abfarad", precise::cgs::abFarad},
        {"stF", precise::cgs::statFarad},
        {"statfarad", precise::cgs::statFarad},
        {"OHM", precise::ohm},
        {"ohm", precise::ohm},
        {"Ohm", precise::ohm},
        {"kilohm", precise::kilo* precise::ohm},  // special case allowed by SI
        {"megohm", precise::mega* precise::ohm},  // special case allowed by SI
        {u8"\u03A9", precise::ohm},  // Greek Omega
        {u8"\u2126", precise::ohm},  // Unicode Ohm symbol
        {"abOhm", precise::cgs::abOhm},
        {"abohm", precise::cgs::abOhm},
        {"statohm", precise::cgs::statOhm},
        {"statOhm", precise::cgs::statOhm},
        {"S", precise::S},
        {"siemen", precise::S},
        {"siemens", precise::S},
        {"SIE", precise::S},
        {"mho", precise::S},
        {"gemmho", precise_unit(1e-6, precise::S)},
        {"MHO", precise::S},
        {"mHO", precise::S},
        {u8"\u01B1", precise::S},
        {"absiemen", precise_unit(1e9, precise::S)},
        {"abmho", precise_unit(1e9, precise::S)},
        {"statmho", precise_unit(1.0 / 8.987551787e11, S)},
        {"Wb", precise::Wb},
        {"weber", precise::Wb},
        {"T", precise::T},
        {"tesla", precise::T},
        {"H", precise::H},
        {"henry", precise::H},
        {"henries", precise::H},
        {"abhenry", precise::cgs::abHenry},
        {"abH", precise::cgs::abHenry},
        {"stathenry", precise::cgs::statHenry},
        {"stH", precise::cgs::statHenry},
        {"statH", precise::cgs::statHenry},
        {"lm", precise::lm},
        {"LM", precise::lm},
        {"lumen", precise::lm},
        {"lux", precise::lx},
        {"luxes", precise::lx},
        {"nox", precise::milli* precise::lx},
        {"lx", precise::lx},
        {"LX", precise::lx},
        {"Bq", precise::Bq},
        {"BQ", precise::Bq},
        {"becquerel", precise::Bq},
        {"rutherford", precise_unit(1e6, precise::Bq)},
        {"activity", precise::Bq},
        {"curie", precise::cgs::curie},
        {"gauss", precise::cgs::gauss},
        {"G", precise::cgs::gauss},
        {"Gs", precise::cgs::gauss},
        {"Ci", precise::cgs::curie},
        {"CI", precise::cgs::curie},
        {"Gal", precise::cgs::gal},
        {"Gals", precise::cgs::gal},
        {"galileos", precise::cgs::gal},
        {"jansky",
         precise_unit(
             1e-26,
             precise::W / precise::m / precise::m / precise::Hz)},
        {"Jy",
         precise_unit(
             1e-26,
             precise::W / precise::m / precise::m / precise::Hz)},
        {"Mm^3", precise::kilo* precise::energy::scm},  // millum cubic meters
        {"Nm^3", precise::energy::ncm},  // normal cubic meter
        {"Sm^3", precise::energy::scm},  // standard cubic meter
        {"MMm^3", precise::mega* precise::energy::scm},  // million cubic meters
        {"bm^3", precise::giga* precise::energy::scm},  // billion cubic meters
        {"tm^3", precise::tera* precise::energy::scm},  // trillion cubic meters
        {"gwp", precise::climate::gwp},  // global warming potential
        {"gtp", precise::climate::gtp},  // global temperature potential
        {"min", precise::min},
        {"mins", precise::min},
        {"mIN", precise::min},
        {"minute", precise::min},
        {"ms", precise::ms},
        {"millisecond", precise::ms},
        {"hr", precise::hr},
        {"HR", precise::hr},
        {"h", precise::hr},
        {"hour", precise::hr},
        {"day", precise::time::day},
        {"dy", precise::time::day},
        {"D", precise::time::day},
        {"d", precise::time::day},
        {"week", precise::time::week},
        {"weekly", precise::one / precise::time::week},
        {"wk", precise::time::week},
        {"WK", precise::time::week},
        {"y", precise::time::year},
        {"YR", precise::time::yr},  // this one gets 365 days exactly
        {"yr", precise::time::yr},  // this one gets 365 days exactly
        {"a", precise::area::are},  // SI symbol is are
        {"year", precise::time::year},  // year SI Definition 365 days
        {"yearly", precise::time::year.inv()},  // year
        {"annum", precise::time::year},  // year
        {"ANN", precise::time::aj},  // year
        {"decade", precise::ten* precise::time::aj},  // year
        {"century", precise::hundred* precise::time::aj},  // year
        {"centuries", precise::hundred* precise::time::aj},  // year
        {"millennia", precise::kilo* precise::time::ag},  // year
        {"millennium", precise::kilo* precise::time::ag},  // year
        {"syr", precise::time::syr},  // sidereal year
        {"year_sdr", precise::time::syr},  // sidereal year
        {"yr_sdr", precise::time::syr},  // sidereal year
        {"month_sdr",
         precise_unit(1.0 / 12.0, precise::time::syr)},  // sidereal month
        {"mo_sdr",
         precise_unit(1.0 / 12.0, precise::time::syr)},  // sidereal month
        {"sday", precise::time::sday},  // sidereal day
        {"day_sdr", precise::time::sday},  // sidereal day
        {"dy_sdr", precise::time::sday},  // sidereal day
        {"d_sdr", precise::time::sday},  // sidereal day
        {"hour_sdr",
         precise_unit(1.0 / 24.0, precise::time::sday)},  // sidereal hour
        {"minute_sdr",
         precise_unit(1.0 / 24.0 / 60.0, precise::time::sday)},  // sidereal
                                                                 // minute
        {"second_sdr",
         precise_unit(
             1.0 / 24.0 / 60.0 / 60.0,
             precise::time::sday)},  // sidereal
                                     // second
        {"hr_sdr",
         precise_unit(1.0 / 24.0, precise::time::sday)},  // sidereal hour
        {"min_sdr",
         precise_unit(1.0 / 24.0 / 60.0, precise::time::sday)},  // sidereal
                                                                 // minute
        {"sec_sdr",
         precise_unit(
             1.0 / 24.0 / 60.0 / 60.0,
             precise::time::sday)},  // sidereal
                                     // second
        {"a_t", precise::time::at},  // tropical year
        {"year_t", precise::time::at},  // tropical year
        {"month_t",
         precise_unit(1.0 / 12.0, precise::time::at)},  // tropical month
        {"mo_t",
         precise_unit(1.0 / 12.0, precise::time::at)},  // tropical month
        {"solaryear", precise::time::at},  // year
        {"ANN_T", precise::time::at},  // year
        {"a_j", precise::time::aj},  // year
        {"meanyear_j", precise::time::aj},  // year
        {"meanyr_j", precise::time::aj},  // year
        {"year_j", precise::time::aj},  // year
        {"yr_j", precise::time::aj},  // year
        {"ANN_J", precise::time::aj},  // year
        {"year(leap)", precise_unit(366.0, precise::time::day)},  // year
        {"commonyear", precise_unit(365.0, precise::time::day)},  // year
        {"leapyear", precise_unit(366.0, precise::time::day)},  // year
        {"yearcommon", precise_unit(365.0, precise::time::day)},  // year
        {"meanyear_g", precise::time::ag},  // year
        {"meanyr_g", precise::time::ag},  // year
        {"a_g", precise::time::ag},  // year
        {"year_g", precise::time::ag},  // year
        {"yr_g", precise::time::ag},  // year
        {"ANN_G", precise::time::ag},  // year
        {"mo", precise::time::mog},  // gregorian month
        {"month", precise::time::mog},  // gregorian month
        {"monthly", precise::time::mog.inv()},  // gregorian month
        {"MO", precise::time::mog},  // gregorian month
        {"mO", precise::time::mog},  // gregorian month
        {"mos", precise::time::mos},  // synodal month
        {"mo_s", precise::time::mos},  // synodal month
        {"mO_S", precise::time::mos},  // synodal month
        {"synodalmonth", precise::time::mos},  // synodal month
        {"month_s", precise::time::mos},  // synodal month
        {"lunarmonth", precise::time::mos},  // synodal month
        {"lunarcycle", precise::time::mos},  // synodal month
        {"moon", precise::time::mos},  // synodal month
        {"mo_j", precise::time::moj},  //
        {"month_j", precise::time::moj},  //
        {"mO_J", precise::time::moj},  //
        {"meanmonth_j", precise::time::moj},  //
        {"mo_g", precise::time::mog},  //
        {"mog", precise::time::mog},  //
        {"month_g", precise::time::mog},  //
        {"mO_G", precise::time::mog},  //
        {"meanmonth_g", precise::time::mog},  //
        {"eon", precise_unit(1e9, precise::time::syr)},
        {"workyear", precise_unit(2056, precise::hr)},
        {"workmonth", precise_unit(2056.0 / 12.0, precise::hr)},
        {"workweek", precise_unit(40.0, precise::hr)},
        {"workday", precise_unit(8.0, precise::hr)},
        {"arcdeg", precise::deg},
        {"deg-planeangle", precise::deg},
        {"deg(planeangle)", precise::deg},
        {"angulardeg", precise::deg},
        {"deg", precise::deg},
        {"DEG", precise::deg},
        {"degE", precise::deg* precise::direction::east},
        {"east", precise::direction::east},
        {"degW", precise::deg* precise::direction::west},
        {"west", precise::direction::west},
        {"degS", precise::deg* precise::direction::south},
        {"south", precise::direction::south},
        {"degN", precise::deg* precise::direction::north},
        {"north", precise::direction::north},
        {"degT", precise::deg* precise::direction::north},
        {"true", precise::direction::north},
        {"o", precise::deg},
        {u8"\u00B0", precise::deg},  // unicode degree symbol
        {"\xB0", precise::deg},  // latin-1 degree
        {u8"\u00B0(s)", precise::deg},  // unicode degree symbol
        {"\xB0(s)", precise::deg},  // latin-1 degree
        {"arcminute", precise::angle::arcmin},
        {"arcmin", precise::angle::arcmin},
        {"amin", precise::angle::arcmin},
        {"MOA", precise::angle::arcmin},
        {"angularminute", precise::angle::arcmin},
        {"'", precise::angle::arcmin},
        {"`", precise::angle::arcmin},
        {u8"\u2032", precise::angle::arcmin},  // single prime
        {"arcsecond", precise::angle::arcsec},
        {"''", precise::angle::arcsec},
        {"``", precise::angle::arcsec},
        {"arcsec", precise::angle::arcsec},
        {"asec", precise::angle::arcsec},
        {"angularsecond", precise::angle::arcsec},
        {"\"", precise::angle::arcsec},
        {u8"\u2033", precise::angle::arcsec},  // double prime
        {"mas", precise_unit(0.001, precise::angle::arcsec)},  // milliarcsec
        {"uas", precise_unit(0.000001, precise::angle::arcsec)},  // microarcsec
        {"rad", precise::rad},
        {"radian", precise::rad},
        {"circ", precise_unit(constants::tau, precise::rad)},
        {"CIRC", precise_unit(constants::tau, precise::rad)},
        {"quadrant", precise_unit(90.0, precise::deg)},
        {"circle", precise_unit(constants::tau, precise::rad)},
        {"cycle", precise_unit(constants::tau, precise::rad)},
        {"rotation", precise_unit(constants::tau, precise::rad)},
        {"turn", precise_unit(constants::tau, precise::rad)},
        {"circle-planeangle", precise_unit(constants::tau, precise::rad)},
        {"sph", precise_unit(4.0 * constants::pi, precise::sr)},
        {"spere-solidangle", precise_unit(4.0 * constants::pi, precise::sr)},
        {"spere", precise_unit(4.0 * constants::pi, precise::sr)},
        {"speres", precise_unit(4.0 * constants::pi, precise::sr)},
        {"Spere", precise_unit(4.0 * constants::pi, precise::sr)},
        {"SPH", precise_unit(4.0 * constants::pi, precise::sr)},
        {"\xB0"
         "C",
         precise::degC},
        {u8"\u00B0C", precise::degC},
        {u8"\u2103", precise::degC},  // direct unicode symbol
        {"\xB0"
         "K",
         precise::K},
        {u8"\u00B0K", precise::K},

        {"degC", precise::degC},
        {"oC", precise::degC},
        {"Cel", precise::degC},
        {"CEL", precise::degC},
        {"K@273.15", precise::degC},
        {"celsius", precise::degC},
        {"degCelsius", precise::degC},
        {"degsC", precise::degC},
        {"cm", precise::cm},
        {"centimeter", precise::cm},
        {"km", precise::km},
        {"kilometer", precise::km},
        {"mm", precise::mm},
        {"millimeter", precise::mm},
        {"nm", precise::nm},
        {"nanometer", precise::nm},
        {"ly", precise::distance::ly},
        {"[LY]", precise::distance::ly},
        {"lightyear", precise::distance::ly},
        {"light-year", precise::distance::ly},
        {"pc", precise::distance::parsec},
        {"parsec", precise::distance::parsec},
        {"PRS", precise::distance::parsec},
        {"pRS", precise::distance::parsec},
        {"[c]", constants::c.as_unit()},
        {"c", constants::c.as_unit()},
        {"[C]", constants::c.as_unit()},
        {"speedoflight", constants::c.as_unit()},
        {"speedoflightinvacuum", constants::c.as_unit()},
        {"light", constants::c.as_unit()},
        {"[h]", constants::h.as_unit()},
        {"[hbar]", constants::hbar.as_unit()},
        {"hbar", constants::hbar.as_unit()},
        {"[H]", constants::h.as_unit()},
        {u8"\u210E", constants::h.as_unit()},
        {u8"\u210F",
         precise_unit(1.0 / constants::tau, constants::h.as_unit())},
        {"[k]", constants::k.as_unit()},
        {"[K]", constants::k.as_unit()},
        {"eps_0", constants::eps0.as_unit()},
        {"[eps_0]", constants::eps0.as_unit()},
        {"vacuumpermittivity", constants::eps0.as_unit()},
        {"[EPS_0]", constants::eps0.as_unit()},
        {u8"\u03B5"
         "0",
         constants::eps0.as_unit()},
        {u8"\u03B5\u2080", constants::eps0.as_unit()},
        {"mu_0", constants::mu0.as_unit()},
        {"[MU_0]", constants::mu0.as_unit()},
        {"[mu0]", constants::mu0.as_unit()},
        {"[e]", constants::e.as_unit()},
        {"e", constants::e.as_unit()},
        {"[E]", constants::e.as_unit()},
        {"elementarycharge", constants::e.as_unit()},
        {"[G]", constants::G.as_unit()},
        {"[GC]", constants::G.as_unit()},
        {"[g]", constants::g0.as_unit()},
        {"[g0]", constants::g0.as_unit()},
        {"standardgravity", constants::g0.as_unit()},
        {"standardfreefall", constants::g0.as_unit()},
        {"freefall", constants::g0.as_unit()},
        {"standardaccelerationoffreefall", constants::g0.as_unit()},
        {"accelerationofgravity", constants::g0.as_unit()},
        {"m_e", constants::me.as_unit()},
        {"[me]", constants::me.as_unit()},
        {"electronmass", constants::me.as_unit()},
        {"[M_E]", constants::me.as_unit()},
        {"m_p", constants::mp.as_unit()},
        {"[M_P]", constants::mp.as_unit()},
        {"[mp]", constants::mp.as_unit()},
        {"protonmass", constants::mp.as_unit()},
        {"m_n", constants::mn.as_unit()},
        {"[M_N]", constants::mn.as_unit()},
        {"[mn]", constants::mn.as_unit()},
        {"neutronmass", constants::mn.as_unit()},
        {"planckmass", constants::planck::mass.as_unit()},
        {"plancklength", constants::planck::length.as_unit()},
        {"plancktime", constants::planck::time.as_unit()},
        {"planckcharge", constants::planck::charge.as_unit()},
        {"plancktemperature", constants::planck::temperature.as_unit()},
        {"planckarea", constants::planck::length.as_unit().pow(2)},
        {"planckvolume", constants::planck::length.as_unit().pow(3)},
        {"planckenergy", precise_unit(1.9561e9, precise::J)},
        {"[fCs]", constants::fCs.as_unit()},
        {"[alpha]", constants::alpha.as_unit()},
        {"[mu]", constants::mu.as_unit()},
        {"[Na]", constants::Na.as_unit()},
        {"[Kcd]", constants::Kcd.as_unit()},
        {"[R]", constants::R.as_unit()},
        {"[s]", constants::s.as_unit()},
        {"[H0]", constants::H0.as_unit()},
        {"[a0]", constants::a0.as_unit()},
        {"[F]", constants::F.as_unit()},
        {"[Kj]", constants::Kj.as_unit()},
        {"[phi0]", constants::phi0.as_unit()},
        {"[Rk]", constants::Rk.as_unit()},
        {"[Rinf]", constants::Rinf.as_unit()},
        {"[planck::length]", constants::planck::length.as_unit()},
        {"[planck::mass]", constants::planck::mass.as_unit()},
        {"[planck::time]", constants::planck::time.as_unit()},
        {"[planck::charge]", constants::planck::charge.as_unit()},
        {"[planck::temperature]", constants::planck::temperature.as_unit()},
        {"[atomic::mass]", constants::atomic::mass.as_unit()},
        {"[atomic::length]", constants::atomic::length.as_unit()},
        {"[atomic::time]", constants::atomic::time.as_unit()},
        {"[atomic::charge]", constants::atomic::charge.as_unit()},
        {"[atomic::energy]", constants::atomic::energy.as_unit()},
        {"[atomic::action]", constants::atomic::action.as_unit()},
        {"[planck::area]", constants::planck::length.as_unit().pow(2)},
        {"[planck::volume]", constants::planck::length.as_unit().pow(3)},
        {"[planck::energy]", precise_unit(1.9561e9, precise::J)},
        {"au", precise::distance::au},
        {"AU", precise::distance::au},
        {"ASU", precise::distance::au},
        {"astronomicalunit", precise::distance::au},
        {"astronomicunit", precise::distance::au},
        {"kph", precise::km / precise::hr},
        {"$", precise::currency},
        {"dollar", precise::currency},
        {"euro", precise::currency},
        {"yen", precise::currency},
        {"ruble", precise::currency},
        {"currency", precise::currency},
        {u8"\u00A2", precise_unit(0.01, precise::currency)},  // cent symbol
        {"\xA2", precise_unit(0.01, precise::currency)},  // cent symbol latin-1
        {u8"\u00A3", precise::currency},  // pound sign
        {"\xA3", precise::currency},  // pound sign latin-1
        {u8"\u00A4", precise::currency},  // currency sign
        {"\xA4", precise::currency},  // currency sign latin-1
        {u8"\u00A5", precise::currency},  // Yen sign
        {"\xA5", precise::currency},  // Yen sign latin-1
        {u8"\u0080", precise::currency},  // Euro sign
        {"\x80", precise::currency},  // Euro sign extended ascii
        {"count", precise::count},
        {"unit", precise::count},
        {"pair", precise_unit(2.0, precise::count)},
        {"dozen", precise_unit(12.0, precise::count)},
        {"bakersdozen", precise_unit(13.0, precise::count)},
        {"baker'sdozen", precise_unit(13.0, precise::count)},
        {"score", precise_unit(20.0, precise::count)},
        {"octet", precise_unit(8.0, precise::count)},
        {"gross", precise_unit(144.0, precise::count)},
        {"greatgross", precise_unit(144.0 * 12.0, precise::count)},
        {"smallgross", precise_unit(10.0 * 12.0, precise::count)},
        {"googol", precise_unit(1e100, precise::one)},
        {"half", precise_unit(0.5, precise::one)},
        {"quarter", precise_unit(0.25, precise::one)},
        {"third", precise_unit(1.0 / 3.0, precise::one)},
        {"fourth", precise_unit(0.25, precise::one)},
        {"fifth", precise_unit(0.2, precise::one)},
        {"sixth", precise_unit(1.0 / 6.0, precise::one)},
        {"eighth", precise_unit(0.125, precise::one)},
        {"tenth", precise_unit(0.1, precise::one)},
        {"cell", precise_unit(1.0, precise::count, commodities::cell)},
        {"cells", precise_unit(1.0, precise::count, commodities::cell)},
        {"{cells}", precise_unit(1.0, precise::count, commodities::cell)},
        // mainly to catch the commodity for using cell
        {"{#}", precise::count},
        {"[#]", precise::count},
        {"#", precise::count},
        {"number", precise::count},
        // {"ct", precise::count},
        {"pix",
         precise_unit(1.0, precise::count, commodities::pixel)},  // Pixel
        {"pixel",
         precise_unit(1.0, precise::count, commodities::pixel)},  // Pixel
        {"dot",
         precise_unit(1.0, precise::count, commodities::voxel)},  // Pixel
        {"voxel",
         precise_unit(1.0, precise::count, commodities::voxel)},  // Pixel
        {"item", precise::count},
        {"part", precise::count},
        {"ratio", precise::ratio},
        {"rat", precise::ratio},
        {"ERR", precise::error},
        {"ERROR", precise::error},
        {"error", precise::error},

        {"Gy", precise::Gy},
        {"gy", precise::Gy},
        {"GY", precise::Gy},
        {"gray", precise::Gy},
        {"Sv", precise::Sv},
        {"SV", precise::Sv},
        {"sievert", precise::Sv},
        {"sverdrup", precise_unit(1e6, precise::m.pow(3) / precise::s)},
        {"rem", precise::cgs::REM},
        {"[REM]", precise::cgs::REM},
        {"REM", precise::cgs::REM},
        {"radiationequivalentman", precise::cgs::REM},
        {"roentgenequivalentinman", precise::cgs::REM},
        {"Ky", precise::cgs::kayser},
        {"KY", precise::cgs::kayser},
        {"kayser", precise::cgs::kayser},
        {"Bi", precise::cgs::biot},
        {"BI", precise::cgs::biot},
        {"biot", precise::cgs::biot},
        {"abamp", precise::cgs::biot},
        {"abcoulomb", precise::cgs::biot* precise::s},
        {"abC", precise::cgs::biot* precise::s},
        {"statamp", precise::cgs::statC_charge / precise::s},
        {"stA", precise::cgs::statC_charge / precise::s},
        {"abA", precise::cgs::biot},
        {"St", precise::cgs::stokes},
        {"ST", precise::cgs::stokes},
        {"stoke", precise::cgs::stokes},
        {"Mx", precise::cgs::maxwell},
        {"MX", precise::cgs::maxwell},
        {"mX", precise::cgs::maxwell},
        {"maxwell", precise::cgs::maxwell},
        {"Oe", precise::cgs::oersted},
        {"OE", precise::cgs::oersted},
        {"oersted", precise::cgs::oersted},
        {"gilbert", precise::cgs::gilbert},
        {"Gb", precise::cgs::gilbert},
        {"Gi", precise::cgs::gilbert},
        {"p", precise::cgs::poise},
        {"cps", precise_unit(0.01, precise::cgs::poise)},
        // centipoise doesn't conflict with ps
        {"P", precise::cgs::poise},
        {"poise", precise::cgs::poise},
        {"rhe", precise_unit(10.0, (precise::Pa * precise::s).inv())},
        {"Ba", precise::cgs::barye},
        {"BA", precise::cgs::barye},
        {"barye", precise::cgs::barye},
        {"barie", precise::cgs::barye},
        {"debye", precise::cgs::debye},
        {"roentgen", precise::cgs::roentgen},
        {"r\xF6ntgen", precise::cgs::roentgen},
        {"parker", precise::cgs::roentgen},
        {u8"r\u00f6ntgen", precise::cgs::roentgen},
        {u8"ro\u0308ntgen", precise::cgs::roentgen},
        {u8"ro\u00A8ntgen", precise::cgs::roentgen},
        {"Roe", precise::cgs::roentgen},
        {"ROE", precise::cgs::roentgen},
        {"R", precise::cgs::roentgen},
        {"Lmb", precise::cgs::lambert},
        {"LMB", precise::cgs::lambert},
        {"lambert", precise::cgs::lambert},
        {"bril", precise_unit(1e-11, precise::cgs::lambert)},
        {"skot", precise_unit(1e-7, precise::cgs::lambert)},
        {"Lb", precise::cgs::lambert},
        {"langley", precise::cgs::langley},
        {"Ly", precise::cgs::langley},
        {"unitpole", precise::cgs::unitpole},
        {"sb", precise::cgs::stilb},
        {"SB", precise::cgs::stilb},
        {"stilb", precise::cgs::stilb},
        {"apostilb", precise_unit(1e-4 / constants::pi, precise::cgs::stilb)},
        {"asb", precise_unit(1e-4 / constants::pi, precise::cgs::stilb)},
        {"blondel", precise_unit(1e-4 / constants::pi, precise::cgs::stilb)},
        {"nit", precise::cd / precise::m.pow(2)},
        {"nt", precise::cd / precise::m.pow(2)},
        {"ph", precise::cgs::phot},
        {"PHT", precise::cgs::phot},
        {"pHT", precise::cgs::phot},
        {"phot", precise::cgs::phot},
        {"[RAD]", precise::cgs::RAD},
        {"RAD", precise::cgs::RAD},
        {"radiationabsorbeddose", precise::cgs::RAD},
        {"Hz", precise::Hz},
        {"HZ", precise::Hz},
        {"hertz", precise::Hz},
        {"rpm", precise::rpm},
        {"rps", precise_unit(constants::tau, precise::rad / precise::s)},
        {"r", precise_unit(constants::tau, precise::rad)},
        {"revolutions", precise_unit(constants::tau, precise::rad)},
        {"revolution", precise_unit(constants::tau, precise::rad)},
        {"rev", precise_unit(constants::tau, precise::rad)},
        {"revs", precise_unit(constants::tau, precise::rad)},
        {"rev/min", precise::rpm},
        {"rad/s", precise::rad / precise::s},
        {"kat", precise::kat},
        {"kats", precise::kat},
        {"KAT", precise::kat},
        {"katal", precise::kat},
        {"sr", precise::sr},
        {"SR", precise::sr},
        {"steradian", precise::sr},
        {"steradian-solidangle", precise::sr},
        {"$/MWh", precise::currency / precise::MWh},
        {"are", precise::area::are},
        {"ar", precise::area::are},
        {"AR", precise::area::are},
        {"hectare", precise::area::hectare},
        {"barn", precise::area::barn},
        {"b", precise::area::barn},
        {"BRN", precise::area::barn},
        {"ha", precise::area::hectare},
        // porous solid permeability
        {"mW", precise::electrical::mW},
        {"milliwatt", precise::electrical::mW},
        {"puW", precise::electrical::puMW / precise::mega},
        {"puMW", precise::electrical::puMW},
        {"puMVA", precise::electrical::puMW},
        {"pumw", precise::electrical::puMW},
        {"pumegawatt", precise::electrical::puMW},
        {"puV", precise::electrical::puV},
        {"puvolt", precise::electrical::puV},
        {"puA", precise::electrical::puA},
        {"puamp", precise::electrical::puA},
        {"mA", precise::electrical::mA},
        {"milliamp", precise::electrical::mA},
        {"kV", precise::electrical::kV},
        {"kilovolt", precise::electrical::kV},
        {"abvolt", precise::cgs::abVolt},
        {"abV", precise::cgs::abVolt},
        {"statvolt", precise::cgs::statV},
        {"stV", precise::cgs::statV},
        {"erg", precise::cgs::erg},
        {"ERG", precise::cgs::erg},
        {"dyn", precise::cgs::dyn},
        {"DYN", precise::cgs::dyn},
        {"dyne", precise::cgs::dyn},
        {"pond", precise::gm::pond},
        {"pu", precise::pu},
        {"perunit", precise::pu},
        {"flag", precise::iflag},
        {"eflag", precise::eflag},
        {"FLAG", precise::iflag},
        {"EFLAG", precise::eflag},
        {"puOhm", precise::pu* precise::ohm},
        {"puohm", precise::pu* precise::ohm},
        {"puHz", precise::electrical::puHz},
        {"puhertz", precise::electrical::puHz},
        {"mps", precise::m / precise::s},
        {"eV", precise::energy::eV},
        {"bev", precise_unit(1e9, precise::energy::eV)},
        {"Ry", precise_unit(13.60583, precise::energy::eV)},  // Rydberg
        {"electronvolt", precise::energy::eV},
        {"electronVolt", precise::energy::eV},
        // Gigatonne carbon
        {"GtC", precise_unit(1000000000.0, precise::t, commodities::carbon)},
        {"hartree", precise::energy::hartree},
        {"Eh", precise::energy::hartree},  // this could be confused with
                                           // Exa-hour, but not likely
        {"E_h", precise::energy::hartree},  // this could be confused with
                                            // Exa-hour, but not likely
        {"Ha", precise::energy::hartree},

        {"fueloilequivalentof1kiloliter",
         precise::energy::foeb / precise::us::barrel* precise::m.pow(3)},
        {"atm", precise::pressure::atm},
        {"ATM", precise::pressure::atm},
        {"atmos", precise::pressure::atm},
        {"atmosphere", precise::pressure::atm},
        {"standardatmosphere", precise::pressure::atm},
        {"stdatmosphere", precise::pressure::atm},
        {"stdatm", precise::pressure::atm},
        {"mmHg", precise::pressure::mmHg},
        {"mm_Hg", precise::pressure::mmHg},
        {"mm[Hg]", precise::pressure::mmHg},
        {"MM[HG]", precise::pressure::mmHg},
        {"cmHg", precise::ten* precise::pressure::mmHg},
        {"cm_Hg", precise::ten* precise::pressure::mmHg},
        {"cm[Hg]", precise::ten* precise::pressure::mmHg},
        {"CM[HG]", precise::ten* precise::pressure::mmHg},
        {"mHg", precise::kilo* precise::pressure::mmHg},
        {"m_Hg", precise::kilo* precise::pressure::mmHg},
        {"m*Hg", precise::kilo* precise::pressure::mmHg},
        {"m[Hg]", precise::kilo* precise::pressure::mmHg},
        {"M[HG]", precise::kilo* precise::pressure::mmHg},
        {"m[HG]", precise::kilo* precise::pressure::mmHg},
        {"metermercury", precise::kilo* precise::pressure::mmHg},
        {"meterofmercury", precise::kilo* precise::pressure::mmHg},
        {"meter{mercury}", precise::kilo* precise::pressure::mmHg},
        {"meter(mercury)", precise::kilo* precise::pressure::mmHg},
        {"metersofmercury", precise::kilo* precise::pressure::mmHg},
        {"meterofmercurycolumn", precise::kilo* precise::pressure::mmHg},
        {"metersofmercurycolumn", precise::kilo* precise::pressure::mmHg},
        {"mmH2O", precise::pressure::mmH2O},
        {"mm_H2O", precise::pressure::mmH2O},
        {"mm[H2O]", precise::pressure::mmH2O},
        {"MM[H2O]", precise::pressure::mmH2O},
        {"cmH2O", precise::ten* precise::pressure::mmH2O},
        {"cm_H2O", precise::ten* precise::pressure::mmH2O},
        {"cm[H2O]", precise::ten* precise::pressure::mmH2O},
        {"CM[H2O]", precise::ten* precise::pressure::mmH2O},
        {"mH2O", precise::kilo* precise::pressure::mmH2O},
        {"m_H2O", precise::kilo* precise::pressure::mmH2O},
        {"m*H2O", precise::kilo* precise::pressure::mmH2O},
        {"m[H2O]", precise::kilo* precise::pressure::mmH2O},
        {"M[H2O]", precise::kilo* precise::pressure::mmH2O},
        {"meterwater", precise::kilo* precise::pressure::mmH2O},
        {"meterofwater", precise::kilo* precise::pressure::mmH2O},
        {"metersofwater", precise::kilo* precise::pressure::mmH2O},
        {"meters{water}", precise::kilo* precise::pressure::mmH2O},
        {"meter{water}", precise::kilo* precise::pressure::mmH2O},
        {"meters(water)", precise::kilo* precise::pressure::mmH2O},
        {"meter(water)", precise::kilo* precise::pressure::mmH2O},
        {"meterofwatercolumn", precise::kilo* precise::pressure::mmH2O},
        {"metersofwatercolumn", precise::kilo* precise::pressure::mmH2O},
        {"torr", precise::pressure::torr},
        {"Torr", precise::pressure::torr},
        {"TORR", precise::pressure::torr},
        {"$/kWh", precise::currency / precise::kWh},
        {"kWh", precise::kWh},
        {"kwh", precise::kWh},
        {"Wh", precise::W* precise::h},
        {"Wa", precise::W* precise::yr},
        {"kilowatthour", precise::kWh},
        {"MWh", precise::MWh},
        {"MWhr", precise::MWh},
        {"megawatthour", precise::MWh},
        {"M$", precise::other::MegaBuck},
        {"mil$", precise::other::MegaBuck},
        {"B$", precise::other::GigaBuck},
        {"bil$", precise::other::GigaBuck},
        {"L", precise::L},  // preferred US notation
        {"l", precise::L},
        {"LT", precise::L},
        {"liter", precise::L},
        {"litre", precise::L},
        {"mL", precise::mL},  // preferred US notation
        {"ml", precise::mL},
        {"LPM", precise::L / precise::min},
        {"LPS", precise::L / precise::s},
        {"t", precise::mass::tonne},  // metric tonne
        {"mt", precise::mass::tonne},  // metric tonne
        {"ton_m", precise::mass::tonne},
        {"tonne", precise::mass::tonne},
        {"tonne_m", precise::mass::tonne},
        {"TNE", precise::mass::tonne},
        {"Da", precise::mass::Da},
        {"dalton", precise::mass::Da},
        {"u", precise::mass::u},
        {"amu", precise::mass::u},
        {"AMU", precise::mass::u},
        {"unifiedatomicmassunit", precise::mass::u},
        {"atomicmassunit", precise::mass::u},

        {"angstrom", precise::distance::angstrom},
        {u8"\u00C5ngstr\u00F6m", precise::distance::angstrom},
        {"\xE5ngstr\xF6m", precise::distance::angstrom},
        {u8"\u00E5ngstr\u00F6m", precise::distance::angstrom},
        {"Ao", precise::distance::angstrom},
        {"AO", precise::distance::angstrom},
        {u8"\u00C5", precise::distance::angstrom},
        {u8"A\u02DA", precise::distance::angstrom},
        {"\xC5", precise::distance::angstrom},
        {u8"\u212B", precise::distance::angstrom},  // unicode
        {"bps", precise::bit / precise::s},
        {"baud", precise::bit / precise::s},
        {"Bd", precise::bit / precise::s},
        {"BD", precise::bit / precise::s},
        {"bit", precise::bit},
        {"BIT", precise::bit},
        {"nibble", precise::data::nibble},
        {"nybble", precise::data::nibble},
        {"bit_s", precise::data::bit_s},
        {"bit-s", precise::data::bit_s},
        {"BIT_S", precise::data::bit_s},
        {"bit-logarithmic", precise::data::bit_s},
        {"bitlogarithmic", precise::data::bit_s},
        {"logbit", precise::data::bit_s},
        // b is for unit barn
        // B is for bel
        {"Bps", precise::B / precise::s},
        {"byte", precise::B},
        {"By", precise::B},
        {"BY", precise::B},
        {"kB", precise::kB},
        {"MB", precise::MB},
        {"GB", precise::GB},
        {"floatingpointoperation", precise::computation::flop},
        {"flops", precise::computation::flops},
        {"mips", precise::computation::mips},
        {"[car_m]", precise::metric::carat},
        {"ct_m", precise::metric::carat},
        {"[CAR_M]", precise::metric::carat},
        {"carat_m", precise::metric::carat},
        {"photometriccarat", precise::metric::carat},
        {"g", precise::g},
        {"gm", precise::g},
        {"gamma", precise::micro* precise::g},
        {u8"\u1D6FE", precise::micro* precise::g},
        {"gamma{mass}", precise::micro* precise::g},
        {"gamma(mass)", precise::micro* precise::g},
        {"gamma{volume}", precise::micro* precise::L},
        {"gamma(volume)", precise::micro* precise::L},
        {"lambda{volume}", precise::micro* precise::L},
        {"lambda(volume)", precise::micro* precise::L},
        {"gamma(geo)",
         precise::nano* precise::T},  // two different uses of gamma
        {"gamma{geo}",
         precise::nano* precise::T},  // two different uses of gamma
        {"gf", precise::g* constants::g0.as_unit()},
        {"gravity", constants::g0.as_unit()},  // force of gravity
        {"geopotential", constants::g0.as_unit()},  // force of gravity
        {"gp", constants::g0.as_unit()},  // force of gravity
        {"force", constants::g0.as_unit()},  // force of gravity
        {"frc", constants::g0.as_unit()},  // force of gravity
        {"kp", precise::kilo* precise::gm::pond},
        // this is probably more common than kilopoise
        {"tonforce_m", precise::mass::tonne* constants::g0.as_unit()},
        {"tonforce(metric)", precise::mass::tonne* constants::g0.as_unit()},
        {"tf", precise::mass::tonne* constants::g0.as_unit()},
        {"hyl", precise::gm::hyl},
        {"GF", precise::g* constants::g0.as_unit()},  // gram-force vs GF
        {"gram", precise::g},
        {"mg", precise::mg},
        {"mcg", precise::micro* precise::g},
        {"milligram", precise::mg},
        {"carat", precise_unit(200.0, precise::mg)},
        {"karat", precise_unit(200.0, precise::mg)},
        {"ct", precise_unit(200.0, precise::mg)},
        // {"kt", precise_unit(200.0, precise::mg)},
        {"EER", precise::energy::EER},
        {"ppv", precise::one},  // parts per volume
        {"ppth", precise::other::ppm* precise::kilo},
        {"PPTH", precise::other::ppm* precise::kilo},
        {"ppm", precise::other::ppm},
        {"ppmv", precise::other::ppm},
        {"PPM", precise::other::ppm},
        {"[PPM]", precise::other::ppm},
        {"partspermillion", precise::other::ppm},
        {"ppb", precise::other::ppb},
        {"ppbv", precise::other::ppb},
        {"PPB", precise::other::ppb},
        {"[PPB]", precise::other::ppb},
        {"partsperbillion", precise::other::ppb},
        {"ppt", precise::other::ppb* precise::milli},
        {"pptv", precise::other::ppb* precise::milli},
        {"pptr", precise::other::ppb* precise::milli},
        {"PPTR", precise::other::ppb* precise::milli},
        {"[PPTR]", precise::other::ppb* precise::milli},
        {"partspertrillion", precise::other::ppb* precise::milli},
        {"ppq", precise::other::ppb* precise::micro},
        {"ppqv", precise::other::ppb* precise::micro},
        {"partsperquadrillion", precise::other::ppb* precise::micro},
        {"PRU", precise::clinical::pru},
        {"peripheralvascularresistanceunit", precise::clinical::pru},
        {"peripheralresistanceunit", precise::clinical::pru},
        {"potentialvorticityunit",
         precise_unit(
             detail::unit_data(2, -1, -1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0),
             1e-6)},
        {"PVU",
         precise_unit(
             detail::unit_data(2, -1, -1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0),
             1e-6)},
        // unit of thermal resistance used in describing the insulating value of
        // clothing; the amount of thermal resistance needed to maintain in
        // comfort a resting subject in a normally ventilated room (air movement
        // 10 cm/sec) at a temperature of 20 degrees C and a humidity less than
        // 50%
        {"clo",
         precise_unit(1.55e-1, precise::K* precise::m.pow(2) / precise::W)},
        {"[MET]", precise::clinical::met},
        {"MET", precise::clinical::met},
        {"metabolicEquivalentofTask", precise::clinical::met},
        {"metabolicequivalents", precise::clinical::met},
        {"[wood'U]", precise::clinical::woodu},
        {"HRU", precise::clinical::woodu},
        {"hybridreferenceunit", precise::clinical::woodu},
        {"[WOOD'U]", precise::clinical::woodu},
        {"woodunit", precise::clinical::woodu},
        {"dpt", precise::clinical::diopter},
        {"diop", precise::clinical::diopter},
        {"[DIOP]", precise::clinical::diopter},
        {"diopter", precise::clinical::diopter},
        {"dioptre", precise::clinical::diopter},
        {"diopter(lens)", precise::clinical::diopter},
        {"[p'diop]", precise::clinical::prism_diopter},
        {"pdiop", precise::clinical::prism_diopter},
        {"[P'DIOP]", precise::clinical::prism_diopter},
        {"PD", precise::clinical::prism_diopter},
        {"prismdiopter", precise::clinical::prism_diopter},
        {"prismdiopter(magnifyingpower)", precise::clinical::prism_diopter},
        {"%slope", precise::clinical::prism_diopter},
        {"%[slope]", precise::clinical::prism_diopter},
        {"%[SLOPE]", precise::clinical::prism_diopter},
        {"percentofslope", precise::clinical::prism_diopter},
        {"drp", precise::clinical::drop},
        {"[drp]", precise::clinical::drop},
        {"[DRP]", precise::clinical::drop},
        {"drop", precise::clinical::drop},
        {"drop(1/20milliliter)", precise::clinical::drop},
        {"drop-metric", precise::clinical::drop},
        {"drop_m", precise::clinical::drop},
        {"drop-metric(1/20mL)", precise::clinical::drop},
        {"Ch", precise::clinical::charriere},
        {"[CH]", precise::clinical::charriere},
        {"french(cathetergauge)", precise::clinical::charriere},
        {"mesh_i", precise::clinical::mesh},
        {"[MESH_I]", precise::clinical::mesh},
        {"mesh", precise::clinical::mesh},
        {"meshes", precise::clinical::mesh},
        {"ASD", precise::special::ASD},
        {"[m/s2/Hz^(1/2)]", precise::special::ASD},
        {"[M/s2/HZ^(1/2)]", precise::special::ASD},
        {"Hz^(1/2)", precise::special::rootHertz},
        {"HZ^(1/2)", precise::special::rootHertz},
        {"squarerootofhertz", precise::special::rootHertz},
        {"squareroothertz", precise::special::rootHertz},
        {"roothertz", precise::special::rootHertz},
        // capitalized version is needed since this is also a generated unit
        {"rootHertz", precise::special::rootHertz},
        {"B", precise::data::byte},
        {"bel", precise::log::bel},
        {"dB", precise::log::dB},
        {"decibel", precise::log::dB},
        {"decibels", precise::log::dB},
        {"DB", precise::log::dB},
        {"Np", precise::log::neper},
        {"NEP", precise::log::neper},
        {"nepers", precise::log::neper},
        {"neper", precise::log::neper},
        {"SPL", precise_unit(2 * 1e-5, precise::pascal)},
        {"soundpressurelevel", precise_unit(2 * 1e-5, precise::pascal)},
        {"soundpressure", precise_unit(2 * 1e-5, precise::pascal)},
        {"BSPL", precise::log::B_SPL},
        {"B[SPL]", precise::log::B_SPL},
        {"B(SPL)", precise::log::B_SPL},
        {"belSPL", precise::log::B_SPL},
        {"belsoundpressure", precise::log::B_SPL},
        {"dB[SPL]", precise::log::dB_SPL},
        {"decibel(SPL)", precise::log::dB_SPL},
        {"DB[SPL]", precise::log::dB_SPL},
        {"DBSPL", precise::log::dB_SPL},
        {"B[V]", precise::log::B_V},
        {"B(V)", precise::log::B_V},
        {"BV", precise::log::B_V},
        {"Bv", precise_unit(0.775, precise::log::B_V)},
        {"Bvolt", precise::log::B_V},
        {"belvolt", precise::log::B_V},
        {"belV", precise::log::B_V},
        {"dB[V]", precise::log::dB_V},
        {"decibelV", precise::log::dB_V},
        {"dB(V)", precise::log::dB_V},
        {"DB[V]", precise::log::dB_V},
        {"B[mV]", precise::log::B_mV},
        {"B(mV)", precise::log::B_mV},
        {"belmillivolt", precise::log::B_mV},
        {"belmV", precise::log::B_mV},
        {"dB[mV]", precise::log::dB_mV},
        {"dB(mV)", precise::log::dB_mV},
        {"B[MV]", precise::log::B_mV},
        {"DB[MV]", precise::log::dB_mV},
        {"B[uV]", precise::log::B_uV},
        {"dB[uV]", precise::log::dB_uV},
        {"decibelmicrovolt", precise::log::dB_uV},
        {"B[UV]", precise::log::B_uV},
        {"B(uV)", precise::log::B_uV},
        {"BuV", precise::log::B_uV},
        {"belmicrovolt", precise::log::B_uV},
        {"DB[UV]", precise::log::dB_uV},
        {"B[10.nV]", precise::log::B_10nV},
        {"B(10nV)", precise::log::B_10nV},
        {"bel10nanovolt", precise::log::B_10nV},
        {"dB[10.nV]", precise::log::dB_10nV},
        {"dB[10*nV]", precise::log::dB_10nV},
        {"decibel10nanovolt", precise::log::B_10nV},
        {"B[10*NV]", precise::log::B_10nV},
        {"B[10*nV]", precise::log::B_10nV},
        {"DB[10*NV]", precise::log::dB_10nV},
        {"B[W]", precise::log::B_W},
        {"B(W)", precise::log::B_W},
        {"belwatt", precise::log::B_W},
        {"belW", precise::log::B_W},
        {"dB[W]", precise::log::dB_W},
        {"DB[W]", precise::log::dB_W},
        {"B[kW]", precise::log::B_kW},
        {"B(kW)", precise::log::B_kW},
        {"DB[kW]", precise::log::dB_kW},
        {"B[KW]", precise::log::B_kW},
        {"belkilowatt", precise::log::B_kW},
        {"belkW", precise::log::B_kW},
        {"DB[KW]", precise::log::dB_kW},
        {"dBZ", precise::log::dBZ},
        {"BZ", precise::log::BZ},
        {"[S]", precise::laboratory::svedberg},
        {"svedbergunit", precise::laboratory::svedberg},
        {"svedberg", precise::laboratory::svedberg},
        {"HPF", precise::laboratory::HPF},
        {"hpf", precise::laboratory::HPF},
        {"highpowerfield", precise::laboratory::HPF},
        {"LPF", precise::laboratory::LPF},
        {"lpf", precise::laboratory::LPF},
        {"lowpowerfield", precise::laboratory::LPF},
        {"[arb'U]", precise::laboratory::arbU},
        {"[ARB'U]", precise::laboratory::arbU},
        {"arbitraryunit", precise::laboratory::arbU},
        {"arbunit", precise::laboratory::arbU},
        {"pdu", precise::laboratory::arbU},
        {"arbu", precise::laboratory::arbU},
        {"proceduredefinedunit", precise::laboratory::arbU},
        {"unknownunit", precise::laboratory::arbU},
        {"[IU]", precise::laboratory::IU},
        {"[iU]", precise::laboratory::IU},
        {"IU", precise::laboratory::IU},
        {"internationalunit", precise::laboratory::IU},
        {"dobson",
         precise_unit(446.2, precise::micro* precise::mol / precise::m.pow(2))},
        {"DU",
         precise_unit(446.2, precise::micro* precise::mol / precise::m.pow(2))},
        {"st", precise::volume::stere},
        {"stere", precise::volume::stere},
        {"STR", precise::volume::stere},
        {"[hnsf'U]", precise::clinical::hounsfield},
        {"[HNSF'U]", precise::clinical::hounsfield},
        {"[PFU]", precise::laboratory::PFU},
        {"PFU", precise::laboratory::PFU},
        {"pfu", precise::laboratory::PFU},
        {"plaqueformingunits", precise::laboratory::PFU},
        {"plaqueformingunit", precise::laboratory::PFU},
        {"[Lf]", precise::laboratory::Lf},
        {"[LF]", precise::laboratory::Lf},
        {"[IR]", precise::laboratory::IR},
        {"[Ir]", precise::laboratory::IR},
        {"50%tissuecultureinfectiousdose",
         precise_unit(0.69, precise::laboratory::PFU)},
        {"50%cellcultureinfectiousdose",
         precise_unit(0.69, precise::laboratory::PFU, commodities::cell)},
        {"50%embryoinfectiousdose",
         precise_unit(0.69, precise::laboratory::PFU, commodities::embryo)},
        {"TCID50", precise_unit(0.69, precise::laboratory::PFU)},
        {"CCID50",
         precise_unit(0.69, precise::laboratory::PFU, commodities::cell)},
        {"EID50",
         precise_unit(0.69, precise::laboratory::PFU, commodities::embryo)},
        {"[hp'_X]", precise_unit(1.0, precise::log::neglog10)},
        {"[HP'_X]", precise_unit(1.0, precise::log::neglog10)},
        {"[hp'_C]", precise_unit(1.0, precise::log::neglog100)},
        {"[HP'_C]", precise_unit(1.0, precise::log::neglog100)},
        {"[hp'_M]", precise_unit(1.0, precise::log::neglog1000)},
        {"[HP'_M]", precise_unit(1.0, precise::log::neglog1000)},
        {"[hp'_Q]", precise_unit(1.0, precise::log::neglog50000)},
        {"[HP'_Q]", precise_unit(1.0, precise::log::neglog50000)},
        {"pH", precise::laboratory::pH},
        {"pHscale", precise::laboratory::pH},
        {"[PH]", precise::laboratory::pH},
    }};

UNITS_CPP14_CONSTEXPR_OBJECT std::array<
    std::pair<const char*, precise_unit>,
    1112>
    defined_unit_strings_customary{
        {{"candle", precise::other::candle},
         {"candlepower", precise::other::candle},
         {"footlambert",
          precise_unit(1.0 / constants::pi, precise::cd / precise::ft.pow(2))},
         {"fl",
          precise_unit(1.0 / constants::pi, precise::cd / precise::ft.pow(2))},
         {"CP", precise::other::candle},
         {"footcandle", precise::lumen / precise::ft.pow(2)},
         {"fc", precise::lumen / precise::ft.pow(2)},
         {"ftc", precise::lumen / precise::ft.pow(2)},
         {"stC", precise::cgs::statC_charge},
         {"statC", precise::cgs::statC_charge},
         {"statC{charge}", precise::cgs::statC_charge},
         {"statC{flux}", precise::cgs::statC_flux},
         {"statcoulomb", precise::cgs::statC_charge},
         {"statcoulomb{charge}", precise::cgs::statC_charge},
         {"statcoulomb{flux}", precise::cgs::statC_flux},
         {"Fr", precise::cgs::statC_charge},
         {"franklin", precise::cgs::statC_charge},
         {"esu", precise::cgs::statC_charge},
         {"emu", precise::cgs::emu},
         {"bar", precise::bar},
         {"BAR", precise::bar},
         {"in", precise::in},
         {"in_i", precise::in},
         {"[IN_I]", precise::in},
         {"inch_i", precise::i::inch},
         {"inches", precise::in},
         {"inches_i", precise::i::inch},
         {"in_us", precise::us::inch},
         {"inus", precise::us::inch},
         {"usin",
          precise::us::inch},  // this is more likely than micro square in
         {"[IN_US]", precise::us::inch},
         {"inch_us", precise::us::inch},
         {"inches_us", precise::us::inch},
         {"caliber", precise_unit(0.01, precise::i::inch)},
         {"rackunit", precise_unit(1.75, precise::i::inch)},
         {"RU", precise_unit(1.75, precise::i::inch)},
         {"rd", precise::us::rod},
         {"rod", precise::us::rod},
         {"rd_us", precise::us::rod},
         {"rdus", precise::us::rod},
         {"[RD_US]", precise::us::rod},
         {"rod_us", precise::us::rod},
         {"rods_us", precise::us::rod},
         {"ch", precise::us::chain},
         {"chain", precise::us::chain},
         {"chain_us", precise::us::chain},
         {"ch_us", precise::us::chain},
         {"[CH_US]", precise::us::chain},
         {"gunter'schain_us", precise::us::chain},
         {"surveyors'schain_us", precise::us::chain},
         {"surveyors'schain", precise::us::chain},
         {"lk_us", precise::us::link},
         {"li", precise::us::link},
         {"lnk", precise::us::link},
         {"link_us", precise::us::link},
         {"li_us", precise::us::link},
         {"link", precise::us::link},
         {"[LK_US]", precise::us::link},
         {"linkforGunter'schain_us", precise::us::link},
         {"rch_us", precise::us::engineers::chain},
         {"[RCH_US]", precise::us::engineers::chain},
         {"ramden'schain_us", precise::us::engineers::chain},
         {"rlk_us", precise::us::engineers::link},
         {"[RLK_US]", precise::us::engineers::link},
         {"linkforRamden'schain_us", precise::us::engineers::link},
         {"fur", precise::us::furlong},
         {"furlong", precise::us::furlong},
         {"fur_us", precise::us::furlong},
         {"[FUR_US]", precise::us::furlong},
         {"furlong_us", precise::us::furlong},
         {"[FUR_I]", precise::us::furlong},

         {"mi_us", precise::us::mile},
         {"[MI_US]", precise::us::mile},
         {"mile_us", precise::us::mile},
         {"lea_us", precise_unit(3.0, precise::us::mile)},
         {"league_us", precise_unit(3.0, precise::us::mile)},

         {"mil_us", precise::us::mil},
         {"[MIL_US]", precise::us::mil},
         {"inch", precise::in},
         {"thou", precise::imp::thou},
         {"thousandth", precise::in* precise::milli},
         {"mil", precise::i::mil},
         {"mil_i", precise::i::mil},
         {"[MIL_I]", precise::i::mil},
         {"cml", precise::i::circ_mil},
         {"circularmil", precise::i::circ_mil},
         {"circularinch",
          precise_unit{constants::pi / 4.0, precise::i::inch.pow(2)}},
         {"cml_i", precise::i::circ_mil},
         {"circularmil_i", precise::i::circ_mil},
         {"[CML_I]", precise::i::circ_mil},
         {"hd", precise::i::hand},
         {"hd_i", precise::i::hand},
         {"[HD_I]", precise::i::hand},
         {"hand", precise::i::hand},
         {"ft", precise::ft},
         {"ft_i", precise::ft},
         {"[FT_I]", precise::ft},
         {"bf", precise::i::board_foot},
         {"BF", precise::i::board_foot},
         {"bf_i", precise::i::board_foot},
         {"BDFT", precise::i::board_foot},
         {"FBM", precise::i::board_foot},
         {"[BF_I]", precise::i::board_foot},
         {"boardfoot", precise::i::board_foot},
         {"boardfeet", precise::i::board_foot},
         {"boardfeet_i", precise::i::board_foot},
         {"cr", precise::i::cord},
         {"crd", precise::i::cord},
         {"cord", precise::i::cord},
         {"crd_i", precise::i::cord},
         {"cords_i", precise::i::cord},
         {"cr_i", precise::i::cord},
         {"[CR_I]", precise::i::cord},
         {"crd_us", precise::us::cord},
         {"[CRD_US]", precise::us::cord},
         {"cord_us", precise::us::cord},
         {"ftus", precise::ft},
         {"ft_us", precise::ft},
         {"[FT_US]", precise::ft},
         {"foot_us", precise::ft},
         {"feet_us", precise::ft},
         {"ft^2", precise::ft* precise::ft},
         {"sin", precise::in* precise::in},
         {"sin_i", precise::in* precise::in},
         {"[SIN_I]", precise::in* precise::in},
         {"ci", precise::in.pow(3)},  // cubic inch instead of centiinch
         {"cin", precise::in.pow(3)},  // cubic inch instead of centiinch
         {"cin_i", precise::in.pow(3)},
         {"[CIN_I]", precise::in.pow(3)},
         {"sf", precise::ft* precise::ft},
         {"sft", precise::ft* precise::ft},
         {"sft_i", precise::ft* precise::ft},
         {"[SFT_I]", precise::ft* precise::ft},
         {"SCF", precise::energy::scf},  // standard cubic foot
         {"CCF", precise::hundred* precise::energy::scf},  // centum cubic foot
         {"MCF", precise::kilo* precise::energy::scf},  // millum cubic foot
         {"MMCF", precise::mega* precise::energy::scf},  // million cubic foot
         {"BCF", precise::giga* precise::energy::scf},  // billion cubic foot
         {"TCF", precise::tera* precise::energy::scf},  // trillion cubic foot
         {"QCF", precise::exa* precise::energy::scf},  // quadrillion cubic foot
         {"Mcf", precise::kilo* precise::energy::scf},  // millum cubic foot
         {"MMcf", precise::mega* precise::energy::scf},  // million cubic foot
         {"Bcf", precise::giga* precise::energy::scf},  // billion cubic foot
         {"Tcf", precise::tera* precise::energy::scf},  // trillion cubic foot
         {"Qcf", precise::exa* precise::energy::scf},  // quadrillion cubic foot
         {"cf", precise::energy::scf},
         {"scf", precise::energy::scf},
         {"std", precise::energy::scf},
         {"ncf", precise::energy::ncf},
         {"mcf", precise::kilo* precise::energy::scf},
         {"ccf", precise::hundred* precise::energy::scf},
         {"cft", precise::ft.pow(3)},
         {"[cft_i]", precise::ft.pow(3)},
         {"[CFT_I]", precise::ft.pow(3)},
         {"foot", precise::ft},
         {"foot_i", precise::i::foot},
         {"feet", precise::ft},
         {"feet_i", precise::i::foot},
         {"YD", precise::yd},
         {"yd", precise::yd},
         {"yd_i", precise::yd},
         {"yard_i", precise::yd},
         {"yards_i", precise::yd},
         {"[YD_I]", precise::yd},
         {"ydus", precise::us::yard},
         {"yd_us", precise::us::yard},
         {"ydsus", precise::us::yard},
         {"[YD_US]", precise::us::yard},
         {"yardus", precise::us::yard},
         {"yard_us", precise::us::yard},
         {"yards_us", precise::us::yard},
         {"marathon", {26.2, precise::mile}},
         {"sy", precise::yd* precise::yd},
         {"syd", precise::yd* precise::yd},
         {"syd_i", precise::yd* precise::yd},
         {"[SYD_I]", precise::yd* precise::yd},
         {"cy", precise::yd.pow(3)},
         {"cyd", precise::yd.pow(3)},
         {"cyd_i", precise::yd.pow(3)},
         {"[CYD_I]", precise::yd.pow(3)},
         {"in_br", precise::imp::inch},
         {"inch_br", precise::imp::inch},
         {"inches_br", precise::imp::inch},
         {"barleycorn", precise::imp::barleycorn},
         {"[IN_BR]", precise::imp::inch},
         {"ft_br", precise::imp::foot},
         {"[FT_BR]", precise::imp::foot},
         {"foot_br", precise::imp::foot},
         {"rd_br", precise::imp::rod},
         {"[RD_BR]", precise::imp::rod},
         {"rod_br", precise::imp::rod},
         {"perch", precise::imp::rod},
         {"pole", precise::imp::rod},
         {"ch_br", precise::imp::chain},
         {"[CH_BR]", precise::imp::chain},
         {"gunter'schain_br", precise::imp::chain},
         {"lk_br", precise::imp::link},
         {"[LK_BR]", precise::imp::link},
         {"linkforGunter'schain_br", precise::imp::link},
         {"fth_br", precise::nautical::fathom},
         {"[FTH_BR]", precise::nautical::fathom},
         {"fathom_br", precise::nautical::fathom},
         {"fathoms_br", precise::nautical::fathom},
         {"pc_br", precise::imp::pace},
         {"[PC_BR]", precise::imp::pace},
         {"pace_br", precise::imp::pace},
         {"yd_br", precise::imp::yard},
         {"[YD_BR]", precise::imp::yard},
         {"yard_br", precise::imp::yard},
         {"nmi_br", precise::imp::nautical_mile},
         {"[NMI_BR]", precise::imp::nautical_mile},
         {"nauticalmile_br", precise::imp::nautical_mile},
         {"cable_br", precise_unit(0.1, precise::imp::nautical_mile)},
         {"nauticalleague_br", precise_unit(3.0, precise::imp::nautical_mile)},
         {"kn_br", precise::imp::nautical_mile / precise::hr},
         {"[KN_BR]", precise::imp::nautical_mile / precise::hr},
         {"knot_br", precise::imp::nautical_mile / precise::hr},
         {"mi_br", precise::imp::mile},
         {"[MI_BR]", precise::imp::mile},
         {"mile_br", precise::imp::mile},
         {"gal_br", precise::imp::gallon},
         {"[GAL_BR]", precise::imp::gallon},
         {"gallon_br", precise::imp::gallon},
         {"yard", precise::yd},
         {"cubit", precise::distance::cubit},
         {"cubit_br", precise::distance::cubit},
         {"cubit(UK)", precise::distance::cubit},
         {"ell", precise_unit(45.0, precise::in)},
         {"ell_br", precise_unit(45.0, precise::in)},
         {"englishell", precise_unit(45.0, precise::in)},
         {"scottishell", precise_unit(0.9398, precise::m)},
         {"frenchell", precise_unit(1.3716, precise::m)},
         {"flemishell", precise_unit(.6858, precise::m)},
         // because cubit has cu in it which indicates cubed
         {"longcubit", precise::distance::longcubit},
         // my kids have taken to measuring thing in number of daddy's so it
         // need to be in the app
         {"daddy", {75.0, precise::in}},
         {"arpent", precise::distance::arpent_us},
         {"arpent_fr", precise::distance::arpent_fr},
         {"arpentlin", precise::distance::arpent_fr},
         {"biweekly", precise_unit(0.5, precise::time::week.inv())},
         {"fortnight", precise::time::fortnight},
         {"yearleap", precise_unit(366.0, precise::time::day)},  // year
         {"draconicyear", precise_unit(346.620075883, precise::time::day)},
         {"lunaryear", precise_unit(12.0, precise::time::mos)},
         {"gon", precise::angle::gon},
         {"gon(grade)", precise::angle::gon},
         {"GON", precise::angle::gon},
         {u8"\u25A1^g", precise::angle::gon},
         {"^g", precise::angle::gon},
         {"grad", precise::angle::grad},
         {"gradians", precise::angle::grad},
         {"grade", precise::angle::grad},
         {"mil(angle)", precise_unit(0.0625, precise::angle::grad)},
         {"brad", precise::angle::brad},
         {"degF", precise::degF},
         {"degsF", precise::degF},
         {"[DEGF]", precise::degF},
         {u8"\u2109", precise::degF},  // direct unicode symbol
         {"degR", precise::temperature::degR},
         {"degsR", precise::temperature::degR},
         {"[DEGR]", precise::temperature::degR},
         {u8"\u00B0R", precise::temperature::degR},
         {u8"\u00B0r", precise::temperature::reaumur},
         {"\xB0R", precise::temperature::degR},
         {"\xB0r", precise::temperature::reaumur},
         {"[DEGRE]", precise::temperature::reaumur},
         {"degRe", precise::temperature::reaumur},
         {"degsRe", precise::temperature::reaumur},
         {u8"degR\u00e9aumur", precise::temperature::reaumur},
         {u8"\u00B0R\u00e9", precise::temperature::reaumur},
         {u8"\u00B0Re", precise::temperature::reaumur},
         {u8"\u00B0Ra", precise::temperature::degR},
         {"\xB0Re", precise::temperature::reaumur},
         {"\xB0Ra", precise::temperature::degR},
         {"degReaumur", precise::temperature::reaumur},
         {"reaumur", precise::temperature::reaumur},
         {u8"r\u00e9aumur", precise::temperature::reaumur},
         {"degFahrenheit", precise::degF},
         {"degRankine", precise::temperature::degR},
         {"degrankine", precise::temperature::degR},
         {"rankine", precise::temperature::degR},
         // gas oven temperature conversions
         {"gasmark", {14.0, precise::temperature::degC}},
         {"gasnumber", {14.0, precise::temperature::degC}},
         {"gas", {14.0, precise::temperature::degC}},
         {"regulomark", {14.0, precise::temperature::degC}},
         {"regulo", {14.0, precise::temperature::degC}},
         {"stufe", {25.0, precise::temperature::degC}},
         {"Thermostat", {27.77778, precise::temperature::degC}},
         {"Th", {27.77778, precise::temperature::degC}},
         {"oF", precise::degF},
         // this is two strings since F could be interpreted as hex and I don't
         // want it to be
         {"\xB0"
          "F",
          precise::degF},
         {u8"\u00b0F", precise::degF},
         {"fahrenheit", precise::degF},
         {"mi", precise::mile},
         {"mi_i", precise::mile},
         {"league", precise::i::league},
         {"lea", precise::i::league},
         {"[MI_I]", precise::mile},
         {"miI", precise::mile},
         {"smi", precise::mile* precise::mile},
         {"smi_us", precise::us::mile* precise::us::mile},
         {"[SMI_US]", precise::us::mile* precise::us::mile},
         {"mile", precise::mile},
         {"mile_i", precise::mile},
         {"miles_i", precise::mile},
         {"srd_us", precise::us::rod* precise::us::rod},
         {"[SRD_US]", precise::us::rod* precise::us::rod},
         {"sct", precise::us::section},
         {"[SCT]", precise::us::section},
         {"section", precise::us::section},
         {"homestead", precise::us::homestead},
         {"twp", precise::us::township},
         {"[TWP]", precise::us::township},
         {"township", precise::us::township},
         {"township_us", precise::us::township},
         {"astronomicalunitBIPM2006", precise::distance::au_old},
         {"auold", precise::distance::au_old},
         {"ua", precise::distance::au_old},
         {"FEU", precise_unit(40.0 * 8.0 * 8.5, precise::ft.pow(3))},
         {"TEU", precise_unit(20.0 * 8.0 * 8.5, precise::ft.pow(3))},
         {"fortyfootequivalent",
          precise_unit(40.0 * 8.0 * 8.5, precise::ft.pow(3))},
         {"twentyfootequivalent",
          precise_unit(20.0 * 8.0 * 8.5, precise::ft.pow(3))},
         {"acre", precise::acre},
         {"acr", precise::acre},
         {"ac", precise::acre},
         {"acft", precise::acre* precise::us::foot},
         {"ac*ft", precise::acre* precise::us::foot},
         {"acre-foot", precise::acre* precise::us::foot},
         {"acrefoot", precise::acre* precise::us::foot},
         {"acre_us", precise::acre},
         {"acr_us", precise::acre},
         {"[ACR_US]", precise::acre},
         {"[ACR_BR]", precise::imp::acre},
         {"acr_br", precise::imp::acre},
         {"acre_br", precise::imp::acre},
         {"acres_br", precise::imp::acre},
         {"buildersacre", precise_unit(40000.0, precise::ft.pow(2))},
         {"cricketpitch", precise_unit(3.66 * 22.56, precise::m.pow(2))},
         {"cricketpitches", precise_unit(3.66 * 22.56, precise::m.pow(2))},
         {"footballpitch", precise_unit(105.0 * 68.0, precise::m.pow(2))},
         {"footballpitches", precise_unit(105.0 * 68.0, precise::m.pow(2))},
         {"footballfield", precise_unit(360.0 * 160.0, precise::ft.pow(2))},
         {"Gasolineat15.5C",
          precise_unit(739.33, precise::kg / precise::m.pow(3))},
         {"rood", precise_unit(0.25, precise::imp::acre)},
         // porous solid permeability
         {"darcy", precise_unit(9.869233e-13, precise::m.pow(2))},
         {"$/gal", precise::currency / precise::gal},
         // allow lower case watt abbreviation
         {"w", precise::watt},
         {"hp", precise::hp},
         {"HP", precise::hp},
         {"horsepower", precise::hp},
         {"donkeypower", {250.0, precise::watt}},
         {"horsepower_i", precise::hp},
         {"horsepower_br", precise::hp},
         {"[HP]", precise::hp},
         {"hpI", precise::hp},
         {"hp(I)", precise::hp},
         {"horsepower-mechanical", precise::hp},
         {"horsepowermechanical", precise::hp},
         {"mechanicalhorsepower", precise::hp},
         {"shafthorsepower", precise::hp},
         {"horsepower(mechanical)", precise::hp},
         {"horsepower(water)", precise::hp},
         {"waterhorsepower", precise::hp},
         {"horsepower(hydraulic)", precise::hp},
         {"horsepower(air)", precise::hp},
         {"hp(mechanical)", precise::hp},
         {"hp(water)", precise::hp},
         {"hp(hydraulic)", precise::hp},
         {"hp(air)", precise::hp},
         {"hpE", precise::power::hpE},
         {"hp(E)", precise::power::hpE},
         {"horsepower-electrical", precise::power::hpE},
         {"horsepower(electrical)", precise::power::hpE},
         {"horsepower(electric)", precise::power::hpE},
         {"electrichorsepower", precise::power::hpE},
         {"hp(electric)", precise::power::hpE},
         {"hpM", precise::power::hpM},
         {"hp_m", precise::power::hpM},
         {"hp(M)", precise::power::hpM},
         {"horsepower_m", precise::power::hpM},
         {"hpS", precise::power::hpS},
         {"hp(S)", precise::power::hpS},
         {"horsepower-steam", precise::power::hpS},
         {"horsepower(steam)", precise::power::hpS},
         {"horsepower(boiler)", precise::power::hpS},
         {"boilerhorsepower", precise::power::hpS},
         {"hp(boiler)", precise::power::hpS},
         {"mph", precise::mph},
         {"mileperhour", precise::mph},

         {"nauticalmile", precise::nautical::mile},
         {"nmile", precise::nautical::mile},
         {"nauticalmile_i", precise::nautical::mile},
         {"nauticalleague", precise::nautical::league},
         {"nauticalleage_i", precise::nautical::league},
         {"nauticalleague_i", precise::nautical::league},
         // this prevents some other issues with the string "br"
         {"br", precise::invalid},
         {"nmi", precise::nautical::mile},
         {"nmi_i", precise::nautical::mile},
         {"[NMI_I]", precise::nautical::mile},
         {"fth", precise::nautical::fathom},
         {"fath", precise::nautical::fathom},
         {"fth_us", precise::nautical::fathom},
         {"fthus", precise::nautical::fathom},
         {"[FTH_US]", precise::nautical::fathom},
         {"fathom_us", precise::nautical::fathom},
         {"fathom", precise::nautical::fathom},
         {"fathom_i", precise::nautical::fathom},
         {"fathoms_i", precise::nautical::fathom},
         {"cable_i", precise_unit(0.1, precise::nautical::mile)},
         {"cable", precise_unit(0.1, precise::nautical::mile)},
         {"cable_us", precise_unit(219.5, precise::m)},
         {"fth_i", precise::nautical::fathom},
         {"[FTH_I]", precise::nautical::fathom},
         {"kn", precise::nautical::knot},
         {"knot", precise::nautical::knot},
         {"knot_i", precise::nautical::knot},
         {"knots_i", precise::nautical::knot},
         {"kn_i", precise::nautical::knot},
         {"[KN_I]", precise::nautical::knot},
         {"knots", precise::nautical::knot},
         {"kts", precise::nautical::knot},
         {"fps", precise::ft / precise::s},
         {"fpm", precise::ft / precise::min},
         {"fph", precise::ft / precise::hr},
         {"cal", precise::cal},
         {"smallcalorie", precise::cal},
         {"Cal", precise::energy::kcal},
         {"CAL", precise::cal},
         {"[Cal]", precise::energy::kcal},
         {"[CAL]", precise::energy::kcal},
         {"nutritionlabelCalories", precise::energy::kcal},
         {"foodcalories", precise::energy::kcal},
         {"largecalories", precise::energy::kcal},
         {"kilogramcalories", precise::energy::kcal},
         {"calorie(nutritional)", precise::energy::cal_it},
         {"cal_[15]", precise::energy::cal_15},
         {u8"cal_15\u00B0C", precise::energy::cal_15},
         {u8"calorieat15\u00B0C", precise::energy::cal_15},
         {"caloriesat15C", precise::energy::cal_15},
         {"calories15C", precise::energy::cal_15},
         {"calorie15C", precise::energy::cal_15},
         {"cal_[20]", precise::energy::cal_20},
         {u8"calorieat20\u00B0C", precise::energy::cal_20},
         {"caloriesat20C", precise::energy::cal_20},
         {"calorie20C", precise::energy::cal_20},
         {"cals20C", precise::energy::cal_20},
         {"cal20C", precise::energy::cal_20},
         {"cals15C", precise::energy::cal_15},
         {"cal15C", precise::energy::cal_15},
         {u8"cal_20\u00B0C", precise::energy::cal_20},
         {"CAL_[15]", precise::energy::cal_15},
         {"CAL_[20]", precise::energy::cal_20},
         {"cal_m", precise::energy::cal_mean},
         {"meancalorie", precise::energy::cal_mean},
         {"cal_IT", precise::energy::cal_it},
         {"CAL_M", precise::energy::cal_mean},
         {"CAL_IT", precise::energy::cal_it},
         {"calorie_IT", precise::energy::cal_it},
         {"caloriesIT", precise::energy::cal_it},
         {"cal_th", precise::energy::cal_th},
         {"CAL_TH", precise::energy::cal_th},
         {"calorie_th", precise::energy::cal_th},
         {"calorie", precise::cal},
         {"kcal", precise::energy::kcal},
         {"kCal", precise::energy::kcal},
         {"KCAL", precise::energy::kcal},
         {"Kcal", precise::energy::kcal},
         {"kilocalorie", precise::energy::kcal},
         {"btu", precise::energy::btu_it},
         {"Btu", precise::energy::btu_it},
         {"BTU", precise::energy::btu_it},
         {"Mbtu", precise_unit(1000.0, precise::energy::btu_it)},
         {"MBtu", precise_unit(1000.0, precise::energy::btu_it)},
         {"MBTU", precise_unit(1000.0, precise::energy::btu_it)},
         {"[Btu]", precise::energy::btu_th},
         {"[BTU]", precise::energy::btu_th},
         {"British thermal unit", precise::energy::btu_th},
         // this is for name matching with the UCUM standard
         {"Btu_39", precise::energy::btu_39},
         {"BTU_39", precise::energy::btu_39},
         {"BTU39F", precise::energy::btu_39},
         {u8"BTU39\u00B0F", precise::energy::btu_39},
         {u8"btu_39\u00B0F", precise::energy::btu_39},
         {"Btu_59", precise::energy::btu_59},
         {"BTU_59", precise::energy::btu_59},
         {"BTU59F", precise::energy::btu_59},
         {u8"BTU59\u00B0F", precise::energy::btu_59},
         {u8"btu_59\u00B0F", precise::energy::btu_59},
         {"Btu_60", precise::energy::btu_60},
         {"BTU_60", precise::energy::btu_60},
         {"BTU60F", precise::energy::btu_60},
         {u8"BTU60\u00B0F", precise::energy::btu_60},
         {u8"btu_60\u00B0F", precise::energy::btu_60},
         {"Btu_m", precise::energy::btu_mean},
         {"BTU_m", precise::energy::btu_mean},
         {"BTU_M", precise::energy::btu_mean},
         {"Btu_IT", precise::energy::btu_it},
         {"BTU_IT", precise::energy::btu_it},
         {"Btu_th", precise::energy::btu_th},
         {"[BTU_TH]", precise::energy::btu_th},
         {"BTU_th", precise::energy::btu_th},
         {"CHU", precise_unit(1899.0, precise::J)},
         {"TNT", precise::energy::ton_tnt},
         {"tontnt", precise::energy::ton_tnt},
         {"tonoftnt", precise::energy::ton_tnt},
         {"tonsoftnt", precise::energy::ton_tnt},
         {"tonsofTNT", precise::energy::ton_tnt},
         {"ton(explosives)", precise::energy::ton_tnt},
         {"kiloton", precise_unit(1000.0, precise::energy::ton_tnt)},
         {"megaton", precise_unit(1000000.0, precise::energy::ton_tnt)},
         {"gigaton", precise_unit(1000000000.0, precise::energy::ton_tnt)},
         {"kton", precise_unit(1000.0, precise::energy::ton_tnt)},
         {"Mton", precise_unit(1000000.0, precise::energy::ton_tnt)},
         {"Gton", precise_unit(1000000000.0, precise::energy::ton_tnt)},
         {"gramoftnt", precise::micro* precise::energy::ton_tnt},
         {"gramsofTNT", precise::micro* precise::energy::ton_tnt},
         {"gramofTNT", precise::micro* precise::energy::ton_tnt},
         {"gramsoftnt", precise::micro* precise::energy::ton_tnt},
         // Gigatonne carbon
         {"C90", precise::conventional::coulomb90},
         {"ohm90", precise::conventional::ohm90},
         {u8"\u03A990", precise::conventional::ohm90},  // Greek Omega
         {u8"\u212690", precise::conventional::ohm90},  // Unicode Ohm symbol
         {"A90", precise::conventional::ampere90},
         {"V90", precise::conventional::volt90},
         {"W90", precise::conventional::watt90},
         {"F90", precise::conventional::farad90},
         {"H90", precise::conventional::henry90},
         {"C_90", precise::conventional::coulomb90},
         {"ohm_90", precise::conventional::ohm90},
         {u8"\u03A9_90", precise::conventional::ohm90},  // Greek Omega
         {u8"\u2126_90", precise::conventional::ohm90},  // Unicode Ohm symbol
         {"A_90", precise::conventional::ampere90},
         {"V_90", precise::conventional::volt90},
         {"W_90", precise::conventional::watt90},
         {"F_90", precise::conventional::farad90},
         {"H_90", precise::conventional::henry90},
         {"coulomb90", precise::conventional::coulomb90},
         {"ampere90", precise::conventional::ampere90},
         {"amp90", precise::conventional::ampere90},
         {"volt90", precise::conventional::volt90},
         {"watt90", precise::conventional::watt90},
         {"farad90", precise::conventional::farad90},
         {"henry90", precise::conventional::henry90},
         {"boe", precise::energy::boe},
         {"barrelofoilequivalent", precise::energy::boe},
         {"barrelsofoilequivalent", precise::energy::boe},
         {"tce", precise::energy::tce},
         {"tonneofcoalequivalent", precise::energy::tce},
         {"lge", precise::energy::lge},
         {"Lge", precise::energy::lge},
         {"literofgasolineequivalent", precise::energy::lge},
         {"foeb", precise::energy::foeb},
         {"fueloilequivalentof1barrel", precise::energy::foeb},

         {"CFM", precise::other::CFM},
         {"CFS", precise::ft.pow(3) / precise::s},
         {"cfm", precise::other::CFM},
         {"cfs", precise::ft.pow(3) / precise::s},
         {"cubicfeetperminute", precise::other::CFM},
         {"gps", precise::gal / precise::s},
         {"gpm", precise::gal / precise::min},
         {"gph", precise::gal / precise::hr},
         {"att", precise::pressure::att},
         {"at", precise::pressure::att},
         {"technicalatmosphere", precise::pressure::att},
         {"atmospheretechnical", precise::pressure::att},
         {"techatmosphere", precise::pressure::att},
         {"techatm", precise::pressure::att},
         {"ATT", precise::pressure::att},
         {"poundal", precise::av::poundal},
         {"pdl", precise::av::poundal},
         {"psi", precise::pressure::psi},
         {"psig", precise::pressure::psig},
         {"ksi", precise_unit(1000.0, precise::pressure::psi)},
         {"psia", precise::pressure::psi},
         {"poundpersquareinch",
          precise::pressure::psi},  // pressure not areal density
         {"poundspersquareinch", precise::pressure::psi},
         {"pound/squareinch",
          precise::pressure::psi},  // pressure not areal density
         {"pounds/squareinch", precise::pressure::psi},
         {"[PSI]", precise::pressure::psi},
         {"[psi]", precise::pressure::psi},
         {"inHg", precise::pressure::inHg},
         {"in_Hg", precise::pressure::inHg},
         {"inchHg", precise::pressure::inHg},
         {"ftH2O", precise_unit(12.0, precise::pressure::inH2O)},
         {"footwater", precise_unit(12.0, precise::pressure::inH2O)},
         {"inH2O", precise::pressure::inH2O},
         {"in_H2O", precise::pressure::inH2O},
         {"inchH2O", precise::pressure::inH2O},
         {"inAq", precise::pressure::inH2O},
         {"in_Aq", precise::pressure::inH2O},
         {"in[Hg]", precise::pressure::inHg},
         {"in[H2O]", precise::pressure::inH2O},
         {"IN[HG]", precise::pressure::inHg},
         {"IN[H2O]", precise::pressure::inH2O},
         {"[in_i'Hg]", precise::pressure::inHg},
         {"[IN_I'HG]", precise::pressure::inHg},
         {"inchofmercury", precise::pressure::inHg},
         {"inchesofmercury", precise::pressure::inHg},
         {"inchmercury", precise::pressure::inHg},
         {"inch{mercury}", precise::pressure::inHg},
         {"inchofmercury_i", precise::pressure::inHg},
         {"inchofmercurycolumn", precise::pressure::inHg},
         {"inchesofmercurycolumn", precise::pressure::inHg},
         {"[in_i'H2O]", precise::pressure::inH2O},
         {"[IN_I'H2O]", precise::pressure::inH2O},
         {"inchwater", precise::pressure::inH2O},
         {"inch(international)ofwater", precise::pressure::inH2O},
         {"inchofwater", precise::pressure::inH2O},
         {"inchesofwater", precise::pressure::inH2O},
         {"inchofwaterguage", precise::pressure::inH2O},
         {"inchofwater_i", precise::pressure::inH2O},
         {"inch{water}", precise::pressure::inH2O},
         {"inchofwatercolumn", precise::pressure::inH2O},
         {"inchesofwatercolumn", precise::pressure::inH2O},
         {"iwg", precise::pressure::inH2O},
         // cgs unit of electric quadrupole
         {"buckingham",
          {1e-26, precise::cgs::statC_charge* precise::cm.pow(2)}},
         {"landauer", {0.0175, precise::energy::eV / precise::data::bit}},
         {"quad", precise::energy::quad},
         {"therm", precise::energy::therm_ec},
         {"thm", precise::energy::therm_ec},
         {"therm(EC)", precise::energy::therm_ec},
         {"thm_ec", precise::energy::therm_ec},
         {"therm_ec", precise::energy::therm_ec},
         {"therm_us", precise::energy::therm_us},
         {"thm_us", precise::energy::therm_us},
         {"therm_br", precise::energy::therm_br},
         {"thermie", precise::MTS::thermie},
         {"gal", precise::gal},
         {"gal_us", precise::us::gallon},
         {"[GAL_US]", precise::us::gallon},
         {"gallon_us", precise::us::gallon},
         {"liquidgallon_us", precise::us::gallon},
         {"gal_wi", precise::us::dry::gallon},
         {"wigal", precise::us::dry::gallon},
         {"[GAL_WI]", precise::us::dry::gallon},
         {"drygallon_us", precise::us::dry::gallon},
         {"gallon-historical", precise::us::dry::gallon},
         {"firkin", precise_unit(9.0, precise::us::dry::gallon)},
         {"rundlet", precise_unit{18.0, precise::imp::gallon}},
         {"rundlet_wi", precise_unit{18.0, precise::imp::gallon}},
         {"tierce", precise_unit{42.0, precise::imp::gallon}},
         {"tierce_wi", precise_unit{42.0, precise::imp::gallon}},
         {"tierce_br", precise_unit{42.0, precise::imp::gallon}},
         {"firkin_wi", precise_unit(70.0, precise::imp::gallon)},
         {"butterfirkin", precise_unit(25.40117, precise::kg)},
         {"soapfirkin", precise_unit(25.40117, precise::kg)},
         {"firkin_br", precise_unit(9.0, precise::imp::gallon)},
         {"kilderkin", precise_unit(18.0, precise::imp::gallon)},
         {"kilderkin_wi", precise_unit(18.0, precise::imp::gallon)},
         {"hogshead_wi", precise_unit(52.5, precise::imp::gallon)},
         {"tun_wi", precise_unit(210.0, precise::imp::gallon)},
         {"butt_wi", precise_unit(105.0, precise::imp::gallon)},
         {"barrel_wi", precise_unit(26.25, precise::imp::gallon)},
         {"beerbarrel_br", precise_unit(26.25, precise::imp::gallon)},
         {"bbl", precise::us::barrel},
         {"barrel", precise::us::barrel},
         {"bbl(oil)", precise::us::barrel},
         {"barrel(oil)", precise::us::barrel},
         {"bbl_us", precise::us::barrel},
         {"[bbl_us]", precise::us::barrel},
         {"[BBL_US]", precise::us::barrel},
         {"barrel_us", precise::us::barrel},
         {"keg", precise_unit(15.5, precise::us::gallon)},
         {"fullkeg", precise_unit(15.5, precise::us::gallon)},
         {"importkeg", precise_unit(13.2, precise::us::gallon)},
         {"minikeg", precise_unit(1.32, precise::us::gallon)},
         {"bubba", precise_unit(1.32, precise::us::gallon)},
         {"sixtel", precise_unit(5.16, precise::us::gallon)},
         {"cornykeg", precise_unit(5.16, precise::us::gallon)},
         {"flbarrel_us", precise::us::flbarrel},
         {"puncheon", precise_unit(84.0, precise::us::gallon)},
         {"puncheon_wi", precise_unit(84.0, precise::us::gallon)},
         {"fluidbarrel_us", precise::us::flbarrel},
         {"liquidbarrel_us", precise::us::flbarrel},
         {"flbarrel", precise::us::flbarrel},
         {"fluidbarrel", precise::us::flbarrel},
         {"liquidbarrel", precise::us::flbarrel},
         {"gal_ca", precise::canada::gallon},
         {"gallon_ca", precise::canada::gallon},
         {"tbsp_ca", precise::canada::tbsp},
         {"tsp_ca", precise::canada::tsp},
         {"tablespoon_ca", precise::canada::tbsp},
         {"teaspoon_ca", precise::canada::tsp},
         {"cup_ca", precise::canada::cup},
         {"traditional_cup_ca", precise::canada::cup_trad},
         {"drum", precise::volume::drum},
         {"gallon", precise::gal},
         {"hogshead", precise::us::hogshead},
         {"canada", precise_unit(1.4, precise::L)},
         {"tonregister", precise_unit(100.0, precise::ft.pow(3))},
         {"tonreg", precise_unit(100.0, precise::ft.pow(3))},
         {"registerton", precise_unit(100.0, precise::ft.pow(3))},
         {"registertonne", precise_unit(100.0, precise::ft.pow(3))},
         {"waterton", precise_unit(224.0, precise::imp::gallon)},
         {"lb", precise::lb},
         {"LB", precise::lb},
         {"kip", precise::kilo* precise::lb},
         {"lb_tr", precise::troy::pound},
         {"[LB_TR]", precise::troy::pound},
         {"pound-troy", precise::troy::pound},
         {"pound", precise::lb},
         {"solarmass", constants::planetary::mass::solar.as_unit()},
         {"solarmasses", constants::planetary::mass::solar.as_unit()},
         {"jupitermass", constants::planetary::mass::jupiter.as_unit()},
         {"jupitermasses", constants::planetary::mass::jupiter.as_unit()},
         {"earthmass", constants::planetary::mass::earth.as_unit()},
         {"earthmasses", constants::planetary::mass::earth.as_unit()},
         {"lunarmass", constants::planetary::mass::moon.as_unit()},
         {"lunarmasses", constants::planetary::mass::moon.as_unit()},
         {"bag", precise_unit(96.0, precise::lb)},

         {"tonc", precise::energy::tonc},  // ton cooling
         {"ton(refrigeration)", precise::energy::tonc},  // ton cooling
         {"tonofrefrigeration", precise::energy::tonc},  // ton cooling
         {"tonsofrefrigeration", precise::energy::tonc},  // ton cooling
         {"refrigerationton", precise::energy::tonc},  // ton cooling
         {"ton(cooling)", precise::energy::tonc},  // ton cooling
         {"ton{refrigeration}", precise::energy::tonc},  // ton cooling
         {"ton{cooling}", precise::energy::tonc},  // ton cooling
         {"tonhour", precise::energy::tonhour},
         {"tonhour(refrigeration)", precise::energy::tonhour},
         {"tonhour{refrigeration}", precise::energy::tonhour},
         {"RT", precise::energy::tonc},  // ton cooling
         {"TR", precise::energy::tonc},  // ton cooling
         // {"tons", precise::energy::tonc* precise::s},
         {"tonh", precise::energy::tonc* precise::hr},
         {"bu", precise::us::dry::bushel},
         {"bushel", precise::us::dry::bushel},
         {"oz fl", precise::us::floz},
         {"ozfl", precise::us::floz},
         {"fl oz", precise::us::floz},
         {"floz", precise::us::floz},
         {"floz_us", precise::us::floz},
         {"foz_us", precise::us::floz},
         {"foz", precise::us::floz},
         {"shot", {1.5, precise::us::floz}},
         {"[FOZ_US]", precise::us::floz},
         {"fluidounce", precise::us::floz},
         {"fluidounce_us", precise::us::floz},
         {"fluiddram", precise_unit(1.0 / 8.0, precise::us::floz)},
         {"fluiddram_us", precise_unit(1.0 / 8.0, precise::us::floz)},
         {"liquidounce", precise::us::floz},
         {"liquidounce_us", precise::us::floz},
         {"liquiddram", precise::us::dram},
         {"liquiddram_us", precise::us::dram},
         {"jigger", precise_unit(1.5, precise::us::floz)},
         {"fdr_us", precise::us::dram},
         {"[FDR_US]", precise::us::dram},
         {"min_us", precise::us::minim},
         {"[MIN_US]", precise::us::minim},
         {"minim_us", precise::us::minim},
         {"minim", precise::us::minim},
         {"ouncefl", precise::us::floz},
         {"fluidoz", precise::us::floz},
         {"liquidoz", precise::us::floz},
         {"oz", precise::oz},
         {"OZ", precise::oz},
         {u8"\u2125", precise::oz},
         {"gr", precise::i::grain},
         {"[GR]", precise::i::grain},
         {"grain", precise::i::grain},
         {"oz_av", precise::av::ounce},
         {"oz_i", precise::av::ounce},
         {"[OZ_AV]", precise::av::ounce},
         {"pound_i", precise::av::pound},
         {"pound_av", precise::av::pound},
         {"lb_av", precise::av::pound},
         {"[LB_AV]", precise::av::pound},
         {"dr",
          precise::us::dram},  // this most commonly implies volume vs weight
         {"dram", precise::us::dram},
         {"dr_av", precise::av::dram},
         {"dr_i", precise::av::dram},
         {"dram_av", precise::av::dram},
         {"dram_i", precise::av::dram},
         {"[DR_AV]", precise::av::dram},
         {"drammassunit", precise::av::dram},
         {"scwt", precise::av::hundredweight},
         {"scwt_av", precise::av::hundredweight},
         {"[SCWT_AV]", precise::av::hundredweight},
         {"shorthundredweight", precise::av::hundredweight},
         {"lcwt_av", precise::av::longhundredweight},
         {"[LCWT_AV]", precise::av::longhundredweight},
         {"longhundredweight", precise::av::longhundredweight},
         {"quintal", precise::mass::quintal},
         {"quintal_m", precise::mass::quintal},
         {"cwt", precise::av::hundredweight},
         {"cwt_us", precise::av::hundredweight},
         {"cwt_br", precise::av::longhundredweight},
         {"hundredweight", precise::av::hundredweight},
         {"hundredweight_us", precise::av::hundredweight},
         {"hundredweight_br", precise::av::longhundredweight},
         {"hundredweight(short)", precise::av::hundredweight},
         {"hundredweight(long)", precise::av::longhundredweight},
         {"assayton", precise::mass::ton_assay},
         {"tonassay", precise::mass::ton_assay},
         {"ton(assayUS)", precise::mass::ton_assay},
         {"ton(assay)", precise::mass::ton_assay},
         {"ton(assay)_br", precise::mass::longton_assay},
         {"longassayton", precise::mass::longton_assay},
         {"stone_av", precise::av::stone},
         {"stone_us", precise_unit(0.125, precise::av::hundredweight)},
         {"[STONE_AV]", precise::av::stone},
         {"ston_av", precise::av::ton},
         {"[STON_AV]", precise::av::ton},
         {"shortton", precise::av::ton},
         {"shortton_us", precise::av::ton},
         {"ton_us", precise::av::ton},
         {"ton_av", precise::av::ton},
         {"stone", precise::av::stone},
         {"stone_br", precise::av::stone},
         {"ton", precise::ton},
         {"ton(short)", precise::ton},
         {"longton", precise_unit(2240.0, precise::lb)},
         {"lton_av", precise::av::longton},
         {"[LTON_AV]", precise::av::longton},
         {"longton_av", precise::av::longton},
         {"ton(long)", precise::av::longton},
         {"longton_br", precise::av::longton},
         {"ton_br", precise::av::longton},
         {"oz_tr", precise::troy::oz},
         {"[OZ_TR]", precise::troy::oz},
         {"ounce_tr", precise::troy::oz},
         {"pound_tr", precise::troy::pound},
         {"pwt_tr", precise::troy::pennyweight},
         {"pwt", precise::troy::pennyweight},
         {"[PWT_TR]", precise::troy::pennyweight},
         {"pennyweight_tr", precise::troy::pennyweight},
         {"pennyweight", precise::troy::pennyweight},
         {"dram_tr", precise_unit{0.00388793458, precise::kg}},
         {"sc_ap", precise::apothecaries::scruple},
         {"[SC_AP]", precise::apothecaries::scruple},
         {"scruple", precise::apothecaries::scruple},
         {"scruple_ap", precise::apothecaries::scruple},
         {u8"\u2108", precise::apothecaries::scruple},
         {"dr_ap", precise::apothecaries::drachm},
         {u8"\u0292", precise::apothecaries::drachm},
         {"dram_ap", precise::apothecaries::drachm},
         {"[DR_AP]", precise::apothecaries::drachm},
         {"oz_ap", precise::apothecaries::ounce},
         {"[OZ_AP]", precise::apothecaries::ounce},
         {"ounce_ap", precise::apothecaries::ounce},
         {"lb_ap", precise::apothecaries::pound},
         {"lbsap", precise::apothecaries::pound},
         {"[LB_AP]", precise::apothecaries::pound},
         {"pound_ap", precise::apothecaries::pound},
         {"oz_m", precise::apothecaries::metric_ounce},
         {"[OZ_M]", precise::apothecaries::metric_ounce},
         {"ounce_m", precise::apothecaries::metric_ounce},
         {"car_Au", precise_unit{1.0 / 24.0, precise::one, commodities::gold}},
         {"carau", precise_unit{1.0 / 24.0, precise::one, commodities::gold}},
         {"[CAR_AU]",
          precise_unit{1.0 / 24.0, precise::one, commodities::gold}},
         {"caratofgoldalloys",
          precise_unit{1.0 / 24.0, precise::one, commodities::gold}},
         {"ounce", precise::oz},
         {"ounce_av", precise::av::ounce},
         {"ounce_i", precise::av::ounce},
         // this is probably more common than kilopoise
         {"kipf", precise::kilo* precise::lbf},
         {"kipforce", precise::kilo* precise::lbf},
         {"tonforce(long)", precise::av::longton* constants::g0.as_unit()},
         {"tonforce(short)", precise::av::ton* constants::g0.as_unit()},
         {"tonforce_us", precise::av::ton* constants::g0.as_unit()},
         {"tonforce_br", precise::av::longton* constants::g0.as_unit()},
         {"tonf_us", precise::av::ton* constants::g0.as_unit()},
         {"tonf_br", precise::av::longton* constants::g0.as_unit()},
         {"sn", precise::MTS::sthene},
         {"sthene", precise::MTS::sthene},
         {"pz", precise::MTS::pieze},
         {"pieze", precise::MTS::pieze},
         {"lbf_av", precise::av::lbf},
         {"lbf_us", precise::av::lbf},
         {"[LBF_AV]", precise::av::lbf},
         {"lbf", precise::lbf},
         {"ozf", precise::av::ozf},
         {"poundforce_us", precise::lbf},
         // the next four are energy even though the breakdown would be power
         {"foot-pound", precise::ft* precise::lbf},
         {"foot-pound of energy", precise::ft* precise::lbf},
         {"foot-pound of torque", precise::ft* precise::lbf},
         {"footpound", precise::ft* precise::lbf},
         {"footpoundforce", precise::ft* precise::lbf},
         {"foot-pounds of energy", precise::ft* precise::lbf},
         {"foot-pounds of torque", precise::ft* precise::lbf},
         {"ft-lb", precise::ft* precise::lbf},
         {"ftlb", precise::ft* precise::lbf},
         {"slug", precise::av::slug},
         // {"kt", precise_unit(200.0, precise::mg)},
         {"cup", precise::us::cup},
         {"cup_us", precise::us::cup},
         {"[CUP_US]", precise::us::cup},
         {"cupUScustomary", precise::us::cup},
         {"sticksofbutter", {0.5, precise::us::cup}},
         {"tsp", precise::us::tsp},
         {"tsp_us", precise::us::tsp},
         {"[TSP_US]", precise::us::tsp},
         {"teaspoon", precise::us::tsp},
         {"teaspoon_us", precise::us::tsp},
         {"tbsp", precise::us::tbsp},
         {"tblsp", precise::us::tbsp},
         {"tbl", precise::us::tbsp},
         {"tbs", precise::us::tbsp},
         {"tbs_us", precise::us::tbsp},
         {"[TBS_US]", precise::us::tbsp},
         {"tablespoon", precise::us::tbsp},
         {"tablespoon_us", precise::us::tbsp},
         {"smidgen", precise_unit(1.0 / 32.0, precise::us::tsp)},
         {"smdg", precise_unit(1.0 / 32.0, precise::us::tsp)},
         {"pinch", precise_unit(1.0 / 24.0, precise::us::tsp)},
         {"pn", precise_unit(1.0 / 24.0, precise::us::tsp)},
         {"dash", precise_unit(1.0 / 16.0, precise::us::tsp)},
         {"tad", precise_unit(1.0 / 8.0, precise::us::tsp)},
         {"saltspoon", precise_unit(1.0 / 4.0, precise::us::tsp)},
         {"ssp", precise_unit(1.0 / 4.0, precise::us::tsp)},
         {"coffeespoon", precise_unit(1.0 / 2.0, precise::us::tsp)},
         {"csp", precise_unit(1.0 / 2.0, precise::us::tsp)},
         {"dessertspoon", precise_unit(2.0, precise::us::tsp)},
         {"dsp", precise_unit(2.0, precise::us::tsp)},
         {"dssp", precise_unit(2.0, precise::us::tsp)},
         {"dstspn", precise_unit(2.0, precise::us::tsp)},
         {"glass_wi", precise_unit(1.0 / 4.0, precise::us::cup)},
         {"wgf", precise_unit(1.0 / 4.0, precise::us::cup)},
         {"teacup", precise_unit(1.0 / 2.0, precise::us::cup)},
         {"tcf", precise_unit(1.0 / 2.0, precise::us::cup)},
         {"cup_m", precise::metric::cup},
         {"[CUP_M]", precise::metric::cup_uslegal},
         {"[cup_m]", precise::metric::cup_uslegal},  // ucum definitions I think
                                                     // it is wrong there
         {"cupUSlegal", precise::metric::cup_uslegal},
         {"tsp_m", precise::metric::tsp},
         {"[TSP_M]", precise::metric::tsp},
         {"teaspoon-metric", precise::metric::tsp},
         {"teaspoon_m", precise::metric::tsp},
         {"tbs_m", precise::metric::tbsp},
         {"tbsm", precise::metric::tbsp},
         {"[TBS_M]", precise::metric::tbsp},
         {"tablespoon-metric", precise::metric::tbsp},
         {"tablespoon_m", precise::metric::tbsp},
         {"foz_m", precise::metric::floz},
         {"[FOZ_M]", precise::metric::floz},
         {"fluidounce-metric", precise::metric::floz},
         {"fluidounce_m", precise::metric::floz},
         {"liquidounce-metric", precise::metric::floz},
         {"liquidounce_m", precise::metric::floz},
         {"quart", precise::us::quart},
         {"qt", precise::us::quart},
         {"QT", precise::us::quart},
         {"qt_us", precise::us::quart},
         {"[QT_US]", precise::us::quart},
         {"quart_us", precise::us::quart},
         {"pt", precise::us::pint},
         {"pint", precise::us::pint},
         {"pint_us", precise::us::pint},
         {"pt_us", precise::us::pint},
         {"[PT_US]", precise::us::pint},
         {"gill", precise::us::gill},
         {"gi", precise::us::gill},
         {"gill_us", precise::us::gill},
         {"gil_us", precise::us::gill},
         {"[GIL_US]", precise::us::gill},
         {"pk", precise::us::dry::peck},
         {"pk_us", precise::us::dry::peck},
         {"[PK_US]", precise::us::dry::peck},
         {"peck", precise::us::dry::peck},
         {"peck_us", precise::us::dry::peck},
         {"bu_us", precise::us::dry::bushel},
         {"bsh_us", precise::us::dry::bushel},
         {"[BU_US]", precise::us::dry::bushel},
         {"bushel_us", precise::us::dry::bushel},
         {"dqt", precise::us::dry::quart},
         {"dqt_us", precise::us::dry::quart},
         {"dryqt", precise::us::dry::quart},
         {"dryqt_us", precise::us::dry::quart},
         {"[DQT_US]", precise::us::dry::quart},
         {"dryquart_us", precise::us::dry::quart},
         {"dryquart", precise::us::dry::quart},
         {"dpt_us", precise::us::dry::pint},
         {"[DPT_US]", precise::us::dry::pint},
         {"drypint_us", precise::us::dry::pint},
         {"drypint", precise::us::dry::pint},
         {"drypt", precise::us::dry::pint},
         {"qt_br", precise::imp::quart},
         {"[QT_BR]", precise::imp::quart},
         {"quart_br", precise::imp::quart},
         {"pt_br", precise::imp::pint},
         {"[PT_BR]", precise::imp::pint},
         {"pint_br", precise::imp::pint},
         {"gil_br", precise::imp::gill},
         {"[GIL_BR]", precise::imp::gill},
         {"gill_br", precise::imp::gill},
         {"gi_br", precise::imp::gill},
         {"pk_br", precise::imp::peck},
         {"[PK_BR]", precise::imp::peck},
         {"peck_br", precise::imp::peck},
         {"bu_br", precise::imp::bushel},
         {"bsh_br", precise::imp::bushel},
         {"[BU_BR]", precise::imp::bushel},
         {"bushel_br", precise::imp::bushel},
         {"foz_br", precise::imp::floz},
         {"floz_br", precise::imp::floz},
         {"[FOZ_BR]", precise::imp::floz},
         {"fluidounce_br", precise::imp::floz},
         {"liquidounce_br", precise::imp::floz},
         {"fdr_br", precise::imp::dram},
         {"[FDR_BR]", precise::imp::dram},
         {"fluiddram_br", precise::imp::dram},
         {"liquiddram_br", precise::imp::dram},
         {"min_br", precise::imp::minim},
         {"[MIN_BR]", precise::imp::minim},
         {"minim_br", precise::imp::minim},
         {"barrel_br", precise::imp::barrel},
         {"bbl_br", precise::imp::barrel},
         {"tablespoon_br", precise::imp::tbsp},
         {"teaspoon_br", precise::imp::tsp},
         {"dessertspoon_br", precise_unit(2.0, precise::imp::tsp)},
         {"tbsp_br", precise::imp::tbsp},
         {"tsp_br", precise::imp::tsp},
         {"cup_br", precise::imp::cup},
         {"[lne]", precise::typographic::dtp::line},
         {"[LNE]", precise::typographic::dtp::line},
         {"line", precise::typographic::dtp::line},
         {"line_us", precise::typographic::american::line},
         {"point_us", precise::typographic::american::point},
         {"pica_us", precise::typographic::american::pica},
         {"twip_us", precise::typographic::american::twip},
         {"line_br", precise::typographic::american::line},
         {"pnt", precise::typographic::dtp::point},
         {"[PNT]", precise::typographic::dtp::point},
         {"desktoppublishingpoint", precise::typographic::dtp::point},
         {"pca", precise::typographic::dtp::pica},
         {"[PCA]", precise::typographic::dtp::pica},
         {"dtppica", precise::typographic::dtp::pica},
         {"dtptwip", precise::typographic::dtp::twip},
         {"dtppoint", precise::typographic::dtp::point},
         {"point", precise::typographic::dtp::point},
         {"pica", precise::typographic::dtp::pica},
         {"twip", precise::typographic::dtp::twip},
         {"printer'spoint", precise::typographic::printers::point},
         {"printerspoint", precise::typographic::printers::point},
         {"pointprinter", precise::typographic::printers::point},
         {"pnt_pr", precise::typographic::printers::point},
         {"pntpr", precise::typographic::printers::point},
         {"[PNT_PR]", precise::typographic::printers::point},
         {"printer'spica", precise::typographic::printers::pica},
         {"printerspica", precise::typographic::printers::pica},
         {"picaprinter", precise::typographic::printers::pica},
         {"ATApica", precise::typographic::printers::pica},
         {"atapica", precise::typographic::printers::pica},
         {"picaata", precise::typographic::printers::pica},
         {"ATApoint", precise::typographic::printers::point},
         {"atapoint", precise::typographic::printers::point},
         {"pointata", precise::typographic::printers::point},
         {"pca_pr", precise::typographic::printers::pica},
         {"pcapr", precise::typographic::printers::pica},
         {"[PCA_PR]", precise::typographic::printers::pica},
         {"INpoint", precise::typographic::IN::point},
         {"INpica", precise::typographic::IN::pica},
         {"INPoint", precise::typographic::IN::point},
         {"INPica", precise::typographic::IN::pica},
         {"pied", precise::typographic::french::pied},
         {"[PIED]", precise::typographic::french::pied},
         {"pouce", precise::typographic::french::pouce},
         {"[POUCE]", precise::typographic::french::pouce},
         {"ligne", precise::typographic::french::ligne},
         {"[LIGNE]", precise::typographic::french::ligne},
         {"didot", precise::typographic::french::didot},
         {"didotpoint", precise::typographic::french::didot},
         {"[DIDOT]", precise::typographic::french::didot},
         {"cicero", precise::typographic::french::cicero},
         {"[CICERO]", precise::typographic::french::cicero},
         {"postscriptpoint", precise::typographic::postscript::point},
         {"pointpostscript", precise::typographic::postscript::point},
         {"postscriptpica", precise::typographic::postscript::pica},
         {"picapostscript", precise::typographic::postscript::pica},
         {"texpoint", precise::typographic::tex::point},
         {"pointtex", precise::typographic::tex::point},
         {"texpica", precise::typographic::tex::pica},
         {"picatex", precise::typographic::tex::pica},
         {"truchetpoint", {0.188, precise::mm}},
         {"truchetpica", {0.188 * 12, precise::mm}},
         {"tex", precise::textile::tex},
         {"texes", precise::textile::tex},
         {"TEX", precise::textile::tex},
         {"denier", precise::textile::denier},
         {"denier(linearmassdensity)", precise::textile::denier},
         {"den", precise::textile::denier},
         {"[DEN]", precise::textile::denier},
         {"span(cloth)", precise::textile::span},
         {"finger(cloth)", precise::textile::finger},
         {"nail(cloth)", precise::textile::nail},
         {"nail", precise::textile::nail},
         {"span", precise::textile::span},
         {"finger", precise::textile::finger},
         {"smoot", precise::distance::smoot},
         {"[SMOOT]", precise::distance::smoot},
         // potrzebie system developed by Dononld Knuth
         // https://polaris93.livejournal.com/2046896.html
         {"potrzebie", precise_unit(2.2633484517438173216473, precise::mm)},
         {"blintz", precise_unit(0.0364253863, precise::kg)},
         {"blintzes", precise_unit(0.0364253863, precise::kg)},
         {"ngogn", precise_unit(22.633484517438173216473, precise::mm).pow(3)},
         {"furshlugginer", precise_unit(1000.0, precise::one)},
         {"farshimmelt", precise_unit(0.000001, precise::one)},
         {"clarke", precise::day},

         {"[hp_X]",
          precise_unit(1.0, precise::log::neglog10, commodities::Hahnemann)},
         {"[HP_X]",
          precise_unit(1.0, precise::log::neglog10, commodities::Hahnemann)},
         {"[hp_C]",
          precise_unit(1.0, precise::log::neglog100, commodities::Hahnemann)},
         {"[HP_C]",
          precise_unit(1.0, precise::log::neglog100, commodities::Hahnemann)},
         {"[hp_M]",
          precise_unit(1.0, precise::log::neglog1000, commodities::Hahnemann)},
         {"[HP_M]",
          precise_unit(1.0, precise::log::neglog1000, commodities::Hahnemann)},
         {"[hp_Q]",
          precise_unit(1.0, precise::log::neglog50000, commodities::Hahnemann)},
         {"[HP_Q]",
          precise_unit(1.0, precise::log::neglog50000, commodities::Hahnemann)},
         {"[kp_X]",
          precise_unit(1.0, precise::log::neglog10, commodities::Korsakov)},
         {"[KP_X]",
          precise_unit(1.0, precise::log::neglog10, commodities::Korsakov)},
         {"[kp_C]",
          precise_unit(1.0, precise::log::neglog100, commodities::Korsakov)},
         {"[KP_C]",
          precise_unit(1.0, precise::log::neglog100, commodities::Korsakov)},
         {"[kp_M]",
          precise_unit(1.0, precise::log::neglog1000, commodities::Korsakov)},
         {"[KP_M]",
          precise_unit(1.0, precise::log::neglog1000, commodities::Korsakov)},
         {"[kp_Q]",
          precise_unit(1.0, precise::log::neglog50000, commodities::Korsakov)},
         {"[KP_Q]",
          precise_unit(1.0, precise::log::neglog50000, commodities::Korsakov)},
         {"shu",
          precise_unit(1.0 / 16, precise::other::ppm, commodities::capsaicin)},
         {"scoville",
          precise_unit(1.0 / 16, precise::other::ppm, commodities::capsaicin)},
         {"scovilleunit",
          precise_unit(1.0 / 16, precise::other::ppm, commodities::capsaicin)},
         {"scovilleheatunit",
          precise_unit(
              1.0 / 16,
              precise::other::ppm,
              commodities::capsaicin)}}};

#if !defined(UNITS_DISABLE_NON_ENGLISH_UNITS) ||                               \
    UNITS_DISABLE_NON_ENGLISH_UNITS == 0
UNITS_CPP14_CONSTEXPR_OBJECT
std::array<std::pair<const char*, precise_unit>, 61>
    defined_unit_strings_non_english{{
        {"ken", precise::japan::ken},
        {"cun", precise::chinese::cun},
        {"cun(Chinese)",
         precise::chinese::cun},  // interaction with cu for cubic
        {"cun_cn", precise::chinese::cun},
        {"chi", precise::chinese::chi},
        {"chi_cn", precise::chinese::chi},
        {"zhang", precise::chinese::zhang},
        {"zhang_cn", precise::chinese::zhang},
        {"li_cn", precise::chinese::li},
        {"jin_cn", precise::chinese::jin},
        {"jin", precise::chinese::jin},
        {"liang_cn", precise::chinese::liang},
        {"liang", precise::chinese::liang},
        {"qian_cn", precise::chinese::qian},
        {"qian", precise::chinese::qian},
        {"funt", precise_unit(409.51718, precise::g)},  // Russian
        {"pood", precise_unit(40.0 * 409.51718, precise::g)},  // Russian
        {"zolotnik", precise_unit(409.51718 / 96.0, precise::g)},  // Russian
        {"halakim", precise_unit(1.0 / 18.0, precise::time::min)},  // Jewish
        {"halek", precise_unit(1.0 / 18.0, precise::time::min)},  // Jewish
        {"lustrum", precise_unit(5.0, precise::time::aj)},  // Roman
        {"lustra", precise_unit(5.0, precise::time::aj)},  // Roman
        // turkey, UAE, levant unit of area
        {"dunam", precise_unit(1000.0, precise::m.pow(2))},
        // taiwan
        {"ping", precise_unit(3.305, precise::m.pow(2))},
        // korea unit of area
        {"pyong", precise_unit(3.305, precise::m.pow(2))},
        // ancient Roman units
        {"amphora", {27.84, precise::L}},
        {"sextarii", {27.84 / 48.0, precise::L}},
        {"libra", {0.722, precise::av::pound}},
        {"uncia", {0.722 / 12.0, precise::av::pound}},
        {"pes", {296.0, precise::mm}},
        {"stadium", {220, precise::yd}},
        {"stadion", {157.0, precise::m}},
        {"itinerarystadion", {157.0, precise::m}},
        {"olympicstadion", {157.0, precise::m}},
        {"leuga", {2.22, precise::km}},
        // chinese units
        {"shi", {71.68, precise::kg}},
        {"dan", {71.68, precise::kg}},
        // metric units in other languages
        {"grama", precise::g},
        {"gramm", precise::g},
        {"gramma", precise::g},
        {"gramme", precise::g},
        {"gramo", precise::g},
        {"segundo", precise::s},
        {"seconde", precise::s},
        {"sekunde", precise::s},
        {"secondo", precise::s},
        {"sekund", precise::s},
        {"sekunda", precise::s},
        {"metro", precise::s},
        // older spanish units
        {"punto", {0.1613, precise::mm}},
        {"linea", {1.935, precise::mm}},
        {"pulgada", {23.216, precise::mm}},
        {"pie", {278.6, precise::mm}},
        {"vara", {3.0 * 278.6, precise::mm}},
        {"paso", {5.0 * 278.6, precise::mm}},
        {"milla", {5000.0 * 278.6, precise::mm}},
        {"legua", {15000.0 * 278.6, precise::mm}},
        {"manzana",
         {100.0 * 100.0 * 3.0 * 278.6 * 3.0 * 278.6, precise::mm.pow(2)}},
        {"labor", {1000000.0 * 3.0 * 278.6 * 3.0 * 278.6, precise::mm.pow(2)}},
        {"zenti", {1.0 / 100.0, precise::one}},  // german for centi
        {"zentner", {50.0, precise::kg}},  // german

    }};

#endif

// Mostly from https://en.wikipedia.org/wiki/International_System_of_Units
UNITS_CPP14_CONSTEXPR_OBJECT
std::array<std::pair<const char*, precise_unit>, 245> defined_measurement_types{
    {
        {"", precise::defunit},
        {"arb", precise::defunit},
        {"arbitrary", precise::defunit},
        {"length", precise::m},
        {"len", precise::m},
        {"distance", precise::m},
        {"height", precise::m},
        {"depth", precise::m},
        {"width", precise::m},
        {"diameter", precise::m},
        {"thickness", precise::m},
        {"gauge", precise::m},
        {"invlen", precise::m.inv()},
        {"volume", precise::m.pow(3)},
        {"dryvolume", precise::m.pow(3)},
        {"vol", precise::m.pow(3)},
        {"fluidvolume", precise::L},
        {"time", precise::s},
        {"duration", precise::s},
        {"mass", precise::kg},
        {"weight", precise::N},
        {"current", precise::A},
        {"electriccurrent", precise::A},
        {"magnetomotiveforce", precise::A},
        {"temperature", precise::K},
        {u8"\u2C90", precise::K},
        {u8"\u2129", precise::K},
        {u8"\u03F4", precise::K},
        {u8"\u0398", precise::K},
        {"celsiustemperature", precise::degC},
        {"temp", precise::K},
        {"thermodynamictemperature", precise::K},
        {"thermalconductivity", precise::W / precise::m / precise::K},
        {"amount", precise::mol},
        {"amountofsubstance", precise::mol},
        {"substance", precise::mol},
        {"sub", precise::mol},
        {"luminousintensity", precise::cd},
        {"area", precise::m.pow(2)},
        {"speed", precise::m / precise::s},
        {"velocity", precise::m / precise::s},
        {"vel", precise::m / precise::s},
        {"wavenumber", precise::m.inv()},
        {"acceleration", precise::m / precise::s.pow(2)},
        {"accel", precise::m / precise::s.pow(2)},
        {"density", precise::kg / precise::m.pow(3)},
        {"massdensity", precise::kg / precise::m.pow(3)},
        {"massconcentration", precise::kg / precise::m.pow(3)},
        {"surfacedensity", precise::kg / precise::m.pow(2)},
        {"lineardensity", precise::kg / precise::m},
        {"lineicmass", precise::kg / precise::m},
        {"linearmassdensity", precise::kg / precise::m},
        {"armass", precise::kg / precise::m.pow(2)},
        {"specificvolume", precise::m.pow(3) / precise::kg},
        {"vcnt", precise::L / precise::kg},
        {"currentdensity", precise::A / precise::m.pow(2)},
        {"magneticfieldstrength", precise::A / precise::m},
        {"magneticfieldintensity", precise::A / precise::m},
        {"magnetictension", precise::Pa / precise::m},
        {"concentration", precise::mol / precise::m.pow(3)},
        {"luminance", precise::cd / precise::m.pow(2)},
        {"brightness", precise::cd / precise::m.pow(2)},
        {"lum.intensitydensity", precise::cd / precise::m.pow(2)},
        {"planeangle", precise::rad},
        {"angle", precise::rad},
        {"solidangle", precise::rad.pow(2)},
        {"angular", precise::rad},
        {"frequency", precise::Hz},
        {"freq", precise::Hz},
        {"frequencies", precise::Hz},
        {"force", precise::N},
        {"pressure", precise::Pa},
        {"pres", precise::Pa},
        {"press", precise::Pa},
        {"stress", precise::Pa},
        {"energy", precise::J},
        {"eng", precise::J},
        {"enrg", precise::J},
        {"work", precise::J},
        {"heat", precise::J},
        {"power", precise::W},
        {"powerlevel", precise::W* precise::log::neper},
        {"radiantflux", precise::W},
        {"heatfluxdensity", precise::W / precise::m.pow(2)},
        {"electriccharge", precise::C},
        {"electricfluxdensity", precise::C / precise::m.pow(2)},
        {"charge", precise::C},
        {"electricchargedensity", precise::C / precise::m.pow(3)},
        {"quantityofelectricity", precise::C},
        {"voltage", precise::V},
        {"electricalpotential", precise::V},
        {"potentialdifference", precise::V},
        {"electricpotentialdifference", precise::V},
        {"elpot", precise::V},
        {"electricpotential", precise::V},
        {"electricpotentiallevel", precise::V* precise::log::neper},
        {"logrtoelp", precise::V* precise::log::neper},
        {"emf", precise::V},
        {"electromotiveforce", precise::V},
        {"capacitance", precise::F},
        {"electriccapacitance", precise::F},
        {"resistance", precise::ohm},
        {"impedance", precise::ohm},
        {"reactance", precise::ohm},
        {"conductance", precise::siemens},
        {"electricdipolemoment", precise::C* precise::m},
        {"electricresistance", precise::ohm},
        {"electricresistivity", precise::ohm* precise::m},
        {"electricimpedance", precise::ohm},
        {"electricreactance", precise::ohm},
        {"electricalconductance", precise::siemens},
        {"electricconductance", precise::siemens},
        {"conduction", precise::siemens},
        {"magneticflux", precise::Wb},
        {"fluxofmagneticinduction", precise::Wb},
        {"magneticfluxdensity", precise::T},
        {"magneticinduction", precise::T},
        {"fluxdensity", precise::T},
        {"noisevoltagedensity", precise::V / precise::special::rootHertz},
        {"inductance", precise::H},
        {"induction", precise::H},
        {"luminousflux", precise::lm},
        {"illuminance", precise::lx},
        {"illumination", precise::lx},
        {"luminousemittance", precise::lx},
        {"intensityoflight", precise::lx},
        {"lightintensity", precise::lx},
        {"radioactivity", precise::Bq},
        {"radionuclideactivity", precise::Bq},
        {"radiation", precise::Bq},
        {"activity", precise::Bq},
        {"activities", precise::Bq},
        {"logarithmofpowerratio", precise::log::dB},
        {"absorbeddose", precise::Gy},
        {"ionizingradiation", precise::Gy},
        {"kerma", precise::Gy},
        {"energydose", precise::Gy},
        {"absorbeddoseenergy", precise::Gy},
        {"engcnt", precise::Gy},
        {"equivalentdose", precise::Sv},
        {"doseequivalent", precise::Sv},
        {"radiationdose", precise::Sv},
        {"radiationdosage", precise::Sv},
        {"effectivedose", precise::Sv},
        {"operationaldose", precise::Sv},
        {"committeddose", precise::Sv},
        {"ionizingradiationdose", precise::Gy},
        {"iondose", precise::Gy},
        {"catalyticactivity", precise::kat},
        {"specificenergy", precise::J / precise::kg},
        {"specificheat", precise::J / precise::kg / precise::K},
        {"engcnc", precise::J / precise::m.pow(3)},
        {"momentofforce", precise::N* precise::m},
        {"moment", precise::N* precise::m},
        {"torque", precise::N* precise::m},
        {"angularvelocity", precise::rad / precise::s},
        {"rotationalspeed", precise::rad / precise::s},
        {"rotationalvelocity", precise::rad / precise::s},
        {"angularacceleration", precise::rad / precise::s.pow(2)},
        {"surfacetension", precise::N / precise::m},
        {"electricfield", precise::V / precise::m},
        {"electricfieldstrength", precise::V / precise::m},
        {"permittivity", precise::F / precise::m},
        {"permeability", precise::H / precise::m},
        {"electricpermittivity", precise::F / precise::m},
        {"magneticpermeability", precise::H / precise::m},
        {"exposure", precise::C / precise::kg},
        {"heatcapacity", precise::J / precise::K},
        {"entropy", precise::J / precise::K},
        {"specificentropy", precise::J / precise::kg / precise::K},
        {"dynamicviscosity", precise::Pa* precise::s},
        {"viscosity", precise::Pa* precise::s},
        {"visc", precise::Pa* precise::s},
        {"fluidity", precise::one / precise::Pa / precise::s},
        {"kinematicviscosity", precise::m.pow(2) / precise::s},
        {"radiantintensity", precise::W / precise::rad.pow(2)},
        {"radiance", precise::W / precise::rad.pow(2) / precise::m.pow(2)},
        {"dimless", precise::one},
        {"ratio", precise::one},
        {"fraction", precise::one},
        {"number", precise::one},
        {"nfr", precise::one},
        {"num", precise::one},
        {"age", precise::time::ag},
        {"strain", precise::pu* precise::m},
        {"impulse", precise::N* precise::s},
        {"imp", precise::N* precise::s},
        {"absorbeddoserate", precise::Gy / precise::s},
        {"mcnc", precise::kg / precise::m.pow(3)},
        {"scnc", precise::mol / precise::m.pow(3)},
        {"scnt", precise::mol / precise::kg},
        {"ncnc", precise::one / precise::m.pow(3)},
        {"ncncn", precise::one / precise::m.pow(3)},
        {"ncnt", precise::one / precise::kg},
        {"substanceconcentration", precise::mol / precise::m.pow(3)},
        {"level", precise::log::neper},
        {"logrto", precise::log::neper},
        {"arvol", precise::m.pow(3) / precise::m.pow(2)},
        {"arsub", precise::mol / precise::m.pow(2)},
        {"lineicvolume", precise::m.pow(3) / precise::m},
        {"lineicnumber", precise::one / precise::m},
        {"refraction", precise::one / precise::m},
        {"naric", precise::one / precise::m.pow(2)},
        {"nlen", precise::one / precise::m},
        {"acidity", precise::laboratory::pH},
        {"cact", precise::kat},
        {"magneticfield", precise::T},
        {"magnetic", precise::T},
        {"fluence", precise::one / precise::m.pow(2)},
        {"mcnt", precise::pu* precise::m},
        {"ccnt", precise::kat / precise::kg},
        {"ccnc", precise::kat / precise::L},
        {"acnc", precise::one / precise::L},
        {"velcnc", precise::m / precise::s / precise::L},
        {"mrat", precise::kg / precise::s},
        {"osmol", precise_unit(1.0, precise::mol, commodities::particles)},
        {"massfraction", precise::pu* precise::kg},
        {"mfr", precise::pu* precise::kg},
        {"amplitudespectraldensity", precise::special::ASD},
        {"fluidresistance", precise::Pa* precise::s / precise::m.pow(3)},
        {"signaltransmissionrate", precise::bit / precise::s},
        {"engmass", precise::J / precise::m.pow(3)},
        {"massicenergy", precise::J / precise::m.pow(3)},
        {"energydensity", precise::J / precise::m.pow(3)},
        {"entsub", precise::mol},
        {"mnum", precise::kg},
        {"cmass", precise::kg / precise::kat},
        {"stiffness", precise::N / precise::m},
        {"signalingrate", precise::bit / precise::s},
        {"datarate", precise::bit / precise::s},
        {"elasticity", precise::N / precise::m.pow(2)},
        {"compliance", precise::m / precise::N},
        {"informationcapacity", precise::data::bit},
        {"compli", precise::m / precise::N},
        {"vascularresistance", precise::Pa* precise::s / precise::m.pow(3)},
        {"enzymaticactivity", precise::kat},
        {"catalyticconcentration", precise::kat / precise::m.pow(3)},
        {"molarenergy", precise::J / precise::mol},
        {"molarentropy", precise::J / precise::mol / precise::K},
        {"datatransferrate", precise::bit / precise::s},
        {"flowrate", precise::m.pow(3) / precise::s},
        {"fuelconsumption", precise::km / precise::L},
        {"information", precise::bit},
        {"unitless", precise::one},
        {"numeric", precise::one},
    }};
}  // namespace UNITS_NAMESPACE
