/*
Copyright (c) 2019-2023,
Lawrence Livermore National Security, LLC;
See the top-level NOTICE for additional details. All rights reserved.
SPDX-License-Identifier: BSD-3-Clause
*/
#include "units.hpp"

#include <algorithm>
#include <array>
#include <cstring>
#include <tuple>
#include <unordered_map>

namespace UNITS_NAMESPACE {
using unitD = std::tuple<const char*, const char*, precise_unit>;
static UNITS_CPP14_CONSTEXPR_OBJECT std::array<unitD, 2088> r20_units = {{
    unitD{"05", "lift", precise::one / precise::count},
    unitD{"06", "small spray", precise::one / precise::count},
    unitD{"08", "heat lot", precise::one / precise::count},
    unitD{"10", "group", precise::one / precise::count},
    unitD{"11", "outfit", precise::one / precise::count},
    unitD{"13", "ration", precise::one / precise::count},
    unitD{"14", "shot", precise::one / precise::count},
    unitD{"15", "stick, military", precise::one / precise::count},
    unitD{"16", "hundred fifteen kg drum", precise::one / precise::count},
    unitD{"17", "hundred lb drum", precise::one / precise::count},
    unitD{"18", "fiftyfive gallon (US) drum", precise::one / precise::count},
    unitD{"19", "tank truck", precise::one / precise::count},
    unitD{"1A", "car mile", precise::one / precise::count},
    unitD{"1B", "car count", precise::one / precise::count},
    unitD{"1C", "locomotive count", precise::one / precise::count},
    unitD{"1D", "caboose count", precise::one / precise::count},
    unitD{"1E", "empty car", precise::one / precise::count},
    unitD{"1F", "train mile", precise::one / precise::count},
    unitD{"1G", "fuel usage gallon (US)", precise::us::gallon},
    unitD{"1H", "caboose mile", precise::one / precise::count},
    unitD{"1I", "fixed rate", precise::one / precise::count},
    unitD{"1J", "ton mile", precise::one / precise::count},
    unitD{"1K", "locomotive mile", precise::one / precise::count},
    unitD{"1L", "total car count", precise::one / precise::count},
    unitD{"1M", "total car mile", precise::one / precise::count},
    unitD{"1X", "quarter mile", precise_unit(0.25, precise::mile)},
    unitD{
        "20",
        "twenty foot container",
        precise_unit(20.0 * 8.0 * 8.5, precise::ft.pow(3))},
    unitD{
        "21",
        "forty foot container",
        precise_unit(40.0 * 8.0 * 8.5, precise::ft.pow(3))},
    unitD{"22", "decilitre per gram", precise::one / precise::count},
    unitD{"23", "gram per cubic centimetre", precise::one / precise::count},
    unitD{"24", "theoretical pound", precise::one / precise::count},
    unitD{"25", "gram per square centimetre", precise::one / precise::count},
    unitD{"26", "actual ton", precise::one / precise::count},
    unitD{"27", "theoretical ton", precise::one / precise::count},
    unitD{"28", "kilogram per square metre", precise::one / precise::count},
    unitD{
        "29",
        "pound per thousand square foot",
        precise::one / precise::count},
    unitD{"2A", "radian per second", precise::rad / precise::s},
    unitD{"2B", "radian per second squared", precise::rad / precise::s.pow(2)},
    unitD{"2C", "roentgen", precise::cgs::roentgen},
    unitD{"2G", "volt AC", precise::one / precise::count},
    unitD{"2H", "volt DC", precise::one / precise::count},
    unitD{
        "2I",
        "British thermal unit (international table) per hour",
        precise::one / precise::count},
    unitD{"2J", "cubic centimetre per second", precise::one / precise::count},
    unitD{"2K", "cubic foot per hour", precise::one / precise::count},
    unitD{"2L", "cubic foot per minute", precise::one / precise::count},
    unitD{"2M", "centimetre per second", precise::one / precise::count},
    unitD{"2N", "decibel", precise::one / precise::count},
    unitD{"2P", "kilobyte", precise::one / precise::count},
    unitD{"2Q", "kilobecquerel", precise::one / precise::count},
    unitD{"2R", "kilocurie", precise::one / precise::count},
    unitD{"2U", "megagram", precise::one / precise::count},
    unitD{"2V", "megagram per hour", precise::one / precise::count},
    unitD{"2W", "bin", precise::one / precise::count},
    unitD{"2X", "metre per minute", precise::one / precise::count},
    unitD{"2Y", "milliroentgen", precise::one / precise::count},
    unitD{"2Z", "millivolt", precise::one / precise::count},
    unitD{
        "30",
        "horse power day per air dry metric ton",
        precise::one / precise::count},
    unitD{"31", "catch weight", precise::one / precise::count},
    unitD{
        "32",
        "kilogram per air dry metric ton",
        precise::one / precise::count},
    unitD{
        "33",
        "kilopascal square metre per gram",
        precise::one / precise::count},
    unitD{"34", "kilopascal per millimetre", precise::one / precise::count},
    unitD{
        "35",
        "millilitre per square centimetre second",
        precise::one / precise::count},
    unitD{
        "36",
        "cubic foot per minute per square foot",
        precise::one / precise::count},
    unitD{"37", "ounce per square foot", precise::one / precise::count},
    unitD{
        "38",
        "ounce per square foot per 0,01inch",
        precise::one / precise::count},
    unitD{"3B", "megajoule", precise::one / precise::count},
    unitD{"3C", "manmonth", precise::one / precise::count},
    unitD{"3E", "pound per pound of product", precise::one / precise::count},
    unitD{"3G", "pound per piece of product", precise::one / precise::count},
    unitD{
        "3H",
        "kilogram per kilogram of product",
        precise::one / precise::count},
    unitD{"3I", "kilogram per piece of product", precise::one / precise::count},
    unitD{"40", "millilitre per second", precise::one / precise::count},
    unitD{"41", "millilitre per minute", precise::one / precise::count},
    unitD{"43", "super bulk bag", precise::one / precise::count},
    unitD{"44", "fivehundred kg bulk bag", precise::one / precise::count},
    unitD{"45", "threehundred kg bulk bag", precise::one / precise::count},
    unitD{"46", "fifty lb bulk bag", precise::one / precise::count},
    unitD{"47", "fifty lb bag", precise::one / precise::count},
    unitD{"48", "bulk car load", precise::one / precise::count},
    unitD{"4A", "bobbin", precise::one / precise::count},
    unitD{"4B", "cap", precise::one / precise::count},
    unitD{"4C", "centistokes", precise::one / precise::count},
    unitD{"4E", "twenty pack", precise::one / precise::count},
    unitD{"4G", "microlitre", precise::one / precise::count},
    unitD{"4H", "micrometre (micron)", precise::one / precise::count},
    unitD{"4K", "milliampere", precise::one / precise::count},
    unitD{"4L", "megabyte", precise::one / precise::count},
    unitD{"4M", "milligram per hour", precise::one / precise::count},
    unitD{"4N", "megabecquerel", precise::one / precise::count},
    unitD{"4O", "microfarad", precise::one / precise::count},
    unitD{"4P", "newton per metre", precise::one / precise::count},
    unitD{"4Q", "ounce inch", precise::one / precise::count},
    unitD{"4R", "ounce foot", precise::one / precise::count},
    unitD{"4T", "picofarad", precise::one / precise::count},
    unitD{"4U", "pound per hour", precise::one / precise::count},
    unitD{"4W", "ton (US) per hour", precise::one / precise::count},
    unitD{"4X", "kilolitre per hour", precise::one / precise::count},
    unitD{"53", "theoretical kilogram", precise::one / precise::count},
    unitD{"54", "theoretical tonne", precise::one / precise::count},
    unitD{"56", "sitas", precise::one / precise::count},
    unitD{"57", "mesh", precise::one / precise::count},
    unitD{"58", "net kilogram", precise::one / precise::count},
    unitD{"59", "part per million", precise::one / precise::count},
    unitD{"5A", "barrel (US) per minute", precise::one / precise::count},
    unitD{"5B", "batch", precise::one / precise::count},
    unitD{"5C", "gallon(US) per thousand", precise::one / precise::count},
    unitD{"5E", "MMSCF/day", precise::one / precise::count},
    unitD{"5F", "pound per thousand", precise::one / precise::count},
    unitD{"5G", "pump", precise::one / precise::count},
    unitD{"5H", "stage", precise::one / precise::count},
    unitD{"5I", "standard cubic foot", precise::one / precise::count},
    unitD{"5J", "hydraulic horse power", precise::one / precise::count},
    unitD{"5K", "count per minute", precise::one / precise::count},
    unitD{"5P", "seismic level", precise::one / precise::count},
    unitD{"5Q", "seismic line", precise::one / precise::count},
    unitD{"60", "percent weight", precise::one / precise::count},
    unitD{"61", "part per billion (US)", precise::one / precise::count},
    unitD{"62", "percent per 1000 hour", precise::one / precise::count},
    unitD{"63", "failure rate in time", precise::one / precise::count},
    unitD{"64", "pound per square inch, gauge", precise::one / precise::count},
    unitD{"66", "oersted", precise::one / precise::count},
    unitD{"69", "test specific scale", precise::one / precise::count},
    unitD{"71", "volt ampere per pound", precise::one / precise::count},
    unitD{"72", "watt per pound", precise::one / precise::count},
    unitD{"73", "ampere tum per centimetre", precise::one / precise::count},
    unitD{"74", "millipascal", precise::one / precise::count},
    unitD{"76", "gauss", precise::one / precise::count},
    unitD{"77", "milli-inch", precise::one / precise::count},
    unitD{"78", "kilogauss", precise::one / precise::count},
    unitD{
        "80",
        "pound per square inch absolute",
        precise::one / precise::count},
    unitD{"81", "henry", precise::one / precise::count},
    unitD{
        "84",
        "kilopound-force per square inch",
        precise::one / precise::count},
    unitD{"85", "foot pound-force", precise::one / precise::count},
    unitD{"87", "pound per cubic foot", precise::one / precise::count},
    unitD{"89", "poise", precise::one / precise::count},
    unitD{"90", "Saybold universal second", precise::one / precise::count},
    unitD{"91", "stokes", precise::one / precise::count},
    unitD{"92", "calorie per cubic centimetre", precise::one / precise::count},
    unitD{"93", "calorie per gram", precise::one / precise::count},
    unitD{"94", "curl unit", precise::one / precise::count},
    unitD{
        "95",
        "twenty thousand gallon (US) tankcar",
        precise::one / precise::count},
    unitD{
        "96",
        "ten thousand gallon (US) tankcar",
        precise::one / precise::count},
    unitD{"97", "ten kg drum", precise::one / precise::count},
    unitD{"98", "fifteen kg drum", precise::one / precise::count},
    unitD{"A1", "15 degC calorie", precise::one / precise::count},
    unitD{
        "A10",
        "ampere square metre per joule second",
        precise::one / precise::count},
    unitD{"A11", "angstrom", precise::one / precise::count},
    unitD{"A12", "astronomical unit", precise::one / precise::count},
    unitD{"A13", "attojoule", precise::one / precise::count},
    unitD{"A14", "barn", precise::one / precise::count},
    unitD{"A15", "barn per electronvolt", precise::one / precise::count},
    unitD{
        "A16",
        "barn per steradian electronvolt",
        precise::one / precise::count},
    unitD{"A17", "barn per steradian", precise::one / precise::count},
    unitD{"A18", "becquerel per kilogram", precise::one / precise::count},
    unitD{"A19", "becquerel per cubic metre", precise::one / precise::count},
    unitD{"A2", "ampere per centimetre", precise::one / precise::count},
    unitD{
        "A20",
        "British thermal unit (international table) per second square foot degree Rankine",
        precise::one / precise::count},
    unitD{
        "A21",
        "British thermal unit (international table) per pound degree Rankine",
        precise::one / precise::count},
    unitD{
        "A22",
        "British thermal unit (international table) per second foot degree Rankine",
        precise::one / precise::count},
    unitD{
        "A23",
        "British thermal unit (international table) per hour square foot degree Rankine",
        precise::one / precise::count},
    unitD{"A24", "candela per square metre", precise::one / precise::count},
    unitD{"A25", "cheval vapeur", precise::one / precise::count},
    unitD{"A26", "coulomb metre", precise::one / precise::count},
    unitD{
        "A27",
        "coulomb metre squared per volt",
        precise::one / precise::count},
    unitD{"A28", "coulomb per cubic centimetre", precise::one / precise::count},
    unitD{"A29", "coulomb per cubic metre", precise::one / precise::count},
    unitD{"A3", "ampere per millimetre", precise::one / precise::count},
    unitD{"A30", "coulomb per cubic millimetre", precise::one / precise::count},
    unitD{"A31", "coulomb per kilogram second", precise::one / precise::count},
    unitD{"A32", "coulomb per mole", precise::one / precise::count},
    unitD{
        "A33",
        "coulomb per square centimetre",
        precise::one / precise::count},
    unitD{"A34", "coulomb per square metre", precise::one / precise::count},
    unitD{
        "A35",
        "coulomb per square millimetre",
        precise::one / precise::count},
    unitD{"A36", "cubic centimetre per mole", precise::one / precise::count},
    unitD{"A37", "cubic decimetre per mole", precise::one / precise::count},
    unitD{"A38", "cubic metre per coulomb", precise::one / precise::count},
    unitD{"A39", "cubic metre per kilogram", precise::one / precise::count},
    unitD{"A4", "ampere per square centimetre", precise::one / precise::count},
    unitD{"A40", "cubic metre per mole", precise::one / precise::count},
    unitD{"A41", "ampere per square metre", precise::one / precise::count},
    unitD{"A42", "curie per kilogram", precise::one / precise::count},
    unitD{"A43", "deadweight tonnage", precise::one / precise::count},
    unitD{"A44", "decalitre", precise::one / precise::count},
    unitD{"A45", "decametre", precise::one / precise::count},
    unitD{"A47", "decitex", precise::one / precise::count},
    unitD{"A48", "degree Rankine", precise::one / precise::count},
    unitD{"A49", "denier", precise::one / precise::count},
    unitD{"A5", "ampere square metre", precise::one / precise::count},
    unitD{
        "A50",
        "dyne second per cubic centimetre",
        precise::one / precise::count},
    unitD{"A51", "dyne second per centimetre", precise::one / precise::count},
    unitD{
        "A52",
        "dyne second per centimetre to the fifth power",
        precise::one / precise::count},
    unitD{"A53", "electronvolt", precise::one / precise::count},
    unitD{"A54", "electronvolt per metre", precise::one / precise::count},
    unitD{"A55", "electronvolt square metre", precise::one / precise::count},
    unitD{
        "A56",
        "electronvolt square metre per kilogram",
        precise::one / precise::count},
    unitD{"A57", "erg", precise::one / precise::count},
    unitD{"A58", "erg per centimetre", precise::one / precise::count},
    unitD{"A59", "8-part cloud cover", precise::one / precise::count},
    unitD{
        "A6",
        "ampere per square metre kelvin squared",
        precise::one / precise::count},
    unitD{"A60", "erg per cubic centimetre", precise::one / precise::count},
    unitD{"A61", "erg per gram", precise::one / precise::count},
    unitD{"A62", "erg per gram second", precise::one / precise::count},
    unitD{"A63", "erg per second", precise::one / precise::count},
    unitD{
        "A64",
        "erg per second square centimetre",
        precise::one / precise::count},
    unitD{
        "A65",
        "erg per square centimetre second",
        precise::one / precise::count},
    unitD{"A66", "erg square centimetre", precise::one / precise::count},
    unitD{
        "A67",
        "erg square centimetre per gram",
        precise::one / precise::count},
    unitD{"A68", "exajoule", precise::one / precise::count},
    unitD{"A69", "farad per metre", precise::one / precise::count},
    unitD{"A7", "ampere per square millimetre", precise::one / precise::count},
    unitD{"A70", "femtojoule", precise::one / precise::count},
    unitD{"A71", "femtometre", precise::one / precise::count},
    unitD{"A73", "foot per second squared", precise::one / precise::count},
    unitD{"A74", "foot pound-force per second", precise::one / precise::count},
    unitD{"A75", "freight ton", precise::one / precise::count},
    unitD{"A76", "gal", precise::one / precise::count},
    unitD{
        "A77",
        "Gaussian CGS (Centimetre-Gram-Second system) unit of displacement",
        precise::one / precise::count},
    unitD{
        "A78",
        "Gaussian CGS (Centimetre-Gram-Second system) unit of electric current",
        precise::one / precise::count},
    unitD{
        "A79",
        "Gaussian CGS (Centimetre-Gram-Second system) unit of electric charge",
        precise::one / precise::count},
    unitD{"A8", "ampere second", precise::one / precise::count},
    unitD{
        "A80",
        "Gaussian CGS (Centimetre-Gram-Second system) unit of electric field strength",
        precise::one / precise::count},
    unitD{
        "A81",
        "Gaussian CGS (Centimetre-Gram-Second system) unit of electric polarization",
        precise::one / precise::count},
    unitD{
        "A82",
        "Gaussian CGS (Centimetre-Gram-Second system) unit of electric potential",
        precise::one / precise::count},
    unitD{
        "A83",
        "Gaussian CGS (Centimetre-Gram-Second system) unit of magnetization",
        precise::one / precise::count},
    unitD{"A84", "gigacoulomb per cubic metre", precise::one / precise::count},
    unitD{"A85", "gigaelectronvolt", precise::one / precise::count},
    unitD{"A86", "gigahertz", precise::one / precise::count},
    unitD{"A87", "gigaohm", precise::one / precise::count},
    unitD{"A88", "gigaohm metre", precise::one / precise::count},
    unitD{"A89", "gigapascal", precise::one / precise::count},
    unitD{"A9", "rate", precise::one / precise::count},
    unitD{"A90", "gigawatt", precise::one / precise::count},
    unitD{"A91", "gon", precise::one / precise::count},
    unitD{"A93", "gram per cubic metre", precise::one / precise::count},
    unitD{"A94", "gram per mole", precise::one / precise::count},
    unitD{"A95", "gray", precise::one / precise::count},
    unitD{"A96", "gray per second", precise::one / precise::count},
    unitD{"A97", "hectopascal", precise::one / precise::count},
    unitD{"A98", "henry per metre", precise::one / precise::count},
    unitD{"A99", "bit", precise::one / precise::count},
    unitD{"AA", "ball", precise::one / precise::count},
    unitD{"AB", "bulk pack", precise::one / precise::count},
    unitD{"ACR", "acre", precise::one / precise::count},
    unitD{"ACT", "activity", precise::one / precise::count},
    unitD{"AD", "byte", precise::one / precise::count},
    unitD{"AE", "ampere per metre", precise::one / precise::count},
    unitD{"AH", "additional minute", precise::one / precise::count},
    unitD{"AI", "average minute per call", precise::one / precise::count},
    unitD{"AJ", "cop", precise::one / precise::count},
    unitD{"AK", "fathom", precise::one / precise::count},
    unitD{"AL", "access line", precise::one / precise::count},
    unitD{"AM", "ampoule", precise::one / precise::count},
    unitD{"AMH", "ampere hour", precise::one / precise::count},
    unitD{"AMP", "ampere", precise::one / precise::count},
    unitD{"ANN", "year", precise::one / precise::count},
    unitD{"AP", "aluminium pound only", precise::one / precise::count},
    unitD{
        "APZ",
        "troy ounce or apothecary ounce",
        precise::one / precise::count},
    unitD{
        "AQ",
        "anti-hemophilic factor (AHF) unit",
        precise::one / precise::count},
    unitD{"AR", "suppository", precise::one / precise::count},
    unitD{"ARE", "are", precise::one / precise::count},
    unitD{"AS", "assortment", precise::one / precise::count},
    unitD{"ASM", "alcoholic strength by mass", precise::one / precise::count},
    unitD{"ASU", "alcoholic strength by volume", precise::one / precise::count},
    unitD{"ATM", "standard atmosphere", precise::one / precise::count},
    unitD{"ATT", "technical atmosphere", precise::one / precise::count},
    unitD{"AV", "capsule", precise::one / precise::count},
    unitD{"AW", "powder filled vial", precise::one / precise::count},
    unitD{"AY", "assembly", precise::one / precise::count},
    unitD{
        "AZ",
        "British thermal unit (international table) per pound",
        precise::one / precise::count},
    unitD{"B0", "Btu per cubic foot", precise::one / precise::count},
    unitD{"B1", "barrel (US) per day", precise::one / precise::count},
    unitD{"B10", "bit per second", precise::one / precise::count},
    unitD{"B11", "joule per kilogram kelvin", precise::one / precise::count},
    unitD{"B12", "joule per metre", precise::one / precise::count},
    unitD{"B13", "joule per square metre", precise::one / precise::count},
    unitD{
        "B14",
        "joule per metre to the fourth power",
        precise::one / precise::count},
    unitD{"B15", "joule per mole", precise::one / precise::count},
    unitD{"B16", "joule per mole kelvin", precise::one / precise::count},
    unitD{"B17", "credit", precise::one / precise::count},
    unitD{"B18", "joule second", precise::one / precise::count},
    unitD{"B19", "digit", precise::one / precise::count},
    unitD{"B2", "bunk", precise::one / precise::count},
    unitD{
        "B20",
        "joule square metre per kilogram",
        precise::one / precise::count},
    unitD{"B21", "kelvin per watt", precise::one / precise::count},
    unitD{"B22", "kiloampere", precise::one / precise::count},
    unitD{"B23", "kiloampere per square metre", precise::one / precise::count},
    unitD{"B24", "kiloampere per metre", precise::one / precise::count},
    unitD{"B25", "kilobecquerel per kilogram", precise::one / precise::count},
    unitD{"B26", "kilocoulomb", precise::one / precise::count},
    unitD{"B27", "kilocoulomb per cubic metre", precise::one / precise::count},
    unitD{"B28", "kilocoulomb per square metre", precise::one / precise::count},
    unitD{"B29", "kiloelectronvolt", precise::one / precise::count},
    unitD{"B3", "batting pound", precise::one / precise::count},
    unitD{"B30", "gibibit", precise::one / precise::count},
    unitD{"B31", "kilogram metre per second", precise::one / precise::count},
    unitD{"B32", "kilogram metre squared", precise::one / precise::count},
    unitD{
        "B33",
        "kilogram metre squared per second",
        precise::one / precise::count},
    unitD{"B34", "kilogram per cubic decimetre", precise::one / precise::count},
    unitD{"B35", "kilogram per litre", precise::one / precise::count},
    unitD{
        "B36",
        "calorie (thermochemical) per gram",
        precise::one / precise::count},
    unitD{"B37", "kilogram-force", precise::one / precise::count},
    unitD{"B38", "kilogram-force metre", precise::one / precise::count},
    unitD{
        "B39",
        "kilogram-force metre per second",
        precise::one / precise::count},
    unitD{"B4", "barrel, imperial", precise::one / precise::count},
    unitD{
        "B40",
        "kilogram-force per square metre",
        precise::one / precise::count},
    unitD{"B41", "kilojoule per kelvin", precise::one / precise::count},
    unitD{"B42", "kilojoule per kilogram", precise::one / precise::count},
    unitD{
        "B43",
        "kilojoule per kilogram kelvin",
        precise::one / precise::count},
    unitD{"B44", "kilojoule per mole", precise::one / precise::count},
    unitD{"B45", "kilomole", precise::one / precise::count},
    unitD{"B46", "kilomole per cubic metre", precise::one / precise::count},
    unitD{"B47", "kilonewton", precise::one / precise::count},
    unitD{"B48", "kilonewton metre", precise::one / precise::count},
    unitD{"B49", "kiloohm", precise::one / precise::count},
    unitD{"B5", "billet", precise::one / precise::count},
    unitD{"B50", "kiloohm metre", precise::one / precise::count},
    unitD{"B51", "kilopond", precise::one / precise::count},
    unitD{"B52", "kilosecond", precise::one / precise::count},
    unitD{"B53", "kilosiemens", precise::one / precise::count},
    unitD{"B54", "kilosiemens per metre", precise::one / precise::count},
    unitD{"B55", "kilovolt per metre", precise::one / precise::count},
    unitD{"B56", "kiloweber per metre", precise::one / precise::count},
    unitD{"B57", "light year", precise::one / precise::count},
    unitD{"B58", "litre per mole", precise::one / precise::count},
    unitD{"B59", "lumen hour", precise::one / precise::count},
    unitD{"B6", "bun", precise::one / precise::count},
    unitD{"B60", "lumen per square metre", precise::one / precise::count},
    unitD{"B61", "lumen per watt", precise::one / precise::count},
    unitD{"B62", "lumen second", precise::one / precise::count},
    unitD{"B63", "lux hour", precise::one / precise::count},
    unitD{"B64", "lux second", precise::one / precise::count},
    unitD{"B65", "maxwell", precise::one / precise::count},
    unitD{"B66", "megaampere per square metre", precise::one / precise::count},
    unitD{"B67", "megabecquerel per kilogram", precise::one / precise::count},
    unitD{"B68", "gigabit", precise::one / precise::count},
    unitD{"B69", "megacoulomb per cubic metre", precise::one / precise::count},
    unitD{"B7", "cycle", precise::one / precise::count},
    unitD{"B70", "megacoulomb per square metre", precise::one / precise::count},
    unitD{"B71", "megaelectronvolt", precise::one / precise::count},
    unitD{"B72", "megagram per cubic metre", precise::one / precise::count},
    unitD{"B73", "meganewton", precise::one / precise::count},
    unitD{"B74", "meganewton metre", precise::one / precise::count},
    unitD{"B75", "megaohm", precise::one / precise::count},
    unitD{"B76", "megaohm metre", precise::one / precise::count},
    unitD{"B77", "megasiemens per metre", precise::one / precise::count},
    unitD{"B78", "megavolt", precise::one / precise::count},
    unitD{"B79", "megavolt per metre", precise::one / precise::count},
    unitD{"B8", "joule per cubic metre", precise::one / precise::count},
    unitD{"B80", "gigabit per second", precise::one / precise::count},
    unitD{
        "B81",
        "reciprocal metre squared reciprocal second",
        precise::one / precise::count},
    unitD{"B82", "inch per linear foot", precise::one / precise::count},
    unitD{"B83", "metre to the fourth power", precise::one / precise::count},
    unitD{"B84", "microampere", precise::one / precise::count},
    unitD{"B85", "microbar", precise::one / precise::count},
    unitD{"B86", "microcoulomb", precise::one / precise::count},
    unitD{"B87", "microcoulomb per cubic metre", precise::one / precise::count},
    unitD{
        "B88",
        "microcoulomb per square metre",
        precise::one / precise::count},
    unitD{"B89", "microfarad per metre", precise::one / precise::count},
    unitD{"B9", "batt", precise::one / precise::count},
    unitD{"B90", "microhenry", precise::one / precise::count},
    unitD{"B91", "microhenry per metre", precise::one / precise::count},
    unitD{"B92", "micronewton", precise::one / precise::count},
    unitD{"B93", "micronewton metre", precise::one / precise::count},
    unitD{"B94", "microohm", precise::one / precise::count},
    unitD{"B95", "microohm metre", precise::one / precise::count},
    unitD{"B96", "micropascal", precise::one / precise::count},
    unitD{"B97", "microradian", precise::one / precise::count},
    unitD{"B98", "microsecond", precise::one / precise::count},
    unitD{"B99", "microsiemens", precise::one / precise::count},
    unitD{"BAR", "bar [unit of pressure]", precise::one / precise::count},
    unitD{"BB", "base box", precise::one / precise::count},
    unitD{"BD", "board", precise::one / precise::count},
    unitD{"BE", "bundle", precise::one / precise::count},
    unitD{"BFT", "board foot", precise::one / precise::count},
    unitD{"BG", "bag", precise::one / precise::count},
    unitD{"BH", "brush", precise::one / precise::count},
    unitD{"BHP", "brake horse power", precise::one / precise::count},
    unitD{"BIL", "billion (EUR)", precise::one / precise::count},
    unitD{"BJ", "bucket", precise::one / precise::count},
    unitD{"BK", "basket", precise::one / precise::count},
    unitD{"BL", "bale", precise::one / precise::count},
    unitD{"BLD", "dry barrel (US)", precise::one / precise::count},
    unitD{"BLL", "barrel (US)", precise::one / precise::count},
    unitD{"BO", "bottle", precise::one / precise::count},
    unitD{"BP", "hundred board foot", precise::one / precise::count},
    unitD{"BQL", "becquerel", precise::one / precise::count},
    unitD{"BR", "bar [unit of packaging]", precise::one / precise::count},
    unitD{"BT", "bolt", precise::one / precise::count},
    unitD{
        "BTU",
        "British thermal unit (international table)",
        precise::one / precise::count},
    unitD{"BUA", "bushel (US)", precise::one / precise::count},
    unitD{"BUI", "bushel (UK)", precise::one / precise::count},
    unitD{"BW", "base weight", precise::one / precise::count},
    unitD{"BX", "box", precise::one / precise::count},
    unitD{"BZ", "million BTUs", precise::one / precise::count},
    unitD{"C0", "call", precise::one / precise::count},
    unitD{
        "C1",
        "composite product pound (total weight)",
        precise::one / precise::count},
    unitD{"C10", "millifarad", precise::one / precise::count},
    unitD{"C11", "milligal", precise::one / precise::count},
    unitD{"C12", "milligram per metre", precise::one / precise::count},
    unitD{"C13", "milligray", precise::one / precise::count},
    unitD{"C14", "millihenry", precise::one / precise::count},
    unitD{"C15", "millijoule", precise::one / precise::count},
    unitD{"C16", "millimetre per second", precise::one / precise::count},
    unitD{
        "C17",
        "millimetre squared per second",
        precise::one / precise::count},
    unitD{"C18", "millimole", precise::one / precise::count},
    unitD{"C19", "mole per kilogram", precise::one / precise::count},
    unitD{"C2", "carset", precise::one / precise::count},
    unitD{"C20", "millinewton", precise::one / precise::count},
    unitD{"C21", "kibibit", precise::one / precise::count},
    unitD{"C22", "millinewton per metre", precise::one / precise::count},
    unitD{"C23", "milliohm metre", precise::one / precise::count},
    unitD{"C24", "millipascal second", precise::one / precise::count},
    unitD{"C25", "milliradian", precise::one / precise::count},
    unitD{"C26", "millisecond", precise::one / precise::count},
    unitD{"C27", "millisiemens", precise::one / precise::count},
    unitD{"C28", "millisievert", precise::one / precise::count},
    unitD{"C29", "millitesla", precise::one / precise::count},
    unitD{"C3", "microvolt per metre", precise::one / precise::count},
    unitD{"C30", "millivolt per metre", precise::one / precise::count},
    unitD{"C31", "milliwatt", precise::one / precise::count},
    unitD{"C32", "milliwatt per square metre", precise::one / precise::count},
    unitD{"C33", "milliweber", precise::one / precise::count},
    unitD{"C34", "mole", precise::one / precise::count},
    unitD{"C35", "mole per cubic decimetre", precise::one / precise::count},
    unitD{"C36", "mole per cubic metre", precise::one / precise::count},
    unitD{"C37", "kilobit", precise::one / precise::count},
    unitD{"C38", "mole per litre", precise::one / precise::count},
    unitD{"C39", "nanoampere", precise::one / precise::count},
    unitD{"C4", "carload", precise::one / precise::count},
    unitD{"C40", "nanocoulomb", precise::one / precise::count},
    unitD{"C41", "nanofarad", precise::one / precise::count},
    unitD{"C42", "nanofarad per metre", precise::one / precise::count},
    unitD{"C43", "nanohenry", precise::one / precise::count},
    unitD{"C44", "nanohenry per metre", precise::one / precise::count},
    unitD{"C45", "nanometre", precise::one / precise::count},
    unitD{"C46", "nanoohm metre", precise::one / precise::count},
    unitD{"C47", "nanosecond", precise::one / precise::count},
    unitD{"C48", "nanotesla", precise::one / precise::count},
    unitD{"C49", "nanowatt", precise::one / precise::count},
    unitD{"C5", "cost", precise::one / precise::count},
    unitD{"C50", "neper", precise::one / precise::count},
    unitD{"C51", "neper per second", precise::one / precise::count},
    unitD{"C52", "picometre", precise::one / precise::count},
    unitD{"C53", "newton metre second", precise::one / precise::count},
    unitD{
        "C54",
        "newton metre squared per kilogram squared",
        precise::one / precise::count},
    unitD{"C55", "newton per square metre", precise::one / precise::count},
    unitD{"C56", "newton per square millimetre", precise::one / precise::count},
    unitD{"C57", "newton second", precise::one / precise::count},
    unitD{"C58", "newton second per metre", precise::one / precise::count},
    unitD{"C59", "octave", precise::one / precise::count},
    unitD{"C6", "cell", precise::one / precise::count},
    unitD{"C60", "ohm centimetre", precise::one / precise::count},
    unitD{"C61", "ohm metre", precise::one / precise::count},
    unitD{"C62", "one", precise::one / precise::count},
    unitD{"C63", "parsec", precise::one / precise::count},
    unitD{"C64", "pascal per kelvin", precise::one / precise::count},
    unitD{"C65", "pascal second", precise::one / precise::count},
    unitD{
        "C66",
        "pascal second per cubic metre",
        precise::one / precise::count},
    unitD{"C67", "pascal second per metre", precise::one / precise::count},
    unitD{"C68", "petajoule", precise::one / precise::count},
    unitD{"C69", "phon", precise::one / precise::count},
    unitD{"C7", "centipoise", precise::one / precise::count},
    unitD{"C70", "picoampere", precise::one / precise::count},
    unitD{"C71", "picocoulomb", precise::one / precise::count},
    unitD{"C72", "picofarad per metre", precise::one / precise::count},
    unitD{"C73", "picohenry", precise::one / precise::count},
    unitD{"C74", "kilobit per second", precise::one / precise::count},
    unitD{"C75", "picowatt", precise::one / precise::count},
    unitD{"C76", "picowatt per square metre", precise::one / precise::count},
    unitD{"C77", "pound gage", precise::one / precise::count},
    unitD{"C78", "pound-force", precise::one / precise::count},
    unitD{"C79", "kilovolt ampere hour", precise::one / precise::count},
    unitD{"C8", "millicoulomb per kilogram", precise::one / precise::count},
    unitD{"C80", "rad", precise::one / precise::count},
    unitD{"C81", "radian", precise::one / precise::count},
    unitD{"C82", "radian square metre per mole", precise::one / precise::count},
    unitD{
        "C83",
        "radian square metre per kilogram",
        precise::one / precise::count},
    unitD{"C84", "radian per metre", precise::one / precise::count},
    unitD{"C85", "reciprocal angstrom", precise::one / precise::count},
    unitD{"C86", "reciprocal cubic metre", precise::one / precise::count},
    unitD{
        "C87",
        "reciprocal cubic metre per second",
        precise::one / precise::count},
    unitD{
        "C88",
        "reciprocal electron volt per cubic metre",
        precise::one / precise::count},
    unitD{"C89", "reciprocal henry", precise::one / precise::count},
    unitD{"C9", "coil group", precise::one / precise::count},
    unitD{
        "C90",
        "reciprocal joule per cubic metre",
        precise::one / precise::count},
    unitD{
        "C91",
        "reciprocal kelvin or kelvin to the power minus one",
        precise::one / precise::count},
    unitD{"C92", "reciprocal metre", precise::one / precise::count},
    unitD{"C93", "reciprocal square metre", precise::one / precise::count},
    unitD{"C94", "reciprocal minute", precise::one / precise::count},
    unitD{"C95", "reciprocal mole", precise::one / precise::count},
    unitD{
        "C96",
        "reciprocal pascal or pascal to the power minus one",
        precise::one / precise::count},
    unitD{"C97", "reciprocal second", precise::one / precise::count},
    unitD{
        "C98",
        "reciprocal second per cubic metre",
        precise::one / precise::count},
    unitD{
        "C99",
        "reciprocal second per metre squared",
        precise::one / precise::count},
    unitD{"CA", "can", precise::one / precise::count},
    unitD{
        "CCT",
        "carrying capacity in metric ton",
        precise::one / precise::count},
    unitD{"CDL", "candela", precise::one / precise::count},
    unitD{"CEL", "degree Celsius", precise::one / precise::count},
    unitD{"CEN", "hundred", precise::one / precise::count},
    unitD{"CG", "card", precise::one / precise::count},
    unitD{"CGM", "centigram", precise::one / precise::count},
    unitD{"CH", "container", precise::one / precise::count},
    unitD{"CJ", "cone", precise::one / precise::count},
    unitD{"CK", "connector", precise::one / precise::count},
    unitD{"CKG", "coulomb per kilogram", precise::one / precise::count},
    unitD{"CL", "coil", precise::one / precise::count},
    unitD{"CLF", "hundred leave", precise::one / precise::count},
    unitD{"CLT", "centilitre", precise::one / precise::count},
    unitD{"CMK", "square centimetre", precise::one / precise::count},
    unitD{"CMQ", "cubic centimetre", precise::one / precise::count},
    unitD{"CMT", "centimetre", precise::one / precise::count},
    unitD{"CNP", "hundred pack", precise::one / precise::count},
    unitD{"CNT", "cental (UK)", precise::one / precise::count},
    unitD{"CO", "carboy", precise::one / precise::count},
    unitD{"COU", "coulomb", precise::one / precise::count},
    unitD{"CQ", "cartridge", precise::one / precise::count},
    unitD{"CR", "crate", precise::one / precise::count},
    unitD{"CS", "case", precise::one / precise::count},
    unitD{"CT", "carton", precise::one / precise::count},
    unitD{"CTG", "content gram", precise::one / precise::count},
    unitD{"CTM", "metric carat", precise::one / precise::count},
    unitD{"CTN", "content ton (metric)", precise::one / precise::count},
    unitD{"CU", "cup", precise::one / precise::count},
    unitD{"CUR", "curie", precise::one / precise::count},
    unitD{"CV", "cover", precise::one / precise::count},
    unitD{
        "CWA",
        "hundred pound (cwt) / hundred weight (US)",
        precise::one / precise::count},
    unitD{"CWI", "hundred weight (UK)", precise::one / precise::count},
    unitD{"CY", "cylinder", precise::one / precise::count},
    unitD{"CZ", "combo", precise::one / precise::count},
    unitD{"D03", "kilowatt hour per hour", precise::one / precise::count},
    unitD{"D04", "lot  [unit of weight]", precise::one / precise::count},
    unitD{
        "D1",
        "reciprocal second per steradian",
        precise::one / precise::count},
    unitD{"D10", "siemens per metre", precise::one / precise::count},
    unitD{"D11", "mebibit", precise::one / precise::count},
    unitD{
        "D12",
        "siemens square metre per mole",
        precise::one / precise::count},
    unitD{"D13", "sievert", precise::one / precise::count},
    unitD{"D14", "thousand linear yard", precise::one / precise::count},
    unitD{"D15", "sone", precise::one / precise::count},
    unitD{"D16", "square centimetre per erg", precise::one / precise::count},
    unitD{
        "D17",
        "square centimetre per steradian erg",
        precise::one / precise::count},
    unitD{"D18", "metre kelvin", precise::one / precise::count},
    unitD{"D19", "square metre kelvin per watt", precise::one / precise::count},
    unitD{
        "D2",
        "reciprocal second per steradian metre squared",
        precise::one / precise::count},
    unitD{"D20", "square metre per joule", precise::one / precise::count},
    unitD{"D21", "square metre per kilogram", precise::one / precise::count},
    unitD{"D22", "square metre per mole", precise::one / precise::count},
    unitD{"D23", "pen gram (protein)", precise::one / precise::count},
    unitD{"D24", "square metre per steradian", precise::one / precise::count},
    unitD{
        "D25",
        "square metre per steradian joule",
        precise::one / precise::count},
    unitD{"D26", "square metre per volt second", precise::one / precise::count},
    unitD{"D27", "steradian", precise::one / precise::count},
    unitD{"D28", "syphon", precise::one / precise::count},
    unitD{"D29", "terahertz", precise::one / precise::count},
    unitD{"D30", "terajoule", precise::one / precise::count},
    unitD{"D31", "terawatt", precise::one / precise::count},
    unitD{"D32", "terawatt hour", precise::one / precise::count},
    unitD{"D33", "tesla", precise::one / precise::count},
    unitD{"D34", "tex", precise::one / precise::count},
    unitD{"D35", "calorie (thermochemical)", precise::one / precise::count},
    unitD{"D36", "megabit", precise::one / precise::count},
    unitD{
        "D37",
        "calorie (thermochemical) per gram kelvin",
        precise::one / precise::count},
    unitD{
        "D38",
        "calorie (thermochemical) per second centimetre kelvin",
        precise::one / precise::count},
    unitD{
        "D39",
        "calorie (thermochemical) per second square centimetre kelvin",
        precise::one / precise::count},
    unitD{"D40", "thousand litre", precise::one / precise::count},
    unitD{"D41", "tonne per cubic metre", precise::one / precise::count},
    unitD{"D42", "tropical year", precise::one / precise::count},
    unitD{"D43", "unified atomic mass unit", precise::one / precise::count},
    unitD{"D44", "var", precise::one / precise::count},
    unitD{
        "D45",
        "volt squared per kelvin squared",
        precise::one / precise::count},
    unitD{"D46", "volt - ampere", precise::one / precise::count},
    unitD{"D47", "volt per centimetre", precise::one / precise::count},
    unitD{"D48", "volt per kelvin", precise::one / precise::count},
    unitD{"D49", "millivolt per kelvin", precise::one / precise::count},
    unitD{
        "D5",
        "kilogram per square centimetre",
        precise::one / precise::count},
    unitD{"D50", "volt per metre", precise::one / precise::count},
    unitD{"D51", "volt per millimetre", precise::one / precise::count},
    unitD{"D52", "watt per kelvin", precise::one / precise::count},
    unitD{"D53", "watt per metre kelvin", precise::one / precise::count},
    unitD{"D54", "watt per square metre", precise::one / precise::count},
    unitD{"D55", "watt per square metre kelvin", precise::one / precise::count},
    unitD{
        "D56",
        "watt per square metre kelvin to the fourth power",
        precise::one / precise::count},
    unitD{"D57", "watt per steradian", precise::one / precise::count},
    unitD{
        "D58",
        "watt per steradian square metre",
        precise::one / precise::count},
    unitD{"D59", "weber per metre", precise::one / precise::count},
    unitD{"D6", "roentgen per second", precise::one / precise::count},
    unitD{"D60", "weber per millimetre", precise::one / precise::count},
    unitD{"D61", "minute [unit of angle]", precise::one / precise::count},
    unitD{"D62", "second [unit of angle]", precise::one / precise::count},
    unitD{"D63", "book", precise::one / precise::count},
    unitD{"D64", "block", precise::one / precise::count},
    unitD{"D65", "round", precise::one / precise::count},
    unitD{"D66", "cassette", precise::one / precise::count},
    unitD{"D67", "dollar per hour", precise::one / precise::count},
    unitD{"D68", "number of words", precise::one / precise::count},
    unitD{"D69", "inch to the fourth power", precise::one / precise::count},
    unitD{"D7", "sandwich", precise::one / precise::count},
    unitD{
        "D70",
        "calorie (international table)",
        precise::one / precise::count},
    unitD{
        "D71",
        "calorie (international table) per second centimetre kelvin",
        precise::one / precise::count},
    unitD{
        "D72",
        "calorie (international table) per second square centimetre kelvin",
        precise::one / precise::count},
    unitD{"D73", "joule square metre", precise::one / precise::count},
    unitD{"D74", "kilogram per mole", precise::one / precise::count},
    unitD{
        "D75",
        "calorie (international table) per gram",
        precise::one / precise::count},
    unitD{
        "D76",
        "calorie (international table) per gram kelvin",
        precise::one / precise::count},
    unitD{"D77", "megacoulomb", precise::one / precise::count},
    unitD{"D78", "megajoule per second", precise::one / precise::count},
    unitD{"D79", "beam", precise::one / precise::count},
    unitD{"D8", "draize score", precise::one / precise::count},
    unitD{"D80", "microwatt", precise::one / precise::count},
    unitD{"D81", "microtesla", precise::one / precise::count},
    unitD{"D82", "microvolt", precise::one / precise::count},
    unitD{"D83", "millinewton metre", precise::one / precise::count},
    unitD{"D85", "microwatt per square metre", precise::one / precise::count},
    unitD{"D86", "millicoulomb", precise::one / precise::count},
    unitD{"D87", "millimole per kilogram", precise::one / precise::count},
    unitD{"D88", "millicoulomb per cubic metre", precise::one / precise::count},
    unitD{
        "D89",
        "millicoulomb per square metre",
        precise::one / precise::count},
    unitD{"D9", "dyne per square centimetre", precise::one / precise::count},
    unitD{"D90", "cubic metre (net)", precise::one / precise::count},
    unitD{"D91", "rem", precise::one / precise::count},
    unitD{"D92", "band", precise::one / precise::count},
    unitD{"D93", "second per cubic metre", precise::one / precise::count},
    unitD{
        "D94",
        "second per cubic metre radian",
        precise::one / precise::count},
    unitD{"D95", "joule per gram", precise::one / precise::count},
    unitD{"D96", "pound gross", precise::one / precise::count},
    unitD{"D97", "pallet/unit load", precise::one / precise::count},
    unitD{"D98", "mass pound", precise::one / precise::count},
    unitD{"D99", "sleeve", precise::one / precise::count},
    unitD{"DAA", "decare", precise::one / precise::count},
    unitD{"DAD", "ten day", precise::one / precise::count},
    unitD{"DAY", "day", precise::one / precise::count},
    unitD{"DB", "dry pound", precise::one / precise::count},
    unitD{"DC", "disk (disc)", precise::one / precise::count},
    unitD{"DD", "degree [unit of angle]", precise::one / precise::count},
    unitD{"DE", "deal", precise::one / precise::count},
    unitD{"DEC", "decade", precise::one / precise::count},
    unitD{"DG", "decigram", precise::one / precise::count},
    unitD{"DI", "dispenser", precise::one / precise::count},
    unitD{"DJ", "decagram", precise::one / precise::count},
    unitD{"DLT", "decilitre", precise::one / precise::count},
    unitD{"DMA", "cubic decametre", precise::one / precise::count},
    unitD{"DMK", "square decimetre", precise::one / precise::count},
    unitD{"DMO", "standard kilolitre", precise::one / precise::count},
    unitD{"DMQ", "cubic decimetre", precise::one / precise::count},
    unitD{"DMT", "decimetre", precise::one / precise::count},
    unitD{"DN", "decinewton metre", precise::one / precise::count},
    unitD{"DPC", "dozen piece", precise::one / precise::count},
    unitD{"DPR", "dozen pair", precise::one / precise::count},
    unitD{"DPT", "displacement tonnage", precise::one / precise::count},
    unitD{"DQ", "data record", precise::one / precise::count},
    unitD{"DR", "drum", precise::one / precise::count},
    unitD{"DRA", "dram (US)", precise::one / precise::count},
    unitD{"DRI", "dram (UK)", precise::one / precise::count},
    unitD{"DRL", "dozen roll", precise::one / precise::count},
    unitD{"DRM", "drachm (UK)", precise::one / precise::count},
    unitD{"DS", "display", precise::one / precise::count},
    unitD{"DT", "dry ton", precise::one / precise::count},
    unitD{"DTN", "decitonne", precise::one / precise::count},
    unitD{"DU", "dyne", precise::one / precise::count},
    unitD{"DWT", "pennyweight", precise::one / precise::count},
    unitD{"DX", "dyne per centimetre", precise::one / precise::count},
    unitD{"DY", "directory book", precise::one / precise::count},
    unitD{"DZN", "dozen", precise::one / precise::count},
    unitD{"DZP", "dozen pack", precise::one / precise::count},
    unitD{"E01", "newton per square centimetre", precise::one / precise::count},
    unitD{"E07", "megawatt hour per hour", precise::one / precise::count},
    unitD{"E08", "megawatt per hertz", precise::one / precise::count},
    unitD{"E09", "milliampere hour", precise::one / precise::count},
    unitD{"E10", "degree day", precise::one / precise::count},
    unitD{"E11", "gigacalorie", precise::one / precise::count},
    unitD{"E12", "mille", precise::one / precise::count},
    unitD{
        "E14",
        "kilocalorie (international table)",
        precise::one / precise::count},
    unitD{
        "E15",
        "kilocalorie (thermochemical) per hour",
        precise::one / precise::count},
    unitD{"E16", "million Btu(IT) per hour", precise::one / precise::count},
    unitD{"E17", "cubic foot per second", precise::one / precise::count},
    unitD{"E18", "tonne per hour", precise::one / precise::count},
    unitD{"E19", "ping", precise::one / precise::count},
    unitD{"E2", "belt", precise::one / precise::count},
    unitD{"E20", "megabit per second", precise::one / precise::count},
    unitD{"E21", "shares", precise::one / precise::count},
    unitD{"E22", "TEU", precise::one / precise::count},
    unitD{"E23", "tyre", precise::one / precise::count},
    unitD{"E25", "active unit", precise::one / precise::count},
    unitD{"E27", "dose", precise::one / precise::count},
    unitD{"E28", "air dry ton", precise::one / precise::count},
    unitD{"E3", "trailer", precise::one / precise::count},
    unitD{"E30", "strand", precise::one / precise::count},
    unitD{"E31", "square metre per litre", precise::one / precise::count},
    unitD{"E32", "litre per hour", precise::one / precise::count},
    unitD{"E33", "foot per thousand", precise::one / precise::count},
    unitD{"E34", "gigabyte", precise::one / precise::count},
    unitD{"E35", "terabyte", precise::one / precise::count},
    unitD{"E36", "petabyte", precise::one / precise::count},
    unitD{"E37", "pixel", precise::one / precise::count},
    unitD{"E38", "megapixel", precise::one / precise::count},
    unitD{"E39", "dots per inch", precise::one / precise::count},
    unitD{"E4", "gross kilogram", precise::one / precise::count},
    unitD{"E40", "part per hundred thousand", precise::one / precise::count},
    unitD{
        "E41",
        "kilogram-force per square millimetre",
        precise::one / precise::count},
    unitD{
        "E42",
        "kilogram-force per square centimetre",
        precise::one / precise::count},
    unitD{"E43", "joule per square centimetre", precise::one / precise::count},
    unitD{
        "E44",
        "kilogram-force metre per square centimetre",
        precise::one / precise::count},
    unitD{"E45", "milliohm", precise::one / precise::count},
    unitD{
        "E46",
        "kilowatt hour per cubic metre",
        precise::one / precise::count},
    unitD{"E47", "kilowatt hour per kelvin", precise::one / precise::count},
    unitD{"E48", "service unit", precise::one / precise::count},
    unitD{"E49", "working day", precise::one / precise::count},
    unitD{"E5", "metric long ton", precise::one / precise::count},
    unitD{"E50", "accounting unit", precise::one / precise::count},
    unitD{"E51", "job", precise::one / precise::count},
    unitD{"E52", "run foot", precise::one / precise::count},
    unitD{"E53", "test", precise::one / precise::count},
    unitD{"E54", "trip", precise::one / precise::count},
    unitD{"E55", "use", precise::one / precise::count},
    unitD{"E56", "well", precise::one / precise::count},
    unitD{"E57", "zone", precise::one / precise::count},
    unitD{"E58", "exabit per second", precise::one / precise::count},
    unitD{"E59", "exbibyte", precise::one / precise::count},
    unitD{"E60", "pebibyte", precise::one / precise::count},
    unitD{"E61", "tebibyte", precise::one / precise::count},
    unitD{"E62", "gibibyte", precise::one / precise::count},
    unitD{"E63", "mebibyte", precise::one / precise::count},
    unitD{"E64", "kibibyte", precise::one / precise::count},
    unitD{"E65", "exbibit per metre", precise::one / precise::count},
    unitD{"E66", "exbibit per square metre", precise::one / precise::count},
    unitD{"E67", "exbibit per cubic metre", precise::one / precise::count},
    unitD{"E68", "gigabyte per second", precise::one / precise::count},
    unitD{"E69", "gibibit per metre", precise::one / precise::count},
    unitD{"E70", "gibibit per square metre", precise::one / precise::count},
    unitD{"E71", "gibibit per cubic metre", precise::one / precise::count},
    unitD{"E72", "kibibit per metre", precise::one / precise::count},
    unitD{"E73", "kibibit per square metre", precise::one / precise::count},
    unitD{"E74", "kibibit per cubic metre", precise::one / precise::count},
    unitD{"E75", "mebibit per metre", precise::one / precise::count},
    unitD{"E76", "mebibit per square metre", precise::one / precise::count},
    unitD{"E77", "mebibit per cubic metre", precise::one / precise::count},
    unitD{"E78", "petabit", precise::one / precise::count},
    unitD{"E79", "petabit per second", precise::one / precise::count},
    unitD{"E80", "pebibit per metre", precise::one / precise::count},
    unitD{"E81", "pebibit per square metre", precise::one / precise::count},
    unitD{"E82", "pebibit per cubic metre", precise::one / precise::count},
    unitD{"E83", "terabit", precise::one / precise::count},
    unitD{"E84", "terabit per second", precise::one / precise::count},
    unitD{"E85", "tebibit per metre", precise::one / precise::count},
    unitD{"E86", "tebibit per cubic metre", precise::one / precise::count},
    unitD{"E87", "tebibit per square metre", precise::one / precise::count},
    unitD{"E88", "bit per metre", precise::one / precise::count},
    unitD{"E89", "bit per square metre", precise::one / precise::count},
    unitD{"E90", "reciprocal centimetre", precise::one / precise::count},
    unitD{"E91", "reciprocal day", precise::one / precise::count},
    unitD{"E92", "cubic decimetre per hour", precise::one / precise::count},
    unitD{"E93", "kilogram per hour", precise::one / precise::count},
    unitD{"E94", "kilomole per second", precise::one / precise::count},
    unitD{"E95", "mole per second", precise::one / precise::count},
    unitD{"E96", "degree per second", precise::one / precise::count},
    unitD{
        "E97",
        "millimetre per degree Celcius metre",
        precise::one / precise::count},
    unitD{"E98", "degree Celsius per kelvin", precise::one / precise::count},
    unitD{"E99", "hectopascal per bar", precise::one / precise::count},
    unitD{"EA", "each", precise::one / precise::count},
    unitD{"EB", "electronic mail box", precise::one / precise::count},
    unitD{"EC", "each per month", precise::one / precise::count},
    unitD{"EP", "eleven pack", precise::one / precise::count},
    unitD{"EQ", "equivalent gallon", precise::one / precise::count},
    unitD{"EV", "envelope", precise::one / precise::count},
    unitD{"F01", "bit per cubic metre", precise::one / precise::count},
    unitD{"F02", "kelvin per kelvin", precise::one / precise::count},
    unitD{"F03", "kilopascal per bar", precise::one / precise::count},
    unitD{"F04", "millibar per bar", precise::one / precise::count},
    unitD{"F05", "megapascal per bar", precise::one / precise::count},
    unitD{"F06", "poise per bar", precise::one / precise::count},
    unitD{"F07", "pascal per bar", precise::one / precise::count},
    unitD{"F08", "milliampere per inch", precise::one / precise::count},
    unitD{"F1", "thousand cubic foot per day", precise::one / precise::count},
    unitD{"F10", "kelvin per hour", precise::one / precise::count},
    unitD{"F11", "kelvin per minute", precise::one / precise::count},
    unitD{"F12", "kelvin per second", precise::one / precise::count},
    unitD{"F13", "slug", precise::one / precise::count},
    unitD{"F14", "gram per kelvin", precise::one / precise::count},
    unitD{"F15", "kilogram per kelvin", precise::one / precise::count},
    unitD{"F16", "milligram per kelvin", precise::one / precise::count},
    unitD{"F17", "pound-force per foot", precise::one / precise::count},
    unitD{"F18", "kilogram square centimetre", precise::one / precise::count},
    unitD{"F19", "kilogram square millimetre", precise::one / precise::count},
    unitD{"F20", "pound inch squared", precise::one / precise::count},
    unitD{"F21", "pound-force inch", precise::one / precise::count},
    unitD{"F22", "pound-force foot per ampere", precise::one / precise::count},
    unitD{"F23", "gram per cubic decimetre", precise::one / precise::count},
    unitD{"F24", "kilogram per kilomol", precise::one / precise::count},
    unitD{"F25", "gram per hertz", precise::one / precise::count},
    unitD{"F26", "gram per day", precise::one / precise::count},
    unitD{"F27", "gram per hour", precise::one / precise::count},
    unitD{"F28", "gram per minute", precise::one / precise::count},
    unitD{"F29", "gram per second", precise::one / precise::count},
    unitD{"F30", "kilogram per day", precise::one / precise::count},
    unitD{"F31", "kilogram per minute", precise::one / precise::count},
    unitD{"F32", "milligram per day", precise::one / precise::count},
    unitD{"F33", "milligram per minute", precise::one / precise::count},
    unitD{"F34", "milligram per second", precise::one / precise::count},
    unitD{"F35", "gram per day kelvin", precise::one / precise::count},
    unitD{"F36", "gram per hour kelvin", precise::one / precise::count},
    unitD{"F37", "gram per minute kelvin", precise::one / precise::count},
    unitD{"F38", "gram per second kelvin", precise::one / precise::count},
    unitD{"F39", "kilogram per day kelvin", precise::one / precise::count},
    unitD{"F40", "kilogram per hour kelvin", precise::one / precise::count},
    unitD{"F41", "kilogram per minute kelvin", precise::one / precise::count},
    unitD{"F42", "kilogram per second kelvin", precise::one / precise::count},
    unitD{"F43", "milligram per day kelvin", precise::one / precise::count},
    unitD{"F44", "milligram per hour kelvin", precise::one / precise::count},
    unitD{"F45", "milligram per minute kelvin", precise::one / precise::count},
    unitD{"F46", "milligram per second kelvin", precise::one / precise::count},
    unitD{"F47", "newton per millimetre", precise::one / precise::count},
    unitD{"F48", "pound-force per inch", precise::one / precise::count},
    unitD{"F49", "rod [unit of distance]", precise::one / precise::count},
    unitD{"F50", "micrometre per kelvin", precise::one / precise::count},
    unitD{"F51", "centimetre per kelvin", precise::one / precise::count},
    unitD{"F52", "metre per kelvin", precise::one / precise::count},
    unitD{"F53", "millimetre per kelvin", precise::one / precise::count},
    unitD{"F54", "milliohm per metre", precise::one / precise::count},
    unitD{"F55", "ohm per mile (statute mile)", precise::one / precise::count},
    unitD{"F56", "ohm per kilometre", precise::one / precise::count},
    unitD{
        "F57",
        "milliampere per pound-force per square inch",
        precise::one / precise::count},
    unitD{"F58", "reciprocal bar", precise::one / precise::count},
    unitD{"F59", "milliampere per bar", precise::one / precise::count},
    unitD{"F60", "degree Celsius per bar", precise::one / precise::count},
    unitD{"F61", "kelvin per bar", precise::one / precise::count},
    unitD{"F62", "gram per day bar", precise::one / precise::count},
    unitD{"F63", "gram per hour bar", precise::one / precise::count},
    unitD{"F64", "gram per minute bar", precise::one / precise::count},
    unitD{"F65", "gram per second bar", precise::one / precise::count},
    unitD{"F66", "kilogram per day bar", precise::one / precise::count},
    unitD{"F67", "kilogram per hour bar", precise::one / precise::count},
    unitD{"F68", "kilogram per minute bar", precise::one / precise::count},
    unitD{"F69", "kilogram per second bar", precise::one / precise::count},
    unitD{"F70", "milligram per day bar", precise::one / precise::count},
    unitD{"F71", "milligram per hour bar", precise::one / precise::count},
    unitD{"F72", "milligram per minute bar", precise::one / precise::count},
    unitD{"F73", "milligram per second bar", precise::one / precise::count},
    unitD{"F74", "gram per bar", precise::one / precise::count},
    unitD{"F75", "milligram per bar", precise::one / precise::count},
    unitD{"F76", "milliampere per millimetre", precise::one / precise::count},
    unitD{"F77", "pascal second per kelvin", precise::one / precise::count},
    unitD{"F78", "inch of water", precise::one / precise::count},
    unitD{"F79", "inch of mercury", precise::one / precise::count},
    unitD{"F80", "water horse power", precise::one / precise::count},
    unitD{"F81", "bar per kelvin", precise::one / precise::count},
    unitD{"F82", "hectopascal per kelvin", precise::one / precise::count},
    unitD{"F83", "kilopascal per kelvin", precise::one / precise::count},
    unitD{"F84", "millibar per kelvin", precise::one / precise::count},
    unitD{"F85", "megapascal per kelvin", precise::one / precise::count},
    unitD{"F86", "poise per kelvin", precise::one / precise::count},
    unitD{"F87", "volt per litre minute", precise::one / precise::count},
    unitD{"F88", "newton centimetre", precise::one / precise::count},
    unitD{"F89", "newton metre per degree", precise::one / precise::count},
    unitD{
        "F9",
        "fibre per cubic centimetre of air",
        precise::one / precise::count},
    unitD{"F90", "newton metre per ampere", precise::one / precise::count},
    unitD{"F91", "bar litre per second", precise::one / precise::count},
    unitD{"F92", "bar cubic metre per second", precise::one / precise::count},
    unitD{"F93", "hectopascal litre per second", precise::one / precise::count},
    unitD{
        "F94",
        "hectopascal cubic metre per second",
        precise::one / precise::count},
    unitD{"F95", "millibar litre per second", precise::one / precise::count},
    unitD{
        "F96",
        "millibar cubic metre per second",
        precise::one / precise::count},
    unitD{"F97", "megapascal litre per second", precise::one / precise::count},
    unitD{
        "F98",
        "megapascal cubic metre per second",
        precise::one / precise::count},
    unitD{"F99", "pascal litre per second", precise::one / precise::count},
    unitD{"FAH", "degree Fahrenheit", precise::one / precise::count},
    unitD{"FAR", "farad", precise::one / precise::count},
    unitD{"FB", "field", precise::one / precise::count},
    unitD{"FBM", "fibre metre", precise::one / precise::count},
    unitD{"FC", "thousand cubic foot", precise::one / precise::count},
    unitD{
        "FD",
        "million particle per cubic foot",
        precise::one / precise::count},
    unitD{"FE", "track foot", precise::one / precise::count},
    unitD{"FF", "hundred cubic metre", precise::one / precise::count},
    unitD{"FG", "transdermal patch", precise::one / precise::count},
    unitD{"FH", "micromole", precise::one / precise::count},
    unitD{"FIT", "failures in time", precise::one / precise::count},
    unitD{"FL", "flake ton", precise::one / precise::count},
    unitD{"FM", "million cubic foot", precise::one / precise::count},
    unitD{"FOT", "foot", precise::one / precise::count},
    unitD{"FP", "pound per square foot", precise::one / precise::count},
    unitD{"FR", "foot per minute", precise::one / precise::count},
    unitD{"FS", "foot per second", precise::one / precise::count},
    unitD{"FTK", "square foot", precise::one / precise::count},
    unitD{"FTQ", "cubic foot", precise::one / precise::count},
    unitD{
        "G01",
        "pascal cubic metre per second",
        precise::one / precise::count},
    unitD{"G04", "centimetre per bar", precise::one / precise::count},
    unitD{"G05", "metre per bar", precise::one / precise::count},
    unitD{"G06", "millimetre per bar", precise::one / precise::count},
    unitD{"G08", "square inch per second", precise::one / precise::count},
    unitD{
        "G09",
        "square metre per second kelvin",
        precise::one / precise::count},
    unitD{"G10", "stokes per kelvin", precise::one / precise::count},
    unitD{
        "G11",
        "gram per cubic centimetre bar",
        precise::one / precise::count},
    unitD{"G12", "gram per cubic decimetre bar", precise::one / precise::count},
    unitD{"G13", "gram per litre bar", precise::one / precise::count},
    unitD{"G14", "gram per cubic metre bar", precise::one / precise::count},
    unitD{"G15", "gram per millilitre bar", precise::one / precise::count},
    unitD{
        "G16",
        "kilogram per cubic centimetre bar",
        precise::one / precise::count},
    unitD{"G17", "kilogram per litre bar", precise::one / precise::count},
    unitD{"G18", "kilogram per cubic metre bar", precise::one / precise::count},
    unitD{"G19", "newton metre per kilogram", precise::one / precise::count},
    unitD{"G2", "US gallon per minute", precise::one / precise::count},
    unitD{"G20", "pound-force foot per pound", precise::one / precise::count},
    unitD{"G21", "cup [unit of volume]", precise::one / precise::count},
    unitD{"G23", "peck", precise::one / precise::count},
    unitD{"G24", "tablespoon (US)", precise::one / precise::count},
    unitD{"G25", "teaspoon (US)", precise::one / precise::count},
    unitD{"G26", "stere", precise::one / precise::count},
    unitD{"G27", "cubic centimetre per kelvin", precise::one / precise::count},
    unitD{"G28", "litre per kelvin", precise::one / precise::count},
    unitD{"G29", "cubic metre per kelvin", precise::one / precise::count},
    unitD{"G3", "Imperial gallon per minute", precise::one / precise::count},
    unitD{"G30", "millilitre per kelvin", precise::one / precise::count},
    unitD{
        "G31",
        "kilogram per cubic centimetre",
        precise::one / precise::count},
    unitD{
        "G32",
        "ounce (avoirdupois) per cubic yard",
        precise::one / precise::count},
    unitD{
        "G33",
        "gram per cubic centimetre kelvin",
        precise::one / precise::count},
    unitD{
        "G34",
        "gram per cubic decimetre kelvin",
        precise::one / precise::count},
    unitD{"G35", "gram per litre kelvin", precise::one / precise::count},
    unitD{"G36", "gram per cubic metre kelvin", precise::one / precise::count},
    unitD{"G37", "gram per millilitre kelvin", precise::one / precise::count},
    unitD{
        "G38",
        "kilogram per cubic centimetre kelvin",
        precise::one / precise::count},
    unitD{"G39", "kilogram per litre kelvin", precise::one / precise::count},
    unitD{
        "G40",
        "kilogram per cubic metre kelvin",
        precise::one / precise::count},
    unitD{"G41", "square metre per second bar", precise::one / precise::count},
    unitD{"G42", "microsiemens per centimetre", precise::one / precise::count},
    unitD{"G43", "microsiemens per metre", precise::one / precise::count},
    unitD{"G44", "nanosiemens per centimetre", precise::one / precise::count},
    unitD{"G45", "nanosiemens per metre", precise::one / precise::count},
    unitD{"G46", "stokes per bar", precise::one / precise::count},
    unitD{"G47", "cubic centimetre per day", precise::one / precise::count},
    unitD{"G48", "cubic centimetre per hour", precise::one / precise::count},
    unitD{"G49", "cubic centimetre per minute", precise::one / precise::count},
    unitD{"G50", "gallon (US) per hour", precise::one / precise::count},
    unitD{"G51", "litre per second", precise::one / precise::count},
    unitD{"G52", "cubic metre per day", precise::one / precise::count},
    unitD{"G53", "cubic metre per minute", precise::one / precise::count},
    unitD{"G54", "millilitre per day", precise::one / precise::count},
    unitD{"G55", "millilitre per hour", precise::one / precise::count},
    unitD{"G56", "cubic inch per hour", precise::one / precise::count},
    unitD{"G57", "cubic inch per minute", precise::one / precise::count},
    unitD{"G58", "cubic inch per second", precise::one / precise::count},
    unitD{"G59", "milliampere per litre minute", precise::one / precise::count},
    unitD{"G60", "volt per bar", precise::one / precise::count},
    unitD{
        "G61",
        "cubic centimetre per day kelvin",
        precise::one / precise::count},
    unitD{
        "G62",
        "cubic centimetre per hour kelvin",
        precise::one / precise::count},
    unitD{
        "G63",
        "cubic centimetre per minute kelvin",
        precise::one / precise::count},
    unitD{
        "G64",
        "cubic centimetre per second kelvin",
        precise::one / precise::count},
    unitD{"G65", "litre per day kelvin", precise::one / precise::count},
    unitD{"G66", "litre per hour kelvin", precise::one / precise::count},
    unitD{"G67", "litre per minute kelvin", precise::one / precise::count},
    unitD{"G68", "litre per second kelvin", precise::one / precise::count},
    unitD{"G69", "cubic metre per day kelvin", precise::one / precise::count},
    unitD{"G7", "microfiche sheet", precise::one / precise::count},
    unitD{"G70", "cubic metre per hour kelvin", precise::one / precise::count},
    unitD{
        "G71",
        "cubic metre per minute kelvin",
        precise::one / precise::count},
    unitD{
        "G72",
        "cubic metre per second kelvin",
        precise::one / precise::count},
    unitD{"G73", "millilitre per day kelvin", precise::one / precise::count},
    unitD{"G74", "millilitre per hour kelvin", precise::one / precise::count},
    unitD{"G75", "millilitre per minute kelvin", precise::one / precise::count},
    unitD{"G76", "millilitre per second kelvin", precise::one / precise::count},
    unitD{
        "G77",
        "millimetre to the fourth power",
        precise::one / precise::count},
    unitD{"G78", "cubic centimetre per day bar", precise::one / precise::count},
    unitD{
        "G79",
        "cubic centimetre per hour bar",
        precise::one / precise::count},
    unitD{
        "G80",
        "cubic centimetre per minute bar",
        precise::one / precise::count},
    unitD{
        "G81",
        "cubic centimetre per second bar",
        precise::one / precise::count},
    unitD{"G82", "litre per day bar", precise::one / precise::count},
    unitD{"G83", "litre per hour bar", precise::one / precise::count},
    unitD{"G84", "litre per minute bar", precise::one / precise::count},
    unitD{"G85", "litre per second bar", precise::one / precise::count},
    unitD{"G86", "cubic metre per day bar", precise::one / precise::count},
    unitD{"G87", "cubic metre per hour bar", precise::one / precise::count},
    unitD{"G88", "cubic metre per minute bar", precise::one / precise::count},
    unitD{"G89", "cubic metre per second bar", precise::one / precise::count},
    unitD{"G90", "millilitre per day bar", precise::one / precise::count},
    unitD{"G91", "millilitre per hour bar", precise::one / precise::count},
    unitD{"G92", "millilitre per minute bar", precise::one / precise::count},
    unitD{"G93", "millilitre per second bar", precise::one / precise::count},
    unitD{"G94", "cubic centimetre per bar", precise::one / precise::count},
    unitD{"G95", "litre per bar", precise::one / precise::count},
    unitD{"G96", "cubic metre per bar", precise::one / precise::count},
    unitD{"G97", "millilitre per bar", precise::one / precise::count},
    unitD{"G98", "microhenry per kiloohm", precise::one / precise::count},
    unitD{"G99", "microhenry per ohm", precise::one / precise::count},
    unitD{"GB", "gallon (US) per day", precise::one / precise::count},
    unitD{"GBQ", "gigabecquerel", precise::one / precise::count},
    unitD{"GC", "gram per 100 gram", precise::one / precise::count},
    unitD{"GD", "gross barrel", precise::one / precise::count},
    unitD{"GDW", "gram, dry weight", precise::one / precise::count},
    unitD{"GE", "pound per gallon (US)", precise::one / precise::count},
    unitD{
        "GF",
        "gram per metre (gram per 100 centimetres)",
        precise::one / precise::count},
    unitD{"GFI", "gram of fissile isotope", precise::one / precise::count},
    unitD{"GGR", "great gross", precise::one / precise::count},
    unitD{"GH", "half gallon (US)", precise::one / precise::count},
    unitD{"GIA", "gill (US)", precise::one / precise::count},
    unitD{"GIC", "gram, including container", precise::one / precise::count},
    unitD{"GII", "gill (UK)", precise::one / precise::count},
    unitD{
        "GIP",
        "gram, including inner packaging",
        precise::one / precise::count},
    unitD{"GJ", "gram per millilitre", precise::one / precise::count},
    unitD{"GK", "gram per kilogram", precise::one / precise::count},
    unitD{"GL", "gram per litre", precise::one / precise::count},
    unitD{"GLD", "dry gallon (US)", precise::one / precise::count},
    unitD{"GLI", "gallon (UK)", precise::one / precise::count},
    unitD{"GLL", "gallon (US)", precise::one / precise::count},
    unitD{"GM", "gram per square metre", precise::one / precise::count},
    unitD{"GN", "gross gallon", precise::one / precise::count},
    unitD{"GO", "milligram per square metre", precise::one / precise::count},
    unitD{"GP", "milligram per cubic metre", precise::one / precise::count},
    unitD{"GQ", "microgram per cubic metre", precise::one / precise::count},
    unitD{"GRM", "gram", precise::one / precise::count},
    unitD{"GRN", "grain", precise::one / precise::count},
    unitD{"GRO", "gross", precise::one / precise::count},
    unitD{"GRT", "gross register ton", precise::one / precise::count},
    unitD{"GT", "gross ton", precise::one / precise::count},
    unitD{"GV", "gigajoule", precise::one / precise::count},
    unitD{
        "GW",
        "gallon per thousand cubic foot",
        precise::one / precise::count},
    unitD{"GWH", "gigawatt hour", precise::one / precise::count},
    unitD{"GY", "gross yard", precise::one / precise::count},
    unitD{"GZ", "gage system", precise::one / precise::count},
    unitD{"H03", "henry per kiloohm", precise::one / precise::count},
    unitD{"H04", "henry per ohm", precise::one / precise::count},
    unitD{"H05", "millihenry per kiloohm", precise::one / precise::count},
    unitD{"H06", "millihenry per ohm", precise::one / precise::count},
    unitD{"H07", "pascal second per bar", precise::one / precise::count},
    unitD{"H08", "microbecquerel", precise::one / precise::count},
    unitD{"H09", "reciprocal year", precise::one / precise::count},
    unitD{"H1", "half page - electronic", precise::one / precise::count},
    unitD{"H10", "reciprocal hour", precise::one / precise::count},
    unitD{"H11", "reciprocal month", precise::one / precise::count},
    unitD{"H12", "degree Celsius per hour", precise::one / precise::count},
    unitD{"H13", "degree Celsius per minute", precise::one / precise::count},
    unitD{"H14", "degree Celsius per second", precise::one / precise::count},
    unitD{"H15", "square centimetre per gram", precise::one / precise::count},
    unitD{"H16", "square decametre", precise::one / precise::count},
    unitD{"H18", "square hectometre", precise::one / precise::count},
    unitD{"H19", "cubic hectometre", precise::one / precise::count},
    unitD{"H2", "half litre", precise::one / precise::count},
    unitD{"H20", "cubic kilometre", precise::one / precise::count},
    unitD{"H21", "blank", precise::one / precise::count},
    unitD{
        "H22",
        "volt square inch per pound-force",
        precise::one / precise::count},
    unitD{"H23", "volt per inch", precise::one / precise::count},
    unitD{"H24", "volt per microsecond", precise::one / precise::count},
    unitD{"H25", "percent per kelvin", precise::one / precise::count},
    unitD{"H26", "ohm per metre", precise::one / precise::count},
    unitD{"H27", "degree per metre", precise::one / precise::count},
    unitD{"H28", "microfarad per kilometre", precise::one / precise::count},
    unitD{"H29", "microgram per litre", precise::one / precise::count},
    unitD{
        "H30",
        "square micrometre (square micron)",
        precise::one / precise::count},
    unitD{"H31", "ampere per kilogram", precise::one / precise::count},
    unitD{"H32", "ampere squared second", precise::one / precise::count},
    unitD{"H33", "farad per kilometre", precise::one / precise::count},
    unitD{"H34", "hertz metre", precise::one / precise::count},
    unitD{"H35", "kelvin metre per watt", precise::one / precise::count},
    unitD{"H36", "megaohm per kilometre", precise::one / precise::count},
    unitD{"H37", "megaohm per metre", precise::one / precise::count},
    unitD{"H38", "megaampere", precise::one / precise::count},
    unitD{"H39", "megahertz kilometre", precise::one / precise::count},
    unitD{"H40", "newton per ampere", precise::one / precise::count},
    unitD{
        "H41",
        "newton metre watt to the power minus 0,5",
        precise::one / precise::count},
    unitD{"H42", "pascal per metre", precise::one / precise::count},
    unitD{"H43", "siemens per centimetre", precise::one / precise::count},
    unitD{"H44", "teraohm", precise::one / precise::count},
    unitD{"H45", "volt second per metre", precise::one / precise::count},
    unitD{"H46", "volt per second", precise::one / precise::count},
    unitD{"H47", "watt per cubic metre", precise::one / precise::count},
    unitD{"H48", "attofarad", precise::one / precise::count},
    unitD{"H49", "centimetre per hour", precise::one / precise::count},
    unitD{"H50", "reciprocal cubic centimetre", precise::one / precise::count},
    unitD{"H51", "decibel per kilometre", precise::one / precise::count},
    unitD{"H52", "decibel per metre", precise::one / precise::count},
    unitD{"H53", "kilogram per bar", precise::one / precise::count},
    unitD{
        "H54",
        "kilogram per cubic decimetre kelvin",
        precise::one / precise::count},
    unitD{
        "H55",
        "kilogram per cubic decimetre bar",
        precise::one / precise::count},
    unitD{
        "H56",
        "kilogram per square metre second",
        precise::one / precise::count},
    unitD{"H57", "inch per two pi radiant", precise::one / precise::count},
    unitD{"H58", "metre per volt second", precise::one / precise::count},
    unitD{"H59", "square metre per newton", precise::one / precise::count},
    unitD{"H60", "cubic metre per cubic metre", precise::one / precise::count},
    unitD{"H61", "millisiemens per centimetre", precise::one / precise::count},
    unitD{"H62", "millivolt per minute", precise::one / precise::count},
    unitD{
        "H63",
        "milligram per square centimetre",
        precise::one / precise::count},
    unitD{"H64", "milligram per gram", precise::one / precise::count},
    unitD{"H65", "millilitre per cubic metre", precise::one / precise::count},
    unitD{"H66", "millimetre per year", precise::one / precise::count},
    unitD{"H67", "millimetre per hour", precise::one / precise::count},
    unitD{"H68", "millimole per gram", precise::one / precise::count},
    unitD{"H69", "picopascal per kilometre", precise::one / precise::count},
    unitD{"H70", "picosecond", precise::one / precise::count},
    unitD{"H71", "percent per month", precise::one / precise::count},
    unitD{"H72", "percent per hectobar", precise::one / precise::count},
    unitD{"H73", "percent per decakelvin", precise::one / precise::count},
    unitD{"H74", "watt per metre", precise::one / precise::count},
    unitD{"H75", "decapascal", precise::one / precise::count},
    unitD{"H76", "gram per millimetre", precise::one / precise::count},
    unitD{"H77", "module width", precise::one / precise::count},
    unitD{
        "H78",
        "conventional centimetre of water",
        precise::one / precise::count},
    unitD{"H79", "French gauge", precise::one / precise::count},
    unitD{"H80", "rack unit", precise::one / precise::count},
    unitD{"H81", "millimetre per minute", precise::one / precise::count},
    unitD{"H82", "big point", precise::one / precise::count},
    unitD{"H83", "litre per kilogram", precise::one / precise::count},
    unitD{"H84", "gram millimetre", precise::one / precise::count},
    unitD{"H85", "reciprocal week", precise::one / precise::count},
    unitD{"H87", "piece", precise::one / precise::count},
    unitD{"H88", "megaohm kilometre", precise::one / precise::count},
    unitD{"H89", "percent per ohm", precise::one / precise::count},
    unitD{"H90", "percent per degree", precise::one / precise::count},
    unitD{"H91", "percent per ten thousand", precise::one / precise::count},
    unitD{
        "H92",
        "percent per one hundred thousand",
        precise::one / precise::count},
    unitD{"H93", "percent per hundred", precise::one / precise::count},
    unitD{"H94", "percent per thousand", precise::one / precise::count},
    unitD{"H95", "percent per volt", precise::one / precise::count},
    unitD{"H96", "percent per bar", precise::one / precise::count},
    unitD{"H98", "percent per inch", precise::one / precise::count},
    unitD{"H99", "percent per metre", precise::one / precise::count},
    unitD{"HA", "hank", precise::one / precise::count},
    unitD{"HAR", "hectare", precise::one / precise::count},
    unitD{"HBA", "hectobar", precise::one / precise::count},
    unitD{"HBX", "hundred boxes", precise::one / precise::count},
    unitD{"HC", "hundred count", precise::one / precise::count},
    unitD{"HD", "half dozen", precise::one / precise::count},
    unitD{"HDW", "hundred kilogram, dry weight", precise::one / precise::count},
    unitD{"HE", "hundredth of a carat", precise::one / precise::count},
    unitD{"HEA", "head", precise::one / precise::count},
    unitD{"HF", "hundred foot", precise::one / precise::count},
    unitD{"HGM", "hectogram", precise::one / precise::count},
    unitD{"HH", "hundred cubic foot", precise::one / precise::count},
    unitD{"HI", "hundred sheet", precise::one / precise::count},
    unitD{"HIU", "hundred international unit", precise::one / precise::count},
    unitD{"HJ", "metric horse power", precise::one / precise::count},
    unitD{"HK", "hundred kilogram", precise::one / precise::count},
    unitD{"HKM", "hundred kilogram, net mass", precise::one / precise::count},
    unitD{"HL", "hundred foot (linear)", precise::one / precise::count},
    unitD{"HLT", "hectolitre", precise::one / precise::count},
    unitD{"HM", "mile per hour (statute mile)", precise::one / precise::count},
    unitD{"HMQ", "million cubic metre", precise::one / precise::count},
    unitD{"HMT", "hectometre", precise::one / precise::count},
    unitD{
        "HN",
        "conventional millimetre of mercury",
        precise::one / precise::count},
    unitD{"HO", "hundred troy ounce", precise::one / precise::count},
    unitD{
        "HP",
        "conventional millimetre of water",
        precise::one / precise::count},
    unitD{"HPA", "hectolitre of pure alcohol", precise::one / precise::count},
    unitD{"HS", "hundred square foot", precise::one / precise::count},
    unitD{"HT", "half hour", precise::one / precise::count},
    unitD{"HTZ", "hertz", precise::one / precise::count},
    unitD{"HUR", "hour", precise::one / precise::count},
    unitD{"HY", "hundred yard", precise::one / precise::count},
    unitD{"IA", "inch pound (pound inch)", precise::one / precise::count},
    unitD{"IC", "count per inch", precise::one / precise::count},
    unitD{"IE", "person", precise::one / precise::count},
    unitD{"IF", "inches of water", precise::one / precise::count},
    unitD{"II", "column inch", precise::one / precise::count},
    unitD{"IL", "inch per minute", precise::one / precise::count},
    unitD{"IM", "impression", precise::one / precise::count},
    unitD{"INH", "inch", precise::one / precise::count},
    unitD{"INK", "square inch", precise::one / precise::count},
    unitD{"INQ", "cubic inch", precise::one / precise::count},
    unitD{"IP", "insurance policy", precise::one / precise::count},
    unitD{"ISD", "international sugar degree", precise::one / precise::count},
    unitD{"IT", "count per centimetre", precise::one / precise::count},
    unitD{"IU", "inch per second", precise::one / precise::count},
    unitD{"IV", "inch per second squared", precise::one / precise::count},
    unitD{"J10", "percent per millimetre", precise::one / precise::count},
    unitD{"J12", "per mille per psi", precise::one / precise::count},
    unitD{"J13", "degree API", precise::one / precise::count},
    unitD{"J14", "degree Baume (origin scale)", precise::one / precise::count},
    unitD{"J15", "degree Baume (US heavy)", precise::one / precise::count},
    unitD{"J16", "degree Baume (US light)", precise::one / precise::count},
    unitD{"J17", "degree Balling", precise::one / precise::count},
    unitD{"J18", "degree Brix", precise::one / precise::count},
    unitD{
        "J19",
        "degree Fahrenheit hour square foot per British thermal unit (thermochemical)",
        precise::one / precise::count},
    unitD{"J2", "joule per kilogram", precise::one / precise::count},
    unitD{"J20", "degree Fahrenheit per kelvin", precise::one / precise::count},
    unitD{"J21", "degree Fahrenheit per bar", precise::one / precise::count},
    unitD{
        "J22",
        "degree Fahrenheit hour square foot per British thermal unit (international table)",
        precise::one / precise::count},
    unitD{"J23", "degree Fahrenheit per hour", precise::one / precise::count},
    unitD{"J24", "degree Fahrenheit per minute", precise::one / precise::count},
    unitD{"J25", "degree Fahrenheit per second", precise::one / precise::count},
    unitD{"J26", "reciprocal degree Fahrenheit", precise::one / precise::count},
    unitD{"J27", "degree Oechsle", precise::one / precise::count},
    unitD{"J28", "degree Rankine per hour", precise::one / precise::count},
    unitD{"J29", "degree Rankine per minute", precise::one / precise::count},
    unitD{"J30", "degree Rankine per second", precise::one / precise::count},
    unitD{"J31", "degree Twaddell", precise::one / precise::count},
    unitD{"J32", "micropoise", precise::one / precise::count},
    unitD{"J33", "microgram per kilogram", precise::one / precise::count},
    unitD{
        "J34",
        "microgram per cubic metre kelvin",
        precise::one / precise::count},
    unitD{
        "J35",
        "microgram per cubic metre bar",
        precise::one / precise::count},
    unitD{"J36", "microlitre per litre", precise::one / precise::count},
    unitD{"J38", "baud", precise::one / precise::count},
    unitD{"J39", "British thermal unit (mean)", precise::one / precise::count},
    unitD{
        "J40",
        "British thermal unit (international table) foot per hour square foot degree Fahrenheit",
        precise::one / precise::count},
    unitD{
        "J41",
        "British thermal unit (international table) inch per hour square foot degree Fahrenheit",
        precise::one / precise::count},
    unitD{
        "J42",
        "British thermal unit (international table) inch per second square foot degree Fahrenheit",
        precise::one / precise::count},
    unitD{
        "J43",
        "British thermal unit (international table) per pound degree Fahrenheit",
        precise::one / precise::count},
    unitD{
        "J44",
        "British thermal unit (international table) per minute",
        precise::one / precise::count},
    unitD{
        "J45",
        "British thermal unit (international table) per second",
        precise::one / precise::count},
    unitD{
        "J46",
        "British thermal unit (thermochemical) foot per hour square foot degree Fahrenheit",
        precise::one / precise::count},
    unitD{
        "J47",
        "British thermal unit (thermochemical) per hour",
        precise::one / precise::count},
    unitD{
        "J48",
        "British thermal unit (thermochemical) inch per hour square foot degree Fahrenheit",
        precise::one / precise::count},
    unitD{
        "J49",
        "British thermal unit (thermochemical) inch per second square foot degree Fahrenheit",
        precise::one / precise::count},
    unitD{
        "J50",
        "British thermal unit (thermochemical) per pound degree Fahrenheit",
        precise::one / precise::count},
    unitD{
        "J51",
        "British thermal unit (thermochemical) per minute",
        precise::one / precise::count},
    unitD{
        "J52",
        "British thermal unit (thermochemical) per second",
        precise::one / precise::count},
    unitD{
        "J53",
        "coulomb square metre per kilogram",
        precise::one / precise::count},
    unitD{"J54", "megabaud", precise::one / precise::count},
    unitD{"J55", "watt second", precise::one / precise::count},
    unitD{"J56", "bar per bar", precise::one / precise::count},
    unitD{"J57", "barrel (UK petroleum)", precise::one / precise::count},
    unitD{
        "J58",
        "barrel (UK petroleum) per minute",
        precise::one / precise::count},
    unitD{
        "J59",
        "barrel (UK petroleum) per day",
        precise::one / precise::count},
    unitD{
        "J60",
        "barrel (UK petroleum) per hour",
        precise::one / precise::count},
    unitD{
        "J61",
        "barrel (UK petroleum) per second",
        precise::one / precise::count},
    unitD{
        "J62",
        "barrel (US petroleum) per hour",
        precise::one / precise::count},
    unitD{
        "J63",
        "barrel (US petroleum) per second",
        precise::one / precise::count},
    unitD{"J64", "bushel (UK) per day", precise::one / precise::count},
    unitD{"J65", "bushel (UK) per hour", precise::one / precise::count},
    unitD{"J66", "bushel (UK) per minute", precise::one / precise::count},
    unitD{"J67", "bushel (UK) per second", precise::one / precise::count},
    unitD{"J68", "bushel (US dry) per day", precise::one / precise::count},
    unitD{"J69", "bushel (US dry) per hour", precise::one / precise::count},
    unitD{"J70", "bushel (US dry) per minute", precise::one / precise::count},
    unitD{"J71", "bushel (US dry) per second", precise::one / precise::count},
    unitD{"J72", "centinewton metre", precise::one / precise::count},
    unitD{"J73", "centipoise per kelvin", precise::one / precise::count},
    unitD{"J74", "centipoise per bar", precise::one / precise::count},
    unitD{"J75", "calorie (mean)", precise::one / precise::count},
    unitD{
        "J76",
        "calorie (international table) per gram degree Celsius",
        precise::one / precise::count},
    unitD{
        "J78",
        "calorie (thermochemical) per centimetre second degree Celsius",
        precise::one / precise::count},
    unitD{
        "J79",
        "calorie (thermochemical) per gram degree Celsius",
        precise::one / precise::count},
    unitD{
        "J81",
        "calorie (thermochemical) per minute",
        precise::one / precise::count},
    unitD{
        "J82",
        "calorie (thermochemical) per second",
        precise::one / precise::count},
    unitD{"J83", "clo", precise::one / precise::count},
    unitD{"J84", "centimetre per second kelvin", precise::one / precise::count},
    unitD{"J85", "centimetre per second bar", precise::one / precise::count},
    unitD{
        "J87",
        "cubic centimetre per cubic metre",
        precise::one / precise::count},
    unitD{"J89", "centimetre of mercury", precise::one / precise::count},
    unitD{"J90", "cubic decimetre per day", precise::one / precise::count},
    unitD{
        "J91",
        "cubic decimetre per cubic metre",
        precise::one / precise::count},
    unitD{"J92", "cubic decimetre per minute", precise::one / precise::count},
    unitD{"J93", "cubic decimetre per second", precise::one / precise::count},
    unitD{"J94", "dyne centimetre", precise::one / precise::count},
    unitD{"J95", "ounce (UK fluid) per day", precise::one / precise::count},
    unitD{"J96", "ounce (UK fluid) per hour", precise::one / precise::count},
    unitD{"J97", "ounce (UK fluid) per minute", precise::one / precise::count},
    unitD{"J98", "ounce (UK fluid) per second", precise::one / precise::count},
    unitD{"J99", "ounce (US fluid) per day", precise::one / precise::count},
    unitD{"JB", "jumbo", precise::one / precise::count},
    unitD{"JE", "joule per kelvin", precise::one / precise::count},
    unitD{"JG", "jug", precise::one / precise::count},
    unitD{"JK", "megajoule per kilogram", precise::one / precise::count},
    unitD{"JM", "megajoule per cubic metre", precise::one / precise::count},
    unitD{"JNT", "pipeline joint", precise::one / precise::count},
    unitD{"JO", "joint", precise::one / precise::count},
    unitD{"JOU", "joule", precise::one / precise::count},
    unitD{"JPS", "hundred metre", precise::one / precise::count},
    unitD{"JR", "jar", precise::one / precise::count},
    unitD{"JWL", "number of jewels", precise::one / precise::count},
    unitD{"K1", "kilowatt demand", precise::one / precise::count},
    unitD{"K10", "ounce (US fluid) per hour", precise::one / precise::count},
    unitD{"K11", "ounce (US fluid) per minute", precise::one / precise::count},
    unitD{"K12", "ounce (US fluid) per second", precise::one / precise::count},
    unitD{"K13", "foot per degree Fahrenheit", precise::one / precise::count},
    unitD{"K14", "foot per hour", precise::one / precise::count},
    unitD{"K15", "foot pound-force per hour", precise::one / precise::count},
    unitD{"K16", "foot pound-force per minute", precise::one / precise::count},
    unitD{"K17", "foot per psi", precise::one / precise::count},
    unitD{
        "K18",
        "foot per second degree Fahrenheit",
        precise::one / precise::count},
    unitD{"K19", "foot per second psi", precise::one / precise::count},
    unitD{
        "K2",
        "kilovolt ampere reactive demand",
        precise::one / precise::count},
    unitD{"K20", "reciprocal cubic foot", precise::one / precise::count},
    unitD{
        "K21",
        "cubic foot per degree Fahrenheit",
        precise::one / precise::count},
    unitD{"K22", "cubic foot per day", precise::one / precise::count},
    unitD{"K23", "cubic foot per psi", precise::one / precise::count},
    unitD{"K24", "foot of water", precise::one / precise::count},
    unitD{"K25", "foot of mercury", precise::one / precise::count},
    unitD{"K26", "gallon (UK) per day", precise::one / precise::count},
    unitD{"K27", "gallon (UK) per hour", precise::one / precise::count},
    unitD{"K28", "gallon (UK) per second", precise::one / precise::count},
    unitD{"K3", "kilovolt ampere reactive hour", precise::one / precise::count},
    unitD{
        "K30",
        "gallon (US liquid) per second",
        precise::one / precise::count},
    unitD{
        "K31",
        "gram-force per square centimetre",
        precise::one / precise::count},
    unitD{"K32", "gill (UK) per day", precise::one / precise::count},
    unitD{"K33", "gill (UK) per hour", precise::one / precise::count},
    unitD{"K34", "gill (UK) per minute", precise::one / precise::count},
    unitD{"K35", "gill (UK) per second", precise::one / precise::count},
    unitD{"K36", "gill (US) per day", precise::one / precise::count},
    unitD{"K37", "gill (US) per hour", precise::one / precise::count},
    unitD{"K38", "gill (US) per minute", precise::one / precise::count},
    unitD{"K39", "gill (US) per second", precise::one / precise::count},
    unitD{
        "K40",
        "standard acceleration of free fall",
        precise::one / precise::count},
    unitD{"K41", "grain per gallon (US)", precise::one / precise::count},
    unitD{"K42", "horsepower (boiler)", precise::one / precise::count},
    unitD{"K43", "horsepower (electric)", precise::one / precise::count},
    unitD{"K45", "inch per degree Fahrenheit", precise::one / precise::count},
    unitD{"K46", "inch per psi", precise::one / precise::count},
    unitD{
        "K47",
        "inch per second degree Fahrenheit",
        precise::one / precise::count},
    unitD{"K48", "inch per second psi", precise::one / precise::count},
    unitD{"K49", "reciprocal cubic inch", precise::one / precise::count},
    unitD{"K5", "kilovolt ampere (reactive)", precise::one / precise::count},
    unitD{"K50", "kilobaud", precise::one / precise::count},
    unitD{"K51", "kilocalorie (mean)", precise::one / precise::count},
    unitD{
        "K52",
        "kilocalorie (international table) per hour metre degree Celsius",
        precise::one / precise::count},
    unitD{"K53", "kilocalorie (thermochemical)", precise::one / precise::count},
    unitD{
        "K54",
        "kilocalorie (thermochemical) per minute",
        precise::one / precise::count},
    unitD{
        "K55",
        "kilocalorie (thermochemical) per second",
        precise::one / precise::count},
    unitD{"K58", "kilomole per hour", precise::one / precise::count},
    unitD{
        "K59",
        "kilomole per cubic metre kelvin",
        precise::one / precise::count},
    unitD{"K6", "kilolitre", precise::one / precise::count},
    unitD{"K60", "kilomole per cubic metre bar", precise::one / precise::count},
    unitD{"K61", "kilomole per minute", precise::one / precise::count},
    unitD{"K62", "litre per litre", precise::one / precise::count},
    unitD{"K63", "reciprocal litre", precise::one / precise::count},
    unitD{
        "K64",
        "pound (avoirdupois) per degree Fahrenheit",
        precise::one / precise::count},
    unitD{
        "K65",
        "pound (avoirdupois) square foot",
        precise::one / precise::count},
    unitD{"K66", "pound (avoirdupois) per day", precise::one / precise::count},
    unitD{"K67", "pound per foot hour", precise::one / precise::count},
    unitD{"K68", "pound per foot second", precise::one / precise::count},
    unitD{
        "K69",
        "pound (avoirdupois) per cubic foot degree Fahrenheit",
        precise::one / precise::count},
    unitD{
        "K70",
        "pound (avoirdupois) per cubic foot psi",
        precise::one / precise::count},
    unitD{
        "K71",
        "pound (avoirdupois) per gallon (UK)",
        precise::one / precise::count},
    unitD{
        "K73",
        "pound (avoirdupois) per hour degree Fahrenheit",
        precise::one / precise::count},
    unitD{
        "K74",
        "pound (avoirdupois) per hour psi",
        precise::one / precise::count},
    unitD{
        "K75",
        "pound (avoirdupois) per cubic inch degree Fahrenheit",
        precise::one / precise::count},
    unitD{
        "K76",
        "pound (avoirdupois) per cubic inch psi",
        precise::one / precise::count},
    unitD{"K77", "pound (avoirdupois) per psi", precise::one / precise::count},
    unitD{
        "K78",
        "pound (avoirdupois) per minute",
        precise::one / precise::count},
    unitD{
        "K79",
        "pound (avoirdupois) per minute degree Fahrenheit",
        precise::one / precise::count},
    unitD{
        "K80",
        "pound (avoirdupois) per minute psi",
        precise::one / precise::count},
    unitD{
        "K81",
        "pound (avoirdupois) per second",
        precise::one / precise::count},
    unitD{
        "K82",
        "pound (avoirdupois) per second degree Fahrenheit",
        precise::one / precise::count},
    unitD{
        "K83",
        "pound (avoirdupois) per second psi",
        precise::one / precise::count},
    unitD{"K84", "pound per cubic yard", precise::one / precise::count},
    unitD{"K85", "pound-force per square foot", precise::one / precise::count},
    unitD{
        "K86",
        "pound-force per square inch degree Fahrenheit",
        precise::one / precise::count},
    unitD{"K87", "psi cubic inch per second", precise::one / precise::count},
    unitD{"K88", "psi litre per second", precise::one / precise::count},
    unitD{"K89", "psi cubic metre per second", precise::one / precise::count},
    unitD{"K90", "psi cubic yard per second", precise::one / precise::count},
    unitD{
        "K91",
        "pound-force second per square foot",
        precise::one / precise::count},
    unitD{
        "K92",
        "pound-force second per square inch",
        precise::one / precise::count},
    unitD{"K93", "reciprocal psi", precise::one / precise::count},
    unitD{"K94", "quart (UK liquid) per day", precise::one / precise::count},
    unitD{"K95", "quart (UK liquid) per hour", precise::one / precise::count},
    unitD{"K96", "quart (UK liquid) per minute", precise::one / precise::count},
    unitD{"K97", "quart (UK liquid) per second", precise::one / precise::count},
    unitD{"K98", "quart (US liquid) per day", precise::one / precise::count},
    unitD{"K99", "quart (US liquid) per hour", precise::one / precise::count},
    unitD{"KA", "cake", precise::one / precise::count},
    unitD{"KAT", "katal", precise::one / precise::count},
    unitD{"KB", "kilocharacter", precise::one / precise::count},
    unitD{"KBA", "kilobar", precise::one / precise::count},
    unitD{"KCC", "kilogram of choline chloride", precise::one / precise::count},
    unitD{"KD", "kilogram decimal", precise::one / precise::count},
    unitD{"KDW", "kilogram drained net weight", precise::one / precise::count},
    unitD{"KEL", "kelvin", precise::one / precise::count},
    unitD{"KF", "kilopacket", precise::one / precise::count},
    unitD{"KG", "keg", precise::one / precise::count},
    unitD{"KGM", "kilogram", precise::one / precise::count},
    unitD{"KGS", "kilogram per second", precise::one / precise::count},
    unitD{
        "KHY",
        "kilogram of hydrogen peroxide",
        precise::one / precise::count},
    unitD{"KHZ", "kilohertz", precise::one / precise::count},
    unitD{"KI", "kilogram per millimetre width", precise::one / precise::count},
    unitD{
        "KIC",
        "kilogram, including container",
        precise::one / precise::count},
    unitD{
        "KIP",
        "kilogram, including inner packaging",
        precise::one / precise::count},
    unitD{"KJ", "kilosegment", precise::one / precise::count},
    unitD{"KJO", "kilojoule", precise::one / precise::count},
    unitD{"KL", "kilogram per metre", precise::one / precise::count},
    unitD{
        "KLK",
        "lactic dry material percentage",
        precise::one / precise::count},
    unitD{"KLX", "kilolux", precise::one / precise::count},
    unitD{"KMA", "kilogram of methylamine", precise::one / precise::count},
    unitD{"KMH", "kilometre per hour", precise::one / precise::count},
    unitD{"KMK", "square kilometre", precise::one / precise::count},
    unitD{"KMQ", "kilogram per cubic metre", precise::one / precise::count},
    unitD{"KMT", "kilometre", precise::one / precise::count},
    unitD{"KNI", "kilogram of nitrogen", precise::one / precise::count},
    unitD{"KNS", "kilogram named substance", precise::one / precise::count},
    unitD{"KNT", "knot", precise::one / precise::count},
    unitD{
        "KO",
        "milliequivalence caustic potash per gram of product",
        precise::one / precise::count},
    unitD{"KPA", "kilopascal", precise::one / precise::count},
    unitD{
        "KPH",
        "kilogram of potassium hydroxide (caustic potash)",
        precise::one / precise::count},
    unitD{"KPO", "kilogram of potassium oxide", precise::one / precise::count},
    unitD{
        "KPP",
        "kilogram of phosphorus pentoxide (phosphoric anhydride)",
        precise::one / precise::count},
    unitD{"KR", "kiloroentgen", precise::one / precise::count},
    unitD{
        "KS",
        "thousand pound per square inch",
        precise::one / precise::count},
    unitD{
        "KSD",
        "kilogram of substance 90 % dry",
        precise::one / precise::count},
    unitD{
        "KSH",
        "kilogram of sodium hydroxide (caustic soda)",
        precise::one / precise::count},
    unitD{"KT", "kit", precise::one / precise::count},
    unitD{"KTM", "kilometre", precise::one / precise::count},
    unitD{"KTN", "kilotonne", precise::one / precise::count},
    unitD{"KUR", "kilogram of uranium", precise::one / precise::count},
    unitD{"KVA", "kilovolt - ampere", precise::one / precise::count},
    unitD{"KVR", "kilovar", precise::one / precise::count},
    unitD{"KVT", "kilovolt", precise::one / precise::count},
    unitD{"KW", "kilogram per millimetre", precise::one / precise::count},
    unitD{"KWH", "kilowatt hour", precise::one / precise::count},
    unitD{
        "KWO",
        "kilogram of tungsten trioxide",
        precise::one / precise::count},
    unitD{"KWT", "kilowatt", precise::one / precise::count},
    unitD{"KX", "millilitre per kilogram", precise::one / precise::count},
    unitD{"L10", "quart (US liquid) per minute", precise::one / precise::count},
    unitD{"L11", "quart (US liquid) per second", precise::one / precise::count},
    unitD{"L12", "metre per second kelvin", precise::one / precise::count},
    unitD{"L13", "metre per second bar", precise::one / precise::count},
    unitD{
        "L14",
        "square metre hour degree Celsius per kilocalorie (international table)",
        precise::one / precise::count},
    unitD{
        "L15",
        "millipascal second per kelvin",
        precise::one / precise::count},
    unitD{"L16", "millipascal second per bar", precise::one / precise::count},
    unitD{
        "L17",
        "milligram per cubic metre kelvin",
        precise::one / precise::count},
    unitD{
        "L18",
        "milligram per cubic metre bar",
        precise::one / precise::count},
    unitD{"L19", "millilitre per litre", precise::one / precise::count},
    unitD{"L2", "litre per minute", precise::one / precise::count},
    unitD{"L20", "reciprocal cubic millimetre", precise::one / precise::count},
    unitD{
        "L21",
        "cubic millimetre per cubic metre",
        precise::one / precise::count},
    unitD{"L23", "mole per hour", precise::one / precise::count},
    unitD{"L24", "mole per kilogram kelvin", precise::one / precise::count},
    unitD{"L25", "mole per kilogram bar", precise::one / precise::count},
    unitD{"L26", "mole per litre kelvin", precise::one / precise::count},
    unitD{"L27", "mole per litre bar", precise::one / precise::count},
    unitD{"L28", "mole per cubic metre kelvin", precise::one / precise::count},
    unitD{"L29", "mole per cubic metre bar", precise::one / precise::count},
    unitD{"L30", "mole per minute", precise::one / precise::count},
    unitD{
        "L31",
        "milliroentgen aequivalent men",
        precise::one / precise::count},
    unitD{"L32", "nanogram per kilogram", precise::one / precise::count},
    unitD{"L33", "ounce (avoirdupois) per day", precise::one / precise::count},
    unitD{"L34", "ounce (avoirdupois) per hour", precise::one / precise::count},
    unitD{
        "L35",
        "ounce (avoirdupois) per minute",
        precise::one / precise::count},
    unitD{
        "L36",
        "ounce (avoirdupois) per second",
        precise::one / precise::count},
    unitD{
        "L37",
        "ounce (avoirdupois) per gallon (UK)",
        precise::one / precise::count},
    unitD{
        "L38",
        "ounce (avoirdupois) per gallon (US)",
        precise::one / precise::count},
    unitD{
        "L39",
        "ounce (avoirdupois) per cubic inch",
        precise::one / precise::count},
    unitD{"L40", "ounce (avoirdupois)-force", precise::one / precise::count},
    unitD{
        "L41",
        "ounce (avoirdupois)-force inch",
        precise::one / precise::count},
    unitD{"L42", "picosiemens per metre", precise::one / precise::count},
    unitD{"L43", "peck (UK)", precise::one / precise::count},
    unitD{"L44", "peck (UK) per day", precise::one / precise::count},
    unitD{"L45", "peck (UK) per hour", precise::one / precise::count},
    unitD{"L46", "peck (UK) per minute", precise::one / precise::count},
    unitD{"L47", "peck (UK) per second", precise::one / precise::count},
    unitD{"L48", "peck (US dry) per day", precise::one / precise::count},
    unitD{"L49", "peck (US dry) per hour", precise::one / precise::count},
    unitD{"L50", "peck (US dry) per minute", precise::one / precise::count},
    unitD{"L51", "peck (US dry) per second", precise::one / precise::count},
    unitD{"L52", "psi per psi", precise::one / precise::count},
    unitD{"L53", "pint (UK) per day", precise::one / precise::count},
    unitD{"L54", "pint (UK) per hour", precise::one / precise::count},
    unitD{"L55", "pint (UK) per minute", precise::one / precise::count},
    unitD{"L56", "pint (UK) per second", precise::one / precise::count},
    unitD{"L57", "pint (US liquid) per day", precise::one / precise::count},
    unitD{"L58", "pint (US liquid) per hour", precise::one / precise::count},
    unitD{"L59", "pint (US liquid) per minute", precise::one / precise::count},
    unitD{"L60", "pint (US liquid) per second", precise::one / precise::count},
    unitD{"L61", "pint (US dry)", precise::one / precise::count},
    unitD{"L62", "quart (US dry)", precise::one / precise::count},
    unitD{"L63", "slug per day", precise::one / precise::count},
    unitD{"L64", "slug per foot second", precise::one / precise::count},
    unitD{"L65", "slug per cubic foot", precise::one / precise::count},
    unitD{"L66", "slug per hour", precise::one / precise::count},
    unitD{"L67", "slug per minute", precise::one / precise::count},
    unitD{"L68", "slug per second", precise::one / precise::count},
    unitD{"L69", "tonne per kelvin", precise::one / precise::count},
    unitD{"L70", "tonne per bar", precise::one / precise::count},
    unitD{"L71", "tonne per day", precise::one / precise::count},
    unitD{"L72", "tonne per day kelvin", precise::one / precise::count},
    unitD{"L73", "tonne per day bar", precise::one / precise::count},
    unitD{"L74", "tonne per hour kelvin", precise::one / precise::count},
    unitD{"L75", "tonne per hour bar", precise::one / precise::count},
    unitD{"L76", "tonne per cubic metre kelvin", precise::one / precise::count},
    unitD{"L77", "tonne per cubic metre bar", precise::one / precise::count},
    unitD{"L78", "tonne per minute", precise::one / precise::count},
    unitD{"L79", "tonne per minute kelvin", precise::one / precise::count},
    unitD{"L80", "tonne per minute bar", precise::one / precise::count},
    unitD{"L81", "tonne per second", precise::one / precise::count},
    unitD{"L82", "tonne per second kelvin", precise::one / precise::count},
    unitD{"L83", "tonne per second bar", precise::one / precise::count},
    unitD{"L84", "ton (UK shipping)", precise::one / precise::count},
    unitD{"L85", "ton long per day", precise::one / precise::count},
    unitD{"L86", "ton (US shipping)", precise::one / precise::count},
    unitD{
        "L87",
        "ton short per degree Fahrenheit",
        precise::one / precise::count},
    unitD{"L88", "ton short per day", precise::one / precise::count},
    unitD{
        "L89",
        "ton short per hour degree Fahrenheit",
        precise::one / precise::count},
    unitD{"L90", "ton short per hour psi", precise::one / precise::count},
    unitD{"L91", "ton short per psi", precise::one / precise::count},
    unitD{"L92", "ton (UK long) per cubic yard", precise::one / precise::count},
    unitD{
        "L93",
        "ton (US short) per cubic yard",
        precise::one / precise::count},
    unitD{"L94", "ton-force (US short)", precise::one / precise::count},
    unitD{"L95", "common year", precise::one / precise::count},
    unitD{"L96", "sidereal year", precise::one / precise::count},
    unitD{"L98", "yard per degree Fahrenheit", precise::one / precise::count},
    unitD{"L99", "yard per psi", precise::one / precise::count},
    unitD{"LA", "pound per cubic inch", precise::one / precise::count},
    unitD{"LAC", "lactose excess percentage", precise::one / precise::count},
    unitD{"LBR", "pound", precise::one / precise::count},
    unitD{"LBT", "troy pound (US)", precise::one / precise::count},
    unitD{"LC", "linear centimetre", precise::one / precise::count},
    unitD{"LD", "litre per day", precise::one / precise::count},
    unitD{"LE", "lite", precise::one / precise::count},
    unitD{"LEF", "leaf", precise::one / precise::count},
    unitD{"LF", "linear foot", precise::one / precise::count},
    unitD{"LH", "labour hour", precise::one / precise::count},
    unitD{"LI", "linear inch", precise::one / precise::count},
    unitD{"LJ", "large spray", precise::one / precise::count},
    unitD{"LK", "link", precise::one / precise::count},
    unitD{"LM", "linear metre", precise::one / precise::count},
    unitD{"LN", "length", precise::one / precise::count},
    unitD{"LO", "lot  [unit of procurement]", precise::one / precise::count},
    unitD{"LP", "liquid pound", precise::one / precise::count},
    unitD{"LPA", "litre of pure alcohol", precise::one / precise::count},
    unitD{"LR", "layer", precise::one / precise::count},
    unitD{"LS", "lump sum", precise::one / precise::count},
    unitD{"LTN", "ton (UK) or long ton (US)", precise::one / precise::count},
    unitD{"LTR", "litre", precise::one / precise::count},
    unitD{"LUB", "metric ton, lubricating oil", precise::one / precise::count},
    unitD{"LUM", "lumen", precise::one / precise::count},
    unitD{"LUX", "lux", precise::one / precise::count},
    unitD{"LX", "linear yard per pound", precise::one / precise::count},
    unitD{"LY", "linear yard", precise::one / precise::count},
    unitD{"M0", "magnetic tape", precise::one / precise::count},
    unitD{"M1", "milligram per litre", precise::one / precise::count},
    unitD{"M10", "reciprocal cubic yard", precise::one / precise::count},
    unitD{
        "M11",
        "cubic yard per degree Fahrenheit",
        precise::one / precise::count},
    unitD{"M12", "cubic yard per day", precise::one / precise::count},
    unitD{"M13", "cubic yard per hour", precise::one / precise::count},
    unitD{"M14", "cubic yard per psi", precise::one / precise::count},
    unitD{"M15", "cubic yard per minute", precise::one / precise::count},
    unitD{"M16", "cubic yard per second", precise::one / precise::count},
    unitD{"M17", "kilohertz metre", precise::one / precise::count},
    unitD{"M18", "gigahertz metre", precise::one / precise::count},
    unitD{"M19", "Beaufort", precise::one / precise::count},
    unitD{
        "M20",
        "reciprocal megakelvin or megakelvin to the power minus one",
        precise::one / precise::count},
    unitD{
        "M21",
        "reciprocal kilovolt - ampere reciprocal hour",
        precise::one / precise::count},
    unitD{
        "M22",
        "millilitre per square centimetre minute",
        precise::one / precise::count},
    unitD{"M23", "newton per centimetre", precise::one / precise::count},
    unitD{"M24", "ohm kilometre", precise::one / precise::count},
    unitD{"M25", "percent per degree Celsius", precise::one / precise::count},
    unitD{"M26", "gigaohm per metre", precise::one / precise::count},
    unitD{"M27", "megahertz metre", precise::one / precise::count},
    unitD{"M29", "kilogram per kilogram", precise::one / precise::count},
    unitD{
        "M30",
        "reciprocal volt - ampere reciprocal second",
        precise::one / precise::count},
    unitD{"M31", "kilogram per kilometre", precise::one / precise::count},
    unitD{"M32", "pascal second per litre", precise::one / precise::count},
    unitD{"M33", "millimole per litre", precise::one / precise::count},
    unitD{
        "M34",
        "newton metre per square metre",
        precise::one / precise::count},
    unitD{"M35", "millivolt - ampere", precise::one / precise::count},
    unitD{"M36", "30-day month", precise::one / precise::count},
    unitD{"M37", "actual/360", precise::one / precise::count},
    unitD{"M38", "kilometre per second squared", precise::one / precise::count},
    unitD{
        "M39",
        "centimetre per second squared",
        precise::one / precise::count},
    unitD{"M4", "monetary value", precise::one / precise::count},
    unitD{"M40", "yard per second squared", precise::one / precise::count},
    unitD{
        "M41",
        "millimetre per second squared",
        precise::one / precise::count},
    unitD{
        "M42",
        "mile (statute mile) per second squared",
        precise::one / precise::count},
    unitD{"M43", "mil", precise::one / precise::count},
    unitD{"M44", "revolution", precise::one / precise::count},
    unitD{
        "M45",
        "degree [unit of angle] per second squared",
        precise::one / precise::count},
    unitD{"M46", "revolution per minute", precise::one / precise::count},
    unitD{"M47", "circular mil", precise::one / precise::count},
    unitD{
        "M48",
        "square mile (based on U.S. survey foot)",
        precise::one / precise::count},
    unitD{
        "M49",
        "chain (based on U.S. survey foot)",
        precise::one / precise::count},
    unitD{"M5", "microcurie", precise::one / precise::count},
    unitD{"M50", "furlong", precise::one / precise::count},
    unitD{"M51", "foot (U.S. survey)", precise::one / precise::count},
    unitD{
        "M52",
        "mile (based on U.S. survey foot)",
        precise::one / precise::count},
    unitD{"M53", "metre per pascal", precise::one / precise::count},
    unitD{"M55", "metre per radiant", precise::one / precise::count},
    unitD{"M56", "shake", precise::one / precise::count},
    unitD{"M57", "mile per minute", precise::one / precise::count},
    unitD{"M58", "mile per second", precise::one / precise::count},
    unitD{"M59", "metre per second pascal", precise::one / precise::count},
    unitD{"M60", "metre per hour", precise::one / precise::count},
    unitD{"M61", "inch per year", precise::one / precise::count},
    unitD{"M62", "kilometre per second", precise::one / precise::count},
    unitD{"M63", "inch per minute", precise::one / precise::count},
    unitD{"M64", "yard per second", precise::one / precise::count},
    unitD{"M65", "yard per minute", precise::one / precise::count},
    unitD{"M66", "yard per hour", precise::one / precise::count},
    unitD{
        "M67",
        "acre-foot (based on U.S. survey foot)",
        precise::one / precise::count},
    unitD{"M68", "cord (128 ft3)", precise::one / precise::count},
    unitD{"M69", "cubic mile (UK statute)", precise::one / precise::count},
    unitD{"M7", "micro-inch", precise::one / precise::count},
    unitD{"M70", "ton, register", precise::one / precise::count},
    unitD{"M71", "cubic metre per pascal", precise::one / precise::count},
    unitD{"M72", "bel", precise::one / precise::count},
    unitD{
        "M73",
        "kilogram per cubic metre pascal",
        precise::one / precise::count},
    unitD{"M74", "kilogram per pascal", precise::one / precise::count},
    unitD{"M75", "kilopound-force", precise::one / precise::count},
    unitD{"M76", "poundal", precise::one / precise::count},
    unitD{
        "M77",
        "kilogram metre per second squared",
        precise::one / precise::count},
    unitD{"M78", "pond", precise::one / precise::count},
    unitD{"M79", "square foot per hour", precise::one / precise::count},
    unitD{"M80", "stokes per pascal", precise::one / precise::count},
    unitD{"M81", "square centimetre per second", precise::one / precise::count},
    unitD{
        "M82",
        "square metre per second pascal",
        precise::one / precise::count},
    unitD{"M83", "denier", precise::one / precise::count},
    unitD{"M84", "pound per yard", precise::one / precise::count},
    unitD{"M85", "ton, assay", precise::one / precise::count},
    unitD{"M86", "pfund", precise::one / precise::count},
    unitD{"M87", "kilogram per second pascal", precise::one / precise::count},
    unitD{"M88", "tonne per month", precise::one / precise::count},
    unitD{"M89", "tonne per year", precise::one / precise::count},
    unitD{
        "M9",
        "million Btu per 1000 cubic foot",
        precise::one / precise::count},
    unitD{"M90", "kilopound per hour", precise::one / precise::count},
    unitD{"M91", "pound per pound", precise::one / precise::count},
    unitD{"M92", "pound-force foot", precise::one / precise::count},
    unitD{"M93", "newton metre per radian", precise::one / precise::count},
    unitD{"M94", "kilogram metre", precise::one / precise::count},
    unitD{"M95", "poundal foot", precise::one / precise::count},
    unitD{"M96", "poundal inch", precise::one / precise::count},
    unitD{"M97", "dyne metre", precise::one / precise::count},
    unitD{
        "M98",
        "kilogram centimetre per second",
        precise::one / precise::count},
    unitD{"M99", "gram centimetre per second", precise::one / precise::count},
    unitD{"MA", "machine per unit", precise::one / precise::count},
    unitD{
        "MAH",
        "megavolt ampere reactive hour",
        precise::one / precise::count},
    unitD{"MAL", "megalitre", precise::one / precise::count},
    unitD{"MAM", "megametre", precise::one / precise::count},
    unitD{"MAR", "megavar", precise::one / precise::count},
    unitD{"MAW", "megawatt", precise::one / precise::count},
    unitD{
        "MBE",
        "thousand standard brick equivalent",
        precise::one / precise::count},
    unitD{"MBF", "thousand board foot", precise::one / precise::count},
    unitD{"MBR", "millibar", precise::one / precise::count},
    unitD{"MC", "microgram", precise::one / precise::count},
    unitD{"MCU", "millicurie", precise::one / precise::count},
    unitD{"MD", "air dry metric ton", precise::one / precise::count},
    unitD{
        "MF",
        "milligram per square foot per side",
        precise::one / precise::count},
    unitD{"MGM", "milligram", precise::one / precise::count},
    unitD{"MHZ", "megahertz", precise::one / precise::count},
    unitD{"MIK", "square mile (statute mile)", precise::one / precise::count},
    unitD{"MIL", "thousand", precise::one / precise::count},
    unitD{"MIN", "minute [unit of time]", precise::one / precise::count},
    unitD{"MIO", "million", precise::one / precise::count},
    unitD{"MIU", "million international unit", precise::one / precise::count},
    unitD{"MK", "milligram per square inch", precise::one / precise::count},
    unitD{"MLD", "milliard", precise::one / precise::count},
    unitD{"MLT", "millilitre", precise::one / precise::count},
    unitD{"MMK", "square millimetre", precise::one / precise::count},
    unitD{"MMQ", "cubic millimetre", precise::one / precise::count},
    unitD{"MMT", "millimetre", precise::one / precise::count},
    unitD{"MND", "kilogram, dry weight", precise::one / precise::count},
    unitD{"MON", "month", precise::one / precise::count},
    unitD{"MPA", "megapascal", precise::one / precise::count},
    unitD{"MQ", "thousand metre", precise::one / precise::count},
    unitD{"MQH", "cubic metre per hour", precise::one / precise::count},
    unitD{"MQS", "cubic metre per second", precise::one / precise::count},
    unitD{"MSK", "metre per second squared", precise::one / precise::count},
    unitD{"MT", "mat", precise::one / precise::count},
    unitD{"MTK", "square metre", precise::one / precise::count},
    unitD{"MTQ", "cubic metre", precise::one / precise::count},
    unitD{"MTR", "metre", precise::one / precise::count},
    unitD{"MTS", "metre per second", precise::one / precise::count},
    unitD{"MV", "number of mults", precise::one / precise::count},
    unitD{"MVA", "megavolt - ampere", precise::one / precise::count},
    unitD{"MWH", "megawatt hour (1000 kW.h)", precise::one / precise::count},
    unitD{"N1", "pen calorie", precise::one / precise::count},
    unitD{"N10", "pound foot per second", precise::one / precise::count},
    unitD{"N11", "pound inch per second", precise::one / precise::count},
    unitD{"N12", "Pferdestaerke", precise::one / precise::count},
    unitD{
        "N13",
        "centimetre of mercury (0 degC)",
        precise::one / precise::count},
    unitD{"N14", "centimetre of water (4 degC)", precise::one / precise::count},
    unitD{"N15", "foot of water (39.2 degF)", precise::one / precise::count},
    unitD{"N16", "inch of mercury (32 degF)", precise::one / precise::count},
    unitD{"N17", "inch of mercury (60 degF)", precise::one / precise::count},
    unitD{"N18", "inch of water (39.2 degF)", precise::one / precise::count},
    unitD{"N19", "inch of water (60 degF)", precise::one / precise::count},
    unitD{"N2", "number of lines", precise::one / precise::count},
    unitD{"N20", "kip per square inch", precise::one / precise::count},
    unitD{"N21", "poundal per square foot", precise::one / precise::count},
    unitD{
        "N22",
        "ounce (avoirdupois) per square inch",
        precise::one / precise::count},
    unitD{"N23", "conventional metre of water", precise::one / precise::count},
    unitD{"N24", "gram per square millimetre", precise::one / precise::count},
    unitD{"N25", "pound per square yard", precise::one / precise::count},
    unitD{"N26", "poundal per square inch", precise::one / precise::count},
    unitD{"N27", "foot to the fourth power", precise::one / precise::count},
    unitD{"N28", "cubic decimetre per kilogram", precise::one / precise::count},
    unitD{"N29", "cubic foot per pound", precise::one / precise::count},
    unitD{"N3", "print point", precise::one / precise::count},
    unitD{"N30", "cubic inch per pound", precise::one / precise::count},
    unitD{"N31", "kilonewton per metre", precise::one / precise::count},
    unitD{"N32", "poundal per inch", precise::one / precise::count},
    unitD{"N33", "pound-force per yard", precise::one / precise::count},
    unitD{
        "N34",
        "poundal second per square foot",
        precise::one / precise::count},
    unitD{"N35", "poise per pascal", precise::one / precise::count},
    unitD{
        "N36",
        "newton second per square metre",
        precise::one / precise::count},
    unitD{"N37", "kilogram per metre second", precise::one / precise::count},
    unitD{"N38", "kilogram per metre minute", precise::one / precise::count},
    unitD{"N39", "kilogram per metre day", precise::one / precise::count},
    unitD{"N40", "kilogram per metre hour", precise::one / precise::count},
    unitD{"N41", "gram per centimetre second", precise::one / precise::count},
    unitD{
        "N42",
        "poundal second per square inch",
        precise::one / precise::count},
    unitD{"N43", "pound per foot minute", precise::one / precise::count},
    unitD{"N44", "pound per foot day", precise::one / precise::count},
    unitD{
        "N45",
        "cubic metre per second pascal",
        precise::one / precise::count},
    unitD{"N46", "foot poundal", precise::one / precise::count},
    unitD{"N47", "inch poundal", precise::one / precise::count},
    unitD{"N48", "watt per square centimetre", precise::one / precise::count},
    unitD{"N49", "watt per square inch", precise::one / precise::count},
    unitD{
        "N50",
        "British thermal unit (international table) per square foot hour",
        precise::one / precise::count},
    unitD{
        "N51",
        "British thermal unit (thermochemical) per square foot hour",
        precise::one / precise::count},
    unitD{
        "N52",
        "British thermal unit (thermochemical) per square foot minute",
        precise::one / precise::count},
    unitD{
        "N53",
        "British thermal unit (international table) per square foot second",
        precise::one / precise::count},
    unitD{
        "N54",
        "British thermal unit (thermochemical) per square foot second",
        precise::one / precise::count},
    unitD{
        "N55",
        "British thermal unit (international table) per square inch second",
        precise::one / precise::count},
    unitD{
        "N56",
        "calorie (thermochemical) per square centimetre minute",
        precise::one / precise::count},
    unitD{
        "N57",
        "calorie (thermochemical) per square centimetre second",
        precise::one / precise::count},
    unitD{
        "N58",
        "British thermal unit (international table) per cubic foot",
        precise::one / precise::count},
    unitD{
        "N59",
        "British thermal unit (thermochemical) per cubic foot",
        precise::one / precise::count},
    unitD{
        "N60",
        "British thermal unit (international table) per degree Fahrenheit",
        precise::one / precise::count},
    unitD{
        "N61",
        "British thermal unit (thermochemical) per degree Fahrenheit",
        precise::one / precise::count},
    unitD{
        "N62",
        "British thermal unit (international table) per degree Rankine",
        precise::one / precise::count},
    unitD{
        "N63",
        "British thermal unit (thermochemical) per degree Rankine",
        precise::one / precise::count},
    unitD{
        "N64",
        "British thermal unit (thermochemical) per pound degree Rankine",
        precise::one / precise::count},
    unitD{
        "N65",
        "kilocalorie (international table) per gram kelvin",
        precise::one / precise::count},
    unitD{
        "N66",
        "British thermal unit (39 degF)",
        precise::one / precise::count},
    unitD{
        "N67",
        "British thermal unit (59 degF)",
        precise::one / precise::count},
    unitD{
        "N68",
        "British thermal unit (60 degF)",
        precise::one / precise::count},
    unitD{"N69", "calorie (20 degC)", precise::one / precise::count},
    unitD{"N70", "quad (1015 BtuIT)", precise::one / precise::count},
    unitD{"N71", "therm (EC)", precise::one / precise::count},
    unitD{"N72", "therm (U.S.)", precise::one / precise::count},
    unitD{
        "N73",
        "British thermal unit (thermochemical) per pound",
        precise::one / precise::count},
    unitD{
        "N74",
        "British thermal unit (international table) per hour square foot degree Fahrenheit",
        precise::one / precise::count},
    unitD{
        "N75",
        "British thermal unit (thermochemical) per hour square foot degree Fahrenheit",
        precise::one / precise::count},
    unitD{
        "N76",
        "British thermal unit (international table) per second square foot degree Fahrenheit",
        precise::one / precise::count},
    unitD{
        "N77",
        "British thermal unit (thermochemical) per second square foot degree Fahrenheit",
        precise::one / precise::count},
    unitD{
        "N78",
        "kilowatt per square metre kelvin",
        precise::one / precise::count},
    unitD{"N79", "kelvin per pascal", precise::one / precise::count},
    unitD{
        "N80",
        "watt per metre degree Celsius",
        precise::one / precise::count},
    unitD{"N81", "kilowatt per metre kelvin", precise::one / precise::count},
    unitD{
        "N82",
        "kilowatt per metre degree Celsius",
        precise::one / precise::count},
    unitD{
        "N83",
        "metre per degree Celcius metre",
        precise::one / precise::count},
    unitD{
        "N84",
        "degree Fahrenheit hour per British thermal unit (international table)",
        precise::one / precise::count},
    unitD{
        "N85",
        "degree Fahrenheit hour per British thermal unit (thermochemical)",
        precise::one / precise::count},
    unitD{
        "N86",
        "degree Fahrenheit second per British thermal unit (international table)",
        precise::one / precise::count},
    unitD{
        "N87",
        "degree Fahrenheit second per British thermal unit (thermochemical)",
        precise::one / precise::count},
    unitD{
        "N88",
        "degree Fahrenheit hour square foot per British thermal unit (international table) inch",
        precise::one / precise::count},
    unitD{
        "N89",
        "degree Fahrenheit hour square foot per British thermal unit (thermochemical) inch",
        precise::one / precise::count},
    unitD{"N90", "kilofarad", precise::one / precise::count},
    unitD{"N91", "reciprocal joule", precise::one / precise::count},
    unitD{"N92", "picosiemens", precise::one / precise::count},
    unitD{"N93", "ampere per pascal", precise::one / precise::count},
    unitD{"N94", "franklin", precise::one / precise::count},
    unitD{"N95", "ampere minute", precise::one / precise::count},
    unitD{"N96", "biot", precise::one / precise::count},
    unitD{"N97", "gilbert", precise::one / precise::count},
    unitD{"N98", "volt per pascal", precise::one / precise::count},
    unitD{"N99", "picovolt", precise::one / precise::count},
    unitD{"NA", "milligram per kilogram", precise::one / precise::count},
    unitD{"NAR", "number of articles", precise::one / precise::count},
    unitD{"NB", "barge", precise::one / precise::count},
    unitD{"NBB", "number of bobbins", precise::one / precise::count},
    unitD{"NC", "car", precise::one / precise::count},
    unitD{"NCL", "number of cells", precise::one / precise::count},
    unitD{"ND", "net barrel", precise::one / precise::count},
    unitD{"NE", "net litre", precise::one / precise::count},
    unitD{"NEW", "newton", precise::one / precise::count},
    unitD{"NF", "message", precise::one / precise::count},
    unitD{"NG", "net gallon (us)", precise::one / precise::count},
    unitD{"NH", "message hour", precise::one / precise::count},
    unitD{"NI", "net imperial gallon", precise::one / precise::count},
    unitD{"NIL", "nil", precise::one / precise::count},
    unitD{
        "NIU",
        "number of international units",
        precise::one / precise::count},
    unitD{"NJ", "number of screens", precise::one / precise::count},
    unitD{"NL", "load", precise::one / precise::count},
    unitD{"NMI", "nautical mile", precise::one / precise::count},
    unitD{"NMP", "number of packs", precise::one / precise::count},
    unitD{"NN", "train", precise::one / precise::count},
    unitD{"NPL", "number of parcels", precise::one / precise::count},
    unitD{"NPR", "number of pairs", precise::one / precise::count},
    unitD{"NPT", "number of parts", precise::one / precise::count},
    unitD{"NQ", "mho", precise::one / precise::count},
    unitD{"NR", "micromho", precise::one / precise::count},
    unitD{"NRL", "number of rolls", precise::one / precise::count},
    unitD{"NT", "net ton", precise::one / precise::count},
    unitD{"NTT", "net register ton", precise::one / precise::count},
    unitD{"NU", "newton metre", precise::one / precise::count},
    unitD{"NV", "vehicle", precise::one / precise::count},
    unitD{"NX", "part per thousand", precise::one / precise::count},
    unitD{"NY", "pound per air dry metric ton", precise::one / precise::count},
    unitD{"OA", "panel", precise::one / precise::count},
    unitD{"ODE", "ozone depletion equivalent", precise::one / precise::count},
    unitD{"OHM", "ohm", precise::one / precise::count},
    unitD{"ON", "ounce per square yard", precise::one / precise::count},
    unitD{"ONZ", "ounce (avoirdupois)", precise::one / precise::count},
    unitD{"OP", "two pack", precise::one / precise::count},
    unitD{"OT", "overtime hour", precise::one / precise::count},
    unitD{"OZ", "ounce av", precise::one / precise::count},
    unitD{"OZA", "fluid ounce (US)", precise::one / precise::count},
    unitD{"OZI", "fluid ounce (UK)", precise::one / precise::count},
    unitD{"P0", "page - electronic", precise::one / precise::count},
    unitD{"P1", "percent", precise::one / precise::count},
    unitD{"P10", "coulomb per metre", precise::one / precise::count},
    unitD{"P11", "kiloweber", precise::one / precise::count},
    unitD{"P12", "gamma", precise::one / precise::count},
    unitD{"P13", "kilotesla", precise::one / precise::count},
    unitD{"P14", "joule per second", precise::one / precise::count},
    unitD{"P15", "joule per minute", precise::one / precise::count},
    unitD{"P16", "joule per hour", precise::one / precise::count},
    unitD{"P17", "joule per day", precise::one / precise::count},
    unitD{"P18", "kilojoule per second", precise::one / precise::count},
    unitD{"P19", "kilojoule per minute", precise::one / precise::count},
    unitD{"P2", "pound per foot", precise::one / precise::count},
    unitD{"P20", "kilojoule per hour", precise::one / precise::count},
    unitD{"P21", "kilojoule per day", precise::one / precise::count},
    unitD{"P22", "nanoohm", precise::one / precise::count},
    unitD{"P23", "ohm circular-mil per foot", precise::one / precise::count},
    unitD{"P24", "kilohenry", precise::one / precise::count},
    unitD{"P25", "lumen per square foot", precise::one / precise::count},
    unitD{"P26", "phot", precise::one / precise::count},
    unitD{"P27", "footcandle", precise::one / precise::count},
    unitD{"P28", "candela per square inch", precise::one / precise::count},
    unitD{"P29", "footlambert", precise::one / precise::count},
    unitD{"P3", "three pack", precise::one / precise::count},
    unitD{"P30", "lambert", precise::one / precise::count},
    unitD{"P31", "stilb", precise::one / precise::count},
    unitD{"P32", "candela per square foot", precise::one / precise::count},
    unitD{"P33", "kilocandela", precise::one / precise::count},
    unitD{"P34", "millicandela", precise::one / precise::count},
    unitD{"P35", "Hefner-Kerze", precise::one / precise::count},
    unitD{"P36", "international candle", precise::one / precise::count},
    unitD{
        "P37",
        "British thermal unit (international table) per square foot",
        precise::one / precise::count},
    unitD{
        "P38",
        "British thermal unit (thermochemical) per square foot",
        precise::one / precise::count},
    unitD{
        "P39",
        "calorie (thermochemical) per square centimetre",
        precise::one / precise::count},
    unitD{"P4", "four pack", precise::one / precise::count},
    unitD{"P40", "langley", precise::one / precise::count},
    unitD{"P41", "decade (logarithmic)", precise::one / precise::count},
    unitD{"P42", "pascal squared second", precise::one / precise::count},
    unitD{"P43", "bel per metre", precise::one / precise::count},
    unitD{"P44", "pound mole", precise::one / precise::count},
    unitD{"P45", "pound mole per second", precise::one / precise::count},
    unitD{"P46", "pound mole per minute", precise::one / precise::count},
    unitD{"P47", "kilomole per kilogram", precise::one / precise::count},
    unitD{"P48", "pound mole per pound", precise::one / precise::count},
    unitD{
        "P49",
        "newton square metre per ampere",
        precise::one / precise::count},
    unitD{"P5", "five pack", precise::one / precise::count},
    unitD{"P50", "weber metre", precise::one / precise::count},
    unitD{"P51", "mol per kilogram pascal", precise::one / precise::count},
    unitD{"P52", "mol per cubic metre pascal", precise::one / precise::count},
    unitD{"P53", "unit pole", precise::one / precise::count},
    unitD{"P54", "milligray per second", precise::one / precise::count},
    unitD{"P55", "microgray per second", precise::one / precise::count},
    unitD{"P56", "nanogray per second", precise::one / precise::count},
    unitD{"P57", "gray per minute", precise::one / precise::count},
    unitD{"P58", "milligray per minute", precise::one / precise::count},
    unitD{"P59", "microgray per minute", precise::one / precise::count},
    unitD{"P6", "six pack", precise::one / precise::count},
    unitD{"P60", "nanogray per minute", precise::one / precise::count},
    unitD{"P61", "gray per hour", precise::one / precise::count},
    unitD{"P62", "milligray per hour", precise::one / precise::count},
    unitD{"P63", "microgray per hour", precise::one / precise::count},
    unitD{"P64", "nanogray per hour", precise::one / precise::count},
    unitD{"P65", "sievert per second", precise::one / precise::count},
    unitD{"P66", "millisievert per second", precise::one / precise::count},
    unitD{"P67", "microsievert per second", precise::one / precise::count},
    unitD{"P68", "nanosievert per second", precise::one / precise::count},
    unitD{"P69", "rem per second", precise::one / precise::count},
    unitD{"P7", "seven pack", precise::one / precise::count},
    unitD{"P70", "sievert per hour", precise::one / precise::count},
    unitD{"P71", "millisievert per hour", precise::one / precise::count},
    unitD{"P72", "microsievert per hour", precise::one / precise::count},
    unitD{"P73", "nanosievert per hour", precise::one / precise::count},
    unitD{"P74", "sievert per minute", precise::one / precise::count},
    unitD{"P75", "millisievert per minute", precise::one / precise::count},
    unitD{"P76", "microsievert per minute", precise::one / precise::count},
    unitD{"P77", "nanosievert per minute", precise::one / precise::count},
    unitD{"P78", "reciprocal square inch", precise::one / precise::count},
    unitD{
        "P79",
        "pascal square metre per kilogram",
        precise::one / precise::count},
    unitD{"P8", "eight pack", precise::one / precise::count},
    unitD{"P80", "millipascal per metre", precise::one / precise::count},
    unitD{"P81", "kilopascal per metre", precise::one / precise::count},
    unitD{"P82", "hectopascal per metre", precise::one / precise::count},
    unitD{
        "P83",
        "standard atmosphere per metre",
        precise::one / precise::count},
    unitD{
        "P84",
        "technical atmosphere per metre",
        precise::one / precise::count},
    unitD{"P85", "torr per metre", precise::one / precise::count},
    unitD{"P86", "psi per inch", precise::one / precise::count},
    unitD{
        "P87",
        "cubic metre per second square metre",
        precise::one / precise::count},
    unitD{"P88", "rhe", precise::one / precise::count},
    unitD{"P89", "pound-force foot per inch", precise::one / precise::count},
    unitD{"P9", "nine pack", precise::one / precise::count},
    unitD{"P90", "pound-force inch per inch", precise::one / precise::count},
    unitD{"P91", "perm (0 degC)", precise::one / precise::count},
    unitD{"P92", "perm (23 degC)", precise::one / precise::count},
    unitD{"P93", "byte per second", precise::one / precise::count},
    unitD{"P94", "kilobyte per second", precise::one / precise::count},
    unitD{"P95", "megabyte per second", precise::one / precise::count},
    unitD{"P96", "reciprocal volt", precise::one / precise::count},
    unitD{"P97", "reciprocal radian", precise::one / precise::count},
    unitD{
        "P98",
        "pascal to the power sum of stoichiometric numbers",
        precise::one / precise::count},
    unitD{
        "P99",
        "mole per cubiv metre to the power sum of stoichiometric numbers",
        precise::one / precise::count},
    unitD{"PA", "packet", precise::one / precise::count},
    unitD{"PAL", "pascal", precise::one / precise::count},
    unitD{"PB", "pair inch", precise::one / precise::count},
    unitD{"PD", "pad", precise::one / precise::count},
    unitD{"PE", "pound equivalent", precise::one / precise::count},
    unitD{"PF", "pallet (lift)", precise::one / precise::count},
    unitD{"PFL", "proof litre", precise::one / precise::count},
    unitD{"PG", "plate", precise::one / precise::count},
    unitD{"PGL", "proof gallon", precise::one / precise::count},
    unitD{"PI", "pitch", precise::one / precise::count},
    unitD{"PK", "pack", precise::one / precise::count},
    unitD{"PL", "pail", precise::one / precise::count},
    unitD{"PLA", "degree Plato", precise::one / precise::count},
    unitD{"PM", "pound percentage", precise::one / precise::count},
    unitD{"PN", "pound net", precise::one / precise::count},
    unitD{"PO", "pound per inch of length", precise::one / precise::count},
    unitD{"PQ", "page per inch", precise::one / precise::count},
    unitD{"PR", "pair", precise::one / precise::count},
    unitD{"PS", "pound-force per square inch", precise::one / precise::count},
    unitD{"PT", "pint (US)", precise::one / precise::count},
    unitD{"PTD", "dry pint (US)", precise::one / precise::count},
    unitD{"PTI", "pint (UK)", precise::one / precise::count},
    unitD{"PTL", "liquid pint (US)", precise::one / precise::count},
    unitD{"PU", "tray / tray pack", precise::one / precise::count},
    unitD{"PV", "half pint (US)", precise::one / precise::count},
    unitD{"PW", "pound per inch of width", precise::one / precise::count},
    unitD{"PY", "peck dry (US)", precise::one / precise::count},
    unitD{"PZ", "peck dry (UK)", precise::one / precise::count},
    unitD{"Q10", "joule per tesla", precise::one / precise::count},
    unitD{"Q11", "erlang", precise::one / precise::count},
    unitD{"Q12", "octet", precise::one / precise::count},
    unitD{"Q13", "octet per second", precise::one / precise::count},
    unitD{"Q14", "shannon", precise::one / precise::count},
    unitD{"Q15", "hartley", precise::one / precise::count},
    unitD{"Q16", "natural unit of information", precise::one / precise::count},
    unitD{"Q17", "shannon per second", precise::one / precise::count},
    unitD{"Q18", "hartley per second", precise::one / precise::count},
    unitD{
        "Q19",
        "natural unit of information per second",
        precise::one / precise::count},
    unitD{"Q20", "second per kilogramm", precise::one / precise::count},
    unitD{"Q21", "watt square metre", precise::one / precise::count},
    unitD{
        "Q22",
        "second per radian cubic metre",
        precise::one / precise::count},
    unitD{"Q23", "weber to the power minus one", precise::one / precise::count},
    unitD{"Q24", "reciprocal inch", precise::one / precise::count},
    unitD{"Q25", "dioptre", precise::one / precise::count},
    unitD{"Q26", "one per one", precise::one / precise::count},
    unitD{"Q27", "newton metre per metre", precise::one / precise::count},
    unitD{
        "Q28",
        "kilogram per square metre pascal second",
        precise::one / precise::count},
    unitD{"Q3", "meal", precise::one / precise::count},
    unitD{"QA", "page - facsimile", precise::one / precise::count},
    unitD{"QAN", "quarter (of a year)", precise::one / precise::count},
    unitD{"QB", "page - hardcopy", precise::one / precise::count},
    unitD{"QD", "quarter dozen", precise::one / precise::count},
    unitD{"QH", "quarter hour", precise::one / precise::count},
    unitD{"QK", "quarter kilogram", precise::one / precise::count},
    unitD{"QR", "quire", precise::one / precise::count},
    unitD{"QT", "quart (US)", precise::one / precise::count},
    unitD{"QTD", "dry quart (US)", precise::one / precise::count},
    unitD{"QTI", "quart (UK)", precise::one / precise::count},
    unitD{"QTL", "liquid quart (US)", precise::one / precise::count},
    unitD{"QTR", "quarter (UK)", precise::one / precise::count},
    unitD{"R1", "pica", precise::one / precise::count},
    unitD{"R4", "calorie", precise::one / precise::count},
    unitD{"R9", "thousand cubic metre", precise::one / precise::count},
    unitD{"RA", "rack", precise::one / precise::count},
    unitD{"RD", "rod", precise::one / precise::count},
    unitD{"RG", "ring", precise::one / precise::count},
    unitD{"RH", "running or operating hour", precise::one / precise::count},
    unitD{"RK", "roll metric measure", precise::one / precise::count},
    unitD{"RL", "reel", precise::one / precise::count},
    unitD{"RM", "ream", precise::one / precise::count},
    unitD{"RN", "ream metric measure", precise::one / precise::count},
    unitD{"RO", "roll", precise::one / precise::count},
    unitD{"ROM", "room", precise::one / precise::count},
    unitD{"RP", "pound per ream", precise::one / precise::count},
    unitD{"RPM", "revolutions per minute", precise::one / precise::count},
    unitD{"RPS", "revolutions per second", precise::one / precise::count},
    unitD{"RS", "reset", precise::one / precise::count},
    unitD{"RT", "revenue ton mile", precise::one / precise::count},
    unitD{"RU", "run", precise::one / precise::count},
    unitD{"S3", "square foot per second", precise::one / precise::count},
    unitD{"S4", "square metre per second", precise::one / precise::count},
    unitD{"S5", "sixty fourths of an inch", precise::one / precise::count},
    unitD{"S6", "session", precise::one / precise::count},
    unitD{"S7", "storage unit", precise::one / precise::count},
    unitD{"S8", "standard advertising unit", precise::one / precise::count},
    unitD{"SA", "sack", precise::one / precise::count},
    unitD{"SAN", "half year (6 months)", precise::one / precise::count},
    unitD{"SCO", "score", precise::one / precise::count},
    unitD{"SCR", "scruple", precise::one / precise::count},
    unitD{"SD", "solid pound", precise::one / precise::count},
    unitD{"SE", "section", precise::one / precise::count},
    unitD{"SEC", "second [unit of time]", precise::one / precise::count},
    unitD{"SET", "set", precise::one / precise::count},
    unitD{"SG", "segment", precise::one / precise::count},
    unitD{"SHT", "shipping ton", precise::one / precise::count},
    unitD{"SIE", "siemens", precise::one / precise::count},
    unitD{"SK", "split tank truck", precise::one / precise::count},
    unitD{"SL", "slipsheet", precise::one / precise::count},
    unitD{"SMI", "mile (statute mile)", precise::one / precise::count},
    unitD{"SN", "square rod", precise::one / precise::count},
    unitD{"SO", "spool", precise::one / precise::count},
    unitD{"SP", "shelf package", precise::one / precise::count},
    unitD{"SQ", "square", precise::one / precise::count},
    unitD{"SQR", "square, roofing", precise::one / precise::count},
    unitD{"SR", "strip", precise::one / precise::count},
    unitD{"SS", "sheet metric measure", precise::one / precise::count},
    unitD{
        "SST",
        "short standard (7200 matches)",
        precise::one / precise::count},
    unitD{"ST", "sheet", precise::one / precise::count},
    unitD{"STC", "stick", precise::one / precise::count},
    unitD{"STI", "stone (UK)", precise::one / precise::count},
    unitD{"STK", "stick, cigarette", precise::one / precise::count},
    unitD{"STL", "standard litre", precise::one / precise::count},
    unitD{
        "STN",
        "ton (US) or short ton (UK/US)",
        precise::one / precise::count},
    unitD{"STW", "straw", precise::one / precise::count},
    unitD{"SV", "skid", precise::one / precise::count},
    unitD{"SW", "skein", precise::one / precise::count},
    unitD{"SX", "shipment", precise::one / precise::count},
    unitD{"SYR", "syringe", precise::one / precise::count},
    unitD{
        "T0",
        "telecommunication line in service",
        precise::one / precise::count},
    unitD{"T1", "thousand pound gross", precise::one / precise::count},
    unitD{"T3", "thousand piece", precise::one / precise::count},
    unitD{"T4", "thousand bag", precise::one / precise::count},
    unitD{"T5", "thousand casing", precise::one / precise::count},
    unitD{"T6", "thousand gallon (US)", precise::one / precise::count},
    unitD{"T7", "thousand impression", precise::one / precise::count},
    unitD{"T8", "thousand linear inch", precise::one / precise::count},
    unitD{"TA", "tenth cubic foot", precise::one / precise::count},
    unitD{
        "TAH",
        "kiloampere hour (thousand ampere hour)",
        precise::one / precise::count},
    unitD{"TAN", "total acid number", precise::one / precise::count},
    unitD{"TC", "truckload", precise::one / precise::count},
    unitD{"TD", "therm", precise::one / precise::count},
    unitD{"TE", "tote", precise::one / precise::count},
    unitD{"TF", "ten square yard", precise::one / precise::count},
    unitD{"TI", "thousand square inch", precise::one / precise::count},
    unitD{
        "TIC",
        "metric ton, including container",
        precise::one / precise::count},
    unitD{
        "TIP",
        "metric ton, including inner packaging",
        precise::one / precise::count},
    unitD{"TJ", "thousand square centimetre", precise::one / precise::count},
    unitD{"TK", "tank, rectangular", precise::one / precise::count},
    unitD{"TKM", "tonne kilometre", precise::one / precise::count},
    unitD{"TL", "thousand foot (linear)", precise::one / precise::count},
    unitD{
        "TMS",
        "kilogram of imported meat, less offal",
        precise::one / precise::count},
    unitD{"TN", "tin", precise::one / precise::count},
    unitD{"TNE", "tonne (metric ton)", precise::one / precise::count},
    unitD{"TP", "ten pack", precise::one / precise::count},
    unitD{"TPR", "ten pair", precise::one / precise::count},
    unitD{"TQ", "thousand foot", precise::one / precise::count},
    unitD{"TQD", "thousand cubic metre per day", precise::one / precise::count},
    unitD{"TR", "ten square foot", precise::one / precise::count},
    unitD{"TRL", "trillion (EUR)", precise::one / precise::count},
    unitD{"TS", "thousand square foot", precise::one / precise::count},
    unitD{"TSD", "tonne of substance 90 % dry", precise::one / precise::count},
    unitD{"TSH", "ton of steam per hour", precise::one / precise::count},
    unitD{"TST", "ten set", precise::one / precise::count},
    unitD{"TT", "thousand linear metre", precise::one / precise::count},
    unitD{"TTS", "ten thousand sticks", precise::one / precise::count},
    unitD{"TU", "tube", precise::one / precise::count},
    unitD{"TV", "thousand kilogram", precise::one / precise::count},
    unitD{"TW", "thousand sheet", precise::one / precise::count},
    unitD{"TY", "tank, cylindrical", precise::one / precise::count},
    unitD{"U1", "treatment", precise::one / precise::count},
    unitD{"U2", "tablet", precise::one / precise::count},
    unitD{"UA", "torr", precise::one / precise::count},
    unitD{
        "UB",
        "telecommunication line in service average",
        precise::one / precise::count},
    unitD{"UC", "telecommunication port", precise::one / precise::count},
    unitD{"UD", "tenth minute", precise::one / precise::count},
    unitD{"UE", "tenth hour", precise::one / precise::count},
    unitD{
        "UF",
        "usage per telecommunication line average",
        precise::one / precise::count},
    unitD{"UH", "ten thousand yard", precise::one / precise::count},
    unitD{"UM", "million unit", precise::one / precise::count},
    unitD{"VA", "volt - ampere per kilogram", precise::one / precise::count},
    unitD{"VI", "vial", precise::one / precise::count},
    unitD{"VLT", "volt", precise::one / precise::count},
    unitD{"VP", "percent volume", precise::one / precise::count},
    unitD{"VQ", "bulk", precise::one / precise::count},
    unitD{"VS", "visit", precise::one / precise::count},
    unitD{"W2", "wet kilo", precise::one / precise::count},
    unitD{"W4", "two week", precise::one / precise::count},
    unitD{"WA", "watt per kilogram", precise::one / precise::count},
    unitD{"WB", "wet pound", precise::one / precise::count},
    unitD{"WCD", "cord", precise::one / precise::count},
    unitD{"WE", "wet ton", precise::one / precise::count},
    unitD{"WEB", "weber", precise::one / precise::count},
    unitD{"WEE", "week", precise::one / precise::count},
    unitD{"WG", "wine gallon", precise::one / precise::count},
    unitD{"WH", "wheel", precise::one / precise::count},
    unitD{"WHR", "watt hour", precise::one / precise::count},
    unitD{"WI", "weight per square inch", precise::one / precise::count},
    unitD{"WM", "working month", precise::one / precise::count},
    unitD{"WR", "wrap", precise::one / precise::count},
    unitD{"WSD", "standard", precise::one / precise::count},
    unitD{"WTT", "watt", precise::one / precise::count},
    unitD{"WW", "millilitre of water", precise::one / precise::count},
    unitD{"X1", "Gunter's chain", precise::one / precise::count},
    unitD{"YDK", "square yard", precise::one / precise::count},
    unitD{"YDQ", "cubic yard", precise::one / precise::count},
    unitD{"YL", "hundred linear yard", precise::one / precise::count},
    unitD{"YRD", "yard", precise::one / precise::count},
    unitD{"YT", "ten yard", precise::one / precise::count},
    unitD{"Z1", "lift van", precise::one / precise::count},
    unitD{"Z11", "hanging container", precise::one / precise::count},
    unitD{"Z2", "chest", precise::one / precise::count},
    unitD{"Z3", "cask", precise::one / precise::count},
    unitD{"Z4", "hogshead", precise::one / precise::count},
    unitD{"Z5", "lug", precise::one / precise::count},
    unitD{"Z6", "conference point", precise::one / precise::count},
    unitD{"Z8", "newspage agate line", precise::one / precise::count},
    unitD{"ZP", "page", precise::one / precise::count},
    unitD{"ZZ", "mutually defined", precise::one / precise::count},
}};

precise_unit r20_unit(const std::string& r20_string)
{
    // NOLINTNEXTLINE (readability-qualified-auto)
    auto ind = std::lower_bound(
        r20_units.begin(),
        r20_units.end(),
        r20_string,
        [](const unitD& u_set, const std::string& val) {
            return (strcmp(std::get<0>(u_set), val.c_str()) < 0);
        });
    if (strcmp(std::get<0>(*ind), r20_string.c_str()) == 0) {
        return std::get<2>(*ind);
    }
    return precise::error;
}

}  // namespace UNITS_NAMESPACE
