#
# x86sets.py
#
# Copyright (C) 2009 Gil Dabah, http://ragestorm.net/disops/
#

from x86header import *

OPT = OperandType
IFlag = InstFlag

class Instructions:
	""" Initializes all instruction of the 80x86 CPU (includes AMD64). """

	def init_INTEGER(self):
		Set = lambda *args: self.SetCallback(ISetClass.INTEGER, *args)

		# V 1.5.13 - Pushes can be affected by operand size prefix. Segment is encoded in flags.

		# SAL is exactly like SHL, so I prefer to use the mnemonic "SHL" (below).

		Set("00", ["ADD"], [OPT.RM8, OPT.REG8], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("01", ["ADD"], [OPT.RM_FULL, OPT.REG_FULL], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("02", ["ADD"], [OPT.REG8, OPT.RM8], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("03", ["ADD"], [OPT.REG_FULL, OPT.RM_FULL], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("04", ["ADD"], [OPT.ACC8, OPT.IMM8], IFlag.INST_FLAGS_NONE)
		Set("05", ["ADD"], [OPT.ACC_FULL, OPT.IMM_FULL], IFlag.INST_FLAGS_NONE)
		Set("06", ["PUSH"], [OPT.SEG], IFlag.PRE_ES | IFlag.INVALID_64BITS)
		Set("07", ["POP"], [OPT.SEG], IFlag.PRE_ES | IFlag.INVALID_64BITS)
		Set("08", ["OR"], [OPT.RM8, OPT.REG8], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("09", ["OR"], [OPT.RM_FULL, OPT.REG_FULL], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("0a", ["OR"], [OPT.REG8, OPT.RM8], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("0b", ["OR"], [OPT.REG_FULL, OPT.RM_FULL], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("0c", ["OR"], [OPT.ACC8, OPT.IMM8], IFlag.INST_FLAGS_NONE)
		Set("0d", ["OR"], [OPT.ACC_FULL, OPT.IMM_FULL], IFlag.INST_FLAGS_NONE)
		Set("0e", ["PUSH"], [OPT.SEG], IFlag.PRE_CS | IFlag.INVALID_64BITS)
		Set("0f, 00 /00", ["SLDT"], [OPT.RM_FULL], IFlag.MODRM_REQUIRED)
		Set("0f, 00 /01", ["STR"], [OPT.RM16], IFlag.MODRM_REQUIRED)
		Set("0f, 00 /02", ["LLDT"], [OPT.RM16], IFlag.MODRM_REQUIRED)
		Set("0f, 00 /03", ["LTR"], [OPT.RM16], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 00 /04", ["VERR"], [OPT.RM16], IFlag.MODRM_REQUIRED)
		Set("0f, 00 /05", ["VERW"], [OPT.RM16], IFlag.MODRM_REQUIRED)
		Set("0f, 01 //00", ["SGDT"], [OPT.MEM16_3264], IFlag.MODRM_REQUIRED | IFlag._64BITS)
		Set("0f, 01 //01", ["SIDT"], [OPT.MEM16_3264], IFlag.MODRM_REQUIRED | IFlag._64BITS)
		Set("0f, 01 //02", ["LGDT"], [OPT.MEM16_3264], IFlag.MODRM_REQUIRED | IFlag._64BITS)
		Set("0f, 01 //03", ["LIDT"], [OPT.MEM16_3264], IFlag.MODRM_REQUIRED | IFlag._64BITS)

		# These two instructions need the whole byte, means they use the whole third byte and are NOT divided.
		# We'll recognize them by their 3 REG bits in their third byte.
		Set("0f, 01 //04", ["SMSW"], [OPT.RFULL_M16], IFlag.MODRM_REQUIRED | IFlag.NOT_DIVIDED)
		Set("0f, 01 //06", ["LMSW"], [OPT.RM16], IFlag.MODRM_REQUIRED | IFlag.NOT_DIVIDED)

		Set("0f, 01 //07", ["INVLPG"], [OPT.MEM], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 01 //c8", ["MONITOR"], [], IFlag._32BITS)
		Set("0f, 01 //c9", ["MWAIT"], [], IFlag._32BITS)
		Set("0f, 01 //f8", ["SWAPGS"], [], IFlag._64BITS_FETCH)
		Set("0f, 01 //f9", ["RDTSCP"], [], IFlag._64BITS_FETCH)
		Set("0f, 02", ["LAR"], [OPT.REG_FULL, OPT.RM16], IFlag.MODRM_REQUIRED)
		Set("0f, 03", ["LSL"], [OPT.REG_FULL, OPT.RM16], IFlag.MODRM_REQUIRED)
		Set("0f, 06", ["CLTS"], [], IFlag._32BITS)
		Set("0f, 08", ["INVD"], [], IFlag._32BITS)
		Set("0f, 09", ["WBINVD"], [], IFlag._32BITS)
		Set("0f, 0b", ["UD2"], [], IFlag._32BITS)

		# MOV: In 64 bits decoding mode REG is 64 bits by default.
		Set("0f, 20", ["MOV"], [OPT.FREG32_64_RM, OPT.CREG], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._64BITS)
		Set("0f, 21", ["MOV"], [OPT.FREG32_64_RM, OPT.DREG], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._64BITS)
		Set("0f, 22", ["MOV"], [OPT.CREG, OPT.FREG32_64_RM], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._64BITS)
		Set("0f, 23", ["MOV"], [OPT.DREG, OPT.FREG32_64_RM], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._64BITS)
		Set("0f, 30", ["WRMSR"], [], IFlag._32BITS)
		Set("0f, 31", ["RDTSC"], [], IFlag._32BITS)
		Set("0f, 32", ["RDMSR"], [], IFlag._32BITS)
		Set("0f, 33", ["RDPMC"], [], IFlag._32BITS)
		Set("0f, 80", ["JO"], [OPT.RELC_FULL], IFlag._32BITS)
		Set("0f, 81", ["JNO"], [OPT.RELC_FULL], IFlag._32BITS)
		Set("0f, 82", ["JB"], [OPT.RELC_FULL], IFlag._32BITS)
		Set("0f, 83", ["JAE"], [OPT.RELC_FULL], IFlag._32BITS)
		Set("0f, 84", ["JZ"], [OPT.RELC_FULL], IFlag._32BITS)
		Set("0f, 85", ["JNZ"], [OPT.RELC_FULL], IFlag._32BITS)
		Set("0f, 86", ["JBE"], [OPT.RELC_FULL], IFlag._32BITS)
		Set("0f, 87", ["JA"], [OPT.RELC_FULL], IFlag._32BITS)
		Set("0f, 88", ["JS"], [OPT.RELC_FULL], IFlag._32BITS)
		Set("0f, 89", ["JNS"], [OPT.RELC_FULL], IFlag._32BITS)
		Set("0f, 8a", ["JP"], [OPT.RELC_FULL], IFlag._32BITS)
		Set("0f, 8b", ["JNP"], [OPT.RELC_FULL], IFlag._32BITS)
		Set("0f, 8c", ["JL"], [OPT.RELC_FULL], IFlag._32BITS)
		Set("0f, 8d", ["JGE"], [OPT.RELC_FULL], IFlag._32BITS)
		Set("0f, 8e", ["JLE"], [OPT.RELC_FULL], IFlag._32BITS)
		Set("0f, 8f", ["JG"], [OPT.RELC_FULL], IFlag._32BITS)
		Set("0f, 90", ["SETO"], [OPT.RM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 91", ["SETNO"], [OPT.RM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 92", ["SETB"], [OPT.RM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 93", ["SETAE"], [OPT.RM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 94", ["SETZ"], [OPT.RM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 95", ["SETNZ"], [OPT.RM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 96", ["SETBE"], [OPT.RM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 97", ["SETA"], [OPT.RM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 98", ["SETS"], [OPT.RM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 99", ["SETNS"], [OPT.RM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 9a", ["SETP"], [OPT.RM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 9b", ["SETNP"], [OPT.RM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 9c", ["SETL"], [OPT.RM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 9d", ["SETGE"], [OPT.RM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 9e", ["SETLE"], [OPT.RM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 9f", ["SETG"], [OPT.RM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, a0", ["PUSH"], [OPT.SEG], IFlag._32BITS | IFlag.PRE_FS | IFlag._64BITS)
		Set("0f, a1", ["POP"], [OPT.SEG], IFlag._32BITS | IFlag.PRE_FS | IFlag._64BITS)
		Set("0f, a2", ["CPUID"], [], IFlag._32BITS)
		Set("0f, a3", ["BT"], [OPT.RM_FULL, OPT.REG_FULL], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, a4", ["SHLD"], [OPT.RM_FULL, OPT.REG_FULL, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, a5", ["SHLD"], [OPT.RM_FULL, OPT.REG_FULL, OPT.REGCL], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, a8", ["PUSH"], [OPT.SEG], IFlag._32BITS | IFlag.PRE_GS | IFlag._64BITS)
		Set("0f, a9", ["POP"], [OPT.SEG], IFlag._32BITS | IFlag.PRE_GS | IFlag._64BITS)
		Set("0f, aa", ["RSM"], [], IFlag._32BITS)
		Set("0f, ab", ["BTS"], [OPT.RM_FULL, OPT.REG_FULL], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_LOCK)
		Set("0f, ac", ["SHRD"], [OPT.RM_FULL, OPT.REG_FULL, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, ad", ["SHRD"], [OPT.RM_FULL, OPT.REG_FULL, OPT.REGCL], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, ae /00", ["FXSAVE", "", "FXSAVE64"], [OPT.MEM], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._64BITS | IFlag.PRE_REX | IFlag.USE_EXMNEMONIC2)
		Set("0f, ae /01", ["FXRSTOR", "", "FXRSTOR64"], [OPT.MEM], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._64BITS | IFlag.PRE_REX | IFlag.USE_EXMNEMONIC2)
		Set("0f, ae /02", ["LDMXCSR"], [OPT.MEM32], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, ae /03", ["STMXCSR"], [OPT.MEM32], IFlag.MODRM_REQUIRED | IFlag._32BITS)

		# MFENCE and XSAVEOPT share the same opcode 0f ae /6. It's MFENCE when MOD=11, else XSAVEOPT or XSAVEOPT64 in 64.
		Set("0f, ae /06", ["MFENCE", "XSAVEOPT", "XSAVEOPT64"], [OPT.MEM_OPT], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.USE_EXMNEMONIC | IFlag.MNEMONIC_MODRM_BASED | IFlag._64BITS | IFlag.PRE_REX | IFlag.USE_EXMNEMONIC2)

		# SFENCE and CLFLUSH share the same opcode 0f ae /7. It's SFENCE when MOD=11, else CLFLUSH.
		# But the operand is used only for CLFLUSH, which means it's optional. MOD=11 for first mnemonic.
		Set("0f, ae /07", ["SFENCE", "CLFLUSH"], [OPT.MEM_OPT], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.USE_EXMNEMONIC | IFlag.MNEMONIC_MODRM_BASED)
		# Same for LFENCE and XRSTOR with 0f ae /5.
		Set("0f, ae /05", ["LFENCE", "XRSTOR", "XRSTOR64"], [OPT.MEM_OPT], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.USE_EXMNEMONIC | IFlag.MNEMONIC_MODRM_BASED | IFlag._64BITS | IFlag.PRE_REX | IFlag.USE_EXMNEMONIC2)

		Set("0f, af", ["IMUL"], [OPT.REG_FULL, OPT.RM_FULL], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, b0", ["CMPXCHG"], [OPT.RM8, OPT.REG8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_LOCK)
		Set("0f, b1", ["CMPXCHG"], [OPT.RM_FULL, OPT.REG_FULL], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_LOCK)
		Set("0f, b2", ["LSS"], [OPT.REG_FULL, OPT.MEM16_FULL], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._64BITS | IFlag.PRE_REX)
		Set("0f, b3", ["BTR"], [OPT.RM_FULL, OPT.REG_FULL], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_LOCK)
		Set("0f, b4", ["LFS"], [OPT.REG_FULL, OPT.MEM16_FULL], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._64BITS | IFlag.PRE_REX)
		Set("0f, b5", ["LGS"], [OPT.REG_FULL, OPT.MEM16_FULL], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._64BITS | IFlag.PRE_REX)
		Set("0f, b6", ["MOVZX"], [OPT.REG_FULL, OPT.RM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, b7", ["MOVZX"], [OPT.REG_FULL, OPT.RM16], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._64BITS | IFlag.PRE_REX)
		Set("0f, b9", ["UD2"], [], IFlag._32BITS)
		Set("0f, ba /04", ["BT"], [OPT.RM_FULL, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, ba /05", ["BTS"], [OPT.RM_FULL, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_LOCK)
		Set("0f, ba /06", ["BTR"], [OPT.RM_FULL, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_LOCK)
		Set("0f, ba /07", ["BTC"], [OPT.RM_FULL, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_LOCK)
		Set("0f, bb", ["BTC"], [OPT.RM_FULL, OPT.REG_FULL], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_LOCK)
		Set("0f, bc", ["BSF"], [OPT.REG_FULL, OPT.RM_FULL], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, bd", ["BSR"], [OPT.REG_FULL, OPT.RM_FULL], IFlag.MODRM_REQUIRED | IFlag._32BITS)

		# V 1.1.6 MOVSX/MOVZX now support 16bits regs.
		Set("0f, be", ["MOVSX"], [OPT.REG_FULL, OPT.RM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, bf", ["MOVSX"], [OPT.REG_FULL, OPT.RM16], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._64BITS | IFlag.PRE_REX)
		Set("0f, c0", ["XADD"], [OPT.RM8, OPT.REG8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_LOCK)
		Set("0f, c1", ["XADD"], [OPT.RM_FULL, OPT.REG_FULL], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_LOCK)
		Set("0f, c7 /01", ["CMPXCHG8B", "", "CMPXCHG16B"], [OPT.MEM64_128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_LOCK | IFlag._64BITS | IFlag.PRE_REX | IFlag.USE_EXMNEMONIC2)
		Set("0f, c8", ["BSWAP"], [OPT.IB_R_FULL], IFlag._32BITS | IFlag._64BITS | IFlag.PRE_REX | IFlag.GEN_BLOCK)
		Set("10", ["ADC"], [OPT.RM8, OPT.REG8], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("11", ["ADC"], [OPT.RM_FULL, OPT.REG_FULL], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("12", ["ADC"], [OPT.REG8, OPT.RM8], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("13", ["ADC"], [OPT.REG_FULL, OPT.RM_FULL], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("14", ["ADC"], [OPT.ACC8, OPT.IMM8], IFlag.INST_FLAGS_NONE)
		Set("15", ["ADC"], [OPT.ACC_FULL, OPT.IMM_FULL], IFlag.INST_FLAGS_NONE)
		Set("16", ["PUSH"], [OPT.SEG], IFlag.PRE_SS | IFlag.INVALID_64BITS)
		Set("17", ["POP"], [OPT.SEG], IFlag.PRE_SS | IFlag.INVALID_64BITS)
		Set("18", ["SBB"], [OPT.RM8, OPT.REG8], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("19", ["SBB"], [OPT.RM_FULL, OPT.REG_FULL], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("1a", ["SBB"], [OPT.REG8, OPT.RM8], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("1b", ["SBB"], [OPT.REG_FULL, OPT.RM_FULL], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("1c", ["SBB"], [OPT.ACC8, OPT.IMM8], IFlag.INST_FLAGS_NONE)
		Set("1d", ["SBB"], [OPT.ACC_FULL, OPT.IMM_FULL], IFlag.INST_FLAGS_NONE)
		Set("1e", ["PUSH"], [OPT.SEG], IFlag.PRE_DS | IFlag.INVALID_64BITS)
		Set("1f", ["POP"], [OPT.SEG], IFlag.PRE_DS | IFlag.INVALID_64BITS)
		Set("20", ["AND"], [OPT.RM8, OPT.REG8], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("21", ["AND"], [OPT.RM_FULL, OPT.REG_FULL], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("22", ["AND"], [OPT.REG8, OPT.RM8], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("23", ["AND"], [OPT.REG_FULL, OPT.RM_FULL], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("24", ["AND"], [OPT.ACC8, OPT.IMM8], IFlag.INST_FLAGS_NONE)
		Set("25", ["AND"], [OPT.ACC_FULL, OPT.IMM_FULL], IFlag.INST_FLAGS_NONE)
		Set("27", ["DAA"], [], IFlag.INVALID_64BITS)
		Set("28", ["SUB"], [OPT.RM8, OPT.REG8], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("29", ["SUB"], [OPT.RM_FULL, OPT.REG_FULL], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("2a", ["SUB"], [OPT.REG8, OPT.RM8], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("2b", ["SUB"], [OPT.REG_FULL, OPT.RM_FULL], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("2c", ["SUB"], [OPT.ACC8, OPT.IMM8], IFlag.INST_FLAGS_NONE)
		Set("2d", ["SUB"], [OPT.ACC_FULL, OPT.IMM_FULL], IFlag.INST_FLAGS_NONE)
		Set("2f", ["DAS"], [], IFlag.INVALID_64BITS)
		Set("30", ["XOR"], [OPT.RM8, OPT.REG8], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("31", ["XOR"], [OPT.RM_FULL, OPT.REG_FULL], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("32", ["XOR"], [OPT.REG8, OPT.RM8], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("33", ["XOR"], [OPT.REG_FULL, OPT.RM_FULL], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("34", ["XOR"], [OPT.ACC8, OPT.IMM8], IFlag.INST_FLAGS_NONE)
		Set("35", ["XOR"], [OPT.ACC_FULL, OPT.IMM_FULL], IFlag.INST_FLAGS_NONE)
		Set("37", ["AAA"], [], IFlag.INVALID_64BITS)
		Set("38", ["CMP"], [OPT.RM8, OPT.REG8], IFlag.MODRM_REQUIRED)
		Set("39", ["CMP"], [OPT.RM_FULL, OPT.REG_FULL], IFlag.MODRM_REQUIRED)
		Set("3a", ["CMP"], [OPT.REG8, OPT.RM8], IFlag.MODRM_REQUIRED)
		Set("3b", ["CMP"], [OPT.REG_FULL, OPT.RM_FULL], IFlag.MODRM_REQUIRED)
		Set("3c", ["CMP"], [OPT.ACC8, OPT.IMM8], IFlag.INST_FLAGS_NONE)
		Set("3d", ["CMP"], [OPT.ACC_FULL, OPT.IMM_FULL], IFlag.INST_FLAGS_NONE)
		Set("3f", ["AAS"], [], IFlag.INVALID_64BITS)
		Set("40", ["INC"], [OPT.IB_R_FULL], IFlag.INVALID_64BITS | IFlag.GEN_BLOCK)
		Set("48", ["DEC"], [OPT.IB_R_FULL], IFlag.INVALID_64BITS | IFlag.GEN_BLOCK)
		Set("50", ["PUSH"], [OPT.IB_R_FULL], IFlag._64BITS | IFlag.GEN_BLOCK)
		Set("58", ["POP"], [OPT.IB_R_FULL], IFlag._64BITS | IFlag.GEN_BLOCK)
		Set("60", ["PUSHA"], [], IFlag.NATIVE | IFlag.INVALID_64BITS)
		Set("61", ["POPA"], [], IFlag.NATIVE | IFlag.INVALID_64BITS)
		Set("62", ["BOUND"], [OPT.REG_FULL, OPT.MEM], IFlag.MODRM_REQUIRED | IFlag.INVALID_64BITS)
		Set("68", ["PUSH"], [OPT.IMM_FULL], IFlag._64BITS)
		Set("69", ["IMUL"], [OPT.REG_FULL, OPT.RM_FULL, OPT.IMM_FULL], IFlag.MODRM_REQUIRED)
		Set("6a", ["PUSH"], [OPT.SEIMM8], IFlag.PRE_OP_SIZE | IFlag._64BITS)
		Set("6b", ["IMUL"], [OPT.REG_FULL, OPT.RM_FULL, OPT.SEIMM8], IFlag.MODRM_REQUIRED)
		# V 1.5.14 - String instructions aren't supposed to be promoted automatically in 64bits, only with a REX prefix.
		# In 64 bits INS/OUTS still supports only 8/16/32 bits.
		Set("6c", ["INS"], [OPT.REGI_EDI, OPT.REGDX], IFlag.PRE_REPNZ | IFlag.PRE_REP) # 8 bit.
		Set("6d", ["INS"], [OPT.REGI_EDI, OPT.REGDX], IFlag._16BITS | IFlag.PRE_REPNZ | IFlag.PRE_REP) # Full size.
		Set("6e", ["OUTS"], [OPT.REGDX, OPT.REGI_ESI], IFlag.PRE_REPNZ | IFlag.PRE_REP) # 8 bit.
		Set("6f", ["OUTS"], [OPT.REGDX, OPT.REGI_ESI], IFlag._16BITS | IFlag.PRE_REPNZ | IFlag.PRE_REP) # Full size.
		Set("70", ["JO"], [OPT.RELCB], IFlag._64BITS)
		Set("71", ["JNO"], [OPT.RELCB], IFlag._64BITS)
		Set("72", ["JB"], [OPT.RELCB], IFlag._64BITS)
		Set("73", ["JAE"], [OPT.RELCB], IFlag._64BITS)
		Set("74", ["JZ"], [OPT.RELCB], IFlag._64BITS)
		Set("75", ["JNZ"], [OPT.RELCB], IFlag._64BITS)
		Set("76", ["JBE"], [OPT.RELCB], IFlag._64BITS)
		Set("77", ["JA"], [OPT.RELCB], IFlag._64BITS)
		Set("78", ["JS"], [OPT.RELCB], IFlag._64BITS)
		Set("79", ["JNS"], [OPT.RELCB], IFlag._64BITS)
		Set("7a", ["JP"], [OPT.RELCB], IFlag._64BITS)
		Set("7b", ["JNP"], [OPT.RELCB], IFlag._64BITS)
		Set("7c", ["JL"], [OPT.RELCB], IFlag._64BITS)
		Set("7d", ["JGE"], [OPT.RELCB], IFlag._64BITS)
		Set("7e", ["JLE"], [OPT.RELCB], IFlag._64BITS)
		Set("7f", ["JG"], [OPT.RELCB], IFlag._64BITS)
		Set("80 /00", ["ADD"], [OPT.RM8, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("80 /01", ["OR"], [OPT.RM8, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("80 /02", ["ADC"], [OPT.RM8, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("80 /03", ["SBB"], [OPT.RM8, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("80 /04", ["AND"], [OPT.RM8, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("80 /05", ["SUB"], [OPT.RM8, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("80 /06", ["XOR"], [OPT.RM8, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("80 /07", ["CMP"], [OPT.RM8, OPT.IMM8], IFlag.MODRM_REQUIRED)
		Set("81 /00", ["ADD"], [OPT.RM_FULL, OPT.IMM_FULL], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("81 /01", ["OR"], [OPT.RM_FULL, OPT.IMM_FULL], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("81 /02", ["ADC"], [OPT.RM_FULL, OPT.IMM_FULL], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("81 /03", ["SBB"], [OPT.RM_FULL, OPT.IMM_FULL], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("81 /04", ["AND"], [OPT.RM_FULL, OPT.IMM_FULL], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("81 /05", ["SUB"], [OPT.RM_FULL, OPT.IMM_FULL], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("81 /06", ["XOR"], [OPT.RM_FULL, OPT.IMM_FULL], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("81 /07", ["CMP"], [OPT.RM_FULL, OPT.IMM_FULL], IFlag.MODRM_REQUIRED)
		Set("82 /00", ["ADD"], [OPT.RM8, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK | IFlag.INVALID_64BITS)
		Set("82 /01", ["OR"], [OPT.RM8, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK | IFlag.INVALID_64BITS)
		Set("82 /02", ["ADC"], [OPT.RM8, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK | IFlag.INVALID_64BITS)
		Set("82 /03", ["SBB"], [OPT.RM8, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK | IFlag.INVALID_64BITS)
		Set("82 /04", ["AND"], [OPT.RM8, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK | IFlag.INVALID_64BITS)
		Set("82 /05", ["SUB"], [OPT.RM8, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK | IFlag.INVALID_64BITS)
		Set("82 /06", ["XOR"], [OPT.RM8, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK | IFlag.INVALID_64BITS)
		Set("82 /07", ["CMP"], [OPT.RM8, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag.INVALID_64BITS)
		Set("83 /00", ["ADD"], [OPT.RM_FULL, OPT.SEIMM8], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("83 /01", ["OR"], [OPT.RM_FULL, OPT.SEIMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_LOCK)
		Set("83 /02", ["ADC"], [OPT.RM_FULL, OPT.SEIMM8], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("83 /03", ["SBB"], [OPT.RM_FULL, OPT.SEIMM8], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("83 /04", ["AND"], [OPT.RM_FULL, OPT.SEIMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_LOCK)
		Set("83 /05", ["SUB"], [OPT.RM_FULL, OPT.SEIMM8], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("83 /06", ["XOR"], [OPT.RM_FULL, OPT.SEIMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_LOCK)
		Set("83 /07", ["CMP"], [OPT.RM_FULL, OPT.SEIMM8], IFlag.MODRM_REQUIRED)
		Set("84", ["TEST"], [OPT.RM8, OPT.REG8], IFlag.MODRM_REQUIRED)
		Set("85", ["TEST"], [OPT.RM_FULL, OPT.REG_FULL], IFlag.MODRM_REQUIRED)
		Set("86", ["XCHG"], [OPT.RM8, OPT.REG8], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("87", ["XCHG"], [OPT.RM_FULL, OPT.REG_FULL], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("88", ["MOV"], [OPT.RM8, OPT.REG8], IFlag.MODRM_REQUIRED)
		Set("89", ["MOV"], [OPT.RM_FULL, OPT.REG_FULL], IFlag.MODRM_REQUIRED)
		Set("8a", ["MOV"], [OPT.REG8, OPT.RM8], IFlag.MODRM_REQUIRED)
		Set("8b", ["MOV"], [OPT.REG_FULL, OPT.RM_FULL], IFlag.MODRM_REQUIRED)
		Set("8c", ["MOV"], [OPT.RFULL_M16, OPT.SREG], IFlag.MODRM_REQUIRED)
		Set("8d", ["LEA"], [OPT.REG_FULL, OPT.MEM], IFlag.MODRM_REQUIRED)
		Set("8e", ["MOV"], [OPT.SREG, OPT.RFULL_M16], IFlag.MODRM_REQUIRED)
		Set("8f /00", ["POP"], [OPT.RM_FULL], IFlag.MODRM_REQUIRED | IFlag._64BITS)
		# V 1.7.28 - NOP in 64bits can be prefixed by REX -
		# Therefore 0x90 in 16/32 bits is NOP. 0x90 with REX base is an XCHG. All else are NOP.
		Set("90", ["XCHG"], [OPT.IB_R_FULL, OPT.ACC_FULL], IFlag.INST_FLAGS_NONE)

		# V 1.7.24 - New instruction multi-byte NOP.
		Set("0f, 1f", ["NOP"], [OPT.RM_FULL], IFlag.MODRM_REQUIRED)
		Set("91", ["XCHG"], [OPT.IB_R_FULL, OPT.ACC_FULL], IFlag.INST_FLAGS_NONE)
		Set("92", ["XCHG"], [OPT.IB_R_FULL, OPT.ACC_FULL], IFlag.INST_FLAGS_NONE)
		Set("93", ["XCHG"], [OPT.IB_R_FULL, OPT.ACC_FULL], IFlag.INST_FLAGS_NONE)
		Set("94", ["XCHG"], [OPT.IB_R_FULL, OPT.ACC_FULL], IFlag.INST_FLAGS_NONE)
		Set("95", ["XCHG"], [OPT.IB_R_FULL, OPT.ACC_FULL], IFlag.INST_FLAGS_NONE)
		Set("96", ["XCHG"], [OPT.IB_R_FULL, OPT.ACC_FULL], IFlag.INST_FLAGS_NONE)
		Set("97", ["XCHG"], [OPT.IB_R_FULL, OPT.ACC_FULL], IFlag.INST_FLAGS_NONE)
		Set("98", ["CBW", "CWDE", "CDQE"], [], IFlag.USE_EXMNEMONIC | IFlag.USE_EXMNEMONIC2)
		Set("99", ["CWD", "CDQ", "CQO"], [], IFlag.USE_EXMNEMONIC | IFlag.USE_EXMNEMONIC2)
		Set("9a", ["CALL FAR"], [OPT.PTR16_FULL], IFlag.INVALID_64BITS)

		# V 1.4.a PUSHF/POPF are supposed to be promoted to 64 bits, without a REX.
		Set("9c", ["PUSHF"], [], IFlag.NATIVE | IFlag._64BITS)
		Set("9d", ["POPF"], [], IFlag.NATIVE | IFlag._64BITS)
		Set("9e", ["SAHF"], [], IFlag.INST_FLAGS_NONE)
		Set("9f", ["LAHF"], [], IFlag.INST_FLAGS_NONE)

		# V 1.6.21 MOV MEM-OFFSET instructions are NOT automatically promoted to 64bits, only with a REX.
		Set("a0", ["MOV"], [OPT.ACC8, OPT.MOFFS8], IFlag.INST_FLAGS_NONE)
		Set("a1", ["MOV"], [OPT.ACC_FULL, OPT.MOFFS_FULL], IFlag.INST_FLAGS_NONE)
		Set("a2", ["MOV"], [OPT.MOFFS8, OPT.ACC8], IFlag.INST_FLAGS_NONE)
		Set("a3", ["MOV"], [OPT.MOFFS_FULL, OPT.ACC_FULL], IFlag.INST_FLAGS_NONE)

		# movs es:di, ds:si*
		# cmps ds:si*, es:di
		# stos es:di, al
		# scas es:di al
		# lods al, ds:si*
		Set("a4", ["MOVS"], [OPT.REGI_EDI, OPT.REGI_ESI], IFlag.PRE_REPNZ | IFlag.PRE_REP)
		Set("a5", ["MOVS"], [OPT.REGI_EDI, OPT.REGI_ESI], IFlag._16BITS | IFlag.PRE_REPNZ | IFlag.PRE_REP | IFlag._64BITS | IFlag.PRE_REX)
		Set("a6", ["CMPS"], [OPT.REGI_ESI, OPT.REGI_EDI], IFlag.PRE_REPNZ | IFlag.PRE_REP)
		Set("a7", ["CMPS"], [OPT.REGI_ESI, OPT.REGI_EDI], IFlag._16BITS | IFlag.PRE_REPNZ | IFlag.PRE_REP | IFlag._64BITS | IFlag.PRE_REX)
		Set("aa", ["STOS"], [OPT.REGI_EDI, OPT.ACC8], IFlag.PRE_REPNZ | IFlag.PRE_REP)
		Set("ab", ["STOS"], [OPT.REGI_EDI, OPT.ACC_FULL], IFlag._16BITS | IFlag.PRE_REPNZ | IFlag.PRE_REP | IFlag._64BITS | IFlag.PRE_REX)
		Set("ac", ["LODS"], [OPT.ACC8, OPT.REGI_ESI], IFlag.PRE_REPNZ | IFlag.PRE_REP)
		Set("ad", ["LODS"], [OPT.ACC_FULL, OPT.REGI_ESI], IFlag._16BITS | IFlag.PRE_REPNZ | IFlag.PRE_REP | IFlag._64BITS | IFlag.PRE_REX)
		Set("ae", ["SCAS"], [OPT.REGI_EDI, OPT.ACC8], IFlag.PRE_REPNZ | IFlag.PRE_REP)
		Set("af", ["SCAS"], [OPT.REGI_EDI, OPT.ACC_FULL], IFlag._16BITS | IFlag.PRE_REPNZ | IFlag.PRE_REP | IFlag._64BITS | IFlag.PRE_REX)
		Set("a8", ["TEST"], [OPT.ACC8, OPT.IMM8], IFlag.INST_FLAGS_NONE)
		Set("a9", ["TEST"], [OPT.ACC_FULL, OPT.IMM_FULL], IFlag.INST_FLAGS_NONE)
		Set("b0", ["MOV"], [OPT.IB_RB, OPT.IMM8], IFlag.GEN_BLOCK)
		Set("b8", ["MOV"], [OPT.IB_R_FULL, OPT.IMM_FULL], IFlag._64BITS | IFlag.PRE_REX | IFlag.GEN_BLOCK)
		Set("c0 /00", ["ROL"], [OPT.RM8, OPT.IMM8], IFlag.MODRM_REQUIRED)
		Set("c0 /01", ["ROR"], [OPT.RM8, OPT.IMM8], IFlag.MODRM_REQUIRED)
		Set("c0 /02", ["RCL"], [OPT.RM8, OPT.IMM8], IFlag.MODRM_REQUIRED)
		Set("c0 /03", ["RCR"], [OPT.RM8, OPT.IMM8], IFlag.MODRM_REQUIRED)
		Set("c0 /04", ["SHL"], [OPT.RM8, OPT.IMM8], IFlag.MODRM_REQUIRED)
		Set("c0 /05", ["SHR"], [OPT.RM8, OPT.IMM8], IFlag.MODRM_REQUIRED)
		Set("c0 /06", ["SAL"], [OPT.RM8, OPT.IMM8], IFlag.MODRM_REQUIRED)
		Set("c0 /07", ["SAR"], [OPT.RM8, OPT.IMM8], IFlag.MODRM_REQUIRED)
		Set("c1 /00", ["ROL"], [OPT.RM_FULL, OPT.IMM8], IFlag.MODRM_REQUIRED)
		Set("c1 /01", ["ROR"], [OPT.RM_FULL, OPT.IMM8], IFlag.MODRM_REQUIRED)
		Set("c1 /02", ["RCL"], [OPT.RM_FULL, OPT.IMM8], IFlag.MODRM_REQUIRED)
		Set("c1 /03", ["RCR"], [OPT.RM_FULL, OPT.IMM8], IFlag.MODRM_REQUIRED)
		Set("c1 /04", ["SHL"], [OPT.RM_FULL, OPT.IMM8], IFlag.MODRM_REQUIRED)
		Set("c1 /05", ["SHR"], [OPT.RM_FULL, OPT.IMM8], IFlag.MODRM_REQUIRED)
		Set("c1 /06", ["SAL"], [OPT.RM_FULL, OPT.IMM8], IFlag.MODRM_REQUIRED)
		Set("c1 /07", ["SAR"], [OPT.RM_FULL, OPT.IMM8], IFlag.MODRM_REQUIRED)
		Set("c2", ["RET"], [OPT.IMM16], IFlag._64BITS)
		Set("c3", ["RET"], [], IFlag._64BITS)
		Set("c4", ["LES"], [OPT.REG_FULL, OPT.MEM16_FULL], IFlag.MODRM_REQUIRED | IFlag.INVALID_64BITS)
		Set("c5", ["LDS"], [OPT.REG_FULL, OPT.MEM16_FULL], IFlag.MODRM_REQUIRED | IFlag.INVALID_64BITS)
		Set("c6 //00", ["MOV"], [OPT.RM8, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag.NOT_DIVIDED)
		Set("c7 //00", ["MOV"], [OPT.RM_FULL, OPT.IMM_FULL], IFlag.MODRM_REQUIRED | IFlag.NOT_DIVIDED)
		Set("c8", ["ENTER"], [OPT.IMM16_1, OPT.IMM8_2], IFlag._64BITS)
		Set("c9", ["LEAVE"], [], IFlag._64BITS)

		# V 1.1.6 RETF is NOT promoted automatically in 64bits. So with REX it should be RETFQ.
		Set("ca", ["RETF"], [OPT.IMM16], IFlag.NATIVE | IFlag._64BITS | IFlag.PRE_REX)
		Set("cb", ["RETF"], [], IFlag.NATIVE | IFlag._64BITS | IFlag.PRE_REX)
		Set("cc", ["INT 3"], [], IFlag.INST_FLAGS_NONE)
		Set("cd", ["INT"], [OPT.IMM8], IFlag.INST_FLAGS_NONE)
		Set("ce", ["INTO"], [], IFlag.INVALID_64BITS)
		Set("cf", ["IRET"], [], IFlag.NATIVE | IFlag._64BITS | IFlag.PRE_REX)
		Set("d0 /00", ["ROL"], [OPT.RM8, OPT.CONST1], IFlag.MODRM_REQUIRED)
		Set("d0 /01", ["ROR"], [OPT.RM8, OPT.CONST1], IFlag.MODRM_REQUIRED)
		Set("d0 /02", ["RCL"], [OPT.RM8, OPT.CONST1], IFlag.MODRM_REQUIRED)
		Set("d0 /03", ["RCR"], [OPT.RM8, OPT.CONST1], IFlag.MODRM_REQUIRED)
		Set("d0 /04", ["SHL"], [OPT.RM8, OPT.CONST1], IFlag.MODRM_REQUIRED)
		Set("d0 /05", ["SHR"], [OPT.RM8, OPT.CONST1], IFlag.MODRM_REQUIRED)
		Set("d0 /06", ["SAL"], [OPT.RM8, OPT.CONST1], IFlag.MODRM_REQUIRED)
		Set("d0 /07", ["SAR"], [OPT.RM8, OPT.CONST1], IFlag.MODRM_REQUIRED)
		Set("d1 /00", ["ROL"], [OPT.RM_FULL, OPT.CONST1], IFlag.MODRM_REQUIRED)
		Set("d1 /01", ["ROR"], [OPT.RM_FULL, OPT.CONST1], IFlag.MODRM_REQUIRED)
		Set("d1 /02", ["RCL"], [OPT.RM_FULL, OPT.CONST1], IFlag.MODRM_REQUIRED)
		Set("d1 /03", ["RCR"], [OPT.RM_FULL, OPT.CONST1], IFlag.MODRM_REQUIRED)
		Set("d1 /04", ["SHL"], [OPT.RM_FULL, OPT.CONST1], IFlag.MODRM_REQUIRED)
		Set("d1 /05", ["SHR"], [OPT.RM_FULL, OPT.CONST1], IFlag.MODRM_REQUIRED)
		Set("d1 /06", ["SAL"], [OPT.RM_FULL, OPT.CONST1], IFlag.MODRM_REQUIRED)
		Set("d1 /07", ["SAR"], [OPT.RM_FULL, OPT.CONST1], IFlag.MODRM_REQUIRED)
		Set("d2 /00", ["ROL"], [OPT.RM8, OPT.REGCL], IFlag.MODRM_REQUIRED)
		Set("d2 /01", ["ROR"], [OPT.RM8, OPT.REGCL], IFlag.MODRM_REQUIRED)
		Set("d2 /02", ["RCL"], [OPT.RM8, OPT.REGCL], IFlag.MODRM_REQUIRED)
		Set("d2 /03", ["RCR"], [OPT.RM8, OPT.REGCL], IFlag.MODRM_REQUIRED)
		Set("d2 /04", ["SHL"], [OPT.RM8, OPT.REGCL], IFlag.MODRM_REQUIRED)
		Set("d2 /05", ["SHR"], [OPT.RM8, OPT.REGCL], IFlag.MODRM_REQUIRED)
		Set("d2 /06", ["SAL"], [OPT.RM8, OPT.REGCL], IFlag.MODRM_REQUIRED)
		Set("d2 /07", ["SAR"], [OPT.RM8, OPT.REGCL], IFlag.MODRM_REQUIRED)
		Set("d3 /00", ["ROL"], [OPT.RM_FULL, OPT.REGCL], IFlag.MODRM_REQUIRED)
		Set("d3 /01", ["ROR"], [OPT.RM_FULL, OPT.REGCL], IFlag.MODRM_REQUIRED)
		Set("d3 /02", ["RCL"], [OPT.RM_FULL, OPT.REGCL], IFlag.MODRM_REQUIRED)
		Set("d3 /03", ["RCR"], [OPT.RM_FULL, OPT.REGCL], IFlag.MODRM_REQUIRED)
		Set("d3 /04", ["SHL"], [OPT.RM_FULL, OPT.REGCL], IFlag.MODRM_REQUIRED)
		Set("d3 /05", ["SHR"], [OPT.RM_FULL, OPT.REGCL], IFlag.MODRM_REQUIRED)
		Set("d3 /06", ["SAL"], [OPT.RM_FULL, OPT.REGCL], IFlag.MODRM_REQUIRED)
		Set("d3 /07", ["SAR"], [OPT.RM_FULL, OPT.REGCL], IFlag.MODRM_REQUIRED)
		Set("d4", ["AAM"], [OPT.IMM8], IFlag.INVALID_64BITS)
		Set("d5", ["AAD"], [OPT.IMM8], IFlag.INVALID_64BITS)
		Set("d6", ["SALC"], [], IFlag.INVALID_64BITS)

		# XLATB / XLAT BYTE PTR DS:[EBX + AL]
		Set("d7", ["XLAT"], [OPT.REGI_EBXAL], IFlag.PRE_DS)

		# LOOPxx are also affected by the ADDRESS-SIZE prefix!
		# But they require a suffix letter indicating their size.
		# LOOPxx are promoted to 64bits.
		Set("e0", ["LOOPNZ"], [OPT.RELCB], IFlag.PRE_ADDR_SIZE | IFlag.NATIVE)
		Set("e1", ["LOOPZ"], [OPT.RELCB], IFlag.PRE_ADDR_SIZE | IFlag.NATIVE)
		Set("e2", ["LOOP"], [OPT.RELCB], IFlag.PRE_ADDR_SIZE | IFlag.NATIVE)

		# JMP CX:
		# This is a special instruction, because the ADDRESS-SIZE prefix affects its register size!!!
		# INST_PRE_ADDR_SIZE isn't supposed to really be a flag of a static instruction, it's quite a hack to distinguish this instruction.
		# J(r/e)CXZ are promoted to 64bits.
		Set("e3", ["JCXZ", "JECXZ", "JRCXZ"], [OPT.RELCB], IFlag.PRE_ADDR_SIZE | IFlag.USE_EXMNEMONIC | IFlag.USE_EXMNEMONIC2)
		Set("e4", ["IN"], [OPT.ACC8, OPT.IMM8], IFlag.INST_FLAGS_NONE)
		Set("e5", ["IN"], [OPT.ACC_FULL_NOT64, OPT.IMM8], IFlag.INST_FLAGS_NONE)
		Set("e6", ["OUT"], [OPT.IMM8, OPT.ACC8], IFlag.INST_FLAGS_NONE)
		Set("e7", ["OUT"], [OPT.IMM8, OPT.ACC_FULL_NOT64], IFlag.INST_FLAGS_NONE)
		Set("e8", ["CALL"], [OPT.RELC_FULL], IFlag._64BITS)
		Set("e9", ["JMP"], [OPT.RELC_FULL], IFlag._64BITS)
		Set("ea", ["JMP FAR"], [OPT.PTR16_FULL], IFlag.INVALID_64BITS)
		Set("eb", ["JMP"], [OPT.RELCB], IFlag._64BITS)
		Set("ec", ["IN"], [OPT.ACC8, OPT.REGDX], IFlag.INST_FLAGS_NONE)
		Set("ed", ["IN"], [OPT.ACC_FULL_NOT64, OPT.REGDX], IFlag.INST_FLAGS_NONE)
		Set("ee", ["OUT"], [OPT.REGDX, OPT.ACC8], IFlag.INST_FLAGS_NONE)
		Set("ef", ["OUT"], [OPT.REGDX, OPT.ACC_FULL_NOT64], IFlag.INST_FLAGS_NONE)
		Set("f1", ["INT1"], [], IFlag.INST_FLAGS_NONE)
		Set("f4", ["HLT"], [], IFlag.INST_FLAGS_NONE)
		Set("f5", ["CMC"], [], IFlag.INST_FLAGS_NONE)
		Set("f6 /00", ["TEST"], [OPT.RM8, OPT.IMM8], IFlag.MODRM_REQUIRED)
		Set("f6 /02", ["NOT"], [OPT.RM8], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("f6 /03", ["NEG"], [OPT.RM8], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("f6 /04", ["MUL"], [OPT.RM8], IFlag.MODRM_REQUIRED)
		Set("f6 /05", ["IMUL"], [OPT.RM8], IFlag.MODRM_REQUIRED)
		Set("f6 /06", ["DIV"], [OPT.RM8], IFlag.MODRM_REQUIRED)
		Set("f6 /07", ["IDIV"], [OPT.RM8], IFlag.MODRM_REQUIRED)
		Set("f7 /00", ["TEST"], [OPT.RM_FULL, OPT.IMM_FULL], IFlag.MODRM_REQUIRED)
		Set("f7 /02", ["NOT"], [OPT.RM_FULL], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("f7 /03", ["NEG"], [OPT.RM_FULL], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("f7 /04", ["MUL"], [OPT.RM_FULL], IFlag.MODRM_REQUIRED)
		Set("f7 /05", ["IMUL"], [OPT.RM_FULL], IFlag.MODRM_REQUIRED)
		Set("f7 /06", ["DIV"], [OPT.RM_FULL], IFlag.MODRM_REQUIRED)
		Set("f7 /07", ["IDIV"], [OPT.RM_FULL], IFlag.MODRM_REQUIRED)
		Set("f8", ["CLC"], [], IFlag.INST_FLAGS_NONE)
		Set("f9", ["STC"], [], IFlag.INST_FLAGS_NONE)
		Set("fa", ["CLI"], [], IFlag.INST_FLAGS_NONE)
		Set("fb", ["STI"], [], IFlag.INST_FLAGS_NONE)
		Set("fc", ["CLD"], [], IFlag.INST_FLAGS_NONE)
		Set("fd", ["STD"], [], IFlag.INST_FLAGS_NONE)
		Set("fe /00", ["INC"], [OPT.RM8], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("fe /01", ["DEC"], [OPT.RM8], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("ff /00", ["INC"], [OPT.RM_FULL], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("ff /01", ["DEC"], [OPT.RM_FULL], IFlag.MODRM_REQUIRED | IFlag.PRE_LOCK)
		Set("ff /02", ["CALL"], [OPT.RM_FULL], IFlag.MODRM_REQUIRED | IFlag._64BITS)
		Set("ff /03", ["CALL FAR"], [OPT.MEM16_FULL], IFlag.MODRM_REQUIRED | IFlag._64BITS | IFlag.PRE_REX)
		Set("ff /04", ["JMP"], [OPT.RM_FULL], IFlag.MODRM_REQUIRED | IFlag._64BITS)
		Set("ff /05", ["JMP FAR"], [OPT.MEM16_FULL], IFlag.MODRM_REQUIRED | IFlag._64BITS | IFlag.PRE_REX)
		Set("ff /06", ["PUSH"], [OPT.RM_FULL], IFlag.MODRM_REQUIRED | IFlag._64BITS)

		# New instructions from AMD July 2007 (POPCNT is already defined in SSE4.2, MONITOR, MWAIT are already defined above):
		# Note LZCNT can be prefixed by 0x66 although it has also a mandatory prefix!
		Set("f3, 0f, bd", ["LZCNT"], [OPT.REG_FULL, OPT.RM_FULL], IFlag.MODRM_REQUIRED | IFlag.PRE_OP_SIZE)

		Set("0f, 38, f0", ["MOVBE"], [OPT.REG_FULL, OPT.RM_FULL], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 38, f1", ["MOVBE"], [OPT.RM_FULL, OPT.REG_FULL], IFlag.MODRM_REQUIRED | IFlag._32BITS)

		# New instructions from Intel 2008:
		Set("0f, 01, d0", ["XGETBV"], [], IFlag._32BITS)
		Set("0f, 01, d1", ["XSETBV"], [], IFlag._32BITS)
		Set("0f, ae /04", ["XSAVE", "", "XSAVE64"], [OPT.MEM], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._64BITS | IFlag.PRE_REX | IFlag.USE_EXMNEMONIC2)
		# XRSTOR is declared below (see LFENCE), cause it is shared with LFENCE.

		# New instruction from Intel September 2009:
		Set("0f, 37", ["GETSEC"], [], IFlag._32BITS)

		# XSAVEOPT is declared below (see SFENCE).

	def init_Exported(self):
		""" Exported instruction are special instructions that create a collision in the DB.
		Therefore they are exported directly so diStorm can use them manually in the
		insts.c instruction look-up code.
		Note that their opcodes are totally ignored here.
		Also the path to the instruction in the trie has to be defined by any instruction with same opcode!
		So for instance, NOP|PAUSE|XCHG -> XHG is really defined, the rest are exported.
		Inside diStorm it will know which one to use. """

		Set = lambda *args: self.SetCallback(ISetClass.INTEGER, *args)
		# 63 /R
		# 16/32: ARPL reg/mem16, reg16
		# 64: MOVSXD OT_REG_FULL, OT_RM_FULL
		# Damn processor, my DB won't support mixing of operands types.
		# Define ARPL!
		Set("63", ["ARPL"], [OPT.RM16, OPT.REG16], IFlag.MODRM_REQUIRED)

		# MOVSXD:
		# This is the worst defined instruction ever. It has so many variations.
		# I decided after a third review, to make it like MOVSXD RAX, EAX when there IS a REX.W.
		# Otherwise it will be MOVSXD EAX, EAX, which really zero extends to RAX.
		# Completely ignoring DB 0x66, which is possible by the docs, BTW.
		Set("63", ["MOVSXD"], [OPT.REG32_64, OPT.RM32], IFlag.MODRM_REQUIRED | IFlag._64BITS | IFlag.PRE_REX | IFlag.EXPORTED)

		Set("90", ["NOP"], [], IFlag.EXPORTED)
		# This instruction is supported directly in diStorm, since it's not a mandatory prefix really.
		Set("f3, 90", ["PAUSE"], [], IFlag._32BITS | IFlag.EXPORTED)

		# Wait instruction is needed, but it can be a prefix. See next page for more info.
		Set("9b", ["WAIT"], [], IFlag.EXPORTED)

		# VMPTRLD and RDRAND use same 2 first bytes and 06 as group (thus 2.3 bytes).
		# When MOD is 3 it's the RDRAND instruction and for the rest it's VMPTRLD.
		# The problem is that they have different operands, so a hack is required in the lookup instruction code.
		# Plus remember that this opcode is prefixed (because of VMCLEAR) sometimes and therefore will be part of a prefixed table!
		Set("0f, c7 /06", ["RDRAND"], [OPT.RM_FULL], IFlag._32BITS | IFlag.MODRM_INCLUDED | IFlag.MODRM_REQUIRED | IFlag._64BITS | IFlag.EXPORTED)

		Set = lambda *args: self.SetCallback(ISetClass._3DNOW, *args)
		# This is not really an instruction, but a gateway to all 3dnow instructions.
		Set("0f, 0f", ["_3DNOW"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._3DNOW_FETCH | IFlag.EXPORTED)

	def init_FPU(self):
		Set = lambda *args: self.SetCallback(ISetClass.FPU, *args)

		# The WAIT instruction is tricky, it starts a 3 bytes instruction series.
		# If you find a 3 bytes long instruction you are on your own.
		# But the problem is that if you don't find a 3 bytes long instruction and the first byte that is going to be DB'ed
		# is this 0x9b byte, which represents the WAIT instruction, thus you'll have to output it as a standalone instruction.
		# Example:
		# 9B DB E3 ~ FINIT
		# 9B DB E4 ~ WAIT; DB 0xDB; ...
		# Get the idea?
		# It might be a part of a long instruction (3 bytes), else it just a simple one byte instruction by its own.
		# This way is a simple rule which is broken easily when dealing with Trie DB, the whole point is that the byte says
		# "read another byte" or "I'm your one", but here both happens.
		# That's why I will have to hardcode the WAIT instruction in the decode function which DB'es unknown bytes.
		# Set(0x9b, "WAIT") ....
		# IFlag.PRE_OP_SIZE is set in order to tell the decoder that 0x9b can be part of the instruction.
		# Because it's found in the prefixed table at the same entry of 0x66 for mandatory prefix.

		Set("9b, d9 //06", ["FSTENV"], [OPT.MEM], IFlag.PRE_OP_SIZE | IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("9b, d9 //07", ["FSTCW"], [OPT.FPUM16], IFlag.PRE_OP_SIZE | IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("9b, db //e2", ["FCLEX"], [], IFlag.PRE_OP_SIZE | IFlag._32BITS)
		Set("9b, db //e3", ["FINIT"], [], IFlag.PRE_OP_SIZE | IFlag._32BITS)
		Set("9b, dd //06", ["FSAVE"], [OPT.MEM], IFlag.PRE_OP_SIZE | IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("9b, dd //07", ["FSTSW"], [OPT.FPUM16], IFlag.PRE_OP_SIZE | IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("9b, df //e0", ["FSTSW"], [OPT.ACC16], IFlag.PRE_OP_SIZE | IFlag._32BITS)


		Set("d8 //00", ["FADD"], [OPT.FPUM32], IFlag.MODRM_REQUIRED)
		Set("d8 //01", ["FMUL"], [OPT.FPUM32], IFlag.MODRM_REQUIRED)
		Set("d8 //02", ["FCOM"], [OPT.FPUM32], IFlag.MODRM_REQUIRED)
		Set("d8 //03", ["FCOMP"], [OPT.FPUM32], IFlag.MODRM_REQUIRED)
		Set("d8 //04", ["FSUB"], [OPT.FPUM32], IFlag.MODRM_REQUIRED)
		Set("d8 //05", ["FSUBR"], [OPT.FPUM32], IFlag.MODRM_REQUIRED)
		Set("d8 //06", ["FDIV"], [OPT.FPUM32], IFlag.MODRM_REQUIRED)
		Set("d8 //07", ["FDIVR"], [OPT.FPUM32], IFlag.MODRM_REQUIRED)
		Set("d8 //c0", ["FADD"], [OPT.FPU_SSI], IFlag.GEN_BLOCK)
		Set("d8 //c8", ["FMUL"], [OPT.FPU_SSI], IFlag.GEN_BLOCK)
		Set("d8 //d0", ["FCOM"], [OPT.FPU_SI], IFlag.GEN_BLOCK)
		Set("d8 //d8", ["FCOMP"], [OPT.FPU_SI], IFlag.GEN_BLOCK)
		Set("d8 //d9", ["FCOMP"], [], IFlag.INST_FLAGS_NONE)
		Set("d8 //e0", ["FSUB"], [OPT.FPU_SSI], IFlag.GEN_BLOCK)
		Set("d8 //e8", ["FSUBR"], [OPT.FPU_SSI], IFlag.GEN_BLOCK)
		Set("d8 //f0", ["FDIV"], [OPT.FPU_SSI], IFlag.GEN_BLOCK)
		Set("d8 //f8", ["FDIVR"], [OPT.FPU_SSI], IFlag.GEN_BLOCK)
		Set("d9 //00", ["FLD"], [OPT.FPUM32], IFlag.MODRM_REQUIRED)
		Set("d9 //02", ["FST"], [OPT.FPUM32], IFlag.MODRM_REQUIRED)
		Set("d9 //03", ["FSTP"], [OPT.FPUM32], IFlag.MODRM_REQUIRED)
		Set("d9 //04", ["FLDENV"], [OPT.MEM], IFlag.MODRM_REQUIRED)
		Set("d9 //05", ["FLDCW"], [OPT.FPUM16], IFlag.MODRM_REQUIRED)
		Set("d9 //06", ["FNSTENV"], [OPT.MEM], IFlag.MODRM_REQUIRED)
		Set("d9 //07", ["FNSTCW"], [OPT.FPUM16], IFlag.MODRM_REQUIRED)
		Set("d9 //c0", ["FLD"], [OPT.FPU_SI], IFlag.GEN_BLOCK)
		Set("d9 //c8", ["FXCH"], [OPT.FPU_SI], IFlag.GEN_BLOCK)
		Set("d9 //c9", ["FXCH"], [], IFlag.INST_FLAGS_NONE)
		Set("d9 //d0", ["FNOP"], [], IFlag.INST_FLAGS_NONE)
		Set("d9 //e0", ["FCHS"], [], IFlag.INST_FLAGS_NONE)
		Set("d9 //e1", ["FABS"], [], IFlag.INST_FLAGS_NONE)
		Set("d9 //e4", ["FTST"], [], IFlag.INST_FLAGS_NONE)
		Set("d9 //e5", ["FXAM"], [], IFlag.INST_FLAGS_NONE)
		Set("d9 //e8", ["FLD1"], [], IFlag.INST_FLAGS_NONE)
		Set("d9 //e9", ["FLDL2T"], [], IFlag.INST_FLAGS_NONE)
		Set("d9 //ea", ["FLDL2E"], [], IFlag.INST_FLAGS_NONE)
		Set("d9 //eb", ["FLDPI"], [], IFlag.INST_FLAGS_NONE)
		Set("d9 //ec", ["FLDLG2"], [], IFlag.INST_FLAGS_NONE)
		Set("d9 //ed", ["FLDLN2"], [], IFlag.INST_FLAGS_NONE)
		Set("d9 //ee", ["FLDZ"], [], IFlag.INST_FLAGS_NONE)
		Set("d9 //f0", ["F2XM1"], [], IFlag.INST_FLAGS_NONE)
		Set("d9 //f1", ["FYL2X"], [], IFlag.INST_FLAGS_NONE)
		Set("d9 //f2", ["FPTAN"], [], IFlag.INST_FLAGS_NONE)
		Set("d9 //f3", ["FPATAN"], [], IFlag.INST_FLAGS_NONE)
		Set("d9 //f4", ["FXTRACT"], [], IFlag.INST_FLAGS_NONE)
		Set("d9 //f5", ["FPREM1"], [], IFlag.INST_FLAGS_NONE)
		Set("d9 //f6", ["FDECSTP"], [], IFlag.INST_FLAGS_NONE)
		Set("d9 //f7", ["FINCSTP"], [], IFlag.INST_FLAGS_NONE)
		Set("d9 //f8", ["FPREM"], [], IFlag.INST_FLAGS_NONE)
		Set("d9 //f9", ["FYL2XP1"], [], IFlag.INST_FLAGS_NONE)
		Set("d9 //fa", ["FSQRT"], [], IFlag.INST_FLAGS_NONE)
		Set("d9 //fb", ["FSINCOS"], [], IFlag.INST_FLAGS_NONE)
		Set("d9 //fc", ["FRNDINT"], [], IFlag.INST_FLAGS_NONE)
		Set("d9 //fd", ["FSCALE"], [], IFlag.INST_FLAGS_NONE)
		Set("d9 //fe", ["FSIN"], [], IFlag.INST_FLAGS_NONE)
		Set("d9 //ff", ["FCOS"], [], IFlag.INST_FLAGS_NONE)
		Set("da //00", ["FIADD"], [OPT.FPUM32], IFlag.MODRM_REQUIRED)
		Set("da //01", ["FIMUL"], [OPT.FPUM32], IFlag.MODRM_REQUIRED)
		Set("da //02", ["FICOM"], [OPT.FPUM32], IFlag.MODRM_REQUIRED)
		Set("da //03", ["FICOMP"], [OPT.FPUM32], IFlag.MODRM_REQUIRED)
		Set("da //04", ["FISUB"], [OPT.FPUM32], IFlag.MODRM_REQUIRED)
		Set("da //05", ["FISUBR"], [OPT.FPUM32], IFlag.MODRM_REQUIRED)
		Set("da //06", ["FIDIV"], [OPT.FPUM32], IFlag.MODRM_REQUIRED)
		Set("da //07", ["FIDIVR"], [OPT.FPUM32], IFlag.MODRM_REQUIRED)
		Set("da //e9", ["FUCOMPP"], [], IFlag.INST_FLAGS_NONE)
		Set("db //00", ["FILD"], [OPT.FPUM32], IFlag.MODRM_REQUIRED)
		Set("db //02", ["FIST"], [OPT.FPUM32], IFlag.MODRM_REQUIRED)
		Set("db //03", ["FISTP"], [OPT.FPUM32], IFlag.MODRM_REQUIRED)
		Set("db //05", ["FLD"], [OPT.FPUM80], IFlag.MODRM_REQUIRED)
		Set("db //07", ["FSTP"], [OPT.FPUM80], IFlag.MODRM_REQUIRED)

		# Obsolete.
		Set("db //e0", ["FENI"], [], IFlag.INST_FLAGS_NONE)

		# Obsolete.
		Set("db //e1", ["FEDISI"], [], IFlag.INST_FLAGS_NONE)
		Set("db //e2", ["FNCLEX"], [], IFlag.INST_FLAGS_NONE)
		Set("db //e3", ["FNINIT"], [], IFlag.INST_FLAGS_NONE)

		# Obsolete.
		Set("db //e4", ["FSETPM"], [], IFlag.INST_FLAGS_NONE)
		Set("db //e8", ["FUCOMI"], [OPT.FPU_SSI], IFlag._32BITS | IFlag.GEN_BLOCK)
		Set("dc //00", ["FADD"], [OPT.FPUM64], IFlag.MODRM_REQUIRED)
		Set("dc //01", ["FMUL"], [OPT.FPUM64], IFlag.MODRM_REQUIRED)
		Set("dc //02", ["FCOM"], [OPT.FPUM64], IFlag.MODRM_REQUIRED)
		Set("dc //03", ["FCOMP"], [OPT.FPUM64], IFlag.MODRM_REQUIRED)
		Set("dc //04", ["FSUB"], [OPT.FPUM64], IFlag.MODRM_REQUIRED)
		Set("dc //05", ["FSUBR"], [OPT.FPUM64], IFlag.MODRM_REQUIRED)
		Set("dc //06", ["FDIV"], [OPT.FPUM64], IFlag.MODRM_REQUIRED)
		Set("dc //07", ["FDIVR"], [OPT.FPUM64], IFlag.MODRM_REQUIRED)
		Set("dc //c0", ["FADD"], [OPT.FPU_SIS], IFlag.GEN_BLOCK)
		Set("dc //c8", ["FMUL"], [OPT.FPU_SIS], IFlag.GEN_BLOCK)
		Set("dc //e0", ["FSUBR"], [OPT.FPU_SIS], IFlag.GEN_BLOCK)
		Set("dc //e8", ["FSUB"], [OPT.FPU_SIS], IFlag.GEN_BLOCK)
		Set("dc //f0", ["FDIVR"], [OPT.FPU_SIS], IFlag.GEN_BLOCK)
		Set("dc //f8", ["FDIV"], [OPT.FPU_SIS], IFlag.GEN_BLOCK)
		Set("dd //00", ["FLD"], [OPT.FPUM64], IFlag.MODRM_REQUIRED)
		Set("dd //02", ["FST"], [OPT.FPUM64], IFlag.MODRM_REQUIRED)
		Set("dd //03", ["FSTP"], [OPT.FPUM64], IFlag.MODRM_REQUIRED)
		Set("dd //04", ["FRSTOR"], [OPT.MEM], IFlag.MODRM_REQUIRED)
		Set("dd //06", ["FNSAVE"], [OPT.MEM], IFlag.MODRM_REQUIRED)
		Set("dd //07", ["FNSTSW"], [OPT.FPUM16], IFlag.MODRM_REQUIRED)
		Set("dd //c0", ["FFREE"], [OPT.FPU_SI], IFlag.GEN_BLOCK)
		Set("dd //d0", ["FST"], [OPT.FPU_SI], IFlag.GEN_BLOCK)
		Set("dd //d8", ["FSTP"], [OPT.FPU_SI], IFlag.GEN_BLOCK)
		Set("dd //e0", ["FUCOM"], [OPT.FPU_SIS], IFlag.GEN_BLOCK)
		Set("dd //e1", ["FUCOM"], [], IFlag.INST_FLAGS_NONE)
		Set("dd //e8", ["FUCOMP"], [OPT.FPU_SI], IFlag.GEN_BLOCK)
		Set("dd //e9", ["FUCOMP"], [], IFlag.INST_FLAGS_NONE)
		Set("de //00", ["FIADD"], [OPT.FPUM16], IFlag.MODRM_REQUIRED)
		Set("de //01", ["FIMUL"], [OPT.FPUM16], IFlag.MODRM_REQUIRED)
		Set("de //02", ["FICOM"], [OPT.FPUM16], IFlag.MODRM_REQUIRED)
		Set("de //03", ["FICOMP"], [OPT.FPUM16], IFlag.MODRM_REQUIRED)
		Set("de //04", ["FISUB"], [OPT.FPUM16], IFlag.MODRM_REQUIRED)
		Set("de //05", ["FISUBR"], [OPT.FPUM16], IFlag.MODRM_REQUIRED)
		Set("de //06", ["FIDIV"], [OPT.FPUM16], IFlag.MODRM_REQUIRED)
		Set("de //07", ["FIDIVR"], [OPT.FPUM16], IFlag.MODRM_REQUIRED)
		Set("de //c0", ["FADDP"], [OPT.FPU_SIS], IFlag.GEN_BLOCK)
		Set("de //c1", ["FADDP"], [], IFlag.INST_FLAGS_NONE)
		Set("de //c8", ["FMULP"], [OPT.FPU_SIS], IFlag.GEN_BLOCK)
		Set("de //c9", ["FMULP"], [], IFlag.INST_FLAGS_NONE)
		Set("de //d9", ["FCOMPP"], [], IFlag.INST_FLAGS_NONE)
		Set("de //e0", ["FSUBRP"], [OPT.FPU_SIS], IFlag.GEN_BLOCK)
		Set("de //e1", ["FSUBRP"], [], IFlag.INST_FLAGS_NONE)
		Set("de //e8", ["FSUBP"], [OPT.FPU_SIS], IFlag.GEN_BLOCK)
		Set("de //e9", ["FSUBP"], [], IFlag.INST_FLAGS_NONE)
		Set("de //f0", ["FDIVRP"], [OPT.FPU_SIS], IFlag.GEN_BLOCK)
		Set("de //f1", ["FDIVRP"], [], IFlag.INST_FLAGS_NONE)
		Set("de //f8", ["FDIVP"], [OPT.FPU_SIS], IFlag.GEN_BLOCK)
		Set("de //f9", ["FDIVP"], [], IFlag.INST_FLAGS_NONE)
		Set("df //00", ["FILD"], [OPT.FPUM16], IFlag.MODRM_REQUIRED)
		Set("df //02", ["FIST"], [OPT.FPUM16], IFlag.MODRM_REQUIRED)
		Set("df //03", ["FISTP"], [OPT.FPUM16], IFlag.MODRM_REQUIRED)
		Set("df //04", ["FBLD"], [OPT.FPUM80], IFlag.MODRM_REQUIRED)
		Set("df //05", ["FILD"], [OPT.FPUM64], IFlag.MODRM_REQUIRED)
		Set("df //06", ["FBSTP"], [OPT.FPUM80], IFlag.MODRM_REQUIRED)
		Set("df //07", ["FISTP"], [OPT.FPUM64], IFlag.MODRM_REQUIRED)
		Set("df //e0", ["FNSTSW"], [OPT.ACC16], IFlag.INST_FLAGS_NONE)
		Set("df //e8", ["FUCOMIP"], [OPT.FPU_SSI], IFlag._32BITS | IFlag.GEN_BLOCK)
		Set("df //f0", ["FCOMIP"], [OPT.FPU_SSI], IFlag._32BITS | IFlag.GEN_BLOCK)

	def init_P6(self):
		Set = lambda *args: self.SetCallback(ISetClass.P6, *args)
		Set("0f, 05", ["SYSCALL"], [], IFlag._32BITS)
		Set("0f, 07", ["SYSRET"], [], IFlag._32BITS)
		Set("0f, 34", ["SYSENTER"], [], IFlag._32BITS) # Only AMD states invalid in 64 bits.
		Set("0f, 35", ["SYSEXIT"], [], IFlag._32BITS) # Only AMD states invalid in 64 bits.
		Set("0f, 40", ["CMOVO"], [OPT.REG_FULL, OPT.RM_FULL], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 41", ["CMOVNO"], [OPT.REG_FULL, OPT.RM_FULL], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 42", ["CMOVB"], [OPT.REG_FULL, OPT.RM_FULL], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 43", ["CMOVAE"], [OPT.REG_FULL, OPT.RM_FULL], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 44", ["CMOVZ"], [OPT.REG_FULL, OPT.RM_FULL], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 45", ["CMOVNZ"], [OPT.REG_FULL, OPT.RM_FULL], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 46", ["CMOVBE"], [OPT.REG_FULL, OPT.RM_FULL], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 47", ["CMOVA"], [OPT.REG_FULL, OPT.RM_FULL], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 48", ["CMOVS"], [OPT.REG_FULL, OPT.RM_FULL], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 49", ["CMOVNS"], [OPT.REG_FULL, OPT.RM_FULL], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 4a", ["CMOVP"], [OPT.REG_FULL, OPT.RM_FULL], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 4b", ["CMOVNP"], [OPT.REG_FULL, OPT.RM_FULL], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 4c", ["CMOVL"], [OPT.REG_FULL, OPT.RM_FULL], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 4d", ["CMOVGE"], [OPT.REG_FULL, OPT.RM_FULL], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 4e", ["CMOVLE"], [OPT.REG_FULL, OPT.RM_FULL], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 4f", ["CMOVG"], [OPT.REG_FULL, OPT.RM_FULL], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("da //c0", ["FCMOVB"], [OPT.FPU_SSI], IFlag._32BITS | IFlag.GEN_BLOCK)
		Set("da //c8", ["FCMOVE"], [OPT.FPU_SSI], IFlag._32BITS | IFlag.GEN_BLOCK)
		Set("da //d0", ["FCMOVBE"], [OPT.FPU_SSI], IFlag._32BITS | IFlag.GEN_BLOCK)
		Set("da //d8", ["FCMOVU"], [OPT.FPU_SSI], IFlag._32BITS | IFlag.GEN_BLOCK)
		Set("db //c0", ["FCMOVNB"], [OPT.FPU_SSI], IFlag._32BITS | IFlag.GEN_BLOCK)
		Set("db //c8", ["FCMOVNE"], [OPT.FPU_SSI], IFlag._32BITS | IFlag.GEN_BLOCK)
		Set("db //d0", ["FCMOVNBE"], [OPT.FPU_SSI], IFlag._32BITS | IFlag.GEN_BLOCK)
		Set("db //d8", ["FCMOVNU"], [OPT.FPU_SSI], IFlag._32BITS | IFlag.GEN_BLOCK)
		Set("db //f0", ["FCOMI"], [OPT.FPU_SSI], IFlag._32BITS | IFlag.GEN_BLOCK)

		# Transactional
		Set("0f, 01, d5", ["XEND"], [], IFlag._64BITS)
		Set("c6 //f8", ["XABORT"], [OPT.IMM8], IFlag._64BITS)
		Set("c7 //f8", ["XBEGIN"], [OPT.RELC_FULL], IFlag._64BITS)
		
	def init_MMX(self):
		Set = lambda *args: self.SetCallback(ISetClass.MMX, *args)

		# Pseudo Opcodes, the second mnemonic is concatenated to the first mnemonic.

		Set("0f, 60", ["PUNPCKLBW"], [OPT.MM, OPT.MM32], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 61", ["PUNPCKLWD"], [OPT.MM, OPT.MM32], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 62", ["PUNPCKLDQ"], [OPT.MM, OPT.MM32], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 63", ["PACKSSWB"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 64", ["PCMPGTB"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 65", ["PCMPGTW"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 66", ["PCMPGTD"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 67", ["PACKUSWB"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 68", ["PUNPCKHBW"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 69", ["PUNPCKHWD"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 6a", ["PUNPCKHDQ"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 6b", ["PACKSSDW"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)

		# V 1.6.22 MOVD/MOVQ are used for 32bits or 64bits correspondignly.
		Set("0f, 6e", ["MOVD", "", "MOVQ"], [OPT.MM, OPT.RM32_64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._64BITS | IFlag.PRE_REX | IFlag.USE_EXMNEMONIC2)
		Set("0f, 6f", ["MOVQ"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 71 /02", ["PSRLW"], [OPT.MM_RM, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.MODRR_REQUIRED)
		Set("0f, 71 /04", ["PSRAW"], [OPT.MM_RM, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.MODRR_REQUIRED)
		Set("0f, 71 /06", ["PSLLW"], [OPT.MM_RM, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.MODRR_REQUIRED)
		Set("0f, 72 /02", ["PSRLD"], [OPT.MM_RM, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.MODRR_REQUIRED)
		Set("0f, 72 /04", ["PSRAD"], [OPT.MM_RM, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.MODRR_REQUIRED)
		Set("0f, 72 /06", ["PSLLD"], [OPT.MM_RM, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.MODRR_REQUIRED)
		Set("0f, 73 /02", ["PSRLQ"], [OPT.MM_RM, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.MODRR_REQUIRED)
		Set("0f, 73 /06", ["PSLLQ"], [OPT.MM_RM, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.MODRR_REQUIRED)
		Set("0f, 74", ["PCMPEQB"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 75", ["PCMPEQW"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 76", ["PCMPEQD"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 77", ["EMMS"], [], IFlag._32BITS)

		# V 1.6.22 MOVD/MOVQ are used for 32bits or 64bits correspondignly.
		Set("0f, 7e", ["MOVD", "", "MOVQ"], [OPT.RM32_64, OPT.MM], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._64BITS | IFlag.PRE_REX | IFlag.USE_EXMNEMONIC2)
		Set("0f, 7f", ["MOVQ"], [OPT.MM64, OPT.MM], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, d1", ["PSRLW"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, d2", ["PSRLD"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, d3", ["PSRLQ"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, d5", ["PMULLW"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, d8", ["PSUBUSB"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, d9", ["PSUBUSW"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, db", ["PAND"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, dc", ["PADDUSB"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, dd", ["PADDUSW"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, df", ["PANDN"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, e1", ["PSRAW"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, e2", ["PSRAD"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, e5", ["PMULHW"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, e8", ["PSUBSB"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, e9", ["PSUBSW"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, eb", ["POR"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, ec", ["PADDSB"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, ed", ["PADDSW"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, ef", ["PXOR"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, f1", ["PSLLW"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, f2", ["PSLLD"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, f3", ["PSLLQ"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, f5", ["PMADDWD"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, f8", ["PSUBB"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, f9", ["PSUBW"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, fa", ["PSUBD"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, fc", ["PADDB"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, fd", ["PADDW"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, fe", ["PADDD"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)

	def init_SSE(self):
		Set = lambda *args: self.SetCallback(ISetClass.SSE, *args)
		Set("0f, 10", ["MOVUPS"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 11", ["MOVUPS"], [OPT.XMM128, OPT.XMM], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		
		# The problem with these instructions (MOVHLPS/MOVLHPS) is that both kinds need partialy the ModR/M byte.
		# mod=11 for first mnemonic.

		Set("0f, 12", ["MOVHLPS", "MOVLPS"], [OPT.XMM, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.USE_EXMNEMONIC | IFlag.MNEMONIC_MODRM_BASED)
		Set("0f, 13", ["MOVLPS"], [OPT.MEM64, OPT.XMM], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 14", ["UNPCKLPS"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 15", ["UNPCKHPS"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 16", ["MOVLHPS", "MOVHPS"], [OPT.XMM, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.USE_EXMNEMONIC | IFlag.MNEMONIC_MODRM_BASED)
		Set("0f, 17", ["MOVHPS"], [OPT.MEM64, OPT.XMM], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 18 /00", ["PREFETCHNTA"], [OPT.MEM], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 18 /01", ["PREFETCHT0"], [OPT.MEM], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 18 /02", ["PREFETCHT1"], [OPT.MEM], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 18 /03", ["PREFETCHT2"], [OPT.MEM], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 28", ["MOVAPS"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 29", ["MOVAPS"], [OPT.XMM128, OPT.XMM], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 2a", ["CVTPI2PS"], [OPT.XMM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 2b", ["MOVNTPS"], [OPT.MEM128, OPT.XMM], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 2c", ["CVTTPS2PI"], [OPT.MM, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 2d", ["CVTPS2PI"], [OPT.MM, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 2e", ["UCOMISS"], [OPT.XMM, OPT.XMM32], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 2f", ["COMISS"], [OPT.XMM, OPT.XMM32], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 50", ["MOVMSKPS"], [OPT.REG32, OPT.XMM_RM], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.MODRR_REQUIRED)
		Set("0f, 51", ["SQRTPS"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 52", ["RSQRTPS"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 53", ["RCPPS"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 54", ["ANDPS"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 55", ["ANDNPS"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 56", ["ORPS"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 57", ["XORPS"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 58", ["ADDPS"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 59", ["MULPS"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 5c", ["SUBPS"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 5d", ["MINPS"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 5e", ["DIVPS"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 5f", ["MAXPS"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 70", ["PSHUFW"], [OPT.MM, OPT.MM64, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, c2", ["CMP", "PS"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.USE_EXMNEMONIC | IFlag.PSEUDO_OPCODE)
		Set("0f, c4", ["PINSRW"], [OPT.MM, OPT.R32_M16, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, c5", ["PEXTRW"], [OPT.REG32, OPT.MM_RM, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.MODRR_REQUIRED)
		Set("0f, c6", ["SHUFPS"], [OPT.XMM, OPT.XMM128, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, d7", ["PMOVMSKB"], [OPT.REG32, OPT.MM_RM], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.MODRR_REQUIRED)
		Set("0f, da", ["PMINUB"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, de", ["PMAXUB"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, e0", ["PAVGB"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, e3", ["PAVGW"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, e4", ["PMULHUW"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, e7", ["MOVNTQ"], [OPT.MEM64, OPT.MM], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, ea", ["PMINSW"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, ee", ["PMAXSW"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, f6", ["PSADBW"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, f7", ["MASKMOVQ"], [OPT.MM, OPT.MM_RM], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.MODRR_REQUIRED)
		Set("f3, 0f, 10", ["MOVSS"], [OPT.XMM, OPT.XMM32], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("f3, 0f, 11", ["MOVSS"], [OPT.XMM32, OPT.XMM], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("f3, 0f, 2a", ["CVTSI2SS"], [OPT.XMM, OPT.RM32_64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._64BITS | IFlag.PRE_REX)
		Set("f3, 0f, 2c", ["CVTTSS2SI"], [OPT.REG32_64, OPT.XMM32], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._64BITS | IFlag.PRE_REX)
		Set("f3, 0f, 2d", ["CVTSS2SI"], [OPT.REG32_64, OPT.XMM32], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._64BITS | IFlag.PRE_REX)
		Set("f3, 0f, 51", ["SQRTSS"], [OPT.XMM, OPT.XMM32], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("f3, 0f, 52", ["RSQRTSS"], [OPT.XMM, OPT.XMM32], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("f3, 0f, 53", ["RCPSS"], [OPT.XMM, OPT.XMM32], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("f3, 0f, 58", ["ADDSS"], [OPT.XMM, OPT.XMM32], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("f3, 0f, 59", ["MULSS"], [OPT.XMM, OPT.XMM32], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("f3, 0f, 5c", ["SUBSS"], [OPT.XMM, OPT.XMM32], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("f3, 0f, 5d", ["MINSS"], [OPT.XMM, OPT.XMM32], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("f3, 0f, 5e", ["DIVSS"], [OPT.XMM, OPT.XMM32], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("f3, 0f, 5f", ["MAXSS"], [OPT.XMM, OPT.XMM32], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("f3, 0f, c2", ["CMP", "SS"], [OPT.XMM, OPT.XMM32], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.USE_EXMNEMONIC | IFlag.PSEUDO_OPCODE)

	def init_SSE2(self):
		Set = lambda *args: self.SetCallback(ISetClass.SSE2, *args)
		Set("0f, 5a", ["CVTPS2PD"], [OPT.XMM, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 5b", ["CVTDQ2PS"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, c3", ["MOVNTI"], [OPT.MEM32_64, OPT.REG32_64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._64BITS | IFlag.PRE_REX)
		Set("0f, d4", ["PADDQ"], [OPT.XMM, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, f4", ["PMULUDQ"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, fb", ["PSUBQ"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 10", ["MOVUPD"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 11", ["MOVUPD"], [OPT.XMM128, OPT.XMM], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 12", ["MOVLPD"], [OPT.XMM, OPT.MEM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 13", ["MOVLPD"], [OPT.MEM64, OPT.XMM], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 14", ["UNPCKLPD"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 15", ["UNPCKHPD"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 16", ["MOVHPD"], [OPT.XMM, OPT.MEM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 17", ["MOVHPD"], [OPT.MEM64, OPT.XMM], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 28", ["MOVAPD"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 29", ["MOVAPD"], [OPT.XMM128, OPT.XMM], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 2a", ["CVTPI2PD"], [OPT.XMM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 2b", ["MOVNTPD"], [OPT.MEM128, OPT.XMM], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 2c", ["CVTTPD2PI"], [OPT.MM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 2d", ["CVTPD2PI"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 2e", ["UCOMISD"], [OPT.XMM, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 2f", ["COMISD"], [OPT.XMM, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 50", ["MOVMSKPD"], [OPT.REG32, OPT.XMM_RM], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.MODRR_REQUIRED)
		Set("66, 0f, 51", ["SQRTPD"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 54", ["ANDPD"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 55", ["ANDNPD"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 56", ["ORPD"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 57", ["XORPD"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 58", ["ADDPD"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 59", ["MULPD"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 5a", ["CVTPD2PS"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 5b", ["CVTPS2DQ"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 5c", ["SUBPD"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 5d", ["MINPD"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 5e", ["DIVPD"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 5f", ["MAXPD"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 60", ["PUNPCKLBW"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 61", ["PUNPCKLWD"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 62", ["PUNPCKLDQ"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 63", ["PACKSSWB"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 64", ["PCMPGTB"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 65", ["PCMPGTW"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 66", ["PCMPGTD"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 67", ["PACKUSWB"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 68", ["PUNPCKHBW"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 69", ["PUNPCKHWD"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 6a", ["PUNPCKHDQ"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 6b", ["PACKSSDW"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 6c", ["PUNPCKLQDQ"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 6d", ["PUNPCKHQDQ"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)

		# V 1.6.22 MOVD/MOVQ are used for 32bits or 64bits correspondignly.
		Set("66, 0f, 6e", ["MOVD", "", "MOVQ"], [OPT.XMM, OPT.RM32_64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._64BITS | IFlag.PRE_REX | IFlag.USE_EXMNEMONIC2)
		Set("66, 0f, 6f", ["MOVDQA"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 70", ["PSHUFD"], [OPT.XMM, OPT.XMM128, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 71 /02", ["PSRLW"], [OPT.XMM_RM, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 71 /04", ["PSRAW"], [OPT.XMM_RM, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 71 /06", ["PSLLW"], [OPT.XMM_RM, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 72 /02", ["PSRLD"], [OPT.XMM_RM, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 72 /04", ["PSRAD"], [OPT.XMM_RM, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 72 /06", ["PSLLD"], [OPT.XMM_RM, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 73 /02", ["PSRLQ"], [OPT.XMM_RM, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 73 /03", ["PSRLDQ"], [OPT.XMM_RM, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 73 /06", ["PSLLQ"], [OPT.XMM_RM, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 73 /07", ["PSLLDQ"], [OPT.XMM_RM, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 74", ["PCMPEQB"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 75", ["PCMPEQW"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 76", ["PCMPEQD"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)

		# V 1.6.22 MOVD/MOVQ are used for 32bits or 64bits correspondignly.
		Set("66, 0f, 7e", ["MOVD", "", "MOVQ"], [OPT.RM32_64, OPT.XMM], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._64BITS | IFlag.PRE_REX | IFlag.USE_EXMNEMONIC2)
		Set("66, 0f, 7f", ["MOVDQA"], [OPT.XMM128, OPT.XMM], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, c2", ["CMP", "PD"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.USE_EXMNEMONIC | IFlag.PSEUDO_OPCODE)
		Set("66, 0f, c4", ["PINSRW"], [OPT.XMM, OPT.R32_M16, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, c5", ["PEXTRW"], [OPT.REG32, OPT.XMM_RM, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, c6", ["SHUFPD"], [OPT.XMM, OPT.XMM128, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, d1", ["PSRLW"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, d2", ["PSRLD"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, d3", ["PSRLQ"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, d4", ["PADDQ"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, d5", ["PMULLW"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, d6", ["MOVQ"], [OPT.XMM64, OPT.XMM], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, d7", ["PMOVMSKB"], [OPT.REG32, OPT.XMM_RM], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.MODRR_REQUIRED)
		Set("66, 0f, d8", ["PSUBUSB"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, d9", ["PSUBUSW"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, da", ["PMINUB"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, db", ["PAND"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, dc", ["PADDUSB"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, dd", ["PADDUSW"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, de", ["PMAXUB"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, df", ["PANDN"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, e0", ["PAVGB"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, e1", ["PSRAW"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, e2", ["PSRAD"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, e3", ["PAVGW"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, e4", ["PMULHUW"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, e5", ["PMULHW"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, e6", ["CVTTPD2DQ"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, e7", ["MOVNTDQ"], [OPT.MEM128, OPT.XMM], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, e8", ["PSUBSB"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, e9", ["PSUBSW"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, ea", ["PMINSW"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, eb", ["POR"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, ec", ["PADDSB"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, ed", ["PADDSW"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, ee", ["PMAXSW"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, ef", ["PXOR"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, f1", ["PSLLW"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, f2", ["PSLLD"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, f3", ["PSLLQ"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, f4", ["PMULUDQ"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, f5", ["PMADDWD"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, f6", ["PSADBW"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, f7", ["MASKMOVDQU"], [OPT.XMM, OPT.XMM_RM], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.MODRR_REQUIRED)
		Set("66, 0f, f8", ["PSUBB"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, f9", ["PSUBW"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, fa", ["PSUBD"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, fb", ["PSUBQ"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, fc", ["PADDB"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, fd", ["PADDW"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, fe", ["PADDD"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("f2, 0f, 10", ["MOVSD"], [OPT.XMM, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("f2, 0f, 11", ["MOVSD"], [OPT.XMM64, OPT.XMM], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("f2, 0f, 2a", ["CVTSI2SD"], [OPT.XMM, OPT.RM32_64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._64BITS | IFlag.PRE_REX)
		Set("f2, 0f, 2c", ["CVTTSD2SI"], [OPT.REG32_64, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._64BITS | IFlag.PRE_REX)
		Set("f2, 0f, 2d", ["CVTSD2SI"], [OPT.REG32_64, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._64BITS | IFlag.PRE_REX)
		Set("f2, 0f, 51", ["SQRTSD"], [OPT.XMM, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("f2, 0f, 58", ["ADDSD"], [OPT.XMM, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("f2, 0f, 59", ["MULSD"], [OPT.XMM, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("f2, 0f, 5a", ["CVTSD2SS"], [OPT.XMM, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("f2, 0f, 5c", ["SUBSD"], [OPT.XMM, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("f2, 0f, 5d", ["MINSD"], [OPT.XMM, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("f2, 0f, 5e", ["DIVSD"], [OPT.XMM, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("f2, 0f, 5f", ["MAXSD"], [OPT.XMM, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("f2, 0f, 70", ["PSHUFLW"], [OPT.XMM, OPT.XMM128, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("f2, 0f, c2", ["CMP", "SD"], [OPT.XMM, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.USE_EXMNEMONIC | IFlag.PSEUDO_OPCODE)
		Set("f2, 0f, d6", ["MOVDQ2Q"], [OPT.MM, OPT.XMM_RM], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.MODRR_REQUIRED)
		Set("f2, 0f, e6", ["CVTPD2DQ"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("f3, 0f, 5a", ["CVTSS2SD"], [OPT.XMM, OPT.XMM32], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("f3, 0f, 5b", ["CVTTPS2DQ"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("f3, 0f, 6f", ["MOVDQU"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("f3, 0f, 70", ["PSHUFHW"], [OPT.XMM, OPT.XMM128, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)

		Set("f3, 0f, 7e", ["MOVQ"], [OPT.XMM, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("f3, 0f, 7f", ["MOVDQU"], [OPT.XMM128, OPT.XMM], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("f3, 0f, d6", ["MOVQ2DQ"], [OPT.XMM, OPT.MM_RM], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.MODRR_REQUIRED)
		Set("f3, 0f, e6", ["CVTDQ2PD"], [OPT.XMM, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)

	def init_SSE3(self):
		Set = lambda *args: self.SetCallback(ISetClass.SSE3, *args)
		Set("66, 0f, 7c", ["HADDPD"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 7d", ["HSUBPD"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, d0", ["ADDSUBPD"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("db //01", ["FISTTP"], [OPT.FPUM32], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("dd //01", ["FISTTP"], [OPT.FPUM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("df //01", ["FISTTP"], [OPT.FPUM16], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("f2, 0f, 12", ["MOVDDUP"], [OPT.XMM, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("f3, 0f, 12", ["MOVSLDUP"], [OPT.XMM, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("f2, 0f, 7c", ["HADDPS"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("f2, 0f, 7d", ["HSUBPS"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("f2, 0f, d0", ["ADDSUBPS"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("f2, 0f, f0", ["LDDQU"], [OPT.XMM, OPT.MEM], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("f3, 0f, 16", ["MOVSHDUP"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)

	def init_SSSE3(self):
		Set = lambda *args: self.SetCallback(ISetClass.SSSE3, *args)
		Set("0f, 38, 00", ["PSHUFB"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 38, 01", ["PHADDW"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 38, 02", ["PHADDD"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 38, 03", ["PHADDSW"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 38, 04", ["PMADDUBSW"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 38, 05", ["PHSUBW"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 38, 06", ["PHSUBD"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 38, 07", ["PHSUBSW"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 38, 08", ["PSIGNB"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 38, 09", ["PSIGNW"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 38, 0a", ["PSIGND"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 38, 0b", ["PMULHRSW"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 38, 1c", ["PABSB"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 38, 1d", ["PABSW"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 38, 1e", ["PABSD"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 3a, 0f", ["PALIGNR"], [OPT.MM, OPT.MM64, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 38, 00", ["PSHUFB"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 38, 01", ["PHADDW"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 38, 02", ["PHADDD"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 38, 03", ["PHADDSW"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 38, 04", ["PMADDUBSW"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 38, 05", ["PHSUBW"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 38, 06", ["PHSUBD"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 38, 07", ["PHSUBSW"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 38, 08", ["PSIGNB"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 38, 09", ["PSIGNW"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 38, 0a", ["PSIGND"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 38, 0b", ["PMULHRSW"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 38, 1c", ["PABSB"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 38, 1d", ["PABSW"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 38, 1e", ["PABSD"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 3a, 0f", ["PALIGNR"], [OPT.XMM, OPT.XMM128, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)

	def init_SSE4_1(self):
		Set = lambda *args: self.SetCallback(ISetClass.SSE4_1, *args)
		Set("66, 0f, 3a, 0d", ["BLENDPD"], [OPT.XMM, OPT.XMM128, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 3a, 0c", ["BLENDPS"], [OPT.XMM, OPT.XMM128, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 38, 15", ["BLENDVPD"], [OPT.XMM, OPT.XMM128, OPT.REGXMM0], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 38, 14", ["BLENDVPS"], [OPT.XMM, OPT.XMM128, OPT.REGXMM0], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 3a, 41", ["DPPD"], [OPT.XMM, OPT.XMM128, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 3a, 40", ["DPPS"], [OPT.XMM, OPT.XMM128, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 3a, 17", ["EXTRACTPS"], [OPT.RM32_64, OPT.XMM, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 3a, 21", ["INSERTPS"], [OPT.XMM, OPT.XMM32, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 38, 2a", ["MOVNTDQA"], [OPT.XMM, OPT.MEM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 3a, 42", ["MPSADBW"], [OPT.XMM, OPT.XMM128, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 38, 2b", ["PACKUSDW"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 38, 10", ["PBLENDVB"], [OPT.XMM, OPT.XMM128, OPT.REGXMM0], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 3a, 0e", ["PBLENDW"], [OPT.XMM, OPT.XMM128, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 38, 29", ["PCMPEQQ"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 3a, 14", ["PEXTRB"], [OPT.R32_64_M8, OPT.XMM, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 3a, 16", ["PEXTRD", "", "PEXTRQ"], [OPT.RM32_64, OPT.XMM, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.USE_EXMNEMONIC2)
		Set("66, 0f, 3a, 15", ["PEXTRW"], [OPT.R32_64_M16, OPT.XMM, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 38, 41", ["PHMINPOSUW"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 3a, 20", ["PINSRB"], [OPT.XMM, OPT.R32_M8, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 3a, 22", ["PINSRD", "", "PINSRQ"], [OPT.XMM, OPT.RM32_64, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.USE_EXMNEMONIC2)
		Set("66, 0f, 38, 3c", ["PMAXSB"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 38, 3d", ["PMAXSD"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 38, 3f", ["PMAXUD"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 38, 3e", ["PMAXUW"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 38, 38", ["PMINSB"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 38, 39", ["PMINSD"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 38, 3b", ["PMINUD"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 38, 3a", ["PMINUW"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 38, 20", ["PMOVSXBW"], [OPT.XMM, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 38, 21", ["PMOVSXBD"], [OPT.XMM, OPT.XMM32], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 38, 22", ["PMOVSXBQ"], [OPT.XMM, OPT.XMM16], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 38, 23", ["PMOVSXWD"], [OPT.XMM, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 38, 24", ["PMOVSXWQ"], [OPT.XMM, OPT.XMM32], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 38, 25", ["PMOVSXDQ"], [OPT.XMM, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 38, 30", ["PMOVZXBW"], [OPT.XMM, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 38, 31", ["PMOVZXBD"], [OPT.XMM, OPT.XMM32], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 38, 32", ["PMOVZXBQ"], [OPT.XMM, OPT.XMM16], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 38, 33", ["PMOVZXWD"], [OPT.XMM, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 38, 34", ["PMOVZXWQ"], [OPT.XMM, OPT.XMM32], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 38, 35", ["PMOVZXDQ"], [OPT.XMM, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 38, 28", ["PMULDQ"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 38, 40", ["PMULLD"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 3a, 09", ["ROUNDPD"], [OPT.XMM, OPT.XMM128, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 3a, 08", ["ROUNDPS"], [OPT.XMM, OPT.XMM128, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 3a, 0b", ["ROUNDSD"], [OPT.XMM, OPT.XMM64, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 3a, 0a", ["ROUNDSS"], [OPT.XMM, OPT.XMM32, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)

	def init_SSE4_2(self):
		Set = lambda *args: self.SetCallback(ISetClass.SSE4_2, *args)
		Set("f2, 0f, 38, f0", ["CRC32"], [OPT.REG32_64, OPT.RM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("f2, 0f, 38, f1", ["CRC32"], [OPT.REG32_64, OPT.RM_FULL], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 38, 17", ["PTEST"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 3a, 61", ["PCMPESTRI"], [OPT.XMM, OPT.XMM128, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 3a, 60", ["PCMPESTRM"], [OPT.XMM, OPT.XMM128, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 3a, 63", ["PCMPISTRI"], [OPT.XMM, OPT.XMM128, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 3a, 62", ["PCMPISTRM"], [OPT.XMM, OPT.XMM128, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 38, 37", ["PCMPGTQ"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)

		# Note POPCNT can be prefixed by 0x66 although it has also a mandatory prefix!
		Set("f3, 0f, b8", ["POPCNT"], [OPT.REG_FULL, OPT.RM_FULL], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_OP_SIZE)

	def init_SSE4_A(self):
		# New instructions from AMD July 2007
		Set = lambda *args: self.SetCallback(ISetClass.SSE4_A, *args)
		# DB can't support a table after Prefixed table (it will really complicate everything and doesn't worth it),
		# therefore we will have to force a REG of 0 in the flags! Beats me. :(
		#Set("66, 0f, 78 /00", ["EXTRQ"], [OPT.XMM_RM, OPT.IMM8_1, OPT.IMM8_2], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, 78", ["EXTRQ"], [OPT.XMM_RM, OPT.IMM8_1, OPT.IMM8_2], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.FORCE_REG0)
		Set("66, 0f, 79", ["EXTRQ"], [OPT.XMM, OPT.XMM_RM], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		# Four operands(!) I want m'mommy
		Set("f2, 0f, 78", ["INSERTQ"], [OPT.XMM, OPT.XMM_RM, OPT.IMM8_1, OPT.IMM8_2], IFlag.MODRM_REQUIRED | IFlag.MODRR_REQUIRED | IFlag._32BITS)
		Set("f2, 0f, 79", ["INSERTQ"], [OPT.XMM, OPT.XMM_RM], IFlag.MODRM_REQUIRED | IFlag.MODRR_REQUIRED | IFlag._32BITS)
		Set("f2, 0f, 2b", ["MOVNTSD"], [OPT.MEM64, OPT.XMM], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("f3, 0f, 2b", ["MOVNTSS"], [OPT.MEM32, OPT.XMM], IFlag.MODRM_REQUIRED | IFlag._32BITS)

	def init_3DNOW(self):
		Set = lambda *args: self.SetCallback(ISetClass._3DNOW, *args)
		Set("0f, 0d /00", ["PREFETCH"], [OPT.MEM], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 0d /01", ["PREFETCHW"], [OPT.MEM], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 0e", ["FEMMS"], [], IFlag.INST_FLAGS_NONE)
		Set("0f, 0f, 0d", ["PI2FD"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._3DNOW_FETCH)
		Set("0f, 0f, 1d", ["PF2ID"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._3DNOW_FETCH)
		Set("0f, 0f, 90", ["PFCMPGE"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._3DNOW_FETCH)
		Set("0f, 0f, 94", ["PFMIN"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._3DNOW_FETCH)
		Set("0f, 0f, 96", ["PFRCP"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._3DNOW_FETCH)
		Set("0f, 0f, 97", ["PFRSQRT"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._3DNOW_FETCH)
		Set("0f, 0f, 9a", ["PFSUB"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._3DNOW_FETCH)
		Set("0f, 0f, 9e", ["PFADD"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._3DNOW_FETCH)
		Set("0f, 0f, a0", ["PFCMPGT"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._3DNOW_FETCH)
		Set("0f, 0f, a4", ["PFMAX"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._3DNOW_FETCH)
		Set("0f, 0f, a6", ["PFRCPIT1"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._3DNOW_FETCH)
		Set("0f, 0f, a7", ["PFRSQIT1"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._3DNOW_FETCH)
		Set("0f, 0f, aa", ["PFSUBR"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._3DNOW_FETCH)
		Set("0f, 0f, ae", ["PFACC"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._3DNOW_FETCH)
		Set("0f, 0f, b0", ["PFCMPEQ"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._3DNOW_FETCH)
		Set("0f, 0f, b4", ["PFMUL"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._3DNOW_FETCH)
		Set("0f, 0f, b6", ["PFRCPIT2"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._3DNOW_FETCH)
		Set("0f, 0f, b7", ["PMULHRW"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._3DNOW_FETCH)
		Set("0f, 0f, bf", ["PAVGUSB"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._3DNOW_FETCH)

	def init_3DNOWEXT(self):
		Set = lambda *args: self.SetCallback(ISetClass._3DNOWEXT, *args)
		Set("0f, 0f, 0c", ["PI2FW"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._3DNOW_FETCH)
		Set("0f, 0f, 1c", ["PF2IW"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._3DNOW_FETCH)
		Set("0f, 0f, 8a", ["PFNACC"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._3DNOW_FETCH)
		Set("0f, 0f, 8e", ["PFPNACC"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._3DNOW_FETCH)
		Set("0f, 0f, bb", ["PSWAPD"], [OPT.MM, OPT.MM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._3DNOW_FETCH)

	def init_VMX(self):
		Set = lambda *args: self.SetCallback(ISetClass.VMX, *args)
		Set("0f, 01 //c1", ["VMCALL"], [], IFlag._32BITS)
		Set("0f, 01 //c2", ["VMLAUNCH"], [], IFlag._32BITS)
		Set("0f, 01 //c3", ["VMRESUME"], [], IFlag._32BITS)
		Set("0f, 01 //c4", ["VMXOFF"], [], IFlag._32BITS)

		# In 64bits the operands are promoted to 64bits automatically.
		Set("0f, 78", ["VMREAD"], [OPT.RM32_64, OPT.REG32_64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._64BITS)
		Set("0f, 79", ["VMWRITE"], [OPT.REG32_64, OPT.RM32_64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._64BITS)
		# VMPTRLD collides with RDRAND (see exported instructions).
		Set("0f, c7 /06", ["VMPTRLD"], [OPT.MEM], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, c7 /07", ["VMPTRST"], [OPT.MEM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("66, 0f, c7 /06", ["VMCLEAR"], [OPT.MEM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("f3, 0f, c7 /06", ["VMXON"], [OPT.MEM64], IFlag.MODRM_REQUIRED | IFlag._32BITS)

		# New VMX instructions from Intel September 2009:
		Set("66, 0f, 38, 80", ["INVEPT"], [OPT.REG32_64, OPT.MEM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._64BITS)
		Set("66, 0f, 38, 81", ["INVVPID"], [OPT.REG32_64, OPT.MEM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._64BITS)

		# New instructions from Intel December 2011.
		Set("0f, 01 //d4", ["VMFUNC"], [], IFlag._32BITS)

		Set("66, 0f, 38, 82", ["INVPCID"], [OPT.REG32_64, OPT.MEM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._64BITS)
		# Can be prefixed with 0x66, see LZCNT.
		Set("f3, 0f, bc", ["TZCNT"], [OPT.REG_FULL, OPT.RM_FULL], IFlag.MODRM_REQUIRED | IFlag.PRE_OP_SIZE)

		Set("f3, 0f, ae /0", ["RDFSBASE"], [OPT.REG32_64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._64BITS)
		Set("f3, 0f, ae /1", ["RDGSBASE"], [OPT.REG32_64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._64BITS)
		Set("f3, 0f, ae /2", ["WRFSBASE"], [OPT.REG32_64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._64BITS)
		Set("f3, 0f, ae /3", ["WRGSBASE"], [OPT.REG32_64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._64BITS)

		# SSE5.
		Set("0f, 7a, 31", ["CVTPS2PH"], [OPT.XMM64, OPT.XMM], IFlag._32BITS, IFlag._64BITS)
		Set("0f, 7a, 30", ["CVTPH2PS"], [OPT.XMM, OPT.XMM64], IFlag._32BITS, IFlag._64BITS)

	def init_SVM(self):
		Set = lambda *args: self.SetCallback(ISetClass.SVM, *args)
		Set("0f, 01 //d8", ["VMRUN"], [OPT.REGI_EAX], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		Set("0f, 01 //d9", ["VMMCALL"], [], IFlag._32BITS)
		Set("0f, 01 //da", ["VMLOAD"], [OPT.REGI_EAX], IFlag._32BITS)
		Set("0f, 01 //db", ["VMSAVE"], [OPT.REGI_EAX], IFlag._32BITS)
		Set("0f, 01 //dc", ["STGI"], [], IFlag._32BITS)
		Set("0f, 01 //dd", ["CLGI"], [], IFlag._32BITS)
		Set("0f, 01 //de", ["SKINIT"], [OPT.REGI_EAX], IFlag._32BITS)
		Set("0f, 01 //df", ["INVLPGA"], [OPT.REGI_EAX, OPT.REGECX], IFlag._32BITS)

	def init_AVX(self):
		# Most SSE/SSE2/SSE3/SSSE3/SSE4 instructions have been promoted, and they are all part of the AVX category.
		# It's easier to keep them separated, also because some of the promoted instructions have different number of operands, etc.

		Set = lambda *args: self.SetCallback(ISetClass.AVX, *args)
		SetAes = lambda *args: self.SetCallback(ISetClass.AES, *args)
		SetClmul = lambda *args: self.SetCallback(ISetClass.CLMUL, *args)

		Set("66, 0f, 58", ["VADDPD"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("0f, 58", ["VADDPS"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("f2, 0f, 58", ["VADDSD"], [OPT.XMM, OPT.VXMM, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("f3, 0f, 58", ["VADDSS"], [OPT.XMM, OPT.VXMM, OPT.XMM32], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)

		Set("66, 0f, d0", ["VADDSUBPD"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("f2, 0f, d0", ["VADDSUBPS"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)

		SetAes("66, 0f, 38, dc", ["AESENC"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		SetAes("66, 0f, 38, dd", ["AESENCLAST"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		SetAes("66, 0f, 38, dc", ["VAESENC"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		SetAes("66, 0f, 38, dd", ["VAESENCLAST"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)

		SetAes("66, 0f, 38, de", ["AESDEC"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		SetAes("66, 0f, 38, df", ["AESDECLAST"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		SetAes("66, 0f, 38, de", ["VAESDEC"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		SetAes("66, 0f, 38, df", ["VAESDECLAST"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)

		SetAes("66, 0f, 38, db", ["AESIMC"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		SetAes("66, 0f, 38, db", ["VAESIMC"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)

		SetAes("66, 0f, 3a, df", ["AESKEYGENASSIST"], [OPT.XMM, OPT.XMM128, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		SetAes("66, 0f, 3a, df", ["VAESKEYGENASSIST"], [OPT.XMM, OPT.XMM128, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)

		Set("66, 0f, 54", ["VANDPD"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("0f, 54", ["VANDPS"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("66, 0f, 55", ["VANDNPD"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("0f, 55", ["VANDNPS"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)

		Set("66, 0f, 3a, 0d", ["VBLENDPD"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("66, 0f, 3a, 0c", ["VBLENDPS"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("66, 0f, 3a, 4b", ["VBLENDVPD"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256, OPT.YXMM_IMM], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("66, 0f, 3a, 4a", ["VBLENDVPS"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256, OPT.YXMM_IMM], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)

		Set("66, 0f, 38, 18", ["VBROADCASTSS"], [OPT.YXMM, OPT.MEM32], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("66, 0f, 38, 19", ["VBROADCASTSD"], [OPT.YMM, OPT.MEM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.FORCE_VEXL)
		Set("66, 0f, 38, 1a", ["VBROADCASTF128"], [OPT.YMM, OPT.MEM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.FORCE_VEXL)

		Set("66, 0f, c2", ["VCMP", "PD"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L | IFlag.PSEUDO_OPCODE)
		Set("0f, c2", ["VCMP", "PS"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L | IFlag.PSEUDO_OPCODE)
		Set("f2, 0f, c2", ["VCMP", "SD"], [OPT.XMM, OPT.VXMM, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.PSEUDO_OPCODE)
		Set("f3, 0f, c2", ["VCMP", "SS"], [OPT.XMM, OPT.VXMM, OPT.XMM32], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.PSEUDO_OPCODE)

		Set("66, 0f, 2f", ["VCOMISD"], [OPT.XMM, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("0f, 2f", ["VCOMISS"], [OPT.XMM, OPT.XMM32], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)

		Set("f3, 0f, e6", ["VCVTDQ2PD"], [OPT.YXMM, OPT.LXMM64_128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("0f, 5b", ["VCVTDQ2PS"], [OPT.YXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("f2, 0f, e6", ["VCVTPD2DQ"], [OPT.XMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("66, 0f, 5a", ["VCVTPD2PS"], [OPT.XMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("66, 0f, 5b", ["VCVTPS2DQ"], [OPT.YXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("0f, 5a", ["VCVTPS2PD"], [OPT.YXMM, OPT.LXMM64_128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("f2, 0f, 2d", ["VCVTSD2SI"], [OPT.WREG32_64, OPT.WXMM32_64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_W)
		Set("f2, 0f, 5a", ["VCVTSD2SS"], [OPT.XMM, OPT.VXMM, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("f2, 0f, 2a", ["VCVTSI2SD"], [OPT.XMM, OPT.VXMM, OPT.WRM32_64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_W)
		Set("f3, 0f, 2a", ["VCVTSI2SS"], [OPT.XMM, OPT.VXMM, OPT.WRM32_64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_W)
		Set("f3, 0f, 5a", ["VCVTSS2SD"], [OPT.XMM, OPT.VXMM, OPT.XMM32], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("f3, 0f, 2d", ["VCVTSS2SI"], [OPT.WREG32_64, OPT.WXMM32_64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_W)
		Set("66, 0f, e6", ["VCVTTPD2DQ"], [OPT.XMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("f3, 0f, 5b", ["VCVTTPS2DQ"], [OPT.YXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("f2, 0f, 2c", ["VCVTTSD2SI"], [OPT.WREG32_64, OPT.WXMM32_64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_W)
		Set("f3, 0f, 2c", ["VCVTTSS2SI"], [OPT.WREG32_64, OPT.WXMM32_64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_W)

		Set("66, 0f, 5e", ["VDIVPD"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("0f, 5e", ["VDIVPS"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("f2, 0f, 5e", ["VDIVSD"], [OPT.XMM, OPT.VXMM, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("f3, 0f, 5e", ["VDIVSS"], [OPT.XMM, OPT.VXMM, OPT.XMM32], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)

		Set("66, 0f, 3a, 41", ["VDPPD"], [OPT.XMM, OPT.VXMM, OPT.XMM128, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 3a, 40", ["VDPPS"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)

		Set("66, 0f, 3a, 19", ["VEXTRACTF128"], [OPT.XMM128, OPT.YMM, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.FORCE_VEXL)
		Set("66, 0f, 3a, 17", ["VEXTRACTPS"], [OPT.RM32, OPT.XMM, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)

		Set("66, 0f, 7c", ["VHADDPD"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("f2, 0f, 7c", ["VHADDPS"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("66, 0f, 7d", ["VHSUBPD"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("f2, 0f, 7d", ["VHSUBPS"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)

		Set("66, 0f, 3a, 18", ["VINSERTF128"], [OPT.YMM, OPT.VYMM, OPT.XMM128, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.FORCE_VEXL)
		Set("66, 0f, 3a, 21", ["VINSERTPS"], [OPT.XMM, OPT.VXMM, OPT.XMM32, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("f2, 0f, f0", ["VLDDQU"], [OPT.YXMM, OPT.LMEM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)

		Set("0f, ae /02", ["VLDMXCSR"], [OPT.MEM32], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)

		Set("66, 0f, f7", ["VMASKMOVDQU"], [OPT.XMM, OPT.XMM_RM], IFlag.MODRM_REQUIRED | IFlag.MODRR_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)

		Set("66, 0f, 38, 2c", ["VMASKMOVPS"], [OPT.YXMM, OPT.VYXMM, OPT.LMEM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("66, 0f, 38, 2d", ["VMASKMOVPD"], [OPT.YXMM, OPT.VYXMM, OPT.LMEM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("66, 0f, 38, 2e", ["VMASKMOVPS"], [OPT.LMEM128_256, OPT.VYXMM, OPT.YXMM], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("66, 0f, 38, 2f", ["VMASKMOVPD"], [OPT.LMEM128_256, OPT.VYXMM, OPT.YXMM], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)

		Set("66, 0f, 5f", ["VMAXPD"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("0f, 5f", ["VMAXPS"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("f2, 0f, 5f", ["VMAXSD"], [OPT.XMM, OPT.VXMM, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("f3, 0f, 5f", ["VMAXSS"], [OPT.XMM, OPT.VXMM, OPT.XMM32], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)

		Set("66, 0f, 5d", ["VMINPD"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("0f, 5d", ["VMINPS"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("f2, 0f, 5d", ["VMINSD"], [OPT.XMM, OPT.VXMM, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("f3, 0f, 5d", ["VMINSS"], [OPT.XMM, OPT.VXMM, OPT.XMM32], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)

		Set("66, 0f, 28", ["VMOVAPD"], [OPT.YXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("66, 0f, 29", ["VMOVAPD"], [OPT.YXMM128_256, OPT.YXMM], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("0f, 28", ["VMOVAPS"], [OPT.YXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("0f, 29", ["VMOVAPS"], [OPT.YXMM128_256, OPT.YXMM], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)

		Set("66, 0f, 6e", ["VMOVD", "VMOVQ"], [OPT.XMM, OPT.WRM32_64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_W | IFlag.MNEMONIC_VEXW_BASED)
		Set("66, 0f, 7e", ["VMOVD", "VMOVQ"], [OPT.WRM32_64, OPT.XMM], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_W | IFlag.MNEMONIC_VEXW_BASED)
		Set("f3, 0f, 7e", ["VMOVQ"], [OPT.XMM, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, d6", ["VMOVQ"], [OPT.XMM64, OPT.XMM], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("f2, 0f, 12", ["VMOVDDUP"], [OPT.YXMM, OPT.YXMM64_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("66, 0f, 6f", ["VMOVDQA"], [OPT.YXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("66, 0f, 7f", ["VMOVDQA"], [OPT.YXMM128_256, OPT.YXMM], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("f3, 0f, 6f", ["VMOVDQU"], [OPT.YXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("f3, 0f, 7f", ["VMOVDQU"], [OPT.YXMM128_256, OPT.YXMM], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)

		Set("0f, 12", ["VMOVHLPS", "VMOVLPS"], [OPT.XMM, OPT.VXMM, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.MNEMONIC_MODRM_BASED | IFlag.USE_EXMNEMONIC)
		Set("66, 0f, 12", ["VMOVLPD"], [OPT.XMM, OPT.VXMM, OPT.MEM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("0f, 13", ["VMOVLPS"], [OPT.MEM64, OPT.XMM], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 13", ["VMOVLPD"], [OPT.MEM64, OPT.XMM], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("0f, 16", ["VMOVLHPS", "VMOVHPS"], [OPT.XMM, OPT.VXMM, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.MNEMONIC_MODRM_BASED | IFlag.USE_EXMNEMONIC)
		Set("66, 0f, 16", ["VMOVHPD"], [OPT.XMM, OPT.VXMM, OPT.MEM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("0f, 17", ["VMOVHPS"], [OPT.MEM64, OPT.XMM], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 17", ["VMOVHPD"], [OPT.MEM64, OPT.XMM], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)

		Set("66, 0f, 50", ["VMOVMSKPD"], [OPT.REG32_64, OPT.YXMM], IFlag.MODRM_REQUIRED | IFlag.MODRR_REQUIRED | IFlag._32BITS | IFlag._64BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("0f, 50", ["VMOVMSKPS"], [OPT.REG32_64, OPT.YXMM], IFlag.MODRM_REQUIRED | IFlag.MODRR_REQUIRED | IFlag._32BITS | IFlag._64BITS | IFlag.PRE_VEX | IFlag.VEX_L)

		Set("66, 0f, e7", ["VMOVNTDQ"], [OPT.LMEM128_256, OPT.YXMM], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("66, 0f, 38, 2a", ["VMOVNTDQA"], [OPT.XMM, OPT.MEM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 2b", ["VMOVNTPD"], [OPT.LMEM128_256, OPT.YXMM], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("0f, 2b", ["VMOVNTPS"], [OPT.LMEM128_256, OPT.YXMM], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)

		# Next two instructions are based on vvvv field.
		Set("f2, 0f, 10", ["VMOVSD"], [OPT.XMM, OPT.VXMM, OPT.XMM_RM], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.MODRM_INCLUDED | IFlag.MODRR_BASED)
		Set("f2, 0f, 10", ["VMOVSD"], [OPT.XMM, OPT.MEM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.MODRM_INCLUDED | IFlag.MODRR_BASED)

		# Next two instructions are based on vvvv field.
		Set("f2, 0f, 11", ["VMOVSD"], [OPT.XMM, OPT.VXMM, OPT.XMM_RM], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.MODRM_INCLUDED | IFlag.MODRR_BASED)
		Set("f2, 0f, 11", ["VMOVSD"], [OPT.MEM64, OPT.XMM], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.MODRM_INCLUDED | IFlag.MODRR_BASED)

		Set("f3, 0f, 16", ["VMOVSHDUP"], [OPT.YXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("f3, 0f, 12", ["VMOVSLDUP"], [OPT.YXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)

		# Next two instructions are based on vvvv field.
		Set("f3, 0f, 10", ["VMOVSS"], [OPT.XMM, OPT.VXMM, OPT.XMM_RM], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.MODRM_INCLUDED | IFlag.MODRR_BASED)
		Set("f3, 0f, 10", ["VMOVSS"], [OPT.XMM, OPT.MEM32], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.MODRM_INCLUDED | IFlag.MODRR_BASED)

		# Next two instructions are based on vvvv field.
		Set("f3, 0f, 11", ["VMOVSS"], [OPT.XMM, OPT.VXMM, OPT.XMM_RM], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.MODRM_INCLUDED | IFlag.MODRR_BASED)
		Set("f3, 0f, 11", ["VMOVSS"], [OPT.MEM32, OPT.XMM], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.MODRM_INCLUDED | IFlag.MODRR_BASED)

		Set("66, 0f, 10", ["VMOVUPD"], [OPT.YXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("66, 0f, 11", ["VMOVUPD"], [OPT.YXMM128_256, OPT.YXMM], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("0f, 10", ["VMOVUPS"], [OPT.YXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("0f, 11", ["VMOVUPS"], [OPT.YXMM128_256, OPT.YXMM], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)

		Set("66, 0f, 3a, 42", ["VMPSADBW"], [OPT.XMM, OPT.VXMM, OPT.XMM128, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)

		Set("66, 0f, 59", ["VMULPD"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("0f, 59", ["VMULPS"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("f2, 0f, 59", ["VMULSD"], [OPT.XMM, OPT.VXMM, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("f3, 0f, 59", ["VMULSS"], [OPT.XMM, OPT.VXMM, OPT.XMM32], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)

		Set("66, 0f, 56", ["VORPD"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("0f, 56", ["VORPS"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)

		Set("66, 0f, 38, 1c", ["VPABSB"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 38, 1d", ["VPABSW"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 38, 1e", ["VPABSD"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 63", ["VPACKSSWB"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 6b", ["VPACKSSDW"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 67", ["VPACKUSWB"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 38, 2b", ["VPACKUSDW"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, fc", ["VPADDB"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, fd", ["VPADDW"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, fe", ["VPADDD"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, d4", ["VPADDQ"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, ec", ["VPADDSB"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, ed", ["VPADDSW"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, dc", ["VPADDUSW"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, dd", ["VPADDUSW"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)

		Set("66, 0f, 3a, 0f", ["VPALIGNR"], [OPT.XMM, OPT.VXMM, OPT.XMM128, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)

		Set("66, 0f, db", ["VPAND"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, df", ["VPANDN"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)

		Set("66, 0f, e0", ["VPAVGB"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, e3", ["VPAVGW"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)

		Set("66, 0f, 3a, 4c", ["VPBLENDVB"], [OPT.XMM, OPT.VXMM, OPT.XMM128, OPT.XMM_IMM], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 3a, 0e", ["VPBLENDW"], [OPT.XMM, OPT.VXMM, OPT.XMM128, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)

		# This instruction is not prefixed with VEX.
		SetClmul("66, 0f, 3a, 44", ["PCLMULQDQ"], [OPT.XMM, OPT.XMM128, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS)
		# Next instruction is prefixed with VEX.
		SetClmul("66, 0f, 3a, 44", ["VPCLMULQDQ"], [OPT.XMM, OPT.VXMM, OPT.XMM128, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)

		Set("66, 0f, 3a, 61", ["VPCMPESTRI"], [OPT.XMM, OPT.XMM128, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 3a, 60", ["VPCMPESTRM"], [OPT.XMM, OPT.XMM128, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 3a, 63", ["VPCMPISTRI"], [OPT.XMM, OPT.XMM128, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 3a, 62", ["VPCMPISTRM"], [OPT.XMM, OPT.XMM128, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)

		Set("66, 0f, 74", ["VPCMPEQB"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 75", ["VPCMPEQW"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 76", ["VPCMPEQD"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 38, 29", ["VPCMPEQQ"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)

		Set("66, 0f, 64", ["VPCMPGTB"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 65", ["VPCMPGTW"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 66", ["VPCMPGTD"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 38, 37", ["VPCMPGTQ"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)

		Set("66, 0f, 38, 0d", ["VPERMILPD"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("66, 0f, 3a, 05", ["VPERMILPD"], [OPT.YXMM, OPT.YXMM128_256, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("66, 0f, 3a, 04", ["VPERMILPS"], [OPT.YXMM, OPT.YXMM128_256, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("66, 0f, 38, 0c", ["VPERMILPS"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("66, 0f, 3a, 06", ["VPERM2F128"], [OPT.YMM, OPT.VYMM, OPT.YMM256, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.FORCE_VEXL)

		Set("66, 0f, 3a, 14", ["VPEXTRB"], [OPT.REG32_64_M8, OPT.XMM, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._64BITS | IFlag.PRE_VEX)
		Set("66, 0f, c5", ["VPEXTRW"], [OPT.REG32_64, OPT.XMM, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._64BITS | IFlag.PRE_VEX)
		Set("66, 0f, 3a, 15", ["VPEXTRW"], [OPT.REG32_64_M16, OPT.XMM, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._64BITS | IFlag.PRE_VEX)
		Set("66, 0f, 3a, 16", ["VPEXTRD", "VPEXTRQ"], [OPT.WRM32_64, OPT.XMM, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_W | IFlag.MNEMONIC_VEXW_BASED)

		Set("66, 0f, 38, 01", ["VPHADDW"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 38, 02", ["VPHADDD"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 38, 03", ["VPHADDSW"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 38, 41", ["VPHMINPOSUW"], [OPT.XMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)

		Set("66, 0f, 38, 05", ["VPHSUBW"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 38, 06", ["VPHSUBD"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 38, 07", ["VPHSUBSW"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)

		Set("66, 0f, 3a, 20", ["VPINSRB"], [OPT.XMM, OPT.VXMM, OPT.REG32_64_M8, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, c4", ["VPINSRW"], [OPT.XMM, OPT.VXMM, OPT.R32_M16, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 3a, 22", ["VPINSRD", "VPINSRQ"], [OPT.XMM, OPT.VXMM, OPT.WRM32_64, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_W | IFlag.MNEMONIC_VEXW_BASED)

		Set("66, 0f, f5", ["VPMADDWD"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 38, 04", ["VPMADDUBSW"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)

		Set("66, 0f, 38, 3c", ["VPMAXSB"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, ee", ["VPMAXSW"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 38, 3d", ["VPMAXSD"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, de", ["VPMAXUB"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 38, 3e", ["VPMAXUW"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 38, 3f", ["VPMAXUD"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)

		Set("66, 0f, 38, 38", ["VPMINSB"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, ea", ["VPMINSW"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 38, 39", ["VPMINSD"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, da", ["VPMINUB"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 38, 3a", ["VPMINUW"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 38, 3b", ["VPMINUD"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)

		Set("66, 0f, d7", ["VPMOVMSKB"], [OPT.REG32_64, OPT.XMM], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag._64BITS | IFlag.PRE_VEX | IFlag.MODRR_REQUIRED)

		Set("66, 0f, 38, 20", ["VPMOVSXBW"], [OPT.XMM, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 38, 21", ["VPMOVSXBD"], [OPT.XMM, OPT.XMM32], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 38, 22", ["VPMOVSXBQ"], [OPT.XMM, OPT.XMM16], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 38, 23", ["VPMOVSXWD"], [OPT.XMM, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 38, 24", ["VPMOVSXWQ"], [OPT.XMM, OPT.XMM32], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 38, 25", ["VPMOVSXDQ"], [OPT.XMM, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)

		Set("66, 0f, 38, 30", ["VPMOVZXBW"], [OPT.XMM, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 38, 31", ["VPMOVZXBD"], [OPT.XMM, OPT.XMM32], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 38, 32", ["VPMOVZXBQ"], [OPT.XMM, OPT.XMM16], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 38, 33", ["VPMOVZXWD"], [OPT.XMM, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 38, 34", ["VPMOVZXWQ"], [OPT.XMM, OPT.XMM32], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 38, 35", ["VPMOVZXDQ"], [OPT.XMM, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)

		Set("66, 0f, e4", ["VPMULHUW"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 38, 0b", ["VPMULHRSW"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, e5", ["VPMULHW"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, d5", ["VPMULLW"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 38, 40", ["VPMULLD"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, f4", ["VPMULUDQ"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 38, 28", ["VPMULDQ"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)

		Set("66, 0f, eb", ["VPOR"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, f6", ["VPSADBW"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 38, 00", ["VPSHUFB"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 70", ["VPSHUFD"], [OPT.XMM, OPT.XMM128, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("f3, 0f, 70", ["VPSHUFHW"], [OPT.XMM, OPT.XMM128, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("f2, 0f, 70", ["VPSHUFLW"], [OPT.XMM, OPT.XMM128, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)

		Set("66, 0f, 38, 08", ["VPSIGNB"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 38, 09", ["VPSIGNW"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 38, 0a", ["VPSIGND"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)

		Set("66, 0f, 73 /07", ["VPSLLDQ"], [OPT.VXMM, OPT.XMM_RM, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.MODRR_REQUIRED)
		Set("66, 0f, 73 /03", ["VPSRLDQ"], [OPT.VXMM, OPT.XMM_RM, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.MODRR_REQUIRED)

		Set("66, 0f, f1", ["VPSLLW"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 71 /06", ["VPSLLW"], [OPT.VXMM, OPT.XMM_RM, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.MODRR_REQUIRED)
		Set("66, 0f, f2", ["VPSLLD"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 72 /06", ["VPSLLD"], [OPT.VXMM, OPT.XMM_RM, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.MODRR_REQUIRED)
		Set("66, 0f, f3", ["VPSLLQ"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 73 /06", ["VPSLLQ"], [OPT.VXMM, OPT.XMM_RM, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.MODRR_REQUIRED)

		Set("66, 0f, e1", ["VPSRAW"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 71 /04", ["VPSRAW"], [OPT.VXMM, OPT.XMM_RM, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.MODRR_REQUIRED)
		Set("66, 0f, e2", ["VPSRAD"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 72 /04", ["VPSRAD"], [OPT.VXMM, OPT.XMM_RM, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.MODRR_REQUIRED)

		Set("66, 0f, d1", ["VPSRLW"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 71 /02", ["VPSRLW"], [OPT.VXMM, OPT.XMM_RM, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.MODRR_REQUIRED)
		Set("66, 0f, d2", ["VPSRLD"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 72 /02", ["VPSRLD"], [OPT.VXMM, OPT.XMM_RM, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.MODRR_REQUIRED)
		Set("66, 0f, d3", ["VPSRLQ"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 73 /02", ["VPSRLQ"], [OPT.VXMM, OPT.XMM_RM, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.MODRR_REQUIRED)

		Set("66, 0f, 38, 17", ["VPTEST"], [OPT.YXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("66, 0f, 38, 0e", ["VTESTPS"], [OPT.YXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("66, 0f, 38, 0f", ["VTESTPD"], [OPT.YXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)

		Set("66, 0f, f8", ["VPSUBB"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, f9", ["VPSUBW"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, fa", ["VPSUBD"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, fb", ["VPSUBQ"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, e8", ["VPSUBSB"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, e9", ["VPSUBSW"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, d8", ["VPSUBUSB"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, d9", ["VPSUBUSW"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)

		Set("66, 0f, 68", ["VPUNPCKHBW"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 69", ["VPUNPCKHWD"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 6a", ["VPUNPCKHDQ"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 6d", ["VPUNPCKHQDQ"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 60", ["VPUNPCKLBW"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 61", ["VPUNPCKLWD"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 62", ["VPUNPCKLDQ"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 6c", ["VPUNPCKLQDQ"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)

		Set("66, 0f, ef", ["VPXOR"], [OPT.XMM, OPT.VXMM, OPT.XMM128], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)

		Set("0f, 53", ["VRCPPS"], [OPT.YXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("f3, 0f, 53", ["VRCPSS"], [OPT.XMM, OPT.VXMM, OPT.XMM32], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("0f, 52", ["VRSQRTPS"], [OPT.YXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("f3, 0f, 52", ["VRSQRTSS"], [OPT.XMM, OPT.VXMM, OPT.XMM32], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)

		Set("66, 0f, 3a, 09", ["VROUNDPD"], [OPT.YXMM, OPT.YXMM128_256, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("66, 0f, 3a, 08", ["VROUNDPS"], [OPT.YXMM, OPT.YXMM128_256, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("66, 0f, 3a, 0b", ["VROUNDSD"], [OPT.XMM, OPT.VXMM, OPT.XMM64, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("66, 0f, 3a, 0a", ["VROUNDSS"], [OPT.XMM, OPT.VXMM, OPT.XMM32, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)

		Set("66, 0f, c6", ["VSHUFPD"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("0f, c6", ["VSHUFPS"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256, OPT.IMM8], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)

		Set("66, 0f, 51", ["VSQRTPD"], [OPT.YXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("0f, 51", ["VSQRTPS"], [OPT.YXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("f2, 0f, 51", ["VSQRTSD"], [OPT.XMM, OPT.VXMM, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("f3, 0f, 51", ["VSQRTSS"], [OPT.XMM, OPT.VXMM, OPT.XMM32], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)

		Set("0f, ae /03", ["VSTMXCSR"], [OPT.MEM32], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)

		Set("66, 0f, 5c", ["VSUBPD"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("0f, 5c", ["VSUBPS"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("f2, 0f, 5c", ["VSUBSD"], [OPT.XMM, OPT.VXMM, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("f3, 0f, 5c", ["VSUBSS"], [OPT.XMM, OPT.VXMM, OPT.XMM32], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)

		Set("66, 0f, 2e", ["VUCOMISD"], [OPT.XMM, OPT.XMM64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)
		Set("0f, 2e", ["VUCOMISS"], [OPT.XMM, OPT.XMM32], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX)

		Set("66, 0f, 15", ["VUNPCKHPD"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("0f, 15", ["VUNPCKHPS"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("66, 0f, 14", ["VUNPCKLPD"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("0f, 14", ["VUNPCKLPS"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)

		Set("66, 0f, 57", ["VXORPD"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)
		Set("0f, 57", ["VXORPS"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L)

		Set("0f, 77", ["VZEROUPPER", "VZEROALL"], [], IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_L | IFlag.MNEMONIC_VEXL_BASED)

	def init_FMA(self):
		Set = lambda *args: self.SetCallback(ISetClass.FMA, *args)

		Set("66, 0f, 38, 98", ["VFMADD132PS", "VFMADD132PD"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_W | IFlag.VEX_L | IFlag.MNEMONIC_VEXW_BASED)
		Set("66, 0f, 38, a8", ["VFMADD213PS", "VFMADD213PD"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_W | IFlag.VEX_L | IFlag.MNEMONIC_VEXW_BASED)
		Set("66, 0f, 38, b8", ["VFMADD231PS", "VFMADD231PD"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_W | IFlag.VEX_L | IFlag.MNEMONIC_VEXW_BASED)

		Set("66, 0f, 38, 99", ["VFMADD132SS", "VFMADD132SD"], [OPT.XMM, OPT.VXMM, OPT.WXMM32_64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_W | IFlag.MNEMONIC_VEXW_BASED)
		Set("66, 0f, 38, a9", ["VFMADD213SS", "VFMADD213SD"], [OPT.XMM, OPT.VXMM, OPT.WXMM32_64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_W | IFlag.MNEMONIC_VEXW_BASED)
		Set("66, 0f, 38, b9", ["VFMADD231SS", "VFMADD231SD"], [OPT.XMM, OPT.VXMM, OPT.WXMM32_64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_W | IFlag.MNEMONIC_VEXW_BASED)

		Set("66, 0f, 38, 96", ["VFMADDSUB132PS", "VFMADDSUB132PD"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_W | IFlag.VEX_L | IFlag.MNEMONIC_VEXW_BASED)
		Set("66, 0f, 38, a6", ["VFMADDSUB213PS", "VFMADDSUB213PD"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_W | IFlag.VEX_L | IFlag.MNEMONIC_VEXW_BASED)
		Set("66, 0f, 38, b6", ["VFMADDSUB231PS", "VFMADDSUB231PD"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_W | IFlag.VEX_L | IFlag.MNEMONIC_VEXW_BASED)

		Set("66, 0f, 38, 97", ["VFMSUBADD132PS", "VFMSUBADD132PD"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_W | IFlag.VEX_L | IFlag.MNEMONIC_VEXW_BASED)
		Set("66, 0f, 38, a7", ["VFMSUBADD213PS", "VFMSUBADD213PD"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_W | IFlag.VEX_L | IFlag.MNEMONIC_VEXW_BASED)
		Set("66, 0f, 38, b7", ["VFMSUBADD231PS", "VFMSUBADD231PD"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_W | IFlag.VEX_L | IFlag.MNEMONIC_VEXW_BASED)

		Set("66, 0f, 38, 9a", ["VFMSUB132PS", "VFMSUB132PD"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_W | IFlag.VEX_L | IFlag.MNEMONIC_VEXW_BASED)
		Set("66, 0f, 38, aa", ["VFMSUB213PS", "VFMSUB213PD"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_W | IFlag.VEX_L | IFlag.MNEMONIC_VEXW_BASED)
		Set("66, 0f, 38, ba", ["VFMSUB231PS", "VFMSUB231PD"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_W | IFlag.VEX_L | IFlag.MNEMONIC_VEXW_BASED)

		Set("66, 0f, 38, 9b", ["VFMSUB132SS", "VFMSUB132SD"], [OPT.XMM, OPT.VXMM, OPT.WXMM32_64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_W | IFlag.MNEMONIC_VEXW_BASED)
		Set("66, 0f, 38, ab", ["VFMSUB213SS", "VFMSUB213SD"], [OPT.XMM, OPT.VXMM, OPT.WXMM32_64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_W | IFlag.MNEMONIC_VEXW_BASED)
		Set("66, 0f, 38, bb", ["VFMSUB231SS", "VFMSUB231SD"], [OPT.XMM, OPT.VXMM, OPT.WXMM32_64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_W | IFlag.MNEMONIC_VEXW_BASED)

		Set("66, 0f, 38, 9c", ["VFNMADD132PS", "VFNMADD132PD"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_W | IFlag.VEX_L | IFlag.MNEMONIC_VEXW_BASED)
		Set("66, 0f, 38, ac", ["VFNMADD213PS", "VFNMADD213PD"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_W | IFlag.VEX_L | IFlag.MNEMONIC_VEXW_BASED)
		Set("66, 0f, 38, bc", ["VFNMADD231PS", "VFNMADD231PD"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_W | IFlag.VEX_L | IFlag.MNEMONIC_VEXW_BASED)

		Set("66, 0f, 38, 9d", ["VFNMADD132SS", "VFNMADD132SD"], [OPT.XMM, OPT.VXMM, OPT.WXMM32_64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_W | IFlag.MNEMONIC_VEXW_BASED)
		Set("66, 0f, 38, ad", ["VFNMADD213SS", "VFNMADD213SD"], [OPT.XMM, OPT.VXMM, OPT.WXMM32_64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_W | IFlag.MNEMONIC_VEXW_BASED)
		Set("66, 0f, 38, bd", ["VFNMADD231SS", "VFNMADD231SD"], [OPT.XMM, OPT.VXMM, OPT.WXMM32_64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_W | IFlag.MNEMONIC_VEXW_BASED)

		Set("66, 0f, 38, 9e", ["VFNMSUB132PS", "VFNMSUB132PD"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_W | IFlag.VEX_L | IFlag.MNEMONIC_VEXW_BASED)
		Set("66, 0f, 38, ae", ["VFNMSUB213PS", "VFNMSUB213PD"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_W | IFlag.VEX_L | IFlag.MNEMONIC_VEXW_BASED)
		Set("66, 0f, 38, be", ["VFNMSUB231PS", "VFNMSUB231PD"], [OPT.YXMM, OPT.VYXMM, OPT.YXMM128_256], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_W | IFlag.VEX_L | IFlag.MNEMONIC_VEXW_BASED)

		Set("66, 0f, 38, 9f", ["VFNMSUB132SS", "VFNMSUB132SD"], [OPT.XMM, OPT.VXMM, OPT.WXMM32_64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_W | IFlag.MNEMONIC_VEXW_BASED)
		Set("66, 0f, 38, af", ["VFNMSUB213SS", "VFNMSUB213SD"], [OPT.XMM, OPT.VXMM, OPT.WXMM32_64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_W | IFlag.MNEMONIC_VEXW_BASED)
		Set("66, 0f, 38, bf", ["VFNMSUB231SS", "VFNMSUB231SD"], [OPT.XMM, OPT.VXMM, OPT.WXMM32_64], IFlag.MODRM_REQUIRED | IFlag._32BITS | IFlag.PRE_VEX | IFlag.VEX_W | IFlag.MNEMONIC_VEXW_BASED)

	def __init__(self, SetCallback):
		""" Initializes all instructions-sets using the given callback.
		The arguments of the callback are as follows:
		(iset-class, opcode-length, list of bytes of opcode, list of string of mnemonics, list of operands, flags) """
		self.SetCallback = SetCallback
		Set = lambda *args: self.SetCallback(ISetClass.INTEGER, *args)
		self.init_INTEGER()
		self.init_Exported()
		self.init_FPU()
		self.init_P6()
		self.init_MMX()
		self.init_SSE()
		self.init_SSE2()
		self.init_SSE3()
		self.init_SSSE3()
		self.init_SSE4_1()
		self.init_SSE4_2()
		self.init_SSE4_A()
		self.init_3DNOW()
		self.init_3DNOWEXT()
		self.init_VMX()
		self.init_SVM()
		self.init_AVX()
		self.init_FMA()
