#ifndef _CEGODISTMANAGER_H_INCLUDED_
#define _CEGODISTMANAGER_H_INCLUDED_
///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoDistManager.h
// -----------------
// Cego distribution manager class definition
// 
// Design and Implementation by Bjoern Lemke
//
// (C)opyright 2000-2019 Bjoern Lemke
//
// INTERFACE MODULE
//
// Class: CegoDistManager
// 
// Description: Distributed table manager
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

#include <lfcbase/Chain.h>
#include <lfcbase/ListT.h>

#include "CegoField.h"
#include "CegoPredDesc.h"
#include "CegoTableManager.h"
#include "CegoDistDbHandler.h"
#include "CegoLockHandler.h"
#include "CegoDatabaseManager.h"

class CegoAction;

class CegoDistManager : public CegoTableManager {

public:
    
    CegoDistManager(CegoDatabaseManager *pDBMng);
    ~CegoDistManager();

    void createTableSet(const Chain& tableSet);

    void startDistTableSet(const Chain& tableSet, const Chain& secondary, bool cleanIt, bool doAutoCorrect, bool doTempReset, bool cpDump);
    void registerObjects(const Chain& tableSet);

    void stopDistTableSet(const Chain& tableSet, bool archComplete);

    bool distObjectExists( const Chain& tableSet, const Chain& objName, CegoObject::ObjectType objType);

    void getDistObject(const Chain& tableSet, const Chain& objName, CegoObject::ObjectType type, CegoDecodableObject& oe);
    void getLocalObject(int tabSetId, const Chain& objName, CegoObject::ObjectType type, CegoDecodableObject& oe);

    void createDistDataTable(const Chain& tableSet, const Chain& tableName, CegoObject::ObjectType type, ListT<CegoField>& fldList, ListT<CegoField>& idxList, bool useColumnId = false);
    void createLocalDataTable(int tabSetId, const Chain& tableName, CegoObject::ObjectType type, ListT<CegoField>& fldList, ListT<CegoField>& idxList, bool useColumnId = false);

    void createDistForeignKey( const Chain& tableSet, const Chain& fkey, const Chain& tableName, const ListT<CegoField>& keyList, const Chain& refTable, const ListT<CegoField>& refList);
    void createLocalForeignKey(int tabSetId, const Chain& fkey, const Chain& tableName, const ListT<CegoField>& keyList, const Chain& refTable, const ListT<CegoField>& refList);    

    void createDistCheck( const Chain& tableSet, const Chain& checkName, const Chain& tableName, CegoPredDesc *pPredDesc);
    void createLocalCheck(int tabSetId, const Chain& checkName, const Chain& tableName, CegoPredDesc *pPredDesc);

    void reorgDistObject(const Chain& tableSet, const Chain& tableName, CegoObject::ObjectType type);

    void createDistIndexTable( const Chain& tableSet, const Chain& indexName, const Chain& tableName, ListT<CegoField>& schema, CegoObject::ObjectType type, bool isCached = false);

    void insertDistDataTable(CegoTableObject& oe, ListT<CegoField>& fvl);
    void insertLocalDataTable(CegoTableObject& oe, ListT<CegoField>& fvl);

    void insertDistDataTable(CegoTableObject& oe, ListT< ListT<CegoField> >& fva);
    void insertLocalDataTable(CegoTableObject& oe, ListT< ListT<CegoField> >& fva);

    unsigned long long deleteDistDataTable(CegoTableObject& oe, CegoPredDesc* pPred, CegoProcBlock* pBlock = 0);
    unsigned long long deleteLocalDataTable(CegoTableObject& oe, CegoPredDesc* pPred, CegoProcBlock* pBlock = 0);

    unsigned long long updateDistDataTable(CegoTableObject& oe,
					   CegoPredDesc* pPred,
					   ListT<CegoField>& updSchema,
					   ListT<CegoExpr*>& exprList,
					   bool returnOnFirst,
					   ListT<CegoField>& returnList,
					   CegoProcBlock* pBlock = 0);

    unsigned long long updateLocalDataTable(CegoTableObject& oe,
					    CegoPredDesc* pPred,
					    ListT<CegoField>& updSchema, 
					    ListT<CegoExpr*>& exprList,
					    bool returnOnFirst,
					    ListT<CegoField>& returnList,
					    CegoProcBlock* pBlock = 0);

    void alterDistDataTable(CegoTableObject& oe, const ListT<CegoAlterDesc>& alterList);
    
    void renameDistObject(const Chain& tableSet, const Chain& objName, CegoObject::ObjectType type,  const Chain& newObjName);
    void renameLocalObject(int tabSetId, const Chain& objName, CegoObject::ObjectType type,  const Chain& newObjName);

    void createDistView(const Chain& tableSet, const Chain& viewName, const ListT<CegoField>& schema, const Chain& viewText);
    void createLocalView(int tabSetId, const Chain& viewName, const ListT<CegoField>& schema, const Chain& viewText);

    void createDistTrigger(const Chain& tableSet, const Chain& triggerName,
			   bool isBefore, bool isOnInsert, bool isOnUpdate, bool isOnDelete,
			   const Chain& tableName, const Chain& triggerText);
    
    void createLocalTrigger(int tabSetId, const Chain& triggerName,
			    bool isBefore, bool isOnInsert, bool isOnUpdate, bool isOnDelete,
			    const Chain& tableName , const Chain& triggerText);

    void createDistAlias(const Chain& tableSet, const Chain& aliasName,
			   const Chain& tableName, const ListT<CegoAttrAlias>& aliasList);
    
    void createLocalAlias(int tabSetId, const Chain& aliasName,
			    const Chain& tableName, const ListT<CegoAttrAlias>& aliasList);

    
    void createDistProc(const Chain& tableSet, const Chain& procName, const Chain& procText);
    void createLocalProc(int tabSetId, const Chain& procName, const Chain& procText);

    void truncateDistTable(const Chain& tableName, const Chain& tableSet);
    void dropDistObject(const Chain& tableName, const Chain& tableSet, CegoObject::ObjectType type);

    void getDistObjectList(const Chain& tableSet, CegoObject::ObjectType type, ListT<Chain> &objList);
    void getLocalObjectList(int tabSetId, CegoObject::ObjectType type, ListT<Chain> &objList);

    void getDistObjectByTableList(const Chain& tableSet,
				  const Chain& tabName,
				  ListT<CegoTableObject> &idxList,
				  ListT<CegoBTreeObject> &btreeList,
				  ListT<CegoKeyObject>& keyList,
				  ListT<CegoCheckObject>& checkList,
				  ListT<CegoTriggerObject>& triggerList,
				  ListT<CegoAliasObject>& aliasList,
				  int &numInvalid);

    int getDistPageCount(const Chain& tableSet, const Chain& tabName, CegoObject::ObjectType type);

    void startDistTransaction( const Chain& tableSet);
    long endDistTransaction( const Chain& tableSet);
    long rollbackDistTransaction( const Chain& tableSet);

    int getDistTid( const Chain& tableSet);
    int getLocalTid(int tabSetId);

    void setAllocatedSortArea(unsigned long long n);
    unsigned long long getAllocatedSortArea() const;

    void syncDistTableSet(const Chain& tableSet, const Chain& msg, const Chain& escCmd, int timeout);

    void enableAuth();
    void disableAuth();

    void setActiveUser(const Chain& tableSet, const Chain& user, const Chain& passwd);

    CegoLockHandler* getLockHandle();
    
    const Chain& getUser() const;

    void getActiveUser(const Chain& tableSet, Chain& user, Chain& password);

    Element* verifyTable(const Chain& tableSet, const Chain& tableName);
    Element* verifyView(const Chain& tableSet, const Chain& viewName);
    Element* verifyProcedure(const Chain& tableSet, const Chain& procName);

    Element* correctTable(const Chain& tableSet, const Chain& tableName);

    void getObjectDesc(const Chain& tableSet, const Chain& objName, CegoObject::ObjectType type, 
		       ListT<CegoField>& schema, ListT< ListT<CegoFieldValue> > &fa);


    CegoView* getView(int tabSetId, const Chain& viewName);
    CegoProcedure* getProcedure(int tabSetId, const Chain& procName);
    CegoTrigger* getTrigger(int tabSetId, const Chain& triggerName);

    CegoAction* getParser();

    void reloadView(int tabSetId, const Chain& viewName);
    void reloadProcedure(int tabSetId, const Chain& procName);
    void reloadTrigger(int tabSetId, const Chain& triggerName);
    
private:

    void startTableSet(const Chain& tableSet, const Chain& secondary, bool cleanIt, bool cpDump, bool doAutoCorrect, bool doTempReset);
    
    bool checkKey(int tabSetId, const Chain& tableName, const ListT<CegoField>& schema, const ListT<CegoKeyObject>& keyList);
    
    class ActiveTSUser {
	
    public:
        
        ActiveTSUser();
        ActiveTSUser(const Chain& tableSet);
        ActiveTSUser (const Chain& tableSet, const Chain& userName, const Chain& passwd );
        ~ActiveTSUser();
	
        const Chain& getUserName() const;
        const Chain& getPasswd() const;

        ActiveTSUser& operator = (const ActiveTSUser& u); 
	bool operator == (const ActiveTSUser& t) const;
        
    private:
        
	Chain _tableSet;
        Chain _userName;
	Chain _passwd;
	
    };

    ListT<ActiveTSUser> _userList;

    CegoAction* _pPA;

    Chain _authUser;
    bool _authEnabled;

    unsigned long long _numAllocated;

    unsigned long _modId;
    
};

#endif
