///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoAliasObject.cc 
// ------------------
// Cego alias object entry implementation
//     
// Design and Implementation by Bjoern Lemke               
//     
// (C)opyright 2000-2019 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: CegoCheckObject
//
// Description: Derived class from CegoDecodable to use as a container class for alias objects
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

// CEGO INCLUDES
#include "CegoAliasObject.h"
#include "CegoXMLdef.h"
#include "CegoXMLHelper.h"

// LFC INCLUDES
#include <string.h>
#include <stdlib.h>

CegoAliasObject::CegoAliasObject()
{
    _subCOList.Insert(this);
}

CegoAliasObject::CegoAliasObject(const CegoAliasObject& ao) : CegoContentObject(ao)
{
    _aliasList = ao._aliasList;
    _subCOList.Insert(this);
}

CegoAliasObject::CegoAliasObject(int tabSetId, const Chain& aliasName) : CegoContentObject(tabSetId, CegoObject::ALIAS, aliasName)
{
    _subCOList.Insert(this);
}

// alias schema information is just retrieved at runtime
CegoAliasObject::CegoAliasObject(int tabSetId, const Chain& aliasName, const Chain& tabName, const ListT<CegoAttrAlias>& aliasList) : CegoContentObject(tabSetId, CegoObject::ALIAS, aliasName, tabName, ListT<CegoField>())
{
    _aliasList = aliasList;
    _subCOList.Insert(this);
}

CegoAliasObject::~CegoAliasObject()
{
}

void  CegoAliasObject::setAliasList(const ListT<CegoAttrAlias>& aliasList)
{
    _aliasList = aliasList;
}

const ListT<CegoAttrAlias>& CegoAliasObject::getAliasList()
{
    return _aliasList;
}

int CegoAliasObject::getEntrySize() const
{
    int entrySize = CegoContentObject::getBaseContentSize();
    
    // alias list size
    entrySize += sizeof(int);
    CegoAttrAlias *pAlias = _aliasList.First();
    while ( pAlias )
    {
	entrySize += pAlias->getEncodingLength();
	pAlias = _aliasList.Next();
    }

    return entrySize;
}

void CegoAliasObject::encode(char *buf)
{
    char* bufPtr = buf;

    int entrySize = getEntrySize();

    CegoContentObject::encodeBaseContent(bufPtr, entrySize);
    bufPtr += CegoContentObject::getBaseContentSize();

    int numAlias = _aliasList.Size();
    memcpy (bufPtr, &numAlias, sizeof(int));
    bufPtr += sizeof(int);
    
    CegoAttrAlias *pAlias = _aliasList.First();
    while ( pAlias )
    {
	pAlias->encode(bufPtr);
	bufPtr = bufPtr + pAlias->getEncodingLength();
	pAlias = _aliasList.Next();
    }
}

void CegoAliasObject::decode(char *buf)
{
    char* bufPtr = buf;

    int size;

    CegoContentObject::decodeBaseContent(bufPtr, size);
    bufPtr += CegoContentObject::getBaseContentSize();

    int numAlias;

    memcpy (&numAlias, bufPtr, sizeof(int));
    bufPtr += sizeof(int);

    _aliasList.Empty();
    while ( numAlias > 0 )
    {
	CegoAttrAlias attrAlias;
	attrAlias.decode(bufPtr);
	bufPtr += attrAlias.getEncodingLength();
	_aliasList.Insert(attrAlias);
	numAlias--;
    }
}

void CegoAliasObject::setSchema(const ListT<CegoField>& schema)
{
    _schema = schema;
}

void CegoAliasObject::mapSchema()
{
        
    CegoAttrAlias *pA = _aliasList.First();
    while ( pA )
    {
	CegoField *pF = _schema.Find( CegoField( getTabName(), pA->getAttrName() ) );
	if ( pF )
	{
	    pF->setTableName( getName() );
	    pF->setTableAlias( getTabAlias() );
	    pF->setAttrName( pA->getAliasName() );	    
	}
	else
	{
	    throw Exception(EXLOC, Chain("Cannot find alias attribute ") + pA->getAttrName() + Chain(" in table ") + getTabName());
	}
	pA = _aliasList.Next();	
    }

    // remove non used field from schema
    CegoField *pF = _schema.First();
    while ( pF )
    {
	if ( pF->getTableName() == getTabName() )
	{	    
	    _schema.Remove(*pF);
	    pF = _schema.First();
	}
	else
	{
	    pF = _schema.Next();
	}
    }    
    
}

CegoAliasObject& CegoAliasObject::operator = ( const CegoAliasObject& ao)
{
    CegoContentObject::operator=(ao);
    _aliasList = ao._aliasList;
    return (*this);
}

bool CegoAliasObject::operator == ( const CegoAliasObject& ao)
{
    return CegoContentObject::operator==(ao);
}

Chain CegoAliasObject::getId() const
{
    return getTabName() + Chain("@") + getTableSet() + Chain("@") + getName();
}

Chain CegoAliasObject::toChain() const
{
    Chain s;
    s = CegoObject::toChain();

    s += Chain("TabName: ") + getTabName() + Chain("\n");
    s += Chain("AliasList: ");

    CegoAttrAlias *pAlias = _aliasList.First();
    while ( pAlias )
    {
	s += Chain("Attr=") + pAlias->getAttrName();
	s += Chain(" Alias=") + pAlias->getAliasName() + Chain("\n");	
	pAlias = _aliasList.Next();
    }

    return s;
}

CegoContentObject* CegoAliasObject::clone(bool isAttrRef)
{
    CegoAliasObject *pClone = new CegoAliasObject(*this);
    return (pClone);
    
}

Element* CegoAliasObject::getElement() const
{
    Element* pRoot = new Element(XML_OBJ_ELEMENT);

    pRoot->setAttribute(XML_TSID_ATTR, Chain(getTabSetId()));
    pRoot->setAttribute(XML_OBJTYPE_ATTR, XML_ALIASOBJ_VALUE);
    pRoot->setAttribute(XML_OBJNAME_ATTR, getName());

    pRoot->setAttribute(XML_TABLENAME_ATTR,  getTabName());

    CegoAttrAlias *pA = _aliasList.First();
    while ( pA ) 
    {
	Element *pColElement = new Element(XML_COL_ELEMENT);

	pColElement->setAttribute(XML_COLNAME_ATTR, pA->getAttrName());
	pColElement->setAttribute(XML_COLALIAS_ATTR, pA->getAliasName());
       
	pRoot->addContent(pColElement);
	pA = _aliasList.Next();
    }

    return pRoot;

}

void CegoAliasObject::putElement(Element* pElement)
{
    Element *pRoot = pElement;
    
    if ( pRoot )
    {	
	Chain objName = pRoot->getAttributeValue(XML_OBJNAME_ATTR); 
	setName(objName);

	int tabSetId = pRoot->getAttributeValue(XML_TSID_ATTR).asInteger();
	setTabSetId(tabSetId);	

	setTabName(pRoot->getAttributeValue(XML_TABLENAME_ATTR)); 

	Chain objType = pRoot->getAttributeValue(XML_OBJTYPE_ATTR);
	setType(CegoObject::ALIAS);

		
	ListT<Element*> colList = pRoot->getChildren(XML_COL_ELEMENT);

	_aliasList.Empty();
	Element **pCol = colList.First();
	while ( pCol ) 
	{
	    Chain colName = (*pCol)->getAttributeValue(XML_COLNAME_ATTR);
	    Chain colAlias = (*pCol)->getAttributeValue(XML_COLALIAS_ATTR);
	    _aliasList.Insert( CegoAttrAlias(colName, colAlias) );
	    pCol = colList.Next();
	}
	
    }
}

Chain CegoAliasObject::getFormatted() const
{
    Chain s;

    int maxAttrLen = 12;
    int maxAttrLen1 = 12;
    int maxAttrLen2 = 12;
    
    s += "+" + fill("-", maxAttrLen + 1) + "+" + "\n";
    s += "| ObjectName : ";
    s += getName();
    s += fill(" ", maxAttrLen - 12 - getName().length()) + "|\n";
	
    s += "| ObjectType : alias " +  fill(" ", maxAttrLen - 18) + "|\n";


    s += "| Table      : " + getTabName();
    s += fill(" ", maxAttrLen - 12 - getTabName().length()) + "|" + "\n";
	
    s += "+---------" + fill("-", maxAttrLen1-9)  + "----------" +  fill("-", maxAttrLen2-9) + "+\n";
    s += "| Alias   " +  fill(" ", maxAttrLen1-9) + "          " +  fill(" ", maxAttrLen2-9) + "|\n";
    s += "+---------" + fill("-", maxAttrLen1-9) + "+---------" +  fill("-", maxAttrLen2-9) + "+\n";

    // TODO : add alias list
    // s += _pPredDesc->toChain();

    s +="+" + fill("-", maxAttrLen1) + "+" + fill("-", maxAttrLen2) + "+\n";

    return s;
}

Chain CegoAliasObject::fill(const Chain& s, int num) const
{    
    Chain fs = Chain("");
    while (num > 0)
    {
	fs = fs + s;
	num--;
    }

    return fs;
}
