/***************************************************************************
 *   Copyright (C) 2011 by Francesco Nwokeka <francesco.nwokeka@gmail.com> *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA            *
 ***************************************************************************/

import Qt 4.7
import org.kde.plasma.graphicswidgets 0.1 as PlasmaWidgets

Flipable {
    id: flipableItem;

    property bool flipEnabled;          // flips only if account is enabled
    property string presenceIconType;
    property string presenceText;

    // Tell the delegate that the user has changed his presence
    signal presenceChanged(variant presenceId, variant presenceIconName, variant presenceText);

    width: parent.width;
    height: 30;
    state: "front";

    front:
    PresenceIcon {
        id: frontPresenceIcon;
        iconName: presenceIconType;
        presenceStatusText: presenceText;

        anchors.left: parent.left;

        onClicked: {
            if (flipableItem.flipEnabled) {
                flip();
            }
        }

        Component.onCompleted: {
            // set correct presence status text
            presenceStatusText = checkPresenceStatusText();
        }
    }

    back:
    Item {
        anchors.fill: parent;

        Row {
            anchors.fill: parent;
            spacing: 2;
            PresenceIcon {
                iconName: "online";
                onClicked: { setPresence(1, "available", i18n("Available")); }
            }
            PresenceIcon {
                iconName: "busy";
                onClicked: { setPresence(2, "busy", i18n("Busy")); }
            }
            PresenceIcon {
                iconName: "away";
                onClicked: { setPresence(3, "away", i18n("Away")); }
            }
            PresenceIcon {
                iconName: "invisible";
                onClicked: { setPresence(4, "invisible", i18n("Invisible")); }
            }
            PresenceIcon {
                iconName: "offline";
                onClicked: { setPresence(5, "offline", i18n("Offline")); }
            }
            PlasmaWidgets.IconWidget {
                height: 22;
                width: 22;

                Component.onCompleted: { setIcon("edit-undo"); }
                onClicked: { flip(); }
            }
        }
    }

    transform:
    Rotation {
        id: rotation;
        origin.x: flipableItem.width/2;
        origin.y: flipableItem.height/2;
        axis.x: 1; axis.y: 0; axis.z: 0;
        angle: 0;   // the default angle
    }

    states: [
        State {
            name: "back"
            PropertyChanges {
                target: rotation;
                angle: 180;
            }
        },
        State {
            name: "front"
            PropertyChanges {
                target: rotation;
                angle: 0;
            }
        }
    ]

    transitions:
        Transition {
            NumberAnimation {
                target: rotation;
                property: "angle";
                duration: 3000;
                easing.type: Easing.OutElastic;
            }
        }

    function flip()
    {
        if (flipableItem.state == "front") {
            flipableItem.state = "back";
        } else {
            flipableItem.state = "front";
        }
    }

    function setPresence(presenceId, presenceIconName, presenceText)
    {
        // emit signal to update the dataEngine
        flipableItem.presenceChanged(presenceId, presenceIconName, presenceText);
        flip();
    }

    // used by global setter. I set the front icon and then emit the "presenceChanged" signal
    function setFrontPresence(presenceIconName, presenceText)
    {
        frontPresenceIcon.iconName = presenceIconName;
        frontPresenceIcon.presenceStatusText = presenceText;
        frontPresenceIcon.setPresenceIcon(presenceIconName);
    }

    // checks presence status text and returns correct string for display
    function checkPresenceStatusText()
    {
        switch (frontPresenceIcon.presenceStatusText) {
            case "available":
                return i18n("Available");
            case "dnd":
                return i18n("Busy");
            case "away":
                return i18n("Away");
            case "hidden":
                return i18n("Invisible");
            case "offline":
                return i18n("Offline");
            default:
                return "";
        }
    }
}
