/* gok-keyboard.c
*
* Copyright 2001,2002 Sun Microsystems, Inc.,
* Copyright 2001,2002 University Of Toronto
*
* This library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Library General Public
* License as published by the Free Software Foundation; either
* version 2 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Library General Public License for more details.
*
* You should have received a copy of the GNU Library General Public
* License along with this library; if not, write to the
* Free Software Foundation, Inc., 59 Temple Place - Suite 330,
* Boston, MA 02111-1307, USA.
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#define XK_LATIN1
#include <X11/keysymdef.h> /* ugh, needed by link following hack */
#include <assert.h>
#include <gdk/gdk.h>
#include <math.h>
#include "gok-keyboard.h"
#include "callbacks.h"
#include "gok-branchback-stack.h"
#include "gok-data.h"
#include "main.h"
#include "gok-button.h"
#include "gok-spy-priv.h"
#include "gok-spy.h"
#include "gok-sound.h"
#include "gok-word-complete.h"
#include "gok-log.h"
#include "gok-modifier.h"
#include "gok-output.h"
#include "gok-predictor.h"
#include "gok-composer.h"
#include "gok-windowlister.h"
#include "gok-settings-dialog.h"
#include "gok-page-keysizespace.h"
#include "gok-feedback.h"
#include "gok-repeat.h"

#define NUM_INTERESTING_ROLES 5
#define GOK_KEYBOARD_MAX_ROWS 24

#define CORE_KEYBOARD "Keyboard"

/* if this flag is set then ignore the resize event (because we generated it) */
static gint m_bIgnoreResizeEvent;

/* keeps track of the key size so we know when to resize the font */
static gint m_WidthKeyFont;
static gint m_HeightKeyFont;

/* storage for the last known GOK pointer location. */
static gint m_oldPointerX;
static gint m_oldPointerY;

/* keep track of number of keyboards - for debugging */
static gint m_NumberOfKeyboards = 0;

/* keep track of interesting roles */
static gint m_InterestingRole[NUM_INTERESTING_ROLES];

/* the XKB keyboard description structure and display */
static XkbDescPtr m_XkbDescPtr = NULL;
static Display   *m_XkbDisplay;
static int       *m_RowLastColumn;
static int       *m_RowTop;
static int       *m_SectionRowStart;
static int       *m_SectionCols;
static int       *m_SectionColStart;
static int        m_MinSectionTop = G_MAXINT;
static int        m_MinSectionLeft = G_MAXINT;

/**
* gok_keyboard_initialize
*
* Initializes global data for all keyboards.
* Call this once at the beginning of the program.
*
* returns: void
**/
void gok_keyboard_initialize ()
{
	m_WidthKeyFont = -1;
	m_HeightKeyFont = -1;
	m_bIgnoreResizeEvent = TRUE;
	m_InterestingRole[0] = SPI_ROLE_MENU;
	m_InterestingRole[1] = SPI_ROLE_MENU_ITEM;
	m_InterestingRole[2] = SPI_ROLE_PUSH_BUTTON;
	m_InterestingRole[3] = SPI_ROLE_RADIO_MENU_ITEM;
	m_InterestingRole[4] = SPI_ROLE_CHECK_MENU_ITEM;
}

/**
 * gok_keyboard_get_xkb_desc: 
 *
 * Returns: a pointer to the XkbDesc structure for the core keyboard.
 **/
XkbDescPtr 
gok_keyboard_get_xkb_desc (void)
{
	/* TODO: check the return values below */
	if (m_XkbDescPtr == NULL) {
		int ir, reason_return;
		char *display_name = getenv ("DISPLAY");
		m_XkbDisplay = XkbOpenDisplay (display_name,
					       &gok_xkb_base_event_type,
					       &ir, NULL, NULL, 
					       &reason_return);
		if (m_XkbDisplay == NULL)
		        g_warning ("Xkb extension could not be initialized! (error code %x)", reason_return);
		else 
			m_XkbDescPtr = XkbGetMap (m_XkbDisplay, 
						       XkbAllComponentsMask,
						       XkbUseCoreKbd);
		if (m_XkbDescPtr == NULL)
		        g_warning ("Xkb keyboard description not available!");
		else {
			XkbGetGeometry (m_XkbDisplay, m_XkbDescPtr);	
			XkbGetNames (m_XkbDisplay, XkbAllNamesMask, m_XkbDescPtr);
		}
	}
	return m_XkbDescPtr;
}

gboolean
gok_keyboard_xkb_select (Display *display)
{
	int opcode_rtn, error_rtn;
	gboolean retval;
	retval = XkbQueryExtension (display, &opcode_rtn, &gok_xkb_base_event_type, 
			   &error_rtn, NULL, NULL);
	if (retval) 
               retval = XkbSelectEvents (display, XkbUseCoreKbd, XkbStateNotifyMask | XkbAccessXNotifyMask, XkbStateNotifyMask | XkbAccessXNotifyMask);
	return retval;
}

void
gok_keyboard_notify_xkb_event (XkbEvent *event)
{
	if (event->any.xkb_type == XkbStateNotify) {
		XkbStateNotifyEvent *sevent = &event->state;
		if (sevent->changed & XkbGroupStateMask) {
			fprintf (stderr, "Group changed to %d\n",
				 sevent->group);
			gok_key_set_effective_group (sevent->group);
		}
	} 
	/* 
	 * TODO: change GOK's shift state notification 
	 * to use XKB instead of at-spi ?
	 */
}

static int
gok_keyboard_get_section_row (gint i)
{
#ifdef NEVER
	int row, i;
	for (i = 0; i < row; i++) {
		if (pXkbSection->rows[row].top <= m_RowTop[i]) {
			row = i;
			break;
		}
	}
	row = MIN (row, GOK_KEYBOARD_MAX_ROWS);
	if (m_RowTop[row] != G_MININT) {
		m_RowTop[row] = pXkbSection->rows[0].top;
	}
	return row;
#endif
	return m_SectionRowStart[i];
}

static gint
gok_keyboard_section_row_columns (XkbGeometryPtr pGeom, XkbRowPtr rowp)
{
	int i, ncols = 0;
	for (i = 0; i < rowp->num_keys; i++) {
		XkbBoundsRec *pBounds;
		pBounds = &pGeom->shapes[rowp->keys[i].shape_ndx].bounds;
		ncols += MAX (1, (pBounds->x2 - pBounds->x1)/
			      (pBounds->y2 - pBounds->y1));
	}
	return ncols;
}

static gint
gok_keyboard_get_section_column (gint i)
{
	int retval;
	int gok_row = gok_keyboard_get_section_row (i);
#ifdef NEVER
	if (pXkbSection->rows[0].vertical == True) {
		int lastRow = 0;
		int i;
		XkbRowPtr rowp = &pXkbSection->rows[0];
		fprintf (stderr, "vertical row\n");
		/* find the row with the largest LastColumn value */
		for (i = gok_row; i < gok_row + rowp->num_keys; i++)
			lastRow = MAX (lastRow, m_RowLastColumn[i]);
		/* set m_RowLastColumn to this value, incremented, for GOK */
		retval = lastRow;
		lastRow++;
		for (i = gok_row; i < gok_row + rowp->num_keys; i++) 
			m_RowLastColumn[i] = lastRow;
	}
	else {
		int lastRow = 0;
		int i;
		/* find the row with the largest LastColumn value */
		for (i = gok_row; i < gok_row + pXkbSection->num_rows; i++)
			lastRow = MAX (lastRow, m_RowLastColumn[i]);
		/* set m_RowLastColumn to this value, incremented, for GOK */
		retval = lastRow;
		for (i = 0; i < pXkbSection->num_rows; i++) {
			XkbRowPtr rowp = &pXkbSection->rows[i];
			fprintf (stderr, "setting RowLastCol[%d] from %d (lastRow %d) ",
				 gok_row + i, m_RowLastColumn[gok_row + i], lastRow);
			m_RowLastColumn[gok_row + i] = lastRow + 
				gok_keyboard_section_row_columns (pXkbDesc->geom, rowp);
			fprintf (stderr, "to %d\n", m_RowLastColumn[gok_row + i]);
		}
	}
	fprintf (stderr, "row %d, first column = %d; last = %d\n", gok_row, retval, 
		 m_RowLastColumn[gok_row] - 1);
#endif
	retval = m_SectionColStart[i];
	return retval;
}

static void
gok_keyboard_xkb_geom_sections_init (XkbSectionPtr sections, int n_sections)
{
        int i, j, num_rows = 0;
	float avg_row_height = 0;
	int min_col_width = G_MAXINT;

	m_RowLastColumn = g_malloc (sizeof (int) * GOK_KEYBOARD_MAX_ROWS);
	m_RowTop = g_malloc (sizeof (int) * GOK_KEYBOARD_MAX_ROWS);
	m_SectionRowStart = g_malloc (sizeof (int) * n_sections);
	m_SectionColStart = g_malloc (sizeof (int) * n_sections);
	m_SectionCols = g_malloc (sizeof (int) * n_sections);

	for (i = 0; i < GOK_KEYBOARD_MAX_ROWS; i++) {
		m_RowLastColumn[i] = 0;
		m_RowTop[i] = G_MININT;
	} 

	m_RowLastColumn[0] = 1;

	for (i = 0; i < n_sections; i++) {
		m_MinSectionTop = MIN (m_MinSectionTop, sections[i].top);
		m_MinSectionLeft = MIN (m_MinSectionLeft, sections[i].left);
		avg_row_height += sections[i].height;
		num_rows += sections[i].num_rows;
		m_SectionCols[i] = 0;
		for (j = 0; j < sections[i].num_rows; j++ ) {
			min_col_width = MIN ((sections[i].width)/
					     sections[i].rows[j].num_keys, 
					     min_col_width);
			m_SectionCols[i] = MAX (m_SectionCols[i], 
						floor (sections[i].rows[j].left/min_col_width)/
						sections[i].rows[j].num_keys);
		}
	}

	avg_row_height /= num_rows;

	for (i = 0; i < n_sections; i++) {
		m_SectionRowStart[i] = floor ((sections[i].top - m_MinSectionTop)/avg_row_height);
		m_SectionColStart[i] = floor ((sections[i].left - m_MinSectionLeft)/min_col_width);
	}

	/* now, do one more pass to remove gaps between sections */
	for (i = 0; i < n_sections; i++) {
		if ((m_SectionRowStart[i] == 0) && (m_SectionColStart[i] == 0)) {
			m_SectionColStart[i] = 1; /* leave room for 'Back' key */
			/* and move other sections aside, if need be */
			if (sections[i].rows[0].num_keys == m_SectionCols[i]) { 
				int n;
				for (n = 0; n < n_sections; n++) {
					int startRow = m_SectionRowStart[n];
					int endRow = startRow + sections[n].num_rows;
					if ((n != i) &&
					    ((startRow >= m_SectionRowStart[i]) &&
					     (startRow < m_SectionRowStart[i] 
					      + sections[i].num_rows)) ||
					    ((endRow >= m_SectionRowStart[i]) &&
					     (endRow < m_SectionRowStart[i]
					      + sections[i].num_rows))) {
						if (m_SectionRowStart[n] > m_SectionRowStart[i]) {
							++m_SectionRowStart[n];
						}
					}
				}
			}
		} 
	}
}

static void
gok_keyboard_add_keys_from_xkb_geom (GokKeyboard *pKeyboard, XkbDescPtr kbd)
{
	GokKey *pKey = pKeyboard->pKeyFirst;
	XkbGeometryPtr geom;
	int row, col, i, rightmost = 0, gok_row = 1, gok_col = 0;

	if (kbd) 
		geom = kbd->geom;
	else
		return;

	gok_modifier_add ("shift");
	gok_modifier_add ("capslock");
	gok_modifier_add ("ctrl");
	gok_modifier_add ("alt");
	gok_modifier_add ("mod2");
	gok_modifier_add ("mod3");
	gok_modifier_add ("mod4");
	gok_modifier_add ("mod5");
        gok_log_x ("core xkb keyboard has %d sections\n",
		 geom->num_sections);
	gok_keyboard_xkb_geom_sections_init (geom->sections, geom->num_sections);
	
	for (i = 0; i < geom->num_sections; i++) {
		XkbSectionPtr section = &geom->sections[i];
		int gok_section_first_col = 
			gok_keyboard_get_section_column (i);
		gok_row = gok_keyboard_get_section_row (i);
		for (row = 0; row < section->num_rows; row++, gok_row++) {
			XkbRowPtr rowp = &section->rows[row];
			gok_col = gok_section_first_col;
			for (col = 0; col < rowp->num_keys; col++) {
				pKey = gok_key_from_xkb_key (pKey, pKeyboard, 
							     gok_keyboard_get_display (),
							     kbd->geom,
							     rowp,
							     section,
							     &rowp->keys[col], 
							     i, gok_row, gok_col);
				if (rowp->vertical == False)
					gok_col = pKey->Right;
				else
					gok_row = pKey->Bottom;
				rightmost = ( gok_col > rightmost ) ? gok_col : rightmost;
			}
		}
	}

	/* add a branch key for special text edit functions */
	pKey = gok_key_new (pKey, NULL, pKeyboard);
	pKey->Style = KEYSTYLE_GENERALDYNAMIC;
	pKey->Type = KEYTYPE_BRANCHCOMPOSE;
	pKey->Target = "text-operations";
	pKey->Top = row + 1;
	pKey->Bottom = row + 2;
	pKey->Left = 0;
	pKey->Right = rightmost;
	gok_key_add_label (pKey, _("Text Manipulation"),NULL);
}

Display *
gok_keyboard_get_display ()
{
	return m_XkbDisplay;
}

/**
* gok_keyboard_get_core:
*
* Returns: a pointer to a #GokKeyboard representing the core system 
* keyboard device, with the same row/column geometry and key symbols.
**/
GokKeyboard *
gok_keyboard_get_core ()
{
	GokKeyboard *pKeyboard;
	GokKey *pKey;
	gok_log_enter();
	pKeyboard = gok_keyboard_new ();
	gok_keyboard_set_name (pKeyboard, CORE_KEYBOARD);
	pKeyboard->bRequiresLayout = FALSE;
	pKeyboard->bLaidOut = TRUE;
	pKeyboard->bSupportsWordCompletion = TRUE;
	pKeyboard->bSupportsCommandPrediction = FALSE;
	pKey = gok_key_new (NULL, NULL, pKeyboard);
	pKey->Type = KEYTYPE_BRANCHBACK;
	pKey->Style = KEYSTYLE_BRANCHBACK;
	pKey->FontSize = -1;
	pKey->FontSizeGroup = FONT_SIZE_GROUP_UNIQUE;
	pKey->Top = 0;
	pKey->Bottom = 1;
	pKey->Left = 0;
	pKey->Right = 1;
	gok_keylabel_new (pKey, _("Back"), NULL);
	gok_keyboard_add_keys_from_xkb_geom (pKeyboard, 
					     gok_keyboard_get_xkb_desc ());
	gok_keyboard_count_rows_columns (pKeyboard);
	gok_log ("created core keyboard with %d rows and %d columns\n",
		 gok_keyboard_get_number_rows (pKeyboard), 
		 gok_keyboard_get_number_columns (pKeyboard));
	gok_log_leave();
	return pKeyboard;
}

/**
* gok_keyboard_read
* @Filename: Name of the keyboard file.
*
* Reads in the given keyboard file. 
* Note: Call 'gok_keyboard_delete' on this keyboard when done with it.
*
* Returns: A pointer to the new keyboard, NULL if not created.
**/
GokKeyboard* gok_keyboard_read (gchar* Filename)
{
	GokKeyboard* pKeyboard;
	xmlDoc* pDoc;
	xmlNode* pNodeFirst;
	xmlNs* pNamespace;

	g_assert (Filename != NULL);

	/* read in the file and create a DOM */
	pDoc = xmlParseFile (Filename);
	if (pDoc == NULL)
	{
		gok_log_x ("Error: gok_keyboard_read failed - xmlParseFile failed. Filename: '%s'", Filename);
		return NULL;
	}

	/* check if the document is empty */
	pNodeFirst = xmlDocGetRootElement (pDoc);
    if (pNodeFirst == NULL)
	 {
		gok_log_x ("Error: gok_keyboard_read failed - first node empty. Filelname: %s", Filename);
		xmlFreeDoc (pDoc);
		return NULL;
	}

	/* check if the document has the correct namespace */
	pNamespace = xmlSearchNsByHref (pDoc, pNodeFirst, 
							(const xmlChar *) "http://www.gnome.org/GOK");
	if (pNamespace == NULL)
	{
		gok_log_x ("Error: Can't create new keyboard '%s'- does not have GOK Namespace.", Filename);
		xmlFreeDoc (pDoc);
		return NULL;
	}

	/* check if this is a "GokFile" */
	if (xmlStrcmp (pNodeFirst->name, (const xmlChar *) "GokFile") != 0)
	{
		gok_log_x ("Error: Can't create new keyboard '%s'- root node is not 'GokFile'.", Filename);
		xmlFreeDoc (pDoc);
		return NULL;
    }

	/* create a new keyboard structure */
    pKeyboard = gok_keyboard_new();
    if (pKeyboard == NULL)
	 {
		gok_log_x ("Error: Can't create new keyboard '%s'!", Filename);
		xmlFreeDoc (pDoc);
		return NULL;
    }

	/* add the keys to the keyboard */
	gok_keyboard_add_keys (pKeyboard, pDoc);

	/* free up the XML doc cause we're done with it */
	xmlFreeDoc (pDoc);

	/* count the number of rows & columns in the keyboard */
	gok_keyboard_count_rows_columns (pKeyboard);

	/* the keyboard is OK, return a pointer to it */
	gok_log("new keyboard created");
	return pKeyboard;
}

/**
* gok_keyboard_add_keys
* @pKeyboard: Pointer to the keyboard that will contain the keys.
* @pDoc: Pointer to the XML document that describes the keys.
*
* Adds the keys from the given DOM to this keyboard.
* The keys will all be deleted when gok_keyboard_delete is called.
*
* returns: TRUE if the keys were added, FALSE if not.
**/
gboolean gok_keyboard_add_keys (GokKeyboard* pKeyboard, xmlDoc* pDoc)
{
	xmlNode* pNodeRoot;
	xmlNode* pNodeKeyboard;
	xmlNode* pNodeKey;
	GokKey* pKeyPrevious;
	GokKey* pKeyNew;
	xmlChar* pStringAttributeValue;

	g_assert (pKeyboard != NULL);
	g_assert (pDoc != NULL);

	/* get the root node of the XML doc */
	pNodeRoot = xmlDocGetRootElement (pDoc);
	if (pNodeRoot == NULL)
	{
		gok_log_x ("Error: gok_keyboard_add_keys failed, pDoc is NULL!");
		return FALSE;
	}

	/* find the 'keyboard' node */
	pNodeKeyboard = gok_keyboard_find_node (pNodeRoot, "keyboard");
	if (pNodeKeyboard == NULL)
	{
		gok_log_x ("Error: gok_keyboard_add_keys failed, can't find 'keyboard' node!");
		return FALSE;	
	}

	/* get the name and type of the keyboard */
	pStringAttributeValue = xmlGetProp (pNodeKeyboard, (const xmlChar *) "name");
	if (pStringAttributeValue != NULL)
	{
		gok_keyboard_set_name (pKeyboard, (char *) pStringAttributeValue);
	}
	else
	{
		/* keyboard must have a name attribute*/
		gok_log_x ("Error: gok_keyboard_add_keys failed: can't find 'name' attribute for keyboard.");
		return FALSE;
	}

	/* get the word completion on/off flag */
	pStringAttributeValue = xmlGetProp (pNodeKeyboard, (const xmlChar *) "wordcompletion");
	if (pStringAttributeValue != NULL)
	{
		if (xmlStrcmp (pStringAttributeValue, (const xmlChar *) "yes") == 0)
		{
			pKeyboard->bSupportsWordCompletion = TRUE;
		}
		else
		{
			pKeyboard->bSupportsWordCompletion = FALSE;
		}
	}

	/* get the command prediction on/off flag */
	pStringAttributeValue = xmlGetProp (pNodeKeyboard, (const xmlChar *) "commandprediction");
	if (pStringAttributeValue != NULL)
	{
		if (xmlStrcmp (pStringAttributeValue, (const xmlChar *) "yes") == 0)
		{
			pKeyboard->bSupportsCommandPrediction = TRUE;
		}
		else
		{
			pKeyboard->bSupportsCommandPrediction = FALSE;
		}
	}

	/* get the type of the keyboard (this attribute is optional) */
	pStringAttributeValue = xmlGetProp (pNodeKeyboard, (const xmlChar *) "layouttype");
	if (pStringAttributeValue != NULL)
	{
		pKeyboard->LayoutType = atoi ((char *)pStringAttributeValue);
	}

	/* get the expansion policy of the keyboard (this attribute is optional) */
	pStringAttributeValue = xmlGetProp (pNodeKeyboard, (const xmlChar *) "expand");
	if (pStringAttributeValue != NULL)
	{
		if (xmlStrcmp (pStringAttributeValue, (const xmlChar *) "never") == 0)
		{
			pKeyboard->expand = GOK_EXPAND_NEVER;
		}
		else if (xmlStrcmp (pStringAttributeValue, (const xmlChar *) "always") == 0)
		{
			pKeyboard->expand = GOK_EXPAND_ALWAYS;
		}
	}

	/* go through all the keys */
	pKeyPrevious = NULL;
	pNodeKey = pNodeKeyboard->xmlChildrenNode;
	while (pNodeKey != NULL)
	{
		/* is this a "key" node ? */
		if (xmlStrcmp (pNodeKey->name, (const xmlChar *) "key") != 0)
		{
			/* not a "key" node, move on to the next node */
			pNodeKey = pNodeKey->next;
			continue;
		}
		
		/* create a new key */
		pKeyNew = gok_key_new (pKeyPrevious, NULL, pKeyboard);
		if (pKeyNew == NULL)
		{
			return FALSE;
		}
		pKeyPrevious = pKeyNew;

		/* initialize the key with data from the XML DOM */
		gok_key_initialize (pKeyNew, pNodeKey);

		/* get the next key */
		pNodeKey = pNodeKey->next;
	}

	return TRUE;
}

/**
* gok_keyboard_delete
* @pKeyboard: Pointer to the keyboard that's getting deleted.
* @bForce: TRUE if the keyboard should be deleted even if it is in the stack.
*
* Deletes the given keyboard. This must be called on every keyboard that has
* been created. Don't use the given keyboard after calling this.
**/
void gok_keyboard_delete (GokKeyboard* pKeyboard, gboolean bForce)
{
	GokKey* pKey;
	GokKey* pKeyTemp;

	gok_log_enter();
		
	/* handle NULL pointers */
	if (pKeyboard == NULL)
	{
		gok_log_leave();
		return;
	}
	
	if ((bForce == FALSE) && (gok_branchbackstack_contains(pKeyboard) == TRUE))
	{
		gok_log_x("keyboard is in the stack, you must force deletion!");
		gok_log_leave();
		return;
	}

	gok_log("deleting keyboard: %s",pKeyboard->Name);
	
	/* delete the AccessibleNode list */
	gok_spy_free(pKeyboard->pNodeAccessible);

	if (pKeyboard->pAccessible != NULL)
	{
		gok_spy_accessible_unref(pKeyboard->pAccessible);
	}
	
	
	/* delete all the keys on the keyboard */
	pKey = pKeyboard->pKeyFirst;
	while (pKey != NULL)
	{
		pKeyTemp = pKey;
		pKey = pKey->pKeyNext;
		gok_key_delete (pKeyTemp, NULL, FALSE);
	}

	/* delete any chunks on the keyboard */
	gok_chunker_delete_chunks (pKeyboard->pChunkFirst, TRUE);
	
	/* unhook it from the keyboard list */
	if (gok_main_get_first_keyboard() == pKeyboard)
	{
		if (pKeyboard->pKeyboardPrevious != NULL)
		{
			gok_main_set_first_keyboard (pKeyboard->pKeyboardPrevious);
		}
		else
		{
			gok_main_set_first_keyboard (pKeyboard->pKeyboardNext);
		}
	}
	
	if (pKeyboard->pKeyboardPrevious != NULL)
	{
		pKeyboard->pKeyboardPrevious->pKeyboardNext = pKeyboard->pKeyboardNext;
	}
	
	if (pKeyboard->pKeyboardNext != NULL)
	{
		pKeyboard->pKeyboardNext->pKeyboardPrevious = pKeyboard->pKeyboardPrevious;
	}

	m_NumberOfKeyboards--;

	g_free (pKeyboard);
	gok_log_leave();
}

/**
* gok_keyboard_new
*
* Allocates memory for a new keyboard and initializes the GokKeyboard structure.
* Call gok_keyboard_delete on this when done with it.
*
* returns: A pointer to the new keyboard, NULL if it can't be created.
**/
GokKeyboard* gok_keyboard_new ()
{
	GokKeyboard* pGokKeyboardNew;

	/* allocate memory for the new keyboard structure */
	pGokKeyboardNew = (GokKeyboard*) g_malloc(sizeof(GokKeyboard));
	
	/* initialize the data members of the structure */
	strcpy(pGokKeyboardNew->Name, "unknown");
	pGokKeyboardNew->LayoutType = KEYBOARD_LAYOUT_NORMAL;
	pGokKeyboardNew->Type = KEYBOARD_TYPE_PLAIN;
	pGokKeyboardNew->bDynamicallyCreated = FALSE;
	pGokKeyboardNew->NumberRows = 0;
	pGokKeyboardNew->NumberColumns = 0;
	pGokKeyboardNew->bRequiresLayout = TRUE;
	pGokKeyboardNew->bLaidOut = FALSE;
	pGokKeyboardNew->bFontCalculated = FALSE;
	pGokKeyboardNew->pKeyFirst = NULL;
	pGokKeyboardNew->pKeyboardNext = NULL;
	pGokKeyboardNew->pKeyboardPrevious = NULL;
	pGokKeyboardNew->bRequiresChunking = FALSE;
	pGokKeyboardNew->pChunkFirst = NULL;
	pGokKeyboardNew->bSupportsWordCompletion = FALSE;
	pGokKeyboardNew->bSupportsCommandPrediction = FALSE;
	pGokKeyboardNew->bWordCompletionKeysAdded = FALSE;
	pGokKeyboardNew->bCommandPredictionKeysAdded = FALSE;
	pGokKeyboardNew->expand = GOK_EXPAND_SOMETIMES;
	pGokKeyboardNew->pAccessible = NULL;	
	pGokKeyboardNew->pNodeAccessible = NULL;	
	pGokKeyboardNew->search_role = SPI_ROLE_MENU;
	
	m_NumberOfKeyboards++;
	
	return pGokKeyboardNew;
}

/**
* gok_keyboard_get_keyboards
* 
* Returns: The number of keyboards loaded.
**/
gint gok_keyboard_get_keyboards ()
{
	return m_NumberOfKeyboards;
}

/**
* gok_keyboard_get_wordcomplete_keys_added
* @pKeyboard: Pointer to the keyboard that we're testing.
*
* Returns: TRUE if the given keyboard has the word completion keys added, FALSE if not.
**/
gboolean gok_keyboard_get_wordcomplete_keys_added (GokKeyboard* pKeyboard)
{
	return pKeyboard->bWordCompletionKeysAdded;	
}

/**
* gok_keyboard_set_wordcomplete_keys_added
* @pKeyboard: Pointer to the keyboard that is changed.
* @bTrueFalse: TRUE if you want the predictor keys added, FALSE if not.
**/
void gok_keyboard_set_wordcomplete_keys_added (GokKeyboard* pKeyboard, gboolean bTrueFalse)
{
	pKeyboard->bWordCompletionKeysAdded = bTrueFalse;	
}

/**
* gok_keyboard_get_commandpredict_keys_added
* @pKeyboard: Pointer to the keyboard that we're testing.
*
* returns: TRUE if the given keyboard has the word completion keys added, FALSE if not.
**/
gboolean gok_keyboard_get_commandpredict_keys_added (GokKeyboard* pKeyboard)
{
	return pKeyboard->bCommandPredictionKeysAdded;	
}

/**
* gok_keyboard_set_commandpredict_keys_added
* @pKeyboard: Pointer to the keyboard that is changed.
* @bTrueFalse: TRUE if you want the prediction keys added, FALSE if not.
**/
void gok_keyboard_set_commandpredict_keys_added (GokKeyboard* pKeyboard, gboolean bTrueFalse)
{
	pKeyboard->bCommandPredictionKeysAdded = bTrueFalse;	
}

/**
* gok_keyboard_get_accessible
* @pKeyboard: Pointer to the keyboard that we're using
*
* Returns: pointer to the accessible (probably shared by keys on this keyboard)
**/
Accessible* gok_keyboard_get_accessible (GokKeyboard* pKeyboard)
{
	return pKeyboard->pAccessible;
}

/**
* gok_keyboard_set_accessible
* @pKeyboard: Pointer to the keyboard that is to be changed.
* @pAccessible: Pointer to the new accessible interface.
**/
void gok_keyboard_set_accessible (GokKeyboard* pKeyboard, Accessible* pAccessible)
{
	g_assert (pKeyboard != NULL);
	
	if (pKeyboard->pAccessible != NULL)
	{
		if (pKeyboard->pAccessible != pAccessible)
		{
			gok_spy_accessible_unref(pKeyboard->pAccessible);
			gok_spy_accessible_ref (pAccessible);
			gok_log("settting keyboard accessible with address: [%d]",pAccessible);
			pKeyboard->pAccessible = pAccessible;
		}
		else
		{
			/* do nothing */
			gok_log("tried to set keyboard accessible to the same value it already has");
		}
	}
	else
	{
		gok_log("settting keyboard accessible with address: [%d]",pAccessible);
		pKeyboard->pAccessible = pAccessible;
		gok_spy_accessible_ref(pAccessible);
	}	
}

/**
* gok_keyboard_get_supports_wordcomplete
* @pKeyboard: Pointer to the keyboard that we're testing.
*
* returns: TRUE if the given keyboard supports word completion.
* Only alphabetic keyboards should support word completion.
**/
gboolean gok_keyboard_get_supports_wordcomplete (GokKeyboard* pKeyboard)
{
	gok_log_enter();
	g_assert (pKeyboard != NULL);
	gok_log_leave();
	return pKeyboard->bSupportsWordCompletion;	
}

/**
* gok_keyboard_get_supports_commandprediction
* @pKeyboard: Pointer to the keyboard that we're testing.
*
* returns: TRUE if the given keyboard supports command prediction.
* Any keyboard can support command prediction..
**/
gboolean gok_keyboard_get_supports_commandprediction (GokKeyboard* pKeyboard)
{
	gok_log_enter();
	g_assert (pKeyboard != NULL);
	gok_log_leave();
	return pKeyboard->bSupportsCommandPrediction;	
}

/**
* gok_keyboard_count_rows_columns
* @pKeyboard: Pointer to the keyboard that we want to get the rows and columns for.
*
* Counts the number of rows and columns in the keyboard and updates members
* of the GokKeyboard structure.
**/
void gok_keyboard_count_rows_columns (GokKeyboard* pKeyboard)
{
	GokKey* pKey;
	gint rows;
	gint columns;

	g_assert (pKeyboard != NULL);

	/* look through all the keys and find the leftmost and bottommost cells */
	rows = 0;
	columns = 0;
	pKey = pKeyboard->pKeyFirst;
	while (pKey != NULL)
	{
		if (pKey->Right > columns)
		{
			columns = pKey->Right;
		}

		if (pKey->Bottom > rows)
		{
			rows = pKey->Bottom;
		}

		pKey = pKey->pKeyNext;
	}

	pKeyboard->NumberRows = rows;
	pKeyboard->NumberColumns = columns;
}

/**
* gok_keyboard_get_number_rows
* @pKeyboard: Pointer to the keyboard that you're concerned about.
*
* returns: The number of rows in the given keyboard.
**/
gint gok_keyboard_get_number_rows (GokKeyboard* pKeyboard)
{
	g_assert (pKeyboard != NULL);
	return pKeyboard->NumberRows;
}

/**
* gok_keyboard_get_number_columns
* @pKeyboard: Pointer to the keyboard you want to know about.
*
* returns: The number of columns in the given keyboard.
**/
gint gok_keyboard_get_number_columns (GokKeyboard* pKeyboard)
{
	g_assert (pKeyboard != NULL);
	return pKeyboard->NumberColumns;
}

/**
* gok_keyboard_find_node
* @pNode: Pointer to the XML node that may contain the node you're looking for.
* @NameNode: Name of the node you're looking for.
*
* returns: A pointer to the first node that has the given name, NULL if it can't be found.
* Note: This is recursive.
**/
xmlNode* gok_keyboard_find_node (xmlNode* pNode, gchar* NameNode)
{
	xmlNode* pNodeChild;
	xmlNode* pNodeReturned;

	g_assert (pNode != NULL);
	g_assert (NameNode != NULL);

	if (xmlStrcmp (pNode->name, (const xmlChar *)NameNode) == 0)
	{
		return pNode;
	}
	
	pNodeChild = pNode->xmlChildrenNode;
	while (pNodeChild != NULL)
	{
		pNodeReturned = gok_keyboard_find_node (pNodeChild, NameNode);
		if (pNodeReturned != NULL)
		{
			return pNodeReturned;
		}
		pNodeChild = pNodeChild->next;
	}

	return NULL;
}

/**
* gok_keyboard_set_name
* @pKeyboard: Pointer to the keyboard that's getting named.
* @Name: Name for the keyboard.
**/
void gok_keyboard_set_name (GokKeyboard* pKeyboard, char* Name)
{
	g_assert (pKeyboard != NULL);
	g_assert (Name != NULL);
	g_assert (strlen (Name) <= MAX_KEYBOARD_NAME);

	strcpy (pKeyboard->Name, Name);
}

/**
* gok_keyboard_get_name
* @pKeyboard: Pointer to the keyboard to get the name from.
*
* returns: gchar* name of keyboard
**/
gchar* gok_keyboard_get_name (GokKeyboard* pKeyboard)
{
	g_assert (pKeyboard != NULL);
	return pKeyboard->Name;
}

/**
* gok_keyboard_calculate_font_size
* @pKeyboard: Pointer to the keyboard that gets the new font size.
* 
* Sets the font size for each key on the given keyboard.
* Each key may be assigned to a a font size group (FSG). If the FSG is
* not specified then the key belongs to group FONT_SIZE_GROUP_UNDEFINED. 
* If the FSG is FONT_SIZE_GROUP_UNIQUE then the key does not belong to 
* any group and calculate a font size for that key.
*
**/
void gok_keyboard_calculate_font_size (GokKeyboard* pKeyboard)
{
	GokKeyboard* pKeyboardTemp;
	GokKey* pKey;
	gint sizeFont;

	/* any time the key size changes, recalculate the font size */
	if ((m_WidthKeyFont != gok_data_get_key_width()) ||
		(m_HeightKeyFont != gok_data_get_key_height()))
	{
		m_WidthKeyFont = gok_data_get_key_width();
		m_HeightKeyFont = gok_data_get_key_height();
		
		pKeyboardTemp = gok_main_get_first_keyboard();
		while (pKeyboardTemp != NULL)
		{
			pKeyboardTemp->bFontCalculated = FALSE;
			pKeyboardTemp = pKeyboardTemp->pKeyboardNext;
		}
	}
	
	/* check this flag before doing the work */
	if (pKeyboard->bFontCalculated == TRUE)
	{
		return;
	}
	
	/* clear the font size for each key */
	pKey = pKeyboard->pKeyFirst;
	while (pKey != NULL)
	{
		pKey->FontSize = -1;
		pKey = pKey->pKeyNext;
	}	
	
	/* go through all the keys on the keyboard, setting their font size */
	pKey = pKeyboard->pKeyFirst;
	while (pKey != NULL)
	{
		/* calculate the font size for each key that has 'unique font size' */
		if (pKey->FontSizeGroup == FONT_SIZE_GROUP_UNIQUE)
		{
			sizeFont = gok_key_calculate_font_size (pKey, TRUE, TRUE);
			gok_key_set_font_size (pKey, sizeFont);
		}
		else /* calculate the font size for the font group */
		{
			/* look at keys that haven't had their font size set yet */
			if (pKey->FontSize == -1)
			{
				gok_keyboard_calculate_font_size_group (pKeyboard, pKey->FontSizeGroup, FALSE);
			}
		}
		pKey = pKey->pKeyNext;
	}				

	pKeyboard->bFontCalculated = TRUE;
}

/**
* gok_keyboard_calculate_font_size_group
* @pKeyboard: Pointer to the keyboard that gets the new font size.
* @GroupNumber: Number of the font size group.
* @bOverride: If TRUE then the font size is set for the key even if it
* already has a font size set. If FALSE then the font size is not set for
* the key if it is already set.
* 
* Sets the font size for each key that belongs to the given group on the 
* given keyboard.
**/
void gok_keyboard_calculate_font_size_group (GokKeyboard* pKeyboard, gint GroupNumber, gboolean bOverride)
{
	GokKey* pKey;
	GokKey* pKeyWidest;
	GokKey* pKeyHighest;
	gint widthLabel;
	gint widthTemp;
	gint heightLabel;
	gint heightTemp;
	gint sizeWidestFont;
	gint sizeHighestFont;
	
	pKeyWidest = NULL;
	widthLabel = 0;
	pKeyHighest = NULL;
	heightLabel = 0;
	sizeWidestFont = 5000; /* minimum font size */
	sizeHighestFont = 5000; /* minimum font size */
	
	/* get font for key width */
	/* go through all the keys on the keyboard */
	pKey = pKeyboard->pKeyFirst;
	while (pKey != NULL)
	{
		/* look at keys that belong to the given group */
		if (pKey->FontSizeGroup == GroupNumber)
		{
			/* change font only if key hasn't had its font size set yet */
			/* or the bOverride flag is TRUE */
			if ((bOverride == TRUE) ||
				(pKey->FontSize == -1))
			{
				widthTemp = gok_key_get_label_lengthpercell (pKey);
				if (widthTemp > widthLabel)
				{
					widthLabel = widthTemp;
					pKeyWidest = pKey;
				}
				
				heightTemp = gok_key_get_label_heightpercell (pKey);
				if (heightTemp > heightLabel)
				{
					heightLabel = heightTemp;
					pKeyHighest = pKey;
				}
			}
		}
		pKey = pKey->pKeyNext;
	}				

	if (pKeyWidest != NULL)
	{
		/* calculate the font for the longest key */
		sizeWidestFont = gok_key_calculate_font_size (pKeyWidest, TRUE, FALSE);
	}
	
	if (pKeyHighest != NULL)
	{
		/* calculate the font for the highest key */
		sizeHighestFont = gok_key_calculate_font_size (pKeyHighest, FALSE, TRUE);
	}
	
	/* sizeWidestFont will be the final font size */
	if (sizeHighestFont < sizeWidestFont)
	{
		sizeWidestFont = sizeHighestFont;
	}
	
	/* set font for all keys that belong to the same group and */
	/* haven't had their font size set yet */
	pKey = pKeyboard->pKeyFirst;
	while (pKey != NULL)
	{
		if (pKey->FontSizeGroup == GroupNumber)
		{
			if ((bOverride == TRUE) ||
				(pKey->FontSize == -1))
			{
				gok_key_set_font_size (pKey, sizeWidestFont);
			}
		}
			
		pKey = pKey->pKeyNext;
	}
}

/**
* gok_keyboard_paint_pointer:
* @pKeyboard: Pointer to the keyboard where the pointer is painted.
* @pWindowMain: Pointer to the main window that holds the keyboards.
* @x : The x coordinate of the GOK pointer relative to the keyboard window.
* @y : The y coordinate of the GOK pointer relative to the keyboard window.
*
* Displays a GOK pointer at the specified location relative to the keyboard window.
*
**/
void
gok_keyboard_paint_pointer (GokKeyboard *pKeyboard, GtkWidget *pWindowMain, 
			    gint x, gint y)
{
	if (pWindowMain->window) {
		GdkGC *gc;
		GdkGCValues values;
		values.function = GDK_INVERT;
		values.line_width = 2;
		gc = gdk_gc_new_with_values (pWindowMain->window, &values, 
					     GDK_GC_FUNCTION | GDK_GC_LINE_WIDTH);	 
		m_oldPointerX = x;
		m_oldPointerY = y;
		gdk_draw_line (pWindowMain->window, gc, x-6, y, x+6, y);
		gdk_draw_line (pWindowMain->window, gc, x, y-6, x, y+6);
	}
}

/**
* gok_keyboard_unpaint_pointer:
* @pKeyboard: Pointer to the keyboard where the pointer is painted.
* @pWindowMain: Pointer to the main window that holds the keyboards.
*
* Hides the GOK pointer if it's currently in a GOK keyboard window.
*
**/
void
gok_keyboard_unpaint_pointer (GokKeyboard *pKeyboard, GtkWidget *pWindowMain)
{
	if (pWindowMain->window) {
		GdkRectangle rect;
		rect.x = m_oldPointerX - 6;
		rect.y = m_oldPointerY - 6;
		rect.width = 13;
		rect.height = 13;
		gdk_window_invalidate_rect (pWindowMain->window, 
					    &rect, True);
		gdk_window_process_updates (pWindowMain->window, True);

	}
}


/**
* gok_keyboard_display:
* @pKeyboard: Pointer to the keyboard that gets displayed.
* @pKeyboardCurrent: Pointer to the current keyboard.
* @pWindowMain: Pointer to the main window that holds the keyboards.
* @CallbackScanner: If TRUE then the keyboard is used by the GOK. If FALSE
* then the keyboard is used by the editor.
*
* Displays the given keyboard in the GOK window.
*
* returns: TRUE if the keyboard was displayed, FALSE if not.
**/
gboolean gok_keyboard_display (GokKeyboard* pKeyboard, GokKeyboard* pKeyboardCurrent, GtkWidget* pWindowMain, gboolean CallbackScanner)
{
	gchar titleWindow[MAX_KEYBOARD_NAME + 10];
	GtkWidget* pFixedContainer;
	GtkWidget* pNewButton;
	GokKey* pKey;
	gint heightWindow;
	gint widthWindow;
	gint frameX;
	gint frameY;
	gint winX;
	gint winY;
	gint widthMax;
	gint heightMax;
	gint widthKeyHold;
	gint heightKeyHold;
	gint widthKeyTemp;
	gint heightKeyTemp;
	GdkRectangle rectFrame;
	GdkRectangle rectTemp;
	GokButton* pGokButton;
	GtkButton* pButton;
	gboolean bKeySizeChanged = FALSE;
	gboolean expand = FALSE;

	g_assert (pKeyboard != NULL);
	g_assert (pWindowMain != NULL);

	/* hide any buttons from the previous keyboard */
	if (pKeyboardCurrent != NULL)
	{
		pKey = pKeyboardCurrent->pKeyFirst;
		while (pKey != NULL)
		{
			if (pKey->pButton != NULL)
			{
				gtk_widget_hide (pKey->pButton);
			}
			pKey = pKey->pKeyNext;
		}
	}

	/* change the name of the window to the keyboard name */
	strcpy (titleWindow, _("GOK - "));
	strcat (titleWindow, pKeyboard->Name);
	gtk_window_set_title (GTK_WINDOW(pWindowMain), titleWindow);
	
	/* get the "fixed container" that holds the buttons */
	pFixedContainer = GTK_BIN(pWindowMain)->child;
	
	/* create all the buttons and add them to the container */
	pKey = pKeyboard->pKeyFirst;
	while (pKey != NULL)
	{
		/* create a new GTK button for the key (if it's not already created) */
		if (pKey->pButton == NULL)
		{
			/* create a new GOK button */
			if (pKey->has_text)
			{
				pNewButton = gok_button_new_with_label (gok_key_get_label (pKey), IMAGE_PLACEMENT_LEFT); 
				if (pKey->has_image) {
					gok_button_set_image (GOK_BUTTON (pNewButton), 
							      GTK_IMAGE (gok_key_create_image_widget (pKey)));
					if (pKey->pImage->type != IMAGE_TYPE_INDICATOR) {
						GOK_BUTTON (pNewButton)->indicator_type = NULL;
					}
				}
			}
			else if (pKey->has_image)
			{
				pNewButton = gok_button_new_with_image (gok_key_create_image_widget (pKey), 
									pKey->pImage->placement_policy);
				if (pKey->pImage->type != IMAGE_TYPE_INDICATOR) {
					GOK_BUTTON (pNewButton)->indicator_type = NULL;
				}
			}
			else {
			        pNewButton = g_object_new (GOK_TYPE_BUTTON, NULL);
			}

			/* for modifier keys, set the indicator type */
			if (pKey->Type == KEYTYPE_MODIFIER) {
				GOK_BUTTON (pNewButton)->indicator_type = "shift";
			}

			/* associate the button with the key */
			pKey->pButton = pNewButton;
			gtk_object_set_data (GTK_OBJECT(pNewButton), "key", pKey);

			/* set the initial state of the button from pKey state*/
			gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (pNewButton),
						      pKey->ComponentState.active);

			/* add these signal handlers to the button */
			if (CallbackScanner == TRUE) /* button is used by GOK */
			{
				gtk_signal_connect (GTK_OBJECT (pNewButton), "button_press_event",
	                      GTK_SIGNAL_FUNC (on_window1_button_press_event),
	                      pKey);

				gtk_signal_connect (GTK_OBJECT (pNewButton), "button_release_event",
	                      GTK_SIGNAL_FUNC (on_window1_button_release_event),
	                      pKey);
	
				gtk_signal_connect (GTK_OBJECT (pNewButton), "toggled",
	                      GTK_SIGNAL_FUNC (on_window1_button_toggle_event),
	                      pKey);
	
				gtk_signal_connect (GTK_OBJECT (pNewButton), "enter_notify_event",
	                      GTK_SIGNAL_FUNC (gok_button_enter_notify),
	                      NULL);

				gtk_signal_connect (GTK_OBJECT (pNewButton), "leave_notify_event",
	                      GTK_SIGNAL_FUNC (gok_button_leave_notify),
	                      NULL);

				gtk_signal_connect (GTK_OBJECT (pNewButton), "state_changed",
						GTK_SIGNAL_FUNC (gok_button_state_changed),
						NULL);

			}
			else /* button is used by editor */
			{
				gtk_signal_connect (GTK_OBJECT (pNewButton), "button_press_event",
	                      GTK_SIGNAL_FUNC (on_editor_button_press_event),
	                      NULL);
			}

			/* set the 'name' of the button */
			/* the 'name' determines the .rc style to apply to the button */
			gok_key_set_button_name (pKey);
		}

		/* ensure all non-modifier buttons are in the 'inactive' state */
		gok_key_update_toggle_state (pKey);

		/* reset the highlight state of this key  - if we need to optimize this then only do it for branch keys... */	
		gok_feedback_unhighlight(pKey, FALSE);
		
		/* show the button */
		gtk_widget_show (pKey->pButton);

		/* get the next key in the list */
		pKey = pKey->pKeyNext;
	}	

	/* calculate the font size for this keyboard */
	gok_keyboard_calculate_font_size (pKeyboard);

	/* calculate the size of the window needed for the keys */
	widthWindow = (gok_keyboard_get_number_columns (pKeyboard) * gok_data_get_key_width()) + 
									((gok_keyboard_get_number_columns (pKeyboard) - 1) * gok_data_get_key_spacing());
	heightWindow = (gok_keyboard_get_number_rows (pKeyboard) * gok_data_get_key_height()) +
									((gok_keyboard_get_number_rows (pKeyboard) - 1) * gok_data_get_key_spacing());

	/* store the current key width and height */
	widthKeyHold = gok_data_get_key_width();
	heightKeyHold = gok_data_get_key_height();

	/* is this window bigger than the screen (or screen geometry)? */
	/* get the frame size */
	gdk_window_get_frame_extents ((GdkWindow*)pWindowMain->window, &rectFrame);
	gdk_window_get_position (pWindowMain->window, &winX, &winY);
	if ((winX != 0) &&
		(winY != 0))
	{
		frameX = (winX - rectFrame.x);
		frameY = (winY - rectFrame.y);
	}
	else
	{
		/* TODO: how can I get the frame size before the window is shown? */
		frameX = 5;/* this is usually true */
		frameY = 22;/* this is usually true */
	}
	
	if (gok_main_get_use_geometry() == TRUE)
	{
		gok_main_get_geometry (&rectTemp);
		widthMax = rectTemp.width;
		heightMax = rectTemp.height;
	}
	else if (pKeyboard->LayoutType == KEYBOARD_LAYOUT_FITWINDOW)
	{
		gdk_window_get_size ((GdkWindow*)pWindowMain->window, &widthMax,
				     &heightMax);
	}
	else
	{
		widthMax = gdk_screen_width();
		heightMax = gdk_screen_height();
	}

	expand = ((pKeyboard->expand == GOK_EXPAND_ALWAYS) || 
		  (gok_data_get_expand () && (pKeyboard->expand != GOK_EXPAND_NEVER)));

	if (((widthWindow + frameX) > widthMax) || 
	    (pKeyboard->LayoutType == KEYBOARD_LAYOUT_FITWINDOW) || expand)

	{
		/* change the key width (for this keyboard) to fit within the screen */
		widthKeyTemp = gok_keyboard_get_keywidth_for_window (widthMax - (frameX * 2), pKeyboard);
		gok_data_set_key_width (widthKeyTemp);
		
		/* calculate a new window size */
		widthWindow = (gok_keyboard_get_number_columns (pKeyboard) * gok_data_get_key_width()) + 
									((gok_keyboard_get_number_columns (pKeyboard) - 1) * gok_data_get_key_spacing());
		bKeySizeChanged = TRUE;
	}
	
	if (((heightWindow + frameY) > heightMax) ||
	    (pKeyboard->LayoutType == KEYBOARD_LAYOUT_FITWINDOW))
	{
		/* change the key height (for this keyboard) to fit within the screen */
		heightKeyTemp = gok_keyboard_get_keyheight_for_window (heightMax - frameY - frameX, pKeyboard);
		gok_data_set_key_height (heightKeyTemp);
		
		/* calculate a new window size */
		heightWindow = (gok_keyboard_get_number_rows (pKeyboard) * gok_data_get_key_height()) +
									((gok_keyboard_get_number_rows (pKeyboard) - 1) * gok_data_get_key_spacing());
		bKeySizeChanged = TRUE;
	}
	
	/* if window resizing forced key resize, resize fonts*/
	if (bKeySizeChanged) 
	       gok_keyboard_calculate_font_size (pKeyboard);

	/* resize the window to hold all the keys */
	gok_main_resize_window (pWindowMain, pKeyboard, widthWindow, heightWindow);

	/* position and resize all the buttons */
	gok_keyboard_position_keys (pKeyboard, pWindowMain);
	
	/* replace the key width and height */
	/* (because we may have changed it for this keyboard only */
	gok_data_set_key_width (widthKeyHold);
	gok_data_set_key_height (heightKeyHold);

	/* update the indicators on all modifier keys */
	gok_modifier_update_modifier_keys (pKeyboard);

	return TRUE;
}

/**
* gok_keyboard_position_keys
* @pKeyboard: Pointer to the keyboard that contains the keys.
* @pWindow: Pointer to the window that displays the keys.
*
* Positions the keys on the keyboard. The key cell coordinates are converted into
* window locations.
**/
void gok_keyboard_position_keys (GokKeyboard* pKeyboard, GtkWidget* pWindow)
{
	GokKey* pKey;
	GtkWidget* pContainer;
	gint left, top, width, height;
	gint widthKey;
	gint heightKey;
	gint spacingKey;

	g_assert (pKeyboard != NULL);
	g_assert (pWindow != NULL);

	/* get the key size */
	/* start with the key size from the gok_data */
	widthKey = gok_data_get_key_width();
	heightKey = gok_data_get_key_height();
	spacingKey = gok_data_get_key_spacing();
		
	/* get the container from the window */
	pContainer = GTK_BIN(pWindow)->child;
	g_assert (pContainer != NULL);

	/* loop through all the keys */
	pKey = pKeyboard->pKeyFirst;
	while (pKey != NULL)
	{
		/* change the size of the button */
		width = (pKey->Right - pKey->Left) * widthKey;
		width += (pKey->Right - pKey->Left - 1) * spacingKey;
		height = (pKey->Bottom - pKey->Top) * heightKey;
		height += (pKey->Bottom - pKey->Top - 1) * spacingKey;
		gtk_widget_set_size_request (pKey->pButton, width, height);

		/* position the button */
		left = pKey->Left * (widthKey + spacingKey);
		top = pKey->Top * (heightKey + spacingKey);
		
		/* if the button has been previously 'put' then 'move' it */
		if (gtk_widget_get_parent (pKey->pButton) == NULL)
		{
			gtk_fixed_put (GTK_FIXED(pContainer), pKey->pButton, left, top);
		}
		else
		{
			/* gtk_fixed_move generates a 'resize' event so set flag to ignore it */
			m_bIgnoreResizeEvent = TRUE;
			gtk_fixed_move (GTK_FIXED(pContainer), pKey->pButton, left, top);
		}

		/* store the new position of the key */
		pKey->TopWin = top;
		pKey->BottomWin = top + height;
		pKey->LeftWin = left;
		pKey->RightWin = left + width;

		pKey = pKey->pKeyNext;
	}
}

static gboolean
gok_keyboard_page_select (AccessibleNode *node) 
{
	AccessibleSelection *selection;
	Accessible *parent;
	int index;
	gboolean retval = FALSE;
       
	parent = Accessible_getParent (node->paccessible);
	if (parent) {
		index = Accessible_getIndexInParent (node->paccessible);
		selection = Accessible_getSelection (parent);
		g_assert (selection != NULL);
		retval = AccessibleSelection_selectChild (selection, index);
	}
	
	return retval;
}


/**
* gok_keyboard_branch_byKey
* @pKey: The key that is causes the branch.
*
* Branch to another keyboard specified by given key.
* The previous keyboard is stored on the "branch back stack".
*
* returns: TRUE if keyboard branched, FALSE if not.
**/
gboolean gok_keyboard_branch_byKey (GokKey* pKey)
{
	gboolean is_branched, is_active;
	AccessibleStateSet *pStateSet;

	gok_log("gok_keyboard_branch_byKey:");
	/* branch according to type */
	switch (pKey->Type)
	{
		case KEYTYPE_BRANCHBACK:
			gok_log("branch back");		
			return gok_main_display_scan_previous();/*_premade();*/
			break;

		case KEYTYPE_BRANCHMENUS:
			gok_log("branch back SPI_ROLE_MENU");		
			return gok_keyboard_branch_gui(pKey->pNodeAccessible, 
						       GOK_SPY_SEARCH_MENU,
						       SPI_ROLE_MENU); 
			break;

		case KEYTYPE_BRANCHMENUITEMS:
			gok_log("branch gui SPI_ROLE_MENU_ITEM");		
			return gok_keyboard_branch_gui(pKey->pNodeAccessible, 
						       GOK_SPY_SEARCH_MENU, 
						       /* does this work for check menu items too? */
						       SPI_ROLE_MENU_ITEM);
			break;
			
		case KEYTYPE_BRANCHTOOLBARS:
			gok_log("branch gui SPI_ROLE_PUSH_BUTTON");		
			return gok_keyboard_branch_gui(pKey->pNodeAccessible, 
						       GOK_SPY_SEARCH_ROLE,
						       SPI_ROLE_PUSH_BUTTON); 
			break;
		case KEYTYPE_BRANCHGUI:
			gok_log("branch gui SPI_ROLE_PUSH_BUTTON");		
			return gok_keyboard_branch_gui(pKey->pNodeAccessible, 
						       GOK_SPY_SEARCH_UI,
						       SPI_ROLE_PUSH_BUTTON); 
			break;			
		case KEYTYPE_PAGESELECTION:
			gok_log("page select");
			gok_keyboard_page_select (pKey->pNodeAccessible);
			return FALSE;
			break;
		case KEYTYPE_BRANCHGUIACTIONS:
			gok_log("branch gui_actions");
			is_branched = gok_keyboard_branch_gui_actions(pKey->pNodeAccessible);
			if (!is_branched) 
			{
				pStateSet = Accessible_getStateSet (pKey->pNodeAccessible->paccessible);
				is_active = AccessibleStateSet_contains (pStateSet, 
									 SPI_STATE_CHECKED);
				pKey->ComponentState.active = is_active;
				AccessibleStateSet_unref (pStateSet);
			}
			return is_branched;
			break;
	        case KEYTYPE_BRANCHHYPERTEXT:
			fprintf (stderr, "HYPERTEXT!\n");
                        break;
		default:
			/* should not be here */
			gok_log ("Unknown branch type");
			break;
	}
	return FALSE;
}


static KeyStyles 
gok_style_if_enabled (AccessibleStateSet *states, KeyStyles style) 
{
	if (AccessibleStateSet_contains (states, SPI_STATE_ENABLED))
		return style;
	else
		return KEYSTYLE_INSENSITIVE;
}

/**
* gok_keyboard_update_dynamic
* @pKeyboard: Pointer to the keyboard that gets updated.
*
* Creates all the keys for the given dynamic keyboard.
*
* returns: TRUE if the keyboard was updated, FALSE if not.
**/
gboolean gok_keyboard_update_dynamic (GokKeyboard* pKeyboard)
{
	AccessibleNode* pNodeAccessible;
	AccessibleNode* pNodeAccessibleNext;
	AccessibleStateSet *pStateSet;
	GokKey* pKey;
	GokKey* pKeyTemp;
	GokKey* pKeyPrevious;
	gint column;
	gboolean is_active;
	
	gok_log_enter();
	g_assert(pKeyboard != NULL);
	
	if (pKeyboard->bDynamicallyCreated == FALSE)
	{
		gok_log_x ("Warning: Keyboard is not dynamic!");
		gok_log_leave();
		return FALSE;
	}

	/* delete the AccessibleNode list if one exists */
	if (pKeyboard->pNodeAccessible != NULL)
	{
		gok_spy_free(pKeyboard->pNodeAccessible);
		pKeyboard->pNodeAccessible = NULL;
	}
	
	/* delete any keys that are currently on the keyboard */
	pKey = pKeyboard->pKeyFirst;
	while (pKey != NULL)
	{
		pKeyTemp = pKey;
		pKey = pKey->pKeyNext;
		gok_key_delete (pKeyTemp, NULL, TRUE);
	}
	pKeyboard->pKeyFirst = NULL;

	
#if REINSTATE_APPLICATIONS
	if (pKeyboard->search_role == GOKSPY_ROLE_APPLICATION)
	{
		pNodeAccessible = gok_spy_get_applications();
	}
	else
	{
#endif
	gok_log("calling get list with accessible [%d] and role [%s]", 
		pKeyboard->pAccessible, AccessibleRole_getName(pKeyboard->search_role));
	/* create new keys for the keyboard */


	/* TODO: REFACTOR: enum KeyType (keyboard role) */
	pNodeAccessible = 
		gok_spy_get_list (pKeyboard->pAccessible, pKeyboard->search_type, pKeyboard->search_role);
		
	if (pNodeAccessible == NULL)
	{
		gok_log_x ("Warning: pNodeAccessible is NULL!");
		gok_log_leave();
		return FALSE;
	}
	
	/* store the list head on the keyboard so that it can be deallocated later */
	pKeyboard->pNodeAccessible = pNodeAccessible;
	
	/* add the new keys to the menu keyboard */
	/* first, add a 'back' key */
	pKey = gok_key_new (NULL, NULL, pKeyboard);
	pKey->Type = KEYTYPE_BRANCHBACK;
	pKey->Style = KEYSTYLE_BRANCHBACK;
	pKey->Top = 0;
	pKey->Bottom = 1;
	pKey->Left = 0;
	pKey->Right = 1;
	gok_key_add_label (pKey, _("back"), NULL);

	pKeyPrevious = pKey;
		
	/* create all the gui keys as one long row */
	/* the keys will be repositioned in gok_keyboard_layout */
	pKeyboard->bLaidOut = FALSE;

	column = 1;
	pNodeAccessibleNext = pNodeAccessible;

	while (pNodeAccessibleNext != NULL)
	{
		pKey = gok_key_new (pKeyPrevious, NULL, pKeyboard);
		pKeyPrevious = pKey;
		
		pKey->Style = KEYSTYLE_GENERALDYNAMIC;
		
		gok_log("node has role: [%s]",Accessible_getRoleName(pNodeAccessibleNext->paccessible));
		
		pStateSet = Accessible_getStateSet (pNodeAccessibleNext->paccessible);
		switch (Accessible_getRole(pNodeAccessibleNext->paccessible))
		{
			/*
			  case SPI_ROLE_INVALID:
			  gok_log_x("invalid role in accessible node list!");
			  pKey->Type = KEYTYPE_NORMAL;
			  break;
			*/
		case SPI_ROLE_MENU:
			gok_log("setting key type BRANCHMENUITEMS");
			pKey->Type = KEYTYPE_BRANCHMENUITEMS;
			pKey->Style = gok_style_if_enabled (pStateSet, KEYSTYLE_BRANCHMENUITEMS);
			break;
			/* add cases here	
			   case SPI_ROLE_XXX:
			   pKey->Type = KEYTYPE_BRANCHXXX;
			   break;
			*/
		case SPI_ROLE_CHECK_BOX:
		case SPI_ROLE_CHECK_MENU_ITEM:
		case SPI_ROLE_TOGGLE_BUTTON:
			pKey->Type = KEYTYPE_BRANCHGUIACTIONS;
			pKey->Style = gok_style_if_enabled (pStateSet, KEYSTYLE_BRANCHGUIACTIONS);
			pKey->has_image = TRUE;
			pKey->pImage = gok_keyimage_new (pKey, NULL);
			pKey->pImage->type = IMAGE_TYPE_INDICATOR;
			is_active = AccessibleStateSet_contains (pStateSet, 
								 SPI_STATE_CHECKED);
			pKey->ComponentState.active = is_active;
			break;
		case SPI_ROLE_RADIO_BUTTON:
		case SPI_ROLE_RADIO_MENU_ITEM:
			pKey->Type = KEYTYPE_BRANCHGUIACTIONS;
			pKey->Style = gok_style_if_enabled (pStateSet, KEYSTYLE_BRANCHGUIACTIONS);
			pKey->has_image = TRUE;
			pKey->pImage = gok_keyimage_new (pKey, NULL);
			pKey->pImage->type = IMAGE_TYPE_INDICATOR;
			is_active = AccessibleStateSet_contains (pStateSet, 
								 SPI_STATE_CHECKED);
			pKey->ComponentState.active = is_active;
			pKey->ComponentState.radio = TRUE;
			break;
		case SPI_ROLE_PAGE_TAB:
			/* no action implemented,, must use selection API on parent */
			pKey->Type = KEYTYPE_PAGESELECTION;
			pKey->Style = gok_style_if_enabled (pStateSet, KEYSTYLE_PAGESELECTION);
			break;
		case SPI_ROLE_TEXT:
			/* should only be in the list if it's editable...*/
			if (pNodeAccessibleNext->is_link) {
			      pKey->Type = KEYTYPE_HYPERLINK;
			      pKey->Style = KEYSTYLE_HYPERLINK;
			      is_active = FALSE;
			}
			else {
			      pKey->Type = KEYTYPE_BRANCHTEXT;
			      if (AccessibleStateSet_contains (pStateSet, SPI_STATE_EDITABLE)) { 
				      pKey->Style = gok_style_if_enabled (pStateSet, KEYSTYLE_BRANCHTEXT);
				      is_active = TRUE;
			      }
			      else {
				      is_active = FALSE;
				      pKey->Style = KEYSTYLE_INSENSITIVE;
			      }
			}
			pKey->ComponentState.active = is_active;
			break;
		case SPI_ROLE_HTML_CONTAINER: /* TODO: check Hypertext interface instead */
			pKey->Type = KEYTYPE_BRANCHHYPERTEXT;
			pKey->Style = KEYSTYLE_BRANCH; /* reuse 'normal' branch style for now. */
			is_active = AccessibleStateSet_contains (pStateSet, 
								 SPI_STATE_SENSITIVE);
			pKey->ComponentState.active = is_active;
			break;
		default:
			if (Accessible_isHypertext (pNodeAccessibleNext->paccessible)) {
				pKey->Type = KEYTYPE_BRANCHHYPERTEXT;
				pKey->Style = KEYSTYLE_BRANCH; /* reuse 'normal' branch style for now. */
				is_active = TRUE; /* FIXME, might not be enabled? */
				pKey->ComponentState.active = is_active;
			}
			else {
				/* this might be a dangerous catch all... */
				gok_log("setting key type BRANCHGUIACTIONS");
				pKey->Type = KEYTYPE_BRANCHGUIACTIONS;
				pKey->Style = gok_style_if_enabled (pStateSet, KEYSTYLE_BRANCHGUIACTIONS);
			}
			break;
		}

		AccessibleStateSet_unref (pStateSet);
		pKey->Top = 0;
		pKey->Bottom = 1;
		pKey->Left = column;
		pKey->Right = column + 1;
		pKey->pNodeAccessible = pNodeAccessibleNext;
		
		gok_log("adding label %s to dynamic key",pNodeAccessibleNext->pname);
		gok_key_add_label (pKey, pNodeAccessibleNext->pname, NULL);
		
		column++;
		pNodeAccessibleNext = pNodeAccessibleNext->pnext;
	}	

	gok_log_leave();
	return TRUE;
}

/**
* gok_keyboard_branch_gui
* @pNodeAccessible: Pointer to the accessible node (parent object).
* @type: Type of dynamic keyboard to branch to (the role of the children).
* @role: role to match, if type is #GOK_SPY_SEARCH_ROLE.
*
* Displays the generic gui keyboard - currently used for widgets inside windowish things
*
* Returns: TRUE if the keyboard was displayed, FALSE if not.
**/
gboolean gok_keyboard_branch_gui (AccessibleNode* pNodeAccessible, 
				  GokSpySearchType type, AccessibleRole role)
{
	Accessible* pAccessibleRoot;
	GokKeyboard* pKeyboard;
	GokKeyboard* pKeyboardTemp;

	gok_log_enter();
	
	pAccessibleRoot = NULL;

	if (pNodeAccessible != NULL)
	{
		gok_log("using accessible from key as root accessible");
		pAccessibleRoot = pNodeAccessible->paccessible;
	}
	
	if (pAccessibleRoot == NULL)
	{
		gok_log("using accessible for the foregound application as root accessible");
		/* get the accessible interface for the foregound application */
		pAccessibleRoot = gok_main_get_foreground_window_accessible();
		
		if (pAccessibleRoot == NULL)
		{
			gok_log_x ("Warning: Can't create gui keyboard because foreground accessible is NULL!");
			gok_log_leave();
			return FALSE;
		}
	}

	/* create a new keyboard */
	pKeyboard = gok_keyboard_new();
	if (pKeyboard == NULL)
	{
		gok_log_leave();
		return FALSE;
	}
		
	/* mark this as a dynamically created keyboard */
	pKeyboard->bDynamicallyCreated = TRUE;
	
	/* store the accessible pointer on the keyboard */
	gok_keyboard_set_accessible(pKeyboard, pAccessibleRoot);

	/* add the new keyboard to the list of keyboards (at the end)*/
	pKeyboardTemp = gok_main_get_first_keyboard();
	g_assert (pKeyboardTemp != NULL);
	while (pKeyboardTemp->pKeyboardNext != NULL)
	{
		pKeyboardTemp = pKeyboardTemp->pKeyboardNext;
	}
	pKeyboardTemp->pKeyboardNext = pKeyboard;
	pKeyboard->pKeyboardPrevious = pKeyboardTemp;
	pKeyboard->search_type = type;
	pKeyboard->search_role = role;

	/* set the name and type of the keyboard */
	switch (type)
	{
	case GOK_SPY_SEARCH_UI:
		pKeyboard->Type = KEYBOARD_TYPE_GUI;
		gok_keyboard_set_name (pKeyboard, _("GUI"));
		break;
	case GOK_SPY_SEARCH_ROLE:
		pKeyboard->Type = KEYBOARD_TYPE_TOOLBAR; /* catch-all ok ? */
		/* TODO: localize the return value from AccessibleRole_getName */
		gok_keyboard_set_name (pKeyboard, AccessibleRole_getName (role));
		break;
	case GOK_SPY_SEARCH_APPLICATIONS:
		pKeyboard->Type = KEYBOARD_TYPE_APPLICATIONS;
		gok_keyboard_set_name (pKeyboard, _("Applications"));   
		break;
	case GOK_SPY_SEARCH_MENU:
	default:
		switch (role) {
		case SPI_ROLE_MENU:
			pKeyboard->Type = KEYBOARD_TYPE_MENUS;
			gok_keyboard_set_name (pKeyboard, _("Menus"));
			pKeyboard->search_type = GOK_SPY_SEARCH_MENU;
			break;
		case SPI_ROLE_MENU_ITEM:
		default:
			pKeyboard->Type = KEYBOARD_TYPE_MENUITEMS;
			gok_keyboard_set_name (pKeyboard, _("Menu"));   
			pKeyboard->search_type = GOK_SPY_SEARCH_MENU;
			break;
		}
		break;
	}
	
	/* set this flag so the keyboard will be laid out when it's displayed */
	pKeyboard->bLaidOut = FALSE;
	pKeyboard->bFontCalculated = FALSE;

	/* display and scan the dynamic keyboard */
	/* note: keys are added in gok_keyboard_update_dynamic which is */
	/* called by gok_main_display_scan */	
	gok_main_display_scan ( pKeyboard, pKeyboard->Name, 
		KEYBOARD_TYPE_UNSPECIFIED, KEYBOARD_LAYOUT_NORMAL );
	
	gok_log_leave();
	return TRUE;
}


/**
* gok_keyboard_branch_gui_actions
* @pNodeAccessible: the node thich represents the gui widget
*
* Widgets can have multiple actions - build a keyboard of them.
*
* returns: TRUE if the keyboard was displayed, FALSE if not.
**/
gboolean gok_keyboard_branch_gui_actions (AccessibleNode* pNodeAccessible)
{
	AccessibleAction* paaction;
	gint i = 0;
	gboolean branch = TRUE;

	paaction = NULL;
		
	gok_log_enter();

	g_assert(pNodeAccessible != NULL);
	g_assert(pNodeAccessible->paccessible != NULL);
	
	/* 
	 * just in case: before we do any actions check to see if this object has 
	 * interesting children and if so, branch to them instead.
	 * Note that this may not be the right thing to do in all cases,
	 * since it implies that we can't activate the parent's actions.
	 */
	gok_log("checking for children with interesting roles");
	for (i=0; i< NUM_INTERESTING_ROLES; i++)
	{
		if (gok_spy_has_child (pNodeAccessible->paccessible, 
				       GOK_SPY_SEARCH_ROLE, m_InterestingRole[i]) == TRUE)
		{
			gok_log_x("Abort action. Branching to child(ren)!");
			gok_log_leave();
			return gok_keyboard_branch_gui(pNodeAccessible, 
						       GOK_SPY_SEARCH_ROLE,
						       m_InterestingRole[i]);
		}
	}
	gok_log("no children with interesting roles");

	/* possible TODO: build a keyboard if more than one action available 
	(and do this before above branch to children!)*/
	paaction = Accessible_getAction(pNodeAccessible->paccessible);
	gok_spy_accessible_implicit_ref(paaction);
	if (paaction != NULL)
	{
		/* for now just perform the first action 
		*** warning repeated code block*/
		{
		Accessible_isAction(paaction);
		AccessibleAction_doAction(paaction, 0);
		gok_spy_accessible_unref(paaction);
		}

		/* no interesting children roles found so now branch back to most 
		recent *static* keyboard */
		branch = FALSE;
	}	
	
	gok_log_leave();
	return branch;	
}

gboolean gok_keyboard_branch_editableTextAction (GokKeyboard* pKeyboard, GokKey* pKey)
{
	/* FIXME: this is suspect */
	pKeyboard->pAccessible = gok_spy_get_accessibleWithText ();
	if (pKeyboard->pAccessible) {
		Accessible_ref (pKeyboard->pAccessible);
	}
	return gok_composer_branch_textAction (pKeyboard, pKey);
}
	
/**
* gok_keyboard_layout
* @pKeyboard: Pointer to the keyboard that is getting laid out.
*
* Arranges the keys on the keyboard.
* Predefined keyboards are already laid out. Runtime keyboards require this.
*
* returns: TRUE if the keyboard was laid out, FALSE if not.
**/
gboolean gok_keyboard_layout (GokKeyboard* pKeyboard)
{
	PangoLayout* pPangoLayout;
	PangoRectangle rectInk;
	PangoRectangle rectLogical;
	GokKey* pKey;
	GtkLabel* pLabel;
	gint maxTextPerCell;
	gint maxCellsRequired;
	gint totalKeys;
	gint totalCells;
	gint maxRowcolumn;
	gint row;
	gint column;
	gint countKeys;
	
	if (pKeyboard->bRequiresLayout == FALSE)
	{
		return TRUE;
	}
	
	if (pKeyboard->bLaidOut == TRUE)
	{
		return TRUE;
	}

	pLabel = (GtkLabel*)gtk_label_new ("");
	totalKeys = 0;
	totalCells = 0;
	maxCellsRequired = 1;
	
	/* calculate all the cells required for this keyboard */
	pKey = pKeyboard->pKeyFirst;
	if (pKey) {
		/* maximum size of text per cell */
		maxTextPerCell = gok_data_get_key_width() - 
			gok_key_get_default_border_width (pKey);
	}

	while (pKey != NULL)
	{
		/* create a label using the key text*/
		gtk_label_set_text (pLabel, gok_key_get_label (pKey));

		/* get the size of the text in the label */
		pPangoLayout = gtk_label_get_layout (pLabel);
		pango_layout_get_pixel_extents (pPangoLayout, &rectInk, &rectLogical);
		
		/* calculate the cells required for this label */
		pKey->CellsRequired = (rectInk.width / maxTextPerCell) + 1;

		totalCells += pKey->CellsRequired;
		if (pKey->CellsRequired > maxCellsRequired)
		{
			maxCellsRequired = pKey->CellsRequired;
		}
		
		totalKeys++;
		
		pKey = pKey->pKeyNext;
	}

	/* guess the proposed width & height of the keyboard */
	
	/* assume the keyboard will be square visually*/
/*
	maxRowcolumn = (int)sqrt (totalCells);
	if (totalCells - (maxRowcolumn * maxRowcolumn) > 0)
	{
		maxRowcolumn++;
	}
*/
	/* make sure the keyboard will hold the longest key */
/*
	if (maxCellsRequired > maxRowcolumn)
	{
		maxRowcolumn = maxCellsRequired;
	}
*/
	/* assign a row and column to each key */
/*
	row = 0;
	column = 0;
	pKey = pKeyboard->pKeyFirst;
	pKeyPrevious = pKey;
	while (pKey != NULL)
	{
		pKey->Top = row;
		pKey->Bottom = row + 1;
		
		pKey->Left = column;
		column += pKey->CellsRequired;
		pKey->Right = column;
				
		if (column > maxRowcolumn)
		{
			pKeyPrevious->Right = maxRowcolumn;
			row++;
			column = 0;
			pKey->Top = row;
			pKey->Bottom = row + 1;
			pKey->Left = column;
			column += pKey->CellsRequired;
			pKey->Right = column;
		}
		else if (column == maxRowcolumn)
		{
			row++;
			column = 0;
		}
		
		pKeyPrevious = pKey;
		pKey = pKey->pKeyNext;
	}
*/

	/* make the last key fill the last row */
/*
	pKeyPrevious->Right = maxRowcolumn;
*/

	/* assume the keyboard will be square in terms of number of keys*/
	maxRowcolumn = (gint)sqrt (totalKeys);
	if (totalKeys - (maxRowcolumn * maxRowcolumn) > 0)
	{
		maxRowcolumn++;
	}
	
	/* assign a row and column to each key */
	row = 0;
	column = 0;
	countKeys = 0;
	pKey = pKeyboard->pKeyFirst;
	while (pKey != NULL)
	{
		pKey->Top = row;
		pKey->Bottom = row + 1;
		
		pKey->Left = column;
		column += pKey->CellsRequired;
		pKey->Right = column;

		countKeys++;
		if (countKeys >= maxRowcolumn)
		{
			countKeys = 0;
			row++;
			column = 0;
		}
		
		pKey = pKey->pKeyNext;
	}

	/* set the number of rows and columns on the keyboard */
	gok_keyboard_count_rows_columns (pKeyboard);
	
	/* fill any empty space at the end of the rows */
	for (row = 0; row < pKeyboard->NumberRows; row++)
	{
		gok_keyboard_fill_row (pKeyboard, row);
	}
	
	/* keyboard is now laid out */
	pKeyboard->bLaidOut = TRUE;
	
/* causes hang - but where is this freed? */	
/*	free (pLabel);*/
		
	return TRUE;
}

/* this array is used to order the keys */
static GokKey* arrayGokKeyPointers[300]; /* largest number of keys in a row */
/**
* gok_keyboard_fill_row
* @pKeyboard: Pointer to the keyboard that contains the row.
* @RowNumber: Number of the row you want filled.
*
* This function resizes the keys in the given row so they fill the entire row.
* This should be used only on keyboards that are synamically created (not predefined).
**/
void gok_keyboard_fill_row (GokKeyboard* pKeyboard, gint RowNumber)
{
	GokKey* pKey;
	gint countColumns;
	gint countKeys;
	gint extraColumns;
	gint perKeyExtraColumns;
	gint x;
	
	g_assert (pKeyboard != NULL);
	g_assert (RowNumber >= 0);
	g_assert (RowNumber < 100);
	
	/* count the number of columns and keys in the target row */
	countColumns = 0;
	countKeys = 0;
	pKey = pKeyboard->pKeyFirst;
	while (pKey != NULL)
	{
		if (pKey->Top == RowNumber)
		{
			countColumns += (pKey->Right - pKey->Left);
			countKeys ++;
		}
		pKey = pKey->pKeyNext;
	}
	
	if (countColumns >= pKeyboard->NumberColumns)
	{
		return;
	}
	
	/* calculate the number of columms that get added to each key */
	extraColumns = pKeyboard->NumberColumns - countColumns;
	perKeyExtraColumns = extraColumns / countKeys;
	if (perKeyExtraColumns < 1)
	{
		perKeyExtraColumns = 1;
	}
	
	/* make a list of the keys in order of right to left */
	arrayGokKeyPointers[0] = NULL;
	pKey = pKeyboard->pKeyFirst;
	while (pKey != NULL)
	{
		if (pKey->Top == RowNumber)
		{
			gok_keyboard_insert_array (pKey);
		}
		pKey = pKey->pKeyNext;
	}

	/* add extra columns to keys until there are no more extra columns */
	for (x = 0; x < 300; x++)
	{
		if (arrayGokKeyPointers[x] == NULL)
		{
			break;
		}
		
		arrayGokKeyPointers[x]->Right += extraColumns;
		extraColumns -= perKeyExtraColumns;
		if (arrayGokKeyPointers[x + 1] == NULL)
		{
			arrayGokKeyPointers[x]->Left = 0;
		}
		else
		{			
			arrayGokKeyPointers[x]->Left += extraColumns;
		}
		
		if (extraColumns <= 0)
		{
			break;
		}
	}
	
	if (arrayGokKeyPointers[0] != NULL)
	{
		arrayGokKeyPointers[0]->Right = pKeyboard->NumberColumns;
	}
}

/**
* gok_keyboard_insert_array
* @pKey: Pointer to the key you want added to the array.
*
* Adds the given key to our array in order of the rightmost key location.
**/
void gok_keyboard_insert_array (GokKey* pKey)
{
	GokKey* pKeyTemp;
	gint x;
	
	for (x = 0; x < 300; x++)
	{
		if (arrayGokKeyPointers[x] == NULL)
		{
			arrayGokKeyPointers[x] = pKey;
			arrayGokKeyPointers[x + 1] = NULL;
			return;
		}
		
		/* if you want the array in left-to-right order then change '<' to '>' here */
		if (arrayGokKeyPointers[x]->Left < pKey->Left)
		{
			pKeyTemp = arrayGokKeyPointers[x];
			arrayGokKeyPointers[x] = pKey;
			pKey = pKeyTemp;
		}
	}
}

static gboolean
gok_keyboard_focus_object (Accessible *accessible)
{
	gboolean retval = FALSE;
	AccessibleComponent *component;
	if (accessible) {
		component = Accessible_getComponent (accessible);
		if (component) {
			retval = AccessibleComponent_grabFocus (component);
			AccessibleComponent_unref (component);
		}
	}
	return retval;
}

/* FIXME: this method is a hack necessitated by the fact that link objects aren't
 * actionable" as they should be, i.e. they don't implement AccessibleAction. 
 * When that's fixed, we should be able to do away with this kludge
 * (which manually focussed the text object, places the cursor, and synthesizes
 * a spacebar key!)
 */
static gboolean
gok_keyboard_follow_link (GokKey *key) 
{
	AccessibleHypertext *hypertext = NULL; 
	AccessibleText *text = NULL;
	AccessibleNode *node = key->pNodeAccessible;
	gboolean is_editable = FALSE;
	if (node && node->paccessible) {
		hypertext = Accessible_getHypertext (node->paccessible);
		text = Accessible_getText (node->paccessible);
		is_editable = Accessible_isEditableText (node->paccessible);
		/* can't activate editable text this way, it will insert a space instead! */
		if (hypertext && text && node->is_link && !is_editable) {
			AccessibleHyperlink *link = AccessibleHypertext_getLink (hypertext, node->link);
			Accessible *target;
			AccessibleAction *action;
			gboolean is_action = FALSE;
			gboolean is_focussed = FALSE;
			long int start, end;
			AccessibleHyperlink_getIndexRange (link, &start, &end);
			is_focussed = gok_keyboard_focus_object (node->paccessible);
			if (!is_focussed) gok_log_x ("Could not focus object\n");
			else {
				AccessibleText_setCaretOffset (text, start);
				SPI_generateKeyboardEvent ((long) XK_space, NULL, SPI_KEY_SYM);
			}
			target = AccessibleHyperlink_getObject (link, 0);
			if (target) {
				action = Accessible_getAction (target);
				if (!action) { /* try the parent object? */
					action = Accessible_getAction (link);
					if (action) gok_log_x ("activating the link object.\n");
				}
				if (!action) { /* try the parent object? */
					action = Accessible_getAction (node->paccessible);
					if (action) gok_log_x ("activating the parent hypertext object!\n");
				}
				if (action) {
					AccessibleAction_doAction (action, 0); /* HACK, should check the action first */
					gok_log ("activating link\n");
					AccessibleAction_unref (action);
					is_action = TRUE;
				}
				Accessible_unref (target);
			}
			AccessibleText_unref (text);
			AccessibleHyperlink_unref (link);
			AccessibleHypertext_unref (text);
			return is_action;
		}
	}
	return FALSE;
}


/* TODO: merge with duplicate code in gok-settings-dialog.c */

/**
 * gok_keyboard_help:
 *
 * @pKey: pointer to the invoking GokKey structure. 
 *
 * Displays the GOK Help text in the gnome-help browser.
 *
 **/
void
gok_keyboard_help (GokKey *pKey) 
{
	GError *error = NULL;
	/* TODO: detect error launching help, and give informative message */
	gnome_help_display ("gok.xml", NULL, &error);
}

/**
 * gok_keyboard_about:
 *
 * @pKey: pointer to the invoking GokKey structure. 
 *
 * Displays the GOK About window.
 *
 **/
void
gok_keyboard_about (GokKey *pKey) 
{
	static GtkWidget *about = NULL;
	GdkPixbuf	 *pixbuf = NULL;
	GError		 *error = NULL;
	gchar		 *file = NULL;
        
        static const gchar *authors [] = {
		"Simon Bates  <simon.bates@utoronto.ca>",
		"David Bolter <david.bolter@utoronto.ca>",
		"Bill Haneman <bill.haneman@sun.com>",
		"Chris Ridpath <chris.ridpath@utoronto.ca>",
		NULL
	};

	const gchar *documenters[] = {
		NULL
	};

	const gchar *translator_credits = _("translator_credits");

	if (about) {
		gtk_window_set_screen (
			GTK_WINDOW (about),
			gtk_widget_get_screen (GTK_WIDGET (gok_main_get_main_window ())));
		gtk_window_present (GTK_WINDOW (about));
		return;
	}
        
	file = gnome_program_locate_file (NULL, GNOME_FILE_DOMAIN_APP_DATADIR, 
					  "gok/goklogo.png", 
					  FALSE, NULL);
	if (file) {
		pixbuf = gdk_pixbuf_new_from_file (file, &error);
		g_free (file);
	}

	if (error) {
		g_warning (G_STRLOC ": cannot open %s: %s", file, error->message);
		g_error_free (error);
	}
	
        about = gnome_about_new (
		_("GOK"), VERSION,
		_("Copyright (C) 2001-2003 Sun Microsystems, Copyright (C) 2001-2003 University of Toronto"),
		_("Dynamic virtual keyboards for the GNOME desktop"),
		authors,
		documenters,
		strcmp (translator_credits, "translator_credits") != 0 ? translator_credits : NULL,
		pixbuf);
		
	if (pixbuf)
		gdk_pixbuf_unref (pixbuf);
			
	gtk_window_set_wmclass (GTK_WINDOW (about), "gok", "GOK");
	gtk_window_set_screen (GTK_WINDOW (about),
			       gtk_widget_get_screen (GTK_WIDGET (gok_main_get_main_window ())));
	g_signal_connect (about, "destroy",
			  G_CALLBACK (gtk_widget_destroyed),
			  &about);
        gtk_widget_show (about);
}

/**
 * gok_keyboard_dock:
 *
 * @pKey: pointer to the GokKey structure whose activation
 *        should relocate the GokKeyboard's docking position 
 *        onscreen.
 *
 * Docks the GOK keyboard to the top or bottom of the screen,
 *        or "floats" the keyboard if "none" is specified as the
 *        docking direction.
 *
 **/
void
gok_keyboard_dock (GokKey *pKey) 
{
	GokKeyboardDirection dir = GOK_DIRECTION_NONE;
	GokDockType old_type = gok_data_get_dock_type ();
	gint top = 0, bottom = 0;

	if (pKey && pKey->pGeneral) 
		dir = *(GokKeyboardDirection *) pKey->pGeneral;
	switch (dir)
	{
	case GOK_DIRECTION_N:
		gok_data_set_dock_type (GOK_DOCK_TOP);
		gok_main_set_wm_dock (TRUE);
		break;
	case GOK_DIRECTION_S:
		gok_data_set_dock_type (GOK_DOCK_BOTTOM);
		gok_main_set_wm_dock (TRUE);
		break;
	default:
		gok_main_update_struts (0, 0);
		gok_data_set_dock_type (GOK_DOCK_NONE);
		gok_main_set_wm_dock (FALSE);
	        break;
	}
	if (dir == GOK_DIRECTION_NONE) {
		GtkWidget *widget = gok_main_get_main_window ();
		if (widget) 
			gdk_window_raise (widget->window); 
/* FIXME: why do we need this? */ 
		gtk_widget_show_now (gok_main_get_main_window ());
	}
}

/**
 * gok_keyboard_move_resize:
 *
 * @pKey: pointer to the GokKey structure whose activation
 *        should relocate the GokKeyboard's default position 
 *        onscreen.
 *
 * Moves the GOK keyboard some distance, or resizes its keys, 
 * according to the value of the 'direction' data in 'pKey->pGeneral'.
 *
 **/
void
gok_keyboard_move_resize (GokKey *pKey) 
{
	GokKeyboardDirection dir = GOK_DIRECTION_NONE;
	GtkWidget *window = gok_main_get_main_window ();
	gboolean resize = FALSE;
	gint x, y;

	gok_log_enter();
	
	if (pKey && pKey->pGeneral) 
		dir = *(GokKeyboardDirection *) pKey->pGeneral;
	gtk_window_get_position (GTK_WINDOW (window), &x, &y);
	switch (dir)
	{
	case GOK_DIRECTION_NE:
		y -= 10;
	case GOK_DIRECTION_E:
		x += 10;
		break;
	case GOK_DIRECTION_NW:
		x -= 10;
	case GOK_DIRECTION_N:
		y -= 10;
		break;
	case GOK_DIRECTION_SW:
		y += 10;
	case GOK_DIRECTION_W:
		x -= 10;
		break;
	case GOK_DIRECTION_SE:
		x += 10;
	case GOK_DIRECTION_S:
		y += 10;
		break;
	case GOK_DIRECTION_FILL_EW:
		/* toggle expand */
		gok_data_set_expand (!gok_data_get_expand ());
		resize = TRUE;
		break;
	case GOK_RESIZE_NARROWER:
		x = gok_data_get_key_width ();
		x -= 2;
		gok_data_set_key_width (MAX (0, x));
		resize = TRUE;
		break;
	case GOK_RESIZE_WIDER:
		gok_data_set_key_width (gok_data_get_key_width () + 2); 
		resize = TRUE;
		break;
	case GOK_RESIZE_SHORTER:
		y = gok_data_get_key_height ();
		y -= 2;
		gok_data_set_key_height (MAX (0, y));
		resize = TRUE;
		break;
	case GOK_RESIZE_TALLER:
		gok_data_set_key_height (gok_data_get_key_height () + 2);
		resize = TRUE;
		break;
	default:
	        break;
	}
	if (resize) {
		gok_keyboard_display (gok_main_get_current_keyboard (),
				      gok_main_get_current_keyboard (), 
				      gok_main_get_main_window (), TRUE);
	}
	else {
		gtk_window_move (GTK_WINDOW (window), x, y);
	}
	gok_log_leave();
}

/**
* gok_keyboard_output_selectedkey
*
* Performs the events associated with the currently selected key
*
* returns: Always 0.
**/
GokKey* gok_keyboard_output_selectedkey (void)
{
	GokKey* pKeySelected = NULL;
	
	gok_log_enter();

	/* get the key selected */
	pKeySelected = gok_feedback_get_selected_key();
	
	/* is a key selected? */
	if (pKeySelected == NULL)
	{
		gok_log_x ("Currently selected key is NULL!");
		gok_log_leave();
		return pKeySelected;
	}
	
	gok_log_leave();
	return gok_keyboard_output_key(pKeySelected);
}

/**
* gok_keyboard_output_key
* @pKeySelected: Pointer to the key that will be output.
*
* Synthesize the keyboard output.
* Possible side effects: makes sound, word completion prediction update.
**/
GokKey* gok_keyboard_output_key(GokKey* pKey)
{
	GokOutput OutputSpace;
	GokKeyboard* pPredictedKeyboard;
	GokKeyboard* pKeyboardTemp;
	GokKey* pKeyTemp;
	
	gok_log_enter();

	g_assert (pKey != NULL);

	/* ignore disabled keys */
	if (strcmp (gtk_widget_get_name (pKey->pButton), "StyleButtonDisabled") == 0)
	{
		gok_log_leave();
		return NULL;
	}

	switch (pKey->Type)
	{
		case KEYTYPE_BRANCHCOMPOSE:
			if (! gok_spy_get_accessibleWithText ()) {
				gok_log_x ("object isn't a text object, can't branch to compose");
				gok_log_leave ();
				return NULL;
			}
			gok_composer_validate (pKey->Target, gok_spy_get_accessibleWithText ());
			/* else fall-through */
		case KEYTYPE_BRANCH:
			gok_main_display_scan ( NULL, pKey->Target, 
				KEYBOARD_TYPE_UNSPECIFIED, KEYBOARD_LAYOUT_NORMAL);
			break;

		case KEYTYPE_BRANCHMODAL:
			gok_main_display_scan ( NULL, pKey->Target, 
				KEYBOARD_TYPE_MODAL, KEYBOARD_LAYOUT_FITWINDOW);
			break;

		/* now handled in default case KEYTYPE_MODIFIER:
			gok_modifier_press (pKey, pKey->ModifierName);
			break;*/
		
		case KEYTYPE_REPEATNEXT:
			gok_repeat_arm();
			pKey = NULL; /* useful hack */
			break;
				
		case KEYTYPE_BRANCHWINDOWS:
			gok_windowlister_show();
			break;
			
		case KEYTYPE_WINDOW:
			gok_windowlister_onKey(pKey);
			break;

		case KEYTYPE_MOUSE:
		case KEYTYPE_MOUSEBUTTON:
			gok_mouse_control (pKey);
			break;

		case KEYTYPE_BRANCHBACK:
gok_log_x("BRANCH BACK %s", pKey->pLabel->Text);			
		case KEYTYPE_BRANCHMENUS:
		case KEYTYPE_BRANCHTOOLBARS:
		case KEYTYPE_BRANCHGUI:
		case KEYTYPE_BRANCHHYPERTEXT:
			gok_keyboard_branch_byKey (pKey);
			break;

		case KEYTYPE_BRANCHALPHABET:
			gok_main_display_scan ( NULL, CORE_KEYBOARD, 
				KEYBOARD_TYPE_UNSPECIFIED, KEYBOARD_LAYOUT_NORMAL);
			break;

		case KEYTYPE_BRANCHTEXT:
			if (pKey->pNodeAccessible && 
				pKey->pNodeAccessible->paccessible) {
					gok_keyboard_focus_object 
						(pKey->pNodeAccessible->paccessible);
				}
			gok_main_display_scan ( NULL, "Keyboard", 
					KEYBOARD_TYPE_UNSPECIFIED, KEYBOARD_LAYOUT_NORMAL);
			break;

		case KEYTYPE_SETTINGS:
			gok_settingsdialog_show();
			break;
				
		case KEYTYPE_COMMANDPREDICT:
			return gok_keyboard_output_key (pKey->pMimicKey);
			break;
			
		case KEYTYPE_POINTERCONTROL:	
			gok_data_set_drive_corepointer (
			!gok_data_get_drive_corepointer ());
			gok_main_set_cursor (NULL);
			break;

		case KEYTYPE_MOVERESIZE:	
			gok_keyboard_move_resize (pKey);
			break;

		case KEYTYPE_DOCK:	
			gok_keyboard_dock (pKey);
			break;

		case KEYTYPE_HELP:	
			gok_keyboard_help (pKey);
			break;

		case KEYTYPE_ABOUT:	
			gok_keyboard_about (pKey);
			break;

		case KEYTYPE_HYPERLINK:
			if (gok_keyboard_follow_link (pKey))
			gok_main_display_scan_reset ();
			break;

		case KEYTYPE_WORDCOMPLETE:
			/* output any modifier keys */
			gok_modifier_output_pre();
			
			/* send the key output to the system */
			gok_output_send_to_system (gok_wordcomplete_get_output (pKey), 
				FALSE);
			
			/* end the word with a 'space' */
			OutputSpace.Type = OUTPUT_KEYSYM;
			OutputSpace.Flag = SPI_KEY_PRESSRELEASE;
			OutputSpace.Name = "space";
			OutputSpace.pOutputNext = NULL;
			gok_output_send_to_system (&OutputSpace, FALSE);
			
			/* output any modifier keys */
			gok_modifier_output_post();
			
			/* turn off any modifier keys that are not locked on */
			gok_modifier_all_off();
			gok_wordcomplete_increment_word_frequency (gok_key_get_label (pKey));
			
			/* reset the word completor */
			gok_wordcomplete_clear_keys();
			gok_wordcomplete_end_word();
			break;
			
		case KEYTYPE_BRANCHMENUITEMS:
		case KEYTYPE_MENUITEM:
		case KEYTYPE_PAGESELECTION:
		case KEYTYPE_BRANCHGUIACTIONS:
			if (!gok_keyboard_branch_byKey (pKey))
				gok_main_display_scan_reset ();
			break;
			
	        case KEYTYPE_TEXTNAV:
	        case KEYTYPE_EDIT:
	        case KEYTYPE_SELECT:
	        case KEYTYPE_TOGGLESELECT:
			gok_keyboard_branch_editableTextAction 
				(gok_main_get_current_keyboard(), pKey);
			break;
			
		default:/* a regular key */
			/* output any modifier keys */
			gok_modifier_output_pre();
		
			/* TODO: add check to see if command prediciton turned on ?*/
	#ifdef WHENCOMMANDPREDICTIONINGOKDATAISHOOKEDUPTOPREFS
			{			
				pPredictedKeyboard = NULL;
				pKeyboardTemp = gok_main_get_first_keyboard();
				g_assert (pKeyboardTemp != NULL);
				while (pKeyboardTemp != NULL)
				{
					pKeyTemp = pKeyboardTemp->pKeyFirst;
					while (pKeyTemp != NULL)
					{
							if (pKeyTemp == pKey)
							{
								pPredictedKeyboard = pKeyboardTemp;
								break;
							}
							pKeyTemp = pKeyTemp->pKeyNext;
					}
					if (pPredictedKeyboard != NULL)
					{
						break;
					}
					pKeyboardTemp = pKeyboardTemp->pKeyboardNext;
				}
				if (pPredictedKeyboard != NULL)
				{								
					gok_predictor_log(gok_main_get_command_predictor(),
						gok_keyboard_get_name(pPredictedKeyboard), 
							gok_key_get_label(pKey) );
				}
			}
	#endif /* WHENCOMMANDPREDICTIONINGOKDATAISHOOKEDUPTOPREFS */
			/* send the key output to the system */
			gok_output_send_to_system (pKey->pOutput, TRUE);
			
			/* output any modifier keys */
			gok_modifier_output_post();
			
			/* turn off any modifier keys that are not locked on */
			if (pKey->Type != KEYTYPE_MODIFIER)
			{
				gok_modifier_all_off();
			}
			break;
	}
	
	gok_log_leave();
	return pKey;
}	

/**
* gok_keyboard_validate_dynamic_keys
* @pAccessibleForeground: Pointer to the foreground accessible pointer.
*
* Enables or disables the keys that branch to the dynamic keyboards
* keyboards.
*
* Returns: TRUE if any of the keys have changed their state (disable/active).
* Returns FALSE if none of the keys change state.
**/
gboolean gok_keyboard_validate_dynamic_keys (Accessible* pAccessibleForeground)
{
	GokKeyboard* pKeyboard;
	GokKey* pKey;
	gboolean bHasMenuBranch;
	gboolean bHasToolbarBranch;
	gboolean bHasGUIBranch;
	gboolean bHasEditTextBranch;
	gboolean bChanged;

	gok_log_enter();

	pKeyboard = gok_main_get_current_keyboard();
	bHasMenuBranch = FALSE;
	bHasToolbarBranch = FALSE;
	bHasGUIBranch = FALSE;
	bHasEditTextBranch = FALSE;
	bChanged = FALSE;
	
	/* are there any keys that branch to a dynamic keyboard? */
	pKey = pKeyboard->pKeyFirst;
	while (pKey != NULL)
	{
		if (pKey->Type == KEYTYPE_BRANCHMENUS)
		{
			bHasMenuBranch = TRUE;
		}
		else if (pKey->Type == KEYTYPE_BRANCHTOOLBARS)
		{
			bHasToolbarBranch = TRUE;
		}
		else if (pKey->Type == KEYTYPE_BRANCHGUI)
		{
			bHasGUIBranch = TRUE;
		}
		else if (pKey->Type == KEYTYPE_BRANCHCOMPOSE)
		{
			bHasEditTextBranch = TRUE;
		}
		pKey = pKey->pKeyNext;
	}
	
	/* don't do anything if there are no menu and no toolbar keys */
	if ((bHasMenuBranch == FALSE) &&
		(bHasToolbarBranch == FALSE) &&
		(bHasGUIBranch == FALSE) &&
		(bHasEditTextBranch == FALSE))
	{
		gok_log("this keyboard has no dynamic branch keys");
		gok_log_leave();
		return bChanged;
	}

	/* enable/disable the branch keys */
	pKey = pKeyboard->pKeyFirst;
	while (pKey != NULL)
	{
		if (pKey->Type == KEYTYPE_BRANCHMENUS)
		{
			if (gok_spy_has_child (pAccessibleForeground, 
					       GOK_SPY_SEARCH_MENU, SPI_ROLE_MENU) == TRUE)
			{
				/* set this flag if the button state changes */
				if (strcmp (gtk_widget_get_name(pKey->pButton), "StyleButtonBranchMenus") != 0)
				{
					bChanged = TRUE;
				}
				gtk_widget_set_name (pKey->pButton, "StyleButtonBranchMenus");
				if (((GokButton*)pKey->pButton)->pLabel != NULL)
					gtk_widget_set_name (((GokButton*)pKey->pButton)->pLabel, 
							     "StyleTextNormal");
			}
			else
			{
				/* set this flag if the button state changes */
				if (strcmp (gtk_widget_get_name(pKey->pButton), "StyleButtonDisabled") != 0)
				{
					bChanged = TRUE;
				}
				gtk_widget_set_name (pKey->pButton, "StyleButtonDisabled"); 
				if (((GokButton*)pKey->pButton)->pLabel != NULL) 
					gtk_widget_set_name (((GokButton*)pKey->pButton)->pLabel, 
							     "StyleTextDisabled");
			}
		}
		else if (pKey->Type == KEYTYPE_BRANCHTOOLBARS)
		{
			if (gok_spy_has_child (pAccessibleForeground, GOK_SPY_SEARCH_ROLE,
					       SPI_ROLE_TOOL_BAR) == TRUE)
			{
				/* set this flag if the button state changes */
				if (strcmp (gtk_widget_get_name(pKey->pButton), "StyleButtonBranchToolbars") != 0)
				{
					bChanged = TRUE;
				}
				gtk_widget_set_name (pKey->pButton, "StyleButtonBranchToolbars"); 
				if (((GokButton*)pKey->pButton)->pLabel != NULL)
					gtk_widget_set_name (((GokButton*)pKey->pButton)->pLabel, 
							     "StyleTextNormal");
			}
			else
			{
				/* set this flag if the button state changes */
				if (strcmp (gtk_widget_get_name(pKey->pButton), "StyleButtonDisabled") != 0)
				{
					bChanged = TRUE;
				}
				gtk_widget_set_name (pKey->pButton, "StyleButtonDisabled"); 
				if (((GokButton*)pKey->pButton)->pLabel != NULL)
					gtk_widget_set_name (((GokButton*)pKey->pButton)->pLabel, 
							     "StyleTextDisabled");
			}
		}
		else if (pKey->Type == KEYTYPE_BRANCHGUI)
		{
			/* TODO: REFACTOR: enum KeyType */
			if ((gok_spy_has_child (pAccessibleForeground, 
						GOK_SPY_SEARCH_ROLE, SPI_ROLE_PUSH_BUTTON) == TRUE) ||
				(gok_spy_has_child (pAccessibleForeground, 
						    GOK_SPY_SEARCH_EDITABLE_TEXT, SPI_ROLE_TEXT) == TRUE))
			{
				/* set this flag if the button state changes */
				if (strcmp (gtk_widget_get_name(pKey->pButton), "StyleButtonBranchGUI") != 0)
				{
					bChanged = TRUE;
				}
				gtk_widget_set_name (pKey->pButton, "StyleButtonBranchGUI"); 
				if (((GokButton*)pKey->pButton)->pLabel != NULL)
					gtk_widget_set_name (((GokButton*)pKey->pButton)->pLabel, 
							     "StyleTextNormal");
			}
			else
			{
				/* set this flag if the button state changes */
				if (strcmp (gtk_widget_get_name(pKey->pButton), "StyleButtonDisabled") != 0)
				{
					bChanged = TRUE;
				}
				gtk_widget_set_name (pKey->pButton, "StyleButtonDisabled"); 
				if (((GokButton*)pKey->pButton)->pLabel != NULL)
					gtk_widget_set_name (((GokButton*)pKey->pButton)->pLabel, 
							     "StyleTextDisabled");
			}
		}
		else if (pKey->Type == KEYTYPE_BRANCHCOMPOSE)
		{
			if (gok_spy_get_accessibleWithText() != NULL)
			{
				/* set this flag if the button state changes */
				if (strcmp (gtk_widget_get_name(pKey->pButton), "StyleButtonBranchGUI") != 0)
				{
					bChanged = TRUE;
				}
				gtk_widget_set_name (pKey->pButton, "StyleButtonBranchGUI"); 
				if (((GokButton*)pKey->pButton)->pLabel != NULL)
					gtk_widget_set_name (((GokButton*)pKey->pButton)->pLabel, 
							     "StyleTextNormal");
			}
			else
			{
				/* set this flag if the button state changes */
				if (strcmp (gtk_widget_get_name(pKey->pButton), "StyleButtonDisabled") != 0)
				{
					bChanged = TRUE;
				}
				gtk_widget_set_name (pKey->pButton, "StyleButtonDisabled"); 
				if (((GokButton*)pKey->pButton)->pLabel != NULL)
					gtk_widget_set_name (((GokButton*)pKey->pButton)->pLabel, 
							     "StyleTextDisabled");
			}
		}
		pKey = pKey->pKeyNext;
	}
	
	gok_log_leave();
	
	return bChanged;
}

/**
* gok_keyboard_on_window_resize
*
* This will be called when the window has been resized.
* Change the key size, update the gok_data and settings dialog with the
* new key size.
* If we resize the window (by branching) then the m_bIgnoreResizeEvent flag
* will be set so we ignore the resize. This flag is needed because we can't get
* a message from the system letting us know that it was the user that resized
* the window.
**/
void gok_keyboard_on_window_resize ()
{
	GokKeyboard* pKeyboard;
	GokKeyboard* pKeyboardFontMod;
	GtkWidget* pWindow;
	gint widthWindow;
	gint heightWindow;
	gint widthWindowPrevious;
	gint heightWindowPrevious;
	gint widthKey;
	gint heightKey;

	/* ignore this resize event if we caused it (by branching the keyboard) */
	/* we want only the resize events generated by user resizing the window */
	if (m_bIgnoreResizeEvent == TRUE)
	{
		m_bIgnoreResizeEvent = FALSE;
		return;
	}
		
	/* these pointers will be NULL at the start of the program so we can't resize */
	pKeyboard = gok_main_get_current_keyboard();
	if (pKeyboard == NULL)
	{
		return;
	}
	pWindow = gok_main_get_main_window();
	if (pWindow == NULL)
	{
		return;
	}
	
	/* compare the size of the window to the size we made the window */
	gdk_window_get_size (pWindow->window, &widthWindow, &heightWindow);
	gok_main_get_our_window_size (&widthWindowPrevious, &heightWindowPrevious);
	
	/* if the size has not changed then don't do anything */
	if ((widthWindow == widthWindowPrevious) &&
		(heightWindow == heightWindowPrevious))
	{
		return;
	}
		
	/* change the key size */
	widthKey = gok_keyboard_get_keywidth_for_window (widthWindow, pKeyboard);
	if (widthKey < MIN_KEY_WIDTH)
	{
		widthKey = MIN_KEY_WIDTH;
	}
	else if (widthKey > MAX_KEY_WIDTH)
	{
		widthKey = MAX_KEY_WIDTH;
	}
	if ((gok_data_get_dock_type () == GOK_DOCK_NONE) || (pKeyboard->expand == GOK_EXPAND_NEVER)) 
		gok_data_set_key_width (widthKey);

	heightKey = gok_keyboard_get_keyheight_for_window (heightWindow, pKeyboard);
	if (heightKey < MIN_KEY_HEIGHT)
	{
		heightKey = MIN_KEY_HEIGHT;
	}
	else if (heightKey > MAX_KEY_HEIGHT)
	{
		heightKey = MAX_KEY_HEIGHT;
	}
	if ((gok_data_get_dock_type () == GOK_DOCK_NONE) || (pKeyboard->expand == GOK_EXPAND_NEVER)) 
	gok_data_set_key_height (heightKey);
	
	/* set this flag on each keyboard so it will calculate a new font size */
	pKeyboardFontMod = gok_main_get_first_keyboard();
	while (pKeyboardFontMod != NULL)
	{
		pKeyboardFontMod->bFontCalculated = FALSE;
		pKeyboardFontMod = pKeyboardFontMod->pKeyboardNext;
	}
	
	/* calculate a new font size for the current keyboard */
	gok_keyboard_calculate_font_size (pKeyboard);
	
	/* redraw all the keys */
	gok_keyboard_position_keys (pKeyboard, pWindow);
	
	/* update the settings dialog */
	gok_settings_page_keysizespace_refresh();
}

/**
* gok_keyboard_get_keywidth_for_window
* @WidthWindow: Width of the target window.
* @pKeyboard: Pointer to the keyboard that will be displayed.
*
* Calculates a key width for the current keyboard given the window width.
*
* returns: The key width.
**/
int gok_keyboard_get_keywidth_for_window (gint WidthWindow, GokKeyboard* pKeyboard)
{
	return (WidthWindow - ((gok_keyboard_get_number_columns (pKeyboard) - 1) * gok_data_get_key_spacing())) / gok_keyboard_get_number_columns (pKeyboard);
}

/**
* gok_keyboard_get_keyheight_for_window
* @HeightWindow: Height of the target window.
* @pKeyboard: Pointer to the keyboard that will be displayed.
*
* Calculates a key height for the current keyboard given the window height.
*
* returns: The key height.
**/
int gok_keyboard_get_keyheight_for_window (gint HeightWindow, GokKeyboard* pKeyboard)
{
	return (HeightWindow - ((gok_keyboard_get_number_rows (pKeyboard) - 1) * gok_data_get_key_spacing())) / gok_keyboard_get_number_rows (pKeyboard);
}

/**
* gok_keyboard_set_ignore_resize
* @bFlag: State of the resize flag.
*
* Sets/clears a flag so that the next resize event will be ignored.
**/
void gok_keyboard_set_ignore_resize (gboolean bFlag)
{
	m_bIgnoreResizeEvent = bFlag;
}

/**
* gok_keyboard_update_labels
*
* Redraws the labels on all the keys. This should be called whenever
* a modifier key changes state.
**/
void gok_keyboard_update_labels ()
{
	GokKeyboard* pKeyboard;
	GokKey* pKey;
	
	pKeyboard = gok_main_get_first_keyboard();
	g_assert (pKeyboard != NULL);
	
	while (pKeyboard != NULL)
	{
		pKey = pKeyboard->pKeyFirst;
		while (pKey != NULL)
		{
			gok_key_update_label (pKey);
			pKey = pKey->pKeyNext;
		}
		pKeyboard = pKeyboard->pKeyboardNext;
	}
}

/**
* gok_keyboard_find_key_at:
* @keyboard: a pointer to a #GokKeyboard structure.
* @x: An x coordinate, in the keyboard window's coordinate system. 
* @y: A y coordinate in the keyboard window's coordinate system.
*
* Find the keyboard key corresponding to @x, @y in window coordinates.
* returns: A pointer to a #GokKey, or NULL if no key in the keyboard 
*          contains the point.
**/
GokKey *
gok_keyboard_find_key_at (GokKeyboard *pKeyboard, gint x, gint y)
{
	GokKey *pKey = NULL;
	gint row, col, n_rows, n_cols;
	if (pKeyboard) {
		pKey = pKeyboard->pKeyFirst;
		while (pKey) {
			if (gok_key_contains_point (pKey, x, y)) {
				return pKey;
			}
			pKey = pKey->pKeyNext;
		}
	}
	return NULL;
}

/**
 **/
gboolean
gok_keyboard_modifier_is_set (guint modifier)
{
	return FALSE;
}

/**
 **/
GokKeyboardDirection 
gok_keyboard_parse_direction (const gchar *string)
{
	if (!strcmp (string, "east"))
		return GOK_DIRECTION_E;
	else if (!strcmp (string, "northeast"))
		return GOK_DIRECTION_NE;
	else if (!strcmp (string, "north"))
		return GOK_DIRECTION_N;
	else if (!strcmp (string, "northwest"))
		return GOK_DIRECTION_NW;
	else if (!strcmp (string, "west"))
		return GOK_DIRECTION_W;
	else if (!strcmp (string, "southwest"))
		return GOK_DIRECTION_SW;
	else if (!strcmp (string, "south"))
		return GOK_DIRECTION_S;
	else if (!strcmp (string, "southeast"))
		return GOK_DIRECTION_SE;
	else if (!strcmp (string, "fillwidth"))
		return GOK_DIRECTION_FILL_EW;
	else if (!strcmp (string, "narrower"))
		return GOK_RESIZE_NARROWER;
	else if (!strcmp (string, "wider"))
		return GOK_RESIZE_WIDER;
	else if (!strcmp (string, "shorter"))
		return GOK_RESIZE_SHORTER;
	else if (!strcmp (string, "taller"))
		return GOK_RESIZE_TALLER;
	return GOK_DIRECTION_NONE;
}
