/* gok-composer.c
*
* Copyright 2001,2002 Sun Microsystems, Inc.,
* Copyright 2001,2002 University Of Toronto
*
* This library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Library General Public
* License as published by the Free Software Foundation; either
* version 2 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Library General Public License for more details.
*
* You should have received a copy of the GNU Library General Public
* License along with this library; if not, write to the
* Free Software Foundation, Inc., 59 Temple Place - Suite 330,
* Boston, MA 02111-1307, USA.
*/

/* 
   This is a runtime created (but mostly static) keyboard that 
   provides gok access to accessible text interface methods
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "gok-composer.h"
#include "gok-log.h"
#include "main.h"
#include "gok-spy.h"

/**
* gok_composer_show
*
* returns: TRUE if the keyboard was displayed, FALSE if not.
**/
gboolean gok_composer_show ()
{
	GokKeyboard* pKeyboard;
	GokKeyboard* pKeyboardNew;
	GokKey* pKey;
	GokKey* pKeyPrevious;
	int column;
	Accessible* pAccessible;
	GokKeyActionType keytype; 
	

	gok_log_enter();	

	pAccessible = gok_spy_get_accessibleWithEditableText();

	if (pAccessible == NULL)
	{
		gok_log_x ("no editable text interface - aborting...");
		gok_log_leave();	
		return FALSE;
	}
	
	/* now build the accessible text keyboard */

	/* first check if there is already such a keyboard created */
	pKeyboard = (GokKeyboard*)gok_main_get_first_keyboard();
	while (pKeyboard != NULL)
	{
		if (pKeyboard->Type == KEYBOARD_TYPE_EDITTEXT)
		{
			break;
		}
		pKeyboard = pKeyboard->pKeyboardNext;
	}
	
	if (pKeyboard == NULL)
	{
		/* no accessible text keyboard so create one */
		gok_log ("creating edit text keyboard..");
		pKeyboardNew = gok_keyboard_new();
		g_assert (pKeyboardNew != NULL);
		
		/* add the new keyboard to the list of keyboards (at the end)*/
		pKeyboard = (GokKeyboard*)gok_main_get_first_keyboard();
		g_assert (pKeyboard != NULL);
		while (pKeyboard->pKeyboardNext != NULL)
		{
			pKeyboard = pKeyboard->pKeyboardNext;
		}
		pKeyboard->pKeyboardNext = pKeyboardNew;
		pKeyboardNew->pKeyboardPrevious = pKeyboard;
		
		pKeyboardNew->Type = KEYBOARD_TYPE_EDITTEXT;
		gok_keyboard_set_name (pKeyboardNew, _("Text Manipulation"));
		
		pKeyboard = pKeyboardNew;
	}
	else
	{
	
		gok_log ("removing old keys.");
		
		/* remove any old keys on the old keyboard */
		pKey = pKeyboard->pKeyFirst;
		while (pKey != NULL)
		{
			gok_key_delete (pKey, pKeyboard, TRUE);
			pKey = pKey->pKeyNext;
		}
	}
	
	/* set this flag so the keyboard will be laid out when it's displayed */
	pKeyboard->bLaidOut = FALSE;

	gok_keyboard_set_accessible(pKeyboard, pAccessible);
	
	gok_log ("adding keys dynamically");

	/* add the new keys to the menu keyboard */
	/* first, add a 'back' key */
	pKey = gok_key_new (NULL, NULL, pKeyboard);
	pKey->Style = KEYSTYLE_BRANCHBACK;
	pKey->Type = KEYTYPE_BRANCHBACK;
	pKey->Top = 0;
	pKey->Bottom = 1;
	pKey->Left = 0;
	pKey->Right = 1;
	gok_key_add_label (pKey, _("back"), NULL);
	
	pKeyPrevious = pKey;
		
	/* create all the keys as one long row */
	/* the keys will be repositioned in gok_keyboard_layout */
	column = 1;
	for (keytype = KEYTYPE_ACCESSIBLETEXT_CHARRIGHT ; 
		 keytype <= KEYTYPE_EDITABLETEXT_CLEAR ;
		 keytype++) {
			pKey = gok_key_new (pKeyPrevious, NULL, pKeyboard);
			pKeyPrevious = pKey;
			pKey->Style = KEYSTYLE_EDITABLETEXTACTION;
			pKey->Top = 0;
			pKey->Bottom = 1;
			pKey->Left = column;
			pKey->Right = column + 1;
			pKey->Type = keytype;
			 
			 
			switch (keytype)
			{
				case KEYTYPE_ACCESSIBLETEXT_CHARRIGHT:
					gok_key_add_label (pKey, _("right"), NULL);
					break;
				case KEYTYPE_ACCESSIBLETEXT_CHARLEFT:
					gok_key_add_label (pKey, _("left"), NULL);
					break;
				case KEYTYPE_ACCESSIBLETEXT_WORDRIGHT:
					gok_key_add_label (pKey, _("word right"), NULL);
					break;
				case KEYTYPE_ACCESSIBLETEXT_WORDLEFT:
					gok_key_add_label (pKey, _("word left"), NULL);
					break;
				case KEYTYPE_ACCESSIBLETEXT_SENTENCERIGHT:
					gok_key_add_label (pKey, _(". right"), NULL);
					break;
				case KEYTYPE_ACCESSIBLETEXT_SENTENCELEFT:
					gok_key_add_label (pKey, _(". left"), NULL);
					break;
				case KEYTYPE_ACCESSIBLETEXT_SELECT_CHARRIGHT:
					gok_key_add_label (pKey, _("select right"), NULL);
					break;
				case KEYTYPE_ACCESSIBLETEXT_SELECT_CHARLEFT:
					gok_key_add_label (pKey, _("select left"), NULL);
					break;
				case KEYTYPE_ACCESSIBLETEXT_SELECT_WORDRIGHT:
					gok_key_add_label (pKey, _("select word right"), NULL);
					break;
				case KEYTYPE_ACCESSIBLETEXT_SELECT_WORDLEFT:
					gok_key_add_label (pKey, _("select word left"), NULL);
					break;
				case KEYTYPE_ACCESSIBLETEXT_SELECT_SENTENCERIGHT:
					gok_key_add_label (pKey, _("select . right"), NULL);
					break;
				case KEYTYPE_ACCESSIBLETEXT_SELECT_SENTENCELEFT:
					gok_key_add_label (pKey, _("select . left"), NULL);
					break;
				case KEYTYPE_EDITABLETEXT_COPY:
					gok_key_add_label (pKey, _("copy"), NULL);
					break;
				case KEYTYPE_EDITABLETEXT_DELETE:
					gok_key_add_label (pKey, _("delete"), NULL);
					break;
				case KEYTYPE_EDITABLETEXT_CUT:
					gok_key_add_label (pKey, _("cut"), NULL);
					break;
				case KEYTYPE_EDITABLETEXT_PASTE:
					gok_key_add_label (pKey, _("paste"), NULL);
					break;
				case KEYTYPE_EDITABLETEXT_CLEAR:
					gok_key_add_label (pKey, _("clear"), NULL);
					break;
				default:
				gok_log_x("unknown key type snuck in!");
				break;
			}
		column++;
	}

	
	/* display and scan the menus keyboard */
	gok_main_display_scan (pKeyboard, NULL, KEYBOARD_TYPE_UNSPECIFIED, KEYBOARD_LAYOUT_NORMAL);
	
	gok_log_leave();	
	return TRUE;
}

/**
* gok_composer_branch_textAction
* @pKeyboard: pointer to the keyboard.
* @pKey: pointer to the key
*
* Performs a text manipulation/navigation action.
*
* returns: TRUE if the action call was made.
**/
gboolean gok_composer_branch_textAction (GokKeyboard* pKeyboard, GokKey* pKey)
{
	Accessible* paccessible;
	AccessibleEditableText* peditableText;
	AccessibleText* paccessibleText;
	long currentCaretPos;
	long selectionStart;
	long selectionEnd;
	long rangeStart;
	long rangeEnd;
	gboolean returncode = FALSE;
	gboolean navigate = FALSE;
	gboolean select = FALSE;
	gboolean backward = FALSE;
	AccessibleTextBoundaryType boundary_type;
	
	currentCaretPos = 0;
	
	gok_log_enter();
	
	/* do we have an accessible object? */
	paccessible = gok_keyboard_get_accessible ( pKeyboard );
	if (paccessible == NULL)
	{
		gok_log_x ("no Accessible object -- aborting");	
		gok_log_leave();
		return FALSE;
	}
	
	/* do we have an accessible text interface? */
	paccessibleText = Accessible_getText ( paccessible );
	if ( paccessibleText == NULL)
	{
		gok_log_x("no AccessibleText object - aborting");
		gok_log_leave();
		return FALSE;
	}
	gok_spy_accessible_implicit_ref(paccessibleText);

	/* do we have an editabletext interface? */
	peditableText = Accessible_getEditableText ( paccessible );
	if (peditableText == NULL)
	{
		gok_spy_accessible_unref(paccessibleText);
		gok_log_x ("no AccessibleEditableText object -- aborting");	
		gok_log_leave();
		return FALSE;
	}
	gok_spy_accessible_implicit_ref(peditableText);

	currentCaretPos = AccessibleText_getCaretOffset( paccessibleText );
	if (currentCaretPos < 0)
	{
		currentCaretPos = 0;
	}
	
	AccessibleText_getSelection( paccessibleText, 0, &selectionStart, &selectionEnd );

	switch (pKey->Type)
	{
		case KEYTYPE_ACCESSIBLETEXT_CHARRIGHT:
			navigate = TRUE;
			boundary_type = SPI_TEXT_BOUNDARY_CHAR;
			gok_log_x("char right");
			break;
		case KEYTYPE_ACCESSIBLETEXT_CHARLEFT:
			navigate = TRUE;
			backward = TRUE;
			boundary_type = SPI_TEXT_BOUNDARY_CHAR;
			gok_log_x("char left");
			break;
		case KEYTYPE_ACCESSIBLETEXT_WORDRIGHT:
			navigate = TRUE;
			boundary_type = SPI_TEXT_BOUNDARY_WORD_END;
			gok_log ("word right");	
			break;
		case KEYTYPE_ACCESSIBLETEXT_WORDLEFT:
			navigate = TRUE;
			backward = TRUE;
			boundary_type = SPI_TEXT_BOUNDARY_WORD_START;
			gok_log ("word left");	
			break;
		case KEYTYPE_ACCESSIBLETEXT_SENTENCERIGHT:
			navigate = TRUE;
			boundary_type = SPI_TEXT_BOUNDARY_SENTENCE_END;
			gok_log ("sentence right");	
			break;
		case KEYTYPE_ACCESSIBLETEXT_SENTENCELEFT:
			navigate = TRUE;
			backward = TRUE;
			boundary_type = SPI_TEXT_BOUNDARY_SENTENCE_START;
			gok_log ("sentence right");	
			break;
		case KEYTYPE_ACCESSIBLETEXT_SELECT_CHARRIGHT:
			select = TRUE;
			boundary_type = SPI_TEXT_BOUNDARY_CHAR;
			gok_log_x("select char right");
			break;
		case KEYTYPE_ACCESSIBLETEXT_SELECT_CHARLEFT:
			select = TRUE;
			backward = TRUE;
			boundary_type = SPI_TEXT_BOUNDARY_CHAR;
			gok_log_x("select char left");
			break;
		case KEYTYPE_ACCESSIBLETEXT_SELECT_WORDRIGHT:
			select = TRUE;
			boundary_type = SPI_TEXT_BOUNDARY_WORD_END;
			gok_log ("select word right");	
			break;
		case KEYTYPE_ACCESSIBLETEXT_SELECT_WORDLEFT:
			select = TRUE;
			backward = TRUE;
			boundary_type = SPI_TEXT_BOUNDARY_WORD_START;
			gok_log ("select word left");	
			break;
		case KEYTYPE_ACCESSIBLETEXT_SELECT_SENTENCERIGHT:
			select = TRUE;
			boundary_type = SPI_TEXT_BOUNDARY_SENTENCE_END;
			gok_log ("select sentence right");	
			break;
		case KEYTYPE_ACCESSIBLETEXT_SELECT_SENTENCELEFT:
			select = TRUE;
			backward = TRUE;
			boundary_type = SPI_TEXT_BOUNDARY_SENTENCE_START;
			gok_log ("select sentence right");	
			break;
		case KEYTYPE_EDITABLETEXT_COPY:
			gok_log ("copy");	
			returncode = AccessibleEditableText_copyText ( peditableText, 
				selectionStart, selectionEnd );
			break;
		case KEYTYPE_EDITABLETEXT_DELETE:
			gok_log ("delete");	
			returncode = AccessibleEditableText_deleteText ( peditableText, 
				selectionStart, selectionEnd );
			break;
		case KEYTYPE_EDITABLETEXT_CUT:
			gok_log ("cut");	
			returncode = AccessibleEditableText_cutText ( peditableText, 
				selectionStart, selectionEnd );
			break;
		case KEYTYPE_EDITABLETEXT_PASTE:
			gok_log ("paste");	
			returncode = AccessibleEditableText_pasteText ( peditableText, 
				currentCaretPos );
			break;
		case KEYTYPE_EDITABLETEXT_CLEAR:
			gok_log ("clear");	
			returncode = AccessibleEditableText_setTextContents ( 
				peditableText, "" );
			break;
		default:
		gok_log_x("unknown key type snuck in!");
		break;
	}
	
	if (navigate || select) {
		if (backward)
		{
			AccessibleText_getTextBeforeOffset (paccessibleText,
						currentCaretPos,
						boundary_type,
						&rangeStart,
						&rangeEnd);
		}
		else {
			AccessibleText_getTextAtOffset (paccessibleText,
						currentCaretPos,
						boundary_type,
						&rangeStart,
						&rangeEnd);
		}
			
	}
	if (navigate) {
		if (backward) 
			currentCaretPos = rangeStart;
		else
			currentCaretPos = rangeEnd;
		returncode = AccessibleText_setCaretOffset (paccessibleText,
							    currentCaretPos);
	}
	else if (select) {
		if (AccessibleText_getNSelections(paccessible) != 0) {
			AccessibleText_getSelection( paccessibleText, 
						(long) 0, /* only interested in contiguous sel */
						&selectionStart, 
						&selectionEnd );
			rangeStart = 
				(selectionStart < rangeStart)? selectionStart: rangeStart;
			rangeEnd = 
				(selectionEnd > rangeEnd)? selectionEnd: rangeEnd;
			returncode = AccessibleText_setSelection(paccessibleText, 
						0, 
						rangeStart,
						rangeEnd);
		}
		else {
			returncode = AccessibleText_addSelection(paccessibleText, 
						rangeStart, 
						rangeEnd);
		}
	}
			
	gok_spy_accessible_unref(peditableText);
	gok_spy_accessible_unref(paccessibleText);
	gok_log_leave();
	return returncode;
}
