/* gok-composer.c
*
* Copyright 2001,2002 Sun Microsystems, Inc.,
* Copyright 2001,2002 University Of Toronto
*
* This library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Library General Public
* License as published by the Free Software Foundation; either
* version 2 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Library General Public License for more details.
*
* You should have received a copy of the GNU Library General Public
* License along with this library; if not, write to the
* Free Software Foundation, Inc., 59 Temple Place - Suite 330,
* Boston, MA 02111-1307, USA.
*/

/* 
   This is a runtime created (but mostly static) keyboard that 
   provides gok access to accessible text interface methods
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "gok-composer.h"
#include "gok-log.h"
#include "main.h"
#include "gok-spy.h"

/**
* gok_composer_show
*
* returns: TRUE if the keyboard was displayed, FALSE if not.
**/
gboolean gok_composer_show ()
{
	GokKeyboard* pKeyboard;
	GokKeyboard* pKeyboardNew;
	GokKey* pKey;
	GokKey* pKeyPrevious;
	int column;
	Accessible* pAccessible;
	

	gok_log_enter();	

	pAccessible = gok_spy_get_accessibleWithEditableText();

	if (pAccessible == NULL)
	{
		gok_log_x ("no editable text interface - aborting...");
		gok_log_leave();	
		return FALSE;
	}
	
	/* now build the accessible text keyboard */

	/* first check if there is already such a keyboard created */
	pKeyboard = (GokKeyboard*)gok_main_get_first_keyboard();
	while (pKeyboard != NULL)
	{
		if (pKeyboard->Type == KEYBOARD_TYPE_EDITTEXT)
		{
			break;
		}
		pKeyboard = pKeyboard->pKeyboardNext;
	}
	
	if (pKeyboard == NULL)
	{
		/* no accessible text keyboard so create one */
		gok_log ("creating edit text keyboard..");
		pKeyboardNew = gok_keyboard_new();
		if (pKeyboardNew == NULL)
		{
			return FALSE;
		}
		
		/* add the new keyboard to the list of keyboards (at the end)*/
		pKeyboard = (GokKeyboard*)gok_main_get_first_keyboard();
		g_assert (pKeyboard != NULL);
		while (pKeyboard->pKeyboardNext != NULL)
		{
			pKeyboard = pKeyboard->pKeyboardNext;
		}
		pKeyboard->pKeyboardNext = pKeyboardNew;
		pKeyboardNew->pKeyboardPrevious = pKeyboard;
		
		pKeyboardNew->Type = KEYBOARD_TYPE_EDITTEXT;
		gok_keyboard_set_name (pKeyboardNew, _("Text Editing"));
		
		pKeyboard = pKeyboardNew;
	}
	else
	{
	
		gok_log ("removing old keys.");
		
		/* remove any old keys on the old keyboard */
		pKey = pKeyboard->pKeyFirst;
		while (pKey != NULL)
		{
			gok_key_delete (pKey, pKeyboard, TRUE);
			pKey = pKey->pKeyNext;
		}
	}
	
	/* set this flag so the keyboard will be laid out when it's displayed */
	pKeyboard->bLaidOut = FALSE;

	gok_keyboard_set_accessible(pKeyboard, pAccessible);
	
	gok_log ("adding keys dynamically");

	/* add the new keys to the menu keyboard */
	/* first, add a 'back' key */
	pKey = gok_key_new (NULL, NULL, pKeyboard);
	pKey->Style = KEYSTYLE_BRANCHBACK;
	pKey->Type = KEYTYPE_BRANCHBACK;
	pKey->Top = 0;
	pKey->Bottom = 1;
	pKey->Left = 0;
	pKey->Right = 1;
	gok_key_add_label (pKey, _("back"), NULL);
	
	pKeyPrevious = pKey;
		
	/* create all the keys as one long row */
	/* the keys will be repositioned in gok_keyboard_layout */
	column = 1;
	

	/* add key */
	{
		pKey = gok_key_new (pKeyPrevious, NULL, pKeyboard);
		pKeyPrevious = pKey;
		
		pKey->Style = KEYSTYLE_GENERALDYNAMIC;
		pKey->Type = KEYTYPE_EDITABLETEXT_CUT;
		pKey->Top = 0;
		pKey->Bottom = 1;
		pKey->Left = column;
		pKey->Right = column + 1;
		
		gok_key_add_label (pKey, _("cut"), NULL);

		column++;
	}
	
	/* add key */
	{
		pKey = gok_key_new (pKeyPrevious, NULL, pKeyboard);
		pKeyPrevious = pKey;
		
		pKey->Style = KEYSTYLE_GENERALDYNAMIC;
		pKey->Type = KEYTYPE_EDITABLETEXT_COPY;
		pKey->Top = 0;
		pKey->Bottom = 1;
		pKey->Left = column;
		pKey->Right = column + 1;
		
		gok_key_add_label (pKey,_("copy"), NULL);

		column++;
	}
	
	/* add key */
	{
		pKey = gok_key_new (pKeyPrevious, NULL, pKeyboard);
		pKeyPrevious = pKey;
		
		pKey->Style = KEYSTYLE_GENERALDYNAMIC;
		pKey->Type = KEYTYPE_EDITABLETEXT_PASTE;
		pKey->Top = 0;
		pKey->Bottom = 1;
		pKey->Left = column;
		pKey->Right = column + 1;
		
		gok_key_add_label (pKey,_("paste"), NULL);

		column++;
	}
	
	/* add key */
	{
		pKey = gok_key_new (pKeyPrevious, NULL, pKeyboard);
		pKeyPrevious = pKey;
		
		pKey->Style = KEYSTYLE_GENERALDYNAMIC;
		pKey->Type = KEYTYPE_EDITABLETEXT_CLEAR;
		pKey->Top = 0;
		pKey->Bottom = 1;
		pKey->Left = column;
		pKey->Right = column + 1;
		
		gok_key_add_label (pKey,_("clear"), NULL);

		column++;
	}
	
	
	/* display and scan the menus keyboard */
	gok_main_display_scan (pKeyboard, NULL, KEYBOARD_TYPE_UNSPECIFIED, KEYBOARD_LAYOUT_NORMAL);
	
	gok_log_leave();	
	return TRUE;
}

/**
* gok_composer_branch_textAction
* @pKeyboard: pointer to the keyboard.
* @pKey: pointer to the key
*
* Performs a text manipulation/navigation action.
*
* returns: TRUE if the action call was made.
**/
gboolean gok_composer_branch_textAction (GokKeyboard* pKeyboard, GokKey* pKey)
{
	Accessible* paccessible;
	AccessibleEditableText* peditableText;
	AccessibleText* paccessibleText;
	long currentCaretPos;
	long selectionStart;
	long selectionEnd;
	
	currentCaretPos = 0;
	
	gok_log_enter();
	
	paccessible = gok_keyboard_get_accessible ( pKeyboard );
	

	if (paccessible == NULL)
	{
		gok_log_x ("no Accessible* -- aborting");	
		gok_log_leave();
		return FALSE;
	}
	
	peditableText = Accessible_getEditableText ( paccessible );
	gok_spy_accessible_implicit_ref(peditableText);
	if (peditableText == NULL)
	{
		gok_log_x ("no AccessibleEditableText* -- aborting");	
		gok_log_leave();
		return FALSE;
	}
	
	/*
	 * put actions NOT requiring the AccessibleText interface here
	 */
	if (pKey->Type == KEYTYPE_EDITABLETEXT_CLEAR)
	{
		gok_log ("clear");	
		AccessibleEditableText_setTextContents ( peditableText, "" );
	}

	paccessibleText = Accessible_getText ( paccessible );
	gok_spy_accessible_implicit_ref(paccessibleText);
	if ( paccessibleText == NULL)
	{
		gok_log ("this action required non existant AccessibleText interface - aborting");
		gok_log_leave();
		return FALSE;
	}

	
	/*
	 * put actions REQUIRING the AccessibleText interface here
	 */
	if (pKey->Type == KEYTYPE_EDITABLETEXT_PASTE)
	{
		gok_log ("paste");	
		currentCaretPos = AccessibleText_getCaretOffset( paccessibleText );
		if (currentCaretPos >= 0)
		{
			AccessibleEditableText_pasteText ( peditableText, currentCaretPos );
		}
		gok_log_leave();
		return TRUE;;
	}
	
	
	/*
	 * put actions EXPECTING a selection of text here
	 */

	AccessibleText_getSelection( paccessibleText, 0, &selectionStart, &selectionEnd );
	
	gok_log("text selection 0 starts[%d] and ends[%d]",selectionStart,selectionEnd);
	
	if (pKey->Type == KEYTYPE_EDITABLETEXT_COPY)
	{
		gok_log ("copy");	
		
		AccessibleEditableText_copyText ( peditableText, selectionStart, selectionEnd );
		
		gok_log_leave();
		return TRUE;
	}

	if (pKey->Type == KEYTYPE_EDITABLETEXT_CUT)
	{
		gok_log ("cut");	
		
		AccessibleEditableText_cutText ( peditableText, selectionStart, selectionEnd );
		
		gok_log_leave();
		return TRUE;;
	}
	
	
	gok_log_leave();
	return FALSE;
}
