/* $Id: email_conduit.c,v 1.16 2002/04/25 16:42:37 jpr Exp $ */

#include <glib.h>
#include <gnome.h>

#include <pi-source.h>
#include <pi-socket.h>
#include <pi-mail.h>
#include <pi-dlp.h>
#include <pi-version.h>

#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <utime.h>
#include <unistd.h>
#include <pwd.h>
#include <signal.h>

#include <gpilotd/gnome-pilot-conduit.h>
#include <gpilotd/gnome-pilot-conduit-standard.h>
#include "email_conduit.h"

#define CONDUIT_VERSION "0.10"

/*#define EC_DEBUG */
#ifdef EC_DEBUG
#define LOG(format,args...) g_log (G_LOG_DOMAIN, \
                                   G_LOG_LEVEL_MESSAGE, \
                                   "email: "##format, ##args)
#else
#define LOG(format,args...)
#endif

GnomePilotConduit *conduit_get_gpilot_conduit( guint32 pilotId ) ;
void conduit_destroy_gpilot_conduit( GnomePilotConduit *c );

static void 
load_configuration(ConduitCfg **c,guint32 pilotId) 
{
	gchar *prefix;

	g_assert(c!=NULL);
	*c = g_new0(ConduitCfg,1);
	(*c)->child = -1;

	prefix = g_strdup_printf("/gnome-pilot.d/email-conduit/Pilot_%u/",pilotId);
  
	gnome_config_push_prefix(prefix);
	(*c)->sendmail = gnome_config_get_string( "sendmail=/usr/lib/sendmail -t -i");
	(*c)->fromAddr = gnome_config_get_string( "from_address" );
	(*c)->sendAction = gnome_config_get_string( "send_action=file");
	(*c)->mhDirectory = gnome_config_get_string( "mh_directory" );
	(*c)->mboxFile = gnome_config_get_string ( "mbox_file" );
	(*c)->receiveAction = gnome_config_get_string( "receive_action=file" );
	gnome_config_pop_prefix();

	(*c)->pilotId = pilotId;
	g_free(prefix);
}

/** this method frees all data from the conduit config */
static void 
destroy_configuration(ConduitCfg **c) 
{
	g_assert(c!=NULL);
	g_assert(*c!=NULL);
	g_free( (*c)->sendmail );
	g_free( (*c)->fromAddr );
	g_free( (*c)->sendAction );
	g_free( (*c)->mhDirectory );
	g_free( (*c)->mboxFile );
	g_free( (*c)->receiveAction );
	g_free(*c);
	*c = NULL;
}


void markline( char *msg ) 
{
    while( (*msg) != '\n' && (*msg) != 0 ) {
        msg++; 
    }
    (*msg) = 0;
}

int openmhmsg( char *dir, int num ) 
{ 
    char filename[1000];
    
    sprintf( filename, "%s/%d", dir, num ); 
    return( open( filename, O_RDONLY ) );
}

char *skipspace( char *c ) 
{
    while ( c && ((*c == ' ') || (*c == '\t')) ) { 
        c++;
    }
    return( c );
}

void header( struct Mail *m, char *t )
{
    static char holding[4096];

    if ( t && strlen(t) && ( t[strlen(t)-1] == '\n' ) ) {
        t[strlen(t)-1] = 0;
    }
         
    if ( t && ((t[0] == ' ') || (t[0] == '\t')) ) { 
        if ( (strlen(t) + strlen(holding)) > 4096 ) { 
            return; /* Just discard approximate overflow */
        }
        strcat( holding, t+1 ); 
        return; 
    }
           
    /* Decide on what we do with m->sendTo */
           
    if ( strncmp( holding, "From:", 5 ) == 0 ) { 
        m->from = strdup( skipspace( holding + 5 ) ); 
    } else if ( strncmp( holding, "To:", 3 ) == 0 ) { 
        m->to = strdup( skipspace( holding + 3 ) ); 
    } else if ( strncmp( holding, "Subject:", 8 ) == 0 ) { 
        m->subject = strdup( skipspace( holding + 8 ) ); 
    } else if ( strncmp( holding, "Cc:", 3 ) == 0 ) {
        m->cc = strdup( skipspace( holding + 3 ) ); 
    } else if ( strncmp( holding, "Bcc:", 4 ) == 0 ) { 
        m->bcc = strdup( skipspace( holding + 4 ) ); 
    } else if ( strncmp( holding, "Reply-To:", 9 ) == 0 ) { 
        m->replyTo = strdup( skipspace( holding + 9 ) ); 
    } else if ( strncmp( holding, "Date:", 4 ) == 0 ) { 
        time_t d = parsedate(skipspace(holding+5)); 
        
        if ( d != -1 ) { 
            struct tm * d2; 
            
            m->dated = 1; 
            d2 = localtime( &d ); 
            m->date = *d2; 
        } 
    } 
    
    holding[0] = 0; 
    
    if ( t ) { 
        strcpy( holding, t );
    }
}

static gboolean
write_message_to_pilot (GnomePilotConduit *c, GnomePilotDBInfo *dbi, 
			int dbHandle, char *buffer, int msg_num)
{
    char *msg;
    int h;
    struct Mail t;
    int len;
    
    t.to = NULL;
    t.from = NULL;
    t.cc = NULL;
    t.bcc = NULL;
    t.subject = NULL;
    t.replyTo = NULL;
    t.sentTo = NULL;
    t.body = NULL;
    t.dated = 0;

    msg = (char *)buffer;
    h = 1;
    
    while ( h == 1 ) {
	markline( msg );
	
	if ( ( msg[0] == 0 ) && ( msg[1] == 0 ) ) {
	    break;
	}
	
	if ( msg[0] == 0 ) {
	    h = 0;
	    header( &t, 0 );
	} else {
	    header( &t, msg );
	}
	msg += strlen(msg)+1;
    }
    
    if ( (*msg) == 0 ) {
	h = 1;
    }
    
    if ( h ) {
	fprintf( stderr, "Incomplete message %d\n", msg_num );
	free_Mail( &t );
	return FALSE;
    }
    
    t.body = strdup( msg );
    
    len = pack_Mail( &t, buffer, 0xffff );
    
    if ( dlp_WriteRecord( dbi->pilot_socket, dbHandle, 0, 0, 0, buffer,
			  len, 0 ) > 0 ) {
	return TRUE;
    } else {
	fprintf( stderr, "Error writing message to Pilot\n" );
	return FALSE;
    }
}

static gint synchronize( GnomePilotConduit *c, GnomePilotDBInfo *dbi ) 
{
    int dbHandle;
    guchar buffer[0xffff];
    struct MailAppInfo tai;
    struct MailSyncPref pref;
    struct MailSignaturePref sig;
    int i;
    int rec;
    int dupe;
   
    g_message ("SendMail Conduit v %s",CONDUIT_VERSION);

    memset( &tai, '\0', sizeof( struct MailAppInfo ) );

    if ( dlp_OpenDB( dbi->pilot_socket, 0, 0x80|0x40, "MailDB", 
                     &dbHandle ) < 0 ) {
        fprintf( stderr, "Unable to open mail database\n" );
        return( -1 );
    }
    
    dlp_ReadAppBlock( dbi->pilot_socket, dbHandle, 0, buffer, 0xffff );
    unpack_MailAppInfo( &tai, buffer, 0xffff );
   
    pref.syncType = 0;
    pref.getHigh = 0;
    pref.getContaining = 0;
    pref.truncate = 8 * 1024;
    pref.filterTo = 0;
    pref.filterFrom = 0;
    pref.filterSubject = 0;

    if ( pi_version( dbi->pilot_socket ) > 0x0100 ) {
        if ( dlp_ReadAppPreference( dbi->pilot_socket, makelong("mail"), 1, 1, 
                                    0xffff, buffer, 0, 0 ) >= 0 ) {
            unpack_MailSyncPref( &pref, buffer, 0xffff );
        } else { 
            if ( dlp_ReadAppPreference( dbi->pilot_socket, makelong("mail"), 1,
                                        1, 0xffff, buffer, 0, 0 ) >= 0 ) { 
                unpack_MailSyncPref( &pref, buffer, 0xffff ); 
            } else {
	      LOG("Couldn't get any mail preferences.\n",0);
            }
        } 

        if ( dlp_ReadAppPreference( dbi->pilot_socket, makelong("mail"), 3, 1, 
                                    0xffff, buffer, 0, 0 ) > 0 ) {
            unpack_MailSignaturePref( &sig, buffer, 0xffff );
        } 
    }

    for ( i = 0; ; i++ ) {
        struct Mail t;
        int attr;
        int size;
        recordid_t recID;
        int length;
        FILE * sendf;
        
        length = dlp_ReadNextRecInCategory( dbi->pilot_socket, dbHandle, 1,
                                            buffer, &recID, 0, &size, &attr );
        
        if ( length < 0 ) {
            break;
        }
        
        if ( ( attr & dlpRecAttrDeleted ) || ( attr & dlpRecAttrArchived ) ) {
            continue;
        }
        
        unpack_Mail( &t, buffer, length );
       
        sendf = popen( GET_CONFIG(c)->sendmail, "w" );
        if ( sendf == NULL ) {
            fprintf( stderr, "Unable to create sendmail process\n" );
            break;
        }

        if ( GET_CONFIG(c)->fromAddr ) {
            fprintf( sendf, "From: %s\n", 
                     GET_CONFIG(c)->fromAddr );
#ifdef EC_DEBUG
            fprintf( stderr, "mail: From: %s\n", 
                     GET_CONFIG(c)->fromAddr );
#endif
        }
        if ( t.to ) {
            fprintf( sendf, "To: %s\n", t.to );
#ifdef EC_DEBUG
            fprintf( stderr, "mail: To: %s\n", t.to );
#endif
        }
        if ( t.cc ) {
            fprintf( sendf, "Cc: %s\n", t.cc );
#ifdef EC_DEBUG
            fprintf( stderr, "mail: Cc: %s\n", t.cc );
#endif
        }
        if ( t.bcc ) {
            fprintf( sendf, "Bcc: %s\n", t.bcc );
#ifdef EC_DEBUG
            fprintf( stderr, "mail: Bcc: %s\n", t.bcc );
#endif
        }
        if ( t.replyTo ) {
            fprintf( sendf, "Reply-To: %s\n", t.replyTo );
#ifdef EC_DEBUG
            fprintf( stderr, "mail: Reply-To: %s\n", t.replyTo );
#endif
        }
        if ( t.subject ) {
            fprintf( sendf, "Subject: %s\n", t.subject );
#ifdef EC_DEBUG
            fprintf( stderr, "mail: Subject: %s\n", t.subject );
#endif
        }
        fprintf( sendf, "\n" );

        if ( t.body ) {
            fputs( t.body, sendf );
            fprintf( sendf, "\n" );
#ifdef EC_DEBUG
            fputs( t.body, stderr );
#endif
        }
       
        if ( t.signature && sig.signature ) {
            char *c = sig.signature;
            
            while ( ( *c == '\r' ) || ( *c == '\n' ) ) {
                c++;
            }
            if ( strncmp( c, "--", 2 ) && strncmp( c, "__", 2 ) ) {
                fprintf( sendf, "\n-- \n" );
            }
            fputs( sig.signature, sendf );
            fprintf( sendf, "\n" );
#ifdef EC_DEBUG
            fputs( sig.signature, stderr );
#endif
        }

        if ( pclose( sendf ) != 0 ) {
            fprintf( stderr, "Error ending sendmail exchange\n" );
            continue;
        }

        if ( !strcmp( GET_CONFIG(c)->sendAction, "delete" ) ) {
            dlp_DeleteRecord( dbi->pilot_socket, dbHandle, 0, recID );
        } else if ( !strcmp( GET_CONFIG(c)->sendAction, 
                             "file" ) ) {
            dlp_WriteRecord( dbi->pilot_socket, dbHandle, attr, recID, 3, 
                             buffer, size, 0);
        }
        free_Mail( &t );
    }
   
    if ( GET_CONFIG(c)->mhDirectory ) {
#ifdef EC_DEBUG
        fprintf( stderr, "Reading inbound mail from %s\n",
                 GET_CONFIG(c)->mhDirectory );
#endif
        
        for( i = 1; ; i++ ) {
            int len;
            int l;
            struct Mail t;
            int mhmsg;
            
            t.to = NULL;
            t.from = NULL;
            t.cc = NULL;
            t.bcc = NULL;
            t.subject = NULL;
            t.replyTo = NULL;
            t.sentTo = NULL;
            t.body = NULL;
            t.dated = 0;
	    
            if ( ( mhmsg = openmhmsg( GET_CONFIG(c)->mhDirectory, i ) ) < 0 ) {
                break;
            }
           
#ifdef EC_DEBUG 
            fprintf( stderr, "Processing message %d", i );
#endif
            
            len = 0;
            while ( ( len < sizeof(buffer) ) &&
                    ( ( l = read( mhmsg, (char *)(buffer+len),
                                  sizeof(buffer)-len ) ) > 0 ) ) {
                len += l;
            }
            buffer[len] = 0;
            
            if ( l < 0 ) {
                fprintf( stderr, "Error processing message %d\n", i );
                break;
            } 
		
	    if (write_message_to_pilot (c, dbi, dbHandle, buffer, i)) {
		rec++;
                if ( strcmp( GET_CONFIG(c)->receiveAction, "delete" ) == 0 ) {
                    char filename[1000];
                    sprintf( filename, "%s/%d", GET_CONFIG(c)->mhDirectory, 
                             i );
                    close( mhmsg );
                    if ( unlink( filename ) ) {
                        fprintf( stderr, "Error deleting message %d\n", i );
                        dupe++;
                    }
                    continue;
                } else {
                    dupe++;
                }
	    }

            close( mhmsg );
        }
    }
    
    if ( GET_CONFIG(c)->mboxFile ) {
	FILE *f;
#ifdef EC_DEBUG
        fprintf( stderr, "Reading inbound mail from %s\n",
                 GET_CONFIG(c)->mboxFile );
#endif
        f = fopen (GET_CONFIG (c)->mboxFile, "r");
	

	if (f) {	
	    fgets (buffer, sizeof (buffer) - 1, f);
	    while (!feof (f) && strncmp (buffer, "From ", 5)) {
		fgets (buffer, sizeof (buffer) - 1, f);
	    }
	    for( i = 1; !feof (f); i++ ) {
		int len;
		int l;
		char *p;
           
#ifdef EC_DEBUG 
		fprintf( stderr, "Processing message %d", i );
#endif
		len = 0;
		while ( ( len < sizeof(buffer) ) &&
			( ( p = fgets ( (char *)(buffer+len),
					sizeof(buffer)-len, f ) ) != 0 ) ) {
		    if (!strncmp (p, "From ", 5)) {
			break;
		    } else {
			len += strlen (p);
		    }
		}
		
		buffer[len] = 0;
		len = 0;
		
		if ( l < 0 ) {
		    fprintf( stderr, "Error processing message %d\n", i );
		    break;
		}
		
		write_message_to_pilot (c, dbi, dbHandle, buffer, i);
	    }
	    fclose (f);
	    if ( strcmp( GET_CONFIG(c)->receiveAction, "delete" ) == 0 ) {
		if ( unlink( GET_CONFIG(c)->mboxFile ) ) {
		    fprintf( stderr, "Error deleting mbox file %s\n", 
			     GET_CONFIG(c)->mboxFile);
		}
	    }
	}   
    }

    free_MailAppInfo( &tai );
    
    dlp_ResetLastSyncPC( dbi->pilot_socket );
    dlp_CloseDB( dbi->pilot_socket, dbHandle );

    return( 0 );
}

gint copy_from_pilot( GnomePilotConduit *c, GnomePilotDBInfo *dbi ) {
    return( synchronize( c, dbi ) );
}

GnomePilotConduit *conduit_get_gpilot_conduit( guint32 pilotId ) 
{
  GtkObject *retval;
  ConduitCfg *cc;

  retval = gnome_pilot_conduit_standard_new("MailDB",0x6d61696c, NULL);

  g_assert(retval != NULL);
  /*
  gtk_signal_connect(retval, "copy_from_pilot", (GtkSignalFunc)copy_from_pilot ,NULL);
    gtk_signal_connect(retval, "copy_to_pilot", (GtkSignalFunc) ,NULL);
    gtk_signal_connect(retval, "merge_to_pilot", (GtkSignalFunc) ,NULL);
  gtk_signal_connect(retval, "merge_from_pilot", (GtkSignalFunc)synchronize ,NULL);
  */
  gtk_signal_connect(retval, "synchronize", (GtkSignalFunc)synchronize ,NULL);

  load_configuration(&cc, pilotId );
  gtk_object_set_data(retval,"conduit_config",(gpointer)cc);
  return GNOME_PILOT_CONDUIT(retval); 
}


void conduit_destroy_gpilot_conduit( GnomePilotConduit *c ) 
{
  ConduitCfg *cc;
  
  cc = GET_CONFIG(c);
  destroy_configuration( &cc );
  gtk_object_destroy(GTK_OBJECT(c));
}

