/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2009 Canonical Services Ltd (www.canonical.com)
 *               2009 Mikkel Kamstrup Erlandsen
 *
 * Authors: Rodrigo Moya <rodrigo.moya@canonical.com>
 *          Mikkel Kamstrup Erlandsen <mikkel.kamstrup@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of version 2 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "couchdb-database-info.h"

struct _CouchdbDatabaseInfo {
	gint ref_count;

	char *dbname;
	gint doc_count;
	gint doc_del_count;
	gint update_seq;
	gboolean compact_running;
	gint disk_size;
};

/*
 * CouchdbDatabaseInfo object
 */

GType
couchdb_database_info_get_type (void)
{
	static GType object_type = 0;

	if (G_UNLIKELY (!object_type))
		object_type = g_boxed_type_register_static (g_intern_static_string ("CouchdbDatabaseInfo"),
							    (GBoxedCopyFunc) couchdb_database_info_ref,
							    (GBoxedFreeFunc) couchdb_database_info_unref);

	return object_type;
}

CouchdbDatabaseInfo *
couchdb_database_info_new (const char *dbname,
			   gint doc_count,
			   gint doc_del_count,
			   gint update_seq,
			   gboolean compact_running,
			   gint disk_size)
{
	CouchdbDatabaseInfo *dbinfo;

	dbinfo = g_slice_new (CouchdbDatabaseInfo);
	dbinfo->ref_count = 1;
	dbinfo->dbname = g_strdup (dbname);
	dbinfo->doc_count = doc_count;
	dbinfo->doc_del_count = doc_del_count;
	dbinfo->update_seq = update_seq;
	dbinfo->compact_running = compact_running;
	dbinfo->disk_size = disk_size;

	return dbinfo;
}

CouchdbDatabaseInfo *
couchdb_database_info_ref (CouchdbDatabaseInfo *dbinfo)
{
	g_return_val_if_fail (dbinfo != NULL, NULL);
	g_return_val_if_fail (dbinfo->ref_count > 0, NULL);

	g_atomic_int_exchange_and_add (&dbinfo->ref_count, 1);

	return dbinfo;
}

void
couchdb_database_info_unref (CouchdbDatabaseInfo *dbinfo)
{
	gint old_ref;

	g_return_if_fail (dbinfo != NULL);
	g_return_if_fail (dbinfo->ref_count > 0);

	old_ref = g_atomic_int_get (&dbinfo->ref_count);
	if (old_ref > 1)
		g_atomic_int_compare_and_exchange (&dbinfo->ref_count, old_ref, old_ref - 1);
	else {
		g_free (dbinfo->dbname);
		g_slice_free (CouchdbDatabaseInfo, dbinfo);
	}
}

const char *
couchdb_database_info_get_dbname (CouchdbDatabaseInfo *dbinfo)
{
	g_return_val_if_fail (dbinfo != NULL, NULL);

	return (const char *) dbinfo->dbname;
}

gint
couchdb_database_info_get_documents_count (CouchdbDatabaseInfo *dbinfo)
{
	g_return_val_if_fail (dbinfo != NULL, 0);

	return dbinfo->doc_count;
}

gint
couchdb_database_info_get_deleted_documents_count (CouchdbDatabaseInfo *dbinfo)
{
	g_return_val_if_fail (dbinfo != NULL, 0);

	return dbinfo->doc_del_count;
}

gint
couchdb_database_info_get_update_sequence (CouchdbDatabaseInfo *dbinfo)
{
	g_return_val_if_fail (dbinfo != NULL, 0);

	return dbinfo->update_seq;
}

gboolean
couchdb_database_info_is_compact_running (CouchdbDatabaseInfo *dbinfo)
{
	g_return_val_if_fail (dbinfo != NULL, FALSE);

	return dbinfo->compact_running;
}

gint
couchdb_database_info_get_disk_size (CouchdbDatabaseInfo *dbinfo)
{
	g_return_val_if_fail (dbinfo != NULL, 0);

	return dbinfo->disk_size;
}

