/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2009 Canonical Services Ltd (www.canonical.com)
 *
 * Authors: Rodrigo Moya <rodrigo.moya@canonical.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of version 2 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include <couchdb-glib.h>

static CouchDB *couchdb;

static void
test_list_databases (void)
{
	GError *error = NULL;
	GSList *dblist;

	dblist = couchdb_list_databases (couchdb, &error);
	g_assert (error == NULL);

	while (dblist != NULL) {
		CouchDBDatabaseInfo *dbinfo;
		GSList *doclist;

		error = NULL;
		dbinfo = couchdb_get_database_info (couchdb, (const char *) dblist->data, &error);
		g_assert (error == NULL);
		g_assert (dbinfo != NULL);
		g_assert (couchdb_database_info_get_dbname (dbinfo) != NULL);

		/* Get list of documents to compare against couchdb_database_info_get_documents_count */
		error = NULL;
		doclist = couchdb_list_documents (couchdb, (const char *) dblist->data, &error);
		g_assert (error == NULL);
		g_assert (g_slist_length (doclist) == couchdb_database_info_get_documents_count (dbinfo));
		couchdb_free_document_list (doclist);

		dblist = g_slist_remove (dblist, dblist->data);
		couchdb_database_info_unref (dbinfo);
	}
}

static void
test_list_documents (void)
{
	GError *error = NULL;
	GSList *dblist;

	dblist = couchdb_list_databases (couchdb, &error);
	g_assert (error == NULL);

	while (dblist != NULL) {
		GSList *doclist;

		error = NULL;
		doclist = couchdb_list_documents (couchdb, (const char *) dblist->data, &error);
		g_assert (error == NULL);

		while (doclist) {
			CouchDBDocumentInfo *doc_info = doclist->data;
			CouchDBDocument *document;
			char *str;

			error = NULL;
			document = couchdb_document_get (couchdb, (const char *) dblist->data,
							 couchdb_document_info_get_docid (doc_info),
							 &error);
			g_assert (error == NULL);
			g_assert (document != NULL);
			g_assert (g_strcmp0 (couchdb_document_info_get_docid (doc_info), couchdb_document_get_id (document)) == 0);
			g_assert (g_strcmp0 (couchdb_document_info_get_revision (doc_info), couchdb_document_get_revision (document)) == 0);

			str = couchdb_document_to_string (document);
			g_assert (str != NULL);
			g_free (str);

			g_object_unref (G_OBJECT (document));

			doclist = g_slist_remove (doclist, doc_info);
			couchdb_document_info_unref (doc_info);
		}

		dblist = g_slist_remove (dblist, dblist->data);
	}
}

int
main (int argc, char *argv[])
{
	g_type_init ();
	g_thread_init (NULL);
	g_test_init (&argc, &argv, NULL);

	/* Initialize data needed for all tests */
	couchdb = couchdb_new (NULL);
	if (!couchdb) {
		g_print ("Could not create CouchDB object\n");
		return -1;
	}

	/* Setup test functions */
	g_test_add_func ("/testcouchdbglib/ListDatabases", test_list_databases);
	g_test_add_func ("/testcouchdbglib/ListDocuments", test_list_documents);

	return g_test_run ();
}
