/*screen-review.c 
 * VERSION : 0.40 (on 0.33)
 * FIXED/DONE:
 *	- no debug messages
 *	- vertical spaces working
 *	- id = 0 for acc_cells that do not belong to AccessibleObjects    
 *	- check y for a text widget (out of the visible arrea or not)
 *      - FIXED:
 * 	FIXME - line 340, (srw_acc_cell_free), for PAGE_TAB, MENU
 *	      - reproducing :1. uncomment 
 *				SPI_ROLE_PAGE_TAB (673)
 *			     2. gtk-demo, line Widget.. Source Info		  
 * 	- changed the "making lines" algorithm
 *	- added FRAME name + other toplevel windows name
 *	- instead of special cases of toplevel windows roles (FRAME, DIALOG..)
 *		use layer == SPI_LAYER_WINDOW
 *	- remove warnnings, check for leaks(unrefs)
 *	- add trailling and leading empty lines
 *	- FLAGS for trailling and leading VSP
 *	- FLAGS for trailling and leading HSP
 *	- removed "//"
 *	- instead of special cases of popup roles (MENU, MENU_ITEM...)
 *		use layer == SPI_LAYER_POPUP
 *	- menu
 *	- fixed clipping (added following case : start_inside = 0 && end_inside =0)
 *	- fixed strip_newline (in order to return correct word_start)
 *	- (srw_text_chunk_list_head_clip) : removed memory leak (free the chunks that are clipped out)
 *	- implemented a cache (0.34)
 *	- implemented a better cache (have to decide what compromise should I do : speed Vs resources)
 *	- changed API
 *	- add SOURCE (the Accessible object) to each AccCell so the user would 
 *	  press the strip sensors and would obtain information about the object
 *	  displayed in the current line at that particular location
 *FIXME/2DO
 *	- add images, scrollbars, etc (objects that don't have a text displayed on screen)
 *	 and figure it out how would be represented on braille
 *	- optimization : do not calculate all the time the baseline ( * 2 / 3), better I should add
 *	  a baseline member to the SRWTextChunk structure
 *	- "lizibilitate" 
 * Copyright 2001, 2002 Sun Microsystems Inc.,
 * Copyright 2001, 2002 Ximian, Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
#include "config.h"
#include "screen-review.h"
#include "cspi/spi.h"
#include "SRMessages.h"

typedef struct _SRWBoundaryRect 
{
	long int	x;
	long int	y;
	long int	width;
	long int	height;
	AccessibleRole 	role; 	/* role of last clipping element */
	gboolean 	isClipped;
	gboolean 	isEmpty;
} SRWBoundaryRect;

typedef struct _SRWTextChunk 
{
	char		*string;
	Accessible	*source;
	int		start_offset;
	int		end_offset;
	SRWBoundaryRect	clip_bounds;
	SRWBoundaryRect	text_bounds;
	SRWBoundaryRect	start_char_bounds;
 	SRWBoundaryRect	end_char_bounds;
	int 		id;
	int		layer;
	gboolean	is_text;
	gboolean	dummy;
} SRWTextChunk;

typedef struct _SRWLine
{
	GList		*cells;
	gint		y1;
	gint		y2;
	gint		baseline;
	gint 		layer;
	
	gboolean	is_empty;
	gint 		cached;
	SRWAccLine	*acc_line;	
	char		*string;		
}SRWLine; 

#define SRW_DELTA_BASELINE_THRESHOLD	4
#define BASELINE_RATIO			0.66
/*__________________________< DEBUG>__________________________________________*/
#undef	SRW_COMPOSITE_DEBUG 
#undef	SRW_BENCHMARKING

#undef	SRW_STRING_DEBUG
#undef	SRW_BOUNDARY_RECT_DEBUG
#undef	SRW_TEXT_CHUNK_DEBUG
#undef	SRW_TEXT_CHUNK_LIST_DEBUG_ 
#undef	SRW_ACC_LINE_DEBUG
#undef	SRW_CLEAN_UP_DEBUG
#undef	SRW_CLIPPING_DEBUG
#undef	SRW_CLIPPING_DEBUG
/*__________________________</DEBUG>__________________________________________*/

/*______________________________< MACROS>_____________________________________*/

#define BOUNDS_CONTAIN_X_BOUNDS(b, p)	( ( (p).x >= (b).x ) 		&&	\
					( ( (p).x + (p).width) <= 		\
                                          ( (b).x + (b).width) )	&& 	\
	                                ( (b).width > 0) 		&& 	\
					( (b).height > 0) )

#define BOUNDS_CONTAIN_Y(b, p)  	( ( (p) > (b)->y) 		&& 	\
					( (p) < ((b)->y + (b)->height) )&&	\
					( (b)->width > 0) 		&& 	\
					( (b)->height > 0) )

#define CHUNK_BOUNDS_BEFORE_START(c, i) ( (i) 	&& 		\
					( (c)->clip_bounds.x < 	\
                                          (i)->clip_bounds.x) )

#define CHUNK_BOUNDS_END_BEFORE_START(c, i) ( (i) 	&& 		\
                                            ( ( (c)->clip_bounds.x + 	\
                                            (c)->clip_bounds.width) < 	\
                                            (i)->clip_bounds.x) )

#define CHUNK_BOUNDS_AFTER_END(c, i) 	( (i) 	&& 			\
					( (c)->clip_bounds.x >= 	\
			                ( (i)->clip_bounds.x + 		\
			                  (i)->clip_bounds.width) ) )

#define CHUNK_BOUNDS_SPANS_END(c, i) 	( (i) 	&& 			\
					( ( (c)->clip_bounds.x + 	\
					  (c)->clip_bounds.width) >= 	\
			                ( (i)->clip_bounds.x + 		\
			                  (i)->clip_bounds.width) ) )

#define IS_CLIPPING_CONTAINER(a) ( ( (a) != SPI_ROLE_PAGE_TAB ) && 	\
				   ( (a) != SPI_ROLE_MENU) )
/*______________________________</MACROS>_____________________________________*/

/*______________________________< GLOBALS>____________________________________*/

static GList		*elements 		= NULL;
static GList		*lines			= NULL;
static GArray 		*lines_index 		= NULL;
static gboolean 	isJava 			= FALSE;
static gint		id 			= 0,
			pixels_per_column 	= G_MAXINT;
static glong		align_flags		= 0;
static SRWBoundaryRect 	clipping_rectangle;
/*______________________________</GLOBALS>____________________________________*/

extern gboolean 
sro_get_from_accessible (Accessible 	*acc, 
			  SRObject	**obj,
			  SRObjectType	type);
/*_________________________< SRWString methods>_______________________________*/
static char *
srw_string_strip_newlines (char *s, long offset, long *start_offset, long *end_offset)
{
    int i, start = 0;
    char *word_start = s;
    
    for (i=0; s && s[i]; ++i)
    {
	if ((s[i] == '\n')  && i > (offset - *start_offset) )
	{
	    s[i] = '\0';
	    *end_offset = *start_offset + i;
	    *start_offset = *start_offset + start;
	    return word_start;
	}
	else
	{
	    if (s[i] == '\n')
	    {
		word_start = &s[i+1];
		start  = i+1;
	    }
	
	}
    }
    *start_offset = *start_offset + start;
    return word_start;
}

static char *
srw_string_guess_clip (SRWTextChunk 	*chunk)
{
	SRWBoundaryRect 	b;
	char 		*s 	= NULL, 
			*sp 	= NULL,
			*ep	= NULL;
	long 		start_offset, 
			end_offset, 
			len;
	if (chunk && chunk->string)
	    sp = chunk->string;
	if (sp) 
	{
	    AccessibleComponent *component = Accessible_getComponent (
						chunk->source);
	    ep 	= sp + (strlen (sp) );
	    len	= g_utf8_strlen (chunk->string, 
				 -1);
	    if (component) 
	    {
		AccessibleComponent_getExtents (component,
						&b.x, 
						&b.y,
						&b.width, 
						&b.height,
						SPI_COORD_TYPE_SCREEN);
		AccessibleComponent_unref (component);
		start_offset 	= len * 
			          (chunk->text_bounds.x - b.x) / 
			    	  b.width;
		end_offset 	= len * 
				  (chunk->text_bounds.x +
				  chunk->text_bounds.width - b.x) / 
				  b.width;
#ifdef SRW_STRING_DEBUG		
		fprintf (stderr, "\nsrw_string_guess_clip: string len=%ld, clipped to %ld-%ld (start-end)",
				 len, 
				 start_offset, 
				 end_offset);
#endif
		len = end_offset - start_offset;
		sp = g_utf8_offset_to_pointer (chunk->string, 
					       start_offset);
		ep = g_utf8_offset_to_pointer (chunk->string, 
					       end_offset);
	    }
	    s = g_new0 (char, ep - sp + 1);
	    s = g_utf8_strncpy (s, sp, len);
	    s [ep - sp] = '\0';
	    g_assert (g_utf8_validate (s, -1, NULL));
	}
	
	return s;
}
/*_________________________</SRWString methods>_______________________________*/


/*_____________________< SRWBoundaryRect methods>________________________________*/
static SRWBoundaryRect **
srw_boundary_rect_new0 (void)
{
	SRWBoundaryRect	**bounds;
	int 		i;
		
	bounds = g_new0 (SRWBoundaryRect *, SPI_LAYER_LAST_DEFINED);
	for (i = 1; i < SPI_LAYER_LAST_DEFINED; ++i) 
	{
		bounds[i] = g_new0 (SRWBoundaryRect, 1);
		bounds[i]->isClipped = FALSE;
		bounds[i]->isEmpty   = FALSE;
	}
	
	return bounds;
}


static SRWBoundaryRect **
srw_boundary_rect_clone (SRWBoundaryRect *bounds[])
{
	SRWBoundaryRect	**bounds_clone;
	int 		i;
		
	bounds_clone = 
		g_new0 (SRWBoundaryRect *, SPI_LAYER_LAST_DEFINED);
	for (i = 1; i < SPI_LAYER_LAST_DEFINED; ++i) 
	{
		bounds_clone[i] = g_new0 (SRWBoundaryRect, 1);
		*bounds_clone[i] = *bounds[i];
	}
	
	return bounds_clone;
}

static void
srw_boundary_rect_free (SRWBoundaryRect **bounds_clone)
{
	int 	i;
	
	for (i = 1; i < SPI_LAYER_LAST_DEFINED; ++i) 
	{
		g_free (bounds_clone[i]);
		bounds_clone[i] = NULL;
	}
	g_free (bounds_clone);
	bounds_clone = NULL;
}

static void
srw_boundary_rect_clip (SRWBoundaryRect	*bounds, 
			SRWBoundaryRect	*clip_bounds)
{
	long int x2 ;
	long int y2 ;

	if (!(bounds && clip_bounds))
	    return;
	    
	x2 = bounds->x + bounds->width;
	y2 = bounds->y + bounds->height;


#ifdef SRW_BOUNDARY_RECT_DEBUG

	fprintf (stderr, "\nsrw_boundary_rect_clip : [x] bounds %ld-%ld, clip_bounds %ld-%ld; ",
		 bounds->x, 
		 x2,
		 clip_bounds->x, 
		 clip_bounds->x + 
		 clip_bounds->width);
	fprintf (stderr, "\nsrw_boundary_rect_clip : [y] bounds %ld-%ld, clip_bounds %ld-%ld; ",
		 bounds->y, 
		 y2,
		 clip_bounds->y, 
		 clip_bounds->y + 
		 clip_bounds->height);
#endif		 
	bounds->x = MAX (bounds->x, clip_bounds->x);
	bounds->y = MAX (bounds->y, clip_bounds->y);
	x2 =  MIN (x2,  clip_bounds->x + clip_bounds->width);
	y2 =  MIN (y2, clip_bounds->y + clip_bounds->height);
	bounds->width  = MAX (x2 - bounds->x, 0);
	bounds->height = MAX (y2 - bounds->y, 0);
	
	if (!bounds->width || !bounds->height)
		bounds->isEmpty = TRUE;
	if (IS_CLIPPING_CONTAINER (bounds->role)) 
	{
		*clip_bounds = *bounds;
	}
#ifdef SRW_BOUNDARY_RECT_DEBUG
	fprintf (stderr, "\nsrw_boundary_rect_clip : [x] bounds %ld-%ld	[y] bounds %ld-%ld",
		 bounds->x, 
		 bounds->x + 
		 bounds->width,
		 bounds->y,
		 bounds->y+
		 bounds->height);
#endif
}

static void
srw_boundary_rect_xclip_head (SRWBoundaryRect	*bounds, 
			      SRWBoundaryRect 	*clip_bounds)
{
	int x2;
	int cx2;
	
	if (!(bounds && clip_bounds))
	    return;
	
	cx2 = clip_bounds->x + clip_bounds->width;
	if (cx2 < bounds->x) 
	    return;
	    
	x2 = bounds->x + bounds->width;
	if (cx2 <= x2)
	     bounds->x = cx2;
	    
	bounds->width = MAX (0, x2 - cx2);
}

static void
srw_boundary_rect_xclip_tail (SRWBoundaryRect 	*bounds, 
			      SRWBoundaryRect 	*clip_bounds)
{
	int x2;

	if (!(bounds && clip_bounds))
	    return;
		
	x2 = bounds->x + bounds->width;
	if (clip_bounds->x > x2) 
	    return;
	
	bounds->width = MAX (0, clip_bounds->x - bounds->x);
}
/*_____________________</SRWBoundaryRect methods>________________________________*/



/*________________________< SRWAccCell methods>__________________________________*/
SRWAccCell *
srw_acc_cell_new (void)
{
    SRWAccCell 	*acc_cell;
    
    acc_cell = g_new0 (SRWAccCell, 1);
    
    return acc_cell;
}

void
srw_acc_cell_free (SRWAccCell *acc_cell)
{
    char *string = NULL;
    if (acc_cell)
    {
	if (acc_cell->ch)
	{
	    string = g_strdup (acc_cell->ch);
	    g_free (acc_cell->ch);
	    acc_cell->ch = NULL;
	}
	/*FIXME : free the SRObject*/
	if (acc_cell->source)
	{
	    sro_release_reference (acc_cell->source);
	}
	g_free (acc_cell);
	acc_cell = NULL;

    }
}

/*________________________</SRWAccCell methods>__________________________________*/



/*________________________< SRWAccLine methods>__________________________________*/
SRWAccLine *
srw_acc_line_new (void)
{
    SRWAccLine 	*acc_line;
    
    acc_line = g_new0 (SRWAccLine , 1);
    acc_line->srw_acc_line =  g_array_new ( TRUE, 
			    		    TRUE, 
			    		    sizeof (SRWAccCell *) );
    return acc_line;
}

void
srw_acc_line_free (SRWAccLine 	*acc_line)
{
    if (!acc_line)
	return;
    if (acc_line->srw_acc_line)
    {
	int i;
	for (i = 0; i < acc_line->srw_acc_line->len; i++)
	{
	    SRWAccCell *acc_cell;
	    acc_cell = g_array_index (acc_line->srw_acc_line, SRWAccCell *, i);

	    srw_acc_cell_free (acc_cell);
	}
	g_array_free (acc_line->srw_acc_line, TRUE);
	acc_line->srw_acc_line = NULL;
	g_free (acc_line);
    }
}

void
srw_acc_line_from_string (SRWAccLine 	*acc_line, 
			  char 		*s, 
			  SRWTextChunk 	*chunk, 
			  glong 	offset,
			  glong 	start_offset)
{
    char	*crt 		= NULL, 
		*nxt 		= NULL, 
		*ch 		= NULL, 
		*s_dup 		= NULL;
    SRWAccCell 	*acc_cell 	= NULL;
    int 	i 		= 0,
		len 		= 0;
    
    
    if (acc_line && acc_line->srw_acc_line)
    {
	s_dup = g_strdup(s);
	crt = s_dup;
	len = g_utf8_strlen (s, -1);

	if (g_utf8_validate (crt, -1, NULL) )
	{
	    while (i < len)
	    {
		nxt = g_utf8_next_char (crt);
		ch  = g_new0 (gchar, nxt - crt + 1);
		ch  = g_utf8_strncpy (ch, crt, nxt - crt + 1);
		ch [nxt - crt] = '\0';
		if (ch && strlen (ch))
		{
		    acc_cell = srw_acc_cell_new ();

		    acc_cell->ch = ch;
		    acc_cell->index = offset + i;
		    if (chunk)
		    {
			acc_cell->id = chunk->id;
			acc_cell->role = chunk->clip_bounds.role;
			sro_get_from_accessible (chunk->source, 
						&(acc_cell->source),
						SR_OBJ_TYPE_VISUAL);
		    }
		    else
		    	acc_cell->id = 0;
		    
		    if (offset + i < start_offset)
			acc_cell->id = SRW_FILL_CELL;/*fill*/
		    else
			if ( (offset + i == start_offset) ||
			    (i == (len - 1) ) )
				acc_cell->id = SRW_DELIMITATOR_CELL;/*delimitator	*/
#ifdef SRW_TEXT_CHUNK_UTF8_DEBUG
		        fprintf (stderr,"\nch =%s index=%d id=%d", 
					acc_cell->ch, 
					acc_cell->index, 
					acc_cell->id);		
#endif	
		    g_array_append_val (acc_line->srw_acc_line, acc_cell);
		}
		else
		{
		    sru_warning ("\nNOT a valid UTF8 string");    		    
		}
		i++;
		crt = nxt;
	    }
	}    
	
	if (s_dup) g_free (s_dup);	
	s_dup = NULL;
	}
}
/*________________________</SRWAccLine methods>__________________________________*/



/*______________________< SRWTextChunk methods>__________________________________*/
static SRWTextChunk *
srw_text_chunk_new0 (char 	*string,
		Accessible	*source,
		SRWBoundaryRect	*clip_bounds,
		SRWBoundaryRect	*text_bounds,
		int		id,
		int		layer)
{
    SRWTextChunk *text_chunk;
    
    text_chunk = g_new0 (SRWTextChunk, 1);

    if (string)
	text_chunk->string = g_strdup(string);
    if (source)
    {
	Accessible_ref (source);
	text_chunk->source = source;
    }
    if (clip_bounds)
	text_chunk->clip_bounds = *clip_bounds;
    if (text_bounds)
	text_chunk->text_bounds = *text_bounds;
	
    text_chunk->layer 	= layer;
    text_chunk->id 	= id;
    text_chunk->dummy 	= FALSE;
    text_chunk->is_text = FALSE;
    return text_chunk;
}

static void
srw_text_chunk_free (SRWTextChunk	*text_chunk)
{
    if (text_chunk)
    {
	if (text_chunk->string)
	{
	    g_free (text_chunk->string);
	    text_chunk->string = NULL;
	}

	if (text_chunk->source)
	{
	    Accessible_unref (text_chunk->source);

	}

	g_free (text_chunk);
	text_chunk = NULL;
    }
}

static SRWTextChunk *
srw_text_chunk_clone (SRWTextChunk	*text_chunk)
{
	SRWTextChunk 	*text_chunk_copy = NULL;
	
	if (!text_chunk)
	    return NULL;
	    
	text_chunk_copy = g_new0 (SRWTextChunk, 1);


	*text_chunk_copy = *text_chunk;

	if (text_chunk->string) 
	{
	    text_chunk_copy->string = g_strdup (text_chunk->string);
	}
	if (text_chunk_copy->source) 
	    Accessible_ref (text_chunk_copy->source);
	

	return text_chunk_copy;
}

static SRWTextChunk *
srw_text_chunk_from_accessible (Accessible	*accessible, 
			        SRWBoundaryRect	*bounds,
			        long 		offset)
{
	SRWTextChunk 		*text_chunk 	= NULL;
	AccessibleText		*text 		= NULL;
	AccessibleRole 		role;
	AccessibleComponent 	*component;
	char 		*s 		= NULL, 
			*temp_string 	= NULL;
	long 		start, end,
			x2, y2;
	gboolean	no_need 	= TRUE,
			skip		= FALSE;
	int 		layer 		= 0;


	Accessible_ref (accessible);
	role = Accessible_getRole (accessible);
	
	component = Accessible_getComponent (accessible);
	layer = AccessibleComponent_getLayer (component); 
	AccessibleComponent_unref (component);
	
	text_chunk = srw_text_chunk_new0 (NULL,
				      accessible,
				      bounds,
				      NULL,
				      0,
				      layer);	

	if (accessible)
	{
	    if (Accessible_isText (accessible)) 
	    {
		text = Accessible_getText (accessible);
		if (!offset)
		{
		    offset = AccessibleText_getOffsetAtPoint (text,
							      bounds->x,
							      bounds->y,
							      SPI_COORD_TYPE_SCREEN);	
		    if (offset > 0) 
			skip = TRUE;
		}
		
		temp_string = AccessibleText_getTextAtOffset (text, offset,
						    SPI_TEXT_BOUNDARY_LINE_START,
						    &start, &end);
#ifdef SRW_TEXT_CHUNK_DEBUG
		fprintf (stderr,"\nsrw_text_chunk_from_accessible : start %d end %d offset %d",
				start,
				end,
				offset);
#endif
		if (temp_string)
		{
		    s = g_strdup (temp_string);
		    SPI_freeString (temp_string);
		}
		
		if (offset >= AccessibleText_getCharacterCount (text) || ((offset > start + 1) && !skip))
		{
		    g_free (s);
		    srw_text_chunk_free (text_chunk);
		    AccessibleText_unref (text);
		    Accessible_unref (accessible);
		    return NULL;
		}
	
#ifdef SRW_TEXT_CHUNK_DEBUG
		fprintf (stderr, "\nsrw_text_chunk_from_accessible: is text |%s| %20s",
				    s, 
				    Accessible_getRoleName(accessible) );
#endif				    
		if (end > start) 
		{

			AccessibleText_getCharacterExtents (
				text, start,
				&text_chunk->start_char_bounds.x,
				&text_chunk->start_char_bounds.y,
				&text_chunk->start_char_bounds.width,
				&text_chunk->start_char_bounds.height,
				SPI_COORD_TYPE_SCREEN);

#ifdef SRW_TEXT_CHUNK_DEBUG
			fprintf (stderr, "\nsrw_text_chunk_from_accessible: %s: start char (%ld) x=%ld y=%ld width=%ld height=%ld; ",
				 s,
				 start,
				 text_chunk->start_char_bounds.x,
				 text_chunk->start_char_bounds.y,
				 text_chunk->start_char_bounds.width,
				 text_chunk->start_char_bounds.height);
#endif
			if (s && strlen (s) && s[strlen (s) - 1] == '\n')
				    end--;
			if (end <= start)
			{
			    end = start + 1;
			}
			AccessibleText_getCharacterExtents (
				text, end-1,
				&text_chunk->end_char_bounds.x,
				&text_chunk->end_char_bounds.y,
				&text_chunk->end_char_bounds.width,
				&text_chunk->end_char_bounds.height,
				SPI_COORD_TYPE_SCREEN);
#ifdef SRW_TEXT_CHUNK_DEBUG			
			fprintf (stderr, "\nsrw_text_chunk_from_accessible: end char (%ld) x=%ld y=%ld width %ld height=%ld",
				 end-1,
				 text_chunk->end_char_bounds.x,
				 text_chunk->end_char_bounds.y,
				 text_chunk->end_char_bounds.width,
				 text_chunk->end_char_bounds.height);
#endif
		}
		/*\n*/
		if ((start == 0 && end == 0) 	|| 
		    start >= end 		||
		    !(s && strlen (s) ) )
		{
			text_chunk->is_text = FALSE;
			g_free (s);
			s = NULL;
		}
		else
		{
		    text_chunk->is_text = TRUE;

		
		    text_chunk->text_bounds.x = MIN (text_chunk->start_char_bounds.x,
						 text_chunk->end_char_bounds.x);
		    text_chunk->text_bounds.y = MIN (text_chunk->start_char_bounds.y,
						 text_chunk->end_char_bounds.y);
		    x2 = MAX (text_chunk->start_char_bounds.x +
			  text_chunk->start_char_bounds.width,
			  text_chunk->end_char_bounds.x +
			  text_chunk->end_char_bounds.width);
		    text_chunk->text_bounds.width = x2 - text_chunk->text_bounds.x;
		    y2 = MAX (text_chunk->start_char_bounds.y +
			  text_chunk->start_char_bounds.height,
			  text_chunk->end_char_bounds.y + 
			  text_chunk->end_char_bounds.height);
		    text_chunk->text_bounds.height = y2 - text_chunk->text_bounds.y;
		    text_chunk->start_offset = start;
		    text_chunk->end_offset = end ;

		    if (text_chunk->text_bounds.width > 0 && s && strlen (s) )
			pixels_per_column = MIN (text_chunk->text_bounds.width / g_utf8_strlen (s, -1),
						pixels_per_column);
		}
		AccessibleText_unref (text);
		no_need = FALSE;
	} 
	if (!text_chunk->is_text)
	{
		if (role == SPI_ROLE_PUSH_BUTTON	||
		    role == SPI_ROLE_RADIO_BUTTON	||
		    role == SPI_ROLE_LIST 		||
		    role == SPI_ROLE_LIST_ITEM 		||
		    role == SPI_ROLE_LABEL 		||
		    role == SPI_ROLE_COLUMN_HEADER 	||
		    role == SPI_ROLE_TABLE_COLUMN_HEADER||
		    role == SPI_ROLE_DIALOG 		||
		    role == SPI_ROLE_COLOR_CHOOSER	||	
		    role == SPI_ROLE_PAGE_TAB 		||
		    role == SPI_ROLE_STATUS_BAR         ||
		    text_chunk->layer == SPI_LAYER_WINDOW||
		    text_chunk->layer == SPI_LAYER_POPUP

		    ) 
		{/*there should be a switch here instead of the fat if*/
			temp_string= Accessible_getName (accessible);
			if (temp_string)
			{
				s = g_strdup (temp_string);
				SPI_freeString (temp_string);
				if ( !(s && strlen (s) ) )
				{	
				    if (s)
					g_free (s);
				    s = g_strdup ("------");
				}
			}
                        /* use name instead if there is no text available*/
			text_chunk->text_bounds = text_chunk->clip_bounds;
			text_chunk->start_offset = 0;
			text_chunk->end_offset = strlen (s);
#ifdef SRW_TEXT_CHUNK_DEBUG
		fprintf (stderr, "\n\nsrw_text_chunk_from_accessible: NOT text |%s| %20s",
				    s, 
				    Accessible_getRoleName(accessible) );
#endif				    
			
			no_need = FALSE;
		}
	if (no_need)
	{
/*
		 fprintf (stderr, "\nno needed role :%s", Accessible_getName (accessible) );
*/		 
		    srw_text_chunk_free (text_chunk);
		    Accessible_unref (accessible);
		    return NULL;
	}
	}
	if (text_chunk->text_bounds.x < text_chunk->clip_bounds.x) 
	{
		text_chunk->text_bounds.x = text_chunk->clip_bounds.x;
		text_chunk->text_bounds.isClipped = TRUE;
	} 
	if ((text_chunk->text_bounds.x +
	     text_chunk->text_bounds.width)
	    > (text_chunk->clip_bounds.x +
	       text_chunk->clip_bounds.width)) 
	{
		text_chunk->text_bounds.width =
			MAX (0, (text_chunk->clip_bounds.x +
				 text_chunk->clip_bounds.width) -
			     text_chunk->text_bounds.x);
		text_chunk->text_bounds.isClipped = TRUE;
	}
/*	have to check y (out of the visible arrea or not)

*/
	if ( !(text_chunk->clip_bounds.y <= text_chunk->text_bounds.y + text_chunk->text_bounds.height/2 &&
	       text_chunk->clip_bounds.y + text_chunk->clip_bounds.height >= 
	       text_chunk->text_bounds.y + text_chunk->text_bounds.height / 2) )
	{
	    if (s)
	    {
		g_free(s);
		s = NULL;
	    }
	    if (text_chunk->is_text && !(text_chunk->clip_bounds.y + text_chunk->clip_bounds.height >= 
	       text_chunk->text_bounds.y + text_chunk->text_bounds.height / 2) )
	    {
		    srw_text_chunk_free (text_chunk);
		    Accessible_unref (accessible);
		    return NULL;
	    }
	    else
		text_chunk->start_offset = text_chunk->end_offset ;	    
	}

	if (s && strlen (s)) 
	{
		if (s[strlen(s)-1] == '\n') s[strlen(s)-1] = ' ';
		text_chunk->string = s;
	} 
	else 
	{
		text_chunk->string = NULL;
	}
    }
    Accessible_unref (accessible);

    return text_chunk;
}


static void
srw_text_chunk_tail_clip (SRWTextChunk	*bottom, 
		      SRWTextChunk	*top)
{
#ifdef SRW_CLIPPING_DEBUG
	fprintf (stderr, "\nsrw_text_chunk_tail_clip : [x] bottom %ld-%ld, top %ld-%ld;",
			bottom->clip_bounds.x,
			bottom->clip_bounds.x + bottom->clip_bounds.width,
			top->clip_bounds.x,
			top->clip_bounds.x + top->clip_bounds.width);
#endif
	srw_boundary_rect_xclip_tail (&bottom->text_bounds, 
			     &top->clip_bounds);
	srw_boundary_rect_xclip_tail (&bottom->clip_bounds, 
			     &top->clip_bounds);
	bottom->text_bounds.isClipped = TRUE;
	bottom->clip_bounds.isClipped = TRUE;
#ifdef SRW_CLIPPING_DEBUG
	fprintf (stderr, "\nsrw_text_chunk_tail_clip: [x] bottom result %ld-%ld\n",
			bottom->clip_bounds.x,
			bottom->clip_bounds.x + bottom->clip_bounds.width);
#endif
}

static void
srw_text_chunk_head_clip (SRWTextChunk	*bottom, 
		    	  SRWTextChunk 	*top)
{
#ifdef SRW_CLIPPING_DEBUG	
    fprintf (stderr, "\nsrw_text_chunk_head_clip: [x] bottom %ld-%ld, top %ld-%ld;",
		 bottom->clip_bounds.x,
		 bottom->clip_bounds.x + bottom->clip_bounds.width,
		 top->clip_bounds.x,
		 top->clip_bounds.x + top->clip_bounds.width);
#endif	
	srw_boundary_rect_xclip_head (&bottom->text_bounds, 
			     &top->clip_bounds);
	srw_boundary_rect_xclip_head (&bottom->clip_bounds, 
			     &top->clip_bounds);
	bottom->text_bounds.isClipped = TRUE;
	bottom->clip_bounds.isClipped = TRUE;
#ifdef SRW_CLIPPING_DEBUG
	fprintf (stderr, "\nsrw_text_chunk_head_clip: [x] bottom result %ld-%ld\n",
		 bottom->clip_bounds.x,
		 bottom->clip_bounds.x + bottom->clip_bounds.width);
#endif	
}

static inline gboolean
srw_text_chunk_bounds_within (SRWTextChunk	*chunk, 
			  SRWTextChunk 	*test_chunk)
{
	int x1, x2, tx1, tx2;
	gboolean gtx1, ltx2;

	x1 	= chunk->clip_bounds.x;
	x2 	= x1 + chunk->clip_bounds.width;
	tx1 	= test_chunk->clip_bounds.x;
	tx2 	= tx1 + test_chunk->clip_bounds.width;
/*	
	gtx1 	= (chunk->clip_bounds.x >= test_chunk->clip_bounds.x);
	ltx2 	= (chunk->clip_bounds.x + chunk->clip_bounds.width
		<= test_chunk->clip_bounds.x + test_chunk->clip_bounds.width);
*/
	gtx1 	= (x1 >= tx1);
	ltx2 	= (x2 <= tx2);
	
	return gtx1 && ltx2;
}

#define TEXT_CHUNK_BOUNDS_WITHIN(a, b) srw_text_chunk_bounds_within(a, b)

#undef SRW_CHARACTER_CLIP_DEBUG

static char*
srw_text_chunk_get_clipped_substring_by_char (SRWTextChunk	*chunk, 
					  int 		start, 
					  int 		end)
{
	SRWBoundaryRect char_bounds;
	int i;
	char *s;
	GString *string = g_string_new ("");
	gunichar c;
	AccessibleText *text = Accessible_getText (chunk->source);
	for (i = start; i < end; ++i) 
	{
		AccessibleText_getCharacterExtents (text,
						    i,
						    &char_bounds.x,
						    &char_bounds.y,
						    &char_bounds.width,
						    &char_bounds.height,
						    SPI_COORD_TYPE_SCREEN);
#ifdef SRW_CLIPPING_SUB_DEBUG
		fprintf (stderr, "\ntesting %d-%d against %d-%d\n",
			 char_bounds.x, char_bounds.x+char_bounds.width,
			 chunk->text_bounds.x,
			 chunk->text_bounds.x + chunk->text_bounds.width);
#endif
		char_bounds.width = char_bounds.width / 3;
		char_bounds.x += char_bounds.width;	
		if (BOUNDS_CONTAIN_X_BOUNDS (chunk->text_bounds,
					     char_bounds)) 
		{
			c = AccessibleText_getCharacterAtOffset (
				text, i);
#ifdef SRW_CLIPPING_DEBUG
			fprintf (stderr, "[%c]", c);
#endif				
			g_string_append_unichar (string, c);
		}
	}
	AccessibleText_unref (text);

	s = string->str;
	g_string_free (string, FALSE);

	return s;
}

static char*
srw_text_chunk_get_clipped_string (SRWTextChunk	*chunk)
{
	char 	*s 		= NULL, 
		*temp_string 	= NULL,
		*string 	= NULL;
	long 	start, 
		end;
	long 	word_start, 
		word_end, 
		range_end;
	SRWBoundaryRect	start_bounds, 
			end_bounds;
	gboolean 	start_inside, 
			end_inside;
	
	string = g_strdup("");
	if (chunk)
	{
	    start 	= chunk->start_offset;
	    end 	= chunk->end_offset;
	}
	else
	    return NULL;

	if (!chunk->text_bounds.isClipped || 
	    !chunk->string)
	    string = chunk->string;
	else 
	if (chunk->source && chunk->is_text )
	{
		/* while words at offset lie within the bounds, add them */
	    AccessibleText *text = Accessible_getText (chunk->source);
#ifdef SRW_CLIPPING_DEBUG
	    fprintf (stderr, "\nclipping1: |%s|", chunk->string);
#endif
	    do 
	    {
		temp_string = AccessibleText_getTextAtOffset (text,
						    start,
						    SPI_TEXT_BOUNDARY_WORD_END,
						    &word_start,
						    &word_end);
		range_end = word_end;
#ifdef SRW_CLIPPING_DEBUG
		fprintf (stderr, "\nclipping2: |%s|", s);
#endif
		s = g_strdup (temp_string);
		SPI_freeString (temp_string);
		if (s[0] == ' ')
		    word_start++;
#ifdef SRW_CLIPPING_DEBUG
		fprintf (stderr, "\nclipping2 before strip newlines: |%s| \nword_start %ld word_end %ld", 
				    s,
				    word_start,
				    word_end);
#endif		
		s = srw_string_strip_newlines (s, start, &word_start, &word_end);
#ifdef SRW_CLIPPING_DEBUG
		fprintf (stderr, "\nclipping2 after strip newlines: |%s| \nword_start %ld word_end %ld", 
				    s,
				    word_start,
				    word_end);
				    
#endif		
		AccessibleText_getCharacterExtents (text,
					    	    word_start,
						    &start_bounds.x,
						    &start_bounds.y,
						    &start_bounds.width,
						    &start_bounds.height,
						    SPI_COORD_TYPE_SCREEN);
				    
		AccessibleText_getCharacterExtents (text,
						    word_end - 1,
						    &end_bounds.x,
						    &end_bounds.y,
						    &end_bounds.width,
						    &end_bounds.height,
						    SPI_COORD_TYPE_SCREEN);
						    
		start_inside = BOUNDS_CONTAIN_X_BOUNDS (chunk->text_bounds,
							start_bounds);
		end_inside = BOUNDS_CONTAIN_X_BOUNDS (chunk->text_bounds,
						      end_bounds);
#ifdef SRW_CLIPPING_DEBUG		
		fprintf (stderr,"\nstart_inside %d end_inside %d",						      
					    start_inside,
					    end_inside);
#endif		     				    
    		if (start_inside && 
		    end_inside) 
		{
		    /* word is contained in bounds */
		    string = g_strconcat (string, s, NULL);
		} 
		else 
		{ 
/*		   if (start_inside ||
		        end_inside) 
		    {
*/			/* one end of word is in OR both end are out*/
			if (word_end > end) word_end = end;
			s = srw_text_chunk_get_clipped_substring_by_char (
				    chunk,
				    MAX (word_start, 
					 chunk->start_offset),
				    MIN (word_end, 
					 chunk->end_offset));
			if (s)
			    string = g_strconcat (string, 
					      s, 
					      NULL);
/*		    }
*/		
		}
	    start = range_end;
	    } while (start < chunk->end_offset - 1);
	    AccessibleText_unref (text);
	
	} 
	else 
	{ 
	    /* we're clipped, but don't implement AccessibleText :-( */
	    /* guess for now, maybe we can do better someday */
	    string = srw_string_guess_clip (chunk);
	}
#ifdef SRW_CLIPPING_DEBUG
	fprintf (stderr, "\nOUT String clipped :|%s|", string);	
#endif	
	if (string && !strlen (string))
	{
	    g_free (string);
	    return NULL;
	} 
	return string;
}


static char*
srw_text_chunk_pad_string (SRWTextChunk	*chunk,
		       char 		*string, 
		       glong 		offset, 
		       glong		*start_offset,
		       const char 	*pad_chars)
{/*HSP flags*/
	char 	*s ="";
	char 	*cp;
	char 	startbuf[6], 
		padbuf[6], 
		endbuf[6];
	glong 	end_padding = -1;
	static	glong	 leading = 0;
	gint 	howmany;

	howmany = g_unichar_to_utf8 (g_utf8_get_char (pad_chars), startbuf);
	startbuf[howmany] = '\0';
	g_assert (howmany < 7 && howmany > 0);
	
	cp = g_utf8_find_next_char (pad_chars, NULL);
	howmany = g_unichar_to_utf8 (g_utf8_get_char (cp), padbuf);
	padbuf[howmany] = '\0';
	g_assert (howmany < 7 && howmany > 0);
	
	cp = g_utf8_find_next_char (cp, NULL);
	howmany = g_unichar_to_utf8 (g_utf8_get_char (cp), endbuf);
	endbuf[howmany] = '\0';
	g_assert (howmany < 7 && howmany > 0);

	offset--;

	/*|leading embedded  embedded embedded embedded      embedded trailling */
	/*|00000000[_______OK--------]========[________Cancel--------]000000000|*/
	/*|00000000111111111111111111100000000111111111111111111111111000000000|*/
/*	
	fprintf (stderr, "\nchunk %p string %s offset %ld",chunk,string, offset);
*/
	if (chunk)
    	    end_padding = (chunk->clip_bounds.x - clipping_rectangle.x) / 
			    pixels_per_column + 1;/*+1*/ 
	else
	    end_padding = clipping_rectangle.width /
			    pixels_per_column + 1;/*+1*/
	if ((align_flags & SRW_ALIGNF_HSP_ADD_LEADING ) == 0)
	{    
	    if (!offset)
	    {
		leading = end_padding;
	    }
	    else
		if (chunk)
		    end_padding -= leading;	
	}    
	if ( ( ( (align_flags & SRW_ALIGNF_HSP_ADD_LEADING ) != 0) && !offset) ||
	     ( ( (align_flags & SRW_ALIGNF_HSP_ADD_EMBEDDED) != 0) &&  offset && chunk) ||
	     ( ( (align_flags & SRW_ALIGNF_HSP_ADD_TRAILING) != 0) && !chunk ) ) 
	{
	    /*0000 */
	    while (offset < end_padding )/*-1*/ 
	    {
		s = g_strconcat (s, padbuf, NULL); /* could be more efficient */
		++offset;
	    }
	}
	if (chunk && string && strlen (string) )
	{
	    /*[*/
	    /* add left delimitator*/
	    offset ++ ;
	    *start_offset = offset;
	    s = g_strconcat (s, startbuf, NULL);
	}
	/*add fill between left part of the obj and location where text begins*/
        if  (align_flags & SRW_ALIGNF_HSP_ADD_EMBEDDED) 
	{
	    /*__*/
	    if (chunk)
		end_padding = (chunk->text_bounds.x - clipping_rectangle.x) / 
			    pixels_per_column - leading; 
	    else
		end_padding = 0;
	    while (offset < end_padding) 
	    {
		s = g_strconcat (s, padbuf, NULL); /* could be more efficient */
		++offset;
	    }
	}
	/*add text : OK*/
	if (chunk && string && strlen (string))
	{
	    s = g_strconcat (s, string, NULL);
	    offset += g_utf8_strlen (string, -1);
	}

	/*--*/
        if  (align_flags & SRW_ALIGNF_HSP_ADD_EMBEDDED)
	{
	    if (chunk)
		end_padding = (chunk->clip_bounds.x + chunk->clip_bounds.width - clipping_rectangle.x) /
			    pixels_per_column - leading;
	    else
		end_padding = 0;	
	    while (offset < end_padding )/*-1*/ 
	    {
		s = g_strconcat (s, padbuf, NULL); /* could be more efficient */
		++offset;
	    }
	}
	/*]*/
	/* add right delimitator*/
	if (chunk && string && strlen (string) )
	    s = g_strconcat (s, endbuf, NULL);
	
	return s;
}

static char*
srw_text_chunk_to_string (SRWTextChunk 	*chunk, 
		    	 glong 		offset,
		        SRWAccLine	*acc_line)
{
	glong 		start_offset 	= 0;
	char 		*s 		= NULL;
	
	if (chunk /*&& chunk->string*/) 
	{
		s = srw_text_chunk_get_clipped_string (chunk);

	/*	if (chunk->dummy)
			s = srw_text_chunk_pad_string (chunk, s, offset, &start_offset, "   ");
		else
	*/	switch (chunk->clip_bounds.role)
		{
		    case SPI_ROLE_PUSH_BUTTON:
			s = srw_text_chunk_pad_string (chunk, s, offset, &start_offset,"[ ]");
			break;
		    case SPI_ROLE_TEXT:
		    	s = srw_text_chunk_pad_string (chunk, s, offset, &start_offset, "\" \"");
			break;
		    case SPI_ROLE_CHECK_BOX:
		    	s = srw_text_chunk_pad_string (chunk, s, offset, &start_offset, "x |");
			break;
		    case SPI_ROLE_RADIO_BUTTON:
		    	s = srw_text_chunk_pad_string (chunk, s, offset, &start_offset, "o |");
			break;
		    default:
			if (chunk->layer == SPI_LAYER_WINDOW) 
			{
			    s = srw_text_chunk_pad_string (chunk, s, offset, &start_offset, "| |");

			}
			else
			{
			    if (chunk->layer == SPI_LAYER_POPUP) 
				s = srw_text_chunk_pad_string (chunk, s, offset, &start_offset, "< >");		    
			    else
				s = srw_text_chunk_pad_string (chunk, s, offset, &start_offset, "   ");
			}
			break;
		}
		
	}
	else
	{
	    s = srw_text_chunk_pad_string (NULL, s, offset, &start_offset, "! !");
	}
/*
	fprintf (stderr,"\npad string %s",s);
*/
	srw_acc_line_from_string (acc_line, s, chunk, offset, start_offset);

	return s;
}
gint 
srw_text_chunk_compare_layer (gconstpointer  a,
  	            	      gconstpointer  b)
{
    int 	rv = 0;
    SRWTextChunk* 	element_a = (SRWTextChunk *)a;
    SRWTextChunk* 	element_b = (SRWTextChunk *)b;
    
    
    if (element_a && element_b)
    {
	rv =  (element_a->layer % SPI_LAYER_WINDOW) -
	      (element_b->layer % SPI_LAYER_WINDOW);
    	if (rv)
	    return rv;
	else
	    return ( element_a->id - element_b->id);
    }
    else 
	return -1;
}

/*______________________</SRWTextChunk methods>__________________________________*/



/*____________________< SRWTextChunkList methods>________________________________*/

#ifdef SRW_TEXT_CHUNK_LIST_DEBUG_
static void
srw_text_chunk_list_debug (GList	*iter)
{
	SRWTextChunk *chunk;
	fprintf (stderr,"\n<DEBUG>");
	while (iter) 
	{
		chunk = (SRWTextChunk *)iter->data;
		if (chunk)
		fprintf (stderr, "\n%d Chunk %40s id %4d layer %4d, clip %4ld-%4ld [%4ld], text %4ld-%4ld [%4ld] \ns%d e%d baseline %d",
			 chunk->is_text,
			 chunk->string,
			 chunk->id,
			 chunk->layer,
			 chunk->clip_bounds.y,
			 chunk->clip_bounds.y + chunk->clip_bounds.height,
			 chunk->clip_bounds.height,
			 chunk->text_bounds.y,
			 chunk->text_bounds.y + chunk->text_bounds.height,
			 chunk->text_bounds.height,
			 chunk->start_offset,
			 chunk->end_offset,
			(int)( chunk->text_bounds.y + chunk->text_bounds.height*2/3));
		iter = iter->next;
	}
	fprintf (stderr,"\n</DEBUG>");
}
#endif

#ifdef SRW_TEXT_CHUNK_DEBUG

static void
print_chunk_debug (SRWTextChunk *chunk, char *opname, GList *prev, GList *next)
{
	fprintf (stderr, "%sing chunk %s between %s and %s; %ld-%ld\n",
		 opname,
		 chunk->string,
		 (prev ? ((SRWTextChunk *) prev->data)->string : "<null>"),
		 (next ? ((SRWTextChunk *) next->data)->string : "<null>"),
		 chunk->clip_bounds.x,
		 chunk->text_bounds.x + chunk->text_bounds.width);
}
#define PRINT_CHUNK_DEBUG(a, b, c, d) print_chunk_debug(a, b, c, d) 

#endif

static GList *
srw_text_chunk_list_split_insert (GList		*chunk_list, 
			    	  GList 	*iter, 
			          SRWTextChunk	*chunk)
{
	SRWTextChunk *iter_chunk = iter->data;
	SRWTextChunk *iter_copy = srw_text_chunk_clone (iter_chunk);
#ifdef SRW_CLIPPING_DEBUG
	fprintf (stderr, "list_split_insert :***clip insert of %s into %s\n", 
		 chunk->string ? chunk->string : "<null>",
		 iter_chunk->string ? iter_chunk->string : "<null>");
#endif	
	chunk_list = g_list_insert_before (chunk_list, 
					   iter, 
					   iter_copy);
	srw_text_chunk_tail_clip (iter_copy, 
			      chunk);
	chunk_list = g_list_insert_before (chunk_list, 
					   iter, 
					   chunk);
	srw_text_chunk_head_clip (iter_chunk, chunk);

	return chunk_list;
}


static GList *
srw_text_chunk_list_head_clip (GList	*text_chunk_list,
			   SRWTextChunk	*chunk,
			   GList	*next)
{
	GList *target, *iter = next, *prev;
	prev = iter->prev;
	if (chunk->string && strlen (chunk->string)) { 
		text_chunk_list =
			g_list_insert_before (text_chunk_list, 
					      next, 
					      chunk);
	}
	while (iter && 
	       CHUNK_BOUNDS_SPANS_END (chunk, 
	    			      (SRWTextChunk *)iter->data)
	       ) 
	{
#ifdef SRW_CLIPPING_DEBUG
			fprintf (stderr, "\nlist_head_clip : deleting %s\n",
				 ((SRWTextChunk *)iter->data)->string);
#endif			
	    target = iter;
	    iter = iter->next;
	    srw_text_chunk_free ( ( (SRWTextChunk *)(target->data)));		  

	    text_chunk_list = g_list_delete_link (text_chunk_list, 
						  target);


	}
	if (iter && 
	    !CHUNK_BOUNDS_END_BEFORE_START (chunk,
					    (SRWTextChunk *)iter->data)
	   ) 
	{
	    srw_text_chunk_head_clip ( (SRWTextChunk *)iter->data,
				    chunk);
	}
	if (prev &&
	    !CHUNK_BOUNDS_AFTER_END (chunk,
				     (SRWTextChunk *)prev->data)
	    ) 
	{
	    srw_text_chunk_tail_clip ( (SRWTextChunk *)prev->data,
				    chunk);
	}
	
	return text_chunk_list;
}

static GList *
srw_text_chunk_list_clip_and_insert (GList	*text_chunk_list,
				 SRWTextChunk	*chunk,
				 GList		*prev,
				 GList		*next)
{
#ifdef SRW_CLIPPING_DEBUG
/*	if (chunk->string) */
		fprintf (stderr, "\nclip-and-insert for %s, between %s (%ld) and %s (%ld)\n",
			 chunk->string,			 
		 	 (prev && ((SRWTextChunk *)prev->data)->string ? ((SRWTextChunk *)prev->data)->string : "<null>"),
			 (prev ? ((SRWTextChunk *)prev->data)->text_bounds.x : -1),
		 	 (next && ((SRWTextChunk *)next->data)->string ? ((SRWTextChunk *)next->data)->string : "<null>"),
 			 (next ? ((SRWTextChunk *)next->data)->text_bounds.x : -1));
#endif
	/* cases: */
	if (!prev && !next) 
	{ /* first element in, no clip needed */
	    text_chunk_list = g_list_append (text_chunk_list, 
					     chunk);
	} 
	else 
	{ 
	    /* check for clip with prev */
	    /* if we split the prev */
	    if (prev &&
		TEXT_CHUNK_BOUNDS_WITHIN (chunk, 
					  (SRWTextChunk *) prev->data)
		) 
	    {
		text_chunk_list = srw_text_chunk_list_split_insert (text_chunk_list,
								prev, 
								chunk);
	    } 
	    else 
		if (next) 
		{ 
		    /* we split the 'next' element */
		    if (TEXT_CHUNK_BOUNDS_WITHIN (chunk, 
						  (SRWTextChunk *)next->data)) 
		    {
			text_chunk_list = srw_text_chunk_list_split_insert ( 
						text_chunk_list,
					    	next, 
					        chunk);
		    } 
		    else 
		    {
			/* do an insert +  head clip */
			text_chunk_list = srw_text_chunk_list_head_clip (
						text_chunk_list,
						chunk,
						next);
		    }
		} 
		else 
		{
		    if (!CHUNK_BOUNDS_AFTER_END (chunk,
						 (SRWTextChunk *) prev->data)) 
		    {
			srw_text_chunk_tail_clip (prev->data, chunk);
		    }
		    text_chunk_list = g_list_append (text_chunk_list, 
						     chunk);
		}
	}

	return text_chunk_list;
}

static GList *
srw_text_chunk_list_insert_chunk (GList *text_chunk_list, 
			      SRWTextChunk *chunk)
{
	GList *iter = g_list_first (text_chunk_list);
	SRWTextChunk *iter_chunk = NULL;

#ifdef SRW_CLIPPING_DEBUG
	fprintf (stderr, "\nsrw_text_chunk_list_insert_chunk: chunk %s", chunk->string);
#endif	
	if (!chunk->string) 
	{    
	    return text_chunk_list;
	}
	
	do 
	{
	    if (iter) 
		iter_chunk = (SRWTextChunk *) iter->data;
            /* if we're ahead of the current element */
	    if (!iter) 
	    {
		text_chunk_list = srw_text_chunk_list_clip_and_insert (
					 text_chunk_list,
					 chunk,
					 iter,
					 NULL);
		break;
	    } 
	    else 
		if (CHUNK_BOUNDS_BEFORE_START (chunk, 
					       iter_chunk)) 
		{
		    text_chunk_list = srw_text_chunk_list_clip_and_insert (
					text_chunk_list,
					chunk,
					iter->prev,
					iter);
		    break;
		} 
		else 
		    if (!iter->next ) 
		    {
			text_chunk_list = srw_text_chunk_list_clip_and_insert (
					    text_chunk_list,
					    chunk,
					    iter,
					    NULL);
			break;
		    }
	    if (iter) 
		iter = iter->next;
	} while (iter);

/*	fprintf (stderr, "list_insert_chunk has %d chunks\n",
			  g_list_length (text_chunk_list) );
*/
	return text_chunk_list;
}

static char*
srw_text_chunk_list_to_string (GList *iter, 
				SRWAccLine *acc_line,
				gboolean line_empty )
{
	char *s = "|";
	char *string;
	SRWTextChunk *chunk = NULL;

	acc_line->is_empty = line_empty;
	while (iter) 
	{
		chunk = (SRWTextChunk *)iter->data;
		if (chunk) {
			string = srw_text_chunk_to_string (chunk, g_utf8_strlen (s, -1) , acc_line);
			if (string)
				s = g_strconcat (s, string, NULL);
			
		}
		else
		{
		    fprintf (stderr, "\nlist_to_string : chunk is NULL");
		}
		iter = iter->next;
	}

	string = srw_text_chunk_to_string (NULL, g_utf8_strlen (s, -1), acc_line);
	if (string)
		s = g_strconcat (s, string, NULL);

	s = g_strconcat (s,"|",NULL);
	return s;
}

/*_________________________</SRWTextChunkList methods>________________________*/


/*____________________________< SRWLine methods>______________________________*/
SRWLine *
srw_line_add_text_chunk (SRWLine	*line, 
			 SRWTextChunk 	*chunk)
{
    SRWLine 		*new_line	= NULL;
    SRWTextChunk 	*chunk_clone 	= NULL;
    
    if (!line)
    {
    	new_line = g_new0 (SRWLine, 1);
	new_line->is_empty = FALSE;
    }
    else
	new_line = line;
    
    new_line->cached = FALSE;
    chunk_clone = srw_text_chunk_clone (chunk);
    new_line->cells = g_list_append (new_line->cells, 
					    chunk_clone);


    return new_line;
}

static gchar *
srw_line_toplevel_composite (SRWLine	*line, 
			     SRWAccLine	*acc_line)
{
	GList 		*chunk_list 	= NULL, 
			*iter		= NULL;
	SRWTextChunk 	*chunk		= NULL;
    
	    line->cells = g_list_sort (line->cells, srw_text_chunk_compare_layer);
	    iter = line->cells;
	
	    while (iter) 
	    {
		chunk = (SRWTextChunk *) iter->data;
		if (chunk) 
		{
#ifdef SRW_CLIP_DEBUG		
		    fprintf (stderr, "inserting chunk <%s>\n",
				 chunk->string ? chunk->string : "<null>");
#endif
		    chunk_list = srw_text_chunk_list_insert_chunk (chunk_list,
							       chunk);
		}
		iter = iter->next;
	    }
	    line->cells = chunk_list;	
	return srw_text_chunk_list_to_string (chunk_list, 
					  acc_line,
					  line->is_empty);
}

int 
srw_lines_compare_line_number (gconstpointer	a,
  	    	    	       gconstpointer 	b)
{
    int 		rv = 0;
    SRWLine		*element_a = (SRWLine *) a;
    SRWLine		*element_b = (SRWLine *) b;
    
    
    if (element_a && element_b)
    {
	rv = (2 * element_a->y2 + element_a->y1)/ 3 - 
	     (2 * element_b->y2 + element_b->y1)/ 3;
	return rv;
    }
    else 
	fprintf (stderr,"\nThis should not happen.");
    return -1;
}


static int
srw_lines_get_n_lines (GList	**lines,
		       glong	align_flags)
{ 
    int 	n_lines = 0,
		height = 0,
		i = 0,
		j = 0,
		y1 = 0,
		y2 = 0,
		empty_lines = 0;
    SRWLine 	*line_crt = NULL,
		*line_nxt = NULL;
    SRWLine	*aux_line = NULL;
    GList 	*iter =NULL,
		*aux_lines = NULL;

    n_lines = g_list_length (*lines); 
    
    iter = g_list_first (*lines);
    
    if (align_flags & SRW_ALIGNF_VSP_ADD)
    {
	if (align_flags & SRW_ALIGNF_VSP_ADD_LEADING)
	    i = -1;
	while (iter)
	{
	    if (i == -1)
	    {
		line_crt = (SRWLine *)iter->data;
		
		i++;
		y1 = clipping_rectangle.y;
		y2 = line_crt->y1;
		height = line_crt->y2 - line_crt->y1;
	    }
	    else    
	    {
    		if (!(align_flags & SRW_ALIGNF_VSP_ADD_EMBEDDED))		    
		{
		    if (i == 0)
		    {
			iter = g_list_nth (*lines, g_list_length (*lines) - 1 );
			i = n_lines - 1;
		    }
		}


		line_crt = (SRWLine *)iter->data;

		if (line_crt )
		{

		    i++;
		    y1 = line_crt->y2;
		}
		else
		    break;
		if (iter->next && (i != n_lines))
		{
		    line_nxt = (SRWLine *)iter->next->data;
		    y2 = line_nxt->y1;
	    	    height =( (line_crt->y2 - line_crt->y1) +
			  (line_nxt->y2 - line_nxt->y1)) / 2;
		}
		else
		{
		    if (align_flags & SRW_ALIGNF_VSP_ADD_TRAILING)
		    {
			y2 = clipping_rectangle.y + clipping_rectangle.height;
	    		height = line_crt->y2 - line_crt->y1;
		    }
		    else
			break;
		}
	    }
	
		    empty_lines = (int)( (y2 - y1)/height);

		    if (!empty_lines && ((y2 - y1) % height > height/2))
			empty_lines++;
		    if (empty_lines > 0)
		    {
			for (j = 1; j <= empty_lines; j++ )
			{
			    aux_line = g_new0 (SRWLine, 1);
			    aux_line->is_empty = TRUE;
			    aux_line->y1 = (y1 + (j-1) * height);
			    if (j == empty_lines)
			    {
			    	aux_line->y2 = y2; 
			    }
			    else
			    {	
				aux_line->y2 = (y1 + j * height); 
			    }
			    aux_lines = g_list_append (aux_lines, (SRWLine *) aux_line);
			}

		    }
		    if (i > 0)
			iter = iter->next;

	    if (i == n_lines ) break;
	}    
    }

    iter = g_list_first (aux_lines);
    while (iter)
    {
	line_crt = (SRWLine *) iter->data;
	*lines = g_list_insert_sorted (*lines, line_crt, srw_lines_compare_line_number);
	iter = iter->next;
    }
    g_list_free (aux_lines);

    n_lines = g_list_length (*lines);
/*
    fprintf (stderr,"\nn_lines = %d - len %d ",n_lines, g_list_length (lines) );
*/
    return n_lines;
}

static char *
srw_line_generate_output (SRWLine	*line, 
			    int		*y1,
			    int		*y2,
			    SRWAccLine	*acc_line)
{
	char 		*string = NULL;
	    
	    if (line) 
	    {
	    	string = srw_line_toplevel_composite (line, acc_line);
		*y1 = line->y1;
		*y2 = line->y2;
	    }
	return string;
}

static void
srw_lines_free (GList	*lines) 
{
    int 		i = 0;
    SRWLine 		*line;
    SRWTextChunk	*chunk;

#ifdef SRW_CLEAN_UP_DEBUG
    fprintf (stderr, "\n\n<FREE LINES> %p",lines);
#endif
    while (lines)
    {
	i++;
	line = (SRWLine *)lines->data;
#ifdef SRW_CLEAN_UP_DEBUG
	fprintf (stderr, "\n\n<FREE LINE %d>",i);
	srw_text_chunk_list_debug (line->cells);
#endif
	if (line->string)
	{
	    g_free (line->string);
	    line->string = NULL;
	}
	srw_acc_line_free (line->acc_line);
	line->acc_line = NULL;
	while (line->cells) 
	{
	    chunk = (SRWTextChunk *)(line->cells->data);
	    if (chunk)
	    {
		srw_text_chunk_free (chunk);
	    }
	    line->cells = line->cells->next;
	}
	g_list_free (line->cells);
	line->cells = NULL;	
	lines = lines->next;
    }
#ifdef SRW_CLEAN_UP_DEBUG    
    fprintf (stderr, "\n\n</FREE LINES>");
#endif
    g_free (lines);
    lines = NULL; 
}
/*________________________</ScreeReviewLine methods>________________________________*/


/*________________________< SRWElements methods>__________________________________*/

gint 
srw_elements_compare_text_chunk_y (gconstpointer	a,
  	    	    	    	    gconstpointer 	b)
{
    int 		rv = 0;
    SRWTextChunk* 	element_a = (SRWTextChunk *) a;
    SRWTextChunk* 	element_b = (SRWTextChunk *) b;
    
    
    if (element_a && element_b)
    {
	rv = element_a->text_bounds.y +  element_a->text_bounds.height*2/3  - 
	    (element_b->text_bounds.y + element_b->text_bounds.height*2/3);
    	if (rv)
	    return rv;
	else
	    return ( element_a->clip_bounds.x - element_b->clip_bounds.x);
    }
    else 
	return -1;
}

static void
srw_elements_from_accessible (Accessible 	*accessible,  
			      SRWBoundaryRect	*parentClipBounds[],
			      gboolean		parent_selected,
			      int		parent_layer)
{
	Accessible 		*child;
	AccessibleRole		role;
	AccessibleComponent 	*component,
				*child_component;
	AccessibleStateSet	*states;
	SRWBoundaryRect 	bounds;
	SRWBoundaryRect		**clip_bounds;
	SRWTextChunk 		*text_chunk;	
	int 			n_children, 
				child_n,
				layer;	
	long 			offset = 0;
		
	Accessible_ref (accessible);
        clip_bounds = srw_boundary_rect_clone (parentClipBounds);

	states = Accessible_getStateSet (accessible);


	if (AccessibleStateSet_contains (states, SPI_STATE_SHOWING)  ||
	    AccessibleStateSet_contains (states, SPI_STATE_SELECTED)  ) 
	{
	    if (Accessible_isComponent (accessible)) 
	    {
		role = Accessible_getRole (accessible);
		component = Accessible_getComponent (accessible);
		layer = AccessibleComponent_getLayer (component);
		bounds = *clip_bounds[layer];
		if (!bounds.isEmpty) 
		{

			AccessibleComponent_getExtents (component,
							&bounds.x,
							&bounds.y,
							&(bounds.width),
							&(bounds.height),
							SPI_COORD_TYPE_SCREEN);							    
			bounds.role = role;
			if (clip_bounds[layer])
				srw_boundary_rect_clip (&bounds, clip_bounds[layer]);
			offset = 0;

			do
			{	
/*		fprintf (stderr, "\nbounds : %d %d %d %d isEmpty %d",
				    bounds.x,
				    bounds.y,
				    bounds.x + bounds.width,
				    bounds.y + bounds.height,
				    bounds.isEmpty);
*/		
    				text_chunk = srw_text_chunk_from_accessible 
					    (accessible, 
					     &bounds, 
					     offset);
				if (text_chunk)
				{
					offset = text_chunk->end_offset + 1;
					if (text_chunk->string)
					{
						id++;
						text_chunk->id 		= id;

						if (text_chunk->layer == SPI_LAYER_WINDOW)
						{
						    child = Accessible_getChildAtIndex (accessible, 0);
						    child_component = Accessible_getComponent (child);
						    AccessibleComponent_getExtents (child_component,
										    &bounds.x,
										    &bounds.y,
										    &(bounds.width),
										    &(bounds.height),
										    SPI_COORD_TYPE_SCREEN);							    
						    text_chunk->text_bounds.height = bounds.y - text_chunk->text_bounds.y; 
						    AccessibleComponent_unref (child_component);
						    Accessible_unref (child);
						}
						if (text_chunk->layer == SPI_LAYER_POPUP && 
						    parent_layer == SPI_LAYER_POPUP)
						{
						    if (!parent_selected) 
						    {
							srw_text_chunk_free (text_chunk);					
							break;
						    }
						}
						if (!text_chunk->clip_bounds.isEmpty && 
						    text_chunk->text_bounds.height > 0)
						    elements = g_list_insert_sorted (elements, 
										text_chunk, 
										srw_elements_compare_text_chunk_y);
						else
						    srw_text_chunk_free (text_chunk);					
					}
					else
					srw_text_chunk_free (text_chunk);					
				}


			}
			while (text_chunk && text_chunk->is_text);
		} 
		Accessible_unref (component);
	    }
	/*
	 * we always descend into children in case they are in a higher layer
	 * this can of course be optimized for the topmost layer...
	 * but nobody uses that one! (SPI_LAYER_OVERLAY)
	 */
	    n_children = Accessible_getChildCount (accessible);
	    parent_selected = AccessibleStateSet_contains (states, SPI_STATE_SELECTED) || 
			      (!AccessibleStateSet_contains (states, SPI_STATE_SELECTABLE) &&
			       AccessibleStateSet_contains (states, SPI_STATE_SHOWING));
	    parent_layer = layer;
	    for (child_n = 0; child_n < n_children; ++child_n) 
	    {
		child = Accessible_getChildAtIndex (accessible, child_n);
		
		srw_elements_from_accessible (child, 
					      clip_bounds,
					      parent_selected,
					      parent_layer); 
		Accessible_unref (child);
	    }

	}

	AccessibleStateSet_unref (states);

	/*free the parent clip bounds */
	srw_boundary_rect_free (clip_bounds);
	Accessible_unref (accessible);
}

GList*
srw_elements_create_lines (GList *elements, glong align_flags)
{
    SRWTextChunk	*chunk_crt	= NULL,		
			*chunk		= NULL,
			*chunk_clone	= NULL;
    SRWLine		*line		= NULL, 
			*prev_line	= NULL;

    static int		n_lines;
    gboolean 		new_line 	= FALSE;		
    GList		*lines 		= NULL,
    			*iter 		= NULL;
    int 		baseline = 0;

    n_lines = 0;
    
    while (elements)
    {
	chunk_crt = (SRWTextChunk *) elements->data;
	if (chunk_crt && !chunk_crt->text_bounds.isEmpty)
	    baseline = chunk_crt->text_bounds.y + chunk_crt->text_bounds.height * 2 / 3;
	else
	    break;
	    
	if (!line)
	    new_line = TRUE;
	else
	    if (baseline - line->baseline < SRW_DELTA_BASELINE_THRESHOLD)
		new_line = FALSE;
	    else
	    	new_line = TRUE;
	  
	if (new_line)
	{/*new line*/
	    n_lines++;
	
	    if (n_lines > 0)
	    {
/*
    		fprintf (stderr, "\n%s %4d - lines %d ", 
			__FILE__,
			__LINE__,
			n_lines);
*/
	    if (prev_line && line)
	    {
		if ((line->layer < prev_line->layer) &&
		    (line->layer != SPI_LAYER_WINDOW) &&
		    ( prev_line->layer != SPI_LAYER_WINDOW))
		{
		    iter = g_list_first (prev_line->cells);
		    while (iter)
		    {
			chunk = (SRWTextChunk *)iter->data;
			if (!chunk->dummy)
			{
			    chunk_clone = srw_text_chunk_clone (chunk);
			    chunk_clone->text_bounds.x = chunk_clone->clip_bounds.x; 
			    chunk_clone->text_bounds.width = chunk_clone->clip_bounds.width; 
			    chunk_clone->dummy = TRUE;
			    chunk_clone->id =0;
			
			    if (chunk_clone->string)
			    {
			        g_free (chunk_clone->string);
				chunk_clone->string = NULL;
				chunk_clone->string = g_strdup(" ");
			    }

			    if (chunk->layer == prev_line->layer &&
				chunk->clip_bounds.y + chunk->clip_bounds.height > line->y1 /*+ (line->y2-line->y1)*2/3*/)
				line->cells = g_list_append (line->cells, chunk_clone);
			    else
			    {
			        srw_text_chunk_free (chunk_clone);
				chunk_clone = NULL;
		    	    }
			    iter = iter->next;
			}
		    }
		}

		if ((line->layer > prev_line->layer) &&
		    (line->layer != SPI_LAYER_WINDOW) &&
		    (prev_line->layer != SPI_LAYER_WINDOW) )
		{
		    iter = g_list_first (line->cells);
		    while (iter)
		    {
			chunk = (SRWTextChunk *)iter->data;
			if (!chunk->dummy)
			{
			    chunk_clone = srw_text_chunk_clone (chunk);
			    chunk_clone->text_bounds.x = chunk_clone->clip_bounds.x;
			    chunk_clone->text_bounds.width = chunk_clone->clip_bounds.width;
			    chunk_clone->dummy = TRUE;
			    chunk_clone->id =0;
			
			    if (chunk_clone->string)
			    {
				g_free (chunk_clone->string);
				chunk_clone->string = NULL;
				chunk_clone->string = g_strdup(" ");
			    }

			    if (chunk->layer == line->layer &&
				chunk->clip_bounds.y < prev_line->y2 /*- (line->y2-line->y1)*2/3*/)
				prev_line->cells = g_list_append (prev_line->cells, chunk_clone);
			    else
			    {
				srw_text_chunk_free (chunk_clone);
				chunk_clone = NULL;
			    }
			    iter = iter->next;
			}
		    }
		}
/*
	    fprintf (stderr,"\n -------PREV_LINE--------------------------");
	    fprintf (stderr,"\n line->y1 %d, line->y2 %d line->baseline %d line->layer %d",
			    prev_line->y1,
			    prev_line->y2,
			    prev_line->baseline,
			    prev_line->layer);
	    srw_text_chunk_list_debug (prev_line->cells);
	    fprintf (stderr,"\n --------------------------------------\n");

	    fprintf (stderr,"\n -------LINE--------------------------");
	    fprintf (stderr,"\n line->y1 %d, line->y2 %d line->baseline %d line->layer %d",
			    line->y1,
			    line->y2,
			    line->baseline,
			    line->layer);
	    srw_text_chunk_list_debug (line->cells);
	    fprintf (stderr,"\n --------------------------------------\n");
*/
	    }

	    prev_line = line;	
	
	    line = NULL;
	    line = srw_line_add_text_chunk (line,
						chunk_crt);
	    	
	    line->y1 = chunk_crt->text_bounds.y;
	    line->y2 = chunk_crt->text_bounds.y + chunk_crt->text_bounds.height;
	    line->baseline = line->y1 + ( chunk_crt->text_bounds.height)* 2/3;
	    if (line->layer)
		line->layer = MIN (line->layer, chunk_crt->layer );
	    else
		line->layer = chunk_crt->layer;
	    lines = g_list_append (lines, (SRWLine *) line);
/*	
	    fprintf (stderr,"\n -------NEW LINE--------------------------");
	    fprintf (stderr,"\n line->y1 %d, line->y2 %d line->baseline %d line->layer %d",
			    line->y1,
			    line->y2,
			    line->baseline,
			    line->layer);
	    srw_text_chunk_list_debug (line->cells);
	    fprintf (stderr,"\n --------------------------------------\n");
*/	    	
	    }
	    else
		fprintf (stderr, "\nThis should not happen");
	}
	else
	{/*same_line*/

		prev_line = line;
		line->y1 = MIN (prev_line->y1, chunk_crt->text_bounds.y );
		line->y2 = MAX (prev_line->y2,chunk_crt->text_bounds.y + chunk_crt->text_bounds.height);
		line->baseline = (line->baseline + baseline )/2;
		if (line->layer)
		    line->layer = MIN (line->layer, chunk_crt->layer );
		else
		    line->layer = chunk_crt->layer;
		
		line = srw_line_add_text_chunk (prev_line, 
						chunk_crt);
/*
	    fprintf (stderr,"\n ----------SAME LINE------------------");
	    fprintf (stderr,"\n line->y1 %d, line->y2 %d line->baseline %d line->layer %d",
			    line->y1,
			    line->y2,
			    line->baseline,
			    line->layer);
	    srw_text_chunk_list_debug (line->cells);
	    fprintf (stderr,"\n --------------------------------------\n");
*/		
	}

	elements = elements->next;					 
    }

    return lines;
}

static void
srw_elements_free (GList *elements)
{
    SRWTextChunk	*chunk;
    
    if (!elements)
	return;
#ifdef SRW_CLEAN_UP_DEBUG    
    fprintf (stderr, "\n\n<FREE ELEMENTS>");
#endif
    while (elements) 
    {
	    chunk = (SRWTextChunk *)elements->data;
	    if (chunk)
	    {
		srw_text_chunk_free (chunk);
	    }
	    elements = elements->next;
    }
    g_list_free (elements);
    elements = NULL;	
#ifdef SRW_CLEAN_UP_DEBUG    
    fprintf (stderr, "\n</FREE ELEMENTS>");
#endif
}
/*________________________</SRWElements methods>__________________________________*/



/*_______________________________<AUX methods>______________________________________*/
GList *
srw_get_toplevels (void)
{
    Accessible		*desktop, 
			*app;
    AccessibleStateSet	*states;
    GList 		*toplevels 	= NULL, 
			*actives 	= NULL, 
			*iter;
    int 		n_apps, 
			n_toplevels,
			app_n, 
			toplevel_n;

    /* how do we decide which desktop ? */
    desktop = SPI_getDesktop (0);
  
    /* for each app */
    n_apps = Accessible_getChildCount (desktop);
    for (app_n = 0; app_n < n_apps; ++app_n) 
    {
	/* for each toplevel in app */
	app =  Accessible_getChildAtIndex (desktop, app_n);
	if (app)
	{
	    n_toplevels = Accessible_getChildCount (app);
	    for (toplevel_n = 0; toplevel_n < n_toplevels; ++toplevel_n) 
	    {
	        Accessible *toplevel = Accessible_getChildAtIndex (app, toplevel_n);
	        if (Accessible_isComponent (toplevel))
		    toplevels = g_list_prepend (toplevels, toplevel);
		else 
		{
		    Accessible_unref (toplevel);
		    fprintf (stderr, "warning, app toplevel not a component.\n");
		}
	    }
	    Accessible_unref (app);
	}    
    }
    Accessible_unref (desktop); 

    /* sort: at the moment we don't have a good way to sort except to put actives on top */
    for (iter = g_list_first (toplevels); iter; iter = iter->next) 
    {
	Accessible *toplevel =
		  (Accessible *) iter->data;
	states = Accessible_getStateSet (toplevel);
	if (AccessibleStateSet_contains (states,
				   SPI_STATE_ACTIVE)) 
	{
	    actives = g_list_prepend (actives, toplevel);
	}
	AccessibleStateSet_unref (states);
    }

    for (iter = g_list_first (actives); iter; iter = actives->next) 
    {
	toplevels = g_list_remove (toplevels, iter->data); /* place at end */
	toplevels = g_list_append (toplevels, iter->data);
    }
    g_list_free (actives);

    return toplevels;
}
/*_______________________________</AUX methods>______________________________________*/

/*_______________________________< API>_______________________________________*/
int
screen_review_init (SRRectangle *clip_rectangle, 
		    glong 	alignment_flags)
{
 
    Accessible 		*toplevel;
    AccessibleComponent	*component;
    AccessibleStateSet	*states;
    
    SRWBoundaryRect	**clip_bounds;
    SRWBoundaryRect 	toplevel_bounds;

    GList 		*toplevels 	= NULL, 
			*iter		= NULL;
    int 		i,
			n_lines;
#ifdef SRW_BENCHMARKING
    GTimer 		*timer = g_timer_new ();
#endif


  /*new && init*/
    clip_bounds	 = srw_boundary_rect_new0 ();
    id = 0;
    pixels_per_column = G_MAXINT;
    lines_index = NULL;
    clipping_rectangle.x 	= clip_rectangle->x;
    clipping_rectangle.y 	= clip_rectangle->y;
    clipping_rectangle.width 	= clip_rectangle->width;
    clipping_rectangle.height 	= clip_rectangle->height;
    align_flags			= alignment_flags;
/*
    fprintf (stderr, "%s %d : clipping_rectangle [%ld %ld %ld %ld]",
		    __FILE__,
		    __LINE__,
		    clipping_rectangle.x,
		    clipping_rectangle.y,
		    clipping_rectangle.x + clipping_rectangle.width,
		    clipping_rectangle.y + clipping_rectangle.height);
*/		    
  /* for each toplevel, ending with the active one(s),
   * clip against children.
   */
    toplevels = srw_get_toplevels ();
    if (toplevels && g_list_first (toplevels))
    for (iter = g_list_first (toplevels); iter; iter = iter->next) 
    { 
	toplevel = (Accessible *) iter->data;
	if (Accessible_isComponent (toplevel)) 
	{
  	    /* make sure toplevel is visible and not iconified or shaded */
	    states = Accessible_getStateSet (toplevel);
	    if ( (AccessibleStateSet_contains (states, SPI_STATE_VISIBLE) && 
	    	  !AccessibleStateSet_contains (states, SPI_STATE_ICONIFIED) )|| 
		 isJava) 
	    { /* isJava hack! */
		component = Accessible_getComponent (toplevel);
		AccessibleComponent_getExtents (component,
		    			        &toplevel_bounds.x,
						&toplevel_bounds.y,
						&toplevel_bounds.width,
						&toplevel_bounds.height,
						SPI_COORD_TYPE_SCREEN);
		toplevel_bounds.isEmpty = FALSE;
		if ( (toplevel_bounds.x == clip_rectangle->x) &&
		     (toplevel_bounds.y == clip_rectangle->y) &&
		     (toplevel_bounds.width == clip_rectangle->width)&&
		     (toplevel_bounds.height == clip_rectangle->height))
		{    
		    if (toplevel_bounds.x < 0)
		    {
			toplevel_bounds.width += toplevel_bounds.x;
		    	toplevel_bounds.x = 0;
		    }
		    if (toplevel_bounds.y < 0)
		    {
			toplevel_bounds.height += toplevel_bounds.y;		
			toplevel_bounds.y = 0;
		    }			
		    for (i = 1; i < SPI_LAYER_LAST_DEFINED; ++i) 
		    {
		        *clip_bounds[i] = toplevel_bounds;
		    }
		    srw_elements_from_accessible (toplevel, 
						  clip_bounds,
						  TRUE,
						  -1);
#ifdef SRW_TEXT_CHUNK_DEBUG			
		    srw_text_chunk_list_debug (elements);
#endif			
		    lines =  srw_elements_create_lines (elements, align_flags);    

		}
		AccessibleComponent_unref (component);
	    }/*states*/
	    AccessibleStateSet_unref (states);
	}/*isComponent*/
	Accessible_unref (toplevel);
    }
    if (lines) 
    {
	n_lines = srw_lines_get_n_lines (&lines, align_flags);
    }

    g_list_free (toplevels);
    toplevels = NULL;

#ifdef SRW_BENCHMARKING
    g_timer_stop (timer);
    fprintf (stderr, "elapsed time = %f s\n", g_timer_elapsed (timer, NULL));
#endif  
 
 

  return n_lines;
}

SRWAccLine *
screen_review_get_line (int line_number, 
			int *y1, 
			int *y2)
{
    char 	*string		= NULL;
    GList	*line		= NULL;
    SRWLine	*line_data	= NULL;
    SRWAccLine	*acc_line	= NULL;
    

    acc_line = srw_acc_line_new ();

    if (lines)
    {
	line = g_list_nth (lines, line_number - 1);
    }
    else
	return NULL;
    if (line && line->data)
    {	
	line_data = (SRWLine *)line->data;
	if (!line_data)
	    return NULL;
	if (!line_data->cached)
	{
	    string = srw_line_generate_output (line_data, y1, y2, acc_line);
	    line_data->cached = TRUE;
	    line_data->acc_line = acc_line;
	    line_data->string = string;
	}
	else
	{
	    *y1 = line_data->y1;
	    *y2 = line_data->y2;
	    acc_line = line_data->acc_line;
	    string = line_data->string; 
/*	    fprintf (stderr, "\ncached %d %d %p",*y1, *y2, acc_line);*/
	}    
    }
    else 
	return NULL;
	
#ifdef SRW_ACC_LINE_DEBUG
    if (acc_line)
    {
	SRWAccCell 	*acc_cell = NULL;    
	int 		i;

	fprintf (stderr, "\nGET LINE len = %d",acc_line->srw_acc_line->len);

	for (i = 0; i < acc_line->srw_acc_line->len; i++)
	{
    	    acc_cell = g_array_index (acc_line->srw_acc_line, SRWAccCell *, i);
	    if (acc_cell)
		fprintf (stderr,"\nch =%s index=%d id=%d", acc_cell->ch, acc_cell->index, acc_cell->id);		
	}
    }

    fprintf (stderr,"\n%d %s",acc_line->is_empty, string);
#endif
    return acc_line;
}


void
screen_review_terminate (void)
{
    srw_lines_free (lines) ;
    lines = NULL;
    srw_elements_free (elements);
    elements = NULL;
    if (lines_index)
    {
	g_array_free (lines_index, TRUE);
	lines_index = NULL;
    }

}

/*_______________________________</API>_______________________________________*/
