/* gok-word-complete.c
*
* Copyright 2001,2002 Sun Microsystems, Inc.,
* Copyright 2001,2002 University Of Toronto
*
* This library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Library General Public
* License as published by the Free Software Foundation; either
* version 2 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Library General Public License for more details.
*
* You should have received a copy of the GNU Library General Public
* License along with this library; if not, write to the
* Free Software Foundation, Inc., 59 Temple Place - Suite 330,
* Boston, MA 02111-1307, USA.
*/

/*
* To use this thing:
* - Call "gok_wordcomplete_open". If it returns TRUE then you're ready to go.
* - Call "gok_wordcomplete_on (TRUE)" to turn it on.
* - Call "gok_wordcomplete_predict" to make the word predictions.
* - Call "gok_wordcomplete_close" when you're done. 
* - To add a word, call "gok_wordcomplete_add_new_word".
*
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "gok-word-complete.h"
#include "gok-keyslotter.h"
#include "main.h"
#include "gok-log.h"
#include "gok-modifier.h"
#include "gok-data.h"
#include "gok-keyboard.h"


/* the word, or part word, that is getting completed */
static gchar m_WordPart[MAXWORDLENGTH + 2];

static GokOutput* m_pOutputListStart;

/**
* gok_wordcomplete_open
* 
* Opens and initializes the word completor engine.
*
* returns: TRUE if it was opend OK, FALSE if not.
**/
gboolean gok_wordcomplete_open (gchar *directory)
{
	m_pOutputListStart = NULL;
	
	gok_wordcomplete_end_word();
	
	gok_wordcomplete_on (gok_data_get_wordcomplete());
	
	return (WordCompleteOpen(directory));
}

/**
* gok_wordcomplete_close
* 
* Closes the word completor engine.
*
* returns: void
**/
void gok_wordcomplete_close ()
{
	if (m_pOutputListStart == NULL)
	{
		gok_output_delete_all (m_pOutputListStart);
	}

	WordCompleteClose();
}

/**
* gok_wordcomplete_on
* 
* Turns on (or off) the word completor.
* This adds (or removes) the word completion keys to every keyboard that supports 
* word completion.
*
* returns: void.
**/
void gok_wordcomplete_on (gboolean bOnOff)
{
	gok_log_enter();
	gok_keyslotter_on(bOnOff, KEYTYPE_WORDCOMPLETE);
	gok_log_leave();
}

/**
* gok_wordcomplete_change_number_predictions
*
* @Number: Number of prediction keys.
*
* Changes the number of word prediction keys displayed on the keyboard. This
* should be called after the user has selected a new number from the settings
* dialog.
*
* returns: void
**/
void gok_wordcomplete_change_number_predictions (int Number)
{
	gok_keyslotter_change_number_predictions( Number, KEYTYPE_WORDCOMPLETE );
}

/**
* gok_wordcomplete_add_prediction_keys
*
* @pKeyboard: Pointer to the keyboard that gets the new prediction keys.
*
* Adds a row of prediction keys to the given keyboard.
*
* returns: TRUE if the prediction keys were added, FALSE if not.
**/
gboolean gok_wordcomplete_add_prediction_keys (GokKeyboard* pKeyboard)
{
	return gok_keyslotter_add_prediction_keys (pKeyboard, KEYTYPE_WORDCOMPLETE);
}

/**
* gok_wordcomplete_predict
* 
* Makes a prediction. If the currently displayed keyboard is showing prediction
* keys then they are filled in with the predictions.
*
* returns: The number of words predicted.
**/
int gok_wordcomplete_predict (const gchar letter)
{
	GokKeyboard* pKeyboard;
	GokKey* pKey;
	gint count;
	gint lengthWordPart;
	gint statusPrediction;
	gchar* pWordsPredicted;
	gchar* token;
	gchar seperators[] = " ,\t\n";

	/* is word completion turned on? */
	if (gok_data_get_wordcomplete() == FALSE)
	{
		return 0;
	}
	
	/* get the currently displayed keyboard */
	pKeyboard = gok_main_get_current_keyboard();
	
	/* does it support word completion? */
	if (gok_keyboard_get_supports_wordcomplete (pKeyboard) == FALSE)
	{
		return 0;
	}

	/* clear the current word predictions */
	gok_wordcomplete_clear_keys();

	/* is this an end-word character? */
	if ((letter == ' ') ||
		(letter == 13) || /* enter */
		(letter == '\t'))
	{
		/* reset the part word */
		gok_wordcomplete_end_word();		
		return 0;
	}
	
	/* have we reached the maximum length of a part word? */
	lengthWordPart = strlen (m_WordPart);
	if (lengthWordPart >= MAXWORDLENGTH)
	{
		return 0;
	}
		
	/* add the letter to the current part word */
	m_WordPart[lengthWordPart] = letter;
	m_WordPart[lengthWordPart + 1] = 0;

	/* make the prediction */
	statusPrediction = WordCompletePredict (m_WordPart, gok_data_get_num_predictions(), &pWordsPredicted);
	if (statusPrediction != 0)
	{
		return 0;
	}
	
	/* fill in the word completion keys */
	count = 0;
	pKey = pKeyboard->pKeyFirst;
	token = strtok (pWordsPredicted, seperators);
	while (token != NULL)
	{
		/* make sure we're not going over our maximum predictions */
		count++;
		if (count > gok_data_get_num_predictions())
		{
			break;
		}
		
		/* get the next word completion key on the keyboard */
		while (pKey != NULL)
		{
			if (pKey->Type == KEYTYPE_WORDCOMPLETE)
			{
				gok_key_set_button_label (pKey, token);
				gok_key_add_label (pKey, token, MODIFIER_NORMAL);
								
				pKey = pKey->pKeyNext;
				break;
			}
			pKey = pKey->pKeyNext;
		}
		if (pKey == NULL)
		{
			break;
		}
		
		/* get the next word predicted */
		token = strtok (NULL, seperators);
	}
	
	/* change the font size for the word completion keys */
	if (count > 0)
	{
		gok_keyboard_calculate_font_size_group (pKeyboard, FONT_SIZE_GROUP_WORDCOMPLETE, TRUE);
	}
	
	return count;
}

/**
* gok_wordcomplete_end_word
* 
* Resets the part word buffer.
*
* returns: void
**/
void gok_wordcomplete_end_word ()
{
	m_WordPart[0] = 0;
}

/**
* gok_wordcomplete_clear_keys
* 
* Clears the word completion keys on the current keyboard.
* (The keys are still displayed, they just have no label or output.)
*
* returns: void
**/
void gok_wordcomplete_clear_keys ()
{
	GokKeyboard* pKeyboard;
	GokKey* pKey;
	
	/* get the currently displayed keyboard */
	pKeyboard = gok_main_get_current_keyboard();
	
	/* clear the current word predictions */
	pKey = pKeyboard->pKeyFirst;
	while (pKey != NULL)
	{
		if (pKey->Type == KEYTYPE_WORDCOMPLETE)
		{
			gok_key_set_button_label (pKey, "");
			gok_key_add_label (pKey, "", MODIFIER_NORMAL);
			gok_key_set_output (pKey, 0, NULL, 0);
		}
		pKey = pKey->pKeyNext;
	}
}

/**
* gok_wordcomplete_add_new_word
* 
* Adds a new word to the predictor dictionary.
*
* returns: TRUE if the word was added to the dictionary, FALSE if not.
**/
gboolean gok_wordcomplete_add_new_word (const gchar* pWord)
{
	return TRUE;
}

/**
* gok_wordcomplete_get_output
* 
* Gets the output for a word prediction key.
*
* @pKey: Pointer to the word completion key that will be output.
*
* returns: A pointer to the output.
**/
GokOutput* gok_wordcomplete_get_output (GokKey* pKey)
{
	GokOutput* pOutputNew;
	GokOutput* pOutputListEnd;
	gchar* LabelText;
	gchar letterOutput[2];
	gint countPartWord;
	gint countOutputWord;
	gint indexBuffer;
	gint indexOutputWord;
	
	g_assert (pKey != NULL);

	/* make sure this is a word completion key */
	if (pKey->Type != KEYTYPE_WORDCOMPLETE)
	{
		gok_log_x ("Hey, this is not a word completion key!\n");
		return NULL;
	}
	
	if (m_pOutputListStart != NULL)
	{
		gok_output_delete_all (m_pOutputListStart);
	}
	
	/* subtract the part word from the output */
	LabelText = gok_key_get_label (pKey);
	if (LabelText == NULL)
	{
		return NULL;
	}
	
	m_pOutputListStart = NULL;
	pOutputListEnd = NULL;
	
	countPartWord = strlen (m_WordPart);
	countOutputWord = strlen (LabelText);
	if (countOutputWord > countPartWord)
	{
		indexOutputWord = countPartWord;
		
		/* move the word completion to the buffer */
		for (indexBuffer = 0; indexBuffer < (countOutputWord - countPartWord); indexBuffer++)
		{
			/* create a series of outputs for the predicted word */
			letterOutput[0] = LabelText[indexOutputWord++];
			letterOutput[1] = 0;
			pOutputNew = gok_output_new (OUTPUT_KEYSYM, letterOutput, SPI_KEY_SYM);
			if (m_pOutputListStart == NULL)
			{
				m_pOutputListStart = pOutputNew;
				pOutputListEnd = pOutputNew;
			}
			else
			{
				pOutputListEnd->pOutputNext = pOutputNew;
				pOutputListEnd = pOutputNew;
			}
		}
	}

	return m_pOutputListStart;
}

