/* spgs.c
 *
 * Copyright 2001, 2002 Sun Microsystems, Inc.,
 * Copyright 2001, 2002 BAUM Retec, A.G.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <libbonobo.h>
#include <gnome-speech/gnome-speech.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "spgs.h"
#include "spgscb.h"
#include "libsrconf.h"
#include "SRMessages.h"

#define SPGS_DEBUG
#undef SPGS_DEBUG
/*_________________________________< GLOBALS >________________________________*/

typedef struct 
{
    float 	min;
    float	max;
} GSMinMaxType;
typedef enum
{
    GS_PARAM_RATE,
    GS_PARAM_PITCH,
    GS_PARAM_VOLUME
}GSParameter;

typedef struct
{
    GNOME_Speech_SynthesisDriver	driver;
    gint                          	voice_idx;
    GNOME_Speech_Speaker          	speaker;
    BonoboObject        		*callback;
    /*parameters MIN and MAX*/
    GSMinMaxType			rate;
    GSMinMaxType			pitch;
    GSMinMaxType			volume;
} GSSpeaker;

TTSEngine 				*tts_engine;

static gint 				driver_cnt 	= 0;
static gint 				voice_cnt	= 0;

static GHashTable  			*gs_speakers_hash_table;
static GSSpeaker  			*default_speaker;
static GNOME_Speech_Speaker 		current_speaker;
/*_________________________________</GLOBALS >________________________________*/

/*___________________________________< AUX >__________________________________*/
void
check_return_value (CORBA_Environment 	*env,
		    int 		line)
{
/*FIXME*/
    if ( (env) && (env)->_major != CORBA_NO_EXCEPTION)
    {
	sru_warning ("Exception \"%s\" occured at line %d.",
		    bonobo_exception_get_text (env), line);
	CORBA_exception_free (env);
    }
}
/*___________________________________</AUX >__________________________________*/


/*_____________________________< GS CALLBACK >________________________________*/
void
gs_callback (gint 	reason, 
	     gpointer 	client_data)
{
    switch (reason)
    {
	 case GNOME_Speech_speech_callback_speech_started:
#ifdef  SPGS_DEBUG
	    sru_message ("gs_callback - event started"); 
#endif
    	    tts_engine->callback (TTS_EV_SPEECH_STARTED, NULL, client_data);
	    break;

	case GNOME_Speech_speech_callback_speech_ended:
#ifdef  SPGS_DEBUG
	    sru_message ("gs_callback - event ended\n");
#endif
    	    tts_engine->callback (TTS_EV_END_OF_SPEECH, NULL, client_data);
	    break;

	default:
    	    sru_warning ("gs_callback - unknown event\n");
    	    break;
    }
}
/*_____________________________</GS CALLBACK >________________________________*/

/*_____________________________< GS SPEAKER >_________________________________*/
GSSpeaker *
gs_speaker_new0 (GNOME_Speech_SynthesisDriver	driver,
		 gint                          	voice_idx)
{
/*FIXME*/
    GSSpeaker *gss = NULL;
    CORBA_Environment 		ev;

    gss = g_new0 (GSSpeaker, 1);

    gss->driver 	= CORBA_OBJECT_NIL;
    gss->speaker	= CORBA_OBJECT_NIL;

    if (driver)
    {
	/* store the driver */
	CORBA_exception_init (&ev);
/*        
	GNOME_Speech_SynthesisDriver_ref (driver, &ev);
	gss->driver = driver;
*/	
        gss->driver = bonobo_object_dup_ref (driver, &ev); 
	check_return_value (&ev, __LINE__);
    }
    
     /* store the voice index in driver */
    gss->voice_idx = voice_idx;  

    return gss;
}

void
gs_speaker_free (GSSpeaker *gss)
{
/*FIXME*/
    CORBA_Environment 		ev;
    
    if (gss)
    {
#ifdef SPGS_DEBUG
	fprintf (stderr,"\ngs_speaker_free : voice_idx = %d",
			    gss->voice_idx);
#endif
	if (gss->callback != CORBA_OBJECT_NIL)
/*
	    GNOME_Speech_SpeechCallback_unref (gss->callback, NULL);
*/
	    bonobo_object_unref (gss->callback);

	if (gss->driver != CORBA_OBJECT_NIL)
	{
	    CORBA_exception_init (&ev);
/*
	    GNOME_Speech_SynthesisDriver_unref(gss->driver, NULL);
*/
	    bonobo_object_release_unref (gss->driver, &ev);
	    check_return_value (&ev, __LINE__);
	    
	}
	
	if (gss->speaker != CORBA_OBJECT_NIL)
	{
	    CORBA_exception_init (&ev);
	    GNOME_Speech_Speaker_unref(gss->speaker, NULL);
	    check_return_value (&ev, __LINE__);
	}
	g_free (gss);
	gss = NULL;
    }
}

float
gs_speaker_procent_to_units (GSSpeaker	*speaker,
			    GSParameter	parameter,
			    float	procent)
{
    float units = 0.0;

    if (!speaker)
	return 0;
    if (procent > 100)
	procent = 100;
    
    switch (parameter)
    {
	default :
	    units = procent;
	case GS_PARAM_RATE:
	    units = speaker->rate.min + 
		    (speaker->rate.max -
		     speaker->rate.min) *
		     procent / 100;
	    break;
	case GS_PARAM_PITCH:
	    units = speaker->pitch.min + 
		    (speaker->pitch.max -
		     speaker->pitch.min) *
		     procent / 100;
	    break;
	case GS_PARAM_VOLUME:
	    units = speaker->volume.min + 
		    (speaker->volume.max -
		     speaker->volume.min) *
		     procent / 100;
	    break;    
    }
#ifdef SPGS_DEBUG
    sru_message ("\n%d. procents %f , units %f",
		    parameter,
		    procent,
		    units);
#endif
    
    return units;
    
}

void
gs_speaker_to_gconf (gint	voice_n, 
		     char 	*voice_name)
{
    gchar	*voice_key; 

     /* !!! TBI !!! voice_#_language, voice_#_gender*/
    voice_key 	= g_strdup_printf ("voice_%d_name", 
				    voice_n);
	    
    srconf_set_data (voice_key, 
		     CFGT_STRING, 
		     voice_name, 
		     SPEECH_PARAMETER_SECTION);
    
    g_free (voice_key);
}
/*_____________________________</GS SPEAKER >_________________________________*/

/*_______________________< GS SPEAKERS CONTAINER >____________________________*/
void
gs_speakers_init (void)
{
    gs_speakers_hash_table = g_hash_table_new (g_str_hash,
				    	       g_str_equal);
}

void
gs_speakers_flush (gpointer 	key,
		   gpointer 	value,
		   gpointer 	user_data)
{
    if (key)
    {
        g_free (key);
        key = NULL;
    }
	
    gs_speaker_free ( (GSSpeaker *) value);
}

void
gs_speakers_terminate (void)
{

	if (!gs_speakers_hash_table) 
	    return;
	
	g_hash_table_foreach (gs_speakers_hash_table,
			      gs_speakers_flush,
			      NULL);
	g_hash_table_destroy (gs_speakers_hash_table);
	gs_speakers_hash_table = NULL;
}

GSSpeaker *
gs_speakers_get_speaker (gchar 	*key)
{
    GSSpeaker 	*speaker = NULL;

#ifdef  SPGS_DEBUG
    sru_message ("gs_speakers_get_speaker key : %s", key);
#endif

    if (key)
    {
	speaker =  (GSSpeaker*) g_hash_table_lookup (gs_speakers_hash_table, 
						     key);
    }
    
    return speaker;
}

void
gs_speakers_add_speaker (GSSpeaker 	*speaker, 
			 gchar 		*speaker_key)
{
    if (speaker && speaker_key)
    {	
        g_hash_table_insert (gs_speakers_hash_table,
	            	    g_strdup (speaker_key),
                    	    speaker);
		
    }
}

GSSpeaker *
gs_speakers_select_speaker (gchar *voice_name)
{
    CORBA_Environment 		ev;
    BonoboObject 		*bo;
    GNOME_Speech_VoiceInfoList 	*voices;
    GNOME_Speech_ParameterList	*parameters;
    
    GSSpeaker 			*gss = NULL;
    Callback 			*cb;
    gint			param_n;

    CORBA_exception_init (&ev);

    if (!voice_name)
    {
	/* default speaker request */
	gss = default_speaker;
    }
    else
    {
	/* specific speaker requested */
	gss = gs_speakers_get_speaker (voice_name);

	/*a speaker with the given key could not be found, 
	  get the default speaker */
	if (!gss)
	{
    	    sru_warning ("gs_select_speaker - Speaker not found, using the default speaker.");
    	    gss = default_speaker;
	}
    }
    /* Instantiation on demand - 
	do I already instantiated a speaker for this voice
    (!!! TBR !!! could change the params here too ) */

    if (gss && !gss->speaker)
    {
	/* create a speaker with this voice */
	voices = GNOME_Speech_SynthesisDriver_getAllVoices (gss->driver, 
							    &ev);

	if (voices)
	{
    	    gss->speaker = GNOME_Speech_SynthesisDriver_createSpeaker (
			    gss->driver,
        		    &voices->_buffer[gss->voice_idx], 
			    &ev);
	    check_return_value (&ev, __LINE__);

	    if (gss->speaker)
	    {
		parameters = GNOME_Speech_Speaker_getSupportedParameters (gss->speaker,
									&ev);
		check_return_value (&ev, __LINE__);
		/* get min and max values for the parameters in wich we are interestead*/
		for ( param_n = 0; param_n < parameters->_length ; param_n ++)
		{
		    GNOME_Speech_Parameter *param = &(parameters->_buffer[param_n]);
		    
		    if (!param)
			continue;
		    if (!strcmp (param->name, "rate") ) 
		    {
			gss->rate.min = param->min;
			gss->rate.max = param->max;
#ifdef SPGS_DEBUG
			sru_message ("param %s min %.2lf max %.2lf current %.2lf",
					param->name,
					gss->rate.min,
					param->max,
					param->current);  
#endif
		    }
		    if (!strcmp (param->name, "pitch fluctuation") ) 
		    {
			gss->pitch.min = param->min;
			gss->pitch.max = param->max;
#ifdef SPGS_DEBUG
			sru_message ("param %s min %.2lf max %.2lf current %.2lf",
					param->name,
					param->min,
					param->max,
					param->current);  
#endif
		    }
		    if (!strcmp (param->name, "volume") ) 
		    {
			gss->volume.min = param->min;
			gss->volume.max = param->max;
#ifdef SPGS_DEBUG		    
			sru_message ("param %s min %.2lf max %.2lf current %.2lf",
					param->name,
					param->min,
					param->max,
					param->current);  
#endif
		    }
		}
/*FIXME*/		
    		/* register a callback for this speaker */
    		cb = callback_new (gs_callback, NULL);
		bo = BONOBO_OBJECT (cb);

    		gss->callback = bo;
	
		GNOME_Speech_Speaker_registerSpeechCallback (gss->speaker,
							bonobo_object_corba_objref (bo),
							&ev);
		check_return_value (&ev, __LINE__);							
		
		CORBA_free (parameters);
	    }
	    CORBA_free (voices);
	}
    }
    /* !!! TBR !!! should have some sanity check and error reporting here */
    return gss;
}

void
gs_speakers_to_gconf (gint 	voice_cnt)
{
    srconf_set_data ("voice_count", 
		     CFGT_INT, 
		     &voice_cnt, 
		     SPEECH_PARAMETER_SECTION);
}
/*_______________________</GS SPEAKERS CONTAINER >____________________________*/

/*______________________________< GS API >____________________________________*/
void
gs_speak (SRSVoice 	*voice, 
	  SRSText 	*text )
{
    GSSpeaker *gss;
    /* sel current speaker for voice->TTSVoiceName */
    gss = gs_speakers_select_speaker (voice->tts_voice_name);
    if (gss)
	current_speaker = gss->speaker;
    if (current_speaker)
    {
	CORBA_Environment 	ev;
	float 			val;
	    
	CORBA_exception_init (&ev);

	val = gs_speaker_procent_to_units (gss,
					  GS_PARAM_RATE,
					  (float)voice->rate);
	GNOME_Speech_Speaker_setParameterValue (current_speaker, 
						"rate", 
						val, 
						&ev);
	check_return_value (&ev, __LINE__);						

	val = gs_speaker_procent_to_units (gss,
					  GS_PARAM_VOLUME,
					  (float)voice->volume);
	GNOME_Speech_Speaker_setParameterValue (current_speaker, 
						"volume", 
						val, 
						&ev);
	check_return_value (&ev, __LINE__);
	
	val = gs_speaker_procent_to_units (gss,
					  GS_PARAM_PITCH,	
					  (float)voice->pitch);
	GNOME_Speech_Speaker_setParameterValue (current_speaker, 
						"pitch fluctuation", 
						val, 
						&ev);
	check_return_value (&ev, __LINE__);

        if (text && text->text) 
	{
#ifdef SPGS_DEBUG
	sru_message ("(%s %d) : gs_speak \"%s\"",
		    __FILE__,
		    __LINE__,
		    text->text);
#endif
    	    /* resume_text = text->Text; */
    	    GNOME_Speech_Speaker_say (current_speaker, 
				      text->text, 
				      &ev);
	    check_return_value (&ev, __LINE__);
        }
    }	
}

void
gs_shut_up (void)
{
    CORBA_Environment ev;
    CORBA_exception_init (&ev);
    
    if (current_speaker)
    {
	GNOME_Speech_Speaker_stop (current_speaker, 
				   &ev);
    	check_return_value (&ev, __LINE__);
    }
}

void gs_pause ()
{
    CORBA_Environment ev;
    
    CORBA_exception_init (&ev);

    /* WAITING FOR SUPPORT FROM GS */
    check_return_value (&ev, __LINE__);
}

void gs_resume ()
{
    CORBA_Environment ev;
    
    CORBA_exception_init (&ev);

    /* WAITING FOR SUPPORT FROM GS */

    check_return_value (&ev, __LINE__);
}


Bonobo_ServerInfoList *
gs_init_get_gs_servers (void)
{
    Bonobo_ServerInfoList 	*servers;
    CORBA_Environment 		ev;


    CORBA_exception_init (&ev);

    if (!bonobo_init (NULL, NULL) )
    { 
	/*this is failure*/
	sru_warning ("Bonobo initialization failed.");
	return NULL;
    }
    servers = bonobo_activation_query (
		"repo_ids.has ('IDL:GNOME/Speech/SynthesisDriver:0.2')",
		NULL, 
		&ev);

    if (BONOBO_EX (&ev)) 
    {
	sru_warning ("Activation of \"gnome-speech\" server failed.");
	return NULL;
    }
    
    if (!servers) 
    {
	sru_warning ("No \"gnome-speech\" drivers were found.");
	return NULL;
    }

    return servers;
}



CORBA_Object
gs_init_activate_server (Bonobo_ServerInfo 	*info)
{
	CORBA_Object		obj = CORBA_OBJECT_NIL;
	CORBA_Environment	ev;

	/* atempt to activate the server */

	CORBA_exception_init (&ev);
	/*obj = a CORBAobject reference to the newly activated server*/
	obj = bonobo_activation_activate_from_id (info->iid, 
						  0, 
						  NULL, 
						  &ev);
	
	/*check the environment for failure*/
	if (BONOBO_EX (&ev) || obj == CORBA_OBJECT_NIL) 
	{
#ifdef SPGS_DEBUG	
	    sru_warning ("Activation of server %s failed.\n", info->iid);
#endif
	}
	else
	{
	    driver_cnt++;
#ifdef SPGS_DEBUG
	    sru_message ("Activation of server %s succedded. Total of %d successes", 
			 info->iid,
			 driver_cnt);
#endif
	}
	return  obj;
}

int
gs_init (TTSEngine *engine)
{
    gint 			server_n = 0, 
				voice_n	 = 0;
    gchar 			*voice_name;

    CORBA_Environment 		ev;
    CORBA_string 		driver_name;/*, driver_version;*/
    Bonobo_ServerInfoList 	*servers;
    GSSpeaker			*gss;

/*FIXME*/
    /*create internal hash table*/
    gs_speakers_init();
	
    servers = gs_init_get_gs_servers();

    if (!servers)
	return FALSE;
	     
    voice_cnt 	= 0;
    driver_cnt 	= 0;	
    
    for (server_n = 0; server_n < servers->_length; server_n++)
    {		
	Bonobo_ServerInfo 		*info;
	GNOME_Speech_VoiceInfoList 	*voices;
	GNOME_Speech_SynthesisDriver 	driver;
	GSSpeaker 			*gss;


	CORBA_exception_init (&ev);
		
	info = &servers->_buffer[server_n];
	driver = (GNOME_Speech_SynthesisDriver) gs_init_activate_server (info);
	
	if (!driver)
	    continue; /* try next server*/

	driver_name = GNOME_Speech_SynthesisDriver__get_driverName (driver, 
								    &ev);
	check_return_value (&ev, __LINE__);

	/* create a list of available voices in gconf */
	voices = GNOME_Speech_SynthesisDriver_getAllVoices (driver, 
							    &ev);
	if (BONOBO_EX (&ev) 	|| 
	    !voices 		|| 
	    !voices->_length) 
	{
	    sru_warning ("Driver named \"%s\" has no available voices.", 
			 driver_name);
	    continue; /* try next server*/
	}
		
	for (voice_n = 0; voice_n < voices->_length  && driver_name; ++voice_n) 
	{
	    /* store the driver together with the voice index to allow a speaker creation later */
    	    gss = gs_speaker_new0 (driver, voice_n);
    	    
	    voice_name 	= g_strdup_printf ("V%d %s - %s", 
					    voice_n, 
					    voices->_buffer[voice_n].name, 
					    driver_name );
	    gs_speaker_to_gconf (voice_cnt++, voice_name);
    	    gs_speakers_add_speaker (gss, voice_name);

    	    /* select a default voice */
    	    if (server_n <= 1 && voice_n == 0) 
		default_speaker = gss; /* !!!!!!! MEGA HACK !!!!!!!!!!!*/

#ifdef SRSP_DEBUG
	    sru_message("gs_init : Driver \"%s\" voice_name %s",
			driver_name,
			voice_name);
#endif
    	    g_free (voice_name);
	}/*end of voices*/
	
	CORBA_free (voices);
	CORBA_free (driver_name);	
	CORBA_Object_release (driver, NULL);
    } /* end of for server */
    gs_speakers_to_gconf (voice_cnt);
    CORBA_free (servers);

    /* select a default speaker */
    gss = gs_speakers_select_speaker (NULL);
    if (gss)
	current_speaker = gss->speaker;
    tts_engine = engine;
    if (driver_cnt)
    {		
	/* fill the engine structure with the actual functions */
   	tts_engine->shut_up	= gs_shut_up;
	tts_engine->speak	= gs_speak;
        tts_engine->pause	= gs_pause;
        tts_engine->resume	= gs_resume;
	tts_engine->terminate	= gs_terminate;
    }
    
    return (driver_cnt > 0) ? TRUE : FALSE;
}

void
gs_terminate (void)
{
    gs_shut_up ();
    gs_speakers_terminate();
}
/*______________________________</GS API >____________________________________*/
