/*
 *  Copyright (C) 2001-2002 Jorn Baayen 
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "gul-general.h"
#include "gul-gui.h"
#include "pixbuf-cache.h"
#include "eel-gconf-extensions.h"

#include <stdlib.h>
#include <string.h>
#include <gdk/gdk.h>
#include <libgnome/gnome-util.h>
#include <gdk-pixbuf/gdk-pixbuf.h>

/* cache of theme pixmaps */
static GHashTable *gul_pixbufs_cache = NULL;

/* Local prototypes */
static gboolean gul_pixbuf_free_cb (gpointer key, GdkPixbuf *value,
			        gpointer user_data);

/**
 * pixbuf_free_cb: free one pixbuf
 */
static gboolean
gul_pixbuf_free_cb (gpointer key, GdkPixbuf *value, gpointer user_data)
{
	g_free (key);
	g_object_unref (value);
	return TRUE;
}

/**
 * pixbuf_cache_reset: reset the pixbuf cache
 */
void
gul_pixbuf_cache_reset (void)
{
	if (gul_pixbufs_cache != NULL)
	{
		g_hash_table_foreach_remove (gul_pixbufs_cache,
					     (GHRFunc) gul_pixbuf_free_cb, NULL);
	}
}

/**
 * pixbuf_cache_get: get 'n cache a pixbuf
 */
GdkPixbuf *
gul_pixbuf_cache_get (const gchar *filename)
{
	GdkPixbuf *pixbuf = NULL;
	gchar *tmp;

	/* lookup in cache */
	if (gul_pixbufs_cache != NULL)
		pixbuf = g_hash_table_lookup (gul_pixbufs_cache, filename);

	if (pixbuf != NULL)
	{
		g_object_ref (G_OBJECT (pixbuf));
		return pixbuf;
	}

	if (g_path_is_absolute (filename))
	{
		tmp = g_strdup (filename);
	}
	else
	{
		tmp = gul_general_user_file (filename, FALSE);
	}

	if (tmp && g_file_test (tmp, G_FILE_TEST_EXISTS))
	{
		pixbuf = gdk_pixbuf_new_from_file (tmp, NULL);
	}

	g_free (tmp);

	if (pixbuf == NULL)
		return NULL;

	/* cache the created GdkPixbuf */
	if (gul_pixbufs_cache == NULL)
	{
		gul_pixbufs_cache = g_hash_table_new (g_str_hash,
						      g_str_equal);
	}

	g_hash_table_insert (gul_pixbufs_cache, g_strdup (filename),
			     pixbuf);
	
	g_object_ref (G_OBJECT (pixbuf));
	return pixbuf;
}

/**
 * pixbuf_cache_register_stocks: register galeon icons with the gnome stock
 * system so people can theme our icons
 */
void
gul_pixbuf_cache_register_stocks (void)
{
	static gboolean done = FALSE;
	GtkIconFactory *factory;
	gint i;

	GtkStockItem items[] = {
		{ GALEON_STOCK_BOOKMARK      ,"_Bookmark",   0, 0, NULL },
		{ GALEON_STOCK_SEPARATOR     ,"_Separator",  0, 0, NULL },
		{ GALEON_STOCK_FOLDER        ,"_Folder",     0, 0, NULL },
		{ GALEON_STOCK_FOLDER_OPEN   ,"_Folder",     0, 0, NULL },
		{ GALEON_STOCK_DEFAULT       ,"_Default",    0, 0, NULL },								     { GALEON_STOCK_DEFAULT_OPEN  ,"_Default",    0, 0, NULL },
		{ GALEON_STOCK_HISTORY       ,"_History",    0, 0, NULL },
		{ GALEON_STOCK_VIEWSOURCE    ,"_Source",     0, 0, NULL },
		{ GALEON_STOCK_SENDLINK      ,"_Send link",  0, 0, NULL }
	};

	if (done) 
	{
		return;
	}
	done = TRUE;

	gtk_stock_add (items, G_N_ELEMENTS (items));

	factory = gtk_icon_factory_new ();
	gtk_icon_factory_add_default (factory);

	for (i = 0; i < (gint) G_N_ELEMENTS (items); i++)
	{
		GtkIconSet *icon_set;
		GdkPixbuf *pixbuf;
		gchar *fn;

		fn = g_strconcat (items[i].stock_id, ".png", NULL);
		pixbuf = gul_pixbuf_cache_get (fn);
		g_free (fn);

		icon_set = gtk_icon_set_new_from_pixbuf (pixbuf);
		gtk_icon_factory_add (factory, items[i].stock_id, icon_set);
		gtk_icon_set_unref (icon_set);
		
		g_object_unref (G_OBJECT (pixbuf));
	}

	g_object_unref (G_OBJECT (factory));
}
