/*
 *  Copyright (C) 2000-2002 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "config.h"
#include "galeon-shell.h"
#include "galeon-automation.h"
#include "galeon-window.h"
#include "GaleonAutomation.h"

#include <libbonoboui.h>
#include <libgnome/gnome-program.h>
#include <libgnomeui/gnome-ui-init.h>
#include <libgnomeui/gnome-window-icon.h>
#include <libgnomevfs/gnome-vfs-init.h>
#include <glade/glade-init.h>

#define GALEON_FACTORY_OAFIID "OAFIID:GNOME_Galeon_Automation_Factory"

static gboolean
galeon_main_automation_init (void);
static gint
galeon_main_translate_url_arguments (poptContext context, gchar ***urls);
static gboolean
galeon_main_start (gpointer data);

/* FIXME */
GnomeProgram *program;
CORBA_Environment corba_env;			/* Global for downloader	*/
static gboolean open_in_existing      = FALSE;  /* load in existing window?     */
static gboolean open_in_new_tab       = FALSE;  /* force open in a new tab?     */
static gboolean noraise               = FALSE;  /* no raise                     */
static gboolean open_in_new_window    = FALSE;  /* force open in a new window?  */
static gboolean open_fullscreen       = FALSE;  /* open galeon in full screen ? */
static gchar   *session_filename      = NULL;   /* the session filename         */
static gchar   *geometry_string       = NULL;   /* the geometry string          */
static gchar   *bookmark_url          = NULL;   /* the temp bookmark to add     */
static gboolean close_option          = FALSE;  /* --close                      */
static gboolean quit_option           = FALSE;  /* --quit                       */
static gboolean galeon_server_mode    = FALSE;
static gboolean open_as_nautilus_view = FALSE;

static BonoboObject *automation_object;
static gint n_urls; 
static gchar **url;
static gboolean first_instance;

/* command line argument parsing structure */
static struct poptOption popt_options[] =
{
	{ NULL, '\0', POPT_ARG_INCLUDE_TABLE, &bonobo_activation_popt_options, 0, NULL,
	  NULL },
	{ "new-tab", 'n', POPT_ARG_NONE, &open_in_new_tab, 0,
	  N_("Open a new tab in an existing Galeon window"),
	  NULL },
	{ "new-window", 'w', POPT_ARG_NONE, &open_in_new_window, 0,
	  N_("Open a new window in an existing Galeon process"),
	  NULL },
	{ "noraise", '\0', POPT_ARG_NONE, &noraise, 0,
	  N_("Do not raise the window when opening a page in an existing Galeon process"),
	  NULL },
	{ "fullscreen", 'f', POPT_ARG_NONE, &open_fullscreen, 0,
	  N_("Run Galeon in full screen mode"),
	  NULL },
	{ "existing", 'x', POPT_ARG_NONE, &open_in_existing, 0,
	  N_("Attempt to load URL in existing Galeon window"),
	  NULL },
	{ "load-session", 'l', POPT_ARG_STRING, &session_filename, 0,
	  N_("Load the given session file"),
	  N_("FILE") },
	{ "server", 's', POPT_ARG_NONE, &galeon_server_mode, 0,
	  N_("Don't open any windows; instead act as a server "
	     "for quick startup of new Galeon instances"),
	  NULL },		  
	{ "add-bookmark", 't', POPT_ARG_STRING, &bookmark_url,
	  0, N_("Add a bookmark (don't open any window)"), 
	  N_("URL")},
	{ "geometry", 'g', POPT_ARG_STRING, &geometry_string,
	  0, N_("Create the initial window with the given geometry, "
		"see X(1) for the GEOMETRY format"),
	  N_("GEOMETRY")},
	{ "close", 'c', POPT_ARG_NONE, &close_option, 0,
	  N_("Close all Galeon windows"),
	  NULL },
	{ "quit", 'q', POPT_ARG_NONE, &quit_option, 0,
	  N_("Same as --close, but exits server mode too"),
	  NULL },
	{ "nautilus-view", 'v', POPT_ARG_NONE, &open_as_nautilus_view, 0,
	  N_("Used internally by the nautilus view"),
	  NULL },

	/* terminator, must be last */
	{ NULL, 0, 0, NULL, 0, NULL, NULL }
};

int
main (int argc, char *argv[])
{
	poptContext context;
        GValue context_as_value = { 0 };
	GnomeProgram *program;
	
#ifdef ENABLE_NLS
	/* Initialize the i18n stuff */
	bindtextdomain(GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);
	bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
	textdomain(GETTEXT_PACKAGE);
#endif

	program = gnome_program_init (PACKAGE, VERSION,
                                      LIBGNOMEUI_MODULE, argc, argv,
                                      GNOME_PARAM_POPT_TABLE, popt_options,
                                      GNOME_PARAM_HUMAN_READABLE_NAME, _("Galeon"),
				      GNOME_PARAM_APP_DATADIR, DATADIR,
                                      NULL);

        g_object_get_property (G_OBJECT (program),
                               GNOME_PARAM_POPT_CONTEXT,
                               g_value_init (&context_as_value, G_TYPE_POINTER));

        context = g_value_get_pointer (&context_as_value);

	/* load arguments that aren't regular options (urls to load) */
        n_urls = galeon_main_translate_url_arguments (context, &url);

	first_instance = galeon_main_automation_init ();

	if (first_instance)
	{
		gnome_vfs_init ();
		
		glade_gnome_init ();
		
		gnome_window_icon_set_default_from_file (PIXMAP_DIR "/galeon.png");
		
		galeon_shell_new ();
		
		g_idle_add ((GSourceFunc) galeon_main_start, NULL);

		galeon_automation_complete_initialization ();

		bonobo_main ();
		
		gnome_vfs_shutdown ();
	}

	return 0;
}

static gboolean
galeon_main_start (gpointer data)
{
	GNOME_GaleonAutomation gaserver;
	int i;

	CORBA_exception_init (&corba_env);
	
	gaserver = bonobo_activation_activate_from_id ("OAFIID:GNOME_Galeon_Automation",
						       0, NULL, &corba_env);

	/* if none of the open in tab/window/existing options is given, fall
	 * back to open in a new window, otherwise the url can get opened in a
	 * different desktop without the user realizing it */
	if (!open_in_existing && !open_in_new_window && !open_in_new_tab)
	{
		open_in_new_window = TRUE;
	}

	if (gaserver == NULL)
	{
		GtkWidget *dialog;
		dialog = gtk_message_dialog_new 
			(NULL,
                         GTK_DIALOG_MODAL,
                         GTK_MESSAGE_ERROR,
                         GTK_BUTTONS_CLOSE,
                         "Galeon can't be used now. "
                         "Running the command \"bonobo-slay\" "
                         "from the console may fix the problem. If not, "
                         "you can try rebooting the computer or "
                         "installing Galeon again.\n\n"
                         "Bonobo couldn't locate the GNOME_Galeon_Automation.server. ");
		gtk_dialog_run (GTK_DIALOG (dialog));
		
	}
	/* FIXME galeon --server doesnt work when not first istance */
	/* Server mode */
	else if (galeon_server_mode)
	{
		g_object_ref (G_OBJECT(galeon_shell));
	}
	/* load the session if requested */
	else if (session_filename)
	{
		GNOME_GaleonAutomation_loadSession 
			(gaserver, session_filename, &corba_env);
	}
	/* if found and we're given a bookmark to add... */
	else if (bookmark_url != NULL)
	{
		GNOME_GaleonAutomation_addBookmark 
			(gaserver, bookmark_url, &corba_env);
	}
	else if (close_option || quit_option)
	{
		GNOME_GaleonAutomation_quit 
			(gaserver, quit_option, &corba_env);
	}
	/* provided with urls? */
	else if (n_urls == 0 && 
		 !open_as_nautilus_view)
	{
		/* no, open a default window */
		GNOME_GaleonAutomation_loadurl 
			(gaserver, "",
			 geometry_string ?
			 geometry_string : "",
			 open_fullscreen,
			 open_in_existing,
			 open_in_new_window,
			 open_in_new_tab,
			 !noraise,
			 &corba_env);
	}
	else
	{
		/* open all of the urls */
		for (i = 0; i < n_urls; i++)
		{
			GNOME_GaleonAutomation_loadurl 
				(gaserver, url[i],
				 geometry_string ?
				 geometry_string : "",
				 open_fullscreen,
				 open_in_existing,
				 open_in_new_window,
				 open_in_new_tab,
				 !noraise,
				 &corba_env);
		}
	}

	/* Unref so it will exit if no more used */
	if (first_instance)
	{
		g_object_unref (G_OBJECT(galeon_shell));
	}

	if (gaserver)
	{
		bonobo_object_release_unref (gaserver, &corba_env);
	}

	CORBA_exception_free (&corba_env);

#if GTK_CHECK_VERSION(2,2,0)
	gdk_notify_startup_complete ();
#endif 
	return FALSE;
}

static gboolean
galeon_main_automation_init (void)
{
	CORBA_Object factory;

	factory = bonobo_activation_activate_from_id 
		(GALEON_FACTORY_OAFIID,
		 Bonobo_ACTIVATION_FLAG_EXISTING_ONLY, 
		 NULL, NULL);

	if (!factory)
	{
		automation_object = galeon_automation_new ();
		return TRUE;
	}
	else
	{
		galeon_main_start (NULL);
		g_message (_("Galeon already running, using existing process"));
		return FALSE;
	}
}

/**
 * translate_url_arguments: gather URL arguments and expand them fully
 * with realpath if they're filenames
 */
static gint
galeon_main_translate_url_arguments (poptContext context, gchar ***urls)
{
        gchar buffer[PATH_MAX];
        gchar **args;
        gint i, n;

        /* any context remaining? */
        if (context == NULL)
        {
                *urls = NULL;
                return 0;
        }

        /* get the args and check */
        args = (gchar **) poptGetArgs (context);
        if (args == NULL)
        {
                poptFreeContext (context);
                *urls = NULL;
                return 0;
        }

        /* count args */
        for (n = 0; args[n] != NULL; n++)
                /* nothing */;

        /* allocate pointer array */
        *urls = g_new0 (gchar *, n + 1);
        
        /* translate each one */
        for (i = 0; i < n; i++)
        {
                /* try to expand as files */
                if (realpath (args[i], buffer) != NULL)
                {
                        (*urls)[i] = g_strconcat ("file://", buffer, NULL);
                }
                else
                {
                        (*urls)[i] = g_strdup (args[i]);
                }
        }
        poptFreeContext (context);
        (*urls)[i] = NULL;

        /* return the number of urls */
        return n;
}
