/*
 *  Copyright (C) 2001 Philip Langdale
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "outputbuffer.h"

#ifdef NOT_PORTED
#include "prefs.h"
#endif

#include <stdlib.h>
#include <string.h>
#include <libgnome/gnome-i18n.h>

#include <nsIIOService.h>
#include <nsIServiceManager.h>
#include <nsCOMPtr.h>
#include <nsIURI.h>
#include <nsIURL.h>
#include <nsNetCID.h>
#include <nsIStorageStream.h>
#include <nsNetUtil.h>
#include <nsIExternalProtocolService.h>
#include <nsCExternalHandlerService.h>
#include "gUrlCID.h"

#include "BaseHelpProtocolHandler.h"

static NS_DEFINE_CID(kStandardURLCID, NS_STANDARDURL_CID);
static NS_DEFINE_CID(kGnomeHelpURLCID, G_GNOMEHELPURL_CID);

/* Implementation file */
NS_IMPL_ISUPPORTS1 (GBaseHelpProtocolHandler, nsIProtocolHandler)

/* nsIURI newURI (in string aSpec, in nsIURI aBaseURI); */
NS_IMETHODIMP GBaseHelpProtocolHandler::NewURI(const nsACString &aSpec,
					       const char *aOriginCharset,
					       nsIURI *aBaseURI,
					       nsIURI **_retval)
{
	nsresult rv = NS_OK;

	nsCOMPtr<nsIURI> gnomeHelpURI;
	rv = nsComponentManager::CreateInstance(kGnomeHelpURLCID, nsnull,
						NS_GET_IID(nsIURI),
						getter_AddRefs(gnomeHelpURI));
	if (NS_FAILED(rv)) return rv;

	if (aSpec.Equals("info:dir"))
		rv = gnomeHelpURI->SetSpec(NS_LITERAL_CSTRING("toc:info"));
	else
		rv = gnomeHelpURI->SetSpec(aSpec);
	if (NS_FAILED(rv)) return rv;

	return gnomeHelpURI->QueryInterface(NS_GET_IID(nsIURI), (void**)_retval);
}

/* nsIChannel newChannel (in nsIURI aURI); */
NS_IMETHODIMP GBaseHelpProtocolHandler::NewChannel(nsIURI *aURI,
					      nsIChannel **_retval)
{
	nsresult rv;
	mURI = aURI;

	nsCOMPtr<nsIURL> url = do_QueryInterface (mURI, &rv);
	if (NS_FAILED(rv) || !url) return rv;

	rv = url->GetQuery (mSection);
	if (NS_FAILED(rv)) return rv;

	rv = url->GetFilePath (mDocName);
	if (NS_FAILED(rv)) return rv;

	//Converters are badly behaved and seem to invoke the shell with the
	//document name as given. Shell control chars, if processed are a bad
	//exploit. So we strip them out.
	mDocName.StripChars("!;:&|^");

	rv = CreatePage ();
	NS_IF_ADDREF (*_retval = mChannel);
	return rv;
}

NS_METHOD GBaseHelpProtocolHandler::CreateGHelpURI (nsIURI **_retval)
{
	nsresult rv;

	nsCAutoString cSpec;
	cSpec.Assign(mScheme + NS_LITERAL_CSTRING(":") + mDocName);
	if ('.' == cSpec.CharAt(cSpec.Length()-2))
	{
		char cat = cSpec.CharAt(cSpec.Length()-1);
		cSpec.Truncate(cSpec.Length()-2);
		cSpec += '(';
		cSpec += cat;
		cSpec += ')';
	}
	
	nsCOMPtr<nsIURI> simpleURI;
	rv = GBaseProtocolHandler::NewURI (cSpec, nsnull, nsnull,
					   getter_AddRefs(simpleURI));
	if (NS_FAILED(rv) || !simpleURI) return NS_ERROR_FAILURE;

	NS_IF_ADDREF (*_retval = simpleURI);
	return NS_OK;
}

NS_METHOD GBaseHelpProtocolHandler::CreatePage (void)
{
	nsresult rv;

	/* Buffer converter output */
	gchar *argv[6];
	argv[0] = g_strdup (mConverter.get());
	argv[1] = g_strdup (mDocName.get());
	if (mScheme.Equals("info"))
	{
		if (mSection.IsEmpty()) mSection.Assign("Top");
		argv[2] = "-a";
		argv[3] = g_strdup (mSection.get());
		argv[4] = "-g";	//Activate galeon mode in gnome-info2html2
		argv[5] = NULL;
	}
	else if (mScheme.Equals("man"))
	{
		argv[2] = "-n";
		argv[3] = g_strconcat ("man:", mDocName.get(), NULL);
		argv[4] = NULL;
	}
	else
	{
		argv[2] = NULL;
	}

	unsigned char *outbuf;
	gint outbuflen;
	int ret = getOutputFrom(argv, NULL, 0, &outbuf, &outbuflen);
	g_free (argv[0]);
	g_free (argv[1]);
	if (mScheme.Equals("info") || mScheme.Equals("man"))
		g_free (argv[3]);

	if (ret == -1)
	{
		nsCOMPtr<nsIExternalProtocolService> ps = 
			do_GetService (NS_EXTERNALPROTOCOLSERVICE_CONTRACTID, &rv);
		if (NS_FAILED(rv) || !ps) return NS_ERROR_FAILURE;

		PRBool haveHandler;
		ps->ExternalProtocolHandlerExists (mScheme.get(), &haveHandler);

#ifdef NOT_PORTED

		if (haveHandler &&
		    !prefs_galeon_used_by_gnome_for_protocol(mScheme.get()))
		{
			nsCOMPtr<nsIURI> ghelpURI;
			rv = CreateGHelpURI (getter_AddRefs(ghelpURI));
			if (NS_SUCCEEDED(rv))
				ps->LoadUrl (ghelpURI);
			else
				ps->LoadUrl (mURI);
		}
#endif
		return NS_ERROR_FAILURE;
	}
		

	/* open the rendering stream */

	nsCOMPtr<nsIStorageStream> sStream;
	nsCOMPtr<nsIOutputStream> stream;

	rv = NS_NewStorageStream(16384, outbuflen, getter_AddRefs(sStream));
	if (NS_FAILED(rv)) return rv;

	rv = sStream->GetOutputStream(0, getter_AddRefs(stream));
	if (NS_FAILED(rv)) return rv;

	/* render the complete document */
	PRUint32 bytesWriten;
	rv = stream->Write ((const char *)outbuf, outbuflen, &bytesWriten);
	g_free (outbuf);
	if (NS_FAILED(rv)) return rv;

	/* finish the rendering */
	nsCOMPtr<nsIInputStream> iStream;

	rv = sStream->NewInputStream(0, getter_AddRefs(iStream));
	if (NS_FAILED(rv)) return rv;

	rv = NS_NewInputStreamChannel(getter_AddRefs(mChannel), mURI,
				      iStream, NS_LITERAL_CSTRING("text/html"),
				      NS_LITERAL_CSTRING("utf-8"));
	if (NS_FAILED(rv)) return rv;

	return rv;
}

