/* 
 *  Copyright (C) 2002  Ricardo Fernndezs Pascual <ric@users.sourceforge.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "bookmarks-tree-model.h"
#include <gtk/gtktreednd.h>
#include <glib-object.h>

#define NOT_IMPLEMENTED g_warning ("not implemented: " G_STRLOC);

//#define DEBUG_MSG(x) g_print x
#define DEBUG_MSG(x)

#define VALID_ITER(iter, tree_model)	(iter!= NULL && iter->user_data != NULL \
					&& tree_model->stamp == iter->stamp)

static void		gb_tree_model_init		(GbTreeModel *btree_model);
static void		gb_tree_model_class_init	(GbTreeModelClass *btree_model_class);
static void		gb_tree_model_tree_model_init	(GtkTreeModelIface *iface);
static void		gb_tree_model_drag_source_init	(GtkTreeDragSourceIface *iface);
static void		gb_tree_model_drag_dest_init	(GtkTreeDragDestIface *iface);
static void		gb_tree_model_finalize_impl	(GObject *object);
static guint		gb_tree_model_get_flags		(GtkTreeModel *tree_model);
static gint		gb_tree_model_get_n_columns	(GtkTreeModel *tree_model);
static GType		gb_tree_model_get_column_type	(GtkTreeModel *tree_model,
							 gint index);
static gboolean		gb_tree_model_get_iter		(GtkTreeModel *tree_model,
							 GtkTreeIter *iter,
							 GtkTreePath *path);
static GtkTreePath *	gb_tree_model_get_path		(GtkTreeModel *tree_model,
							 GtkTreeIter *iter);
static void		gb_tree_model_get_value		(GtkTreeModel *tree_model,
							 GtkTreeIter *iter,
							 gint column,
							 GValue *value);
static gboolean		gb_tree_model_iter_next		(GtkTreeModel *tree_model,
							 GtkTreeIter *iter);
static gboolean		gb_tree_model_iter_children	(GtkTreeModel *tree_model,
							 GtkTreeIter *iter,
							 GtkTreeIter *parent);
static gboolean		gb_tree_model_iter_has_child	(GtkTreeModel *tree_model,
							 GtkTreeIter *iter);
static gint		gb_tree_model_iter_n_children	(GtkTreeModel *tree_model,
							 GtkTreeIter *iter);
static gboolean		gb_tree_model_iter_nth_child	(GtkTreeModel *tree_model,
							 GtkTreeIter *iter,
							 GtkTreeIter *parent,
							 gint n);
static gboolean		gb_tree_model_iter_parent	(GtkTreeModel *tree_model,
							 GtkTreeIter *iter,
							 GtkTreeIter *child);

static void		gb_tree_model_set_roots		(GbTreeModel *btree_model,
							 GSList *roots);


/* DND interfaces */
static gboolean		gb_tree_model_drag_data_delete	(GtkTreeDragSource *drag_source,
							 GtkTreePath *path);
static gboolean		gb_tree_model_drag_data_get	(GtkTreeDragSource *drag_source,
							 GtkTreePath *path,
							 GtkSelectionData *selection_data);
static gboolean		gb_tree_model_drag_data_received (GtkTreeDragDest *drag_dest,
							  GtkTreePath *dest,
							  GtkSelectionData *selection_data);
static gboolean		gb_tree_model_row_drop_possible	(GtkTreeDragDest *drag_dest,
							 GtkTreePath *dest_path,
							 GtkSelectionData *selection_data);

/* helper functions */
static gboolean		gb_tree_model_is_root		(GbTreeModel *m, GbFolder *f, int *i);
static void		gb_tree_model_descendant_modified_cb (GbFolder *f, GbBookmark *b, gpointer data);
static void 		gb_tree_model_descendant_added_cb (GbFolder *f, GbFolder *p, GbBookmark *b,
							   int pos, gpointer data);
static void 		gb_tree_model_descendant_removed_cb (GbFolder *f, GbFolder *p, GbBookmark *b, 
							     int pos, gpointer data);


static GObjectClass *parent_class = NULL;


GtkType
gb_tree_model_get_type (void)
{
	static GType btree_model_type = 0;
	
	if (!btree_model_type)
	{
		static const GTypeInfo btree_model_info =
			{
				sizeof (GbTreeModelClass),
				NULL,		/* base_init */
				NULL,		/* base_finalize */
				(GClassInitFunc) gb_tree_model_class_init,
				NULL,		/* class_finalize */
				NULL,		/* class_data */
				sizeof (GbTreeModel),
				0, /* n_preallocs */
				(GInstanceInitFunc) gb_tree_model_init
			};
		
		static const GInterfaceInfo tree_model_info =
			{
				(GInterfaceInitFunc) gb_tree_model_tree_model_init,
				NULL,
				NULL
			};
		
		static const GInterfaceInfo drag_source_info =
			{
				(GInterfaceInitFunc) gb_tree_model_drag_source_init,
				NULL,
				NULL
			};
		
		static const GInterfaceInfo drag_dest_info =
			{
				(GInterfaceInitFunc) gb_tree_model_drag_dest_init,
				NULL,
				NULL
			};
		
		btree_model_type = g_type_register_static (G_TYPE_OBJECT, "GbTreeModel", 
							   &btree_model_info, 0);
		
		g_type_add_interface_static (btree_model_type,
					     GTK_TYPE_TREE_MODEL,
					     &tree_model_info);
		g_type_add_interface_static (btree_model_type,
					     GTK_TYPE_TREE_DRAG_SOURCE,
					     &drag_source_info);
		g_type_add_interface_static (btree_model_type,
					     GTK_TYPE_TREE_DRAG_DEST,
					     &drag_dest_info);
		
	}
	
	return btree_model_type;
}

static void
gb_tree_model_class_init (GbTreeModelClass *class)
{
	GObjectClass *object_class;
	
	parent_class = g_type_class_peek_parent (class);
	object_class = (GObjectClass *) class;
	
	object_class->finalize = gb_tree_model_finalize_impl;
}

static void
gb_tree_model_tree_model_init (GtkTreeModelIface *iface)
{
	iface->get_flags = gb_tree_model_get_flags;
	iface->get_n_columns = gb_tree_model_get_n_columns;
	iface->get_column_type = gb_tree_model_get_column_type;
	iface->get_iter = gb_tree_model_get_iter;
	iface->get_path = gb_tree_model_get_path;
	iface->get_value = gb_tree_model_get_value;
	iface->iter_next = gb_tree_model_iter_next;
	iface->iter_children = gb_tree_model_iter_children;
	iface->iter_has_child = gb_tree_model_iter_has_child;
	iface->iter_n_children = gb_tree_model_iter_n_children;
	iface->iter_nth_child = gb_tree_model_iter_nth_child;
	iface->iter_parent = gb_tree_model_iter_parent;
}

static void
gb_tree_model_drag_source_init (GtkTreeDragSourceIface *iface)
{
	iface->drag_data_delete = gb_tree_model_drag_data_delete;
	iface->drag_data_get = gb_tree_model_drag_data_get;
}

static void
gb_tree_model_drag_dest_init (GtkTreeDragDestIface *iface)
{
	iface->drag_data_received = gb_tree_model_drag_data_received;
	iface->row_drop_possible = gb_tree_model_row_drop_possible;
}

static void
gb_tree_model_init (GbTreeModel *btree_model)
{
	btree_model->roots = NULL;
	do
	{
		btree_model->stamp = g_random_int ();
	}
	while (btree_model->stamp == 0);
}

/**
 * gb_tree_model_new:
 * @root: the root bookmark for this model
 *
 * Return value: a new #GbTreeModel
 **/
GbTreeModel *
gb_tree_model_new (GSList *roots)
{
	GbTreeModel *retval;
	
	retval = GB_TREE_MODEL (g_object_new (GB_TYPE_TREE_MODEL, NULL));
	gb_tree_model_set_roots (retval, roots);
	
	return retval;
}

GbTreeModel *
gb_tree_model_new_for_set (GbBookmarkSet *set)
{
	GbTreeModel *ret;
	GSList *l;
	
	g_return_val_if_fail (GB_IS_BOOKMARK_SET (set), NULL);

	l = g_slist_prepend (NULL, set->root);
	ret = gb_tree_model_new (l);
	g_slist_free (l);
	return ret;
}

static void 
gb_tree_model_set_roots (GbTreeModel *btree_model, GSList *roots)
{
	GSList *l;

	/* call this only once, without NULL as roots */
	g_return_if_fail (btree_model->roots == NULL);
	g_return_if_fail (roots != NULL);

	btree_model->roots = g_slist_copy (roots);
	if (btree_model->roots)
	{
		if (GB_BOOKMARK (btree_model->roots->data)->set)
		{
			btree_model->set = GB_BOOKMARK (btree_model->roots->data)->set;
			g_object_ref (G_OBJECT (btree_model->set));
		}
	}
	else
	{
		btree_model->set = NULL;
	}

	for (l = btree_model->roots; l != NULL; l = l->next)
	{
		g_signal_connect (l->data, "descendant-modified", 
				  G_CALLBACK (gb_tree_model_descendant_modified_cb), btree_model);
		g_signal_connect (l->data, "descendant-added", 
				  G_CALLBACK (gb_tree_model_descendant_added_cb), btree_model);
		g_signal_connect (l->data, "descendant-removed", 
				  G_CALLBACK (gb_tree_model_descendant_removed_cb), btree_model);
		g_object_ref (l->data);
	}
}

static void
gb_tree_model_finalize_impl (GObject *object)
{
	GSList *l;
	GbTreeModel *m = GB_TREE_MODEL (object);
	
	DEBUG_MSG (("Finalizing a GbTreeModel\n"));

	if (m->set)
	{
		g_object_unref (G_OBJECT (m->set));
	}
	for (l = m->roots; l != NULL; l = l->next)
	{
		g_signal_handlers_disconnect_matched (l->data, G_SIGNAL_MATCH_DATA, 0, 0, 
						      NULL, NULL, m);
		g_object_unref (G_OBJECT (l->data));
	}
	g_slist_free (m->roots);
	(* parent_class->finalize) (object);
}

/* fulfill the GtkTreeModel requirements */

static guint
gb_tree_model_get_flags (GtkTreeModel *tree_model)
{
	return 0;
}

static gint
gb_tree_model_get_n_columns (GtkTreeModel *tree_model)
{
	return GB_TREE_MODEL_NUM_COLUMS;
}

static GType
gb_tree_model_get_column_type (GtkTreeModel *tree_model,
			       gint index)
{
	g_return_val_if_fail (GB_IS_TREE_MODEL (tree_model), G_TYPE_INVALID);
	g_return_val_if_fail ((index < GB_TREE_MODEL_NUM_COLUMS) && (index >= 0), G_TYPE_INVALID);
	
	switch (index)
	{
	case GB_TREE_MODEL_COL_ICON:
		return GDK_TYPE_PIXBUF;
		break;
	case GB_TREE_MODEL_COL_TITLE:
	case GB_TREE_MODEL_COL_URL:
		return G_TYPE_STRING;
		break;
	default:
		g_assert_not_reached ();
		return G_TYPE_INVALID;
		break;
	}
}

static gboolean
gb_tree_model_get_iter (GtkTreeModel *tree_model,
			GtkTreeIter *iter,
			GtkTreePath *path)
{
	GbTreeModel *btree_model = (GbTreeModel *) tree_model;
	GtkTreeIter parent;
	gint *indices;
	gint depth, i;
	
	g_return_val_if_fail (GB_IS_TREE_MODEL (btree_model), FALSE);
	
	indices = gtk_tree_path_get_indices (path);
	depth = gtk_tree_path_get_depth (path);
	
	g_return_val_if_fail (depth > 0, FALSE);
	
	parent.stamp = btree_model->stamp;
	parent.user_data = btree_model->roots;
	
	if (! gb_tree_model_iter_nth_child (tree_model, iter, &parent, indices[0]))
		return FALSE;
	
	for (i = 1; i < depth; i++)
	{
		parent = *iter;
		if (! gb_tree_model_iter_nth_child (tree_model, iter, &parent, indices[i]))
			return FALSE;
	}
	return TRUE;
}

static GtkTreePath *
gb_tree_model_get_path (GtkTreeModel *tree_model,
			GtkTreeIter *iter)
{
	GtkTreePath *retval;
	GbBookmark *b_it;
	gint i = 0;
	GbTreeModel *btree_model = (GbTreeModel *) tree_model;
	
	g_return_val_if_fail (GB_IS_TREE_MODEL (tree_model), NULL);
	g_return_val_if_fail (iter != NULL, NULL);
	g_return_val_if_fail (iter->user_data != NULL, NULL);
	g_return_val_if_fail (iter->stamp == btree_model->stamp, NULL);
	
	if (iter->user_data == btree_model->roots)
	{
		return gtk_tree_path_new ();
	}
	
	b_it = iter->user_data;

	if (gb_tree_model_is_root (btree_model, (GbFolder *) b_it, &i))
	{
		retval = gtk_tree_path_new ();
		gtk_tree_path_append_index (retval, i);
	}
	else
	{
		GbFolder *parent = b_it->parent;
		GtkTreeIter tmp_iter = *iter;

		g_assert (parent != NULL);
		tmp_iter.user_data = parent;
		
		retval = gb_tree_model_get_path (tree_model,
						 &tmp_iter);
		if (retval != NULL) 
		{
			i = gb_folder_get_child_index (parent, b_it);
			gtk_tree_path_append_index (retval, i);
		}
	}

	return retval;
}


static void
gb_tree_model_get_value (GtkTreeModel *tree_model,
			 GtkTreeIter *iter,
			 gint column,
			 GValue *value)
{
	GbBookmark *b;
	GdkPixbuf *pb;
	g_return_if_fail (GB_IS_TREE_MODEL (tree_model));
	g_return_if_fail (iter != NULL);
	g_return_if_fail (iter->stamp == GB_TREE_MODEL (tree_model)->stamp);
	g_return_if_fail (GB_IS_BOOKMARK (iter->user_data));
	g_return_if_fail (column < GB_TREE_MODEL_NUM_COLUMS);
	
	b = iter->user_data;
	
	switch (column) {
	case GB_TREE_MODEL_COL_ICON:
		g_value_init (value, GDK_TYPE_PIXBUF);
		pb = gb_bookmark_get_icon (b);
		g_value_set_object (value, pb);
		break;
	case GB_TREE_MODEL_COL_TITLE:
		g_value_init (value, G_TYPE_STRING);
		if (GB_IS_SEPARATOR (b))
		{
			g_value_set_string (value, "-----");
		}
		else
		{
			g_value_set_string (value, b->name);
		}
		break;
	case GB_TREE_MODEL_COL_URL:
		g_value_init (value, G_TYPE_STRING);
		if (GB_IS_SITE (b))
		{
			g_value_set_string (value, ((GbSite *) b)->url);
		}
		break;
	default:
		g_assert_not_reached ();
		break;
	}
}

static gboolean
gb_tree_model_iter_next (GtkTreeModel *tree_model,
			 GtkTreeIter *iter)
{
	GbBookmark *b;
	gint i;
	GbTreeModel *btree_model = (GbTreeModel *) tree_model;
	g_return_val_if_fail (iter != NULL, FALSE);
	g_return_val_if_fail (iter->user_data != NULL, FALSE);
	g_return_val_if_fail (iter->stamp == GB_TREE_MODEL (tree_model)->stamp, FALSE);
	
	b = iter->user_data;
	
	if (gb_tree_model_is_root (btree_model, (GbFolder *) b, &i))
	{
		GSList *nl = g_slist_nth (btree_model->roots, i + 1);
		if (nl != NULL) 
		{
			iter->user_data = nl->data;
			return TRUE;
		}
		else
		{
			return FALSE;
		}
	}
	else
	{
		GbBookmark *n;
		n = b->next;
		if (n != NULL) 
		{
			iter->user_data = n;
			return TRUE;
		}
		else
		{
			return FALSE;
		}
	}
}

static gboolean
gb_tree_model_iter_children (GtkTreeModel *tree_model,
			     GtkTreeIter *iter,
			     GtkTreeIter *parent)
{
	GbBookmark *c1;
	GbTreeModel *btree_model = (GbTreeModel *) tree_model;
	g_return_val_if_fail (parent == NULL || parent->user_data != NULL, FALSE);
	g_return_val_if_fail (parent == NULL || parent->stamp == btree_model->stamp, FALSE);
	
	if (!parent) 
	{
		c1 = btree_model->roots->data;
	}
	else
	{
		GbFolder *p = parent->user_data;
		if (!gb_bookmark_is_alias (p))
			c1 = p->child;
		else 
			c1 = NULL;
	}
	
	if (c1)
	{
		iter->stamp = GB_TREE_MODEL (tree_model)->stamp;
		iter->user_data = c1;
		return TRUE;
	}
	else
	{
		return FALSE;
	}
}

static gboolean
gb_tree_model_iter_has_child (GtkTreeModel *tree_model,
			      GtkTreeIter *iter)
{
	GbFolder *b_it;
	g_return_val_if_fail (GB_IS_TREE_MODEL (tree_model), FALSE);
	g_return_val_if_fail (iter->stamp == GB_TREE_MODEL (tree_model)->stamp, FALSE);
	g_return_val_if_fail (iter->user_data != NULL, FALSE);
	
	if (!GB_IS_FOLDER (iter->user_data))
		return FALSE;

	b_it = iter->user_data;
	
	return b_it->child != NULL && !gb_bookmark_is_alias (b_it);
}

static gint
gb_tree_model_iter_n_children (GtkTreeModel *tree_model,
			       GtkTreeIter *iter)
{
	g_return_val_if_fail (GB_IS_TREE_MODEL (tree_model), 0);
	g_return_val_if_fail (iter == NULL || iter->user_data != NULL, 0);
	
	if (iter == NULL)
	{
		GSList *l = GB_TREE_MODEL (tree_model)->roots;
		return g_slist_length (l);
	}
	else 
	{
		GbFolder *b = iter->user_data;
		int i = 0;
		if (GB_IS_FOLDER (b) && !gb_bookmark_is_alias (b))
		{
			GbBookmark *c = b->child;
			while (c != NULL)
			{
				i++;
				c = c->next;
			}
		}
		return i;
	}
}

static gboolean
gb_tree_model_iter_nth_child (GtkTreeModel *tree_model,
			      GtkTreeIter *iter,
			      GtkTreeIter *parent,
			      gint n)
{
	g_return_val_if_fail (GB_IS_TREE_MODEL (tree_model), FALSE);
	g_return_val_if_fail (parent == NULL || parent->user_data != NULL, FALSE);
	g_return_val_if_fail (iter != NULL, FALSE);
	
	if (parent == NULL || parent->user_data == GB_TREE_MODEL (tree_model)->roots)
	{
		GSList *l = g_slist_nth (GB_TREE_MODEL (tree_model)->roots, n);
		if (l != NULL)
		{
			iter->user_data = l->data;
			iter->stamp = GB_TREE_MODEL (tree_model)->stamp;
			return TRUE;
		}
		else
		{
			return FALSE;
		}
	}
	else 
	{
		GbFolder *b = parent->user_data;
		int i = 0;
		if (GB_IS_FOLDER (b) && !gb_bookmark_is_alias (b))
		{
			GbBookmark *c = b->child;
			while (c != NULL)
			{
				if (i == n) 
				{
					iter->user_data = c;
					iter->stamp = GB_TREE_MODEL (tree_model)->stamp;
					return TRUE;
				}
				i++;
				c = c->next;
			}
			return FALSE;
		}
		else 
		{
			return FALSE;
		}
	}
}

static gboolean
gb_tree_model_iter_parent (GtkTreeModel *tree_model,
			   GtkTreeIter *iter,
			   GtkTreeIter *child)
{
	g_return_val_if_fail (iter != NULL, FALSE);
	g_return_val_if_fail (child != NULL, FALSE);
	g_return_val_if_fail (child->user_data != NULL, FALSE);
	g_return_val_if_fail (child->stamp == GB_TREE_MODEL (tree_model)->stamp, FALSE);
	
	if (child->user_data == GB_TREE_MODEL (tree_model)->roots)
	{
		return FALSE;
	}
	else if (gb_tree_model_is_root (GB_TREE_MODEL (tree_model), child->user_data, NULL))
	{
		iter->user_data = GB_TREE_MODEL (tree_model)->roots;
		iter->stamp = GB_TREE_MODEL (tree_model)->stamp;
		return TRUE;
	}
	else 
	{
		GbFolder *parent = GB_FOLDER (GB_BOOKMARK (child->user_data)->parent);
		g_assert (parent != NULL);
		iter->user_data = parent;
		iter->stamp = GB_TREE_MODEL (tree_model)->stamp;
		return TRUE;
	}
}

/**
 * gb_tree_model_iter_depth:
 * @btree_model: A #GbTreeModel
 * @iter: A valid #GtkTreeIter
 * 
 * Returns the depth of @iter. This will be 0 for anything on the root level, 1
 * for anything down a level, etc.
 * 
 * Return value: The depth of @iter
 **/
gint
gb_tree_model_iter_depth (GbTreeModel *btree_model,
			  GtkTreeIter *iter)
{
	gint depth = 0;
	GbBookmark *b_it;
	g_return_val_if_fail (GB_IS_TREE_MODEL (btree_model), 0);
	g_return_val_if_fail (VALID_ITER (iter, btree_model), 0);
	
	b_it = iter->user_data;
	while (b_it->parent && !gb_tree_model_is_root (btree_model, (GbFolder *) b_it, NULL))
	{
		depth++;
		b_it = (GbBookmark *) b_it->parent;
	}
	return depth;
}




/* DND */


static gboolean
gb_tree_model_drag_data_delete (GtkTreeDragSource *drag_source,
				GtkTreePath *path)
{
	GtkTreeIter iter;
	GbTreeModel *tm;

	g_return_val_if_fail (GB_IS_TREE_MODEL (drag_source), FALSE);

	tm = GB_TREE_MODEL (drag_source);

	if (gb_tree_model_get_iter (GTK_TREE_MODEL (tm), &iter, path))
	{
		GbBookmark *b = gb_tree_model_bookmark_from_iter (tm, &iter);
		gb_bookmark_unparent (b);
		return TRUE;
	}
	else
	{
		return FALSE;
	}
}

static gboolean
gb_tree_model_drag_data_get (GtkTreeDragSource *drag_source,
			     GtkTreePath *path,
			     GtkSelectionData *selection_data)
{
	g_return_val_if_fail (GB_IS_TREE_MODEL (drag_source), FALSE);
	
	/* Note that we don't need to handle the GTK_TREE_MODEL_ROW
	 * target, because the default handler does it for us, but
	 * we do anyway for the convenience of someone maybe overriding the
	 * default handler.
	 */
	
	if (gtk_tree_set_row_drag_data (selection_data,
					GTK_TREE_MODEL (drag_source),
					path))
	{
		return TRUE;
	}
	else
	{
		/* export XBEL */
	}

	return FALSE;
}


static gboolean
gb_tree_model_drag_data_received (GtkTreeDragDest *drag_dest,
				  GtkTreePath *dest,
				  GtkSelectionData *selection_data)
{
	GbTreeModel *gtm;
	GtkTreeModel *src_model = NULL;
	GtkTreePath *src_path = NULL;
	GbBookmark *b = NULL;
	GbBookmark *bparent;
	gint bindex;
	
	g_return_val_if_fail (GB_IS_TREE_MODEL (drag_dest), FALSE);

	DEBUG_MSG (("in gb_tree_model_drag_data_received\n"));

	gtm = GB_TREE_MODEL (drag_dest);
	
	/* get the bookmark */
	if (gtk_tree_get_row_drag_data (selection_data,
					&src_model,
					&src_path))
	{
		/* in the same process */

		/* Copy the given row to a new position */
		GtkTreeIter src_iter;
		GbBookmark *srcb;
		
		if (!gtk_tree_model_get_iter (src_model,
					      &src_iter,
					      src_path))
		{
			gtk_tree_path_free (src_path);
			return FALSE;
		}

		srcb = gb_tree_model_bookmark_from_iter (gtm, &src_iter);
		if (gb_bookmark_is_alias (srcb))
		{
			/* if the source bookmark is an alias, we create an alias to it instead 
			   of copying it. This avoid turning aliases into normal bookmarks when 
			   moving them */
			b = gb_bookmark_alias_create (srcb, NULL);
		}
		else
		{
			b = gb_bookmark_copy (srcb);
		}
		gtk_tree_path_free (src_path);
	}
	else
	{
		/* import XBEL */
		NOT_IMPLEMENTED;
	}

	if (!b)
	{
		return FALSE;
	}

	{
		gchar *s = gtk_tree_path_to_string (dest);
		DEBUG_MSG (("dest = %s\n", s));
		g_free (s);
	}

	/* get the destination */
	{
		GtkTreePath *dest_parent_path = gtk_tree_path_copy (dest);
		GtkTreeIter dest_parent_iter; 
		if (gtk_tree_path_up (dest_parent_path))
		{
			if (gtk_tree_model_get_iter (GTK_TREE_MODEL (gtm), 
						     &dest_parent_iter, dest_parent_path))
			{
				bparent = gb_tree_model_bookmark_from_iter (gtm, &dest_parent_iter);
			}
			else
			{
				bparent = NULL;
			}
		}
		else
		{
			bparent = NULL;
		}
		gtk_tree_path_free (dest_parent_path);
	}

	if (!bparent)
	{
		bparent = gb_tree_model_get_first_root (gtm);
		DEBUG_MSG (("Defaulting the parent to the first root...\n"));
	}

	g_return_val_if_fail (bparent, FALSE);

	if (!GB_IS_FOLDER (bparent))
	{
		bparent = (GbBookmark *) bparent->parent;
	}

	g_return_val_if_fail (GB_IS_FOLDER (bparent), FALSE);

	if (g_object_get_data (G_OBJECT (gtm),
			       "gtk-tree-model-drop-append"))
	{
		bindex = -1;
		g_object_set_data (G_OBJECT (gtm), "gtk-tree-model-drop-append", NULL);
	}
	else if (gtk_tree_path_get_depth (dest) > 0)
	{
		bindex = gtk_tree_path_get_indices (dest)[gtk_tree_path_get_depth (dest) - 1];
	}
	else
	{
		bindex = -1;
	}
	
	DEBUG_MSG (("will insert into %s (at %d)\n", bparent->name, bindex));

	gb_folder_add_child (GB_FOLDER (bparent), b, bindex);

	return TRUE;
}

static gboolean
gb_tree_model_row_drop_possible (GtkTreeDragDest *drag_dest,
				 GtkTreePath *dest_path,
				 GtkSelectionData *selection_data)
{
	GtkTreeModel *src_model = NULL;
	GtkTreePath *src_path = NULL;
	GtkTreePath *tmp = NULL;
	gboolean retval = TRUE;
	
	if (gtk_tree_get_row_drag_data (selection_data,
					&src_model,
					&src_path))
	{	
		/* in-process drag */

		if (src_model == GTK_TREE_MODEL (drag_dest))
		{
			/* in the same tree model */
	
			/* Can't drop into ourself. */
			if (gtk_tree_path_is_ancestor (src_path,
						       dest_path))
			{
				retval = FALSE;
			}
			else
			{
				retval = TRUE;
			}
		}
		else
		{
			retval = TRUE;
		}
		gtk_tree_path_free (src_path);
	}

	if (retval)
	{
		/* Can't drop if dest_path's parent doesn't exist and is a folder */

		GtkTreeIter iter;
		
		if (gtk_tree_path_get_depth (dest_path) > 1)
		{
			tmp = gtk_tree_path_copy (dest_path);
			gtk_tree_path_up (tmp);
			
			if (!gtk_tree_model_get_iter (GTK_TREE_MODEL (drag_dest),
						      &iter, tmp))
			{
				retval = GB_IS_FOLDER (gb_tree_model_bookmark_from_iter 
						       (GB_TREE_MODEL (drag_dest), &iter));
			}
			else
			{
				retval = TRUE;
			}
			gtk_tree_path_free (tmp);
		}
		else
		{
			retval = TRUE;
		}
	}
	
	DEBUG_MSG (("gb_tree_model_row_drop_possible () = %s\n", retval ? "TRUE" : "FALSE"));
	
	return retval;
	
}



static gboolean
gb_tree_model_is_root (GbTreeModel *m, GbFolder *f, int *i)
{
	GSList *l = m->roots;
	int j = 0;
	while (l != NULL)
	{
		if (l->data == f)
		{
			if (i != NULL)
			{
				*i = j;
			}
			return TRUE;
		}
		i++;
		l = l->next;
	}
	return FALSE;
}

GbBookmark *
gb_tree_model_bookmark_from_iter (GbTreeModel *m, GtkTreeIter *iter)
{
	return GB_BOOKMARK (iter->user_data);
}

void
gb_tree_model_iter_from_bookmark (GbTreeModel *model, GbBookmark *b, GtkTreeIter *iter)
{
	iter->stamp = model->stamp;
	iter->user_data = b;
}

static void 
gb_tree_model_descendant_modified_cb (GbFolder *f, GbBookmark *b, gpointer data)
{
	GbTreeModel *m = data;
	GtkTreeIter it;
	GtkTreePath *p;
	it.user_data = b;
	it.stamp = m->stamp;
	p = gb_tree_model_get_path ((GtkTreeModel *) m, &it);
	gtk_tree_model_row_changed ((GtkTreeModel *) m, p, &it);
	gtk_tree_path_free (p);
}

static void
gb_tree_model_descendant_added_cb (GbFolder *f, GbFolder *p, GbBookmark *b, int pos, gpointer data)
{
	if (!gb_bookmark_is_alias (p)) 
	{
		GbTreeModel *m = data;
		GtkTreeIter it;
		GtkTreePath *path;
		it.user_data = b;
		it.stamp = m->stamp;
		path = gb_tree_model_get_path ((GtkTreeModel *) m, &it);
		gtk_tree_model_row_inserted ((GtkTreeModel *) m, path, &it);
		gtk_tree_path_free (path);

		/* probably this could be done less inneficient */
		if (GB_IS_FOLDER (b) && !gb_bookmark_is_alias (b))
		{
			GbBookmark *bi;
			int i = 0;
			for (bi = GB_FOLDER (b)->child; bi; bi = bi->next)
			{
				gb_tree_model_descendant_added_cb (f, bi->parent, bi, i, data);
				++i;
			}
		}
	}
}

static void
gb_tree_model_descendant_removed_cb (GbFolder *f, GbFolder *p, GbBookmark *b, int pos, gpointer data)
{
	GbTreeModel *m = data;
	if (!gb_bookmark_is_alias (p))
	{
		GtkTreeIter it;
		GtkTreePath *path;
		it.user_data = p;
		it.stamp = m->stamp;
		path = gb_tree_model_get_path ((GtkTreeModel *) m, &it);
		gtk_tree_path_append_index (path, pos);

		gtk_tree_model_row_deleted ((GtkTreeModel *) m, path);

		if (p->child == NULL)
		{
			gtk_tree_path_up (path);
			gtk_tree_model_row_has_child_toggled ((GtkTreeModel *) m, path, &it);
		}
		gtk_tree_path_free (path);
	}
}



GbBookmark *
gb_tree_model_get_first_root (GbTreeModel *model)
{
	if (model->roots)
	{
		return model->roots->data;
	}
	else
	{
		return NULL;
	}
}

GbBookmarkSet *
gb_tree_model_get_set (GbTreeModel *model)
{
	return model->set;
}


