/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*-
 *
 * Copyright (C) 2006-2009 Richard Hughes <richard@hughsie.com>
 *
 * Licensed under the GNU General Public License Version 2
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"

#include <glib.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <glib/gi18n.h>
#include <string.h>
#include <gconf/gconf-client.h>

#include "egg-debug.h"
#include "gpm-common.h"
#include "gpm-inhibit.h"

static void     gpm_inhibit_finalize   (GObject		*object);

#define GPM_INHIBIT_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), GPM_TYPE_INHIBIT, GpmInhibitPrivate))
#define GPM_SESSION_MANAGER_SERVICE			"org.gnome.SessionManager"
#define GPM_SESSION_MANAGER_PATH			"/org/gnome/SessionManager"
#define GPM_SESSION_MANAGER_INTERFACE			"org.gnome.SessionManager"

typedef struct
{
	gchar			*application;
	gchar			*reason;
	gchar			*connection;
	guint32			 cookie;
} GpmInhibitData;

struct GpmInhibitPrivate
{
	GSList			*list;
	DBusGProxy		*proxy;
	DBusGProxy		*proxy_session;
	GConfClient		*conf;
	gboolean		 ignore_inhibits;
};

enum {
	HAS_INHIBIT_CHANGED,
	LAST_SIGNAL
};

static guint signals [LAST_SIGNAL] = { 0 };

G_DEFINE_TYPE (GpmInhibit, gpm_inhibit, G_TYPE_OBJECT)

/**
 * gpm_inhibit_error_quark:
 * Return value: Our personal error quark.
 **/
GQuark
gpm_inhibit_error_quark (void)
{
	static GQuark quark = 0;
	if (!quark)
		quark = g_quark_from_static_string ("gpm_inhibit_error");
	return quark;
}

/**
 * gpm_inhibit_cookie_compare_func
 * @a: Pointer to the data to test
 * @b: Pointer to a cookie to compare
 *
 * A GCompareFunc for comparing a cookie to a list.
 *
 * Return value: 0 if cookie matches
 **/
static gint
gpm_inhibit_cookie_compare_func (gconstpointer a, gconstpointer b)
{
	GpmInhibitData *data;
	guint32 cookie;
	data = (GpmInhibitData*) a;
	cookie = *((guint32*) b);
	if (cookie == data->cookie)
		return 0;
	return 1;
}

/**
 * gpm_inhibit_find_cookie:
 * @inhibit: This inhibit instance
 * @cookie: The cookie we are looking for
 *
 * Finds the data in the cookie list.
 *
 * Return value: The cookie data, or NULL if not found
 **/
static GpmInhibitData *
gpm_inhibit_find_cookie (GpmInhibit *inhibit, guint32 cookie)
{
	GpmInhibitData *data;
	GSList *ret;
	/* search list */
	ret = g_slist_find_custom (inhibit->priv->list, &cookie,
				   gpm_inhibit_cookie_compare_func);
	if (ret == NULL)
		return NULL;
	data = (GpmInhibitData *) ret->data;
	return data;
}

/**
 * gpm_inhibit_inhibit:
 * @application: Application name, e.g. "Nautilus"
 * @reason: Reason for inhibiting, e.g. "Copying files"
 *
 * Allocates a random cookie used to identify the connection, as multiple
 * inhibit requests can come from one caller sharing a dbus connection.
 * We need to refcount internally, and data is saved in the GpmInhibitData
 * struct.
 *
 * Return value: a new random cookie.
 **/
void
gpm_inhibit_inhibit (GpmInhibit  *inhibit, const gchar *application, const gchar *reason, DBusGMethodInvocation *context)
{
	gboolean ret;
	GError *error = NULL;
	const gchar *connection;
	GpmInhibitData *data;
	guint inhibit_cookie = 0;

	/* as we are async, we can get the sender */
	connection = dbus_g_method_get_sender (context);

	/* handle where the application does not add required data */
	if (application == NULL || reason == NULL) {
		egg_warning ("Recieved Inhibit, but application "
			     "did not set the parameters correctly");
		dbus_g_method_return (context, -1);
		return;
	}

	/* proxy to gnome-session */
	ret = dbus_g_proxy_call (inhibit->priv->proxy_session, "Inhibit", &error,
				 G_TYPE_STRING, application, /* app_id */
				 G_TYPE_UINT, 0, /* toplevel_xid */
				 G_TYPE_STRING, reason, /* reason*/
				 G_TYPE_UINT, 8, /* flags */
				 G_TYPE_INVALID,
				 G_TYPE_UINT, &inhibit_cookie,
				 G_TYPE_INVALID);
	if (!ret) {
		egg_warning ("failed to proxy: %s", error->message);
		dbus_g_method_return_error (context, error);
		return;
	}

	/* seems okay, add to list */
	data = g_new (GpmInhibitData, 1);
	data->cookie = inhibit_cookie;
	data->application = g_strdup (application);
	data->connection = g_strdup (connection);
	data->reason = g_strdup (reason);

	inhibit->priv->list = g_slist_append (inhibit->priv->list, (gpointer) data);

	egg_debug ("Recieved Inhibit from '%s' (%s) because '%s' saving as #%i",
		   data->application, data->connection, data->reason, data->cookie);

	/* only emit event on the first one */
	if (g_slist_length (inhibit->priv->list) == 1)
		g_signal_emit (inhibit, signals [HAS_INHIBIT_CHANGED], 0, TRUE);

	dbus_g_method_return (context, data->cookie);
}

/* free one element in GpmInhibitData struct */
static void
gpm_inhibit_free_data_object (GpmInhibitData *data)
{
	g_free (data->application);
	g_free (data->reason);
	g_free (data->connection);
	g_free (data);
}

/**
 * gpm_inhibit_un_inhibit:
 * @application: Application name
 * @cookie: The cookie that we used to register
 *
 * Removes a cookie and associated data from the GpmInhibitData struct.
 **/
gboolean
gpm_inhibit_un_inhibit (GpmInhibit *inhibit, guint32 cookie, GError **error)
{
	gboolean ret;
	GpmInhibitData *data;

	/* Only remove the correct cookie */
	data = gpm_inhibit_find_cookie (inhibit, cookie);
	if (data == NULL) {
		*error = g_error_new (gpm_inhibit_error_quark (),
				      GPM_INHIBIT_ERROR_GENERAL,
				      "Cannot find registered program for #%i, so "
				      "cannot do UnInhibit!", cookie);
		return FALSE;
	}
	egg_debug ("UnInhibit okay #%i", cookie);

	/* proxy to gnome-session */
	ret = dbus_g_proxy_call (inhibit->priv->proxy_session, "Uninhibit", error,
				 G_TYPE_UINT, cookie,
				 G_TYPE_INVALID,
				 G_TYPE_INVALID);
	if (!ret) {
		egg_warning ("failed to proxy: %s", (*error)->message);
		return FALSE;
	}

	gpm_inhibit_free_data_object (data);

	/* remove it from the list */
	inhibit->priv->list = g_slist_remove (inhibit->priv->list, (gconstpointer) data);

	/* only emit event on the last one */
	if (g_slist_length (inhibit->priv->list) == 0)
		g_signal_emit (inhibit, signals [HAS_INHIBIT_CHANGED], 0, FALSE);

	return TRUE;
}

/**
 * gpm_inhibit_remove_dbus:
 * @connection: Connection name
 * @application: Application name
 * @cookie: The cookie that we used to register
 *
 * Checks to see if the dbus closed session is registered, in which case
 * unregister it.
 **/
static void
gpm_inhibit_remove_dbus (GpmInhibit *inhibit, const gchar *connection)
{
	guint a;
	GpmInhibitData *data;
	gboolean ret;
	GError *error = NULL;

	/* Remove *any* connections that match the connection */
	for (a=0; a<g_slist_length (inhibit->priv->list); a++) {
		data = (GpmInhibitData *) g_slist_nth_data (inhibit->priv->list, a);
		if (strcmp (data->connection, connection) == 0) {
			egg_debug ("Auto-revoked idle inhibit on '%s'.", data->application);

			/* proxy to gnome-session */
			ret = dbus_g_proxy_call (inhibit->priv->proxy_session, "Uninhibit", &error,
						 G_TYPE_UINT, data->cookie,
						 G_TYPE_INVALID,
						 G_TYPE_INVALID);
			if (!ret) {
				egg_warning ("failed to proxy: %s", error->message);
				g_error_free (error);
			}

			gpm_inhibit_free_data_object (data);
			/* remove it from the list */
			inhibit->priv->list = g_slist_remove (inhibit->priv->list, (gconstpointer) data);
		}
	}
	return;
}

/**
 * gpm_inhibit_name_owner_changed_cb:
 **/
static void
gpm_inhibit_name_owner_changed_cb (DBusGProxy *proxy, const gchar *name,
				   const gchar *prev, const gchar *new,
				   GpmInhibit *inhibit)
{
	if (strlen (new) == 0)
		gpm_inhibit_remove_dbus (inhibit, name);
}

/**
 * gpm_inhibit_has_inhibit
 *
 * Checks to see if we are being stopped from performing an action.
 *
 * TRUE if the action is OK, i.e. we have *not* been inhibited.
 **/
gboolean
gpm_inhibit_has_inhibit (GpmInhibit *inhibit, gboolean *has_inihibit, GError **error)
{
	guint length;

	length = g_slist_length (inhibit->priv->list);

	if (inhibit->priv->ignore_inhibits) {
		egg_debug ("Inhibit ignored through gconf policy!");
		*has_inihibit = FALSE;
	}

	/* An inhibit can stop the action */
	if (length == 0) {
		egg_debug ("Valid as no inhibitors");
		*has_inihibit = FALSE;
	} else {
		/* we have at least one application blocking the action */
		egg_debug ("We have %i valid inhibitors", length);
		*has_inihibit = TRUE;
	}

	/* we always return successful for DBUS */
	return TRUE;
}

/**
 * gpm_inhibit_get_message:
 *
 * @message: Description string, e.g. "Nautilus because 'copying files'"
 * @action: Action we wanted to do, e.g. "suspend"
 *
 * Returns a localised message text describing what application has inhibited
 * the action, and why.
 *
 **/
void
gpm_inhibit_get_message (GpmInhibit *inhibit, GString *message, const gchar *action)
{
	guint a;
	GpmInhibitData *data;
	gchar *boldstr;
	gchar *italicstr;

	if (g_slist_length (inhibit->priv->list) == 1) {
		data = (GpmInhibitData *) g_slist_nth_data (inhibit->priv->list, 0);
		boldstr = g_strdup_printf ("<b>%s</b>", data->application);
		italicstr = g_strdup_printf ("<b>%s</b>", data->reason);
		
		if (strcmp (action, "suspend") == 0) {
			/*Translators: first %s is an application name, second %s is the reason*/
			g_string_append_printf (message, _("%s has stopped the suspend from taking place: %s."),
				boldstr, italicstr);

		} else if (strcmp (action, "hibernate") == 0) {
			/*Translators: first %s is an application name, second %s is the reason*/
			g_string_append_printf (message, _("%s has stopped the hibernate from taking place: %s."),
					boldstr, italicstr); 

		} else if (strcmp (action, "policy action") == 0) {
			/*Translators: first %s is an application name, second %s is the reason*/
			g_string_append_printf (message, _("%s has stopped the policy action from taking place: %s."),
					boldstr, italicstr); 

		} else if (strcmp (action, "reboot") == 0) {
			/*Translators: first %s is an application name, second %s is the reason*/
			g_string_append_printf (message, _("%s has stopped the reboot from taking place: %s."),
					boldstr, italicstr); 

		} else if (strcmp (action, "shutdown") == 0) {
			/*Translators: first %s is an application name, second %s is the reason*/
			g_string_append_printf (message, _("%s has stopped the shutdown from taking place: %s."),
				boldstr, italicstr); 

		} else if (strcmp (action, "timeout action") == 0) {
			/*Translators: first %s is an application name, second %s is the reason*/
			g_string_append_printf (message, _("%s has stopped the timeout action from taking place: %s."),
				boldstr, italicstr); 
		}

		g_free (boldstr);
		g_free (italicstr);

	} else {
		if (strcmp (action, "suspend") == 0) {
			g_string_append (message, _("Multiple applications have stopped the suspend from taking place."));

		} else if (strcmp (action, "hibernate") == 0) {
			g_string_append (message, _("Multiple applications have stopped the hibernate from taking place."));

		} else if (strcmp (action, "policy action") == 0) {
			g_string_append (message, _("Multiple applications have stopped the policy action from taking place."));

		} else if (strcmp (action, "reboot") == 0) {
			g_string_append (message, _("Multiple applications have stopped the reboot from taking place.")); 

		} else if (strcmp (action, "shutdown") == 0) {
			g_string_append (message, _("Multiple applications have stopped the shutdown from taking place."));

		} else if (strcmp (action, "timeout action") == 0) {
			g_string_append (message, _("Multiple applications have stopped the suspend from taking place."));
		}
		
		for (a=0; a<g_slist_length (inhibit->priv->list); a++) {
			data = (GpmInhibitData *) g_slist_nth_data (inhibit->priv->list, a);
			g_string_append_printf (message,
						"\n<b>%s</b> : <i>%s</i>",
						data->application, data->reason);
		}
	}
}

/** intialise the class */
static void
gpm_inhibit_class_init (GpmInhibitClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	object_class->finalize = gpm_inhibit_finalize;

	signals [HAS_INHIBIT_CHANGED] =
		g_signal_new ("has-inhibit-changed",
			      G_TYPE_FROM_CLASS (object_class), G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (GpmInhibitClass, has_inhibit_changed),
			      NULL, NULL, g_cclosure_marshal_VOID__BOOLEAN,
			      G_TYPE_NONE, 1, G_TYPE_BOOLEAN);
	g_type_class_add_private (klass, sizeof (GpmInhibitPrivate));
}

/** intialise the object */
static void
gpm_inhibit_init (GpmInhibit *inhibit)
{
	DBusGConnection *connection;
	GError *error = NULL;

	inhibit->priv = GPM_INHIBIT_GET_PRIVATE (inhibit);
	inhibit->priv->list = NULL;
	inhibit->priv->conf = gconf_client_get_default ();

	connection = dbus_g_bus_get (DBUS_BUS_SESSION, &error);
	if (error != NULL) {
		egg_warning ("Cannot connect to bus: %s", error->message);
		g_error_free (error);
	}
	inhibit->priv->proxy = dbus_g_proxy_new_for_name_owner (connection,
								DBUS_SERVICE_DBUS, DBUS_PATH_DBUS,
						 		DBUS_INTERFACE_DBUS, &error);
	if (inhibit->priv->proxy == NULL) {
		egg_warning ("failed to get proxy: %s", error->message);
		g_error_free (error);
		return;
	}
	dbus_g_proxy_add_signal (inhibit->priv->proxy, "NameOwnerChanged",
				 G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_INVALID);
	dbus_g_proxy_connect_signal (inhibit->priv->proxy, "NameOwnerChanged",
				     G_CALLBACK (gpm_inhibit_name_owner_changed_cb),
				     inhibit, NULL);

	inhibit->priv->proxy_session = dbus_g_proxy_new_for_name_owner (connection,
									GPM_SESSION_MANAGER_SERVICE, GPM_SESSION_MANAGER_PATH,
						 			GPM_SESSION_MANAGER_INTERFACE, NULL);
	if (inhibit->priv->proxy_session == NULL) {
		egg_warning ("failed to get proxy: %s", error->message);
		g_error_free (error);
		return;
	}

	/* Do we ignore inhibit requests? */
	inhibit->priv->ignore_inhibits = gconf_client_get_bool (inhibit->priv->conf, GPM_CONF_IGNORE_INHIBITS, NULL);
}

/** finalise the object */
static void
gpm_inhibit_finalize (GObject *object)
{
	GpmInhibit *inhibit;
	guint a;
	GpmInhibitData *data;

	g_return_if_fail (object != NULL);
	g_return_if_fail (GPM_IS_INHIBIT (object));

	inhibit = GPM_INHIBIT (object);
	inhibit->priv = GPM_INHIBIT_GET_PRIVATE (inhibit);

	for (a=0; a<g_slist_length (inhibit->priv->list); a++) {
		data = (GpmInhibitData *) g_slist_nth_data (inhibit->priv->list, a);
		gpm_inhibit_free_data_object (data);
	}
	g_slist_free (inhibit->priv->list);

	g_object_unref (inhibit->priv->conf);
	g_object_unref (inhibit->priv->proxy);
	g_object_unref (inhibit->priv->proxy_session);
	G_OBJECT_CLASS (gpm_inhibit_parent_class)->finalize (object);
}

/** create the object */
GpmInhibit *
gpm_inhibit_new (void)
{
	GpmInhibit *inhibit;
	inhibit = g_object_new (GPM_TYPE_INHIBIT, NULL);
	return GPM_INHIBIT (inhibit);
}

/***************************************************************************
 ***                          MAKE CHECK TESTS                           ***
 ***************************************************************************/
#ifdef EGG_TEST
#include "egg-test.h"
#include "egg-dbus-proxy.h"
#include "gpm-common.h"

/** cookie is returned as an unsigned integer */
static gboolean
inhibit (EggDbusProxy       *gproxy,
	 const gchar     *appname,
	 const gchar     *reason,
	 guint           *cookie)
{
	GError  *error = NULL;
	gboolean ret;
	DBusGProxy *proxy;

	g_return_val_if_fail (cookie != NULL, FALSE);

	proxy = egg_dbus_proxy_get_proxy (gproxy);
	if (proxy == NULL) {
		g_warning ("not connected");
		return FALSE;
	}

	ret = dbus_g_proxy_call (proxy, "Inhibit", &error,
				 G_TYPE_STRING, appname,
				 G_TYPE_STRING, reason,
				 G_TYPE_INVALID,
				 G_TYPE_UINT, cookie,
				 G_TYPE_INVALID);
	if (error) {
		g_debug ("ERROR: %s", error->message);
		g_error_free (error);
		*cookie = 0;
	}
	if (!ret) {
		/* abort as the DBUS method failed */
		g_warning ("Inhibit failed!");
	}

	return ret;
}

static gboolean
uninhibit (EggDbusProxy *gproxy,
	   guint      cookie)
{
	GError *error = NULL;
	gboolean ret;
	DBusGProxy *proxy;

	proxy = egg_dbus_proxy_get_proxy (gproxy);
	if (proxy == NULL) {
		g_warning ("not connected");
		return FALSE;
	}

	ret = dbus_g_proxy_call (proxy, "UnInhibit", &error,
				 G_TYPE_UINT, cookie,
				 G_TYPE_INVALID,
				 G_TYPE_INVALID);
	if (error) {
		egg_debug ("ERROR: %s", error->message);
		g_error_free (error);
	}
	return ret;
}

static gboolean
gpm_inhibit_test_has_inhibit (EggDbusProxy *gproxy, gboolean *has_inhibit)
{
	GError  *error = NULL;
	gboolean ret;
	DBusGProxy *proxy;

	proxy = egg_dbus_proxy_get_proxy (gproxy);
	if (proxy == NULL) {
		g_warning ("not connected");
		return FALSE;
	}

	ret = dbus_g_proxy_call (proxy, "HasInhibit", &error,
				 G_TYPE_INVALID,
				 G_TYPE_BOOLEAN, has_inhibit,
				 G_TYPE_INVALID);
	if (error) {
		g_debug ("ERROR: %s", error->message);
		g_error_free (error);
	}
	if (!ret) {
		/* abort as the DBUS method failed */
		g_warning ("HasInhibit failed!");
	}

	return ret;
}

void
gpm_inhibit_test (gpointer data)
{
	gboolean ret;
	gboolean valid;
	guint cookie1 = 0;
	guint cookie2 = 0;
	DBusGConnection *connection;
	EggDbusProxy *gproxy;
	EggTest *test = (EggTest *) data;

	if (egg_test_start (test, "GpmInhibit") == FALSE) {
		return;
	}

	gproxy = egg_dbus_proxy_new ();
	connection = dbus_g_bus_get (DBUS_BUS_SESSION, NULL);
	egg_dbus_proxy_assign (gproxy, connection, GPM_DBUS_SERVICE,
			       GPM_DBUS_PATH_INHIBIT, GPM_DBUS_INTERFACE_INHIBIT);

	if (gproxy == NULL) {
		g_warning ("Unable to get connection to power manager");
		return;
	}

	/************************************************************/
	egg_test_title (test, "make sure we are not inhibited");
	ret = gpm_inhibit_test_has_inhibit (gproxy, &valid);
	if (!ret) {
		egg_test_failed (test, "Unable to test validity");
	} else if (valid) {
		egg_test_failed (test, "Already inhibited");
	} else {
		egg_test_success (test, NULL);
	}

	/************************************************************/
	egg_test_title (test, "clear an invalid cookie");
	ret = uninhibit (gproxy, 123456);
	if (!ret) {
		egg_test_success (test, "invalid cookie failed as expected");
	} else {
		egg_test_failed (test, "should have rejected invalid cookie");
	}

	/************************************************************/
	egg_test_title (test, "get auto cookie 1");
	ret = inhibit (gproxy,
				  "gnome-power-self-test",
				  "test inhibit",
				  &cookie1);
	if (!ret) {
		egg_test_failed (test, "Unable to inhibit");
	} else if (cookie1 == 0) {
		egg_test_failed (test, "Cookie invalid (cookie: %u)", cookie1);
	} else {
		egg_test_success (test, "cookie: %u", cookie1);
	}

	/************************************************************/
	egg_test_title (test, "make sure we are auto inhibited");
	ret = gpm_inhibit_test_has_inhibit (gproxy, &valid);
	if (!ret) {
		egg_test_failed (test, "Unable to test validity");
	} else if (valid) {
		egg_test_success (test, "inhibited");
	} else {
		egg_test_failed (test, "inhibit failed");
	}

	/************************************************************/
	egg_test_title (test, "get cookie 2");
	ret = inhibit (gproxy,
				  "gnome-power-self-test",
				  "test inhibit",
				  &cookie2);
	if (!ret) {
		egg_test_failed (test, "Unable to inhibit");
	} else if (cookie2 == 0) {
		egg_test_failed (test, "Cookie invalid (cookie: %u)", cookie2);
	} else {
		egg_test_success (test, "cookie: %u", cookie2);
	}

	/************************************************************/
	egg_test_title (test, "clear cookie 1");
	ret = uninhibit (gproxy, cookie1);
	if (!ret) {
		egg_test_failed (test, "cookie failed to clear");
	} else {
		egg_test_success (test, NULL);
	}

	/************************************************************/
	egg_test_title (test, "make sure we are still inhibited");
	ret = gpm_inhibit_test_has_inhibit (gproxy, &valid);
	if (!ret) {
		egg_test_failed (test, "Unable to test validity");
	} else if (valid) {
		egg_test_success (test, "inhibited");
	} else {
		egg_test_failed (test, "inhibit failed");
	}

	/************************************************************/
	egg_test_title (test, "clear cookie 2");
	ret = uninhibit (gproxy, cookie2);
	if (!ret) {
		egg_test_failed (test, "cookie failed to clear");
	} else {
		egg_test_success (test, NULL);
	}

	g_object_unref (gproxy);

	egg_test_end (test);
}

#endif

