/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */

/* eel-gtk-extensions.c - implementation of new functions that operate on
  			  gtk classes. Perhaps some of these should be
  			  rolled into gtk someday.

   Copyright (C) 1999, 2000, 2001 Eazel, Inc.

   The Gnome Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   The Gnome Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the Gnome Library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.

   Authors: John Sullivan <sullivan@eazel.com>
            Ramiro Estrugo <ramiro@eazel.com>
	    Darin Adler <darin@eazel.com>
*/

#include <config.h>
#include "eel-gtk-extensions.h"

#include "eel-gdk-pixbuf-extensions.h"
#include "eel-glib-extensions.h"
#include "eel-gnome-extensions.h"
#include "eel-pango-extensions.h"
#include "eel-string.h"
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <gdk/gdk.h>
#include <gdk/gdkkeysyms.h>
#include <gdk/gdkprivate.h>
#include <gdk/gdkx.h>
#include <gtk/gtkdialog.h>
#include <gtk/gtkdnd.h>
#include <gtk/gtkmain.h>
#include <gtk/gtkrc.h>
#include <gtk/gtkselection.h>
#include <gtk/gtkseparatormenuitem.h>
#include <gtk/gtksignal.h>
#include <pango/pangoft2.h>

#include "eel-marshal.h"
#include "eel-marshal.c"

/* This number is fairly arbitrary. Long enough to show a pretty long
 * menu title, but not so long to make a menu grotesquely wide.
 */
#define MAXIMUM_MENU_TITLE_LENGTH	48

/* Used for window position & size sanity-checking. The sizes are big enough to prevent
 * at least normal-sized gnome panels from obscuring the window at the screen edges. 
 */
#define MINIMUM_ON_SCREEN_WIDTH		100
#define MINIMUM_ON_SCREEN_HEIGHT	100

/* How far down the window tree will we search when looking for top-level
 * windows? Some window managers doubly-reparent the client, so account
 * for that, and add some slop.
 */
#define MAXIMUM_WM_REPARENTING_DEPTH 4

/**
 * eel_gtk_button_set_padding
 * 
 * Adds some padding around the contained widget in the button (typically the label).
 * @button: a GtkButton
 * @pad_amount: number of pixels of space to add around the button's contents.
 * GNOME_PAD_SMALL is a typical value.
 **/
void
eel_gtk_button_set_padding (GtkButton *button, int pad_amount)
{
	g_return_if_fail (GTK_IS_BUTTON (button));
	g_return_if_fail (pad_amount > 0);

	gtk_misc_set_padding (GTK_MISC (GTK_BIN(button)->child), 
			      pad_amount, 
			      pad_amount);
}

/**
 * eel_gtk_signal_connect_free_data_custom:
 * 
 * Attach a function pointer and user data to a signal, and call a
 * a destroy function on the user data when the signal is disconnected.
 * @object: the object which emits the signal. For example, a button in the button press signal.
 * @name: the name of the signal.
 * @func: function pointer to attach to the signal.
 * @data: the user data associated with the function.
 * @destroy_func: the function to call on the user data when the signal
 * is disconnected.
 **/
guint eel_gtk_signal_connect_free_data_custom (GtkObject *object,
				  	     	    const gchar *name,
				  	     	    GtkSignalFunc func,
				  	     	    gpointer data,
				  	     	    GtkDestroyNotify destroy_func)
{
	return gtk_signal_connect_full (object, 
					name, 
					func, 
					NULL, /* marshal */
					data, 
					destroy_func, 
					FALSE, /* is this an object signal? */
					FALSE); /* invoke func after signal? */
}

/**
 * eel_gtk_signal_connect_free_data:
 * 
 * Attach a function pointer and user data to a signal, and free
 * the user data when the signal is disconnected.
 * @object: the object which emits the signal. For example, a button in the button press signal.
 * @name: the name of the signal.
 * @func: function pointer to attach to the signal.
 * @data: the user data associated with the function. g_free() will be called on
 * this user data when the signal is disconnected.
 **/
guint
eel_gtk_signal_connect_free_data (GtkObject *object,
				       const gchar *name,
				       GtkSignalFunc func,
				       gpointer data)
{
	return eel_gtk_signal_connect_free_data_custom
		(object, name, func, data, g_free);
}

/**
 * eel_gtk_window_present:
 * 
 * Presents to the user a window that may be hidden, iconified, or buried.
 * @window: The GtkWindow to be presented to the user.
 **/
void
eel_gtk_window_present (GtkWindow *window)
{
	gtk_window_present (window);
}

/**
 * eel_gtk_window_get_geometry_string:
 * @window: a #GtkWindow
 * 
 * Obtains the geometry string for this window, suitable for
 * set_geometry_string(); assumes the window has NorthWest gravity
 * 
 * Return value: geometry string, must be freed
 **/
char*
eel_gtk_window_get_geometry_string (GtkWindow *window)
{
	char *str;
	int w, h, x, y;
	
	g_return_val_if_fail (GTK_IS_WINDOW (window), NULL);
	g_return_val_if_fail (gtk_window_get_gravity (window) ==
			      GDK_GRAVITY_NORTH_WEST, NULL);

	gtk_window_get_position (window, &x, &y);
	gtk_window_get_size (window, &w, &h);
	
	str = g_strdup_printf ("%dx%d+%d+%d", w, h, x, y);

	return str;
}

static void
send_delete_event (GtkWindow *window)
{
	/* Synthesize delete_event to close window. */
	
	GdkEvent event;
	GtkWidget *widget;
	
	widget = GTK_WIDGET (window);
	
	event.any.type = GDK_DELETE;
	event.any.window = widget->window;
	event.any.send_event = TRUE;
	
	g_object_ref (event.any.window);
	gtk_main_do_event (&event);
	g_object_unref (event.any.window);
}

static int
handle_standard_close_accelerator (GtkWindow *window, 
				   GdkEventKey *event, 
				   gpointer user_data)
{
	g_assert (GTK_IS_WINDOW (window));
	g_assert (event != NULL);
	g_assert (user_data == NULL);

	if (eel_gtk_window_event_is_close_accelerator (window, event)) {
		send_delete_event (window);
		gtk_signal_emit_stop_by_name 
			(GTK_OBJECT (window), "key_press_event");
		return TRUE;
	}

	return FALSE;
}

/**
 * eel_gtk_window_event_is_close_accelerator:
 * 
 * Tests whether a key event is a standard window close accelerator.
 * Not needed for clients that use eel_gtk_window_set_up_close_accelerator;
 * use only if you must set up your own key_event handler for your own reasons.
 **/
gboolean
eel_gtk_window_event_is_close_accelerator (GtkWindow *window, GdkEventKey *event)
{
	g_return_val_if_fail (GTK_IS_WINDOW (window), FALSE);
	g_return_val_if_fail (event != NULL, FALSE);

	if (event->state & GDK_CONTROL_MASK) {
		/* Note: menu item equivalents are case-sensitive, so we will
		 * be case-sensitive here too.
		 */		
		if (event->keyval == EEL_STANDARD_CLOSE_WINDOW_CONTROL_KEY) {
			return TRUE;
		}
	}

	if (event->keyval == GDK_Escape) {
		return TRUE;
	}

	return FALSE;	
}

/**
 * eel_gtk_window_set_up_close_accelerator:
 * 
 * Sets up the standard keyboard equivalent to close the window.
 * Call this for windows that don't set up a keyboard equivalent to
 * close the window some other way, e.g. via a menu item accelerator.
 *
 * NOTE: do not use for GtkDialog, it already sets up the right
 * stuff here.
 * 
 * @window: The GtkWindow that should be hidden when the standard
 * keyboard equivalent is typed.
 **/
void
eel_gtk_window_set_up_close_accelerator (GtkWindow *window)
{
	g_return_if_fail (GTK_IS_WINDOW (window));

	if (GTK_IS_DIALOG (window)) {
		g_warning ("eel_gtk_window_set_up_close_accelerator: Should not mess with close accelerator on GtkDialogs");
		return;
	}
	
	g_signal_connect (window,
			    "key_press_event",
			    G_CALLBACK (handle_standard_close_accelerator),
			    NULL);
}

static void
sanity_check_window_position (int *left, int *top)
{
	g_assert (left != NULL);
	g_assert (top != NULL);

	/* Make sure the top of the window is on screen, for
	 * draggability (might not be necessary with all window managers,
	 * but seems reasonable anyway). Make sure the top of the window
	 * isn't off the bottom of the screen, or so close to the bottom
	 * that it might be obscured by the panel.
	 */
	*top = CLAMP (*top, 0, gdk_screen_height() - MINIMUM_ON_SCREEN_HEIGHT);
	
	/* FIXME bugzilla.eazel.com 669: 
	 * If window has negative left coordinate, set_uposition sends it
	 * somewhere else entirely. Not sure what level contains this bug (XWindows?).
	 * Hacked around by pinning the left edge to zero, which just means you
	 * can't set a window to be partly off the left of the screen using
	 * this routine.
	 */
	/* Make sure the left edge of the window isn't off the right edge of
	 * the screen, or so close to the right edge that it might be
	 * obscured by the panel.
	 */
	*left = CLAMP (*left, 0, gdk_screen_width() - MINIMUM_ON_SCREEN_WIDTH);
}

static void
sanity_check_window_dimensions (int *width, int *height)
{
	g_assert (width != NULL);
	g_assert (height != NULL);

	/* Pin the size of the window to the screen, so we don't end up in
	 * a state where the window is so big essential parts of it can't
	 * be reached (might not be necessary with all window managers,
	 * but seems reasonable anyway).
	 */
	*width = MIN (*width, gdk_screen_width());
	*height = MIN (*height, gdk_screen_height());
}

/**
 * eel_gtk_window_set_initial_geometry:
 * 
 * Sets the position and size of a GtkWindow before the
 * GtkWindow is shown. It is an error to call this on a window that
 * is already on-screen. Takes into account screen size, and does
 * some sanity-checking on the passed-in values.
 * 
 * @window: A non-visible GtkWindow
 * @geometry_flags: A EelGdkGeometryFlags value defining which of
 * the following parameters have defined values
 * @left: pixel coordinate for left of window
 * @top: pixel coordinate for top of window
 * @width: width of window in pixels
 * @height: height of window in pixels
 */
void
eel_gtk_window_set_initial_geometry (GtkWindow *window, 
					  EelGdkGeometryFlags geometry_flags,
					  int left,
					  int top,
					  guint width,
					  guint height)
{
	int real_left, real_top;

	g_return_if_fail (GTK_IS_WINDOW (window));
	g_return_if_fail (!(geometry_flags & EEL_GDK_WIDTH_VALUE) || width > 0);
	g_return_if_fail (!(geometry_flags & EEL_GDK_HEIGHT_VALUE) || height > 0);

	/* Setting the default size doesn't work when the window is already showing.
	 * Someday we could make this move an already-showing window, but we don't
	 * need that functionality yet. 
	 */
	g_return_if_fail (!GTK_WIDGET_VISIBLE (window));

	if ((geometry_flags & EEL_GDK_X_VALUE) && (geometry_flags & EEL_GDK_Y_VALUE)) {
		real_left = left;
		real_top = top;

		/* This is sub-optimal. GDK doesn't allow us to set win_gravity
		 * to South/East types, which should be done if using negative
		 * positions (so that the right or bottom edge of the window
		 * appears at the specified position, not the left or top).
		 * However it does seem to be consistent with other GNOME apps.
		 */
		if (geometry_flags & EEL_GDK_X_NEGATIVE) {
			real_left = gdk_screen_width () - real_left;
		}
		if (geometry_flags & EEL_GDK_Y_NEGATIVE) {
			real_top = gdk_screen_height () - real_top;
		}

		sanity_check_window_position (&real_left, &real_top);
		gtk_window_move (window, real_left, real_top);
	}

	if ((geometry_flags & EEL_GDK_WIDTH_VALUE) && (geometry_flags & EEL_GDK_HEIGHT_VALUE)) {
		sanity_check_window_dimensions (&width, &height);
		gtk_window_set_default_size (GTK_WINDOW (window), width, height);
	}
}

/**
 * eel_gtk_window_set_initial_geometry_from_string:
 * 
 * Sets the position and size of a GtkWindow before the
 * GtkWindow is shown. The geometry is passed in as a string. 
 * It is an error to call this on a window that
 * is already on-screen. Takes into account screen size, and does
 * some sanity-checking on the passed-in values.
 * 
 * @window: A non-visible GtkWindow
 * @geometry_string: A string suitable for use with gnome_parse_geometry
 * @minimum_width: If the width from the string is smaller than this,
 * use this for the width.
 * @minimum_height: If the height from the string is smaller than this,
 * use this for the height.
 */
void
eel_gtk_window_set_initial_geometry_from_string (GtkWindow *window, 
						 const char *geometry_string,
						 guint minimum_width,
						 guint minimum_height)
{
	int left, top;
	guint width, height;
	EelGdkGeometryFlags geometry_flags;

	g_return_if_fail (GTK_IS_WINDOW (window));
	g_return_if_fail (geometry_string != NULL);

	/* Setting the default size doesn't work when the window is already showing.
	 * Someday we could make this move an already-showing window, but we don't
	 * need that functionality yet. 
	 */
	g_return_if_fail (!GTK_WIDGET_VISIBLE (window));

	geometry_flags = eel_gdk_parse_geometry (geometry_string, &left, &top, &width, &height);

	/* Make sure the window isn't smaller than makes sense for this window.
	 * Other sanity checks are performed in set_initial_geometry.
	 */
	if (geometry_flags & EEL_GDK_WIDTH_VALUE) {
		width = MAX (width, minimum_width);
	}
	if (geometry_flags & EEL_GDK_HEIGHT_VALUE) {
		height = MAX (height, minimum_height);
	}

	eel_gtk_window_set_initial_geometry (window, geometry_flags, left, top, width, height);
}

/**
 * eel_gtk_selection_data_copy_deep:
 * 
 * Copies a GtkSelectionData, and copies the data too.
 * @data: The GtkSelectionData to be copied.
 **/
GtkSelectionData *
eel_gtk_selection_data_copy_deep (const GtkSelectionData *data)
{
	GtkSelectionData *copy;

	copy = g_new0 (GtkSelectionData, 1);
	gtk_selection_data_set (copy, data->type, data->format, data->data, data->length);

	return copy;
}

/**
 * eel_gtk_selection_data_free_deep:
 * 
 * Frees a GtkSelectionData, and frees the data too.
 * @data: The GtkSelectionData to be freed.
 **/
void
eel_gtk_selection_data_free_deep (GtkSelectionData *data)
{
	g_free (data->data);
	gtk_selection_data_free (data);
}

/**
 * eel_gtk_signal_connect_free_data:
 * 
 * Function to displace the popup menu some, otherwise the first item
 * gets selected right away.
 * This function gets called by gtk_menu_popup ().
 *
 * @menu: the popup menu.
 * @x: x coord where gtk want to place the menu
 * @y: y coord where gtk want to place the menu
 * @user_data: something
 **/
static void
eel_popup_menu_position_func (GtkMenu  *menu,
			      int      *x,
			      int      *y,
			      gboolean *push_in,
			      gpointer  user_data)
{
	GdkPoint *offset;
	GtkRequisition requisition;

	g_assert (x != NULL);
	g_assert (y != NULL);

	offset = (GdkPoint *) user_data;

	g_assert (offset != NULL);

	gtk_widget_size_request (GTK_WIDGET (menu), &requisition);
	  
	*x = CLAMP (*x + (int) offset->x, 0, MAX (0, gdk_screen_width () - requisition.width));
	*y = CLAMP (*y + (int) offset->y, 0, MAX (0, gdk_screen_height () - requisition.height));

	*push_in = FALSE;
}

/**
 * eel_truncate_text_for_menu_item:
 * 
 * Given an arbitrary string, returns a newly-allocated string
 * suitable for use as a menu item label. Truncates long strings 
 * in the middle.
 */
char *
eel_truncate_text_for_menu_item (const char *text)
{
	return eel_str_middle_truncate (text, MAXIMUM_MENU_TITLE_LENGTH);
}

/**
 * eel_pop_up_context_menu:
 * 
 * Pop up a context menu under the mouse.
 * The menu is sunk after use, so it will be destroyed unless the 
 * caller first ref'ed it.
 * 
 * This function is more of a helper function than a gtk extension,
 * so perhaps it belongs in a different file.
 * 
 * @menu: The menu to pop up under the mouse.
 * @offset_x: Number of pixels to displace the popup menu vertically
 * @offset_y: Number of pixels to displace the popup menu horizontally
 * @event: The event that invoked this popup menu.
 **/
void 
eel_pop_up_context_menu (GtkMenu	     *menu,
			      gint16	      offset_x,
			      gint16	      offset_y,
			      GdkEventButton *event)
{
	GdkPoint offset;
	int button;

	g_return_if_fail (GTK_IS_MENU (menu));

	offset.x = offset_x;
	offset.y = offset_y;

	/* The event button needs to be 0 if we're popping up this menu from
	 * a button release, else a 2nd click outside the menu with any button
	 * other than the one that invoked the menu will be ignored (instead
	 * of dismissing the menu). This is a subtle fragility of the GTK menu code.
	 */
	button = event->type == GDK_BUTTON_RELEASE
		? 0
		: event->button;
	
	gtk_menu_popup (menu,					/* menu */
			NULL,					/* parent_menu_shell */
			NULL,					/* parent_menu_item */
			eel_popup_menu_position_func,	/* func */
			&offset,			        /* data */
			button,					/* button */
			event->time);				/* activate_time */

	gtk_object_sink (GTK_OBJECT (menu));
}

GtkMenuItem *
eel_gtk_menu_append_separator (GtkMenu *menu)
{
	return eel_gtk_menu_insert_separator (menu, -1);
}

GtkMenuItem *
eel_gtk_menu_insert_separator (GtkMenu *menu, int index)
{
	GtkWidget *menu_item;

	menu_item = gtk_separator_menu_item_new ();
	gtk_widget_show (menu_item);
	gtk_menu_shell_insert (GTK_MENU_SHELL (menu), menu_item, index);

	return GTK_MENU_ITEM (menu_item);
}

void
eel_gtk_menu_set_item_visibility (GtkMenu *menu, int index, gboolean visible)
{
	GList *children;
	GtkWidget *menu_item;

	g_return_if_fail (GTK_IS_MENU (menu));

	children = gtk_container_children (GTK_CONTAINER (menu));
	g_return_if_fail (index >= 0 && index < (int) g_list_length (children));

	menu_item = GTK_WIDGET (g_list_nth_data (children, index));
	if (visible) {
		gtk_widget_show (menu_item);
	} else {
		gtk_widget_hide (menu_item);
	}

	g_list_free (children);
}

gboolean
eel_point_in_allocation (const GtkAllocation *allocation,
			      int x, int y)
{
	g_return_val_if_fail (allocation != NULL, FALSE);
	return x >= allocation->x
		&& y >= allocation->y
		&& x < allocation->x + allocation->width 
		&& y < allocation->y + allocation->height;
}

/* FIXME this function is dangerous, because widget->window coords (or
 * other window-belonging-to-widget coords) do not need to be in the
 * same coordinate system as widget->allocation.
 * If you use this function, be aware of that. Someone should probably
 * audit all uses, too.
 */
gboolean
eel_point_in_widget (GtkWidget *widget,
			  int x, int y)
{
	if (widget == NULL) {
		return FALSE;
	}
	g_return_val_if_fail (GTK_IS_WIDGET (widget), FALSE);
	return eel_point_in_allocation (&widget->allocation, x, y);
}

/**
 * eel_gtk_object_list_ref
 *
 * Ref all the objects in a list.
 * @list: GList of objects.
 **/
GList *
eel_gtk_object_list_ref (GList *list)
{
	g_list_foreach (list, (GFunc) g_object_ref, NULL);
	return list;
}

/**
 * eel_gtk_object_list_unref
 *
 * Unref all the objects in a list.
 * @list: GList of objects.
 **/
void
eel_gtk_object_list_unref (GList *list)
{
	g_list_foreach (list, (GFunc) g_object_unref, NULL);
}

/**
 * eel_gtk_object_list_free
 *
 * Free a list of objects after unrefing them.
 * @list: GList of objects.
 **/
void
eel_gtk_object_list_free (GList *list)
{
	eel_gtk_object_list_unref (list);
	g_list_free (list);
}

/**
 * eel_gtk_object_list_copy
 *
 * Copy the list of objects, ref'ing each one.
 * @list: GList of objects.
 **/
GList *
eel_gtk_object_list_copy (GList *list)
{
	return g_list_copy (eel_gtk_object_list_ref (list));
}

/**
 * eel_gtk_widget_set_shown
 *
 * Show or hide a widget.
 * @widget: The widget.
 * @shown: Boolean value indicating whether the widget should be shown or hidden.
 **/
void
eel_gtk_widget_set_shown (GtkWidget *widget, gboolean shown)
{
	g_return_if_fail (GTK_IS_WIDGET (widget));

	if (shown) {
		gtk_widget_show (widget);
	} else {
		gtk_widget_hide (widget);
	}
}

/**
 * eel_gtk_widget_set_font_by_name
 *
 * Sets the font for a widget, managing the font and style objects.
 * @widget: The widget
 **/
void
eel_gtk_widget_set_font_by_name (GtkWidget *widget, const char *font_name)
{
	PangoFontDescription *font_desc;
	
	g_return_if_fail (GTK_IS_WIDGET (widget));
	g_return_if_fail (font_name != NULL);

	font_desc = pango_font_description_from_string (font_name);

	if (font_desc == NULL) {
		g_warning ("Bad font name '%s'", font_name);
		return;
	}

	gtk_widget_modify_font (widget, font_desc);

	pango_font_description_free (font_desc);
}

/* This stuff is stolen from Gtk. */

typedef struct DisconnectInfo {
	GtkObject *object1;
	guint disconnect_handler1;
	guint signal_handler;
	GtkObject *object2;
	guint disconnect_handler2;
} DisconnectInfo;

static void
alive_disconnecter (GtkObject *object, DisconnectInfo *info)
{
	g_assert (info != NULL);
	g_assert (GTK_IS_OBJECT (info->object1));
	g_assert (info->disconnect_handler1 != 0);
	g_assert (info->signal_handler != 0);
	g_assert (GTK_IS_OBJECT (info->object2));
	g_assert (info->disconnect_handler2 != 0);
	g_assert (object == info->object1 || object == info->object2);
	
	gtk_signal_disconnect (info->object1, info->disconnect_handler1);
	gtk_signal_disconnect (info->object1, info->signal_handler);
	gtk_signal_disconnect (info->object2, info->disconnect_handler2);
	
	g_free (info);
}

/**
 * eel_gtk_signal_connect_full_while_alive
 *
 * Like gtk_signal_connect_while_alive, but works with full parameters.
 **/
void
eel_gtk_signal_connect_full_while_alive (GtkObject *object,
					 const gchar *name,
					 GtkSignalFunc func,
					 GtkCallbackMarshal marshal,
					 gpointer data,
					 GtkDestroyNotify destroy_func,
					 gboolean object_signal,
					 gboolean after,
					 GtkObject *alive_object)
{
	DisconnectInfo *info;
	
	g_return_if_fail (GTK_IS_OBJECT (object));
	g_return_if_fail (name != NULL);
	g_return_if_fail (func != NULL || marshal != NULL);
	g_return_if_fail (object_signal == FALSE || object_signal == TRUE);
	g_return_if_fail (after == FALSE || after == TRUE);
	g_return_if_fail (GTK_IS_OBJECT (alive_object));
	
	info = g_new (DisconnectInfo, 1);
	info->object1 = object;
	info->object2 = alive_object;
	
	info->signal_handler = gtk_signal_connect_full (object,
							name,
							func,
							marshal,
							data,
							destroy_func,
							object_signal,
							after);
	info->disconnect_handler1 = gtk_signal_connect (object,
							"destroy",
							G_CALLBACK (alive_disconnecter),
							info);
	info->disconnect_handler2 = gtk_signal_connect (alive_object,
							"destroy",
							G_CALLBACK (alive_disconnecter),
							info);
}

typedef struct
{
	GtkObject *object;
	guint object_destroy_handler;
	
	GtkWidget *realized_widget;
	guint realized_widget_destroy_handler;
	guint realized_widget_unrealized_handler;

	guint signal_handler;
} RealizeDisconnectInfo;

static void
while_realized_disconnecter (GtkObject *object,
			     RealizeDisconnectInfo *info)
{
	g_return_if_fail (GTK_IS_OBJECT (object));
	g_return_if_fail (info != NULL);
	g_return_if_fail (GTK_IS_OBJECT (info->object));
	g_return_if_fail (info->object_destroy_handler != 0);
	g_return_if_fail (info->object_destroy_handler != 0);
	g_return_if_fail (info->realized_widget_destroy_handler != 0);
	g_return_if_fail (info->realized_widget_unrealized_handler != 0);

 	gtk_signal_disconnect (info->object, info->object_destroy_handler);
 	gtk_signal_disconnect (info->object, info->signal_handler);
 	gtk_signal_disconnect (GTK_OBJECT (info->realized_widget), info->realized_widget_destroy_handler);
 	gtk_signal_disconnect (GTK_OBJECT (info->realized_widget), info->realized_widget_unrealized_handler);
	g_free (info);
}

/**
 * eel_gtk_signal_connect_while_realized:
 *
 * @object: Object to connect to.
 * @name: Name of signal to connect to.
 * @callback: Caller's callback.
 * @callback_data: Caller's callback_data.
 * @realized_widget: Widget to monitor for realized state.  Signal is connected
 *                   while this wigget is realized.
 *
 * Connect to a signal of an object while another widget is realized.  This is 
 * useful for non windowed widgets that need to monitor events in their ancestored
 * windowed widget.  The signal is automatically disconnected when &widget is
 * unrealized.  Also, the signal is automatically disconnected when either &object
 * or &widget are destroyed.
 **/
void
eel_gtk_signal_connect_while_realized (GtkObject *object,
					    const char *name,
					    GtkSignalFunc callback,
					    gpointer callback_data,
					    GtkWidget *realized_widget)
{
	RealizeDisconnectInfo *info;

	g_return_if_fail (GTK_IS_OBJECT (object));
	g_return_if_fail (name != NULL);
	g_return_if_fail (name[0] != '\0');
	g_return_if_fail (callback != NULL);
	g_return_if_fail (GTK_IS_WIDGET (realized_widget));
	g_return_if_fail (GTK_WIDGET_REALIZED (realized_widget));

	info = g_new0 (RealizeDisconnectInfo, 1);
	
	info->object = object;
	info->object_destroy_handler = 
		gtk_signal_connect (info->object,
				    "destroy",
				    G_CALLBACK (while_realized_disconnecter),
				    info);
	
	info->realized_widget = realized_widget;
	info->realized_widget_destroy_handler = 
		g_signal_connect (info->realized_widget,
				    "destroy",
				    G_CALLBACK (while_realized_disconnecter),
				    info);
	info->realized_widget_unrealized_handler = 
		gtk_signal_connect_after (GTK_OBJECT (info->realized_widget),
					  "unrealize",
					  G_CALLBACK (while_realized_disconnecter),
					  info);

	info->signal_handler = gtk_signal_connect (info->object, name, callback, callback_data);
}

/**
 * eel_nullify_when_destroyed.
 *
 * Nulls out a saved reference to an object when the object gets destroyed.
 * @data: Address of the saved reference.
 **/

void 
eel_nullify_when_destroyed (gpointer data)
{
	GObject **object_reference;

	object_reference = (GObject **)data;
	if (*object_reference == NULL) {
		/* the reference is  NULL, nothing to do. */
		return;
	}

	g_assert (G_IS_OBJECT (*object_reference));

	g_object_add_weak_pointer (*object_reference,
				   (void*) object_reference);
}

/**
 * eel_nullify_cancel.
 *
 * Disconnects the signal used to make eel_nullify_when_destroyed.
 * Used when the saved reference is no longer needed, the structure it is in is
 * being destroyed, etc. Nulls out the refernce when done.
 * @data: Address of the saved reference.
 **/

void 
eel_nullify_cancel (gpointer data)
{
	GObject **object_reference;

	object_reference = (GObject **)data;	
	if (*object_reference == NULL) {
		/* the object was already destroyed and the reference nulled out,
		 * nothing to do.
		 */
		return;
	}

	g_assert (G_IS_OBJECT (*object_reference));

	g_object_remove_weak_pointer (*object_reference,
				      (void*) object_reference);
	
	*object_reference = NULL;
}


/**
 * eel_gtk_container_get_first_child.
 *
 * Returns the first child of a container.
 * @container: The container.
 **/

static void
get_first_callback (GtkWidget *widget, gpointer callback_data)
{
	GtkWidget **first_child_slot;

	g_assert (GTK_IS_WIDGET (widget));
	g_assert (callback_data != NULL);
	
	first_child_slot = callback_data;

	if (*first_child_slot == NULL) {
		*first_child_slot = widget;
		/* We'd stop the iterating now if we could. */
	} else {
		g_assert (GTK_IS_WIDGET (*first_child_slot));
	}
}

GtkWidget *
eel_gtk_container_get_first_child (GtkContainer *container)
{
	GtkWidget *first_child;

	g_return_val_if_fail (GTK_IS_CONTAINER (container), NULL);
	
	first_child = NULL;
	gtk_container_foreach (container, get_first_callback, &first_child);
	g_assert (first_child == NULL || GTK_IS_WIDGET (first_child));
	return first_child;
}

typedef struct {
	GtkCallback   callback;
	gpointer      callback_data;
} container_foreach_deep_callback_data;

static void
container_foreach_deep_callback (GtkWidget *child, gpointer data)
{
	container_foreach_deep_callback_data *deep_data;

	deep_data = (container_foreach_deep_callback_data *) data;

	deep_data->callback (child, deep_data->callback_data);

	if (GTK_IS_CONTAINER (child)) {
		gtk_container_foreach (GTK_CONTAINER (child), container_foreach_deep_callback, data);
	}
}

void
eel_gtk_container_foreach_deep (GtkContainer *container,
				     GtkCallback callback,
				     gpointer callback_data)
{
	container_foreach_deep_callback_data deep_data;
	deep_data.callback = callback;
	deep_data.callback_data = callback_data;
	gtk_container_foreach (container, container_foreach_deep_callback, &deep_data);
}

/* The standard gtk_adjustment_set_value ignores page size, which
 * disagrees with the logic used by scroll bars, for example.
 */
void
eel_gtk_adjustment_set_value (GtkAdjustment *adjustment,
				   float value)
{
	float upper_page_start, clamped_value;

	g_return_if_fail (GTK_IS_ADJUSTMENT (adjustment));
	
	upper_page_start = MAX (adjustment->upper - adjustment->page_size, adjustment->lower);
	clamped_value = CLAMP (value, adjustment->lower, upper_page_start);
	if (clamped_value != adjustment->value) {
		adjustment->value = clamped_value;
		gtk_adjustment_value_changed (adjustment);
	}
}

/* Clamp a value if the minimum or maximum has changed. */
void
eel_gtk_adjustment_clamp_value (GtkAdjustment *adjustment)
{
	g_return_if_fail (GTK_IS_ADJUSTMENT (adjustment));
	
	eel_gtk_adjustment_set_value (adjustment, adjustment->value);
}

/**
 * eel_gtk_label_make_bold.
 *
 * Switches the font of label to a bold equivalent.
 * @label: The label.
 **/
void
eel_gtk_label_make_bold (GtkLabel *label)
{
	PangoFontDescription *font_desc;

	font_desc = pango_font_description_new ();

	pango_font_description_set_weight (font_desc,
					   PANGO_WEIGHT_BOLD);

	/* This will only affect the weight of the font, the rest is
	 * from the current state of the widget, which comes from the
	 * theme or user prefs, since the font desc only has the
	 * weight flag turned on.
	 */
	gtk_widget_modify_font (GTK_WIDGET (label), font_desc);

	pango_font_description_free (font_desc);
}

/**
 * eel_gtk_label_set_scale:
 * @label: 
 * @num_steps: 
 *
 * Function is broken, see eel_gtk_label_make_larger() for explanation
 * 
 **/
void
eel_gtk_label_set_scale (GtkLabel *label,
			 double scale_factor)
{
	PangoAttrList *old_attr_list;
	PangoAttrList *attr_list;
	
	g_return_if_fail (GTK_IS_LABEL (label));
	g_return_if_fail (scale_factor > 0);

	old_attr_list = gtk_label_get_attributes (label);
	attr_list = eel_pango_attr_list_apply_global_attribute (old_attr_list,
								pango_attr_scale_new (scale_factor));
	gtk_label_set_attributes (label, attr_list);
	pango_attr_list_unref (attr_list);
}

void
eel_gtk_widget_set_background_color (GtkWidget *widget,
				     const char *color_spec)
{
	GdkColor color;

	g_return_if_fail (GTK_IS_WIDGET (widget));

	eel_gdk_color_parse_with_white_default (color_spec, &color);

	gtk_widget_modify_bg (widget, GTK_STATE_NORMAL, &color);
	gtk_widget_modify_base (widget, GTK_STATE_NORMAL, &color);
	gtk_widget_modify_bg (widget, GTK_STATE_ACTIVE, &color);
	gtk_widget_modify_base (widget, GTK_STATE_ACTIVE, &color);
}

void
eel_gtk_widget_set_foreground_color (GtkWidget *widget,
				     const char *color_spec)
{
	GdkColor color;

	g_return_if_fail (GTK_IS_WIDGET (widget));

	eel_gdk_color_parse_with_white_default (color_spec, &color);

	gtk_widget_modify_fg (widget, GTK_STATE_NORMAL, &color);
	gtk_widget_modify_text (widget, GTK_STATE_NORMAL, &color);
	gtk_widget_modify_fg (widget, GTK_STATE_ACTIVE, &color);
	gtk_widget_modify_text (widget, GTK_STATE_ACTIVE, &color);
}

GtkWidget *
eel_gtk_widget_find_windowed_ancestor (GtkWidget *widget)
{
	g_return_val_if_fail (GTK_IS_WIDGET (widget), NULL);

	while (widget && GTK_WIDGET_NO_WINDOW (widget)) {
		widget = widget->parent;
	}

	return widget;
}



/*following code shamelessly stolen from gtk*/
static void
rgb_to_hls (gdouble *r,
            gdouble *g,
            gdouble *b)
{
	gdouble min;
	gdouble max;
	gdouble red;
	gdouble green;
	gdouble blue;
	gdouble h, l, s;
	gdouble delta;

	red = *r;
	green = *g;
	blue = *b;

	if (red > green) {
		if (red > blue)
			max = red;
		else
			max = blue;
		
		if (green < blue)
			min = green;
		else
			min = blue;
	} else {
		if (green > blue)
			max = green;
		else
			max = blue;
		
		if (red < blue)
			min = red;
		else
			min = blue;
	}

	l = (max + min) / 2;
	s = 0;
	h = 0;

	if (max != min) {
		if (l <= 0.5)
			s = (max - min) / (max + min);
		else
			s = (max - min) / (2 - max - min);
		
		delta = max -min;
		if (red == max)
			h = (green - blue) / delta;
		else if (green == max)
			h = 2 + (blue - red) / delta;
		else if (blue == max)
			h = 4 + (red - green) / delta;
		
		h *= 60;
		if (h < 0.0)
			h += 360;
	}
	
	*r = h;
	*g = l;
	*b = s;
}

static void
hls_to_rgb (gdouble *h,
            gdouble *l,
            gdouble *s)
{
	gdouble hue;
	gdouble lightness;
	gdouble saturation;
	gdouble m1, m2;
	gdouble r, g, b;
	
	lightness = *l;
	saturation = *s;

	if (lightness <= 0.5)
		m2 = lightness * (1 + saturation);
	else
		m2 = lightness + saturation - lightness * saturation;
	m1 = 2 * lightness - m2;

	if (saturation == 0) {
		*h = lightness;
		*l = lightness;
		*s = lightness;
	} else {
		hue = *h + 120;
		while (hue > 360)
			hue -= 360;
		while (hue < 0)
			hue += 360;
		
		if (hue < 60)
			r = m1 + (m2 - m1) * hue / 60;
		else if (hue < 180)
			r = m2;
		else if (hue < 240)
			r = m1 + (m2 - m1) * (240 - hue) / 60;
		else
			r = m1;

		hue = *h;
		while (hue > 360)
			hue -= 360;
		while (hue < 0)
			hue += 360;
		
		if (hue < 60)
			g = m1 + (m2 - m1) * hue / 60;
		else if (hue < 180)
			g = m2;
		else if (hue < 240)
			g = m1 + (m2 - m1) * (240 - hue) / 60;
		else
			g = m1;
		
		hue = *h - 120;
		while (hue > 360)
			hue -= 360;
		while (hue < 0)
			hue += 360;
		
		if (hue < 60)
			b = m1 + (m2 - m1) * hue / 60;
		else if (hue < 180)
			b = m2;
		else if (hue < 240)
			b = m1 + (m2 - m1) * (240 - hue) / 60;
		else
			b = m1;
		
		*h = r;
		*l = g;
		*s = b;
	}
}

void
eel_gtk_style_shade (GdkColor *a,
			  GdkColor *b,
			  gdouble   k)
{

	gdouble red;
	gdouble green;
	gdouble blue;
	
	red = (gdouble) a->red / 65535.0;
	green = (gdouble) a->green / 65535.0;
	blue = (gdouble) a->blue / 65535.0;
	
	rgb_to_hls (&red, &green, &blue);

	green *= k;
	if (green > 1.0)
		green = 1.0;
	else if (green < 0.0)
		green = 0.0;

	blue *= k;
	if (blue > 1.0)
		blue = 1.0;
	else if (blue < 0.0)
		blue = 0.0;

	hls_to_rgb (&red, &green, &blue);

	b->red = red * 65535.0;
	b->green = green * 65535.0;
	b->blue = blue * 65535.0;
}

/**
 * eel_gtk_class_name_make_like_existing_type:
 * @class_name: The class name for the custom widget.
 * @existing_gtk_type: The GtkType of the existing GtkWidget.
 *
 * Make the given class name act like an existing GtkType for
 * gtk theme/style purposes.  This can be used by custom 
 * widget to emulate the styles of stock Gtk widgets.  For
 * example:
 *
 * eel_gtk_class_name_make_like_existing ("EelCustomButton",
 *                                             GTK_TYPE_BUTTON);
 *
 *
 * You should call this function only once from the class_init()
 * method of the custom widget.
 **/
void
eel_gtk_class_name_make_like_existing_type (const char *class_name,
						 GtkType existing_gtk_type)
{
	GtkWidget *temporary;
	GtkStyle *style;

	g_return_if_fail (class_name != NULL);

	temporary = gtk_widget_new (existing_gtk_type, NULL);
	gtk_widget_ensure_style (temporary);
		
	style = gtk_widget_get_style (temporary);

	if (style->rc_style != NULL) {
		gtk_rc_add_widget_class_style (style->rc_style, class_name);
	}
		
	gtk_object_sink (GTK_OBJECT (temporary));
}

/* helper function for eel_get_window_list_ordered_front_to_back () */
static GtkWidget *
window_at_or_below (int depth, Window xid, gboolean *keep_going)
{
	static Atom wm_state = 0;

	GtkWidget *widget;

	Atom actual_type;
	int actual_format;
	gulong nitems, bytes_after;
	gulong *prop;

	GdkWindow *window;
	gpointer data;

	Window root, parent, *children;
	int nchildren, i;

	if (wm_state == 0) {
		wm_state = XInternAtom (GDK_DISPLAY (), "WM_STATE", False);
	}

	/* Check if the window is a top-level client window.
	 * Windows will have a WM_STATE property iff they're top-level.
	 */
	if (XGetWindowProperty (GDK_DISPLAY (), xid, wm_state, 0, 1,
				False, AnyPropertyType, &actual_type,
				&actual_format, &nitems, &bytes_after,
				(guchar **) &prop) == Success
	    && prop != NULL && actual_format == 32 && prop[0] == NormalState)
	{
		/* Found a top-level window */

		if (prop != NULL) {
			XFree (prop);
		}

		/* Does GDK know anything about this window? */
		window = gdk_window_lookup (xid);
		if (window != NULL) {
			gdk_window_get_user_data (window, &data);
			if (data != NULL)
			{
				/* Found one of the widgets we're after */
				*keep_going = FALSE;
				return GTK_WIDGET (data);
			}
		}

		/* No point in searching past here. It's a top-level
		 * window, but not from this application.
		 */
		*keep_going = FALSE;
		return NULL;
	}

	/* Not found a top-level window yet, so keep recursing. */
	if (depth < MAXIMUM_WM_REPARENTING_DEPTH) {
		if (XQueryTree (GDK_DISPLAY (), xid, &root,
				&parent, &children, &nchildren) != 0)
		{
			widget = NULL;

			for (i = 0; *keep_going && i < nchildren; i++) {
				widget = window_at_or_below (depth + 1,
							     children[i],
							     keep_going);
			}

			if (children != NULL) {
				XFree (children);
			}

			if (! *keep_going) {
				return widget;
			}
		}
	}

	return NULL;
}

/* eel_get_window_list_ordered_front_to_back:
 *
 * Return a list of GtkWindows's, representing the stacking order (top to
 * bottom) of all windows (known to the local GDK).
 *
 * (Involves a large number of X server round trips, so call sparingly)
 */
GList *
eel_get_window_list_ordered_front_to_back (void)
{
	Window root, parent, *children;
	int nchildren, i;
	GList *windows;
	GtkWidget *widget;
	gboolean keep_going;

	/* There's a possibility that a window will be closed in
	 * the period between us querying the child-of-root windows
	 * and getting round to search _their_ children. So arrange
	 * for errors to be caught and ignored.
	 */

	gdk_error_trap_push ();

	windows = NULL;

	if (XQueryTree (GDK_DISPLAY (), GDK_ROOT_WINDOW (),
			&root, &parent, &children, &nchildren) != 0)
	{
		for (i = 0; i < nchildren; i++) {
			keep_going = TRUE;
			widget = window_at_or_below (0, children[i],
						     &keep_going);
			if (widget != NULL) {
				/* XQueryTree returns window in bottom ->
				 * top order, so consing up the list in
				 * the normal manner will reverse this
				 * giving the desired top -> bottom order
				 */
				windows = g_list_prepend (windows, widget);
			}
		}
		if (children != NULL) {
			XFree (children);
		}
	}

	gdk_flush ();
	gdk_error_trap_pop ();

	return windows;
}

/* eel_gtk_get_system_font:
 *
 * Return the system font as selected in the control center. Need to 
 * g_object_unref() the result when done with it.
 *
 * Perhaps there is a better way to figure out what that font is, but
 * the following is simple enough and it works.
 */
PangoFontDescription *
eel_gtk_get_system_font (void)
{
	GtkWidget *label;
	PangoFontDescription *font;

	label = gtk_label_new ("");
	
	gtk_widget_ensure_style (label);

	font = pango_font_description_copy (label->style->font_desc);

 	gtk_object_sink (GTK_OBJECT (label));

	return font;
}

/**
 * eel_gtk_widget_standard_realize:
 *
 * @widget: A GtkWidget (must be unrealized).
 *
 * A standard implementation of GtkWidget::realize for widgets
 * that need a GdkWindow (GTK_WIDGET_NO_WINDOW (widget) == FALSE)
 * 
 * You should call gtk_widget_set_events() in your "GtkWidget::initialize"
 * method to set the event mask for your widget.
 */
void
eel_gtk_widget_standard_realize (GtkWidget *widget)
{
	GdkWindowAttr attributes;
	gint attributes_mask;
	gint border_width;
	
	g_return_if_fail (GTK_IS_WIDGET (widget));
	g_return_if_fail (!GTK_WIDGET_REALIZED (widget));

	GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);

	/* Its a little weird to check for the specific container type
	 * here, but it makes this function convenient to use for
	 * for containers as well.
	 */
	border_width = 
		GTK_IS_CONTAINER (widget) ?
		GTK_CONTAINER (widget)->border_width :
		0;
	
	attributes.x = widget->allocation.x + border_width;
	attributes.y = widget->allocation.y + border_width;
	attributes.width = widget->allocation.width - 2 * border_width;
	attributes.height = widget->allocation.height - 2 * border_width;
	attributes.window_type = GDK_WINDOW_CHILD;
	attributes.wclass = GDK_INPUT_OUTPUT;
	attributes.visual = gtk_widget_get_visual (widget);
	attributes.colormap = gtk_widget_get_colormap (widget);
	attributes.event_mask = gtk_widget_get_events (widget);
	
	attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;
	
	widget->window = gdk_window_new (gtk_widget_get_parent_window (widget), &attributes, attributes_mask);
	gdk_window_set_user_data (widget->window, widget);
	
	widget->style = gtk_style_attach (widget->style, widget->window);
	gtk_style_set_background (widget->style, widget->window, GTK_STATE_NORMAL);
}

/**
 * eel_gtk_bin_standard_size_allocate:
 *
 * @widget: A GtkBin widget.
 * @allocation: GtkAllocation pointer from size_allocate method.
 *
 * An implementation of GtkWidget::size_allocate for GtkBin widgets.
 */
void
eel_gtk_bin_standard_size_allocate (GtkWidget *widget,
				    GtkAllocation *allocation)
{
	GtkWidget *child;
	GtkAllocation child_allocation;
	
	g_return_if_fail (GTK_IS_BIN (widget));
	g_return_if_fail (allocation != NULL);

	child = GTK_BIN (widget)->child;
	
	if (child != NULL) {
		g_return_if_fail (GTK_IS_WIDGET (child));
	}
	
	widget->allocation = *allocation;
	
	child_allocation.x = 0;
	child_allocation.y = 0;
	child_allocation.width = MAX (allocation->width - GTK_CONTAINER (widget)->border_width * 2, 0);
	child_allocation.height = MAX (allocation->height - GTK_CONTAINER (widget)->border_width * 2, 0);
	
	if (GTK_WIDGET_REALIZED (widget)) {
		gdk_window_move_resize (widget->window,
					allocation->x + GTK_CONTAINER (widget)->border_width,
					allocation->y + GTK_CONTAINER (widget)->border_width,
					child_allocation.width,
					child_allocation.height);
	}
	
	if (child != NULL) {
		gtk_widget_size_allocate (child, &child_allocation);
	}
}

/**
 * eel_gtk_bin_standard_size_request:
 *
 * @container: A GtkContainer widget.
 * @requisition: GtkRequisition pointer from size_request method.
 *
 * An implementation of GtkWidget::size_request for GtkBin widgets.
 */
void
eel_gtk_bin_standard_size_request (GtkWidget *widget,
				   GtkRequisition *requisition)
{
	GtkRequisition child_requisition;
	GtkWidget *child;
	
	g_return_if_fail (GTK_IS_BIN (widget));
	g_return_if_fail (requisition != NULL);

	child = GTK_BIN (widget)->child;

	if (child != NULL) {
		g_return_if_fail (GTK_IS_WIDGET (child));
	}
	
	requisition->width = GTK_CONTAINER (widget)->border_width * 2;
	requisition->height = GTK_CONTAINER (widget)->border_width * 2;
	
	if (child != NULL && GTK_WIDGET_VISIBLE (child)) {
		gtk_widget_size_request (child, &child_requisition);
		
		requisition->width += child_requisition.width;
		requisition->height += child_requisition.height;
	}
}

static PangoContext *
create_pango_ft2_context (PangoContext *base_context)
{
	PangoContext *context;
	double dpi_x, dpi_y;
			
	dpi_x = gdk_screen_width () * 25.4 / gdk_screen_width_mm ();
	dpi_y = gdk_screen_height () * 25.4 / gdk_screen_height_mm ();
	context = pango_ft2_get_context (dpi_x, dpi_y);

	pango_context_set_language (context, pango_context_get_language (base_context));
	pango_context_set_base_dir (context, pango_context_get_base_dir (base_context));
	pango_context_set_font_description (context, pango_context_get_font_description (base_context));

	return context;
}

PangoContext *
eel_gtk_widget_get_pango_ft2_context (GtkWidget *widget)
{
	static GQuark quark_widget_pango_ft2_context;
	PangoContext *context;

	if (quark_widget_pango_ft2_context == 0) {
		quark_widget_pango_ft2_context = g_quark_from_static_string ("eel-ft2-context");
	}
  
	context = g_object_get_qdata (G_OBJECT (widget), quark_widget_pango_ft2_context);
	if (context == NULL) {
		context = create_pango_ft2_context (gtk_widget_get_pango_context (widget));
		g_object_set_qdata_full (G_OBJECT (widget), quark_widget_pango_ft2_context,
					 context, g_object_unref);
	}
	
	return context;
}
