/* Simple socket operations
 *
 * Written by Matthias Hensler 1999, Copyright WSPse 1999
 * Licensed under GPL
 *
 * "Routines for socket operations"
 *
 * Created: Matthias Hensler, 1999/03/03
 * Updated: Matthias Hensler, 1999/11/20
 *          Matthias Hensler, 2006/07/30: fix includes
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <sys/un.h>
#ifndef SOCK_STREAM
#include <socketbits.h>
#endif
#include "mp3creat.h"
#ifdef HAVE_CTYPE_H
#include <ctype.h>
#endif

extern int cache_remain;
extern void print_stat_win(char *text);
extern int cache_remain;

static union {
  struct sockaddr    sa;
  struct sockaddr_in si;
  struct sockaddr_un su;
  } addr;

int open_socket (const char *address)
{
  int sockfd;                /* Descriptor for Socket */
  int connst;
  char *hostname, *servname; /* Pointers */
  char buffer[256];
  unsigned char protocol_typ;
  struct hostent *hp;
  struct servent *sp;

#ifdef DEBUG
  fprintf(stderr, "DEBUG: open_socket(\"%s\")\n", address);
#endif
  if(strlen(address) > 255) {
    print_stat_win(_("socketaddress is illegal (usage host:port)"));
    sleep(1);
    return -1;
  }

  memset( (char *) &addr, 0, sizeof(addr));
  hostname = strcpy(buffer, address);
  servname = strchr(buffer, ':');
  if(servname == NULL) {
    print_stat_win(_("socket is illegal (usage host:port)"));
    sleep(1);
    return -1;
  }
  *servname++ = 0;
  if(hostname == NULL) {
    print_stat_win(_("hostname missing (usage host:port)"));
    sleep(1);
    return -1;
  }
  if(servname == NULL) {
    print_stat_win(_("port missing (usage host:port)"));
    sleep(1);
    return -1;
  }

  cache_remain = 0;

  /* UNIX-Domain Socket */
  if((strcasecmp(hostname, "local")==0) || (strcasecmp(hostname, "unix")==0)) {
#ifdef DEBUG
    fprintf(stderr, "DEBUG: start filling parameters for UNIX-Domain socket\n");
#endif
    addr.su.sun_family = AF_UNIX;
    strcpy(addr.su.sun_path, servname);
    protocol_typ = 1;

  } else {

  /* INET Socket */
#ifdef DEBUG
    fprintf(stderr, "DEBUG: start filling parameters for INET socket\n");
#endif
    addr.si.sin_family = AF_INET;
    /* Find ip-address from hostname */
    if(strcmp(hostname, "*")==0) {
      addr.si.sin_addr.s_addr = INADDR_ANY;
    } else if(strcmp(hostname, "loopback")==0) {
      addr.si.sin_addr.s_addr = inet_addr("127.0.0.1");
    } else {
      addr.si.sin_addr.s_addr = inet_addr(hostname);
      if(addr.si.sin_addr.s_addr == -1) {
	print_stat_win(_("lookup hostname"));
#ifdef DEBUG
        fprintf(stderr, "DEBUG: Looking up hostname \"%s\"\n", hostname);
#endif
        hp = gethostbyname(hostname); /* get IP address */
        endhostent();                 /* close connection to DNS Server */
        if(hp == NULL) {
	  print_stat_win(_("hostname unknown"));
	  sleep(1);
          return -1;
        }
        addr.si.sin_addr.s_addr = ( (struct in_addr *) (hp->h_addr) )->s_addr;
      }
    }
#ifdef DEBUG
    fprintf(stderr, "DEBUG: got Hostaddress\n");
#endif
    /* find port from servicename */
    if (isdigit(*servname & 0xff)) {   /* Portaddress explicity given */
      addr.si.sin_port = htons(atoi(servname));
    } else {
      print_stat_win(_("lookup servicename"));
#ifdef DEBUG
      fprintf(stderr, "DEBUG: looking up servicename \"%s\"\n", servname);
#endif
      sp = getservbyname(servname, "tcp"); /* get port */
      endservent();                    /* close connection to /etc/services */
      if(sp == NULL) {
	sprintf(buffer, _("service \"%s\" is unknown"), servname);
	print_stat_win(buffer);
        return -1;
      }
      addr.si.sin_port = sp->s_port;
    }
    protocol_typ = 2;
  }

  /* open socket */
  print_stat_win(_("open connection to server"));
  if(protocol_typ == 1) { /* UNIX-Domain */
    sockfd = socket(addr.su.sun_family, SOCK_STREAM, 0);
  } else {                /* INET        */
    sockfd = socket(addr.si.sin_family, SOCK_STREAM, 0);
  }
  if(sockfd == -1) {
    print_stat_win(_("opening socket failed"));
#ifdef DEBUG
    perror("socket");
#endif
    return -1;
  }

#ifdef DEBUG
  fprintf(stderr, "DEBUG: success while opening socket.\n");
#endif

  /* connect socket */
  print_stat_win(_("connect to server"));
  if(protocol_typ == 1) { /* UNIX-Domain */
    connst = connect(sockfd, &addr.sa, sizeof(struct sockaddr_un));
  } else {                /* INET        */
    connst = connect(sockfd, &addr.sa, sizeof(struct sockaddr_in));
  }
  if(connst == -1) {
    print_stat_win(_("connecting socket failed"));
#ifdef DEBUG
    perror("connect");
#endif
    return -1;
  }

#ifdef DEBUG
  fprintf(stderr, "DEBUG: success while connecting socket.\n");
#endif

  connst = fcntl(sockfd, F_SETFL, O_NONBLOCK);
  if(connst == -1) {
    fprintf(stderr, _("setting socket to nonblock mode failed.\n"));
#ifdef DEBUG
    perror("fcntl");
#endif
    return -1;
  }

  cache_remain = 0;

  return (sockfd);           /* Return descriptor */
}

