/*******************************************************************************
 * Copyright (c) 2005 - 2006 Joel Cheuoua & others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Joel Cheuoua - initial API and implementation
 *******************************************************************************/
package org.eclipse.emf.codegen.jet.editor;

import java.io.File;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IPath;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;

/**
 * @author Joel Cheuoua
 */
public class JavaClassLoaderFactory {

  private Map classpathEntriesMap = new HashMap();

  private Map exportedClasspathEntriesMap = new HashMap();

  private HashSet exportedURLList;

  private class OrderedUniqueList extends ArrayList {
    
    private static final long serialVersionUID = 1L;

    public boolean add(Object element) {
      if (contains(element))
        return false;
      return super.add(element);
    }
  };
  public URLClassLoader getClassloader(IJavaProject javaProject, ClassLoader parent) {
    Collection classpathEntries = getClasspathEntries(javaProject);
    List classpathEntryFiles = new ArrayList();
    for (Iterator iter = classpathEntries.iterator(); iter.hasNext();) {
      URL entryURL = (URL) iter.next();
      if ("file".equals(entryURL.getProtocol())) {
        File entryFile = new File(entryURL.getFile());
        classpathEntryFiles.add(entryFile);
      }
    }
    File[] files = new File[classpathEntryFiles.size()];
    System.arraycopy(classpathEntryFiles.toArray(), 0, files, 0, files.length);
    JETURLClassLoader classLoader = new JETURLClassLoader(files, parent);
    return classLoader;
  }

  public URLClassLoader getClassloader(IJavaProject javaProject) {
    return getClassloader(javaProject, getClass().getClassLoader());
  }

  public Collection getClasspathEntries(IJavaProject javaProject) {
    // Return the cache value if any
    if (classpathEntriesMap.get(javaProject) != null)
      return (Collection) classpathEntriesMap.get(javaProject);
    try {
      if (!javaProject.exists())
        return Collections.EMPTY_LIST;
      IClasspathEntry[] entries = javaProject.getResolvedClasspath(true);

      // URL entries should be stored in an ordered list with no duplicates
      List urlList = new OrderedUniqueList();

      exportedURLList = new HashSet();
      IPath baseAbsolutePath = javaProject.getProject().getLocation();
      IPath outputAbsolutePath = baseAbsolutePath;
      if (javaProject.getOutputLocation().segmentCount() > 1)
        outputAbsolutePath = baseAbsolutePath.append(javaProject.getOutputLocation().removeFirstSegments(1));
      URL outputURL = new URL("file", null, outputAbsolutePath.toString());
      urlList.add(outputURL);
      exportedURLList.add(outputURL);
      for (int i = 0; i < entries.length; i++) {
        IClasspathEntry entry = entries[i];
        processEntry(urlList, entry);
      }

      // Update the exported Map
      exportedClasspathEntriesMap.put(javaProject, exportedURLList);

      // Cache the classpath list
      classpathEntriesMap.put(javaProject, urlList);

      return urlList;
    } catch (JavaModelException e) {
      JETEditorPlugin.getDefault().log(e);
    } catch (MalformedURLException e) {
      JETEditorPlugin.getDefault().log(e);
    }
    return null;
  }

  private void processEntry(List urlList, IClasspathEntry entry) throws MalformedURLException {
    // This source output ... always included & exported
    if (entry.getEntryKind() == IClasspathEntry.CPE_SOURCE) {
      IPath outputLocation = entry.getOutputLocation();
      if (outputLocation != null) {
        URL url = new URL("file", null, outputLocation.toString());
        exportedURLList.add(url);
        urlList.add(url);
      }
    }
    // Referenced project classpath. If this project is exported,
    // Then all *exported* entries are exported with respect to this project,
    else if (entry.getEntryKind() == IClasspathEntry.CPE_PROJECT) {
      IProject ijproject = ResourcesPlugin.getWorkspace().getRoot().getProject(entry.getPath().segment(0));
      IJavaProject ref = JavaCore.create(ijproject);
      Collection cpEntries = getClasspathEntries(ref);
      if (entry.isExported()) {
        Collection refExp = (Collection) exportedClasspathEntriesMap.get(ref);
        if (refExp != null)
          exportedURLList.addAll(refExp);
      }
      urlList.addAll(cpEntries);
    }
    // This is Directories classpath
    else if (entry.getEntryKind() == IClasspathEntry.CPE_CONTAINER) {
      IPath entryPath = entry.getPath();
      URL url = new URL("file", null, entryPath.toString());
      IResource res = ResourcesPlugin.getWorkspace().getRoot().findMember(entryPath);
      if (res != null && res.exists())
        url = new URL("file", null, res.getLocation().toString());
      if (entry.isExported())
        exportedURLList.add(url);
      urlList.add(url);
    }
    // This is Library classpath
    else if (entry.getEntryKind() == IClasspathEntry.CPE_LIBRARY) {
      IPath entryPath = entry.getPath();
      URL url = new URL("file", null, entryPath.toString());
      IResource res = ResourcesPlugin.getWorkspace().getRoot().findMember(entryPath);
      if (res != null && res.exists())
        url = new URL("file", null, res.getLocation().toString());
      if (entry.isExported())
        exportedURLList.add(url);
      urlList.add(url);
    }
    // This is Variables classpath
    else if (entry.getEntryKind() == IClasspathEntry.CPE_VARIABLE) {
      String variableName = entry.getPath().segment(0);
      IPath variablePath = JavaCore.getClasspathVariable(variableName);
      if (variablePath != null) {
        URL url = new URL("file", null, variablePath.toString());
        if (entry.isExported())
          exportedURLList.add(url);
        urlList.add(url);
      }
    }
  }

  /**
   * Gets the exported URL list which was generated by the method <code>getClasspathEntries</code>. This method
   * should always be called after <code>getClasspathEntries</code>.
   * 
   * @return
   * @see getClasspathEntries(IJavaProject)
   */
  public Collection getCurrentExportedURLList() {
    return exportedURLList;
  }

  public void removeFromClassPathEntryMap(IJavaProject javaProject) {
    classpathEntriesMap.remove(javaProject);
  }

  public void clearCaches() {
    classpathEntriesMap.clear();
    exportedClasspathEntriesMap.clear();
  }

  /**
   * @param javaProject
   * @return
   */
  public static URLClassLoader getClassLoader(IJavaProject javaProject) {
    JavaClassLoaderFactory factory = new JavaClassLoaderFactory();
    return factory.getClassloader(javaProject);
  }

  /**
   * @param javaProject
   * @return
   */
  public static URLClassLoader getClassLoader(IJavaProject javaProject, ClassLoader parent) {
    JavaClassLoaderFactory factory = new JavaClassLoaderFactory();
    return factory.getClassloader(javaProject, parent);
  }
}
