/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.kura.core.data;

import java.util.Date;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import java.util.function.UnaryOperator;
import org.eclipse.kura.core.data.AutoConnectStrategy;
import org.eclipse.kura.core.data.DataServiceOptions;
import org.eclipse.kura.message.store.StoredMessage;
import org.quartz.CronExpression;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ScheduleStrategy
implements AutoConnectStrategy {
    private static final Logger logger = LoggerFactory.getLogger(ScheduleStrategy.class);
    private final ScheduledExecutorService executor;
    private final CronExpression expression;
    private final long disconnectTimeoutMs;
    private final AutoConnectStrategy.ConnectionManager connectionManager;
    private final Supplier<Date> currentTimeProvider;
    private State state;
    private Optional<ScheduledFuture<?>> timeout = Optional.empty();
    private DataServiceOptions dataServiceOptions;

    public ScheduleStrategy(CronExpression expression, DataServiceOptions dataServiceOptions, AutoConnectStrategy.ConnectionManager connectionManager) {
        this(expression, dataServiceOptions.getConnectionScheduleDisconnectDelay() * 1000L, connectionManager, Executors.newSingleThreadScheduledExecutor(), Date::new, dataServiceOptions);
    }

    public ScheduleStrategy(CronExpression expression, long disconnectTimeoutMs, AutoConnectStrategy.ConnectionManager connectionManager, ScheduledExecutorService executor, Supplier<Date> currentTimeProvider, DataServiceOptions dataServiceOptions) {
        this.expression = expression;
        this.disconnectTimeoutMs = disconnectTimeoutMs;
        this.connectionManager = connectionManager;
        this.state = new AwaitConnectTime();
        this.executor = executor;
        this.currentTimeProvider = currentTimeProvider;
        this.dataServiceOptions = dataServiceOptions;
        this.updateState(State::onEnterState);
        executor.scheduleWithFixedDelay(new TimeShiftDetector(60000L), 0L, 1L, TimeUnit.MINUTES);
    }

    private void rescheduleTimeout(long timeoutMs) {
        this.cancelTimeout();
        this.timeout = Optional.of(this.executor.schedule(() -> this.updateState(State::onTimeout), timeoutMs, TimeUnit.MILLISECONDS));
    }

    private void cancelTimeout() {
        Optional<ScheduledFuture<?>> currentFuture = this.timeout;
        if (currentFuture.isPresent()) {
            currentFuture.get().cancel(false);
        }
    }

    private void updateState(UnaryOperator<State> transition) {
        this.executor.execute(() -> this.updateStateInternal(transition));
    }

    private void updateStateInternal(UnaryOperator<State> transitionFunction) {
        Optional<ScheduledFuture<?>> currentFuture = this.timeout;
        State nextState = (State)transitionFunction.apply(this.state);
        if (nextState != this.state) {
            logger.info("State change: {} -> {}", (Object)this.state.getClass().getSimpleName(), (Object)nextState.getClass().getSimpleName());
            currentFuture.ifPresent(c -> {
                boolean bl = c.cancel(false);
            });
            this.state = nextState;
            this.updateStateInternal(State::onEnterState);
        }
    }

    @Override
    public void shutdown() {
        this.executor.execute(() -> {
            this.cancelTimeout();
            this.executor.shutdown();
        });
        try {
            this.executor.awaitTermination(30L, TimeUnit.SECONDS);
        }
        catch (InterruptedException interruptedException) {
            Thread.currentThread().interrupt();
            logger.warn("Interrupted while waiting for executor shutdown");
        }
    }

    public void onConnectionEstablished() {
        this.updateState(State::onConnectionEstablished);
    }

    public void onDisconnecting() {
    }

    public void onDisconnected() {
        this.updateState(State::onConnectionLost);
    }

    public void onConnectionLost(Throwable cause) {
        this.updateState(State::onConnectionLost);
    }

    public void onMessageArrived(String topic, byte[] payload, int qos, boolean retained) {
    }

    public void onMessagePublished(int messageId, String topic) {
        this.updateState(State::onMessageEvent);
    }

    public void onMessageConfirmed(int messageId, String topic) {
        this.updateState(State::onMessageEvent);
    }

    @Override
    public void onPublishRequested(String topic, byte[] payload, int qos, boolean retain, int priority) {
        this.updateState(c -> this.state.onPublish(topic, payload, qos, retain, priority));
    }

    private class AwaitConnect
    implements State {
        private AwaitConnect() {
        }

        @Override
        public State onEnterState() {
            if (ScheduleStrategy.this.connectionManager.isConnected()) {
                return new AwaitDisconnectTime();
            }
            ScheduleStrategy.this.connectionManager.startConnectionTask();
            return this;
        }

        @Override
        public State onConnectionLost() {
            ScheduleStrategy.this.connectionManager.startConnectionTask();
            return this;
        }

        @Override
        public State onConnectionEstablished() {
            return new AwaitDisconnectTime();
        }
    }

    private class AwaitConnectTime
    implements State {
        private AwaitConnectTime() {
        }

        @Override
        public State onEnterState() {
            Optional<StoredMessage> dm = ScheduleStrategy.this.connectionManager.getNextMessage();
            if (dm.isPresent() && dm.get().getPriority() <= ScheduleStrategy.this.dataServiceOptions.getConnectionSchedulePriorityOverridePriority()) {
                logger.info("Priority message sent while disconnecting. Initiating Connection to send message with a high priority.");
                return new AwaitConnect();
            }
            Date now = (Date)ScheduleStrategy.this.currentTimeProvider.get();
            Date nextTick = ScheduleStrategy.this.expression.getNextValidTimeAfter(now);
            long delay = Math.max(1L, nextTick.getTime() - now.getTime());
            logger.info("Connection scheduled at {} in {} ms", (Object)nextTick, (Object)delay);
            ScheduleStrategy.this.rescheduleTimeout(delay);
            return this;
        }

        @Override
        public State onTimeout() {
            return new AwaitConnect();
        }

        @Override
        public State onPublish(String topic, byte[] payload, int qos, boolean retain, int priority) {
            if (ScheduleStrategy.this.dataServiceOptions.isConnectionSchedulePriorityOverrideEnabled() && priority <= ScheduleStrategy.this.dataServiceOptions.getConnectionSchedulePriorityOverridePriority() && !ScheduleStrategy.this.connectionManager.isConnected()) {
                logger.info("Initiating Connection to send message with a high priority.");
                return new AwaitConnect();
            }
            return this;
        }
    }

    private class AwaitDisconnect
    implements State {
        private AwaitDisconnect() {
        }

        @Override
        public State onEnterState() {
            ScheduleStrategy.this.connectionManager.stopConnectionTask();
            ScheduleStrategy.this.connectionManager.disconnect();
            return this;
        }

        @Override
        public State onConnectionLost() {
            return new AwaitConnectTime();
        }

        @Override
        public State onMessageEvent() {
            return this;
        }
    }

    private class AwaitDisconnectTime
    implements State {
        private AwaitDisconnectTime() {
        }

        @Override
        public State onEnterState() {
            return this.onMessageEvent();
        }

        @Override
        public State onConnectionLost() {
            return new AwaitConnect();
        }

        @Override
        public State onMessageEvent() {
            ScheduleStrategy.this.rescheduleTimeout(ScheduleStrategy.this.disconnectTimeoutMs);
            return this;
        }

        @Override
        public State onTimeout() {
            if (ScheduleStrategy.this.connectionManager.hasInFlightMessages()) {
                return this;
            }
            return new AwaitDisconnect();
        }
    }

    private static interface State {
        default public State onEnterState() {
            return this;
        }

        default public State onConnectionEstablished() {
            return this;
        }

        default public State onMessageEvent() {
            return this;
        }

        default public State onConnectionLost() {
            return this;
        }

        default public State onTimeout() {
            return this;
        }

        default public State onPublish(String topic, byte[] payload, int qos, boolean retain, int priority) {
            return this;
        }
    }

    private class TimeShiftDetector
    implements Runnable {
        private OptionalLong previousTimestamp = OptionalLong.empty();
        private final long expectedDelay;

        public TimeShiftDetector(long expectedTickRate) {
            this.expectedDelay = expectedTickRate;
        }

        @Override
        public void run() {
            long now = System.currentTimeMillis();
            OptionalLong previous = this.previousTimestamp;
            if (!previous.isPresent()) {
                this.previousTimestamp = OptionalLong.of(now);
                return;
            }
            if (now < previous.getAsLong() || Math.abs(now - previous.getAsLong() - this.expectedDelay) > 60000L) {
                logger.warn("Time shift detected, reinitializing connection schedule");
                ScheduleStrategy.this.updateState(c -> new AwaitConnectTime());
            }
            this.previousTimestamp = OptionalLong.of(now);
        }
    }
}

