/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.kura.core.internal.linux.executor;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.exec.CommandLine;
import org.apache.commons.exec.DefaultExecutor;
import org.apache.commons.exec.ExecuteException;
import org.apache.commons.exec.ExecuteResultHandler;
import org.apache.commons.exec.ExecuteStreamHandler;
import org.apache.commons.exec.ExecuteWatchdog;
import org.apache.commons.exec.Executor;
import org.apache.commons.exec.PumpStreamHandler;
import org.apache.commons.exec.environment.EnvironmentUtils;
import org.apache.commons.io.output.NullOutputStream;
import org.eclipse.kura.core.internal.linux.executor.FlushPumpStreamHandler;
import org.eclipse.kura.core.linux.executor.LinuxExitStatus;
import org.eclipse.kura.core.linux.executor.LinuxPid;
import org.eclipse.kura.core.linux.executor.LinuxResultHandler;
import org.eclipse.kura.core.linux.executor.LinuxSignal;
import org.eclipse.kura.executor.Command;
import org.eclipse.kura.executor.CommandStatus;
import org.eclipse.kura.executor.ExitStatus;
import org.eclipse.kura.executor.Pid;
import org.eclipse.kura.executor.Signal;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ExecutorUtil {
    private static final Logger logger = LoggerFactory.getLogger(ExecutorUtil.class);
    private static final String COMMAND_MESSAGE = "Command ";
    private static final String FAILED_TO_GET_PID_MESSAGE = "Failed to get pid for command '{}'";
    private static final File TEMP_DIR = new File(System.getProperty("java.io.tmpdir"));
    private static final String DEFAULT_COMMAND_USERNAME = "kura";
    private String commandUsername;

    public ExecutorUtil() {
        this.commandUsername = DEFAULT_COMMAND_USERNAME;
    }

    public ExecutorUtil(String commandUsername) {
        this.commandUsername = commandUsername;
    }

    public String getCommandUsername() {
        return this.commandUsername;
    }

    public void setCommandUsername(String commandUsername) {
        this.commandUsername = commandUsername;
    }

    public CommandStatus executeUnprivileged(Command command) {
        CommandLine commandLine = this.buildUnprivilegedCommand(command);
        return this.executeSync(command, commandLine);
    }

    public void executeUnprivileged(Command command, Consumer<CommandStatus> callback) {
        CommandLine commandLine = this.buildUnprivilegedCommand(command);
        this.executeAsync(command, commandLine, callback);
    }

    public CommandStatus executePrivileged(Command command) {
        CommandLine commandLine = this.buildPrivilegedCommand(command);
        return this.executeSync(command, commandLine);
    }

    public void executePrivileged(Command command, Consumer<CommandStatus> callback) {
        CommandLine commandLine = this.buildPrivilegedCommand(command);
        this.executeAsync(command, commandLine, callback);
    }

    public boolean stopUnprivileged(Pid pid, Signal signal) {
        boolean isStopped = true;
        if (this.isRunning(pid)) {
            Command killCommand = new Command(this.buildKillCommand(pid, signal));
            killCommand.setTimeout(60);
            killCommand.setSignal(signal);
            CommandStatus commandStatus = this.executeUnprivileged(killCommand);
            isStopped = commandStatus.getExitStatus().isSuccessful();
        }
        return isStopped;
    }

    public boolean killUnprivileged(String[] commandLine, Signal signal) {
        boolean isKilled = true;
        Map<String, Pid> pids = this.getPids(commandLine);
        for (Pid pid : pids.values()) {
            isKilled &= this.stopUnprivileged(pid, signal);
        }
        return isKilled;
    }

    public boolean stopPrivileged(Pid pid, Signal signal) {
        boolean isStopped = true;
        if (this.isRunning(pid)) {
            Command killCommand = new Command(this.buildKillCommand(pid, signal));
            killCommand.setTimeout(60);
            killCommand.setSignal(signal);
            CommandStatus commandStatus = this.executePrivileged(killCommand);
            isStopped = commandStatus.getExitStatus().isSuccessful();
        }
        return isStopped;
    }

    public boolean killPrivileged(String[] commandLine, Signal signal) {
        boolean isKilled = true;
        Map<String, Pid> pids = this.getPids(commandLine);
        for (Pid pid : pids.values()) {
            isKilled &= this.stopPrivileged(pid, signal);
        }
        return isKilled;
    }

    public boolean isRunning(Pid pid) {
        boolean isRunning = false;
        String pidString = Integer.valueOf(pid.getPid()).toString();
        String psCommand = "ps -p " + pidString;
        CommandLine commandLine = CommandLine.parse((String)psCommand);
        Executor executor = this.getExecutor();
        ByteArrayOutputStream out = this.createStream();
        ByteArrayOutputStream err = this.createStream();
        PumpStreamHandler handler = new PumpStreamHandler((OutputStream)out, (OutputStream)err);
        executor.setStreamHandler((ExecuteStreamHandler)handler);
        executor.setWorkingDirectory(TEMP_DIR);
        executor.setExitValue(0);
        try {
            int exitValue = executor.execute(commandLine);
            if (exitValue == 0 && new String(out.toByteArray(), StandardCharsets.UTF_8).contains(pidString)) {
                isRunning = true;
            }
        }
        catch (IOException iOException) {
            logger.warn("Failed to check if process with pid {} is running", (Object)pidString);
        }
        return isRunning;
    }

    public boolean isRunning(String[] commandLine) {
        return !this.getPids(commandLine).isEmpty();
    }

    public Map<String, Pid> getPids(String[] commandLine) {
        Map<String, Pid> pids = new HashMap<String, Pid>();
        CommandLine psCommandLine = new CommandLine("ps");
        psCommandLine.addArgument("-ax");
        Executor executor = this.getExecutor();
        ByteArrayOutputStream out = this.createStream();
        ByteArrayOutputStream err = this.createStream();
        PumpStreamHandler handler = new PumpStreamHandler((OutputStream)out, (OutputStream)err);
        executor.setStreamHandler((ExecuteStreamHandler)handler);
        executor.setWorkingDirectory(TEMP_DIR);
        executor.setExitValue(0);
        int exitValue = 1;
        try {
            exitValue = executor.execute(psCommandLine);
        }
        catch (IOException e) {
            logger.debug(FAILED_TO_GET_PID_MESSAGE, (Object)commandLine, (Object)e);
        }
        if (exitValue == 0) {
            pids = this.parsePids(out, commandLine);
        }
        return pids;
    }

    public static void stopStreamHandler(Executor executor) {
        try {
            if (executor.getStreamHandler() != null) {
                executor.getStreamHandler().stop();
            }
        }
        catch (IOException e) {
            logger.warn("Failed to stop stream handlers", (Throwable)e);
        }
    }

    private Map<String, Pid> parsePids(ByteArrayOutputStream out, String[] commandLine) {
        String[] output;
        HashMap<String, Integer> pids = new HashMap<String, Integer>();
        String[] stringArray = output = new String(out.toByteArray(), StandardCharsets.UTF_8).split("\n");
        int n = output.length;
        int n2 = 0;
        while (n2 < n) {
            String line = stringArray[n2];
            StringTokenizer st = new StringTokenizer(line);
            String pid = st.nextToken();
            st.nextElement();
            st.nextElement();
            st.nextElement();
            line = line.substring(line.indexOf(st.nextToken()));
            if (this.checkLine(line, commandLine)) {
                pids.put(line, Integer.parseInt(pid));
            }
            ++n2;
        }
        return pids.entrySet().stream().sorted(Map.Entry.comparingByValue().reversed()).collect(Collectors.toMap(Map.Entry::getKey, e -> new LinuxPid((Integer)e.getValue()), (e1, e2) -> e1, LinkedHashMap::new));
    }

    private boolean checkLine(String line, String[] tokens) {
        return ((Stream)Arrays.stream(tokens).parallel()).allMatch(line::contains);
    }

    private CommandStatus executeSync(Command command, CommandLine commandLine) {
        CommandStatus commandStatus = new CommandStatus(command, (ExitStatus)new LinuxExitStatus(0));
        commandStatus.setOutputStream(command.getOutputStream());
        commandStatus.setErrorStream(command.getErrorStream());
        commandStatus.setInputStream(command.getInputStream());
        Executor executor = this.configureExecutor(command);
        int exitStatus = 0;
        logger.debug("Executing: {}", (Object)commandLine);
        try {
            try {
                Map environment = command.getEnvironment();
                if (environment != null && !environment.isEmpty()) {
                    Map currentEnv = EnvironmentUtils.getProcEnvironment();
                    currentEnv.putAll(environment);
                    exitStatus = executor.execute(commandLine, currentEnv);
                } else {
                    exitStatus = executor.execute(commandLine);
                }
            }
            catch (ExecuteException e) {
                exitStatus = e.getExitValue();
                logger.debug("Command  {} returned error code {}", new Object[]{commandLine, exitStatus, e});
                ExecutorUtil.stopStreamHandler(executor);
                commandStatus.setExitStatus((ExitStatus)new LinuxExitStatus(exitStatus));
                commandStatus.setTimedout(executor.getWatchdog().killedProcess());
            }
            catch (IOException e) {
                exitStatus = 1;
                logger.debug("Command  {} failed", (Object)commandLine, (Object)e);
                ExecutorUtil.stopStreamHandler(executor);
                commandStatus.setExitStatus((ExitStatus)new LinuxExitStatus(exitStatus));
                commandStatus.setTimedout(executor.getWatchdog().killedProcess());
            }
        }
        finally {
            ExecutorUtil.stopStreamHandler(executor);
            commandStatus.setExitStatus((ExitStatus)new LinuxExitStatus(exitStatus));
            commandStatus.setTimedout(executor.getWatchdog().killedProcess());
        }
        return commandStatus;
    }

    private Executor configureExecutor(Command command) {
        Executor executor = this.getExecutor();
        int timeout = command.getTimeout();
        ExecuteWatchdog watchdog = timeout <= 0 ? new ExecuteWatchdog(-1L) : new ExecuteWatchdog((long)timeout * 1000L);
        executor.setWatchdog(watchdog);
        OutputStream out = command.getOutputStream();
        OutputStream err = command.getErrorStream();
        InputStream in = command.getInputStream();
        FlushPumpStreamHandler handler = out != null && err != null ? new FlushPumpStreamHandler(out, err, in) : (out != null ? new FlushPumpStreamHandler(out, (OutputStream)new NullOutputStream(), in) : (err != null ? new FlushPumpStreamHandler((OutputStream)new NullOutputStream(), err, in) : new FlushPumpStreamHandler((OutputStream)new NullOutputStream(), (OutputStream)new NullOutputStream(), in)));
        executor.setStreamHandler((ExecuteStreamHandler)handler);
        String directory = command.getDirectory();
        File workingDir = directory == null || directory.isEmpty() || !Files.isDirectory(Paths.get(directory, new String[0]), new LinkOption[0]) ? TEMP_DIR : new File(directory);
        executor.setWorkingDirectory(workingDir);
        return executor;
    }

    protected Executor getExecutor() {
        return new DefaultExecutor();
    }

    private void executeAsync(Command command, CommandLine commandLine, Consumer<CommandStatus> callback) {
        CommandStatus commandStatus = new CommandStatus(command, (ExitStatus)new LinuxExitStatus(0));
        commandStatus.setOutputStream(command.getOutputStream());
        commandStatus.setErrorStream(command.getErrorStream());
        commandStatus.setInputStream(command.getInputStream());
        Executor executor = this.configureExecutor(command);
        LinuxResultHandler resultHandler = new LinuxResultHandler(callback, executor);
        resultHandler.setStatus(commandStatus);
        logger.debug("Executing: {}", (Object)commandLine);
        try {
            Map environment = command.getEnvironment();
            if (environment != null && !environment.isEmpty()) {
                Map currentEnv = EnvironmentUtils.getProcEnvironment();
                currentEnv.putAll(environment);
                executor.execute(commandLine, currentEnv, (ExecuteResultHandler)resultHandler);
            } else {
                executor.execute(commandLine, (ExecuteResultHandler)resultHandler);
            }
        }
        catch (IOException e) {
            ExecutorUtil.stopStreamHandler(executor);
            commandStatus.setExitStatus((ExitStatus)new LinuxExitStatus(1));
            logger.error(COMMAND_MESSAGE + commandLine + " failed", (Throwable)e);
        }
    }

    private String[] buildKillCommand(Pid pid, Signal signal) {
        Integer pidNumber = pid.getPid();
        if (logger.isInfoEnabled()) {
            logger.info("Attempting to send {} to process with pid {}", (Object)((LinuxSignal)signal).name(), (Object)pidNumber);
        }
        return new String[]{"kill", "-" + signal.getSignalNumber(), String.valueOf(pidNumber)};
    }

    private CommandLine buildUnprivilegedCommand(Command command) {
        int timeout;
        CommandLine commandLine = new CommandLine("su");
        commandLine.addArgument(this.commandUsername);
        commandLine.addArgument("-c");
        ArrayList<String> c = new ArrayList<String>();
        Map env = command.getEnvironment();
        if (env != null && !env.isEmpty()) {
            env.entrySet().stream().forEach(entry -> {
                boolean bl = c.add(String.valueOf((String)entry.getKey()) + "=" + (String)entry.getValue());
            });
        }
        if ((timeout = command.getTimeout()) != -1) {
            c.add("timeout");
            c.add("-s");
            c.add(((LinuxSignal)command.getSignal()).name());
            c.add(Integer.toString(timeout));
        }
        Arrays.asList(command.getCommandLine()).stream().forEach(c::add);
        commandLine.addArgument(String.join((CharSequence)" ", c), false);
        return commandLine;
    }

    private CommandLine buildPrivilegedCommand(Command command) {
        CommandLine commandLine;
        if (command.isExecutedInAShell()) {
            commandLine = new CommandLine("/bin/sh");
            commandLine.addArgument("-c");
            commandLine.addArgument(command.toString(), false);
        } else {
            String[] tokens = command.getCommandLine();
            commandLine = new CommandLine(tokens[0]);
            int i = 1;
            while (i < tokens.length) {
                commandLine.addArgument(tokens[i], false);
                ++i;
            }
        }
        return commandLine;
    }

    protected ByteArrayOutputStream createStream() {
        return new ByteArrayOutputStream();
    }
}

