/**
 * Copyright (c) 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package fr.inria.diverse.melange.utils;

import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import fr.inria.diverse.melange.lib.EcoreExtensions;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.xtext.diagnostics.Severity;
import org.eclipse.xtext.validation.EObjectDiagnosticImpl;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

@SuppressWarnings("all")
public class ErrorHelper {
  @Inject
  @Extension
  private EcoreExtensions _ecoreExtensions;
  
  private final static String DEFAULT_PROBLEM_CODE = "MelangeProblemCode";
  
  public void addError(final EObject eo, final String message) {
    this.addError(eo, message, null);
  }
  
  /**
   * Add an error diagnostic (and marker) to an EObject
   */
  public void addError(final EObject eo, final String message, final String problemCode) {
    EList<Resource.Diagnostic> _errors = eo.eResource().getErrors();
    String _elvis = null;
    if (problemCode != null) {
      _elvis = problemCode;
    } else {
      _elvis = ErrorHelper.DEFAULT_PROBLEM_CODE;
    }
    EObjectDiagnosticImpl _eObjectDiagnosticImpl = new EObjectDiagnosticImpl(
      Severity.ERROR, _elvis, message, eo, 
      null, 
      (-1), 
      null);
    _errors.add(_eObjectDiagnosticImpl);
  }
  
  /**
   * Add a warning diagnostic (and marker) to an EObject
   */
  public void addWarning(final EObject eo, final String message, final String problemCode) {
    EList<Resource.Diagnostic> _errors = eo.eResource().getErrors();
    String _elvis = null;
    if (problemCode != null) {
      _elvis = problemCode;
    } else {
      _elvis = ErrorHelper.DEFAULT_PROBLEM_CODE;
    }
    EObjectDiagnosticImpl _eObjectDiagnosticImpl = new EObjectDiagnosticImpl(
      Severity.WARNING, _elvis, message, eo, 
      null, 
      (-1), 
      null);
    _errors.add(_eObjectDiagnosticImpl);
  }
  
  /**
   * Checks whether {@link o} or one of its children
   * has an error marker produced by the validator
   */
  public boolean isInError(final EObject o) {
    final Function1<EObjectDiagnosticImpl, Boolean> _function = new Function1<EObjectDiagnosticImpl, Boolean>() {
      @Override
      public Boolean apply(final EObjectDiagnosticImpl it) {
        return Boolean.valueOf(ErrorHelper.this._ecoreExtensions.isContainedBy(it.getProblematicObject(), o));
      }
    };
    return IterableExtensions.<EObjectDiagnosticImpl>exists(Iterables.<EObjectDiagnosticImpl>filter(o.eResource().getErrors(), EObjectDiagnosticImpl.class), _function);
  }
}
