/*
 * Copyright (c) 2015 the original author or authors.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Simon Scholz (vogella GmbH) - initial API and implementation and initial documentation
 */

package org.eclipse.buildship.ui.internal.view.execution;

import java.util.List;

import org.gradle.tooling.events.OperationDescriptor;
import org.gradle.tooling.events.test.JvmTestOperationDescriptor;

import com.google.common.base.Predicate;
import com.google.common.collect.FluentIterable;
import com.google.common.collect.ImmutableList;

import org.eclipse.jface.action.Action;

import org.eclipse.buildship.core.internal.console.ProcessDescription;
import org.eclipse.buildship.ui.internal.util.nodeselection.NodeSelection;
import org.eclipse.buildship.ui.internal.util.nodeselection.SelectionSpecificAction;

/**
 * Opens the sources files of the the selected test operation items. Delegates to
 * {@link OpenTestSourceFileJob} to do the actual search and navigation.
 *
 * @see OpenTestSourceFileJob
 */
public final class OpenTestSourceFileAction extends Action implements SelectionSpecificAction {

    private final ExecutionPage executionPage;

    public OpenTestSourceFileAction(ExecutionPage executionPage) {
        super(ExecutionViewMessages.Action_OpenTestSourceFile_Text);
        this.executionPage = executionPage;
    }

    @Override
    public void run() {
        ProcessDescription processDescription = this.executionPage.getProcessDescription();
        List<OperationItem> selectedTestOperationItems = this.executionPage.getSelection().toList(OperationItem.class);
        OpenTestSourceFileJob openTestSourceFileJob = new OpenTestSourceFileJob(selectedTestOperationItems, processDescription.getRunConfig());
        openTestSourceFileJob.schedule();
    }

    @Override
    public boolean isVisibleFor(NodeSelection selection) {
        if (selection.isEmpty()) {
            return false;
        }

        if (!selection.hasAllNodesOfType(OperationItem.class)) {
            return false;
        }

        ImmutableList<OperationItem> operationItems = selection.toList(OperationItem.class);
        return FluentIterable.from(operationItems).allMatch(new Predicate<OperationItem>() {

            @Override
            public boolean apply(OperationItem operationItem) {
                OperationDescriptor adapter = (OperationDescriptor) operationItem.getAdapter(OperationDescriptor.class);
                return adapter instanceof JvmTestOperationDescriptor;
            }
        });
    }

    @Override
    public boolean isEnabledFor(NodeSelection selection) {
        if (selection.isEmpty()) {
            return false;
        }

        if (!selection.hasAllNodesOfType(OperationItem.class)) {
            return false;
        }

        ImmutableList<OperationItem> operationItems = selection.toList(OperationItem.class);
        return FluentIterable.from(operationItems).allMatch(new Predicate<OperationItem>() {

            @Override
            public boolean apply(OperationItem operationItem) {
                OperationDescriptor adapter = (OperationDescriptor) operationItem.getAdapter(OperationDescriptor.class);
                return adapter instanceof JvmTestOperationDescriptor && ((JvmTestOperationDescriptor) adapter).getClassName() != null;
            }
        });
    }

    @Override
    public void setEnabledFor(NodeSelection selection) {
        setEnabled(isEnabledFor(selection));
    }

}
