/*
Copyright (C) 2016, 2017 Siep Kroonenberg

This file is part of TLaunch.

TLaunch is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

TLaunch is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with TLaunch.  If not, see <http://www.gnu.org/licenses/>.
*/

// Main program: launcher for TeX Live.
// First-time initialization includes adjustment of user searchpath,
// setting up file associations and creating an forgetter/un-initializer.

#include "tla_utils.h"
#include <time.h>
#include "tlaunch.h"
#include "versioninfo.h"

// some globals only needed here
wchar_t * tlaunchbase; // not used for forget
wchar_t * this_exe; // _wpgmptr, except that M$ declares this obsolete
int is_forgetter = 0; // original launcher or copy under configdir?
// try a name which does not trigger windows' checks for (un)installers
wchar_t * forget_basename;

// optional argument to program invocation
wchar_t * prog_arg = NULL;
// for debugging:
wchar_t cmd_global[MAX_PATH];
// The code below is the easy way to initialize an array of structs and
// its length, but cannot be used in a shared header file, because then
// these variables would be multiply defined. Therefore,
// wchar_t * perl_envs is declared in tlaunch.h and set to perl_envs_arr
// in set_simple_globals, where its length nperl_envs is also calculated.

PerlEnv perl_envs_arr [] = {
  { L"PERL5OPT", NULL },
  { L"PERLIO", NULL },
  { L"PERLIO_DEBUG", NULL },
  { L"PERLLIB", NULL },
  { L"PERL5DB", NULL },
  { L"PERL5DB_THREADED", NULL },
  { L"PERL5SHELL", NULL },
  { L"PERL_ALLOW_NON_IFS_LSP", NULL },
  { L"PERL_DEBUG_MSTATS", NULL },
  { L"PERL_DESTRUCT_LEVEL", NULL },
  { L"PERL_DL_NONLAZY", NULL },
  { L"PERL_ENCODING", NULL },
  { L"PERL_HASH_SEED", NULL },
  { L"PERL_HASH_SEED_DEBUG", NULL },
  { L"PERL_ROOT", NULL },
  { L"PERL_SIGNALS", NULL },
  { L"PERL_UNICODE", NULL },
};
// the val members of the array elements will be retrieved right after
// processing the ini file

// note that globals without explicit initialization
// are already automatically set to 0 or NULL by the compiler
// so some of the initializations below are no-ops.

void set_simple_globals() {

  errmess = NULL; // global variable for errors and warnings
  errkeep = 0; // replace errmess next time
  keep_capts = 0; // do not keep capture files

  // log file; failure to create is no error
  { DWORD maxl = MAX_PATH + wcslen( L"TeXLive_Launcher.log" ) + 1;
    wchar_t tmp_path[ maxl ];
    DWORD ln;
    logstream = NULL;
    ln = GetTempPath( maxl, tmp_path );
    if( ln <= MAX_PATH ) {
      logpath = tl_wcsnconcat( 2, tmp_path, L"TeXLive_Launcher.log" );
      logstream = _wfopen( logpath, L"w" );
      // if still NULL, wr_log will do nothing
    }
  }

  version = NULL;

  // basename of forgetter
  forget_basename = L"tl_forget";

  // basename of current exe
  this_exe = get_own_path( );
  if( _wcsicmp( this_exe + wcslen( this_exe ) - 4, L".exe" ))
      tldie( NULL, L"Unexpected filename: %ls", this_exe );

  do_searchpath = 1;
  // default filetype policy (0: do nothing; 1: do not overwrite; 2: always):
  assoc_mode = 1;
  // roaming fix: create startup item to recreate missing file associations
  create_rememberer = 0;

  ed_initdone = FALSE;

  perl_envs = perl_envs_arr;
  nperl_envs = ((int)( sizeof( perl_envs_arr ) / sizeof( perl_envs[0] )));

  // latex-related extensions
  latex_exts[0] = L".cls";
  latex_exts[1] = L".sty";
  latex_exts[2] = L".tex";
  tl_wasprintf( &lx_exts, L"%ls %ls %ls",
      latex_exts[0], latex_exts[1], latex_exts[2]);

  // default and custom editor
  ed_pgid = NULL;
  cust_pgid = NULL;

  // announcement string
  tl_announce = NULL;

} // set_simple_globals

// Programs subdirectory of the start menu
wchar_t * get_startmenudir( int global ) {
  wchar_t * menuprefix;
  menuprefix = get_shell_folder(
      global ? FOLDERID_CommonPrograms : FOLDERID_Programs );
  // get_shell_folder may return NULL, but never an empty string
  return menuprefix;
} // get_startmenudir

int user_config_exists ( ) {
  wchar_t * forgetpath;
  int ret = 1;
  forgetpath = tl_wcsnconcat( 4, configdir, L"\\", forget_basename, L".exe" );
  if( !FILE_EXISTS( forgetpath )) {
    ret = 0;
  } else {
    wcscpy( forgetpath + wcslen( forgetpath ) - 3, L"ini" );
    if( !FILE_EXISTS( forgetpath )) ret = 0;
  }
  FREE0( forgetpath );
  return ret;
} // user_config_exists

wchar_t * tl_shortcut_path( wchar_t * target, int global ) {
  wchar_t * mdir, * msh;
  if( !wcscmp( target, L"remember" )) {
    mdir = get_shell_folder( FOLDERID_Startup );
    msh = tl_wcsnconcat( 2, mdir, L"\\tl_FTAs.lnk" );
  } else {
    mdir = get_startmenudir( global );
    if( !wcscmp( target, L"dir" ))
      msh = tl_wcsnconcat( 3, mdir, L"\\TeX Live ", version );
    else if( !wcscmp( target, L"forget" ))
      msh = tl_wcsnconcat( 4, mdir, L"\\TeX Live ", version,
        L"\\Forget settings.lnk" );
    else if( !wcscmp( target, L"launcher" ))
      msh = tl_wcsnconcat( 4, mdir, L"\\TeX Live ", version,
          L"\\Launcher.lnk" );
    else
      tldie( NULL, L"Illegal parameter %ls to tl_shortcut_path", target );
  }
  FREE0( mdir );
  return msh;
} // tl_shortcut_path

// TL uninstall registry key, not including HKCU or HKLM
// regular run: key for user settings
// tl [un]installation run: for actual TL installation
// key does NOT depend on tl_name
wchar_t * tl_regkey_path( wchar_t * target ) {
  wchar_t * ret;
  wchar_t * uninst_path =
      L"software\\microsoft\\windows\\currentversion\\uninstall\\TeXLive";
  if( !wcscmp( target, L"forget" ))
    ret = tl_wcsnconcat( 3, uninst_path, version, L" Settings" );
  else if( !wcscmp( target, L"tl" ))
    ret = tl_wcsnconcat( 2, uninst_path, version );
  else {
    tldie( NULL, L"Illegal parameter %ls to tl_regkey_path", target );
    return NULL;
  }
  FREE0( uninst_path );
  return ret;
} // tl_regkey_path

wchar_t * kpse_answer( wchar_t * question ) {
  wchar_t * command, * out_buf;
  command = tl_wcsnconcat( 4, L"\"", tlroot,
        L"\\bin\\win32\\kpsewhich.exe\" ", question );
  out_buf = capture_output( command, FALSE );
  FREE0( command );
  if( out_buf && !out_buf[0] ) FREE0( out_buf );
  return out_buf;
} // kpse_answer

// first-time initialization: path, file associations, forgetter

int is_latex_ext( wchar_t * ext ) {
  int i;
  if( !ext || !( *ext )) return 0;
  for( i=0; i<NEXTS; i++ ) {
    if( !_wcsicmp( ext, latex_exts[i] )) return 1;
  }
  return 0;
} // is_latex_ext

void do_file_assocs( wchar_t ** mess ) {
  if( assoc_mode ) {
    int i, exts_len;
    wchar_t * ext, * exts, * p;
    wr_log( L"Doing file associations\n" );
    for( i=0; i<nprogs; i++ ) {
      Pgid * pgid = pgids[i];
      DWORD tp;
      if( !pgid->shell_cmd ) { // skip invalid pgids including cust_pgid
        continue;
      }
      // register progid
      tp = wcschr( pgid->command, L'%' ) ? REG_EXPAND_SZ : REG_SZ;
      if( !register_progid( pgid->progid, pgid->shell_cmd, tp,
          pgid->iconspec, pgid->name )) {
        add_to_mess( mess,
            L"Failed to register filetype %ls", pgid->progid );
        continue;
      }
      // register command, with or without path prefix
      if( pgid->basename ) {
        wchar_t * prog;
        prog = prog_from_cmd( pgid->command );
        register_prog( prog, pgid->path_prefix ? tlbin : NULL );
        FREE0( prog );
      }
      // loop through its extensions for primary and secondary associations,
      // skipping the latex extensions as to primary associations
      if( pgid->extensions && pgid->command ) {
        exts_len = wcslen( pgid->extensions );
        exts = tl_wcscpy( pgid->extensions );
        ext = exts;
        while( ext && *ext ) {
          // find next separator/space
          p = wcschr( ext, L' ' );
          if( p ) *p = L'\0'; // now ext is 1 extension
          if( is_latex_ext( ext ) || !pgid->primary ) {
            // here only secondary association
            set_2nd_assoc( ext, pgid->progid );
          } else {
            if( !set_assoc( ext, pgid->progid, assoc_mode ))
              add_to_mess( mess, L"Failed to associate %ls "
                  L"with filetype %ls", ext, pgid->progid );
          }
          if( !p ) break;
          ext = ( p < ( exts + exts_len )) ? p + 1 : NULL;
        } // end running through extensions
        FREE0( exts );
      }
    } // end running through pgids
    // now also primary association for default editor.
    // there is at least a slot for a custom editor
    if( assoc_mode > 1 ) {
      ed_pgid = first_ed;
      if( !ed_pgid ) ed_pgid = cust_pgid;
    }
    if( ed_pgid && ed_pgid->shell_cmd ) {
      for( i=0; i<NEXTS; i++ ) {
      if( !set_assoc( latex_exts[i], ed_pgid->progid, assoc_mode ))
        add_to_mess( mess, L"Failed to associate %ls with filetype %ls",
            latex_exts[i], ed_pgid->progid );
      }
    }
  }
}


// Note. Re-initialization consists of forgetting and restarting.
void first_time( wchar_t ** mess ) {
  wchar_t * str = NULL;
  *mess = NULL;

  // run pre_config batchfile, if any, e.g. for forgetting previous version
  if( pre_config && FILE_EXISTS( pre_config )) {
    wr_log( L"Running %ls\n", pre_config );
    str = capture_tmp_quoted( pre_config, TRUE );
    if( str ) {
      add_to_mess( mess, L"%ls", str );
      FREE0( str );
    }
  }

  // user path
  if( do_searchpath ) {
    wr_log( L"Doing searchpath\n" );
    if( !modify_user_path( tlbin, 1 )) {
      add_to_mess( mess, L"Failed to add %ls to user searchpath", tlbin );
    }
  }

  // file associations
  do_file_assocs( mess );

  // create forgetter: copy launcher and ini file
  // to configdir/tl_forget.[exe|ini]
  // mainly useful if TL itself was not installed by the current user,
  // but also convenient for automatically undoing config
  // when activating/installing a newer version
  {
    wr_log( L"Creating forgetter\n" );
    wchar_t * forget_file = NULL, * ini_path, *keypath;
    BYTE * buf;
    __int64 exesize; // is signed
    HANDLE h;
    DWORD written, lread;
    int ok = 1; // valid forgetter?
    int is_tl_user_install;
    if( !configdir ) {
      add_to_mess( mess,
          L"No config directory specified, should not have arrived here" );
      tldie( NULL, *mess );
    }
    if( !mkdir_recursive( configdir )) {
      add_to_mess( mess, L"Failed to create %ls", configdir );
      ok = 0;
      goto end_forget;
    }
    // used in forget; failure not fatal
    forget_file = tl_wcsnconcat( 4, configdir, L"\\", forget_basename,
        L".ini" );

    // Parameters CreateFile for creating or overwriting a file:
    // 1. filename
    // 2. desired access: GENERIC_READ | GENERIC_WRITE
    // 3. share mode: in the current case, 0 (exclusive access) is ok.
    //    this access mode holds while the handle is open.
    // 4. security attributes. this is the reason why we do not use CopyFile
    //    which copies security attributes; we just want
    //    default attributes. So we set this parameter NULL.
    // 5. CreationDisposition, set to CREATE_ALWAYS i.e. overwrite
    //    if already existing
    // 6. flags; FILE_ATTRIBUTE_NORMAL will do
    // 7. optional template handle; set to NULL
    // Return value is a file handle or otherwise INVALID_HANDLE_VALUE

    h = CreateFile( forget_file, GENERIC_READ|GENERIC_WRITE, 0,
        NULL, CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL );
    if( h == INVALID_HANDLE_VALUE ) {
      add_to_mess( mess, L"Failed to create %ls", forget_file );
      h = NULL;
      ok = 0;
      goto end_forget;
    }
    // first write root section, with definition for tlroot
    tl_wasprintf( &str, L"[Root]\r\ntlroot=%ls\r\n", tlroot );
    // note: write strings without terminating 0
    ok = WriteFile( h, (void *)str, wcslen( str ) * sizeof( wchar_t ),
          &written, NULL );
    FREE0( str );
    if( !ok ) {
      CloseHandle( h );
      h = NULL;
      add_to_mess( mess, L"Problem writing %ls", forget_file );
      goto end_forget;
    }
    // add VERSION, and TEXMFVAR and TEXMFCONFIG if applicable
    str = tl_wcsnconcat( 3, L"version=", version, L"\r\n" );
    ok = WriteFile( h, (void *)str, wcslen( str ) * sizeof( wchar_t ),
        &written, NULL );
    FREE0( str );
    if( !ok ) {
      CloseHandle( h );
      h = NULL;
      add_to_mess( mess, L"Problem writing %ls", forget_file );
      goto end_forget;
    }
    if( texmfvar ) {
      str = tl_wcsnconcat( 3, L"texmfvar=", texmfvar, L"\r\n" );
      ok = WriteFile( h, (void *)str, wcslen( str ) * sizeof( wchar_t ),
          &written, NULL );
      FREE0( str );
      if( !ok ) {
        CloseHandle( h );
        h = NULL;
        add_to_mess( mess, L"Problem writing %ls", forget_file );
        goto end_forget;
      }
    }
    if( texmfconfig ) {
      str = tl_wcsnconcat( 3, L"texmfconfig=", texmfconfig, L"\r\n" );
      ok = WriteFile( h, (void *)str, wcslen( str ) * sizeof( wchar_t ),
          &written, NULL );
      FREE0( str );
      if( !ok ) {
        CloseHandle( h );
        h = NULL;
        add_to_mess( mess, L"Problem writing %ls", forget_file );
        goto end_forget;
      }
    }
    // append ini file wholesale
    ini_path = tl_wcsnconcat( 4, tlroot, L"\\", tlaunchbase, L".ini" );
    if( !FILE_EXISTS( ini_path )) {
      FREE0( ini_path );
      str = tl_wcsnconcat( 3, L"--format=web2c ", tlaunchbase, L".ini" );
      ini_path = kpse_answer( str );
      FREE0( str );
    }
    if( !ini_path )
        // this shouldn't happen since the launcher just read this file
        tldie( NULL, L"Config file %ls.ini disappeared", tlaunchbase );
    str = slurp_file_to_wcs( ini_path, MAX_CONFIG );
    ok = WriteFile( h, (void *)str, wcslen( str ) * sizeof(wchar_t ),
        &written, NULL );
    FREE0( str );
    CloseHandle( h );
    h = NULL;
    FREE0( ini_path );
    if( !ok ) {
      add_to_mess( mess, L"Problem writing %ls", forget_file );
      goto end_forget;
    }
    // done with forget ini file; now copy the launcher executable.
    // first, copy exe to in-memory buffer buf
    exesize = win_filesize( this_exe );
    buf = slurp_file( this_exe, exesize, &lread );
    // launcher binary read to buf; now write buf
    // just need to change extension of forget_file
    wcscpy( forget_file + wcslen( forget_file ) - 3, L"exe" );
    h = CreateFile( forget_file, GENERIC_READ|GENERIC_WRITE, 0,
        NULL, CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL );
    if( h == INVALID_HANDLE_VALUE ) {
      add_to_mess( mess, L"Failed to create %ls", forget_file );
      ok = 0;
      goto end_forget;
    }
    ok = WriteFile( h, buf, lread, &written, NULL );
    FREE0( buf );
    CloseHandle( h );
    if( !ok ) {
      add_to_mess( mess, L"Cannot copy launcher" );
      goto end_forget;
    } // done with actual forgetter

    // TL user install? If so, no forgetter shortcut or registry entry
    keypath = tl_regkey_path( L"tl" );
    is_tl_user_install = reg_value_exists(
        HKEY_CURRENT_USER, keypath, L"UninstallString" );
    FREE0( keypath );

    // forgetter shortcut;  make_shortcut creates directory if necessary
    if( !is_tl_user_install ) {
      str = tl_shortcut_path( L"forget", 0 );
      if( !make_shortcut( forget_file, NULL, str )) {
        add_to_mess( mess, L"Cannot create shortcut for forgetter (%ls)",
            forget_file );
        rmdir_ifempty( str );
      }
      FREE0( str );
    }
    // rememberer restores missing filetype associations on login;
    // might be needed even in a TL user install
    // if TL was installed on a network drive
    if( create_rememberer ) {
      str = tl_shortcut_path( L"remember", 0 );
      if( !make_shortcut( forget_file, L"remember", str ))
        add_to_mess( mess, L"Cannot create rememberer shortcut" );
      FREE0( str );
    }

    // Create registry entry for the forgetter, to convince windows
    // that it was a proper and successful 'installation'.
    if( !is_tl_user_install ) {
      int ok_reg = 1;
      keypath = tl_regkey_path( L"forget" );
      str = tl_wcsnconcat( 2, tl_name, L" Settings" );
      if( !reg_set_stringvalue( HKEY_CURRENT_USER, keypath, L"DisplayName",
          str )) ok_reg = 0;
      FREE0( str );
      if( !reg_set_stringvalue( HKEY_CURRENT_USER, keypath, L"DisplayVersion",
          version )) ok_reg = 0;
      str = tl_wcsnconcat( 3, L"\"", forget_file, L"\"" );
      if( !reg_set_stringvalue( HKEY_CURRENT_USER, keypath, L"UninstallString",
          str )) ok_reg = 0;
      FREE0( str );
      // the next two are purely cosmetic; no error checking
      reg_set_stringvalue( HKEY_CURRENT_USER, keypath, L"Publisher",
        L"User settings" );
      reg_set_stringvalue( HKEY_CURRENT_USER, keypath, L"URLInfoAbout",
        TLA_URL );
      FREE0( keypath );
      if( !ok_reg ) add_to_mess( mess,
          L"Problems creating registry entry for forgetter" );
      FREE0( keypath );
    }
    // cleanup: forget files? configdir?
    end_forget:;
    if( forget_file ) FREE0( forget_file );
    if( !ok && configdir && ( FILE_EXISTS( configdir )))
      rmdir_recursive( configdir );
  } // done creating forgetter

  // run post-inst batchfile, if any.
  // provide some useful environment variables,
  // including tlroot, since its calculation is cumbersome
  if( post_config && FILE_EXISTS( post_config )) {
    wchar_t * tlroot_old, * tlconfig_old;
    wr_log( L"Running %ls\n", post_config );
    // no special handling necessary for NULL values
    tlroot_old = get_env( L"TLROOT" );
    SetEnvironmentVariable( L"TLROOT", tlroot );
    tlconfig_old = get_env( L"TLCONFIG" );
    SetEnvironmentVariable( L"TLCONFIG", configdir );

    str = capture_tmp_quoted( post_config, TRUE );
    if( str ) {
      add_to_mess( mess, L"%ls", str );
      FREE0( str );
    }
    SetEnvironmentVariable( L"TLROOT", tlroot_old );
    FREE0( tlroot_old );
    SetEnvironmentVariable( L"TLCONFIG", tlconfig_old );
    FREE0( tlconfig_old );
  }
} // first_time

// called by clear_tl and forget_self
void forget( wchar_t ** mess ) {
  wchar_t * ext, * exts, * p, * str;
  int i, exts_len;

  // run pre-forget batchfile, if any
  if( pre_forget && FILE_EXISTS( pre_forget )) {
    add_to_mess( mess, L"Running pre-forget" );
    str = capture_tmp_quoted( pre_forget, TRUE );
    if( str ) {
      add_to_mess( mess, L"%ls", str );
      FREE0( str );
    }
  }

  // file associations
  for( i=0; i<nprogs; i++ ) {
    Pgid * pgid = pgids[i];
    // loop through all extensions for primary and secondary associations
    if( pgid->extensions ) {
      exts_len = wcslen( pgid->extensions );
      exts = tl_wcscpy( pgid->extensions );
      ext = exts;
      while( ext && *ext ) {
        // find next separator/space
        p = wcschr( ext, L' ' );
        if( p ) *p = L'\0'; // now ext is 1 extension
        if( !remove_assoc( ext, pgid->progid ))
          add_to_mess( mess,
            L"Failed to remove association %ls with filetype %ls",
            ext, pgid->progid );
        if( !p ) break;
        ext = ( p < ( exts + exts_len )) ? p + 1 : NULL;
      } // end running through extensions
      FREE0( exts );
    }
    // unregister progid
    if( !remove_progid( pgid->progid )) {
      add_to_mess( mess,
          L"Failed to remove filetype %ls", pgid->progid );
      continue;
    }
    // unregister prog
    if( pgid->basename ) unregister_prog( pgid->basename );
  } // end running through pgids

  add_to_mess( mess, L"Finished clearing file associations" );

  // texmfvar and texmfconfig and their parent, if empty.
  // kpse_answer will return NULL rather than an empty string.

  if( !texmfvar ) texmfvar = kpse_answer( L"-var-value=TEXMFVAR" );
  if( texmfvar ) {
    wchar_subst( texmfvar, L'/', L'\\' );
    if( FILE_EXISTS( texmfvar )) rmdir_recursive( texmfvar );
    if( FILE_EXISTS( texmfvar ))
      add_to_mess( mess, L"Failed to delete texmfvar: %ls", texmfvar );
  }
  if( !texmfconfig ) texmfconfig = kpse_answer( L"-var-value=TEXMFCONFIG" );
  if( texmfconfig ) {
    wchar_subst( texmfconfig, L'/', L'\\' );
    if( FILE_EXISTS( texmfconfig )) rmdir_recursive( texmfconfig );
    if( FILE_EXISTS( texmfconfig ))
      add_to_mess( mess, L"Failed to delete texmfconfig: %ls", texmfconfig );
  }
  if( texmfvar ) {
    // remove parent if it exists and is empty
    p = wcsrchr( texmfvar, L'\\' );
    if( p && p - texmfvar > 0 ) {
      DWORD last_err;
      *p = L'\0';
      // RemoveDirectory only removes non-empty directories
      if( FILE_EXISTS( texmfvar )) {
        if( !RemoveDirectory( texmfvar )) {
          last_err = GetLastError();
          if( last_err == ERROR_DIR_NOT_EMPTY )
            add_to_mess( mess, L"%ls not removed because non-empty",
                texmfvar );
          else
            add_to_mess( mess, L"%ls not removed because of error %ul",
                texmfvar, last_err );
        }
      }
    }
  }

  // user path
  if( !modify_user_path( tlbin, 0 ))
    add_to_mess( mess, L"Failed to remove %ls from user searchpath", tlbin );

  // remove forgetter shortcut and its subdirectory if empty;
  // leave removing forgetter to caller, i.e. uninstall-self or clear_tl
  str = tl_shortcut_path( L"forget", 0 );
  if( !delete_file( str ))
    add_to_mess( mess, L"Failed to remove forgetter shortcut %ls", str );
  p = wcsrchr( str, L'\\' );
  if( p ) {
    *p = L'\0'; // now str is just the shortcut directory
    rmdir_ifempty( str );
  }
  FREE0( str );

  // remove rememberer shortcut but not its subdirectory
  str = tl_shortcut_path( L"remember", 0 );
  if( !delete_file( str ))
    add_to_mess( mess, L"Failed to remove rememberer shortcut %ls", str );
  FREE0( str );

  // remove registry key
  str = tl_regkey_path( L"forget" );
  if( !reg_del_key( HKEY_CURRENT_USER, str ))
    add_to_mess( mess,
        L"Failed to remove %ls settings registry entry", tl_name );
  FREE0( str );
  // the caller should handle any accumulated messages and terminate.
} // forget

// remove directory recursively, even if it contains the current exe,
// by creating a batchfile that does the job and removes itself besides
void remove_selfdir( wchar_t * dirname ) {
  wchar_t * cmd, * sysroot, * tfname;
  BOOL ok = FALSE;
  FILE * BFILE;
  PROCESS_INFORMATION pi;
  STARTUPINFO si;
  get_tempfilename( L".cmd", &tfname );
  if( tfname ) {
    BFILE = _wfopen( tfname, L"w" );
    if( BFILE ) {
      fwprintf( BFILE,
        L"timeout /t 3 2>NUL\r\n"
        L"for %%%%I in (a b c d e f g h i j k l m n) do (\r\n"
        L"  if not exist \"%ls\" goto delbatch\r\n"
        L"  rmdir /s /q \"%ls\"\r\n"
        L")\r\n"
        L":delbatch\r\n"
        L"del \"%ls\"\r\n",
        dirname, dirname, tfname );
      fclose( BFILE );
      if( win_filesize( tfname ) > 0 ) ok = TRUE;
    }
  }
  if( ok ) {
    cmd = tl_wcsnconcat( 3, L"cmd /c \"", tfname, L"\"" );
    // create child process
    ZeroMemory( &pi, sizeof( pi ));
    ZeroMemory( &si, sizeof( si ));
    si.cb = sizeof(STARTUPINFO);
    si.dwFlags = STARTF_USESHOWWINDOW;
    si.wShowWindow = SW_HIDE;
    // cmd should NOT run from the to be deleted directory
    sysroot = get_env( L"SystemRoot" );
    if( CreateProcess(
        NULL,
        cmd,
        NULL,
        NULL,
        FALSE,
        CREATE_NO_WINDOW,
        NULL,
        sysroot,
        &si,
        &pi )) {
      CloseHandle( pi.hThread ); // this does NOT kill anything
      CloseHandle( pi.hProcess );
    } else {
      ok = FALSE;
    }
  }
  if( !ok ) tldie( NULL, L"Cannot remove %ls", dirname );
  ExitProcess( GetLastError( ));
} // remove_selfdir

// used when program is called as forgetter
void forget_self( int do_feedback ) {
  wchar_t * mess = NULL;
  int ans;
  if( !do_feedback ) // invoked from first-time regular run of newer version
    ans = IDOK;
  else
    ans = MessageBox( NULL, L"Remove TL settings?", tl_name, MB_OKCANCEL );
  if( ans != IDOK ) tldie( NULL, L"Aborting." );
  forget( &mess );
  // can't come back to report; instead, display warnings so far now
  if( mess && do_feedback ) tlinfo( NULL, L"%ls", mess );
  remove_selfdir( configdir ); // also exits program
} // uninstall-self

// redo file associations on login.  this will be triggered by an
// optional shortcut created in FOLDERID_Startup, i.e. normally in
// /Users/<userid>/appdata/roaming/microsoft/windows/
// /start menu/programs/startup.
// Run completely silent; no error reporting
void remember( ) {
  wchar_t * mess = NULL;
  do_file_assocs( &mess );
  wr_log( mess );
}
// TODO: parser should detect remember option

void restart_elevated( wchar_t * action, int do_feedback ) {
  BOOL have_com;
  SHELLEXECUTEINFO shei;
  // initializing COM is recommended before invoking ShellExecute,
  // just in case, but I believe it is not needed here
  have_com = init_com();
  // this is the recommended way to get an elevation prompt!!!
  shei.cbSize = sizeof(SHELLEXECUTEINFO);
  shei.fMask = SEE_MASK_NOASYNC;
  // according to MS, this flag is necessary if the caller
  // exits immediately afterwards
  shei.hwnd = NULL;
  shei.lpVerb = L"runas";
  shei.lpFile = this_exe;
  if( do_feedback ) shei.lpParameters = action;
  else shei.lpParameters = tl_wcsnconcat( 2, action, L" silent" );
  shei.lpDirectory = NULL;
  shei.nShow = SW_SHOW;
  shei.hInstApp = NULL;
  if ( !ShellExecuteEx( &shei )) tlwarn( NULL, L"Failed to elevate" );
  if( have_com ) uninit_com();
  ExitProcess( 0 );
} // restart-elevated

// installing or uninstalling TL

void do_inst_uninst( wchar_t * action, int do_feedback ) {
  HKEY hk;
  BOOL is_el, wants_el;
  wchar_t * keypath, * mess = NULL, * p;
  int ok = 1, ok_reg = 1, ok_menu = 1;

  // prelims: elevation

  // elevation available?
  is_el = is_elevated();

  // uninst registry entry
  keypath = tl_regkey_path( L"tl" );

  // elevation required?
  if( !wcscmp( L"uninst_all", action ) || !wcscmp( L"uninst", action ))
    wants_el = reg_key_exists( HKEY_LOCAL_MACHINE, keypath );
  else // user_inst, admin_inst
    wants_el = !wcscmp( L"admin_inst", action );

  // if necessary, restart with a UAC prompt
  if( wants_el && !is_el ) {
    restart_elevated( action, do_feedback );
    ExitProcess( 1 );
  }

  hk = wants_el ? HKEY_LOCAL_MACHINE : HKEY_CURRENT_USER;

  // sufficient elevation, appropriate registry hive
  // time to go ahead

  if( !wcscmp( action, L"uninst_all" ) || !wcscmp( action, L"uninst")) {
    int ans, do_all;
    wchar_t * confirm, * entrypath;

    do_all = !wcscmp( action, L"uninst_all" ) ? 1 : 0;

    if( do_feedback ) {
      // ask for confirmation
      tl_wasprintf(
        &confirm, L"About to uninstall %ls;\n\n"
        "also remove all TeX Live files (%ls)?", tl_name, tlroot );
      ans = MessageBox(
        NULL, confirm, tl_name, MB_YESNOCANCEL | MB_ICONQUESTION |
        ( do_all ? MB_DEFBUTTON1 : MB_DEFBUTTON2 ));
    } else {
      ans = do_all ? IDYES : IDNO;
    }
    if( ans == IDCANCEL ) {
      // if( !delete_file( lockfile )) {
      //   add_to_mess( &mess,
      //   L"Cannot remove lockfile %ls, please do this manually", lockfile );
      // }
      add_to_mess( &mess, L"Aborting..." );
      tldie( NULL, L"%ls", mess );
      return;
    }

    // run forget function if forgetter exists; no feedback
    if( user_config_exists( )) {
      forget( &mess );
      FREE0( mess );
      // remove forgetter directory
      rmdir_recursive( configdir );
    }

    // remove TL registry entry
    FREE0( errmess );
    reg_del_key( hk, keypath );
    Sleep( 250 );
    if( reg_key_exists( hk, keypath )) {
      if( errmess ) {
        add_to_mess( &mess, L"%ls", errmess );
        FREE0( errmess );
      }
      ok_reg = 0;
    }

    // remove launcher start menu entry
    FREE0( errmess );
    entrypath = tl_shortcut_path( L"launcher", wants_el );
    if( !delete_file( entrypath )) {
      add_to_mess( &mess, L"%ls", errmess );
      FREE0( errmess );
      // add_to_mess( &mess, L"Failure to remove start menu entry" );
      ok_menu = 0;
    } else {
      p = wcsrchr( entrypath, L'\\' );
      if( p ) {
        // and maybe submenu
        *p = L'\0'; // now entrypath truncated to submenu
        FREE0( errmess );
        if( !rmdir_ifempty( entrypath )) {
          add_to_mess( &mess, L"%ls", errmess );
          FREE0( errmess );
          // add_to_mess( &mess, L"Failure to remove %ls start menu", tl_name );
          ok_menu = 0;
        }
      FREE0( entrypath );
      }
    }

    ok = ok && ok_menu && ok_reg;

    if( ans == IDYES ) {
      add_to_mess( &mess, L"%ls uninstalled", tl_name );
      // cannot come back after remove_selfdir
      // so handle message output now
      wr_log( mess );
      if( !ok && do_feedback )
        tlwarn( NULL, L"%ls", mess );
      else if( do_feedback )
        tlinfo( NULL, L"%ls", mess );
      // Remove TL itself. This terminates the program.
      remove_selfdir( tlroot );
      return;
    } else { // uninst; faster for testing
      // if( !delete_file( lockfile )) {
      //   add_to_mess( &mess, L"Failed to delete lock file %ls\n"
      //        L"please delete this file manually", lockfile );
      //   ok = 0;
      // }
      add_to_mess( &mess, L"%ls uninstalled; its files remain", tl_name );
    } // uninst

  } else { // install, with or without feedback

    wchar_t * str;

    if( !wcscmp( action, L"admin_inst_silent" ) ||
        !wcscmp( action, L"user_inst_silent" ))
      do_feedback = 0;
    // uninstaller registry key
    if( !reg_set_stringvalue( hk, keypath, L"DisplayName", tl_name ))
      ok_reg = 0;
    if( !reg_set_stringvalue( hk, keypath, L"DisplayVersion", version ))
      ok_reg = 0;
    str = tl_wcsnconcat( 3, L"\"", this_exe, L"\" uninst_all" );
    if( !reg_set_stringvalue( hk, keypath, L"UninstallString", str ))
      ok_reg = 0;
    FREE0( str );
    // next two cosmetic; no error checking
    reg_set_stringvalue( hk, keypath, L"Publisher", L"TeX Live" );
    reg_set_stringvalue( hk, keypath, L"URLInfoAbout", TLA_URL );
    FREE0( keypath );
    if( !ok_reg )
      add_to_mess( &mess, L"Problems creating uninstaller registry key" );
    ok = ok && ok_reg;

    // launcher entry in start menu
    str = tl_shortcut_path( L"launcher", wants_el );
    // make_shortcut creates directory if necessary
    if( !make_shortcut( this_exe, NULL, str )) ok_menu = 0;
    FREE0( str );
    if( !ok_menu ) add_to_mess( &mess,
        L"Problems creating start menu shortcut" );
    else
      add_to_mess( &mess, L"%ls now available via Start Menu", tl_name );

    // user initialization for single-user installation
    if( !wants_el ) {
      wchar_t * fmess;
      first_time( &fmess );
      if( fmess && fmess[0] ) {
        // wr_log( L"%ls", fmess );
        add_to_mess( &mess,
          L"\nAdditional information may be available in %ls\n"
          L"after closing this dialog.\n", logpath );
      }
    }
    if( !do_feedback ) {
      int exitcode = 0;
      if( !ok_reg ) exitcode = 1;
      if( !ok_menu ) exitcode += 2;
      ExitProcess( exitcode );
      return;
    };
  } // install
  if( mess ) wr_log( mess );
  if( !do_feedback ) return;
  if( !ok && mess )
    tlwarn( NULL, L"%ls", mess );
  else if( mess )
    tlinfo( NULL, L"%ls", mess );
} // do_inst_uninst

void clean_perl_env() {
  int i;
  SetEnvironmentVariable( L"PATH", tlp_searchpath );
  for( i=0; i<nperl_envs; i++ )
    SetEnvironmentVariable( perl_envs[i].name, NULL );
} // clean_perl_env

void restore_perl_env() {
  int i;
  SetEnvironmentVariable( L"PATH", new_searchpath );
  for( i=0; i<nperl_envs; i++ )
    SetEnvironmentVariable( perl_envs[i].name, perl_envs[i].val );
} // restore_perl_env

// program initializations to do before parsing the ini file
void find_file_locations( wchar_t ** ini_path ) {
  // Run as launcher if kpsewhich is in the same directory as the program
  // or in its bin/win32 subdirectory. Otherwise, run as forgetter.
  // In the forgetter case, the ini file should be in the same directory.
  // This function also sets either:
  // - if launcher, the global variable configdir. Otherwise
  // - the globals tlroot and tlaunchbase i.e. basename of launcher

  wchar_t * exepath, * p, * kp = NULL, * cmd = NULL;

  // the inifile should have the name of the launcher or forgetter,
  // but with extension .ini
  // try ini file in same directory
  *ini_path = tl_wcscpy( this_exe );
  wcscpy( *ini_path + wcslen( *ini_path ) - 3, L"ini" );
  exepath = tl_wcscpy( this_exe ); // need copy; will be modified
  p = wcsrchr( exepath, L'\\' );
  if( !p ) tldie( NULL, L"Own path %ls not an absolute filename", exepath );
  *p = L'\0';
  // now exepath is directory; p+1 is basename
  if( FILE_EXISTS( *ini_path )) {
    kp = tl_wcsnconcat( 2, exepath, L"\\bin\\win32\\kpsewhich.exe" );
    if( FILE_EXISTS( kp )) { // we are launcher, in installation root
      is_forgetter = 0;
      tlroot = tl_wcscpy( exepath );
      p++; // start of basename
      tlaunchbase = tl_wcsncpy( p, wcslen( p ) - 4 );
    } else {
      is_forgetter = 1;
      configdir = tl_wcscpy( exepath );
    }
    FREE0( kp );
  } else { // launcher in bin/win32, config found by kpsewhich
    FREE0( *ini_path ); // get rid of wrong guess
    kp = tl_wcsnconcat( 2, exepath, L"\\kpsewhich.exe" );
    if( !FILE_EXISTS( kp ))
        tldie( NULL, L"The launcher should be either in the root of TeX Live "
        L"or in the bin\\win32 subdirectory" );
    p++;
    tlaunchbase = tl_wcsncpy( p, wcslen( p ) - 4 );
    tlroot = tl_wcsncpy( exepath, wcslen( exepath ) - wcslen( L"\\bin\\win32" ));
    cmd = tl_wcsnconcat( 3, L"-format=web2c ",
        tlaunchbase, L".ini" );
    *ini_path = kpse_answer( cmd );
    if( !( *ini_path ))
        tldie( NULL, L"Config file %ls not found", tlaunchbase );
    FREE0( cmd );
    FREE0( kp );
  }
  FREE0( exepath );
} // find_file_locations


void get_config( void ) {

  wchar_t * config, * ini_path;

  find_file_locations( &ini_path );
  errkeep = 0;
  config = slurp_file_to_wcs( ini_path, MAX_CONFIG );
  if( !config || !config[0] ) tldie( NULL, L"%ls", errmess );
  parse_ini( config );
  FREE0( config ); // don't save; when needed, get fresh, unmangled copy
  FREE0( ini_path );
  if( !configdir )
      tldie( NULL, L"Should not happen: no forgetter folder defined" );
  /* // ATM no non-fatal errors generated by parse_ini
  if( errmess && errmess[0] ) {
    tlwarn( NULL, L"%ls", errmess );
    FREE0( errmess );
  }
  return;
  */
} // get_config

////////////////////////////////////////////////
// the _tl functions

// terminate the launcher normally
void quit_tl( HWND hwnd ) {
  PostMessage( hwnd, WM_CLOSE, 0, 0 );
}

// TODO: replace with something prettier
void about_tl( HWND hwnd ) {
  wchar_t * cpr;

  tl_wasprintf( &cpr,
      L"TLaunch Version %ls\n\nCopyright (C) %ls, %ls\n\nWebpage %ls\n",
      TLA_VERSION, TLA_AUTHOR, TLA_CR_YEAR, TLA_URL );
  MessageBox( hwnd, cpr, tl_name, 0 );
  FREE0( cpr );
} // about_tl

// initialization: clear settings and restart
// init_tl and clear_tl are called by the launcher, not by the forgetter
// so the config directory can be removed without trickery

void init_tl( HWND hwnd ) {
  wchar_t * mess = NULL;
  wchar_t * prog;
  // wchar_t * buf;
  PROCESS_INFORMATION pi;
  STARTUPINFO si;

  forget( &mess );
  // remove forgetter directory
  rmdir_recursive( configdir );
  if( FILE_EXISTS( configdir ))
    add_to_mess( &mess, L"Failed to remove %ls with forgetter", configdir );
  add_to_mess( &mess, L"If necessary, restart manually." );
  display_string( hwnd, mess, L"Re-initialize" );
  FREE0( mess );

  prog = get_own_path( );
  ZeroMemory( &pi, sizeof( pi ));
  ZeroMemory( &si, sizeof( si ));
  si.cb = sizeof(STARTUPINFO);
  if( !CreateProcess( prog, NULL, NULL, NULL, FALSE, 0, NULL, NULL, &si, &pi ))
      tlwarn( NULL, L"Failure to restart; do this manually" );
  SendMessage( hwnd, WM_CLOSE, 0, 0 );
  // PostQuitMessage( 0 );
} // init_tl

// undo configuration
void clear_tl( HWND hwnd ) {
  wchar_t * mess = NULL;

  forget( &mess );
  // remove forgetter directory
  rmdir_recursive( configdir );
  if( FILE_EXISTS( configdir ))
    add_to_mess( &mess, L"Failed to remove %ls with forgetter", configdir );
  add_to_mess( &mess, L"Done with cleanup." );
  display_string( hwnd, mess, L"Forget" );
  FREE0( mess );
  SendMessage( hwnd, WM_CLOSE, 0, 0 );
  // PostQuitMessage( 0 );
} // clear_tl

void uninst_tl( HWND hwnd ) {
  // keep the actual TL installation
  do_inst_uninst( L"uninst", 1 ); // does its own reporting
  ExitProcess( 0 );
} // uninst_tl

void uninst_all_tl( HWND hwnd ) {
  do_inst_uninst( L"uninst_all", 1 ); // does its own reporting
  ExitProcess( 0 );
} // uninst_all_tl

// ed_sel_tl defined in tla_editors

/////////////////////////////////////////

BOOL run_control( HWND hwnd, UINT_PTR id ) {
  Control * ctrl = NULL;
  int i;
  for( i=0; i<ncontrols; i++ ) {
    if( controls[i] && ( controls[i]->id == id )) {
      ctrl = controls[i];
      break;
    }
  }
  if( !ctrl ) return FALSE; // i.e. message not handled
  if( ctrl->enabled ) {
    switch( ctrl->type ) {
      case L'S': { // script
        HWND hsplash;
        wchar_t * outp;
        Script * prs = ctrl->script;
        // existence of command and splashtext members is guaranteed
        hsplash = splash( hwnd, prs->splashtext );
        // outp = capture_output( prs->command, TRUE );
        outp = capture_tmp( prs->command, TRUE );
        SendMessage( hsplash, WM_CLOSE, 0, 0 );
        if( !outp )
          tlwarn( hwnd, L"No output from %ls\n", prs->command );
         else {
           display_string( hwnd, outp, prs->command ); // modal dialog
          FREE0( outp );
        }
        return TRUE;
      }
      case L'D': { // 'document'
        int l;
        BOOL have_com;
        have_com = init_com();
        l = (int)ShellExecute( NULL, NULL, (wchar_t *)(ctrl->action),
            NULL, NULL, SW_SHOWNORMAL ); // (int) cast as per MS docu
        if( l <= 32 )
          tlwarn( hwnd, L"Failure to open %ls, error %d",
              (wchar_t *)(ctrl->action), l );
        // A previous implementation closed the launcher
        // at the start of a GUI program.
        // else
        //   PostMessage( hwnd, WM_CLOSE, 0, 0 );

        break;
        // TODO: test against specific error codes
        if( have_com ) uninit_com();
      }
      case L'C': { // arbitrary command
        STARTUPINFO si;
        PROCESS_INFORMATION pi;
        wchar_t * cdir;
        ZeroMemory( &si, sizeof(si) );
        si.cb = sizeof(si);
        si.lpTitle = ctrl->display;
        ZeroMemory( &pi, sizeof(pi) );
        cdir = get_shell_folder( FOLDERID_Documents ); // NULL if failure
        if( !CreateProcess( NULL, ctrl->action, NULL, NULL, TRUE, 0,
            NULL, cdir, &si, &pi ))
          tlwarn( hwnd, L"Failure to start %ls", ctrl->action );
        FREE0( cdir );
        // else
        //   PostMessage( hwnd, WM_CLOSE, 0, 0 );
        break;
      }
      case L'E':
      case L'P': {
        STARTUPINFO si;
        PROCESS_INFORMATION pi;
        wchar_t * cdir;
        ZeroMemory( &si, sizeof(si) );
        si.cb = sizeof(si);
        si.lpTitle = ctrl->display;
        ZeroMemory( &pi, sizeof(pi) );
        cdir = get_shell_folder( FOLDERID_Documents ); // NULL if failure
        if( !CreateProcess( NULL, ctrl->pgid->command, NULL, NULL, TRUE, 0,
            NULL, cdir, &si, &pi ))
          tlwarn( hwnd, L"Failure to start %ls", ctrl->pgid->command );
        FREE0( cdir );
        // else
        //   PostMessage( hwnd, WM_CLOSE, 0, 0 );
        break;
      }
      case L'F': {
        if( ctrl->fun ) {
          ( ctrl->fun )( hwnd );
          break;
        } else {
          return FALSE;
        }
      }
      default: tldie( hwnd, L"Internal error" );
    } // switch
  } // ctrl->enabled
  return TRUE; // message handled
} // run_control

void set_icon( Control * ctrl ) {
  wchar_t * iconfile = NULL, * iconspec = NULL, * cmd = NULL;
  wchar_t * p = NULL;
  UINT iconindex = 0;

  if( !ctrl->hw ) tldie(
      NULL, L"set_icon invoked for a control without a window" );

  if( ctrl->hicon ) {
    SendMessage( ctrl->hw, BM_SETIMAGE,
      ( WPARAM )IMAGE_ICON, ( LPARAM )NULL );
    if( !ctrl->icon_from_resource) DestroyIcon( ctrl->hicon );
  }
  ctrl->hicon = NULL; // icon handle

  // pick suitable member from Control struct
  if( ctrl->type == L'D' ) {
    p = wcsrchr(( wchar_t * )ctrl->action, L'.' );
    if( p ) {
      get_assoc_data_ext( p, NULL, &cmd, &iconspec, NULL );
      if( iconspec ) FREE0( cmd );
    }
  } else if( ctrl->type == L'P' || ctrl->type == L'E' ) {
    iconspec = ctrl->pgid->iconspec;
  } else if( ctrl->type == L'S' ) {
      if( ctrl->script->command ) {
        cmd = ctrl->script->command;
      }
  } else if( ctrl->type == L'C' ) {
    cmd = ctrl->action;
  }

  if( iconspec ) {
    parse_iconspec( iconspec, &iconfile, &iconindex );
    if( iconfile ) {
      ctrl->hicon = ExtractIcon( hInst, iconfile, iconindex );
      if( ctrl->hicon ) ctrl->icon_from_resource = FALSE;
      FREE0( iconfile );
    }
    // ctrl->type D: iconspec was allocated by get_assoc_data
    // and should be released when done
    if( ctrl->type == L'D' ) FREE0( iconspec );
  } else if( cmd ) {
    iconfile = prog_from_cmd( cmd );
    if( iconfile ) {
      ctrl->hicon = ExtractIcon( hInst, iconfile, iconindex );
      if( ctrl->hicon ) ctrl->icon_from_resource = FALSE;
      FREE0( iconfile );
      // ctrl->type D: iconfile was allocated by get_assoc_data
      if( ctrl->type == L'D' ) FREE0( cmd );
    }
  }

  if( ctrl->type == L'F' && ctrl->fun == ed_sel_tl ) {
    ctrl->hicon = LoadIcon( hInst, MAKEINTRESOURCE( IDI_SELECT ));
    ctrl->icon_from_resource = TRUE;
  } else if( !ctrl->hicon ) {
    ctrl->hicon = LoadIcon( hInst, MAKEINTRESOURCE( IDI_DEFLT ));
    ctrl->icon_from_resource = TRUE;
  }
  if( ctrl->hicon ) {
      SendMessage( ctrl->hw, BM_SETIMAGE, ( WPARAM )IMAGE_ICON,
        ( LPARAM )ctrl->hicon );
  } // otherwise leave button face blank
  // note: no error if ctrl->icon remains NULL
} //set_icon

void resize_toplevel( HWND hwnd, int w, int h, UINT flags ) {
  RECT r_cl, r_wn;
    GetClientRect( hwnd, &r_cl );
    GetWindowRect( hwnd, &r_wn );
    // adjust size for difference  between window size and client size.
    // top left client coordinate is always 0,0
    SetWindowPos(
        hwnd, HWND_TOP, 0, 0,
        w + r_wn.right - r_wn.left - r_cl.right,
        h + r_wn.bottom - r_wn.top - r_cl.bottom,
        flags );
} // resize_toplevel

// Build the interface; only applies to a regular run
void build_gui( HWND hwnd ) {
  int i, j, w, h, mw;
  Control * ctrl;
  int mwidth; // menu width
  HMENU hMenu, hSubMenu;
  HWND hlabel;
  // global variable UINT_PTR id_next_control
  // is set to ID_FIRST_CONTROL in wWinMain.
  // tlauncher-specific structs:
  SubMenu * smenu;

  SendMessage( hwnd, WM_SETTEXT, 0, (LPARAM)tl_name );

  // start of resource numbering for menu items and push buttons
  id_next_control = ID_FIRST_CONTROL;
  // same for other child controls
  id_next_child = ID_FIRST_CHILD;

  // run through first all menu- and submenu items,
  // then through all button controls.
  // the ini parser has already checked whether
  // a control references an existing item.

  hMenu = CreateMenu();
  mwidth = 0; // keep track of menu width
  for( i=0; i<nmenus; i++ ) {
    smenu = menus[i];
    BOOL smenu_enabled = FALSE; // until an enabled item is encountered
    if( !smenu ) continue; // should not happen
    if( smenu->nitems<1 ) continue;
    hSubMenu = CreatePopupMenu();
    for( j=0; j<smenu->nitems; j++ ) {
      ctrl = controls[smenu->first_item + j];
      if( ctrl->type == L'N' ) { // separator
        AppendMenu( hSubMenu, MF_SEPARATOR, 0, NULL );
      } else {
        ctrl->id = id_next_control++;
        if( ctrl->enabled || ctrl->type == L'E' ) {
          AppendMenu( hSubMenu, MF_STRING | MF_POPUP |
              ( ctrl->enabled ? MF_ENABLED : ( MF_DISABLED | MF_GRAYED )),
              ctrl->id, ctrl->display );
          smenu_enabled = TRUE;
        }
      }
    }
    if( smenu_enabled ) {
      AppendMenu( hMenu, MF_STRING | MF_POPUP |
          ( smenu_enabled ? MF_ENABLED : ( MF_DISABLED | MF_GRAYED )),
          (UINT_PTR)hSubMenu, smenu->name );
      mwidth = mwidth + cx * ( wcslen( smenu->name ) + 4 );
    }
  }
  SetMenu( hwnd, hMenu );

  // window width needed for menu, incl margins
  w = mwidth + 4 * cx;
  // window height if no buttons or announcement
  h = cy;

  // button measurements
  // each button control has a label underneath of width 20 * cx and height 2 * cy,
  // separated by cy / 2

  if( nbuttons > 0 ) {
    int iconside, blwidth, hoffset, voffset, wb;
    // Control * gctrl; // for grouping rectangle
    // BOOL ingroup;
    // int gleft;

    iconside = iconsize + 6; // icon button: add room for border
    blwidth = 20 * cx; // width of text label
    if( blwidth < iconside ) blwidth = iconside;
    hoffset = ( blwidth - iconside ) / 2; // for button wrt text label
    voffset = iconside + cy / 2; // for text label wrt button
    // now correct window size
    wb =  nbuttons * ( blwidth + 2 * cx ) + 2 * cx;
    if( w < wb ) w = wb;
    h += h + voffset + 3 * cy; // includes new bottom margin
    for ( i=0; i<nbuttons; i++ ) {
      HWND hlabel;
      ctrl = controls[first_button + i];
      ctrl->id = id_next_control++;
      ctrl->hw = CreateWindow(
          L"BUTTON",
          NULL,
          WS_CHILD | WS_VISIBLE | WS_TABSTOP | BS_PUSHBUTTON | BS_ICON,
          hoffset + 2 * cx + i * ( blwidth + 2 * cx ),
          cy,
          iconside,
          iconside,
          hwnd,
          ( HMENU )( ctrl->id ),
          hInst,
          NULL
      );
      if( !( ctrl->hw )) tldie( NULL, L"Failure to create icon button" );
      set_icon( ctrl ); // sets ctrl->hicon, which may remain NULL

      if( !ctrl->enabled ) EnableWindow( ctrl->hw, FALSE );
      hlabel = CreateWindow(
          L"STATIC",
          NULL,
          WS_CHILD | WS_VISIBLE | SS_CENTER,
          2 * cx + i * ( blwidth + 2 * cx ),
          cy + voffset,
          blwidth,
          2 * cy,
          hwnd,
          ( HMENU )( IDC_STATIC ),
          hInst,
          NULL
      );
      if( !hlabel ) tldie( NULL, L"Failure to create text label" );
      SendMessage( hlabel, WM_SETFONT, ( WPARAM )hfnt, TRUE );
      SendMessage( hlabel, WM_SETTEXT, 0, (LPARAM)( ctrl->display ));
    } // for
  } // nbuttons > 0

  // sanity check wrt. width
  mw = GetSystemMetrics( SM_CXFULLSCREEN );
  if( mw && w > mw ) w = mw;

  // announcement text, if any
  if( tl_announce ) {
    h += 2 * cy; // includes new bottom margin
    hlabel = CreateWindow(
        L"STATIC",
        NULL,
        WS_CHILD | WS_VISIBLE | SS_CENTER,
        cx,
        h - 2 * cy,
        w - 2 * cx,
        cy,
        hwnd,
        ( HMENU )IDC_STATIC,
        hInst,
        NULL
    );
    if( hlabel ) {
      SendMessage( hlabel, WM_SETFONT, ( WPARAM )hfnt, TRUE );
      SendMessage( hlabel, WM_SETTEXT, 0, (LPARAM)tl_announce );
    }
    // separator between announcement and buttons
    if( nbuttons > 0 ) {
      CreateWindow(
        L"STATIC",
        NULL,
        WS_CHILD | WS_VISIBLE | SS_BLACKRECT,
        1,
        h - 11 * cy / 4,
        w - 2,
        1,
        hwnd,
        ( HMENU )IDC_STATIC,
        hInst,
        NULL
      );
    }
  }
  // now resize window
  resize_toplevel( hwnd, w, h,
      SWP_NOACTIVATE | SWP_NOMOVE | SWP_NOZORDER | SWP_NOREDRAW );

  // label and icon for default editor
  update_mainwin_ed_control( );

} // build_gui

LRESULT CALLBACK WndProc(HWND hwnd, UINT Mess, WPARAM wParam, LPARAM lParam) {
  switch (Mess) {
    case WM_CREATE: {
      HDC hdc = GetDC( hwnd );
      // select stock font
      SelectObject( hdc, hfnt );
      ReleaseDC( NULL, hdc );

      build_gui( hwnd );
      return 0;
    }
    case WM_CTLCOLORSTATIC: {
      HDC hdcStatic = ( HDC )wParam;
      // it makes no sense to use both an RGB value
      // and a stock brush, but that is what everybody tells me to do.
      // at least, it appears to work (knock on wood)
      SetBkColor(hdcStatic, RGB(255,255,255));
      return (INT_PTR) GetStockObject( WHITE_BRUSH );
      // Stock objects do not need to be deleted.
    }
    // no entries any more for a forget run:
    // a forget run now just uses a message box
    // and does not use WndProc any more
    case WM_COMMAND: {
      // launcher buttons
      if( LOWORD(wParam) >= ID_FIRST_CONTROL &&
          LOWORD(wParam) < id_next_control ) {
        // menu item or button
        if( run_control( hwnd, LOWORD(wParam) )) return 0;
        else return DefWindowProc(hwnd, Mess, wParam, lParam);
      }
      break;
    }
    case WM_CLOSE:
      DestroyWindow(hwnd);
      return 0;
    case WM_DESTROY:
      PostQuitMessage(0);
      return 0;
  }
  return DefWindowProc(hwnd, Mess, wParam, lParam);
} // WndProc

// wWinMain: initialization, special runs, boilerplate

int WINAPI wWinMain (HINSTANCE hInstance, HINSTANCE hPrevInstance,
    PWSTR cmdline, int iCmdShow) {

  WNDCLASSEX wclass;
  wchar_t * mainwin_class;
  MSG Msg;
  int do_feedback = 1;
  wchar_t * action = NULL;
  // initialize random generator
  time_t sec;
  srand( time( &sec ));

  set_simple_globals(); // sets this_exe, among others

  // get_config: configuration, both from ini and otherwise
  // also sets is_forgetter
  get_config( );

  // The MSDN documentation explicitly states in the
  // "WinMain entry point" page:
  // "ANSI GUI applications can use the lpCmdLine parameter of the WinMain
  // function to access the command-line string, excluding the program name."
  // I found no such explicit statement about wWinMain.

  // Parse cmdline, which should consist of a number of unquoted,
  // space-separated keywords. Allowed keywords:
  // silent, [user|admin]_inst[_silent], uninst_all, uninst, remember.
  // Note. Tab characters are treated as word characters.
  if( cmdline && cmdline[0] ) { // otherwise retain initialized values
    wchar_t * p, * p_next, * c;
    c = tl_wcscpy( cmdline );
    p = c;
    while( p && *p ) {
      // next non-space character
      while( *p && *p == L' ' ) p++;
      if( !*p ) break;
      p_next = wcschr( p, L' ' );
      if( p_next ) *p_next = L'\0';
      if( !wcscmp( p, L"silent" )) {
        do_feedback = 0;
      } else if( !wcscmp( p, L"remember" )) {
        FREE0( action );
        action = tl_wcscpy( L"remember" );
      } else if( !wcscmp( p, L"uninst_all" )) {
        FREE0( action );
        action = tl_wcscpy( L"uninst_all" );
      } else if( !wcscmp( p, L"uninst" )) {
        FREE0( action );
        action = tl_wcscpy( L"uninst" );
      } else if( !wcscmp( p, L"user_inst" )) {
        FREE0( action );
        action = tl_wcscpy( L"user_inst" );
      } else if( !wcscmp( p, L"user_inst_silent" )) {
        FREE0( action );
        action = tl_wcscpy( L"user_inst" );
        do_feedback = 0;
      } else if( !wcscmp( p, L"admin_inst" )) {
        FREE0( action );
        action = tl_wcscpy( L"admin_inst" );
      } else if( !wcscmp( p, L"admin_inst_silent" )) {
        FREE0( action );
        action = tl_wcscpy( L"admin_inst" );
        do_feedback = 0;
      } else if( !wcscmp( p, L"--version" ) || !wcscmp( p, L"-v" )) {
        tlinfo( NULL, L"tlaunch (TeX Live Launcher) %ls.\n\n"
          L"See %ls", TLA_VERSION, TLA_URL );
        ExitProcess( 0 );
      } else {
        tlwarn( NULL, L"tlaunch is normally started without a parameter.\n"
          L"See the tlaunch manual for full documentation." );
        ExitProcess( 1 );
      }
      // start looking for next word
      if( p_next ) p_next++;
      p = p_next;
    }
    FREE0( c );
  }

  if( !is_forgetter && action &&
      ( !wcscmp( action, L"uninst" ) || !wcscmp( action, L"uninst_all" ) ||
      !wcscmp( action, L"user_inst" ) || !wcscmp( action, L"admin_inst" ))) {
    do_inst_uninst( action, do_feedback );
    ExitProcess( 0 ); // dont trust windows to really quit right now:
    return 0;
  } else if( action && !wcscmp( action, L"remember" )) {
    remember( ); // should work with either tlauch or forgetter
    ExitProcess( 0 );
    return 0;
  } else if( action && is_forgetter ) {
    tldie( NULL, L"Illegal action %ls for forgetter", action );
    return 0;
  } else if( action && !is_forgetter ) {
    tldie( NULL, L"Illegal action %ls for %ls", action, tlaunchbase );
    return 0;
  } else if( is_forgetter ) {
    forget_self( do_feedback );
    ExitProcess( 0 );
    return 0;
  }

  // remaining case: normal run
  { // avoid running multiple copies per user session:
    // compose name of main window class from exe basename, username and
    // session id, then check for occurrences of this window class.
    // Since this may be a restart, wait a moment for any parent to shut down.
    DWORD pid, sid;
    wchar_t * username, * bas;
    HWND hDup; // hMain is global

    bas = wcsrchr( this_exe, L'\\' );
    bas++;
    // bas is basename; for simplicity, we just keep the trailing '.exe'
    username = get_env( L"USERNAME" );
    pid = GetCurrentProcessId();
    if( username && ProcessIdToSessionId( pid, &sid )) { // otherwise dont check
      tl_wasprintf( &mainwin_class, L"%ls-%ls%010lu", bas, username, sid );
      Sleep( 1000 );
      hDup = FindWindow( mainwin_class, NULL );
      if( hDup ) {
        ShowWindow( hDup, SW_RESTORE );
        if( !SetForegroundWindow( hDup )) tlinfo( NULL, L"Already running" );
        return 0 ;
      }
    }
    FREE0( username );
  } // avoid multiple

  hInst = hInstance; // copy to global variable

  // GUI preliminaries
  { // fonts and font metrics
    HDC hdc;
    TEXTMETRIC tm;
    LOGFONT lfnt;
    // icons
    iconsize = GetSystemMetrics( SM_CXICON );

    hdc = GetDC( NULL ); // NULL: device context for entire screen
    // default and big font
    hfnt = GetStockObject( DEFAULT_GUI_FONT );
    SelectObject( hdc, hfnt );
    GetTextMetrics( hdc, &tm );
    // cx, cy, cxb and cyb are declared in tlaunch.h
    cx = tm.tmAveCharWidth;
    cy = tm.tmHeight + tm.tmExternalLeading;
    // big font, for splash screen
    hfntbig = NULL; //should already be true
    ZeroMemory( &lfnt, sizeof( lfnt ));
    if( GetObject( hfnt, sizeof( lfnt ), &lfnt )) {
      lfnt.lfHeight = 1.5 * lfnt.lfHeight;
      lfnt.lfWidth = 0;
      lfnt.lfPitchAndFamily = VARIABLE_PITCH | FF_SWISS;
      hfntbig = CreateFontIndirect( &lfnt );
    }
    if( hfntbig ) {
      SelectObject( hdc, hfntbig );
      GetTextMetrics( hdc, &tm );
      cxb = tm.tmAveCharWidth;
      cyb = tm.tmHeight + tm.tmExternalLeading;
    } else {
      hfntbig = hfnt;
      cxb = cx;
      cyb = cy;
    }
    ReleaseDC( NULL, NULL );
  } // fonts and font metrics

  // window classes
  wclass.cbSize     = sizeof(WNDCLASSEX);
  wclass.style     = CS_HREDRAW | CS_VREDRAW;
  wclass.lpfnWndProc   = WndProc;
  wclass.cbClsExtra   = 0;
  wclass.cbWndExtra   = 0;
  wclass.hInstance   = hInstance;
  wclass.hIcon       = LoadIcon( GetModuleHandle( NULL ),
      MAKEINTRESOURCE( IDI_TL ));
  wclass.hCursor     = LoadCursor(NULL, IDC_ARROW);
  wclass.hbrBackground = (HBRUSH)GetStockObject (WHITE_BRUSH);
  wclass.lpszMenuName  = NULL;
  wclass.lpszClassName = mainwin_class;
  wclass.hIconSm     = NULL;

  if(!RegisterClassEx(&wclass)) {
    tldie( NULL, L"Window Registration Failed!" );
    return 0;
  }

  // window for editors dialog
  register_eds_wnd_class( );

  // splash screen for long-running scripts
  register_splash( );

  if( !user_config_exists( )) {
    HWND hsplash;
    wchar_t * mess = NULL;
    hsplash = splash( NULL, L"Initializing..." );
    first_time( &mess ); // path, file associations, forgetter
    SendMessage( hsplash, WM_CLOSE, 0, 0 );
    add_to_mess( &mess, L"Done initializing" );
    display_string( NULL, mess, L"Initializations" );
    // wait a while to allow build_gui to see the changed registry settings
    // Sleep( 100 );
    FREE0( mess );
  }

  hMain = CreateWindow( mainwin_class, L"Launcher",
      WS_OVERLAPPEDWINDOW,
      CW_USEDEFAULT, CW_USEDEFAULT,
      640, 480, // width and height to be reset during WM_CREATE callback
      NULL, NULL, hInst, NULL );
  // The menu will be hooked up to the main window in build_gui;
  // for now it is still NULL
  if( !hMain ) {
    tldie( NULL, L"Failure to create main window" );
  }
  ShowWindow( hMain, iCmdShow );
  UpdateWindow( hMain );

  while(GetMessage(&Msg, NULL, 0, 0) > 0) {
    TranslateMessage(&Msg);
    DispatchMessage(&Msg);
  }
  return Msg.wParam;
  // end regular run
} // wWinMain
