from __future__ import division

#############################################################################		
# --------------------------------------------------------------------------
# BMAE v1.2
# --------------------------------------------------------------------------
# ***** BEGIN GPL LICENSE BLOCK *****
#
# Copyright (C) 2006 - 2007: Ron Walker
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software Foundation
# Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
#
# ***** END GPL LICENCE BLOCK *****
# --------------------------------------------------------------------------
############################################################################

import Blender
from Blender import Mesh, Object, Window

import math
from math import pi,sin,cos



#############################################################################################################
#Base Section                                                                                               #
#############################################################################################################

#==============
#Measure class
#==============
class Measure(object):
	"""Determine the measurement to use and use it."""
	global mos_l, mis_l, ml_l
	
	def __init__(self,unit,sn, o_s_b,l_b=[[0,0,0.0],[0,0,0.0],[0,0,0.0]], i_s_b=[[0,0,0.0],[0,0,0.0],[0,0,0.0]], floor=0,core=0):
		global mos_l, mis_l, ml_l, f_z
		self.unit = unit
		self.sn = sn
		self.o_s_b = o_s_b
		self.l_b = l_b
		self.i_s_b = i_s_b
		self.floor = floor
		self.core = core
		mos_l = []
		mis_l = []
		ml_l = []
		
		if self.unit == 2 or self.unit == 3:
			"""Set the normal measument types."""
			
			#Select
			if self.unit == 2:
				m_k = 5280
				i_c = 12
			elif self.unit == 3:
				m_k = 1000
				i_c = 100

			#Size - Outer		
			xos = (((self.o_s_b[0][0]*m_k) + (self.o_s_b[0][1]) + (self.o_s_b[0][2]/i_c)) / 2) * self.sn
			yos = (((self.o_s_b[1][0]*m_k) + (self.o_s_b[1][1]) + (self.o_s_b[1][2]/i_c)) / 2) * self.sn
			zos = (((self.o_s_b[2][0]*m_k) + (self.o_s_b[2][1]) + (self.o_s_b[2][2]/i_c)) / 2) * self.sn
			#Size - Inner		
			xis = (((self.i_s_b[0][0]*m_k) + (self.i_s_b[0][1]) + (self.i_s_b[0][2]/i_c)) / 2) * self.sn
			yis = (((self.i_s_b[1][0]*m_k) + (self.i_s_b[1][1]) + (self.i_s_b[1][2]/i_c)) / 2) * self.sn
			zis = (((self.i_s_b[2][0]*m_k) + (self.i_s_b[2][1]) + (self.i_s_b[2][2]/i_c)) / 2) * self.sn
			#Location
			xl = ((self.l_b[0][0]*m_k) + (self.l_b[0][1]) + (self.l_b[0][2]/i_c)) * self.sn
			yl = ((self.l_b[1][0]*m_k) + (self.l_b[1][1]) + (self.l_b[1][2]/i_c)) * self.sn
			if self.floor == 1:
				if zis > zos and self.core == 1:
					zl = zis * self.sn
				else:
					zl = zos * self.sn
			elif self.floor == 2:
				if zis > zos and self.core == 1:
					zl = -zis * self.sn
				else:
					zl = -zos * self.sn
			else:
				zl= ((self.l_b[2][0]*m_k) + (self.l_b[2][1]) + (self.l_b[2][2]/i_c)) * self.sn
			#Values
			mos_l = [xos,yos,zos]
			mis_l = [xis,yis,zis]
			ml_l = [xl,yl,zl]
			
		elif self.unit == 4 or self.unit == 5:
			"""Set the conversion measurement types."""
	
			#Select
			if self.unit == 4:
				m_k   = 5280
				m_k_c = 0.621371192
				f_m   =	3.2808399
				i_c   = 12
				i_c_c = 0.393700787
			elif self.unit == 5:
				m_k   = 1000
				m_k_c = 1.609344
				f_m   =	0.3048
				i_c   = 100
				i_c_c = 2.54
			
			#Size - Outer
			xos = ((((self.o_s_b[0][0]*m_k)*m_k_c) + (self.o_s_b[0][1]*f_m) + ((self.o_s_b[0][2]/i_c)*i_c_c)) / 2) * self.sn
			yos = ((((self.o_s_b[1][0]*m_k)*m_k_c) + (self.o_s_b[1][1]*f_m) + ((self.o_s_b[1][2]/i_c)*i_c_c)) / 2) * self.sn 
			zos = ((((self.o_s_b[2][0]*m_k)*m_k_c) + (self.o_s_b[2][1]*f_m) + ((self.o_s_b[2][2]/i_c)*i_c_c)) / 2) * self.sn 
			#Size - Inner
			xis = ((((self.i_s_b[0][0]*m_k)*m_k_c) + (self.i_s_b[0][1]*f_m) + ((self.i_s_b[0][2]/i_c)*i_c_c)) / 2) * self.sn
			yis = ((((self.i_s_b[1][0]*m_k)*m_k_c) + (self.i_s_b[1][1]*f_m) + ((self.i_s_b[1][2]/i_c)*i_c_c)) / 2) * self.sn 
			zis = ((((self.i_s_b[2][0]*m_k)*m_k_c) + (self.i_s_b[2][1]*f_m) + ((self.i_s_b[2][2]/i_c)*i_c_c)) / 2) * self.sn 
			#Location
			xl = (((self.l_b[0][0]*m_k)*m_k_c) + (self.l_b[0][1]*f_m) + ((self.l_b[0][2]/i_c)*i_c_c)) * self.sn
			yl = (((self.l_b[1][0]*m_k)*m_k_c) + (self.l_b[1][1]*f_m) + ((self.l_b[1][2]/i_c)*i_c_c)) * self.sn
			if self.floor == 1:
				if zis > zos and self.core == 1:
					zl = zis * self.sn
				else:
					zl = zos * self.sn
			elif self.floor == 2:
				if zis > zos and self.core == 1:
					zl = -zis * self.sn
				else:
					zl = -zos * self.sn
			else:
				zl = (((self.l_b[2][0]*m_k)*m_k_c) + (self.l_b[2][1]*f_m) + ((self.l_b[2][2]/i_c)*i_c_c)) * self.sn
			#Values
			mos_l = [xos,yos,zos]
			mis_l = [xis,yis,zis]
			ml_l = [xl,yl,zl]		

				
#======================
#Mesh creation classes
#======================
class CreateMesh(object):
	"""Create a mesh shape."""
	
	def __init__(self,type,ax,e_set,name,flip_y=0,rt_c=0):
		self.o_radius = mos_l
		self.i_radius = mis_l
		self.loc = ml_l
		self.type = type
		self.ax = ax
		self.e_set = e_set
		self.flip_y = flip_y
		self.rt_c = rt_c
		self.name = name
		ob = []
		
	def __str__(self):
		ns = self.name
		return ns
		
	def create(self):
		"""Create the mesh."""
		global me, ob, x_,y_,z_
				
		Window.EditMode(0)
		
		me = Mesh.New(self.name)	
		
		#Axis change
		if self.ax == 0:#x axis
			x_=2; y_=1; z_=0
		elif self.ax == 1:#y axis
			x_=0; y_=2; z_=1
		else:#z axis
			x_=0; y_=1; z_=2
		
		scn = Blender.Scene.GetCurrent()              
		ob = scn.objects.new(me, self.name)
			
		ob.setLocation(self.loc)		
	
		if self.ax == 0:#x
			if self.rt_c:
				ob.setEuler((pi/2,0.0,pi/2))
			else:
				ob.setEuler((0.0,pi/2,0.0))
		elif self.ax == 1:#y
			if self.flip_y == 1:
				ob.setEuler((pi/2,0.0,0.0))
			else:
				ob.setEuler((-pi/2,pi,0.0))		
		else:
			pass		
		
		ob.select(1)
		

#############################################################################################################
#2D Meshes Section                                                                                          #
#############################################################################################################

			
#============
#Plane class
#============
class PlaneMesh(CreateMesh):
	"""Create a plane - inherited from CreateMesh"""
	def __init__(self,type,ax,e_set,core,name,flip_y):
		super(PlaneMesh, self).__init__(type,ax,e_set,name,flip_y)
		self.core = core

	def create(self):
		"""Extend the verts and faces"""
		super(PlaneMesh, self).create()

		#Vertices
		#=========
		plv = []
		for x in xrange(-1,2,2):
			x = x * self.o_radius[x_]
			for y in xrange(-1,2,2):
				y = y * self.o_radius[y_]
				plv.append([x,y,0.0])
		me.verts.extend(plv)  
 
		if self.core:
			plv = []
			for x in xrange(-1,2,2):
				x = x * self.i_radius[x_]
				for y in xrange(-1,2,2):
					y = y * self.i_radius[y_]
					plv.append([x,y,0.0])
			me.verts.extend(plv)
		
		#Faces
		#======			
		if self.core:
			PLF = ([0,2,6,4],[2,3,7,6],[3,1,5,7],[1,0,4,5])
			me.faces.extend(PLF)
		else:
			me.faces.extend(0,2,3,1)
					
		Window.EditMode(self.e_set)

#=============
#Circle class
#=============
class CircleMesh(CreateMesh):
	"""Create a circle - inherited from CreateMesh"""
	
	def __init__(self,segments,type,ax,e_set,name):
		super(CircleMesh, self).__init__(type,ax,e_set,name)
		self.segments = segments
			
	def create(self):
		"""Extend the verts and faces"""
		super(CircleMesh, self).create()
	
		# Vertices
		#==========
		civ = []
		for vN in xrange(self.segments):
			u = (vN * 1.0/self.segments)#change
			x = self.o_radius[x_]*cos(2*pi*u)
			y = self.o_radius[y_]*sin(2*pi*u)
			z = self.o_radius[z_]*0.0
			civ.append([x,y,z])
		me.verts.extend(civ)
	
		#Edges
		#======  
		cif = []
		for ce in xrange(0,self.segments):
			ce0 = ce
			ce1 = (ce+1) %self.segments
			cif.append([ce0,ce1])
		me.edges.extend(cif)
		
		Window.EditMode(self.e_set)
		
		civ=[]
		cif=[]
		
#===========
#Disc class
#===========
class DiscMesh(CreateMesh):
	"""Create a disc - inherited from CreateMesh"""
	
	def __init__(self,segments,type,ax,e_set,core,name,flip_y):
		super(DiscMesh, self).__init__(type,ax,e_set,name,flip_y)
		self.segments = segments
		self.core = core
			
	def create(self):
		"""Extend the verts and faces"""
		super(DiscMesh, self).create()
		
		#Vertices
		#==========
		#Outer Ring
		def outer_ring():
			div = []
			for rs in xrange(self.segments):
				v = (rs/self.segments)
				x = self.o_radius[x_]*cos(2*pi*v) 
				y = self.o_radius[y_]*sin(2*pi*v) 
				z = self.o_radius[z_]
				div.append([x,y,z])
			me.verts.extend(div)
				
		outer_ring()
		
		#Center	
		if self.core != 1:
			me.verts.extend(0.0,0.0,0.0)
	
		#inner Ring 
		def inner_ring():
			div = []
			for rs in xrange(self.segments):
				v = (rs/self.segments)
				x = self.i_radius[x_]*cos(2*pi*v) 
				y = self.i_radius[y_]*sin(2*pi*v)
				z = self.i_radius[z_] 
				div.append([x,y,z])
			me.verts.extend(div)
	
		if self.core:
			inner_ring()
	
		#Faces
		#======  
		def cap_core():
			dif = []
			ad = self.segments*2
			for df in xrange(self.segments):
				df0 = df
				df1 = (df+1) %self.segments
				df2 = df + self.segments
				df3 = (df + self.segments)+1
				if df3 >= ad:
					df3 = ad - self.segments
				dif.append([df0,df1,df3,df2])
			me.faces.extend(dif)
				
		def cap_fill():
			dif = []
			for bf in xrange(0,self.segments):
				bf0 = bf
				bf1 = (bf+1) %self.segments
				dif.append([self.segments,bf0,bf1])
			me.faces.extend(dif)
				
		if self.core:		
			cap_core()
		else:
			cap_fill()

		Window.EditMode(self.e_set)

		div=[]
		dif=[]

#===============
#Triangle class
#===============
class TriangleMesh(CreateMesh):
	"""Create a 2d triangle - inherited from CreateMesh"""
	
	def __init__(self,type,ax,e_set,core,name,flip_y):
		super(TriangleMesh, self).__init__(type,ax,e_set,name,flip_y)
		self.core = core
		
	def create(self):
		"""Extend the verts and faces"""
		super(TriangleMesh, self).create()
	
		#Vertices
		#=========
		tr2v = []
		for y in xrange(-1,0):
			y = y * self.o_radius[y_]
			for x in xrange(-1,2,2):
				x = x * self.o_radius[x_]
				tr2v.append([x,y,0.0])
		me.verts.extend(tr2v) 
		me.verts.extend(0.0,self.o_radius[y_],0.0) 
 
		if self.core:
			tr2v = []
			for y in xrange(-1,0):
				y = y * self.i_radius[y_]
				for x in xrange(-1,2,2):
					x = x * self.i_radius[x_]
					tr2v.append([x,y,0.0])
			me.verts.extend(tr2v) 
			me.verts.extend(0.0,self.i_radius[y_],0.0) 
		
		#Faces
		#======			
		if self.core:
			TR2F = ([0,1,4,3],[1,2,5,4],[2,0,3,5])
			me.faces.extend(TR2F)  	         
		else: 
			me.faces.extend(0,1,2)
					
		Window.EditMode(self.e_set)
		
#=====================
#Right Triangle class
#=====================
class RightTriangleMesh(CreateMesh):
	"""Create a 2d right triangle - inherited from CreateMesh"""
	
	def __init__(self,type,ax,e_set,core,direction,name,flip_y,rt_c):
		super(RightTriangleMesh, self).__init__(type,ax,e_set,name,flip_y,rt_c)
		self.core = core
		self.direction = direction
		
	def create(self):
		"""Extend the verts and faces"""
		super(RightTriangleMesh, self).create()
		
		#Vertices
		#=========	
		#Outer 
		def r_triangle(a,b,c,d):
			rt2v = []
			for y in xrange(a,b):
				y = y * self.o_radius[y_]
				for x in xrange(-1,2,2):
					x = x * self.o_radius[x_]
					rt2v.append([x,y,0.0])
			me.verts.extend(rt2v) 
			me.verts.extend(c,d,0.0)
		#BL
		if self.direction == 0:	
			r_triangle(-1,0,-self.o_radius[x_],self.o_radius[y_])
		#BR
		elif self.direction == 1:
			r_triangle(-1,0,self.o_radius[x_],self.o_radius[y_])
		#TL
		elif self.direction == 2:
			r_triangle(1,2,-self.o_radius[x_],-self.o_radius[y_])
		#TR
		elif self.direction == 3:
			r_triangle(1,2,self.o_radius[x_],-self.o_radius[y_])

		#Inner
		def r_triangle_core(a,b,c,d):
			rt2v = []
			for y in xrange(a,b):
				y = y * self.i_radius[y_]
				for x in xrange(-1,2,2):
					x = x * self.i_radius[x_]
					rt2v.append([x,y,0.0])
			me.verts.extend(rt2v)
			me.verts.extend(c,d,0.0)
						
		if self.core:
			#BL
			if self.direction == 0:
				r_triangle_core(-1,0,-self.i_radius[x_],self.i_radius[y_])
			#BR
			elif self.direction == 1:
				r_triangle_core(-1,0,self.i_radius[x_],self.i_radius[y_])
			#TL
			elif self.direction == 2:
				r_triangle_core(1,2,-self.i_radius[x_],-self.i_radius[y_])
			#TR
			elif self.direction == 3:
				r_triangle_core(1,2,self.i_radius[x_],-self.i_radius[y_])
		
		#Faces
		#======			
		if self.core:
			#BL
			if self.direction == 0:
				RT2F0 = ([0,1,4,3],[0,3,5,2])
				me.faces.extend(RT2F0)
			#BR
			elif self.direction == 1:
				RT2F1 = ([0,1,4,3],[1,2,5,4])
				me.faces.extend(RT2F1)
			#TL
			elif self.direction == 2:
				RT2F2 = ([0,3,4,1],[2,5,3,0])
				me.faces.extend(RT2F2)
			#TR
			elif self.direction == 3:
				RT2F3 = ([0,3,4,1],[1,4,5,2])
				me.faces.extend(RT2F3)        
		else: 
			#BL & BR
			if self.direction == 0 or self.direction == 1: 
				me.faces.extend(0,1,2)
			#TL & TR
			if self.direction == 2 or self.direction == 3: 
				me.faces.extend(0,2,1)
					
		Window.EditMode(self.e_set)



#############################################################################################################
#3D Meshes Section                                                                                          #
#############################################################################################################
	
#===========
#Cube class
#===========
class CubeMesh(CreateMesh):
	"""Create a cube - inherited from CreateMesh"""
	
	def __init__(self,type,ax,e_set,core,fill,name):
		super(CubeMesh, self).__init__(type,ax,e_set,name)
		self.core = core
		self.fill = fill
		
	def create(self):
		"""Extend the verts and faces"""
		super(CubeMesh, self).create()

		#Vertices
		#=========
		#Outer - Top
		cuv = []
		for x in xrange(-1,2,2):
			x = x * self.o_radius[x_]
			for y in xrange(-1,2,2):
				y = y * self.o_radius[y_]
				cuv.append([x,y,self.o_radius[z_]])		
		me.verts.extend(cuv)  
		#Outer - Bottom
		cuv = []
		for x in xrange(-1,2,2):
			x = x * self.o_radius[x_]
			for y in xrange(-1,2,2):
				y = y * self.o_radius[y_]
				cuv.append([x,y,-self.o_radius[z_]])
		me.verts.extend(cuv) 
 
		if self.core:
			#Inner - Top
			cuv = []
			for x in xrange(-1,2,2):
				x = x * self.i_radius[x_]
				for y in xrange(-1,2,2):
					y = y * self.i_radius[y_]
					cuv.append([x,y,self.i_radius[z_]])
			me.verts.extend(cuv)  
			#Inner - Bottom
			cuv = []
			for x in xrange(-1,2,2):
				x = x * self.i_radius[x_]
				for y in xrange(-1,2,2):
					y = y * self.i_radius[y_]
					cuv.append([x,y,-self.i_radius[z_]])
			me.verts.extend(cuv) 
		
		#Faces
		#======		
		if self.core:	
			CUF0 = ([0,2,10,8],[1,0,8,9],[2,3,11,10],[3,1,9,11],      #Top
					[0,4,6,2],[2,6,7,3],[3,7,5,1],[1,5,4,0],          #Center
					[4,12,14,6],[5,13,12,4],[6,14,15,7],[7,15,13,5])  #Bottom	
			me.faces.extend(CUF0)
			if self.fill == 1:
				#Center fill
				CUF1 = ([8,10,14,12],[10,11,15,14],[11,9,13,15],[9,8,12,13])
				me.faces.extend(CUF1)	         
		else: 
			CUF2 = ([0,2,3,1],								 #Top
				   [0,4,6,2],[2,6,7,3],[3,7,5,1],[1,5,4,0],  #Center
				   [4,5,7,6])                                #Bottom
			me.faces.extend(CUF2)
				
		Window.EditMode(self.e_set)
		
#===========
#Tube class      
#===========
class TubeMesh(CreateMesh):
	"""Create a tube - inherited from CreateMesh"""
	
	def __init__(self,segments,type,ax,e_set,name):
		super(TubeMesh, self).__init__(type,ax,e_set,name)
		self.segments = segments
	
	def create(self):
		"""Extend the verts and faces"""
		super(TubeMesh, self).create()
		
		#Vertices
		#==========
		def ring(z):
			tuv = []
			for sv in xrange(self.segments):
				u = (sv/self.segments)
				x = self.o_radius[x_]*cos(2*pi*u) 
				y = self.o_radius[y_]*sin(2*pi*u)
				tuv.append([x,y,z])
			me.verts.extend(tuv)
			
				
		ring(self.o_radius[z_])
		ring(-self.o_radius[z_])
		
		#Faces
		#======    
		tuf = []      
		for i in xrange(1):
			i0 = i*self.segments
			i1 = (i+1)*self.segments 
			for j in xrange(self.segments):
				j0 = j
				j1 = (j+1) %self.segments
				tw0 = (i0 + j1)
				tw1 = (i0 + j0)
				tw2 = (i1 + j0)
				tw3 = (i1 + j1)
				tuf.append([tw0,tw1,tw2,tw3])
		me.faces.extend(tuf)
		
		Window.EditMode(self.e_set)
		
		tuv=[]
		tuf=[]
			
#===============
#Cylinder class  
#===============
class CylinderMesh(CreateMesh):
	"""Create a cylinder - inherited from CreateMesh"""
	
	def __init__(self, segments,type,ax,e_set,core,fill,name):
		super(CylinderMesh, self).__init__(type,ax,e_set,name)
		self.segments = segments
		self.core = core
		self.fill = fill
	
	def create(self):
		"""Extend the verts and faces"""
		super(CylinderMesh, self).create()

		#Vertices
		#==========
		#Outer Rings
		def outer_ring(z):
			cyv = []
			for sv in xrange(self.segments):
				u = (sv/self.segments)
				x = self.o_radius[x_] * cos(2*pi*u) 
				y = self.o_radius[y_] * sin(2*pi*u) 
				cyv.append([x,y,z])
			me.verts.extend(cyv)
		
		outer_ring(self.o_radius[z_])
		outer_ring(-self.o_radius[z_])
		
		if self.core != 1:
			me.verts.extend(0.0,0.0,self.o_radius[z_])
			me.verts.extend(0.0,0.0,-self.o_radius[z_])
		
		#inner Rings 
		def inner_ring(z):
			cyv = []
			for rs in xrange(self.segments):
				v = (rs/self.segments)
				x = self.i_radius[x_]*cos(2*pi*v) 
				y = self.i_radius[y_]*sin(2*pi*v) 
				cyv.append([x,y,z])
			me.verts.extend(cyv)
	
		if self.core:
			inner_ring(self.i_radius[z_])
			inner_ring(-self.i_radius[z_])
		
		#Faces
		#====== 
		#Outer Rings
		cyf = []          
		for i in xrange(self.segments):
			i0 = i
			i1 = (i + 1) %self.segments
			i2 = ((i + self.segments) + 1) %self.segments + self.segments
			i3 = i + self.segments	
			cyf.append([i0,i3,i2,i1])
		me.faces.extend(cyf)
		
		if self.core:
			#Top Core Cap
			cyf = []
			for i in xrange(self.segments):
				i0 = i
				i1 = (i + 1) %self.segments
				i2 = ((i + self.segments*2) + 1)
				if i2 == self.segments*3:
					i2 = self.segments*2
				i3 = i + self.segments*2
				cyf.append([i0,i1,i2,i3])
			me.faces.extend(cyf)
			#Bottom Core Cap
			cyf = []
			for i in xrange(self.segments):
				i0 = i + self.segments
				i1 = ((i + self.segments)+ 1)
				if i1 == self.segments*2:
					i1 = self.segments
				i2 = ((i + self.segments*3) + 1)
				if i2 == self.segments*4:
					i2 = self.segments*3
				i3 = i + self.segments*3
				cyf.append([i0,i3,i2,i1])
			me.faces.extend(cyf)
			#Center Core Fill
			cyf = []
			if self.fill:
				for i in xrange(self.segments):
					i0 = i + self.segments*2
					i1 = (i + self.segments*2) + 1
					if i1 == self.segments*3:
						i1 = self.segments*2
					i2 = ((i + self.segments*3) + 1)
					if i2 == self.segments*4:
						i2 = self.segments*3
					i3 = i + self.segments*3
					cyf.append([i0,i1,i2,i3])
				me.faces.extend(cyf)
		else:
			#Top Cap
			cyf = []		
			for ct in xrange(self.segments):
				ct0 = ct
				ct1 = (ct + 1) %self.segments
				s0 = (self.segments*2)
				cyf.append([s0,ct0,ct1])	
			me.faces.extend(cyf)
			#Bottom Cap	
			cyf = []		
			for cb in xrange(self.segments, self.segments*2):
				cb0 = cb
				cb1 = ((cb + 1) %self.segments) + self.segments
				s1 = (self.segments*2) + 1
				cyf.append([s1,cb1,cb0])
			me.faces.extend(cyf)

		Window.EditMode(self.e_set)	
		
		cyv=[]
		cyf=[]
	
#===============
#Triangle class
#===============
class Triangle3dMesh(CreateMesh):
	"""Create a 3d triangle - inherited from CreateMesh"""
	
	def __init__(self,type,ax,e_set,core,fill,name):
		super(Triangle3dMesh, self).__init__(type,ax,e_set,name)
		self.core = core
		self.fill = fill
		
	def create(self):
		"""Extend the verts and faces"""
		super(Triangle3dMesh, self).create()
		
		#Vertices
		#=========
		def triangle_2(z):
			tr3v = []
			for y in xrange(-1,0):
				y = y * self.o_radius[y_]
				for x in xrange(-1,2,2):
					x = x * self.o_radius[x_]
					tr3v.append([x,y,z])
			me.verts.extend(tr3v) 
			me.verts.extend(0.0,self.o_radius[y_],z)
 
		triangle_2(-self.o_radius[z_])
		triangle_2(self.o_radius[z_])

		def triangle_2_core(z):
			tr3v = []
			for y in xrange(-1,0):
				y = y * self.i_radius[y_]
				for x in xrange(-1,2,2):
					x = x * self.i_radius[x_]
					tr3v.append([x,y,z])
			me.verts.extend(tr3v)
			me.verts.extend(0.0,self.i_radius[y_],z)
				
		if self.core: 
			triangle_2_core(-self.i_radius[z_])
			triangle_2_core(self.i_radius[z_])
		
		#Faces
		#======			
		if self.core:
			TR3F0 = ([3,4,10,9],[4,5,11,10],[5,3,9,11],#top
				    [0,1,4,3],[1,2,5,4],[2,0,3,5],     #center
				    [0,6,7,1],[1,7,8,2],[2,8,6,0])     #bottom
			me.faces.extend(TR3F0)	
			if self.fill:
				TR3F1 = ([6,9,10,7],[7,10,11,8],[8,11,9,6])
				me.faces.extend(TR3F1)         
		else:	
			TR3F2 = ([3,4,5],						#Top
					 [0,1,4,3],[1,2,5,4],[2,0,3,5], #Center
					 [0,2,1])						#Bottom
			me.faces.extend(TR3F2)
					
		Window.EditMode(self.e_set)
	
#=====================
#Right Triangle class
#=====================
class RightTriangle3dMesh(CreateMesh):
	"""Create a 3d right triangle - inherited from CreateMesh"""
	
	def __init__(self,type,ax,e_set,core,fill,direction,name,flip_y,rt_c):
		super(RightTriangle3dMesh, self).__init__(type,ax,e_set,name,flip_y,rt_c)
		self.core = core
		self.fill = fill
		self.direction = direction
		
	def create(self):
		"""Extend the verts and faces"""
		super(RightTriangle3dMesh, self).create()
		
		#Vertices
		#=========	
		#Outer 
		def r_triangle_2(a,b,c,d,e,f):
			rt3v = []
			for y in xrange(a,b):
				y = y * self.o_radius[y_]
				for x in xrange(-1,2,2):
					x = x * self.o_radius[x_]
					rt3v.append([x,y,c])
			me.verts.extend(rt3v) 
			me.verts.extend(d,e,f)
		#BL
		if self.direction == 0:	
			r_triangle_2(-1,0,-self.o_radius[z_],-self.o_radius[x_],self.o_radius[y_],-self.o_radius[z_])
			r_triangle_2(-1,0,self.o_radius[z_],-self.o_radius[x_],self.o_radius[y_],self.o_radius[z_])
		#BR
		elif self.direction == 1:
			r_triangle_2(-1,0,-self.o_radius[z_],self.o_radius[x_],self.o_radius[y_],-self.o_radius[z_])
			r_triangle_2(-1,0,self.o_radius[z_],self.o_radius[x_],self.o_radius[y_],self.o_radius[z_])
		#TL
		elif self.direction == 2:
			r_triangle_2(1,2,-self.o_radius[z_],-self.o_radius[x_],-self.o_radius[y_],-self.o_radius[z_])
			r_triangle_2(1,2,self.o_radius[z_],-self.o_radius[x_],-self.o_radius[y_],self.o_radius[z_])
		#TR
		elif self.direction == 3:
			r_triangle_2(1,2,-self.o_radius[z_],self.o_radius[x_],-self.o_radius[y_],-self.o_radius[z_])
			r_triangle_2(1,2,self.o_radius[z_],self.o_radius[x_],-self.o_radius[y_],self.o_radius[z_])

		#Inner
		def r_triangle_2_core(a,b,c,d,e,f):
			rt3v=[]
			for y in xrange(a,b):
				y = y * self.i_radius[y_]
				for x in xrange(-1,2,2):
					x = x * self.i_radius[x_]
					rt3v.append([x,y,c])
			me.verts.extend(rt3v)
			me.verts.extend(d,e,f)
						
		if self.core:
			#BL
			if self.direction == 0:
				r_triangle_2_core(-1,0,-self.i_radius[z_],-self.i_radius[x_],self.i_radius[y_],-self.i_radius[z_])
				r_triangle_2_core(-1,0,self.i_radius[z_],-self.i_radius[x_],self.i_radius[y_],self.i_radius[z_])
			#BR
			elif self.direction == 1:
				r_triangle_2_core(-1,0,-self.i_radius[z_],self.i_radius[x_],self.i_radius[y_],-self.i_radius[z_])
				r_triangle_2_core(-1,0,self.i_radius[z_],self.i_radius[x_],self.i_radius[y_],self.i_radius[z_])
			#TL
			elif self.direction == 2:
				r_triangle_2_core(1,2,-self.i_radius[z_],-self.i_radius[x_],-self.i_radius[y_],-self.i_radius[z_])
				r_triangle_2_core(1,2,self.i_radius[z_],-self.i_radius[x_],-self.i_radius[y_],self.i_radius[z_])
			#TR
			elif self.direction == 3:
				r_triangle_2_core(1,2,-self.i_radius[z_],self.i_radius[x_],-self.i_radius[y_],-self.i_radius[z_])
				r_triangle_2_core(1,2,self.i_radius[z_],self.i_radius[x_],-self.i_radius[y_],self.i_radius[z_])
		
		#Faces
		#======			
		if self.core:
			#BL
			if self.direction == 0:
				RT3F0 = ([3,4,10,9],[5,3,9,11],[0,1,4,3],[1,7,10,4],[8,2,5,11],[2,0,3,5],[0,6,7,1],[2,8,6,0])
				me.faces.extend(RT3F0)
				if self.fill:
					RT3F1 = ([7,6,9,10],[6,8,11,9])
					me.faces.extend(RT3F1)
			#BR
			elif self.direction == 1:
				RT3F2 = ([3,4,10,9],[4,5,11,10],[0,1,4,3],[1,2,5,4],[2,8,11,5],[6,0,3,9],[0,6,7,1],[1,7,8,2])
				me.faces.extend(RT3F2)
				if self.fill:
					RT3F3 = ([6,9,10,7],[7,10,11,8])
					me.faces.extend(RT3F3)	
			#TL
			elif self.direction == 2:
				RT3F4 = ([3,5,11,9],[4,3,9,10],[0,2,5,3],[2,8,11,5],[7,1,4,10],[1,0,3,4],[0,6,8,2],[1,7,6,0])
				me.faces.extend(RT3F4)
				if self.fill:
					RT3F5 = ([6,9,11,8],[7,10,9,6])
					me.faces.extend(RT3F5)
			#TR
			elif self.direction == 3:
				RT3F6 = ([3,9,10,4],[4,10,11,5],[0,3,4,1],[1,4,5,2],[2,5,11,8],[6,9,3,0],[0,1,7,6],[1,2,8,7])
				me.faces.extend(RT3F6)
				if self.fill:
					RT3F7 = ([6,7,10,9],[7,8,11,10])
					me.faces.extend(RT3F7)	         
		else: 
			#BL & BR
			if self.direction == 0 or self.direction == 1:
				RT3F8 = ([3,4,5],[0,1,4,3],[1,2,5,4],[2,0,3,5],[0,2,1])
				me.faces.extend(RT3F8)
			#TL & TR
			elif self.direction == 2 or self.direction == 3:
				RT3F9 = ([3,5,4],[0,3,4,1],[1,4,5,2],[2,5,3,0],[0,1,2])
				me.faces.extend(RT3F9)
		
		Window.EditMode(self.e_set)
		
	
#==================
#Tetrahedron class
#==================
class TetrahedronMesh(CreateMesh):
	"""Create a tetrahedron - inherited from CreateMesh"""
	
	def __init__(self,type,ax,e_set,name):
		super(TetrahedronMesh, self).__init__(type,ax,e_set,name)
		
	def create(self):
		"""Extend the verts and faces"""
		super(TetrahedronMesh, self).create()
		
		#Vertices
		#=========
		tev=[]
		for y in xrange(-1,0):
			y = y * self.o_radius[y_]
			for x in xrange(-1,2,2):
				x = x * self.o_radius[x_]
				tev.append([x,y,-self.o_radius[z_]])
		me.verts.extend(tev)			 
		me.verts.extend(0.0,self.o_radius[y_],-self.o_radius[z_])
		me.verts.extend(0.0,0.0,self.o_radius[z_])
		
		#Faces
		#======	
		TEF = ([0,2,1],[0,1,3],[1,2,3],[2,0,3])	
		me.faces.extend(TEF)
		
		Window.EditMode(self.e_set)

#==============
#Pyramid class
#==============
class PyramidMesh(CreateMesh):
	"""Create a pyramid - inherited from CreateMesh"""
	
	def __init__(self,type,ax,e_set,name):
		super(PyramidMesh, self).__init__(type,ax,e_set,name)
		
	def create(self):
		"""Extend the verts and faces"""
		super(PyramidMesh, self).create()
		
		#Vertices
		#=========
		pyv=[]
		for x in xrange(-1,2,2):
			x = x * self.o_radius[x_]
			for y in xrange(-1,2,2):
				y = y * self.o_radius[y_]
				pyv.append([x,y,-self.o_radius[z_]])
		me.verts.extend(pyv) 
		me.verts.extend(0.0,0.0,self.o_radius[z_])
		
		#Faces
		#======			
		PYF = ([4,0,2],[4,2,3],[4,3,1],[4,1,0],[0,1,3,2])
		me.faces.extend(PYF)
		
		Window.EditMode(self.e_set)

#==============
#Diamond class
#==============
class DiamondMesh(CreateMesh):
	"""Create a diamond - inherited from CreateMesh"""
	
	def __init__(self,type,ax,e_set,name):
		super(DiamondMesh, self).__init__(type,ax,e_set,name)
		
	def create(self):
		"""Extend the verts and faces"""
		super(DiamondMesh, self).create()
		
		#Vertices
		#=========
		div=[]
		for x in xrange(-1,2,2):
			x = x * self.o_radius[x_]
			for y in xrange(-1,2,2):
				y = y * self.o_radius[y_]
				div.append([x,y,0.0])
		me.verts.extend(div) 
		me.verts.extend(0.0,0.0,self.o_radius[z_])
		me.verts.extend(0.0,0.0,-self.o_radius[z_])
		
		#Faces
		#======			
		DIF = ([4,0,2],[4,2,3],[4,3,1],[4,1,0],[5,2,0],[5,3,2],[5,3,1],[5,1,0])
		me.faces.extend(DIF)

		Window.EditMode(self.e_set)	

#===========
#Cone class
#===========
class ConeMesh(CreateMesh):
	"""Create a cone - inherited from CreateMesh"""
	
	def __init__(self,segments,type,ax,e_set,name):
		super(ConeMesh, self).__init__(type,ax,e_set,name)
		self.segments = segments
	
	def create(self):
		"""Extend the verts and faces"""
		super(ConeMesh, self).create()
		
		# Vertices
		#==========
		cov=[]
		for sv in xrange(self.segments):
			u = (sv/self.segments)
			x = self.o_radius[x_]*cos(2*pi*u) 
			y = self.o_radius[y_]*sin(2*pi*u) 
			z = -self.o_radius[z_] 
			cov.append([x,y,z])
		me.verts.extend(cov)
		
		me.verts.extend(0.0,0.0,-self.o_radius[z_])		
		me.verts.extend(0.0,0.0,self.o_radius[z_])
		
		#Faces
		#======  
		cof=[]
		for bf in xrange(0,self.segments):
			bf0 = bf
			bf1 = (bf+1) %self.segments
			cof.append([self.segments,bf1,bf0])
		me.faces.extend(cof)
		
		cof=[]	
		for cw in xrange(0,self.segments):
			cw0 = cw
			cw1 = (cw+1) %self.segments
			cw2 = (self.segments + 1)
			cof.append([cw2,cw0,cw1])
		me.faces.extend(cof)

		Window.EditMode(self.e_set)

		cov=[]
		cof=[]

#=============
#Sphere class    
#=============		
class SphereMesh(CreateMesh):
	"""Create a sphere - inherited from CreateMesh"""
	
	def __init__(self, segments,rings,type,ax,e_set,name):
		super(SphereMesh, self).__init__(type,ax,e_set,name)
		self.segments = segments
		self.rings = rings
	
	def create(self):
		"""Extend the verts and faces"""
		super(SphereMesh, self).create()
		
		spv = []
		spf = []
		
		#Vertices
		#==========
		for rv in xrange(1,self.rings):
			v = (rv/self.rings)
			for sv in xrange(self.segments):
				u = (sv/self.segments)
				x = self.o_radius[x_]*sin(2*pi*u)*sin(pi*v) 
				y =	self.o_radius[y_]*cos(2*pi*u)*sin(pi*v) 
				z = self.o_radius[z_]*cos(pi*v)
				spv.append([x,y,z])
				
		me.verts.extend(spv)
		
		me.verts.extend(0,0,self.o_radius[z_])
		me.verts.extend(0,0,-self.o_radius[z_])
		
		#Faces
		#======
		for rf in xrange(1,self.rings - 1):
			v = rf %self.rings
			for sf in xrange(1,self.segments + 1):
				u = sf %self.segments
				f0 = (v * self.segments) + u
				f1 = (v * self.segments) + sf - 1
				f2 = ((rf - 1) * self.segments) + sf -1
				f3 = ((rf - 1) * self.segments) + u
				spf.append([f0,f1,f2,f3])
		
		me.faces.extend(spf)
		
		spf=[]					
		total = (self.rings*self.segments)
		cv_t = total - (self.segments)	
		for ft in xrange(self.segments):
			ft_0 = ft
			ft_1 = (ft + 1) %self.segments
			spf.append([cv_t,ft_1,ft_0])
			
		me.faces.extend(spf)
			
		spf=[]
		adjust_b = cv_t - self.segments
		cv_b = cv_t + 1	
		for fb in xrange(adjust_b, cv_t):
			fb_0 = fb
			fb_1 = (fb+1) %self.segments + adjust_b
			spf.append([cv_b,fb_0,fb_1])
			
		me.faces.extend(spf)

		Window.EditMode(self.e_set)
		
		spv = []
		spf = []

#============
#Torus class     
#============
class TorusMesh(CreateMesh):
	"""Create a torus - inherited from CreateMesh"""
	
	def __init__(self,mp,segments,rings,type,ax,e_set,name):
		super(TorusMesh, self).__init__(type,ax,e_set,name)
		self.mp = mp
		self.segments = segments
		self.rings = rings
	
	def create(self):
		"""Extend the verts and faces"""
		super(TorusMesh, self).create()
		
		tov = []
		tof = []
		
		#Vertices
		#==========
		for rv in range(self.rings):
			v = (rv/self.rings)
			for sv in range(self.segments):
				u = (sv/self.segments)
				x = (self.o_radius[x_]*sin(2*pi*u)*(2+(0.5*sin(2*pi*v))))/self.mp 
				y =	(self.o_radius[y_]*cos(2*pi*u)*(2+(0.5*sin(2*pi*v))))/self.mp 
				z = (self.o_radius[z_]*cos(2*pi*v)*2.5)/self.mp 
				tov.append([x,y,z])
				
		me.verts.extend(tov)
		
		#Faces
		#======
		for rf in range(1,self.rings+1):
			v = rf %self.rings
			for sf in range(1,self.segments + 1):
				u = sf %self.segments
				f0 = (v * self.segments) + u
				f1 = (v * self.segments) + (sf - 1)
				f2 = ((rf - 1) * self.segments) + (sf -1)
				f3 = ((rf - 1) * self.segments) + u
				tof.append([f0,f1,f2,f3])
				
		me.faces.extend(tof)

		Window.EditMode(self.e_set)
		
		tov = []
		tof = []
		
		
		
		
######################################################################################################################################
#                                                                                                                                    #
#																																	 #
#																																	 #

"""This is still a work in progress and although not final all parts work the way they are supposed to. Better structure design 
would be needed if anyone ever wanted to import this module to use the mesh classes but right now, although possible, that's not 
it's purpose. 
		
"""