#ifndef MEMORYALLOCATOR_H
#define MEMORYALLOCATOR_H

#include <stdio.h>
#include <stdlib.h>

#define HEAP_BASE 16
#define UCHAR unsigned char

/**
 * Customized memory allocators that allocates/deallocates memory in chunks
 *
 * @author Tao Ju
 */



/**
 * Base class of memory allocators
 */
class VirtualMemoryAllocator
{
public:
	virtual UCHAR * allocate( ) = 0 ;
	virtual void deallocate( UCHAR * obj ) = 0 ;
	virtual void destroy( ) = 0 ;
	virtual void printInfo( ) = 0 ;

	virtual int getAllocated( ) = 0 ;
	virtual int getAll( ) = 0 ;
	virtual int getBytes( ) = 0 ;
};

/**
 * Dynamic memory allocator - allows allocation/deallocation
 * 
 * Note: there are 4 bytes overhead for each allocated yet unused object.
 */
template < int N >
class MemoryAllocator : public VirtualMemoryAllocator
{
private:

	/// Constants
	int HEAP_UNIT, HEAP_MASK ;

	/// Data array
	UCHAR ** data ;

	/// Allocation stack
	UCHAR *** stack ;

	/// Number of data blocks
	int datablocknum ;

	/// Number of stack blocks
	int stackblocknum ;

	/// Size of stack
	int stacksize ;

	/// Number of available objects on stack
	int available ;

	/**
	 * Allocate a memory block
	 */
	void allocateDataBlock ( )
	{
		// Allocate a data block
		datablocknum += 1 ;
		data = ( UCHAR ** )realloc( data, sizeof ( UCHAR * ) * datablocknum ) ;
		data[ datablocknum - 1 ] = ( UCHAR * )malloc( HEAP_UNIT * N ) ;

		// Update allocation stack
		for ( int i = 0 ; i < HEAP_UNIT ; i ++ )
		{
			stack[ 0 ][ i ] = ( data[ datablocknum - 1 ] + i * N ) ;
		}
		available = HEAP_UNIT ;
	}

	/**
	 * Allocate a stack block, to store more deallocated objects
	 */
	void allocateStackBlock( )
	{
		// Allocate a stack block
		stackblocknum += 1 ;
		stacksize += HEAP_UNIT ;
		stack = ( UCHAR *** )realloc( stack, sizeof ( UCHAR ** ) * stackblocknum ) ;
		stack[ stackblocknum - 1 ] = ( UCHAR ** )malloc( HEAP_UNIT * sizeof ( UCHAR * ) ) ;
	}


public:
	/**
	 * Constructor
	 */
	MemoryAllocator( )
	{
		HEAP_UNIT = 1 << HEAP_BASE ;
		HEAP_MASK = ( 1 << HEAP_BASE ) - 1 ;

		data = ( UCHAR ** )malloc( sizeof( UCHAR * ) ) ;
		data[ 0 ] = ( UCHAR * )malloc( HEAP_UNIT * N ) ;
		datablocknum = 1 ;

		stack = ( UCHAR *** )malloc( sizeof ( UCHAR ** ) ) ;
		stack[ 0 ] = ( UCHAR ** )malloc( HEAP_UNIT * sizeof ( UCHAR * ) ) ;
		stackblocknum = 1 ;
		stacksize = HEAP_UNIT ;
		available = HEAP_UNIT ;

		for ( int i = 0 ; i < HEAP_UNIT ; i ++ )
		{
			stack[ 0 ][ i ] = ( data[ 0 ] + i * N ) ;
		}
	}

	/**
	 * Destructor
	 */
	void destroy( )
	{
		int i ;
		for ( i = 0 ; i < datablocknum ; i ++ )
		{
			delete[] data[ i ] ;
		}
		for ( i = 0 ; i < stackblocknum ; i ++ )
		{
			delete[] stack[ i ] ;
		}
		delete[] data ;
		delete[] stack ;
	}

	/**
	 * Allocation method
	 */
	UCHAR * allocate ( )
	{
		if ( available == 0 )
		{
			allocateDataBlock ( ) ;
		}

		// printf("Allocating %d\n", header[ allocated ]) ;
		available -- ;
		return stack[ available >> HEAP_BASE ][ available & HEAP_MASK ] ;
	}

	/**
	 * De-allocation method
	 */
	void deallocate ( UCHAR * obj )
	{
		if ( available == stacksize )
		{
			allocateStackBlock ( ) ;
		}

		// printf("De-allocating %d\n", ( obj - data ) / N ) ;
		stack[ available >> HEAP_BASE ][ available & HEAP_MASK ] = obj ;
		available ++ ;
		// printf("%d %d\n", allocated, header[ allocated ]) ;
	}

	/**
	 * Print information
	 */
	void printInfo ( )
	{
		printf("Bytes: %d Used: %d Allocated: %d Maxfree: %d\n", getBytes(), getAllocated(), getAll(), stacksize ) ;
	}

	/**
	 * Query methods
	 */
	int getAllocated( )
	{
		return HEAP_UNIT * datablocknum - available ;	
	};

	int getAll( )
	{
		return HEAP_UNIT * datablocknum ;
	};

	int getBytes( )
	{
		return N ;	
	};
};

/**
 * Monotonic memory allocator - allows for allocation only
 * 
 * Note: does not provide de-allocation method
 */
template < int N >
class MonotonicMemoryAllocator : public VirtualMemoryAllocator
{
private:

	/// Constants
	int HEAP_UNIT, HEAP_MASK ;

	/// Data array
	UCHAR ** data ;

	/// Number of blocks
	int blocknum ;

	/// Total size of objects
	int totalsize ;

	/// Number of allocated objects
	int allocated ;

	/// Size of the first block
	int firstblocksize ;

	/**
	 * Allocate a block
	 */
	void allocateBlock ( )
	{
		int oldsize = totalsize ;
		totalsize += HEAP_UNIT ;
		blocknum += 1 ;

		// printf("Allocating... block size: %d %d\n", totalsize, totalsize * N ) ;
		data = ( UCHAR ** )realloc( data, sizeof ( UCHAR * ) * blocknum ) ;
		data[ blocknum - 1 ] = ( UCHAR * )malloc( HEAP_UNIT * N ) ;
	}


public:
	/**
	 * Constructor
	 */
	MonotonicMemoryAllocator( int length )
	{
		HEAP_UNIT = 1 << HEAP_BASE ;
		HEAP_MASK = ( 1 << HEAP_BASE ) - 1 ;

		firstblocksize = length ;
		totalsize = length ;
		blocknum = 1 ;
		allocated = 0 ;
		data = ( UCHAR ** )malloc( sizeof( UCHAR * ) ) ;
		data[ 0 ] = ( UCHAR * )malloc( totalsize * N ) ;
	}

	/**
	 * Destructor
	 */
	void destroy( )
	{
		for ( int i = 0 ; i < blocknum ; i ++ )
		{
			delete[] data[ i ] ;
		}
		delete[] data ;
	}

	/**
	 * Allocation method
	 */
	UCHAR * allocate ( )
	{
		if ( allocated == totalsize )
		{
			allocateBlock ( ) ;
		}

		// printf("Allocating %d\n", header[ allocated ]) ;
		int nallocated = allocated ;
		allocated ++ ;

		if ( nallocated < firstblocksize )
		{
			return ( data[ 0 ] + nallocated * N ) ;
		}
		else
		{
			nallocated -= firstblocksize ;
			return ( data[ 1 + ( nallocated >> HEAP_BASE ) ] + ( nallocated & HEAP_MASK ) * N ) ;
		}
	}

	/**
	 * De-allocation method
	 */
	void deallocate ( UCHAR * obj )
	{
		printf("Attempt to deallocate from an allocation-only heap...Exit.\n") ;
		exit( 0 ) ;
	} ;

	/**
	 * Print information
	 */
	void printInfo ( )
	{
		printf("Bytes: %d Allocated: %d All: %d\n", N, allocated, totalsize ) ;
	}

	/**
	 * Query methods
	 */
	int getAllocated( )
	{
		return allocated ;	
	};

	int getAll( )
	{
		return totalsize ;
	};

	int getBytes( )
	{
		return N ;	
	};
};

/**
 * Serial memory allocator - allows for serial allocation and deallocation
 */
template < int N >
class SerialMemoryAllocator : public VirtualMemoryAllocator
{
private:

	/// Constants
	int HEAP_UNIT, HEAP_MASK ;

	/// Data array
	UCHAR ** data ;

	/// Number of blocks
	int blocknum ;

	/// Total size of objects
	int totalsize ;

	/// Number of allocated objects
	int allocated ;

	/**
	 * Allocate a block
	 */
	void allocateBlock ( )
	{
		int oldsize = totalsize ;
		totalsize += HEAP_UNIT ;
		blocknum += 1 ;

		// printf("Allocating... block size: %d %d\n", totalsize, totalsize * N ) ;
		data = ( UCHAR ** )realloc( data, sizeof ( UCHAR * ) * blocknum ) ;
		data[ blocknum - 1 ] = ( UCHAR * )malloc( HEAP_UNIT * N ) ;
	}


public:
	/**
	 * Constructor
	 */
	SerialMemoryAllocator( )
	{
		HEAP_UNIT = 1 << HEAP_BASE ;
		HEAP_MASK = ( 1 << HEAP_BASE ) - 1 ;

		totalsize = HEAP_UNIT ;
		blocknum = 1 ;
		allocated = 0 ;
		data = ( UCHAR ** )malloc( sizeof( UCHAR * ) ) ;
		data[ 0 ] = ( UCHAR * )malloc( totalsize * N ) ;
	}

	/**
	 * Destructor
	 */
	void destroy( )
	{
		for ( int i = 0 ; i < blocknum ; i ++ )
		{
			delete[] data[ i ] ;
		}
		delete[] data ;
	}

	/**
	 * Allocation method
	 */
	UCHAR * allocate ( )
	{
		if ( allocated == totalsize )
		{
			allocateBlock ( ) ;
		}

		// printf("Allocating %d\n", header[ allocated ]) ;
		int nallocated = allocated ;
		allocated ++ ;
		return ( data[ nallocated >> HEAP_BASE ] + ( nallocated & HEAP_MASK ) * N ) ;
	}

	/**
	 * Random-access de-allocation method: not implemented
	 */
	void deallocate ( UCHAR * obj )
	{
		printf("Attempt to deallocate from an allocation-only heap...Exit.\n") ;
		exit( 0 ) ;
	} ;

	/**
	 * Serial De-allocation method
	 */
	void deallocateSerial ( int num )
	{
		allocated -= num ;
	}

	/**
	 * Get element
	 */
	UCHAR * getAllocatedAt ( int index )
	{
		if ( index >= 0 )
		{
			return ( data[ index >> HEAP_BASE ] + ( index & HEAP_MASK ) * N ) ;
		}

		return NULL ;
	}

	/**
	 * Print information
	 */
	void printInfo ( )
	{
		printf("Bytes: %d Allocated: %d All: %d\n", N, allocated, totalsize ) ;
	}

	/**
	 * Query methods
	 */
	int getAllocated( )
	{
		return allocated ;	
	};

	int getAll( )
	{
		return totalsize ;
	};

	int getBytes( )
	{
		return N ;	
	};
};



#endif