/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.iotdb.db.queryengine.plan.execution.config.metadata.relational;

import org.apache.iotdb.commons.schema.column.ColumnHeader;
import org.apache.iotdb.commons.schema.column.ColumnHeaderConstant;
import org.apache.iotdb.commons.schema.table.TableNodeStatus;
import org.apache.iotdb.commons.schema.table.TableType;
import org.apache.iotdb.confignode.rpc.thrift.TTableInfo;
import org.apache.iotdb.db.queryengine.common.header.DatasetHeader;
import org.apache.iotdb.db.queryengine.common.header.DatasetHeaderFactory;
import org.apache.iotdb.db.queryengine.plan.execution.config.ConfigTaskResult;
import org.apache.iotdb.db.queryengine.plan.execution.config.IConfigTask;
import org.apache.iotdb.db.queryengine.plan.execution.config.executor.IConfigTaskExecutor;
import org.apache.iotdb.rpc.TSStatusCode;

import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.SettableFuture;
import org.apache.tsfile.common.conf.TSFileConfig;
import org.apache.tsfile.enums.TSDataType;
import org.apache.tsfile.read.common.block.TsBlockBuilder;
import org.apache.tsfile.utils.Binary;

import java.util.Comparator;
import java.util.List;
import java.util.function.Predicate;
import java.util.stream.Collectors;

public class ShowTablesDetailsTask implements IConfigTask {

  private final String database;
  private final Predicate<String> checkCanShowTable;

  public ShowTablesDetailsTask(final String database, final Predicate<String> checkCanShowTable) {
    this.database = database;
    this.checkCanShowTable = checkCanShowTable;
  }

  @Override
  public ListenableFuture<ConfigTaskResult> execute(final IConfigTaskExecutor configTaskExecutor)
      throws InterruptedException {
    return configTaskExecutor.showTables(database, checkCanShowTable, true);
  }

  public static void buildTsBlock(
      final List<TTableInfo> tableInfoList,
      final SettableFuture<ConfigTaskResult> future,
      final Predicate<String> checkCanShowTable) {
    final List<TSDataType> outputDataTypes =
        ColumnHeaderConstant.showTablesDetailsColumnHeaders.stream()
            .map(ColumnHeader::getColumnType)
            .collect(Collectors.toList());

    final TsBlockBuilder builder = new TsBlockBuilder(outputDataTypes);

    tableInfoList.stream()
        .filter(t -> checkCanShowTable.test(t.getTableName()))
        .sorted(Comparator.comparing(TTableInfo::getTableName))
        .forEach(
            tableInfo -> {
              builder.getTimeColumnBuilder().writeLong(0L);

              builder
                  .getColumnBuilder(0)
                  .writeBinary(new Binary(tableInfo.getTableName(), TSFileConfig.STRING_CHARSET));
              builder
                  .getColumnBuilder(1)
                  .writeBinary(new Binary(tableInfo.getTTL(), TSFileConfig.STRING_CHARSET));
              builder
                  .getColumnBuilder(2)
                  .writeBinary(
                      new Binary(
                          TableNodeStatus.values()[tableInfo.getState()].toString(),
                          TSFileConfig.STRING_CHARSET));
              if (tableInfo.isSetComment()) {
                builder
                    .getColumnBuilder(3)
                    .writeBinary(new Binary(tableInfo.getComment(), TSFileConfig.STRING_CHARSET));
              } else {
                builder.getColumnBuilder(3).appendNull();
              }
              builder
                  .getColumnBuilder(4)
                  .writeBinary(
                      new Binary(
                          tableInfo.isSetType()
                              ? TableType.values()[tableInfo.getType()].getName()
                              : TableType.BASE_TABLE.getName(),
                          TSFileConfig.STRING_CHARSET));

              builder.declarePosition();
            });

    final DatasetHeader datasetHeader = DatasetHeaderFactory.getShowTablesDetailsHeader();
    future.set(new ConfigTaskResult(TSStatusCode.SUCCESS_STATUS, builder.build(), datasetHeader));
  }
}
