/*
 * Decompiled with CFR 0.152.
 */
package org.apache.spark.network.client;

import io.netty.channel.Channel;
import io.netty.util.concurrent.Future;
import io.netty.util.concurrent.GenericFutureListener;
import java.io.Closeable;
import java.io.IOException;
import java.net.SocketAddress;
import java.nio.ByteBuffer;
import java.util.UUID;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import javax.annotation.Nullable;
import org.apache.commons.lang3.builder.ToStringBuilder;
import org.apache.commons.lang3.builder.ToStringStyle;
import org.apache.spark.network.buffer.ManagedBuffer;
import org.apache.spark.network.buffer.NioManagedBuffer;
import org.apache.spark.network.client.BaseResponseCallback;
import org.apache.spark.network.client.ChunkReceivedCallback;
import org.apache.spark.network.client.MergedBlockMetaResponseCallback;
import org.apache.spark.network.client.RpcResponseCallback;
import org.apache.spark.network.client.StreamCallback;
import org.apache.spark.network.client.TransportResponseHandler;
import org.apache.spark.network.protocol.ChunkFetchRequest;
import org.apache.spark.network.protocol.MergedBlockMetaRequest;
import org.apache.spark.network.protocol.OneWayMessage;
import org.apache.spark.network.protocol.RpcRequest;
import org.apache.spark.network.protocol.StreamChunkId;
import org.apache.spark.network.protocol.StreamRequest;
import org.apache.spark.network.protocol.UploadStream;
import org.apache.spark.network.util.NettyUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.sparkproject.guava.annotations.VisibleForTesting;
import org.sparkproject.guava.base.Preconditions;
import org.sparkproject.guava.base.Throwables;
import org.sparkproject.guava.util.concurrent.SettableFuture;

public class TransportClient
implements Closeable {
    private static final Logger logger = LoggerFactory.getLogger(TransportClient.class);
    private final Channel channel;
    private final TransportResponseHandler handler;
    @Nullable
    private String clientId;
    private volatile boolean timedOut;

    public TransportClient(Channel channel, TransportResponseHandler handler) {
        this.channel = Preconditions.checkNotNull(channel);
        this.handler = Preconditions.checkNotNull(handler);
        this.timedOut = false;
    }

    public Channel getChannel() {
        return this.channel;
    }

    public boolean isActive() {
        return !this.timedOut && (this.channel.isOpen() || this.channel.isActive());
    }

    public SocketAddress getSocketAddress() {
        return this.channel.remoteAddress();
    }

    public String getClientId() {
        return this.clientId;
    }

    public void setClientId(String id) {
        Preconditions.checkState(this.clientId == null, "Client ID has already been set.");
        this.clientId = id;
    }

    public void fetchChunk(long streamId, final int chunkIndex, final ChunkReceivedCallback callback) {
        if (logger.isDebugEnabled()) {
            logger.debug("Sending fetch chunk request {} to {}", (Object)chunkIndex, (Object)NettyUtils.getRemoteAddress(this.channel));
        }
        final StreamChunkId streamChunkId = new StreamChunkId(streamId, chunkIndex);
        StdChannelListener listener = new StdChannelListener(streamChunkId){

            @Override
            void handleFailure(String errorMsg, Throwable cause) {
                TransportClient.this.handler.removeFetchRequest(streamChunkId);
                callback.onFailure(chunkIndex, new IOException(errorMsg, cause));
            }
        };
        this.handler.addFetchRequest(streamChunkId, callback);
        this.channel.writeAndFlush((Object)new ChunkFetchRequest(streamChunkId)).addListener((GenericFutureListener)listener);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void stream(final String streamId, final StreamCallback callback) {
        StdChannelListener listener = new StdChannelListener(streamId){

            @Override
            void handleFailure(String errorMsg, Throwable cause) throws Exception {
                callback.onFailure(streamId, new IOException(errorMsg, cause));
            }
        };
        if (logger.isDebugEnabled()) {
            logger.debug("Sending stream request for {} to {}", (Object)streamId, (Object)NettyUtils.getRemoteAddress(this.channel));
        }
        TransportClient transportClient = this;
        synchronized (transportClient) {
            this.handler.addStreamCallback(streamId, callback);
            this.channel.writeAndFlush((Object)new StreamRequest(streamId)).addListener((GenericFutureListener)listener);
        }
    }

    public long sendRpc(ByteBuffer message, RpcResponseCallback callback) {
        if (logger.isTraceEnabled()) {
            logger.trace("Sending RPC to {}", (Object)NettyUtils.getRemoteAddress(this.channel));
        }
        long requestId = TransportClient.requestId();
        this.handler.addRpcRequest(requestId, callback);
        RpcChannelListener listener = new RpcChannelListener(requestId, callback);
        this.channel.writeAndFlush((Object)new RpcRequest(requestId, new NioManagedBuffer(message))).addListener((GenericFutureListener)listener);
        return requestId;
    }

    public void sendMergedBlockMetaReq(String appId, int shuffleId, int shuffleMergeId, int reduceId, MergedBlockMetaResponseCallback callback) {
        long requestId = TransportClient.requestId();
        if (logger.isTraceEnabled()) {
            logger.trace("Sending RPC {} to fetch merged block meta to {}", (Object)requestId, (Object)NettyUtils.getRemoteAddress(this.channel));
        }
        this.handler.addRpcRequest(requestId, callback);
        RpcChannelListener listener = new RpcChannelListener(requestId, callback);
        this.channel.writeAndFlush((Object)new MergedBlockMetaRequest(requestId, appId, shuffleId, shuffleMergeId, reduceId)).addListener((GenericFutureListener)listener);
    }

    public long uploadStream(ManagedBuffer meta, ManagedBuffer data, RpcResponseCallback callback) {
        if (logger.isTraceEnabled()) {
            logger.trace("Sending RPC to {}", (Object)NettyUtils.getRemoteAddress(this.channel));
        }
        long requestId = TransportClient.requestId();
        this.handler.addRpcRequest(requestId, callback);
        RpcChannelListener listener = new RpcChannelListener(requestId, callback);
        this.channel.writeAndFlush((Object)new UploadStream(requestId, meta, data)).addListener((GenericFutureListener)listener);
        return requestId;
    }

    public ByteBuffer sendRpcSync(ByteBuffer message, long timeoutMs) {
        final SettableFuture result = SettableFuture.create();
        this.sendRpc(message, new RpcResponseCallback(){

            @Override
            public void onSuccess(ByteBuffer response) {
                try {
                    ByteBuffer copy = ByteBuffer.allocate(response.remaining());
                    copy.put(response);
                    copy.flip();
                    result.set(copy);
                }
                catch (Throwable t) {
                    logger.warn("Error in responding RPC callback", t);
                    result.setException(t);
                }
            }

            @Override
            public void onFailure(Throwable e) {
                result.setException(e);
            }
        });
        try {
            return (ByteBuffer)result.get(timeoutMs, TimeUnit.MILLISECONDS);
        }
        catch (ExecutionException e) {
            throw Throwables.propagate(e.getCause());
        }
        catch (Exception e) {
            throw Throwables.propagate(e);
        }
    }

    public void send(ByteBuffer message) {
        this.channel.writeAndFlush((Object)new OneWayMessage(new NioManagedBuffer(message)));
    }

    public void removeRpcRequest(long requestId) {
        this.handler.removeRpcRequest(requestId);
    }

    public void timeOut() {
        this.timedOut = true;
    }

    @VisibleForTesting
    public TransportResponseHandler getHandler() {
        return this.handler;
    }

    @Override
    public void close() {
        this.channel.close().awaitUninterruptibly(10L, TimeUnit.SECONDS);
    }

    public String toString() {
        return new ToStringBuilder((Object)this, ToStringStyle.SHORT_PREFIX_STYLE).append("remoteAddress", (Object)this.channel.remoteAddress()).append("clientId", (Object)this.clientId).append("isActive", this.isActive()).toString();
    }

    private static long requestId() {
        return Math.abs(UUID.randomUUID().getLeastSignificantBits());
    }

    private class RpcChannelListener
    extends StdChannelListener {
        final long rpcRequestId;
        final BaseResponseCallback callback;

        RpcChannelListener(long rpcRequestId, BaseResponseCallback callback) {
            super("RPC " + rpcRequestId);
            this.rpcRequestId = rpcRequestId;
            this.callback = callback;
        }

        @Override
        void handleFailure(String errorMsg, Throwable cause) {
            TransportClient.this.handler.removeRpcRequest(this.rpcRequestId);
            this.callback.onFailure(new IOException(errorMsg, cause));
        }
    }

    private class StdChannelListener
    implements GenericFutureListener<Future<? super Void>> {
        final long startTime = System.currentTimeMillis();
        final Object requestId;

        StdChannelListener(Object requestId) {
            this.requestId = requestId;
        }

        public void operationComplete(Future<? super Void> future) throws Exception {
            if (future.isSuccess()) {
                if (logger.isTraceEnabled()) {
                    long timeTaken = System.currentTimeMillis() - this.startTime;
                    logger.trace("Sending request {} to {} took {} ms", new Object[]{this.requestId, NettyUtils.getRemoteAddress(TransportClient.this.channel), timeTaken});
                }
            } else {
                String errorMsg = String.format("Failed to send RPC %s to %s: %s", this.requestId, NettyUtils.getRemoteAddress(TransportClient.this.channel), future.cause());
                logger.error(errorMsg, future.cause());
                TransportClient.this.channel.close();
                try {
                    this.handleFailure(errorMsg, future.cause());
                }
                catch (Exception e) {
                    logger.error("Uncaught exception in RPC response callback handler!", (Throwable)e);
                }
            }
        }

        void handleFailure(String errorMsg, Throwable cause) throws Exception {
        }
    }
}

