/* $Id: bitwise.c,v 1.9 2005/12/28 23:54:29 jwp Exp $
 *
 * Copyright 2005, PostgresPy Project.
 * http://python.projects.postgresql.org
 *
 *//*
 * Interface to "bitwise" types.
 */
#include <setjmp.h>
#include <postgres.h>
#include <fmgr.h>
#include <access/heapam.h>
#include <access/htup.h>
#include <access/tupdesc.h>
#include <catalog/pg_proc.h>
#include <catalog/pg_type.h>
#include <catalog/pg_namespace.h>
#include <catalog/pg_conversion.h>
#include <catalog/pg_operator.h>
#include <catalog/pg_opclass.h>
#include <catalog/namespace.h>
#include <lib/stringinfo.h>
#include <nodes/params.h>
#include <parser/parse_func.h>
#include <tcop/dest.h>
#include <tcop/tcopprot.h>
#include <utils/array.h>
#include <utils/datum.h>
#include <utils/elog.h>
#include <utils/palloc.h>
#include <utils/builtins.h>
#include <utils/syscache.h>
#include <utils/typcache.h>
#include <utils/relcache.h>
#include <pypg/environment.h>
#include <pypg/postgres.h>

#include <Python.h>
#include <structmember.h>
#include <pypg/python.h>

#include <pypg/externs.h>
#include <pypg/type.h>
#include <pypg/type/object.h>
#include <pypg/type/bitwise.h>
#include <pypg/error.h>

#ifdef PGTYPE_BOOL
/*
 * The bool type is basically a formality that transparently creates
 * a True or False depending on the results of bool_in. Normally, BOOLOID types
 * will never reach here as it will already be converted the interface that does
 * the coercion.
 */
static PyObj
bool_typinput(PyObj subtype, PyObj ob)
{
	Datum datum = 0;
	PyObj obstr, rob;

	obstr = PyObject_Str(ob);
	PG_TRY();
	{
		datum = DirectFunctionCall1(boolin,
			PointerGetDatum(PyString_AS_STRING(obstr)));
	}
	PYPG_CATCH_END();

	Py_DECREF(obstr);
	if (PyErr_Occurred()) return(NULL);

	rob = datum == 0 ? Py_False : Py_True;
	Py_INCREF(rob);

	return(rob);
}

static PyObj
bool_typoutput(PyObj subtype, PyObj ob)
{
	int truth;
	char *str = NULL;
	PyObj rob;

	truth = PyObject_IsTrue(ob);
	if (truth < 0) return(NULL);

	PgError_TRAP(str = (char *) DirectFunctionCall1(boolout, (Datum) truth));
	if (PyErr_Occurred()) return(NULL);

	rob = PyString_FromString(str);
	pfree(str);
	return(rob);
}

static PyObj
bool_typreceive(PyObj subtype, PyObj ob)
{
	Datum datum = 0;
	PyObj obstr, rob;
	StringInfoData buf;

	obstr = PyObject_Str(ob);
	PG_TRY();
	{
		initStringInfo(&buf);
		appendBinaryStringInfo(&buf,
			PyString_AS_STRING(obstr),
			PyString_GET_SIZE(obstr)
		);

		datum = DirectFunctionCall1(boolrecv, PointerGetDatum(&buf));
	}
	PYPG_CATCH_END();

	Py_DECREF(obstr);
	pfree(buf.data);

	if (PyErr_Occurred()) return(NULL);

	rob = datum == 0 ? Py_False : Py_True;

	Py_INCREF(rob);
	return(rob);
}

static PyObj
bool_typsend(PyObj subtype, PyObj ob)
{
	int truth;
	Datum datum = 0;
	PyObj rob;

	truth = PyObject_IsTrue(ob);
	if (truth < 0) return(NULL);
	PgError_TRAP(datum = DirectFunctionCall1(boolsend, (Datum) truth));
	if (PyErr_Occurred()) return(NULL);

	rob = PyString_FromStringAndSize(VARDATA(datum), VARSIZE(datum)-VARHDRSZ);
	return(rob);
}

static PyMethodDef bool_methods[] = {
	{"typinput", (PyCFunction) bool_typinput, METH_O | METH_CLASS,
	PyDoc_STR("special bool typinput that returns True or False")},
	{"typoutput", (PyCFunction) bool_typoutput, METH_O | METH_CLASS,
	PyDoc_STR("special bool typoutput that returns 't' or 'f'")},
	{"typreceive", (PyCFunction) bool_typreceive, METH_O | METH_CLASS,
	PyDoc_STR("special bool typreceive that returns True or False")},
	{"typsend", (PyCFunction) bool_typsend, METH_O | METH_CLASS,
	PyDoc_STR("special bool typsend that returns '\\x00' or '\\x01'")},
	{NULL}
};

static PyObj
bool_new(PyTypeObject *subtype, PyObj args, PyObj kw)
{
	PyObj ob, rob = NULL;
	Datum datum = 0;

	if (!PyArg_ParseTuple(args, "O", &ob))
		return(NULL);

	if (ob == Py_False || ob == Py_True)
		return(ob);

	ob = PyObject_Str(ob);

	PgError_TRAP(
		datum = DirectFunctionCall1(boolin,
						PointerGetDatum(PyString_AS_STRING(ob)))
	);
	Py_DECREF(ob);

	if (!PyErr_Occurred())
	{
		if (datum == 0)
			rob = Py_False;
		else
			rob = Py_True;
		Py_INCREF(rob);
	}
	return(rob);
}

PyDoc_STRVAR(PyPg_bool_Type_Doc,
	"Base type for arbitrary object types"
);

PyPgTypeObject PyPg_bool_Type = {{
	PyObject_HEAD_INIT(&PyPgType_Type)
	0,												/* ob_size */
	"bool",										/* tp_name */
	sizeof(struct PyPgObject),				/* tp_basicsize */
	0,												/* tp_itemsize */
	NULL,											/* tp_dealloc */
	NULL,											/* tp_print */
	NULL,											/* tp_getattr */
	NULL,											/* tp_setattr */
	NULL,											/* tp_compare */
	NULL,											/* tp_repr */
	NULL,											/* tp_as_number */
	NULL,											/* tp_as_sequence */
	NULL,											/* tp_as_mapping */
	NULL,											/* tp_hash */
	NULL,											/* tp_call */
	NULL,											/* tp_str */
	NULL,											/* tp_getattro */
	NULL,											/* tp_setattro */
	NULL,											/* tp_as_buffer */
	PyPg_TPFLAGS_DEFAULT,							/* tp_flags */
	PyPg_bool_Type_Doc,								/* tp_doc */
	NULL,											/* tp_traverse */
	NULL,											/* tp_clear */
	NULL,											/* tp_richcompare */
	0,												/* tp_weaklistoffset */
	NULL,											/* tp_iter */
	NULL,											/* tp_iternext */
	bool_methods,								/* tp_methods */
	NULL,											/* tp_members */
	NULL,											/* tp_getset */
	(PyTypeObject *) &PyPgObject_Type,	/* tp_base */
	NULL,											/* tp_dict */
	NULL,											/* tp_descr_get */
	NULL,											/* tp_descr_set */
	0,												/* tp_dictoffset */
	NULL,											/* tp_init */
	NULL,											/* tp_alloc */
	bool_new,									/* tp_new */
},
};
#endif /* BOOL */

#ifdef PGTYPE_BYTEA
static int
bytea_length(PyObj self)
{
	Datum d;
	d = PyPgObject_FetchDatum(self);

	Assert(d != 0);

	PgError_TRAP(d = DirectFunctionCall1(byteaoctetlen, d));

	if (PyErr_Occurred()) return(-1);
	return((int) d);
}

static PyObj
bytea_concat(PyObj self, PyObj seq)
{
	return(PyPgObject_Operate("||", self, seq));
}

static PyObj
bytea_item(PyObj self, int item)
{
	Datum rd = 0, datum = PyPgObject_FetchDatum(self);
	int32 len = 0;
	PyObj rob;

	Assert(datum != 0);

	PgError_TRAP(len = DatumGetInt32(DirectFunctionCall1(byteaoctetlen, datum)));
	if (PyErr_Occurred()) return(NULL);

	if (item >= len)
	{
		PyErr_Format(PyExc_IndexError,
			"index %d is out of range(%d)", item, len);
		return(NULL);
	}

	PgError_TRAP(rd = (Datum) DatumGetByteaPSlice(datum, item, 1));
	if (PyErr_Occurred()) return(NULL);

	rob = PyPgObject_New(self->ob_type, rd);
	pfree(DatumGetPointer(rd));

	return(rob);
}

static PyObj
bytea_slice(PyObj self, int from, int to)
{
	Datum rd = 0, datum = PyPgObject_FetchDatum(self);
	PyObj rob = NULL;

	Assert(datum != 0);

	PgError_TRAP(rd = (Datum) DatumGetByteaPSlice(datum, from, to - from));
	if (PyErr_Occurred()) return(NULL);

	rob = PyPgObject_New(self->ob_type, rd);
	pfree(DatumGetPointer(rd));

	return(rob);
}

static int
bytea_contains(PyObj self, PyObj item)
{
	return(0);
}

static PySequenceMethods bytea_as_sequence = {
	bytea_length,		/* sq_length */
	bytea_concat,		/* sq_concat */
	NULL,					/* sq_repeat */
	bytea_item,			/* sq_item */
	bytea_slice,		/* sq_slice */
	NULL,					/* sq_ass_item */
	NULL,					/* sq_ass_slice */
	bytea_contains,	/* sq_contains */
	NULL,					/* sq_inplace_concat */
	NULL,					/* sq_inplace_repeat */
};

static PyNumberMethods bytea_as_number = {
	bytea_concat,	/* nb_add */
};

static PyObj
bytea_new(PyTypeObject *subtype, PyObj args, PyObj kw)
{
	PyObj source, srcstr, rob = NULL;
	Size sz;
	Datum datum = 0;

	if (!PyArg_ParseTuple(args, "O", &source))
		return(NULL);
	if (source->ob_type == subtype) return(source);

	srcstr = PyObject_Str(source);
	if (srcstr == NULL) return(NULL);

	sz = PyString_GET_SIZE(srcstr) + VARHDRSZ;
	PgError_TRAP(datum = PointerGetDatum(palloc(sz)));
	if (!PyErr_Occurred())
	{
		VARATT_SIZEP(datum) = sz;
		memcpy(
			VARDATA(datum),
			PyString_AS_STRING(srcstr),
			PyString_GET_SIZE(srcstr)
		);
		rob = PyPgObject_New(subtype, datum);
	}

	Py_DECREF(srcstr);
	return(rob);
}

PyDoc_STRVAR(PyPg_bytea_Type_Doc, "bytea interface type");

PyPgTypeObject PyPg_bytea_Type = {{
	PyObject_HEAD_INIT(&PyPgType_Type)
	0,												/* ob_size */
	"bytea",										/* tp_name */
	sizeof(struct PyPg_bytea),				/* tp_basicsize */
	0,												/* tp_itemsize */
	NULL,											/* tp_dealloc */
	NULL,											/* tp_print */
	NULL,											/* tp_getattr */
	NULL,											/* tp_setattr */
	NULL,											/* tp_compare */
	NULL,											/* tp_repr */
	&bytea_as_number,							/* tp_as_number */
	&bytea_as_sequence,						/* tp_as_sequence */
	NULL,											/* tp_as_mapping */
	NULL,											/* tp_hash */
	NULL,											/* tp_call */
	NULL,											/* tp_str */
	NULL,											/* tp_getattro */
	NULL,											/* tp_setattro */
	NULL,											/* tp_as_buffer */
	PyPg_TPFLAGS_DEFAULT,						/* tp_flags */
	PyPg_bytea_Type_Doc,						/* tp_doc */
	NULL,											/* tp_traverse */
	NULL,											/* tp_clear */
	NULL,											/* tp_richcompare */
	0,												/* tp_weaklistoffset */
	NULL,											/* tp_iter */
	NULL,											/* tp_iternext */
	NULL,											/* tp_methods */
	NULL,											/* tp_members */
	NULL,											/* tp_getset */
	(PyTypeObject *) &PyPgObject_Type,	/* tp_base */
	NULL,											/* tp_dict */
	NULL,											/* tp_descr_get */
	NULL,											/* tp_descr_set */
	0,												/* tp_dictoffset */
	NULL,											/* tp_init */
	NULL,											/* tp_alloc */
	bytea_new,									/* tp_new */
},
};
#endif /* BYTEA */

#ifdef PGTYPE_BIT
PyDoc_STRVAR(PyPg_bit_Type_Doc, "bit interface type");

PyPgTypeObject PyPg_bit_Type = {{
	PyObject_HEAD_INIT(&PyPgType_Type)
	0,												/* ob_size */
	"bit",										/* tp_name */
	sizeof(struct PyPgObject),				/* tp_basicsize */
	0,												/* tp_itemsize */
	NULL,											/* tp_dealloc */
	NULL,											/* tp_print */
	NULL,											/* tp_getattr */
	NULL,											/* tp_setattr */
	NULL,											/* tp_compare */
	NULL,											/* tp_repr */
	NULL,											/* tp_as_number */
	NULL,											/* tp_as_sequence */
	NULL,											/* tp_as_mapping */
	NULL,											/* tp_hash */
	NULL,											/* tp_call */
	NULL,											/* tp_str */
	NULL,											/* tp_getattro */
	NULL,											/* tp_setattro */
	NULL,											/* tp_as_buffer */
	PyPg_TPFLAGS_DEFAULT,						/* tp_flags */
	PyPg_bit_Type_Doc,						/* tp_doc */
	NULL,											/* tp_traverse */
	NULL,											/* tp_clear */
	NULL,											/* tp_richcompare */
	0,												/* tp_weaklistoffset */
	NULL,											/* tp_iter */
	NULL,											/* tp_iternext */
	NULL,											/* tp_methods */
	NULL,											/* tp_members */
	NULL,											/* tp_getset */
	(PyTypeObject *) &PyPgObject_Type,	/* tp_base */
	NULL,											/* tp_dict */
	NULL,											/* tp_descr_get */
	NULL,											/* tp_descr_set */
	0,												/* tp_dictoffset */
	NULL,											/* tp_init */
	NULL,											/* tp_alloc */
	PyPgArbitrary_new,						/* tp_new */
},
};
#endif /* BIT */

#ifdef PGTYPE_VARBIT
PyDoc_STRVAR(PyPg_varbit_Type_Doc, "varbit interface type");

PyPgTypeObject PyPg_varbit_Type = {{
	PyObject_HEAD_INIT(&PyPgType_Type)
	0,												/* ob_size */
	"varbit",									/* tp_name */
	sizeof(struct PyPgObject),				/* tp_basicsize */
	0,												/* tp_itemsize */
	NULL,											/* tp_dealloc */
	NULL,											/* tp_print */
	NULL,											/* tp_getattr */
	NULL,											/* tp_setattr */
	NULL,											/* tp_compare */
	NULL,											/* tp_repr */
	NULL,											/* tp_as_number */
	NULL,											/* tp_as_sequence */
	NULL,											/* tp_as_mapping */
	NULL,											/* tp_hash */
	NULL,											/* tp_call */
	NULL,											/* tp_str */
	NULL,											/* tp_getattro */
	NULL,											/* tp_setattro */
	NULL,											/* tp_as_buffer */
	PyPg_TPFLAGS_DEFAULT,						/* tp_flags */
	PyPg_varbit_Type_Doc,					/* tp_doc */
	NULL,											/* tp_traverse */
	NULL,											/* tp_clear */
	NULL,											/* tp_richcompare */
	0,												/* tp_weaklistoffset */
	NULL,											/* tp_iter */
	NULL,											/* tp_iternext */
	NULL,											/* tp_methods */
	NULL,											/* tp_members */
	NULL,											/* tp_getset */
	(PyTypeObject *) &PyPgObject_Type,	/* tp_base */
	NULL,											/* tp_dict */
	NULL,											/* tp_descr_get */
	NULL,											/* tp_descr_set */
	0,												/* tp_dictoffset */
	NULL,											/* tp_init */
	NULL,											/* tp_alloc */
	PyPgArbitrary_new,						/* tp_new */
},
};
#endif /* VARBIT */

/*
 * vim: ts=3:sw=3:noet:
 */
