/* Module:          SQLTables.c
 *
 * Description:     Returns the list of table, catalog, or schema names, and table types, 
 *					stored in a specific data source. 
 *
 * Classes:         
 *
 * API functions:   SQLTables
 *
 * Comments:        See "notice.txt" for copyright and license information.
 *
 */

#include "driver.h"

SQLRETURN SQL_API SQLTables(
							SQLHSTMT		hDrvStmt,
							SQLCHAR			* szTableQualifier,
							SQLSMALLINT		cbTableQualifier,
							SQLCHAR			* szTableOwner,
							SQLSMALLINT		cbTableOwner,
							SQLCHAR			* szTableName,
							SQLSMALLINT		cbTableName,
							SQLCHAR			* szTableType,
							SQLSMALLINT		cbTableType
							)
{
	static char *func = "SQLTables";
	StatementClass *stmt = (StatementClass *) hDrvStmt;
	StatementClass *tbl_stmt;
	TupleNode *row;
	SQLHSTMT htbl_stmt;
	RETCODE result;
	char *tableType;
	char tables_query[STD_STATEMENT_LEN];
	char table_name[MAX_INFO_STRING], table_owner[MAX_INFO_STRING], relkind_or_hasrules[MAX_INFO_STRING];
	ConnectionClass *conn;
	ConnInfo *ci;
	char *prefix[32], prefixes[MEDIUM_REGISTRY_LEN];
	char *table_type[32], table_types[MAX_INFO_STRING];
	char show_system_tables, show_regular_tables, show_views;
	char regular_table, view, systable;
	int i;

	mylog("%s: entering...stmt=%u\n", func, stmt);

	if( ! stmt)
	{
		SC_log_error(func, "", NULL);
		return SQL_INVALID_HANDLE;
	}

	stmt->manual_result = TRUE;
	stmt->errormsg_created = TRUE;

	conn = (ConnectionClass *) (stmt->hdbc);
	ci = &stmt->hdbc->connInfo;

	result = SQLAllocStmt( stmt->hdbc, &htbl_stmt);
	if((result != SQL_SUCCESS) && (result != SQL_SUCCESS_WITH_INFO))
	{
		stmt->errornumber = STMT_NO_MEMORY_ERROR;
		stmt->errormsg = "Couldn't allocate statement for SQLTables result.";
		SC_log_error(func, "", stmt);
		return SQL_ERROR;
	}
	tbl_stmt = (StatementClass *) htbl_stmt;

	/* ********************************************************************** */
	/*	Create the query to find out the tables */
	/* ********************************************************************** */

	if (PG_VERSION_GE(conn, 7.1))
	{ /* view is represented by its relkind since 7.1 */
		strcpy(tables_query, "select relname, usename, relkind from pg_class, pg_user");
		strcat(tables_query, " where relkind in ('r', 'v')");
	}
	else
	{
		strcpy(tables_query, "select relname, usename, relhasrules from pg_class, pg_user");
		strcat(tables_query, " where relkind = 'r'");
	}

	my_strcat(tables_query, " and usename like '%.*s'", szTableOwner, cbTableOwner);
	my_strcat(tables_query, " and relname like '%.*s'", szTableName, cbTableName);


	/*	Parse the extra systable prefix  */
	strcpy(prefixes, globals.extra_systable_prefixes);
	i = 0;
	prefix[i] = strtok(prefixes, ";");
	while (prefix[i] && i<32) {
		prefix[++i] = strtok(NULL, ";");
	}

	/*	Parse the desired table types to return */
	show_system_tables = FALSE;
	show_regular_tables = FALSE;
	show_views = FALSE;

	/*	make_string mallocs memory */
	tableType = make_string(szTableType, cbTableType, NULL);
	if (tableType)
	{
		strcpy(table_types, tableType);
		free(tableType);
		i = 0;
		table_type[i] = strtok(table_types, ",");
		while (table_type[i] && i<32)
		{
			table_type[++i] = strtok(NULL, ",");
		}

		/*	Check for desired table types to return */
		i = 0;
		while (table_type[i])
		{
			if ( strstr(table_type[i], "SYSTEM TABLE"))
			{
				show_system_tables = TRUE;
			}
			else if ( strstr(table_type[i], "TABLE"))
			{
				show_regular_tables = TRUE;
			}
			else if ( strstr(table_type[i], "VIEW"))
			{
				show_views = TRUE;
			}

			i++;
		}
	}
	else 
	{
		show_regular_tables = TRUE;
		show_views = TRUE;
	}

	/*  If not interested in SYSTEM TABLES then filter them out
		to save some time on the query.  If treating system tables
		as regular tables, then dont filter either.
	*/
	if ( ! atoi(ci->show_system_tables) && ! show_system_tables)
	{
		strcat(tables_query, " and relname !~ '^" POSTGRES_SYS_PREFIX);

		/*	Also filter out user-defined system table types */
		i = 0;
		while(prefix[i])
		{
			strcat(tables_query, "|^");
			strcat(tables_query, prefix[i]);
			i++;
		}

		strcat(tables_query, "'");
	}


	/* match users */
	if (PG_VERSION_LT(conn, 7.1)) /* filter out large objects in older versions */
	{
		strcat(tables_query, " and relname !~ '^xinv[0-9]+'"); 
	}

	strcat(tables_query, " and usesysid = relowner");
	strcat(tables_query, " order by relname");

	/* ********************************************************************** */

	result = SQLExecDirect(htbl_stmt, tables_query, strlen(tables_query));
	if((result != SQL_SUCCESS) && (result != SQL_SUCCESS_WITH_INFO))
	{
		stmt->errormsg = SC_create_errormsg(htbl_stmt);
		stmt->errornumber = tbl_stmt->errornumber;
		SC_log_error(func, "", stmt);
		SQLFreeStmt(htbl_stmt, SQL_DROP);
		return SQL_ERROR;
	}

    result = SQLBindCol(htbl_stmt, 1, SQL_C_CHAR,
                        table_name, MAX_INFO_STRING, NULL);
    if((result != SQL_SUCCESS) && (result != SQL_SUCCESS_WITH_INFO))
	{
		stmt->errormsg = tbl_stmt->errormsg;
		stmt->errornumber = tbl_stmt->errornumber;
		SC_log_error(func, "", stmt);
		SQLFreeStmt(htbl_stmt, SQL_DROP);
        return SQL_ERROR;
    }

    result = SQLBindCol(htbl_stmt, 2, SQL_C_CHAR,
                        table_owner, MAX_INFO_STRING, NULL);
    if((result != SQL_SUCCESS) && (result != SQL_SUCCESS_WITH_INFO))
	{
		stmt->errormsg = tbl_stmt->errormsg;
		stmt->errornumber = tbl_stmt->errornumber;
		SC_log_error(func, "", stmt);
		SQLFreeStmt(htbl_stmt, SQL_DROP);
        return SQL_ERROR;
    }
    result = SQLBindCol(htbl_stmt, 3, SQL_C_CHAR,
                        relkind_or_hasrules, MAX_INFO_STRING, NULL);
    if((result != SQL_SUCCESS) && (result != SQL_SUCCESS_WITH_INFO)) 
	{
		stmt->errormsg = tbl_stmt->errormsg;
		stmt->errornumber = tbl_stmt->errornumber;
		SC_log_error(func, "", stmt);
		SQLFreeStmt(htbl_stmt, SQL_DROP);
        return SQL_ERROR;
    }

	stmt->result = QR_Constructor();
	if(!stmt->result) 
	{
		stmt->errormsg = "Couldn't allocate memory for SQLTables result.";
		stmt->errornumber = STMT_NO_MEMORY_ERROR;
		SC_log_error(func, "", stmt);
		SQLFreeStmt(htbl_stmt, SQL_DROP);
		return SQL_ERROR;
	}

	/* the binding structure for a statement is not set up until */
	/* a statement is actually executed, so we'll have to do this ourselves. */
	extend_bindings(stmt, 5);
	
	/* set the field names */
	QR_set_num_fields(stmt->result, 5);
	QR_set_field_info(stmt->result, 0, "TABLE_QUALIFIER", PG_TYPE_TEXT, MAX_INFO_STRING);
	QR_set_field_info(stmt->result, 1, "TABLE_OWNER", PG_TYPE_TEXT, MAX_INFO_STRING);
	QR_set_field_info(stmt->result, 2, "TABLE_NAME", PG_TYPE_TEXT, MAX_INFO_STRING);
	QR_set_field_info(stmt->result, 3, "TABLE_TYPE", PG_TYPE_TEXT, MAX_INFO_STRING);
	QR_set_field_info(stmt->result, 4, "REMARKS", PG_TYPE_TEXT, 254);

	/* add the tuples */
	result = SQLFetch(htbl_stmt);
	while((result == SQL_SUCCESS) || (result == SQL_SUCCESS_WITH_INFO))
	{

		/*	Determine if this table name is a system table.
			If treating system tables as regular tables, then 
			no need to do this test.
		*/		
		systable = FALSE;
		if( ! atoi(ci->show_system_tables))
		{

			if ( strncmp(table_name, POSTGRES_SYS_PREFIX, strlen(POSTGRES_SYS_PREFIX)) == 0)
			{
				systable = TRUE;
			}

			else
			{			/* Check extra system table prefixes */
				i = 0;
				while (prefix[i])
				{
					mylog("table_name='%s', prefix[%d]='%s'\n", table_name, i, prefix[i]);
					if (strncmp(table_name, prefix[i], strlen(prefix[i])) == 0) 
					{
						systable = TRUE;
						break;
					}
					i++;
				}
			}
		}

		/*	Determine if the table name is a view */
		if (PG_VERSION_GE(conn, 7.1)) /* view is represented by its relkind since 7.1 */
		{
			view = (relkind_or_hasrules[0] == 'v');
		}
		else
		{
			view = (relkind_or_hasrules[0] == '1');
		}

		/*	It must be a regular table */
		regular_table = ( ! systable && ! view);


		/*	Include the row in the result set if meets all criteria */
		/*	NOTE: Unsupported table types (i.e., LOCAL TEMPORARY, ALIAS, etc)
					will return nothing */
		if ( (systable && show_system_tables) ||
			 (view && show_views) || 
			 (regular_table && show_regular_tables))
		{

			row = (TupleNode *)malloc(sizeof(TupleNode) + (5 - 1) * sizeof(TupleField));

			set_tuplefield_string(&row->tuple[0], "");

			/* I have to hide the table owner from Access, otherwise it */
			/* insists on referring to the table as 'owner.table'. */
			/* (this is valid according to the ODBC SQL grammar, but */
			/* Postgres won't support it.) */
			/* set_tuplefield_string(&row->tuple[1], table_owner); */

			mylog("SQLTables: table_name = '%s'\n", table_name);

			set_tuplefield_string(&row->tuple[1], "");
			set_tuplefield_string(&row->tuple[2], table_name);
			set_tuplefield_string(&row->tuple[3], systable ? "SYSTEM TABLE" : (view ? "VIEW" : "TABLE"));
			set_tuplefield_string(&row->tuple[4], "");

			QR_add_tuple(stmt->result, row);
		}
		result = SQLFetch(htbl_stmt);
    }
	if(result != SQL_NO_DATA_FOUND)
	{
		stmt->errormsg = SC_create_errormsg(htbl_stmt);
		stmt->errornumber = tbl_stmt->errornumber;
		SC_log_error(func, "", stmt);
		SQLFreeStmt(htbl_stmt, SQL_DROP);
		return SQL_ERROR;
	}

	/* also, things need to think that this statement is finished so */
	/* the results can be retrieved. */
	stmt->status = STMT_FINISHED;

	/* set up the current tuple pointer for SQLFetch */
	stmt->currTuple = -1;
	stmt->rowset_start = -1;
	stmt->current_col = -1;

	SQLFreeStmt(htbl_stmt, SQL_DROP);
	mylog("SQLTables(): EXIT,  stmt=%u\n", stmt);
	return SQL_SUCCESS;
}
