import { IrcBridge } from "./IrcBridge";
import { BridgeRequest } from "../models/BridgeRequest";
import { IrcServer } from "../irc/IrcServer";
import { PowerLevelContent } from "matrix-appservice-bridge";
/**
 * This class is supplimentary to the IrcHandler class. This
 * class handles incoming mode changes as well as computing the new
 * power level state.
 */
export declare class RoomAccessSyncer {
    private ircBridge;
    private pendingPLChanges;
    private accessRefreshCount;
    private powerLevelsForRoom;
    constructor(ircBridge: IrcBridge);
    get powerLevelGracePeriod(): number;
    /**
     * Called when a m.room.power_levels is forwarded to the bridge. This should
     * happen when a Matrix user or the bridge changes the power levels for a room.
     * @param \event The matrix event.
     */
    onMatrixPowerlevelEvent(event: {
        room_id: string;
        content: PowerLevelContent;
    }): void;
    /**
     * Function to determine current power levels. Will either use
     * cached value or fetch from the homeserver.
     * @param roomId The room to fetch the state from.
     */
    private getCurrentPowerlevels;
    removePowerLevels(roomId: string, users: string[], req: BridgeRequest): Promise<void>;
    /**
     * Bulk change a set of users permissions from a room. If users is empty
     * or no changes were made, this will no-op.
     * @param roomId A roomId
     * @param users A set of userIds
     */
    private changePowerLevels;
    setPowerLevel(roomId: string, userId: string, level: number | null, req: BridgeRequest): Promise<void>;
    /**
     * Called when an IRC user sets a mode on another user or channel.
     * @param {BridgeReqeust} req The request tracking the operation.
     * @param {IrcServer} server The server the channel and users are part of
     * @param {string} channel Which channel was the mode set in.
     * @param {string} by Which user set the mode
     * @param {string} mode The mode string
     * @param {boolean} enabled Whether the mode was enabled or disabled.
     * @param {string|null} arg This is usually the affected user, if applicable.
     */
    onMode(req: BridgeRequest, server: IrcServer, channel: string, by: string, mode: string, enabled: boolean, arg: string | null): Promise<void>;
    /**
     * Called when an IRC server responds to a mode request.
     * @param {BridgeRequest} req The request tracking the operation.
     * @param {IrcServer} server The server the channel and users are part of
     * @param {string} channel Which channel was the mode(s) set in.
     * @param {string} mode The mode string, which may contain many modes.
     */
    onModeIs(req: BridgeRequest, server: IrcServer, channel: string, mode: string): Promise<void | void[]>;
    onFailedMessage(req: BridgeRequest, server: IrcServer, channel: string): Promise<void>;
    /**
     * If a mode given in PRIVATE_MODES is found, change a room's join rules
     * to match.
     * @param {BridgeReqeust} req The request tracking the operation.
     * @param {IrcServer} server The server the channel and users are part of
     * @param {string} channel Which channel was the mode(s) set in.
     * @param {string} mode The mode string.
     * @param {boolean} enabled Was the mode enabled or disabled.
     */
    private onPrivateMode;
    /**
     * This is called when a "m" mode is toggled in a channel. This will either
     * enable or disable a users permission to speak unless they are voiced.
     * @param {BridgeReqeust} req The request tracking the operation.
     * @param {IrcServer} server The server the channel and users are part of
     * @param {string} channel Which channel was the mode(s) set in.
     * @param {boolean} enabled Has moderation been turned on or off.
     */
    private onModeratedChannelToggle;
    /**
     * Modify the join rules of a room, setting it either to invite only or public.
     * @param {MatrixRoom} room The room to set the join_rules for.
     * @param {boolean} isInviteOnly Set to true to make invite only, set to false to
     *                               make the room public
     */
    private setMatrixRoomAsInviteOnly;
}
